/*====================================================================

filename:     trx_ppc_cpu.h
project:      GCemu
created:      2004-6-18
mail:		  duddie@walla.com

Copyright (c) 2005 Duddie & Tratax

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

====================================================================*/
#pragma once

#include "system/types.h"

extern double *fpu_as_double;// points to fpr register for double usage.
extern double *ps0_double; // ps0 points to same as fpr registers
extern double *ps1_double; // ps1 points (wrongly) to another set of registers 
extern uint64 *ps0_int;
extern uint64 ps1_int[32]; // ps1 is (wrongly) another set of registers

// GEKKO SPECIFICS !!!
// defines related to gekko special purpose registers
#define HID2_LSQE (1<<31)
#define HID2_WPE (1<<30)
#define HID2_PSE (1<<29)
#define HID2_LCE (1<<28)
#define DMAQL    (1<<24)

#define DMAL_LD  (1<<4)
#define DMAL_T	 (1<<1)
#define DMAL_F   (1)

#define ICTC_EN  (1)
#define ICTC_FI  (1<<1)

#define MSR_SF		(1<<31)
#define MSR_UNKNOWN	(1<<30)
#define MSR_UNKNOWN2	(1<<27)
#define MSR_VEC		(1<<25)
#define MSR_KEY		(1<<19)		// 603e
#define MSR_POW		(1<<18)
#define MSR_TGPR	(1<<15)		// 603(e)
#define MSR_ILE		(1<<16)
#define MSR_EE		(1<<15)
#define MSR_PR		(1<<14)
#define MSR_FP		(1<<13)
#define MSR_ME		(1<<12)
#define MSR_FE0		(1<<11)
#define MSR_SE		(1<<10)
#define MSR_BE		(1<<9)
#define MSR_FE1		(1<<8)
#define MSR_IP		(1<<6)
#define MSR_IR		(1<<5)
#define MSR_DR		(1<<4)
#define MSR_PM		(1<<2)
#define MSR_RI		(1<<1)
#define MSR_LE		(1)

/*
xer bits:
0 so
1 ov
2 carry
3-24 res
25-31 number of bytes for lswx/stswx
*/

#define XER_SO (1<<31)
#define XER_OV (1<<30)
#define XER_CA (1<<29)
#define XER_n(v) ((v)&0x7f)

/*
cr: .67
 0- 3 cr0
 4- 7 cr1
 8-11 cr2
12-15 cr3
16-19 cr4
20-23 cr5
24-27 cr6
28-31 cr7
*/

#define CR_CR0(v) ((v)>>28)
#define CR_CR1(v) (((v)>>24)&0xf)
#define CR_CRx(v, x) (((v)>>(4*(7-(x))))&0xf)

/*
cr0 bits: .68
lt
gt
eq
so
*/

#define CR_CR0_LT (1<<31)
#define CR_CR0_GT (1<<30)
#define CR_CR0_EQ (1<<29)
#define CR_CR0_SO (1<<28)

/*
cr1 bits: .68
4 Floating-point exception (FX)
5 Floating-point enabled exception (FEX)
6 Floating-point invalid exception (VX)
7 Floating-point overflow exception (OX)
*/

#define CR_CR1_FX (1<<27)
#define CR_CR1_FEX (1<<26)
#define CR_CR1_VX (1<<25)
#define CR_CR1_OX (1<<24)

/*
FPSCR bits: .70

*/
 
#define FPSCR_FX (1<<31)
#define FPSCR_FEX (1<<30)
#define FPSCR_VX (1<<29)
#define FPSCR_OX (1<<28)
#define FPSCR_UX (1<<27)
#define FPSCR_ZX (1<<26)
#define FPSCR_XX (1<<25)
#define FPSCR_VXSNAN (1<<24)
#define FPSCR_VXISI (1<<23)
#define FPSCR_VXIDI (1<<22)
#define FPSCR_VXZDZ (1<<21)
#define FPSCR_VXIMZ (1<<20)
#define FPSCR_VXVC (1<<19)
#define FPSCR_FR (1<<18)
#define FPSCR_FI (1<<17)

#define FPSCR_FPRF(v) (((v)>>12)&0x1f)

#define FPSCR_res0 (1<<11)
#define FPSCR_VXSOFT (1<<10)
#define FPSCR_VXSQRT (1<<9)
#define FPSCR_VXCVI (1<<8)
#define FPSCR_VXVE (1<<7)
#define FPSCR_VXOE (1<<6)
#define FPSCR_VXUE (1<<5)
#define FPSCR_VXZE (1<<4)
#define FPSCR_VXXE (1<<3)
#define FPSCR_VXNI (1<<2)
#define FPSCR_RN(v) ((v)&3)

#define FPSCR_RN_NEAR 0
#define FPSCR_RN_ZERO 1
#define FPSCR_RN_PINF 2
#define FPSCR_RN_MINF 3 

#define BLOCKEND_CONT 0
#define BLOCKEND_LOOP 1
#define BLOCKEND_STOP 2

// maximum amount of instructions in a single block
//#define MAX_BLOCK_SIZE 0x1fff
#define MAX_BLOCK_SIZE 256

#define PPC_DEC 22
#define PPC_SRR0 26
#define PPC_SRR1 27
#define PPC_TBL 284
#define PPC_TBH 285
#define PPC_GQR0 912
#define PPC_HID2 920
#define PPC_WPAR 921
#define PPC_DMAU 922
#define PPC_DMAL 923
#define PPC_ICTC 1019

struct TRX_PPC_Registers
{	
	// UISA
	uint32 gpr[32];
	uint64 fpr[32];
	uint64 ps1[32];
	uint32 cr;
	uint32 fpscr;
	uint32 xer;	// spr 1
	uint32 xer_ca;
	uint32 lr;	// spr 8
	uint32 ctr;	// spr 9
	uint32 msr;
	uint32 sr[16]; // segment registers
		
	//   misc 
	uint32 spr[4096];
	
	// not part of PPC registers but keeps state
	uint32 pc;
	uint32 npc;
	uint32 opcode;

	bool   exception_pending;
	bool   dec_exception;
	bool   ext_exception;
	bool   stop_exception;
	uint32 blockend;
	uint32 block_instr;
	uint32 block_startPC;
	int	   interrupt_cycles_leftover; // when we break a block early by setting cycles_left to 0, here we store how much was REALLY left
	bool   interrupt_signalled;
};

extern struct TRX_PPC_Registers *CPUcurrmode;

void trx_ppc_step();
bool trx_cpu_init(void);
void trx_ppc_signal_interrupt(void);
int trx_ppc_exception(uint32 type, uint32 flags = 0);
void trx_ppc_debug_run();
void trx_ppc_run(void);

#define LOCKEDCACHE_MASK 0x3fff

extern uint8 lockedcache[16*1024];
extern uint8 *virtualmemory;

extern void *pLockedcache;

extern	uint32 cpuslice;
extern	int cpuslice_left;

bool is_breakpoint(uint32 address);
bool add_breakpoint(uint32 address);
bool remove_breakpoint(uint32 address);


extern	uint32 cpu_instr_breakpoint_flag;
extern	uint32 cpu_stop_running;
extern	uint32 cpu_is_running;
extern	uint32 cpu_instr_breakpoint_list[];

// handles DMA transfers
void dma_engine(void);

// fpu control 
extern uint16 fpucontrol_roundzero; 
extern uint16 fpucontrol_default;

extern uint32 fcmpo_nan_flags;
extern uint32 fcmpu_nan_flags;


///////////////
// move to trx_ppc_int.h later on
#define PPC_EXC_UNKNOWN 0
#define PPC_EXC_SYS_RESET 0x100
#define PPC_EXC_DSI 0x00300
#define PPC_EXC_EXT_INT 0x00500
#define PPC_EXC_ALIGNMENT 0x00600
#define PPC_EXC_PROGRAM 0x00700
#define PPC_EXC_NO_FPU 0x00800
#define PPC_EXC_DEC 0x00900
//Reserved 0x00A00
//Reserved 0x00B00
#define PPC_EXC_SC 0x00C00
#define PPC_EXC_PERF_MON 0xF00
#define PPC_EXC_ALTIVEC 0xF20
#define PPC_EXC_ALTIVEC_ASSIST 0x1600
#define PPC_EXC_TAU 0x1700

#define PPC_EXC_PROGRAM_FLOAT (1<<20)
#define PPC_EXC_PROGRAM_ILL   (1<<19)
#define PPC_EXC_PROGRAM_PRIV  (1<<18)
#define PPC_EXC_PROGRAM_TRAP  (1<<17)
#define PPC_EXC_PROGRAM_NEXT  (1<<16)

#define FPU_SIGN_BIT (0x8000000000000000ULL)

#define HID0_ICFI (1<<20)

#define CR0_LT (1<<31)
#define CR0_GT (1<<30)
#define CR0_EQ (1<<29)
#define CR0_SO (1<<28)

//
// helper pointers for assembly test version of interpreter
extern uint32 *prd, *prs, *pra, *prb, *prc;
extern double *frd, *frs, *fra, *frb, *frc, *fps1;
extern uint64 *qrd, *qrb;
extern uint32 zero, one;
extern uint32 cr_gt, cr_lt, cr_eq, cr_so;
extern uint32 trx_ppc_cmp_and_mask[8];

// repeated for interpreter and recompiler which do not include memory_interface
extern uint32 mem_iread(uint32 address);
// direct access to GC main memory is NOT recommended!
extern uint8 *gMemory;

// memory access routines for selfcheck mode
uint8 master_mem_read8(uint32 address);
uint16 master_mem_read16(uint32 address);
uint32 master_mem_read32(uint32 address);
void master_mem_write8(uint32 address, uint8 val);
void master_mem_write16(uint32 address, uint16 val);
void master_mem_write32(uint32 address, uint32 val);

uint8 slave_mem_read8(uint32 address);
uint16 slave_mem_read16(uint32 address);
uint32 slave_mem_read32(uint32 address);
void slave_mem_write8(uint32 address, uint8 val);
void slave_mem_write16(uint32 address, uint16 val);
void slave_mem_write32(uint32 address, uint32 val);
