#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <windows.h>

#include "gxD3D.h"

#include "system/types.h"
#include "gp_defs.h"
#include "bp_regs.h"
#include "cp_regs.h"
#include "xf_regs.h"
#include "vertex_processor.h"
#include "debug/tracers.h"
#include "texture_environment.h"
#include "d3d_fun.h"

#define WITH_VERTEX_CACHE	0
#define WITH_MULTI_TEXTURE	1


uint32 test_buffer[1024*16];
uint32 *test_buf;

uint16 vx_swap16(uint16 data)
{
	return ((data >> 8) | (data << 8));
}

uint32 vx_swap32(uint32 data)
{
	return (data>>24)|((data>>8)&0xff00)|((data<<8)&0xff0000)|(data<<24);
}

#if WITH_VERTEX_CACHE
uint8 vx_pos_cache[65536 * 4 * 4];
uint8 vx_nor_cache[65536 * 4 * 4];
uint8 vx_col_cache[65536 * 4 * 4];
uint8 vx_tex_cache[65536 * 4 * 4];
uint8 vx_col_valid[65536];
uint8 vx_pos_valid[65536];
uint8 vx_nor_valid[65536];
uint8 vx_tex_valid[65536];

uint32 vx_col_watermark = 65536;
uint32 vx_pos_watermark = 65536;
uint32 vx_nor_watermark = 65536;
uint32 vx_tex_watermark = 65536;
#endif


#define STRUT_LN        130     // long side of strut
#define STRUT_SD        4       // short side of strut
#define JOINT_SD        10      // joint is a cube

#if USE_VERTEX_PTR
vx_vertex_ptr_t	vx_vertex_ptr;
uint8			vx_vertex_data_ub[VX_MAX_CMDS * 4 * 4];
#else
vx_vertex_t *vx_vertex_data;
vx_vertex_t vx_vertex_data_static[VX_MAX_CMDS];
#endif
uint8		*vx_vertex_data_size;

// three parameters describe almost every data: type, cnt, fmt
// fmt is the 

enum vx_type
{
	VX_T_NONE = 0,
	VX_T_NORM,
	VX_T_IDX8,
	VX_T_IDX16
};

enum vx_form
{
	VX_F_U8 = 0,
	VX_F_S8,
	VX_F_U16,
	VX_F_S16,
	VX_F_F32,
};

enum vx_col_fmt
{
	VX_F_RGB565 = 0,
    VX_F_RGB8,
    VX_F_RGBX8,
    VX_F_RGBA4,
    VX_F_RGBA6,
    VX_F_RGBA8


};

void vx_invalidate_cache(uint32 pos)
{
#if WITH_VERTEX_CACHE
	uint32 i;
	switch(pos)
	{
	case 0x0:
		for(i = 0 ; i < vx_pos_watermark ; i++)
			vx_pos_valid[i] = 0;
		vx_pos_watermark = 0;
		break;
	case 0x1:
		for(i = 0 ; i < vx_nor_watermark ; i++)
			vx_nor_valid[i] = 0;
		vx_nor_watermark = 0;
		break;
	case 0x2:
		for(i = 0 ; i < vx_col_watermark ; i++)
			vx_col_valid[i] = 0;
		vx_col_watermark = 0;
		break;
	case 0x4:
		for(i = 0 ; i < vx_tex_watermark ; i++)
			vx_tex_valid[i] = 0;
		vx_tex_watermark = 0;
		break;
	default:
		printf("GX: Vertex Cache invalidate called with wrong parameter: %d\n", pos);
		exit(1);
		break;
	}
#endif
}



typedef struct
{
	// maksimum 22 different values describing vertex. Actually 21 + 1 NULL
	// however position 0 is reserved for POS processor
	void	(*jump_table[22])(uint32);
	uint32	param[22];
	uint8	expected_size[VX_MAX_CMDS];
	uint32	size;			// one packet size
	uint32	vcd_lo, vcd_hi, vat_a, vat_b, vat_c;
	uint32	cmd_count;
	uint32	dxx_size;
	uint32	fvf;
} vx_table_t;

typedef struct
{
	void	(*processor)(uint32);
	uint32	exp_size;
	uint32	exp_count;
} vx_command_t;



// For debug purposes
uint32	vx_nor_debug_fmt;
uint32	vx_nor_debug_cnt;
uint32	vx_nor_debug_type;

uint32	vx_pos_debug_fmt;
uint32	vx_pos_debug_cnt;
uint32	vx_pos_debug_type;

uint32	vx_col_debug_fmt;
uint32	vx_col_debug_cnt;
uint32	vx_col_debug_type;

uint32	vx_tex_debug_fmt[8];
uint32	vx_tex_debug_cnt[8];
uint32	vx_tex_debug_type[8];

char *vx_str_type[4] = 
{
	"NONE", "NORM", "IDX8", "IDX16"
};

char *vx_str_fmt[8] = 
{
	"U8", "S8", "U16", "S16", 
	"F32", "UNK", "UNK", "UNK"
};

char *vx_str_col_fmt[8] = 
{
	"RGB565", "RGB8", "RGBX8", "RGBA4", "RGBA6", "RGBA8", "UNK", "UNK"
};

char *vx_str_pos_cnt[2] = 
{
	"XY", "XYZ"
};

char *vx_str_nor_cnt[2] = 
{
	"NOR", "N/B/T"
};

char *vx_str_col_cnt[2] = 
{
	"RGB", "RGBA"
};

char *vx_str_tex_cnt[2] = 
{
	"S", "ST"
};

// 8 tables for 8 vertex descriptors
vx_table_t vx_table[8];
vx_table_t *vx_table_selected;

extern vx_command_t vx_cmds_pos_n[16];
extern vx_command_t vx_cmds_pos_i8[16];
extern vx_command_t vx_cmds_pos_i16[16];
extern vx_command_t vx_cmds_col_n[16];
extern vx_command_t vx_cmds_col_i8[16];
extern vx_command_t vx_cmds_col_i16[16];
extern vx_command_t vx_cmds_tex_n[16];
extern vx_command_t vx_cmds_tex_i8[16];
extern vx_command_t vx_cmds_tex_i16[16];
extern vx_command_t vx_cmds_nor_n[16];
extern vx_command_t vx_cmds_nor_i8[16];
extern vx_command_t vx_cmds_nor_i16[16];




uint32 vx_get_text_vat(uint32 vat, uint32 i);

uint32 vx_prim_count;
uint32 vx_prim_mode;

D3DPRIMITIVETYPE	vx_prim_type;
uint32	vx_quad_mode = true;
uint32	vx_quad_counter;

uint32	*ex_buf;
uint32	empty_buf[1024];


void te_set_tev(void);

void vx_begin(vxb_t mode)
{
#if WITH_FRAGMENT_SHADER
	if (te_modified) 
		fsg_generate();
#else
//	if (te_modified) 
//		te_set_stage();
#endif // WITH_FRAGMENT_SHADER
	//	glBegin(mode)

	if (te_modified) 
		te_set_tev();

	vx_prim_mode = mode;
	test_buf = test_buffer;
	ex_buf = test_buffer;

	vx_quad_mode = true;	
	vx_quad_counter = 0;

}

void vx_begin_count(uint32 count)
{
	vx_prim_count = count;

	switch(vx_prim_mode)
	{
	case VX_TRIANGLES:
		vx_quad_mode = false;
		vx_prim_type = D3DPT_TRIANGLELIST;
		vx_prim_count = count / 3;
		break;
	case VX_TRIANGLE_STRIP:
		vx_quad_mode = false;
		vx_prim_type = D3DPT_TRIANGLESTRIP;
		vx_prim_count = count - 2;
		break;
	case VX_TRIANGLE_FAN:
		vx_quad_mode = false;
		vx_prim_type = D3DPT_TRIANGLEFAN;
		vx_prim_count = count - 2;
		break;
	case VX_LINE_STRIP:
		vx_quad_mode = false;
		vx_prim_type = D3DPT_LINESTRIP;
		vx_prim_count = count - 1;
		break;
	case VX_LINES:
		vx_quad_mode = false;
		vx_prim_type = D3DPT_LINELIST;
		vx_prim_count = count / 2;
		break;
	case VX_QUADS:
		vx_prim_type = D3DPT_TRIANGLELIST;
		vx_prim_count = count / 2;
		break;
	default:
		syslog_error(VX, "Unsupported MODE");
		return;
	}
}


void vx_end(void)
{
/*	if (vx_prim_type == D3DPT_LINELIST)
	{
		int i;
		float *tf = (float *)test_buffer;
		syslog(VX, "LINES\n");
		for(i = 0 ; i < vx_prim_count * 2 ; i++)
		{
			uint32 p = i * vx_table_selected->dxx_size/4;
			syslog(VX, "%08x %d %f, %f, %f %08x\n", test_buffer+p, i, tf[p+0], tf[p+1], tf[p+2], test_buffer[p+3]);
		}
	}
*/

	g_pd3dDevice->SetFVF(vx_table_selected->fvf);
	g_pd3dDevice->DrawPrimitiveUP(vx_prim_type, vx_prim_count, test_buffer, vx_table_selected->dxx_size);
}

void vx_add_vertex(float x, float y, float z)
{
	vx_vertex_t *xx;
	//syslog(VX, "ADD VERTEX: %08x %f, %f, %f\n", test_buf, x, y, z);
	xx = (vx_vertex_t *)test_buf;
	xx[0].f = x;
	xx[1].f = y;
	xx[2].f = z;
	test_buf+=3;
}

void vx_add_texc(float x, float y)
{
	vx_vertex_t *xx;
	//syslog(VX, "ADD VERTEX: %08x %f, %f, %f\n", test_buf, x, y, z);
	xx = (vx_vertex_t *)test_buf;
	xx[0].f = x;
	xx[1].f = y;
	test_buf += 2;
}

void vx_add_colour(uint32 val)
{
	*test_buf++ = val;
}


void vx_set_requested_size(uint32 pos, uint32 count, uint8 size)
{
	do
	{
#if USE_VERTEX_PTR
		vx_vertex_data_size[pos] = (count << 4) * size;
#endif
		vx_vertex_data_size[pos++] = (count << 4) | size;
	} while(--count);
}

void vertex_null_function(uint32 pos)
{
	//syslog("Matrix Index %d\n", vx_vertex_ptr.ubp[pos]);
}

extern uint32 xf_mat_col0;

void vx_null_col(uint32 pos)
{
	vx_add_colour(xf_mat_col0);
}


uint32 vx_prepare_table(uint32 vat)
{
	uint32 jump_table_pos;
	
	uint32 vcd_hi, vcd_lo;

	uint32 cmd_count;
	uint32 vat_a;
	uint32 mode;

	vx_table_selected = &vx_table[vat];
	vx_vertex_data_size = vx_table_selected->expected_size;

	vcd_lo = gp_cp_regs[0x50];
	vcd_hi = gp_cp_regs[0x60];

	vat_a = gp_cp_regs[0x70 + vat];

	if (vx_table_selected->vcd_lo == vcd_lo)
		if (vx_table_selected->vcd_hi == vcd_hi)
			if (vx_table_selected->vat_a == vat_a)
				if (vx_table_selected->vat_b == gp_cp_regs[0x80 + vat])
					if (vx_table_selected->vat_c == gp_cp_regs[0x90 + vat])
						return vx_table_selected->cmd_count;

	vx_table_selected->vcd_lo = vcd_lo;
	vx_table_selected->vcd_hi = vcd_hi;
	vx_table_selected->vat_a = vat_a;
	vx_table_selected->vat_b = gp_cp_regs[0x80 + vat];
	vx_table_selected->vat_c = gp_cp_regs[0x90 + vat];


	syslog(VX,"Check for changed VAT and VCD should be added here\n");
	// for now lets build table every time

	jump_table_pos = 1;


	cmd_count = 0;

	vx_table_selected->dxx_size = 0;
	vx_table_selected->fvf = 0;

	if (vcd_lo & (1 << 0))
	{
		// PMIDX
		// Position/Normal Matrix Index
		syslog(VX,"PT: Position/Normal Matrix Index - NOT SUPPORTED\n");
		vx_table_selected->jump_table[jump_table_pos] = vertex_null_function;
		vx_table_selected->param[jump_table_pos] = cmd_count;
		cmd_count += 1;
		jump_table_pos++;
	}
	if (vcd_lo & (0xff << 1))
	{
		// TnMIDX
		// Texture Matrix Index
		syslog_error(VX, "PT: Texture Matrix Index (%d) - NOT SUPPORTED\n", (vcd_lo >> 1) & 0xff);
	}
	if (vcd_lo & (0x3 << 9))
	{
		// POS
		// Position

		// for debugging purposes
		vx_pos_debug_type =	(vcd_lo >> 9) & 0x3;
		vx_pos_debug_fmt =	(vat_a >> 1) & 0x7;
		vx_pos_debug_cnt =	(vat_a) & 0x1;
		
		vx_table_selected->dxx_size += 12;		// 4 * 3 (X, Y, Z)
		vx_table_selected->fvf |= D3DFVF_XYZ;
		switch((vcd_lo >> 9) & 0x3)
		{
		case VX_T_NORM:
			mode = (vat_a) & 0xf;
			vx_table_selected->jump_table[0] = vx_cmds_pos_n[mode].processor;
			vx_table_selected->param[0] = cmd_count | ((vat_a << 4) & 0x1f00);
#if USE_VERTEX_PTR
			cmd_count += vx_cmds_pos_n[mode].exp_count * vx_cmds_pos_n[mode].exp_size;
#else
			vx_set_requested_size(cmd_count, vx_cmds_pos_n[mode].exp_count, vx_cmds_pos_n[mode].exp_size);
			cmd_count += vx_cmds_pos_n[mode].exp_count;
#endif
			break;
		case VX_T_IDX8:
			mode = (vat_a) & 0xf;
			vx_table_selected->jump_table[0] = vx_cmds_pos_i8[mode].processor;
			vx_table_selected->param[0] = cmd_count | ((vat_a << 4) & 0x1f00);
#if USE_VERTEX_PTR
			cmd_count += vx_cmds_pos_i8[mode].exp_count * vx_cmds_pos_i8[mode].exp_size;
#else
			vx_set_requested_size(cmd_count, vx_cmds_pos_i8[mode].exp_count, vx_cmds_pos_i8[mode].exp_size);
			cmd_count += vx_cmds_pos_i8[(vat_a) & 0xf].exp_count;
#endif
			break;
		case VX_T_IDX16:
			mode = (vat_a) & 0xf;
			vx_table_selected->jump_table[0] = vx_cmds_pos_i16[mode].processor;
			vx_table_selected->param[0] = cmd_count | ((vat_a << 4) & 0x1f00);
#if USE_VERTEX_PTR
			cmd_count += vx_cmds_pos_i16[mode].exp_count * vx_cmds_pos_i16[mode].exp_size;
#else
			vx_set_requested_size(cmd_count, vx_cmds_pos_i16[mode].exp_count, vx_cmds_pos_i16[mode].exp_size);
			cmd_count += vx_cmds_pos_i16[(vat_a) & 0xf].exp_count;
#endif
			break;
		default:
			syslog_error(VX, "PT: Position (%d) - NOT SUPPORTED\n", (vcd_lo >> 9) & 0x03);
			break;
		}
	}
	if (vcd_lo & (0x3 << 11))
	{
		// NRM
		// Normal
		if (vat_a & (1 << 9))
		{
			// NBT
			syslog_error(VX, "PT: NBT not supported\n");
		}

		vx_table_selected->dxx_size += 12;		// 4 * 3 (X, Y, Z)
		vx_table_selected->fvf |= D3DFVF_NORMAL;


		vx_nor_debug_type =	(vcd_lo >> 11) & 0x03;
		vx_nor_debug_fmt =	(vat_a >> 10) & 0x7;
		vx_nor_debug_cnt =	(vat_a >> 9) & 0x1;

		switch((vcd_lo >> 11) & 0x3)
		{
		case VX_T_NORM:
			mode = (vat_a >> 9) & 0xf;
			vx_table_selected->jump_table[jump_table_pos] = vx_cmds_nor_n[mode].processor;
			vx_table_selected->param[jump_table_pos] = cmd_count;
#if USE_VERTEX_PTR
			cmd_count += vx_cmds_nor_n[mode].exp_count * vx_cmds_nor_n[mode].exp_size;
#else
			vx_set_requested_size(cmd_count, vx_cmds_nor_n[mode].exp_count, vx_cmds_nor_n[mode].exp_size);
			cmd_count += vx_cmds_nor_n[mode].exp_count;
#endif
			jump_table_pos++;
			break;
		case VX_T_IDX8:
			mode = (vat_a >> 9) & 0xf;
			vx_table_selected->jump_table[jump_table_pos] = vx_cmds_nor_i8[mode].processor;
			vx_table_selected->param[jump_table_pos] = cmd_count;
#if USE_VERTEX_PTR
			cmd_count += vx_cmds_nor_i8[mode].exp_count * vx_cmds_nor_i8[mode].exp_size;
#else
			vx_set_requested_size(cmd_count, vx_cmds_nor_i8[mode].exp_count, vx_cmds_nor_i8[mode].exp_size);
			cmd_count += vx_cmds_nor_i8[mode].exp_count;
#endif
			jump_table_pos++;
			break;
		case VX_T_IDX16:
			mode = (vat_a >> 9) & 0xf;
			vx_table_selected->jump_table[jump_table_pos] = vx_cmds_nor_i16[mode].processor;
			vx_table_selected->param[jump_table_pos] = cmd_count;
#if USE_VERTEX_PTR
			cmd_count += vx_cmds_nor_i16[mode].exp_count * vx_cmds_nor_i16[mode].exp_size;
#else
			vx_set_requested_size(cmd_count, vx_cmds_nor_i16[mode].exp_count, vx_cmds_nor_i16[mode].exp_size);
			cmd_count += vx_cmds_nor_i16[mode].exp_count;
#endif
			jump_table_pos++;
			break;
		default:
			syslog_error(VX,"PT: Normal/Binormal/Tangent (%d) - NOT SUPPORTED\n", (vcd_lo >> 11) & 0x03);
			break;
		}
	} // NRM

	if (vcd_lo & (0x3 << 13))
	{
		// COL0
		// Color0
		// for debugging purposes
		vx_col_debug_type =	(vcd_lo >> 13) & 0x03;
		vx_col_debug_fmt =	(vat_a >> 14) & 0x7;
		vx_col_debug_cnt =	(vat_a >> 13) & 0x1;

		vx_table_selected->dxx_size += 4;	// ARGB (8bit ch)
		vx_table_selected->fvf |= D3DFVF_DIFFUSE;

		switch((vcd_lo >> 13) & 0x3)
		{
		case VX_T_NORM:
			mode = (vat_a >> 13) & 0xf;
			vx_table_selected->jump_table[jump_table_pos] = vx_cmds_col_n[mode].processor;
			vx_table_selected->param[jump_table_pos] = cmd_count;
#if USE_VERTEX_PTR
			cmd_count += vx_cmds_col_n[mode].exp_count * vx_cmds_col_n[mode].exp_size;
#else
			vx_set_requested_size(cmd_count, vx_cmds_col_n[mode].exp_count, vx_cmds_col_n[mode].exp_size);
			cmd_count += vx_cmds_col_n[mode].exp_count;
#endif
			jump_table_pos++;
			break;
		case VX_T_IDX8:
			mode = (vat_a >> 13) & 0xf;
			vx_table_selected->jump_table[jump_table_pos] = vx_cmds_col_i8[mode].processor;
			vx_table_selected->param[jump_table_pos] = cmd_count;
#if USE_VERTEX_PTR
			cmd_count += vx_cmds_col_i8[mode].exp_count * vx_cmds_col_i8[mode].exp_size;
#else
			vx_set_requested_size(cmd_count, vx_cmds_col_i8[mode].exp_count, vx_cmds_col_i8[mode].exp_size);
			cmd_count += vx_cmds_col_i8[mode].exp_count;
#endif
			jump_table_pos++;
			break;
		case VX_T_IDX16:
			mode = (vat_a >> 13) & 0xf;
			vx_table_selected->jump_table[jump_table_pos] = vx_cmds_col_i16[mode].processor;
			vx_table_selected->param[jump_table_pos] = cmd_count;
#if USE_VERTEX_PTR
			cmd_count += vx_cmds_col_i16[mode].exp_count * vx_cmds_col_i16[mode].exp_size;
#else
			vx_set_requested_size(cmd_count, vx_cmds_col_i16[mode].exp_count, vx_cmds_col_i16[mode].exp_size);
			cmd_count += vx_cmds_col_i16[mode].exp_count;
#endif
			jump_table_pos++;
			break;
		default:
			syslog_error(VX,"PT: Color0 (%d) - NOT SUPPORTED\n", (vcd_lo >> 13) & 0x03);
			break;
		}
	} // COL0
	else
	{
		// HACK: Material colour does not get applied correctly. sad. 
		vx_table_selected->dxx_size += 4;	// ARGB (8bit ch)
		vx_table_selected->fvf |= D3DFVF_DIFFUSE;
		vx_table_selected->jump_table[jump_table_pos] = vx_null_col;
		vx_table_selected->param[jump_table_pos] = cmd_count;
		jump_table_pos++;
	}


	if (vcd_lo & (0x3 << 15))
	{
		// COL1
		// Color1
		// for debugging purposes
		vx_col_debug_type =	(vcd_lo >> 15) & 0x03;
		vx_col_debug_fmt =	(vat_a >> 18) & 0x7;
		vx_col_debug_cnt =	(vat_a >> 17) & 0x1;

		mode = (vat_a >> 17) & 0xf;
		switch((vcd_lo >> 15) & 0x3)
		{
		case VX_T_NORM:
			vx_table_selected->jump_table[jump_table_pos] = vx_cmds_col_n[mode].processor;
			vx_table_selected->param[jump_table_pos] = cmd_count;
			vx_set_requested_size(cmd_count, vx_cmds_col_n[mode].exp_count, vx_cmds_col_n[mode].exp_size);
			cmd_count += vx_cmds_col_n[mode].exp_count;
			jump_table_pos++;
			break;
		case VX_T_IDX8:
			vx_table_selected->jump_table[jump_table_pos] = vx_cmds_col_i8[mode].processor;
			vx_table_selected->param[jump_table_pos] = cmd_count;
			vx_set_requested_size(cmd_count, vx_cmds_col_i8[mode].exp_count, vx_cmds_col_i8[mode].exp_size);
			cmd_count += vx_cmds_col_i8[mode].exp_count;
			jump_table_pos++;
			break;
		case VX_T_IDX16:
			vx_table_selected->jump_table[jump_table_pos] = vx_cmds_col_i16[mode].processor;
			vx_table_selected->param[jump_table_pos] = cmd_count;
			vx_set_requested_size(cmd_count, vx_cmds_col_i16[mode].exp_count, vx_cmds_col_i16[mode].exp_size);
			cmd_count += vx_cmds_col_i16[mode].exp_count;
			jump_table_pos++;
			break;
		default:
			syslog_error(VX,"PT: Color1 (%d) - NOT SUPPORTED\n", (vcd_lo >> 15) & 0x03);
			break;
		}
	} // COL1

	if (vcd_hi & 0xffff)
	{
		// TEXn
		// Texture Coordinate n
		uint16 temp;
		uint32 text_vat;
		int i;
		temp = vcd_hi & 0xffff;
		for(i = 0 ; i < 8 ; i++)
		{
			vx_tex_debug_type[i] = temp & 0x3;

			if (i != 0 && temp != VX_T_NONE)
			{
//				syslog_error(VX,"Unsupported multitexturing\n");
			}

			if (temp != VX_T_NONE)
			{
				vx_table_selected->dxx_size += 8;	// 2 texture coords
				vx_table_selected->fvf |= D3DFVF_TEX1 + i;// | D3DFVF_TEXCOORDSIZE2(0);
			}

			switch(temp & 0x3)
			{
			case VX_T_NONE:
				break;
			case VX_T_NORM:
				text_vat = vx_get_text_vat(vat, i);
				vx_tex_debug_fmt[i] = (text_vat >> 1) & 0x7;
				vx_tex_debug_cnt[i] = (text_vat >> 0) & 0x1;
				mode = text_vat & 0xf;
				vx_table_selected->jump_table[jump_table_pos] = vx_cmds_tex_n[mode].processor;
				vx_table_selected->param[jump_table_pos] = cmd_count | (i << 24) | ((text_vat << 4) & 0x1f00);
#if USE_VERTEX_PTR
				cmd_count += vx_cmds_tex_n[mode].exp_count * vx_cmds_tex_n[mode].exp_size;
#else
				vx_set_requested_size(cmd_count, vx_cmds_tex_n[mode].exp_count, vx_cmds_tex_n[mode].exp_size);
				cmd_count += vx_cmds_tex_n[mode].exp_count;
#endif
				jump_table_pos++;
				break;
			case VX_T_IDX8:
				text_vat = vx_get_text_vat(vat, i);
				vx_tex_debug_fmt[i] = (text_vat >> 1) & 0x7;
				vx_tex_debug_cnt[i] = (text_vat >> 0) & 0x1;
				mode = text_vat & 0xf;
				vx_table_selected->jump_table[jump_table_pos] = vx_cmds_tex_i8[mode].processor;
				vx_table_selected->param[jump_table_pos] = cmd_count | (i << 24) | ((text_vat << 4) & 0x1f00);
#if USE_VERTEX_PTR
				cmd_count += vx_cmds_tex_i8[mode].exp_count * vx_cmds_tex_i8[mode].exp_size;
#else
				vx_set_requested_size(cmd_count, vx_cmds_tex_i8[mode].exp_count, vx_cmds_tex_i8[mode].exp_size);
				cmd_count += vx_cmds_tex_i8[mode].exp_count;
#endif
				jump_table_pos++;
				break;
			case VX_T_IDX16:
				text_vat = vx_get_text_vat(vat, i);
				vx_tex_debug_fmt[i] = (text_vat >> 1) & 0x7;
				vx_tex_debug_cnt[i] = (text_vat >> 0) & 0x1;
				mode = text_vat & 0xf;
				vx_table_selected->jump_table[jump_table_pos] = vx_cmds_tex_i16[mode].processor;
				vx_table_selected->param[jump_table_pos] = cmd_count | (i << 24) | ((text_vat << 4) & 0x1f00);
#if USE_VERTEX_PTR
				cmd_count += vx_cmds_tex_i16[mode].exp_count * vx_cmds_tex_i16[mode].exp_size;
#else
				vx_set_requested_size(cmd_count, vx_cmds_tex_i16[mode].exp_count, vx_cmds_tex_i16[mode].exp_size);
				cmd_count += vx_cmds_tex_i16[mode].exp_count;
#endif
				jump_table_pos++;
				break;
			default:
				syslog_error(VX,"PT: TEX%d Texture Coordinate (%d) - NOT SUPPORTED\n", i, temp & 0x3);
				break;
			}
			temp >>= 2;
		}
	} // TEXn
	
	vx_table[vat].jump_table[jump_table_pos] = NULL;
	vx_table[vat].cmd_count = cmd_count;
	return cmd_count;
}

//--------------------------------------------------------------------------
//--------------------------------------------------------------------------

// processing commands main loop

void vx_process_commands(void)
{
	uint32 pos;
	pos = 0;
	//syslog(VX,"Processor START\n");

	while(vx_table_selected->jump_table[pos])
	{
		vx_table_selected->jump_table[pos](vx_table_selected->param[pos]);
		pos++;
	};
//	if(vx_table_selected->jump_table[0])
//		vx_table_selected->jump_table[0](vx_table_selected->param[0]);
	if (vx_quad_mode)
	{
		vx_quad_counter++;
		vx_quad_counter &= 3;
		if (vx_quad_counter == 3)
		{
			uint8 *src;
			uint8 *dst;

			src = (uint8 *)test_buf;
			dst = (uint8 *)test_buf;

			memcpy(dst, src - (vx_table_selected->dxx_size * 3), vx_table_selected->dxx_size);
			dst += vx_table_selected->dxx_size;
			memcpy(dst, src - (vx_table_selected->dxx_size * 1), vx_table_selected->dxx_size);
			dst += vx_table_selected->dxx_size;
			test_buf = (uint32 *)dst;
		}
	}
	//syslog(VX,"Processor END\n");
}








//--------------------------------------------------------------------------
//--------------------------------------------------------------------------
// POS Functions
// param:
// -------- -------s ssssssss pppppppp
// p - pos in vertex table
// s - shift (fractional bits)



void vx_pos_unknown(uint32)
{
	syslog_error(VX,"POS *NOT IMPLEMENTED* Type %s Cnt %s Format %s\n", vx_str_type[vx_pos_debug_type], vx_str_pos_cnt[vx_pos_debug_cnt], vx_str_fmt[vx_pos_debug_fmt]);
}
void vx_pos_n_xy_u8(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 s = (param >> 8) & 0x1f;

	syslog(VX,"pos_n_xyz_u8(%d)\n", pos);

	float divider, x, y;
	divider = (1 << s);

#if USE_VERTEX_PTR
	uint8 *tptr;
	tptr = (uint8 *)(vx_vertex_ptr.ubp + pos);
	x = tptr[0];
	y = tptr[1];
#else
	x = vx_vertex_data[pos + 0].ub;
	y = vx_vertex_data[pos + 1].ub;
#endif

	x /= divider;
	y /= divider;

	syslog(VX,"// glVertex2f(%f, %f)\n", x, y);
	vx_add_vertex(x, y, 0);
}

void vx_pos_n_xyz_u8(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 s = (param >> 8) & 0x1f;

	syslog(VX,"pos_n_xyz_u8(%d)\n", pos);

	float divider, x, y, z;
	divider = (1 << s);

#if USE_VERTEX_PTR
	uint8 *tptr;
	tptr = (uint8 *)(vx_vertex_ptr.ubp + pos);
	x = tptr[0];
	y = tptr[1];
	z = tptr[2];
#else
	x = vx_vertex_data[pos + 0].ub;
	y = vx_vertex_data[pos + 1].ub;
	z = vx_vertex_data[pos + 2].ub;
#endif

	x /= divider;
	y /= divider;
	z /= divider;

	syslog(VX,"// glVertex3f(%f, %f, %f)\n", x, y, z);
	vx_add_vertex(x, y, z);
}

void vx_pos_n_xy_s8(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 s = (param >> 8) & 0x1f;

	syslog(VX,"pos_n_xy_s8(%d) // glVertex2f()\n", pos);

	float divider, x, y;
	divider = (1 << s);

	sint8 *tptr;
	tptr = (sint8 *)(vx_vertex_ptr.ubp + pos);

	x = tptr[0];
	y = tptr[1];

	x /= divider;
	y /= divider;

	syslog(VX,"// glVertex2f(%f, %f)\n", x, y);
	vx_add_vertex(x, y, 0);
}

void vx_pos_n_xyz_s8(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 s = (param >> 8) & 0x1f;

	syslog(VX,"pos_n_xyz_s8(%d) // glVertex3f()\n", pos);

	float divider, x, y, z;
	divider = (1 << s);

	sint8 *tptr;
	tptr = (sint8 *)(vx_vertex_ptr.ubp + pos);

	x = tptr[0];
	y = tptr[1];
	z = tptr[2];

	x /= divider;
	y /= divider;
	z /= divider;

	syslog(VX,"// glVertex3f(%f, %f, %f)\n", x, y, z);
	vx_add_vertex(x, y, z);
}

void vx_pos_n_xy_u16(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 s = (param >> 8) & 0x1f;

	syslog_error(VX,"pos_n_xy_u16(%d)\n", pos);

	float divider, x, y;
	divider = (1 << s);

#if USE_VERTEX_PTR
	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	x = (uint16)vx_swap16(tptr[0]);
	y = (uint16)vx_swap16(tptr[1]);
#else
	x = vx_vertex_data[pos + 0].us;
	y = vx_vertex_data[pos + 1].us;
#endif

	x /= divider;
	y /= divider;

	syslog(VX,"// glVertex2f(%f, %f)\n", x, y);
	// glVertex2f(x, y);
}

void vx_pos_n_xyz_u16(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 s = (param >> 8) & 0x1f;

	syslog_error(VX,"pos_n_xyz_u16(%d)\n", pos);

	float divider, x, y, z;
	divider = (1 << s);

#if USE_VERTEX_PTR
	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	x = (uint16)vx_swap16(tptr[0]);
	y = (uint16)vx_swap16(tptr[1]);
	z = (uint16)vx_swap16(tptr[2]);
#else
	x = vx_vertex_data[pos + 0].ss;
	y = vx_vertex_data[pos + 1].ss;
	z = vx_vertex_data[pos + 2].ss;
#endif

	x /= divider;
	y /= divider;
	z /= divider;

	syslog(VX,"// glVertex3f(%f, %f, %f)\n", x, y, z);
	// glVertex3f(x, y, z);
}

void vx_pos_n_xy_s16(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 s = (param >> 8) & 0x1f;

	syslog(VX,"pos_n_xy_s16(%d)\n", pos);

	float divider, x, y;
	divider = (1 << s);

#if USE_VERTEX_PTR
	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	x = (sint16)vx_swap16(tptr[0]);
	y = (sint16)vx_swap16(tptr[1]);
#else
	x = vx_vertex_data[pos + 0].ss;
	y = vx_vertex_data[pos + 1].ss;
#endif

	x /= divider;
	y /= divider;

	syslog(VX,"// glVertex2f(%f, %f)\n", x, y);
	vx_add_vertex(x, y, 0);
}

void vx_pos_n_xyz_s16(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 s = (param >> 8) & 0x1f;

	syslog(VX,"pos_n_xyz_s16(%d)\n", pos);

	if (s)
	{
		syslog(VX,"Fractional bits: %d\n", s);
	}

	float divider, x, y, z;

	divider = (1 << s);

#if USE_VERTEX_PTR
	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	x = (sint16)vx_swap16(tptr[0]);
	y = (sint16)vx_swap16(tptr[1]);
	z = (sint16)vx_swap16(tptr[2]);
#else
	x = vx_vertex_data[pos + 0].ss;
	y = vx_vertex_data[pos + 1].ss;
	z = vx_vertex_data[pos + 2].ss;
#endif

	x /= divider;
	y /= divider;
	z /= divider;

	syslog(VX,"Vertex3f(%f, %f, %f)\n", x, y, z);
	vx_add_vertex(x, y, z);
}

void vx_pos_n_xy_f32(uint32 param)
{
	uint32 pos = param & 0xff;

	syslog_error(VX,"pos_n_xy_f32(%d)\n", pos);

	uint32 s = (param >> 8) & 0x1f;
	if (s)
	{
		syslog_error(VX,"Fractional bits in float: %d\n", s);
	}

	vx_vertex_t x, y;

	uint32 *tptr;
	tptr = (uint32 *)(vx_vertex_ptr.ubp + pos);

	x.i = (uint32)vx_swap32(tptr[0]);
	y.i = (uint32)vx_swap32(tptr[1]);

	syslog(VX,"// glVertex2f(%f, %f)\n", x.f, y.f);
	// glVertex2f(x.f, y.f);
}

void vx_pos_n_xyz_f32(uint32 param)
{
	uint32 pos = param & 0xff;

	syslog(VX,"pos_n_xyz_f32(%d)\n", pos);

	uint32 s = (param >> 8) & 0x1f;
	if (s)
	{
		syslog_error(VX,"Fractional bits in float: %d\n", s);
	}

	vx_vertex_t x, y, z;

	uint32 *tptr;
	tptr = (uint32 *)(vx_vertex_ptr.ubp + pos);

	x.i = (uint32)vx_swap32(tptr[0]);
	y.i = (uint32)vx_swap32(tptr[1]);
	z.i = (uint32)vx_swap32(tptr[2]);

	syslog(VX,"Vertex(%f, %f, %f)\n", x.f, y.f, z.f);
	vx_add_vertex(x.f, y.f ,z.f);
}

void vx_pos_i8_xy_s8(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;

	syslog_error(VX,"pos_i8_xy_s8(%d)\n", pos);

	uint32 s = (param >> 8) & 0x1f;
	if (s)
	{
		syslog(VX,"Fractional bits: %d\n", s);
	}

	index_mem = gp_cp_regs[0xa0];
	index_size = gp_cp_regs[0xb0];

#if USE_VERTEX_PTR
	index_pos = vx_vertex_ptr.ubp[pos];
#else
	index_pos = vx_vertex_data[pos].ub;
#endif

	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	float divider, x, y;
	divider = (1 << s);

	x = index_ptr.sbp[0];
	y = index_ptr.sbp[1];

	x /= divider;
	y /= divider;

	syslog(VX,"// glVertex2f(%f, %f, %f)\n", x, y);
	// glVertex2f(x, y);
}

void vx_pos_i8_xyz_s8(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;

	syslog(VX,"pos_i8_xyz_s8(%d)\n", pos);

	uint32 s = (param >> 8) & 0x1f;
	if (s)
	{
		syslog(VX,"Fractional bits: %d\n", s);
	}

	index_mem = gp_cp_regs[0xa0];
	index_size = gp_cp_regs[0xb0];

#if USE_VERTEX_PTR
	index_pos = vx_vertex_ptr.ubp[pos];
#else
	index_pos = vx_vertex_data[pos].ub;
#endif

	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	float divider, x, y, z;
	divider = (1 << s);

	x = index_ptr.sbp[0];
	y = index_ptr.sbp[1];
	z = index_ptr.sbp[2];

	x /= divider;
	y /= divider;
	z /= divider;

	syslog(VX,"// glVertex3f(%f, %f, %f)\n", x, y, z);
	vx_add_vertex(x, y ,z);
}

void vx_pos_i8_xyz_u8(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;

	syslog_error(VX,"pos_i8_xyz_s8(%d)\n", pos);

	uint32 s = (param >> 8) & 0x1f;
	if (s)
	{
		syslog(VX,"Fractional bits: %d\n", s);
	}

	index_mem = gp_cp_regs[0xa0];
	index_size = gp_cp_regs[0xb0];

#if USE_VERTEX_PTR
	index_pos = vx_vertex_ptr.ubp[pos];
#else
	index_pos = vx_vertex_data[pos].ub;
#endif

	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	float divider, x, y, z;
	divider = (1 << s);

	x = index_ptr.ubp[0];
	y = index_ptr.ubp[1];
	z = index_ptr.ubp[2];

	x /= divider;
	y /= divider;
	z /= divider;

	syslog(VX,"// glVertex3f(%f, %f, %f)\n", x, y, z);
	// glVertex3f(x, y, z);
}


void vx_pos_i8_xy_s16(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;

	syslog(VX,"pos_i8_xy_s16(%d)\n", pos);

	uint32 s = (param >> 8) & 0x1f;
	if (s)
	{
		syslog(VX,"Fractional bits: %d\n", s);
	}

	index_mem = gp_cp_regs[0xa0];
	index_size = gp_cp_regs[0xb0];

	index_pos = vx_vertex_ptr.ubp[pos];
	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	float x, y;
	float divider = (1 << s);

	x = (sint16)vx_swap16(index_ptr.usp[0]);
	y = (sint16)vx_swap16(index_ptr.usp[1]);

	x /= divider;
	y /= divider;

	syslog(VX,"// glVertex2f(%f, %f)\n", x, y);
	vx_add_vertex(x, y, 0);
}

void vx_pos_i8_xyz_s16(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;

	syslog(VX,"pos_i8_xyz_s16(%d)\n", pos);

	uint32 s = (param >> 8) & 0x1f;
	if (s)
	{
		syslog(VX,"Fractional bits: %d\n", s);
	}

	index_mem = gp_cp_regs[0xa0];
	index_size = gp_cp_regs[0xb0];

#if USE_VERTEX_PTR
	index_pos = vx_vertex_ptr.ubp[pos];
#else
	index_pos = vx_vertex_data[pos].ub;
#endif
	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	float x, y, z;
	float divider = (1 << s);

	x = (sint16)vx_swap16(index_ptr.usp[0]);
	y = (sint16)vx_swap16(index_ptr.usp[1]);
	z = (sint16)vx_swap16(index_ptr.usp[2]);

	x /= divider;
	y /= divider;
	z /= divider;

	syslog(VX,"Vertex3f(%f, %f, %f)\n", x, y, z);
	vx_add_vertex(x, y, z);
	//vx_add_vertex(0,0,0);
}

void vx_pos_i8_xyz_f32(uint32 param)
{
	uint32	pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;

	uint32	i;
	uint32 s = (param >> 8) & 0x1f;
	
	syslog(VX,"pos_i8_xyz_f32(%d)\n", pos);
	if (s)
	{
		syslog_error(VX,"Fractional bits in float: %d\n", s);
	}

	index_mem = gp_cp_regs[0xa0];
	index_size = gp_cp_regs[0xb0];

#if USE_VERTEX_PTR
	index_pos = vx_vertex_ptr.ubp[pos];
#else
	index_pos = vx_vertex_data[pos].ub;
#endif

	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	vx_vertex_t	x, y, z;

	x.i = vx_swap32(index_ptr.ulp[0]);
	y.i = vx_swap32(index_ptr.ulp[1]);
	z.i = vx_swap32(index_ptr.ulp[2]);

	syslog(VX,"Vertex3f(%f, %f, %f)\n", x.f, y.f, z.f);
	vx_add_vertex(x.f, y.f, z.f);
}

void vx_pos_i16_xyz_s8(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;

	syslog_error(VX,"pos_i8_xyz_s8(%d)\n", pos);

	uint32 s = (param >> 8) & 0x1f;
	if (s)
	{
		syslog(VX,"Fractional bits: %d\n", s);
	}

	index_mem = gp_cp_regs[0xa0];
	index_size = gp_cp_regs[0xb0];

#if USE_VERTEX_PTR
	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	index_pos = (uint16)vx_swap16(tptr[0]);
#else
	index_pos = vx_vertex_data[pos].us;
#endif
	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	float divider, x, y, z;
	divider = (1 << s);

	x = index_ptr.sbp[0];
	y = index_ptr.sbp[1];
	z = index_ptr.sbp[2];

	x /= divider;
	y /= divider;
	z /= divider;

	syslog(VX,"// glVertex3f(%f, %f, %f)\n", x, y, z);
	// glVertex3f(x, y, z);
}


void vx_pos_i16_xyz_s16(uint32 param)
{
	uint32	pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;
	uint32	i;

	syslog_error(VX,"pos_i16_xyz_s16(%d)\n", pos);

	uint32 s = (param >> 8) & 0x1f;
	if (s)
	{
		syslog(VX,"Fractional bits: %d\n", s);
	}

	index_mem = gp_cp_regs[0xa0];
	index_size = gp_cp_regs[0xb0];
#if USE_VERTEX_PTR
	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	index_pos = (uint16)vx_swap16(tptr[0]);
#else
	index_pos = vx_vertex_data[pos].us;
#endif
	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	float divider, x, y, z;

	divider = (1 << s);

	x = (sint16)vx_swap16(index_ptr.usp[0]);
	y = (sint16)vx_swap16(index_ptr.usp[1]);
	z = (sint16)vx_swap16(index_ptr.usp[2]);

	x /= divider;
	y /= divider;
	z /= divider;

	syslog(VX,"// glVertex3f(%f, %f, %f)\n", x, y, z);
	// glVertex3f(x, y, z);

#if WITH_VERTEX_CACHE
	// for debugging purposes
	if ((index_mem & 0x1) || (index_size & 0x1))
	{
		syslog_error(VX,"pos_i16_xyz_s16(%d %x) Unaligned memory access\n", index_size, index_mem);
	}
	syslog(VX,"size %d\n", index_size);
	if(vx_pos_valid[index_pos] != index_size)
	{
		for(i = 0 ; i < index_size ; i++)
		{
			vx_pos_cache[index_pos * index_size + i] = gx_memory[index_mem + index_pos * index_size + (i ^ 1)];
		}
		vx_pos_valid[index_pos] = index_size;
		if (index_pos > vx_pos_watermark)
			vx_pos_watermark = index_pos;
	}
	syslog(VX,"// glVertex3sv(%04x, %04x, %04x)\n", ((sint16 *)(&vx_pos_cache[index_pos * index_size]))[0], ((sint16 *)(&vx_pos_cache[index_pos * index_size]))[1], ((sint16 *)(&vx_pos_cache[index_pos * index_size]))[2]);
	// glVertex3sv((GLshort *)(&vx_pos_cache[index_pos * index_size]));
#endif
}

void vx_pos_i16_xyz_f32(uint32 param)
{
	uint32	pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;
	uint32	i;
	uint32 s = (param >> 8) & 0x1f;
	if (s)
	{
		syslog(VX,"Fractional bits: %d\n", s);
	}

	syslog(VX,"pos_i16_xyz_f32(%d)\n", pos);
//	syslog(VX,"// glVertex3fv)

	index_mem = gp_cp_regs[0xa0];
	index_size = gp_cp_regs[0xb0];

	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	index_pos = (uint16)vx_swap16(tptr[0]);

	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	vx_vertex_t	x, y, z;

	x.i = vx_swap32(index_ptr.ulp[0]);
	y.i = vx_swap32(index_ptr.ulp[1]);
	z.i = vx_swap32(index_ptr.ulp[2]);

	syslog(VX,"// glVertex3f(%f, %f, %f)\n", x.f, y.f, z.f);
	vx_add_vertex(x.f, y.f, z.f);

}



//--------------------------------------------------------------------------
// NOR Functions
// param:
// -------- -------- -------- pppppppp
// p - pos in vertex table


void vx_nor_unknown(uint32 param)
{
	uint32 pos = param & 0xff;
	syslog_error(VX,"NOR *NOT IMPLEMENTED* Pos: %d Type %s Cnt %s Format %s\n", pos, vx_str_type[vx_nor_debug_type], vx_str_nor_cnt[vx_nor_debug_cnt], vx_str_fmt[vx_nor_debug_fmt]);
}

void vx_nor_n_nor_s8(uint32 param)
{
	uint32 pos = param & 0xff;
	syslog_error(VX,"nor_n_nor_s8(%d)\n", pos);

	float x, y, z;

#if USE_VERTEX_PTR
	x = vx_vertex_ptr.sbp[pos + 0];
	y = vx_vertex_ptr.sbp[pos + 1];
	z = vx_vertex_ptr.sbp[pos + 2];
#else
	x = vx_vertex_data[pos + 0].sb;
	y = vx_vertex_data[pos + 1].sb;
	z = vx_vertex_data[pos + 2].sb;
#endif

	syslog_error(VX,"// glVormal3f(%f, %f, %f)\n", x, y, z);
	// glVormal3f(x, y, z);
}

void vx_nor_n_nor_s16(uint32 param)
{
	uint32 pos = param & 0xff;
	syslog_error(VX,"nor_n_nor_s8(%d)\n", pos);

	float x, y, z;

#if USE_VERTEX_PTR
	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	x = (sint16)vx_swap16(tptr[0]);
	y = (sint16)vx_swap16(tptr[1]);
	z = (sint16)vx_swap16(tptr[2]);
#else
	x = vx_vertex_data[pos + 0].sb;
	y = vx_vertex_data[pos + 1].sb;
	z = vx_vertex_data[pos + 2].sb;
#endif

	syslog_error(VX,"// glVormal3f(%f, %f, %f)\n", x, y, z);
	// glVormal3f(x, y, z);
}


void vx_nor_n_nor_f32(uint32 param)
{
	uint32 pos = param & 0xff;
	syslog(VX,"nor_n_nor_f32(%d)\n", pos);

	vx_vertex_t x, y, z;

#if USE_VERTEX_PTR
	uint32 *tptr;
	tptr = (uint32 *)(vx_vertex_ptr.ubp + pos);

	x.i = (uint32)vx_swap32(tptr[0]);
	y.i = (uint32)vx_swap32(tptr[1]);
	z.i = (uint32)vx_swap32(tptr[2]);
#else
	x = vx_vertex_data[pos + 0].f;
	y = vx_vertex_data[pos + 1].f;
	z = vx_vertex_data[pos + 2].f;
#endif

	syslog(VX,"Normal3f(%f, %f, %f)\n", x.f, y.f, z.f);
	vx_add_vertex(x.f, y.f, z.f);
}

void vx_nor_i8_nor_s8(uint32 param)
{
	uint32	pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;

	syslog_error(VX,"nor_i8_nor_s8(%d)\n", pos);

/*	uint32 s = (param >> 8) & 0x1f;
	if (s)
	{
		syslog_error(VX,"Fractional bits: %d\n", s);
	}
*/

	index_mem = gp_cp_regs[0xa1];
	index_size = gp_cp_regs[0xb1];

#if USE_VERTEX_PTR
	index_pos = vx_vertex_ptr.ubp[pos];
#else
	index_pos = vx_vertex_data[pos].ub;
#endif
	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	float x, y, z;

	x = index_ptr.sbp[0];
	y = index_ptr.sbp[1];
	z = index_ptr.sbp[2];

	syslog_error(VX,"// glVormal3f(%f, %f, %f)\n", x, y, z);
	// glVormal3f(x, y, z);
}

void vx_nor_i8_nor_s16(uint32 param)
{
	uint32	pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;

	syslog_error(VX,"nor_i8_nor_s16(%d)\n", pos);

/*	uint32 s = (param >> 8) & 0x1f;
	if (s)
	{
		syslog_error(VX,"Fractional bits: %d\n", s);
	}
*/

	index_mem = gp_cp_regs[0xa1];
	index_size = gp_cp_regs[0xb1];

#if USE_VERTEX_PTR
	index_pos = vx_vertex_ptr.ubp[pos];
#else
	index_pos = vx_vertex_data[pos].ub;
#endif
	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	float x, y, z;

	x = (sint16)vx_swap16(index_ptr.usp[0]);
	y = (sint16)vx_swap16(index_ptr.usp[1]);
	z = (sint16)vx_swap16(index_ptr.usp[2]);

	syslog_error(VX,"// glVormal3f(%f, %f, %f)\n", x, y, z);
	// glVormal3f(x, y, z);
}
void vx_nor_i8_nor_f32(uint32 param)
{
	uint32	pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;
	uint32	i;

/*	uint32 s = (param >> 8) & 0x1f;
	if (s)
	{
		syslog_error(VX,"Fractional bits: %d\n", s);
	}
*/
	syslog(VX,"nor_i8_nor_f32(%d)\n", pos);

	index_mem = gp_cp_regs[0xa1];
	index_size = gp_cp_regs[0xb1];

#if USE_VERTEX_PTR
	index_pos = vx_vertex_ptr.ubp[pos];
#else
	index_pos = vx_vertex_data[pos].ub;
#endif
	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	vx_vertex_t	x, y, z;

	x.i = vx_swap32(index_ptr.ulp[0]);
	y.i = vx_swap32(index_ptr.ulp[1]);
	z.i = vx_swap32(index_ptr.ulp[2]);

	syslog(VX,"Normal3f(%f, %f, %f)\n", x.f, y.f, z.f);
	vx_add_vertex(x.f, y.f, z.f);
}

void vx_nor_i16_nor_s8(uint32 param)
{
	uint32	pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;

	syslog_error(VX,"nor_i8_nor_s8(%d)\n", pos);

/*	uint32 s = (param >> 8) & 0x1f;
	if (s)
	{
		syslog_error(VX,"Fractional bits: %d\n", s);
	}
*/

	index_mem = gp_cp_regs[0xa1];
	index_size = gp_cp_regs[0xb1];

#if USE_VERTEX_PTR
	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	index_pos = (uint16)vx_swap16(tptr[0]);
#else
	index_pos = vx_vertex_data[pos].ub;
#endif
	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	float x, y, z;

	x = index_ptr.sbp[0];
	y = index_ptr.sbp[1];
	z = index_ptr.sbp[2];

	syslog_error(VX,"// glVormal3f(%f, %f, %f)\n", x, y, z);
	// glVormal3f(x, y, z);
}

void vx_nor_i16_nor_s16(uint32 param)
{
	uint32	pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;
	uint32	i;

/*	uint32 s = (param >> 8) & 0x1f;
	if (s)
	{
		syslog_error(VX,"Fractional bits: %d\n", s);
	}
*/
	syslog_error(VX,"nor_i16_nor_s16(%d)\n", pos);

	index_mem = gp_cp_regs[0xa1];
	index_size = gp_cp_regs[0xb1];
#if USE_VERTEX_PTR
	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	index_pos = (uint16)vx_swap16(tptr[0]);
#else
	index_pos = vx_vertex_data[pos].us;
#endif

	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	float x, y, z;

	x = (sint16)vx_swap16(index_ptr.usp[0]);
	y = (sint16)vx_swap16(index_ptr.usp[1]);
	z = (sint16)vx_swap16(index_ptr.usp[2]);

	syslog_error(VX,"// glVormal3f(%f, %f, %f)\n", x, y, z);
	// glVormal3f(x, y, z);

#if WITH_VERTEX_CACHE
	// for debugging purposes
	if ((index_mem & 0x3) || (index_size & 0x3))
	{
		syslog_error(VX,"nor_i8_nor_f32() Unaligned memory access\n");
	}
	if(!vx_nor_valid[index_pos])
	{
		for(i = 0 ; i < index_size ; i++)
		{
			vx_nor_cache[index_pos * index_size + i] = gx_memory[index_mem + ((index_pos * index_size + i) ^ 3)];
		}
		vx_nor_valid[index_pos] = 1;
		if (index_pos > vx_nor_watermark)
			vx_nor_watermark = index_pos;
	}
	syslog_error(VX,"// glVormal3fv(%f, %f, %f)\n", ((GLfloat *)(&vx_nor_cache[index_pos * index_size]))[0], ((GLfloat *)(&vx_nor_cache[index_pos * index_size]))[1], ((GLfloat *)(&vx_nor_cache[index_pos * index_size]))[2]);
	// glVormal3fv((GLfloat *)(&vx_nor_cache[index_pos * index_size]));
#endif
}

void vx_nor_i16_nor_f32(uint32 param)
{
	uint32	pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;
	uint32	i;

	syslog(VX,"nor_i16_nor_f32(%d)\n", pos);

/*	uint32 s = (param >> 8) & 0x1f;
	if (s)
	{
		syslog_error(VX,"Fractional bits: %d\n", s);
	}
*/

	index_mem = gp_cp_regs[0xa1];
	index_size = gp_cp_regs[0xb1];

	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	index_pos = (uint16)vx_swap16(tptr[0]);

	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	vx_vertex_t	x, y, z;

	x.i = vx_swap32(index_ptr.ulp[0]);
	y.i = vx_swap32(index_ptr.ulp[1]);
	z.i = vx_swap32(index_ptr.ulp[2]);

	syslog(VX,"// glVormal3f(%f, %f, %f)\n", x.f, y.f, z.f);
	vx_add_vertex(x.f, y.f, z.f);
}


/*
void vx_nor_i8_xyz_u8(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 index_pos;
	uint32 index_mem;
	uint32 index_size;

	index_mem = gp_cp_regs[0xa1];
	index_size = gp_cp_regs[0xb1];
	index_pos = vx_vertex_data[pos].ub;

	syslog_error(VX,"nor_i8_xyz_u8(%d) * CACHE NOT IMPLEMENTED *\n", pos);
	//// glVertex3f(((char)gx_memory[index_mem + index_pos * index_size])/1.0f, ((char)gx_memory[index_mem + index_pos * index_size + 1])/1.0f, ((char)gx_memory[index_mem + index_pos * index_size + 2])/1.0f);
	// glVormal3f(((uint8 *)&gx_memory[index_mem + index_pos * index_size])[0], ((uint8 *)&gx_memory[index_mem + index_pos * index_size])[1], ((uint8 *)&gx_memory[index_mem + index_pos * index_size])[2]);
}
*/


//--------------------------------------------------------------------------
// COLn Functions

// param:
// -------n -------- -------- pppppppp
// n - COLn
// p - pos in vertex table


void vx_col_unknown(uint32 param)
{
	uint32 pos = param & 0xff;
	syslog_error(VX,"COL *NOT IMPLEMENTED* Pos: %d Type %s Cnt %s Format %s\n", pos, vx_str_type[vx_col_debug_type], vx_str_col_cnt[vx_col_debug_cnt], vx_str_col_fmt[vx_col_debug_fmt]);
}

void vx_col_n_rgb_rgb8(uint32 param)
{
	uint32 pos = param & 0xff;
	syslog(VX,"col_n_rgb_rgb8(%d)\n", pos);
#if USE_VERTEX_PTR
//	// glColor3ub(vx_vertex_ptr.ubp[pos + 0], vx_vertex_ptr.ubp[pos + 1], vx_vertex_ptr.ubp[pos + 2]);
	// glColor3ubv(&vx_vertex_ptr.ubp[pos]);
	uint32 val = *((uint32 *)&vx_vertex_ptr.ubp[pos]);
	val |= 0xff000000;
	vx_add_colour(val);
#else
	// glColor3ub(vx_vertex_data[pos].ub, vx_vertex_data[pos + 1].ub, vx_vertex_data[pos + 2].ub);
#endif
}

void vx_col_n_rgba_rgba4(uint32 param)
{
	uint32 pos = param & 0xff;
	syslog_error(VX,"col_n_rgba_rgba4(%d)\n", pos);
	uint16 data;
#if USE_VERTEX_PTR
	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);
	data = tptr[0];
#else
	data = vx_vertex_data[pos].us;
#endif
	// glColor4ub((data >> 8) & 0xf0, (data >> 4) & 0xf0, (data >> 0) & 0xf0, (data << 4) & 0xf0);
}

void vx_col_n_rgba_rgba8(uint32 param)
{
	uint32 pos = param & 0xff;
	syslog(VX,"col_n_rgba_rgba8(%d) // glColor4ub()\n", pos);
	//syslog_error(VX,"// glColor4ub(%d, %d, %d, %d)", vx_vertex_ptr.ubp[pos + 0], vx_vertex_ptr.ubp[pos + 1], vx_vertex_ptr.ubp[pos + 2], vx_vertex_ptr.ubp[pos + 3]);
	uint32 val = *((uint32 *)(vx_vertex_ptr.ubp + pos));
	vx_add_colour(val);

	//// glColor4ub(vx_vertex_ptr.ubp[pos + 0], vx_vertex_ptr.ubp[pos + 1], vx_vertex_ptr.ubp[pos + 2], vx_vertex_ptr.ubp[pos + 3]);
	// glColor4ubv(vx_vertex_ptr.ubp + pos);
}

void vx_col_i16_rgba_rgba8(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 index_pos;
	uint32 index_mem;
	uint32 index_size;

	syslog(VX,"col_i16_rgba_rgba8(%d)\n", pos);

	index_mem = gp_cp_regs[0xa2];
	index_size = gp_cp_regs[0xb2];

	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	index_pos = (uint16)vx_swap16(tptr[0]);

	uint32 val = *((uint32 *)(&gx_memory[index_mem + index_pos * index_size]));
	vx_add_colour(val);
}

void vx_col_i8_rgb_rgb8(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 index_pos;
	uint32 index_mem;
	uint32 index_size;

	syslog_error(VX,"col_i8_rgba_rgb8(%d)\n", pos);

	index_mem = gp_cp_regs[0xa2];
	index_size = gp_cp_regs[0xb2];

#if USE_VERTEX_PTR
	index_pos = vx_vertex_ptr.ubp[pos];
#else
	index_pos = vx_vertex_data[pos].ub;
#endif

	// glColor3ubv(&gx_memory[index_mem + index_pos * index_size]);
}



void vx_col_i8_rgba_rgba8(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 index_pos;
	uint32 index_mem;
	uint32 index_size;

	syslog(VX,"col_i8_rgba_rgba8(%d)\n", pos);

	index_mem = gp_cp_regs[0xa2];
	index_size = gp_cp_regs[0xb2];

#if USE_VERTEX_PTR
	index_pos = vx_vertex_ptr.ubp[pos];
#else
	index_pos = vx_vertex_data[pos].ub;
#endif
	uint32 val = *((uint32 *)(&gx_memory[index_mem + index_pos * index_size]));
	vx_add_colour(val);
}





void vx_col_i8_rgb_rgb565(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;

	syslog_error(VX,"col_i8_rgb_rgb565(%d)\n", pos);

	index_mem = gp_cp_regs[0xa2];
	index_size = gp_cp_regs[0xb2];
#if USE_VERTEX_PTR
	uint8 *tptr;
	tptr = (uint8 *)(vx_vertex_ptr.ubp + pos);
	index_pos = tptr[0];
#else
	index_pos = vx_vertex_data[pos].us;
#endif
	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);
	uint16 data;
	data = index_ptr.usp[0];
	data = vx_swap16(data);
	// glColor3ub((data >> 8) & 0xf8, (data >> 3) & 0xfc, (data << 3) & 0xf8);
}

void vx_col_i8_rgba_rgba4(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;

	syslog_error(VX,"col_i8_rgba_rgba4(%d)\n", pos);

	index_mem = gp_cp_regs[0xa2];
	index_size = gp_cp_regs[0xb2];
#if USE_VERTEX_PTR
	uint8 *tptr;
	tptr = (uint8 *)(vx_vertex_ptr.ubp + pos);
	index_pos = tptr[0];
#else
	index_pos = vx_vertex_data[pos].us;
#endif
	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);
	uint16 data;
	data = index_ptr.usp[0];
	data = vx_swap16(data);
	// glColor4ub((data >> 8) & 0xf0, (data >> 4) & 0xf0, (data >> 0) & 0xf0, (data << 4) & 0xf0);
}

void vx_col_i16_rgb_rgb565(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;

	syslog_error(VX,"col_i16_rgb_rgb565(%d)\n", pos);

	index_mem = gp_cp_regs[0xa2];
	index_size = gp_cp_regs[0xb2];
#if USE_VERTEX_PTR
	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	index_pos = (uint16)vx_swap16(tptr[0]);
#else
	index_pos = vx_vertex_data[pos].us;
#endif
	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);
	uint16 data;
	data = index_ptr.usp[0];
	data = vx_swap16(data);
	// glColor3ub((data >> 8) & 0xf8, (data >> 3) & 0xfc, (data << 3) & 0xf8);
}

void vx_col_i16_rgba_rgba4(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;

	syslog_error(VX,"col_i16_rgba_rgba4(%d)\n", pos);

	index_mem = gp_cp_regs[0xa2];
	index_size = gp_cp_regs[0xb2];
#if USE_VERTEX_PTR
	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	index_pos = (uint16)vx_swap16(tptr[0]);
#else
	index_pos = vx_vertex_data[pos].us;
#endif
	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);
	uint16 data;
	data = index_ptr.usp[0];
	data = vx_swap16(data);
	// glColor4ub((data >> 8) & 0xf0, (data >> 4) & 0xf0, (data >> 0) & 0xf0, (data << 4) & 0xf0);
}

void vx_col_i16_rgb_rgb8(uint32 param)
{
	uint32 pos = param & 0xff;
	uint32 index_pos;
	uint32 index_mem;
	uint32 index_size;

	syslog(VX,"col_i16_rgb_rgb8(%d) // glColor3ub()\n", pos);

	index_mem = gp_cp_regs[0xa2];
	index_size = gp_cp_regs[0xb2];
#if USE_VERTEX_PTR
	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	index_pos = (uint16)vx_swap16(tptr[0]);
#else
	index_pos = vx_vertex_data[pos].us;
#endif
	uint32 val = *((uint32 *)(&gx_memory[index_mem + index_pos * index_size]));
	vx_add_colour(val);
}

//--------------------------------------------------------------------------
// TEXn Functions
// param:
// -----nnn -------s ssssssss pppppppp
// n - TEXn
// p - pos in vertex table
// s - shift (fractional bits)

void vx_tex_unknown(uint32 param)
{
	uint32 n = (param >> 24) & 0x7;
	uint32 s = (param >> 8) & 0x1f;
	syslog_error(VX,"TEX%d *NOT IMPLEMENTED* Type %s Cnt %s Format %s Fractional bits: %d\n", n, vx_str_type[vx_tex_debug_type[n]], vx_str_tex_cnt[vx_tex_debug_cnt[n]], vx_str_fmt[vx_tex_debug_fmt[n]], s);
}

void vx_tex_n_st_s8(uint32 param)
{
	uint32 n = (param >> 24) & 0x7;
	uint32 pos = param & 0xff;
	uint32 s = (param >> 8) & 0x1f;
	
	syslog(VX,"tex_n_st_s8(%d)\n", pos);

	float divider = (1 << s);
	float x,y;

	sint8 *tptr;
	tptr = (sint8 *)(vx_vertex_ptr.ubp + pos);
	x = tptr[0];
	y = tptr[1];

	x /= divider;
	y /= divider;

	syslog(VX,"MultiTexCoord2f(GL_TEXTURE%d_ARB, %f, %f)\n", n, x, y);
	vx_add_texc(x, y);
}

void vx_tex_n_st_u8(uint32 param)
{
	uint32 n = (param >> 24) & 0x7;
	uint32 pos = param & 0xff;
	uint32 s = (param >> 8) & 0x1f;
	
	syslog(VX,"tex_n_st_u8(%d)\n", pos);

	float divider = (1 << s);
	float x,y;
	uint8 *tptr;
	tptr = (uint8 *)(vx_vertex_ptr.ubp + pos);
	x = tptr[0];
	y = tptr[1];
	x /= divider;
	y /= divider;

	syslog(VX,"MultiTexCoord2f(GL_TEXTURE%d_ARB, %f, %f)\n", n, x, y);
	vx_add_texc(x, y);
}


void vx_tex_n_st_s16(uint32 param)
{
	uint32 n = (param >> 24) & 0x7;
	uint32 pos = param & 0xff;
	uint32 s = (param >> 8) & 0x1f;
	
	syslog(VX,"tex_n_st_s16(%d)\n", pos);

	float divider = (1 << s);
	float x,y;

	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	x = (sint16)vx_swap16(tptr[0]);
	y = (sint16)vx_swap16(tptr[1]);

	x /= divider;
	y /= divider;

	syslog(VX,"glMultiTexCoord2f(GL_TEXTURE%d_ARB, %f, %f)\n", n, x, y);
	vx_add_texc(x, y);
}

void vx_tex_n_st_u16(uint32 param)
{
	uint32 n = (param >> 24) & 0x7;
	uint32 pos = param & 0xff;
	uint32 s = (param >> 8) & 0x1f;
	
	syslog(VX,"tex_n_st_u16(%d)\n", pos);

	float divider = (1 << s);
	float x,y;
#if USE_VERTEX_PTR
	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	x = (uint16)vx_swap16(tptr[0]);
	y = (uint16)vx_swap16(tptr[1]);
#else
	x = vx_vertex_data[pos].us;
	y = vx_vertex_data[pos + 1].us;
#endif

	x /= divider;
	y /= divider;
	
	syslog(VX,"MultiTexCoord(%d, %f, %f)\n", n, x, y);
	vx_add_texc(x, y);
}

void vx_tex_n_st_f32(uint32 param)
{
	uint32 n = (param >> 24) & 0x7;
	uint32 pos = param & 0xff;
	
	syslog(VX,"tex_n_st_f32(%d)\n", pos);

	uint32 s = (param >> 8) & 0x1f;
	if (s)
	{
		syslog_error(VX,"Fractional bits in float: %d\n", s);
	}
	
	vx_vertex_t x, y;

	uint32 *tptr;
	tptr = (uint32 *)(vx_vertex_ptr.ubp + pos);

	x.i = (uint32)vx_swap32(tptr[0]);
	y.i = (uint32)vx_swap32(tptr[1]);

	syslog(VX,"glMultiTexCoord2f(GL_TEXTURE%d_ARB, %f, %f)\n", n, x.f, y.f);
	vx_add_texc(x.f, y.f);
}

void vx_tex_i8_st_u8(uint32 param)
{
	uint32 n = (param >> 24) & 0x7;
	uint32 s = (param >> 8) & 0x1f;
	uint32 pos = param & 0xff;

	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;

	syslog(VX,"tex_i8_st_u8(%d)\n", pos);
	if (s)
	{
		syslog_error(VX,"Fractional bits: %d\n", s);
	}
	index_mem = gp_cp_regs[0xa4];
	index_size = gp_cp_regs[0xb4];

#if USE_VERTEX_PTR
	index_pos = vx_vertex_ptr.ubp[pos];
#else
	index_pos = vx_vertex_data[pos].ub;
#endif

	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	float divider, x, y;
	divider = (1 << s);

	x = index_ptr.ubp[0];
	y = index_ptr.ubp[1];

	x /= divider;
	y /= divider;

	syslog(VX,"TEX%d_COORD(%f, %f)\n", n, x, y);
	vx_add_texc(x, y);
}

void vx_tex_i8_st_s8(uint32 param)
{
	uint32 n = (param >> 24) & 0x7;
	uint32 pos = param & 0xff;

	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;

	syslog(VX,"tex_i8_st_s8(%d)\n", pos);

	uint32 s = (param >> 8) & 0x1f;
	if (s)
	{
		syslog_error(VX,"Fractional bits: %d\n", s);
	}
	index_mem = gp_cp_regs[0xa4];
	index_size = gp_cp_regs[0xb4];

#if USE_VERTEX_PTR
	index_pos = vx_vertex_ptr.ubp[pos];
#else
	index_pos = vx_vertex_data[pos].ub;
#endif
	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	float divider, x, y;
	divider = (1 << s);

	x = index_ptr.sbp[0];
	y = index_ptr.sbp[1];

	x /= divider;
	y /= divider;

	syslog(VX,"MultiTexCoord2f(GL_TEXTURE%d_ARB, %f, %f)\n", n, x, y);
	vx_add_texc(x, y);
}

void vx_tex_i8_st_s16(uint32 param)
{
	uint32 n = (param >> 24) & 0x7;
	uint32	pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;
	uint32	i;

	syslog_error(VX,"tex_i8_st_s16(%d)\n", pos);

	uint32 s = (param >> 8) & 0x1f;
	if (s)
	{
		syslog_error(VX,"Fractional bits: %d\n", s);
	}

	index_mem = gp_cp_regs[0xa4];
	index_size = gp_cp_regs[0xb4];
#if USE_VERTEX_PTR
	index_pos = vx_vertex_ptr.ubp[pos];
#else
	index_pos = vx_vertex_data[pos].ub;
#endif
	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	float x, y;
	float divider = (1 << s);

	x = (sint16)vx_swap16(index_ptr.usp[0]);
	y = (sint16)vx_swap16(index_ptr.usp[1]);

	x /= divider;
	y /= divider;

	syslog_error(VX,"glMultiTexCoord2f(GL_TEXTURE%d_ARB, %f, %f)\n", n, x, y);

#if WITH_VERTEX_CACHE
	// for debugging purposes
	if ((index_mem & 0x1) || (index_size & 0x1))
	{
		syslog_error(VX,"tex_i8_st_s16() Unaligned memory access\n");
	}
	if(!vx_tex_valid[index_pos])
	{
		for(i = 0 ; i < index_size ; i++)
		{
			vx_tex_cache[index_pos * index_size + i] = gx_memory[index_mem + ((index_pos * index_size + i) ^ 1)];
		}
		vx_tex_valid[index_pos] = 1;
		if (index_pos > vx_tex_watermark)
			vx_tex_watermark = index_pos;
	}
	float divider = (1 << s);
	glTexCoord2f(((sint16 *)(&vx_tex_cache[index_pos * index_size]))[0] / divider, ((sint16*)(&vx_tex_cache[index_pos * index_size]))[1] / divider);
#endif
}


void vx_tex_i8_st_f32(uint32 param)
{
	uint32 n = (param >> 24) & 0x7;
	uint32	pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;
	uint32	i;
	uint32 s = (param >> 8) & 0x1f;

	syslog_error(VX,"tex_i8_st_f32(%d)\n", pos);
	if (s)
	{
		syslog_error(VX,"Fractional bits in float: %d\n", s);
	}

	index_mem = gp_cp_regs[0xa4];
	index_size = gp_cp_regs[0xb4];
#if USE_VERTEX_PTR
	index_pos = vx_vertex_ptr.ubp[pos];
#else
	index_pos = vx_vertex_data[pos].ub;
#endif
	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	vx_vertex_t	x, y;

	x.i = vx_swap32(index_ptr.ulp[0]);
	y.i = vx_swap32(index_ptr.ulp[1]);

	syslog_error(VX,"glMultiTexCoord2f(GL_TEXTURE%d_ARB, %f, %f)\n", n, x.f, y.f);

#if WITH_VERTEX_CACHE
	// for debugging purposes
	if ((index_mem & 0x3) || (index_size & 0x3))
	{
		syslog_error(VX,"tex_i8_st_f32() Unaligned memory access\n");
	}
	if(!vx_tex_valid[index_pos])
	{
		for(i = 0 ; i < index_size ; i++)
		{
			vx_tex_cache[index_pos * index_size + i] = gx_memory[index_mem + ((index_pos * index_size + i) ^ 3)];
		}
		vx_tex_valid[index_pos] = 1;
		if (index_pos > vx_tex_watermark)
			vx_tex_watermark = index_pos;
	}
	syslog_error(VX,"glTexCoord2fv(%f, %f)\n", ((GLfloat *)(&vx_tex_cache[index_pos * index_size]))[0], ((GLfloat *)(&vx_tex_cache[index_pos * index_size]))[1]);
	glTexCoord2fv((GLfloat *)(&vx_tex_cache[index_pos * index_size]));
#endif
}

void vx_tex_i16_st_u16(uint32 param)
{
	uint32 n = (param >> 24) & 0x7;
	uint32	pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;
	uint32	i;
	uint32 s = (param >> 8) & 0x1f;

	syslog(VX,"tex_i16_st_f32(%d)\n", pos);
	if (s)
	{
		syslog_error(VX,"Fractional bits: %d\n", s);
	}

	index_mem = gp_cp_regs[0xa4];
	index_size = gp_cp_regs[0xb4];

	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	index_pos = (uint16)vx_swap16(tptr[0]);

	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	float divider = (1 << s);
	float x, y;

	x = (uint16)vx_swap16(index_ptr.usp[0]);
	y = (uint16)vx_swap16(index_ptr.usp[1]);

	x /= divider;
	y /= divider;

	syslog_error(VX,"glMultiTexCoord2f(GL_TEXTURE%d_ARB, %f, %f)\n", n, x, y);
	vx_add_texc(x, y);
}

void vx_tex_i16_st_s16(uint32 param)
{
	uint32 n = (param >> 24) & 0x7;
	uint32	pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;
	uint32	i;
	uint32 s = (param >> 8) & 0x1f;

	syslog_error(VX,"tex_i16_st_f32(%d)\n", pos);
	if (s)
	{
		syslog_error(VX,"Fractional bits: %d\n", s);
	}

	index_mem = gp_cp_regs[0xa4];
	index_size = gp_cp_regs[0xb4];

#if USE_VERTEX_PTR
	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	index_pos = (uint16)vx_swap16(tptr[0]);
#else
	index_pos = vx_vertex_data[pos].us;
#endif

	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	float divider = (1 << s);
	float x, y;

	x = (sint16)vx_swap16(index_ptr.usp[0]);
	y = (sint16)vx_swap16(index_ptr.usp[1]);

	x /= divider;
	y /= divider;

	syslog_error(VX,"MultiTexCoord2f(GL_TEXTURE%d_ARB, %f, %f)\n", n, x, y);
}

void vx_tex_i16_st_f32(uint32 param)
{
	uint32 n = (param >> 24) & 0x7;
	uint32	pos = param & 0xff;
	uint32 index_pos, index_mem, index_size;
	vx_vertex_ptr_t index_ptr;
	uint32	i;
	uint32 s = (param >> 8) & 0x1f;

	syslog(VX,"tex_i16_st_f32(%d)\n", pos);
	if (s)
	{
		syslog_error(VX,"Fractional bits in float: %d\n", s);
	}

	index_mem = gp_cp_regs[0xa4];
	index_size = gp_cp_regs[0xb4];

	uint16 *tptr;
	tptr = (uint16 *)(vx_vertex_ptr.ubp + pos);

	index_pos = (uint16)vx_swap16(tptr[0]);

	index_ptr.ubp = gx_memory + index_mem + (index_pos * index_size);

	vx_vertex_t	x, y;

	x.i = vx_swap32(index_ptr.ulp[0]);
	y.i = vx_swap32(index_ptr.ulp[1]);

	syslog(VX,"glMultiTexCoord2f(GL_TEXTURE%d_ARB, %f, %f)\n", n, x.f, y.f);
	vx_add_texc(x.f, y.f);
}



//---------------------------------------------------------------------------

// pos 
vx_command_t vx_cmds_pos_n[16] =
{
	// VAT_A bit 0 - XY/XYZ bits 1-3 FMT
	vx_pos_n_xy_u8,			1,	2,		// VX_F_U8 | VX_C_XY
	vx_pos_n_xyz_u8,		1,	3,		// VX_F_U8 | VX_C_XYZ
	vx_pos_n_xy_s8,			1,	2,		// VX_F_S8 | VX_C_XY
	vx_pos_n_xyz_s8,		1,	3,		// VX_F_S8 | VX_C_XYZ
	vx_pos_n_xy_u16,		2,	2,		// VX_F_U16 | VX_C_XY
	vx_pos_n_xyz_u16,		2,	3,		// VX_F_U16 | VX_C_XYZ
	vx_pos_n_xy_s16,		2,	2,		// VX_F_S16 | VX_C_XY
	vx_pos_n_xyz_s16,		2,	3,		// VX_F_S16 | VX_C_XYZ
	vx_pos_n_xy_f32,		4,	2,		// VX_F_F32 | VX_C_XY
	vx_pos_n_xyz_f32,		4,	3,		// VX_F_F32 | VX_C_XYZ
	vx_pos_unknown,			7,	5,		// VX_F_UNK | VX_C_XY
	vx_pos_unknown,			7,	5,		// VX_F_UNK | VX_C_XYZ
	vx_pos_unknown,			7,	5,		// VX_F_UNK | VX_C_XY
	vx_pos_unknown,			7,	5,		// VX_F_UNK | VX_C_XYZ
	vx_pos_unknown,			7,	5,		// VX_F_UNK | VX_C_XY
	vx_pos_unknown,			7,	5,		// VX_F_UNK | VX_C_XYZ
};

// pos 
vx_command_t vx_cmds_pos_i8[16] =
{
	// VAT_A bit 0 - XY/XYZ bits 1-3 FMT
	vx_pos_unknown,			1,	1,		// VX_F_U8 | VX_C_XY
	vx_pos_i8_xyz_u8,		1,	1,		// VX_F_U8 | VX_C_XYZ
	vx_pos_i8_xy_s8,		1,	1,		// VX_F_S8 | VX_C_XY
	vx_pos_i8_xyz_s8,		1,	1,		// VX_F_S8 | VX_C_XYZ
	vx_pos_unknown,			1,	1,		// VX_F_U16 | VX_C_XY
	vx_pos_unknown,			1,	1,		// VX_F_U16 | VX_C_XYZ
	vx_pos_i8_xy_s16,		1,	1,		// VX_F_S16 | VX_C_XY
	vx_pos_i8_xyz_s16,		1,	1,		// VX_F_S16 | VX_C_XYZ
	vx_pos_unknown,			1,	1,		// VX_F_F32 | VX_C_XY
	vx_pos_i8_xyz_f32,		1,	1,		// VX_F_F32 | VX_C_XYZ
	vx_pos_unknown,			1,	1,		// VX_F_UNK | VX_C_XY
	vx_pos_unknown,			1,	1,		// VX_F_UNK | VX_C_XYZ
	vx_pos_unknown,			1,	1,		// VX_F_UNK | VX_C_XY
	vx_pos_unknown,			1,	1,		// VX_F_UNK | VX_C_XYZ
	vx_pos_unknown,			1,	1,		// VX_F_UNK | VX_C_XY
	vx_pos_unknown,			1,	1,		// VX_F_UNK | VX_C_XYZ
};

vx_command_t vx_cmds_pos_i16[16] =
{
	// VAT_A bit 0 - XY/XYZ bits 1-3 FMT
	vx_pos_unknown,			2,	1,		// VX_F_U8 | VX_C_XY
	vx_pos_unknown,			2,	1,		// VX_F_U8 | VX_C_XYZ
	vx_pos_unknown,			2,	1,		// VX_F_S8 | VX_C_XY
	vx_pos_i16_xyz_s8,		2,	1,		// VX_F_S8 | VX_C_XYZ
	vx_pos_unknown,			2,	1,		// VX_F_U16 | VX_C_XY
	vx_pos_unknown,			2,	1,		// VX_F_U16 | VX_C_XYZ
	vx_pos_unknown,			2,	1,		// VX_F_S16 | VX_C_XY
	vx_pos_i16_xyz_s16,		2,	1,		// VX_F_S16 | VX_C_XYZ
	vx_pos_unknown,			2,	1,		// VX_F_F32 | VX_C_XY
	vx_pos_i16_xyz_f32,		2,	1,		// VX_F_F32 | VX_C_XYZ
	vx_pos_unknown,			2,	1,		// VX_F_UNK | VX_C_XY
	vx_pos_unknown,			2,	1,		// VX_F_UNK | VX_C_XYZ
	vx_pos_unknown,			2,	1,		// VX_F_UNK | VX_C_XY
	vx_pos_unknown,			2,	1,		// VX_F_UNK | VX_C_XYZ
	vx_pos_unknown,			2,	1,		// VX_F_UNK | VX_C_XY
	vx_pos_unknown,			2,	1,		// VX_F_UNK | VX_C_XYZ
};

// ---------------------- Normal Tables ---------------------------------

vx_command_t vx_cmds_nor_n[16] =
{
	vx_nor_unknown,			7,	5,		// VX_F_U8 | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_U8 | VX_N_NBT
	vx_nor_n_nor_s8,		1,	3,		// VX_F_S8 | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_S8 | VX_N_NBT
	vx_nor_unknown,			7,	5,		// VX_F_U16 | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_U16 | VX_N_NBT
	vx_nor_n_nor_s16,		2,	3,		// VX_F_S16 | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_S16 | VX_N_NBT
	vx_nor_n_nor_f32,		4,	3,		// VX_F_F32 | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_F32 | VX_N_NBT
	vx_nor_unknown,			7,	5,		// VX_F_UNK | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_UNK | VX_N_NBT
	vx_nor_unknown,			7,	5,		// VX_F_UNK | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_UNK | VX_N_NBT
	vx_nor_unknown,			7,	5,		// VX_F_UNK | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_UNK | VX_N_NBT
};

vx_command_t vx_cmds_nor_i8[16] =
{
	vx_nor_unknown,			7,	5,		// VX_F_U8 | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_U8 | VX_N_NBT
	vx_nor_i8_nor_s8,		1,	1,		// VX_F_S8 | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_S8 | VX_N_NBT
	vx_nor_unknown,			7,	5,		// VX_F_U16 | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_U16 | VX_N_NBT
	vx_nor_i8_nor_s16,		1,	1,		// VX_F_S16 | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_S16 | VX_N_NBT
	vx_nor_i8_nor_f32,		1,	1,		// VX_F_F32 | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_F32 | VX_N_NBT
	vx_nor_unknown,			7,	5,		// VX_F_UNK | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_UNK | VX_N_NBT
	vx_nor_unknown,			7,	5,		// VX_F_UNK | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_UNK | VX_N_NBT
	vx_nor_unknown,			7,	5,		// VX_F_UNK | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_UNK | VX_N_NBT
};

vx_command_t vx_cmds_nor_i16[16] =
{
	vx_nor_unknown,			7,	5,		// VX_F_U8 | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_U8 | VX_N_NBT
	vx_nor_i16_nor_s16,		2,	1,		// VX_F_S8 | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_S8 | VX_N_NBT
	vx_nor_unknown,			7,	5,		// VX_F_U16 | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_U16 | VX_N_NBT
	vx_nor_i16_nor_s16,		2,	1,		// VX_F_S16 | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_S16 | VX_N_NBT
	vx_nor_i16_nor_f32,		2,	1,		// VX_F_F32 | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_F32 | VX_N_NBT
	vx_nor_unknown,			7,	5,		// VX_F_UNK | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_UNK | VX_N_NBT
	vx_nor_unknown,			7,	5,		// VX_F_UNK | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_UNK | VX_N_NBT
	vx_nor_unknown,			7,	5,		// VX_F_UNK | VX_N_NOR
	vx_nor_unknown,			7,	5,		// VX_F_UNK | VX_N_NBT
};
// ----------------------------------------------------------------------

// col
vx_command_t vx_cmds_col_n[16] =
{
	vx_col_unknown,			7,	5,		// VX_F_RGB565 | VX_C_RGB
	vx_col_unknown,			7,	5,		// VX_F_RGB565 | VX_C_RGBA
	vx_col_n_rgb_rgb8,		1,	3,		// VX_F_RGB8   | VX_C_RGB
	vx_col_unknown,			7,	5,		// VX_F_RGB8   | VX_C_RGBA
	vx_col_unknown,			7,	5,		// VX_F_RGBX8  | VX_C_RGB
	vx_col_unknown,			7,	5,		// VX_F_RGBX8  | VX_C_RGBA
	vx_col_unknown,			7,	5,		// VX_F_RGBA4  | VX_C_RGB
	vx_col_n_rgba_rgba4,	2,	1,		// VX_F_RGBA4  | VX_C_RGBA
	vx_col_unknown,			7,	5,		// VX_F_RGBA6  | VX_C_RGB
	vx_col_unknown,			7,	5,		// VX_F_RGBA6  | VX_C_RGBA
	vx_col_unknown,			7,	5,		// VX_F_RGBA8  | VX_C_RGB
	vx_col_n_rgba_rgba8,	1,	4,		// VX_F_RGBA8  | VX_C_RGBA
	vx_col_unknown,			7,	5,		// VX_F_UNK    | VX_C_RGB
	vx_col_unknown,			7,	5,		// VX_F_UNK    | VX_C_RGBA
	vx_col_unknown,			7,	5,		// VX_F_UNK    | VX_C_RGB
	vx_col_unknown,			7,	5,		// VX_F_UNK    | VX_C_RGBA
};

// col
vx_command_t vx_cmds_col_i8[16] =
{
	vx_col_i8_rgb_rgb565,	1,	1,		// VX_F_RGB565 | VX_C_RGB
	vx_col_unknown,			1,	1,		// VX_F_RGB565 | VX_C_RGBA
	vx_col_i8_rgb_rgb8,		1,	1,		// VX_F_RGB8   | VX_C_RGB
	vx_col_unknown,			1,	1,		// VX_F_RGB8   | VX_C_RGBA
	vx_col_unknown,			1,	1,		// VX_F_RGBX8  | VX_C_RGB
	vx_col_unknown,			1,	1,		// VX_F_RGBX8  | VX_C_RGBA
	vx_col_unknown,			1,	1,		// VX_F_RGBA4  | VX_C_RGB
	vx_col_i8_rgba_rgba4,	1,	1,		// VX_F_RGBA4  | VX_C_RGBA
	vx_col_unknown,			1,	1,		// VX_F_RGBA6  | VX_C_RGB
	vx_col_unknown,			1,	1,		// VX_F_RGBA6  | VX_C_RGBA
	vx_col_unknown,			1,	1,		// VX_F_RGBA8  | VX_C_RGB
	vx_col_i8_rgba_rgba8,	1,	1,		// VX_F_RGBA8  | VX_C_RGBA
	vx_col_unknown,			1,	1,		// VX_F_UNK    | VX_C_RGB
	vx_col_unknown,			1,	1,		// VX_F_UNK    | VX_C_RGBA
	vx_col_unknown,			1,	1,		// VX_F_UNK    | VX_C_RGB
	vx_col_unknown,			1,	1,		// VX_F_UNK    | VX_C_RGBA
};

vx_command_t vx_cmds_col_i16[16] =
{
	vx_col_i16_rgb_rgb565,	2,	1,		// VX_F_RGB565 | VX_C_RGB
	vx_col_unknown,			2,	1,		// VX_F_RGB565 | VX_C_RGBA
	vx_col_i16_rgb_rgb8,	2,	1,		// VX_F_RGB8   | VX_C_RGB
	vx_col_unknown,			2,	1,		// VX_F_RGB8   | VX_C_RGBA
	vx_col_unknown,			2,	1,		// VX_F_RGBX8  | VX_C_RGB
	vx_col_unknown,			2,	1,		// VX_F_RGBX8  | VX_C_RGBA
	vx_col_unknown,			2,	1,		// VX_F_RGBA4  | VX_C_RGB
	vx_col_i16_rgba_rgba4,	2,	1,		// VX_F_RGBA4  | VX_C_RGBA
	vx_col_unknown,			2,	1,		// VX_F_RGBA6  | VX_C_RGB
	vx_col_unknown,			2,	1,		// VX_F_RGBA6  | VX_C_RGBA
	vx_col_unknown,			2,	1,		// VX_F_RGBA8  | VX_C_RGB
	vx_col_i16_rgba_rgba8,	2,	1,		// VX_F_RGBA8  | VX_C_RGBA
	vx_col_unknown,			2,	1,		// VX_F_UNK    | VX_C_RGB
	vx_col_unknown,			2,	1,		// VX_F_UNK    | VX_C_RGBA
	vx_col_unknown,			2,	1,		// VX_F_UNK    | VX_C_RGB
	vx_col_unknown,			2,	1,		// VX_F_UNK    | VX_C_RGBA
};

vx_command_t vx_cmds_tex_n[16] =
{
	vx_tex_unknown,			7,	5,		// VX_F_U8 | VX_T_S
	vx_tex_n_st_u8,			1,	2,		// VX_F_U8 | VX_T_ST
	vx_tex_unknown,			7,	5,		// VX_F_S8 | VX_T_S
	vx_tex_n_st_s8,			1,	2,		// VX_F_S8 | VX_T_ST
	vx_tex_unknown,			7,	5,		// VX_F_U16 | VX_T_S
	vx_tex_n_st_u16,		2,	2,		// VX_F_U16 | VX_T_ST
	vx_tex_unknown,			7,	5,		// VX_F_S16 | VX_T_S
	vx_tex_n_st_s16,		2,	2,		// VX_F_S16 | VX_T_ST
	vx_tex_unknown,			7,	5,		// VX_F_F32 | VX_T_S
	vx_tex_n_st_f32,		4,	2,		// VX_F_F32 | VX_T_ST
	vx_tex_unknown,			7,	5,		// VX_F_UNK | VX_T_S
	vx_tex_unknown,			7,	5,		// VX_F_UNK | VX_T_ST
	vx_tex_unknown,			7,	5,		// VX_F_UNK | VX_T_S
	vx_tex_unknown,			7,	5,		// VX_F_UNK | VX_T_ST
	vx_tex_unknown,			7,	5,		// VX_F_UNK | VX_T_S
	vx_tex_unknown,			7,	5,		// VX_F_UNK | VX_T_ST
};

vx_command_t vx_cmds_tex_i8[16] =
{
	vx_tex_unknown,			1,	1,		// VX_F_U8 | VX_T_S
	vx_tex_i8_st_u8,		1,	1,		// VX_F_U8 | VX_T_ST
	vx_tex_unknown,			1,	1,		// VX_F_S8 | VX_T_S
	vx_tex_i8_st_s8,		1,	1,		// VX_F_S8 | VX_T_ST
	vx_tex_unknown,			1,	1,		// VX_F_U16 | VX_T_S
	vx_tex_unknown,			1,	1,		// VX_F_U16 | VX_T_ST
	vx_tex_unknown,			1,	1,		// VX_F_S16 | VX_T_S
	vx_tex_i8_st_s16,		1,	1,		// VX_F_S16 | VX_T_ST
	vx_tex_unknown,			1,	1,		// VX_F_F32 | VX_T_S
	vx_tex_i8_st_f32,		1,	1,		// VX_F_F32 | VX_T_ST
	vx_tex_unknown,			1,	1,		// VX_F_UNK | VX_T_S
	vx_tex_unknown,			1,	1,		// VX_F_UNK | VX_T_ST
	vx_tex_unknown,			1,	1,		// VX_F_UNK | VX_T_S
	vx_tex_unknown,			1,	1,		// VX_F_UNK | VX_T_ST
	vx_tex_unknown,			1,	1,		// VX_F_UNK | VX_T_S
	vx_tex_unknown,			1,	1,		// VX_F_UNK | VX_T_ST
};

vx_command_t vx_cmds_tex_i16[16] =
{
	vx_tex_unknown,			2,	1,		// VX_F_U8 | VX_T_S
	vx_tex_unknown,			2,	1,		// VX_F_U8 | VX_T_ST
	vx_tex_unknown,			2,	1,		// VX_F_S8 | VX_T_S
	vx_tex_unknown,			2,	1,		// VX_F_S8 | VX_T_ST
	vx_tex_unknown,			2,	1,		// VX_F_U16 | VX_T_S
	vx_tex_i16_st_u16,		2,	1,		// VX_F_U16 | VX_T_ST
	vx_tex_unknown,			2,	1,		// VX_F_S16 | VX_T_S
	vx_tex_i16_st_s16,		2,	1,		// VX_F_S16 | VX_T_ST
	vx_tex_unknown,			2,	1,		// VX_F_F32 | VX_T_S
	vx_tex_i16_st_f32,		2,	1,		// VX_F_F32 | VX_T_ST
	vx_tex_unknown,			2,	1,		// VX_F_UNK | VX_T_S
	vx_tex_unknown,			2,	1,		// VX_F_UNK | VX_T_ST
	vx_tex_unknown,			2,	1,		// VX_F_UNK | VX_T_S
	vx_tex_unknown,			2,	1,		// VX_F_UNK | VX_T_ST
	vx_tex_unknown,			2,	1,		// VX_F_UNK | VX_T_S
	vx_tex_unknown,			2,	1,		// VX_F_UNK | VX_T_ST
};

/* generic
vx_command_t vx_cmds_pos[16] =
{
	vx_pos_unknown,			7,	5,		// VX_F_U8 | VX_C_XY
	vx_pos_unknown,			7,	5,		// VX_F_U8 | VX_C_XYZ
	vx_pos_unknown,			7,	5,		// VX_F_S8 | VX_C_XY
	vx_pos_unknown,			7,	5,		// VX_F_S8 | VX_C_XYZ
	vx_pos_unknown,			7,	5,		// VX_F_U16 | VX_C_XY
	vx_pos_unknown,			7,	5,		// VX_F_U16 | VX_C_XYZ
	vx_pos_unknown,			7,	5,		// VX_F_S16 | VX_C_XY
	vx_pos_n_xyz_s16,		2,	3,		// VX_F_S16 | VX_C_XYZ
	vx_pos_unknown,			7,	5,		// VX_F_F32 | VX_C_XY
	vx_pos_unknown,			7,	5,		// VX_F_F32 | VX_C_XYZ
	vx_pos_unknown,			7,	5,		// VX_F_UNK | VX_C_XY
	vx_pos_unknown,			7,	5,		// VX_F_UNK | VX_C_XYZ
	vx_pos_unknown,			7,	5,		// VX_F_UNK | VX_C_XY
	vx_pos_unknown,			7,	5,		// VX_F_UNK | VX_C_XYZ
	vx_pos_unknown,			7,	5,		// VX_F_UNK | VX_C_XY
	vx_pos_unknown,			7,	5,		// VX_F_UNK | VX_C_XYZ
};
*/






uint32 vx_get_text_vat(uint32 vat, uint32 i)
{
	uint32 text_vat, vat_x;
	if (i == 0)
	{
		// for TEX0 params are in vat_a
		// VAT_A
		vat_x = gp_cp_regs[0x70 + vat];
		text_vat = vat_x >> 21;
	}
	else if (i < 5)
	{
		// for TEX1-4 params are in vat_b
		vat_x = gp_cp_regs[0x80 + vat];
		text_vat = vat_x >> (i - 1) * 9;
	}
	else
	{
		// for TEX5-7 params are in vat_c
		vat_x = gp_cp_regs[0x90 + vat];
		text_vat = vat_x >> (i - 5) * 9;
	}
	text_vat &= 0x1ff;
	return text_vat;
}






















