/*====================================================================

filename:     pad.cpp
project:      GCemu
created:      2004-6-18
mail:		  duddie@walla.com

Copyright (c) 2005 Duddie & Tratax

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

====================================================================*/
#define DIRECTINPUT_VERSION 0x0300
#include <dinput.h>
#include <stdio.h>
#include "system/types.h"
#include "pad.h"

extern struct pad_data pad[4];

// this should come from config file

typedef struct
{
	uint32 key;
	uint32 bit;
} key_t;

key_t key_table[12] = {
	DIK_1, 28, // 28: BUTTON_START
	DIK_Y, 27, // 27: BUTTON_X
	DIK_X, 26, // 26: BUTTON_Y
	DIK_B, 25, // 25: BUTTON_B
	DIK_V, 24, // 24: BUTTON_A
	DIK_LBRACKET, 22, // 22: BUTTON_L
	DIK_RBRACKET, 21, // 21: BUTTON_R
	DIK_Z, 20, // 20: BUTTON_Z
	DIK_I, 19, // 19: BUTTON_UP
	DIK_K, 18, // 18: BUTTON_DOWN
	DIK_L, 17,// 17: BUTTON_RIGHT
	DIK_J, 16,// 16: BUTTON_LEFT
};

#define ANALOG_LEFT DIK_LEFT
#define ANALOG_RIGHT DIK_RIGHT
#define ANALOG_UP DIK_UP
#define ANALOG_DOWN DIK_DOWN
#define ANALOG_LEFT DIK_LEFT

#define ANALOG_C_RIGHT DIK_F
#define ANALOG_C_UP DIK_E
#define ANALOG_C_DOWN DIK_D
#define ANALOG_C_LEFT DIK_S

BYTE keys[256];
IDirectInput       *lpDI;
IDirectInputDevice *lpDID;

// analog emulation
// swing joystick from left to right slowly
// and not go from one extreme to another

#define ANA_NEUTRAL 0
#define ANA_PLUS 1
#define ANA_MINUS 2

#define ANA_DELTA 1
#define ANA_MIN 24
#define ANA_MIDDLE 128
#define ANA_MAX 224

uint8 analog_emu(uint8 prev_val, uint8 direction)
{	
	if(direction == ANA_NEUTRAL)
	{
		if(prev_val < ANA_MIDDLE) 
		{
			prev_val += ANA_DELTA;
		}
		else if(prev_val > ANA_MIDDLE)
		{
			prev_val -= ANA_DELTA;
		}
	}
	else if(direction == ANA_PLUS)
	{
		if(prev_val < ANA_MAX)
		{
			prev_val += ANA_DELTA;
		}
	}
	else if(direction == ANA_MIN)
	{
		if(prev_val > ANA_MIN)
		{
			prev_val -= ANA_DELTA;
		}
	}
	return prev_val;
}

int pad_close()
{
	if (lpDID != NULL)
	{
		lpDID->Unacquire();
		lpDID->Release();
		lpDID = NULL;
	}

	if (lpDI != NULL)
	{
		lpDI->Release();
		lpDI=NULL;
	}
	return 0;
}


int pad_open(HWND hwnd, HINSTANCE hinstance)
{
	HRESULT res;

	lpDI = NULL;
	lpDID = NULL;

	if((res = DirectInputCreate(hinstance, DIRECTINPUT_VERSION, &lpDI, NULL))!=DI_OK)
	{
		switch(res)
		{
		case	DIERR_BETADIRECTINPUTVERSION:
				printf("ERROR: DirectInput DIERR_BETADIRECTINPUTVERSION\n");
				break;
		case	DIERR_INVALIDPARAM:
				printf("ERROR: DirectInput DIERR_INVALIDPARAM\n");
				break;
		case	DIERR_OLDDIRECTINPUTVERSION:
				printf("ERROR: DirectInput DIERR_OLDDIRECTINPUTVERSION\n");
				break;
		case	DIERR_OUTOFMEMORY:
				printf("ERROR: DirectInput DIERR_OUTOFMEMORY\n");
				break;
		default:
				printf("ERROR: unknown DirectInputCreate Error!\n");
				break;
		}
		pad_close();
		return -1;
	}

	res = lpDI->CreateDevice(GUID_SysKeyboard, &lpDID, NULL);
	if (res != DI_OK)
	{
		printf("ERROR: DirectInput: Create Device Failed");
		pad_close();
		return -1;
	}

	res = lpDID->SetDataFormat(&c_dfDIKeyboard);
	if (res != DI_OK)
	{
		printf("ERROR: DirectInput:Set Data Format failed\n");
		pad_close();
		return -1;
	}
	// for fullscreen: DISCL_NONEXCLUSIVE | DISCL_BACKGROUND
	// for window: DISCL_NONEXCLUSIVE | DISCL_FOREGROUND
	res = lpDID->SetCooperativeLevel(hwnd, DISCL_NONEXCLUSIVE | DISCL_BACKGROUND);

	if (res != DI_OK)
	{
		printf("ERROR: DirectInput:SetCooperative Level Failed\n");
		pad_close();
		return -1;
	}

	res = lpDID->Acquire();
	if (res != DI_OK)
	{
		printf("ERROR: DirectInput: Acquire Failed\n");
		pad_close();
		return -1;
	}

	// fill in starter data
	pad[0].pad_datah = 0x8080;
	pad[0].pad_datal = 0x80800000;

	printf("PAD driver 0.1 initialized\n");
	return 0;
}

bool pad_read(void)
{
	HRESULT res;
	unsigned short i;
	uint8 analog_val;

	// grab keyboard state into array
	do
	{
		res = lpDID->GetDeviceState(256, &keys);
		if (res == DIERR_INPUTLOST) lpDID->Acquire();
	}
	while (res == DIERR_INPUTLOST);

	if (res != DI_OK)
		return false;

	// parse keyboard info and fill in the pad structure
	pad[0].connected = 1;
	// analog left and right
	if(keys[ANALOG_LEFT])
	{
		analog_val = (pad[0].pad_datah >> 8);
		analog_val = analog_emu(analog_val, ANA_MIN);
	}
	else if(keys[ANALOG_RIGHT])
	{
		analog_val = (pad[0].pad_datah >> 8);
		analog_val = analog_emu(analog_val, ANA_PLUS);
	}
	else
	{
		// both not pressed, so go neutral
		analog_val = (pad[0].pad_datah >> 8);
		analog_val = analog_emu(analog_val, ANA_NEUTRAL);
	}
	pad[0].pad_datah &= 0xff; // mask off old value
	pad[0].pad_datah |= (analog_val<<8);

	// analog y up and down
	if(keys[ANALOG_DOWN])
	{
		analog_val = pad[0].pad_datah;
		analog_val = analog_emu(analog_val, ANA_MIN);
	}
	else if(keys[ANALOG_UP])
	{
		analog_val = pad[0].pad_datah;
		analog_val = analog_emu(analog_val, ANA_PLUS);
	}
	else
	{
		// both not pressed, so go neutral
		analog_val = pad[0].pad_datah;
		analog_val = analog_emu(analog_val, ANA_NEUTRAL);
	}
	pad[0].pad_datah &= 0xff00; // mask off old value
	pad[0].pad_datah |= analog_val;
	
	// buttons
	for(i = 0 ; i < 12 ; i++)
	{
		if (keys[key_table[i].key])
			pad[0].pad_datah |= (1 << key_table[i].bit);
	}

	// C stick left and right
	if(keys[ANALOG_C_LEFT])
	{
		analog_val = (pad[0].pad_datal >> 24);
		analog_val = analog_emu(analog_val, ANA_MIN);
	}
	else if(keys[ANALOG_C_RIGHT])
	{
		analog_val = (pad[0].pad_datal >> 24);
		analog_val = analog_emu(analog_val, ANA_PLUS);
	}
	else
	{
		// both not pressed, so go neutral
		analog_val = (pad[0].pad_datal >> 24);
		analog_val = analog_emu(analog_val, ANA_NEUTRAL);
	}
	pad[0].pad_datal &= 0x00ff0000; // mask off old value
	pad[0].pad_datal |= (analog_val<<24);

	// C stick up and down
	if(keys[ANALOG_C_DOWN])
	{
		analog_val = (pad[0].pad_datal >> 16);
		analog_val = analog_emu(analog_val, ANA_MIN);
	}
	else if(keys[ANALOG_C_UP])
	{
		analog_val = (pad[0].pad_datal >> 16);
		analog_val = analog_emu(analog_val, ANA_PLUS);
	}
	else
	{
		// both not pressed, so go neutral
		analog_val = (pad[0].pad_datal >> 16);
		analog_val = analog_emu(analog_val, ANA_NEUTRAL);
	}
	pad[0].pad_datal &= 0xff000000; // mask off old value
	pad[0].pad_datal |= (analog_val << 16);

	// l and r triggers .. copy of digital values
	pad[0].pad_datal &= 0xffff0000;
	if(pad[0].pad_datah & (1<<22))pad[0].pad_datal |= 0xff00;
	if(pad[0].pad_datah & (1<<21))pad[0].pad_datal |= 0xff;

	// we do not handle pads 2-4 right now
	pad[1].connected = 0;
	pad[1].pad_datah = 0x8080;
	pad[1].pad_datal = 0x80800000;
	pad[2].connected = 0;
	pad[2].pad_datah = 0x8080;
	pad[2].pad_datal = 0x80800000;
	pad[3].connected = 0;
	pad[3].pad_datah = 0x8080;
	pad[3].pad_datal = 0x80800000;

#if PAD_RECORD
	static FILE *padfile = NULL;
	if (padfile == NULL)
		padfile = fopen("pad_capture.dat", "wb");
	fwrite(&pad0_buttons, 1, 4, padfile);
#endif
#if PAD_PLAY
	static FILE *padfile = NULL;
	if (padfile == NULL)
		padfile = fopen("pad_capture.dat", "rb");
	fread(&pad0_buttons, 1, 4, padfile);
#endif

/*
	if (keys[DIK_F12])
	{
		di_open_cover();
	}
	if (keys[DIK_F11])
	{
		di_close_cover();
	}
	if (keys[DIK_F10])
	{
		mc_irq();
	}
*/
#if PAD_RECORD || PAD_PLAY
		fclose(padfile);
#endif
	return true;
}
