/*====================================================================

filename:     processor_interface.cpp
project:      GCemu
created:      2004-6-18
mail:		  duddie@walla.com

Copyright (c) 2005 Duddie & Tratax

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

====================================================================*/
#include "system/types.h"
#include "processor_interface.h"
#include "video_interface.h"
#include "hardware/hw_io.h"
#include "dsp_interface.h"
#include "dvd_interface.h"
#include "external_interface.h"
#include "plugins/gx_plugin.h"

#define PI_INTSR 0x00
#define PI_INTMR 0x04
#define PI_FIFO_START 0x0c
#define PI_FIFO_END 0x10
#define PI_FIFO_WRITEPOINTER 0x14

uint32	pi_intmr;
uint32	pi_fifo_start;
uint32	pi_fifo_end;
uint32	pi_fifo_writepointer;

typedef struct
{
	void	(*callback)(void);
	uint32	count;
} pi_irq_t;




pi_irq_t	pi_irq_table[PI_MAX_IRQ_SOURCES];



uint32 pi_read_intsr(void);

void pi_init(void)
{
	int i;
	pi_intmr = 0;
	for(i = 0 ; i < PI_MAX_IRQ_SOURCES ; i++)
	{
		pi_irq_table[i].count = 0;
		pi_irq_table[i].callback = NULL;
	}
}

void pi_close(void)
{
}

void pi_write_register8(uint32 addr, uint8 data)
{
	switch(addr & 0xff)
	{
	case 0x04:
		pi_intmr = data;
		break;
	default:
		break;
	}
}

void pi_write_register16(uint32 addr, uint16 data)
{
	switch(addr & 0xff)
	{
	case 0x04:
		pi_intmr = data;
		break;
	default:
		break;
	}
}

void pi_write_register32(uint32 addr, uint32 data)
{
	switch(addr & 0xff)
	{
	case PI_INTMR:
		pi_intmr = data;
		break;
	case PI_FIFO_START:
		pi_fifo_start = data & ALIGN_32BYTES;
		break;
	case PI_FIFO_END:
		pi_fifo_end = data & ALIGN_32BYTES;
		break;
	case PI_FIFO_WRITEPOINTER:
		pi_fifo_writepointer = data & ALIGN_32BYTES;
		break;
	default:
		syslog_warn(IO_CORE,"unhandled write to %08x (PI) (%08x)\n", addr, data);
		break;
	}
}

uint8 pi_read_register8(uint32 addr)
{
	uint8 data;
	switch(addr & 0xff)
	{
	case 0x00:
		data = pi_read_intsr();
		break;
	case 0x04:
		data = pi_intmr;
		break;
	default:
		data = 0;
		break;
	}
	return data;
}

uint16 pi_read_register16(uint32 addr)
{
	uint16 data;
	switch(addr & 0xff)
	{
	case PI_INTSR:
		data = pi_read_intsr();
		break;
	case PI_INTMR:
		data = pi_intmr;
		break;
	default:
		data = 0;
		break;
	}
	return data;
}

uint32 pi_read_register32(uint32 addr)
{
	uint32 data;
	switch(addr & 0xff)
	{
	case PI_INTSR:
		data = pi_read_intsr();
		break;
	case PI_INTMR:
		data = pi_intmr;
		break;
	case PI_FIFO_START:
		data = pi_fifo_start;
		break;
	case PI_FIFO_END:
		data = pi_fifo_end;
		break;
	case PI_FIFO_WRITEPOINTER:
		data = pi_fifo_writepointer;
		break;
	default:
		data = 0;
		break;
	}
	return data;
}

#define PI_INT_CP			11
#define PI_INT_PE_FINISH	10
#define PI_INT_VI			8
#define PI_INT_DSP			6
#define PI_INT_AI			5
#define PI_INT_EXI			4
#define PI_INT_SI			3
#define PI_INT_DI			2

uint32 pi_read_intsr(void)
{
	uint32 res;
	res = 0;
	if (vi_check_interrupt())
		res |= (1 << PI_INT_VI);
	if (di_check_interrupt())
		res |= (1 << PI_INT_DI);
	if (dspi_check_interrupt())
		res |= (1 << PI_INT_DSP);
	if (GX_PECheckInterrupt())
		res |= (1 << PI_INT_PE_FINISH);
	if (exi_check_interrupt())
		res |= (1 << PI_INT_EXI);
	if (si_check_interrupt())
		res |= (1 << PI_INT_SI);
	if (cp_check_interrupt())
		res |= (1 << PI_INT_CP);

	res &= pi_intmr;

	return res;
}


void pi_check_scheduler(void)
{
	int i;
	for(i = 0 ; i < PI_MAX_IRQ_SOURCES ; i++)
	{
		if(pi_irq_table[i].count != 0)
		{
			pi_irq_table[i].count--;
			if(pi_irq_table[i].count == 0)
			{
				if(pi_irq_table[i].callback != NULL)
				{
					pi_irq_table[i].callback();
				}
			}
		}
	}
}

bool pi_check_interrupt_lines(void)
{
	pi_check_scheduler();

	if (pi_read_intsr())
		return true;
	return false;
}

void pi_schedule_irq(uint32 irq, uint32 count, void (*callback)(void))
{
	pi_irq_table[irq].count = count;
	pi_irq_table[irq].callback = callback;
}
