/*====================================================================

filename:     dsp_interface.cpp
project:      GCemu
created:      2004-6-18
mail:		  duddie@walla.com

Copyright (c) 2005 Duddie & Tratax

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

====================================================================*/
#include "config.h"

#include <stdio.h>
#include "system/types.h"
#include "dsp_interface.h"
#include "hardware/hw_io.h"
#include "hardware/memory_interface.h"
#include "dsp_interface_defines.h"
#include "memory_card.h"
#include "debug/tracers.h"
#include "dsp/gdsp_interface.h"
#include "dsp/gdsp_interpreter.h"



unsigned long crc32(unsigned long crc, const unsigned char *buf, unsigned int len);

char dspi_regs[DSPI_REGS_SIZE];

uint8 *aram;

uint32	dspi_cnt_valid;
uint32	dspi_feed;
uint32	dspi_halted;

#define	DSPI_MB_QUEUE_SIZE	256
uint32	dspi_mb_queue[DSPI_MB_QUEUE_SIZE];
uint32	dspi_mb_queue_ptr_write;
uint32	dspi_mb_queue_ptr_read;
uint32	dspi_mail_to_dsp;

void dspi_init(void)
{
	int i;
	for (i = 0 ; i < DSPI_REGS_SIZE ; i++)
	{
		dspi_regs[i] = 0;
	}
	dspi_cnt_valid = 0;
	aram = (uint8 *)malloc(ARAM_SIZE);
}

void dspi_close(void)
{
	if (aram)
	{
		free(aram);
		aram = NULL;
	}
}

#define	DSPI_IRQ_DSP	0
#define	DSPI_IRQ_AR	1

void dspi_raise_irq(uint32 type)
{
	uint16	temp16;

	temp16 = DSPI_READ_UINT16(0x0a);
	switch(type & 0x3)
	{
	case DSPI_IRQ_DSP:
		temp16 |= AIDCR_DSPINT;
		break;
	case DSPI_IRQ_AR:
		temp16 |= AIDCR_ARINT;
		break;
	default:
		break;
	}
	DSPI_WRITE_UINT16(0x0a, temp16);
	trx_ppc_signal_interrupt();
}

void dspi_req_dsp_irq(void)
{
	dspi_raise_irq(DSPI_IRQ_DSP);
}

uint32 dsp_dump_code(uint32 addr, uint32 len)
{
	char filename[256];
	uint8 *save_data = (uint8 *)gMemory + (addr & 0x01ffffff);
	
	uint32 crc = crc32(0, save_data, len);
// uncomment to dump DSP programs for further analysis
//	sprintf(filename, "dsp/dsp_dump_%08x.bin", crc);
	
//	FILE *out = fopen(filename, "wb");
	
//	fwrite(save_data, 1, len, out);
//	fclose(out);
	return crc;
}

void dspi_aram_dma(void)
{
	uint32 cnt, araddr, mmaddr;
	uint32 i;

	// !!!!!!!!
	// addresses and count must be advanced after transfer completed

	mmaddr = DSPI_READ_UINT32(DSPI_REG_AR_DMA_MMADDR);
	araddr = DSPI_READ_UINT32(DSPI_REG_AR_DMA_ARADDR);
	cnt = DSPI_READ_UINT32(DSPI_REG_AR_DMA_CNT);

	if (dspi_cnt_valid == 0x3)
	{
		syslog(ARAM,"DMA %s addr: %08x size: %08x aram: %08x\n", cnt & 0x80000000 ? "READ" : "WRITE", mmaddr, cnt & 0x7fffffff, araddr);
		if(cnt & 0x80000000)
		{
			mmaddr &= 0x3fffffff;
			// read from ARAM
			if (araddr > ARAM_SIZE)
			{
				for(i = 0 ; i < (cnt & 0x7fffffff) ; i++)
					mem_write8(mmaddr, 0);
			}
			else
			{
				memcpy(gMemory + (mmaddr & 0x0fffffff), aram + araddr, cnt & 0x7fffffff);
			}
		}
		else
		{
			if (araddr > ARAM_SIZE)
			{
			}
			else
			{
				if (araddr < 0x1000)
				{
#if WITH_REAL_DSP
					gdsp_idma_in(araddr, mmaddr, (cnt & 0x7fffffff) * 4);
#else
					dsp_dump_code(mmaddr, (cnt & 0x7fffffff) * 4);
#endif	// WITH_REAL_DSP
				}
				memcpy(aram + araddr, gMemory + (mmaddr & 0x0fffffff), cnt & 0x7fffffff);
			}
			// write to ARAM
		}
		dspi_raise_irq(DSPI_IRQ_AR);

		dspi_cnt_valid = 0;
	}
}


void dspi_mb_queue_flush(void)
{
	dspi_mb_queue_ptr_read = dspi_mb_queue_ptr_write = 0;
}

void dspi_mb_queue_add(uint32 mail)
{
	syslog(DSPI,"MB Add: %08x pos: %d\n", mail, dspi_mb_queue_ptr_write);
	dspi_mb_queue[dspi_mb_queue_ptr_write++] = mail;
}

uint32 dspi_mb_queue_get(void)
{
	uint32 mail;
	mail = dspi_mb_queue[dspi_mb_queue_ptr_read];
	syslog(DSPI,"MB Get: %08x pos: %d\n", mail, dspi_mb_queue_ptr_read);
	return mail;
}

void dspi_mb_queue_advance(void)
{
//	if(dspi_halted == 0)
	{
		if (dspi_mb_queue_ptr_read < dspi_mb_queue_ptr_write)
		{
			dspi_mb_queue_ptr_read++;
			if (dspi_mb_queue_ptr_read == dspi_mb_queue_ptr_write)
			{
				dspi_mb_queue[0] = dspi_mb_queue[dspi_mb_queue_ptr_read - 1];
				dspi_mb_queue_ptr_read = dspi_mb_queue_ptr_write = 0;
			}
		}
	}
}

uint32	dspi_cmd_expected_params;
uint32	dspi_cmd;
uint32	dspi_cmd_param[10];
uint32	dspi_cmd_param_count;
void dspi_cmd_parser_reset(void)
{
	dspi_cmd_expected_params = 0;
}

uint32	dspi_idma_addr;
uint32	dspi_idma_imem;
uint32	dspi_idma_len;

typedef struct
{
	void	(*init)(void);
	uint32	(*parse_mail)(void);
	void	(*run_cmd)(void);
} dsp_task_t;

dsp_task_t	dsp_tasks[10];

//----------------------------------------------------------------------

uint32 dsp_hle_unknown_parse_mail(void)
{
	switch(dspi_cmd)
	{
	case 0xbabe0180:
	case 0xbabe0024:
	case 0xbabe0030:
	case 0xbabe0040:
	case 0xbabe0038:
		return 1;
	case 0x000000005:
		return 0;
	default:
//		fprintf(stderr, "Unknown Mail %08x\n", dspi_cmd);
		break;
	}
	return 0;
}
void dsp_hle_unknown_run_cmd(void)
{
	if ((dspi_cmd & 0xffff0000) == 0xbabe0000)
	{
/*	

	case 0xbabe0180:
	case 0xbabe0024:
	case 0xbabe0030:
	case 0xbabe0040:
	case 0xbabe0038:
*/
		dspi_raise_irq(DSPI_IRQ_DSP);
		dspi_mb_queue_add(0xdcd10002);
	}
//	fprintf(stderr, "Running CMD: %08x\n", dspi_cmd);
}
//----------------------------------------------------------------------
uint32 dsp_hle_axstream1_parse_mail(void)
{
	switch(dspi_cmd)
	{
	case 0xbabe0180:
	case 0xbabe0024:
	case 0xbabe0030:
	case 0xbabe0040:
	case 0xbabe0038:
		return 1;
	default:
//		fprintf(stderr, "Unknown Mail %08x\n", dspi_cmd);
		break;
	}
	return 0;
}

void dsp_hle_axstream1_run_cmd(void)
{
//	fprintf(stderr, "Running CMD: %08x\n", dspi_cmd);
	if ((dspi_cmd & 0xffff0000) == 0xbabe0000)
	{
/*	

	case 0xbabe0180:
	case 0xbabe0024:
	case 0xbabe0030:
	case 0xbabe0040:
	case 0xbabe0038:
*/
		dspi_raise_irq(DSPI_IRQ_DSP);
		dspi_mb_queue_add(0xdcd10002);
	}
/*	if (dspi_cmd == 0xcdd10003)
	{
		dspi_raise_irq(DSPI_IRQ_DSP);
		dspi_mb_queue_add(0xdcd10003);
	}
*/
}
void dsp_add_task(uint32 crc)
{
	uint32 i;
	fprintf(stderr, "adding dsp task ");
	i = 0;
	switch(crc)
	{
	case 0x6a696ce7:	// axstream1
		fprintf(stderr, "axstream1\n");
		dspi_raise_irq(DSPI_IRQ_DSP);
		dspi_mb_queue_add(0xdcd10000);
		dsp_tasks[i].parse_mail = dsp_hle_axstream1_parse_mail;
		dsp_tasks[i].run_cmd = dsp_hle_axstream1_run_cmd;
		break;
	case 0xb177bdec:
		fprintf(stderr, "Pikmin DSP code: %08x\n", crc);
		dspi_raise_irq(DSPI_IRQ_DSP);
		dspi_mb_queue_add(0x88881111);
		dsp_tasks[i].parse_mail = dsp_hle_unknown_parse_mail;
		dsp_tasks[i].run_cmd = dsp_hle_unknown_run_cmd;
		break;
	case 0x4cb8233b:
		fprintf(stderr, "Robotech Battle DSP code: %08x\n", crc);
		dspi_raise_irq(DSPI_IRQ_DSP);
		dspi_mb_queue_add(0xdcd10000);
		dsp_tasks[i].parse_mail = dsp_hle_unknown_parse_mail;
		dsp_tasks[i].run_cmd = dsp_hle_unknown_run_cmd;
		break;
	default:
		fprintf(stderr, "Unknown DSP code: %08x\n", crc);
		dspi_raise_irq(DSPI_IRQ_DSP);
		dspi_mb_queue_add(0xdcd10000);
		dsp_tasks[i].parse_mail = dsp_hle_unknown_parse_mail;
		dsp_tasks[i].run_cmd = dsp_hle_unknown_run_cmd;
		break;
	}
	//dsp_tasks[i].init();
}


void dspi_cmd_parse(void)
{
	//uint16	temp16;
	//uint32	temp32;

	switch(dspi_cmd)
	{
	case 0x80f3a001:
		dspi_idma_addr = dspi_cmd_param[0];
		break;
	case 0x80f3a002:
		dspi_idma_len = dspi_cmd_param[0];
		break;
	case 0x80f3c002:
		dspi_idma_imem = dspi_cmd_param[0];
		break;
	case 0x80f3d001:		// init vector
		syslog(DSPI,"IDMA %08x -> %08x (size: %08x) start: %08x\n", dspi_idma_addr, dspi_idma_imem, dspi_idma_len, dspi_cmd_param[0]);
		{
			uint32 crc;
			crc = dsp_dump_code(dspi_idma_addr, dspi_idma_len);
			dsp_add_task(crc);
		}
//		dspi_raise_irq(DSPI_IRQ_DSP);
//		dspi_mb_queue_add(0xdcd10000);
		break;
	
	case 0xff000000:
		dspi_raise_irq(DSPI_IRQ_DSP);
		dspi_mb_queue_add(0xdcd10003);
		mc_unlock();
		break;

	case 0x81000040:
//	case 0x00000003:
		// init vector
		//printf("Setting IRQ!!!\n");
		dspi_raise_irq(DSPI_IRQ_DSP);
		dspi_mb_queue_add(0xdcd10004);
		dspi_mb_queue_add(0xf3558100);
		break;

		//
/*	case 0xcdd10002:
		// DELETE TASK (SEEMS SO)
		break;
*/
	default:
		if (dsp_tasks[0].run_cmd)
			dsp_tasks[0].run_cmd();
		break;
	}
}
void dspi_parse_mail(uint32 mail)
{
	uint32	temp32;

	syslog(DSPI,"%08x Mail to DSP: %08x\n", CPUcurrmode->pc, mail);
	//fprintf(stderr, "Mail to DSP: %08x\n", mail);
	if (dspi_cmd_expected_params)
	{
		dspi_cmd_param[dspi_cmd_param_count] = mail;
		dspi_cmd_expected_params--;
	}
	else
	{
		dspi_cmd = mail;
		switch(dspi_cmd)
		{
		case 0x80f3a001:	// RAM addr
			dspi_cmd_expected_params = 1;
			break;
		case 0x80f3c002:	// IMEM addr
			dspi_cmd_expected_params = 1;
			break;
		case 0x80f3a002:	// len
			dspi_cmd_expected_params = 1;
			break;
		case 0x80f3b002:	// ???
			dspi_cmd_expected_params = 1;
			break;
		case 0x80f3d001:	// Execute addr
			dspi_cmd_expected_params = 1;
			break;
		case 0xff000000:	// card data
			dspi_cmd_expected_params = 1;
			break;
		default:
			if (dsp_tasks[0].parse_mail)
				dspi_cmd_expected_params = dsp_tasks[0].parse_mail();
			break;
		}
	}
	if(dspi_cmd_expected_params == 0)
		dspi_cmd_parse();

	temp32 = DSPI_READ_UINT32(DSPI_REG_MB_CPU);
	temp32 &= ~0x80000000; // mail received
	DSPI_WRITE_UINT32(DSPI_REG_MB_CPU, temp32);
}

void dspi_ai_irq_callback(void)
{
	uint16 temp16;
	temp16 = DSPI_READ_UINT16(0x0a);
	temp16 |= AIDCR_AIINT;
	DSPI_WRITE_UINT16(0x0a, temp16);

	uint32 temp32;

	temp16 = DSPI_READ_UINT16(0x36);
	temp32 = DSPI_READ_UINT32(0x30);
	temp32 += (temp16 & 0x7fff) * 32;
	DSPI_WRITE_UINT32(0x30, temp32);
	temp16 = 0x8000;
    DSPI_WRITE_UINT16(0x36, temp16);
}

void dspi_write_register8(uint32 addr, uint32 data)
{
#pragma warning(push)
#pragma warning(disable:4065)
	switch(addr & 0xff)
	{
	default:
		DSPI_WRITE_UINT8(addr & 0xff, data);
		syslog_error(DSPI,"W8 %04x %08x\n", addr, data);
		break;
	}
#pragma warning(pop)
}

void dspi_write_register16(uint32 addr, uint32 data)
{
	uint16 temp16;

	syslog(DSPI,"W16 %04x %08x\n", addr, data);

	if ((addr & 0xff) == 0x0a)
	{
		temp16 = DSPI_READ_UINT16(addr & 0xff);
		temp16 &= 0xa8;
	    if(data & AIDCR_DSPINT) temp16 &= ~AIDCR_DSPINT;
		if(data & AIDCR_ARINT) temp16 &= ~AIDCR_ARINT;
		if(data & AIDCR_AIINT) temp16 &= ~AIDCR_AIINT;
		data &= ~(AIDCR_DSPINT | AIDCR_ARINT | AIDCR_AIINT);
		data |= temp16;
	}

	DSPI_WRITE_UINT16(addr & 0xff, data);

	switch(addr & 0xff)
	{
	case 0x00:
#if WITH_REAL_DSP
		gdsp_mbox_write_h(GDSP_MBOX_CPU, data & 0x7fff);
#else
		if ((data & 0xffff) == 0x0000)
		{
			//printf("Setting IRQ!!!\n");
			//dspi_raise_interrupt();
			if (dspi_halted)
			{
				dspi_halted = 0;
				//dspi_mb_queue_add(0x8071feed);
				dspi_mb_queue_ptr_write = 0;
				dspi_mb_queue_ptr_read = 0;
				dspi_mb_queue_flush();
				dspi_mb_queue_add(0x0);
				dspi_mb_queue_add(0x80544348);
			}
		}
		dspi_mail_to_dsp = data << 16;
#endif
		//fprintf(stderr, ".");
		break;

	case DSPI_REG_MB_CPU_L:
#if WITH_REAL_DSP
		gdsp_mbox_write_l(GDSP_MBOX_CPU, data);
#else
		dspi_mail_to_dsp = (dspi_mail_to_dsp & 0xffff0000) | data;
		dspi_parse_mail(dspi_mail_to_dsp);
#endif
		break;

	case DSPI_REG_AI_DSP_CSR:
#if WITH_REAL_DSP
		gdsp_write_cr(data);
		syslog(DSPI,"CSR %04x\n", data);
		//fprintf(stderr, "WR CSR: %04x\n", data);
		data &= ~0x01;
		DSPI_WRITE_UINT16(addr & 0xff, data);
#else
		temp16 = DSPI_READ_UINT16(addr & 0xff);
		
		syslog(DSPI,"CSR %04x\n", temp16);
		
		if(temp16 & AIDCR_RES)
		{
			syslog(DSPI,"RESET\n");
			// reset dsp
			temp16 &= ~0x01;
			// ready for feed
			dspi_feed = 1;
			dspi_cmd_parser_reset();
			dspi_mb_queue_flush();
			dspi_mb_queue_add(0x8071feed);
		}
		dspi_halted = temp16 & AIDCR_HALT;
		DSPI_WRITE_UINT16(addr & 0xff, temp16);
#endif
		break;
	case DSPI_REG_AR_DMA_CNT_H:
		if (dspi_cnt_valid & 0x2)
		{
			syslog(ARAM,"WARNING: DSP: higher half of AR_DMA_CNT has been already set\n");
		}
		dspi_cnt_valid |= 0x2;
		dspi_aram_dma();
		break;

	case DSPI_REG_AR_DMA_CNT_L:
		if (dspi_cnt_valid & 0x1)
		{
			syslog(ARAM,"WARNING: DSP: lower half of AR_DMA_CNT has been already set\n");
		}
		dspi_cnt_valid |= 0x1;
		dspi_aram_dma();
		break;

	case 0x12:
		// probably InitOf DSP or other shit
		if (data == 0x43)
		{
			// halt DSP
			temp16 = 0x0004;
			DSPI_WRITE_UINT16(0x0a, temp16);
//			DSPI_WRITE_UINT32(0x04, 0x80544348);
		}
		if (data == 0x63)
		{
//			dspi_mb_queue_add(0x8071feed);
		}
		if (data == 0x64)
		{
//			dspi_mb_queue_add(0x8071feed);
		}
		break;

	case DSPI_REG_AI_DMA_CR:
		if (data & 0x8000)
		{
//			static FILE *wavout = NULL;
			uint32 waddr;
			uint32 wsize;
			syslog(DSPI,"Play Sample: addr: %08x len: %06x\n", DSPI_READ_UINT32(DSPI_REG_AI_DMA_ADDR), (data & 0x7fff) * 32);
			waddr = DSPI_READ_UINT32(DSPI_REG_AI_DMA_ADDR);
			waddr &= 0x3fffffff;
			wsize = (data & 0x7fff) * 32;
//			if (wavout == NULL)
//				wavout = fopen("wavout.raw", "wb");
//			fwrite(gMemory + waddr, 1, wsize, wavout);

			pi_schedule_irq(PI_IRQ_DSPI_AI, 300, dspi_ai_irq_callback);
		}
		else
		{
			syslog(DSPI,"Stop Sample\n");
		}
		break;
	default:
		break;
	}
}

void dspi_write_register32(uint32 addr, uint32 data)
{
	switch(addr & 0xff)
	{
	case DSPI_REG_AR_DMA_MMADDR:
	case DSPI_REG_AR_DMA_ARADDR:
		DSPI_WRITE_UINT32(addr & 0xff, data);
		break;
	case DSPI_REG_AR_DMA_CNT:
		DSPI_WRITE_UINT32(addr & 0xff, data);
		dspi_cnt_valid |= 0x3;
		dspi_aram_dma();
		break;
	default:
		DSPI_WRITE_UINT8(addr & 0xff, data);
		syslog_error(DSPI,"W8 %04x %08x\n", addr, data);
		break;
	}
}

uint8 dspi_read_register8(uint32 addr)
{
	uint32 data;

#pragma warning(push)
#pragma warning(disable:4065)
	switch(addr & 0xff)
	{
	default:
		data = DSPI_READ_UINT32(addr & 0xff);
		syslog_error(DSPI,"R8 %04x %08x\n", addr, data);
		break;
	}
#pragma warning(pop)

	return data;
}

uint16 dspi_read_register16(uint32 addr)
{
	uint32 data;
	data = DSPI_READ_UINT16(addr & 0xff);

	switch(addr & 0xff)
	{
	case 0x00:
#if WITH_REAL_DSP
		data = gdsp_mbox_read_h(GDSP_MBOX_CPU);
#endif
		break;
	case 0x04:
#if WITH_REAL_DSP
		data = gdsp_mbox_read_h(GDSP_MBOX_DSP);
#else
		data = dspi_mb_queue_get();
		syslog(DSPI,"%08x\n", data);
		DSPI_WRITE_UINT32(DSPI_REG_MB_DSP, data);
		data >>= 16;
#endif
		//fprintf(stderr, "MailH: %04x\n", data);

		break;
	case 0x06:
#if WITH_REAL_DSP
		data = gdsp_mbox_read_l(GDSP_MBOX_DSP);
#else
		dspi_mb_queue_advance();
#endif
		/*fprintf(stderr, "MailL: %04x\n", data);
		{
			static xcount = 0;
		if (data == 0x0002)
			xcount++;
		if (xcount == 20)
			exit(0);
		}
		*/
		break;
	case DSPI_REG_AI_DSP_CSR:
#if WITH_REAL_DSP
		data = (data & ~0x805) | gdsp_read_cr();
#endif
		break;
	case 0x16:
		data = 0x1;
		break;
	default:
		break;
	}
	return data;
}

uint32 dspi_read_register32(uint32 addr)
{
	uint32 data;

#pragma warning(push)
#pragma warning(disable:4065)
	switch(addr & 0xff)
	{
	default:
		data = DSPI_READ_UINT32(addr & 0xff);
		syslog_error(DSPI,"R32 %04x %08x\n", addr, data);
		break;
	}
#pragma warning(pop)

	return data;
}

bool dspi_check_interrupt(void)
{
	uint16 csr, mask;
	csr = DSPI_READ_UINT16(DSPI_REG_AI_DSP_CSR);

	mask = (csr >> 1) & 0xa8;
	if (csr & mask)
		return true;
	return false;
}