/*====================================================================

filename:     elf_loader.cpp
project:      GCemu
created:      2004-6-18
mail:		  duddie@walla.com

Copyright (c) 2005 Duddie & Tratax

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

====================================================================*/
#include <stdlib.h>
#include <string.h>
#include "elf.h"
#include "elf_loader.h"
#include "debug/syslog.h"

#define RAM_MASK	0x01ffffff

elf_file * elf_create(unsigned char *mem)
{
	elf_file *elf = (elf_file *)malloc(sizeof(elf_file));
	elf->symbols = NULL;
	elf->file_in = NULL;
	elf->memory = mem;
	elf->debug = false;
	return elf;
}

void elf_enabledebug(elf_file *elf, bool debug)
{
	elf->debug = debug;
}

bool elf_open(elf_file *elf, char *name)
{
	if (elf->file_in = fopen(name,"rb"))
		return true;
	return false;
}

void elf_close(elf_file *elf)
{
	if (elf->file_in) fclose(elf->file_in);
	elf->file_in = NULL;
}

void elf_free(elf_file *elf)
{
	if (elf->section_header) free(elf->section_header);
	if (elf->program_header) free(elf->program_header);
	if (elf->sections_names) free(elf->sections_names);
	free(elf);
}

int elf_readfile(elf_file *elf, unsigned char *ptr, int file_ptr, int size)
{
	fseek(elf->file_in, file_ptr, SEEK_SET);
	return (int)fread(ptr, 1, size, elf->file_in);
}

void elf_swapendian_h(unsigned short *variable)
{
	*variable = ((*variable >> 8) & 0x00ff) | ((*variable << 8) & 0xff00);
}

void elf_swapendian_w(unsigned long *variable)
{
	*variable = ((*variable >> 24) & 0x000000ff) | ((*variable >> 8) & 0x0000ff00) | ((*variable << 8) & 0x00ff0000) | ((*variable << 24) & 0xff000000);
}

uint32 elf_top_used;

int elf_loadheaders(elf_file *elf)
{
	elf_readfile(elf, (unsigned char *)&elf->header, 0, sizeof(Elf32_Ehdr));

	// if BigEndian switch endianess of header
	if (elf->header.e_ident[5] == 2)
	{
		elf_swapendian_h(&elf->header.e_type);
		elf_swapendian_h(&elf->header.e_machine);
		elf_swapendian_w(&elf->header.e_version);
		elf_swapendian_w(&elf->header.e_entry);
		elf_swapendian_w(&elf->header.e_phoff);
		elf_swapendian_w(&elf->header.e_shoff);
		elf_swapendian_w(&elf->header.e_flags);
		elf_swapendian_h(&elf->header.e_ehsize);
		elf_swapendian_h(&elf->header.e_phentsize);
		elf_swapendian_h(&elf->header.e_phnum);
		elf_swapendian_h(&elf->header.e_shentsize);
		elf_swapendian_h(&elf->header.e_shnum);
		elf_swapendian_h(&elf->header.e_shstrndx);
	}

	if (elf->header.e_shentsize != sizeof(Elf32_Shdr) )
		printf("size of section elf->header not standard expected %d is %d\n", sizeof(Elf32_Shdr), elf->header.e_shentsize);

	if((elf->header.e_shnum * sizeof(Elf32_Shdr)) != 0)
		elf->section_header = (Elf32_Shdr *) malloc(elf->header.e_shnum * sizeof(Elf32_Shdr));
	else
		elf->section_header = NULL;

	if((elf->header.e_phnum * elf->header.e_phentsize) != 0)
		elf->program_header = (Elf32_Phdr *) malloc(elf->header.e_phnum * elf->header.e_phentsize);
	else
		elf->program_header = NULL;

	if (elf->debug)
	{
		printf("type:      ");
		switch(elf->header.e_type)
		{
		default:
			printf("unknown %x\n",elf->header.e_type);
			break;
		case 0x0:
			printf("no file type\n");
			break;
		case 0x1:
			printf("relocatable\n");
			break;
		case 0x2:
			printf("executable\n");
			break;
		}
		printf("machine:   ");
		switch(elf->header.e_machine)
		{
		default:
			printf("unknown: %08x\n", elf->header.e_machine);
			break;
		case 0x14:
			printf("gekko\n");
			break;
		case 0x8:
			printf("mips_rs3000\n");
			break;
		}
		printf("version:   %d\n",elf->header.e_version);
		printf("entry:     %08x\n",elf->header.e_entry);
		printf("flags:     %08x\n",elf->header.e_flags);
		printf("eh size:   %08x\n",elf->header.e_ehsize);
		printf("ph off:    %08x\n",elf->header.e_phoff);
		printf("ph entsiz: %08x\n",elf->header.e_phentsize);
		printf("ph num:    %08x\n", elf->header.e_phnum);
		printf("sh off:    %08x\n", elf->header.e_shoff);
		printf("sh entsiz: %08x\n", elf->header.e_shentsize);
		printf("sh num:    %08x\n", elf->header.e_shnum);
		printf("sh strndx: %08x\n", elf->header.e_shstrndx);
		
		printf("\n");
	}
	elf_loadprogramheaders(elf);
	elf_loadsectionheaders(elf);
	return true;
}


int elf_loadprogramheaders(elf_file *elf)
{
	int i;

	elf_top_used = 0;

	// is this critical, or warning?
	if ( elf->header.e_phentsize != sizeof(Elf32_Phdr) )
		printf("size of program header not standard");

	
	if (elf->header.e_phnum != 0)
	{
		elf_readfile(elf, (unsigned char *)&elf->program_header[0], elf->header.e_phoff, elf->header.e_phentsize * elf->header.e_phnum);

		// if BigEndian switch endianess of header
		if (elf->header.e_ident[5] == 2)
		{
			for(i = 0 ; i < elf->header.e_phnum ; i++)
			{
				elf_swapendian_w(&elf->program_header[i].p_type);
				elf_swapendian_w(&elf->program_header[i].p_offset);
				elf_swapendian_w(&elf->program_header[i].p_vaddr);
				elf_swapendian_w(&elf->program_header[i].p_paddr);
				elf_swapendian_w(&elf->program_header[i].p_filesz);
				elf_swapendian_w(&elf->program_header[i].p_memsz);
				elf_swapendian_w(&elf->program_header[i].p_flags);
				elf_swapendian_w(&elf->program_header[i].p_align);
			}
		}
	
		if (elf->debug)
		{
			for(i = 0 ; i < elf->header.e_phnum ; i++)
			{
				printf("Elf32 Program Header");
		
				printf("type:      ");
				switch(elf->program_header[i].p_type)
				{
				default:
					printf("unknown %x\n",  (int)elf->program_header[i].p_type);
					break;
				case 0x1:
					printf("load\n");
					printf("from file: %08x size: %08x to: %08x\n", elf->program_header[i].p_offset, elf->program_header[i].p_filesz, elf->program_header[i].p_vaddr & RAM_MASK);
					elf_readfile(elf, (unsigned char*)(elf->memory + (elf->program_header[i].p_vaddr & RAM_MASK)), elf->program_header[i].p_offset,  elf->program_header[i].p_filesz);
					uint32 temp_top = elf->program_header[i].p_vaddr + elf->program_header[i].p_memsz;
					if (elf_top_used < temp_top)
						elf_top_used = temp_top;
					if (elf->debug)
					{
						printf("* LOADED *\n");
					}
					break;
				}
				printf("offset:    %08x\n", (int)elf->program_header[i].p_offset);
				printf("vaddr:     %08x\n", (int)elf->program_header[i].p_vaddr);
				printf("paddr:     %08x\n", elf->program_header[i].p_paddr);
				printf("file size: %08x\n", elf->program_header[i].p_filesz);
				printf("mem size:  %08x\n", elf->program_header[i].p_memsz);
				printf("flags:     %08x\n", elf->program_header[i].p_flags);
				printf("palign:    %08x\n", elf->program_header[i].p_align);
				printf("\n");
			}
		}
	}
	return true;
}


int elf_loadsectionheaders(elf_file * elf)
{
	int i_st = -1;
	int i_dt = -1;
	
	int i,j;
	
	if (elf->header.e_shnum != 0)
	{
		elf_readfile(elf, (unsigned char *)&elf->section_header[0], elf->header.e_shoff, elf->header.e_shentsize * elf->header.e_shnum);
		
		// if BigEndian switch endianess of header
		if (elf->header.e_ident[5] == 2)
		{
			for(i = 0 ; i < elf->header.e_shnum ; i++)
			{
				elf_swapendian_w(&elf->section_header[i].sh_name);
				elf_swapendian_w(&elf->section_header[i].sh_type);
				elf_swapendian_w(&elf->section_header[i].sh_flags);
				elf_swapendian_w(&elf->section_header[i].sh_addr);
				elf_swapendian_w(&elf->section_header[i].sh_offset);
				elf_swapendian_w(&elf->section_header[i].sh_size);
				elf_swapendian_w(&elf->section_header[i].sh_link);
				elf_swapendian_w(&elf->section_header[i].sh_info);
				elf_swapendian_w(&elf->section_header[i].sh_addralign);
				elf_swapendian_w(&elf->section_header[i].sh_entsize);
			}
		}
		if (elf->header.e_shstrndx < elf->header.e_shnum)
		{
			elf->sections_names = (char *)  malloc(elf->section_header[elf->header.e_shstrndx].sh_size);
			elf_readfile(elf, (unsigned char *)elf->sections_names, elf->section_header[elf->header.e_shstrndx].sh_offset, elf->section_header[elf->header.e_shstrndx].sh_size);
		}
		
		for(i = 0 ; i < elf->header.e_shnum ; i++)
		{
			if (elf->debug)
			{
				printf("Elf32 Section Header [%x] %s\n",  i, &elf->sections_names[elf->section_header[i].sh_name]);
			}
			if (elf->section_header[i].sh_flags & 0x2)
			{
				if (elf->debug)
				{
					printf("from file: %08x size: %08x to: %08x\n", elf->section_header[i].sh_offset, elf->section_header[i].sh_size, elf->section_header[i].sh_addr & RAM_MASK);
				}
				elf_readfile(elf, (elf->memory + (elf->section_header[i].sh_addr & RAM_MASK)), elf->section_header[i].sh_offset,  elf->section_header[i].sh_size);

				if (elf->debug)
				{
					printf("* LOADED *\n");
				}
			}
			if (elf->debug)
			{
				printf("type:      ");
				switch(elf->section_header[i].sh_type)
				{
				default:
					printf("unknown %08x\n", elf->section_header[i].sh_type);
					break;
				case 0x0:
					printf("null\n");
					break;
				case 0x1:
					printf("progbits\n");
					break;
				case 0x2:
					printf("symtab\n");
					break;
				case 0x3:
					printf("strtab\n");
					break;
				case 0x4:
					printf("rela\n");
					break;
				case 0x8:
					printf("no bits\n");
					break;
				case 0x9:
					printf("rel\n");
					break;
				}
				printf("flags:     %08x\n", elf->section_header[i].sh_flags);
				printf("addr:      %08x\n", elf->section_header[i].sh_addr);
				printf("offset:    %08x\n", elf->section_header[i].sh_offset);
				printf("size:      %08x\n", elf->section_header[i].sh_size);
				printf("link:      %08x\n", elf->section_header[i].sh_link);
				printf("info:      %08x\n", elf->section_header[i].sh_info);
				printf("addralign: %08x\n", elf->section_header[i].sh_addralign);
				printf("entsize:   %08x\n", elf->section_header[i].sh_entsize);
				printf("\n");
			}			
			// dump symbol table
			
			if (elf->section_header[i].sh_type == 0x02)
			{
				i_st = i;
				i_dt = elf->section_header[i].sh_link; 
			}
			
		}
		
		if (i_st >= 0 && i_dt >= 0)
		{
			elf->symbol_names = (char *)malloc(elf->section_header[i_dt].sh_size);
			elf->symbols = (Elf32_Sym *)malloc(elf->section_header[i_st].sh_size);
			elf_readfile(elf,(unsigned char *)elf->symbol_names, elf->section_header[i_dt].sh_offset, elf->section_header[i_dt].sh_size);
			
			Elf32_Sym *eS = (Elf32_Sym *)malloc(elf->section_header[i_st].sh_size);
			
			elf_readfile(elf, (unsigned char *)eS, elf->section_header[i_st].sh_offset, elf->section_header[i_st].sh_size);
			if (elf->header.e_ident[5] == 2)
			{
				for(i = 0 ; i < (int)(elf->section_header[i_st].sh_size / sizeof(Elf32_Sym)) ; i++)
				{
					elf_swapendian_w(&eS[i].st_name);
					elf_swapendian_w(&eS[i].st_value);
					elf_swapendian_w(&eS[i].st_size);
					elf_swapendian_h(&eS[i].st_shndx);
				}
			}
			// find min and max
			int k;
			int count = 0;
			for (k = 1 ; k < (int)(elf->section_header[i_st].sh_size / sizeof(Elf32_Sym)) ; k ++)
			{
				//if (eS[k].st_value != 0 && ELF32_ST_TYPE(eS[k].st_info) != 3)
				if (eS[k].st_value != 0)
				{
					if ( NULL == strstr(&elf->symbol_names[eS[k].st_name], "compiled") )
					{
						memcpy(&elf->symbols[count], &eS[k] , sizeof(Elf32_Sym)) ;
						count++;
					}
				}
			}
			elf->count = count;

			// so marvelous bubble sort on magic symbol table
			for (j = 0 ; j < count ; j++)
			{
				for (i = 0 ; i < (count-1) ; i++)
				{
					if (elf->symbols[i].st_value > elf->symbols[i+1].st_value )
					{
						memcpy(&eS[0], &elf->symbols[i], sizeof(Elf32_Sym)) ;
						memcpy(&elf->symbols[i], &elf->symbols[i+1] , sizeof(Elf32_Sym)) ;
						memcpy(&elf->symbols[i+1], &eS[0] , sizeof(Elf32_Sym)) ;
					}
				}

			}

			if (elf->debug)
			{
				for( int j = 0; j < elf->count ; j++ )
				{
					printf("%08x %1d %1d %02d %s\n",  elf->symbols[j].st_value, ELF32_ST_BIND(elf->symbols[j].st_info), ELF32_ST_TYPE(elf->symbols[j].st_info), elf->symbols[j].st_shndx, &elf->symbol_names[elf->symbols[j].st_name]);
				}
			}
			free(eS);
		}
	}
	return true;	
}



char * elf_getfunctionname(elf_file * elf, uint32 addr)
{
	if (!elf) return NULL;
	if (elf->symbols == NULL) return NULL;

	for( int j = 0; j < elf->count ; j++ )
	{
//		if (elf->symbols[j].st_info & 0xf )
		{
			uint32 temp = elf->symbols[j].st_value;
			if( (addr & RAM_MASK) == (temp & RAM_MASK))
			{
				return &elf->symbol_names[elf->symbols[j].st_name];
			}
/*			else if( (addr & 0x00ffffff) > (elf->symbols[j].st_value & 0x00ffffff))
			{
				return NULL;
			}
*/

		}
	}
	return NULL;
}

uint32 elf_getfunctionaddr(elf_file * elf, char *name)
{
	if (!elf) return NULL;
	if (elf->symbols == NULL) return NULL;

	for( int j = 0; j < elf->count ; j++ )
	{
		if (strcmp(&elf->symbol_names[elf->symbols[j].st_name], name) == 0)
			return elf->symbols[j].st_value;
	}
	return 0xffffffff;
}

uint32 elf_getentrypoint(elf_file * elf)
{
	printf("Entry Point: %08x\n", elf->header.e_entry);
	return elf->header.e_entry;
}

bool elf_hassymbols(elf_file * elf)
{
	if (!elf) return false;
	if (elf->symbols == NULL) return false;
	return true;
}

uint32 elf_get_top(elf_file * elf)
{
	return elf_top_used;
}