/***************************************************************************

	driver.c

	NEOGEO CDhCoRA (based on M.A.M.E.'s neogeo driver)

***************************************************************************/

/***************************************************************************
    M.A.M.E. Neo Geo driver presented to you by the Shin Emu Keikaku team.

    The following people have all spent probably far too much time on this:

    AVDB
    Bryan McPhail
    Fuzz
    Ernesto Corvi
    Andrew Prime

    Neogeo Motherboard (info - courtesy of Guru)
    --------------------------------------------

    PCB Layout (single slot, older version)

    NEO-MVH MV1
    |---------------------------------------------------------------------|
    |       4558                                                          |
    |                                          HC04  HC32                 |
    |                      SP-S2.SP1  NEO-E0   000-L0.L0   LS244  AS04    |
    |             YM2610                                                  |
    | 4558                                                                |
    |       4558                        5814  HC259   SFIX.SFIX           |
    |                                                             NEO-I0  |
    | HA13001 YM3016                    5814                              |
    --|                                                                   |
      |     4558                                                          |
    --|                                                 SM1.SM1   LS32    |
    |                                                                     |
    |                           LSPC-A0         PRO-C0            LS244   |
    |                                                                     |
    |J              68000                                                 |
    |A                                                                    |
    |M                                                                    |
    |M                                                      NEO-ZMC2      |
    |A                                                                    |
    |   LS273  NEO-G0                          58256  58256     Z80A      |
    |                           58256  58256   58256  58256     6116      |
    |   LS273 5864                                                        |
    --| LS05  5864  PRO-B0                                                |
      |                                                                   |
    --|             LS06   HC32           D4990A    NEO-F0   24.000MHz    |
    |                      DSW1    BATT3.6V 32.768kHz       NEO-D0        |
    |                                           2003  2003                |
    |---------------------------------------------------------------------|

    Notes:
          68k clock: 12.000MHz
          Z80 clock: 4.000MHz
       YM2610 clock: 8.000MHz
              VSync: 60Hz
              HSync: 15.21kHz

             Custom SNK chips
             ----------------
             NEO-G0: QFP64
             NEO-E0: QFP64
             PRO-B0: QFP136
            LSPC-A0: QFP160
             PRO-C0: QFP136
             NEO-F0: QFP64
             NEO-D0: QFP64
           NEO-ZMC2: QFP80
             NEO-I0: QFP64

             ROMs        Type
             ----------------------------
             SP-S2.SP1   TC531024 (DIP40)
             000-L0.L0   TC531000 (DIP28)
             SFIX.SFIX   D27C1000 (DIP32)
             SM1.SM1     MB832001 (DIP32)

    ------------------------------------------------------

    GRAPHICAL ISSUES :

    - Effects created using the Raster Interrupt are probably not 100% correct,
      e.g.:
      - full screen zoom in trally and tpgolf is broken again :-( I think this was
        caused by the fix for kof94 japan stage.
      - Tests on the hardware show that there are 264 raster lines; however, there
        are one or two line alignemnt issues with some games, SCANLINE_ADJUST is
        a kludge to get the alignment almost right in most cases.
        Some good places to test raster effects handling and alignment:
        - aodk 100 mega shock logo
        - viewpoin Sammy logo
        - zedblade parallax scrolling
        - ridhero road
        - turfmast Japan course hole 4 (the one with the waterfall)
        - fatfury3, 7th stage (Port Town). Raster effects are used for the background.
      - spinmast uses IRQ2 with no noticeable effect (it seems to be always near
        the bottom of the screen).
      - garoup enables IRQ2 on Terry's stage, but with no noticeable effect. Note
        that it is NOT enabled in 2 players mode, only vs cpu.
      - strhoop enables IRQ2 on every scanline during attract mode, with no
        noticeable effect.
      - Money Idol Exchanger runs slow during the "vs. Computer" mode. Solo mode
        works fine.

    - Full screen zoom has some glitches in tpgolf.

    - Gururin has bad tiles all over the place (used to work ..)

    - Bad clipping during scrolling at the sides on some games.
        (tpgolf for example)

    AUDIO ISSUES :

    - Sound (Music) was cutting out in ncommand and ncombat due to a bug in the
      original code, which should obviously have no ill effect on the real YM2610 but
      confused the emulated one. This is fixed in the YM2610 emulator.

    - Some rather bad sounding parts in a couple of Games
        (shocktro End of Intro)

    - In mahretsu music should stop when you begin play (correct after a continue) *untested*

    GAMEPLAY ISSUES / LOCKUPS :

    - Viewpoint resets halfway through level 1. This is a bug in the asm 68k core.

    - magdrop2 behaves strangely when P2 wins a 2 Player game (reports both as losing)

    - popbounc without a patch this locks up when sound is disabled, also for this game 'paddle'
      conroller can be selected in the setup menus, but Mame doesn't support this.

    - ssideki2 locks up sometimes during play *not tested recently, certainly used to*

    - 2020bb apparently resets when the batter gets hit by the pitcher *not tested*

    - some games apparently crash / reset when you finish them before you get the ending *untested*

    - fatfury3 locks up when you complete the game.

    NON-ISSUES / FIXED ISSUES :

    - Auto Animation Speed is not quite right in Many Games
        (mslug waterfalls, some bg's in samsho4, blazstar lev 2 etc.)

    - shocktro locking up at the bosses, this was fixed a long long time ago, it was due to bugs
      in the 68k Core.

    - sound, graphic, the odd game crash & any other strange happenings in kof99p and garoup are
      probably because these machines are prototypes, the games are therefore not finished.  There
      are 'patched' versions of these romsets available in some locations, however these will not
      be supported.

    OTHER MINOR THINGS :

    - 2020bb version display, the program roms contains two version numbers, the one which always
      get displayed when running in Mame is that which would be displayed on a console.
      This depends on location 0x46 of nvram. That location is the BIOS "Demo sound" bit ('00' for
      'set up in each game' and '01' for 'without'). If you set 0x46 to '01' ALL Demosound
      (Neo Splash screen and in game attract mode) is off, and version number is 1.32X. If you set
      0x46 to '00' and set 0x229 (Demosound bit for the game itself. '00' for 'ON' and '01' for
      'OFF') to '01' The Neo splashscreen has sound but the ingame attract mode does not and
      version is set to 1.32X. So it would seem that 1.32X gets displayed when demosund is off
      and 1.02C when Demosound is on.

    NOTES ABOUT UNSUPPORTED GAMES :

    - Diggerman (???, 2000) - Not A Real Arcade Game .. Will Not Be Supported.

    VIEWPOINT CRASH

    "Viewpoint resets under the ASM core due to nested IRQ1."


=============================================================================

Points to note, known and proven information deleted from this map:

    0x3000001       Dipswitches
                bit 0 : Selftest
                bit 1 : Unknown (Unused ?) \ something to do with
                bit 2 : Unknown (Unused ?) / auto repeating keys ?
                bit 3 : \
                bit 4 :  | communication setting ?
                bit 5 : /
                bit 6 : free play
                bit 7 : stop mode ?

    0x320001        bit 0 : COIN 1
                bit 1 : COIN 2
                bit 2 : SERVICE
                bit 3 : UNKNOWN
                bit 4 : UNKNOWN
                bit 5 : UNKNOWN
                bit 6 : 4990 test pulse bit.
                bit 7 : 4990 data bit.

    0x380051        4990 control write register
                bit 0: C0
                bit 1: C1
                bit 2: C2
                bit 3-7: unused.

                0x02 = shift.
                0x00 = register hold.
                0x04 = ????.
                0x03 = time read (reset register).

    0x3c000c        IRQ acknowledge

    0x380011        Backup bank select

    0x3a0001        Enable display.
    0x3a0011        Disable display

    0x3a0003        Swap in Bios (0x80 bytes vector table of BIOS)
    0x3a0013        Swap in Rom  (0x80 bytes vector table of ROM bank)

    0x3a000d        lock backup ram
    0x3a001d        unlock backup ram

    0x3a000b        set game vector table (?)  mirror ?
    0x3a001b        set bios vector table (?)  mirror ?

    0x3a000c        Unknown (ghost pilots)
    0x31001c        Unknown (ghost pilots)

    IO word read

    0x3c0002        return vidram word (pointed to by 0x3c0000)
    0x3c0006        ?????.
    0x3c0008        shadow adress for 0x3c0000 (not confirmed).
    0x3c000a        shadow adress for 0x3c0002 (confirmed, see
                               Puzzle de Pon).
    IO word write

    0x3c0006        Unknown, set vblank counter (?)

    0x3c0008        shadow address for 0x3c0000 (not confirmed)
    0x3c000a        shadow address for 0x3c0002 (not confirmed)

    The Neo Geo contains an NEC 4990 Serial I/O calendar & clock.
    accesed through 0x320001, 0x380050, 0x280050 (shadow adress).
    A schematic for this device can be found on the NEC webpages.

******************************************************************************/

/* Changes 25/03/2003 DH

-- using elsemi's information all the bios patches have been removed.
this required
bios / game vector selection to be emulated
calender emulation fixes
removal of hacks to change region / get info memory card manager
 *note1 memory card manager is reportedly only accessable on the
  neogeo console so should be a job for MESS, not MAME
 *note2 to change region you must now recompile with the relevant
  part of the code uncommented so the correct bios roms are loaded
 *note3 startup now takes longer, this is because it is performing
  memory checks which were previously skipped
-- added correct Bang Bead set (again based on Elsemi's Information)
-- fixed bios filename (based on info from Guru)

-- are the EURO bios roms infact ASIA bios roms?

*/

/**********************************************************************************************/

#include "neogeocd.h"

/* VBLANK should fire on line 248 */
#define RASTER_COUNTER_START 0x1f0	/* value assumed right after vblank */
#define RASTER_COUNTER_RELOAD 0x0f8	/* value assumed after 0x1ff */
#define RASTER_LINE_RELOAD (0x200 - RASTER_COUNTER_START)

#define SCANLINE_ADJUST 3	/* in theory should be 0, give or take an off-by-one mistake */

/*  flags for irq1control:

    0x07 unused? kof94 sets some random combination of these at the character
         selection screen but only because it does andi.w #$ff2f, $3c0006. It
         clears them immediately after.

    0x08 shocktro2, stops autoanim counter

    Maybe 0x07 writes to the autoanim counter, meaning that in conjunction with
    0x08 one could fine control it. However, if that was the case, writing the
    the IRQ2 control bits would interfere with autoanimation, so I'm probably
    wrong.

    0x10 irq1 enable, tile engine scanline irq that is triggered
    when a certain scanline is reached.

    0x20 when set, the next values written in the irq position register
    sets irq1 to happen N lines after the current one

    0x40 when set, irq position register is automatically loaded at vblank to
    set the irq1 line.

    0x80 when set, every time irq1 is triggered the irq position register is
    automatically loaded to set the next irq1 line.

    0x80 and 0x40 may be set at the same time (Viewpoint does this).
*/

#define IRQ1CTRL_AUTOANIM_STOP		0x08
#define IRQ1CTRL_ENABLE				0x10
#define IRQ1CTRL_LOAD_RELATIVE		0x20
#define IRQ1CTRL_AUTOLOAD_VBLANK	0x40
#define IRQ1CTRL_AUTOLOAD_REPEAT	0x80

/**********************************************************************************************/

static int irq1start = 1000;
static int irq1control;
static int current_rastercounter;
static int current_rasterline;
static int scanline_read;
static UINT32 irq1pos_value;
static int vblank_int;
static int scanline_int;
static int neogeo_raster_enable;
static UINT32 frame_counter = 0;
static int pending_command;
static int result_code;

static int nowait_irqack;

/**********************************************************************************************/

UINT32 neogeo_frame_counter = 0;
UINT32 neogeo_frame_counter_speed = 4;

double video_fps = 15625.0 / 264; // = 59.1856;
#if 1
struct rectangle visible_area = { 8, 311, FIRST_VISIBLE_LINE, LAST_VISIBLE_LINE };
#else
// ƒperƂ̂炢̕\͈
struct rectangle visible_area = { 16, 293, FIRST_VISIBLE_LINE + 8, LAST_VISIBLE_LINE };
#endif


/**********************************************************************************************/

/*------------------------------------------------------

	hCȍ

	  : Ȃ
	߂l: Ȃ

 -----------------------------------------------------*/

void neogeo_driver_reset(void)
{
	frame_counter              = 0;
	neogeo_frame_counter       = 0;
	neogeo_frame_counter_speed = 4;

	irq1start                  = 1000;
	irq1control                = 0;

	current_rastercounter      = 0;
	current_rasterline         = 0;
	scanline_read              = 0;

	if (game_index >= 0)
		neogeo_raster_enable   = games[game_index].driver > 1;
	else
		neogeo_raster_enable   = 0;

	irq1pos_value              = 0;

	vblank_int                 = 0;
	scanline_int               = 0;

	pending_command            = 0;
	result_code                = 0;

	if (GAME_NAME("lastbld2"))
		nowait_irqack = 1;
	else
		nowait_irqack = 0;
}


/****************************************************************************
	荞
 ***************************************************************************/

/*------------------------------------------------------

	荞ݏԂ̍XV

	  : Ȃ
	߂l: Ȃ

 -----------------------------------------------------*/

static void update_interrupts(void)
{
	int level = 0;

	/* determine which interrupt is active */
	if (vblank_int) level = 2;
	if (scanline_int) level = 1;

	/* either set or clear the appropriate lines */
	if (level)
	{
		cpu_set_irq_line(0, level, ASSERT_LINE);

		if (nowait_irqack)
		{
			if (level == 2) vblank_int = 0;
			if (level == 1) scanline_int = 0;
		}
	}
	else
		cpu_set_irq_line(0, 7, CLEAR_LINE);
}


/*------------------------------------------------------

	NEOGEO荞ݏ(VBLANK荞)

	  : Ȃ
	߂l: Ȃ

 -----------------------------------------------------*/

void neogeo_interrupt(void)
{
	int line = RASTER_LINES - cpu_getiloops();

	current_rasterline = line;

	{
		int l = line;

		if (l == RASTER_LINES) l = 0;	/* vblank */
		if (l < RASTER_LINE_RELOAD)
			current_rastercounter = RASTER_COUNTER_START + l;
		else
			current_rastercounter = RASTER_COUNTER_RELOAD + l - RASTER_LINE_RELOAD;
	}

	if (line == RASTER_LINES)	/* vblank */
	{
		current_rasterline = 0;

		/* Add a timer tick to the pd4990a */
		pd4990a_addretrace();

		if (!(irq1control & IRQ1CTRL_AUTOANIM_STOP))
		{
			/* Animation counter */
			if (frame_counter++ > neogeo_frame_counter_speed)	/* fixed animation speed */
			{
				frame_counter = 0;
				neogeo_frame_counter++;
			}
		}

		if (irq1control & IRQ1CTRL_ENABLE)
			ui_popup("IRQ1 enabled, need raster driver.");

		/* return a standard vblank interrupt */
		vblank_int = 1; /* vertical blank */
	}

	update_interrupts();
}


/*------------------------------------------------------

	NEOGEO荞ݏ(XLC荞)

	  : int busy  busytO
	߂l: Ȃ

	busy=1̏ꍇirq control̒lɊ֌WȂʍXV

 -----------------------------------------------------*/

static void raster_interrupt(int busy)
{
	int line = RASTER_LINES - cpu_getiloops();
	int do_refresh = 0;

	current_rasterline = line;

	{
		int l = line;

		if (l == RASTER_LINES) l = 0;	/* vblank */
		if (l < RASTER_LINE_RELOAD)
			current_rastercounter = RASTER_COUNTER_START + l;
		else
			current_rastercounter = RASTER_COUNTER_RELOAD + l - RASTER_LINE_RELOAD;
	}

	if (busy)
	{
		if (neogeo_raster_enable && scanline_read)
		{
			do_refresh = 1;
//logerror("partial refresh at raster line %d (raster counter %03x)\n", line, current_rastercounter);
			scanline_read = 0;
		}
	}

	if (irq1control & IRQ1CTRL_ENABLE)
	{
		if (line == irq1start)
		{
//logerror("trigger IRQ1 at raster line %d (raster counter %d)\n", line, current_rastercounter);
			if (!busy)
			{
				if (neogeo_raster_enable)
					do_refresh = 1;
			}

			if (irq1control & IRQ1CTRL_AUTOLOAD_REPEAT)
				irq1start += (irq1pos_value + 3) / 0x180;	/* ridhero gives 0x17d */

			scanline_int = 1;
		}
	}

	if (line == RASTER_LINES)	/* vblank */
	{
		current_rasterline = 0;

		if (osd_is_key_pressed_memory(KEYCODE_F1))
		{
			neogeo_raster_enable ^= 1;
			ui_popup("raster effects %s", neogeo_raster_enable ? "enabled" : "disabled");
		}

		if (irq1control & IRQ1CTRL_AUTOLOAD_VBLANK)
			irq1start = (irq1pos_value + 3) / 0x180;	/* ridhero gives 0x17d */
		else
			irq1start = 1000;

		/* Add a timer tick to the pd4990a */
		pd4990a_addretrace();

		/* Animation counter */
		if (!(irq1control & IRQ1CTRL_AUTOANIM_STOP))
		{
			if (frame_counter++ > neogeo_frame_counter_speed)	/* fixed animation speed */
			{
				frame_counter = 0;
				neogeo_frame_counter++;
			}
		}

//logerror("trigger IRQ2 (VBLANK)\n");
		vblank_int = 1;
	}

	if (do_refresh)
	{
		if (line > RASTER_LINE_RELOAD)	/* avoid unnecessary updates after start of vblank */
			neogeo_partial_screenrefresh((current_rastercounter - 256) - 1 + SCANLINE_ADJUST);
	}

	update_interrupts();
}


void neogeo_raster_interrupt(void)
{
	raster_interrupt(0);
}

void neogeo_raster_interrupt_busy(void)
{
	raster_interrupt(1);
}

/****************************************************************************
	ANZX
 ***************************************************************************/

/*------------------------------------------------------

	pbgBANKύX    (0x3a000e - 0x3a000f)
						(0x3a001e - 0x3a001f)

	  : offs_t offset  ItZbg(gp)
	        UINT16 data    f[^(gp)
	߂l: Ȃ

 -----------------------------------------------------*/

static WRITE16_HANDLER( neogeo_setpalbank_16_w )
{
	neogeo_setpalbank(offset);
}


/*------------------------------------------------------

	pbgRAMǂݍ (0x400000 - 0x7fffff)

	  : offs_t offset  ǂݍރItZbg
	߂l: pbgf[^ (16bit)

 -----------------------------------------------------*/

static READ16_HANDLER( neogeo_paletteram16_r )
{
	offset &= 0xfff;

	return neogeo_paletteram16[offset];
}


/*------------------------------------------------------

	pbgRAM (0x400000 - 0x7fffff)

	  : offs_t offset  ރItZbg
			UINT16 datat   ރpbgf[^
	߂l: Ȃ

 -----------------------------------------------------*/

static WRITE16_HANDLER( neogeo_paletteram16_w )
{
	UINT16 newword;

	offset &= 0xfff;

	newword = neogeo_paletteram16[offset];

	COMBINE_DATA(&newword);

	neogeo_paletteram16[offset] = newword;
	video_palette[offset] = video_color_lut[newword & 0x7fff];
}


/*------------------------------------------------------

	rfIRAMItZbgǂݍ (0x3c0008 - 0x3c0009)

	  : offs_t offset  ItZbg(gp)
	߂l: rfIRAM̃ItZbg

 -----------------------------------------------------*/

static READ16_HANDLER( neogeo_vidram16_offset_r )
{
	return neogeo_vidram16_offset;
}


/*------------------------------------------------------

	rfIRAMItZbg (0x3c0000 - 0x3c0001)

	  : offs_t offset  ItZbg(gp)
	        UINT16 data    rfIRAM̃ItZbg
	߂l: Ȃ

 -----------------------------------------------------*/

static WRITE16_HANDLER( neogeo_vidram16_offset_w )
{
	COMBINE_DATA(&neogeo_vidram16_offset);
}


/*------------------------------------------------------

	rfIRAMf[^ǂݍ (0x3c0000 - 0x3c0001)
	                        (0x3c0002 - 0x3c0003)
	                        (0x3c000a - 0x3c000b)

	  : offs_t offset  ItZbg(gp)
	߂l: rfIRAMǂݍ񂾃f[^

 -----------------------------------------------------*/

static READ16_HANDLER( neogeo_vidram16_data_r )
{
	return neogeo_vidram16[neogeo_vidram16_offset];
}


/*------------------------------------------------------

	rfIRAMf[^ (0x3c0002 - 0x3c0003)

	  : offs_t offset  ItZbg(gp)
	        UINT16 data    ރf[^
	߂l: Ȃ

 -----------------------------------------------------*/

static WRITE16_HANDLER( neogeo_vidram16_data_w )
{
	COMBINE_DATA(&neogeo_vidram16[neogeo_vidram16_offset]);

	neogeo_vidram16_offset = (neogeo_vidram16_offset & 0x8000)	/* gururin fix */
			| ((neogeo_vidram16_offset + neogeo_vidram16_modulo) & 0x7fff);
}


/*------------------------------------------------------

	rfIRAMWǂݍ (0x3c0004 - 0x3c0005)

	  : offs_t offset  ItZbg(gp)
	߂l: rfIRAMItZbg̃W

 -----------------------------------------------------*/

static READ16_HANDLER( neogeo_vidram16_modulo_r )
{
	return neogeo_vidram16_modulo;
}


/*------------------------------------------------------

	rfIRAMW (0x3c0004 - 0x3c0005)

	  : offs_t offset  ItZbg(gp)
	        UINT16 data    rfIRAMW
	߂l: Ȃ

 -----------------------------------------------------*/

static WRITE16_HANDLER( neogeo_vidram16_modulo_w )
{
	COMBINE_DATA(&neogeo_vidram16_modulo);
}


/*---------------------------------------------------------

	rfIRg[f[^ǂݍ (0x3c0006 - 0x3c0007)

	  : offs_t offset  ItZbg(gp)
	߂l: Ȃ

 --------------------------------------------------------*/

static READ16_HANDLER( neogeo_control_16_r )
{
	int res;

	/*
        The format of this very important location is:  AAAA AAAA A??? BCCC

        A is the raster line counter. mosyougi relies solely on this to do the
          raster effects on the title screen; sdodgeb loops waiting for the top
          bit to be 1; zedblade heavily depends on it to work correctly (it
          checks the top bit in the IRQ2 handler).
        B is definitely a PAL/NTSC flag. Evidence:
          1) trally changes the position of the speed indicator depending on
             it (0 = lower 1 = higher).
          2) samsho3 sets a variable to 60 when the bit is 0 and 50 when it's 1.
             This is obviously the video refresh rate in Hz.
          3) samsho3 sets another variable to 256 or 307. This could be the total
             screen height (including vblank), or close to that.
          Some games (e.g. lstbld2, samsho3) do this (or similar):
          bclr    #$0, $3c000e.l
          when the bit is set, so 3c000e (whose function is unknown) has to be
          related
        C is a variable speed counter. In blazstar, it controls the background
          speed in level 2.
    */

	scanline_read = 1;	/* needed for raster_busy optimization */

	res =	((current_rastercounter << 7) & 0xff80) |	/* raster counter */
			(neogeo_frame_counter & 0x0007);			/* frame counter */

//	logerror("PC %06x: neogeo_control_16_r (%04x)\n", m68000_get_reg(M68K_PC), res);

	return res;
}


/*---------------------------------------------------------

	rfIRg[f[^ (0x3c0006 - 0x3c0007)

	  : offs_t offset  ItZbg(gp)
	        UINT16 data    f[^
	߂l: Ȃ

 --------------------------------------------------------*/

static WRITE16_HANDLER( neogeo_control_16_w )
{
//	logerror("PC %06x: neogeo_control_16_w %04x\n", m68000_get_reg(M68K_PC), data);

	/* Auto-Anim Speed Control */
	neogeo_frame_counter_speed = (data >> 8) & 0xff;

	irq1control = data & 0xff;
}


/*------------------------------------------------------

	IRQ1ʒuf[^ (0x3c0008 - 0x3c0009)
	                       (0x3c000a - 0x3c000b)

	  : offs_t offset  ItZbg(gp)
	        UINT16 data    荞݊Jnʒũf[^
	߂l: Ȃ

 -----------------------------------------------------*/

static WRITE16_HANDLER( neogeo_irq1pos_16_w )
{
//	logerror("PC %06x: neo_irq2pos_16_w offset %d %04x\n", m68000_get_reg(M68K_PC), offset, data);

	if (offset)
		irq1pos_value = (irq1pos_value & 0xffff0000) | (UINT32)data;
	else
		irq1pos_value = (irq1pos_value & 0x0000ffff) | ((UINT32)data << 16);

	if (irq1control & IRQ1CTRL_LOAD_RELATIVE)
	{
//      int line = (irq2pos_value + 3) / 0x180; /* ridhero gives 0x17d */
		int line = (irq1pos_value + 0x3b) / 0x180;

		irq1start = current_rasterline + line;

//		logerror("irq1start = %d, current_rasterline = %d, current_rastercounter = %d\n", irq1start, current_rasterline, current_rastercounter);
	}
}


/*------------------------------------------------------

	荞݊ (0x3c000c - 0x3c000d)

	  : offs_t offset  ItZbg(gp)
	        UINT16 data    荞݂̒l
	߂l: Ȃ

   data = 4: IRQ2(VBLANK)
     data = 2: IRQ1(SCANLINE)

 -----------------------------------------------------*/

static WRITE16_HANDLER( neogeo_irqack_16_w )
{
	if (ACCESSING_LSB)
	{
		if (data & 4) vblank_int = 0;
		if (data & 2) scanline_int = 0;
		update_interrupts();
	}
}


/*------------------------------------------------------

	VXeXCb`̏ (0x3a0000 - 0x3a001f)

	  : offs_t offset  ItZbg
	        UINT16 data    f[^
	߂l: Ȃ

 -----------------------------------------------------*/

static WRITE16_HANDLER( neogeo_syscontrol1_16_w )
{
	offset &= 0x7f;

	switch (offset << 1)
	{
	case 0x00: break;
	case 0x10: break;
	case 0x50: pd4990a_control_16_w(0, data, mem_mask); break;
	case 0xd0: pd4990a_control_16_w(0, data, mem_mask); break;
	default:
		logerror("neogeo_syscontrol_16_w(): write (%04x) to unmapped address %06x\n", data, offset);
		break;
	}
}


/*------------------------------------------------------

	VXeXCb`̏ (0x3a0000 - 0x3a001f)

	  : offs_t offset  ItZbg
	        UINT16 data    f[^
	߂l: Ȃ

 -----------------------------------------------------*/

static WRITE16_HANDLER( neogeo_syscontrol2_16_w )
{
	offset &= 0xf;

	switch (offset << 1)
	{
	case 0x00:	break;	// unknown
	case 0x02:	neogeo_select_bios_vectors(0, data, mem_mask); break;
	case 0x04:	break;	// unknown
	case 0x06:	break;	// unknown
	case 0x08:	break;	// unknown
	case 0x0a:	break;	// unknown
	case 0x0c:	break;	// unknown
	case 0x0e:	neogeo_setpalbank_16_w(0, data, mem_mask); break;
	case 0x10:	break;	// unknown
	case 0x12:	neogeo_select_game_vectors(0, data, mem_mask);break;
	case 0x14:	break;	// unknown
	case 0x16:	break;	// unknown
	case 0x18:	break;	// unknown
	case 0x1a:	break;	// unknown
	case 0x1c:	break;	// unknown
	case 0x1e:	neogeo_setpalbank_16_w(1, data, mem_mask); break;
#if 0
	default:
		logerror("neogeo_syscontrol2_16_w(): write (%04x) to unmapped address %06x\n", data, offset);
		break;
#endif
	}
}


/*------------------------------------------------------

	rfIRAMǂݍ (0x3c0000 - 0x3c000f)

	  : offs_t offset  ItZbg
	        UINT16 data    f[^
	߂l: Ȃ

 -----------------------------------------------------*/

static READ16_HANDLER( neogeo_video_16_r )
{
	UINT16 retdata = 0xffff;

	offset &= 0x3;

	switch (offset << 1)
	{
		case 0: retdata = neogeo_vidram16_data_r(0, mem_mask); break;
		case 2: retdata = neogeo_vidram16_data_r(0, mem_mask); break;
		case 4:	retdata = neogeo_vidram16_modulo_r(0, mem_mask); break;
		case 6:	retdata = neogeo_control_16_r(0, mem_mask); break;
	}

	return retdata;
}


/*------------------------------------------------------

	rfIRAM (0x3c0000 - 0x3c000f)

	  : offs_t offset  ItZbg
	        UINT16 data    f[^
	߂l: Ȃ

 -----------------------------------------------------*/

static WRITE16_HANDLER( neogeo_video_16_w )
{
	offset &= 0x7;

	switch (offset << 1)
	{
		case 0x0: neogeo_vidram16_offset_w(0, data, mem_mask); break;
		case 0x2: neogeo_vidram16_data_w(0, data, mem_mask); break;
		case 0x4: neogeo_vidram16_modulo_w(0, data, mem_mask); break;
		case 0x6: neogeo_control_16_w(0, data, mem_mask); break;
		case 0x8: neogeo_irq1pos_16_w(0, data, mem_mask); break;
		case 0xa: neogeo_irq1pos_16_w(1, data, mem_mask); break;
		case 0xc: neogeo_irqack_16_w(0, data, mem_mask); break;
		case 0xe: break; /* Unknown, see control_r */
	}
}


/*------------------------------------------------------

	Z80Agf[^[h (0x320000 - 0x320001)

	  : offs_t offset  ItZbg(gp)
	߂l: f[^

 -----------------------------------------------------*/

/* Calendar, coins + Z80 communication */
static READ16_HANDLER( neogeo_z80_r )
{
	UINT16 res;
	int coinflip = pd4990a_testbit_r(0);
	int databit = pd4990a_databit_r(0);

//  logerror("PC %06x: Read timer\n", m68000_get_reg(M68K_PC));

	res = 0 ^ (coinflip << 6) ^ (databit << 7);

	if (sample_rate)
	{
		res |= result_code << 8;
		if (pending_command) res &= 0x7fff;
	}
	else
		res = 0x0100;

	return res;
}


/*------------------------------------------------------

	Z80Agf[^Cg (0x320000 - 0x320001)

	  : offs_t offset  ItZbg(gp)
	        UINT16 data    f[^
	߂l: Ȃ

 -----------------------------------------------------*/

static WRITE16_HANDLER( neogeo_z80_w )
{
	/* tpgold uses 16-bit writes, this can't be correct */
//  if (ACCESSING_LSB)
//      return;

	soundlatch_w(0, (data >> 8) & 0xff);
	pending_command = 1;
	z80_set_irq_line(IRQ_LINE_NMI, PULSE_LINE);
	cpu_spinuntil_time(TIME_IN_USEC(50));
}


/*------------------------------------------------------

	Rg[[1̓[h (0x300000 - 0x300001)

	  : offs_t offset  ItZbg(gp)
	߂l: |[g̓͏Ԃ̃f[^

 -----------------------------------------------------*/

READ16_HANDLER( neogeo_controler1_16_r )
{
	return read_controler1();
}


/*------------------------------------------------------

	Rg[[2̓[h (0x340000 - 0x340001)

	  : offs_t offset  ItZbg(gp)
	߂l: |[g̓͏Ԃ̃f[^

 -----------------------------------------------------*/

READ16_HANDLER( neogeo_controler2_16_r )
{
	return read_controler2();
}


/*------------------------------------------------------

	X^[g/ZNg[h (0x380000 - 0x380001)

	  : offs_t offset  ItZbg(gp)
	߂l: |[g̓͏Ԃ̃f[^

 -----------------------------------------------------*/

READ16_HANDLER( neogeo_controler3_16_r )
{
	int button;
	UINT16 mask = 0x01, value = 0x0f;

	for (button = NEOGEO_INPUT_START; button <= NEOGEO_INPUT_SELECT; button++, mask <<= 1)
		if (osd_is_key_pressed(key_code[PLAYER1][button]) || osd_is_joy_pressed(joy_code[PLAYER1][button]))
			value &= ~mask;

	for (button = NEOGEO_INPUT_START; button <= NEOGEO_INPUT_SELECT; button++, mask <<= 1)
		if (osd_is_key_pressed(key_code[PLAYER2][button]) || osd_is_joy_pressed(joy_code[PLAYER2][button]))
			value &= ~mask;

	return value << 8;
}


/****************************************************************************
	Z80 |[g[hCg
 ***************************************************************************/

/*------------------------------------------------------

	Z80|[g[h

	  : UINT16 port  |[g
	߂l: f[^

 -----------------------------------------------------*/

UINT8 neogeo_z80_port_r(UINT16 port)
{
	switch (port & 0xff)
	{
	case 0x00:
		pending_command = 0;
		return soundlatch_r(0);

	case 0x04:
		return YM2610_status_port_A_r(0);

	case 0x05:
		return YM2610_read_port_r(0);

	case 0x06:
		return YM2610_status_port_B_r(0);

	case 0x08:
	case 0x09:
	case 0x0a:
	case 0x0b:
		// set bank
		return 0;

	default:
		logerror("CPU #1: read unmapped port %02x\n", port & 0xff);
		break;
	};

	return 0;
}


/*------------------------------------------------------

	Z80|[gCg

	  : UINT16 port  |[g
			UINT8 data
	߂l: Ȃ

 -----------------------------------------------------*/

void neogeo_z80_port_w(UINT16 port, UINT8 data)
{
	switch (port & 0xff)
	{
	case 0x00:	// 0x0cƓ
		break;

	case 0x04:
		YM2610_control_port_A_w(0, data);
		break;

	case 0x05:
		YM2610_data_port_A_w(0, data);
		break;

	case 0x06:
		YM2610_control_port_B_w(0, data);
		break;

	case 0x07:
		YM2610_data_port_B_w(0, data);
		break;

	case 0x08:
		/* NMI enable / acknowledge? (the data written doesn't matter) */
		break;

	case 0x0c:
		result_code = data;
		break;

	case 0x18:
		/* NMI disable? (the data written doesn't matter) */
		break;

//	case 0x80:
//		neogeo_cdda_stop();
//		break;

	case 0xc0:
	case 0xc1:
	case 0xc2:
		// unknown
		break;

	default:
		logerror("CPU #1: write %02x to unmapped port %02x\n", data, port & 0xff);
		break;
	}
}


/****************************************************************************
	̈
 ***************************************************************************/

MEMREGION neogeo_memory_region[] =
{
	{ NULL, 0x200000, REGION_CPU1,   0 },   // MC68000
	{ NULL, 0x10000,  REGION_CPU2,   0 },   // Z80
	{ NULL, 0x20000,  REGION_GFX1,   0 },   // FIX
	{ NULL, 0x400000, REGION_GFX2,   0 },   // SPRITE
	{ NULL, 0x20000,  REGION_GFX3,   0 },   // LO ROM
	{ NULL, 0x100000, REGION_SOUND1, 0 },   // YM2610 ADPCM-A
	{ NULL, 0x80000,  REGION_USER1,  0 },   // BIOS
	{ NULL, 0x2000,   REGION_USER2,  0 },   // obNAbv
	{ NULL, 0x200,    REGION_USER3,  0 },   // s

	{ NULL, 0x2000,   REGION_GFX4,   0 },   // EmulatorpVXetHg
#if ENABLE_SYSTEM_CHECK
	{ NULL, 0x20000,  REGION_GFX5,   0 },   // LO ROM
	{ NULL, 0x400000, REGION_GFX6,   0 },   // SPRITE
#endif
	{ NULL, }
};


/****************************************************************************
	[hCg}bv
 ***************************************************************************/

READMEM neogeo_readmem[] = 
{
	{ MEM_RAM, 0x000000, 0x1fffff, NULL                     },
	{ MEM_MAP, 0x300000, 0x31ffff, neogeo_controler1_16_r   },
	{ MEM_MAP, 0x320000, 0x33ffff, neogeo_z80_r             },
	{ MEM_MAP, 0x340000, 0x35ffff, neogeo_controler2_16_r   },
	{ MEM_MAP, 0x380000, 0x39ffff, neogeo_controler3_16_r   },
	{ MEM_MAP, 0x3c0000, 0x3dffff, neogeo_video_16_r        },
	{ MEM_MAP, 0x400000, 0x7fffff, neogeo_paletteram16_r    },
	{ MEM_MAP, 0x800000, 0x803fff, neogeo_memcard16_r       },
	{ MEM_ROM, 0xc00000, 0xc7ffff, NULL                     },
	{ MEM_MAP, 0xe00000, 0xefffff, neogeo_externalmem_16_r  },
	{ MEM_MAP, 0xff0000, 0xffffff, neogeo_hardcontrol_16_r  },
	{ MEM_END, }
};

WRITEMEM neogeo_writemem[] = 
{
	{ MEM_RAM, 0x000000, 0x1fffff, NULL                     },
	{ MEM_MAP, 0x300000, 0x31ffff, watchdog_reset_16_w      },
	{ MEM_MAP, 0x320000, 0x33ffff, neogeo_z80_w             },
	{ MEM_MAP, 0x380000, 0x39ffff, neogeo_syscontrol1_16_w  },
	{ MEM_MAP, 0x3a0000, 0x3affff, neogeo_syscontrol2_16_w  },
	{ MEM_MAP, 0x3c0000, 0x3dffff, neogeo_video_16_w        },
	{ MEM_MAP, 0x400000, 0x7fffff, neogeo_paletteram16_w    },
	{ MEM_MAP, 0x800000, 0x803fff, neogeo_memcard16_w       },
	{ MEM_MAP, 0xe00000, 0xefffff, neogeo_externalmem_16_w  },
	{ MEM_MAP, 0xff0000, 0xffffff, neogeo_hardcontrol_16_w  },
	{ MEM_END, }
};


/*----------------------------------------------------------------------------------------

  hCo̍\

  RpCoCiJۂ́Ȁ폜A`FbNXLbv
  肵Ȃ悤ɂĂB

  NJLĂQ[CD-ROM̏񂵂o^Ă܂B
  oOC肵CD-ROM̃vXŐςĂꍇ͂炭܂B
  iłȂ̂ɓ삵ȂCD-ROMꍇ́AR}hC-cIvVtA
  o͂Aȉ̓o^ς݂̃f[^Qlɓo^ĂB

  gbNf[^̏o

  ncdz.exe -c [hCu] [O]

  ) EhCuCD-ROM̏ NGCD012_B01 ƂOŏo͂   ncdz.exe -c E NGCD012_B01

  ncdz.exeƓtH_ driver_trackdata.c Ƃt@Cŏo͂܂B

 ----------------------------------------------------------------------------------------*/

// Track Data                         Leadout Track 1 Track 2 ...
static const UINT32 NGCD001_B01[]  = { 222259,      0,   3976,   6239,   8508,  30445,  32121,  48090,  49971,  51330,  52844,  55182,  57442,  77402,  78928,  99831, 110384, 127887, 138888, 156038, 158131, 159885, 161235, 178045, 192909, 195577, 200259, 201195, 206451, 207402 };
static const UINT32 NGCD002_B01[]  = { 122250,      0,   4748,   7595,  25220,  42934,  59797,  77979,  88397,  89748,  90996,  92209,  93456,  94803, 100040, 103352, 104383, 105711, 107376, 111041, 112402, 116368, 120843 };
static const UINT32 NGCD003_B01[]  = { 193091,      0,   5067,   6156,   7187,   8263,   9324,  22118,  64734, 100720, 139012, 178593, 179677, 180775, 183487, 185745, 188019, 189082 };
static const UINT32 NGCD004_B03[]  = { 140156,      0,   2712,   7324,  23836,  41560,  60974,  79027,  94946, 109610, 119458, 120457, 132606, 135540, 138715 };
static const UINT32 NGCD006_B01[]  = { 207348,      0,   3085,   4823,   6966,  10408,  11478,  14265,  16390,  18522,  21712,  24113,  26703,  31990,  34908,  37817,  41902,  45364,  49350,  54002,  58352,  60628,  61699,  65611,  81372, 141242, 157274, 173076, 188337, 199566, 203878 };
static const UINT32 NGCD007_B01[]  = { 228061,      0,   4932,  23657,  25116,  36877,  55269,  73694,  74837,  93278, 111827, 113136, 125434, 143945, 162308, 164390, 182657, 184088, 202518, 213279, 218466, 219959, 220848, 225500, 226865 };
static const UINT32 NGCD010_B01[]  = { 249158,      0,   4369,   9820,  14483,  18935,  20219,  35319,  49505,  54798,  69840,  82759,  97687, 112136, 113786, 115189, 124177, 125980, 131040, 144457, 159418, 173707, 188700, 202886, 219227, 233649, 235492, 242949, 246072, 247244 };
static const UINT32 NGCD011_B01[]  = { 197106,      0,   3976,   5802,  24520,  44619,  64484,  83757, 104253, 124980, 133972, 144705, 156184, 166460, 176685, 184744, 186077 };
static const UINT32 NGCD014_B01[]  = { 241601,      0,   4168,  18978,  33564,  34621,  41157,  65403,  89179, 112653, 136683, 160343, 163108, 163992, 178556, 193231, 217622, 218618, 220119, 231401, 232498, 235598, 236532, 237655, 238655, 240321 };
static const UINT32 NGCD017_B01[]  = { 287912,      0,   4237,   7982,  19838,  41850,  55694,  63985,  65766,  77983,  86995,  98698, 100473, 102734, 116949, 127450, 139656, 150717, 160185, 171440, 178902, 191656, 202423, 203819, 216211, 226132, 234020, 241822, 250828, 252663, 262784, 274315, 275684, 283842, 284811 };
static const UINT32 NGCD018_B01[]  = { 167032,      0,   3848,  14493,  25035,  35403,  36389,  38529,  48363,  58056,  67806,  68904,  70392,  80454,  90515, 100798, 101856, 110694, 119325, 128615, 137955, 147610, 154565, 161840, 165857 };
static const UINT32 NGCD019_B02[]  = {   4248,      0,   2227 };
static const UINT32 NGCD020_B01[]  = { 279095,      0,   3976,   5661,  11469,  32554,  53073,  75251,  96667, 117785, 139586, 161459, 196294, 231267, 247850, 264500, 265602, 266779, 274027, 275142, 277474 };
static const UINT32 NGCD021_B02[]  = { 258224,      0,   2056,  17330,  22697,  30557,  39143,  44761,  53221,  61154,  68362,  74910,  75950,  77151,  82718,  83791,  91235,  97890, 105609, 114034, 120799, 127415, 134192, 140407, 146324, 152572, 159528, 166505, 173382, 180375, 187063, 192370, 195217, 198536, 205438, 212144, 219267, 225916, 232232, 238634, 244710, 251386 };
static const UINT32 NGCD024_B01[]  = { 172825,      0,   5668,  16339,  25270,  42431,  56338,  72626,  73743,  75638,  77682,  78951,  89976, 103946, 118694, 121236, 134303, 146175, 149881, 166610, 171448 };
static const UINT32 NGCD030_B01[]  = { 167436,      0,   3912,  20086,  21530,  36371,  38422,  42819,  46066,  49386,  64901,  84047,  85251,  86851,  88100,  89238,  90390,  91846,  95791,  96863, 101209, 105723, 109684, 113659, 116885, 118505, 121731, 126584, 127639, 128662, 129648, 133742, 139548, 144771, 150032, 151218, 152394, 153564, 157426, 161993, 166024 };
static const UINT32 NGCD031_B01[]  = { 102806,      0,   3919,   5060,   8308,  12812,  16660,  21195,  22631,  35934,  49609,  63534,  77971,  86821,  91105,  94665,  95655,  98922 };
static const UINT32 NGCD032_B01[]  = { 161601,      0,   4978,   8256,   9314,  32687,  48584,  61962,  76770,  89269, 102186, 129438, 137976, 150709, 153634, 160368 };
static const UINT32 NGCD033_B02[]  = { 283869,      0,   4107,  22889,  41639,  46891,  49205,  71209,  72295,  92664, 113395, 123583, 124459, 146036, 166170, 186721, 188268, 189216, 194952, 195979, 201003, 201972, 202968, 204606, 206254, 208596, 229352, 238975, 244261, 249486, 259767, 262146, 269623, 281130 };
static const UINT32 NGCD034_B01[]  = { 121058,      0,   3913,  23113,  42016,  61087,  79977,  81950,  86064,  89811,  92926,  94394,  95444,  96945, 100545, 101661, 102582, 105583, 119782 };
static const UINT32 NGCD039_B01[]  = { 314138,      0,   8261,  25720,  43191,  45033,  49360,  51361,  69118,  86099, 103095, 119923, 127874, 129131, 146166, 147379, 164357, 181240, 198386, 215352, 231902, 235169, 236654, 237817, 252644, 269458, 270959, 272984, 278816, 283800, 285566, 287307, 288786, 306186, 307838 };
static const UINT32 NGCD040_B01[]  = { 267304,      0,   6004,   8920,  23226,  36957,  50715,  65682,  76426,  90225, 116462, 127207, 140552, 153182, 156025, 157229, 158418, 160257, 162084, 163244, 171336, 181739, 192083, 202783, 229270, 230952, 238482, 242981, 246986, 248570, 250436, 254049, 263043, 264087 };
static const UINT32 NGCD041_B02[]  = { 127291,      0,   3912,   5667,  11827,  13354,  16580,  19663,  20714,  21945,  47070,  70498,  94575, 118274, 119725, 122829 };
static const UINT32 NGCD043_B01[]  = { 192621,      0,  18669,  19671,  28697,  32026,  35326,  38626,  41927,  45227,  48528,  51829,  55130,  58430,  61730,  78553,  95323, 112094, 128860, 145632, 162400, 165401, 168402, 169385, 170596, 177634, 178963, 184385 };
static const UINT32 NGCD044_B02[]  = { 226835,      0,  10475,  29259,  48010,  66805,  85598, 104294, 123215, 141969, 160761, 164242, 169196, 173783, 178317, 183567, 186570, 188193, 189175, 190824, 192445, 194100, 195779, 197473, 199144, 200943, 202796, 203772, 204744, 205716, 209396, 212470, 215777, 220272, 221220, 222061, 223100, 224809 };
static const UINT32 NGCD045_B05[]  = { 310683,      0,  10996,  27167,  41870,  56573,  71166,  87781, 107136, 111353, 128191, 142752, 158409, 159739, 164377, 166153, 167833, 188060, 203219, 219401, 238276, 254212, 269936, 276190, 294341, 296440, 301684, 303550, 305079, 306936 };
static const UINT32 NGCD046_B01[]  = { 247051,      0,   9551,  11610,  14316,  19033,  30866,  38789,  45384,  51898,  63087,  68093,  75752,  86037,  99632, 107442, 116508, 125530, 136518, 143513, 150635, 160403, 176964, 178637, 180197, 196729, 197700, 198754, 199831, 201732, 205487, 206821, 214747, 222883, 229519, 236525, 244940, 245929 };
static const UINT32 NGCD047_B01[]  = { 266168,      0,  16148,  18286,  20304,  30197,  33257,  47698,  62182,  76845,  91479, 106146, 120940, 135531, 150070, 159167, 173718, 188161, 202729, 217322, 226464, 236053, 240389, 242226, 243512, 253104, 254619, 256624 };
static const UINT32 NGCD052_B01[]  = { 100851,      0,   3903,   5557,  11356,  28154,  30103,  32084,  35062,  36103,  37571,  38815,  55910,  57859,  73910,  97744 };
static const UINT32 NGCD055_B01[]  = { 299278,      0,  16870,  38246,  59248,  81430, 103330, 125275, 146278, 167310, 191410, 212094, 234442, 235651, 237478, 241370, 243445, 253103, 264664, 268491, 282742, 286005, 289523, 293427 };
static const UINT32 NGCD056_B03[]  = { 307418,      0,  15475,  19473,  22526,  24978,  25834,  28788,  31598,  35195,  38999,  42751,  43736,  44791,  46439,  47528,  62611,  65739,  81271,  97552, 112990, 126268, 128852, 144185, 158951, 175327, 189004, 206676, 221925, 238570, 241682, 258138, 261187, 264546, 267051, 269851, 272538, 275452, 278198, 281376, 294394 };
static const UINT32 NGCD058_B03[]  = { 305513,      0,  11124,  12802,  14552,  24445,  27505,  36602,  38439,  39725,  49317,  63758,  78242,  92905, 107539, 122206, 137000, 151591, 166130, 180681, 195124, 209692, 224285, 233874, 248621, 263319, 277968, 292536 };
static const UINT32 NGCD058_B06[]  = { 305549,      0,  11160,  12838,  14588,  24481,  27541,  36638,  38475,  39761,  49353,  63794,  78278,  92941, 107575, 122242, 137036, 151627, 166166, 180717, 195160, 209728, 224321, 233910, 248657, 263355, 278004, 292572 };
static const UINT32 NGCD059_B02[]  = { 274222,      0,  25549,  28377,  32444,  37693,  39446,  40473,  41905,  49914,  65188,  66189,  81187,  95791,  96792, 112061, 126822, 127823, 142888, 157728, 158729, 173204, 187778, 188779, 202999, 205259, 209126, 210252, 224322, 229575, 234943, 242441, 244443, 252381, 259773, 265261 };
static const UINT32 NGCD061_B01[]  = { 253623,      0,  20434,  21969,  25849,  27242,  28731,  37809,  85726, 133948, 184165, 230898, 232425, 247517, 249388, 251276 };
static const UINT32 NGCD063_B02[]  = { 307569,      0,  16969,  22108,  30281,  48306,  62184,  77627,  94409, 101873, 113733, 128463, 143711, 149213, 164859, 178074, 191466, 207113, 220369, 229680, 242686, 255727, 257642, 260763, 262038, 263087, 264963, 266388, 269091, 270816, 274271, 277949, 281981, 283850, 289459, 292158, 297197, 300864, 302511, 303486, 305488 };
static const UINT32 NGCD068_B01[]  = { 224585,      0,   3435,   4280,   6893,  11419,  12453,  13756,  33605,  52855,  72127,  73266,  77781,  80852,  82029, 100708, 119943, 138945, 140164, 144344, 147518, 148709, 169440, 188653, 208251, 209343, 213746, 218399 };
static const UINT32 NGCD069_B06[]  = { 314916,      0,  16729,  20794,  24937,  28313,  31689,  35066,  38442,  41819,  45196,  47125,  48250,  49375,  50539,  56167,  70824,  71806,  86461, 101117, 115592, 130219, 144694, 159169, 173824, 188479, 203480, 206858, 221513, 236151, 239063, 241314, 244690, 259316, 262693, 265406, 267421, 270498, 284973, 287496, 301972 };
static const UINT32 NGCD075_B01[]  = { 309854,      0,   9689,  10534,  13740,  17542,  30615,  44243,  51421,  63957,  75456,  83498,  96198, 109102, 124078, 137561, 149377, 154671, 170724, 188043, 189382, 192044, 194500, 197113, 199724, 202159, 204718, 206994, 209262, 211716, 214202, 216705, 219117, 224680, 259251 };
static const UINT32 NGCD078_B01[]  = { 307164,      0,   9507,  11553,  16056,  37058,  58060,  79062, 100064, 121066, 142068, 163070, 184072, 205074, 226076, 247753, 268755, 270449, 271501, 276378, 281256, 286133, 291011, 296263, 300091 };
static const UINT32 NGCD080_B04[]  = { 331595,      0,  10587,  18865,  31083,  32853,  35313,  37866,  47541,  57271,  67188,  77414,  87630,  89698,  99734, 112481, 122669, 132705, 142653, 152868, 162723, 173226, 177156, 178432, 182293, 186239, 190088, 200228, 210379, 220838, 222208, 225573, 235729, 239161, 240755, 241957, 244134, 245412, 257069, 267116, 277333, 285224, 296322, 297731, 301321, 305374, 307550, 311795, 317557, 327189 };
static const UINT32 NGCD081_B02[]  = { 243468,      0,  26136,  28211,  37825,  39175,  42635,  44654,  95614, 146292, 196745, 199103, 201461, 203819, 206182, 208540, 210898, 213256, 224281, 227733, 231504, 232747, 235027 };
static const UINT32 NGCD082_B01[]  = { 303917,      0,   9985,  17522,  25100,  27264,  28769,  47997,  64542,  84817, 103300, 122501, 142832, 161555, 181983, 201707, 222517, 242522, 262585, 280748, 292169 };
static const UINT32 NGCD082_B06[]  = { 303931,      0,   9999,  17536,  25114,  27278,  28783,  48011,  64556,  84831, 103314, 122515, 142846, 161569, 181997, 201721, 222531, 242536, 262599, 280762, 292183 };
static const UINT32 NGCD083_B01[]  = {   1165,      0 };
static const UINT32 NGCD084_B03[]  = { 309347,      0,  19263,  24183,  27559,  32811,  34313,  35557,  37059,  38047,  50051,  57055,  68156,  75395,  87396,  94179,  95681, 107682, 115183, 116685, 128840, 134799, 146800, 154799, 166799, 174299, 186299, 192955, 202706, 207957, 220305, 228069, 237820, 239322, 251322, 258822, 264119, 269371, 274668, 279920, 289671, 298908 };
static const UINT32 NGCD087_A03[]  = { 325299,      0,  20328,  23039,  36975,  50891,  52693,  53919,  67617,  81423,  85179,  86288,  88090,  89316,  90876,  91837,  94256,  97556, 111368, 125520, 127322, 128548, 142482, 156341, 158143, 159369, 173163, 187066, 188868, 190094, 204106, 218036, 219838, 221064, 234761, 238048, 239481, 241082, 255001, 256080, 262020, 268931, 287744, 306585, 311485, 315665, 320564 };
static const UINT32 NGCD088_B01[]  = { 186173,      0,   4600,   8793,  10631,  12552,  14368,  24103,  29680,  37724,  39557,  41455,  43031,  47686,  55955,  62987,  75213,  88671,  97614, 108230, 115284, 122015, 128818, 135290, 146014, 164434, 167086, 169372 };
static const UINT32 NGCD089_B01[]  = { 295742,      0,  62302,  77528,  95285, 110357, 122901, 137430, 150410, 162303, 185635, 201327, 222980, 234525, 254705, 258230, 266440, 268160, 269893, 274000, 280215, 281341 };
static const UINT32 NGCD092_B02[]  = { 248277,      0,  12447,  16348,  28498,  40649,  52799,  64949,  66225,  73126,  85277,  97428, 109579, 121730, 133881, 146032, 158183, 170334, 171610, 172886, 174162, 175438, 180839, 185865, 187141, 188417, 191568, 194719, 197870, 201021, 204172, 207323, 210474, 213625, 229526 };
static const UINT32 NGCD094_B01[]  = { 305890,      0,  30312,  31682,  33782,  35554,  40763,  46687,  49829,  54328,  65230,  71230,  77037,  83189,  88476,  95322, 112406, 125458, 142542, 157247, 175083, 192937, 209402, 225164, 243231, 252325, 272410, 290324 };
static const UINT32 NGCD095_B02[]  = { 332467,      0,  18050,  20662,  24056,  27443,  42229,  57279,  72417,  87085,  89035, 103516, 118180, 132662, 137821, 140029, 155114, 169601, 184259, 199318, 203626, 218110, 232774, 247745, 262903, 264681, 279421, 294458, 297632, 304217, 306055, 311387, 312557, 315245 };
static const UINT32 NGCD0961_A01[] = { 271319,      0,  30313,  31565,  35844,  38190,  42034,  45798,  48765,  53971,  59202,  62275,  80036,  98098, 115408, 133799, 149823, 167809, 168966, 169977, 187115, 205221, 222552, 239204, 255657, 260842, 268099, 269941 };
static const UINT32 NGCD097_B01[]  = { 209714,      0,   7592,   8267,  17682,  21882,  29637,  39685,  49491,  57416,  67365,  78094,  88577,  99048, 108484, 116615, 126240, 136261, 147321, 157601, 167861, 178160, 189221, 199395, 200720, 201806, 206829 };
static const UINT32 NGCD098_B03[]  = { 208725,      0,   3974,   4970,   7368,  10940,  14982,  18587,  28258,  31192,  37410,  40935,  51346,  55323,  65681,  71289,  76897,  80555,  91152,  94466, 100538, 104107, 115793, 119306, 125500, 131686, 137324, 142878, 146088, 149397, 150755, 152180, 153399, 154536, 155657, 156855, 158005, 159351, 160620, 161801, 163080, 164334, 165411, 166642, 167917, 169162, 170396, 171623, 172668, 174025, 175370, 176511, 177748, 178891, 179979, 181399, 182632, 183851, 185062, 186118, 187200, 188342, 189465, 190786, 192151, 193282, 194340, 195581, 196700, 197712, 198842, 200003, 201162, 202481, 203825, 205045, 206247, 207557 };
static const UINT32 NGCD200_A01[]  = { 184144,      0,  19056,  21084,  23336,  25365,  27904,  33504,  43569,  67680,  68771,  92816,  93907, 117908, 118999, 143000, 144091, 159093, 161344, 163971, 179040, 181141 };
static const UINT32 NGCD201_A01[]  = { 287889,      0,  13145,  14238,  24670,  49311,  74738,  76615,  78289, 103713, 129140, 154567, 156444, 158118, 183545, 208970, 234397, 236274, 237948, 263374, 275376 };
static const UINT32 NGCD205_B06[]  = { 325264,      0,  46505,  48198,  57609,  61315,  64655,  67641,  69447,  74308,  78593,  81974,  93980, 100968, 112965, 119621, 120872, 124878, 128255, 140256, 152255, 159080, 168521, 176700, 181672, 184791, 186067, 198069, 210069, 211945, 214020, 217884, 219237, 236523, 238882, 240144, 242162, 246149, 258149, 270152, 273196, 275256, 287088, 295022, 301617, 308185, 309265, 310396, 312318, 315860, 323998 };
static const UINT32 NGCD213_A01[]  = { 122364,      0,   6062,  20615,  35169,  49722,  63673,  78853,  93408, 107961 };
static const UINT32 NGCD214_A02[]  = { 328850,      0,  25972,  29423,  39174,  45910,  55660,  62755,  72505,  79667,  80790,  90541,  98099, 107888, 115019, 124773, 132633, 136916, 138206, 141293, 142211, 151961, 159143, 168897, 176193, 185945, 192996, 202747, 209798, 220070, 227046, 233140, 234267, 244849, 251899, 260098, 269848, 276899, 284632, 292138, 299708, 306952, 316447, 327561 };
static const UINT32 NGCD218_A01[]  = { 303221,      0,  16995,  20394,  22464,  26221,  27599,  29851,  32316,  38219,  42937,  47289,  55027,  56604,  57416,  78259,  98733, 119736, 139912, 160238, 180938, 200132, 221957, 242659, 262830, 282668 };
static const UINT32 NGCD219_B02[]  = { 293488,      0,   8975,  24458,  26357,  41327,  56811,  58710,  73723,  91640,  93539, 110936, 126655, 128554, 143828, 160663, 162562, 178949, 197431, 199330, 211706, 228464, 230363, 241730, 243184, 247145, 253841, 260680, 264681, 265969, 271288, 279940, 283647, 287523, 292130 };
static const UINT32 NGCD220_A01[]  = { 299915,      0,  20162,  22570,  24982,  31538,  52940,  73400,  97314, 122389, 153964, 179123, 187494, 191469, 223907, 241083, 257927, 280503, 281929, 284854, 296764 };
static const UINT32 NGCD221_A01[]  = { 287245,      0,   5087,  14986,  21510,  26911,  42064,  57215,  72366,  87512, 102663, 131011, 132289, 133568, 134920, 136422, 144520, 160042, 188310, 203461, 218609, 233759, 248909, 264060, 279211, 280789, 284242 };
static const UINT32 NGCD222_A07[]  = { 316241,      0,  28086,  29928,  30847,  34224,  35348,  37233,  48109,  58997,  69873,  78115,  79239,  81124,  93903, 104972, 109173, 120931, 122055, 123940, 134906, 146006, 154173, 155374, 166102, 167226, 169111, 177149, 181282, 192404, 203280, 204404, 206289, 217165, 228041, 238917, 249793, 250917, 252802, 256928, 266520, 269686, 280085, 281396, 292298, 295706, 301051, 309681, 310555 };
static const UINT32 NGCD229_B06[]  = { 326026,      0,  38396,  41847,  51598,  54435,  64185,  67371,  77121,  80282,  83206,  92957,  95378, 105167, 109321, 119075, 122152, 126435, 127725, 130812, 133566, 143316, 146174, 155928, 159250, 169002, 171578, 181329, 185027, 195299, 199419, 216569, 218532, 229114, 230856, 234591, 244341, 247024, 251405, 258581, 263968, 270051, 279546, 289915, 295638, 298022, 300893, 304868, 307309, 310638, 313979, 317271 };
static const UINT32 NGCD0850_A01[] = { 319249,      0, 122201, 130299, 135865, 142843, 148506, 152356, 155777, 162788, 169284, 178270, 184256, 188007, 196442, 199668, 204244, 211220, 215496, 220929, 227121, 230272, 236603, 242513, 255563, 259723, 266334, 267985, 274221, 275872, 279972, 285072, 292931 };
static const UINT32 NGCD2230_A01[] = { 306764,      0,  32664,  34888,  39755,  40667,  52511,  63558,  75891,  87123,  99125, 110163, 111697, 113858, 125113, 138196, 149192, 160870, 172100, 183163, 184697, 196644, 207867, 219121, 230341, 242341, 249914, 251514, 264339, 267206, 273360, 275365, 276489 };
static const UINT32 NGCD2300_A01[] = { 223328,      0,  10956,  14043,  29166,  44286,  45539,  46755,  48630,  49882,  65007,  80126,  95260, 110383, 125503, 140626, 155747, 170871, 185994, 201747, 207748, 208885, 210101, 211317, 212534, 213749, 214965, 216181, 217397, 218613, 219830, 221046, 222263 };
static const UINT32 NGCD2320_A03[] = { 330316,      0,  35556,  38714,  41023,  43529,  50475,  56880,  63241,  70186,  76762,  83258,  91793,  98026, 106861, 112975, 121720, 128008, 129588, 130879, 139624, 145714, 154685, 160866, 169656, 175748, 184538, 190179, 198970, 205061, 213851, 219942, 228652, 234861, 243057, 251446, 257586, 263685, 272205, 278320, 288063, 296430, 300184, 306325, 312413, 315252, 326292 };
static const UINT32 NGCD2340_A01[] = { 323784,      0,  41727,  45718,  47104,  48914,  50036,  51836,  53187,  54537,  64438,  74340,  84240,  94140, 104041, 113942, 117093, 126994, 136895, 146796, 156697, 166599, 176500, 186401, 197052, 207703, 218354, 229005, 234211, 244862, 248388, 259195, 260621, 271272, 273673, 275106, 278627, 280853, 284125, 291062, 297213, 310524 };
static const UINT32 NGCD2400_A01[] = { 323900,      0,  33245,  36412,  42343,  48525,  50819,  52026,  53617,  55364,  56723,  68663,  81027,  92398, 103584, 116548, 127683, 139078, 140406, 141718, 143045, 156967, 168360, 179729, 190904, 203045, 214222, 226308, 238125, 249484, 262549, 274688, 282400, 294389, 306352, 318666 };
static const UINT32 NGCD2410_A01[] = { 308505,      0,  20587,  43722,  53374,  60347,  84912, 105950, 127531, 150678, 177606, 199873, 222372, 242088, 264194, 278895, 295291, 296618, 298040, 300578, 307755 };
static const UINT32 NGCD2420_A04[] = { 330734,      0,  47414,  48473,  49964,  51208,  55111,  59093,  64130,  67695,  70108,  80238,  82415,  85887,  95602, 104214, 113700, 122800, 132281, 142045, 152517, 161998, 171043, 179922, 188804, 197348, 198344, 208275, 217011, 227097, 236881, 246725, 256911, 265294, 275530, 282371, 292498, 301418, 311411, 321236, 325535 };
static const UINT32 NGCD2430_A03[] = { 281722,      0,  43345,  49046,  50547,  53473,  59324,  69975,  79876,  89777, 100428, 110329, 120230, 130131, 140032, 150683, 160584, 170485, 180386, 190287, 200938, 211589, 222240, 232891, 240392, 242193, 257719, 273245, 277596 };
static const UINT32 NGCD2510_B01[] = { 322559,      0,  39867,  45571,  47299,  51653,  55480,  63237,  69516,  74320,  77364,  80104,  82564,  85180,  94349,  97656, 102270, 104883, 114636, 122106, 131859, 139362, 149115, 156618, 166371, 173885, 183671, 191176, 200931, 208436, 218189, 225694, 235525, 243008, 252763, 260331, 270123, 277647, 287464, 294973, 304796, 312347, 314704, 320408 };

static const UINT32 ADCD001_B01[]  = { 215302,      0,   3643,  11482,  12951,  15727,  18728,  22404,  26950,  28976,  29852,  30953,  37074,  44545,  45869,  47895,  49021,  54287,  59012,  64503,  66154,  71014,  72039,  76018,  91859, 109663, 124417, 138367, 151491, 164586, 181449, 195535, 208581 };
static const UINT32 ADCD002_B02[]  = { 231483,      0,   3387,   6687,  11458,  14833,  19649,  27974,  30224,  36614,  39840,  49021,  54196,  56971,  59971,  61696,  63046,  75826,  92566, 104806, 119476, 135946, 152596, 170416, 184412, 200657, 216902, 220052 };
static const UINT32 ADCD003_B01[]  = { 140830,      0,   3067,   5167,  10432,  15023,  15998,  21038,  22088,  23813,  24939,  45415,  63145,  80695,  95770, 112420, 127450 };
static const UINT32 ADCD004_B01[]  = { 225521,      0,   3643,   5669,   7019,  11790,  16141,  21091,  25906,  31532,  35432,  40338,  41388,  59748,  73878,  89494, 103985, 121445, 135035, 147770, 148820, 150171, 157236, 164706, 170736, 178791, 186171, 193147, 200078, 206694, 211960 };
static const UINT32 ADCD005_B01[]  = { 272142,      0,   3812,   6275,  22562,  38787,  54982,  71493,  87705, 103501, 119081, 134653, 151761, 168011, 169572, 173931, 175429, 179603, 181180, 183266, 185386, 187472, 189614, 191779, 193940, 196063, 198173, 203833, 209295, 212746, 221550, 225210, 231075, 237454, 247422, 255224, 262934 };
static const UINT32 ADCD006_B02[]  = { 302542,      0,   8584,  11577,  21249,  30920,  40900,  50964,  60681,  70346,  79903,  89424,  98967, 109765, 114132, 117769, 119136, 120921, 122388, 125807, 136011, 145704, 156178, 165864, 175805, 185685, 195636, 205581, 216294, 226323, 239024, 244076, 247746, 252751, 255542, 260738, 263696, 267226, 270093, 271535, 274372, 277882, 282843, 287160, 288552, 293763, 297639, 301578 };
static const UINT32 ADCD007_B01[]  = { 230481,      0,  11627,  14327,  18962,  20313,  21730,  24898,  29093,  30437,  31855,  33795,  35970,  52200,  68882,  83435,  86961,  89574, 104173, 117629, 132298, 148139, 164564, 166057, 167626, 182254, 184648, 186143, 189139, 205187, 208078, 210699, 213768, 216169, 218795 };
static const UINT32 ADCD008_B02[]  = { 315240,      0,   9562,  12036,  13532,  16233,  21558,  24329,  27926,  29424,  31374,  41738,  52054,  55500,  66225,  77127,  79150,  89516,  99740, 101460, 108810, 118321, 119819, 121769, 124544, 127315, 130165, 132112, 143286, 153645, 155521, 165976, 176161, 178212, 187363, 197679, 201504, 212226, 223041, 235968, 247140, 260516, 272947, 279438, 290789, 301380 };
static const UINT32 ADCD009_B01[]  = { 329711,      0,  12892,  14343,  17099,  20193,  34954,  49707,  64403,  78782,  93883, 109082, 125125, 126303, 129894, 131145, 132486, 134294, 135615, 139035, 153968, 170459, 184807, 188086, 202856, 206662, 221441, 224700, 227242, 230697, 234200, 238739, 242171, 248765, 255420, 261848, 266405, 276041, 279094, 282169, 286710, 290024, 295790, 302481, 305958, 312421, 316748, 323258 };
static const UINT32 ADCD010_B02[]  = { 193990,      0,   3269,   8971,  23418,  60608,  97671,  99376, 101079, 106685, 143757, 149501, 186626, 188331 };
static const UINT32 ADCD011_B01[]  = { 241145,      0,  13310,  16222,  31012,  56764,  83008, 108329, 134109, 135421, 149971, 151899, 153453, 163481, 189388, 215279 };
static const UINT32 ADCD012_A01[]  = { 316932,      0,  16413,  21086,  28360,  40233,  56545,  71311,  86618, 102125, 117580, 134550, 135823, 144115, 145632, 147870, 163214, 178818, 194120, 209476, 224733, 239732, 241023, 248139, 255249, 259276, 263487, 267591, 273466, 278226, 282816, 286708, 292765, 297349, 301308, 305220, 309141 };
static const UINT32 ADCD013_A01[]  = { 267643,      0,   8468,  14221,  15502,  19511,  23792,  25071,  39716,  54085,  70900,  87669, 107929, 125519, 126819, 143792, 160703, 175172, 189647, 206577, 210723, 214664, 228384, 245468, 246887, 248699, 249974, 259980 };
static const UINT32 ADCD100_B01[]  = { 204679,      0,   2619,   9325,  10825,  15595,  17096,  18146,  19046,  32321,  49331,  63912,  78852,  91587,  98922, 112602, 126552, 129703, 132853, 136003, 139154, 142304, 145454, 148604, 151754, 154905, 160170, 176101, 179326, 182476, 185626, 188777, 191928, 195079, 198229, 201379 };
static const UINT32 ADCD101_B01[]  = { 250867,      0,   2875,   4976,   8952,  12178,  13304,  23249,  30405,  37920,  45570,  52545,  58531,  66046,  67922,  68965,  71891,  86786, 103475, 121836, 136911, 155946, 172417, 190507, 204862, 219262, 238072 };
static const UINT32 ADCD102_B01[]  = { 231005,      0,   4461,   7101,  21587,  36036,  50516,  64966,  79620,  94134, 108512, 123006, 126161, 127709, 130939, 132780, 147201, 161636, 176148, 178179, 192629, 203296, 206423, 229900 };
static const UINT32 ADCD103_B01[]  = { 306406,      0,   9027,  12342,  22401,  32487,  34066,  36055,  37323,  38754,  48846,  52259,  55519,  65858,  77015,  87426,  97870, 108142, 117555, 118960, 122469, 132761, 143470, 153429, 154944, 156210, 166255, 176101, 177440, 179353, 180699, 181982, 191913, 201866, 203782, 214357, 228645, 242785, 252333, 262558, 271561, 281374, 292548, 302649 };
static const UINT32 ADCD104_B01[]  = { 174951,      0,   4803,  10333,  35298,  60204,  61405,  62471,  63626,  88890, 114070, 117744, 132891, 134383, 135498, 138646, 142281, 145696, 148974, 156285, 159813, 164264, 167377, 170841 };

static const UINT32 AICD048_B01[]  = { 230738,      0,   3534,   4584,  10734,  21460,  25415,  31703,  48353,  62753,  68979,  76630,  82883,  87207,  93912,  99657, 107778, 115131, 126204, 132639, 137505, 142958, 150709, 157235, 164286, 170630, 178123, 183562, 190178, 196911, 203240, 209480, 210927, 215619, 218103, 219453, 229428 };
static const UINT32 AICD051_B01[]  = { 143612,      0,   3350,   6350,   7176,  18501,  27801,  47900,  59075,  74300,  91025, 106775, 115775, 129575, 131525 };

static const UINT32 DECO002_B01[]  = { 102005,      0,   3452,  14495,  25791,  37141,  45559,  46982,  48567,  49876,  62120,  71967,  83836,  92629,  95460,  96787,  97630,  98547,  99466, 100319, 101162 };
static const UINT32 DECO003_B01[]  = { 313506,      0,   7842,  11517,  13617,  30788,  47900,  65078,  82202,  99261, 116415, 133555, 137417, 138892, 141156, 142609, 143947, 161018, 178001, 194982, 211985, 229129, 246121, 250547, 253350, 257744, 261045, 263773, 267969, 272285, 279648, 285921, 295188, 297907, 300735, 304932, 307787, 310784 };
static const UINT32 DECO004_B04[]  = { 161626,      0,   4461,  28519,  52717,  76547, 100989, 124842, 130485, 132582, 134067, 149670, 150996 };


//---------------------------------------------
// CD-ROM Info
//---------------------------------------------
// int    num_tracks
// int    num_trackdata
// UINt32 *tracks[]
//---------------------------------------------
static const CDROM NGCD001  = { 29, 1, { NGCD001_B01  } };				// nam1975
static const CDROM NGCD002  = { 22, 1, { NGCD002_B01  } };				// bstars
static const CDROM NGCD003  = { 17, 1, { NGCD003_B01  } };				// tpgolf
static const CDROM NGCD004  = { 14, 1, { NGCD004_B03  } };				// mahretsu    (JPN only)
static const CDROM NGCD006  = { 30, 1, { NGCD006_B01  } };				// ridhero
static const CDROM NGCD007  = { 24, 1, { NGCD007_B01  } };				// alpham2
static const CDROM NGCD010  = { 29, 1, { NGCD010_B01  } };				// cyberlip
static const CDROM NGCD011  = { 16, 1, { NGCD011_B01  } };				// superspy    = EB01
static const CDROM NGCD014  = { 25, 1, { NGCD014_B01  } };				// mutnat      = EB01
static const CDROM NGCD017  = { 34, 1, { NGCD017_B01  } };				// sengoku     = EB01
static const CDROM NGCD018  = { 24, 1, { NGCD018_B01  } };				// burningf
static const CDROM NGCD019  = { 02, 1, { NGCD019_B02  } };				// lbowling
static const CDROM NGCD020  = { 20, 1, { NGCD020_B01  } };				// gpilots
static const CDROM NGCD021  = { 41, 1, { NGCD021_B02  } };				// joyjoy
static const CDROM NGCD024  = { 20, 1, { NGCD024_B01  } };				// lresort
static const CDROM NGCD030  = { 40, 1, { NGCD030_B01  } };				// 2020bb
static const CDROM NGCD031  = { 17, 1, { NGCD031_B01  } };				// socbrawl
static const CDROM NGCD032  = { 15, 1, { NGCD032_B01  } };				// roboarmy
static const CDROM NGCD033  = { 33, 1, { NGCD033_B02  } };				// fatfury1    = EB03
static const CDROM NGCD034  = { 18, 1, { NGCD034_B01  } };				// fbfrenzy
static const CDROM NGCD039  = { 34, 1, { NGCD039_B01  } };				// kotm2
static const CDROM NGCD040  = { 33, 1, { NGCD040_B01  } };				// sengoku2
static const CDROM NGCD041  = { 15, 1, { NGCD041_B02  } };				// bstars2
static const CDROM NGCD043  = { 27, 1, { NGCD043_B01  } };				// 3countb     = EB01
static const CDROM NGCD044  = { 37, 1, { NGCD044_B02  } };				// aof
static const CDROM NGCD045  = { 29, 1, { NGCD045_B05  } };				// samsho      = B08
static const CDROM NGCD046  = { 37, 1, { NGCD046_B01  } };				// tophuntr    = EB01
static const CDROM NGCD047  = { 27, 1, { NGCD047_B01  } };				// fatfury2    = EB01
static const CDROM NGCD052  = { 15, 1, { NGCD052_B01  } };				// ssideki     = EB01
static const CDROM NGCD055  = { 23, 1, { NGCD055_B01  } };				// kof94       = B02,EB01
static const CDROM NGCD056  = { 40, 1, { NGCD056_B03  } };				// aof2        = B04
static const CDROM NGCD058  = { 27, 2, { NGCD058_B06, NGCD058_B03 } };	// fatfursp    B06 = B09
static const CDROM NGCD059  = { 35, 1, { NGCD059_B02  } };				// savgere     = EB01
static const CDROM NGCD061  = { 15, 1, { NGCD061_B01  } };				// ssideki2    = B04,EB01
static const CDROM NGCD063  = { 40, 1, { NGCD063_B02  } };				// samsho2     = EB02
static const CDROM NGCD068  = { 27, 1, { NGCD068_B01  } };				// pspikes2
static const CDROM NGCD069  = { 40, 1, { NGCD069_B06  } };				// fatfury3    = EB02,EB03
static const CDROM NGCD075  = { 34, 1, { NGCD075_B01  } };				// sonicwi2
static const CDROM NGCD078  = { 24, 1, { NGCD078_B01  } };				// galaxyfg    = E01
static const CDROM NGCD080  = { 49, 1, { NGCD080_B04  } };				// quizkof     (JPN only)
static const CDROM NGCD081  = { 22, 1, { NGCD081_B02  } };				// ssideki3
static const CDROM NGCD082  = { 20, 2, { NGCD082_B06, NGCD082_B01 } };	// doubledr    B06 = EB05, B01 = EB04
static const CDROM NGCD083  = {  1, 1, { NGCD083_B01  } };				// pbobblen
static const CDROM NGCD084  = { 41, 1, { NGCD084_B03  } };				// kof95       = B05,B07,EB01
static const CDROM NGCD087  = { 46, 1, { NGCD087_A03  } };				// samsho3     = A12,EA05
static const CDROM NGCD088  = { 27, 1, { NGCD088_B01  } };				// stakwin     (JPN only)
static const CDROM NGCD089  = { 21, 1, { NGCD089_B01  } };				// pulstar     = EB01
static const CDROM NGCD092  = { 34, 1, { NGCD092_B02  } };				// kabukikl    (JPN only)
static const CDROM NGCD094  = { 27, 1, { NGCD094_B01  } };				// gowcaizr    = EB01
static const CDROM NGCD095  = { 33, 1, { NGCD095_B02  } };				// rbff1       = B08,E01
static const CDROM NGCD096  = { 27, 1, { NGCD0961_A01 } };				// aof3
static const CDROM NGCD097  = { 26, 1, { NGCD097_B01  } };				// sonicwi3
static const CDROM NGCD098  = { 77, 1, { NGCD098_B03  } };				// fromance2   (JPN only)
static const CDROM NGCD200  = { 21, 1, { NGCD200_A01  } };				// turfmast
static const CDROM NGCD201  = { 20, 1, { NGCD201_A01  } };				// mslug       = EA01
static const CDROM NGCD205  = { 50, 1, { NGCD205_B06  } };				// ngcdsp      (JPN only)
static const CDROM NGCD213  = {  9, 1, { NGCD213_A01  } };				// neodrift    (JPN only)
static const CDROM NGCD214  = { 43, 1, { NGCD214_A02  } };				// kof96       = A03,EA01
static const CDROM NGCD218  = { 25, 1, { NGCD218_A01  } };				// ragnagrd    (JPN only)
static const CDROM NGCD219  = { 34, 1, { NGCD219_B02  } };				// pgoal
static const CDROM NGCD220  = { 20, 1, { NGCD220_A01  } };				// ironclad    (JPN only)
static const CDROM NGCD221  = { 26, 1, { NGCD221_A01  } };				// magdrop2    (JPN only)
static const CDROM NGCD222  = { 48, 1, { NGCD222_A07  } };				// samsho4
static const CDROM NGCD229  = { 51, 1, { NGCD229_B06  } };				// kof96ngc    (JPN only)
static const CDROM NGCD0850 = { 32, 1, { NGCD0850_A01 } };				// ssrpg       (JPN only)
static const CDROM NGCD2230 = { 32, 1, { NGCD2230_A01 } };				// rbffspec
static const CDROM NGCD2300 = { 32, 1, { NGCD2300_A01 } };				// breakers    (JPN only)
static const CDROM NGCD2320 = { 46, 1, { NGCD2320_A03 } };				// kof97       = EB01
static const CDROM NGCD2340 = { 41, 1, { NGCD2340_A01 } };				// lastblad
static const CDROM NGCD2400 = { 35, 1, { NGCD2400_A01 } };				// rbff2       = EA02
static const CDROM NGCD2410 = { 20, 1, { NGCD2410_A01 } };				// lastbld2
static const CDROM NGCD2420 = { 40, 1, { NGCD2420_A04 } };				// kof98       = 2421 A01 (łQ[̃fBXNNGCD-2420)
static const CDROM NGCD2430 = { 28, 1, { NGCD2430_A03 } };				// mslug2
static const CDROM NGCD2510 = { 43, 1, { NGCD2510_B01 } };				// kof99       = 2511 B01

static const CDROM ADCD001  = { 32, 1, { ADCD001_B01  } };				// bjourney    (JPN only)
static const CDROM ADCD002  = { 27, 1, { ADCD002_B02  } };				// crswords
static const CDROM ADCD003  = { 16, 1, { ADCD003_B01  } };				// trally      = B02
static const CDROM ADCD004  = { 30, 1, { ADCD004_B01  } };				// ncommand    (JPN only)
static const CDROM ADCD005  = { 36, 1, { ADCD005_B01  } };				// wh1
static const CDROM ADCD006  = { 47, 1, { ADCD006_B02  } };				// wh2
static const CDROM ADCD007  = { 34, 1, { ADCD007_B01  } };				// wh2j
static const CDROM ADCD008  = { 45, 1, { ADCD008_B02  } };				// aodk        = E01
static const CDROM ADCD009  = { 47, 1, { ADCD009_B01  } };				// whp
static const CDROM ADCD010  = { 13, 1, { ADCD010_B02  } };				// mosyougi    (JPN only)
static const CDROM ADCD011  = { 15, 1, { ADCD011_B01  } };				// overtop     (JPN only)
static const CDROM ADCD012  = { 36, 1, { ADCD012_A01  } };				// ninjamas    (JPN only)
static const CDROM ADCD013  = { 27, 1, { ADCD013_A01  } };				// twinspri    (JPN only)
static const CDROM ADCD100  = { 35, 1, { ADCD100_B01  } };				// maglord     (JPN only)
static const CDROM ADCD101  = { 26, 1, { ADCD101_B01  } };				// ncombat     (JPN only)
static const CDROM ADCD102  = { 23, 1, { ADCD102_B01  } };				// crsword2    (JPN only)
static const CDROM ADCD103  = { 43, 1, { ADCD103_B01  } };				// adkworld    (JPN only)
static const CDROM ADCD104  = { 23, 1, { ADCD104_B01  } };				// zintrick    (JPN only)

static const CDROM AICD048  = { 36, 1, { AICD048_B01  } };				// jyanshin    (JPN only)
static const CDROM AICD051  = { 14, 1, { AICD051_B01  } };				// viewpoin    = B04,EB01

static const CDROM DECO002  = { 20, 1, { DECO002_B01  } };				// wjammers
static const CDROM DECO003  = { 37, 1, { DECO003_B01  } };				// karnovr
static const CDROM DECO004  = { 12, 1, { DECO004_B04  } };				// strhoop     = EB03

GAMES games[] =
{
	//---------------------------------------------
	// Driver Info
	//---------------------------------------------
	// char  *name
	// char  *id
	// int   year
	// char  *manufacturer
	// int   driver_type
	// int   driver_flag
	// CDROM *cdrom
	// char  title[3]   (0:Japan 1:USA 2:Europe)
	//---------------------------------------------

	{ "nam1975",  "NGCD-001",  1990, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD001,  { "NAM-1975",                                             NULL,                                           }},
	{ "bstars",   "NGCD-002",  1990, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD002,  { "BASEBALL STARS PROFESSIONAL",                          NULL,                                           }},
	{ "tpgolf",   "NGCD-003",  1990, "SNK",           RASTER,      LOADING_TYPE2,                  &NGCD003,  { "TOP PLAYER'S GOLF",                                    NULL,                                           }},
	{ "mahretsu", "NGCD-004",  1990, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD004,  { "` {",                                 "mahjong-kyo retsuden",                         }},
	{ "ridhero",  "NGCD-006",  1991, "SNK",           RASTER_FAST, LOADING_TYPE2,                  &NGCD006,  { "RIDING HERO",                                          NULL,                                           }},
	{ "alpham2",  "NGCD-007",  1991, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD007,  { "ASO U - LAST GUARDIAN",                               "ALPHA MISSION 2",                              }},
	{ "cyberlip", "NGCD-010",  1991, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD010,  { "CYBER-LIP",                                            NULL,                                           }},
	{ "superspy", "NGCD-011",  1990, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD011,  { "THE SUPER SPY",                                        NULL,                                           }},
	{ "mutnat",   "NGCD-014",  1991, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD014,  { "MUTATION NATION",                                      NULL,                                           }},
	{ "sengoku",  "NGCD-017",  1991, "SNK",           RASTER_FAST, LOADING_TYPE2,                  &NGCD017,  { "퍑`",                                             "SENGOKU",                                      }},
	{ "burningf", "NGCD-018",  1991, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD018,  { "BURNING FIGHT",                                        NULL,                                           }},
	{ "lbowling", "NGCD-019",  1990, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD019,  { "LEAGUE BOWLING",                                       NULL,                                           }},
	{ "gpilots",  "NGCD-020",  1991, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD020,  { "GHOST PILOTS",                                         NULL,                                           }},
	{ "joyjoy",   "NGCD-021",  1990, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD021,  { "Joy Joy Kid",                                          "PUZZLED",                                      }},
	{ "lresort",  "NGCD-024",  1991, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD024,  { "LAST RESORT",                                          NULL,                                           }},
	{ "2020bb",   "NGCD-030",  1991, "PALLAS/SNK",    NEOGEO,      LOADING_TYPE2,                  &NGCD030,  { "2020 SUPER BASEBALL",                                  NULL,                                           }},
	{ "socbrawl", "NGCD-031",  1991, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD031,  { "SOCCER BRAWL",                                         NULL,                                           }},
	{ "roboarmy", "NGCD-032",  1991, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD032,  { "ROBO ARMY",                                            NULL,                                           }},
	{ "fatfury",  "NGCD-033",  1991, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD033,  { "T` - h̓",                                "FATAL FURY",                                   }},
	{ "fbfrenzy", "NGCD-034",  1991, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD034,  { "FOOTBALL FRENZY",                                      NULL,                                           }},
	{ "kotm2",    "NGCD-039",  1992, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD039,  { "KING OF THE MONSTERS 2 - THE NEXT THING",              NULL,                                           }},
	{ "sengoku2", "NGCD-040",  1993, "SNK",           RASTER_FAST, LOADING_TYPE2,                  &NGCD040,  { "퍑`2",                                            "SENGOKU 2",                                    }},
	{ "bstars2",  "NGCD-041",  1992, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD041,  { "BASEBALL STARS 2",                                     NULL,                                           }},
	{ "3countb",  "NGCD-043",  1993, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD043,  { "FIRE SPLEX",                                           "3 COUNT BOUT",                                 }},
	{ "aof",      "NGCD-044",  1992, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD044,  { "Ղ̌",                                             "ART OF FIGHTING",                              }},
	{ "samsho",   "NGCD-045",  1995, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD045,  { "SAMURAI SPIRITS",                                      "SAMURAI SHODOWN",                              }},
	{ "tophuntr", "NGCD-046",  1994, "SNK",           RASTER_FAST, LOADING_TYPE2,                  &NGCD046,  { "TOP HUNTER - RODDY & CATHY",                           NULL,                                           }},
	{ "fatfury2", "NGCD-047",  1992, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD047,  { "T`2 - VȂ铬",                             "FATAL FURY 2",                                 }},
	{ "ssideki",  "NGCD-052",  1992, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD052,  { "_ - SUPER SIDEKICSKS",                            "SUPER SIDEKICKS",                              }},
	{ "kof94",    "NGCD-055",  1994, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD055,  { "THE KING OF FIGHTERS '94",                             NULL,                                           }},
	{ "aof2",     "NGCD-056",  1994, "SNK",           RASTER_FAST, LOADING_TYPE2 | PATCH_AOF2,     &NGCD056,  { "Ղ̌2",                                            "ART OF FIGHTING 2",                            }},
	{ "fatfursp", "NGCD-058",  1993, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD058,  { "T` SPECIAL",                                     "FATAL FURY SPECIAL",                           }},
	{ "savagere", "NGCD-059",  1995, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD059,  { "_َ^ in",                                  "SAVAGE REIGN",                                 }},
	{ "ssideki2", "NGCD-061",  1994, "SNK",           RASTER,      LOADING_TYPE2,                  &NGCD061,  { "_2 - At @Cg tbg{[",               "SUPER SIDEKICKS 2",                            }},
	{ "samsho2",  "NGCD-063",  1994, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD063,  { "^SAMULAI SPIRITS eےn",                       "SAMURAI SHODOWN II",                           }},
	{ "pspikes2", "NGCD-068",  1994, "VIDEO SYSTEM",  RASTER_FAST, LOADING_TYPE2,                  &NGCD068,  { "POWER SPIKES U",                                      "POWER SPIKES II",                              }},
	{ "fatfury3", "NGCD-069",  1995, "SNK",           RASTER_FAST, LOADING_TYPE2,                  &NGCD069,  { "T`3 - yȂ铬",                             "FATAL FURY 3",                                 }},
	{ "sonicwi2", "NGCD-075",  1994, "VIDEO SYSTEM",  NEOGEO,      LOADING_TYPE2,                  &NGCD075,  { "SONIC WINGS 2",                                        "AERO FIGHTERS 2",                              }},
	{ "galaxyfg", "NGCD-078",  1995, "SUNSOFT",       RASTER_FAST, LOADING_TYPE2,                  &NGCD078,  { "GALAXY FIGHT",                                         NULL,                                           }},
	{ "quizkof",  "NGCD-080",  1995, "SAURUS",        RASTER_FAST, LOADING_TYPE2,                  &NGCD080,  { "NCYELOEIuEt@C^[Y",                   "quiz king of fighters",                        }},
	{ "ssideki3", "NGCD-081",  1995, "SNK",           RASTER,      LOADING_TYPE2,                  &NGCD081,  { "_3 hւ̒",                                 "SUPER SIDEKICKS 3",                            }},
	{ "doubledr", "NGCD-082",  1995, "TECHNOS",       RASTER_FAST, LOADING_TYPE2,                  &NGCD082,  { "DOUBLE DRAGON",                                        NULL,                                           }},
	{ "pbobblen", "NGCD-083",  1994, "TAITO",         NEOGEO,      LOADING_TYPE2,                  &NGCD083,  { "PUZZLE BOBBLE",                                        "BUST-A-MOVE",                                  }},
	{ "kof95",    "NGCD-084",  1995, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD084,  { "THE KING OF FIGHTERS '95",                             NULL,                                           }},
	{ "samsho3",  "NGCD-087",  1995, "SNK",           RASTER_FAST, LOADING_TYPE2,                  &NGCD087,  { "TCXsbc agYo",                      "SAMURAI SHODOWN III",                          }},
	{ "stakwin",  "NGCD-088",  1996, "SAURUS",        NEOGEO,      LOADING_TYPE2,                  &NGCD088,  { "Xe[NXEBi[ GISeւ̓",                  "STAKES WINNER",                                }},
	{ "pulstar",  "NGCD-089",  1995, "AICOM",         RASTER_FAST, LOADING_TYPE2,                  &NGCD089,  { "PULSTAR",                                              NULL,                                           }},
	{ "kabukikl", "NGCD-092",  1995, "HUDSON SOFT",   NEOGEO,      LOADING_TYPE2,                  &NGCD092,  { "VO ^`",                                        "FAR EAST OF EDEN - KABUKIi KLASH",             }},
	{ "gowcaizr", "NGCD-094",  1995, "TECHNOS",       NEOGEO,      LOADING_TYPE2,                  &NGCD094,  { "lwSEJCU[",                                 "VOLTAGE FIGHTER - GOWCAIZER",                  }},
	{ "rbff1",    "NGCD-095",  1995, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD095,  { "AoEgT`",                                 "REAL BOUT FATAL FURY",                         }},
	{ "aof3",     "NGCD-096",  1996, "SNK",           NEOGEO,      LOADING_TYPE3,                  &NGCD096,  { "ART OF FIGHTING Ղ̌ O`",                        "ART OF FIGHTING 3",                            }},
	{ "sonicwi3", "NGCD-097",  1995, "VIDEO SYSTEM",  NEOGEO,      LOADING_TYPE2,                  &NGCD097,  { "SONIC WINGS 3",                                        "AERO FIGHTERS 3",                              }},
	{ "fromanc2", "NGCD-098",  1995, "VIDEO SYSTEM",  NEOGEO,      LOADING_TYPE2,                  &NGCD098,  { "ACh t@Ci}X2",                     "Final Romance 2",                              }},
	{ "turfmast", "NGCD-200",  1996, "NAZCA",         RASTER_FAST, LOADING_TYPE2,                  &NGCD200,  { "BIG TORNAMENT GOLF",                                   "NEO TURF MASTERS",                             }},
	{ "mslug",    "NGCD-201",  1996, "NAZCA",         NEOGEO,      LOADING_TYPE2,                  &NGCD201,  { "METAL SLUG",                                           NULL,                                           }},
	{ "ngcdsp",   "NGCD-205",  1995, "SNK",           RASTER,      LOADING_TYPE1,                  &NGCD205,  { "NEOGEO CD SPECIAL",                                    NULL,                                           }},
	{ "neodrift", "NGCD-213",  1996, "VISCO",         RASTER,      LOADING_TYPE2,                  &NGCD213,  { "NEO Drift Out - NEW TECHNOLOGY",                       NULL,                                           }},
	{ "kof96",    "NGCD-214",  1996, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD214,  { "THE KING OF FIGHTERS '96",                             NULL,                                           }},
	{ "ragnagrd", "NGCD-218",  1996, "SAURUS",        NEOGEO,      LOADING_TYPE2,                  &NGCD218,  { "_",                                               "RAGNAGARD",                                    }},
	{ "pgoal",    "NGCD-219",  1996, "SAURUS",        NEOGEO,      LOADING_TYPE2,                  &NGCD219,  { "FOOTSAL - 5 on 5 mini SOCCER",                         "PLEASURE GOAL",                                }},
	{ "ironclad", "NGCD-220",  1996, "SAURUS",        NEOGEO,      LOADING_TYPE2,                  &NGCD220,  { "SuLK[",                                     "IRONCLAD",                                     }},
	{ "magdrop2", "NGCD-221",  1996, "DATA EAST",     NEOGEO,      LOADING_TYPE2,                  &NGCD221,  { "}WJhbv2",                                    "MAGICAL DROP II",                              }},
	{ "samsho4",  "NGCD-222",  1996, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD222,  { "TCXsbc V~",                          "SAMURAI SHODOWN IV",                           }},
	{ "kof96ngc", "NGCD-229",  1997, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD229,  { "THE KING OF FIGHTERS '96 NEOGEO Collection",          "THE KING OF FIGHTERS '96 NEOGEO Collection",   }},
	{ "ssrpg",    "NGCD-0850", 1996, "SNK",           NEOGEO,      LOADING_TYPE2 | PATCH_SSRPG,    &NGCD0850, { "^ TCXsbc m`",                   "shinsetsu samurai spirits - bushido-retsuden", }},
	{ "rbffspec", "NGCD-2230", 1996, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD2230, { "AoEgT`XyV",                       "REAL BOUT FATAL FURY SPECIAL",                 }},
	{ "breakers", "NGCD-2300", 1996, "VISCO",         RASTER,      LOADING_TYPE2,                  &NGCD2300, { "BREAKER'S",                                            NULL,                                           }},
	{ "kof97",    "NGCD-2320", 1997, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD2320, { "THE KING OF FIGHTERS '97",                             NULL,                                           }},
	{ "lastblad", "NGCD-2340", 1997, "SNK",           RASTER_FAST, LOADING_TYPE3,                  &NGCD2340, { "Q ؂̌m",                                  "THE LAST BLADE",                               }},
	{ "rbff2",    "NGCD-2400", 1998, "SNK",           NEOGEO,      LOADING_TYPE2 | PATCH_RBFF2,    &NGCD2400, { "AoEgT`2",                                "REAL BOUT FATAL FURY 2",                       }},
	{ "mslug2",   "NGCD-2410", 1998, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD2410, { "METAL SLUG 2",                                         NULL,                                           }},
	{ "kof98",    "NGCD-2420", 1998, "SNK",           RASTER_FAST, LOADING_TYPE2,                  &NGCD2420, { "THE KING OF FIGHTERS '98",                             NULL,                                           }},
	{ "lastbld2", "NGCD-2430", 1998, "SNK",           RASTER_FAST, LOADING_TYPE3,                  &NGCD2430, { "Q ؂̌m `ɍ炭؁AU䂭ԁ`", "THE LAST BLADE 2",                             }},
	{ "kof99",    "NGCD-2510", 1999, "SNK",           NEOGEO,      LOADING_TYPE2,                  &NGCD2510, { "THE KING OF FIGHTERS '99",                             NULL,                                           }},

	{ "bjourney", "ADCD-001",  1990, "SNK/ADK",       NEOGEO,      LOADING_TYPE2,                  &ADCD001,  { "RAGUY",                                                "BLUE'S JOURNEY",                               }},
	{ "crswords", "ADCD-002",  1991, "SNK/ADK",       NEOGEO,      LOADING_TYPE2,                  &ADCD002,  { "CLOSSED SWORDS",                                       NULL,                                           }},
	{ "trally",   "ADCD-003",  1991, "SNK/ADK",       RASTER,      LOADING_TYPE2,                  &ADCD003,  { "RALLY CHASE",                                          "THRASH RALLY",                                 }},
	{ "ncommand", "ADCD-004",  1992, "SNK/ADK",       NEOGEO,      LOADING_TYPE2,                  &ADCD004,  { "NINJA COMMANDO",                                       NULL,                                           }},
	{ "wh1",      "ADCD-005",  1992, "SNK/ADK",       RASTER_FAST, LOADING_TYPE2,                  &ADCD005,  { "WORLD HEROS",                                          NULL,                                           }},
	{ "wh2",      "ADCD-006",  1993, "SNK/ADK",       RASTER_FAST, LOADING_TYPE2,                  &ADCD006,  { "WORLD HEROS 2",                                        NULL,                                           }},
	{ "wh2j",     "ADCD-007",  1994, "SNK/ADK",       RASTER_FAST, LOADING_TYPE2,                  &ADCD007,  { "WORLD HEROS 2 JET",                                    NULL,                                           }},
	{ "aodk",     "ADCD-008",  1994, "SNK/ADK",       RASTER_FAST, LOADING_TYPE2,                  &ADCD008,  { "ɉGANGANsi",                                     "AGGRESSORS OF DARK KOMBAT",                    }},
	{ "whp",      "ADCD-009",  1995, "SNK/ADK",       NEOGEO,      LOADING_TYPE2,                  &ADCD009,  { "WORLD HEROS PERFECT",                                  NULL,                                           }},
	{ "mosyougi", "ADCD-010",  1995, "SNK/ADK",       RBUSY_FAST,  LOADING_TYPE2,                  &ADCD010,  { "̒Bl",                                           "syougi no tatsujin",                           }},
	{ "overtop",  "ADCD-011",  1996, "SNK/ADK",       RASTER,      LOADING_TYPE2,                  &ADCD011,  { "OVER TOP",                                             NULL,                                           }},
	{ "ninjamas", "ADCD-012",  1996, "SNK/ADK",       NEOGEO,      LOADING_TYPE2,                  &ADCD012,  { "NINJA MASTER'S eE@",                            "NINJA MASTER'S - haoh-ninpo-cho",              }},
	{ "twinspri", "ADCD-013",  1996, "SNK/ADK",       NEOGEO,      LOADING_TYPE2,                  &ADCD013,  { "eBNX^[ XvCc",                          "TWINKLE STAR SPRITES",                         }},
	{ "maglord",  "ADCD-100",  1990, "SNK/ADK",       NEOGEO,      LOADING_TYPE2,                  &ADCD100,  { "Magican Lord",                                         NULL,                                           }},
	{ "ncombat",  "ADCD-101",  1990, "SNK/ADK",       RASTER_FAST, LOADING_TYPE2,                  &ADCD101,  { "NINJA COMBAT",                                         NULL,                                           }},
	{ "crsword2", "ADCD-102",  1995, "SNK/ADK",       NEOGEO,      LOADING_TYPE2 | PATCH_CRSWORD2, &ADCD102,  { "CROSSED SWORDS U",                                    "CROSSED SWORDS II",                            }},
	{ "adkworld", "ADCD-103",  1995, "SNK/ADK",       NEOGEO,      LOADING_TYPE2 | PATCH_ADKWORLD, &ADCD103,  { "ADK[h",                                          "ADK WORLD",                                    }},
	{ "zintrick", "ADCD-104",  1996, "SNK/ADK",       NEOGEO,      LOADING_TYPE2,                  &ADCD104,  { "oWgbN",                                 "ZINTRICK",                                     }},

	{ "janshin",  "AICD-048",  1994, "AICOM",         RASTER,      LOADING_TYPE2,                  &AICD048,  { "_`",                                             "jyanshin densetsu",                            }},
	{ "viewpoin", "AICD-051",  1992, "SUMMY/AICOM",   RASTER_FAST, LOADING_TYPE2,                  &AICD051,  { "VIEWPOINT" ,                                           NULL,                                           }},

	{ "wjammers", "DECO-002",  1994, "DATA EAST",     RASTER_FAST, LOADING_TYPE2,                  &DECO002,  { "FLYING POWER DISC",                                    "WINDJAMMERS",                                  }},
	{ "karnovr",  "DECO-003",  1994, "DATA EAST",     RASTER_FAST, LOADING_TYPE2,                  &DECO003,  { "FIGHTER'S HISTORY DYNAMITE",                           "KARNOV'S REVENGE",                             }},
	{ "strhoop",  "DECO-004",  1994, "DATA EAST",     RASTER_FAST, LOADING_TYPE2,                  &DECO004,  { "DUNK DREAM",                                           "STREET HOOP",                                  "STREET SLAM" }},

	{ NULL,       "CD-T02EMU", 2005, "SNK",           NEOGEO,      LOADING_TYPE2,                  NULL,      { APPNAME,                                                NULL,                                           }}
};
