/***************************************************************************

	cpuintrf.h

	Core CPU interface functions and definitions.

***************************************************************************/

#ifndef CPUINTRF_H
#define CPUINTRF_H


/*************************************
 *
 *	Enum listing all the CPUs
 *
 *************************************/

enum
{
	CPU_M68000 = 0,
	CPU_Z80,
	MAX_CPU
};



/*************************************
 *
 *	CPU flag constants
 *
 *************************************/

enum
{
	/* flags for CPU go into upper byte */
	CPU_FLAGS_MASK = 0xff00,

	/* set this if the CPU is used as a slave for audio. It will not be emulated if */
	/* sound is disabled, therefore speeding up a lot the emulation. */
	CPU_AUDIO_CPU = 0x8000,

	/* the Z80 can be wired to use 16 bit addressing for I/O ports */
	CPU_16BIT_PORT = 0x4000
};



/*************************************
 *
 *	Interrupt line constants
 *
 *************************************/

enum
{
	/* line states */
	CLEAR_LINE = 0,				/* clear (a fired, held or pulsed) line */
	ASSERT_LINE,				/* assert an interrupt immediately */
	HOLD_LINE,					/* hold interrupt line until acknowledged */
	PULSE_LINE,					/* pulse interrupt line for one instruction */

	/* internal flags (not for use by drivers!) */
	INTERNAL_CLEAR_LINE = 100 + CLEAR_LINE,
	INTERNAL_ASSERT_LINE = 100 + ASSERT_LINE,

	/* interrupt parameters */
	MAX_IRQ_LINES =	8,			/* maximum number of IRQ lines per CPU */
	IRQ_LINE_NMI = 127			/* IRQ line for NMIs */
};



/*************************************
 *
 *	CPU information constants
 *
 *************************************/

/* get_reg/set_reg constants */
enum
{
	MAX_REGS = 128,				/* maximum number of register of any CPU */

	/* This value is passed to activecpu_get_reg to retrieve the previous
	 * program counter value, ie. before a CPU emulation started
	 * to fetch opcodes and arguments for the current instrution. */
	REG_PREVIOUSPC = -1,

	/* This value is passed to activecpu_get_reg to retrieve the current
	 * program counter value. */
	REG_PC = -2,

	/* This value is passed to activecpu_get_reg to retrieve the current
	 * stack pointer value. */
	REG_SP = -3,

	/* This value is passed to activecpu_get_reg/activecpu_set_reg, instead of one of
	 * the names from the enum a CPU core defines for it's registers,
	 * to get or set the contents of the memory pointed to by a stack pointer.
	 * You can specify the n'th element on the stack by (REG_SP_CONTENTS-n),
	 * ie. lower negative values. The actual element size (UINT16 or UINT32)
	 * depends on the CPU core. */
	REG_SP_CONTENTS = -4
};



/*************************************
 *
 *	Core CPU interface structure
 *
 *************************************/

struct cpu_interface
{
	/* type (used to make sure we mach the enum above */
	int			cpu_type;

	/* CPU description for drivers */
	int			cpu_clock;					/* in Hertz */
	void		(*vblank_interrupt)(void);	/* for interrupts tied to VBLANK */
	int 		vblank_interrupts_per_frame;/* usually 1 */

	/* table of core functions */
	void		(*init)(void);
	void		(*reset)(int param);
	void		(*exit)(void);
	int			(*execute)(int cycles);
	void		(*burn)(int cycles);
	void		(*set_irq_line)(int line, int state);
	void		(*set_irq_callback)(int(*callback)(int irqline));

	/* IRQ and clock information */
	unsigned	num_irqs;
	int			default_vector;
	int			*icount;
	double		overclock;
	int			irq_int;
};


extern struct cpu_interface cpuintf[];
extern int cpu_reset_flag;



/*************************************
 *
 *	Core CPU execution
 *
 *************************************/

/* Prepare CPUs for execution */
void cpu_init(void);

void cpu_change_driver(void);

/* Run CPUs until the user quits */
void cpu_run(void);

/* Clean up after quitting */
void cpu_exit(void);

/* Force a reset after the current timeslice */
void machine_reset(void);



/*************************************
 *
 *	Watchdog reser
 *
 *************************************/

/* 16-bit watchdog read/write handlers */
WRITE16_HANDLER( watchdog_reset_16_w );



/*************************************
 *
 *	CPU halt/reset lines
 *
 *************************************/

/* Set the logical state (ASSERT_LINE/CLEAR_LINE) of the HALT line on a CPU */
void cpu_set_halt_line(int cpu,int state);

/* Returns status (1=running, 0=suspended for some reason) of a CPU */
int cpu_getstatus(int cpunum);



/*************************************
 *
 *	Interrupt handling
 *
 *************************************/

/* Install a driver callback for IRQ acknowledge */
void cpu_set_irq_callback(int cpunum, int (*callback)(int irqline));

/* set the IRQ line state for a specific irq line of a CPU */
/* normally use state HOLD_LINE, irqline 0 for first IRQ type of a cpu */
void cpu_set_irq_line(int cpunum, int irqline, int state);



/*************************************
 *
 *	Synchronization
 *
 *************************************/

/* generate a trigger now */
void cpu_trigger(int trigger);

/* generate a trigger corresponding to an interrupt on the given CPU */
void cpu_triggerint(int cpunum);

/* Burn/yield CPU cycles for a specific period of time */
void cpu_spinuntil_time(double duration);



/*************************************
 *
 *	Core timing
 *
 *************************************/

/* Returns the number of times the interrupt handler will be called before
   the end of the current video frame. This is can be useful to interrupt
   handlers to synchronize their operation. If you call this from outside
   an interrupt handler, add 1 to the result, i.e. if it returns 0, it means
   that the interrupt handler will be called once. */
int cpu_getiloops(void);


#endif
