/***************************************************************************

	razeintf.c

	Handling RAZE Z80 core.

***************************************************************************/

#include "neogeocd.h"

static int cpu_enable;
static UINT8 cpu_context[7256];
static UINT8 z80_port_read(UINT16 port);
static void  z80_port_write(UINT16 port, UINT8 value);


void z80_init(void)
{
	UINT8 *mem = memory_region(REGION_CPU2);

	/* Map memory and port handlers */
	z80_init_memmap();
	z80_map_fetch(0x0000, 0xffff, mem);
	z80_add_read(0x0000, 0xffff, Z80_MAP_DIRECT, mem);
	z80_add_write(0x0000, 0xffff, Z80_MAP_DIRECT, mem);
	z80_set_in(&z80_port_read);
	z80_set_out(&z80_port_write);
	z80_end_memmap();

	cpu_enable = 0;
}


void z80_reset(int param)
{
	z80_Reset();

	cpu_enable = param;
}


void z80_exit(void)
{
	/* nothing to do ? */
}

int z80_execute(int cycles)
{
	if (cpu_enable)
		return z80_emulate(cycles);
	else
		return cycles;
}


void z80_burn(int cycles)
{
	z80_do_wait_states(cycles);
}


void z80_assert_irq(int irqline)
{
	if (irqline == IRQ_LINE_NMI)
	{
		int save_icount;

		z80_cause_NMI();

		save_icount = z80_ICount;
		// ̂炢܂킳ȂƁA_uhS̉EEE
		z80_execute(4000);
		z80_ICount = save_icount;
	}
	else
	{
		z80_raise_IRQ(irqline);
	}
}


void z80_clear_irq(int irqline)
{
	z80_lower_IRQ();
}


void z80_set_irq_line(int irqline, int state)
{
	switch (state)
	{
		case CLEAR_LINE:
			z80_clear_irq(irqline);
			return;
		case ASSERT_LINE:
			z80_assert_irq(irqline);
			return;
		default:
			z80_assert_irq(irqline);
			return;
	}
}


void z80_set_irq_callback(int (*callback)(int irqline))
{
	// dummy
}


static UINT8 z80_port_read(UINT16 port)
{
	return neogeo_z80_port_r(port);
}


static void z80_port_write(UINT16 port, UINT8 value)
{
	neogeo_z80_port_w(port, value);
}


void z80_enable(int enable)
{
	cpu_enable = enable;
}


void z80_save_context(void)
{
	z80_get_context(cpu_context);
}

void z80_restore_context(void)
{
	z80_set_context(cpu_context);
}
