#include <string.h>
#include "gba.h"
#include "minilzo.107/minilzo.h"

#define STATEID 0x57a731dc

#define STATESAVE 0
#define SRAMSAVE 1
#define CONFIGSAVE 2

#define STATESIZE 0xE140					//from equates.h
#define STATEPTR (u8*)0x2040000-STATESIZE	//from equates.h

extern u8 Image$$RO$$Limit;
extern u8 g_cartflags;		//(from SMS header)
extern char g_scaling;		//(cart.s) current display mode
extern char g_scaling_set;	//(cart.s) display mode, setting
extern char g_config;		//(cart.s) current bios setting
extern char g_flicker;		//(gfx.s)
extern char gammavalue;		//(gfx.s) current gammavalue
extern char bcolor;			//(gfx.s) border color
extern char sprcollision;	//(z80.s) sprite collision on/off
//extern u32 soundmode;		//(sound.s) current soundmode
extern u8 stime;			//from ui.c
extern u8 autostate;		//from ui.c
extern u8 *textstart;		//from main.c

extern char pogoshell;

int totalstatesize;			//how much SRAM is used

//-------------------
void sleepset(void);						//ui.c
u8 *findrom(int);
void cls(int);								//main.c
void drawtext(int,char*,int);
void setdarknessgs(int dark);
void scrolll(int offset, int f);
void scrollr(int offset);
void waitframe(void);
u32 getmenuinput(int);
void writeconfig(void);
//void setup_sram_after_loadstate(void);

extern int roms;							//main.c
extern int selected;						//ui.c
extern char pogoshell_romname[32];			//main.c
//----asm stuff------
int savestate(void);						//cart.s
void loadstate(int);						//cart.s
void bytecopy_(u8 *dst,u8 *src,int count);	//memory.s

extern u8 *romstart;						//from cart.s
extern u32 romnum;							//from cart.s
extern u32 frametotal;						//from z80.s
//-------------------

typedef struct {
	u16 size;			//header+data
	u16 type;			//=STATESAVE or SRAMSAVE
	u32 compressedsize;
	u32 framecount;
	u32 checksum;
	char title[32];
} stateheader;

typedef struct {		//(modified stateheader)
	u16 size;
	u16 type;			//=CONFIGSAVE
	char displaytype;
	char gammavalue;
	char soundmode;
	char sleepflick;
	char config;
	char bcolor;
	char reserved1;
	char reserved2;
	u32 sram_checksum;	//checksum of rom using SRAM e000-ffff	
	u32 zero;	//=0
	char reserved3[32];	//="CFG"
} configdata;

//we have a big chunk of memory starting at Image$$RO$$Limit free to use
#define BUFFER1 (&Image$$RO$$Limit)
#define BUFFER2 (&Image$$RO$$Limit+0x10000)


int using_flashcart() {
	return (u32)textstart&0x8000000;
}

void getsram() {		//copy GBA sram to BUFFER1
	u8 *sram=MEM_SRAM;
	u8 *buff1=BUFFER1;
	u32 *p;

	p=(u32*)buff1;
	if(*p!=STATEID) {	//if sram hasn't been copied already
		bytecopy_(buff1,sram,0x10000);	//copy everything to BUFFER1
		if(*p!=STATEID) {	//valid savestate data?
			*p=STATEID;	//nope.  initialize
			*(p+1)=0;
		}
	}
}

//quick & dirty rom checksum
u32 checksum(u8 *p) {
	u32 sum=0;
	int i;
	for(i=0;i<128;i++) {
		sum+=*p|(*(p+1)<<8)|(*(p+2)<<16)|(*(p+3)<<24);
		p+=128;
	}
	return sum;
}

void writeerror(int scr) {
	int i;
	cls(1+scr);
	drawtext(9+(scr<<5), "  Write error! Memory full.",0);
	drawtext(10+(scr<<5),"     Delete some games.",0);
	for(i=90;i;--i)
		waitframe();
}

//(BUFFER1=copy of GBA SRAM, BUFFER2=new data)
//overwrite:  index=state#, erase=0
//new:  index=big number (anything >=total saves), erase=0
//erase:  index=state#, erase=1
//returns TRUE if successful
//IMPORTANT!!! totalstatesize is assumed to be current
int updatestates(int index,int erase,int type) {
	int i;
	int srcsize;
	int total=totalstatesize;
	u8 *src=BUFFER1+4;//skip STATEID
	u8 *dst=BUFFER1+4;
	stateheader *newdata=(stateheader*)BUFFER2;


	//skip ahead to where we want to write

	srcsize=((stateheader*)dst)->size;
	i=(type==((stateheader*)dst)->type)?0:-1;
	while(i<index && srcsize) {	//while (looking for state) && (not out of data)
		dst+=srcsize;
		srcsize=((stateheader*)dst)->size;
		if(((stateheader*)dst)->type==type)
			i++;
	}
	src = dst+srcsize;

	//write new data

	total-=srcsize;
	if(!erase) {
		i=newdata->size;
		total+=i;
		if(total>0x10000) //**OUT OF MEMORY**
			return 0;
		src=(u8*)newdata + newdata->size;
		//copy trailing data to 2nd buffer
		memcpy(src,dst+((stateheader*)dst)->size,total-i);
		//overwrite
		memcpy(dst,newdata,i);
		dst+=i;
	}

	srcsize=((stateheader*)src)->size;

	//get trailing data

	while(srcsize) {
		memcpy(dst,src,srcsize);
		src+=srcsize;
		dst+=srcsize;
		srcsize=((stateheader*)src)->size;
	}
	*(u32*)dst=0;	//terminate
	*(u32*)(dst+4)=0xffffffff;	//terminate
	dst+=8;
	total+=8;

	//copy everything to GBA sram

	totalstatesize=total;
	while(total<0x10000)
	{
		*dst++=0;
		total++;
	}
	bytecopy_(MEM_SRAM,BUFFER1,total);	//copy to sram
	return 1;
}

//more dumb stuff so we don't waste space by using sprintf
int twodigits(int n,char *s) {
	int mod=n%10;
	n=n/10;
	*(s++)=(n+'0');
	*s=(mod+'0');
	return n;
}

void getstatetimeandsize(char *s,int time,u32 size,u32 totalsize) {
	strcpy(s,"00:00:00 - 00/00k");
	twodigits(time/216000,s);
	s+=3;
	twodigits((time/3600)%60,s);
	s+=3;
	twodigits((time/60)%60,s);
	s+=5;
	twodigits(size/1024,s);
	s+=3;
	twodigits(totalsize/1024,s);
}

#define LOADMENU 0
#define SAVEMENU 1
#define SRAMMENU 2
#define FIRSTLINE 2
#define LASTLINE 16

//BUFFER1 holds copy of SRAM
//draw save/loadstate menu and update global totalstatesize
//returns a pointer to current selected state
//update *states on exit
stateheader* drawstates(int menutype,int *menuitems,int *menuoffset) {
	int type;
	int offset=*menuoffset;
	int sel=selected;
	int startline;
	int size;
	int statecount;
	int total;
	char s[30];
	stateheader *selectedstate;
	int time;
	int selectedstatesize;
	stateheader *sh=(stateheader*)(BUFFER1+4);

	type=(menutype==SRAMMENU)?SRAMSAVE:STATESAVE;

	statecount=*menuitems;
	if(sel-offset>LASTLINE-FIRSTLINE-3 && statecount>LASTLINE-FIRSTLINE+1) {		//scroll down
		offset=sel-(LASTLINE-FIRSTLINE-3);
		if(offset>statecount-(LASTLINE-FIRSTLINE+1))	//hit bottom
			offset=statecount-(LASTLINE-FIRSTLINE+1);
	}
	if(sel-offset<3) {				//scroll up
		offset=sel-3;
		if(offset<0)					//hit top
			offset=0;
	}
	*menuoffset=offset;
	
	startline=FIRSTLINE-offset;
	cls(1);
	statecount=0;
	total=8;	//header+null terminator
	while(sh->size) {
		size=sh->size;
		if(sh->type==type) {
			if(startline+statecount>=FIRSTLINE && startline+statecount<=LASTLINE) {
				drawtext(startline+statecount,sh->title,sel==statecount);
			}
			if(sel==statecount) {		//keep info for selected state
				time=sh->framecount;
				selectedstatesize=size;
				selectedstate=sh;
			}
			statecount++;
		}
		total+=size;
		sh=(stateheader*)((u8*)sh+size);
	}

	if(sel!=statecount) {//not <NEW>
		getstatetimeandsize(s,time,selectedstatesize,total);
		drawtext(18,s,0);
	}
	if(statecount)
		drawtext(19,"Push SELECT to delete",0);
	if(menutype==SAVEMENU) {
		if(startline+statecount<=LASTLINE)
			drawtext(startline+statecount,"<NEW>",sel==statecount);
		drawtext(0,"Save state:",0);
		statecount++;	//include <NEW> as a menuitem
	} else if(menutype==LOADMENU) {
		drawtext(0,"Load state:",0);
	} else {
		drawtext(0,"Erase SRAM:",0);
	}
	*menuitems=statecount;
	totalstatesize=total;
	return selectedstate;
}

//compress src into BUFFER2 (adding header), using 64k of workspace
void compressstate(lzo_uint size,u16 type,u8 *src,void *workspace) {
	lzo_uint compressedsize;
	stateheader *sh;

	lzo1x_1_compress(src,size,BUFFER2+sizeof(stateheader),&compressedsize,workspace);	//workspace needs to be 64k

	//setup header:
	sh=(stateheader*)BUFFER2;
	sh->size=(compressedsize+sizeof(stateheader)+3)&~3;	//size of compressed state+header, word aligned
	sh->type=type;
	sh->compressedsize=compressedsize;	//size of compressed state
	sh->framecount=frametotal;
	sh->checksum=checksum((u8*)romstart);	//checksum
    if(pogoshell)
    {
		strcpy(sh->title,pogoshell_romname);
    }
    else
    {
		strcpy(sh->title,(char*)findrom(romnum)+32);
    }
}

void managesram() {
	int i;
	int menuitems;
	int offset=0;

	getsram();

	selected=0;
	drawstates(SRAMMENU,&menuitems,&offset);
	if(!menuitems)
		return;		//nothing to do!

	scrolll(0x100,0);
	do {
		i=getmenuinput(menuitems);
		if(i&SELECT) {
			updatestates(selected,1,SRAMSAVE);
			if(selected==menuitems-1) selected--;	//deleted last entry.. move up one
		}
		if(i&(SELECT+UP+DOWN+LEFT+RIGHT))
			drawstates(SRAMMENU,&menuitems,&offset);
	} while(menuitems && !(i&(L_BTN+R_BTN+B_BTN)));
	scrollr(0x100);
}

void savestatemenu() {
	int i;
	int menuitems;
	int offset=0;

	i=savestate();
	compressstate(i,STATESAVE,STATEPTR,BUFFER1);

	getsram();

	selected=0;
	drawstates(SAVEMENU,&menuitems,&offset);
	scrolll(0x100,0);
	do {
		i=getmenuinput(menuitems);
		if(i&(A_BTN)) {
			if(!updatestates(selected,0,STATESAVE))
				writeerror(0);
		}
		if(i&SELECT)
			updatestates(selected,1,STATESAVE);
		if(i&(SELECT+UP+DOWN+LEFT+RIGHT))
			drawstates(SAVEMENU,&menuitems,&offset);
	} while(!(i&(L_BTN+R_BTN+A_BTN+B_BTN)));
	scrollr(0x100);
}

//locate last save by checksum
//returns save index (-1 if not found) and updates stateptr
//updates totalstatesize (so quicksave can use updatestates)
int findstate(u32 checksum,int type,stateheader **stateptr) {
	int state,size,foundstate,total;
	stateheader *sh;

	getsram();
	sh=(stateheader*)(BUFFER1+4);

	state=-1;
	foundstate=-1;
	total=8;
	size=sh->size;
	while(size) {
		if(sh->type==type) {
			state++;
			if(sh->checksum==checksum) {
				foundstate=state;
				*stateptr=sh;
			}
		}
		total+=size;
		sh=(stateheader*)(((u8*)sh)+size);
		size=sh->size;
	}
	totalstatesize=total;
	return foundstate;
}

void uncompressstate(int rom,stateheader *sh) {
	lzo_uint statesize=sh->compressedsize;
	lzo1x_decompress((u8*)(sh+1),statesize,STATEPTR,&statesize,NULL);
	loadstate(rom);
	frametotal=sh->framecount;			//restore global frame counter
//	setup_sram_after_loadstate();		//handle sram packing
}


void quickload() {
	stateheader *sh;
	int i;

	if(!using_flashcart())
		return;

	i=findstate(checksum((u8*)romstart),STATESAVE,&sh);
	if(i>=0)
		uncompressstate(romnum,sh);
}

void quicksave() {
	stateheader *sh;
	int i;

	if(!using_flashcart())
		return;

	cls(2);
	setdarknessgs(7);	//darken
	drawtext(32+9,"           Saving.",0);

	i=savestate();
	compressstate(i,STATESAVE,STATEPTR,BUFFER1);
	i=findstate(checksum((u8*)romstart),STATESAVE,&sh);
	if(i<0) i=65536;	//make new save if one doesn't exist
	if(!updatestates(i,0,STATESAVE))
		writeerror(1);
	cls(2);
}

//make new saved sram (using SMS_SRAM contents)
//this is to ensure that we have all info for this rom and can save it even after this rom is removed
void save_sms_sram() {
	stateheader *sh;
	int i;

	if(!using_flashcart())
		return;

	cls(2);
	drawtext(32+9,"           Saving.",0);

	compressstate(0x8000,SRAMSAVE,SMS_SRAM,BUFFER1);
	i=findstate(checksum((u8*)romstart),SRAMSAVE,&sh);	//see if packed SRAM exists
	if(i<0) i=65536;	//make new save if one doesn't exist
	if(!updatestates(i,0,SRAMSAVE))
		writeerror(1);
}

void load_sms_sram() {
	int i;
	u32 chk;
	stateheader *sh;
	lzo_uint statesize;

	if(!using_flashcart())
		return;

	chk=checksum(romstart);
	i=findstate(chk,SRAMSAVE,&sh);	//see if packed SRAM exists

	if(i>=0) {//packed SRAM exists: unpack into SMS_SRAM
		statesize=sh->compressedsize;
		lzo1x_decompress((u8*)(sh+1),statesize,SMS_SRAM,&statesize,NULL);
	}
}


void loadstatemenu() {
	stateheader *sh;
	u32 key;
	int i;
	int offset=0;
	int menuitems;
	u32 sum;

	getsram();

	selected=0;
	sh=drawstates(LOADMENU,&menuitems,&offset);
	if(!menuitems)
		return;		//nothing to load!

	scrolll(0x100,0);
	do {
		key=getmenuinput(menuitems);
		if(key&(A_BTN)) {
			sum=sh->checksum;
			i=0;
			do {
				if(sum==checksum(findrom(i)+sizeof(romheader))) {	//find rom with matching checksum
					uncompressstate(i,sh);
					i=8192;
				}
				i++;
			} while(i<roms);
			if(i<8192) {
				cls(2);
				drawtext(9,"       ROM not found.",0);
				for(i=0;i<60;i++)	//(1 second wait)
					waitframe();
			}
		} else if(key&SELECT) {
			updatestates(selected,1,STATESAVE);
			if(selected==menuitems-1) selected--;	//deleted last entry? move up one
		}
		if(key&(SELECT+UP+DOWN+LEFT+RIGHT))
			sh=drawstates(LOADMENU,&menuitems,&offset);
	} while(menuitems && !(key&(L_BTN+R_BTN+A_BTN+B_BTN)));
	scrollr(0x100);
}


const configdata configtemplate={
	sizeof(configdata),
	CONFIGSAVE,
	3,2,0,0,0x40,0,0,0,0,0,
	"CFG"
};

void writeconfig() {
	configdata *cfg;
	int i,j;

	if(!using_flashcart())
		return;

	i=findstate(0,CONFIGSAVE,(stateheader**)&cfg);
	if(i<0) {//make new config
		memcpy(BUFFER2,&configtemplate,sizeof(configdata));
		cfg=(configdata*)BUFFER2;
	}
	cfg->config      = g_config;				//store current bios setting
	cfg->bcolor      = bcolor;					//store current border color
	cfg->displaytype = g_scaling_set;			//store current display type
	cfg->gammavalue  = gammavalue;				//store current gammavalue
//	cfg->soundmode   = (char)soundmode;			//store current soundmode
	j = stime & 0xF;							//store current autosleep time
	j |= ((g_flicker & 0x1)^1)<<4;				//store current flicker setting
	j |= ((sprcollision & 0x20)^0x20);			//store current sprite collision
	j |= (autostate & 0x1)<<6;					//store current autostate setting
	cfg->sleepflick = j;

	if(i<0) {	//create new config
		updatestates(0,0,CONFIGSAVE);
	} else {		//config already exists, update sram directly (faster)
		bytecopy_((u8*)cfg-BUFFER1+MEM_SRAM,(u8*)cfg,sizeof(configdata));
	}
}

void readconfig() {
	int j;
	const configdata *cfg;
	if(!using_flashcart())
		return;

	j=findstate(0,CONFIGSAVE,(stateheader**)&cfg);
	if(j<0) {
		cfg = &configtemplate;
	}

	g_config        = cfg->config;				//restore current bios setting
	bcolor          = cfg->bcolor;				//restore current border color
	g_scaling       = g_scaling_set = cfg->displaytype;
	gammavalue	    = cfg->gammavalue;			//restore gamma value
//	soundmode       = (u32)cfg->soundmode;
	j = cfg->sleepflick;
	g_flicker = ((j & 0x10)^0x10)>>4;			//restore current flicker setting
	sprcollision = ((j & 0x20)^0x20);			//restore current sprite collision
	autostate = (j & 0x40)>>6;					//restore autostate setting
	stime = ((j-1) & 0x3);						//restore current autosleep time
	sleepset();

}

