#include "EFPluginManager.hpp"
#include "EFException.hpp"
#include <io.h>

namespace EFramework
{
	EFPluginManager* EFSingleton<EFPluginManager>::mSingleton = NULL;

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	EFPluginManager::EFPluginManager()
	{
		mLoadedPlugins.clear();
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	EFPluginManager::~EFPluginManager()
	{
		UnloadPlugins();
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	void EFPluginManager::LoadPlugins(const EFString& directory)
	{
		//======================================================
		// Retrieve the names of all the files in the directory.
		//======================================================
		std::vector< EFString > files;
		GetFileNames(directory, files);

		//======================================================
		// Load all of the files that were found as plugins.
		//======================================================
		std::vector< EFString >::const_iterator iter;
		for(iter = files.begin(); iter != files.end(); iter++)
		{
			LoadPlugin(directory, *iter);
		}
	}

	void EFPluginManager::LoadPlugin(const EFString& directory, const EFString& file)
	{
		//======================================================
		// Attempt to load the specified library file.
		//======================================================
		EFString fullName = directory + EFString("\\") + file;
		HMODULE LibHandle = ::LoadLibrary( fullName.c_str() );

		//======================================================
		// If an error occurred then thrown an exception.
		//======================================================
		if(LibHandle == NULL)
		{
			LPVOID lpMsgBuf;
			::FormatMessage(
				FORMAT_MESSAGE_ALLOCATE_BUFFER | 
				FORMAT_MESSAGE_FROM_SYSTEM     | 
				FORMAT_MESSAGE_IGNORE_INSERTS,
				NULL,
				GetLastError(),
				MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
				(LPTSTR) &lpMsgBuf,
				0,
				NULL 
			);

			::OutputDebugString( static_cast< LPCSTR >( lpMsgBuf ) );

			::LocalFree( lpMsgBuf );
			EF_THROW(EFException::EC_PLUGIN_ERROR, "Could not load dynamic link library " + file);
		}

		//======================================================
		// Add this plugins information to our vector.
		//======================================================
		PluginData data;
		data.FileName  = file.c_str();
		data.DllHandle = LibHandle;
		mLoadedPlugins.push_back( data );
	}

	void EFPluginManager::UnloadPlugins()
	{
		//======================================================
		// Loop through and free all loaded libraries.
		//======================================================
		std::vector< PluginData >::const_iterator iter;
		for(iter = mLoadedPlugins.begin(); iter != mLoadedPlugins.end(); iter++)
		{
			::FreeLibrary( iter->DllHandle );
		}

		mLoadedPlugins.clear();
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	EFString EFPluginManager::GetPluginFileName(ulong index) const
	{
		//======================================================
		// Ensure the index is within the appropriate bounds.
		//======================================================
		if(mLoadedPlugins.size() < index)
			return "NULL";

		//======================================================
		// Loop through the directory until no files are left.
		//======================================================
		return EFString( mLoadedPlugins.at( index ).FileName );
	}

	ulong EFPluginManager::GetNumberOfPlugins() const
	{
		return static_cast< ulong >( mLoadedPlugins.size() );
	}

	void* EFPluginManager::GetProcedure(const EFString& plugin, const EFString& procedureName)
	{
		//======================================================
		// Loop until we find the specified library file.
		//======================================================
		std::vector< PluginData >::const_iterator iter;
		for(iter = mLoadedPlugins.begin(); iter != mLoadedPlugins.end(); iter++)
		{
			if(EFString(iter->FileName) == plugin)
			{
				return static_cast< void* >( ::GetProcAddress(iter->DllHandle, procedureName.c_str()) );
			}
		}

		//======================================================
		// We couldn't find it so we'll return a null pointer.
		//======================================================
		return NULL;
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	void EFPluginManager::GetFileNames(const EFString& directory, std::vector< EFString >& fileNames)
	{
		//======================================================
		// Constructor our search mask.
		//======================================================
		EFString searchMask = directory + EFString("\\*.dll");

		//======================================================
		// Find the first file meeting our criteria.
		//======================================================
		struct _finddata_t fileInfo;
		long handle = static_cast< long >( ::_findfirst(searchMask.c_str(), &fileInfo) );
		long file   = handle;

		//======================================================
		// Loop through the directory until no files are left.
		//======================================================
		while(file >= 0)
		{
			fileNames.push_back( EFString( fileInfo.name ) );
			file = ::_findnext(handle, &fileInfo);
		}

		//======================================================
		// Perform our cleanup here.
		//======================================================
		::_findclose(handle);
	}
} // Namespace EFramework