#ifndef EF_WINDOW_APPLICATION_HPP
#define EF_WINDOW_APPLICATION_HPP

#include <atlbase.h>
#include <atlapp.h>
#include "EFException.hpp"

namespace EFramework
{
	/// Wraps the Windows Template Library (WTL) application class.
	template<class TWindowClass>
	class EFWindowApplication
	{
	public:
		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Constructors

		/** The default constructor.
		  */
		EFWindowApplication(HINSTANCE hInstance)
		{
			Initialize(hInstance);
		}

		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Constructors

		/** The default destructor.
		  */
		~EFWindowApplication()
		{
			Shutdown();
		}

		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Application Configuration Functions

		/** Adds a message processing loop to the application.
		  */
		void AddMessageProcessingLoop(CMessageLoop loop, bool removeCurrentLoop = false)
		{
			//========================================================================
			// Remove the default message loop if this action is requested.
			//========================================================================
			if(removeCurrentLoop)
			{
				mWindowModule.RemoveMessageLoop();
			}

			//========================================================================
			// Add the message loop item passed to the function.
			//========================================================================
			if(!mWindowModule.AddMessageLoop(&loop))
			{
				EF_THROW(EFException::EC_GUI_ERROR, "Could not add message processing loop to application module.");
			}
		}
		/** Adds an idle time handler to the loop processor.
		  */
		void AddIdleHandler(CIdleHandler* handler)
		{
			//========================================================================
			// Add a new idle time handler to the current message processing loop.
			//========================================================================
			if(!(mWindowModule.GetMessageLoop()->AddIdleHandler(handler)))
			{
				EF_THROW(EFException::EC_GUI_ERROR, "Could not add an idle time handler.");
			}
		}

		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Window Execution Functions

		/** Manages all areas of application initialization and shutdown.
		  */
		int Run()
		{
			//========================================================================
			// Create an instance of our window class and run the message loop.
			//========================================================================
			TWindowClass tWindow;
			return (mWindowModule.GetMessageLoop()->Run());
		}
	protected:
		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Initialization and Shutdown Functions

		/** Initializes the WTL application.
		  */
		void Initialize(HINSTANCE hInstance)
		{
			//========================================================================
			// Attempt to initialization COM on this thread.
			//========================================================================
			if(!SUCCEEDED(::CoInitialize(NULL)))
				EF_THROW(EFException::EC_FATAL_ERROR, "Could not initialize the COM architecture.");

			//========================================================================
			// Initialize the common control system.
			//========================================================================
			::InitCommonControls();

			//========================================================================
			// Attempt to initialize the application module.
			//========================================================================
			if(!SUCCEEDED(mWindowModule.Init(NULL, hInstance, NULL)))
			{
				EF_THROW(EFException::EC_FATAL_ERROR, "Could not initialize CAppModule.");
			}

			//========================================================================
			// Attempt to add a message processing loop to our application.
			//========================================================================
			if(!mWindowModule.AddMessageLoop(&mMessageLoop))
			{
				EF_THROW(EFException::EC_FATAL_ERROR, "Could not add message processing loop.");
			}
		}
		/** Shuts down the WTL application.
		  */
		void Shutdown()
		{
			//========================================================================
			// Attempt to remove the message loop for our application.
			//========================================================================
			mWindowModule.RemoveMessageLoop();

			//========================================================================
			// Terminate the WTL application module.
			//========================================================================
			mWindowModule.Term();

			//========================================================================
			// Uninitialize the COM architecture on this thread.
			//========================================================================
			::CoUninitialize();
		}

		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Variables

		CMessageLoop mMessageLoop;  ///< The windows message processing loop.
		CAppModule   mWindowModule; ///< The windows template library module.
	};
} // Namespace EFramework

#endif // EF_WINDOW_APPLICATION_HPP