#ifndef EF_TRACE_HPP
#define EF_TRACE_HPP

#include "EFPlatform.hpp"

namespace EFramework
{
	/// Adds a trace of the current function to the stack.
	#define EF_TRACE(trace) { \
		trace.AddTrace(__FUNCTION__); \
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	/// Used to trace the execution of a program in a log file.
	class EF_EXPORT EFTrace
	{
	public:
		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Enumerations

		/// Miscellaneous information about the trace.
		enum TraceInformation
		{
			TRACE_DEPTH = 256
		};

		/// Predefined values for trace level detail.
		enum TraceLevels
		{
			TL_SHALLOW =  5,
			TL_DEEP    = 10,
			TL_DEEPEER = 15,
			TL_DEEPEST = (TRACE_DEPTH-1)
		};

		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Constructors

		/** The default constructor.
		  */
		EFTrace();

		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Tracing Functions

		/** Adds a trace function to our managed list of functions.
		  */
		void AddTrace(const EFString& function);
		/** Returns a string containing a trace of most recently called functions.
		  */
		EFString Trace(ulong TraceDepth = TL_SHALLOW);

		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Debug Mode Functions

		/** Sets the current debug mode for tracing.
		  */
		const void SetDebugMode(bool debugOn);
		/** Retrieves the current debug mode for tracing.
		  */
		bool GetDebugMode() const;
	protected:
		///////////////////////////////////////////////////////////////////////////////////////////////////
		// Variables

		bool     mDebugMode;           ///< The current debug mode for this trace item.
		uint32_t mStackPointer;        ///< The current point in the stack where data is being added.
		EFString mTraces[TRACE_DEPTH]; ///< The names of all the functions which have been traced.
	};
} // Namespace EFramework

#endif // EF_TRACE_HPP