#ifndef PSX_INTERPRETER_HPP
#define PSX_INTERPRETER_HPP

#include <list>
#include <memory>
#include "PSXMemory.hpp"
#include "PSXUtility.hpp"
#include "EFramework.hpp"

namespace NeoPSX
{
	/// The context for emulation of the CPU.
	struct TR3000Context
	{
		uint32_t  Reg[32];
		uint32_t  Cp0[17];
		uint32_t  Hi;
		uint32_t  Lo;
		uint32_t  Pc;
		uint32_t  Opcode;
		uint32_t  Cycle;
	};

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	/// The R3K CPU interpreter for the Playstation.
	class PSXInterpreter
	{
	public:
		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Constructors

		/** Initializes internal variables.
		  */
		PSXInterpreter();

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// CPU Emulation Functions

		/** Emulates a single CPU instruction.
		  */
		void Execute();
		/** Resets the emulator and its componenets.
		  */
		void Reset();

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// CPU Context Functions

		/** Retrieves the R3000 CPU context.
		  */
		TR3000Context& GetContext() const;
		/** Retrieves the PSXMemory class pointer.
		  */
		PSXMemory& GetMemory() const;

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// CPU Register Functions

		/** Sets the specified register to a specific value.
		  */
		const void SetRegisterGPR(uint32_t reg, uint32_t value);
		/** Retrieves the value in the specified register.
		  */
		const uint32_t GetRegisterGPR(uint32_t reg) const;

		/** Sets the program counter value.
		  */
		const void SetPC(uint32_t value);
		/** Retrieves the current program counter value.
		  */
		const uint32_t GetPC() const;
	private:
		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Helper Functions

		/** Pushes exception status register onto the stack.
		  */
		void  PushStatusCP0();
		/** Pops exception status register off top of stack.
		  */
		void PopStatusCP0();
		/** Performs and branch or jump instruction.
		  */
		void PerformBranchOrJump(uint32_t address, bool link = false, uint32_t linkRegister = 0);

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Macros

		//== Opcode Parameter Retrieval Macros ==//
		#define _Opcode_     (((mContext.Opcode)&0xFC000000)>>26)
		#define _Target_     ((mContext.Opcode)&0x03FFFFFF)
		#define _Shift_      (((mContext.Opcode)&0x000007C0)>>6)
		#define _UImmediate_ ((uint16_t)(mContext.Opcode))
		#define _SImmediate_ ((uint16_t)(mContext.Opcode))
		#define _Function_   ((mContext.Opcode)&0x0000003F)
		#define Rt           (((mContext.Opcode)>>16)&0x1F)
		#define Rs           (((mContext.Opcode)>>21)&0x1F)
		#define Rd           (((mContext.Opcode)>>11)&0x1F)
		#define _Rt_         (mContext.Reg[Rt])
		#define _Rs_         (mContext.Reg[Rs])
		#define _Rd_         (mContext.Reg[Rd])

		//== Jump Address Calculation ==//
		#define CALCULATE_JUMP_ADDRESS()   ((_Target_<<2) + (mContext.Pc&0xF0000000))
		#define CALCULATE_BRANCH_ADDRESS() ((_SImmediate_<<2) + mContext.Pc)

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// R3K Opcode Functions

		//----------------------------------------------------------
		// Jump Instructions
		//----------------------------------------------------------

		/** Jump instruction. */
		void cpuJ();
		/** Jump and link instruction. */
		void cpuJAL();
		/** Jump register instruction. */
		void cpuJR();
		/** Jump and link register instruction. */
		void cpuJALR();
		
		//----------------------------------------------------------
		// Branch Instructions
		//----------------------------------------------------------

		/** Branch on equal instruction. */
		void cpuBEQ();
		/** Branch on not equal instruction. */
		void cpuBNE();
		/** Branch on less than or equal to zero instruction. */
		void cpuBLEZ();
		/** Branch on greater than or equal to zero instruction */
		void cpuBGTZ();
		/** Branch on less than zero instruction. */
		void cpuBLTZ();
		/** Branch on greater than or equal to zero instruction. */
		void cpuBGEZ();
		/** Branch on less than zero and link instruction. */
		void cpuBLTZAL();
		/** Branch on greater than or equal to zero and link instruction. */
		void cpuBGEZAL();

		//----------------------------------------------------------
		// Immediate Instructions
		//----------------------------------------------------------

		/** Add immediate instruction. */
		void cpuADDI();
		/** Add immediate unsigned instruction. */
		void cpuADDIU();
		/** Set on less than immediate instruction. */
		void cpuSLTI();
		/** Set on less than immediate unsigned instruction. */
		void cpuSLTIU();
		/** AND immediate instruction. */
		void cpuANDI();
		/** OR immediate instruction. */
		void cpuORI();
		/** XOR immediate instruction. */
		void cpuXORI();
		/** Load upper immediate instruction. */
		void cpuLUI();

		//----------------------------------------------------------
		// Load Instructions
		//----------------------------------------------------------

		/** Load byte instruction. */
		void cpuLB();
		/** Load halfword instruction. */
		void cpuLH();
		/** Load word left instruction. */
		void cpuLWL();
		/** Load word instruction. */
		void cpuLW();
		/** Load byte unsigned. */
		void cpuLBU();
		/** Load word unsigned. */
		void cpuLHU();
		/** Load word right instruction. */
		void cpuLWR();

		//----------------------------------------------------------
		// Store Instructions
		//----------------------------------------------------------

		/** Store byte instruction. */
		void cpuSB();
		/** Store halfword instruction. */
		void cpuSH();
		/** Store word left instruction. */
		void cpuSWL();
		/** Store word instruction. */
		void cpuSW();
		/** Store word right instruction. */
		void cpuSWR();

		//----------------------------------------------------------
		// Shift Instructions
		//----------------------------------------------------------
		
		/** Shift left logical instruction. */
		void cpuSLL();
		/** Shift right logical instruction. */
		void cpuSRL();
		/** Shift right arithmetic instruction. */
		void cpuSRA();
		/** Shift left logical variable instruction. */
		void cpuSLLV();
		/** Shift right logical variable instruction. */
		void cpuSRLV();
		/** Shift right arithmetic variable instruction. */
		void cpuSRAV();

		//----------------------------------------------------------
		// Special Instructions
		//----------------------------------------------------------

		/** The SYSCALL instruction. */
		void cpuSYSCALL();
		/** The BREAK instruction. */
		void cpuBREAK();

		//----------------------------------------------------------
		// Multiply and Divide Instructions
		//----------------------------------------------------------

		/** Multiply instruction. */
		void cpuMULT();
		/** Multiply unsigned instruction. */
		void cpuMULTU();
		/** Divide instruction. */
		void cpuDIV();
		/** Divide unsigned instruction. */
		void cpuDIVU();
		/** Move from HI instruction. */
		void cpuMFHI();
		/** Move to HI instruction. */
		void cpuMTHI();
		/** Move from LO instruction. */
		void cpuMFLO();
		/** Move to LO instruction. */
		void cpuMTLO();

		//----------------------------------------------------------
		// Three Operand Register Type Instructions
		//----------------------------------------------------------

		/** Add instruction. */
		void cpuADD();
		/** Add unsigned instruction. */
		void cpuADDU();
		/** Subtract instruction. */
		void cpuSUB();
		/** Subtract unsigned instruction. */
		void cpuSUBU();
		/** And instruction. */
		void cpuAND();
		/** Or instruction. */
		void cpuOR();
		/** Xor instruction. */
		void cpuXOR();
		/** Nor instruction. */
		void cpuNOR();
		/** Set on less than instruction. */
		void cpuSLT();
		/** Set on less than unsigned instruction. */
		void cpuSLTU();

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Cop0 Opcode Functions

		/** Move from CoProcessor0 instruction. */
		void cpuMFC0();
		/** Move to CoProcessor0 instruction. */
		void cpuMTC0();

		/*********************************************************
		 * All TLB instructions have been stripped from PSX Cop0 *
		 *********************************************************/

		/** Restore from exception instruction. */
		void cpuRFE();

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Table Jump Functions

		/** The unknown CPU instruction function. */
		void cpuUNK();
		/** Jumps to the special instruction table. */
		void cpuSPECIAL();
		/** Jumps to the conditional branch instruction table. */
		void cpuBCOND();
		/** Jumps to COP0 instruction table. */
		void cpuCOP0();
		/** Jumps to COP2 instruction table. */
		void cpuCOP2();

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Private Variables

		TR3000Context            mContext;  ///< The context for the R3K CPU.
		std::auto_ptr<PSXMemory> mMemory;   ///< The playstation memory area, emulated.
		std::list<uint32_t>      mCp0Stack; ///< The CoProcessor0 status register stack.

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Private Typedefs

		typedef ClassFunctionPointer<PSXInterpreter>::ResultType FUNCPTR;

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Private Enumerations

		enum RegisterDefinitions
		{
			RD_COP0_STATUS = 12,
			RD_COP0_CAUSE  = 13,
			RD_COP0_EPC    = 14,
			RD_COP0_PRID   = 15
		};

		/////////////////////////////////////////////////////////////////////////////////////////////////
		// Private Function Pointer Tables

		static const FUNCPTR mTableBASIC    [64]; ///< The basic opcode function table.
		static const FUNCPTR mTableSPECIAL  [64]; ///< The special function table.
		static const FUNCPTR mTableBCOND    [32]; ///< The conditional break function table.
		static const FUNCPTR mTableCOP0     [32]; ///< The CoProcessor 0 function table.
		static const FUNCPTR mTableCOP2     [64]; ///< The CoProcessor 2 function table.
		static const FUNCPTR mTableCOP2Basic[32]; ///< The CoProcessor 2 basic function table.
	};
} // Namespace NeoPSX

#endif // PSX_INTERPRETER_HPP