#include "EFRegistryKey.hpp"
#include "EFException.hpp"

namespace EFramework
{
	EFRegistryKey::EFRegistryKey(const EFString& registryDirectoryName, const EFString& registryDirectory)
	{
		//========================================================================
		// Create a string describing the desired registry path.
		//========================================================================
		EFString pathString = registryDirectory + registryDirectoryName;

		//========================================================================
		// Create a new registry access key.
		//========================================================================
		mRegistryKey.Create(HKEY_CURRENT_USER, _T(pathString.c_str()));
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	void EFRegistryKey::WriteInteger(const EFString& valueName, ulong integer)
	{
		//========================================================================
		// Attempt to write our integer value to the registry.
		//========================================================================
		mRegistryKey.SetDWORDValue(_T(valueName.c_str()), static_cast< DWORD >( integer ));
	}

	void EFRegistryKey::WriteString(const EFString& valueName, const EFString& string)
	{
		//========================================================================
		// Attempt to write our string value to the registry.
		//========================================================================
		if(mRegistryKey.SetStringValue(_T(valueName.c_str()), _T(string.c_str())) != ERROR_SUCCESS)
		{
			EF_THROW(EFException::EC_GUI_ERROR, "Could not write String value for key: " + valueName);
		}
	}

	void EFRegistryKey::WriteString(const EFString& valueName, char* string)
	{
		//========================================================================
		// Attempt to write our array of characters value to the registry.
		//========================================================================
		if(mRegistryKey.SetStringValue(_T(valueName.c_str()), _T(string)) != ERROR_SUCCESS)
		{
			EF_THROW(EFException::EC_GUI_ERROR, "Could not write String value for key: " + valueName);
		}
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	ulong EFRegistryKey::ReadInteger(const EFString& valueName, ulong defaultValue = 0)
	{
		//========================================================================
		// Attempt to read our integer value from the registry.
		//========================================================================
		DWORD buffer = 0;
		if(mRegistryKey.QueryDWORDValue(_T(valueName.c_str()), buffer) != ERROR_SUCCESS)
			return defaultValue;

		//========================================================================
		// Return the value that from the registry.
		//========================================================================
		return (static_cast< ulong >( buffer ));
	}

	char* EFRegistryKey::ReadString(const EFString& valueName,  char* defaultValue, ulong stringSize)
	{
		//========================================================================
		// Attempt to read our string value from the registry.
		//========================================================================
		static char buffer[4096];
		if(mRegistryKey.QueryStringValue(_T(valueName.c_str()), buffer, &stringSize) != ERROR_SUCCESS)
			return defaultValue;

		//========================================================================
		// Return the value that from the registry.
		//========================================================================
		return buffer;
	}
} // Namespace EFramework