#include "arm7.h"
#include "mmu.h"

OPCODE Thumb7opUND() { return 1; }

#define Thumb7RdToEax() \
    __asm { \
        __asm mov eax,[arm7reg+ARMREG_OP*4] \
        __asm and eax,7 \
    }

#define Thumb7RdToEdx() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4] \
        __asm and edx,7 \
    }

#define Thumb7addrRegC() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]    /* Read opcode */ \
        __asm mov ecx,edx                      /* Make a copy */ \
        __asm shr ecx,3                        /* Shift in Rn */ \
        __asm and ecx,7                        /* Mask to range */ \
        __asm mov ebx,[arm7reg+ecx*4]          /* Read value of Rn */ \
        __asm and edx,7                        /* Mask to Rd */ \
        __asm mov eax,[arm7reg+edx*4]          /* Read value of Rd */ \
    }

#define Thumb7addrImm5() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]    /* Read opcode */ \
        __asm mov ecx,edx                      /* Make a copy */ \
        __asm shr ecx,3                        /* Shift in Rn */ \
        __asm and ecx,7                        /* Mask to range */ \
        __asm mov eax,[arm7reg+ecx*4]          /* Read value of Rn */ \
        __asm mov ebx,edx                      /* Another copy */ \
        __asm shr ebx,6                        /* Shift in shftamount */ \
        __asm and ebx,31                       /* and mask off */ \
        __asm and edx,7                        /* Mask to Rd */ \
    }

#define Thumb7addrImm5shft() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]    /* Read opcode */ \
        __asm mov ecx,edx                      /* Make a copy */ \
        __asm shr ecx,3                        /* Shift in Rn */ \
        __asm and ecx,7                        /* Mask to range */ \
        __asm mov eax,[arm7reg+ecx*4]          /* Read value of Rn */ \
        __asm mov ebx,edx                      /* Another copy */ \
        __asm shr ebx,6                        /* Shift in shftamount */ \
        __asm and ebx,31                       /* and mask off */ \
        __asm jnz shft                         /* If zero.. */ \
        __asm mov ebx,32                       /* make 32 */ \
        __asm shft:                            /* Otherwise */ \
        __asm and edx,7                        /* Mask to Rd */ \
    }

#define Thumb7addrImm8() \
    __asm { \
        __asm mov ebx,[arm7reg+ARMREG_OP*4]    /* Read opcode */ \
        __asm and ebx,255 \
    }

#define Thumb7addrImm3() \
    __asm { \
        __asm mov ebx,[arm7reg+ARMREG_OP*4] \
        __asm mov ecx,ebx \
        __asm shr ecx,3                        /* Shift in Rn */ \
        __asm and ecx,7                        /* Mask to range */ \
        __asm mov ecx,[arm7reg+ecx*4]          /* Read value of Rn */ \
        __asm mov [arm7reg+eax*4],ecx \
        __asm shr ebx,6 \
        __asm and ebx,7                        /* Mask to Rm */ \
    }

#define Thumb7addrReg() \
    __asm { \
        __asm mov ebx,[arm7reg+ARMREG_OP*4] \
        __asm mov ecx,ebx \
        __asm shr ecx,3                        /* Shift in Rn */ \
        __asm and ecx,7                        /* Mask to range */ \
        __asm mov ecx,[arm7reg+ecx*4]          /* Read value of Rn */ \
        __asm mov [arm7reg+eax*4],ecx \
        __asm shr ebx,6 \
        __asm and ebx,7                        /* Mask to Rm */ \
        __asm mov ebx,[arm7reg+ebx*4] \
    }

#define Thumb7addrRegLS() \
    __asm { \
        __asm mov ebx,[arm7reg+ARMREG_OP*4] \
        __asm mov ecx,ebx \
        __asm shr ecx,3                        /* Shift in Rn */ \
        __asm and ecx,7                        /* Mask to range */ \
        __asm mov eax,[arm7reg+ecx*4]          /* Read value of Rn */ \
        __asm shr ebx,6 \
        __asm and ebx,7                        /* Mask to Rm */ \
        __asm mov ebx,[arm7reg+ebx*4] \
    }

#define Thumb7addrH() \
    __asm { \
        __asm mov eax,[arm7reg+ARMREG_OP*4] \
        __asm mov ecx,eax \
        __asm and eax,7 \
        __asm shr ecx,4 \
        __asm and ecx,8 \
        __asm or eax,ecx \
        __asm mov ecx,[arm7reg+ARMREG_OP*4] \
        __asm shr ecx,3                        /* Shift in Rn */ \
        __asm and ecx,15                       /* Mask to range */ \
        __asm mov ebx,[arm7reg+ecx*4]          /* Read value of Rn */ \
        __asm cmp ecx,15 \
        __asm jne nopc \
        __asm add ebx,2 \
        __asm nopc: \
    }

#define Thumb7addrPC() \
    __asm { \
        __asm mov ebx,[arm7reg+ARMREG_OP*4] \
        __asm and ebx,255\
        __asm shl ebx,2 \
        __asm mov eax,[arm7reg+15*4] \
        __asm add eax,2 \
        __asm and eax,0xFFFFFFFC \
    }

#define Thumb7addrSP() \
    __asm { \
        __asm mov ebx,[arm7reg+ARMREG_OP*4] \
        __asm and ebx,255\
        __asm shl ebx,2 \
        __asm mov eax,[arm7reg+13*4] \
    }

#define Thumb7addrPCalu() \
    __asm { \
        __asm mov ebx,[arm7reg+ARMREG_OP*4] \
        __asm and ebx,255\
        __asm shl ebx,2 \
        __asm mov ecx,[arm7reg+15*4] \
        __asm add ecx,2 \
        __asm and ecx,0xFFFFFFFC \
        __asm mov [arm7reg+eax*4],ecx \
    }

#define Thumb7addrSPalu() \
    __asm { \
        __asm mov ebx,[arm7reg+ARMREG_OP*4] \
        __asm and ebx,255\
        __asm shl ebx,2 \
        __asm mov ecx,[arm7reg+13*4] \
        __asm mov [arm7reg+eax*4],ecx \
    }

#define Thumb7addrImm7() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4] \
        __asm mov ebx,edx \
        __asm and ebx,127 \
        __asm shl ebx,2 \
        __asm test edx,0x0080 \
        __asm je imm7done \
        __asm neg ebx \
        __asm imm7done: \
    }

#define Thumb7opTST() \
    __asm { \
        __asm and eax,ebx \
    }

#define Thumb7opCMPdp() \
    __asm { \
        __asm sub eax,ebx \
    }

#define Thumb7opCMN() \
    __asm { \
        __asm add eax,ebx \
    }

#define Thumb7opNEG() \
    __asm { \
        __asm xor eax,eax \
        __asm sub eax,ebx \
        __asm mov [arm7reg+edx*4],eax \
    }

#define Thumb7opAND() \
    __asm { \
        __asm and eax,ebx \
        __asm mov [arm7reg+edx*4],eax \
    }

#define Thumb7opEOR() \
    __asm { \
        __asm xor eax,ebx \
        __asm mov [arm7reg+edx*4],eax \
    }

#define Thumb7opORR() \
    __asm { \
        __asm or eax,ebx \
        __asm mov [arm7reg+edx*4],eax \
    }

#define Thumb7opBIC() \
    __asm { \
        __asm not ebx \
        __asm and eax,ebx \
        __asm mov [arm7reg+edx*4],eax \
    }

#define Thumb7opMUL() \
    __asm { \
        __asm mov ecx,[arm7reg+ARMREG_OP*4]    /* Read opcode */ \
        __asm and ecx,7 \
        __asm mul ebx \
        __asm mov [arm7reg+ecx*4],eax \
    }

#define Thumb7opMVN() \
    __asm { \
        __asm not ebx \
        __asm mov [arm7reg+edx*4],ebx \
    }

#define Thumb7opADC() \
    __asm { \
        __asm mov ecx,[arm7reg+ARMREG_FC*4]    /* Read in carry flag */ \
        __asm shr ecx,1                        /* Shift into x86 C */ \
        __asm adc eax,ebx                      /* Do teh adc */ \
        __asm mov [arm7reg+edx*4],eax          /* And save the result */ \
    }

#define Thumb7opSBC() \
    __asm { \
        __asm mov ecx,[arm7reg+ARMREG_FC*4] \
        __asm not ecx \
        __asm shr ecx,1 \
        __asm sbb eax,ebx \
        __asm mov [arm7reg+edx*4],eax \
    }

#define Thumb7opLSL() \
    __asm { \
        __asm mov ecx,ebx \
        __asm and ecx,255 \
        __asm jz noshft_lsl                   /* Skip if no shift */ \
        __asm cmp cl,32                       /* If greater than 32 */ \
        __asm jg overshft_lsl                 /* Shift all the way */ \
        __asm je fullshft_lsl                 /* If equal, shift to carry */ \
        __asm shl eax,cl                      /* Shift operand1 */ \
        __asm jmp shftdone_lsl                /* And skip carry read */ \
        __asm noshft_lsl:                     /* Done shifting */ \
        \
        __asm mov ecx,[arm7reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm jmp shftdone_lsl                /* Skip over */ \
        __asm fullshft_lsl:                   /* Shift by 32 */ \
        __asm mov ecx,eax                     /* Make a copy */ \
        __asm xor eax,eax                     /* Rm=0 */ \
        __asm shr ecx,1                       /* Carry = Rm[0] */ \
        __asm jmp shftdone_lsl                /* Skip over */ \
        __asm overshft_lsl:                   /* Shift over 32 */ \
        __asm xor eax,eax                     /* Rm=0 */ \
        __asm shr eax,1                       /* And carry=0 */ \
        __asm shftdone_lsl:                   /* all done! */ \
        __asm mov [arm7reg+edx*4],eax \
    }

#define Thumb7opLSR() \
    __asm { \
        __asm mov ecx,ebx \
        __asm and ecx,255 \
        __asm jz noshft                       /* Skip if no shift */ \
        __asm cmp cl,32                       /* If greater than 32 */ \
        __asm jg overshft                     /* Shift all the way */ \
        __asm je fullshft                     /* If equal, shift to carry */ \
        __asm shr eax,cl                      /* Shift operand1 */ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
        \
        __asm mov ecx,[arm7reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm jmp shftdone                    /* Skip over */ \
        __asm fullshft:                       /* Shift by 32 */ \
        __asm mov ecx,eax                     /* Make a copy */ \
        __asm xor eax,eax                     /* Rm=0 */ \
        __asm shl ecx,1                       /* Carry = Rm[31] */ \
        __asm jmp shftdone                    /* Skip over */ \
        __asm overshft:                       /* Shift over 32 */ \
        __asm xor eax,eax                     /* Rm=0 */ \
        __asm shr eax,1                       /* And carry=0 */ \
        __asm shftdone:                       /* all done! */ \
        __asm mov [arm7reg+edx*4],eax \
    }

#define Thumb7opASR() \
    __asm { \
        __asm mov ecx,ebx \
        __asm and ecx,255 \
        __asm jz noshft                       /* Skip if no shift */ \
        __asm cmp cl,32                       /* If greater than 32 */ \
        __asm jge fullshft                    /* Shift all the way */ \
        __asm sar eax,cl                      /* Shift operand1 */ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
	\
        __asm mov ecx,[arm7reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm jmp shftdone                    /* and skip over */ \
        __asm fullshft:                       /* shift over 31 */ \
 	__asm mov ecx,eax                     /* Make a copy */ \
        __asm sar eax,31                      /* Fill with hi bit */ \
        __asm shl ecx,1                       /* Carry = Rm[31] */ \
        __asm shftdone:                       /* all done! */ \
        __asm mov [arm7reg+edx*4],eax \
    }

#define Thumb7opROR() \
    __asm { \
        __asm mov ecx,ebx                     /* Make a copy of Rn */ \
        __asm and ecx,255                     /* Get the low byte */ \
        __asm jz noshft_ror                   /* Skip if no shift */ \
        __asm and cl,31                       /* Check [4:0] */ \
        __asm jz fullshft_ror                 /* And skip if full */ \
        __asm ror eax,cl                      /* Shift operand1 */ \
        __asm jmp shftdone_ror                /* And skip carry read */ \
        __asm noshft_ror:                     /* Done shifting */ \
	\
        __asm mov ecx,[arm7reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm jmp shftdone_ror                /* Skip over */ \
        __asm fullshft_ror:                   /* Shifting by full */ \
        __asm mov ecx,eax                     /* Make a copy of Rm */ \
        __asm shl ecx,1                       /* Carry = hi bit */ \
        __asm shftdone_ror:                   /* all done! */ \
        __asm mov [arm7reg+edx*4],eax \
    }

#define Thumb7opLDR() \
    __asm { \
        __asm add eax,ebx \
        __asm push edx                  /* Save Rd */ \
        __asm push eax                  /* Address */ \
        __asm xor ecx,ecx               /* Bus = 0 */ \
        __asm push ecx                  /* Push "bus number" */ \
        __asm call MMUrdW               /* Read a word */ \
        __asm pop ecx                   /* Clear stack */ \
        __asm pop ecx                   /* of params */ \
        __asm pop edx                   /* Get Rd back */\
	__asm mov [arm7reg+edx*4],eax   /* Write value to Rd */ \
	__asm mov eax,3 \
    }

#define Thumb7opSTR() \
    __asm { \
        __asm add eax,ebx \
        __asm mov ecx,[arm7reg+edx*4]     /* Push Rd */ \
        __asm push ecx \
        __asm push eax                    /* Push address */ \
        __asm xor ecx,ecx                 /* Bus = 0 */ \
        __asm push ecx                    /* Push "bus number" */ \
        __asm call MMUwrW                 /* Write thru MMU */ \
        __asm pop eax                     /* Clear stack */ \
        __asm pop eax                     /* of pushed */ \
        __asm pop eax                     /* parameters */ \
        __asm mov eax,2 \
    }

#define Thumb7opLDRH() \
    __asm { \
        __asm add eax,ebx \
        __asm push edx                  /* Save Rd */ \
        __asm push eax                  /* Address */ \
        __asm xor ecx,ecx               /* Bus = 0 */ \
        __asm push ecx                  /* Push "bus number" */ \
        __asm call MMUrdH               /* Read a word */ \
        __asm pop ecx                   /* Clear stack */ \
        __asm pop ecx                   /* of params */ \
        __asm pop edx                   /* Get Rd back */\
        __asm movzx ecx,ax \
	__asm mov [arm7reg+edx*4],ecx   /* Write value to Rd */ \
	__asm mov eax,3 \
    }

#define Thumb7opSTRH() \
    __asm { \
        __asm add eax,ebx \
        __asm mov ecx,[arm7reg+edx*4]     /* Push Rd */ \
        __asm and ecx,0x0000FFFF \
        __asm push ecx \
        __asm push eax                    /* Push address */ \
        __asm xor ecx,ecx                 /* Bus = 0 */ \
        __asm push ecx                    /* Push "bus number" */ \
        __asm call MMUwrH                 /* Write thru MMU */ \
        __asm pop eax                     /* Clear stack */ \
        __asm pop eax                     /* of pushed */ \
        __asm pop eax                     /* parameters */ \
        __asm mov eax,2 \
    }

#define Thumb7opLDRB() \
    __asm { \
        __asm add eax,ebx \
        __asm push edx                  /* Save Rd */ \
        __asm push eax                  /* Address */ \
        __asm xor ecx,ecx               /* Bus = 0 */ \
        __asm push ecx                  /* Push "bus number" */ \
        __asm call MMUrdB               /* Read a word */ \
        __asm pop ecx                   /* Clear stack */ \
        __asm pop ecx                   /* of params */ \
        __asm pop edx                   /* Get Rd back */\
        __asm movzx ecx,al \
	__asm mov [arm7reg+edx*4],ecx   /* Write value to Rd */ \
	__asm mov eax,3 \
    }

#define Thumb7opSTRB() \
    __asm { \
        __asm add eax,ebx \
        __asm mov ecx,[arm7reg+edx*4]     /* Push Rd */ \
        __asm and ecx,0x000000FF \
        __asm push ecx \
        __asm push eax                    /* Push address */ \
        __asm xor ecx,ecx                 /* Bus = 0 */ \
        __asm push ecx                    /* Push "bus number" */ \
        __asm call MMUwrB                 /* Write thru MMU */ \
        __asm pop eax                     /* Clear stack */ \
        __asm pop eax                     /* of pushed */ \
        __asm pop eax                     /* parameters */ \
        __asm mov eax,2 \
    }

#define Thumb7opLDRSH() \
    __asm { \
        __asm add eax,ebx \
        __asm push edx                  /* Save Rd */ \
        __asm push eax                  /* Address */ \
        __asm xor ecx,ecx               /* Bus = 0 */ \
        __asm push ecx                  /* Push "bus number" */ \
        __asm call MMUrdH               /* Read a word */ \
        __asm pop ecx                   /* Clear stack */ \
        __asm pop ecx                   /* of params */ \
        __asm pop edx                   /* Get Rd back */\
        __asm movsx ecx,ax \
	__asm mov [arm7reg+edx*4],ecx   /* Write value to Rd */ \
	__asm mov eax,3 \
    }

#define Thumb7opLDRSB() \
    __asm { \
        __asm add eax,ebx \
        __asm push edx                  /* Save Rd */ \
        __asm push eax                  /* Address */ \
        __asm xor ecx,ecx               /* Bus = 0 */ \
        __asm push ecx                  /* Push "bus number" */ \
        __asm call MMUrdB               /* Read a word */ \
        __asm pop ecx                   /* Clear stack */ \
        __asm pop ecx                   /* of params */ \
        __asm pop edx                   /* Get Rd back */\
        __asm movsx ecx,al \
	__asm mov [arm7reg+edx*4],ecx   /* Write value to Rd */ \
	__asm mov eax,3 \
    }

#define Thumb7flagsShft() \
    __asm { \
        __asm setc  [arm7reg+ARMREG_FC*4]      /* C flag */ \
        __asm mov ecx,eax \
        __asm or eax,eax \
        __asm setz [arm7reg+ARMREG_FZ*4] \
        __asm shr ecx,31 \
        __asm mov [arm7reg+ARMREG_FN*4],ecx \
        __asm mov eax,1 \
    }

#define Thumb7flagsBIT() \
    __asm { \
        __asm sets  [arm7reg+ARMREG_FN*4]      /* Copy N flag */ \
        __asm setz  [arm7reg+ARMREG_FZ*4]      /* and Z flag */ \
        __asm setc  [arm7reg+ARMREG_FC*4]      /* C flag */ \
        __asm mov eax,1 \
    }

#define Thumb7flagsADD() \
    __asm { \
        __asm sets  [arm7reg+ARMREG_FN*4]      /* Copy N flag */ \
        __asm setz  [arm7reg+ARMREG_FZ*4]      /* and Z flag */ \
        __asm seto  [arm7reg+ARMREG_FV*4]      /* V too */ \
        __asm setc  [arm7reg+ARMREG_FC*4]      /* C flag */ \
        __asm mov eax,1 \
    }

#define Thumb7flagsSUB() \
    __asm { \
        __asm sets  [arm7reg+ARMREG_FN*4]      /* Copy N flag */ \
        __asm setz  [arm7reg+ARMREG_FZ*4]      /* and Z flag */ \
        __asm seto  [arm7reg+ARMREG_FV*4]      /* V too */ \
        __asm setnc [arm7reg+ARMREG_FC*4]      /* C flag */ \
        __asm mov eax,1 \
    }

#define Thumb7opMOV()                          /* eax = Rd */ \
    __asm { \
        __asm mov ecx,[arm7reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm mov [arm7reg+eax*4],ebx \
        __asm or ebx,ebx \
    }

#define Thumb7opADD()                          /* eax = Rd */ \
    __asm { \
        __asm add [arm7reg+eax*4],ebx \
    }

#define Thumb7opSUB() \
    __asm { \
        __asm sub [arm7reg+eax*4],ebx \
    }

#define Thumb7opCMP() \
    __asm { \
        __asm mov ecx,[arm7reg+eax*4] \
        __asm sub ecx,ebx \
    }

#define Thumb7addrB() \
    arm7reg.tmp2=arm7reg.curop&0x00FF; \
    arm7reg.tmp1=(arm7reg.tmp2&0x0080)? \
                 (0xFFFFFF00|arm7reg.tmp2): \
		 (0x00000000|arm7reg.tmp2)

OPCODE Thumb7opLSLimm() { Thumb7addrImm5shft(); Thumb7opLSL(); Thumb7flagsShft(); }
OPCODE Thumb7opLSRimm() { Thumb7addrImm5shft(); Thumb7opLSR(); Thumb7flagsBIT(); }
OPCODE Thumb7opASRimm() { Thumb7addrImm5shft(); Thumb7opASR(); Thumb7flagsShft(); }

OPCODE Thumb7opADDreg() { Thumb7RdToEax(); Thumb7addrReg(); Thumb7opADD(); Thumb7flagsADD(); }
OPCODE Thumb7opSUBreg() { Thumb7RdToEax(); Thumb7addrReg(); Thumb7opSUB(); Thumb7flagsSUB(); }

OPCODE Thumb7opADDimm3() { Thumb7RdToEax(); Thumb7addrImm3(); Thumb7opADD(); Thumb7flagsADD(); }
OPCODE Thumb7opSUBimm3() { Thumb7RdToEax(); Thumb7addrImm3(); Thumb7opSUB(); Thumb7flagsSUB(); }

OPCODE Thumb7opMOVimm8r0() { __asm { mov eax,0 } Thumb7addrImm8(); Thumb7opMOV(); Thumb7flagsBIT(); }
OPCODE Thumb7opMOVimm8r1() { __asm { mov eax,1 } Thumb7addrImm8(); Thumb7opMOV(); Thumb7flagsBIT(); }
OPCODE Thumb7opMOVimm8r2() { __asm { mov eax,2 } Thumb7addrImm8(); Thumb7opMOV(); Thumb7flagsBIT(); }
OPCODE Thumb7opMOVimm8r3() { __asm { mov eax,3 } Thumb7addrImm8(); Thumb7opMOV(); Thumb7flagsBIT(); }
OPCODE Thumb7opMOVimm8r4() { __asm { mov eax,4 } Thumb7addrImm8(); Thumb7opMOV(); Thumb7flagsBIT(); }
OPCODE Thumb7opMOVimm8r5() { __asm { mov eax,5 } Thumb7addrImm8(); Thumb7opMOV(); Thumb7flagsBIT(); }
OPCODE Thumb7opMOVimm8r6() { __asm { mov eax,6 } Thumb7addrImm8(); Thumb7opMOV(); Thumb7flagsBIT(); }
OPCODE Thumb7opMOVimm8r7() { __asm { mov eax,7 } Thumb7addrImm8(); Thumb7opMOV(); Thumb7flagsBIT(); }
OPCODE Thumb7opCMPimm8r0() { __asm { mov eax,0 } Thumb7addrImm8(); Thumb7opCMP(); Thumb7flagsSUB(); }
OPCODE Thumb7opCMPimm8r1() { __asm { mov eax,1 } Thumb7addrImm8(); Thumb7opCMP(); Thumb7flagsSUB(); }
OPCODE Thumb7opCMPimm8r2() { __asm { mov eax,2 } Thumb7addrImm8(); Thumb7opCMP(); Thumb7flagsSUB(); }
OPCODE Thumb7opCMPimm8r3() { __asm { mov eax,3 } Thumb7addrImm8(); Thumb7opCMP(); Thumb7flagsSUB(); }
OPCODE Thumb7opCMPimm8r4() { __asm { mov eax,4 } Thumb7addrImm8(); Thumb7opCMP(); Thumb7flagsSUB(); }
OPCODE Thumb7opCMPimm8r5() { __asm { mov eax,5 } Thumb7addrImm8(); Thumb7opCMP(); Thumb7flagsSUB(); }
OPCODE Thumb7opCMPimm8r6() { __asm { mov eax,6 } Thumb7addrImm8(); Thumb7opCMP(); Thumb7flagsSUB(); }
OPCODE Thumb7opCMPimm8r7() { __asm { mov eax,7 } Thumb7addrImm8(); Thumb7opCMP(); Thumb7flagsSUB(); }
OPCODE Thumb7opADDimm8r0() { __asm { mov eax,0 } Thumb7addrImm8(); Thumb7opADD(); Thumb7flagsADD(); }
OPCODE Thumb7opADDimm8r1() { __asm { mov eax,1 } Thumb7addrImm8(); Thumb7opADD(); Thumb7flagsADD(); }
OPCODE Thumb7opADDimm8r2() { __asm { mov eax,2 } Thumb7addrImm8(); Thumb7opADD(); Thumb7flagsADD(); }
OPCODE Thumb7opADDimm8r3() { __asm { mov eax,3 } Thumb7addrImm8(); Thumb7opADD(); Thumb7flagsADD(); }
OPCODE Thumb7opADDimm8r4() { __asm { mov eax,4 } Thumb7addrImm8(); Thumb7opADD(); Thumb7flagsADD(); }
OPCODE Thumb7opADDimm8r5() { __asm { mov eax,5 } Thumb7addrImm8(); Thumb7opADD(); Thumb7flagsADD(); }
OPCODE Thumb7opADDimm8r6() { __asm { mov eax,6 } Thumb7addrImm8(); Thumb7opADD(); Thumb7flagsADD(); }
OPCODE Thumb7opADDimm8r7() { __asm { mov eax,7 } Thumb7addrImm8(); Thumb7opADD(); Thumb7flagsADD(); }
OPCODE Thumb7opSUBimm8r0() { __asm { mov eax,0 } Thumb7addrImm8(); Thumb7opSUB(); Thumb7flagsSUB(); }
OPCODE Thumb7opSUBimm8r1() { __asm { mov eax,1 } Thumb7addrImm8(); Thumb7opSUB(); Thumb7flagsSUB(); }
OPCODE Thumb7opSUBimm8r2() { __asm { mov eax,2 } Thumb7addrImm8(); Thumb7opSUB(); Thumb7flagsSUB(); }
OPCODE Thumb7opSUBimm8r3() { __asm { mov eax,3 } Thumb7addrImm8(); Thumb7opSUB(); Thumb7flagsSUB(); }
OPCODE Thumb7opSUBimm8r4() { __asm { mov eax,4 } Thumb7addrImm8(); Thumb7opSUB(); Thumb7flagsSUB(); }
OPCODE Thumb7opSUBimm8r5() { __asm { mov eax,5 } Thumb7addrImm8(); Thumb7opSUB(); Thumb7flagsSUB(); }
OPCODE Thumb7opSUBimm8r6() { __asm { mov eax,6 } Thumb7addrImm8(); Thumb7opSUB(); Thumb7flagsSUB(); }
OPCODE Thumb7opSUBimm8r7() { __asm { mov eax,7 } Thumb7addrImm8(); Thumb7opSUB(); Thumb7flagsSUB(); }

OPCODE Thumb7opDPg1()
{
    switch(arm7reg.curop&0x00C0)
    {
        case 0x0000: Thumb7addrRegC(); Thumb7opAND(); Thumb7flagsBIT(); break;
        case 0x0040: Thumb7addrRegC(); Thumb7opEOR(); Thumb7flagsBIT(); break;
        case 0x0080: Thumb7addrRegC(); Thumb7opLSL(); Thumb7flagsShft(); break;
        case 0x00C0: Thumb7addrRegC(); Thumb7opLSR(); Thumb7flagsBIT(); break;
    }
    return 1;
}

OPCODE Thumb7opDPg2()
{
    switch(arm7reg.curop&0x00C0)
    {
        case 0x0000: Thumb7addrRegC(); Thumb7opASR(); Thumb7flagsShft(); break;
        case 0x0040: Thumb7addrRegC(); Thumb7opADC(); Thumb7flagsADD(); break;
        case 0x0080: Thumb7addrRegC(); Thumb7opSBC(); Thumb7flagsSUB(); break;
        case 0x00C0: Thumb7addrRegC(); Thumb7opROR(); Thumb7flagsShft(); break;
    }
    return 1;
}

OPCODE Thumb7opDPg3()
{
    switch(arm7reg.curop&0x00C0)
    {
        case 0x0000: Thumb7addrRegC(); Thumb7opTST(); Thumb7flagsBIT(); break;
        case 0x0040: Thumb7addrRegC(); Thumb7opNEG(); Thumb7flagsSUB(); break;
        case 0x0080: Thumb7addrRegC(); Thumb7opCMPdp(); Thumb7flagsSUB(); break;
        case 0x00C0: Thumb7addrRegC(); Thumb7opCMN(); Thumb7flagsADD(); break;
    }
    return 1;
}

OPCODE Thumb7opDPg4()
{
    switch(arm7reg.curop&0x00C0)
    {
        case 0x0000: Thumb7addrRegC(); Thumb7opORR(); Thumb7flagsBIT(); break;
        case 0x0040: Thumb7addrRegC(); Thumb7opMUL(); Thumb7flagsShft(); break;
        case 0x0080: Thumb7addrRegC(); Thumb7opBIC(); Thumb7flagsBIT(); break;
        case 0x00C0: Thumb7addrRegC(); Thumb7opMVN(); Thumb7flagsBIT(); break;
    }
    return 1;
}

OPCODE Thumb7opADDH() { Thumb7addrH(); Thumb7opADD(); return 1; }
OPCODE Thumb7opMOVH() { Thumb7addrH(); Thumb7opMOV(); return 1; }
OPCODE Thumb7opCMPH() { Thumb7addrH(); Thumb7opCMP(); Thumb7flagsSUB(); }

OPCODE Thumb7opLDRPCr0() { __asm { mov edx,0 } Thumb7addrPC(); Thumb7opLDR(); }
OPCODE Thumb7opLDRPCr1() { __asm { mov edx,1 } Thumb7addrPC(); Thumb7opLDR(); }
OPCODE Thumb7opLDRPCr2() { __asm { mov edx,2 } Thumb7addrPC(); Thumb7opLDR(); }
OPCODE Thumb7opLDRPCr3() { __asm { mov edx,3 } Thumb7addrPC(); Thumb7opLDR(); }
OPCODE Thumb7opLDRPCr4() { __asm { mov edx,4 } Thumb7addrPC(); Thumb7opLDR(); }
OPCODE Thumb7opLDRPCr5() { __asm { mov edx,5 } Thumb7addrPC(); Thumb7opLDR(); }
OPCODE Thumb7opLDRPCr6() { __asm { mov edx,6 } Thumb7addrPC(); Thumb7opLDR(); }
OPCODE Thumb7opLDRPCr7() { __asm { mov edx,7 } Thumb7addrPC(); Thumb7opLDR(); }

OPCODE Thumb7opLDRSPr0() { __asm { mov edx,0 } Thumb7addrSP(); Thumb7opLDR(); }
OPCODE Thumb7opLDRSPr1() { __asm { mov edx,1 } Thumb7addrSP(); Thumb7opLDR(); }
OPCODE Thumb7opLDRSPr2() { __asm { mov edx,2 } Thumb7addrSP(); Thumb7opLDR(); }
OPCODE Thumb7opLDRSPr3() { __asm { mov edx,3 } Thumb7addrSP(); Thumb7opLDR(); }
OPCODE Thumb7opLDRSPr4() { __asm { mov edx,4 } Thumb7addrSP(); Thumb7opLDR(); }
OPCODE Thumb7opLDRSPr5() { __asm { mov edx,5 } Thumb7addrSP(); Thumb7opLDR(); }
OPCODE Thumb7opLDRSPr6() { __asm { mov edx,6 } Thumb7addrSP(); Thumb7opLDR(); }
OPCODE Thumb7opLDRSPr7() { __asm { mov edx,7 } Thumb7addrSP(); Thumb7opLDR(); }
OPCODE Thumb7opSTRSPr0() { __asm { mov edx,0 } Thumb7addrSP(); Thumb7opSTR(); }
OPCODE Thumb7opSTRSPr1() { __asm { mov edx,1 } Thumb7addrSP(); Thumb7opSTR(); }
OPCODE Thumb7opSTRSPr2() { __asm { mov edx,2 } Thumb7addrSP(); Thumb7opSTR(); }
OPCODE Thumb7opSTRSPr3() { __asm { mov edx,3 } Thumb7addrSP(); Thumb7opSTR(); }
OPCODE Thumb7opSTRSPr4() { __asm { mov edx,4 } Thumb7addrSP(); Thumb7opSTR(); }
OPCODE Thumb7opSTRSPr5() { __asm { mov edx,5 } Thumb7addrSP(); Thumb7opSTR(); }
OPCODE Thumb7opSTRSPr6() { __asm { mov edx,6 } Thumb7addrSP(); Thumb7opSTR(); }
OPCODE Thumb7opSTRSPr7() { __asm { mov edx,7 } Thumb7addrSP(); Thumb7opSTR(); }

OPCODE Thumb7opSTRreg()   { Thumb7RdToEdx(); Thumb7addrRegLS(); Thumb7opSTR(); }
OPCODE Thumb7opLDRreg()   { Thumb7RdToEdx(); Thumb7addrRegLS(); Thumb7opLDR(); }
OPCODE Thumb7opSTRHreg()  { Thumb7RdToEdx(); Thumb7addrRegLS(); Thumb7opSTRH(); }
OPCODE Thumb7opLDRHreg()  { Thumb7RdToEdx(); Thumb7addrRegLS(); Thumb7opLDRH(); }
OPCODE Thumb7opSTRBreg()  { Thumb7RdToEdx(); Thumb7addrRegLS(); Thumb7opSTRB(); }
OPCODE Thumb7opLDRBreg()  { Thumb7RdToEdx(); Thumb7addrRegLS(); Thumb7opLDRB(); }
OPCODE Thumb7opLDRSBreg() { Thumb7RdToEdx(); Thumb7addrRegLS(); Thumb7opLDRSB(); }
OPCODE Thumb7opLDRSHreg() { Thumb7RdToEdx(); Thumb7addrRegLS(); Thumb7opLDRSH(); }

OPCODE Thumb7opSTRimm5()  { Thumb7RdToEdx(); Thumb7addrImm5(); __asm { shl ebx,2 } Thumb7opSTR(); }
OPCODE Thumb7opLDRimm5()  { Thumb7RdToEdx(); Thumb7addrImm5(); __asm { shl ebx,2 } Thumb7opLDR(); }
OPCODE Thumb7opSTRHimm5() { Thumb7RdToEdx(); Thumb7addrImm5(); __asm { shl ebx,1 } Thumb7opSTRH(); }
OPCODE Thumb7opLDRHimm5() { Thumb7RdToEdx(); Thumb7addrImm5(); __asm { shl ebx,1 } Thumb7opLDRH(); }
OPCODE Thumb7opSTRBimm5() { Thumb7RdToEdx(); Thumb7addrImm5(); Thumb7opSTRB(); }
OPCODE Thumb7opLDRBimm5() { Thumb7RdToEdx(); Thumb7addrImm5(); Thumb7opLDRB(); }

OPCODE Thumb7opADDPCr0() { __asm { mov eax,0 } Thumb7addrPCalu(); Thumb7opADD(); return 1; }
OPCODE Thumb7opADDPCr1() { __asm { mov eax,1 } Thumb7addrPCalu(); Thumb7opADD(); return 1; }
OPCODE Thumb7opADDPCr2() { __asm { mov eax,2 } Thumb7addrPCalu(); Thumb7opADD(); return 1; }
OPCODE Thumb7opADDPCr3() { __asm { mov eax,3 } Thumb7addrPCalu(); Thumb7opADD(); return 1; }
OPCODE Thumb7opADDPCr4() { __asm { mov eax,4 } Thumb7addrPCalu(); Thumb7opADD(); return 1; }
OPCODE Thumb7opADDPCr5() { __asm { mov eax,5 } Thumb7addrPCalu(); Thumb7opADD(); return 1; }
OPCODE Thumb7opADDPCr6() { __asm { mov eax,6 } Thumb7addrPCalu(); Thumb7opADD(); return 1; }
OPCODE Thumb7opADDPCr7() { __asm { mov eax,7 } Thumb7addrPCalu(); Thumb7opADD(); return 1; }
OPCODE Thumb7opADDSPr0() { __asm { mov eax,0 } Thumb7addrSPalu(); Thumb7opADD(); return 1; }
OPCODE Thumb7opADDSPr1() { __asm { mov eax,1 } Thumb7addrSPalu(); Thumb7opADD(); return 1; }
OPCODE Thumb7opADDSPr2() { __asm { mov eax,2 } Thumb7addrSPalu(); Thumb7opADD(); return 1; }
OPCODE Thumb7opADDSPr3() { __asm { mov eax,3 } Thumb7addrSPalu(); Thumb7opADD(); return 1; }
OPCODE Thumb7opADDSPr4() { __asm { mov eax,4 } Thumb7addrSPalu(); Thumb7opADD(); return 1; }
OPCODE Thumb7opADDSPr5() { __asm { mov eax,5 } Thumb7addrSPalu(); Thumb7opADD(); return 1; }
OPCODE Thumb7opADDSPr6() { __asm { mov eax,6 } Thumb7addrSPalu(); Thumb7opADD(); return 1; }
OPCODE Thumb7opADDSPr7() { __asm { mov eax,7 } Thumb7addrSPalu(); Thumb7opADD(); return 1; }

OPCODE Thumb7opADDSPimm7() { __asm { mov eax,13 } Thumb7addrImm7(); Thumb7opADD(); return 1; }

OPCODE Thumb7opBEQ() {if(ARM7condEQ()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;} return 1; }
OPCODE Thumb7opBNE() {if(ARM7condNE()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;} return 1; }
OPCODE Thumb7opBCS() {if(ARM7condCS()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;} return 1; }
OPCODE Thumb7opBCC() {if(ARM7condCC()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;} return 1; }
OPCODE Thumb7opBMI() {if(ARM7condMI()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;} return 1; }
OPCODE Thumb7opBPL() {if(ARM7condPL()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;} return 1; }
OPCODE Thumb7opBVS() {if(ARM7condVS()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;} return 1; }
OPCODE Thumb7opBVC() {if(ARM7condVC()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;} return 1; }
OPCODE Thumb7opBHI() {if(ARM7condHI()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;} return 1; }
OPCODE Thumb7opBLS() {if(ARM7condLS()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;} return 1; }
OPCODE Thumb7opBGE() {if(ARM7condGE()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;} return 1; }
OPCODE Thumb7opBLT() {if(ARM7condLT()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;} return 1; }
OPCODE Thumb7opBGT() {if(ARM7condGT()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;} return 1; }
OPCODE Thumb7opBLE() {if(ARM7condLE()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;} return 1; }
OPCODE Thumb7opBAL() {if(ARM7condAL()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;} return 1; }
OPCODE Thumb7opBNV() {if(ARM7condNV()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;} return 1; }

OPCODE Thumb7opB()
{
    arm7reg.tmp2=arm7reg.curop&0x07FF;
    arm7reg.tmp1=(arm7reg.tmp2&0x0400)?
                 (0xFFFFF800|arm7reg.tmp2):
		 (0x00000000|arm7reg.tmp2);
    arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;
    return 3;
}

OPCODE Thumb7opBXreg()
{
    if(arm7reg.curop&0x0080) arm7reg.r[14]=arm7reg.r[15]|1;
    arm7reg.r[15]=arm7reg.r[ARM7TOP_RNH]&0xFFFFFFFE;
/*    if(arm7reg.r[ARM7TOP_RNH]&1) arm7reg.cpsr|=ARMS_T;
                          else arm7reg.cpsr&=(0xFFFFFFFF-ARMS_T);*/
    arm7reg.flags[ARMFLAG_T]=arm7reg.r[ARM7TOP_RNH]&1;
    ARM7updateCPSR();
    return 3;
}

OPCODE Thumb7opBLsetup()
{
    arm7reg.tmp2=arm7reg.curop&0x07FF;
    arm7reg.tmp1=(arm7reg.tmp2&0x0400)?
                 (0xFFFFF800|arm7reg.tmp2):
		 (0x00000000|arm7reg.tmp2);
    arm7reg.r[14]=arm7reg.r[15]+((signed)arm7reg.tmp1)*4096+2;
    return 4;
}

OPCODE Thumb7opBLoff()
{
    arm7reg.tmp2=arm7reg.curop&0x07FF;
    arm7reg.tmp1=arm7reg.r[14];
    arm7reg.r[14]=arm7reg.r[15]|1;
    arm7reg.r[15]=arm7reg.tmp1+arm7reg.tmp2*2;
    return 4;
}

#define Thumb7opLDMIA(rg) \
    int a; \
    arm7reg.tmp1=arm7reg.r[(rg)]; \
    arm7reg.r[(rg)]+=ARM7SideSum(arm7reg.curop&0x00FF)*4; \
    for(a=0;a<=7;a++) \
    { \
        if(arm7reg.curop&(1<<a)) \
        { \
            arm7reg.r[a]=MMUrdW(0,arm7reg.tmp1); \
            arm7reg.tmp1+=4; \
        } \
    }

#define Thumb7opSTMIA(rg) \
    int a; \
    arm7reg.tmp1=arm7reg.r[(rg)]; \
    arm7reg.r[(rg)]+=ARM7SideSum(arm7reg.curop&0x00FF)*4; \
    for(a=0;a<=7;a++) \
    { \
        if(arm7reg.curop&(1<<a)) \
        { \
            MMUwrW(0,arm7reg.tmp1,arm7reg.r[a]); \
            arm7reg.tmp1+=4; \
        } \
    }

OPCODE Thumb7opPUSH()
{
    int a;
    arm7reg.tmp2=0;
    arm7reg.tmp1=arm7reg.r[13]-ARM7SideSum(arm7reg.curop&0x00FF)*4;
    arm7reg.r[13]-=ARM7SideSum(arm7reg.curop&0x00ff)*4;
    for(a=0;a<=7;a++)
    {
        if(arm7reg.curop&(1<<a))
        {
            MMUwrW(0,arm7reg.tmp1,arm7reg.r[a]);
            arm7reg.tmp1+=4;
        }
    }
    return 1;
}

OPCODE Thumb7opPUSHlr()
{
    int a;
    arm7reg.tmp1=arm7reg.r[13]-ARM7SideSum(arm7reg.curop&0x01FF)*4;
    arm7reg.r[13]-=ARM7SideSum(arm7reg.curop&0x01ff)*4;
    for(a=0;a<=7;a++)
    {
        if(arm7reg.curop&(1<<a))
        {
            MMUwrW(0,arm7reg.tmp1,arm7reg.r[a]);
            arm7reg.tmp1+=4;
        }
    }
    if(arm7reg.curop&0x0100)
        MMUwrW(0,arm7reg.tmp1,arm7reg.r[14]);
    return 1;
}

OPCODE Thumb7opPOP()
{
    int a;
    arm7reg.tmp1=arm7reg.r[13];
    arm7reg.r[13]+=ARM7SideSum(arm7reg.curop&0x00FF)*4;
    for(a=0;a<=7;a++)
    {
        if(arm7reg.curop&(1<<a))
        {
            arm7reg.r[a]=MMUrdW(0,arm7reg.tmp1);
            arm7reg.tmp1+=4;
        }
    }
    return 1;
}

OPCODE Thumb7opPOPpc()
{
    int a;
    arm7reg.tmp1=arm7reg.r[13];
    arm7reg.r[13]+=ARM7SideSum(arm7reg.curop&0x01FF)*4;
    for(a=0;a<=7;a++)
    {
        if(arm7reg.curop&(1<<a))
        {
            arm7reg.r[a]=MMUrdW(0,arm7reg.tmp1);
            arm7reg.tmp1+=4;
        }
    }
    if(arm7reg.curop&0x0100)
    {
        arm7reg.r[15]=MMUrdW(0,arm7reg.tmp1)&0xFFFFFFFE;
        //ARM5: ThumbMode=r15&1
    }
    return 1;
}

OPCODE Thumb7opBKPT() {return 1;}
OPCODE Thumb7opSWI()
{
    arm7reg.curop<<=16; 
    return ARM7opSWI();
}

OPCODE Thumb7opLDMIAr0() {Thumb7opLDMIA(0); return 3; }
OPCODE Thumb7opLDMIAr1() {Thumb7opLDMIA(1); return 3; }
OPCODE Thumb7opLDMIAr2() {Thumb7opLDMIA(2); return 3; }
OPCODE Thumb7opLDMIAr3() {Thumb7opLDMIA(3); return 3; }
OPCODE Thumb7opLDMIAr4() {Thumb7opLDMIA(4); return 3; }
OPCODE Thumb7opLDMIAr5() {Thumb7opLDMIA(5); return 3; }
OPCODE Thumb7opLDMIAr6() {Thumb7opLDMIA(6); return 3; }
OPCODE Thumb7opLDMIAr7() {Thumb7opLDMIA(7); return 3; }
OPCODE Thumb7opSTMIAr0() {Thumb7opSTMIA(0); return 2; }
OPCODE Thumb7opSTMIAr1() {Thumb7opSTMIA(1); return 2; }
OPCODE Thumb7opSTMIAr2() {Thumb7opSTMIA(2); return 2; }
OPCODE Thumb7opSTMIAr3() {Thumb7opSTMIA(3); return 2; }
OPCODE Thumb7opSTMIAr4() {Thumb7opSTMIA(4); return 2; }
OPCODE Thumb7opSTMIAr5() {Thumb7opSTMIA(5); return 2; }
OPCODE Thumb7opSTMIAr6() {Thumb7opSTMIA(6); return 2; }
OPCODE Thumb7opSTMIAr7() {Thumb7opSTMIA(7); return 2; }

