/**************************************************************************
* DSemu: GameBoy Advance interface and encapsulation (gba.c)              *
* Released under the terms of the BSD Public Licence                      *
* Imran Nazar (tf@oopsilon.com), 2005                                     *
* Code acknowledgements: Blatant rips from ector and NeHe.                *
**************************************************************************/

#include <windows.h>
#include <stdio.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glaux.h>
#include "arm7.h"
#include "mmu.h"
#include "gpu.h"
#include "defs.h"
#include "vtbl.h"
#include "win.h"
#include "err.h"
#include "emu.h"
#include "gba.h"
#include "resource.h"
#include "key.h"
#include "ioreg.h"

LOGVTBL *logvt=NULL;

IOREG GBAio[512];

u16 ioregflags[]={
    3,3,3,1,3,3,3,3,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
    2,2,2,2,3,3,2,0,3,2,2,0,0,0,0,0,3,3,3,0,3,0,3,0,3,3,3,0,3,0,3,0,
    3,3,3,0,0,0,0,0,3,3,3,3,3,3,3,3,3,3,3,3,0,0,0,0,2,2,2,2,2,3,2,2,
    2,2,2,3,2,2,2,2,2,3,2,2,2,2,2,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,0,3,3,3,3,3,3,0,0,1,3,3,0,0,0,0,0,
    3,0,0,0,0,0,0,0,3,3,3,3,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    3,3,3,0,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
};

u32 *screenbuf=NULL, *outbuf=NULL, *dbgbuf=NULL, *palbuf=NULL, *membuf=NULL;
HDC hDCMain, hDCPal, hDCDbg, hDCMem;
HGLRC hRC;
int statbarSize;
int dbgmodesub;
static GLuint backbufTex;
BITMAPV4HEADER BmpInfo;
HWND hWnd, hWndPal, hWndDbg, hWndMem;
int stepclk, dbgmode, memmode; u32 dumpaddr;
int childwinmask=0;

LRESULT CALLBACK WinPalProc(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK WinDbgProc(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK WinMemProc(HWND, UINT, WPARAM, LPARAM);

extern intfptr ARM7execforPtr;

u16 KeyMap[256]={0};

float winX=240.0, winY=160.0;
int bufferFlipped=0, usingGL=0;

void ResizeWin(HWND wnd, int x, int y)
{
    RECT rWnd, rCli, rDiff, rStat;
    HWND hStatus;
    int neww, newh;
    char str[80];

    GetWindowRect(wnd, &rWnd);
    GetClientRect(wnd, &rCli);

    SetRect(&rDiff,
            0,
            0,
            (rWnd.right - rWnd.left) - rCli.right,
            (rWnd.bottom - rWnd.top) - rCli.bottom);

    hStatus = GetDlgItem(wnd, ID_STATUS);
    if(hStatus)
    {
        SendMessage(hStatus, WM_SIZE, 0, 0);
        GetWindowRect(hStatus, &rStat);
    }
    else SetRect(&rStat,0,0,0,0);

    neww = x + rDiff.right;
    newh = y + rDiff.bottom + (rStat.bottom - rStat.top);

    SetWindowPos(wnd, HWND_TOP, 0, 0, neww, newh, SWP_NOMOVE|SWP_NOZORDER);
}

EMUVTBL *getVt()
{
    return &vtbl;
}

#define REGCLASS(name,menu,curs,proc) \
    wc.cbSize        = sizeof(wc); \
    wc.style         = CS_VREDRAW | CS_HREDRAW; \
    wc.lpfnWndProc   = (proc); \
    wc.cbClsExtra    = 0; \
    wc.cbWndExtra    = 0; \
    wc.hInstance     = hInst; \
    wc.hIcon         = LoadImage(hInst,MAKEINTRESOURCE(ID_DSICON32),IMAGE_ICON,32,32,0); \
    wc.hCursor       = (curs); \
    wc.hbrBackground = (HBRUSH)GetStockObject(BLACK_BRUSH); \
    wc.lpszMenuName  = (menu); \
    wc.lpszClassName = (name); \
    wc.hIconSm       = LoadImage(hInst,MAKEINTRESOURCE(ID_DSICON16),IMAGE_ICON,16,16,0); \
    if(!RegisterClassEx(&wc)) RETFAIL("FAIL: Main: WNDCLASS registration.")

#define MAKECHILDWIN(win,class,title) \
    (win) = CreateWindowEx(0, \
	 		     (class), \
	 		     (title), \
			     WS_CLIPSIBLINGS | WS_CLIPCHILDREN | WS_CAPTION | WS_SYSMENU, \
			     CW_USEDEFAULT, CW_USEDEFAULT, \
			     0,0, \
			     hWnd, \
			     NULL, \
			     hInst, \
			     NULL); \
    if(!(win)) RETFAIL("FAIL: Main: Window creation.")

int emuInit(char *file, LOGVTBL *lvtbl, HWND wnd)
{
    char szWndPal[]="DSWndPal";
    char szWndDbg[]="DSWndDbg";
    char szWndMem[]="DSWndMem";

    char str[40], inifile[MAX_PATH*2]; int a, inipathlen;
    PIXELFORMATDESCRIPTOR pfd; GLuint pf;
    HWND hStat; RECT rStat;
    WNDCLASSEX wc; HINSTANCE hInst;

    vtbl.running=0;
    vtbl.animate=0;
    logvt=lvtbl;
    hWnd=wnd;

    hInst=GetModuleHandle(NULL);
    REGCLASS(szWndPal, NULL,LoadCursor(NULL,IDC_ARROW),WinPalProc);
    REGCLASS(szWndDbg, NULL,LoadCursor(NULL,IDC_ARROW),WinDbgProc);
    REGCLASS(szWndMem, NULL,LoadCursor(NULL,IDC_ARROW),WinMemProc);

    MAKECHILDWIN(hWndPal,szWndPal,"Palette");
    MAKECHILDWIN(hWndDbg,szWndDbg,"Debugger");
    MAKECHILDWIN(hWndMem,szWndMem,"Memory View");

    stepclk=0;

    dbgbuf = (u32*)malloc(468*144*2);
    if(!dbgbuf) RETFAIL("FAIL: Main: Buffer init.");
    memset(dbgbuf, 0, 468*144*2);

    if(ARM7init(file,dbgbuf)) RETFAIL("FAIL: Main: ARM7 init.");
    if(GPUinit()) RETFAIL("FAIL: Main: GPU init.");

    ResizeWin(hWndPal, 280, 160);
    ResizeWin(hWndDbg, 480, 160);
    ResizeWin(hWndMem, 360, 160);

    screenbuf = (u32*)malloc(240*160*2);
    if(!screenbuf) RETFAIL("FAIL: Main: Buffer init.");
    memset(screenbuf, 0, 240*160*2);

    outbuf = (u32*)malloc(240*160*2);
    if(!outbuf) RETFAIL("FAIL: Main: Buffer init.");
    memset(outbuf, 0, 240*160*2);

    palbuf = (u32*)malloc(264*152*2);
    if(!palbuf) RETFAIL("FAIL: Main: Buffer init.");
    memset(palbuf, 0, 264*152*2);

    membuf = (u32*)malloc(360*160*2);
    if(!membuf) RETFAIL("FAIL: Main: Buffer init.");
    memset(membuf, 0, 360*160*2);

    hDCMain = GetDC(hWnd);
    hDCPal  = GetDC(hWndPal);
    hDCDbg  = GetDC(hWndDbg);
    hDCMem  = GetDC(hWndMem);

    memset(&BmpInfo, 0, sizeof(BmpInfo));
    BmpInfo.bV4Size = sizeof(BmpInfo);
    BmpInfo.bV4Planes = 1;           // Must be one, apparently
    BmpInfo.bV4BitCount = 16;
    BmpInfo.bV4V4Compression = BI_RGB|BI_BITFIELDS;
    BmpInfo.bV4RedMask = 0x001F;
    BmpInfo.bV4GreenMask = 0x03E0;
    BmpInfo.bV4BlueMask = 0x7C00;

    inipathlen=strstr(logvt->file,"\\log.txt")-logvt->file;
    strncpy(inifile, logvt->file, inipathlen); inifile[inipathlen]=0;
    sprintf(inifile, "%s\\dsemu.ini", inifile);
    GetPrivateProfileString("Keys","UP","26",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_UP;
    GetPrivateProfileString("Keys","DOWN","28",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_DOWN;
    GetPrivateProfileString("Keys","LEFT","25",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_LEFT;
    GetPrivateProfileString("Keys","RIGHT","27",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_RIGHT;
    GetPrivateProfileString("Keys","A","5A",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_A;
    GetPrivateProfileString("Keys","B","58",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_B;
    GetPrivateProfileString("Keys","L","41",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_L;
    GetPrivateProfileString("Keys","R","53",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_R;
    GetPrivateProfileString("Keys","START","0D",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_START;
    GetPrivateProfileString("Keys","SELECT","08",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_SELECT;

    GetPrivateProfileString("General","FixFPS","1",str,4,inifile);
    vtbl.fixfps=strtol(str,NULL,10)&1;

    hStat = GetDlgItem(wnd, ID_STATUS);
    SendMessage(hStat, WM_SIZE, 0, 0);
    GetWindowRect(hStat, &rStat);
    statbarSize=rStat.bottom-rStat.top;

    //---BEGIN-STEAL-------------------------------------------------------
    memset(&pfd,0,sizeof(pfd));
    pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
    pfd.nVersion = 1;
    pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
    pfd.iPixelType = PFD_TYPE_RGBA;
    pfd.cColorBits = 16;
    pfd.cDepthBits = 16;
    pfd.cAccumBits = 0;
    pfd.cStencilBits = 0;
    if(!(pf=ChoosePixelFormat(hDCMain,&pfd)))
    {
        logvt->append("FAIL: OpenGL: Pixel format resolution.");
        usingGL=0; goto glInitEnd;
    }
//    logvt->append("OpenGL: Pixel format found.");
    SetPixelFormat(hDCMain, pf, &pfd);
//    logvt->append("OpenGL: Pixel format selected.");
    hRC=wglCreateContext(hDCMain);
//    logvt->append("OpenGL: Rendering context created.");
    if(wglMakeCurrent(hDCMain, hRC)==FALSE)
    {
        logvt->append("FAIL: OpenGL: Rendering context selection.");
        usingGL=0; goto glInitEnd;
    }
//    logvt->append("OpenGL: Rendering context selected.");

    glPolygonMode (GL_FRONT_AND_BACK, GL_FILL);

    glDisable(GL_CULL_FACE);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_BLEND);

    glShadeModel(GL_SMOOTH);

    glEnable(GL_TEXTURE_2D);
    glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);

    glGenTextures(1,&backbufTex);

    glBindTexture(GL_TEXTURE_2D,backbufTex);
    glPixelStorei(GL_UNPACK_ALIGNMENT,1);
    glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
    glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

    glTexImage2D(GL_TEXTURE_2D,0,GL_RGB5_A1,256,256,0, GL_RGBA,0x8034,outbuf);
    if(glGetError()!=GL_NO_ERROR)
    {
        logvt->append("FAIL: OpenGL: Texture image creation.");
        usingGL=0; goto glInitEnd;
    }
    usingGL=1;
    //---END-STEAL---------------------------------------------------------
glInitEnd:
    if(usingGL) logvt->append("GPU: Using OpenGL.");
    else logvt->append("GPU: Using GDI.");

    ResizeWin(hWnd, 240, 160);

    dbgmode=3; memmode=1;
    dumpaddr=0x04000000;
    for(a=0;a<512;a++) GBAio[a].flags=ioregflags[a];
    if(emuReset()) RETFAIL("FAIL: Main: Initialisation.");
    RETPASS("Main: Emulator initialised.");
}

void gbaGLblit()
{
    __int64 maskr=0x001F001F001F001F;
    __int64 maskg=0x03E003E003E003E0;
    __int64 maskb=0x7C007C007C007C00;
    float ypos=(winY)/((winY)+(float)(statbarSize*2.0))-0.02; //<3 magic numbers

    if(!bufferFlipped)
    {
        __asm {
        push esi
        push edi
        mov esi,screenbuf
        mov edi,outbuf
        mov ecx,240*160/4
fliplp:
        movq mm0,[esi]
        movq mm1,mm0
        movq mm2,mm0
        pand mm0,[maskr]
        pand mm1,[maskg]
        pand mm2,[maskb]
        psllw mm0,11
        psllw mm1,1
        psrlw mm2,9
        por mm0,mm1
        por mm0,mm2
        movq [edi],mm0
        add esi,8
        add edi,8
        dec ecx
        jnz fliplp
        emms
        pop edi
        pop esi
        }
        bufferFlipped=1;
    }

	glClearColor(0,0,0,0);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glBindTexture(GL_TEXTURE_2D,backbufTex);
	glTexSubImage2D(GL_TEXTURE_2D,0,0,0,240,160, GL_RGBA, 0x8034, outbuf);
        if(glGetError()!=GL_NO_ERROR) logvt->append("FAIL: OpenGL: Texture image selection.");

		glEnable(GL_TEXTURE_2D);
		glBegin(GL_QUADS);
			glTexCoord2f(0.9375f,0);
			glVertex2f(1,1);
			glTexCoord2f(0,0);
			glVertex2f(-1,1);
			glTexCoord2f(0,0.625f);
			glVertex2f(-1,-ypos);
			glTexCoord2f(0.9375f,0.625f);
			glVertex2f(1,-ypos);
		glEnd();
		glDisable(GL_TEXTURE_2D);

	SwapBuffers(hDCMain);
}

void emuFini()
{
    char szWndPal[]="DSWndPal";
    char szWndDbg[]="DSWndDbg";
    char szWndMem[]="DSWndMem";

    logvt->append("Main: Emulator shutting down.");
    ARM7status(0,dbgmodesub);
    GPUfini();
    ARM7fini();

    glDeleteTextures(1,&backbufTex);
    if(hRC)
    {
        wglMakeCurrent(NULL,NULL);
        wglDeleteContext(hRC); hRC=NULL;
    }

    memset(screenbuf, 0, 240*160*2);
    memset(dbgbuf, 0, 468*144*2);
    memset(palbuf, 0, 264*152*2);
    memset(membuf, 0, 360*160*2);

        BmpInfo.bV4Width = 240;
        BmpInfo.bV4Height = -160;
        SetDIBitsToDevice(hDCMain, 0, 0, 240, 160, 0, 0, 0, 160,
                          screenbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 468;
    BmpInfo.bV4Height = -144;
    SetDIBitsToDevice(hDCDbg, 6, 8, 468, 144, 0, 0, 0, 144,
                      dbgbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 264;
    BmpInfo.bV4Height = -152;
    SetDIBitsToDevice(hDCPal, 8, 8, 264, 152, 0, 0, 0, 152,
                      palbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 360;
    BmpInfo.bV4Height = -160;
    SetDIBitsToDevice(hDCMem, 6, 0, 360, 160, 0, 0, 0, 160,
                      membuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    DestroyWindow(hWndPal);
    DestroyWindow(hWndDbg);
    DestroyWindow(hWndMem);

    ReleaseDC(hWnd, hDCMain);
    ReleaseDC(hWndPal, hDCPal);
    ReleaseDC(hWndDbg, hDCDbg);
    ReleaseDC(hWndMem, hDCMem);
    hWnd=NULL;    hWndPal=NULL; hWndDbg=NULL; hWndMem=NULL;
    hDCMain=NULL; hDCPal=NULL;  hDCDbg=NULL;  hDCMem=NULL;

    UnregisterClass(szWndPal,GetModuleHandle(NULL));
    UnregisterClass(szWndDbg,GetModuleHandle(NULL));
    UnregisterClass(szWndMem,GetModuleHandle(NULL));

    if(membuf) free(membuf);
    if(dbgbuf) free(dbgbuf);
    if(palbuf) free(palbuf);
    if(screenbuf) free(screenbuf);
    membuf=NULL; dbgbuf=NULL; palbuf=NULL; screenbuf=NULL;
    logvt->append("Main: Shutdown.");
}

void emuRefresh()
{
//    logvt->append("Main: If we fail, we shouldn't be here.");
    GPUpal((u16*)palbuf);
    SendMessage(hWndDbg, WM_PAINT, 0, 0);

    if(usingGL) gbaGLblit();
    else {
        BmpInfo.bV4Width = 240;
        BmpInfo.bV4Height = -160;
        SetDIBitsToDevice(hDCMain, 0, 0, 240, 160, 0, 0, 0, 160,
                          screenbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
    }

    BmpInfo.bV4Width = 468;
    BmpInfo.bV4Height = -144;
    SetDIBitsToDevice(hDCDbg, 6, 8, 468, 144, 0, 0, 0, 144,
                      dbgbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 264;
    BmpInfo.bV4Height = -152;
    SetDIBitsToDevice(hDCPal, 8, 8, 264, 152, 0, 0, 0, 152,
                      palbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 360;
    BmpInfo.bV4Height = -160;
    SetDIBitsToDevice(hDCMem, 6, 0, 360, 160, 0, 0, 0, 160,
                      membuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
}

void emuStep()
{
    static int stepwhere=1;
    stepclk+=ARM7exec();
    ARM7status(0,dbgmode);
    emuMMUDump(dumpaddr,memmode);
    SendMessage(hWndDbg, WM_OFFRESET, 0, 0);
    if(stepclk>=960 && stepwhere==1)
    {
        stepclk=0; stepwhere=2;
//        logvt->append("Line!");
        if(vtbl.gpuon) GPUscanline((u16*)screenbuf);
    }
    if(stepclk>=272 && stepwhere==2)
    {
        stepclk=0; stepwhere=1;
        if(vtbl.gpuon) GPUclearHBL();
    }
    emuRefresh();
}

int emuReset()
{
    int a;
    for(a=0;a<512;a++) GBAio[a].data=0;
    GBAio[REG_KEYINPUT].data=0x03FF;

    if(ARM7reset()) RETFAIL("FAIL: ARM7: Reset.");
    GPUreset();

    memset(screenbuf, 0, 240*160*2);
    memset(dbgbuf, 0, 468*144*2);
    memset(palbuf, 0, 264*152*2);
    memset(membuf, 0, 360*160*2);
    SendMessage(hWndDbg, WM_OFFRESET, 0, 0);

    ARM7status(0,dbgmode);
    emuMMUDump(dumpaddr,memmode);
    if(usingGL) { bufferFlipped=0; gbaGLblit(); }
    else {
        BmpInfo.bV4Width = 240;
        BmpInfo.bV4Height = -160;
        SetDIBitsToDevice(hDCMain, 0, 0, 240, 160, 0, 0, 0, 160,
                          screenbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
    }

    BmpInfo.bV4Width = 468;
    BmpInfo.bV4Height = -144;
    SetDIBitsToDevice(hDCDbg, 6, 8, 468, 144, 0, 0, 0, 144,
                      dbgbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 264;
    BmpInfo.bV4Height = -152;
    SetDIBitsToDevice(hDCPal, 8, 8, 264, 152, 0, 0, 0, 152,
                      palbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 360;
    BmpInfo.bV4Height = -160;
    SetDIBitsToDevice(hDCMem, 6, 0, 360, 160, 0, 0, 0, 160,
                      membuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
    return 0;
}

void emuFrame()
{
    int a,clk;
    for(a=0;a<228;a++)
    {
        clk=0; do clk+=ARM7execforPtr(960-clk); while(clk<960);
        if(vtbl.gpuon) GPUscanline((u16*)screenbuf);
        do clk+=ARM7execforPtr(1232-clk); while(clk<1232);
        if(vtbl.gpuon) GPUclearHBL();
    }
    if(usingGL) { bufferFlipped=0; gbaGLblit(); }
    else {
        BmpInfo.bV4Width = 240;
        BmpInfo.bV4Height = -160;
        SetDIBitsToDevice(hDCMain, 0, 0, 240, 160, 0, 0, 0, 160,
                          screenbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
    }
}

void emuLine()
{
    int clk=0;
    do clk+=ARM7execforPtr(960-clk); while(clk<960);
    if(vtbl.gpuon) GPUscanline((u16*)screenbuf);
    do clk+=ARM7execforPtr(1232-clk); while(clk<1232);
    if(vtbl.gpuon) GPUclearHBL();
    if(usingGL) { bufferFlipped=0; gbaGLblit(); }
    else {
        BmpInfo.bV4Width = 240;
        BmpInfo.bV4Height = -160;
        SetDIBitsToDevice(hDCMain, 0, 0, 240, 160, 0, 0, 0, 160,
                          screenbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
    }
}

void emuLineDbg()
{
    int a,clk=0;
    do clk+=ARM7execforPtr(960-clk); while(clk<960);
    if(vtbl.gpuon) GPUscanline((u16*)screenbuf);
    do clk+=ARM7execforPtr(1232-clk); while(clk<1232);
    if(vtbl.gpuon) GPUclearHBL();
    ARM7status(0,dbgmode);
    emuMMUDump(dumpaddr,memmode);
    SendMessage(hWndDbg, WM_OFFRESET, 0, 0);
    bufferFlipped=0; emuRefresh();
}

void emuPause()
{
    SendMessage(hWnd, WM_COMMAND, ID_MENU_DBG_PAUSE, 0);
    SendMessage(hWndDbg, WM_PAINT, 0, 0);
}

void emuResize(int lParam)
{
    char str[160];
    winX=(float)LOWORD(lParam);
    winY=(float)(HIWORD(lParam)-statbarSize);
//    sprintf(str,"Main: Window resized to %dx%d.",LOWORD(lParam),HIWORD(lParam));
//    logvt->append(str);
    if(usingGL) glViewport(0,0,LOWORD(lParam),HIWORD(lParam));
}

void emuDebugCPU(int off,int mode)
{
    dbgmode=mode;
    ARM7status(off,dbgmode);
}

#define dump16b(p) \
    for(a=0;a<16;a++) \
    { \
        sprintf(str, \
	    "%08X: %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X\n", \
            (p)+a*16, \
	    MMUrdB(1,(p)+a*16+0x0),MMUrdB(1,(p)+a*16+0x1), \
	    MMUrdB(1,(p)+a*16+0x2),MMUrdB(1,(p)+a*16+0x3), \
	    MMUrdB(1,(p)+a*16+0x4),MMUrdB(1,(p)+a*16+0x5), \
	    MMUrdB(1,(p)+a*16+0x6),MMUrdB(1,(p)+a*16+0x7), \
	    MMUrdB(1,(p)+a*16+0x8),MMUrdB(1,(p)+a*16+0x9), \
	    MMUrdB(1,(p)+a*16+0xA),MMUrdB(1,(p)+a*16+0xB), \
	    MMUrdB(1,(p)+a*16+0xC),MMUrdB(1,(p)+a*16+0xD), \
	    MMUrdB(1,(p)+a*16+0xE),MMUrdB(1,(p)+a*16+0xF)); \
        dbgOut(membuf,360,160,str,0,24+a*8,0x7FFF); \
    }

#define dump16h(p) \
    (p)&=0xFFFFFFF0; \
    for(a=0;a<16;a++) \
    { \
        sprintf(str, \
	    "%08X: %04X %04X %04X %04X %04X %04X %04X %04X\n", \
            (p)+a*16, \
	    MMUrdH(1,(p)+a*16+0x0),MMUrdH(1,(p)+a*16+0x2), \
	    MMUrdH(1,(p)+a*16+0x4),MMUrdH(1,(p)+a*16+0x6), \
	    MMUrdH(1,(p)+a*16+0x8),MMUrdH(1,(p)+a*16+0xA), \
	    MMUrdH(1,(p)+a*16+0xC),MMUrdH(1,(p)+a*16+0xE)); \
        dbgOut(membuf,360,160,str,0,24+a*8,0x7FFF); \
    }

#define dump16w(p) \
    for(a=0;a<16;a++) \
    { \
        sprintf(str, \
	    "%08X: %08X %08X %08X %08X\n", \
            (p)+a*16, \
            MMUrdW(1,(p)+a*16+0x0), \
            MMUrdW(1,(p)+a*16+0x4), \
            MMUrdW(1,(p)+a*16+0x8), \
            MMUrdW(1,(p)+a*16+0xC)); \
        dbgOut(membuf,360,160,str,0,24+a*8,0x7FFF); \
    }

void emuMMUDump(u32 addr, int mode)
{
    int a; char str[512];
    dumpaddr=addr;
    dbgOutClear(membuf,360,160);
    memmode=mode;
    switch(mode)
    {
        case 1: dump16b(addr); break;
        case 2: dump16h(addr); break;
        case 3: dump16w(addr); break;
    }
}

void emuGPUCol(int idx)
{
    GPUpalcol((u16*)palbuf,idx);
    BmpInfo.bV4Width = 264;
    BmpInfo.bV4Height = -152;
    SetDIBitsToDevice(hDCPal, 8, 8, 264, 152, 0, 0, 0, 152,
                      palbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
}

void emuKeySet(u32 vk, u32 updown)
{
    if(updown==1)
        GBAio[REG_KEYINPUT].data&=(0xFFFF-KeyMap[vk&255]);
    else if(updown==0)
        GBAio[REG_KEYINPUT].data|=(KeyMap[vk&255]);
}

int emuChildWin(int win)
{
    childwinmask^=win;
    switch(win)
    {
        case 1:
            if(childwinmask&1) { ShowWindow(hWndPal,SW_SHOW); UpdateWindow(hWndPal); }
            else ShowWindow(hWndPal,SW_HIDE);
            break;
        case 2:
            if(childwinmask&2) { ShowWindow(hWndDbg,SW_SHOW); UpdateWindow(hWndDbg); }
            else ShowWindow(hWndDbg,SW_HIDE);
            break;
        case 4:
            if(childwinmask&4) { ShowWindow(hWndMem,SW_SHOW); UpdateWindow(hWndMem); }
            else ShowWindow(hWndMem,SW_HIDE);
            break;
    }
    return (childwinmask&win)?MF_CHECKED:MF_UNCHECKED;
}

LRESULT CALLBACK WinPalProc(HWND wnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    PAINTSTRUCT ps;
    int x,y;
    switch(msg)
    {
	case WM_CLOSE:
            SendMessage(hWnd,WM_COMMAND,ID_MENU_VIEW_PAL,0);
	    return 0;

        case WM_PAINT:
            BeginPaint(wnd, &ps);
	    emuRefresh();
            EndPaint(wnd, &ps);
            return 0;

	case WM_MOUSEMOVE:
	        x=(LOWORD(lParam)/8)-1; y=(HIWORD(lParam)/8)-2;
                if(y>15 && y<20) y=15;
                else if(y>19 && y<36) { y-=20; y+=32; }
                else if(y>35) y=32+15;
	        if(x>16) { x-=17; y+=16; }
	        if(x>=0 && x<=15 && y>=0 && y<=63) emuGPUCol(y*16+x);
	    return 0;

        case WM_KEYDOWN:
            emuKeySet(wParam, 1);
            return DefWindowProc(wnd, msg, wParam, lParam);

        case WM_KEYUP:
            emuKeySet(wParam, 0);
            return DefWindowProc(wnd, msg, wParam, lParam);

        default:
            return DefWindowProc(wnd, msg, wParam, lParam);
    }
}

LRESULT CALLBACK WinDbgProc(HWND wnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    PAINTSTRUCT ps;
    SCROLLINFO si;
    HDC hDC;
    static int offset=0, mode=DASM_TRACK;
    static HWND hWndScrl, hWndBut1, hWndBut2, hWndBut3;
    static HFONT hf;

    switch(msg)
    {
        case WM_CREATE:
            hWndScrl=CreateWindowEx(0, "SCROLLBAR", NULL,
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|SBS_VERT,320,24,12,128,
			 wnd,NULL,GetModuleHandle(NULL),NULL);
	    hWndBut1=CreateWindowEx(0, "BUTTON", "ARM",
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|WS_GROUP|BS_AUTORADIOBUTTON,
	                 6,2,50,20,wnd,(HMENU)ID_BUT_DASM_ARM0,
	                 GetModuleHandle(NULL),NULL);
	    hWndBut2=CreateWindowEx(0, "BUTTON", "Thumb",
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|BS_AUTORADIOBUTTON,
	                 60,2,60,20,wnd,(HMENU)ID_BUT_DASM_THUMB0,
	                 GetModuleHandle(NULL),NULL);
	    hWndBut3=CreateWindowEx(0, "BUTTON", "Track T bit",
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|BS_AUTORADIOBUTTON,
	                 120,2,75,20,wnd,(HMENU)ID_BUT_DASM_TRACK0,
	                 GetModuleHandle(NULL),NULL);
	    SendMessage(hWndBut1, BM_SETCHECK, BST_UNCHECKED,0);
	    SendMessage(hWndBut2, BM_SETCHECK, BST_UNCHECKED,0);
	    SendMessage(hWndBut3, BM_SETCHECK, BST_CHECKED,0);
	    SendMessage(wnd, WM_OFFRESET, 0, 0);
            return 0;

	case WM_CLOSE:
            SendMessage(hWnd,WM_COMMAND,ID_MENU_VIEW_DBG,0);
	    return 0;

        case WM_OFFRESET:
	    si.cbSize=sizeof(si);
	    si.fMask =SIF_RANGE|SIF_PAGE;
	    si.nMin  =-1048576;
	    si.nMax  =1048576;
	    si.nPage =8;
	    SetScrollInfo(hWndScrl, SB_CTL, &si, TRUE);
            offset=0;
            return 0;

	case WM_CTLCOLORSTATIC:
	case WM_CTLCOLORBTN:
	    hDC=(HDC)wParam;
	    SetTextColor(hDC,RGB(255,255,255));
	    SetBkColor(hDC,RGB(0,0,0));
    	    SetBkMode(hDC,OPAQUE);
    	    hf=CreateFont(-MulDiv(6, GetDeviceCaps(hDC, LOGPIXELSY), 72),
		    0, 0, 0, 0, FALSE, 0, 0, 0, 0,
		    0, 0, FF_DONTCARE, "MS Sans Serif");
    	    SelectObject(hDC,hf);
    	    DeleteObject(hf);
    	    return (LONG)GetStockObject(BLACK_BRUSH);

        case WM_PAINT:
            BeginPaint(wnd, &ps);
	    emuDebugCPU(offset,mode);
            BmpInfo.bV4Width = 468;
            BmpInfo.bV4Height = -144;
            SetDIBitsToDevice(hDCDbg, 6, 8, 468, 144, 0, 0, 0, 144,
                      dbgbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
            EndPaint(wnd, &ps);
            return 0;

        case WM_VSCROLL:
            si.cbSize=sizeof(si);
            si.fMask =SIF_ALL;
            GetScrollInfo(hWndScrl, SB_CTL, &si);
            switch(LOWORD(wParam))
            {
                case SB_LINEUP:   offset--; break;
                case SB_LINEDOWN: offset++; break;
                case SB_PAGEUP:   offset-=8; break;
                case SB_PAGEDOWN: offset+=8; break;
            }
            si.fMask=SIF_POS;
            si.nPos=offset;
            SetScrollInfo(hWndScrl, SB_CTL, &si, TRUE);
            InvalidateRect(wnd, NULL, FALSE);
            return 0;

	case WM_COMMAND:
	    switch(wParam)
	    {
	        case ID_BUT_DASM_ARM0:
	            mode=DASM_ARM;
	            InvalidateRect(wnd, NULL, FALSE);
	            break;
	        case ID_BUT_DASM_THUMB0:
	            mode=DASM_THUMB;
	            InvalidateRect(wnd, NULL, FALSE);
	            break;
	        case ID_BUT_DASM_TRACK0:
	            mode=DASM_TRACK;
	            InvalidateRect(wnd, NULL, FALSE);
	            break;
	    }
	    return 0;

        case WM_KEYDOWN:
            emuKeySet(wParam, 1);
            return DefWindowProc(wnd, msg, wParam, lParam);

        case WM_KEYUP:
            emuKeySet(wParam, 0);
            return DefWindowProc(wnd, msg, wParam, lParam);

        default:
            return DefWindowProc(wnd, msg, wParam, lParam);
    }
}

LRESULT CALLBACK WinMemProc(HWND wnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    PAINTSTRUCT ps;
    HFONT hf; HDC hDC;
    static HWND hWndEdit, hWndBut, hWndBut1, hWndBut2, hWndBut3;
    char buf[512], *bufstop;
    static u32 addr, mode;

    switch(msg)
    {
        case WM_CREATE:
//            hWndScrl=CreateWindowEx(0, "SCROLLBAR", NULL,
//	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|SBS_VERT,320,24,12,128,
//			 hWnd,NULL,GetModuleHandle(NULL),NULL);
	    hWndEdit=CreateWindowEx(WS_EX_CLIENTEDGE, "EDIT", NULL,
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|ES_UPPERCASE|ES_LEFT,
	                 6,2,80,18,wnd,(HMENU)ID_EDIT_MEMV_ADDR,
	                 GetModuleHandle(NULL),NULL);
	    SetFocus(hWndEdit);
	    SendMessage(hWndEdit, EM_SETLIMITTEXT, 8, 0);
	    hWndBut =CreateWindowEx(0, "BUTTON", "Go",
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|BS_DEFPUSHBUTTON,
	                 90,2,24,18,wnd,(HMENU)ID_BUT_MEMV_GO,
	                 GetModuleHandle(NULL),NULL);
	    hWndBut1=CreateWindowEx(0, "BUTTON", "8bit",
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|WS_GROUP|BS_AUTORADIOBUTTON,
	                 120,2,50,20,wnd,(HMENU)ID_BUT_MEMV_8,
	                 GetModuleHandle(NULL),NULL);
	    hWndBut2=CreateWindowEx(0, "BUTTON", "16bit",
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|BS_AUTORADIOBUTTON,
	                 170,2,50,20,wnd,(HMENU)ID_BUT_MEMV_16,
	                 GetModuleHandle(NULL),NULL);
	    hWndBut3=CreateWindowEx(0, "BUTTON", "32bit",
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|BS_AUTORADIOBUTTON,
	                 220,2,50,20,wnd,(HMENU)ID_BUT_MEMV_32,
	                 GetModuleHandle(NULL),NULL);
	    SendMessage(hWndBut1, BM_SETCHECK, BST_CHECKED,0);
	    SendMessage(hWndBut2, BM_SETCHECK, BST_UNCHECKED,0);
	    SendMessage(hWndBut3, BM_SETCHECK, BST_UNCHECKED,0);
	    addr=0x04000000; mode=1;
	    return 0;

	case WM_CLOSE:
            SendMessage(hWnd,WM_COMMAND,ID_MENU_VIEW_MEM,0);
	    return 0;

	case WM_CTLCOLORSTATIC:
	case WM_CTLCOLOREDIT:
	case WM_CTLCOLORBTN:
	    hDC=(HDC)wParam;
	    SetTextColor(hDC,RGB(255,255,255));
	    SetBkColor(hDC,RGB(0,0,0));
    	    SetBkMode(hDC,OPAQUE);
    	    hf=CreateFont(-MulDiv(6, GetDeviceCaps(hDC, LOGPIXELSY), 72),
		    0, 0, 0, 0, FALSE, 0, 0, 0, 0,
		    0, 0, FF_DONTCARE, "MS Sans Serif");
    	    SelectObject(hDC,hf);
    	    DeleteObject(hf);
    	    return (LONG)GetStockObject(BLACK_BRUSH);

        case WM_PAINT:
            BeginPaint(hWnd, &ps);
	        emuMMUDump(addr,mode);
	        emuRefresh();
            EndPaint(hWnd, &ps);
            return 0;

	case WM_COMMAND:
	    switch(LOWORD(wParam))
	    {
	        case ID_BUT_MEMV_GO:
	            if(HIWORD(wParam)==BN_CLICKED)
	            {
		        GetWindowText(hWndEdit, (LPSTR)&buf, 512);
		        addr=strtol(buf,&bufstop,16);
 	   	        if((bufstop-buf)!=8)
		        {
		            MessageBox(NULL, "plz 2 enter 8 hex digits.", "Errorthx", MB_OK);
		            return 0;
		        }
		        SendMessage(wnd, WM_PAINT, 0, 0);
	            }
	            break;
	        case ID_BUT_MEMV_8:
	            mode=1; SendMessage(wnd, WM_PAINT,0,0); break;
	        case ID_BUT_MEMV_16:
	            mode=2; SendMessage(wnd, WM_PAINT,0,0); break;
	        case ID_BUT_MEMV_32:
	            mode=3; SendMessage(wnd, WM_PAINT,0,0); break;
	    }
	    return 0;

        case WM_KEYDOWN:
            emuKeySet(wParam, 1);
            return DefWindowProc(wnd, msg, wParam, lParam);

        case WM_KEYUP:
            emuKeySet(wParam, 0);
            return DefWindowProc(wnd, msg, wParam, lParam);

        default:
            return DefWindowProc(wnd, msg, wParam, lParam);
    }
}

/*** EOF:gba.c ***********************************************************/

