/**************************************************************************
* DSemu: GameBoy Advance interface and encapsulation (gba.c)              *
* Released under the terms of the BSD Public Licence                      *
* Imran Nazar (tf@oopsilon.com), 2004                                     *
**************************************************************************/

#include <windows.h>
#include <stdio.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glaux.h>
#include "arm9.h"
#include "arm7.h"
#include "dsmmumain.h"
#include "dsmmusub.h"
#include "gpu.h"
#include "cache.h"
#include "defs.h"
#include "vtbl.h"
#include "win.h"
#include "err.h"
#include "emu.h"
#include "gba.h"
#include "resource.h"
#include "key.h"
#include "dsioreg.h"
#include "int.h"
#include "dbgout.h"
#include "dsregisters.h"
#include "dsaccelerators.h"

#define TOUCHIPC 0x027FF000

RAMWORD *EWRAM, *IWRAM;

LOGVTBL *logvt=NULL;

IOREG DSio_cpu[3][4096];
int DSio_cpu_specific[4096];



u16 ioregflags[]={
    3,3,3,1,3,3,3,3, 2,2,2,2,2,2,2,2, 2,2,2,2,2,2,2,2, 2,2,2,2,2,2,2,2,
    2,2,2,2,3,3,2,0, 3,2,2,0,0,0,0,0, 3,3,3,0,3,0,3,0, 3,3,3,0,3,0,3,0,
    3,3,3,0,0,0,0,0, 3,3,3,3,3,3,3,3, 3,3,3,3,0,0,0,0, 2,2,2,2,2,3,2,2,
    2,2,2,3,2,2,2,2, 2,3,2,2,2,2,2,3, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,

    3,3,3,3,3,3,3,3, 0,0,0,0,0,0,0,0, 3,3,3,3,3,3,0,0, 1,3,3,0,0,0,0,0,
    3,0,0,0,0,0,0,0, 3,3,3,3,1,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,

	// 0x100
    3,3,3,0,3,0,0,0, 3,3,3,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,
    3,3,3,3,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,

    0,0,3,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,
};

u16 *screenbuf=NULL, *outbuf=NULL, *dbgmainbuf=NULL;
u16 *dbgsubbuf=NULL, *palbuf=NULL, *membuf=NULL, *logbuf=NULL;
HDC hDCMain, hDCPal, hDCDbgMain, hDCDbgSub, hDCMem, hDCLog;
BITMAPV4HEADER BmpInfo;
HWND hWnd, hWndPal, hWndDbgMain, hWndDbgSub, hWndMem, hWndLog;
int stepclk, dbgmodemain, dbgmodesub, memmode;
u32 dumpaddr;
int childwinmask=0;
int unscramble = 0;

LRESULT CALLBACK WinPalProc(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK WinDbgMainProc(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK WinDbgSubProc(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK WinMemProc(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK WinLogProc(HWND, UINT, WPARAM, LPARAM);

u16 KeyMap[256]={0};
char inifile[MAX_PATH*2];

HGLRC hRC;
int statbarSize;
static GLuint backbufTex;
float winX=256.0, winY=384.0;
int bufferFlipped=0, usingGL=0;

// Swap the two screens so that what was on the top screen is now
// displayed on the bottom and vice versa.
void SwapScreens() {
	// TODO: Implement this properly by copying screen data around.
	memset(screenbuf, 0, 256*384*2);
}

void ResizeWin(HWND wnd, int x, int y)
{
    RECT rWnd, rCli, rDiff, rStat;
    HWND hStatus;
    int neww, newh;
//    char str[80];

    GetWindowRect(wnd, &rWnd);
    GetClientRect(wnd, &rCli);

    SetRect(&rDiff,
            0,
            0,
            (rWnd.right - rWnd.left) - rCli.right,
            (rWnd.bottom - rWnd.top) - rCli.bottom);

    hStatus = GetDlgItem(wnd, ID_STATUS);
    if(hStatus)
    {
        SendMessage(hStatus, WM_SIZE, 0, 0);
        GetWindowRect(hStatus, &rStat);
    }
    else SetRect(&rStat,0,0,0,0);

    neww = x + rDiff.right;
    newh = y + rDiff.bottom + (rStat.bottom - rStat.top);

    SetWindowPos(wnd, HWND_TOP, 0, 0, neww, newh, SWP_NOMOVE|SWP_NOZORDER);
}

EMUVTBL *getVt()
{
    return &vtbl;
}

#define REGCLASS(name,menu,curs,proc) \
    wc.cbSize        = sizeof(wc); \
    wc.style         = CS_VREDRAW | CS_HREDRAW; \
    wc.lpfnWndProc   = (proc); \
    wc.cbClsExtra    = 0; \
    wc.cbWndExtra    = 0; \
    wc.hInstance     = hInst; \
    wc.hIcon         = LoadImage(hInst,MAKEINTRESOURCE(ID_DSICON32),IMAGE_ICON,32,32,0); \
    wc.hCursor       = (curs); \
    wc.hbrBackground = (HBRUSH)GetStockObject(BLACK_BRUSH); \
    wc.lpszMenuName  = (menu); \
    wc.lpszClassName = (name); \
    wc.hIconSm       = LoadImage(hInst,MAKEINTRESOURCE(ID_DSICON16),IMAGE_ICON,16,16,0); \
    if(!RegisterClassEx(&wc)) RETFAIL("FAIL: Main: WNDCLASS registration.")

#define MAKECHILDWIN(win,class,title) \
    (win) = CreateWindowEx(0, \
	 		     (class), \
	 		     (title), \
			     WS_CLIPSIBLINGS | WS_CLIPCHILDREN | WS_CAPTION | WS_SYSMENU, \
			     CW_USEDEFAULT, CW_USEDEFAULT, \
			     0,0, \
			     hWnd, \
			     NULL, \
			     hInst, \
			     NULL); \
    if(!(win)) RETFAIL("FAIL: Main: Window creation.")

int emuInit(char *file, LOGVTBL *lvtbl, HWND wnd)
{
    char szWndPal[]="DSWndPal";
    char szWndDbgMain[]="DSWndDbgM";
    char szWndDbgSub[]="DSWndDbgS";
    char szWndMem[]="DSWndMem";
	char szWndLog[]="DSWndLog";

    char str[40];
	int a,inipathlen;
    PIXELFORMATDESCRIPTOR pfd; GLuint pf;
    HWND hStat; RECT rStat; WNDCLASSEX wc; HINSTANCE hInst;

    vtbl.running=0;
    vtbl.animate=0;
    logvt=lvtbl;

    stepclk=0;
    hWnd=wnd;

    hInst=GetModuleHandle(NULL);
    REGCLASS(szWndPal, NULL,LoadCursor(NULL,IDC_ARROW),WinPalProc);
    REGCLASS(szWndDbgMain, NULL,LoadCursor(NULL,IDC_ARROW),WinDbgMainProc);
    REGCLASS(szWndDbgSub,  NULL,LoadCursor(NULL,IDC_ARROW),WinDbgSubProc);
    REGCLASS(szWndMem, NULL,LoadCursor(NULL,IDC_ARROW),WinMemProc);
	REGCLASS(szWndLog, NULL,LoadCursor(NULL,IDC_ARROW),WinLogProc);

    MAKECHILDWIN(hWndPal,szWndPal,"Palette");
    MAKECHILDWIN(hWndDbgMain,szWndDbgMain,"Debugger: ARM9");
    MAKECHILDWIN(hWndDbgSub,szWndDbgSub,"Debugger: ARM7");
    MAKECHILDWIN(hWndMem,szWndMem,"Memory View");
	MAKECHILDWIN(hWndLog,szWndLog,"Logging View");

    ResizeWin(hWndPal, 280, 320);
    ResizeWin(hWndDbgMain, 480, 160);
    ResizeWin(hWndDbgSub, 480, 160);
    ResizeWin(hWndMem, 360, 160);
	ResizeWin(hWndLog, 360, 260);

    dbgmainbuf = (u16*)malloc(468*144*2);
    if(!dbgmainbuf) RETFAIL("FAIL: Main: Buffer init.");
    memset(dbgmainbuf, 0, 468*144*2);

    screenbuf = (u16*)malloc(256*384*2);
    if(!screenbuf) RETFAIL("FAIL: Main: Buffer init.");
    memset(screenbuf, 0, 256*384*2);

    outbuf = (u16*)malloc(256*512*2);
    if(!outbuf) RETFAIL("FAIL: Main: Buffer init.");
    memset(outbuf, 0, 256*512*2);

    palbuf = (u16*)malloc(264*312*2);
    if(!palbuf) RETFAIL("FAIL: Main: Buffer init.");
    memset(palbuf, 0, 264*312*2);

    membuf = (u16*)malloc(360*160*2);
    if(!membuf) RETFAIL("FAIL: Main: Buffer init.");
    memset(membuf, 0, 360*160*2);

    logbuf = (u16*)malloc(360*160*2);
    if(!logbuf) RETFAIL("FAIL: Main: Buffer init.");
    memset(logbuf, 0, 360*160*2);

    dbgsubbuf = (u16*)malloc(468*144*2);
    if(!dbgsubbuf) RETFAIL("FAIL: Main: Buffer init.");
    memset(dbgsubbuf, 0, 468*144*2);

    logvt->append("Main: Memory allocated.");

    if(ARM9init(file,(u32*)dbgmainbuf)) RETFAIL("FAIL: Main: ARM9 init.");
    if(ARM7init(file,(u32*)dbgsubbuf)) RETFAIL("FAIL: Main: ARM7 init.");
    if(GPUinit()) RETFAIL("FAIL: Main: GPU init.");

    hDCMain = GetDC(hWnd);
    hDCPal  = GetDC(hWndPal);
    hDCDbgMain  = GetDC(hWndDbgMain);
    hDCDbgSub  = GetDC(hWndDbgSub);
    hDCMem  = GetDC(hWndMem);
	hDCLog = GetDC(hWndLog);

    memset(&BmpInfo, 0, sizeof(BmpInfo));
    BmpInfo.bV4Size = sizeof(BmpInfo);
    BmpInfo.bV4Planes = 1;           // Must be one, apparently
    BmpInfo.bV4BitCount = 16;
    BmpInfo.bV4V4Compression = BI_RGB|BI_BITFIELDS;
    BmpInfo.bV4RedMask = 0x001F;
    BmpInfo.bV4GreenMask = 0x03E0;
    BmpInfo.bV4BlueMask = 0x7C00;

    inipathlen=strstr(logvt->file,"\\log.txt")-logvt->file;
    strncpy(inifile, logvt->file, inipathlen); inifile[inipathlen]=0;
    sprintf(inifile, "%s\\dsemu.ini", inifile);
    GetPrivateProfileString("Keys","UP","26",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_UP;
    GetPrivateProfileString("Keys","DOWN","28",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_DOWN;
    GetPrivateProfileString("Keys","LEFT","25",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_LEFT;
    GetPrivateProfileString("Keys","RIGHT","27",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_RIGHT;
    GetPrivateProfileString("Keys","A","5A",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_A;
    GetPrivateProfileString("Keys","B","58",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_B;
    GetPrivateProfileString("Keys","L","41",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_L;
    GetPrivateProfileString("Keys","R","53",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_R;
	GetPrivateProfileString("Keys","X","47",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=DSKEY_X;
    GetPrivateProfileString("Keys","Y","48",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=DSKEY_Y;
    GetPrivateProfileString("Keys","START","0D",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_START;
    GetPrivateProfileString("Keys","SELECT","08",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_SELECT;

    GetPrivateProfileString("General","FixFPS","1",str,4,inifile);
    vtbl.fixfps=strtol(str,NULL,10)&1;

    logvt->append("Main: INI file read.");

    hStat = GetDlgItem(wnd, ID_STATUS);
    SendMessage(hStat, WM_SIZE, 0, 0);
    GetWindowRect(hStat, &rStat);
    statbarSize=rStat.bottom-rStat.top;

    //---BEGIN-STEAL-------------------------------------------------------
    memset(&pfd,0,sizeof(pfd));
    pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
    pfd.nVersion = 1;
    pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
    pfd.iPixelType = PFD_TYPE_RGBA;
    pfd.cColorBits = 16;
    pfd.cDepthBits = 16;
    pfd.cAccumBits = 0;
    pfd.cStencilBits = 0;
    if(!(pf=ChoosePixelFormat(hDCMain,&pfd)))
    {
        logvt->append("FAIL: OpenGL: Pixel format resolution.");
        usingGL=0;
    } else {
    logvt->append("OpenGL: Pixel format found.");
    SetPixelFormat(hDCMain, pf, &pfd);
    logvt->append("OpenGL: Pixel format selected.");
    hRC=wglCreateContext(hDCMain);
    logvt->append("OpenGL: Rendering context created.");
    if(wglMakeCurrent(hDCMain, hRC)==FALSE)
    {
        logvt->append("FAIL: OpenGL: Rendering context selection.");
        usingGL=0;
    }
    else {
    logvt->append("OpenGL: Rendering context selected.");
    glPolygonMode (GL_FRONT_AND_BACK, GL_FILL);

    glDisable(GL_CULL_FACE);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_BLEND);

    glShadeModel(GL_SMOOTH);

    glEnable(GL_TEXTURE_2D);
    glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);

    glGenTextures(1,&backbufTex);

    glBindTexture(GL_TEXTURE_2D,backbufTex);
    glPixelStorei(GL_UNPACK_ALIGNMENT,1);
    glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
    glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

    logvt->append("OpenGL: About to create texture image.");

    glTexImage2D(GL_TEXTURE_2D,0,GL_RGB5_A1,256,512,0, GL_RGBA,0x8034,outbuf);
    if(glGetError()!=GL_NO_ERROR)
    {
        logvt->append("FAIL: OpenGL: Texture image creation.");
        usingGL=0;
    }
    else usingGL=1;
    }}
    //---END-STEAL---------------------------------------------------------
    if(usingGL) logvt->append("GPU: Using OpenGL.");
    else logvt->append("GPU: Using GDI.");

    ResizeWin(hWnd, 256, 384);

    dbgmodemain=3; memmode=1;
    dbgmodesub=3;
    dumpaddr=0x04000000;
    for(a=0;a<4096;a++) {
		DSio_cpu[0][a].flags=0;
		DSio_cpu[1][a].flags=0;
		DSio_cpu[2][a].flags=0;
	}

    for(a=0;a<512;a++)
    {
        DSio_cpu[0][a].flags=ioregflags[a];
        DSio_cpu[1][a].flags=ioregflags[a];
        DSio_cpu[2][a].flags=ioregflags[a];
        DSio_cpu[0][a+2048].flags=ioregflags[a];
        DSio_cpu[1][a+2048].flags=ioregflags[a];
        DSio_cpu[2][a+2048].flags=ioregflags[a];
		switch(a) {
			case REG_POWERCNT:
			case REG_SERIALCNT:
			case REG_SERIALDATA:
			case REG_IE:
			case REG_IF:
			case REG_IME:
			case REG_TM0D:
			case REG_TM1D:
			case REG_TM2D:
			case REG_TM3D:
			case REG_TM0CNT:
			case REG_TM1CNT:
			case REG_TM2CNT:
			case REG_TM3CNT:
			case REG_IPCSYNC:
				DSio_cpu_specific[a] = 1;
				break;

			default:
				DSio_cpu_specific[a] = 0;
				break;
		}
    }

	// Need to convert these to use the ioflags lookup table.
	DSio_cpu[0][REG_SERIALCNT].flags = 3;
	DSio_cpu[1][REG_SERIALCNT].flags = 3;
	DSio_cpu[2][REG_SERIALCNT].flags = 3;
	DSio_cpu[0][REG_SERIALDATA].flags = 3;
	DSio_cpu[1][REG_SERIALDATA].flags = 3;
	DSio_cpu[2][REG_SERIALDATA].flags = 3;
	DSio_cpu[0][REG_IPCSYNC].flags = 3;
	DSio_cpu[1][REG_IPCSYNC].flags = 3;
	DSio_cpu[2][REG_IPCSYNC].flags = 3;

	DSio_cpu[0][REG_XKEYS].flags = 3;
	DSio_cpu[1][REG_XKEYS].flags = 3;
	DSio_cpu[2][REG_XKEYS].flags = 3;

	// Timer register defauls
	for(a=0x80;a<0x87;++a) {
		DSio_cpu[0][a].flags = 3;
		DSio_cpu[1][a].flags = 3;
		DSio_cpu[2][a].flags = 3;
	}

    if(emuReset()) RETFAIL("FAIL: Main: Initialisation.");
    RETPASS("Main: Emulator initialised.");
}

void gbaGLblit()
{
    __int64 maskr=0x001F001F001F001F;
    __int64 maskg=0x03E003E003E003E0;
    __int64 maskb=0x7C007C007C007C00;
    float ypos=(winY)/((winY)+(float)(statbarSize*2.0f))-0.005f; //<3 magic numbers

    if(!bufferFlipped)
    {
        __asm {
        push esi
        push edi
        mov esi,screenbuf
        mov edi,outbuf
        mov ecx,256*384/4
fliplp:
        movq mm0,[esi]
        movq mm1,mm0
        movq mm2,mm0
        pand mm0,[maskr]
        pand mm1,[maskg]
        pand mm2,[maskb]
        psllw mm0,11
        psllw mm1,1
        psrlw mm2,9
        por mm0,mm1
        por mm0,mm2
        movq [edi],mm0
        add esi,8
        add edi,8
        dec ecx
        jnz fliplp
        emms
        pop edi
        pop esi
        }
        bufferFlipped=1;
    }

	glClearColor(0,0,0,0);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glBindTexture(GL_TEXTURE_2D,backbufTex);
	glTexSubImage2D(GL_TEXTURE_2D,0,0,0,256,384, GL_RGBA, 0x8034, outbuf);
        if(glGetError()!=GL_NO_ERROR) logvt->append("FAIL: OpenGL: Texture image selection.");

		glEnable(GL_TEXTURE_2D);
		glBegin(GL_QUADS);
			glTexCoord2f(1,0);
			glVertex2f(1,1);
			glTexCoord2f(0,0);
			glVertex2f(-1,1);
			glTexCoord2f(0,0.75f);
			glVertex2f(-1,-ypos);
			glTexCoord2f(1,0.75f);
			glVertex2f(1,-ypos);
		glEnd();
		glDisable(GL_TEXTURE_2D);

	SwapBuffers(hDCMain);
}

void emuFini()
{
    char szWndPal[]="DSWndPal";
    char szWndDbgMain[]="DSWndDbgM";
    char szWndDbgSub[]="DSWndDbgS";
    char szWndMem[]="DSWndMem";
	char szWndLog[]="DSWndLog";

    logvt->append("Main: Emulator shutting down.");
    GPUfini();
    ARM7fini();
    ARM9fini();

    glDeleteTextures(1,&backbufTex);
    if(hRC)
    {
        wglMakeCurrent(NULL,NULL);
        wglDeleteContext(hRC); hRC=NULL;
    }

    memset(screenbuf, 0, 256*384*2);
    memset(dbgmainbuf, 0, 468*144*2);
    memset(dbgsubbuf, 0, 468*144*2);
    memset(palbuf, 0, 128*144*2);
    memset(membuf, 0, 360*160*2);
	memset(logbuf, 0, 360*160*2);

    BmpInfo.bV4Width = 256;
    BmpInfo.bV4Height = -384;
    SetDIBitsToDevice(hDCMain, 0, 0, 256, 384, 0, 0, 0, 384,
                      screenbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 468;
    BmpInfo.bV4Height = -144;
    SetDIBitsToDevice(hDCDbgMain, 6, 8, 468, 144, 0, 0, 0, 144,
                      dbgmainbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 468;
    BmpInfo.bV4Height = -144;
    SetDIBitsToDevice(hDCDbgSub, 6, 8, 468, 144, 0, 0, 0, 144,
                      dbgsubbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 264;
    BmpInfo.bV4Height = -312;
    SetDIBitsToDevice(hDCPal, 8, 8, 264, 312, 0, 0, 0, 312,
                      palbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 360;
    BmpInfo.bV4Height = -160;
    SetDIBitsToDevice(hDCMem, 6, 0, 360, 160, 0, 0, 0, 160,
                      membuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 360;
    BmpInfo.bV4Height = -160;
    SetDIBitsToDevice(hDCLog, 6, 0, 360, 160, 0, 0, 0, 160,
                      logbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    ReleaseDC(hWnd, hDCMain);
    ReleaseDC(hWndPal, hDCPal);
    ReleaseDC(hWndDbgMain, hDCDbgMain);
    ReleaseDC(hWndDbgSub, hDCDbgSub);
    ReleaseDC(hWndMem, hDCMem);
	ReleaseDC(hWndLog, hDCLog);

    DestroyWindow(hWndPal);
    DestroyWindow(hWndDbgMain);
    DestroyWindow(hWndDbgSub);
    DestroyWindow(hWndMem);
	DestroyWindow(hWndLog);

    hWnd=NULL;    hWndPal=NULL; hWndDbgMain=NULL; hWndDbgSub=NULL; hWndMem=NULL; hWndLog=NULL;
    hDCMain=NULL; hDCPal=NULL;  hDCDbgMain=NULL;  hDCDbgSub=NULL;  hDCMem=NULL;
    
    UnregisterClass(szWndPal,GetModuleHandle(NULL));
    UnregisterClass(szWndDbgMain,GetModuleHandle(NULL));
    UnregisterClass(szWndDbgSub,GetModuleHandle(NULL));
    UnregisterClass(szWndMem,GetModuleHandle(NULL));
	UnregisterClass(szWndLog,GetModuleHandle(NULL));

	if(logbuf) free(logbuf);
    if(membuf) free(membuf);
    if(dbgmainbuf) free(dbgmainbuf);
    if(dbgsubbuf) free(dbgsubbuf);
    if(palbuf) free(palbuf);
    if(screenbuf) free(screenbuf);
    logbuf=NULL; membuf=NULL; dbgmainbuf=NULL; dbgsubbuf=NULL; palbuf=NULL; screenbuf=NULL;
}

void emuRefresh()
{
//    logvt->append("Main: If we fail, we shouldn't be here.");
    GPUpal((u16*)palbuf);
    SendMessage(hWndDbgMain, WM_PAINT, 0, 0);
    SendMessage(hWndDbgSub, WM_PAINT, 0, 0);

    if(usingGL) gbaGLblit();
    else {
    BmpInfo.bV4Width = 256;
    BmpInfo.bV4Height = -384;
    SetDIBitsToDevice(hDCMain, 0, 0, 256, 384, 0, 0, 0, 384,
                      screenbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
    }

    BmpInfo.bV4Width = 468;
    BmpInfo.bV4Height = -144;
    SetDIBitsToDevice(hDCDbgMain, 6, 8, 468, 144, 0, 0, 0, 144,
                      dbgmainbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 468;
    BmpInfo.bV4Height = -144;
    SetDIBitsToDevice(hDCDbgSub, 6, 8, 468, 144, 0, 0, 0, 144,
                      dbgsubbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 264;
    BmpInfo.bV4Height = -312;
    SetDIBitsToDevice(hDCPal, 8, 8, 264, 312, 0, 0, 0, 312,
                      palbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 360;
    BmpInfo.bV4Height = -160;
    SetDIBitsToDevice(hDCMem, 6, 0, 360, 160, 0, 0, 0, 160,
                      membuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
    BmpInfo.bV4Width = 360;
    BmpInfo.bV4Height = -160;
    SetDIBitsToDevice(hDCLog, 6, 0, 360, 160, 0, 0, 0, 160,
                      logbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);


}

void emuStep()
{
	// TODO: What if the ARM9 is halted but the ARM7 isn't?
    static int stepwhere=1;
    stepclk+=ARM9execfor(1);
    ARM7execfor(1);
    ARM9status(0,dbgmodemain);
    ARM7status(0,dbgmodesub);
    emuMMUDump(dumpaddr,memmode);
    SendMessage(hWndDbgMain, WM_OFFRESET, 0, 0);
    if((cpu9Halted || stepclk>=1536) && stepwhere==1)
    {
        stepclk=0; stepwhere=2;
        if(vtbl.gpuon) GPUscanline((u16*)screenbuf);
		if(usingGL) { bufferFlipped=0; gbaGLblit(); }
		else {
			BmpInfo.bV4Width = 256;
			BmpInfo.bV4Height = -384;
			SetDIBitsToDevice(hDCMain, 0, 0, 256, 384, 0, 0, 0, 384,
							screenbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
		}
        InvalidateRect(hWnd, NULL, FALSE);
		UpdateWindow(hWnd);
    }
    if((cpu9Halted || stepclk>=594) && stepwhere==2)
    {
        stepclk=0; stepwhere=1;
        if(vtbl.gpuon) GPUclearHBL();
		if(usingGL) { bufferFlipped=0; gbaGLblit(); }
		else {
			BmpInfo.bV4Width = 256;
			BmpInfo.bV4Height = -384;
			SetDIBitsToDevice(hDCMain, 0, 0, 256, 384, 0, 0, 0, 384,
							screenbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
		}
        InvalidateRect(hWnd, NULL, FALSE);
		UpdateWindow(hWnd);
    }

    emuRefresh();
}

int emuReset()
{
    int a;
    for(a=0;a<4096;a++) {
		DSio_cpu[0][a].data=0;
		DSio_cpu[1][a].data=0;
		DSio_cpu[2][a].data=0;
	}
    DSio_cpu[0][REG_KEYINPUT].data=0xFFFF;
    DSio_cpu[1][REG_KEYINPUT].data=0xFFFF;
    DSio_cpu[2][REG_KEYINPUT].data=0xFFFF;

	/* Set default values for registers (verified on hardware) */
	REG(ARM7_REG,POWERCNT) = 3;
	REG(ARM9_REG,POWERCNT) = 0x8000;

	// Set default XKEYS
	REG(ARMX_REG, XKEYS) = 0x43;

	// Other registers
	initialize_registers();


	// Set touchscreen calibration data
	MMUwrH(0,0x027FFCD8, 400);	//X1
	MMUwrH(0,0x027FFCDE, 3100);	//X2
	MMUwrH(0,0x027FFCDA, 500);	//Y1
	MMUwrH(0,0x027FFCE0, 2700);	//Y2

    logvt->append("Main: Resetting.");

    if(ARM9reset()) RETFAIL("FAIL: ARM9: Reset.");
    if(ARM7reset()) RETFAIL("FAIL: ARM7: Reset.");
    GPUreset();

#if 0
    MMUMainwrH(0,TOUCHIPC+20,0xFFFF);
#endif
    memset(screenbuf, 0, 256*384*2);
    memset(dbgmainbuf, 0, 468*144*2);
    memset(dbgsubbuf, 0, 468*144*2);
    memset(palbuf, 0, 128*144*2);
    memset(membuf, 0, 360*160*2);
	memset(logbuf, 0, 360*160*2);
    SendMessage(hWndDbgMain, WM_OFFRESET, 0, 0);
    SendMessage(hWndDbgSub, WM_OFFRESET, 0, 0);

    ARM9status(0,dbgmodemain);
    ARM7status(0,dbgmodesub);
    emuMMUDump(dumpaddr,memmode);

    emuRefresh();
    return 0;
}

// Process any outstanding GUI messages. Returns true if
// emulation is to continue or false to exit.
int ProcessGUIMessages()
{
	MSG msg;
	while(PeekMessage(&msg, 0, 0, 0, PM_NOREMOVE)) {
       if(msg.message == WM_QUIT)
	   {
		   return 0;
	   }
	   else
	   {
			PeekMessage(&msg, 0, 0, 0, PM_REMOVE);
			if(!TranslateAccelerator(hWnd, accelTable, &msg))
			{
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
		}		
	}
	return 1;
}



void emuFrame()
{
    int a,clk9,clk7,quit=0;
	char str[100];
    GetPrivateProfileString("General","Unscramble","0",str,16,inifile);
    unscramble=strtol(str,NULL,10);

    for(a=0;a<263;a++)
    {
        clk9=0; clk7=0;
		quit=!ProcessGUIMessages();
        do {
			int clk = ARM9execfor(1536-clk9); 
			clk9 += clk;
//			UpdateTimer0(clk);
		} while(!quit && clk9<1536 && !cpu9Halted);
			quit=!ProcessGUIMessages();
        do {
			clk7+=ARM7execfor(768-clk7); 
		} while(!quit && clk7<768);

        if(!quit && vtbl.gpuon) 
			GPUscanline((u16*)screenbuf);
			quit=!ProcessGUIMessages();

        do {
			int clk = ARM9execfor(2130-clk9); 
			clk9+=clk;
//			UpdateTimer0(clk);
            UpdateAcceleratorsRegisters(clk);
		} while(!quit &&  clk9<2130 && !cpu9Halted);
        do {
			clk7+=ARM7execfor(1065-clk7); 
		} while(!quit &&  clk7<1065);

        if(!quit && vtbl.gpuon) 
			GPUclearHBL();
    }
    if(usingGL) { bufferFlipped=0; gbaGLblit(); }
    else {
    BmpInfo.bV4Width = 256;
    BmpInfo.bV4Height = -384;
    SetDIBitsToDevice(hDCMain, 0, 0, 256, 384, 0, 0, 0, 384,
                      screenbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
    }
}

void emuLine()
{
    int clk9=0, clk7=0;
        do clk9+=ARM9execfor(1536-clk9); while(clk9<1536);
        do clk7+=ARM7execfor(768-clk7); while(clk7<768);
        if(vtbl.gpuon) GPUscanline((u16*)screenbuf);
        do clk9+=ARM9execfor(2130-clk9); while(clk9<2130);
        do clk7+=ARM7execfor(1065-clk7); while(clk7<1065);
        if(vtbl.gpuon) GPUclearHBL();
    if(usingGL) { bufferFlipped=0; gbaGLblit(); }
    else {
    BmpInfo.bV4Width = 256;
    BmpInfo.bV4Height = -384;
    SetDIBitsToDevice(hDCMain, 0, 0, 256, 384, 0, 0, 0, 384,
                      screenbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
    }
}

void emuLineDbg()
{
    int /*a,*/clk9=0, clk7=0;
        do clk9+=ARM9execfor(1536-clk9); while(clk9<1536);
        do clk7+=ARM7execfor(768-clk7); while(clk7<768);
        if(vtbl.gpuon) GPUscanline((u16*)screenbuf);
        do clk9+=ARM9execfor(2130-clk9); while(clk9<2130);
        do clk7+=ARM7execfor(1065-clk7); while(clk7<1065);
        if(vtbl.gpuon) GPUclearHBL();
    ARM9status(0,dbgmodemain);
    ARM7status(0,dbgmodesub);
    emuMMUDump(dumpaddr,memmode);
    SendMessage(hWndDbgMain, WM_OFFRESET, 0, 0);
    SendMessage(hWndDbgSub, WM_OFFRESET, 0, 0);
    bufferFlipped=0; emuRefresh();
}

void emuPause()
{
    SendMessage(hWnd, WM_COMMAND, ID_MENU_DBG_PAUSE, 0);
    SendMessage(hWndDbgMain, WM_PAINT, 0, 0);
    SendMessage(hWndDbgSub, WM_PAINT, 0, 0);
}

void emuResize(int lParam)
{
//    char str[160];
    winX=(float)LOWORD(lParam);
    winY=(float)(HIWORD(lParam)-statbarSize);
//    sprintf(str,"Main: Window resized to %dx%d.",LOWORD(lParam),HIWORD(lParam));
//    logvt->append(str);
    if(usingGL) glViewport(0,0,LOWORD(lParam),HIWORD(lParam));
}

void emuDebugCPU(int offmain,int modemain,int offsub,int modesub)
{
    dbgmodemain=modemain;
    dbgmodesub=modesub;
    ARM9status(offmain,dbgmodemain);
    ARM7status(offsub,dbgmodesub);
}

#define dump16b(p) \
    for(a=0;a<16;a++) \
    { \
        sprintf(str, \
	    "%08X: %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X\n", \
            (p)+a*16, \
	    MMUMainrdB(1,(p)+a*16+0x0),MMUMainrdB(1,(p)+a*16+0x1), \
	    MMUMainrdB(1,(p)+a*16+0x2),MMUMainrdB(1,(p)+a*16+0x3), \
	    MMUMainrdB(1,(p)+a*16+0x4),MMUMainrdB(1,(p)+a*16+0x5), \
	    MMUMainrdB(1,(p)+a*16+0x6),MMUMainrdB(1,(p)+a*16+0x7), \
	    MMUMainrdB(1,(p)+a*16+0x8),MMUMainrdB(1,(p)+a*16+0x9), \
	    MMUMainrdB(1,(p)+a*16+0xA),MMUMainrdB(1,(p)+a*16+0xB), \
	    MMUMainrdB(1,(p)+a*16+0xC),MMUMainrdB(1,(p)+a*16+0xD), \
	    MMUMainrdB(1,(p)+a*16+0xE),MMUMainrdB(1,(p)+a*16+0xF)); \
        dbgOut(membuf,360,160,str,0,24+a*8,0x7FFF); \
    }

#define dump16h(p) \
    (p)&=0xFFFFFFF0; \
    for(a=0;a<16;a++) \
    { \
        sprintf(str, \
	    "%08X: %04X %04X %04X %04X %04X %04X %04X %04X\n", \
            (p)+a*16, \
	    MMUMainrdH(1,(p)+a*16+0x0),MMUMainrdH(1,(p)+a*16+0x2), \
	    MMUMainrdH(1,(p)+a*16+0x4),MMUMainrdH(1,(p)+a*16+0x6), \
	    MMUMainrdH(1,(p)+a*16+0x8),MMUMainrdH(1,(p)+a*16+0xA), \
	    MMUMainrdH(1,(p)+a*16+0xC),MMUMainrdH(1,(p)+a*16+0xE)); \
        dbgOut(membuf,360,160,str,0,24+a*8,0x7FFF); \
    }

#define dump16w(p) \
    for(a=0;a<16;a++) \
    { \
        sprintf(str,"%08X: ",(p)+a*16); \
        dbgOut(membuf,360,160,str,0,24+a*8,0x7FFF); \
        w=(p)+a*16+0x0; sprintf(str,"%08X",MMUMainrdW(1,w)); \
        if(arm9reg.r[13]==w) dbgOut(membuf,360,160,str,60,24+a*8,0x03FF); \
        else                 dbgOut(membuf,360,160,str,60,24+a*8,0x7FFF); \
        w=(p)+a*16+0x4; sprintf(str,"%08X",MMUMainrdW(1,w)); \
        if(arm9reg.r[13]==w) dbgOut(membuf,360,160,str,114,24+a*8,0x03FF); \
        else                 dbgOut(membuf,360,160,str,114,24+a*8,0x7FFF); \
        w=(p)+a*16+0x8; sprintf(str,"%08X",MMUMainrdW(1,w)); \
        if(arm9reg.r[13]==w) dbgOut(membuf,360,160,str,168,24+a*8,0x03FF); \
        else                 dbgOut(membuf,360,160,str,168,24+a*8,0x7FFF); \
        w=(p)+a*16+0xC; sprintf(str,"%08X",MMUMainrdW(1,w)); \
        if(arm9reg.r[13]==w) dbgOut(membuf,360,160,str,222,24+a*8,0x03FF); \
        else                 dbgOut(membuf,360,160,str,222,24+a*8,0x7FFF); \
    }

void emuMMUDump(u32 addr, int mode)
{
    int a; u32 w; char str[512];
    dumpaddr=addr;
    dbgOutClear(membuf,360,160);
    memmode=mode;
    switch(mode)
    {
        case 1: dump16b(addr); break;
        case 2: dump16h(addr); break;
        case 3: dump16w(addr); break;
    }
}

void emuGPUCol(int idx)
{
    GPUpalcol((u16*)palbuf,idx);
    BmpInfo.bV4Width = 264;
    BmpInfo.bV4Height = -312;
    SetDIBitsToDevice(hDCPal, 8, 8, 264, 312, 0, 0, 0, 312,
                      palbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
}

void emuKeySet(u32 vk, u32 updown)
{
    int touchx, touchy; //u16 t;
//    char str[80];
    if(vk==VK_LBUTTON)
    {
		extern int spi_touch_x;
		extern int spi_touch_y;

		if(updown & 0x80000000) {
			touchx=(LOWORD(updown)&0xFFFF)*14;
			touchy=(((HIWORD(updown)&0x7FFF)-192)*8917)>>9;

			spi_touch_x = touchx;
			spi_touch_y = touchy;

			// We're touching the screen so clear bit 6 of XKEYS
			REG(ARMX_REG, XKEYS) &= ~0x40;
		}
		else {
			// Not touching the screen. Set bit 6 of XKEYS
			REG(ARMX_REG, XKEYS) |= 0x40;
		}
#if 0
        MMUMainwrH(0,TOUCHIPC+4, touchx);
        MMUMainwrH(0,TOUCHIPC+6, touchy);
        t=MMUMainrdH(0,TOUCHIPC+20); t&=0xFFBF;
        if(!(updown&0x80000000)) t|=0x40;
        MMUMainwrH(0,TOUCHIPC+20,t);
#endif
/*        sprintf(str,"WND: Touch: %d,%d - %d",LOWORD(updown),
                                             (HIWORD(updown)&0x7FFF),
                                             ((updown&0x80000000)?1:0));
        logvt->append(str);
*/
    } else {
        if(updown==1) {
			int keys_cr  = REG(ARMX_REG, KEYCNT);
			int fire_int = 0;
			int bitmask  = 0xFFFF-KeyMap[vk&255];
            REG(0, KEYINPUT) &=bitmask;
            REG(1, KEYINPUT) &=bitmask;
            REG(2, KEYINPUT) &=bitmask;

			if(!(bitmask & DSKEY_X))
				REG(ARMX_REG, XKEYS) &= ~(1<<0);
			if(!(bitmask & DSKEY_Y))
				REG(ARMX_REG, XKEYS) &= ~(1<<1);
			// Check the keys control register to see if interrupts are
			// enabled. If so, and the given keys are pressed, fire the 
			// interrupt. 
			//
			// TODO: Check if the ARM7 gets the interrupt as well. Currently
			// assuming yes.
			if(keys_cr & (1<<14)) {
				int bm = ~(REG(ARMX_REG, KEYINPUT));
				if(keys_cr & (1<<15)) {
					// All keys must be pressed
					fire_int = !((bm&keys_cr)^keys_cr);
				}
				else {
					// Any key can be pressed
					fire_int = bm&keys_cr;
				}
				if(fire_int) 
					IntFire(INT_KEY);
			}					
		}
        else if(updown==0) {
			int bitmask = KeyMap[vk&255];
            DSio_cpu[0][REG_KEYINPUT].data|=bitmask;
            DSio_cpu[1][REG_KEYINPUT].data|=bitmask;
            DSio_cpu[2][REG_KEYINPUT].data|=bitmask;
			if(bitmask & DSKEY_X)
				REG(ARMX_REG, XKEYS) |= (1<<0);
			if(bitmask & DSKEY_Y)
				REG(ARMX_REG, XKEYS) |= (1<<1);
		}
    }
}

int emuChildWin(int win)
{
    childwinmask^=win;
    switch(win)
    {
        case 1:
            if(childwinmask&1) { ShowWindow(hWndPal,SW_SHOW); UpdateWindow(hWndPal); }
            else ShowWindow(hWndPal,SW_HIDE);
            break;
        case 2:
            if(childwinmask&2)
            {
                ShowWindow(hWndDbgMain,SW_SHOW); UpdateWindow(hWndDbgMain);
                ShowWindow(hWndDbgSub,SW_SHOW);  UpdateWindow(hWndDbgSub);
            } else {
                ShowWindow(hWndDbgMain,SW_HIDE);
                ShowWindow(hWndDbgSub,SW_HIDE);
            }
            break;
        case 4:
            if(childwinmask&4) { ShowWindow(hWndMem,SW_SHOW); UpdateWindow(hWndMem); }
            else ShowWindow(hWndMem,SW_HIDE);
            break;
        case 5:
            if(childwinmask&5) { ShowWindow(hWndLog,SW_SHOW); UpdateWindow(hWndLog); }
            else ShowWindow(hWndLog,SW_HIDE);
            break;
    }
    return (childwinmask&win)?MF_CHECKED:MF_UNCHECKED;
}

LRESULT CALLBACK WinPalProc(HWND wnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    PAINTSTRUCT ps;
    int x,y;
    switch(msg)
    {
	case WM_CLOSE:
            SendMessage(hWnd,WM_COMMAND,ID_MENU_VIEW_PAL,0);
	    return 0;

        case WM_PAINT:
            BeginPaint(wnd, &ps);
	    emuRefresh();
            EndPaint(wnd, &ps);
            return 0;

	case WM_MOUSEMOVE:
	        x=(LOWORD(lParam)/8)-1; y=(HIWORD(lParam)/8)-2;
                if(y>15 && y<20) y=15;
                else if(y>19 && y<36) { y-=20; y+=32; }
                else if(y>35) y=32+15;
	        if(x>16) { x-=17; y+=16; }
	        if(x>=0 && x<=15 && y>=0 && y<=63) emuGPUCol(y*16+x);
	    return 0;

        case WM_KEYDOWN:
            emuKeySet(wParam, 1);
            return DefWindowProc(wnd, msg, wParam, lParam);

        case WM_KEYUP:
            emuKeySet(wParam, 0);
            return DefWindowProc(wnd, msg, wParam, lParam);

        default:
            return DefWindowProc(wnd, msg, wParam, lParam);
    }
}

LRESULT CALLBACK WinDbgMainProc(HWND wnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    PAINTSTRUCT ps;
    SCROLLINFO si;
    HDC hDC;
    static int offsetmain=0, modemain=DASM_TRACK;
    static HWND hWndScrlMain, hWndBut1Main, hWndBut2Main, hWndBut3Main;
    static HFONT hf;

    switch(msg)
    {
        case WM_CREATE:
            hWndScrlMain=CreateWindowEx(0, "SCROLLBAR", NULL,
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|SBS_VERT,320,24,12,128,
			 wnd,NULL,GetModuleHandle(NULL),NULL);
	    hWndBut1Main=CreateWindowEx(0, "BUTTON", "ARM",
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|WS_GROUP|BS_AUTORADIOBUTTON,
	                 6,2,50,20,wnd,(HMENU)ID_BUT_DASM_ARM0,
	                 GetModuleHandle(NULL),NULL);
	    hWndBut2Main=CreateWindowEx(0, "BUTTON", "Thumb",
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|BS_AUTORADIOBUTTON,
	                 60,2,60,20,wnd,(HMENU)ID_BUT_DASM_THUMB0,
	                 GetModuleHandle(NULL),NULL);
	    hWndBut3Main=CreateWindowEx(0, "BUTTON", "Track T bit",
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|BS_AUTORADIOBUTTON,
	                 120,2,75,20,wnd,(HMENU)ID_BUT_DASM_TRACK0,
	                 GetModuleHandle(NULL),NULL);
	    SendMessage(hWndBut1Main, BM_SETCHECK, BST_UNCHECKED,0);
	    SendMessage(hWndBut2Main, BM_SETCHECK, BST_UNCHECKED,0);
	    SendMessage(hWndBut3Main, BM_SETCHECK, BST_CHECKED,0);
	    SendMessage(wnd, WM_OFFRESET, 0, 0);
            return 0;

	case WM_CLOSE:
            SendMessage(hWnd,WM_COMMAND,ID_MENU_VIEW_DBG,0);
	    return 0;

        case WM_OFFRESET:
	    si.cbSize=sizeof(si);
	    si.fMask =SIF_RANGE|SIF_PAGE;
	    si.nMin  =-1048576;
	    si.nMax  =1048576;
	    si.nPage =8;
	    SetScrollInfo(hWndScrlMain, SB_CTL, &si, TRUE);
            offsetmain=0;
            return 0;

	case WM_CTLCOLORSTATIC:
	case WM_CTLCOLORBTN:
	    hDC=(HDC)wParam;
	    SetTextColor(hDC,RGB(255,255,255));
	    SetBkColor(hDC,RGB(0,0,0));
    	    SetBkMode(hDC,OPAQUE);
    	    hf=CreateFont(-MulDiv(6, GetDeviceCaps(hDC, LOGPIXELSY), 72),
		    0, 0, 0, 0, FALSE, 0, 0, 0, 0,
		    0, 0, FF_DONTCARE, "MS Sans Serif");
    	    SelectObject(hDC,hf);
    	    DeleteObject(hf);
    	    return (LONG)GetStockObject(BLACK_BRUSH);

        case WM_PAINT:
            BeginPaint(wnd, &ps);
            dbgmodemain=modemain;
            ARM9status(offsetmain,dbgmodemain);
            BmpInfo.bV4Width = 468;
            BmpInfo.bV4Height = -144;
            SetDIBitsToDevice(hDCDbgMain, 6, 8, 468, 144, 0, 0, 0, 144,
                      dbgmainbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
            EndPaint(wnd, &ps);
            return 0;

        case WM_VSCROLL:
            si.cbSize=sizeof(si);
            si.fMask =SIF_ALL;
                    GetScrollInfo(hWndScrlMain, SB_CTL, &si);
                    switch(LOWORD(wParam))
                    {
                        case SB_LINEUP:   offsetmain--; break;
                        case SB_LINEDOWN: offsetmain++; break;
                        case SB_PAGEUP:   offsetmain-=8; break;
                        case SB_PAGEDOWN: offsetmain+=8; break;
                    }
                    si.fMask=SIF_POS;
                    si.nPos=offsetmain;
                    SetScrollInfo(hWndScrlMain, SB_CTL, &si, TRUE);
            InvalidateRect(wnd, NULL, FALSE);
            return 0;

	case WM_COMMAND:
	    switch(wParam)
	    {
	        case ID_BUT_DASM_ARM0:
	            modemain=DASM_ARM;
	            InvalidateRect(wnd, NULL, FALSE);
	            break;
	        case ID_BUT_DASM_THUMB0:
	            modemain=DASM_THUMB;
	            InvalidateRect(wnd, NULL, FALSE);
	            break;
	        case ID_BUT_DASM_TRACK0:
	            modemain=DASM_TRACK;
	            InvalidateRect(wnd, NULL, FALSE);
	            break;
	    }
	    return 0;

        case WM_KEYDOWN:
            emuKeySet(wParam, 1);
            return DefWindowProc(wnd, msg, wParam, lParam);

        case WM_KEYUP:
            emuKeySet(wParam, 0);
            return DefWindowProc(wnd, msg, wParam, lParam);

        default:
            return DefWindowProc(wnd, msg, wParam, lParam);
    }
}

LRESULT CALLBACK WinDbgSubProc(HWND wnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    PAINTSTRUCT ps;
    SCROLLINFO si;
    HDC hDC;
    static int offsetsub=0, modesub=DASM_TRACK;
    static HWND hWndScrlSub, hWndBut1Sub, hWndBut2Sub, hWndBut3Sub;
    static HFONT hf;

    switch(msg)
    {
        case WM_CREATE:
            hWndScrlSub=CreateWindowEx(0, "SCROLLBAR", NULL,
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|SBS_VERT,320,24,12,128,
			 wnd,NULL,GetModuleHandle(NULL),NULL);
	    hWndBut1Sub=CreateWindowEx(0, "BUTTON", "ARM",
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|WS_GROUP|BS_AUTORADIOBUTTON,
	                 6,2,50,20,wnd,(HMENU)ID_BUT_DASM_ARM1,
	                 GetModuleHandle(NULL),NULL);
	    hWndBut2Sub=CreateWindowEx(0, "BUTTON", "Thumb",
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|BS_AUTORADIOBUTTON,
	                 60,2,60,20,wnd,(HMENU)ID_BUT_DASM_THUMB1,
	                 GetModuleHandle(NULL),NULL);
	    hWndBut3Sub=CreateWindowEx(0, "BUTTON", "Track T bit",
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|BS_AUTORADIOBUTTON,
	                 120,2,75,20,wnd,(HMENU)ID_BUT_DASM_TRACK1,
	                 GetModuleHandle(NULL),NULL);
	    SendMessage(hWndBut1Sub, BM_SETCHECK, BST_UNCHECKED,0);
	    SendMessage(hWndBut2Sub, BM_SETCHECK, BST_UNCHECKED,0);
	    SendMessage(hWndBut3Sub, BM_SETCHECK, BST_CHECKED,0);

	    SendMessage(wnd, WM_OFFRESET, 0, 0);
            return 0;

	case WM_CLOSE:
            SendMessage(hWnd,WM_COMMAND,ID_MENU_VIEW_DBG,0);
	    return 0;

        case WM_OFFRESET:
	    si.cbSize=sizeof(si);
	    si.fMask =SIF_RANGE|SIF_PAGE;
	    si.nMin  =-1048576;
	    si.nMax  =1048576;
	    si.nPage =8;
	    SetScrollInfo(hWndScrlSub, SB_CTL, &si, TRUE);
            offsetsub=0;
            return 0;

	case WM_CTLCOLORSTATIC:
	case WM_CTLCOLORBTN:
	    hDC=(HDC)wParam;
	    SetTextColor(hDC,RGB(255,255,255));
	    SetBkColor(hDC,RGB(0,0,0));
    	    SetBkMode(hDC,OPAQUE);
    	    hf=CreateFont(-MulDiv(6, GetDeviceCaps(hDC, LOGPIXELSY), 72),
		    0, 0, 0, 0, FALSE, 0, 0, 0, 0,
		    0, 0, FF_DONTCARE, "MS Sans Serif");
    	    SelectObject(hDC,hf);
    	    DeleteObject(hf);
    	    return (LONG)GetStockObject(BLACK_BRUSH);

        case WM_PAINT:
            BeginPaint(wnd, &ps);
            dbgmodesub=modesub;
            ARM7status(offsetsub,dbgmodesub);
            BmpInfo.bV4Width = 468;
            BmpInfo.bV4Height = -144;
            SetDIBitsToDevice(hDCDbgSub, 6, 8, 468, 144, 0, 0, 0, 144,
                      dbgsubbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
            EndPaint(wnd, &ps);
            return 0;

        case WM_VSCROLL:
            si.cbSize=sizeof(si);
            si.fMask =SIF_ALL;
                    GetScrollInfo(hWndScrlSub, SB_CTL, &si);
                    switch(LOWORD(wParam))
                    {
                        case SB_LINEUP:   offsetsub--; break;
                        case SB_LINEDOWN: offsetsub++; break;
                        case SB_PAGEUP:   offsetsub-=8; break;
                        case SB_PAGEDOWN: offsetsub+=8; break;
                    }
                    si.fMask=SIF_POS;
                    si.nPos=offsetsub;
                    SetScrollInfo(hWndScrlSub, SB_CTL, &si, TRUE);
            InvalidateRect(wnd, NULL, FALSE);
            return 0;

	case WM_COMMAND:
	    switch(wParam)
	    {
	        case ID_BUT_DASM_ARM1:
	            modesub=DASM_ARM;
	            InvalidateRect(wnd, NULL, FALSE);
	            break;
	        case ID_BUT_DASM_THUMB1:
	            modesub=DASM_THUMB;
	            InvalidateRect(wnd, NULL, FALSE);
	            break;
	        case ID_BUT_DASM_TRACK1:
	            modesub=DASM_TRACK;
	            InvalidateRect(wnd, NULL, FALSE);
	            break;
	    }
	    return 0;

        case WM_KEYDOWN:
            emuKeySet(wParam, 1);
            return DefWindowProc(wnd, msg, wParam, lParam);

        case WM_KEYUP:
            emuKeySet(wParam, 0);
            return DefWindowProc(wnd, msg, wParam, lParam);

        default:
            return DefWindowProc(wnd, msg, wParam, lParam);
    }
}

LRESULT CALLBACK WinMemProc(HWND wnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    PAINTSTRUCT ps;
    HFONT hf; HDC hDC;
    static HWND hWndEdit, hWndBut, hWndBut1, hWndBut2, hWndBut3;
    char buf[512], *bufstop;
    static u32 addr, mode;

    switch(msg)
    {
        case WM_CREATE:
//            hWndScrl=CreateWindowEx(0, "SCROLLBAR", NULL,
//	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|SBS_VERT,320,24,12,128,
//			 hWnd,NULL,GetModuleHandle(NULL),NULL);
	    hWndEdit=CreateWindowEx(WS_EX_CLIENTEDGE, "EDIT", NULL,
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|ES_UPPERCASE|ES_LEFT,
	                 6,2,80,18,wnd,(HMENU)ID_EDIT_MEMV_ADDR,
	                 GetModuleHandle(NULL),NULL);
	    SetFocus(hWndEdit);
	    SendMessage(hWndEdit, EM_SETLIMITTEXT, 8, 0);
	    hWndBut =CreateWindowEx(0, "BUTTON", "Go",
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|BS_DEFPUSHBUTTON,
	                 90,2,24,18,wnd,(HMENU)ID_BUT_MEMV_GO,
	                 GetModuleHandle(NULL),NULL);
	    hWndBut1=CreateWindowEx(0, "BUTTON", "8bit",
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|WS_GROUP|BS_AUTORADIOBUTTON,
	                 120,2,50,20,wnd,(HMENU)ID_BUT_MEMV_8,
	                 GetModuleHandle(NULL),NULL);
	    hWndBut2=CreateWindowEx(0, "BUTTON", "16bit",
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|BS_AUTORADIOBUTTON,
	                 170,2,50,20,wnd,(HMENU)ID_BUT_MEMV_16,
	                 GetModuleHandle(NULL),NULL);
	    hWndBut3=CreateWindowEx(0, "BUTTON", "32bit",
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|BS_AUTORADIOBUTTON,
	                 220,2,50,20,wnd,(HMENU)ID_BUT_MEMV_32,
	                 GetModuleHandle(NULL),NULL);
	    SendMessage(hWndBut1, BM_SETCHECK, BST_CHECKED,0);
	    SendMessage(hWndBut2, BM_SETCHECK, BST_UNCHECKED,0);
	    SendMessage(hWndBut3, BM_SETCHECK, BST_UNCHECKED,0);
	    addr=0x04000000; mode=1;
	    return 0;

	case WM_CLOSE:
            SendMessage(hWnd,WM_COMMAND,ID_MENU_VIEW_MEM,0);
	    return 0;

	case WM_CTLCOLORSTATIC:
	case WM_CTLCOLOREDIT:
	case WM_CTLCOLORBTN:
	    hDC=(HDC)wParam;
	    SetTextColor(hDC,RGB(255,255,255));
	    SetBkColor(hDC,RGB(0,0,0));
    	    SetBkMode(hDC,OPAQUE);
    	    hf=CreateFont(-MulDiv(6, GetDeviceCaps(hDC, LOGPIXELSY), 72),
		    0, 0, 0, 0, FALSE, 0, 0, 0, 0,
		    0, 0, FF_DONTCARE, "MS Sans Serif");
    	    SelectObject(hDC,hf);
    	    DeleteObject(hf);
    	    return (LONG)GetStockObject(BLACK_BRUSH);

        case WM_PAINT:
            BeginPaint(hWnd, &ps);
	        emuMMUDump(addr,mode);
	        emuRefresh();
            EndPaint(hWnd, &ps);
            return 0;

	case WM_COMMAND:
	    switch(LOWORD(wParam))
	    {
	        case ID_BUT_MEMV_GO:
	            if(HIWORD(wParam)==BN_CLICKED)
	            {
		        GetWindowText(hWndEdit, (LPSTR)&buf, 512);
		        addr=strtol(buf,&bufstop,16);
 	   	        if((bufstop-buf)!=8)
		        {
		            MessageBox(NULL, "plz 2 enter 8 hex digits.", "Errorthx", MB_OK);
		            return 0;
		        }
		        SendMessage(wnd, WM_PAINT, 0, 0);
	            }
	            break;
	        case ID_BUT_MEMV_8:
	            mode=1; SendMessage(wnd, WM_PAINT,0,0); break;
	        case ID_BUT_MEMV_16:
	            mode=2; SendMessage(wnd, WM_PAINT,0,0); break;
	        case ID_BUT_MEMV_32:
	            mode=3; SendMessage(wnd, WM_PAINT,0,0); break;
	    }
	    return 0;

        case WM_KEYDOWN:
            emuKeySet(wParam, 1);
            return DefWindowProc(wnd, msg, wParam, lParam);

        case WM_KEYUP:
            emuKeySet(wParam, 0);
            return DefWindowProc(wnd, msg, wParam, lParam);

        default:
            return DefWindowProc(wnd, msg, wParam, lParam);
    }
}
static HWND hWndLogEdit;
void setlogbuf(char * test){
	char tmp[360*160*2];
	//SendMessage (hWndEdit, WM_GETTEXT,(WPARAM) 1000 - 1,(LPARAM) logbuf);
	//sprintf(tmp,"size of %d\r\n",strlen(logbuf));
	if( (strlen((char*)logbuf)+strlen(test)) > (360*160*2) )
		memset(logbuf+(360*160), 0, 360*160);
	strcat(test,"\r\n");
	strcpy(tmp,test);
	strcat(tmp,(char*)logbuf);
	strcpy((char*)logbuf,tmp);
	SendMessage(hWndLogEdit, WM_SETTEXT, 0, (LPARAM)(logbuf));
}
LRESULT CALLBACK WinLogProc(HWND wnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
     PAINTSTRUCT ps;
    SCROLLINFO si;
    HDC hDC;
    static int offsetsub=0, modesub=DASM_TRACK;
    static HWND hWndLogBut1 ,hWndLogBut2;
    static HFONT hf;
	
    switch(msg)
    {
        case WM_CREATE:
        /*    hWndLogEdit=CreateWindowEx(0, "SCROLLBAR", NULL,
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS,320,24,12,128,
			 wnd,NULL,GetModuleHandle(NULL),NULL);*/
	    hWndLogEdit=CreateWindowEx(WS_EX_CLIENTEDGE, "EDIT", NULL,
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|ES_UPPERCASE|ES_LEFT|ES_MULTILINE|WS_VSCROLL|SBS_RIGHTALIGN,
	                 5,25,360,235,wnd,(HMENU)ID_EDIT_LOGGER,
	                 GetModuleHandle(NULL),NULL);

	    hWndLogBut1 =CreateWindowEx(0, "BUTTON", "Clear",
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|BS_DEFPUSHBUTTON,
	                 150,5,35,20,wnd,(HMENU)ID_BUT_LOGCLEAR,
	                 GetModuleHandle(NULL),NULL);

	    hWndLogBut2 =CreateWindowEx(0, "BUTTON", "Close",
	                 WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|BS_DEFPUSHBUTTON,
	                 190,5,35,20,wnd,(HMENU)ID_BUT_LOGCLOSE,
	                 GetModuleHandle(NULL),NULL);
	    SendMessage(wnd, WM_OFFRESET, 0, 0);
            return 0;

	case WM_CLOSE:
            SendMessage(hWnd,WM_COMMAND,ID_MENU_VIEW_LOG,0);
	    return 0;

        case WM_OFFRESET:
	    si.cbSize=sizeof(si);
	    si.fMask =SIF_RANGE|SIF_PAGE;
	    si.nMin  =-1048576;
	    si.nMax  =1048576;
	    si.nPage =8;
	    SetScrollInfo(hWndLogEdit, SB_CTL, &si, TRUE);
            offsetsub=0;
            return 0;

	case WM_CTLCOLORSTATIC:
	case WM_CTLCOLORBTN:
	    hDC=(HDC)wParam;
	    SetTextColor(hDC,RGB(255,255,255));
	    SetBkColor(hDC,RGB(0,0,0));
    	    SetBkMode(hDC,OPAQUE);
    	    hf=CreateFont(-MulDiv(6, GetDeviceCaps(hDC, LOGPIXELSY), 72),
		    0, 0, 0, 0, FALSE, 0, 0, 0, 0,
		    0, 0, FF_DONTCARE, "MS Sans Serif");
    	    SelectObject(hDC,hf);
    	    DeleteObject(hf);
    	    return (LONG)GetStockObject(BLACK_BRUSH);

        case WM_PAINT:
            BeginPaint(wnd, &ps);

            EndPaint(wnd, &ps);
			
            return 0;

        case WM_VSCROLL:
            si.cbSize=sizeof(si);
            si.fMask =SIF_ALL;
                    GetScrollInfo(hWndLogEdit, SB_CTL, &si);
                    switch(LOWORD(wParam))
                    {
                        case SB_LINEUP:   offsetsub--; break;
                        case SB_LINEDOWN: offsetsub++; break;
                        case SB_PAGEUP:   offsetsub-=8; break;
                        case SB_PAGEDOWN: offsetsub+=8; break;
                    }
                    si.fMask=SIF_POS;
                    si.nPos=offsetsub;
                    SetScrollInfo(hWndLogEdit, SB_CTL, &si, TRUE);
            InvalidateRect(wnd, NULL, FALSE);
            return 0;

	case WM_COMMAND:
	    switch(LOWORD(wParam))
	    {
	        case ID_BUT_LOGCLEAR:
	            if(HIWORD(wParam)==BN_CLICKED)
	            {
				
					memset(logbuf, 0, 360*160*2);
					SendMessage(hWndLogEdit, WM_SETTEXT, 0, (LPARAM)(logbuf));
	            }
	            break;
	        case ID_BUT_LOGCLOSE:
	            if(HIWORD(wParam)==BN_CLICKED)
	            {
					SendMessage(hWnd,WM_COMMAND,ID_MENU_VIEW_LOG,0);
	            }
	            break;
	    }
	    return 0;

        case WM_KEYDOWN:
            emuKeySet(wParam, 1);
            return DefWindowProc(wnd, msg, wParam, lParam);

        case WM_KEYUP:
            emuKeySet(wParam, 0);
            return DefWindowProc(wnd, msg, wParam, lParam);

        default:
            return DefWindowProc(wnd, msg, wParam, lParam);
    }
}
/*** EOF:gba.c ***********************************************************/
