/**************************************************************************
* DSemu: Computationnal Accelerators: divide and square root (dsaccelerators.c)*
* Released under the terms of the BSD Public Licence                      *
* Crazyjul                      , 2005                                    *
**************************************************************************/

#include <math.h>
#include "arm.h"
#include "dsaccelerators.h"
#include "vtbl.h"

//Implementation of DS computationnal registers
#define DIVISION_MODE_32_32         0
#define DIVISION_MODE_64_32         1
#define DIVISION_MODE_64_64         2
#define DIVISION_MODE_PROHIBITED    3

#define SQUARE_ROOT_MODE_32         0
#define SQUARE_ROOT_MODE_64         1

#define BUSY_FLAG               (1<<15)
#define DIVISION_BY_ZERO_FLAG   (1<<14)

//Register in the array
#define REG_DIVCNT          0

#define REG_DIV_NUMER_L     4
#define REG_DIV_NUMER_H     5
#define REG_DIV_DENOM_L     6
#define REG_DIV_DENOM_H     7
#define REG_DIV_RESULT_L    8
#define REG_DIV_RESULT_H    9
#define REG_DIV_REMAIN_L    10
#define REG_DIV_REMAIN_H    11

#define REG_SQRTCNT         12

#define REG_SQRT_RESULT     13
#define REG_SQRT_PARAM_L    14
#define REG_SQRT_PARAM_H    15



//Cycle counts
#define DIV_32_32_CYCLECOUNT    18
#define DIV_64_32_CYCLECOUNT    34
#define DIV_64_64_CYCLECOUNT    34

#define SQRT_32_CYCLECOUNT      13
#define SQRT_64_CYCLECOUNT      13

#define SIGN_BIT (1<<31)

#define REG_ACCEL_BASE_ADDRESS 0x04000280
s32 REG_ACCEL[ 16 ];
u8 div_cycle_count_left;
u8 sqrt_cycle_count_left;


void initialize_accelerators_registers()
{
    div_cycle_count_left = 0;
    sqrt_cycle_count_left = 0;

    REG_ACCEL[ REG_DIVCNT ] = 0;
    REG_ACCEL[ REG_SQRTCNT ] = 0;
}

void UpdateAcceleratorsRegisters(u8 cycle_count)
{
    div_cycle_count_left -= cycle_count;
    if( div_cycle_count_left <= 0 )
    {
        div_cycle_count_left = 0;
        REG_ACCEL[ REG_DIVCNT ] &= ~BUSY_FLAG;
    }
    else
    {
        REG_ACCEL[ REG_DIVCNT ] |= BUSY_FLAG;
    }

    sqrt_cycle_count_left -= cycle_count;
    if( sqrt_cycle_count_left <=0 )
    {
        sqrt_cycle_count_left = 0;
        REG_ACCEL[ REG_SQRTCNT ] &= ~BUSY_FLAG;
    }
    else
    {
        REG_ACCEL[ REG_SQRTCNT ] |= BUSY_FLAG;
    }
}

static void compute_div_accelerator_registers()
{
    s32 numerator32, denominator32;
    s64 numerator64, denominator64;
    s32 result32, remain32;
    s64 result64, remain64;

    /*if( REG_ACCEL[ REG_DIVCNT ] & BUSY_FLAG )
    {
        // Busy flag is set, so any register modif is unpredictable
        // I decide to do nothing, will do some test on the ds to see what's happening
        return;
    }*/

    if( REG_ACCEL[REG_DIV_DENOM_L] == 0 && REG_ACCEL[REG_DIV_DENOM_H] == 0 )
    {
        REG_ACCEL[REG_DIVCNT] |= DIVISION_BY_ZERO_FLAG;
        div_cycle_count_left = 0;
        return;
    }
    else
    {
        REG_ACCEL [REG_DIVCNT] &= ~DIVISION_BY_ZERO_FLAG;
    }

    //First implementation, if it's called often, then I will optimize it

    switch( REG_ACCEL[REG_DIVCNT] & 0x3 )
    {
    case DIVISION_MODE_32_32:
        {
            numerator32 = REG_ACCEL[REG_DIV_NUMER_L];
            denominator32 = REG_ACCEL[REG_DIV_DENOM_L];

            result32 = numerator32 / denominator32;
            remain32 = numerator32 % denominator32;

            REG_ACCEL[REG_DIV_RESULT_L] = result32;
            //don't know if used, but a sign extend to 64 bit
            if( result32 & SIGN_BIT )
                REG_ACCEL[REG_DIV_RESULT_H] = ~0;
            else
                REG_ACCEL[REG_DIV_RESULT_H] = 0;

            REG_ACCEL[REG_DIV_REMAIN_L] = remain32;
            //don't know if used, but a sign extend to 64 bit
            if( remain32 & SIGN_BIT )
                REG_ACCEL[REG_DIV_REMAIN_H] = ~0;
            else
                REG_ACCEL[REG_DIV_REMAIN_H] = 0;
                
            div_cycle_count_left = DIV_32_32_CYCLECOUNT;
        }
        break;
    case DIVISION_MODE_64_32:
        {
            numerator64 = REG_ACCEL[REG_DIV_NUMER_H];
            numerator64 <<= 32;
            numerator64 |= REG_ACCEL[REG_DIV_NUMER_L];
            denominator32 = REG_ACCEL[REG_DIV_DENOM_L];

            result64 = numerator64 / denominator32;
            remain64 = numerator64 % denominator32;

            REG_ACCEL[REG_DIV_RESULT_L] = (s32) (result64 && 0XFFFFFFFF);
            REG_ACCEL[REG_DIV_RESULT_H] = (s32) (result64 >> 32);

            REG_ACCEL[REG_DIV_REMAIN_L] = (s32) (remain64 && 0XFFFFFFFF);
            REG_ACCEL[REG_DIV_REMAIN_H] = (s32) (remain64 >> 32);

            div_cycle_count_left = DIV_64_32_CYCLECOUNT;
        }
        break;
    case DIVISION_MODE_64_64:
        {
            numerator64 = REG_ACCEL[REG_DIV_NUMER_H];
            numerator64 <<= 32;
            numerator64 |= REG_ACCEL[REG_DIV_NUMER_L];

            denominator64 = REG_ACCEL[REG_DIV_DENOM_H];
            denominator64 <<= 32;
            denominator64 |= REG_ACCEL[REG_DIV_DENOM_L];

            result64 = numerator64 / denominator64;
            remain64 = numerator64 % denominator64;

            REG_ACCEL[REG_DIV_RESULT_L] = (s32) (result64 && 0XFFFFFFFF);
            REG_ACCEL[REG_DIV_RESULT_H] = (s32) (result64 >> 32);

            REG_ACCEL[REG_DIV_REMAIN_L] = (s32) (remain64 && 0XFFFFFFFF);
            REG_ACCEL[REG_DIV_REMAIN_H] = (s32) (remain64 >> 32);

            div_cycle_count_left = DIV_64_64_CYCLECOUNT;
        }
        break;
    case DIVISION_MODE_PROHIBITED:
        {
            //Log it
            logvt->append("Prohibited division mode encountered.");
        }
        break;
    }
}

static void compute_sqrt_accelerator_registers()
{
    double data;
    s64 idata;

    /*if( REG_ACCEL[ REG_DIVCNT ] & BUSY_FLAG )
    {
        // Busy flag is set, so any register modif is unpredictable
        // I decide to do nothing, will do some test on the ds to see what's happening
        return;
    }*/

    switch( REG_ACCEL[ REG_SQRTCNT ] & 0x1 )
    {
    case SQUARE_ROOT_MODE_32:
        {
            data = (s32) REG_ACCEL[REG_SQRT_PARAM_L];
            data = sqrt( data );
            REG_ACCEL[REG_SQRT_RESULT]= (s32) data;
            sqrt_cycle_count_left = SQRT_64_CYCLECOUNT;
        }
        break;
    case SQUARE_ROOT_MODE_64:
        {
            idata = REG_ACCEL[REG_SQRT_PARAM_H];
            idata <<= 32;
            idata |= REG_ACCEL[REG_SQRT_PARAM_L];
            data = (double)idata;
            data = sqrt(data);
            REG_ACCEL[REG_SQRT_RESULT]= (s32) data;
            sqrt_cycle_count_left = SQRT_64_CYCLECOUNT;
        }
        break;
    }
}

u8 do_accelerators_read_byte(int cpu, u32 addr)
{
    (void) cpu;
    (void) addr;
    return 0;
}
u16 do_accelerators_read_half(int cpu, u32 addr)
{
     int array_index;
    
    (void) cpu;
    array_index = addr - 0x04000280; 
    array_index >>= 2;
    return REG_ACCEL[array_index] && 0xFFFF; 
}
u32 do_accelerators_read_word(int cpu, u32 addr)
{
    int array_index;
    
    (void) cpu;
    array_index = addr - 0x04000280; 
    array_index >>= 2;
    return REG_ACCEL[array_index];       
}
void do_accelerators_write_byte(int cpu, u32 addr, u8 value)
{
    (void) cpu;
    (void) addr;
    (void) value;
}
void do_accelerators_write_half(int cpu, u32 addr, u16 value)
{
    int array_index;
    (void) cpu;

    array_index = addr - 0x04000280; 
    array_index >>= 2;
    REG_ACCEL[array_index] = value;

    if( array_index < REG_SQRTCNT )
    {
        compute_div_accelerator_registers();
    }
    else
    {
        compute_sqrt_accelerator_registers();
    }
}
void do_accelerators_write_word(int cpu, u32 addr, u32 value)
{
    int array_index;
    (void) cpu;

    array_index = addr - 0x04000280; 
    array_index >>= 2;
    REG_ACCEL[array_index] = value;

    if( array_index < REG_SQRTCNT )
    {
        compute_div_accelerator_registers();
    }
    else
    {
        compute_sqrt_accelerator_registers();
    }
}
