/**************************************************************************
* DSemu: ARM9 assembly core, x86 (arm9x86.c)                              *
* Released under the terms of the BSD Public Licence                      *
* Imran Nazar (tf@oopsilon.com), 2004                                     *
**************************************************************************/

#include <stdio.h>
#include "err.h"
#include "arm9.h"
#include "arm9func.h"
#include "cache.h"
#include "vtbl.h"
#include "dbgout.h"
#include "bkpt.h"
#include "dsmmumain.h"
#include "dsioreg.h"
#include "timers.h"
#include <math.h>

// TODO: ARM7 and ARM9 cores are 90% identical. Merge the common code.

#define REGION_DEBUG 1
#define ARM9REGDUMP

ARMREGS arm9reg;

extern void emuPause();
void DoCP15C1(unsigned int value);
void DoCP15C6(unsigned int region, unsigned int value);
void DoCP15C5(unsigned int region, unsigned int opcode2, unsigned int value);
void DoCP15C9(unsigned int tcm, unsigned int value);

u32 *dbuf9;
FILE *ARM9regdump=NULL;

int cpu9Halted; u16 intr9Bitmask;
int ARM9execforThumb(int), ARM9execforARM(int);
int (*ARM9execforPtr)(int) = ARM9execforARM;
int ARM9waitstate;

int ARM9instrCnt;

BREAKPOINT ARM9bkpts[16];

int ARM9init(char *file, u32 *dbgbuf)
{
    char str[8192];
    sprintf(str,"ARM9: ROM file: %s",file);
    logvt->append(str);

    MMUMaininit(file);
    dbuf9=dbgbuf;

//    if(ARM9reset()) RETFAIL("FAIL: ARM9: Reset.");
    RETPASS("ARM9: Initialised.");
}

void ARM9fini()
{
    MMUMainfini();
    #ifdef ARM9REGDUMP
    fclose(ARM9regdump);
    #endif
    logvt->append("ARM9: Shutdown.");
}

int ARM9reset()
{
    int a;

    #ifdef ARM9REGDUMP
    if(ARM9regdump){fclose(ARM9regdump);ARM9regdump=NULL;}
    ARM9regdump=fopen("C:\\regdump.arm9.txt","w");
    #endif

    arm9reg.r0=0;  arm9reg.r1=0;  arm9reg.r2=0;  arm9reg.r3=0;
    arm9reg.r4=0;  arm9reg.r5=0;  arm9reg.r6=0;  arm9reg.r7=0;
    arm9reg.r8=0;  arm9reg.r9=0;  arm9reg.r10=0; arm9reg.r11=0;
    arm9reg.r12=0; arm9reg.r13=0x03007F00; arm9reg.r14=0; arm9reg.r15=0x08000000;

    arm9reg.r[0]=arm9reg.r0;   arm9reg.r[1]=arm9reg.r1;   arm9reg.r[2]=arm9reg.r2;   arm9reg.r[3]=arm9reg.r3;
    arm9reg.r[4]=arm9reg.r4;   arm9reg.r[5]=arm9reg.r5;   arm9reg.r[6]=arm9reg.r6;   arm9reg.r[7]=arm9reg.r7;
    arm9reg.r[8]=arm9reg.r8;   arm9reg.r[9]=arm9reg.r9;   arm9reg.r[10]=arm9reg.r10; arm9reg.r[11]=arm9reg.r11;
    arm9reg.r[12]=arm9reg.r12; arm9reg.r[13]=arm9reg.r13; arm9reg.r[14]=arm9reg.r14; arm9reg.r[15]=arm9reg.r15;

    arm9reg.cpsr=ARMS_M_SYS|ARMS_F;
    arm9reg.clock=0;

    for(a=0;a<7;a++) arm9reg.spsr[a]=0;
    arm9reg.r8fiq=0;  arm9reg.r9fiq=0;  arm9reg.r10fiq=0; arm9reg.r11fiq=0;
    arm9reg.r12fiq=0; arm9reg.r13fiq=0; arm9reg.r14fiq=0;
    arm9reg.r13irq=0x03007FA0; arm9reg.r14irq=0;
    arm9reg.r13abt=0; arm9reg.r14abt=0;
    arm9reg.r13svc=0x03007FE0; arm9reg.r14svc=0;
    arm9reg.r13und=0; arm9reg.r14und=0;

    cpu9Halted=0; intr9Bitmask=0xFFFF;
    ARM9execforPtr=ARM9execforARM;
    ARM9waitstate=0;
    ARM9instrCnt=0;

	// TODO: Find out what the defaults really are
	for(a=0;a<8;++a) 
		arm9reg.cp15_regions[a] = 0;
	for(a=0;a<2;++a) 
		arm9reg.cp15_tcm_base[a] = 0;

	// Defaults to vector address being 0xffff0000
	arm9reg.cp15_control_register = 0x2000;
	
    arm9reg.curmode=ARMMD_SYS;
    for(a=0;a<8;a++) arm9reg.flags[a]=0;
    logvt->append("ARM9: Registers zeroed.");
    if(MMUMainreset()) RETFAIL("FAIL: MMUMain: Reset.");
    RETPASS("ARM9: Reset.");
}

__int64 ARM9getClock() { return arm9reg.clock; }
void ARM9addClock(int clk) { arm9reg.clock+=clk; ARM9waitstate+=clk; }

void ARM9DUMP(u32 op)
{
        u32 r15,a;
	ARM9updateCPSR();
        fprintf(ARM9regdump,"%08X - PC=%08X BANK=3 ",ARM9instrCnt,arm9reg.r[15]);
        for(a=0;a<=14;a++) fprintf(ARM9regdump,"R%02d=%08X ",a,arm9reg.r[a]);
        r15=arm9reg.r[15]+((arm9reg.cpsr&ARMS_T)?4:8);
        fprintf(ARM9regdump,"R15=%08X\n",r15);
}


// TODO: Reorganise code so timers can be taken out of the
//       ARM9 file.
#include "int.h"

static void UpdateIRQ() 
{
	extern u16 int9Fired;
	if(int9Fired && !(arm9reg.cpsr & ARMS_I))
	{
		if(REG(ARM9_REG,IME) && (REG(ARM9_REG,IE)&int9Fired))
		{
			REG(ARM9_REG,IF)|=int9Fired;
			ARM9InterruptFlagChanged();
			ARM9irq();
		}
		int9Fired = 0;
	}
}

// TODO: For some reason UpdateIRQ2() needs to be used to
// increment r15 by 4 when called from within the ARM do loop.
// DSLinux won't run without doing this. I suspect I'm working
// around a bug in the ARM9 core that isn't increment r15 at the
// right place. Need to investigate why this is needed.
static void UpdateIRQ2() 
{
	extern u16 int9Fired;
	if(int9Fired && !(arm9reg.cpsr & ARMS_I))
	{
		if(REG(ARM9_REG,IME) && (REG(ARM9_REG,IE)&int9Fired))
		{
			REG(ARM9_REG,IF)|=int9Fired;
			ARM9InterruptFlagChanged();			
			arm9reg.r[0xf]+=4;
			ARM9irq();
		}
		int9Fired = 0;
	}
}

#if 0
static void CheckCPSR()
{
    if(arm9reg.flags[0] != ((arm9reg.cpsr>>31)&1))
	{
		logvt->append("CPSR Flag 0 invalid");
	}

    if(arm9reg.flags[1] != ((arm9reg.cpsr>>30)&1))
	{
		logvt->append("CPSR Flag 1 invalid");
	}

	if(arm9reg.flags[2]!=((arm9reg.cpsr>>29)&1))
	{
		logvt->append("CPSR Flag 2 invalid");
	}

	if(arm9reg.flags[3] != ((arm9reg.cpsr>>28)&1))
	{
		logvt->append("CPSR Flag 3 invalid");
	}

    if(arm9reg.flags[4]!=((arm9reg.cpsr>>5)&1))
	{
		logvt->append("CPSR Flag 4 invalid");
	}

    if(arm9reg.flags[5]!=((arm9reg.cpsr>>27)&1))
	{
		logvt->append("CPSR Flag 5 invalid");
	}
	if(arm9reg.flags[ARMFLAG_T] && ARM9execforPtr != ARM9execforThumb)
	{
		logvt->append("arm9execforptr invalid");
	}
	if(ARM9modetocpsr[arm9reg.curmode] != (arm9reg.cpsr&0x1f))
	{
		logvt->append("current mode invalid");
	}

}
#endif

int ARM9execforThumb(int count)
{
    u32 clk=0/*,opclk*/; /*u32 r15;*/ RAMWORD op;
	if(!is_any_timer_active(ARM9_REG) && cpu9Halted)
	{
		UpdateIRQ();
        UpdateAcceleratorsRegisters( count );
		return count;
	}
    do
    {
		if(cpu9Halted && is_any_timer_active(ARM9_REG)) {
			int i = 0;
			int min_timer = 1000000;
			int opclk = 0;
			for(i=0; i < 4; ++i) {
				int ticks = timer_data[ARM9_REG][i].rollover_ticks - timer_data[ARM9_REG][i].current_ticks;
				if(!timer_data[ARM9_REG][i].cascade && timer_data[ARM9_REG][i].active && ticks < min_timer)
					min_timer = ticks;
			}
			opclk = min(count-(int)clk, min_timer);
			clk += opclk;
			UpdateTimers(ARM9_REG, opclk);
			UpdateIRQ();
            UpdateAcceleratorsRegisters( opclk );
		}
		else if(!cpu9Halted)
		{
			int c = 0;
			UpdateIRQ();
			op=MMUMainrdS(0,arm9reg.r[15]);
#if 0
			{
				char str[255];
				sprintf(str, "addr: %8X - THUMB", arm9reg.r[15]);
				logvt->append(str);
			}
#endif
			ARM9waitstate=0;
			arm9reg.curop=(arm9reg.r[15]&2)?op.h[1]:op.h[0];
			arm9reg.r[15]+=2;
//			CheckCPSR();
			c = thumb9ops[arm9reg.curop>>8]()+ARM9waitstate;
//			CheckCPSR();
			clk+= c;
			UpdateTimers(ARM9_REG, c);
            UpdateAcceleratorsRegisters( c );
		}
		else 
			break;
    } while(((int)clk<count) && arm9reg.flags[ARMFLAG_T]);
    return (int)clk;
}


int ARM9execforARM(int count)
{
			static int do_log = 0;
    int clk=0,opclk; /*u32 r15; */RAMWORD op;
	if(!is_any_timer_active(ARM9_REG) && cpu9Halted) {
		UpdateIRQ();
        UpdateAcceleratorsRegisters( count );
		return count;
	}
    do
    {
		if(cpu9Halted && is_any_timer_active(ARM9_REG)) {
			int i = 0;
			int min_timer = 1000000;
			int opclk = 0;
			for(i=0; i < 4; ++i) {
				int ticks = timer_data[ARM9_REG][i].rollover_ticks - timer_data[ARM9_REG][i].current_ticks;
				if(!timer_data[ARM9_REG][i].cascade && timer_data[ARM9_REG][i].active && ticks < min_timer)
					min_timer = ticks;
			}
			opclk = min(count-clk, min_timer);
			clk += opclk;
			UpdateTimers(ARM9_REG, opclk);
			UpdateIRQ();
            UpdateAcceleratorsRegisters( opclk );
		}
		else if(!cpu9Halted)
		{
			u32 r15 = 0;
			UpdateIRQ2();
			r15 = arm9reg.r[0xf];
#if 0
			{
				char str[255];
				sprintf(str, "addr: %8X - ARM", arm9reg.r[15]);
				logvt->append(str);
			}
#endif
#if 0
			// DSLinux testing
			if((r15 <0x02005000 || r15 >= 0x20d21ff) && (r15 != 0x18 && r15 <=0x0200 && r15 >=0x023c))
			{
				int x = 5;
			}
			if(r15 == 0x2021cf4) {
				// return from interrupt
				int x=5;
			}

			if(r15 == 0x2021c40) {
				int x = 5;
				// Data abort
			}
			if(r15 == 0x2022008) {
				int x = 5;
				//no work
			}
			if(r15 == 0x2081bdc) {
				char str[256];
				char str2[256];
				char ch = MMUMainrdB(0, arm9reg.r[2]);
				int i = 0;
				str2[0] = ch;
				for(i=1;i < sizeof(str2)-1 && ch != 0;++i) {
					ch=MMUMainrdB(0, arm9reg.r[2]+i);
					str2[i] = ch;
				}
				str2[i] = 0;
				sprintf(str, "vsnprintf: buffer=%8X, size: %8X, str: %s",
					arm9reg.r[0], arm9reg.r[1], str2);
				logvt->append(str);				
			}
			if(r15 == 0x2026e3c) {
				char str[256];
				sprintf(str, "Fault at %8X, %8X", arm9reg.r[0], arm9reg.r[1]);
				logvt->append(str);
			}
			if(r15 == 0x02010014) {
				char str[256];
				logvt->append("Called uclinux_mtd_init");
			}
			if(r15 == 0x0201011c) {
				char str[256];
				logvt->append("before add_mtd_partitions");
			}
			if(r15 == 0x0205d5b8) {
				char str[256];
				logvt->append("sys_mount");
			}


			// Debug
			if(do_log)
			{
				char str[250];
				sprintf(str, "%8X", r15);
				logvt->append(str);
			}
#endif
			op=MMUMainrdS(0,r15);
			ARM9waitstate=0;
			arm9reg.r[0xf] = r15 + 4;
			if(op.cond && op.cond())
			{			
	            arm9reg.curop=op.data;
//				CheckCPSR();
				opclk=op.op();
//				CheckCPSR();
				if(opclk) 
					clk+=opclk+ARM9waitstate; 
				else 
					return clk;
				UpdateTimers(ARM9_REG, opclk+ARM9waitstate);
                UpdateAcceleratorsRegisters( opclk+ARM9waitstate );
			}
		}
		else
			break;
    } while((clk<count) && (!arm9reg.flags[ARMFLAG_T]));
    return clk;
}

int ARM9execfor(int count)
{
	return ARM9execforPtr(count);
}

void ARM9updateCPSR()
{
    if(arm9reg.flags[0]) 
		arm9reg.cpsr|=ARMS_N;
    else 
		arm9reg.cpsr&=(0xFFFFFFFF-ARMS_N);
    if(arm9reg.flags[1]) 
		arm9reg.cpsr|=ARMS_Z;
    else 
		arm9reg.cpsr&=(0xFFFFFFFF-ARMS_Z);
    if(arm9reg.flags[2]) 
		arm9reg.cpsr|=ARMS_C;
    else 
		arm9reg.cpsr&=(0xFFFFFFFF-ARMS_C);
    if(arm9reg.flags[3]) 
		arm9reg.cpsr|=ARMS_V;
    else 
		arm9reg.cpsr&=(0xFFFFFFFF-ARMS_V);
    if(arm9reg.flags[4]) 
		arm9reg.cpsr|=ARMS_T;
    else 
		arm9reg.cpsr&=(0xFFFFFFFF-ARMS_T);
    if(arm9reg.flags[5]) 
		arm9reg.cpsr|=ARMS_Q;
    else 
		arm9reg.cpsr&=(0xFFFFFFFF-ARMS_Q);
    ARM9execforPtr=(arm9reg.flags[ARMFLAG_T]?ARM9execforThumb:ARM9execforARM);
}

void ARM9splitCPSR()
{
    arm9reg.flags[0]=(arm9reg.cpsr>>31)&1;
    arm9reg.flags[1]=(arm9reg.cpsr>>30)&1;
    arm9reg.flags[2]=(arm9reg.cpsr>>29)&1;
    arm9reg.flags[3]=(arm9reg.cpsr>>28)&1;
    arm9reg.flags[4]=(arm9reg.cpsr>>5)&1;
    arm9reg.flags[5]=(arm9reg.cpsr>>27)&1;
    ARM9execforPtr=(arm9reg.flags[ARMFLAG_T]?ARM9execforThumb:ARM9execforARM);
}

void ARM9modesw(int mdin, int mdout)
{
    ARM9updateCPSR();
    arm9reg.r0=arm9reg.r[0]; arm9reg.r1=arm9reg.r[1];
    arm9reg.r2=arm9reg.r[2]; arm9reg.r3=arm9reg.r[3];
    arm9reg.r4=arm9reg.r[4]; arm9reg.r5=arm9reg.r[5];
    arm9reg.r6=arm9reg.r[6]; arm9reg.r7=arm9reg.r[7];
    arm9reg.r15=arm9reg.r[15];

    switch(mdin)
    {
    	case ARMMD_USR: case ARMMD_SYS:
    	    arm9reg.r8=arm9reg.r[8];   arm9reg.r9=arm9reg.r[9];
    	    arm9reg.r10=arm9reg.r[10]; arm9reg.r11=arm9reg.r[11];
    	    arm9reg.r12=arm9reg.r[12]; arm9reg.r13=arm9reg.r[13];
    	    arm9reg.r14=arm9reg.r[14];
    	    break;
    	case ARMMD_IRQ:
    	    arm9reg.r8=arm9reg.r[8];   arm9reg.r9=arm9reg.r[9];
    	    arm9reg.r10=arm9reg.r[10]; arm9reg.r11=arm9reg.r[11];
    	    arm9reg.r12=arm9reg.r[12];
            arm9reg.r13irq=arm9reg.r[13]; arm9reg.r14irq=arm9reg.r[14];
    	    break;
        case ARMMD_SVC:
    	    arm9reg.r8=arm9reg.r[8];   arm9reg.r9=arm9reg.r[9];
    	    arm9reg.r10=arm9reg.r[10]; arm9reg.r11=arm9reg.r[11];
    	    arm9reg.r12=arm9reg.r[12];
            arm9reg.r13svc=arm9reg.r[13]; arm9reg.r14svc=arm9reg.r[14];
    	    break;
        case ARMMD_ABT:
    	    arm9reg.r8=arm9reg.r[8];   arm9reg.r9=arm9reg.r[9];
    	    arm9reg.r10=arm9reg.r[10]; arm9reg.r11=arm9reg.r[11];
    	    arm9reg.r12=arm9reg.r[12];
            arm9reg.r13abt=arm9reg.r[13]; arm9reg.r14abt=arm9reg.r[14];
    	    break;
        case ARMMD_UND:
    	    arm9reg.r8=arm9reg.r[8];   arm9reg.r9=arm9reg.r[9];
    	    arm9reg.r10=arm9reg.r[10]; arm9reg.r11=arm9reg.r[11];
    	    arm9reg.r12=arm9reg.r[12];
            arm9reg.r13und=arm9reg.r[13]; arm9reg.r14und=arm9reg.r[14];
    	    break;
    	case ARMMD_FIQ:
            arm9reg.r8fiq=arm9reg.r[8];   arm9reg.r9fiq=arm9reg.r[9];
    	    arm9reg.r10fiq=arm9reg.r[10]; arm9reg.r11fiq=arm9reg.r[11];
    	    arm9reg.r12fiq=arm9reg.r[12]; arm9reg.r13fiq=arm9reg.r[13];
    	    arm9reg.r14fiq=arm9reg.r[14];
    	    break;
    }
    arm9reg.cpsr&=(0xFFFFFFFF-ARMS_M);
    switch(mdout)
    {
    	case ARMMD_USR:
    	    arm9reg.r[8]=arm9reg.r8;   arm9reg.r[9]=arm9reg.r9;
    	    arm9reg.r[10]=arm9reg.r10; arm9reg.r[11]=arm9reg.r11;
    	    arm9reg.r[12]=arm9reg.r12; arm9reg.r[13]=arm9reg.r13;
    	    arm9reg.r[14]=arm9reg.r14;
    	    break;
        case ARMMD_SYS:
    	    arm9reg.r[8]=arm9reg.r8;   arm9reg.r[9]=arm9reg.r9;
    	    arm9reg.r[10]=arm9reg.r10; arm9reg.r[11]=arm9reg.r11;
    	    arm9reg.r[12]=arm9reg.r12; arm9reg.r[13]=arm9reg.r13;
    	    arm9reg.r[14]=arm9reg.r14;
    	    break;
    	case ARMMD_IRQ:
    	    arm9reg.r[8]=arm9reg.r8;   arm9reg.r[9]=arm9reg.r9;
    	    arm9reg.r[10]=arm9reg.r10; arm9reg.r[11]=arm9reg.r11;
    	    arm9reg.r[12]=arm9reg.r12;
            arm9reg.r[13]=arm9reg.r13irq; arm9reg.r[14]=arm9reg.r14irq;
    	    break;
        case ARMMD_SVC:
    	    arm9reg.r[8]=arm9reg.r8;   arm9reg.r[9]=arm9reg.r9;
    	    arm9reg.r[10]=arm9reg.r10; arm9reg.r[11]=arm9reg.r11;
    	    arm9reg.r[12]=arm9reg.r12;
            arm9reg.r[13]=arm9reg.r13svc; arm9reg.r[14]=arm9reg.r14svc;
    	    break;
        case ARMMD_ABT:
    	    arm9reg.r[8]=arm9reg.r8;   arm9reg.r[9]=arm9reg.r9;
    	    arm9reg.r[10]=arm9reg.r10; arm9reg.r[11]=arm9reg.r11;
    	    arm9reg.r[12]=arm9reg.r12;
            arm9reg.r[13]=arm9reg.r13abt; arm9reg.r[14]=arm9reg.r14abt;
    	    break;
        case ARMMD_UND:
    	    arm9reg.r[8]=arm9reg.r8;   arm9reg.r[9]=arm9reg.r9;
    	    arm9reg.r[10]=arm9reg.r10; arm9reg.r[11]=arm9reg.r11;
    	    arm9reg.r[12]=arm9reg.r12;
            arm9reg.r[13]=arm9reg.r13und; arm9reg.r14und=arm9reg.r[14];
    	    break;
    	case ARMMD_FIQ:
            arm9reg.r[8]=arm9reg.r8fiq;   arm9reg.r[9]=arm9reg.r9fiq;
    	    arm9reg.r[10]=arm9reg.r10fiq; arm9reg.r[11]=arm9reg.r11fiq;
    	    arm9reg.r[12]=arm9reg.r12fiq; arm9reg.r[13]=arm9reg.r13fiq;
    	    arm9reg.r[14]=arm9reg.r14fiq;
    	    break;
    }
    arm9reg.cpsr|=ARM9modetocpsr[mdout];
    arm9reg.curmode=mdout;
	ARM9splitCPSR();
}

void ARM9status(int offset, int mode)
{
    char str[512]; int a; u16 oph;
    u32 r15=arm9reg.r[15],r15orig=r15; static u32 r15old9;
    static u32 dbg9oldr[16],r9changed[16],dbg9oldf[6],f9changed[6];
    RAMWORD op; u32 col; u16 *outbuf=(u16*)dbuf9;

    ARM9updateCPSR();
    dbgOutClear(outbuf,468,144);
//    logvt->append("ARM9status did get called.");

    if((mode==3&&arm9reg.cpsr&ARMS_T)||(mode==2))
        r15-=(16-offset*2);
    else r15-=(32-offset*4);

    for(a=0;a<16;a++)
    {
        if(r15==r15orig) col=0x7E10; else col=0x7FFF;
        if((r15&0x0F000000)==(r15orig&0x0F000000)) op=MMUMainrdS(0,r15);
        switch(mode)
        {
            case 1:
                if((r15&0x0F000000)==(r15orig&0x0F000000))
                    sprintf(str,"%08X: %08X | %s",r15,op.data,ARM9DASM(op.data));
                else sprintf(str,"%08X",r15);
                r15+=4; break;
            case 2:
                if((r15&0x0F000000)==(r15orig&0x0F000000))
                {
                    if(r15&2) oph=op.data>>16; else oph=op.data&65535;
                    sprintf(str,"%08X: %04X | %s",r15,oph,Thumb9DASM(oph));
                }
                else sprintf(str,"%08X",r15);
                r15+=2; break;
            case 3:
                switch(arm9reg.cpsr&ARMS_T)
                {
                    case 0:
                        if((r15&0x0F000000)==(r15orig&0x0F000000))
                            sprintf(str,"%08X: %08X | %s",r15,op.data,ARM9DASM(op.data));
                        else sprintf(str,"%08X",r15);
                        r15+=4; break;
                    case ARMS_T:
                        if((r15&0x0F000000)==(r15orig&0x0F000000))
                        {
                            if(r15&2) oph=op.data>>16; else oph=op.data&65535;
                            sprintf(str,"%08X: %04X | %s",r15,oph,Thumb9DASM(oph));
                        }
                        else sprintf(str,"%08X",r15);
                        r15+=2; break;
                }
                break;
        }
        dbgOut(outbuf,468,144,str,0,16+a*8,col);
    }
    for(a=0;a<=15;a++)
    {
        sprintf(str,"r%02d: %08X",a,arm9reg.r[a]);
        if(r15old9!=r15orig)
        {
            if(arm9reg.r[a]!=dbg9oldr[a]) { dbg9oldr[a]=arm9reg.r[a]; r9changed[a]=1; }
            else r9changed[a]=0;
        }
        if(r9changed[a]) dbgOut(outbuf,468,144,str,56*6,16+a*8,0x001F);
	else dbgOut(outbuf,468,144,str,56*6,16+a*8,0x7FFF);
    }
    sprintf(str,"%s",ARM9modes[arm9reg.curmode]);
    dbgOut(outbuf, 468,144, str, 72*6,72,0x7FFF);

    a=(arm9reg.cpsr&ARMS_N)>>31; sprintf(str,"N: %d",a);
    if(r15old9!=r15orig)
        if(a!=(int)dbg9oldf[0]) { dbg9oldf[0]=a; f9changed[0]=1; } else f9changed[0]=0;
    if(f9changed[0]) dbgOut(outbuf,468,144,str,72*6,16,0x001F);
    else dbgOut(outbuf,468,144,str,72*6,16,0x7FFF);

    a=(arm9reg.cpsr&ARMS_Z)>>30; sprintf(str,"Z: %d",a);
    if(r15old9!=r15orig)
        if(a!=(int)dbg9oldf[1]) { dbg9oldf[1]=a; f9changed[1]=1; } else f9changed[1]=0;
    if(f9changed[1]) dbgOut(outbuf,468,144,str,72*6,24,0x001F);
    else dbgOut(outbuf,468,144,str,72*6,24,0x7FFF);

    a=(arm9reg.cpsr&ARMS_C)>>29; sprintf(str,"C: %d",a);
    if(r15old9!=r15orig)
        if(a!=(int)dbg9oldf[2]) { dbg9oldf[2]=a; f9changed[2]=1; } else f9changed[2]=0;
    if(f9changed[2]) dbgOut(outbuf,468,144,str,72*6,32,0x001F);
    else dbgOut(outbuf,468,144,str,72*6,32,0x7FFF);

    a=(arm9reg.cpsr&ARMS_V)>>28; sprintf(str,"V: %d",a);
    if(r15old9!=r15orig)
        if(a!=(int)dbg9oldf[3]) { dbg9oldf[3]=a; f9changed[3]=1; } else f9changed[3]=0;
    if(f9changed[3]) dbgOut(outbuf,468,144,str,72*6,40,0x001F);
    else dbgOut(outbuf,468,144,str,72*6,40,0x7FFF);

    a=(arm9reg.cpsr&ARMS_Q)>>27;  sprintf(str,"Q: %d",a);
    if(r15old9!=r15orig)
        if(a!=(int)dbg9oldf[4]) { dbg9oldf[4]=a; f9changed[4]=1; } else f9changed[4]=0;
    if(f9changed[4]) dbgOut(outbuf,468,144,str,72*6,48,0x001F);
    else dbgOut(outbuf,468,144,str,72*6,48,0x7FFF);

    a=(arm9reg.cpsr&ARMS_T)>>5;  sprintf(str,"T: %d",a);
    if(r15old9!=r15orig)
        if(a!=(int)dbg9oldf[5]) { dbg9oldf[5]=a; f9changed[5]=1; } else f9changed[5]=0;
    if(f9changed[5]) dbgOut(outbuf,468,144,str,72*6,56,0x001F);
    else dbgOut(outbuf,468,144,str,72*6,56,0x7FFF);

    if(r15old9!=r15orig) r15old9=r15orig;
//    logvt->append("ARM9status finished.");
}

//---Undefined/unimplemented opcode messaging------------------------------

OPCODE ARM9opUND()
{
    char str[80];
    sprintf(str,"Undefined opcode encountered: %08X.",arm9reg.curop);
    logvt->append(str);
    return 1;
}

OPCODE ARM9opUNP()
{
    char str[80];
    sprintf(str,"Unpredictable opcode encountered: %08X.",arm9reg.curop);
    logvt->append(str);
    return 1;
}

OPCODE ARM9opUNI()
{
    char str[80];
    sprintf(str,"Unimplemented opcode encountered at %08X: %08X.",arm9reg.r[0xf],arm9reg.curop);
    logvt->append(str);
    return 1;
}

OPCODE ARM9opUNL()
{
    RAMWORD op;
    u32 idx=((arm9reg.curop&0x0FF00000)>>16)|((arm9reg.curop&0x000000F0)>>4);
    u16 cond=(arm9reg.curop&0xF0000000)>>28;
    op.data=arm9reg.curop;
    op.op=arm9ops[idx];
    op.tidxlo=(arm9reg.curop>>8)&255;
    op.tidxhi=(arm9reg.curop>>24);
    op.cond=ARM9cond[cond];
//    op.cyc=arm9ops[idx].cyc;
    op.flags=0;

	if(arm9reg.flags[ARMFLAG_T] || (arm9reg.r[0xf] % 2 == 1))
	{
		// Somehow we ended up executing thumb instructions in ARM mode.
		// This happens in the current libnds as the interrupt handler is
		// in thumb but the interrupt routine executes the handler in ARM.
		// I check if the instruction is a thumb address and execute the 
		// correct ARM codes.

		int clock = 0;
		    arm9reg.curop=op.data&65535;
		    clock = thumb9ops[arm9reg.curop>>8]();
		    arm9reg.curop=op.data>>16;
		    clock+=thumb9ops[arm9reg.curop>>8]();
			return clock;
        } else {
            MMUMainwrS(0,arm9reg.r[15]-4,op);
            if(op.cond())
            {
                arm9reg.curop=op.data;
                return op.op();
//                arm9reg.clock+=op.cyc;
            }
            return 1;
        }
}

RAMWORD ARM9opDecode(u32 input)
{
    RAMWORD op;
    u32 idx=((input&0x0FF00000)>>16)|((input&0x000000F0)>>4);
    u16 cond=(input&0xF0000000)>>28;
    op.data=input;
    op.op=arm9ops[idx];
    op.tidxlo=(input>>8)&255;
    op.tidxhi=(input>>24);
    op.cond=ARM9cond[cond];
    op.flags=0;

    return op;
}

//---Conditional execution implementation----------------------------------

int ARM9condEQ(){return  arm9reg.flags[ARMFLAG_Z];}
int ARM9condNE(){return !arm9reg.flags[ARMFLAG_Z];}
int ARM9condCS(){return  arm9reg.flags[ARMFLAG_C];}
int ARM9condCC(){return !arm9reg.flags[ARMFLAG_C];}
int ARM9condMI(){return  arm9reg.flags[ARMFLAG_N];}
int ARM9condPL(){return !arm9reg.flags[ARMFLAG_N];}
int ARM9condVS(){return  arm9reg.flags[ARMFLAG_V];}
int ARM9condVC(){return !arm9reg.flags[ARMFLAG_V];}

int ARM9condHI(){return (!arm9reg.flags[ARMFLAG_Z])&arm9reg.flags[ARMFLAG_C];}
int ARM9condLS(){return arm9reg.flags[ARMFLAG_Z]|(!arm9reg.flags[ARMFLAG_C]);}
int ARM9condGE(){return !(arm9reg.flags[ARMFLAG_N]^arm9reg.flags[ARMFLAG_V]);}
int ARM9condLT(){return  (arm9reg.flags[ARMFLAG_N]^arm9reg.flags[ARMFLAG_V]);}
int ARM9condGT(){return (!arm9reg.flags[ARMFLAG_Z])&(!(arm9reg.flags[ARMFLAG_N]^arm9reg.flags[ARMFLAG_V]));}
int ARM9condLE(){return arm9reg.flags[ARMFLAG_Z]|(arm9reg.flags[ARMFLAG_N]^arm9reg.flags[ARMFLAG_V]);}
int ARM9condAL(){return 1;}
int ARM9condNV(){return 0;}

//---Branching-------------------------------------------------------------

OPCODE ARM9opB()
{
    __asm {
        mov eax,[arm9reg+ARMREG_OP*4]     /* Get opcode */
        shl eax,8                         /* Extract branch address */
        sar eax,6                         /* addr*4, with sign extend */
        add eax,4                         /* Add 4 to new pc */
        add [arm9reg+ARMREG_R15*4],eax    /* and write to r15 */
        mov eax,3
    }
}

OPCODE ARM9opBL()
{
    __asm {
	mov ecx,[arm9reg+ARMREG_R15*4]    /* Get r15 */
        mov [arm9reg+ARMREG_R14*4],ecx    /* r14=r15 */
        mov eax,[arm9reg+ARMREG_OP*4]     /* Get opcode */
        shl eax,8                         /* Extract branch address */
        sar eax,6                         /* addr*4, with sign extend */
        add eax,4                         /* Add 4 to new pc */
        add [arm9reg+ARMREG_R15*4],eax    /* and write to r15 */
        mov eax,3
    }
}

OPCODE ARM9opBX()
{
    __asm {
        mov eax,[arm9reg+ARMREG_OP*4]      /* Read opcode */
        and eax,15                         /* Mask Rm */
        mov eax,[arm9reg+eax*4]            /* Read Rm's value */
        mov ebx,eax                        /* Make a copy */
	and eax,0FFFFFFFEh                 /* Mask out T bit */
	mov [arm9reg+ARMREG_R15*4],eax             /* and write r15 */
	and ebx,1                          /* Mask in T bit */
	cmp ebx,1                          /* Is it 1? */
	sbb ecx,ecx                        /* (==0)?-1:0 */
	not ecx                            /* (==0)?0:-1 */
//	and ecx,ARMS_T                     /* Mask cpsr T bit */
	//and [arm9reg+ARMREG_CPSR*4],0FFFFFFFFh-ARMS_T
	//or [arm9reg+ARMREG_CPSR*4],ecx     /* Add T */
        and ecx,1
        mov [arm9reg+ARMREG_FT*4],ecx
        call ARM9updateCPSR
        mov eax,3
    }
}

OPCODE ARM9opBLX()
{
  // Implemented in ARM5
  return 1;
}

OPCODE ARM9opBLXr()
{
    __asm {
        mov eax,[arm9reg+ARMREG_R15*4]
        mov [arm9reg+ARMREG_R14*4],eax
        mov eax,[arm9reg+ARMREG_OP*4]      /* Read opcode */
        and eax,15                         /* Mask Rm */
        mov eax,[arm9reg+eax*4]            /* Read Rm's value */
        mov ebx,eax                        /* Make a copy */
	and eax,0FFFFFFFEh                 /* Mask out T bit */
	mov [arm9reg+ARMREG_R15*4],eax             /* and write r15 */
	and ebx,1                          /* Mask in T bit */
	cmp ebx,1                          /* Is it 1? */
	sbb ecx,ecx                        /* (==0)?-1:0 */
	not ecx                            /* (==0)?0:-1 */
//	and ecx,ARMS_T                     /* Mask cpsr T bit */
//	and [arm9reg+ARMREG_CPSR*4],0FFFFFFFFh-ARMS_T
//	or [arm9reg+ARMREG_CPSR*4],ecx     /* Add T */
        and ecx,1
        mov [arm9reg+ARMREG_FT*4],ecx
        call ARM9updateCPSR
        mov eax,3
    }
}

//---Data Processing addressing modes, opcodes-----------------------------

#define ARM9addrDPops() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask to range */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Read value of Rn */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt2                          /* If not, skip next */ \
        __asm add eax,4                       /* Hack r15 */ \
        __asm pt2:                            /* OPERAND2 set */ \
        \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm and ecx,15                      /* Mask to Rm */ \
        __asm mov ebx,[arm9reg+ecx*4]         /* Read value of Rm */ \
        __asm cmp ecx,15                      /* Is Rm 15? */ \
        __asm jl pt1                          /* If not, skip next */ \
        __asm add ebx,4                       /* Hack r15 value */ \
        __asm pt1:                            /* OPERAND1 set */ \
    }

#define ARM9addrDPreg() \
    ARM9addrDPops(); \
    __asm { \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        __asm mov ecx,[arm9reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm setc [arm9reg+ARMREG_TC*4] \
    }

#define ARM9addrDPimm() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask to range */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Read value of Rn */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt2                          /* If not, skip next */ \
        __asm add eax,4                       /* Hack r15 */ \
        __asm pt2:                            /* OPERAND2 set */ \
        \
        __asm mov ecx,edx                     /* Make op copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        \
        __asm and ebx,255                     /* Get the low byte */ \
        __asm shr ecx,7                       /* Shift in rotamnt */ \
        __asm and ecx,30                      /* and mask to 31 */ \
        __asm ror ebx,cl                      /* Rotate constant */ \
        __asm or ecx,ecx                      /* Check rotamnt */ \
        __asm jz noshft                       /* Skip if no rot */ \
        __asm mov ecx,eax                     /* Copy operand1 */ \
        __asm shl ecx,1                       /* Carry = high bit */ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
        __asm mov ecx,[arm9reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
	\
        __asm shftdone:                       /* all done! */ \
        __asm setc [arm9reg+ARMREG_TC*4] \
    }

#define ARM9addrDPlli() \
    ARM9addrDPops(); \
    __asm { \
        __asm mov ecx,edx                     /* Make op copy */ \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        \
        __asm shr ecx,7                       /* Shift in shftamount */ \
        __asm and ecx,31                      /* Mask off amount */ \
        __asm jz noshft                       /* Skip if no shift */ \
        __asm shl ebx,cl                      /* Shift operand1 */ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
	\
        __asm mov ecx,[arm9reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm shftdone:                       /* all done! */ \
        __asm setc [arm9reg+ARMREG_TC*4] \
    }

#define ARM9addrDPllr() \
    ARM9addrDPops(); \
    __asm { \
        __asm cmp ecx,15 \
        __asm jl pt3 \
        __asm add ebx,4 \
        __asm pt3: \
        \
        __asm mov ecx,edx                     /* Make op copy */ \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        \
        __asm shr ecx,8                       /* Shift in Rs */ \
        __asm and ecx,15                      /* Mask off Rs */ \
        __asm mov ecx,[arm9reg+ecx*4]         /* Read Rs's value */ \
        __asm and ecx,255                     /* Get the low byte */ \
        __asm jz noshft                       /* Skip if no shift */ \
        __asm shl ebx,cl                      /* Shift operand1 */ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
	\
        __asm mov ecx,[arm9reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm shftdone:                       /* all done! */ \
        __asm setc [arm9reg+ARMREG_TC*4] \
    }

#define ARM9addrDPlri() \
    ARM9addrDPops(); \
    __asm { \
        __asm mov ecx,edx                     /* Make op copy */ \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        \
        __asm shr ecx,7                       /* Shift in shftamount */ \
        __asm and ecx,31                      /* Mask off amount */ \
        __asm jz noshft                       /* Skip if no shift */ \
        __asm shr ebx,cl                      /* Shift operand1 */ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
	\
        __asm mov ecx,ebx                     /* Make a copy */ \
        __asm xor ebx,ebx                     /* Rm=0 */ \
        __asm shl ecx,1                       /* Carry = Rm[31] */ \
        __asm shftdone:                       /* all done! */ \
        __asm setc [arm9reg+ARMREG_TC*4] \
    }

#define ARM9addrDPlrr() \
    ARM9addrDPops(); \
    __asm { \
        __asm cmp ecx,15 \
        __asm jl pt3 \
        __asm add ebx,4 \
        __asm pt3: \
        \
        __asm mov ecx,edx                     /* Make op copy */ \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        \
        __asm shr ecx,8                       /* Shift in Rs */ \
        __asm and ecx,15                      /* Mask off Rs */ \
        __asm mov ecx,[arm9reg+ecx*4]         /* Read Rs's value */ \
        __asm and ecx,255                     /* Get the low byte */ \
        __asm jz noshft                       /* Skip if no shift */ \
        __asm cmp cl,32                       /* If greater than 32 */ \
        __asm jg overshft                     /* Shift all the way */ \
        __asm je fullshft                     /* If equal, shift to carry */ \
        __asm shr ebx,cl                      /* Shift operand1 */ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
        \
        __asm mov ecx,[arm9reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm jmp shftdone                    /* Skip over */ \
        __asm fullshft:                       /* Shift by 32 */ \
        __asm mov ecx,ebx                     /* Make a copy */ \
        __asm xor ebx,ebx                     /* Rm=0 */ \
        __asm shl ecx,1                       /* Carry = Rm[31] */ \
        __asm jmp shftdone                    /* Skip over */ \
        __asm overshft:                       /* Shift over 32 */ \
        __asm xor ebx,ebx                     /* Rm=0 */ \
        __asm shr ebx,1                       /* And carry=0 */ \
        __asm shftdone:                       /* all done! */ \
        __asm setc [arm9reg+ARMREG_TC*4] \
    }

#define ARM9addrDPari() \
    ARM9addrDPops(); \
    __asm { \
        __asm mov ecx,edx                     /* Make op copy */ \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        \
        __asm shr ecx,7                       /* Shift in shftamount */ \
        __asm and ecx,31                      /* Mask off amount */ \
        __asm jz noshft                       /* Skip if no shift */ \
        __asm sar ebx,cl                      /* Shift operand1 */ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
	\
 	__asm mov ecx,ebx                     /* Make a copy */ \
        __asm sar ebx,31                      /* Fill with hi bit */ \
        __asm shl ecx,1                       /* Carry = Rm[31] */ \
        __asm shftdone:                       /* all done! */ \
        __asm setc [arm9reg+ARMREG_TC*4] \
    }

#define ARM9addrDParr() \
    ARM9addrDPops(); \
    __asm { \
        __asm cmp ecx,15 \
        __asm jl pt3 \
        __asm add ebx,4 \
        __asm pt3: \
        \
        __asm mov ecx,edx                     /* Make op copy */ \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        \
        __asm shr ecx,8                       /* Shift in Rs */ \
        __asm and ecx,15                      /* Mask off Rs */ \
        __asm mov ecx,[arm9reg+ecx*4]         /* Read Rs's value */ \
        __asm and ecx,255                     /* Get the low byte */ \
        __asm jz noshft                       /* Skip if no shift */ \
        __asm cmp cl,32                       /* If greater than 32 */ \
        __asm jge fullshft                    /* Shift all the way */ \
        __asm sar ebx,cl                      /* Shift operand1 */ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
	\
        __asm mov ecx,[arm9reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm jmp shftdone                    /* and skip over */ \
        __asm fullshft:                       /* shift over 31 */ \
 	__asm mov ecx,ebx                     /* Make a copy */ \
        __asm sar ebx,31                      /* Fill with hi bit */ \
        __asm shl ecx,1                       /* Carry = Rm[31] */ \
        __asm shftdone:                       /* all done! */ \
        __asm setc [arm9reg+ARMREG_TC*4] \
    }

#define ARM9addrDPrri() \
    ARM9addrDPops(); \
    __asm { \
        __asm mov ecx,edx                     /* Make op copy */ \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        \
        __asm shr ecx,7                       /* Shift in shftamount */ \
        __asm and ecx,31                      /* Mask off amount */ \
        __asm jz noshft                       /* Skip if no shift */ \
        __asm ror ebx,cl                      /* Shift operand1 */ \
/*        __asm mov ecx,ebx */ \
/*        __asm shr ecx,1 */ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
	\
	__asm push ebx                        /* Keep orig value */ \
        __asm mov ecx,[arm9reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shl ecx,31                      /* Shift to high */ \
        __asm shr ebx,1                       /* 33-bit shift */ \
        __asm or ebx,ecx                      /* Add together */ \
        __asm pop ecx                         /* Get orig back */ \
        __asm shr ecx,1                       /* Carry = Rm[0] */ \
        __asm shftdone:                       /* all done! */ \
        __asm setc [arm9reg+ARMREG_TC*4] \
    }

#define ARM9addrDPrrr() \
    ARM9addrDPops(); \
    __asm { \
        __asm cmp ecx,15 \
        __asm jl pt3 \
        __asm add ebx,4 \
        __asm pt3: \
        \
        __asm mov ecx,edx                     /* Make op copy */ \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        \
        __asm shr ecx,8                       /* Shift in Rs */ \
        __asm and ecx,15                      /* Mask off Rs */ \
        __asm mov ecx,[arm9reg+ecx*4]         /* Read Rs's value */ \
        __asm and ecx,255                     /* Get the low byte */ \
        __asm jz noshft                       /* Skip if no shift */ \
        __asm and cl,31                       /* Check [4:0] */ \
        __asm jz fullshft                     /* And skip if full */ \
        __asm ror ebx,cl                      /* Shift operand1 */ \
/*        __asm mov ecx,ebx*/ \
/*        __asm shr ecx,1*/ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
	\
        __asm mov ecx,[arm9reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm jmp shftdone                    /* Skip over */ \
        __asm fullshft:                       /* Shifting by full */ \
        __asm mov ecx,ebx                     /* Make a copy of Rm */ \
        __asm shl ecx,1                       /* Carry = hi bit */ \
        __asm shftdone:                       /* all done! */ \
        __asm setc [arm9reg+ARMREG_TC*4] \
    }

/*---eax=operand1, ebx=operand2, edx=destreg----*/

#define ARM9opAND() \
    __asm { \
        __asm and eax,ebx \
        __asm mov [arm9reg+edx*4],eax \
    }

#define ARM9opEOR() \
    __asm { \
        __asm xor eax,ebx \
        __asm mov [arm9reg+edx*4],eax \
    }

#define ARM9opSUB() \
    __asm { \
        __asm sub eax,ebx \
        __asm mov [arm9reg+edx*4],eax \
    }

#define ARM9opRSB() \
    __asm { \
        __asm mov ecx,[arm9reg+ARMREG_TC*4] \
        __asm not ecx \
        __asm shr ecx,1 \
        __asm sub ebx,eax \
        __asm mov [arm9reg+edx*4],ebx \
    }

#define ARM9opADD() \
    __asm { \
        __asm add eax,ebx \
        __asm mov [arm9reg+edx*4],eax \
    }

#define ARM9opADC() \
    __asm { \
        __asm adc eax,ebx \
        __asm mov [arm9reg+edx*4],eax \
    }

#define ARM9opSBC() \
    __asm { \
        __asm mov ecx,[arm9reg+ARMREG_TC*4] \
        __asm not ecx \
        __asm shr ecx,1 \
        __asm sbb eax,ebx \
        __asm mov [arm9reg+edx*4],eax \
    }

#define ARM9opRSC() \
    __asm { \
        __asm mov ecx,[arm9reg+ARMREG_TC*4] \
        __asm not ecx \
        __asm shr ecx,1 \
        __asm sbb ebx,eax \
        __asm mov [arm9reg+edx*4],ebx \
    }

#define ARM9opTST() \
    __asm { \
        __asm and eax,ebx \
    }

#define ARM9opTEQ() \
    __asm { \
        __asm xor eax,ebx \
    }

#define ARM9opCMP() \
    __asm { \
        __asm sub eax,ebx \
    }

#define ARM9opCMN() \
    __asm { \
        __asm add eax,ebx \
    }

#define ARM9opORR() \
    __asm { \
        __asm or eax,ebx \
        __asm mov [arm9reg+edx*4],eax \
    }

#define ARM9opMOV() \
    __asm { \
        __asm or ebx,ebx \
        __asm mov [arm9reg+edx*4],ebx \
    }

#define ARM9opBIC() \
    __asm { \
        __asm not ebx \
        __asm and ebx,eax \
        __asm mov [arm9reg+edx*4],ebx \
    }

#define ARM9opMVN() \
    __asm { \
        __asm not ebx \
        __asm mov [arm9reg+edx*4],ebx \
    }

#define ARM9addrDPflagR15() \
    __asm { \
        __asm cmp edx,15                       /* Is Rd 15? */ \
        __asm jl flagdone                      /* If not, skip */ \
        __asm mov eax,[arm9reg+ARMREG_MODE*4]  /* Get current mode */ \
        __asm mov eax,[arm9reg+ARMREG_SUSR*4+eax*4]     /* Read mode's SPSR */ \
        __asm mov ecx,eax                      /* Make a copy of it */ \
	__asm mov ebx,[arm9reg+ARMREG_CPSR*4]  /* Read CPSR */ \
	__asm and ecx,15                       /* Mask spsr mode */ \
	__asm and ebx,15                       /* and cpsr mode */ \
	__asm cmp ecx,ebx                      /* They the same? */ \
	__asm je flageq                        /* If so, skip */ \
	__asm push eax                         /* modesw mods eax */ \
	__asm push [ARM9cpsrtomode+ecx*4]       /* New mode */ \
	__asm push [ARM9cpsrtomode+ebx*4]       /* Old mode */ \
	__asm call ARM9modesw                  /* Switch modes */ \
	__asm pop eax                          /* Pull param2 */ \
	__asm pop eax                          /* and param1 */ \
	__asm pop eax                          /* Get eax back */ \
	__asm flageq:                          /* Modes unchanged */ \
	__asm mov [arm9reg+ARMREG_CPSR*4],eax  /* CPSR=SPSR */ \
	__asm call ARM9splitCPSR               /* Redo flags */ \
        __asm flagdone:                        /* All done! */ \
    }

#define ARM9addrDPflagBIT() \
    __asm { \
        __asm sets  [arm9reg+ARMREG_FN*4]      /* Copy N flag */ \
        __asm setz  [arm9reg+ARMREG_FZ*4]      /* and Z flag */ \
        __asm mov ecx,[arm9reg+ARMREG_TC*4] \
        __asm mov [arm9reg+ARMREG_FC*4],ecx \
		__asm push edx \
    } \
	ARM9updateCPSR(); \
	__asm pop edx \
    ARM9addrDPflagR15()

#define ARM9addrDPflagADD() \
    __asm { \
        __asm sets  [arm9reg+ARMREG_FN*4]    /* Copy N flag */ \
        __asm setz  [arm9reg+ARMREG_FZ*4]    /* and Z flag */ \
        __asm seto  [arm9reg+ARMREG_FV*4]    /* V too */ \
        __asm setc [arm9reg+ARMREG_FC*4]    /* REVERSE C flag */ \
		__asm push edx \
    } \
	ARM9updateCPSR(); \
	__asm pop edx \
    ARM9addrDPflagR15()

#define ARM9addrDPflagADC() \
    __asm { \
        __asm sets  [arm9reg+ARMREG_FN*4]    /* Copy N flag */ \
        __asm setz  [arm9reg+ARMREG_FZ*4]    /* and Z flag */ \
        __asm seto  [arm9reg+ARMREG_FV*4]    /* V too */ \
        __asm setc [arm9reg+ARMREG_FC*4]    /* REVERSE C flag */ \
		__asm push edx \
    } \
	ARM9updateCPSR(); \
	__asm pop edx \
    ARM9addrDPflagR15()

#define ARM9addrDPflagSUB() \
    __asm { \
        __asm sets  [arm9reg+ARMREG_FN*4]    /* Copy N flag */ \
        __asm setz  [arm9reg+ARMREG_FZ*4]    /* and Z flag */ \
        __asm seto  [arm9reg+ARMREG_FV*4]    /* V too */ \
        __asm setnc [arm9reg+ARMREG_FC*4]    /* REVERSE C flag */ \
		__asm push edx \
    } \
	ARM9updateCPSR(); \
	__asm pop edx \
    ARM9addrDPflagR15()

#define ARM9addrDPflagSBC() \
    __asm { \
        __asm sets  [arm9reg+ARMREG_FN*4]    /* Copy N flag */ \
        __asm setz  [arm9reg+ARMREG_FZ*4]    /* and Z flag */ \
        __asm seto  [arm9reg+ARMREG_FV*4]    /* V too */ \
        __asm setnc [arm9reg+ARMREG_FC*4]    /* REVERSE C flag */ \
		__asm push edx \
    } \
	ARM9updateCPSR(); \
	__asm pop edx \
    ARM9addrDPflagR15()

#define ARM9addrDPflagRSB() \
    __asm { \
        __asm sets  [arm9reg+ARMREG_FN*4]    /* Copy N flag */ \
        __asm setz  [arm9reg+ARMREG_FZ*4]    /* and Z flag */ \
        __asm seto  [arm9reg+ARMREG_FV*4]    /* V too */ \
        __asm setnc [arm9reg+ARMREG_FC*4]    /* REVERSE C flag */ \
		__asm push edx \
    } \
	ARM9updateCPSR(); \
	__asm pop edx \
    ARM9addrDPflagR15()

#define ARM9addrDPflagRSC() \
    __asm { \
        __asm sets  [arm9reg+ARMREG_FN*4]    /* Copy N flag */ \
        __asm setz  [arm9reg+ARMREG_FZ*4]    /* and Z flag */ \
        __asm seto  [arm9reg+ARMREG_FV*4]    /* V too */ \
        __asm setnc [arm9reg+ARMREG_FC*4]    /* REVERSE C flag */ \
		__asm push edx \
    } \
	ARM9updateCPSR(); \
	__asm pop edx \
    ARM9addrDPflagR15()

#define ARM9addrDPflagCMN() \
    __asm { \
        __asm sets  [arm9reg+ARMREG_FN*4]    /* Copy N flag */ \
        __asm setz  [arm9reg+ARMREG_FZ*4]    /* and Z flag */ \
        __asm seto  [arm9reg+ARMREG_FV*4]    /* V too */ \
        __asm setc [arm9reg+ARMREG_FC*4]    /* REVERSE C flag */ \
		__asm push edx \
    } \
	ARM9updateCPSR(); \
	__asm pop edx \
    ARM9addrDPflagR15()

#define ARM9addrDPflagCMP() \
    __asm { \
        __asm sets  [arm9reg+ARMREG_FN*4]    /* Copy N flag */ \
        __asm setz  [arm9reg+ARMREG_FZ*4]    /* and Z flag */ \
        __asm seto  [arm9reg+ARMREG_FV*4]    /* V too */ \
        __asm setnc [arm9reg+ARMREG_FC*4]    /* REVERSE C flag */ \
		__asm push edx \
    } \
	ARM9updateCPSR(); \
	__asm pop edx \
    ARM9addrDPflagR15()

OPCODE ARM9opANDreg() { ARM9addrDPreg(); ARM9opAND(); return 2; }
OPCODE ARM9opANDimm() { ARM9addrDPimm(); ARM9opAND(); return 1; }
OPCODE ARM9opANDlli() { ARM9addrDPlli(); ARM9opAND(); return 2; }
OPCODE ARM9opANDllr() { ARM9addrDPllr(); ARM9opAND(); return 2; }
OPCODE ARM9opANDlri() { ARM9addrDPlri(); ARM9opAND(); return 2; }
OPCODE ARM9opANDlrr() { ARM9addrDPlrr(); ARM9opAND(); return 2; }
OPCODE ARM9opANDari() { ARM9addrDPari(); ARM9opAND(); return 2; }
OPCODE ARM9opANDarr() { ARM9addrDParr(); ARM9opAND(); return 2; }
OPCODE ARM9opANDrri() { ARM9addrDPrri(); ARM9opAND(); return 2; }
OPCODE ARM9opANDrrr() { ARM9addrDPrrr(); ARM9opAND(); return 2; }

OPCODE ARM9opEORreg() { ARM9addrDPreg(); ARM9opEOR(); return 2; }
OPCODE ARM9opEORimm() { ARM9addrDPimm(); ARM9opEOR(); return 1; }
OPCODE ARM9opEORlli() { ARM9addrDPlli(); ARM9opEOR(); return 2; }
OPCODE ARM9opEORllr() { ARM9addrDPllr(); ARM9opEOR(); return 2; }
OPCODE ARM9opEORlri() { ARM9addrDPlri(); ARM9opEOR(); return 2; }
OPCODE ARM9opEORlrr() { ARM9addrDPlrr(); ARM9opEOR(); return 2; }
OPCODE ARM9opEORari() { ARM9addrDPari(); ARM9opEOR(); return 2; }
OPCODE ARM9opEORarr() { ARM9addrDParr(); ARM9opEOR(); return 2; }
OPCODE ARM9opEORrri() { ARM9addrDPrri(); ARM9opEOR(); return 2; }
OPCODE ARM9opEORrrr() { ARM9addrDPrrr(); ARM9opEOR(); return 2; }

OPCODE ARM9opSUBreg() { ARM9addrDPreg(); ARM9opSUB(); return 2; }
OPCODE ARM9opSUBimm() { ARM9addrDPimm(); ARM9opSUB(); return 1; }
OPCODE ARM9opSUBlli() { ARM9addrDPlli(); ARM9opSUB(); return 2; }
OPCODE ARM9opSUBllr() { ARM9addrDPllr(); ARM9opSUB(); return 2; }
OPCODE ARM9opSUBlri() { ARM9addrDPlri(); ARM9opSUB(); return 2; }
OPCODE ARM9opSUBlrr() { ARM9addrDPlrr(); ARM9opSUB(); return 2; }
OPCODE ARM9opSUBari() { ARM9addrDPari(); ARM9opSUB(); return 2; }
OPCODE ARM9opSUBarr() { ARM9addrDParr(); ARM9opSUB(); return 2; }
OPCODE ARM9opSUBrri() { ARM9addrDPrri(); ARM9opSUB(); return 2; }
OPCODE ARM9opSUBrrr() { ARM9addrDPrrr(); ARM9opSUB(); return 2; }

OPCODE ARM9opRSBreg() { ARM9addrDPreg(); ARM9opRSB(); return 2; }
OPCODE ARM9opRSBimm() { ARM9addrDPimm(); ARM9opRSB(); return 1; }
OPCODE ARM9opRSBlli() { ARM9addrDPlli(); ARM9opRSB(); return 2; }
OPCODE ARM9opRSBllr() { ARM9addrDPllr(); ARM9opRSB(); return 2; }
OPCODE ARM9opRSBlri() { ARM9addrDPlri(); ARM9opRSB(); return 2; }
OPCODE ARM9opRSBlrr() { ARM9addrDPlrr(); ARM9opRSB(); return 2; }
OPCODE ARM9opRSBari() { ARM9addrDPari(); ARM9opRSB(); return 2; }
OPCODE ARM9opRSBarr() { ARM9addrDParr(); ARM9opRSB(); return 2; }
OPCODE ARM9opRSBrri() { ARM9addrDPrri(); ARM9opRSB(); return 2; }
OPCODE ARM9opRSBrrr() { ARM9addrDPrrr(); ARM9opRSB(); return 2; }

OPCODE ARM9opADDreg() { ARM9addrDPreg(); ARM9opADD(); return 2; }
OPCODE ARM9opADDimm() { ARM9addrDPimm(); ARM9opADD(); return 1; }
OPCODE ARM9opADDlli() { ARM9addrDPlli(); ARM9opADD(); return 2; }
OPCODE ARM9opADDllr() { ARM9addrDPllr(); ARM9opADD(); return 2; }
OPCODE ARM9opADDlri() { ARM9addrDPlri(); ARM9opADD(); return 2; }
OPCODE ARM9opADDlrr() { ARM9addrDPlrr(); ARM9opADD(); return 2; }
OPCODE ARM9opADDari() { ARM9addrDPari(); ARM9opADD(); return 2; }
OPCODE ARM9opADDarr() { ARM9addrDParr(); ARM9opADD(); return 2; }
OPCODE ARM9opADDrri() { ARM9addrDPrri(); ARM9opADD(); return 2; }
OPCODE ARM9opADDrrr() { ARM9addrDPrrr(); ARM9opADD(); return 2; }

OPCODE ARM9opADCreg() { ARM9addrDPreg(); ARM9opADC(); return 2; }
OPCODE ARM9opADCimm() { ARM9addrDPimm(); ARM9opADC(); return 1; }
OPCODE ARM9opADClli() { ARM9addrDPlli(); ARM9opADC(); return 2; }
OPCODE ARM9opADCllr() { ARM9addrDPllr(); ARM9opADC(); return 2; }
OPCODE ARM9opADClri() { ARM9addrDPlri(); ARM9opADC(); return 2; }
OPCODE ARM9opADClrr() { ARM9addrDPlrr(); ARM9opADC(); return 2; }
OPCODE ARM9opADCari() { ARM9addrDPari(); ARM9opADC(); return 2; }
OPCODE ARM9opADCarr() { ARM9addrDParr(); ARM9opADC(); return 2; }
OPCODE ARM9opADCrri() { ARM9addrDPrri(); ARM9opADC(); return 2; }
OPCODE ARM9opADCrrr() { ARM9addrDPrrr(); ARM9opADC(); return 2; }

OPCODE ARM9opSBCreg() { ARM9addrDPreg(); ARM9opSBC(); return 2; }
OPCODE ARM9opSBCimm() { ARM9addrDPimm(); ARM9opSBC(); return 1; }
OPCODE ARM9opSBClli() { ARM9addrDPlli(); ARM9opSBC(); return 2; }
OPCODE ARM9opSBCllr() { ARM9addrDPllr(); ARM9opSBC(); return 2; }
OPCODE ARM9opSBClri() { ARM9addrDPlri(); ARM9opSBC(); return 2; }
OPCODE ARM9opSBClrr() { ARM9addrDPlrr(); ARM9opSBC(); return 2; }
OPCODE ARM9opSBCari() { ARM9addrDPari(); ARM9opSBC(); return 2; }
OPCODE ARM9opSBCarr() { ARM9addrDParr(); ARM9opSBC(); return 2; }
OPCODE ARM9opSBCrri() { ARM9addrDPrri(); ARM9opSBC(); return 2; }
OPCODE ARM9opSBCrrr() { ARM9addrDPrrr(); ARM9opSBC(); return 2; }

OPCODE ARM9opRSCreg() { ARM9addrDPreg(); ARM9opRSC(); return 2; }
OPCODE ARM9opRSCimm() { ARM9addrDPimm(); ARM9opRSC(); return 1; }
OPCODE ARM9opRSClli() { ARM9addrDPlli(); ARM9opRSC(); return 2; }
OPCODE ARM9opRSCllr() { ARM9addrDPllr(); ARM9opRSC(); return 2; }
OPCODE ARM9opRSClri() { ARM9addrDPlri(); ARM9opRSC(); return 2; }
OPCODE ARM9opRSClrr() { ARM9addrDPlrr(); ARM9opRSC(); return 2; }
OPCODE ARM9opRSCari() { ARM9addrDPari(); ARM9opRSC(); return 2; }
OPCODE ARM9opRSCarr() { ARM9addrDParr(); ARM9opRSC(); return 2; }
OPCODE ARM9opRSCrri() { ARM9addrDPrri(); ARM9opRSC(); return 2; }
OPCODE ARM9opRSCrrr() { ARM9addrDPrrr(); ARM9opRSC(); return 2; }

OPCODE ARM9opTSTreg() { ARM9addrDPreg(); ARM9opTST(); return 2; }
OPCODE ARM9opTSTimm() { ARM9addrDPimm(); ARM9opTST(); return 1; }
OPCODE ARM9opTSTlli() { ARM9addrDPlli(); ARM9opTST(); return 2; }
OPCODE ARM9opTSTllr() { ARM9addrDPllr(); ARM9opTST(); return 2; }
OPCODE ARM9opTSTlri() { ARM9addrDPlri(); ARM9opTST(); return 2; }
OPCODE ARM9opTSTlrr() { ARM9addrDPlrr(); ARM9opTST(); return 2; }
OPCODE ARM9opTSTari() { ARM9addrDPari(); ARM9opTST(); return 2; }
OPCODE ARM9opTSTarr() { ARM9addrDParr(); ARM9opTST(); return 2; }
OPCODE ARM9opTSTrri() { ARM9addrDPrri(); ARM9opTST(); return 2; }
OPCODE ARM9opTSTrrr() { ARM9addrDPrrr(); ARM9opTST(); return 2; }

OPCODE ARM9opTEQreg() { ARM9addrDPreg(); ARM9opTEQ(); return 2; }
OPCODE ARM9opTEQimm() { ARM9addrDPimm(); ARM9opTEQ(); return 1; }
OPCODE ARM9opTEQlli() { ARM9addrDPlli(); ARM9opTEQ(); return 2; }
OPCODE ARM9opTEQllr() { ARM9addrDPllr(); ARM9opTEQ(); return 2; }
OPCODE ARM9opTEQlri() { ARM9addrDPlri(); ARM9opTEQ(); return 2; }
OPCODE ARM9opTEQlrr() { ARM9addrDPlrr(); ARM9opTEQ(); return 2; }
OPCODE ARM9opTEQari() { ARM9addrDPari(); ARM9opTEQ(); return 2; }
OPCODE ARM9opTEQarr() { ARM9addrDParr(); ARM9opTEQ(); return 2; }
OPCODE ARM9opTEQrri() { ARM9addrDPrri(); ARM9opTEQ(); return 2; }
OPCODE ARM9opTEQrrr() { ARM9addrDPrrr(); ARM9opTEQ(); return 2; }

OPCODE ARM9opCMPreg() { ARM9addrDPreg(); ARM9opCMP(); return 2; }
OPCODE ARM9opCMPimm() { ARM9addrDPimm(); ARM9opCMP(); return 1; }
OPCODE ARM9opCMPlli() { ARM9addrDPlli(); ARM9opCMP(); return 2; }
OPCODE ARM9opCMPllr() { ARM9addrDPllr(); ARM9opCMP(); return 2; }
OPCODE ARM9opCMPlri() { ARM9addrDPlri(); ARM9opCMP(); return 2; }
OPCODE ARM9opCMPlrr() { ARM9addrDPlrr(); ARM9opCMP(); return 2; }
OPCODE ARM9opCMPari() { ARM9addrDPari(); ARM9opCMP(); return 2; }
OPCODE ARM9opCMParr() { ARM9addrDParr(); ARM9opCMP(); return 2; }
OPCODE ARM9opCMPrri() { ARM9addrDPrri(); ARM9opCMP(); return 2; }
OPCODE ARM9opCMPrrr() { ARM9addrDPrrr(); ARM9opCMP(); return 2; }

OPCODE ARM9opCMNreg() { ARM9addrDPreg(); ARM9opCMN(); return 2; }
OPCODE ARM9opCMNimm() { ARM9addrDPimm(); ARM9opCMN(); return 1; }
OPCODE ARM9opCMNlli() { ARM9addrDPlli(); ARM9opCMN(); return 2; }
OPCODE ARM9opCMNllr() { ARM9addrDPllr(); ARM9opCMN(); return 2; }
OPCODE ARM9opCMNlri() { ARM9addrDPlri(); ARM9opCMN(); return 2; }
OPCODE ARM9opCMNlrr() { ARM9addrDPlrr(); ARM9opCMN(); return 2; }
OPCODE ARM9opCMNari() { ARM9addrDPari(); ARM9opCMN(); return 2; }
OPCODE ARM9opCMNarr() { ARM9addrDParr(); ARM9opCMN(); return 2; }
OPCODE ARM9opCMNrri() { ARM9addrDPrri(); ARM9opCMN(); return 2; }
OPCODE ARM9opCMNrrr() { ARM9addrDPrrr(); ARM9opCMN(); return 2; }

OPCODE ARM9opORRreg() { ARM9addrDPreg(); ARM9opORR(); return 2; }
OPCODE ARM9opORRimm() { ARM9addrDPimm(); ARM9opORR(); return 1; }
OPCODE ARM9opORRlli() { ARM9addrDPlli(); ARM9opORR(); return 2; }
OPCODE ARM9opORRllr() { ARM9addrDPllr(); ARM9opORR(); return 2; }
OPCODE ARM9opORRlri() { ARM9addrDPlri(); ARM9opORR(); return 2; }
OPCODE ARM9opORRlrr() { ARM9addrDPlrr(); ARM9opORR(); return 2; }
OPCODE ARM9opORRari() { ARM9addrDPari(); ARM9opORR(); return 2; }
OPCODE ARM9opORRarr() { ARM9addrDParr(); ARM9opORR(); return 2; }
OPCODE ARM9opORRrri() { ARM9addrDPrri(); ARM9opORR(); return 2; }
OPCODE ARM9opORRrrr() { ARM9addrDPrrr(); ARM9opORR(); return 2; }

OPCODE ARM9opMOVreg() { ARM9addrDPreg(); ARM9opMOV(); return 2; }
OPCODE ARM9opMOVimm() { ARM9addrDPimm(); ARM9opMOV(); return 1; }
OPCODE ARM9opMOVlli() { ARM9addrDPlli(); ARM9opMOV(); return 2; }
OPCODE ARM9opMOVllr() { ARM9addrDPllr(); ARM9opMOV(); return 2; }
OPCODE ARM9opMOVlri() { ARM9addrDPlri(); ARM9opMOV(); return 2; }
OPCODE ARM9opMOVlrr() { ARM9addrDPlrr(); ARM9opMOV(); return 2; }
OPCODE ARM9opMOVari() { ARM9addrDPari(); ARM9opMOV(); return 2; }
OPCODE ARM9opMOVarr() { ARM9addrDParr(); ARM9opMOV(); return 2; }
OPCODE ARM9opMOVrri() { ARM9addrDPrri(); ARM9opMOV(); return 2; }
OPCODE ARM9opMOVrrr() { ARM9addrDPrrr(); ARM9opMOV(); return 2; }

OPCODE ARM9opBICreg() { ARM9addrDPreg(); ARM9opBIC(); return 2; }
OPCODE ARM9opBICimm() { ARM9addrDPimm(); ARM9opBIC(); return 1; }
OPCODE ARM9opBIClli() { ARM9addrDPlli(); ARM9opBIC(); return 2; }
OPCODE ARM9opBICllr() { ARM9addrDPllr(); ARM9opBIC(); return 2; }
OPCODE ARM9opBIClri() { ARM9addrDPlri(); ARM9opBIC(); return 2; }
OPCODE ARM9opBIClrr() { ARM9addrDPlrr(); ARM9opBIC(); return 2; }
OPCODE ARM9opBICari() { ARM9addrDPari(); ARM9opBIC(); return 2; }
OPCODE ARM9opBICarr() { ARM9addrDParr(); ARM9opBIC(); return 2; }
OPCODE ARM9opBICrri() { ARM9addrDPrri(); ARM9opBIC(); return 2; }
OPCODE ARM9opBICrrr() { ARM9addrDPrrr(); ARM9opBIC(); return 2; }

OPCODE ARM9opMVNreg() { ARM9addrDPreg(); ARM9opMVN(); return 2; }
OPCODE ARM9opMVNimm() { ARM9addrDPimm(); ARM9opMVN(); return 1; }
OPCODE ARM9opMVNlli() { ARM9addrDPlli(); ARM9opMVN(); return 2; }
OPCODE ARM9opMVNllr() { ARM9addrDPllr(); ARM9opMVN(); return 2; }
OPCODE ARM9opMVNlri() { ARM9addrDPlri(); ARM9opMVN(); return 2; }
OPCODE ARM9opMVNlrr() { ARM9addrDPlrr(); ARM9opMVN(); return 2; }
OPCODE ARM9opMVNari() { ARM9addrDPari(); ARM9opMVN(); return 2; }
OPCODE ARM9opMVNarr() { ARM9addrDParr(); ARM9opMVN(); return 2; }
OPCODE ARM9opMVNrri() { ARM9addrDPrri(); ARM9opMVN(); return 2; }
OPCODE ARM9opMVNrrr() { ARM9addrDPrrr(); ARM9opMVN(); return 2; }

OPCODE ARM9opANDSreg() { ARM9addrDPreg(); ARM9opAND(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opANDSimm() { ARM9addrDPimm(); ARM9opAND(); ARM9addrDPflagBIT(); return 1; }
OPCODE ARM9opANDSlli() { ARM9addrDPlli(); ARM9opAND(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opANDSllr() { ARM9addrDPllr(); ARM9opAND(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opANDSlri() { ARM9addrDPlri(); ARM9opAND(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opANDSlrr() { ARM9addrDPlrr(); ARM9opAND(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opANDSari() { ARM9addrDPari(); ARM9opAND(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opANDSarr() { ARM9addrDParr(); ARM9opAND(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opANDSrri() { ARM9addrDPrri(); ARM9opAND(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opANDSrrr() { ARM9addrDPrrr(); ARM9opAND(); ARM9addrDPflagBIT(); return 2; }

OPCODE ARM9opEORSreg() { ARM9addrDPreg(); ARM9opEOR(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opEORSimm() { ARM9addrDPimm(); ARM9opEOR(); ARM9addrDPflagBIT(); return 1; }
OPCODE ARM9opEORSlli() { ARM9addrDPlli(); ARM9opEOR(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opEORSllr() { ARM9addrDPllr(); ARM9opEOR(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opEORSlri() { ARM9addrDPlri(); ARM9opEOR(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opEORSlrr() { ARM9addrDPlrr(); ARM9opEOR(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opEORSari() { ARM9addrDPari(); ARM9opEOR(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opEORSarr() { ARM9addrDParr(); ARM9opEOR(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opEORSrri() { ARM9addrDPrri(); ARM9opEOR(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opEORSrrr() { ARM9addrDPrrr(); ARM9opEOR(); ARM9addrDPflagBIT(); return 2; }


OPCODE ARM9opSUBSreg() { ARM9addrDPreg(); ARM9opSUB(); ARM9addrDPflagSUB(); return 2; }
OPCODE ARM9opSUBSimm() { ARM9addrDPimm(); ARM9opSUB(); ARM9addrDPflagSUB(); return 1; }
OPCODE ARM9opSUBSlli() { ARM9addrDPlli(); ARM9opSUB(); ARM9addrDPflagSUB(); return 2; }
OPCODE ARM9opSUBSllr() { ARM9addrDPllr(); ARM9opSUB(); ARM9addrDPflagSUB(); return 2; }
OPCODE ARM9opSUBSlri() { ARM9addrDPlri(); ARM9opSUB(); ARM9addrDPflagSUB(); return 2; }
OPCODE ARM9opSUBSlrr() { ARM9addrDPlrr(); ARM9opSUB(); ARM9addrDPflagSUB(); return 2; }
OPCODE ARM9opSUBSari() { ARM9addrDPari(); ARM9opSUB(); ARM9addrDPflagSUB(); return 2; }
OPCODE ARM9opSUBSarr() { ARM9addrDParr(); ARM9opSUB(); ARM9addrDPflagSUB(); return 2; }
OPCODE ARM9opSUBSrri() { ARM9addrDPrri(); ARM9opSUB(); ARM9addrDPflagSUB(); return 2; }
OPCODE ARM9opSUBSrrr() { ARM9addrDPrrr(); ARM9opSUB(); ARM9addrDPflagSUB(); return 2; }

OPCODE ARM9opRSBSreg() { ARM9addrDPreg(); ARM9opRSB(); ARM9addrDPflagRSB(); return 2; }
OPCODE ARM9opRSBSimm() { ARM9addrDPimm(); ARM9opRSB(); ARM9addrDPflagRSB(); return 1; }
OPCODE ARM9opRSBSlli() { ARM9addrDPlli(); ARM9opRSB(); ARM9addrDPflagRSB(); return 2; }
OPCODE ARM9opRSBSllr() { ARM9addrDPllr(); ARM9opRSB(); ARM9addrDPflagRSB(); return 2; }
OPCODE ARM9opRSBSlri() { ARM9addrDPlri(); ARM9opRSB(); ARM9addrDPflagRSB(); return 2; }
OPCODE ARM9opRSBSlrr() { ARM9addrDPlrr(); ARM9opRSB(); ARM9addrDPflagRSB(); return 2; }
OPCODE ARM9opRSBSari() { ARM9addrDPari(); ARM9opRSB(); ARM9addrDPflagRSB(); return 2; }
OPCODE ARM9opRSBSarr() { ARM9addrDParr(); ARM9opRSB(); ARM9addrDPflagRSB(); return 2; }
OPCODE ARM9opRSBSrri() { ARM9addrDPrri(); ARM9opRSB(); ARM9addrDPflagRSB(); return 2; }
OPCODE ARM9opRSBSrrr() { ARM9addrDPrrr(); ARM9opRSB(); ARM9addrDPflagRSB(); return 2; }

OPCODE ARM9opADDSreg() { ARM9addrDPreg(); ARM9opADD(); ARM9addrDPflagADD(); return 2; }
OPCODE ARM9opADDSimm() { ARM9addrDPimm(); ARM9opADD(); ARM9addrDPflagADD(); return 1; }
OPCODE ARM9opADDSlli() { ARM9addrDPlli(); ARM9opADD(); ARM9addrDPflagADD(); return 2; }
OPCODE ARM9opADDSllr() { ARM9addrDPllr(); ARM9opADD(); ARM9addrDPflagADD(); return 2; }
OPCODE ARM9opADDSlri() { ARM9addrDPlri(); ARM9opADD(); ARM9addrDPflagADD(); return 2; }
OPCODE ARM9opADDSlrr() { ARM9addrDPlrr(); ARM9opADD(); ARM9addrDPflagADD(); return 2; }
OPCODE ARM9opADDSari() { ARM9addrDPari(); ARM9opADD(); ARM9addrDPflagADD(); return 2; }
OPCODE ARM9opADDSarr() { ARM9addrDParr(); ARM9opADD(); ARM9addrDPflagADD(); return 2; }
OPCODE ARM9opADDSrri() { ARM9addrDPrri(); ARM9opADD(); ARM9addrDPflagADD(); return 2; }
OPCODE ARM9opADDSrrr() { ARM9addrDPrrr(); ARM9opADD(); ARM9addrDPflagADD(); return 2; }

OPCODE ARM9opADCSreg() { ARM9addrDPreg(); ARM9opADC(); ARM9addrDPflagADC(); return 2; }
OPCODE ARM9opADCSimm() { ARM9addrDPimm(); ARM9opADC(); ARM9addrDPflagADC(); return 1; }
OPCODE ARM9opADCSlli() { ARM9addrDPlli(); ARM9opADC(); ARM9addrDPflagADC(); return 2; }
OPCODE ARM9opADCSllr() { ARM9addrDPllr(); ARM9opADC(); ARM9addrDPflagADC(); return 2; }
OPCODE ARM9opADCSlri() { ARM9addrDPlri(); ARM9opADC(); ARM9addrDPflagADC(); return 2; }
OPCODE ARM9opADCSlrr() { ARM9addrDPlrr(); ARM9opADC(); ARM9addrDPflagADC(); return 2; }
OPCODE ARM9opADCSari() { ARM9addrDPari(); ARM9opADC(); ARM9addrDPflagADC(); return 2; }
OPCODE ARM9opADCSarr() { ARM9addrDParr(); ARM9opADC(); ARM9addrDPflagADC(); return 2; }
OPCODE ARM9opADCSrri() { ARM9addrDPrri(); ARM9opADC(); ARM9addrDPflagADC(); return 2; }
OPCODE ARM9opADCSrrr() { ARM9addrDPrrr(); ARM9opADC(); ARM9addrDPflagADC(); return 2; }

OPCODE ARM9opSBCSreg() { ARM9addrDPreg(); ARM9opSBC(); ARM9addrDPflagSBC(); return 2; }
OPCODE ARM9opSBCSimm() { ARM9addrDPimm(); ARM9opSBC(); ARM9addrDPflagSBC(); return 1; }
OPCODE ARM9opSBCSlli() { ARM9addrDPlli(); ARM9opSBC(); ARM9addrDPflagSBC(); return 2; }
OPCODE ARM9opSBCSllr() { ARM9addrDPllr(); ARM9opSBC(); ARM9addrDPflagSBC(); return 2; }
OPCODE ARM9opSBCSlri() { ARM9addrDPlri(); ARM9opSBC(); ARM9addrDPflagSBC(); return 2; }
OPCODE ARM9opSBCSlrr() { ARM9addrDPlrr(); ARM9opSBC(); ARM9addrDPflagSBC(); return 2; }
OPCODE ARM9opSBCSari() { ARM9addrDPari(); ARM9opSBC(); ARM9addrDPflagSBC(); return 2; }
OPCODE ARM9opSBCSarr() { ARM9addrDParr(); ARM9opSBC(); ARM9addrDPflagSBC(); return 2; }
OPCODE ARM9opSBCSrri() { ARM9addrDPrri(); ARM9opSBC(); ARM9addrDPflagSBC(); return 2; }
OPCODE ARM9opSBCSrrr() { ARM9addrDPrrr(); ARM9opSBC(); ARM9addrDPflagSBC(); return 2; }

OPCODE ARM9opRSCSreg() { ARM9addrDPreg(); ARM9opRSC(); ARM9addrDPflagRSC(); return 2; }
OPCODE ARM9opRSCSimm() { ARM9addrDPimm(); ARM9opRSC(); ARM9addrDPflagRSC(); return 1; }
OPCODE ARM9opRSCSlli() { ARM9addrDPlli(); ARM9opRSC(); ARM9addrDPflagRSC(); return 2; }
OPCODE ARM9opRSCSllr() { ARM9addrDPllr(); ARM9opRSC(); ARM9addrDPflagRSC(); return 2; }
OPCODE ARM9opRSCSlri() { ARM9addrDPlri(); ARM9opRSC(); ARM9addrDPflagRSC(); return 2; }
OPCODE ARM9opRSCSlrr() { ARM9addrDPlrr(); ARM9opRSC(); ARM9addrDPflagRSC(); return 2; }
OPCODE ARM9opRSCSari() { ARM9addrDPari(); ARM9opRSC(); ARM9addrDPflagRSC(); return 2; }
OPCODE ARM9opRSCSarr() { ARM9addrDParr(); ARM9opRSC(); ARM9addrDPflagRSC(); return 2; }
OPCODE ARM9opRSCSrri() { ARM9addrDPrri(); ARM9opRSC(); ARM9addrDPflagRSC(); return 2; }
OPCODE ARM9opRSCSrrr() { ARM9addrDPrrr(); ARM9opRSC(); ARM9addrDPflagRSC(); return 2; }

OPCODE ARM9opTSTSreg() { ARM9addrDPreg(); ARM9opTST(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opTSTSimm() { ARM9addrDPimm(); ARM9opTST(); ARM9addrDPflagBIT(); return 1; }
OPCODE ARM9opTSTSlli() { ARM9addrDPlli(); ARM9opTST(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opTSTSllr() { ARM9addrDPllr(); ARM9opTST(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opTSTSlri() { ARM9addrDPlri(); ARM9opTST(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opTSTSlrr() { ARM9addrDPlrr(); ARM9opTST(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opTSTSari() { ARM9addrDPari(); ARM9opTST(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opTSTSarr() { ARM9addrDParr(); ARM9opTST(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opTSTSrri() { ARM9addrDPrri(); ARM9opTST(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opTSTSrrr() { ARM9addrDPrrr(); ARM9opTST(); ARM9addrDPflagBIT(); return 2; }

OPCODE ARM9opTEQSreg() { ARM9addrDPreg(); ARM9opTEQ(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opTEQSimm() { ARM9addrDPimm(); ARM9opTEQ(); ARM9addrDPflagBIT(); return 1; }
OPCODE ARM9opTEQSlli() { ARM9addrDPlli(); ARM9opTEQ(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opTEQSllr() { ARM9addrDPllr(); ARM9opTEQ(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opTEQSlri() { ARM9addrDPlri(); ARM9opTEQ(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opTEQSlrr() { ARM9addrDPlrr(); ARM9opTEQ(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opTEQSari() { ARM9addrDPari(); ARM9opTEQ(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opTEQSarr() { ARM9addrDParr(); ARM9opTEQ(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opTEQSrri() { ARM9addrDPrri(); ARM9opTEQ(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opTEQSrrr() { ARM9addrDPrrr(); ARM9opTEQ(); ARM9addrDPflagBIT(); return 2; }

OPCODE ARM9opCMPSreg() { ARM9addrDPreg(); ARM9opCMP(); ARM9addrDPflagCMP(); return 2; }
OPCODE ARM9opCMPSimm() { ARM9addrDPimm(); ARM9opCMP(); ARM9addrDPflagCMP(); return 1; }
OPCODE ARM9opCMPSlli() { ARM9addrDPlli(); ARM9opCMP(); ARM9addrDPflagCMP(); return 2; }
OPCODE ARM9opCMPSllr() { ARM9addrDPllr(); ARM9opCMP(); ARM9addrDPflagCMP(); return 2; }
OPCODE ARM9opCMPSlri() { ARM9addrDPlri(); ARM9opCMP(); ARM9addrDPflagCMP(); return 2; }
OPCODE ARM9opCMPSlrr() { ARM9addrDPlrr(); ARM9opCMP(); ARM9addrDPflagCMP(); return 2; }
OPCODE ARM9opCMPSari() { ARM9addrDPari(); ARM9opCMP(); ARM9addrDPflagCMP(); return 2; }
OPCODE ARM9opCMPSarr() { ARM9addrDParr(); ARM9opCMP(); ARM9addrDPflagCMP(); return 2; }
OPCODE ARM9opCMPSrri() { ARM9addrDPrri(); ARM9opCMP(); ARM9addrDPflagCMP(); return 2; }
OPCODE ARM9opCMPSrrr() { ARM9addrDPrrr(); ARM9opCMP(); ARM9addrDPflagCMP(); return 2; }

OPCODE ARM9opCMNSreg() { ARM9addrDPreg(); ARM9opCMN(); ARM9addrDPflagCMN(); return 2; }
OPCODE ARM9opCMNSimm() { ARM9addrDPimm(); ARM9opCMN(); ARM9addrDPflagCMN(); return 1; }
OPCODE ARM9opCMNSlli() { ARM9addrDPlli(); ARM9opCMN(); ARM9addrDPflagCMN(); return 2; }
OPCODE ARM9opCMNSllr() { ARM9addrDPllr(); ARM9opCMN(); ARM9addrDPflagCMN(); return 2; }
OPCODE ARM9opCMNSlri() { ARM9addrDPlri(); ARM9opCMN(); ARM9addrDPflagCMN(); return 2; }
OPCODE ARM9opCMNSlrr() { ARM9addrDPlrr(); ARM9opCMN(); ARM9addrDPflagCMN(); return 2; }
OPCODE ARM9opCMNSari() { ARM9addrDPari(); ARM9opCMN(); ARM9addrDPflagCMN(); return 2; }
OPCODE ARM9opCMNSarr() { ARM9addrDParr(); ARM9opCMN(); ARM9addrDPflagCMN(); return 2; }
OPCODE ARM9opCMNSrri() { ARM9addrDPrri(); ARM9opCMN(); ARM9addrDPflagCMN(); return 2; }
OPCODE ARM9opCMNSrrr() { ARM9addrDPrrr(); ARM9opCMN(); ARM9addrDPflagCMN(); return 2; }

OPCODE ARM9opORRSreg() { ARM9addrDPreg(); ARM9opORR(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opORRSimm() { ARM9addrDPimm(); ARM9opORR(); ARM9addrDPflagBIT(); return 1; }
OPCODE ARM9opORRSlli() { ARM9addrDPlli(); ARM9opORR(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opORRSllr() { ARM9addrDPllr(); ARM9opORR(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opORRSlri() { ARM9addrDPlri(); ARM9opORR(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opORRSlrr() { ARM9addrDPlrr(); ARM9opORR(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opORRSari() { ARM9addrDPari(); ARM9opORR(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opORRSarr() { ARM9addrDParr(); ARM9opORR(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opORRSrri() { ARM9addrDPrri(); ARM9opORR(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opORRSrrr() { ARM9addrDPrrr(); ARM9opORR(); ARM9addrDPflagBIT(); return 2; }

OPCODE ARM9opMOVSreg() { ARM9addrDPreg(); ARM9opMOV(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opMOVSimm() { ARM9addrDPimm(); ARM9opMOV(); ARM9addrDPflagBIT(); return 1; }
OPCODE ARM9opMOVSlli() { ARM9addrDPlli(); ARM9opMOV(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opMOVSllr() { ARM9addrDPllr(); ARM9opMOV(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opMOVSlri() { ARM9addrDPlri(); ARM9opMOV(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opMOVSlrr() { ARM9addrDPlrr(); ARM9opMOV(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opMOVSari() { ARM9addrDPari(); ARM9opMOV(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opMOVSarr() { ARM9addrDParr(); ARM9opMOV(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opMOVSrri() { ARM9addrDPrri(); ARM9opMOV(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opMOVSrrr() { ARM9addrDPrrr(); ARM9opMOV(); ARM9addrDPflagBIT(); return 2; }

OPCODE ARM9opBICSreg() { ARM9addrDPreg(); ARM9opBIC(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opBICSimm() { ARM9addrDPimm(); ARM9opBIC(); ARM9addrDPflagBIT(); return 1; }
OPCODE ARM9opBICSlli() { ARM9addrDPlli(); ARM9opBIC(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opBICSllr() { ARM9addrDPllr(); ARM9opBIC(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opBICSlri() { ARM9addrDPlri(); ARM9opBIC(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opBICSlrr() { ARM9addrDPlrr(); ARM9opBIC(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opBICSari() { ARM9addrDPari(); ARM9opBIC(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opBICSarr() { ARM9addrDParr(); ARM9opBIC(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opBICSrri() { ARM9addrDPrri(); ARM9opBIC(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opBICSrrr() { ARM9addrDPrrr(); ARM9opBIC(); ARM9addrDPflagBIT(); return 2; }

OPCODE ARM9opMVNSreg() { ARM9addrDPreg(); ARM9opMVN(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opMVNSimm() { ARM9addrDPimm(); ARM9opMVN(); ARM9addrDPflagBIT(); return 1; }
OPCODE ARM9opMVNSlli() { ARM9addrDPlli(); ARM9opMVN(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opMVNSllr() { ARM9addrDPllr(); ARM9opMVN(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opMVNSlri() { ARM9addrDPlri(); ARM9opMVN(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opMVNSlrr() { ARM9addrDPlrr(); ARM9opMVN(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opMVNSari() { ARM9addrDPari(); ARM9opMVN(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opMVNSarr() { ARM9addrDParr(); ARM9opMVN(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opMVNSrri() { ARM9addrDPrri(); ARM9opMVN(); ARM9addrDPflagBIT(); return 2; }
OPCODE ARM9opMVNSrrr() { ARM9addrDPrrr(); ARM9opMVN(); ARM9addrDPflagBIT(); return 2; }

//---Load/Store addressing modes, opcodes----------------------------------

#define ARM9addrLSofim() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm and ebx,00000FFFh               /* Mask offset */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* Subtract offset */ \
    }

#define ARM9addrLSofip() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm and ebx,00000FFFh               /* Mask offset */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* Subtract offset */ \
    }

#define ARM9addrLSlmofim() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm mov eax,edx                     /* yet another */ \
        __asm shr eax,4                       /* Shift top nybble */ \
        __asm and ebx,15                      /* Mask low nybble */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm and eax,240                     /* Mask high nyb */ \
        __asm or ebx,eax                      /* Add nybbles */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* Subtract offset */ \
    }

#define ARM9addrLSlmofip() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm mov eax,edx                     /* yet another */ \
        __asm shr eax,4                       /* Shift top nybble */ \
        __asm and ebx,15                      /* Mask low nybble */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm and eax,240                     /* Mask high nyb */ \
        __asm or ebx,eax                      /* Add nybbles */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* Add offset */ \
    }

#define ARM9addrLSofrm() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM9addrLSofrmll() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
	__asm shl ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM9addrLSofrmlr() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm shl ebx,1                       /* Set carry */ \
        __asm xor ebx,ebx                     /* Operand=0 */ \
        __asm jmp shftend                     /* and skip over */\
        __asm shft1:                          /* Otherwise.. */ \
	__asm shr ebx,cl                      /* Shift Rm */ \
	__asm shftend:                        /* Good to go */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM9addrLSofrmar() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,31                      /* Set to 32 */ \
        __asm shft1:                          /* Otherwise.. */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
	__asm sar ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM9addrLSofrmrr() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,[arm9reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm rcr ebx,1                       /* 33-bit shift */ \
        __asm jmp shftend \
        __asm shft1:                          /* Otherwise.. */ \
	__asm ror ebx,cl                      /* Shift Rm */ \
	__asm shftend: \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM9addrLSofrp() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM9addrLSofrpll() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
	__asm shl ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM9addrLSofrplr() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm shl ebx,1                       /* Set carry */ \
        __asm xor ebx,ebx                     /* Operand=0 */ \
        __asm jmp shftend                     /* and skip over */\
        __asm shft1:                          /* Otherwise.. */ \
	__asm shr ebx,cl                      /* Shift Rm */ \
	__asm shftend:                        /* Good to go */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM9addrLSofrpar() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,31                      /* Set to 32 */ \
        __asm shft1:                          /* Otherwise.. */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
	__asm sar ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM9addrLSofrprr() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,[arm9reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm rcr ebx,1                       /* 33-bit shift */ \
        __asm jmp shftend                     /* And skip over this */ \
        __asm shft1:                          /* Otherwise.. */ \
	__asm ror ebx,cl                      /* Shift Rm */ \
	__asm shftend: \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM9addrLSprim() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm and ebx,00000FFFh               /* Mask offset */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* Subtract offset */ \
        __asm mov [arm9reg+ecx*4],eax         /* Write back */ \
    }

#define ARM9addrLSprip() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm and ebx,00000FFFh               /* Mask offset */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* Add offset */ \
        __asm mov [arm9reg+ecx*4],eax         /* Write back */ \
    }

#define ARM9addrLSlmprim() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm mov eax,edx                     /* yet another */ \
        __asm shr eax,4                       /* Shift top nybble */ \
        __asm and ebx,15                      /* Mask low nybble */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm and eax,240                     /* Mask high nyb */ \
        __asm or ebx,eax                      /* Add nybbles */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* Subtract offset */ \
        __asm mov [arm9reg+ecx*4],eax         /* Write back */ \
    }

#define ARM9addrLSlmprip() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm mov eax,edx                     /* yet another */ \
        __asm shr eax,4                       /* Shift top nybble */ \
        __asm and ebx,15                      /* Mask low nybble */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm and eax,240                     /* Mask high nyb */ \
        __asm or ebx,eax                      /* Add nybbles */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* Add offset */ \
        __asm mov [arm9reg+ecx*4],eax         /* Write back */ \
    }

#define ARM9addrLSprrm() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm9reg+ecx*4],eax         /* Write back */ \
    }

#define ARM9addrLSprrmll() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
	__asm shl ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm9reg+ecx*4],eax         /* Write back */ \
    }

#define ARM9addrLSprrmlr() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm shl ebx,1                       /* Set carry */ \
        __asm xor ebx,ebx                     /* Operand=0 */ \
        __asm jmp shftend                     /* and skip over */\
        __asm shft1:                          /* Otherwise.. */ \
	__asm shr ebx,cl                      /* Shift Rm */ \
	__asm shftend:                        /* Good to go */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm9reg+ecx*4],eax         /* Write back */ \
    }

#define ARM9addrLSprrmar() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,31                      /* Set to 32 */ \
        __asm shft1:                          /* Otherwise.. */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
	__asm sar ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm9reg+ecx*4],eax         /* Write back */ \
    }

#define ARM9addrLSprrmrr() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,[arm9reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm rcr ebx,1                       /* 33-bit shift */ \
        __asm jmp shftend                     /* And skip over this */ \
        __asm shft1:                          /* Otherwise.. */ \
	__asm ror ebx,cl                      /* Shift Rm */ \
	__asm shftend: \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm9reg+ecx*4],eax         /* Write back */ \
    }

#define ARM9addrLSprrp() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm9reg+ecx*4],eax         /* Write back */ \
    }

#define ARM9addrLSprrpll() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
	__asm shl ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm9reg+ecx*4],eax         /* Write back */ \
    }

#define ARM9addrLSprrplr() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm shl ebx,1                       /* Set carry */ \
        __asm xor ebx,ebx                     /* Operand=0 */ \
        __asm jmp shftend                     /* and skip over */\
        __asm shft1:                          /* Otherwise.. */ \
	__asm shr ebx,cl                      /* Shift Rm */ \
	__asm shftend:                        /* Good to go */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm9reg+ecx*4],eax         /* Write back */ \
    }

#define ARM9addrLSprrpar() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,31                      /* Set to 32 */ \
        __asm shft1:                          /* Otherwise.. */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
	__asm sar ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm9reg+ecx*4],eax         /* Write back */ \
    }

#define ARM9addrLSprrprr() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,[arm9reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm rcr ebx,1                       /* 33-bit shift */ \
        __asm jmp shftend                     /* And skip over this */ \
        __asm shft1:                          /* Otherwise.. */ \
	__asm ror ebx,cl                      /* Shift Rm */ \
	__asm shftend: \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm9reg+ecx*4],eax         /* Write back */ \
    }

#define ARM9addrLSptim() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm and ebx,00000FFFh               /* Mask offset */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub [arm9reg+ecx*4],ebx         /* Don't touch eax */ \
    }

#define ARM9addrLSptip() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm and ebx,00000FFFh               /* Mask offset */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add [arm9reg+ecx*4],ebx         /* Don't touch eax */ \
    }

#define ARM9addrLSlmptim() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm mov eax,edx                     /* yet another */ \
        __asm shr eax,4                       /* Shift top nybble */ \
        __asm and ebx,15                      /* Mask low nybble */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm and eax,240                     /* Mask high nyb */ \
        __asm or ebx,eax                      /* Add nybbles */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub [arm9reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM9addrLSlmptip() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm mov eax,edx                     /* yet another */ \
        __asm shr eax,4                       /* Shift top nybble */ \
        __asm and ebx,15                      /* Mask low nybble */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm and eax,240                     /* Mask high nyb */ \
        __asm or ebx,eax                      /* Add nybbles */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add [arm9reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM9addrLSptrm() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub [arm9reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM9addrLSptrmll() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
	__asm shl ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub [arm9reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM9addrLSptrmlr() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm shl ebx,1                       /* Set carry */ \
        __asm xor ebx,ebx                     /* Operand=0 */ \
        __asm jmp shftend                     /* and skip over */\
        __asm shft1:                          /* Otherwise.. */ \
	__asm shr ebx,cl                      /* Shift Rm */ \
	__asm shftend:                        /* Good to go */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub [arm9reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM9addrLSptrmar() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,31                      /* Set to 32 */ \
        __asm shft1:                          /* Otherwise.. */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
	__asm sar ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub [arm9reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM9addrLSptrmrr() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,[arm9reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm rcr ebx,1                       /* 33-bit shift */ \
        __asm jmp shftend                     /* And skip over this */ \
        __asm shft1:                          /* Otherwise.. */ \
	__asm ror ebx,cl                      /* Shift Rm */ \
	__asm shftend: \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub [arm9reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM9addrLSptrp() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add [arm9reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM9addrLSptrpll() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
	__asm shl ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add [arm9reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM9addrLSptrplr() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm shl ebx,1                       /* Set carry */ \
        __asm xor ebx,ebx                     /* Operand=0 */ \
        __asm jmp shftend                     /* and skip over */\
        __asm shft1:                          /* Otherwise.. */ \
	__asm shr ebx,cl                      /* Shift Rm */ \
	__asm shftend:                        /* Good to go */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add [arm9reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM9addrLSptrpar() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,31                      /* Set to 32 */ \
        __asm shft1:                          /* Otherwise.. */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
	__asm sar ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add [arm9reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM9addrLSptrprr() \
    __asm { \
        __asm mov edx,[arm9reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm9reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,[arm9reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm rcr ebx,1                       /* 33-bit shift */ \
        __asm jmp shftend                     /* And skip over this */ \
        __asm shft1:                          /* Otherwise.. */ \
	__asm ror ebx,cl                      /* Shift Rm */ \
	__asm shftend: \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm9reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add [arm9reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM9opLDR() \
    __asm { \
        __asm shr edx,12                /* Shift in Rd */ \
        __asm and edx,15                /* Mask to 15 */ \
        __asm cmp edx,15                /* Is Rd 15? */ \
        __asm jl ldr1                   /* If not, skip */ \
        __asm and eax,0FFFFFFFCh        /* Discount low 2 bits */ \
        __asm ldr1:                     /* Otherwise.. */ \
        __asm push edx                  /* Save Rd */ \
        __asm push eax                  /* Address */ \
        __asm xor ecx,ecx               /* Bus = 0 */ \
        __asm push ecx                  /* Push "bus number" */ \
        __asm call MMUMainrdW               /* Read a word */ \
        __asm pop ecx                   /* Clear stack */ \
        __asm pop ecx                   /* of params */ \
        __asm pop edx                   /* Get Rd back */\
	__asm mov [arm9reg+edx*4],eax   /* Write value to Rd */ \
	__asm mov eax,3 \
    }

#define ARM9opLDRH() \
    __asm { \
        __asm push edx                       /* Save opcode */ \
        __asm push eax                       /* Push address */ \
        __asm xor ecx,ecx                    /* Bus = 0 */ \
        __asm push ecx                       /* Push "bus number" */ \
        __asm call MMUMainrdH                    /* Read a hword */ \
        __asm pop ecx                        /* Pop param1 */ \
        __asm pop ecx                        /* Pop param2 */ \
        __asm pop edx                        /* Get opcode back */ \
        __asm shr edx,12                     /* Shift in Rd */ \
        __asm and edx,15                     /* Mask to 15 */ \
        __asm movzx ebx,ax                   /* Extend to 32b */ \
	__asm mov [arm9reg+edx*4],ebx        /* and write */ \
	__asm mov eax,3 \
    }

#define ARM9opLDRB() \
    __asm { \
        __asm push edx                       /* Save opcode */ \
        __asm push eax                       /* Push address */ \
        __asm xor ecx,ecx                    /* Bus = 0 */ \
        __asm push ecx                       /* Push "bus number" */ \
        __asm call MMUMainrdB                    /* Read a byte */ \
        __asm pop ecx                        /* Pop param1 */ \
        __asm pop ecx                        /* Pop param2 */ \
        __asm pop edx                        /* Get opcode back */ \
        __asm shr edx,12                     /* Shift in Rd */ \
        __asm and edx,15                     /* Mask to 15 */ \
        __asm movzx ebx,al                   /* Extend to 32b */ \
	__asm mov [arm9reg+edx*4],ebx        /* and write */ \
	__asm mov eax,3 \
    }

#define ARM9opLDRSH() \
    __asm { \
        __asm push edx                       /* Save opcode */ \
        __asm push eax                       /* Push address */ \
        __asm xor ecx,ecx                    /* Bus = 0 */ \
        __asm push ecx                       /* Push "bus number" */ \
        __asm call MMUMainrdH                    /* Read a byte */ \
        __asm pop ecx                        /* Pop param1 */ \
        __asm pop ecx                        /* Pop param2 */ \
        __asm pop edx                        /* Get opcode back */ \
        __asm shr edx,12                     /* Shift in Rd */ \
        __asm and edx,15                     /* Mask to 15 */ \
        __asm movsx ebx,ax                   /* Extend to 32b */ \
	__asm mov [arm9reg+edx*4],ebx        /* and write */ \
	__asm mov eax,3 \
    }

#define ARM9opLDRSB() \
    __asm { \
        __asm push edx                       /* Save opcode */ \
        __asm push eax                       /* Push address */ \
        __asm xor ecx,ecx                    /* Bus = 0 */ \
        __asm push ecx                       /* Push "bus number" */ \
        __asm call MMUMainrdB                    /* Read a byte */ \
        __asm pop ecx                        /* Pop param1 */ \
        __asm pop ecx                        /* Pop param2 */ \
        __asm pop edx                        /* Get opcode back */ \
        __asm shr edx,12                     /* Shift in Rd */ \
        __asm and edx,15                     /* Mask to 15 */ \
        __asm movsx ebx,al                   /* Extend to 32b */ \
	__asm mov [arm9reg+edx*4],ebx        /* and write */ \
	__asm mov eax,3 \
    }

#define ARM9opLDRT() \
    __asm { \
        __asm mov ecx,eax               /* Make copy of addr */ \
        __asm and ecx,3                 /* Get low 2 bits */ \
        __asm shl ecx,3                 /* Multiples of 8 */ \
        __asm ror eax,cl                /* Rotate addr */ \
        __asm shr edx,12                /* Shift in Rd */ \
        __asm and edx,15                /* Mask to 15 */ \
        __asm push edx                  /* Save Rd */ \
        __asm push eax                  /* Address */ \
        __asm xor ecx,ecx               /* Bus = 0 */ \
        __asm push ecx                  /* Push "bus number" */ \
        __asm call MMUMainrdW               /* Read a word */ \
        __asm pop ecx                   /* Clear stack */ \
        __asm pop ecx                   /* of params */ \
        __asm pop edx                   /* Get Rd back */\
        /*__asm mov ecx,[arm9reg+ARMREG_MODE*4] \
        __asm cmp ecx,ARMMD_SYS         // Set flags on mode  \
        __asm jle usrmode               // If USR or SYS, skip  \
	__asm mov [arm9reg+ARMREG_USER*4+edx*4],eax \
	__asm jmp modedone              // Read usr-reg, end  \
	__asm usrmode:                  // Skip here if USR  \
	__asm mov [arm9reg+edx*4],eax   // Read normal reg  \
	__asm modedone:                 // Done */ \
	__asm mov [arm9reg+edx*4],eax \
	__asm mov eax,3 \
    }

#define ARM9opLDRBT() \
    __asm { \
        __asm shr edx,12                /* Shift in Rd */ \
        __asm and edx,15                /* Mask to 15 */ \
        __asm push edx                  /* Save Rd */ \
        __asm push eax                  /* Address */ \
        __asm xor ecx,ecx               /* Bus = 0 */ \
        __asm push ecx                  /* Push "bus number" */ \
        __asm call MMUMainrdB               /* Read a word */ \
        __asm pop ecx                   /* Clear stack */ \
        __asm pop ecx                   /* of params */ \
        __asm pop edx                   /* Get Rd back */\
        __asm movzx ebx,al              /* Extend to 32b */ \
        __asm mov ecx,[arm9reg+ARMREG_MODE*4] \
        __asm cmp ecx,ARMMD_SYS         /* Set flags on mode */ \
        __asm jle usrmode               /* If USR or SYS, skip */ \
	__asm mov [arm9reg+ARMREG_USER*4+edx*4],ebx \
	__asm jmp modedone              /* Read usr-reg, end */ \
	__asm usrmode:                  /* Skip here if USR */ \
	__asm mov [arm9reg+edx*4],ebx   /* Read normal reg */ \
	__asm modedone:                 /* Done */ \
	__asm mov eax,3 \
    }

#define ARM9opSTR() \
    __asm { \
        __asm shr edx,12                  /* Shift in Rd */ \
        __asm and edx,15                  /* Mask to 15 */ \
        __asm mov ebx,[arm9reg+edx*4]     /* Get Rd value */ \
        __asm push ebx                    /* Push Rd */ \
        __asm push eax                    /* Push address */ \
        __asm xor ecx,ecx                 /* Bus = 0 */ \
        __asm push ecx                    /* Push "bus number" */ \
        __asm call MMUMainwrW                 /* Write thru MMUMain */ \
        __asm pop eax                     /* Clear stack */ \
        __asm pop eax                     /* of pushed */ \
        __asm pop eax                     /* parameters */ \
	__asm mov eax,2 \
    }

#define ARM9opSTRH() \
    __asm { \
        __asm shr edx,12                  /* Shift in Rd */ \
        __asm and edx,15                  /* Mask to 15 */ \
        __asm mov ebx,[arm9reg+edx*4]     /* Get Rd value */ \
        __asm and ebx,0000FFFFh           /* Mask low hword */ \
        __asm push ebx                    /* Push Rd */ \
        __asm push eax                    /* Push address */ \
        __asm xor ecx,ecx                 /* Bus = 0 */ \
        __asm push ecx                    /* Push "bus number" */ \
        __asm call MMUMainwrH                 /* Write thru MMUMain */ \
        __asm pop eax                     /* Clear stack */ \
        __asm pop eax                     /* of pushed */ \
        __asm pop eax                     /* parameters */ \
	__asm mov eax,2 \
    }

#define ARM9opSTRB() \
    __asm { \
        __asm shr edx,12                  /* Shift in Rd */ \
        __asm and edx,15                  /* Mask to 15 */ \
        __asm mov ebx,[arm9reg+edx*4]     /* Get Rd value */ \
        __asm and ebx,000000FFh           /* Mask low byte */ \
        __asm push ebx                    /* Push Rd */ \
        __asm push eax                    /* Push address */ \
        __asm xor ecx,ecx                 /* Bus = 0 */ \
        __asm push ecx                    /* Push "bus number" */ \
        __asm call MMUMainwrB                 /* Write thru MMUMain */ \
        __asm pop eax                     /* Clear stack */ \
        __asm pop eax                     /* of pushed */ \
        __asm pop eax                     /* parameters */ \
	__asm mov eax,2 \
    }

#define ARM9opSTRT() \
    __asm { \
        __asm shr edx,12                  /* Shift in Rd */ \
        __asm and edx,15                  /* Mask to 15 */ \
        __asm mov ecx,[arm9reg+ARMREG_MODE*4] \
        __asm cmp ecx,ARMMD_SYS         /* Set flags on mode */ \
        __asm jle usrmode               /* If USR or SYS, skip */ \
	__asm mov ebx,[arm9reg+ARMREG_USER*4+edx*4] \
	__asm jmp modedone                /* Read usr-reg, end */ \
	__asm usrmode:                    /* Skip here if USR */ \
	__asm mov ebx,[arm9reg+edx*4]     /* Read normal reg */ \
	__asm modedone:                   /* Done */ \
        __asm push ebx                    /* Push value */ \
        __asm push eax                    /* Push address */ \
        __asm xor ecx,ecx                 /* Bus = 0 */ \
        __asm push ecx                    /* Push "bus number" */ \
        __asm call MMUMainwrW                 /* Write thru MMUMain */ \
        __asm pop eax                     /* Clear stack */ \
        __asm pop eax                     /* of pushed */ \
        __asm pop eax                     /* parameters */ \
	__asm mov eax,2 \
    }

#define ARM9opSTRBT() \
    __asm { \
        __asm shr edx,12                  /* Shift in Rd */ \
        __asm and edx,15                  /* Mask to 15 */ \
        __asm mov ecx,[arm9reg+ARMREG_MODE*4] \
        __asm cmp ecx,ARMMD_SYS         /* Set flags on mode */ \
        __asm jle usrmode               /* If USR or SYS, skip */ \
	__asm mov ebx,[arm9reg+ARMREG_USER*4+edx*4] \
	__asm jmp modedone              /* Read usr-reg, end */ \
	__asm usrmode:                  /* Skip here if USR */ \
	__asm mov ebx,[arm9reg+edx*4]   /* Read normal reg */ \
	__asm modedone:                 /* Done */ \
        __asm and ebx,000000FFh           /* Mask low byte */ \
        __asm push ebx                    /* Push Rd */ \
        __asm push eax                    /* Push address */ \
        __asm xor ecx,ecx                 /* Bus = 0 */ \
        __asm push ecx                    /* Push "bus number" */ \
        __asm call MMUMainwrB                 /* Write thru MMUMain */ \
        __asm pop eax                     /* Clear stack */ \
        __asm pop eax                     /* of pushed */ \
        __asm pop eax                     /* parameters */ \
	__asm mov eax,2 \
    }

OPCODE ARM9opLDRptim() { ARM9addrLSptim(); ARM9opLDR(); }
OPCODE ARM9opLDRptip() { ARM9addrLSptip(); ARM9opLDR(); }
OPCODE ARM9opLDRptrm() { ARM9addrLSptrm(); ARM9opLDR(); }
OPCODE ARM9opLDRptrp() { ARM9addrLSptrp(); ARM9opLDR(); }
OPCODE ARM9opLDRprim() { ARM9addrLSprim(); ARM9opLDR(); }
OPCODE ARM9opLDRprip() { ARM9addrLSprip(); ARM9opLDR(); }
OPCODE ARM9opLDRprrm() { ARM9addrLSprrm(); ARM9opLDR(); }
OPCODE ARM9opLDRprrp() { ARM9addrLSprrp(); ARM9opLDR(); }
OPCODE ARM9opLDRofim() { ARM9addrLSofim(); ARM9opLDR(); }
OPCODE ARM9opLDRofip() { ARM9addrLSofip(); ARM9opLDR(); }
OPCODE ARM9opLDRofrm() { ARM9addrLSofrm(); ARM9opLDR(); }
OPCODE ARM9opLDRofrp() { ARM9addrLSofrp(); ARM9opLDR(); }

OPCODE ARM9opSTRptim() { ARM9addrLSptim(); ARM9opSTR(); }
OPCODE ARM9opSTRptip() { ARM9addrLSptip(); ARM9opSTR(); }
OPCODE ARM9opSTRptrm() { ARM9addrLSptrm(); ARM9opSTR(); }
OPCODE ARM9opSTRptrp() { ARM9addrLSptrp(); ARM9opSTR(); }
OPCODE ARM9opSTRprim() { ARM9addrLSprim(); ARM9opSTR(); }
OPCODE ARM9opSTRprip() { ARM9addrLSprip(); ARM9opSTR(); }
OPCODE ARM9opSTRprrm() { ARM9addrLSprrm(); ARM9opSTR(); }
OPCODE ARM9opSTRprrp() { ARM9addrLSprrp(); ARM9opSTR(); }
OPCODE ARM9opSTRofim() { ARM9addrLSofim(); ARM9opSTR(); }
OPCODE ARM9opSTRofip() { ARM9addrLSofip(); ARM9opSTR(); }
OPCODE ARM9opSTRofrm() { ARM9addrLSofrm(); ARM9opSTR(); }
OPCODE ARM9opSTRofrp() { ARM9addrLSofrp(); ARM9opSTR(); }

OPCODE ARM9opLDRBptim() { ARM9addrLSptim(); ARM9opLDRB(); }
OPCODE ARM9opLDRBptip() { ARM9addrLSptip(); ARM9opLDRB(); }
OPCODE ARM9opLDRBptrm() { ARM9addrLSptrm(); ARM9opLDRB(); }
OPCODE ARM9opLDRBptrp() { ARM9addrLSptrp(); ARM9opLDRB(); }
OPCODE ARM9opLDRBprim() { ARM9addrLSprim(); ARM9opLDRB(); }
OPCODE ARM9opLDRBprip() { ARM9addrLSprip(); ARM9opLDRB(); }
OPCODE ARM9opLDRBprrm() { ARM9addrLSprrm(); ARM9opLDRB(); }
OPCODE ARM9opLDRBprrp() { ARM9addrLSprrp(); ARM9opLDRB(); }
OPCODE ARM9opLDRBofim() { ARM9addrLSofim(); ARM9opLDRB(); }
OPCODE ARM9opLDRBofip() { ARM9addrLSofip(); ARM9opLDRB(); }
OPCODE ARM9opLDRBofrm() { ARM9addrLSofrm(); ARM9opLDRB(); }
OPCODE ARM9opLDRBofrp() { ARM9addrLSofrp(); ARM9opLDRB(); }

OPCODE ARM9opSTRBptim() { ARM9addrLSptim(); ARM9opSTRB(); }
OPCODE ARM9opSTRBptip() { ARM9addrLSptip(); ARM9opSTRB(); }
OPCODE ARM9opSTRBptrm() { ARM9addrLSptrm(); ARM9opSTRB(); }
OPCODE ARM9opSTRBptrp() { ARM9addrLSptrp(); ARM9opSTRB(); }
OPCODE ARM9opSTRBprim() { ARM9addrLSprim(); ARM9opSTRB(); }
OPCODE ARM9opSTRBprip() { ARM9addrLSprip(); ARM9opSTRB(); }
OPCODE ARM9opSTRBprrm() { ARM9addrLSprrm(); ARM9opSTRB(); }
OPCODE ARM9opSTRBprrp() { ARM9addrLSprrp(); ARM9opSTRB(); }
OPCODE ARM9opSTRBofim() { ARM9addrLSofim(); ARM9opSTRB(); }
OPCODE ARM9opSTRBofip() { ARM9addrLSofip(); ARM9opSTRB(); }
OPCODE ARM9opSTRBofrm() { ARM9addrLSofrm(); ARM9opSTRB(); }
OPCODE ARM9opSTRBofrp() { ARM9addrLSofrp(); ARM9opSTRB(); }

OPCODE ARM9opLDRHptim() { ARM9addrLSlmptim(); ARM9opLDRH(); }
OPCODE ARM9opLDRHptip() { ARM9addrLSlmptip(); ARM9opLDRH(); }
OPCODE ARM9opLDRHptrm() { ARM9addrLSptrm(); ARM9opLDRH(); }
OPCODE ARM9opLDRHptrp() { ARM9addrLSptrp(); ARM9opLDRH(); }
OPCODE ARM9opLDRHprim() { ARM9addrLSlmprim(); ARM9opLDRH(); }
OPCODE ARM9opLDRHprip() { ARM9addrLSlmprip(); ARM9opLDRH(); }
OPCODE ARM9opLDRHprrm() { ARM9addrLSprrm(); ARM9opLDRH(); }
OPCODE ARM9opLDRHprrp() { ARM9addrLSprrp(); ARM9opLDRH(); }
OPCODE ARM9opLDRHofim() { ARM9addrLSlmofim(); ARM9opLDRH(); }
OPCODE ARM9opLDRHofip() { ARM9addrLSlmofip(); ARM9opLDRH(); }
OPCODE ARM9opLDRHofrm() { ARM9addrLSofrm(); ARM9opLDRH(); }
OPCODE ARM9opLDRHofrp() { ARM9addrLSofrp(); ARM9opLDRH(); }

OPCODE ARM9opSTRHptim() { ARM9addrLSlmptim(); ARM9opSTRH(); }
OPCODE ARM9opSTRHptip() { ARM9addrLSlmptip(); ARM9opSTRH(); }
OPCODE ARM9opSTRHptrm() { ARM9addrLSptrm(); ARM9opSTRH(); }
OPCODE ARM9opSTRHptrp() { ARM9addrLSptrp(); ARM9opSTRH(); }
OPCODE ARM9opSTRHprim() { ARM9addrLSlmprim(); ARM9opSTRH(); }
OPCODE ARM9opSTRHprip() { ARM9addrLSlmprip(); ARM9opSTRH(); }
OPCODE ARM9opSTRHprrm() { ARM9addrLSprrm(); ARM9opSTRH(); }
OPCODE ARM9opSTRHprrp() { ARM9addrLSprrp(); ARM9opSTRH(); }
OPCODE ARM9opSTRHofim() { ARM9addrLSlmofim(); ARM9opSTRH(); }
OPCODE ARM9opSTRHofip() { ARM9addrLSlmofip(); ARM9opSTRH(); }
OPCODE ARM9opSTRHofrm() { ARM9addrLSofrm(); ARM9opSTRH(); }
OPCODE ARM9opSTRHofrp() { ARM9addrLSofrp(); ARM9opSTRH(); }

OPCODE ARM9opLDRTptim() { ARM9addrLSptim(); ARM9opLDRT(); }
OPCODE ARM9opLDRTptip() { ARM9addrLSptip(); ARM9opLDRT(); }
OPCODE ARM9opLDRTptrm() { ARM9addrLSptrm(); ARM9opLDRT(); }
OPCODE ARM9opLDRTptrp() { ARM9addrLSptrp(); ARM9opLDRT(); }
OPCODE ARM9opLDRTprim() { ARM9addrLSprim(); ARM9opLDRT(); }
OPCODE ARM9opLDRTprip() { ARM9addrLSprip(); ARM9opLDRT(); }
OPCODE ARM9opLDRTprrm() { ARM9addrLSprrm(); ARM9opLDRT(); }
OPCODE ARM9opLDRTprrp() { ARM9addrLSprrp(); ARM9opLDRT(); }
OPCODE ARM9opLDRTofim() { ARM9addrLSofim(); ARM9opLDRT(); }
OPCODE ARM9opLDRTofip() { ARM9addrLSofip(); ARM9opLDRT(); }
OPCODE ARM9opLDRTofrm() { ARM9addrLSofrm(); ARM9opLDRT(); }
OPCODE ARM9opLDRTofrp() { ARM9addrLSofrp(); ARM9opLDRT(); }

OPCODE ARM9opSTRTptim() { ARM9addrLSptim(); ARM9opSTRT(); }
OPCODE ARM9opSTRTptip() { ARM9addrLSptip(); ARM9opSTRT(); }
OPCODE ARM9opSTRTptrm() { ARM9addrLSptrm(); ARM9opSTRT(); }
OPCODE ARM9opSTRTptrp() { ARM9addrLSptrp(); ARM9opSTRT(); }
OPCODE ARM9opSTRTprim() { ARM9addrLSprim(); ARM9opSTRT(); }
OPCODE ARM9opSTRTprip() { ARM9addrLSprip(); ARM9opSTRT(); }
OPCODE ARM9opSTRTprrm() { ARM9addrLSprrm(); ARM9opSTRT(); }
OPCODE ARM9opSTRTprrp() { ARM9addrLSprrp(); ARM9opSTRT(); }
OPCODE ARM9opSTRTofim() { ARM9addrLSofim(); ARM9opSTRT(); }
OPCODE ARM9opSTRTofip() { ARM9addrLSofip(); ARM9opSTRT(); }
OPCODE ARM9opSTRTofrm() { ARM9addrLSofrm(); ARM9opSTRT(); }
OPCODE ARM9opSTRTofrp() { ARM9addrLSofrp(); ARM9opSTRT(); }

OPCODE ARM9opLDRBTptim() { ARM9addrLSptim(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTptip() { ARM9addrLSptip(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTptrm() { ARM9addrLSptrm(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTptrp() { ARM9addrLSptrp(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTprim() { ARM9addrLSprim(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTprip() { ARM9addrLSprip(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTprrm() { ARM9addrLSprrm(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTprrp() { ARM9addrLSprrp(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTofim() { ARM9addrLSofim(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTofip() { ARM9addrLSofip(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTofrm() { ARM9addrLSofrm(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTofrp() { ARM9addrLSofrp(); ARM9opLDRBT(); }

OPCODE ARM9opSTRBTptim() { ARM9addrLSptim(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTptip() { ARM9addrLSptip(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTptrm() { ARM9addrLSptrm(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTptrp() { ARM9addrLSptrp(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTprim() { ARM9addrLSprim(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTprip() { ARM9addrLSprip(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTprrm() { ARM9addrLSprrm(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTprrp() { ARM9addrLSprrp(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTofim() { ARM9addrLSofim(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTofip() { ARM9addrLSofip(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTofrm() { ARM9addrLSofrm(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTofrp() { ARM9addrLSofrp(); ARM9opSTRBT(); }

OPCODE ARM9opLDRSBptim() { /*u8 c;*/ ARM9addrLSlmptim(); ARM9opLDRSB(); }
OPCODE ARM9opLDRSBptip() { /*u8 c;*/ ARM9addrLSlmptip(); ARM9opLDRSB(); }
OPCODE ARM9opLDRSBptrm() { /*u8 c;*/ ARM9addrLSptrm(); ARM9opLDRSB(); }
OPCODE ARM9opLDRSBptrp() { /*u8 c;*/ ARM9addrLSptrp(); ARM9opLDRSB(); }
OPCODE ARM9opLDRSBprim() { /*u8 c;*/ ARM9addrLSlmprim(); ARM9opLDRSB(); }
OPCODE ARM9opLDRSBprip() { /*u8 c;*/ ARM9addrLSlmprip(); ARM9opLDRSB(); }
OPCODE ARM9opLDRSBprrm() { /*u8 c;*/ ARM9addrLSprrm(); ARM9opLDRSB(); }
OPCODE ARM9opLDRSBprrp() { /*u8 c;*/ ARM9addrLSprrp(); ARM9opLDRSB(); }
OPCODE ARM9opLDRSBofim() { /*u8 c;*/ ARM9addrLSlmofim(); ARM9opLDRSB(); }
OPCODE ARM9opLDRSBofip() { /*u8 c;*/ ARM9addrLSlmofip(); ARM9opLDRSB(); }
OPCODE ARM9opLDRSBofrm() { /*u8 c;*/ ARM9addrLSofrm(); ARM9opLDRSB(); }
OPCODE ARM9opLDRSBofrp() { /*u8 c;*/ ARM9addrLSofrp(); ARM9opLDRSB(); }

OPCODE ARM9opLDRSHptim() { /*u16 c;*/ ARM9addrLSlmptim(); ARM9opLDRSH(); }
OPCODE ARM9opLDRSHptip() { /*u16 c;*/ ARM9addrLSlmptip(); ARM9opLDRSH(); }
OPCODE ARM9opLDRSHptrm() { /*u16 c;*/ ARM9addrLSptrm(); ARM9opLDRSH(); }
OPCODE ARM9opLDRSHptrp() { /*u16 c;*/ ARM9addrLSptrp(); ARM9opLDRSH(); }
OPCODE ARM9opLDRSHprim() { /*u16 c;*/ ARM9addrLSlmprim(); ARM9opLDRSH(); }
OPCODE ARM9opLDRSHprip() { /*u16 c;*/ ARM9addrLSlmprip(); ARM9opLDRSH(); }
OPCODE ARM9opLDRSHprrm() { /*u16 c;*/ ARM9addrLSprrm(); ARM9opLDRSH(); }
OPCODE ARM9opLDRSHprrp() { /*u16 c;*/ ARM9addrLSprrp(); ARM9opLDRSH(); }
OPCODE ARM9opLDRSHofim() { /*u16 c;*/ ARM9addrLSlmofim(); ARM9opLDRSH(); }
OPCODE ARM9opLDRSHofip() { /*u16 c;*/ ARM9addrLSlmofip(); ARM9opLDRSH(); }
OPCODE ARM9opLDRSHofrm() { /*u16 c;*/ ARM9addrLSofrm(); ARM9opLDRSH(); }
OPCODE ARM9opLDRSHofrp() { /*u16 c;*/ ARM9addrLSofrp(); ARM9opLDRSH(); }

OPCODE ARM9opLDRptrmll() { ARM9addrLSptrmll(); ARM9opLDR(); }
OPCODE ARM9opLDRptrmlr() { ARM9addrLSptrmlr(); ARM9opLDR(); }
OPCODE ARM9opLDRptrmar() { ARM9addrLSptrmar(); ARM9opLDR(); }
OPCODE ARM9opLDRptrmrr() { ARM9addrLSptrmrr(); ARM9opLDR(); }
OPCODE ARM9opLDRptrpll() { ARM9addrLSptrpll(); ARM9opLDR(); }
OPCODE ARM9opLDRptrplr() { ARM9addrLSptrplr(); ARM9opLDR(); }
OPCODE ARM9opLDRptrpar() { ARM9addrLSptrpar(); ARM9opLDR(); }
OPCODE ARM9opLDRptrprr() { ARM9addrLSptrprr(); ARM9opLDR(); }
OPCODE ARM9opLDRprrmll() { ARM9addrLSprrmll(); ARM9opLDR(); }
OPCODE ARM9opLDRprrmlr() { ARM9addrLSprrmlr(); ARM9opLDR(); }
OPCODE ARM9opLDRprrmar() { ARM9addrLSprrmar(); ARM9opLDR(); }
OPCODE ARM9opLDRprrmrr() { ARM9addrLSprrmrr(); ARM9opLDR(); }
OPCODE ARM9opLDRprrpll() { ARM9addrLSprrpll(); ARM9opLDR(); }
OPCODE ARM9opLDRprrplr() { ARM9addrLSprrplr(); ARM9opLDR(); }
OPCODE ARM9opLDRprrpar() { ARM9addrLSprrpar(); ARM9opLDR(); }
OPCODE ARM9opLDRprrprr() { ARM9addrLSprrprr(); ARM9opLDR(); }
OPCODE ARM9opLDRofrmll() { ARM9addrLSofrmll(); ARM9opLDR(); }
OPCODE ARM9opLDRofrmlr() { ARM9addrLSofrmlr(); ARM9opLDR(); }
OPCODE ARM9opLDRofrmar() { ARM9addrLSofrmar(); ARM9opLDR(); }
OPCODE ARM9opLDRofrmrr() { ARM9addrLSofrmrr(); ARM9opLDR(); }
OPCODE ARM9opLDRofrpll() { ARM9addrLSofrpll(); ARM9opLDR(); }
OPCODE ARM9opLDRofrplr() { ARM9addrLSofrplr(); ARM9opLDR(); }
OPCODE ARM9opLDRofrpar() { ARM9addrLSofrpar(); ARM9opLDR(); }
OPCODE ARM9opLDRofrprr() { ARM9addrLSofrprr(); ARM9opLDR(); }

OPCODE ARM9opSTRptrmll() { ARM9addrLSptrmll(); ARM9opSTR(); }
OPCODE ARM9opSTRptrmlr() { ARM9addrLSptrmlr(); ARM9opSTR(); }
OPCODE ARM9opSTRptrmar() { ARM9addrLSptrmar(); ARM9opSTR(); }
OPCODE ARM9opSTRptrmrr() { ARM9addrLSptrmrr(); ARM9opSTR(); }
OPCODE ARM9opSTRptrpll() { ARM9addrLSptrpll(); ARM9opSTR(); }
OPCODE ARM9opSTRptrplr() { ARM9addrLSptrplr(); ARM9opSTR(); }
OPCODE ARM9opSTRptrpar() { ARM9addrLSptrpar(); ARM9opSTR(); }
OPCODE ARM9opSTRptrprr() { ARM9addrLSptrprr(); ARM9opSTR(); }
OPCODE ARM9opSTRprrmll() { ARM9addrLSprrmll(); ARM9opSTR(); }
OPCODE ARM9opSTRprrmlr() { ARM9addrLSprrmlr(); ARM9opSTR(); }
OPCODE ARM9opSTRprrmar() { ARM9addrLSprrmar(); ARM9opSTR(); }
OPCODE ARM9opSTRprrmrr() { ARM9addrLSprrmrr(); ARM9opSTR(); }
OPCODE ARM9opSTRprrpll() { ARM9addrLSprrpll(); ARM9opSTR(); }
OPCODE ARM9opSTRprrplr() { ARM9addrLSprrplr(); ARM9opSTR(); }
OPCODE ARM9opSTRprrpar() { ARM9addrLSprrpar(); ARM9opSTR(); }
OPCODE ARM9opSTRprrprr() { ARM9addrLSprrprr(); ARM9opSTR(); }
OPCODE ARM9opSTRofrmll() { ARM9addrLSofrmll(); ARM9opSTR(); }
OPCODE ARM9opSTRofrmlr() { ARM9addrLSofrmlr(); ARM9opSTR(); }
OPCODE ARM9opSTRofrmar() { ARM9addrLSofrmar(); ARM9opSTR(); }
OPCODE ARM9opSTRofrmrr() { ARM9addrLSofrmrr(); ARM9opSTR(); }
OPCODE ARM9opSTRofrpll() { ARM9addrLSofrpll(); ARM9opSTR(); }
OPCODE ARM9opSTRofrplr() { ARM9addrLSofrplr(); ARM9opSTR(); }
OPCODE ARM9opSTRofrpar() { ARM9addrLSofrpar(); ARM9opSTR(); }
OPCODE ARM9opSTRofrprr() { ARM9addrLSofrprr(); ARM9opSTR(); }

OPCODE ARM9opLDRBptrmll() { ARM9addrLSptrmll(); ARM9opLDRB(); }
OPCODE ARM9opLDRBptrmlr() { ARM9addrLSptrmlr(); ARM9opLDRB(); }
OPCODE ARM9opLDRBptrmar() { ARM9addrLSptrmar(); ARM9opLDRB(); }
OPCODE ARM9opLDRBptrmrr() { ARM9addrLSptrmrr(); ARM9opLDRB(); }
OPCODE ARM9opLDRBptrpll() { ARM9addrLSptrpll(); ARM9opLDRB(); }
OPCODE ARM9opLDRBptrplr() { ARM9addrLSptrplr(); ARM9opLDRB(); }
OPCODE ARM9opLDRBptrpar() { ARM9addrLSptrpar(); ARM9opLDRB(); }
OPCODE ARM9opLDRBptrprr() { ARM9addrLSptrprr(); ARM9opLDRB(); }
OPCODE ARM9opLDRBprrmll() { ARM9addrLSprrmll(); ARM9opLDRB(); }
OPCODE ARM9opLDRBprrmlr() { ARM9addrLSprrmlr(); ARM9opLDRB(); }
OPCODE ARM9opLDRBprrmar() { ARM9addrLSprrmar(); ARM9opLDRB(); }
OPCODE ARM9opLDRBprrmrr() { ARM9addrLSprrmrr(); ARM9opLDRB(); }
OPCODE ARM9opLDRBprrpll() { ARM9addrLSprrpll(); ARM9opLDRB(); }
OPCODE ARM9opLDRBprrplr() { ARM9addrLSprrplr(); ARM9opLDRB(); }
OPCODE ARM9opLDRBprrpar() { ARM9addrLSprrpar(); ARM9opLDRB(); }
OPCODE ARM9opLDRBprrprr() { ARM9addrLSprrprr(); ARM9opLDRB(); }
OPCODE ARM9opLDRBofrmll() { ARM9addrLSofrmll(); ARM9opLDRB(); }
OPCODE ARM9opLDRBofrmlr() { ARM9addrLSofrmlr(); ARM9opLDRB(); }
OPCODE ARM9opLDRBofrmar() { ARM9addrLSofrmar(); ARM9opLDRB(); }
OPCODE ARM9opLDRBofrmrr() { ARM9addrLSofrmrr(); ARM9opLDRB(); }
OPCODE ARM9opLDRBofrpll() { ARM9addrLSofrpll(); ARM9opLDRB(); }
OPCODE ARM9opLDRBofrplr() { ARM9addrLSofrplr(); ARM9opLDRB(); }
OPCODE ARM9opLDRBofrpar() { ARM9addrLSofrpar(); ARM9opLDRB(); }
OPCODE ARM9opLDRBofrprr() { ARM9addrLSofrprr(); ARM9opLDRB(); }

OPCODE ARM9opSTRBptrmll() { ARM9addrLSptrmll(); ARM9opSTRB(); }
OPCODE ARM9opSTRBptrmlr() { ARM9addrLSptrmlr(); ARM9opSTRB(); }
OPCODE ARM9opSTRBptrmar() { ARM9addrLSptrmar(); ARM9opSTRB(); }
OPCODE ARM9opSTRBptrmrr() { ARM9addrLSptrmrr(); ARM9opSTRB(); }
OPCODE ARM9opSTRBptrpll() { ARM9addrLSptrpll(); ARM9opSTRB(); }
OPCODE ARM9opSTRBptrplr() { ARM9addrLSptrplr(); ARM9opSTRB(); }
OPCODE ARM9opSTRBptrpar() { ARM9addrLSptrpar(); ARM9opSTRB(); }
OPCODE ARM9opSTRBptrprr() { ARM9addrLSptrprr(); ARM9opSTRB(); }
OPCODE ARM9opSTRBprrmll() { ARM9addrLSprrmll(); ARM9opSTRB(); }
OPCODE ARM9opSTRBprrmlr() { ARM9addrLSprrmlr(); ARM9opSTRB(); }
OPCODE ARM9opSTRBprrmar() { ARM9addrLSprrmar(); ARM9opSTRB(); }
OPCODE ARM9opSTRBprrmrr() { ARM9addrLSprrmrr(); ARM9opSTRB(); }
OPCODE ARM9opSTRBprrpll() { ARM9addrLSprrpll(); ARM9opSTRB(); }
OPCODE ARM9opSTRBprrplr() { ARM9addrLSprrplr(); ARM9opSTRB(); }
OPCODE ARM9opSTRBprrpar() { ARM9addrLSprrpar(); ARM9opSTRB(); }
OPCODE ARM9opSTRBprrprr() { ARM9addrLSprrprr(); ARM9opSTRB(); }
OPCODE ARM9opSTRBofrmll() { ARM9addrLSofrmll(); ARM9opSTRB(); }
OPCODE ARM9opSTRBofrmlr() { ARM9addrLSofrmlr(); ARM9opSTRB(); }
OPCODE ARM9opSTRBofrmar() { ARM9addrLSofrmar(); ARM9opSTRB(); }
OPCODE ARM9opSTRBofrmrr() { ARM9addrLSofrmrr(); ARM9opSTRB(); }
OPCODE ARM9opSTRBofrpll() { ARM9addrLSofrpll(); ARM9opSTRB(); }
OPCODE ARM9opSTRBofrplr() { ARM9addrLSofrplr(); ARM9opSTRB(); }
OPCODE ARM9opSTRBofrpar() { ARM9addrLSofrpar(); ARM9opSTRB(); }
OPCODE ARM9opSTRBofrprr() { ARM9addrLSofrprr(); ARM9opSTRB(); }

OPCODE ARM9opLDRTptrmll() { ARM9addrLSptrmll(); ARM9opLDRT(); }
OPCODE ARM9opLDRTptrmlr() { ARM9addrLSptrmlr(); ARM9opLDRT(); }
OPCODE ARM9opLDRTptrmar() { ARM9addrLSptrmar(); ARM9opLDRT(); }
OPCODE ARM9opLDRTptrmrr() { ARM9addrLSptrmrr(); ARM9opLDRT(); }
OPCODE ARM9opLDRTptrpll() { ARM9addrLSptrpll(); ARM9opLDRT(); }
OPCODE ARM9opLDRTptrplr() { ARM9addrLSptrplr(); ARM9opLDRT(); }
OPCODE ARM9opLDRTptrpar() { ARM9addrLSptrpar(); ARM9opLDRT(); }
OPCODE ARM9opLDRTptrprr() { ARM9addrLSptrprr(); ARM9opLDRT(); }
OPCODE ARM9opLDRTprrmll() { ARM9addrLSprrmll(); ARM9opLDRT(); }
OPCODE ARM9opLDRTprrmlr() { ARM9addrLSprrmlr(); ARM9opLDRT(); }
OPCODE ARM9opLDRTprrmar() { ARM9addrLSprrmar(); ARM9opLDRT(); }
OPCODE ARM9opLDRTprrmrr() { ARM9addrLSprrmrr(); ARM9opLDRT(); }
OPCODE ARM9opLDRTprrpll() { ARM9addrLSprrpll(); ARM9opLDRT(); }
OPCODE ARM9opLDRTprrplr() { ARM9addrLSprrplr(); ARM9opLDRT(); }
OPCODE ARM9opLDRTprrpar() { ARM9addrLSprrpar(); ARM9opLDRT(); }
OPCODE ARM9opLDRTprrprr() { ARM9addrLSprrprr(); ARM9opLDRT(); }
OPCODE ARM9opLDRTofrmll() { ARM9addrLSofrmll(); ARM9opLDRT(); }
OPCODE ARM9opLDRTofrmlr() { ARM9addrLSofrmlr(); ARM9opLDRT(); }
OPCODE ARM9opLDRTofrmar() { ARM9addrLSofrmar(); ARM9opLDRT(); }
OPCODE ARM9opLDRTofrmrr() { ARM9addrLSofrmrr(); ARM9opLDRT(); }
OPCODE ARM9opLDRTofrpll() { ARM9addrLSofrpll(); ARM9opLDRT(); }
OPCODE ARM9opLDRTofrplr() { ARM9addrLSofrplr(); ARM9opLDRT(); }
OPCODE ARM9opLDRTofrpar() { ARM9addrLSofrpar(); ARM9opLDRT(); }
OPCODE ARM9opLDRTofrprr() { ARM9addrLSofrprr(); ARM9opLDRT(); }

OPCODE ARM9opSTRTptrmll() { ARM9addrLSptrmll(); ARM9opSTRT(); }
OPCODE ARM9opSTRTptrmlr() { ARM9addrLSptrmlr(); ARM9opSTRT(); }
OPCODE ARM9opSTRTptrmar() { ARM9addrLSptrmar(); ARM9opSTRT(); }
OPCODE ARM9opSTRTptrmrr() { ARM9addrLSptrmrr(); ARM9opSTRT(); }
OPCODE ARM9opSTRTptrpll() { ARM9addrLSptrpll(); ARM9opSTRT(); }
OPCODE ARM9opSTRTptrplr() { ARM9addrLSptrplr(); ARM9opSTRT(); }
OPCODE ARM9opSTRTptrpar() { ARM9addrLSptrpar(); ARM9opSTRT(); }
OPCODE ARM9opSTRTptrprr() { ARM9addrLSptrprr(); ARM9opSTRT(); }
OPCODE ARM9opSTRTprrmll() { ARM9addrLSprrmll(); ARM9opSTRT(); }
OPCODE ARM9opSTRTprrmlr() { ARM9addrLSprrmlr(); ARM9opSTRT(); }
OPCODE ARM9opSTRTprrmar() { ARM9addrLSprrmar(); ARM9opSTRT(); }
OPCODE ARM9opSTRTprrmrr() { ARM9addrLSprrmrr(); ARM9opSTRT(); }
OPCODE ARM9opSTRTprrpll() { ARM9addrLSprrpll(); ARM9opSTRT(); }
OPCODE ARM9opSTRTprrplr() { ARM9addrLSprrplr(); ARM9opSTRT(); }
OPCODE ARM9opSTRTprrpar() { ARM9addrLSprrpar(); ARM9opSTRT(); }
OPCODE ARM9opSTRTprrprr() { ARM9addrLSprrprr(); ARM9opSTRT(); }
OPCODE ARM9opSTRTofrmll() { ARM9addrLSofrmll(); ARM9opSTRT(); }
OPCODE ARM9opSTRTofrmlr() { ARM9addrLSofrmlr(); ARM9opSTRT(); }
OPCODE ARM9opSTRTofrmar() { ARM9addrLSofrmar(); ARM9opSTRT(); }
OPCODE ARM9opSTRTofrmrr() { ARM9addrLSofrmrr(); ARM9opSTRT(); }
OPCODE ARM9opSTRTofrpll() { ARM9addrLSofrpll(); ARM9opSTRT(); }
OPCODE ARM9opSTRTofrplr() { ARM9addrLSofrplr(); ARM9opSTRT(); }
OPCODE ARM9opSTRTofrpar() { ARM9addrLSofrpar(); ARM9opSTRT(); }
OPCODE ARM9opSTRTofrprr() { ARM9addrLSofrprr(); ARM9opSTRT(); }

OPCODE ARM9opLDRBTptrmll() { ARM9addrLSptrmll(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTptrmlr() { ARM9addrLSptrmlr(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTptrmar() { ARM9addrLSptrmar(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTptrmrr() { ARM9addrLSptrmrr(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTptrpll() { ARM9addrLSptrpll(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTptrplr() { ARM9addrLSptrplr(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTptrpar() { ARM9addrLSptrpar(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTptrprr() { ARM9addrLSptrprr(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTprrmll() { ARM9addrLSprrmll(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTprrmlr() { ARM9addrLSprrmlr(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTprrmar() { ARM9addrLSprrmar(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTprrmrr() { ARM9addrLSprrmrr(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTprrpll() { ARM9addrLSprrpll(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTprrplr() { ARM9addrLSprrplr(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTprrpar() { ARM9addrLSprrpar(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTprrprr() { ARM9addrLSprrprr(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTofrmll() { ARM9addrLSofrmll(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTofrmlr() { ARM9addrLSofrmlr(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTofrmar() { ARM9addrLSofrmar(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTofrmrr() { ARM9addrLSofrmrr(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTofrpll() { ARM9addrLSofrpll(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTofrplr() { ARM9addrLSofrplr(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTofrpar() { ARM9addrLSofrpar(); ARM9opLDRBT(); }
OPCODE ARM9opLDRBTofrprr() { ARM9addrLSofrprr(); ARM9opLDRBT(); }

OPCODE ARM9opSTRBTptrmll() { ARM9addrLSptrmll(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTptrmlr() { ARM9addrLSptrmlr(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTptrmar() { ARM9addrLSptrmar(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTptrmrr() { ARM9addrLSptrmrr(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTptrpll() { ARM9addrLSptrpll(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTptrplr() { ARM9addrLSptrplr(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTptrpar() { ARM9addrLSptrpar(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTptrprr() { ARM9addrLSptrprr(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTprrmll() { ARM9addrLSprrmll(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTprrmlr() { ARM9addrLSprrmlr(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTprrmar() { ARM9addrLSprrmar(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTprrmrr() { ARM9addrLSprrmrr(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTprrpll() { ARM9addrLSprrpll(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTprrplr() { ARM9addrLSprrplr(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTprrpar() { ARM9addrLSprrpar(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTprrprr() { ARM9addrLSprrprr(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTofrmll() { ARM9addrLSofrmll(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTofrmlr() { ARM9addrLSofrmlr(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTofrmar() { ARM9addrLSofrmar(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTofrmrr() { ARM9addrLSofrmrr(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTofrpll() { ARM9addrLSofrpll(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTofrplr() { ARM9addrLSofrplr(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTofrpar() { ARM9addrLSofrpar(); ARM9opSTRBT(); }
OPCODE ARM9opSTRBTofrprr() { ARM9addrLSofrprr(); ARM9opSTRBT(); }

//-------------------------------------------------------------------------

OPCODE ARM9opMUL()
{
    __asm {
        mov ebx,[arm9reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm9reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm9reg+eax*4]          /* Read Rs' value */
	mul ecx                          /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	mov [arm9reg+ecx*4],eax          /* Write low word */
    }
    return 3;
}

OPCODE ARM9opMLA()
{
    __asm {
        mov ebx,[arm9reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm9reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm9reg+eax*4]          /* Read Rs' value */
	mul ecx                          /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	add eax,[arm9reg+ebx*4]          /* Add Rd to result */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	mov [arm9reg+ecx*4],eax          /* Add low word */
    }
    return 4;
}

OPCODE ARM9opMULS()
{
    __asm {
        mov ebx,[arm9reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm9reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm9reg+eax*4]          /* Read Rs' value */
	mul ecx                          /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	mov [arm9reg+ecx*4],eax          /* Write low word */
	or eax,eax                       /* Check word=0 */
	setz [arm9reg+ARMREG_FZ*4]       /* And set ARM Z flag */
	and eax,80000000h                /* Check MSB of word */
	sets [arm9reg+ARMREG_FN*4]       /* set ARM N flag */
    }
    return 3;
}

OPCODE ARM9opMLAS()
{
    __asm {
        mov ebx,[arm9reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm9reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm9reg+eax*4]          /* Read Rs' value */
	mul ecx                          /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	add eax,[arm9reg+ebx*4]          /* Add Rd to result */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	mov [arm9reg+ecx*4],eax          /* Add low word */
	or eax,eax                       /* Check word=0 */
	setz [arm9reg+ARMREG_FZ*4]       /* And set ARM Z flag */
	and eax,80000000h                /* Check MSB of word */
	sets [arm9reg+ARMREG_FN*4]       /* set ARM N flag */
    }
    return 4;
}

OPCODE ARM9opUMULL()
{
    __asm {
        mov ebx,[arm9reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm9reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm9reg+eax*4]          /* Read Rs' value */
	mul ecx                          /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	mov [arm9reg+ebx*4],eax          /* Write high word */
	mov [arm9reg+ecx*4],edx          /* Write low word */
    }
    return 4;
}

OPCODE ARM9opUMLAL()
{
    __asm {
        mov ebx,[arm9reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm9reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm9reg+eax*4]          /* Read Rs' value */
	mul ecx                          /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	add [arm9reg+ebx*4],eax          /* Add high word */
	adc [arm9reg+ecx*4],edx          /* Add low word */
    }
    return 5;
}

OPCODE ARM9opSMULL()
{
    __asm {
        mov ebx,[arm9reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm9reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm9reg+eax*4]          /* Read Rs' value */
	imul ecx                         /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	mov [arm9reg+ebx*4],eax          /* Write high word */
	mov [arm9reg+ecx*4],edx          /* Write low word */
    }
    return 4;
}

OPCODE ARM9opSMLAL()
{
    __asm {
        mov ebx,[arm9reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm9reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm9reg+eax*4]          /* Read Rs' value */
	imul ecx                         /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	add [arm9reg+ebx*4],eax          /* Add high word */
	adc [arm9reg+ecx*4],edx          /* Add low word */
    }
    return 5;
}

OPCODE ARM9opUMULLS()
{
    __asm {
        mov ebx,[arm9reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm9reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm9reg+eax*4]          /* Read Rs' value */
	mul ecx                          /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	mov [arm9reg+ebx*4],eax          /* Write high word */
	mov [arm9reg+ecx*4],edx          /* Write low word */
	or eax,edx                       /* Check both words are 0 */
	setz [arm9reg+ARMREG_FZ*4]       /* And set ARM Z flag */
	and edx,80000000h                /* Check MSB of hiword */
	sets [arm9reg+ARMREG_FN*4]       /* set ARM N flag */
    }
    return 4;
}

OPCODE ARM9opUMLALS()
{
    __asm {
        mov ebx,[arm9reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm9reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm9reg+eax*4]          /* Read Rs' value */
	mul ecx                          /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	add eax,[arm9reg+ebx*4]          /* Add high word */
	adc edx,[arm9reg+ecx*4]          /* Add low word */
	mov [arm9reg+ebx*4],eax
	mov [arm9reg+ecx*4],edx
	or eax,edx                       /* Check both words are 0 */
	setz [arm9reg+ARMREG_FZ*4]       /* And set ARM Z flag */
	and edx,80000000h                /* Check MSB of hiword */
	sets [arm9reg+ARMREG_FN*4]       /* set ARM N flag */
    }
    return 5;
}

OPCODE ARM9opSMULLS()
{
    __asm {
        mov ebx,[arm9reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm9reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm9reg+eax*4]          /* Read Rs' value */
	imul ecx                         /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	mov [arm9reg+ebx*4],eax          /* Write high word */
	mov [arm9reg+ecx*4],edx          /* Write low word */
	or eax,edx                       /* Check both words are 0 */
	setz [arm9reg+ARMREG_FZ*4]       /* And set ARM Z flag */
	and edx,80000000h                /* Check MSB of hiword */
	sets [arm9reg+ARMREG_FN*4]       /* set ARM N flag */
    }
    return 4;
}

OPCODE ARM9opSMLALS()
{
    __asm {
        mov ebx,[arm9reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm9reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm9reg+eax*4]          /* Read Rs' value */
	imul ecx                         /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	add eax,[arm9reg+ebx*4]          /* Add high word */
	adc edx,[arm9reg+ecx*4]          /* Add low word */
	mov [arm9reg+ebx*4],eax
	mov [arm9reg+ecx*4],edx
	or eax,edx                       /* Check both words are 0 */
	setz [arm9reg+ARMREG_FZ*4]       /* And set ARM Z flag */
	and edx,80000000h                /* Check MSB of hiword */
	sets [arm9reg+ARMREG_FN*4]       /* set ARM N flag */
    }
    return 5;
}

//-------------------------------------------------------------------------

int ARM9SideSum(u16 val)
{
    int res=val;
    res=((res>>1)&0x5555)+(res&0x5555);
    res=((res>>2)&0x3333)+(res&0x3333);
    res=((res>>4)&0x0F0F)+(res&0x0F0F);
    res=((res>>8)&0x00FF)+(res&0x00FF);
    return res;
}

#define ARM9addrLMIA() /*int a;*/ arm9reg.tmp1=arm9reg.r[ARM9OP_RN]
#define ARM9addrLMIB() /*int a;*/ arm9reg.tmp1=arm9reg.r[ARM9OP_RN]+4
#define ARM9addrLMDA() /*int a;*/ arm9reg.tmp1=arm9reg.r[ARM9OP_RN]-(ARM9SideSum(arm9reg.curop&65535)*4)+4
#define ARM9addrLMDB() /*int a;*/ arm9reg.tmp1=arm9reg.r[ARM9OP_RN]-(ARM9SideSum(arm9reg.curop&65535)*4)

#define ARM9addrLMwi() arm9reg.r[ARM9OP_RN]+=ARM9SideSum(arm9reg.curop&65535)*4
#define ARM9addrLMwd() arm9reg.r[ARM9OP_RN]-=ARM9SideSum(arm9reg.curop&65535)*4

#define ARM9opLDM() \
    for(a=0;a<=14;a++) \
    { \
        if(arm9reg.curop&(1<<a)) \
        { \
            arm9reg.r[a]=MMUMainrdW(0,arm9reg.tmp1); \
            arm9reg.tmp1+=4; \
        } \
    } \
    if(arm9reg.curop&0x00008000) \
    { \
        arm9reg.r[15]=MMUMainrdW(0,arm9reg.tmp1)&0xFFFFFFFC; \
        arm9reg.tmp1+=4; \
    } \
    return 3

#define ARM9opSTM() \
    for(a=0;a<=15;a++) \
    { \
        if(arm9reg.curop&(1<<a)) \
        { \
		MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[a]); \
            arm9reg.tmp1+=4; \
        } \
    } \
    return 2

#define ARM9opLDMu() \
    if(arm9reg.curop&0x0001) { arm9reg.r[0]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    if(arm9reg.curop&0x0002) { arm9reg.r[1]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    if(arm9reg.curop&0x0004) { arm9reg.r[2]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    if(arm9reg.curop&0x0008) { arm9reg.r[3]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    if(arm9reg.curop&0x0010) { arm9reg.r[4]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    if(arm9reg.curop&0x0020) { arm9reg.r[5]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    if(arm9reg.curop&0x0040) { arm9reg.r[6]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    if(arm9reg.curop&0x0080) { arm9reg.r[7]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    switch(arm9reg.curmode) \
    { \
    	case ARMMD_USR: case ARMMD_SYS: \
    	    if(arm9reg.curop&0x0100) { arm9reg.r[8]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4;  } \
    	    if(arm9reg.curop&0x0200) { arm9reg.r[9]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4;  } \
    	    if(arm9reg.curop&0x0400) { arm9reg.r[10]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x0800) { arm9reg.r[11]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x1000) { arm9reg.r[12]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x2000) { arm9reg.r[13]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x4000) { arm9reg.r[14]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    	    break;\
    	case ARMMD_IRQ: case ARMMD_ABT: case ARMMD_SVC: case ARMMD_UND: \
    	    if(arm9reg.curop&0x0100) { arm9reg.r[8]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4;  } \
    	    if(arm9reg.curop&0x0200) { arm9reg.r[9]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4;  } \
    	    if(arm9reg.curop&0x0400) { arm9reg.r[10]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x0800) { arm9reg.r[11]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x1000) { arm9reg.r[12]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x2000) { arm9reg.r[13]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; }   \
    	    if(arm9reg.curop&0x4000) { arm9reg.r[14]=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; }   \
            break; \
    	case ARMMD_FIQ: \
    	    if(arm9reg.curop&0x0100) { arm9reg.r8=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4;  } \
    	    if(arm9reg.curop&0x0200) { arm9reg.r9=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4;  } \
    	    if(arm9reg.curop&0x0400) { arm9reg.r10=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x0800) { arm9reg.r11=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x1000) { arm9reg.r12=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x2000) { arm9reg.r13=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x4000) { arm9reg.r14=MMUMainrdW(0,arm9reg.tmp1); arm9reg.tmp1+=4; } \
            break; \
    } \
    if(arm9reg.curop&0x8000) \
    { \
    	arm9reg.r[15]=MMUMainrdW(0,arm9reg.tmp1)&0xFFFFFFFC; arm9reg.tmp1+=4; \
		arm9reg.cpsr=arm9reg.spsr[arm9reg.curmode]; ARM9splitCPSR();\
    } \
    return 3

#define ARM9opSTMu() \
    if(arm9reg.curop&0x0001) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[0]); arm9reg.tmp1+=4; } \
    if(arm9reg.curop&0x0002) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[1]); arm9reg.tmp1+=4; } \
    if(arm9reg.curop&0x0004) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[2]); arm9reg.tmp1+=4; } \
    if(arm9reg.curop&0x0008) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[3]); arm9reg.tmp1+=4; } \
    if(arm9reg.curop&0x0010) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[4]); arm9reg.tmp1+=4; } \
    if(arm9reg.curop&0x0020) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[5]); arm9reg.tmp1+=4; } \
    if(arm9reg.curop&0x0040) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[6]); arm9reg.tmp1+=4; } \
    if(arm9reg.curop&0x0080) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[7]); arm9reg.tmp1+=4; } \
    switch(arm9reg.curmode) \
    { \
    	case ARMMD_USR: case ARMMD_SYS: \
    	    if(arm9reg.curop&0x0100) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[8]); arm9reg.tmp1+=4;  } \
    	    if(arm9reg.curop&0x0200) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[9]); arm9reg.tmp1+=4;  } \
    	    if(arm9reg.curop&0x0400) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[10]); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x0800) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[11]); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x1000) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[12]); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x2000) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[13]); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x4000) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[14]); arm9reg.tmp1+=4; } \
    	    break; \
    	case ARMMD_IRQ: case ARMMD_ABT: case ARMMD_SVC: case ARMMD_UND: \
    	    if(arm9reg.curop&0x0100) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[8]); arm9reg.tmp1+=4;  } \
    	    if(arm9reg.curop&0x0200) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[9]); arm9reg.tmp1+=4;  } \
    	    if(arm9reg.curop&0x0400) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[10]); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x0800) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[11]); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x1000) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[12]); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x2000) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[13]); arm9reg.tmp1+=4; }   \
    	    if(arm9reg.curop&0x4000) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[14]); arm9reg.tmp1+=4; }   \
            break; \
    	case ARMMD_FIQ: \
    	    if(arm9reg.curop&0x0100) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r8); arm9reg.tmp1+=4;  } \
    	    if(arm9reg.curop&0x0200) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r9); arm9reg.tmp1+=4;  } \
    	    if(arm9reg.curop&0x0400) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r10); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x0800) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r11); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x1000) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r12); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x2000) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r13); arm9reg.tmp1+=4; } \
    	    if(arm9reg.curop&0x4000) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r14); arm9reg.tmp1+=4; } \
            break; \
    } \
    if(arm9reg.curop&0x8000) { MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[15]); arm9reg.tmp1+=4; } \
    return 2

OPCODE ARM9opLDMIA() 
{ 
#if 0
	ARM9addrLMIA(); ARM9opLDM(); 
#endif
	int a; 
	arm9reg.tmp1=arm9reg.r[ARM9OP_RN];
    for(a=0;a<=14;a++) 
    { 
        if(arm9reg.curop&(1<<a)) 
        { 
            arm9reg.r[a]=MMUMainrdW(0,arm9reg.tmp1); 
            arm9reg.tmp1+=4; 
        } 
    } 
    if(arm9reg.curop&0x00008000) 
    { 
        arm9reg.r[15]=MMUMainrdW(0,arm9reg.tmp1)&0xFFFFFFFC; 
        arm9reg.tmp1+=4; 
    } 
    return 3;
}

OPCODE ARM9opLDMIB() { int a; ARM9addrLMIB(); ARM9opLDM(); }
OPCODE ARM9opLDMDA() { int a; ARM9addrLMDA(); ARM9opLDM(); }
OPCODE ARM9opLDMDB() { int a; ARM9addrLMDB(); ARM9opLDM(); }

OPCODE ARM9opLDMIAw() { int a; ARM9addrLMIA(); ARM9addrLMwi(); ARM9opLDM(); }
OPCODE ARM9opLDMIBw() { int a; ARM9addrLMIB(); ARM9addrLMwi(); ARM9opLDM(); }
OPCODE ARM9opLDMDAw() { int a; ARM9addrLMDA(); ARM9addrLMwd(); ARM9opLDM(); }
OPCODE ARM9opLDMDBw() { int a; ARM9addrLMDB(); ARM9addrLMwd(); ARM9opLDM(); }

OPCODE ARM9opLDMIAu() 
{ 
#if 0
	ARM9addrLMIA(); ARM9opLDMu(); 
#endif
//	int a; 
	arm9reg.tmp1=arm9reg.r[ARM9OP_RN];
    if(arm9reg.curop&0x0001) 
	{ 
		arm9reg.r[0]=MMUMainrdW(0,arm9reg.tmp1); 
		arm9reg.tmp1+=4; 
	} 
    if(arm9reg.curop&0x0002) 
	{ 
		arm9reg.r[1]=MMUMainrdW(0,arm9reg.tmp1); 
		arm9reg.tmp1+=4; 
	} 
    if(arm9reg.curop&0x0004) 
	{ 
		arm9reg.r[2]=MMUMainrdW(0,arm9reg.tmp1); 
		arm9reg.tmp1+=4; 
	} 
    if(arm9reg.curop&0x0008) 
	{ 
		arm9reg.r[3]=MMUMainrdW(0,arm9reg.tmp1); 
		arm9reg.tmp1+=4; 
	} 
    if(arm9reg.curop&0x0010) 
	{ 
		arm9reg.r[4]=MMUMainrdW(0,arm9reg.tmp1); 
		arm9reg.tmp1+=4; 
	} 
    if(arm9reg.curop&0x0020) 
	{ 
		arm9reg.r[5]=MMUMainrdW(0,arm9reg.tmp1); 
		arm9reg.tmp1+=4; 
	} 
    if(arm9reg.curop&0x0040) 
	{ 
		arm9reg.r[6]=MMUMainrdW(0,arm9reg.tmp1); 
		arm9reg.tmp1+=4; 
	} 
    if(arm9reg.curop&0x0080) 
	{ 
		arm9reg.r[7]=MMUMainrdW(0,arm9reg.tmp1); 
		arm9reg.tmp1+=4; 
	} 
    switch(arm9reg.curmode) 
    { 
    	case ARMMD_USR: case ARMMD_SYS: 
    	    if(arm9reg.curop&0x0100) 
			{ 
				arm9reg.r[8]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4;  
			} 
    	    if(arm9reg.curop&0x0200) 
			{ 
				arm9reg.r[9]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4;  
			} 
    	    if(arm9reg.curop&0x0400) 
			{ 
				arm9reg.r[10]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x0800) 
			{ 
				arm9reg.r[11]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x1000) 
			{ 
				arm9reg.r[12]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x2000) 
			{ 
				arm9reg.r[13]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x4000) 
			{ 
				arm9reg.r[14]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    break;
    	case ARMMD_IRQ: case ARMMD_ABT: case ARMMD_SVC: case ARMMD_UND: 
    	    if(arm9reg.curop&0x0100) 
			{ 
				arm9reg.r[8]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4;  
			} 
    	    if(arm9reg.curop&0x0200) 
			{ 
				arm9reg.r[9]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4;  
			} 
    	    if(arm9reg.curop&0x0400) 
			{ 
				arm9reg.r[10]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x0800) 
			{ 
				arm9reg.r[11]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x1000) 
			{ 
				arm9reg.r[12]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x2000) 
			{ 
//				arm9reg.r13=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.r[13]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			}   
    	    if(arm9reg.curop&0x4000) 
			{ 
//				arm9reg.r14=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.r[14]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			}   
            break; 
    	case ARMMD_FIQ: 
    	    if(arm9reg.curop&0x0100) 
			{ 
				arm9reg.r8=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4;  
			} 
    	    if(arm9reg.curop&0x0200) 
			{ 
				arm9reg.r9=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4;  
			} 
    	    if(arm9reg.curop&0x0400) 
			{ 
				arm9reg.r10=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x0800) 
			{ 
				arm9reg.r11=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x1000) 
			{ 
				arm9reg.r12=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x2000) 
			{ 
				arm9reg.r13=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x4000) 
			{ 
				arm9reg.r14=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
            break; 
    } 
    if(arm9reg.curop&0x8000) 
    { 
    	arm9reg.r[15]=MMUMainrdW(0,arm9reg.tmp1)&0xFFFFFFFC; 
		arm9reg.tmp1+=4; 
		arm9reg.cpsr=arm9reg.spsr[arm9reg.curmode]; 
		ARM9splitCPSR();
    } 
    return 3;

}

OPCODE ARM9opLDMIBu() { ARM9addrLMIB(); ARM9opLDMu(); }
OPCODE ARM9opLDMDAu() { ARM9addrLMDA(); ARM9opLDMu(); }


OPCODE ARM9opLDMDBu() 
{ 
#if 0
	ARM9addrLMDB(); ARM9opLDMu(); 
#endif
	arm9reg.tmp1=arm9reg.r[ARM9OP_RN]-(ARM9SideSum(arm9reg.curop&65535)*4);
    if(arm9reg.curop&0x0001) 
	{ 
		arm9reg.r[0]=MMUMainrdW(0,arm9reg.tmp1); 
		arm9reg.tmp1+=4; 
	} 
    if(arm9reg.curop&0x0002) 
	{ 
		arm9reg.r[1]=MMUMainrdW(0,arm9reg.tmp1); 
		arm9reg.tmp1+=4; 
	} 
    if(arm9reg.curop&0x0004) 
	{ 
		arm9reg.r[2]=MMUMainrdW(0,arm9reg.tmp1); 
		arm9reg.tmp1+=4; 
	} 
    if(arm9reg.curop&0x0008) 
	{ 
		arm9reg.r[3]=MMUMainrdW(0,arm9reg.tmp1); 
		arm9reg.tmp1+=4; 
	} 
    if(arm9reg.curop&0x0010) 
	{ 
		arm9reg.r[4]=MMUMainrdW(0,arm9reg.tmp1); 
		arm9reg.tmp1+=4; 
	} 
    if(arm9reg.curop&0x0020) 
	{ 
		arm9reg.r[5]=MMUMainrdW(0,arm9reg.tmp1); 
		arm9reg.tmp1+=4; 
	} 
    if(arm9reg.curop&0x0040) 
	{ 
		arm9reg.r[6]=MMUMainrdW(0,arm9reg.tmp1); 
		arm9reg.tmp1+=4; 
	} 
    if(arm9reg.curop&0x0080) 
	{ 
		arm9reg.r[7]=MMUMainrdW(0,arm9reg.tmp1); 
		arm9reg.tmp1+=4; 
	} 
    switch(arm9reg.curmode) 
    { 
    	case ARMMD_USR: case ARMMD_SYS: 
    	    if(arm9reg.curop&0x0100) 
			{ 
				arm9reg.r[8]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4;  
			} 
    	    if(arm9reg.curop&0x0200) 
			{ 
				arm9reg.r[9]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4;  
			} 
    	    if(arm9reg.curop&0x0400) 
			{ 
				arm9reg.r[10]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x0800) 
			{ 
				arm9reg.r[11]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x1000) 
			{ 
				arm9reg.r[12]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x2000) 
			{ 
				arm9reg.r[13]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x4000) 
			{ 
				arm9reg.r[14]=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    break;
    	case ARMMD_IRQ: case ARMMD_ABT: case ARMMD_SVC: case ARMMD_UND: 
    	    if(arm9reg.curop&0x0100) 
			{ 
				arm9reg.r8=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4;  
			} 
    	    if(arm9reg.curop&0x0200) 
			{ 
				arm9reg.r9=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4;  
			} 
    	    if(arm9reg.curop&0x0400) 
			{ 
				arm9reg.r10=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x0800) 
			{ 
				arm9reg.r11=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x1000) 
			{ 
				arm9reg.r12=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x2000) 
			{ 
				arm9reg.r13=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			}   
    	    if(arm9reg.curop&0x4000) 
			{ 
				arm9reg.r14=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			}   
            break; 
    	case ARMMD_FIQ: 
    	    if(arm9reg.curop&0x0100) 
			{ 
				arm9reg.r8=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4;  
			} 
    	    if(arm9reg.curop&0x0200) 
			{ 
				arm9reg.r9=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4;  
			} 
    	    if(arm9reg.curop&0x0400) 
			{ 
				arm9reg.r10=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x0800) 
			{ 
				arm9reg.r11=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x1000) 
			{ 
				arm9reg.r12=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x2000) 
			{ 
				arm9reg.r13=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
    	    if(arm9reg.curop&0x4000) 
			{ 
				arm9reg.r14=MMUMainrdW(0,arm9reg.tmp1); 
				arm9reg.tmp1+=4; 
			} 
            break; 
    } 
    return 3;
}

OPCODE ARM9opLDMIAuw() { ARM9addrLMIA(); ARM9addrLMwi(); ARM9opLDMu(); }
OPCODE ARM9opLDMIBuw() { ARM9addrLMIB(); ARM9addrLMwi(); ARM9opLDMu(); }
OPCODE ARM9opLDMDAuw() { ARM9addrLMDA(); ARM9addrLMwd(); ARM9opLDMu(); }
OPCODE ARM9opLDMDBuw() { ARM9addrLMDB(); ARM9addrLMwd(); ARM9opLDMu(); }

OPCODE ARM9opSTMIA() { int a; ARM9addrLMIA(); ARM9opSTM(); }
OPCODE ARM9opSTMIB() { int a; ARM9addrLMIB(); ARM9opSTM(); }
OPCODE ARM9opSTMDA() { int a; ARM9addrLMDA(); ARM9opSTM(); }
OPCODE ARM9opSTMDB() { int a; ARM9addrLMDB(); ARM9opSTM(); }

OPCODE ARM9opSTMIAw() { int a; ARM9addrLMIA(); ARM9addrLMwi(); ARM9opSTM(); }
OPCODE ARM9opSTMIBw() { int a; ARM9addrLMIB(); ARM9addrLMwi(); ARM9opSTM(); }
OPCODE ARM9opSTMDAw() { int a; ARM9addrLMDA(); ARM9addrLMwd(); ARM9opSTM(); }
OPCODE ARM9opSTMDBw() { int a; ARM9addrLMDB(); ARM9addrLMwd(); ARM9opSTM(); }

OPCODE ARM9opSTMIAu() { ARM9addrLMIA(); ARM9opSTMu(); }
OPCODE ARM9opSTMIBu() { ARM9addrLMIB(); ARM9opSTMu(); }
OPCODE ARM9opSTMDAu() { ARM9addrLMDA(); ARM9opSTMu(); }
OPCODE ARM9opSTMDBu() { ARM9addrLMDB(); ARM9opSTMu(); }

OPCODE ARM9opSTMIAuw() { ARM9addrLMIA(); ARM9addrLMwi(); ARM9opSTMu(); }
OPCODE ARM9opSTMIBuw() { ARM9addrLMIB(); ARM9addrLMwi(); ARM9opSTMu(); }
OPCODE ARM9opSTMDAuw() { ARM9addrLMDA(); ARM9addrLMwd(); ARM9opSTMu(); }
OPCODE ARM9opSTMDBuw() { ARM9addrLMDB(); ARM9addrLMwd(); ARM9opSTMu(); }

//-------------------------------------------------------------------------

OPCODE ARM9opMRSrs()
{
    ARM9updateCPSR();
    if(arm9reg.curmode>1) arm9reg.tmp1=arm9reg.spsr[arm9reg.curmode];
    else arm9reg.tmp1=arm9reg.cpsr;
    arm9reg.r[ARM9OP_RD]=arm9reg.tmp1;
    return 1;
}

OPCODE ARM9opMRSrc()
{
    ARM9updateCPSR();
    __asm {
	mov ebx,[arm9reg+ARMREG_CPSR*4]
	mov eax,[arm9reg+ARMREG_OP*4]
	shr eax,12
	and eax,15
	mov [arm9reg+eax*4],ebx
    }
    return 1;
}

OPCODE ARM9opMSRic()
{
    arm9reg.tmp3 = arm9reg.curop&255;
    arm9reg.tmp4 = (arm9reg.curop&0x00000F00)>>7;
    arm9reg.tmp1 = (arm9reg.tmp3>>arm9reg.tmp4)|((arm9reg.tmp3&((1<<arm9reg.tmp4)-1))<<(32-arm9reg.tmp4));
    arm9reg.tmp2=arm9reg.cpsr;
    if(arm9reg.curop&0x00080000)
    {
        arm9reg.tmp2&=0x00FFFFFF;
        arm9reg.tmp2|=(arm9reg.tmp1&0xFF000000);
    }
    if(arm9reg.curmode>0)
    {
        if(arm9reg.curop&0x00040000)
        {
            arm9reg.tmp2&=0xFF00FFFF;
            arm9reg.tmp2|=(arm9reg.tmp1&0x00FF0000);
        }
        if(arm9reg.curop&0x00020000)
        {
            arm9reg.tmp2&=0xFFFF00FF;
            arm9reg.tmp2|=(arm9reg.tmp1&0x0000FF00);
        }
        if(arm9reg.curop&0x00010000)
        {
            arm9reg.tmp2&=0xFFFFFF00;
            arm9reg.tmp2|=(arm9reg.tmp1&0x000000FF);
        }
    }
    arm9reg.cpsr=arm9reg.tmp2; ARM9splitCPSR();
    ARM9modesw(arm9reg.curmode,ARM9cpsrtomode[arm9reg.cpsr&0x0F]);
    return 1;
}

OPCODE ARM9opMSRis()
{
    arm9reg.tmp3 = arm9reg.curop&255;
    arm9reg.tmp4 = (arm9reg.curop&0x00000F00)>>7;
    arm9reg.tmp1 = (arm9reg.tmp3>>arm9reg.tmp4)|((arm9reg.tmp3&((1<<arm9reg.tmp4)-1))<<(32-arm9reg.tmp4));
    arm9reg.tmp2=0;
    if(arm9reg.curop&0x00080000)
    {
        arm9reg.tmp2&=0x00FFFFFF;
        arm9reg.tmp2|=(arm9reg.tmp1&0xFF000000);
    }
    if(arm9reg.curmode>0)
    {
        if(arm9reg.curop&0x00040000)
        {
            arm9reg.tmp2&=0xFF00FFFF;
            arm9reg.tmp2|=(arm9reg.tmp1&0x00FF0000);
        }
        if(arm9reg.curop&0x00020000)
        {
            arm9reg.tmp2&=0xFFFF00FF;
            arm9reg.tmp2|=(arm9reg.tmp1&0x0000FF00);
        }
        if(arm9reg.curop&0x00010000)
        {
            arm9reg.tmp2&=0xFFFFFF00;
            arm9reg.tmp2|=(arm9reg.tmp1&0x000000FF);
        }
    }
    if(arm9reg.curmode>1) arm9reg.spsr[arm9reg.curmode]=arm9reg.tmp2;
    return 1;
}

OPCODE ARM9opMSRrc()
{
    arm9reg.tmp1=arm9reg.r[ARM9OP_RM];
    arm9reg.tmp2=arm9reg.cpsr;
    if(arm9reg.curop&0x00080000)
    {
        arm9reg.tmp2&=0x00FFFFFF;
        arm9reg.tmp2|=(arm9reg.tmp1&0xFF000000);
    }
    if(arm9reg.curmode>0)
    {
        if(arm9reg.curop&0x00040000)
        {
            arm9reg.tmp2&=0xFF00FFFF;
            arm9reg.tmp2|=(arm9reg.tmp1&0x00FF0000);
        }
        if(arm9reg.curop&0x00020000)
        {
            arm9reg.tmp2&=0xFFFF00FF;
            arm9reg.tmp2|=(arm9reg.tmp1&0x0000FF00);
        }
        if(arm9reg.curop&0x00010000)
        {
            arm9reg.tmp2&=0xFFFFFF00;
            arm9reg.tmp2|=(arm9reg.tmp1&0x000000FF);
        }
    }
    arm9reg.cpsr=arm9reg.tmp2; ARM9splitCPSR();
    ARM9modesw(arm9reg.curmode,ARM9cpsrtomode[arm9reg.cpsr&0x0F]);
    return 1;
}

OPCODE ARM9opMSRrs()
{
    arm9reg.tmp1=arm9reg.r[ARM9OP_RM];
    arm9reg.tmp2=0;
    if(arm9reg.curop&0x00080000)
    {
        arm9reg.tmp2&=0x00FFFFFF;
        arm9reg.tmp2|=(arm9reg.tmp1&0xFF000000);
    }
    if(arm9reg.curmode>0)
    {
        if(arm9reg.curop&0x00040000)
        {
            arm9reg.tmp2&=0xFF00FFFF;
            arm9reg.tmp2|=(arm9reg.tmp1&0x00FF0000);
        }
        if(arm9reg.curop&0x00020000)
        {
            arm9reg.tmp2&=0xFFFF00FF;
            arm9reg.tmp2|=(arm9reg.tmp1&0x0000FF00);
        }
        if(arm9reg.curop&0x00010000)
        {
            arm9reg.tmp2&=0xFFFFFF00;
            arm9reg.tmp2|=(arm9reg.tmp1&0x000000FF);
        }
    }
    if(arm9reg.curmode>1) arm9reg.spsr[arm9reg.curmode]=arm9reg.tmp2;
    return 1;
}

//-------------------------------------------------------------------------

#define ARM9addrSWP() \
    arm9reg.tmp2=arm9reg.r[ARM9OP_RN]; \
    switch(arm9reg.tmp2&3) \
    { \
        case 0: arm9reg.tmp1=arm9reg.tmp2; break; \
	case 1: arm9reg.tmp1=(arm9reg.tmp2>> 8)|((arm9reg.tmp2&0x000000FF)<<24); break; \
	case 2: arm9reg.tmp1=(arm9reg.tmp2>>16)|((arm9reg.tmp2&0x0000FFFF)<<16); break; \
	case 3: arm9reg.tmp1=(arm9reg.tmp2>>24)|((arm9reg.tmp2&0x00FFFFFF)<< 8); break; \
    }

OPCODE ARM9opSWP()
{
    ARM9addrSWP();
    arm9reg.tmp4=MMUMainrdW(0,arm9reg.tmp1);
    MMUMainwrW(0,arm9reg.tmp1,arm9reg.r[ARM9OP_RM]);
    arm9reg.r[ARM9OP_RD]=arm9reg.tmp4;
    return 4;
}

OPCODE ARM9opSWPB()
{
    ARM9addrSWP();
    arm9reg.tmp4=0|MMUMainrdB(0,arm9reg.tmp1);
    MMUMainwrB(0,arm9reg.tmp1,arm9reg.r[ARM9OP_RM]&255);
    arm9reg.r[ARM9OP_RD]=0|arm9reg.tmp4;
    return 4;
}

//-------------------------------------------------------------------------

void ARM9InterruptFlagChanged() {
	// If the interrupt occured we're waiting for it, restart the CPU.
#if 0
	if(cpu9Halted && MMUMainrdH(0,0x04000214)&intr9Bitmask) {
		cpu9Halted = 0;
	}
#endif
}

void handleSWI()
{
    arm9reg.r14svc=arm9reg.r[15];
    arm9reg.spsr[ARMMD_SVC]=arm9reg.cpsr;
    ARM9modesw(arm9reg.curmode, ARMMD_SVC);
    arm9reg.cpsr&=(0xFFFFFFFF-ARMS_T);
    arm9reg.cpsr|=ARMS_I;
    arm9reg.r[15]= (arm9reg.cp15_control_register & 0x2000) ? 0xFFFF0008 : 0x00000008;
	ARM9execforPtr=ARM9execforARM;
	arm9reg.flags[ARMFLAG_T] = 0;
}

OPCODE ARM9opSWI()
{
    u32 a,ret; u32 src/*,dest,size*/;

/*    arm9reg.r14svc=arm9reg.r[15];
    arm9reg.spsr[ARMMD_SVC]=arm9reg.cpsr;
    ARM9modesw(arm9reg.curmode, ARMMD_SVC);
    arm9reg.cpsr&=(0xFFFFFFFF-ARMS_T);
    arm9reg.cpsr|=ARMS_I;
    arm9reg.r[15]=0x00000008;
*/
    switch((arm9reg.curop&0x00FF0000)>>16)
    {
        /*SoftReset*/ case 0:
            for(a=0;a<=12;a++) arm9reg.r[a]=0;
            if(MMUMainrdB(0,0x03007FFA)) ret=0x02000000; else ret=0x08000000;
            for(a=0x03007E00;a<=0x03007FFF;a+=4) MMUMainwrW(0,a,0);
            arm9reg.r13=0x03007F00;
            arm9reg.r13irq=0x03007FA0;
            arm9reg.r13svc=0x03007FE0;
            ARM9modesw(arm9reg.curmode,ARMMD_SYS);
            arm9reg.flags[ARMFLAG_T]=0;
            arm9reg.r[15]=ret; break;

        /*Delay*/ case 3:
			{
			int delay = arm9reg.r[0];
			arm9reg.r[0] = 0;
			return delay+3;
			}
			break;

        /*IntrWait*/ case 4:
			// Let the BIOS handle it
			handleSWI();
			break;

        /*VblankIntrWait*/ case 5:
			// Let the BIOS handle it
			handleSWI();
			break;

#if 0
		/*Halt*/ case 6:
         cpu9Halted=1; intr9Bitmask=0xFFFF; break;

        /*Stop*/ case 7:
            cpu9Halted=1; intr9Bitmask=0x0000; break;
#endif
        /*Div*/ case 9:
			{
				s32 lvalue = arm9reg.r[0];
				s32 rvalue = arm9reg.r[1];
				s32 divresult = lvalue / rvalue;
				s32 divmodulous = lvalue % rvalue;
				arm9reg.r[0] = divresult;
				arm9reg.r[1] = divmodulous;
			}
			break;

		/*Sqrt*/ case 0x0d:
            src=arm9reg.r[0]; arm9reg.r[0]=(u32)sqrt(src);
            break;
#if 0
        /*CpuSet*/ case 11:
            src=arm9reg.r[0]; dest=arm9reg.r[1]; size=arm9reg.r[2]&0xFFFF;
            for(a=0;a<size;a++)
            {
                if(arm9reg.r[2]&0x04000000)
                {
                    MMUMainwrW(0,dest,MMUMainrdW(0,src));
                    dest+=4;
                    if(!(arm9reg.r[2]&0x01000000)) src+=4;
                } else {
                    MMUMainwrH(0,dest,MMUMainrdH(0,src));
                    dest+=2;
                    if(!(arm9reg.r[2]&0x01000000)) src+=2;
                }
            }
            break;

        /*cpuFastSet*/ case 12:
            src=arm9reg.r[0]; dest=arm9reg.r[1]; size=arm9reg.r[2]&0xFFF8;
            for(a=0;a<size;a++)
            {
                MMUMainwrW(0,dest,MMUMainrdW(0,src));
                dest+=4;
                if(!(arm9reg.r[2]&0x01000000)) src+=4;
            }
#endif
	case 0xFF:
		{
			char str[64+256];
			char str2[256];
			char ch = MMUMainrdB(0, arm9reg.r[0]);
			int i = 0;
			sprintf(str, "SWI 0xFF - %8X - ", arm9reg.r[0xf]);
			str2[0] = ch;
			for(i=1;i < sizeof(str2)-1 && ch != 0;++i) {
				ch=MMUMainrdB(0, arm9reg.r[0]+i);
				str2[i] = ch;
			}
			str2[i] = 0;
			setlogbuf(str2);
		    logvt->append(strcat(str,str2));
		}		
		break;

    }
    return 3;
}

void ARM9irq()
{
#if 0
	{
		char str[250];
			extern u16 int9Fired;

		sprintf(str, "%8X ARM9IRQ %8X CPSR&I: %4X CPSR_svc&I: %4X curmode: %4X", 
			arm9reg.r[0xf], int9Fired, arm9reg.cpsr& ARMS_I, arm9reg.spsr[4]& ARMS_I,
			arm9reg.curmode);
		logvt->append(str);
	}
#endif
    arm9reg.r14irq=arm9reg.r[15]; 
    arm9reg.spsr[ARMMD_IRQ]=arm9reg.cpsr;
    ARM9modesw(arm9reg.curmode, ARMMD_IRQ);
    arm9reg.cpsr&=(0xFFFFFFFF-ARMS_T);
    arm9reg.cpsr|=ARMS_I;
    cpu9Halted=0; intr9Bitmask=0xFFFF;
    arm9reg.r[15]= (arm9reg.cp15_control_register & 0x2000) ? 0xFFFF0018 : 0x00000018;
	ARM9execforPtr=ARM9execforARM;
	arm9reg.flags[ARMFLAG_T] = 0;
//	CheckCPSR();
}

//---eDSP extension--------------------------------------------------------

OPCODE ARM9opCLZ()
{
    __asm {
        mov ebx,[arm9reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,12                       /* Shift in Rd */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
	and eax,15                       /* Mask Rd to 15 */
	mov edx,[arm9reg+ecx*4]
	mov ecx,31
        or edx,edx
        jz opzero
	bsr ebx,edx                      /* Count leading zeros */
        sub ecx,ebx
	jmp clzend
opzero: mov ecx,32
clzend:	mov [arm9reg+eax*4],ecx          /* Write result */
    }
    return 1;
}

OPCODE ARM9opBKPT()
{
    int bk=arm9reg.curop&0x0000000F;
    arm9reg.r[15]-=4;
    emuPause();
    return 0;
}

#define ARM9opLDRD() \
    __asm { \
        __asm shr edx,12                /* Shift in Rd */ \
        __asm and edx,15                /* Mask off */ \
        __asm cmp edx,14                /* Is Rd 14? */ \
        __asm je ldr1                   /* If so, end */ \
        __asm push edx                  /* Save Rd */ \
        __asm push eax                  /* Address */ \
        __asm xor ecx,ecx               /* Bus = 0 */ \
        __asm push ecx                  /* Push "bus number" */ \
        __asm call MMUMainrdW               /* Read a word */ \
        __asm pop ecx                   /* Clear stack */ \
        __asm pop ebx                   /* of params */ \
        __asm pop edx                   /* Get Rd back */\
	__asm mov [arm9reg+edx*4],eax   /* Write value to Rd */ \
	__asm inc edx                   /* Move along one reg */ \
	__asm add ebx,4                 /* And one word */ \
        __asm push edx                  /* Save Rd */ \
        __asm push ebx                  /* Address */ \
        __asm xor ecx,ecx               /* Bus = 0 */ \
        __asm push ecx                  /* Push "bus number" */ \
        __asm call MMUMainrdW               /* Read a word */ \
        __asm pop ecx                   /* Clear stack */ \
        __asm pop ecx                   /* of params */ \
        __asm pop edx                   /* Get Rd back */\
	__asm mov [arm9reg+edx*4],eax   /* Write value to Rd */ \
        __asm ldr1:                     /* Otherwise.. */ \
	__asm mov eax,3 \
    }

#define ARM9opSTRD() \
    __asm { \
        __asm shr edx,12                  /* Shift in Rd */ \
        __asm and edx,14                  /* Mask to 15 */ \
        __asm mov ebx,[arm9reg+edx*4]     /* Get Rd value */ \
        __asm push ebx                    /* Push Rd */ \
        __asm push eax                    /* Push address */ \
        __asm xor ecx,ecx                 /* Bus = 0 */ \
        __asm push ecx                    /* Push "bus number" */ \
        __asm call MMUMainwrW                 /* Write thru MMUMain */ \
        __asm pop eax                     /* Clear stack */ \
        __asm pop eax                     /* of pushed */ \
        __asm pop ecx                     /* parameters */ \
	__asm inc edx                     /* Move along one reg */ \
	__asm add eax,4                   /* And one word */ \
        __asm mov ebx,[arm9reg+edx*4]     /* Get Rd value */ \
        __asm push ebx                    /* Push Rd */ \
        __asm push eax                    /* Push address */ \
        __asm xor ecx,ecx                 /* Bus = 0 */ \
        __asm push ecx                    /* Push "bus number" */ \
        __asm call MMUMainwrW                 /* Write thru MMUMain */ \
        __asm pop eax                     /* Clear stack */ \
        __asm pop eax                     /* of pushed */ \
        __asm pop eax                     /* parameters */ \
	__asm mov eax,2 \
    }

OPCODE ARM9opLDRDptim() { ARM9addrLSlmptim(); ARM9opLDRD(); }
OPCODE ARM9opLDRDptip() { ARM9addrLSlmptip(); ARM9opLDRD(); }
OPCODE ARM9opLDRDptrm() { ARM9addrLSptrm(); ARM9opLDRD(); }
OPCODE ARM9opLDRDptrp() { ARM9addrLSptrp(); ARM9opLDRD(); }
OPCODE ARM9opLDRDprim() { ARM9addrLSlmprim(); ARM9opLDRD(); }
OPCODE ARM9opLDRDprip() { ARM9addrLSlmprip(); ARM9opLDRD(); }
OPCODE ARM9opLDRDprrm() { ARM9addrLSprrm(); ARM9opLDRD(); }
OPCODE ARM9opLDRDprrp() { ARM9addrLSprrp(); ARM9opLDRD(); }
OPCODE ARM9opLDRDofim() { ARM9addrLSlmofim(); ARM9opLDRD(); }
OPCODE ARM9opLDRDofip() { ARM9addrLSlmofip(); ARM9opLDRD(); }
OPCODE ARM9opLDRDofrm() { ARM9addrLSofrm(); ARM9opLDRD(); }
OPCODE ARM9opLDRDofrp() { ARM9addrLSofrp(); ARM9opLDRD(); }

OPCODE ARM9opSTRDptim() { ARM9addrLSlmptim(); ARM9opSTRD(); }
OPCODE ARM9opSTRDptip() { ARM9addrLSlmptip(); ARM9opSTRD(); }
OPCODE ARM9opSTRDptrm() { ARM9addrLSptrm(); ARM9opSTRD(); }
OPCODE ARM9opSTRDptrp() { ARM9addrLSptrp(); ARM9opSTRD(); }
OPCODE ARM9opSTRDprim() { ARM9addrLSlmprim(); ARM9opSTRD(); }
OPCODE ARM9opSTRDprip() { ARM9addrLSlmprip(); ARM9opSTRD(); }
OPCODE ARM9opSTRDprrm() { ARM9addrLSprrm(); ARM9opSTRD(); }
OPCODE ARM9opSTRDprrp() { ARM9addrLSprrp(); ARM9opSTRD(); }
OPCODE ARM9opSTRDofim() { ARM9addrLSlmofim(); ARM9opSTRD(); }
OPCODE ARM9opSTRDofip() { ARM9addrLSlmofip(); ARM9opSTRD(); }
OPCODE ARM9opSTRDofrm() { ARM9addrLSofrm(); ARM9opSTRD(); }
OPCODE ARM9opSTRDofrp() { ARM9addrLSofrp(); ARM9opSTRD(); }

OPCODE ARM9opQADD()
{
    __asm {
        mov ecx,[arm9reg+ARMREG_OP*4]
        mov eax,ecx
        mov ebx,ecx
        and eax,15
        shr ebx,16
        and ebx,15
        shr ecx,12
        and ecx,15
        mov ebx,[arm9reg+ebx*4]
        add ebx,[arm9reg+eax*4]
        mov eax,1
        jno no_over
        mov [arm9reg+ARMREG_FQ*4],eax
        add eax,1
        test ebx,0x80000000
        jnz over_neg
        mov ebx,0x80000000
        jmp no_over
        over_neg:
        mov ebx,0x7FFFFFFF
        no_over:
        mov [arm9reg+ecx*4],ebx
    }
}

OPCODE ARM9opSMULBB()
{
    __asm {
        mov ecx,[arm9reg+ARMREG_OP*4]
        mov eax,ecx
        mov ebx,ecx
        and eax,15
        shr ebx,8
        and ebx,15
        shr ecx,16
        and ecx,15
        movsx eax,word ptr [arm9reg+eax*4]
        movsx ebx,word ptr [arm9reg+ebx*4]
        imul eax,ebx
        mov [arm9reg+ecx*4],eax
        mov eax,1
    }
}

OPCODE ARM9opSMULBT()
{
    __asm {
        mov ecx,[arm9reg+ARMREG_OP*4]
        mov eax,ecx
        mov ebx,ecx
        and eax,15
        shr ebx,8
        and ebx,15
        shr ecx,16
        and ecx,15
        movsx eax,word ptr [arm9reg+eax*4]
        movsx ebx,word ptr [arm9reg+ebx*4+2]
        imul eax,ebx
        mov [arm9reg+ecx*4],eax
        mov eax,1
    }
}

OPCODE ARM9opSMULTB()
{
    __asm {
        mov ecx,[arm9reg+ARMREG_OP*4]
        mov eax,ecx
        mov ebx,ecx
        and eax,15
        shr ebx,8
        and ebx,15
        shr ecx,16
        and ecx,15
        movsx eax,word ptr [arm9reg+eax*4+2]
        movsx ebx,word ptr [arm9reg+ebx*4]
        imul eax,ebx
        mov [arm9reg+ecx*4],eax
        mov eax,1
    }
}

OPCODE ARM9opSMULTT()
{
    __asm {
        mov ecx,[arm9reg+ARMREG_OP*4]
        mov eax,ecx
        mov ebx,ecx
        and eax,15
        shr ebx,8
        and ebx,15
        shr ecx,16
        and ecx,15
        movsx eax,word ptr [arm9reg+eax*4+2]
        movsx ebx,word ptr [arm9reg+ebx*4+2]
        imul eax,ebx
        mov [arm9reg+ecx*4],eax
        mov eax,1
    }
}

OPCODE ARM9opSMLABB()
{
    __asm {
        mov ecx,[arm9reg+ARMREG_OP*4]
        mov eax,ecx
        mov ebx,ecx
        and eax,15
        shr ebx,8
        and ebx,15
        movsx eax,word ptr [arm9reg+eax*4]
        movsx ebx,word ptr [arm9reg+ebx*4]
        imul eax,ebx
        mov ebx,ecx
        shr ebx,12
        and ebx,15
        shr ecx,16
        and ecx,15
        mov ebx,[arm9reg+ebx*4]
        add eax,ebx
        jno smla_end
        seto [arm9reg+ARMREG_FQ*4]
        smla_end:
        mov [arm9reg+ecx*4],eax
        mov eax,1
    }
}

OPCODE ARM9opSMLABT()
{
    __asm {
        mov ecx,[arm9reg+ARMREG_OP*4]
        mov eax,ecx
        mov ebx,ecx
        and eax,15
        shr ebx,8
        and ebx,15
        movsx eax,word ptr [arm9reg+eax*4]
        movsx ebx,word ptr [arm9reg+ebx*4+2]
        imul eax,ebx
        mov ebx,ecx
        shr ebx,12
        and ebx,15
        shr ecx,16
        and ecx,15
        mov ebx,[arm9reg+ebx*4]
        add eax,ebx
        jno smla_end
        seto [arm9reg+ARMREG_FQ*4]
        smla_end:
        mov [arm9reg+ecx*4],eax
        mov eax,1
    }
}

OPCODE ARM9opSMLATB()
{
    __asm {
        mov ecx,[arm9reg+ARMREG_OP*4]
        mov eax,ecx
        mov ebx,ecx
        and eax,15
        shr ebx,8
        and ebx,15
        movsx eax,word ptr [arm9reg+eax*4+2]
        movsx ebx,word ptr [arm9reg+ebx*4]
        imul eax,ebx
        mov ebx,ecx
        shr ebx,12
        and ebx,15
        shr ecx,16
        and ecx,15
        mov ebx,[arm9reg+ebx*4]
        add eax,ebx
        jno smla_end
        seto [arm9reg+ARMREG_FQ*4]
        smla_end:
        mov [arm9reg+ecx*4],eax
        mov eax,1
    }
}

OPCODE ARM9opSMLATT()
{
    __asm {
        mov ecx,[arm9reg+ARMREG_OP*4]
        mov eax,ecx
        mov ebx,ecx
        and eax,15
        shr ebx,8
        and ebx,15
        movsx eax,word ptr [arm9reg+eax*4+2]
        movsx ebx,word ptr [arm9reg+ebx*4+2]
        imul eax,ebx
        mov ebx,ecx
        shr ebx,12
        and ebx,15
        shr ecx,16
        and ecx,15
        mov ebx,[arm9reg+ebx*4]
        add eax,ebx
        jno smla_end
        seto [arm9reg+ARMREG_FQ*4]
        smla_end:
        mov [arm9reg+ecx*4],eax
        mov eax,1
    }
}

//---Copro handling--------------------------------------------------------

OPCODE ARM9opMRC()
{
#if 0
    char str[160];
    sprintf(str,"COPRO: Read from cp%1X c%1X/c%1X op %1X/%1X to r%1X",
            ((arm9reg.curop&0x00000F00)>>8),
            ((arm9reg.curop&0x000F0000)>>16),
            ((arm9reg.curop&0x0000000F)),
            ((arm9reg.curop&0x00E00000)>>21),
            ((arm9reg.curop&0x000000E0)>>5),
            ((arm9reg.curop&0x0000F000)>>12));
    logvt->append(str);
#endif
    switch((arm9reg.curop&0x000000F00)>>8)
    {
        case 15:
            arm9reg.r[(arm9reg.curop&0x0000F000)>>12]=
                ARM9cp15Get((arm9reg.curop&0x000F0000)>>16,
                            (arm9reg.curop&0x0000000F),
                            (arm9reg.curop&0x00E00000)>>21,
                            (arm9reg.curop&0x000000E0)>>5);
            break;
    }
    return 1;
}

OPCODE ARM9opMCR()
{
#if 0
    char str[160];
    sprintf(str,"COPRO: Write to cp%1X c%1X/c%1X op %1X/%1X from r%1X",
            ((arm9reg.curop&0x00000F00)>>8),
            ((arm9reg.curop&0x000F0000)>>16),
            ((arm9reg.curop&0x0000000F)),
            ((arm9reg.curop&0x00E00000)>>21),
            ((arm9reg.curop&0x000000E0)>>5),
            ((arm9reg.curop&0x0000F000)>>12));
    logvt->append(str);
#endif
    switch((arm9reg.curop&0x000000F00)>>8)
    {
        case 15:
            ARM9cp15Set((arm9reg.curop&0x000F0000)>>16,
                        (arm9reg.curop&0x0000000F),
						(arm9reg.curop&0x0000F000)>>12,
                        (arm9reg.curop&0x00E00000)>>21,
                        (arm9reg.curop&0x000000E0)>>5);
            break;
    }

    return 1;
}

OPCODE ARM9opCDP()
{
    return 1;
}

u32 ARM9cp15Get(int Rn, int Rm, int op1, int op2)
{
    char str[160];
#if 0
    sprintf(str,"ARM9cp15Get: p15, %1X, r?, c%1X, c%1X", op1, Rn, Rm, op2);
    logvt->append(str);
#endif
    switch(Rn)
    {
        case 0:
            switch(op2)
            {
                case 0: return 0x41059461; /*ARM5TE, 946-1, by ARM*/
                case 1: return 0x0F0D2112; /*000 0111 1 [000 011 010 0 10] [000 100 010 0 10]*/
                default: return 0;
            }
            break;

			case 1:
#if 0
				sprintf(str, "get cp15_control_register = %8X", arm9reg.cp15_control_register);
				logvt->append(str);
#endif
				return arm9reg.cp15_control_register;
		break;
			    sprintf(str,"ARM9cp15Get: p15, %1X, r?, c%1X, c%1X", op1, Rn, Rm, op2);
				logvt->append(str);
				sprintf(str, "ARM9cp15Get: c%1X not implemented", Rn);
				logvt->append(str);
				break;

			case 9:
				return arm9reg.cp15_tcm_base[op2];
				break;

        default: return 0;
    }
}

void ARM9cp15Set(int Rn, int Rm, int Rd, int op1, int op2)
{
    char str[160];
	switch(Rn) {
	case 1:
		DoCP15C1(arm9reg.r[Rd]);
		break;

	case 5:
		DoCP15C5(Rm, op2, arm9reg.r[Rd]);
		break;

	case 6:
		DoCP15C6(Rm, arm9reg.r[Rd]);
		break;

	case 7:
		if(Rm == 0) {
			// mcr 15, 0, lr, cr7. cr0 = powersave mode
			if(arm9reg.flags[ARMFLAG_T]) {
				arm9reg.r[0xf] += 2;
			} 
			else {
				arm9reg.r[0xf] += 4;
			}
			cpu9Halted = 1;
		}
		else {
		    sprintf(str,"ARM9cp15Set: p15, %1X, r%1X, c%1X, c%1X", op1, Rd, Rn, Rm, op2);
			logvt->append(str);
			sprintf(str, "ARM9cp15Set: c%1X not implemented", Rn);
			logvt->append(str);
		}

		break;

	case 9:
		if(Rm == 1)
		{
			DoCP15C9(op2, arm9reg.r[Rd]);
		}
		else {
		    sprintf(str,"ARM9cp15Set: p15, %1X, r%1X, c%1X, c%1X", op1, Rd, Rn, Rm, op2);
			logvt->append(str);
			sprintf(str, "ARM9cp15Set: c%1X not implemented", Rn);
			logvt->append(str);
		}
		break;
	default:
	    sprintf(str,"ARM9cp15Set: p15, %1X, r%1X, c%1X, c%1X", op1, Rd, Rn, Rm, op2);
	    logvt->append(str);
		sprintf(str, "ARM9cp15Set: c%1X not implemented", Rn);
		logvt->append(str);
		break;
	}
}

void ARM9cp15DP(int Rn, int Rm, int Rd, int op1, int op2)
{
}

void DoCP15C1(unsigned int value)
{
	const int protection_unit_enabled = value & 1;
	const int dcache_enabled = value & (1<<2);
	const int big_endian = value & (1<<7);
	const int icache_enabled = value & (1<<12);
	const int alternate_vector_select = value & (1<<13);
	const int round_robin_replacement = value & (1<<14);
	const int tbit = value & (1<<15);
	const int data_ram_enable = value & (1<<16);
	const int data_ram_load_mode = value & (1<<17);
	const int instruction_ram_enable = value & (1<<18);
	const int instruction_ram_load_mode = value & (1<<19);
#if defined(REGION_DEBUG)
	{
		char str[120];
		sprintf(str,"CP15-C1: Value: %8X", value);
	    logvt->append(str);
		sprintf(str,"         Protection unit %s", protection_unit_enabled ? "enabled" : "disabled");
	    logvt->append(str);
		sprintf(str,"         DCache %s", dcache_enabled ? "enabled" : "disabled");
	    logvt->append(str);
		sprintf(str,"         Endian is %s", big_endian ? "big" : "little");
	    logvt->append(str);
		sprintf(str,"         ICache %s", icache_enabled ? "enabled" : "disabled");
	    logvt->append(str);
		sprintf(str,"         Vector Address %s", alternate_vector_select ? "FFFF0000" : "00000000");
	    logvt->append(str);
		sprintf(str,"         round robin replacement %s", round_robin_replacement ? "enabled" : "disabled");
	    logvt->append(str);
		sprintf(str,"         configure disable loading TBUT %s", tbit ? "enabled" : "disabled");
	    logvt->append(str);
		sprintf(str,"         Data RAM enable %s", data_ram_enable ? "enabled" : "disabled");
	    logvt->append(str);
		sprintf(str,"         Data RAM load mode %s", data_ram_load_mode ? "enabled" : "disabled");
	    logvt->append(str);
		sprintf(str,"         Instruction RAM enable %s", instruction_ram_enable ? "enabled" : "disabled");
	    logvt->append(str);
		sprintf(str,"         Instruction RAM load mode %s", instruction_ram_load_mode ? "enabled" : "disabled");
	    logvt->append(str);
	}
#endif
	arm9reg.cp15_control_register = value;
}

// Return the string given a CP15 memory return size
static char* region_size(unsigned int size)
{
	static char* sizes[] = {
		"Reserved0",
		"Reserved1",
		"Reserved2",
		"Reserved3",
		"Reserved4",
		"Reserved5",
		"Reserved6",
		"Reserved7",
		"Reserved8",
		"Reserved9",
		"Reserved10",
		"4KB",
		"8KB",
		"16KB",
		"32KB",
		"64KB",
		"128KB",
		"256KB",
		"512KB",
		"1MB",
		"2MB",
		"4MB",
		"8MB",
		"16MB",
		"32MB",
		"64MB",
		"128MB",
		"256MB",
		"512MB",
		"1GB",
		"2GB",
		"4GB"
	};

	return sizes[size];
}

// Do Register 6 of CP15 - Protection region/base size
void DoCP15C6(unsigned int region, unsigned int value)
{
	const unsigned int enable = value & 1;
	const unsigned int area_size = (value >> 1) & 0x1F;
	const unsigned int base_address = (value >> 12) & 0xFFFFF;

#if defined(REGION_DEBUG)
	{
		char str[120];
		sprintf(str,"CP15-C6: Region: %1X Enable: %1X Address: %8X Size: %s", 
			region, enable, base_address * 0x1000, region_size(area_size));
	    logvt->append(str);
	}
#endif
	arm9reg.cp15_regions[region] = value;
}

static char* access_permissions(unsigned int value)
{
	static char* permissions[] = {
		"None, None",
		"R/W,  None",
		"R/W,  R",
		"R/W,  R/W",
		"UNP,  UNP",
		"R,    None",
		"R,    R",
		"UNP,  UNP",
		"UNP,  UNP",
		"UNP,  UNP",
		"UNP,  UNP",
		"UNP,  UNP",
		"UNP,  UNP",
		"UNP,  UNP",
		"UNP,  UNP",
		"UNP,  UNP"
	};

	return permissions[value];
}

static char* c5opcode_string(unsigned int opcode) 
{
	static char* s[] = {
		"0           ",
		"1           ",
		"data        ",
		"instruction "
	};
	return s[opcode];
}

void DoCP15C5(unsigned int region, unsigned int opcode2, unsigned int value)
{
	unsigned int area0 = value & 0xf;
	unsigned int area1 = (value >> 4) & 0xf;
	unsigned int area2 = (value >> 8) & 0xf;
	unsigned int area3 = (value >> 12) & 0xf;
	unsigned int area4 = (value >> 16) & 0xf;
	unsigned int area5 = (value >> 20) & 0xf;
	unsigned int area6 = (value >> 24) & 0xf;
	unsigned int area7 = (value >> 28) & 0xf;


#if defined(REGION_DEBUG)
	{
		char str[120];
		sprintf(str,"CP15-C5: Region: 0 %s %s",  c5opcode_string(opcode2), access_permissions(area0));
	    logvt->append(str);
		sprintf(str,"CP15-C5: Region: 1 %s %s",  c5opcode_string(opcode2), access_permissions(area1));
	    logvt->append(str);
		sprintf(str,"CP15-C5: Region: 2 %s %s",  c5opcode_string(opcode2), access_permissions(area2));
	    logvt->append(str);
		sprintf(str,"CP15-C5: Region: 3 %s %s",  c5opcode_string(opcode2), access_permissions(area3));
	    logvt->append(str);
		sprintf(str,"CP15-C5: Region: 4 %s %s",  c5opcode_string(opcode2), access_permissions(area4));
	    logvt->append(str);
		sprintf(str,"CP15-C5: Region: 5 %s %s",  c5opcode_string(opcode2), access_permissions(area5));
	    logvt->append(str);
		sprintf(str,"CP15-C5: Region: 6 %s %s",  c5opcode_string(opcode2), access_permissions(area6));
	    logvt->append(str);
		sprintf(str,"CP15-C5: Region: 7 %s %s",  c5opcode_string(opcode2), access_permissions(area7));
	    logvt->append(str);
	}
#endif
	// TODO: implement
}

// Return the string given a CP15 memory return size
static char* tcm_region_size(unsigned int size)
{
	static char* sizes[] = {
		"Reserved0",
		"Reserved1",
		"Reserved2",
		"4KB",
		"8KB",
		"16KB",
		"32KB",
		"64KB",
		"128KB",
		"256KB",
		"512KB",
		"1MB",
		"2MB",
		"4MB",
		"8MB",
		"16MB",
		"32MB",
		"64MB",
		"128MB",
		"256MB",
		"512MB",
		"1GB",
		"2GB",
		"4GB",
		"Reserved3",
		"Reserved4",
		"Reserved5",
		"Reserved6",
		"Reserved7",
		"Reserved8",
		"Reserved9",
		"Reserved10"
	};

	return sizes[size];
}

// tck == 0 for itcm, 1 for dtcm
void DoCP15C9(unsigned int tcm, unsigned int value)
{
	const unsigned int area_size = (value >> 1) & 0x1F;
	const unsigned int base_address = (value >> 12) & 0xFFFFF;

#if defined(REGION_DEBUG)
	{
		char str[120];
		sprintf(str,"CP15-C9: Region: %s Address: %8X Size: %s", 
			tcm ? "ITCM" : "DTCM", base_address * 0x1000, tcm_region_size(area_size));
	    logvt->append(str);
	}
#endif
	arm9reg.cp15_tcm_base[tcm] = value;
}
		

//=========================================================================
//---Thumb time!-----------------------------------------------------------

#include "thumb9x86.c"

/*** EOF:arm9.c **********************************************************/
