/**************************************************************************
* DSemu: Scanline-based tiled and bitmap graphics (gpu.c)                 *
* Released under the terms of the BSD Public Licence                      *
* gladius, 2004-5; ported to DSemu by Imran Nazar                         *
**************************************************************************/

#include <windows.h>
#include <stdio.h>
#include "err.h"
#include "defs.h"
#include "ioreg.h"
#include "gpu.h"
#include "dma.h"
#include "vtbl.h"
#include "int.h"
#include "resource.h"

//#define GPUDEBUG

u16 *VRAM;
u8  *VRAM8;

u16 OAM[1024];
u16 BGPAL[256];
u16 OBJPAL[256];
u8 *BGPAL8=(u8*)BGPAL;
u8 *OBJPAL8=(u8*)OBJPAL;

u32 BGX[2],BGY[2];
u8 mospy[4];

u16 curline;
__int64 framecount;

extern u32 VRAMmap[553];

int sprsize[16][2]={
    { 8, 8},{16,16},{32,32},{64,64},
    {16, 8},{32, 8},{32,16},{64,32},
    { 8,16},{ 8,32},{16,32},{32,64},
    {-1,-1},{-1,-1},{-1,-1},{-1,-1},
};

int GPUinit()
{
    u8 r,g,b; char str[50]; int a;

    VRAM = (u16*)malloc(1024*1024);
    if(!VRAM) RETFAIL("GPU: FAIL: VRAM allocation.");
    logvt->append("GPU: VRAM allocated.");
    VRAM8=(u8*)VRAM;

    GPUreset();

    RETPASS("GPU: Initialised.");
}

void GPUfini()
{
    char str[2048];
    free(VRAM);
    sprintf(str,"GPU: Shutdown after %lld frames.",framecount);
    logvt->append(str);
}

void GPUreset()
{
    int a;
    memset(VRAM, 0, 1024*1024);
    memset(BGPAL,0, 256*2);
    memset(OBJPAL,0, 256*2);
    memset(OAM, 0, 512*2);
    curline=0;
    BGX[0]=0; BGY[0]=0;
    BGX[1]=0; BGY[1]=0;
    REG(BG2PA)=256; REG(BG2PD)=256;
    REG(BG3PA)=256; REG(BG3PD)=256;
    framecount=0; for(a=0;a<4;a++) mospy[a]=0;
}

void GPUpal(u16 *pbuf)
{
    int a,b,c,d; u32 xoff=0,yoff=8,idx=0;
    dbgOut(pbuf,264,8,"BGPAL",0,0,0x7FFF);
    for(a=0;a<16;a++)
    {
        for(b=0;b<16;b++)
        {
	    for(c=0;c<8;c++)
	    {
	        for(d=0;d<8;d++)
	        {
//	            pbuf[(143-(a*8+c))*128+(b*8+d)]=LCDcolour[BGPAL[a*16+b]];
//                  pbuf[off]=LCDcolour[BGPAL[idx]];
//                  off++;
	            pbuf[yoff*264+xoff]=BGPAL[idx];
                    xoff++;
	        }
//	        off+=137;
                yoff++; xoff-=8;
	    }
//	    off-=1008;
	    idx++;
            xoff+=8; yoff-=8;
	}
//	idx+=16;
	xoff=0; yoff+=8;
    }

    xoff=136; yoff=8; idx=0;
    dbgOut(pbuf,264,8,"OBJPAL",136,0,0x7FFF);
    for(a=0;a<16;a++)
    {
        for(b=0;b<16;b++)
        {
	    for(c=0;c<8;c++)
	    {
	        for(d=0;d<8;d++)
	        {
//	            pbuf[(143-(a*8+c))*128+(b*8+d)]=LCDcolour[BGPAL[a*16+b]];
//                  pbuf[off]=LCDcolour[BGPAL[idx]];
//                  off++;
	            pbuf[yoff*264+xoff]=OBJPAL[idx];
                    xoff++;
	        }
//	        off+=137;
                yoff++; xoff-=8;
	    }
//	    off-=1008;
	    idx++;
            xoff+=8; yoff-=8;
	}
//	idx+=16;
	xoff=136; yoff+=8;
    }
}

void GPUpalcol(u16 *pbuf, int idx)
{
    char str[80];
    u16 col; u8 r,g,b;
    if(idx&256) col=OBJPAL[idx&255]; else col=BGPAL[idx&255];
    r=(col&0x001F); g=(col&0x03E0)>>5; b=(col&0x7C00)>>10;
    dbgOutClear(pbuf+264*144,264,8);
    sprintf(str,"R:%02Xh  G:%02Xh  B:%02Xh",r,g,b);
    dbgOut(pbuf+264*144,264,8,str,(idx&256)?136:0,0,0x7FFF);
}

//---START: Gladius' code--------------------------------------------------

void GPUscanTile(u16 *scanline, int bg)
{
    u16 bgcnt, scrnBase, charBase, hofs, vofs, x, y, tileIdx, tileY;
    u16 tmpTileIdx, width, height, a, bgx, bgy, tileChar; u8 b, palnum;
    u8 mosx, mospx, mosy;
    bgcnt=GBAio[REG_BG0CNT+bg].data;
    switch((bgcnt>>14)&3)
    {
        case 0: width=256; height=256; break;
        case 1: width=512; height=256; break;
        case 2: width=256; height=512; break;
        case 3: width=512; height=512; break;
    }
    scrnBase=((bgcnt>>8)&0x1F)*0x400;
    charBase=((bgcnt>>2)&0x3)*0x4000;
    hofs=GBAio[REG_BG0HOFS+bg*2].data;
    vofs=GBAio[REG_BG0VOFS+bg*2].data;
/*    vramptr=VRAM+(VRAMmap[(lcd)*128]<<13);
    vram8ptr=VRAM8+(VRAMmap[(lcd)*128]<<14);*/
    if(bgcnt&0x0040)
    {
        mosx=(REG(MOSAIC)&0x000F);
        mosy=((REG(MOSAIC)&0x00F0)>>4);
        mospx=mosx; if(!curline) mospy[bg]=mosy;
        mospy[bg]++; if(mospy[bg]<=mosy) vofs-=mospy[bg];
        else mospy[bg]=0;
    } else { mospx=0; mosx=0; }
    if(bgcnt&0x0080)
    {
        bgy=((curline+vofs)&(height-1))/8;
        tileIdx=scrnBase+(((bgy&31)*32));
        switch((bgcnt>>14)&3)
        {
            case 2: if(bgy>=32) tileIdx += 32*32; break;
            case 3: if(bgy>=32) tileIdx += 32*32*2; break;
        }
        tileY=((curline+vofs)&7)*8;
        for(a=0;a<240;a++)
        {
            bgx=((a+hofs)&(width-1))/8;
            tmpTileIdx=tileIdx+((bgx&31));
            if(bgx>=32) tmpTileIdx += 32*32;
            tileChar=VRAM[tmpTileIdx];
            x=(a+hofs)&7; y=tileY;
            if(tileChar&0x0400) x=7-x;
            if(tileChar&0x0800) y=56-y;
            if(mospx>=mosx) b=VRAM8[charBase+((tileChar&0x03FF)*64)+x+y];
            mospx++; if(mospx>mosx) mospx=0;
            if(b) scanline[a]=BGPAL[b];
        }
    } else {
        bgy=((curline+vofs)&(height-1))/8;
        tileIdx=scrnBase+(((bgy&31)*32));
        switch((bgcnt>>14)&3)
        {
            case 2: if(bgy>=32) tileIdx += 32*32; break;
            case 3: if(bgy>=32) tileIdx += 32*32*2; break;
        }
        tileY=((curline+vofs)&7)*4;
        for(a=0;a<240;a++)
        {
            bgx=((a+hofs)&(width-1))/8;
            tmpTileIdx=tileIdx+((bgx&31));
            if(bgx>=32) tmpTileIdx += 32*32;
            tileChar=VRAM[tmpTileIdx];
            x=(a+hofs)&7; y=tileY;
            if(tileChar&0x0400) x=7-x;
            if(tileChar&0x0800) y=28-y;
            if(mospx>=mosx)
            {
                b=VRAM8[charBase+((tileChar&0x03FF)*32)+(x/2)+y];
                if(x&1) b>>=4; b&=15;
            }
            palnum=((tileChar>>12)&15)*16;
            mospx++; if(mospx>mosx) mospx=0;
            if(b) scanline[a]=BGPAL[b+palnum];
        }
    }
}

void GPUscanRot(u16 *scanline, int bg)
{
    u16 bgcnt, scrnBase, charBase, hofs, vofs, tileIdx, tileY;
    u32 x,y; u16 ax,ay; s16 dx,dy; int trans; u8 mosx, mospx, mosy;
    u16 tmpTileIdx, width, height, a, bgx, bgy, tileChar; u8 b, palnum;
    bgcnt=REG(BG0CNT+bg); bg-=2;
    trans=(bgcnt&0x2000)==0;
    switch((bgcnt>>14)&3)
    {
        case 0: width=128; height=128; break;
        case 1: width=256; height=256; break;
        case 2: width=512; height=512; break;
        case 3: width=1024; height=1024; break;
    }
    scrnBase=((bgcnt>>8)&0x1F)*0x800;
    charBase=((bgcnt>>2)&0x3)*0x4000;
    x=BGX[bg]; y=BGY[bg];
    dx=REG(BG2PA+(bg*8)); dy=REG(BG2PC+(bg*8));
    if(bgcnt&0x0040)
    {
        mosx=(REG(MOSAIC)&0x000F);
        mosy=((REG(MOSAIC)&0x00F0)>>4);
        mospx=mosx; if(!curline) mospy[bg]=mosy;
        mospy[bg]++; if(mospy[bg]<=mosy) y-=mospy[bg]*256;
        else mospy[bg]=0;
    } else { mospx=0; mosx=0; }
    for(a=0;a<240;a++)
    {
        ax=x>>8; ay=y>>8;
        if ((ax >= 0 && ax < width && ay >= 0 && ay < height) || !trans)
        {
            tmpTileIdx=scrnBase+(((ay&(height-1))/8)*(width/8)+((ax&(width-1))/8));
            tileChar=VRAM8[tmpTileIdx];
            if(mospx>=mosx) b=VRAM8[charBase+tileChar*64+((ay&7)*8)+(ax&7)];
            mospx+=(dx>>8); if(mospx>mosx) mospx=0;
            if(b) scanline[a]=BGPAL[b];
        }
        x+=dx; y+=dy;
    }
}

void GPUsprites(u16 *scanline, int pri)
{
    u16 dispcnt=REG(DISPCNT), oamNum, attr0,attr1,attr2;
    int x, y, i, width=-1, height=-1, rwidth, rheight, scale=1;
    int spritey, baseSprite, baseInc, curIdx, palIdx; u8 b;
    s16 dx,dmx,rx,dy,dmy,ry; int tx,ty,cx,cy,pitch,rotScaleParam;
    char str[100];

    if(!(dispcnt&0x1000)) return;
    pri<<=10;
    oamNum=128; do
    {
        oamNum--;
        attr2=OAM[oamNum*4+2];
        if((attr2&0x0C00)!=pri) continue;
        attr0=OAM[oamNum*4+0];
        attr1=OAM[oamNum*4+1];
        x=attr1&0x01FF; y=attr0&0x00FF;

                switch ((attr0 >> 14) & 3)
                {
                    case 0:
                        // Square
                        switch ((attr1 >> 14) & 3)
                        {
                            case 0: width = 8; height = 8; break;
                            case 1: width = 16; height = 16; break;
                            case 2: width = 32; height = 32; break;
                            case 3: width = 64; height = 64; break;
                        }
                        break;
                    case 1:
                        // Horizontal Rectangle
                        switch ((attr1 >> 14) & 3)
                        {
                            case 0: width = 16; height = 8; break;
                            case 1: width = 32; height = 8; break;
                            case 2: width = 32; height = 16; break;
                            case 3: width = 64; height = 32; break;
                        }
                        break;
                    case 2:
                        // Vertical Rectangle
                        switch ((attr1 >> 14) & 3)
                        {
                            case 0: width = 8; height = 16; break;
                            case 1: width = 8; height = 32; break;
                            case 2: width = 16; height = 32; break;
                            case 3: width = 32; height = 64; break;
                        }
                        break;
                }

                rwidth=width; rheight=height;
                if (attr0&0x0100)
                    // Rot-scale on
                    if (attr0&0x0200) { rwidth*=2; rheight*=2; }
                else
                    // Invalid sprite
                    if (attr0&0x0200) width = -1;

                if (width == -1)
                    // Invalid sprite
                    continue;

                // Y clipping
                if (y > ((y + rheight) & 0xff))
                    if (curline >= ((y + rheight) & 0xff) && !(y < curline)) continue;
                else
                    if (curline < y || curline >= ((y + rheight) & 0xff)) continue;

                if (attr0&0x2000) scale = 2;

                spritey = curline - y;
                if (spritey < 0) spritey += 256;

                if (!(attr0&0x0100))
                {
                    if (attr1&0x2000) spritey = (height - 1) - spritey;

                    if (dispcnt&0x0040)
                        // 1 dimensional
                        baseSprite = (attr2 & 0x3FF) + ((spritey / 8) * (width / 8)) * scale;
                    else
                        // 2 dimensional
                        baseSprite = (attr2 & 0x3FF) + ((spritey / 8) * 0x20);

                    baseInc = scale;
                    if (attr1&0x1000)
                    {
                        baseSprite += ((width / 8) * scale) - scale;
                        baseInc = -baseInc;
                    }

                    if (attr0&0x2000)
                    {
                        // 256 colors
//                        sprintf(str,"GPU: Line #%d: Sprite #%d: %dx%d, non-rotating, 8-bit",curline,oamNum,width,height);
//                        logvt->append(str);
                        for (i = x; i < x + width; i++)
                        {
                            if ((i & 0x1ff) < 240 && spritey >= 0 && spritey < height)
                            {
                                tx = (i - x) & 7;
                                if (attr1&0x1000) tx = 7 - tx;
                                curIdx = baseSprite * 32 + ((spritey & 7) * 8) + tx;
                                b = VRAM8[0x10000 + curIdx];
                                if (b) scanline[i&0x1ff]=OBJPAL[b];
                            }
                            if (((i - x) & 7) == 7) baseSprite += baseInc;
                        }
                    }
                    else
                    {
                        // 16 colors
//                        sprintf(str,"GPU: Line #%d: Sprite #%d: %dx%d, non-rotating, 4-bit",curline,oamNum,width,height);
//                        logvt->append(str);
                        palIdx = ((attr2 >> 8) & 0xF0);
                        for (i = x; i < x + width; i++)
                        {
                            if ((i & 0x1ff) < 240 && spritey >= 0 && spritey < height)
                            {
                                tx = (i - x) & 7;
                                if ((attr1 & (1 << 12)) != 0) tx = 7 - tx;
                                curIdx = baseSprite * 32 + ((spritey & 7) * 4) + (tx / 2);
                                b = VRAM8[0x10000 + curIdx];
                                if (tx & 1) b>>=4; b&=15;
                                if(b) scanline[i&0x1ff]=OBJPAL[palIdx+b];
                            }
                            if (((i - x) & 7) == 7) baseSprite += baseInc;
                        }
                    }
                }
                else
                {
                    rotScaleParam = (attr1 >> 9) & 0x1F;

                    dx =  (s16)OAM[(rotScaleParam * 4 * 4) + 3];
                    dmx = (s16)OAM[(rotScaleParam * 4 * 4) + 7];
                    dy =  (s16)OAM[(rotScaleParam * 4 * 4) + 11];
                    dmy = (s16)OAM[(rotScaleParam * 4 * 4) + 15];

                    cx = rwidth / 2;
                    cy = rheight / 2;

                    baseSprite = attr2 & 0x3FF;

                    if (dispcnt&0x0040)
                        // 1 dimensional
                        pitch = (width / 8) * scale;
                    else
                        // 2 dimensional
                        pitch = 0x20;

                    rx = (s16)((dmx * (spritey - cy)) - (cx * dx) + (width << 7));
                    ry = (s16)((dmy * (spritey - cy)) - (cx * dy) + (height << 7));

                    // Draw a rot/scale sprite
                    if (attr0&0x2000)
                    {
//                        sprintf(str,"GPU: Line #%d: Sprite #%d: %dx%d, rotating, 8-bit",curline,oamNum,width,height);
//                        logvt->append(str);
                        // 256 colors
                        for (i = x; i < x + rwidth; i++)
                        {
                            tx = rx >> 8;
                            ty = ry >> 8;

                            if ((i & 0x1ff) < 240 && tx >= 0 && tx < width && ty >= 0 && ty < height)
                            {
                                curIdx = (baseSprite + ((ty / 8) * pitch) + ((tx / 8) * scale)) * 32 + ((ty & 7) * 8) + (tx & 7);
                                b = VRAM8[0x10000 + curIdx];
                                if(b) scanline[i&0x1ff]=OBJPAL[b];
                            }

                            rx += dx;
                            ry += dy;
                        }
                    }
                    else
                    {
                        // 16 colors
//                        sprintf(str,"GPU: Line #%d: Sprite #%d: %dx%d, rotating, 8-bit",curline,oamNum,width,height);
//                        logvt->append(str);
                        palIdx = ((attr2 >> 8) & 0xF0);
                        for (i = x; i < x + rwidth; i++)
                        {
                            tx = rx >> 8;
                            ty = ry >> 8;

                            if ((i & 0x1ff) < 240 && tx >= 0 && tx < width && ty >= 0 && ty < height)
                            {
                                curIdx = (baseSprite + ((ty / 8) * pitch) + ((tx / 8) * scale)) * 32 + ((ty & 7) * 4) + ((tx & 7) / 2);
                                b = VRAM8[0x10000 + curIdx];
                                if (tx & 1) b>>=4; b&=15;
                                if(b) scanline[i&0x1ff]=OBJPAL[palIdx+b];
                            }

                            rx += dx;
                            ry += dy;
                        }
                    }
                }
    } while(oamNum);
}

void GPUscanMode0(u16 *scanline)
{
    u16 dispcnt=REG(DISPCNT), i; int pri;
    for(pri=3;pri>=0;pri--)
    {
        if(dispcnt&0x0800) if((REG(BG3CNT)&3)==pri) GPUscanTile(scanline,3);
        if(dispcnt&0x0400) if((REG(BG2CNT)&3)==pri) GPUscanTile(scanline,2);
        if(dispcnt&0x0200) if((REG(BG1CNT)&3)==pri) GPUscanTile(scanline,1);
        if(dispcnt&0x0100) if((REG(BG0CNT)&3)==pri) GPUscanTile(scanline,0);
        GPUsprites(scanline,pri);
    }
}

void GPUscanMode1(u16 *scanline)
{
    u16 dispcnt=REG(DISPCNT), i; int pri;
    for(pri=3;pri>=0;pri--)
    {
        if(dispcnt&0x0400) if((REG(BG2CNT)&3)==pri) GPUscanRot(scanline,2);
        if(dispcnt&0x0200) if((REG(BG1CNT)&3)==pri) GPUscanTile(scanline,1);
        if(dispcnt&0x0100) if((REG(BG0CNT)&3)==pri) GPUscanTile(scanline,0);
        GPUsprites(scanline,pri);
    }
}

void GPUscanMode2(u16 *scanline)
{
    u16 dispcnt=REG(DISPCNT), i; int pri;
    for(pri=3;pri>=0;pri--)
    {
        if(dispcnt&0x0800) if((REG(BG3CNT)&3)==pri) GPUscanRot(scanline,3);
        if(dispcnt&0x0400) if((REG(BG2CNT)&3)==pri) GPUscanRot(scanline,2);
        GPUsprites(scanline,pri);
    }
}

void GPUscanMode3(u16 *scanline)
{
    u16 dispcnt=REG(DISPCNT), bg2cnt=REG(BG2CNT), i; int pri;
    u32 x,y,ax,ay; s16 dx,dy; u16 w; u16 mosx,mospx,mosy;
    for(pri=3;pri>(bg2cnt&3);pri--) GPUsprites(scanline,pri);
    if(dispcnt&0x0400)
    {
        x=BGX[0]; y=BGY[0];
        if(bg2cnt&0x0040)
        {
            mosx=(REG(MOSAIC)&0x000F);
            mosy=((REG(MOSAIC)&0x00F0)>>4);
            mospx=mosx; if(!curline) mospy[2]=mosy;
            mospy[2]++; if(mospy[2]<=mosy) y-=mospy[2]*256;
            else mospy[2]=0;
        } else { mospx=0; mosx=0; }
        dx=(signed)REG(BG2PA); dy=(signed)REG(BG2PC);
        for(i=0;i<240;i++)
        {
            ax=x>>8; ay=y>>8;
            if (ax >= 0 && ax < 240 && ay >= 0 && ay < 160)
            {
                if(mospx>=mosx) w=VRAM[ay*240+ax];
                scanline[i]=w;
            }
            mospx+=(dx>>8); if(mospx>mosx) mospx=0;
            x+=dx; y+=dy;
        }
    }
    for(pri=(bg2cnt&3);pri>=0;pri--) GPUsprites(scanline,pri);
}

void GPUscanMode4(u16 *scanline)
{
    u16 dispcnt=REG(DISPCNT), bg2cnt=REG(BG2CNT), i; int pri;
    u32 x,y,ax,ay; s16 dx,dy; u8 b; u16 mosx,mospx,mosy;
    u16 baseIdx=(bg2cnt&0x0010)?0xA000:0;
    for(pri=3;pri>(bg2cnt&3);pri--) GPUsprites(scanline,pri);
    if(dispcnt&0x0400)
    {
        x=BGX[0]; y=BGY[0];
        if(bg2cnt&0x0040)
        {
            mosx=(REG(MOSAIC)&0x000F);
            mosy=((REG(MOSAIC)&0x00F0)>>4);
            mospx=mosx; if(!curline) mospy[2]=mosy;
            mospy[2]++; if(mospy[2]<=mosy) y-=mospy[2]*256;
            else mospy[2]=0;
        } else { mospx=0; mosx=0; }
        dx=(signed)REG(BG2PA); dy=(signed)REG(BG2PC);
        for(i=0;i<240;i++)
        {
            ax=x>>8; ay=y>>8;
            if (ax >= 0 && ax < 240 && ay >= 0 && ay < 160)
            {
                if(mospx>=mosx) b=VRAM8[baseIdx+ay*240+ax];
                if(b) scanline[i]=BGPAL[b];
            }
            mospx+=(dx>>8); if(mospx>mosx) mospx=0;
            x+=dx; y+=dy;
        }
    }
    for(pri=(bg2cnt&3);pri>=0;pri--) GPUsprites(scanline,pri);
}

void GPUscanMode5(u16 *scanline)
{
    u16 dispcnt=REG(DISPCNT), bg2cnt=REG(BG2CNT), i; int pri;
    u32 x,y,ax,ay; s16 dx,dy; u16 w; u16 mosx,mospx,mosy;
    u16 baseIdx=(bg2cnt&0x0010)?(160*128):0;
    for(pri=3;pri>(bg2cnt&3);pri--) GPUsprites(scanline,pri);
    if(dispcnt&0x0400)
    {
        x=BGX[0]; y=BGY[0];
        if(bg2cnt&0x0040)
        {
            mosx=(REG(MOSAIC)&0x000F);
            mosy=((REG(MOSAIC)&0x00F0)>>4);
            mospx=mosx; if(!curline) mospy[2]=mosy;
            mospy[2]++; if(mospy[2]<=mosy) y-=mospy[2]*256;
            else mospy[2]=0;
        } else { mospx=0; mosx=0; }
        dx=(signed)REG(BG2PA); dy=(signed)REG(BG2PC);
        for(i=0;i<240;i++)
        {
            ax=x>>8; ay=y>>8;
            if (ax >= 0 && ax < 160 && ay >= 0 && ay < 128)
            {
                if(mospx>=mosx) w=VRAM[baseIdx+ay*160+ax];
                scanline[i]=w;
            }
            mospx+=(dx>>8); if(mospx>mosx) mospx=0;
            x+=dx; y+=dy;
        }
    }
    for(pri=(bg2cnt&3);pri>=0;pri--) GPUsprites(scanline,pri);
}

void GPUscanNULL(u16 *scanline)
{
}

fptr GPUscanModes[]={GPUscanMode0,GPUscanMode1,GPUscanMode2,
                     GPUscanMode3,GPUscanMode4,GPUscanMode5,
                     GPUscanNULL,GPUscanNULL};

void GPUscanline(u16 *screen)
{
    u16 dispcnt=REG(DISPCNT), voff=curline*240, scanline[240], i;
    if(curline<160)
    {
        if(dispcnt&0x0080) for(i=0;i<240;i++) scanline[i]=0x7FFF;
        else
        {
            for(i=0;i<240;i++) scanline[i]=BGPAL[0];
            GPUscanModes[dispcnt&7](scanline);
        }
        memcpy(screen+voff, scanline, 480);
        if(REG(DISPSTAT)&STAT_INTHBL) IntFire7(INT_HBLANK);
        DMAcheck(DMA_TIMEHBL);
    }
    REG(DISPSTAT)|=STAT_HBLANK;
}

//---END: Gladius' code----------------------------------------------------

void GPUclearHBL()
{
    REG(DISPSTAT)&=(0xFFFF-STAT_HBLANK);
    curline++; if(curline>227) curline=0;
    BGX[0]+=(s16)REG(BG2PB); BGY[0]+=(s16)REG(BG2PD);
    BGX[1]+=(s16)REG(BG3PB); BGY[1]+=(s16)REG(BG3PD);
    REG(VCOUNT)=curline;
    switch(curline)
    {
        case 0:
           REG(DISPSTAT)&=(0xFFFF-STAT_VBLANK);
           BGX[0]=REG(BG2XL)+(REG(BG2XH)<<16);
           BGX[1]=REG(BG3XL)+(REG(BG3XH)<<16);
           BGY[0]=REG(BG2YL)+(REG(BG2YH)<<16);
           BGY[1]=REG(BG3YL)+(REG(BG3YH)<<16);
           break;
        case 160:
           framecount++;
	   REG(DISPSTAT)|=STAT_VBLANK;
           if(REG(DISPSTAT)&STAT_INTVBL) IntFire7(INT_VBLANK);
           DMAcheck(DMA_TIMEVBL);
	   break;
    }
    if(curline==(REG(DISPSTAT)>>6))
    {
        REG(DISPSTAT)|=STAT_VCOUNT;
        if(REG(DISPSTAT)&STAT_INTVCT) IntFire7(INT_VCOUNT);
    }
    else
        REG(DISPSTAT)&=(0xFFFF-STAT_VCOUNT);
}

/*** EOF:gpu.c ***********************************************************/
