// dsregisters.c
#include <assert.h>
#include "arm.h"
#include "dsint.h"
#include "dsioreg.h"
#include "dsregisters.h"
#include "dsaccelerators.h"

static void initialize_fifo_registers();

static u8 do_fifocnt_read_byte(int cpu, u32 addr);
static u16 do_fifocnt_read_half(int cpu, u32 addr);
static u32 do_fifocnt_read_word(int cpu, u32 addr);
static void do_fifocnt_write_byte(int cpu, u32 addr, u8 value);
static void do_fifocnt_write_half(int cpu, u32 addr, u16 value);
static void do_fifocnt_write_word(int cpu, u32 addr, u32 value);

static u8 do_fifosend_read_byte(int cpu, u32 addr);
static u16 do_fifosend_read_half(int cpu, u32 addr);
static u32 do_fifosend_read_word(int cpu, u32 addr);
static void do_fifosend_write_byte(int cpu, u32 addr, u8 value);
static void do_fifosend_write_half(int cpu, u32 addr, u16 value);
static void do_fifosend_write_word(int cpu, u32 addr, u32 value);

static u8 do_fiforecv_read_byte(int cpu, u32 addr);
static u16 do_fiforecv_read_half(int cpu, u32 addr);
static u32 do_fiforecv_read_word(int cpu, u32 addr);
static void do_fiforecv_write_byte(int cpu, u32 addr, u8 value);
static void do_fiforecv_write_half(int cpu, u32 addr, u16 value);
static void do_fiforecv_write_word(int cpu, u32 addr, u32 value);


int is_register_address(int cpu, u32 addr)
{
	(void)cpu;
	return ((addr>= 0x04000184 && addr < (0x04000184+2)) ||
		   (addr >= 0x04000188 && addr < (0x04000188+4)) ||
           (addr >= 0x04000280 && addr <= 0x040002BC )    || 
		   (addr >= 0x04100000 && addr < (0x04100000+4)));
}

u8 do_register_read_byte(int cpu, u32 addr)
{
	if(addr >= 0x04000184 && addr < (0x04000184+2))
		return do_fifocnt_read_byte(cpu, addr);
	else if(addr >= 0x04000188 && addr < (0x04000188+4))
		return do_fifosend_read_byte(cpu, addr);
	else if(addr >= 0x04100000 && addr < (0x04100000+4))
		return do_fiforecv_read_byte(cpu, addr);
	else if (addr >= 0x04000280 && addr <= 0x040002BC ) 
        return do_accelerators_read_byte(cpu, addr);
    else
		return 0;
}

u16 do_register_read_half(int cpu, u32 addr)
{
	if(addr >= 0x04000184 && addr < (0x04000184+2))
		return do_fifocnt_read_half(cpu, addr);
	else if(addr >= 0x04000188 && addr < (0x04000188+4))
		return do_fifosend_read_half(cpu, addr);
	else if(addr >= 0x04100000 && addr < (0x04100000+4))
		return do_fiforecv_read_half(cpu, addr);
    else if (addr >= 0x04000280 && addr <= 0x040002BC ) 
        return do_accelerators_read_half(cpu, addr);
	else
		return 0;
}

u32 do_register_read_word(int cpu, u32 addr)
{
	if(addr >= 0x04000184 && addr < (0x04000184+2))
		return do_fifocnt_read_word(cpu, addr);
	else if(addr >= 0x04000188 && addr < (0x04000188+4))
		return do_fifosend_read_word(cpu, addr);
	else if(addr >= 0x04100000 && addr < (0x04100000+4))
		return do_fiforecv_read_word(cpu, addr);
	else if (addr >= 0x04000280 && addr <= 0x040002BC ) 
        return do_accelerators_read_word(cpu, addr);
	else
		return 0;
}


void do_register_write_byte(int cpu, u32 addr, u8 value)
{
	if(addr >= 0x04000184 && addr < (0x04000184+2))
		do_fifocnt_write_byte(cpu, addr, value);
	else if(addr >= 0x04000188 && addr < (0x04000188+4))
		do_fifosend_write_byte(cpu, addr, value);
	else if(addr >= 0x04100000 && addr < (0x04100000+4))
		do_fiforecv_write_byte(cpu, addr, value);
    else if (addr >= 0x04000280 && addr <= 0x040002BC ) 
        do_accelerators_write_byte(cpu, addr,value);
}

void do_register_write_half(int cpu, u32 addr, u16 value)
{
	if(addr >= 0x04000184 && addr < (0x04000184+2))
		do_fifocnt_write_half(cpu, addr, value);
	else if(addr >= 0x04000188 && addr < (0x04000188+4))
		do_fifosend_write_half(cpu, addr, value);
	else if(addr >= 0x04100000 && addr < (0x04100000+4))
		do_fiforecv_write_half(cpu, addr, value);
    else if (addr >= 0x04000280 && addr <= 0x040002BC ) 
        do_accelerators_write_half(cpu, addr,value);
}

void do_register_write_word(int cpu, u32 addr, u32 value)
{
	if(addr >= 0x04000184 && addr < (0x04000184+2))
		do_fifocnt_write_word(cpu, addr, value);
	else if(addr >= 0x04000188 && addr < (0x04000188+4))
		do_fifosend_write_word(cpu, addr, value);
	else if(addr >= 0x04100000 && addr < (0x04100000+4))
		do_fiforecv_write_word(cpu, addr, value);
    else if (addr >= 0x04000280 && addr <= 0x040002BC ) 
        do_accelerators_write_word(cpu, addr,value);
}

void initialize_registers()
{
	initialize_fifo_registers();
    initialize_accelerators_registers();
}

/* FIFO Register Implementation */
static u16 fifocnt[NO_CPU_REG];
static u32 fifoqueue[NO_CPU_REG][16];
static int fifoqueuehead[NO_CPU_REG];
static int fifoqueuetail[NO_CPU_REG];

static int queue_empty(int cpu) {
	return fifoqueuehead[cpu] == fifoqueuetail[cpu];
}

static int queue_full(int cpu) {
	return (fifoqueuehead[cpu] != 0 && fifoqueuehead[cpu] == fifoqueuetail[cpu] + 1) ||
		(fifoqueuehead[cpu] == 0 && fifoqueuetail[cpu] == 15);
}

/* Push on the end, pop from the front */
static void push_queue(int cpu, u32 value) {
	/* Only if the queue is enabled */
	if(fifocnt[cpu] & (1<<15)) {
		/* If the queue is full, error */
		if(queue_full(cpu)) {
			fifocnt[cpu] |= (1<<14);
		} 
		else {
			if(queue_empty(cpu)) {
				fifocnt[cpu] &= ~(1<<0);
				fifocnt[OTHER_CPU(cpu)] &= ~(1<<9);				
			}
			fifoqueue[cpu][fifoqueuetail[cpu]++] = value;
			if(fifoqueuetail[cpu] == 16)
				fifoqueuetail[cpu] = 0;
			if(queue_full(cpu)) {
				fifocnt[cpu] |= (1<<1);
				fifocnt[cpu] &= ~(1<<0);
				fifocnt[OTHER_CPU(cpu)] |= (1<<9);
				fifocnt[OTHER_CPU(cpu)] &= ~(1<<0);
				if(fifocnt[OTHER_CPU(cpu)] & (1<<10)) {
					if(OTHER_CPU(cpu) == ARM7_REG)
						IntFire7(18);
					else
						IntFire9(18);
				}
			}
		}
	}
}

static u32 pop_queue(int cpu) {
	u32 value = 0;
	/* Only if the queue is enabled */
	if(fifocnt[cpu] & (1<<15)) {
		if(queue_empty(cpu)) {
			fifocnt[cpu] |= (1<<14);
		} 
		else {
			if(queue_full(cpu)) {
				fifocnt[cpu] &= ~(1<<9);
				fifocnt[OTHER_CPU(cpu)] &= ~(1<<1);
			}
			value = fifoqueue[cpu][fifoqueuehead[cpu]++];
			if(fifoqueuehead[cpu] == 16)
				fifoqueuehead[cpu] = 0;
			if(queue_empty(cpu)) {
				fifocnt[cpu] |= (1<<8);
				fifocnt[OTHER_CPU(cpu)] |= (1<<0);
				if(fifocnt[OTHER_CPU(cpu)] & (1<<2)) {
					if(OTHER_CPU(cpu) == ARM7_REG)
						IntFire7(17);
					else
						IntFire9(17);
				}
			}
		}
	}
	return value;
}

static void initialize_fifo_registers()
{
	fifocnt[ARM7_REG] = 0x101;
	fifoqueuehead[ARM7_REG] = 0;
	fifoqueuetail[ARM7_REG] = 0;
	fifocnt[ARM9_REG] = 0x101;
	fifoqueuehead[ARM9_REG] = 0;
	fifoqueuetail[ARM9_REG] = 0;
}

static u8 do_fifocnt_read_byte(int cpu, u32 addr)
{
	if(addr == 0x04000184)
		return (u8)(fifocnt[cpu] & 0xFF);
	else if(addr == 0x04000185)
		return (u8)((fifocnt[cpu] & 0xFF00) >> 8);

	// We should never get here
	assert(1!=0);
	return 0;
}

static u16 do_fifocnt_read_half(int cpu, u32 addr)
{
	if(addr == 0x04000184)
		return fifocnt[cpu];
	else if(addr == 0x04000185)
		return (u16)(((fifocnt[cpu] & 0xFF00) >> 8) | 
				(do_register_read_byte(cpu, addr+1) << 8));
	// We should never get here
	assert(1!=0);
	return 0;
}

static u32 do_fifocnt_read_word(int cpu, u32 addr)
{
	if(addr == 0x04000184)
		return fifocnt[cpu] | (do_register_read_half(cpu, addr+2) << 16);
	else if(addr == 0x04000185)
		return ((fifocnt[cpu] & 0xFF00) >> 8) | 
				(do_register_read_byte(cpu, addr+1) << 8);
	// We should never get here
	assert(1!=0);
	return 0;
}

static void do_fifocnt_write_byte(int cpu, u32 addr, u8 value)
{
	if(addr == 0x04000184) {
		const int send_irq = value & (1<<2);
		const int send_clear = value & (1<<3);
		const int enabled = fifocnt[cpu] & (1<<15);
		
		if(send_irq)
			fifocnt[cpu] |= (1<<3);
		else
			fifocnt[cpu] &= ~(1<<3);

		if(enabled && send_clear) {
			fifoqueuehead[cpu] = 0;
			fifoqueuetail[cpu] = 0;
		}

		if(enabled && send_irq && !queue_empty(cpu)) {
			// TODO:Set correct interrrupt
			if(cpu == ARM9_REG)
				IntFire9(17);
			else 
				IntFire7(17);
		}
	}
	else if(addr == 0x04000185) {
		const int recv_irq = value & (1<<2);
		const int enabled = value & (1<<7);

		if(recv_irq)
			fifocnt[cpu] |= (1<<10);
		else
			fifocnt[cpu] &= ~(1<<10);

		if(enabled)
			fifocnt[cpu] |= (1<<15);
		else
			fifocnt[cpu] &= ~(1<<15);

		if(enabled) {
			if((fifocnt[cpu] & (1<<3))) {
				fifoqueuehead[cpu] = 0;
				fifoqueuetail[cpu] = 0;
			}

			if((fifocnt[cpu] & (1<<2)) && !queue_empty(cpu)) {
				// TODO:Set correct interrrupt
				if(cpu == ARM9_REG)
					IntFire9(99);
				else 
					IntFire7(99);
			}
			if((fifocnt[cpu] & (1<<10)) && !queue_empty(OTHER_CPU(cpu))) {
				// TODO:Set correct interrrupt
				if(cpu == ARM9_REG)
					IntFire9(18);
				else 
					IntFire7(18);
			}
		}
	}
}

static void do_fifocnt_write_half(int cpu, u32 addr, u16 value)
{
	if(addr == 0x04000184) {
		do_fifocnt_write_byte(cpu, addr, (u8)(value & 0xFF));
		do_fifocnt_write_byte(cpu, addr+1, (u8)((value & 0xFF00)>>8));
	}
	else if(addr == 0x04000185) {
		do_fifocnt_write_byte(cpu, addr, (u8)(value & 0xFF));
		do_register_write_byte(cpu, addr+1, (u8)((value & 0xFF00)>>8));
	}
}

static void do_fifocnt_write_word(int cpu, u32 addr, u32 value)
{
	if(addr == 0x04000184) {
		do_fifocnt_write_half(cpu, addr, (u16)(value & 0xFFFF));
		do_register_write_half(cpu, addr+2, (u16)((value & 0xFFFF0000)>>16));
	}
	else if(addr == 0x04000185) {
		do_fifocnt_write_byte(cpu, addr, (u8)(value & 0xFF));
		do_register_write_byte(cpu, addr+1, (u8)((value & 0xFF00)>>8));
		do_register_write_half(cpu, addr+2, (u16)((value & 0xFFFF0000)>>16));
	}
}

static u8 do_fifosend_read_byte(int cpu, u32 addr)
{
	(void)cpu;
	(void)addr;
	return 0;
}

static u16 do_fifosend_read_half(int cpu, u32 addr)
{
	if(addr == 0x04000188)
		return 0;
	else if(addr == 0x04000188+1)
		return 0;
	else if(addr == 0x04000188+2)
		return 0;
	else if(addr == 0x04000188+3)
		return (u16)(do_register_read_byte(cpu, addr+3) << 8);
	// We should never get here
	assert(1!=0);
	return 0;
}

static u32 do_fifosend_read_word(int cpu, u32 addr)
{
	if(addr == 0x04000188)
		return 0;
	else if(addr == 0x04000188+1)
		return (do_register_read_byte(cpu, addr+3) << 24);
	else if(addr == 0x04000188+2)
		return (do_register_read_half(cpu, addr+2) << 16);
	else if(addr == 0x04000188+3)
		return (do_register_read_half(cpu, addr+3) << 8) |
			   (do_register_read_byte(cpu, addr+3) << 24);
	// We should never get here
	assert(1!=0);
	return 0;
}

static void do_fifosend_write_byte(int cpu, u32 addr, u8 value)
{
	if(addr == 0x04000188) 
		do_fifosend_write_word(cpu, 0x04000188, value);
	else if(addr == 0x04000188+1)
		do_fifosend_write_word(cpu, 0x04000188, value<<8);
	else if(addr == 0x04000188+2)
		do_fifosend_write_word(cpu, 0x04000188, value<<16);
	else if(addr == 0x04000188+3)
		do_fifosend_write_word(cpu, 0x04000188, value<<24);
}

static void do_fifosend_write_half(int cpu, u32 addr, u16 value)
{
	if(addr == 0x04000188) 
		do_fifosend_write_word(cpu, 0x04000188, value);
	else if(addr == 0x04000188+1)
		do_fifosend_write_word(cpu, 0x04000188, value<<8);
	else if(addr == 0x04000188+2)
		do_fifosend_write_word(cpu, 0x04000188, value<<16);
	else if(addr == 0x04000188+3) {
		do_fifosend_write_word(cpu, 0x04000188, (value&0xFF)<<24);
		do_register_write_byte(cpu, 0x04000188+4, (u8)((value & 0xFF00)>>8));
	}
}

static void do_fifosend_write_word(int cpu, u32 addr, u32 value)
{
	if(addr == 0x04000188) {
		push_queue(cpu, value);
	}
	else if(addr == 0x04000188+1) {
		do_fifosend_write_word(cpu, 0x04000188, (value & 0x00FFFFFF)<<8);
		do_register_write_byte(cpu, addr+4, (u8)((value & 0xFF000000)>>24));
	}
	else if(addr == 0x04000188+2) {
		do_fifosend_write_word(cpu, 0x04000188, (value & 0x00FFFF)<<16);
		do_register_write_half(cpu, addr+4, (u16)((value & 0xFFFF0000)>>16));
	}
	else if(addr == 0x04000188+3) {
		do_fifosend_write_word(cpu, 0x04000188, (value & 0x00FF)<<24);
		do_register_write_half(cpu, addr+4, (u16)((value & 0x00FFFF00)>>8));
		do_register_write_byte(cpu, addr+6, (u8)((value & 0xFF000000)>>24));
	}
}

static u8 do_fiforecv_read_byte(int cpu, u32 addr)
{
	if(addr == 0x04100000) 		
		return (u8)(do_fiforecv_read_word(cpu, addr) & 0xFF);
	else if(addr == 0x04100000+1) 		
		return (u8)((do_fiforecv_read_word(cpu, addr) & 0xFF00)>>8);
	else if(addr == 0x04100000+2) 		
		return (u8)((do_fiforecv_read_word(cpu, addr) & 0xFF0000)>>16);
	else if(addr == 0x04100000+3) 		
		return (u8)((do_fiforecv_read_word(cpu, addr) & 0xFF000000)>>24);

	// We should never get here
	assert(1!=0);
	return 0;
}

static u16 do_fiforecv_read_half(int cpu, u32 addr)
{
	if(addr == 0x04100000) 		
		return (u16)(do_fiforecv_read_word(cpu, addr) & 0xFFFF);
	else if(addr == 0x04100000+1) 		
		return (u16)((do_fiforecv_read_word(cpu, addr) & 0xFFFF00)>>8);
	else if(addr == 0x04100000+2) 		
		return (u16)((do_fiforecv_read_word(cpu, addr) & 0xFFFF0000)>>16);
	else if(addr == 0x04100000+3) 		
		return (u16)(((do_fiforecv_read_word(cpu, addr) & 0xFF000000)>>24) | 
		        (do_register_read_byte(cpu, 0x0410000+6) << 24));

	// We should never get here
	assert(1!=0);
	return 0;
}

static u32 do_fiforecv_read_word(int cpu, u32 addr)
{
	if(addr == 0x04100000) {
		return pop_queue(OTHER_CPU(cpu));
	}
	else if(addr == 0x4100000+1) {
		return ((do_fiforecv_read_word(cpu, 0x4100000) & 0xFFFFFF00) >> 8) | 
			   ((do_register_read_byte(cpu, 0x4100000+4) & 0xFF) << 24);
	}
	else if(addr == 0x4100000+2) {
		return ((do_fiforecv_read_word(cpu, 0x4100000) & 0xFFFF0000) >> 16) | 
			   ((do_register_read_half(cpu, 0x4100000+4) & 0xFFFF) << 16);
	}
	else if(addr == 0x4100000+3) {
		return ((do_fiforecv_read_word(cpu, 0x4100000) & 0xFF000000) >> 24) | 
			   ((do_register_read_half(cpu, 0x4100000+4) & 0xFFFF) << 16) |
			   ((do_register_read_byte(cpu, 0x4100000+6) & 0xFF) << 24) ;
	}

	// We should never get here
	assert(1!=0);
	return 0;
}

static void do_fiforecv_write_byte(int cpu, u32 addr, u8 value)
{
	(void)cpu;
	(void)addr;
	(void)value;
}

static void do_fiforecv_write_half(int cpu, u32 addr, u16 value)
{
	(void)cpu;
	(void)addr;
	(void)value;
}

static void do_fiforecv_write_word(int cpu, u32 addr, u32 value)
{
	(void)cpu;
	(void)addr;
	(void)value;
}

