/**************************************************************************
* DSemu: Memory management and handling (mmu.c)                           *
* Released under the terms of the BSD Public Licence                      *
* Imran Nazar (tf@oopsilon.com), 2004                                     *
**************************************************************************/

#include <stdio.h>
#include "vtbl.h"
#include "err.h"
#include "arm.h"
#include "dsmmusub.h"
#include "dma.h"
#include "dsioreg.h"
#include "gpu.h"
#include "unzip.h"
#include "timers.h"
#include "sram.h"
#include "arm7.h"
#include "dsint.h"
#include "dsregisters.h"

//#define MMUDEBUG
//#define BIOSsize 1024

extern u16 *VRAM;
extern u8  *VRAM8;
extern RAMWORD *EWRAM, *IWRAM, *GameROM;
RAMWORD *BIOS;

// The ARM7 has access to 64K of RAM exclusive to its own use.
RAMWORD* ARM7_IWRAM;	

extern ARMREGS arm7reg;

extern u16 OAM[512];
extern u32 LCDcolour[32768];
extern u16 GPUPAL[1024];
extern u8 *GPUPAL8;

extern ARMREGS arm7reg;

RAMWORD MMUSubEmpty;
u32 MMUSubROMsize; char *MMUSubROMfile;
u32 MMUSubBIOSsize;
extern u8* VRAMmap[];

RAMWORD DTCM[4096], ITCM[4096];

void doSPIWriteH(u16 value);
void doSPIWriteW(u32 value);
void doSPIWriteB(u8 value);
void doSPIDataWriteH(u16 value);
void doSPIDataWriteW(u32 value);
void doSPIDataWriteB(u8 value);
u16 doSPIDataReadH();


int MMUinit(char *file)
{
    //FILE *fp; u8 c; int a;

    MMUSubROMfile=file;

    MMUSubEmpty.data=0xFFFFFFFF;
    MMUSubEmpty.op=ARM7opUNL;
    MMUSubEmpty.cond=ARM7condAL;

    ARM7_IWRAM=(RAMWORD*)malloc(16384*sizeof(RAMWORD));
    if(!ARM7_IWRAM) RETFAIL("FAIL: MMU7: ARM7_IWRAM allocation.");


    RETPASS("MMU7: Memoryspace allocation complete.");
}

int MMUreset()
{
    HANDLE hFile; /*u8 c; */int a,done=0,bytesread; u8 *tempROM;
    char BIOSpath[1024], INIpath[1024], ARM7name[1024], str[256], strout[384];
    int pathsize;
    HZIP hz; ZIPENTRY ze; int cnt;
    u32 ARM7size, ARM7source=0x08000000, ARM7dest=0x02380000, ARM7start=0x02380000;

	// Reset the ARM7's own 64K RAM area to 0.
    memset(ARM7_IWRAM,0,0xFFFF);

    pathsize=strstr(logvt->file,"\\log.txt")-logvt->file;
    strncpy(BIOSpath, logvt->file, pathsize); BIOSpath[pathsize]=0;
    sprintf(INIpath, "%s\\dsemu.ini", BIOSpath);
    GetPrivateProfileString("General","BIOS","bioshack.bin",str,256,INIpath);
    sprintf(BIOSpath, "%s\\%s", BIOSpath,str);

    if(!strncmp((MMUSubROMfile+strlen(MMUSubROMfile)-3),"zip",3))
    {
        hz=OpenZipFromName(MMUSubROMfile,0);
        if(!hz) RETFAIL("FAIL: MMU7: Zip file open.");
        #ifdef MMUDEBUG
        logvt->append("MMU7: Zip file open.");
        #endif
        GetZipItem(hz,-1,&ze); cnt=ze.index; a=0;
        do {
            GetZipItem(hz,a,&ze);
            if(ze.unc_size && (strstr(ze.name,".gba") || strstr(ze.name,".bin")))
	        done=1;
	    else a++;
	} while(!done && a<cnt);
	if(a==cnt) RETFAIL("FAIL: MMU7: Zip file has no GBA binary.");

        MMUSubROMsize=ze.unc_size; strcpy(ARM7name,ze.name);
        sprintf(strout,"MMU7: Zipped ROM: %s, %d bytes.",ze.name,MMUSubROMsize);
        logvt->append(strout);
        CloseZip(hz);
//        #ifdef MMUDEBUG
        logvt->append("MMU7: Zip not read, closed.");
//        #endif
    } else {
        strcpy(ARM7name,MMUSubROMfile);
        hFile=CreateFile(MMUSubROMfile,GENERIC_READ,0,NULL,OPEN_EXISTING,0,NULL);
        if(hFile==INVALID_HANDLE_VALUE) {
            FormatMessage(
                          FORMAT_MESSAGE_FROM_SYSTEM |
                          FORMAT_MESSAGE_IGNORE_INSERTS,
                          NULL, GetLastError(), 0, (LPTSTR)str, 256, NULL);
            sprintf(strout,"FAIL: MMU7: ROM file open: %s,%s",str,MMUSubROMfile);
            RETFAIL(strout);
        }
//        #ifdef MMUDEBUG
        logvt->append("MMU7: ROM file open.");
//       #endif

        MMUSubROMsize=GetFileSize(hFile,NULL);
        sprintf(str, "MMU7: ROM size: %d bytes.",MMUSubROMsize);
        logvt->append(str);
        CloseHandle(hFile); hFile=NULL;
//        #ifdef MMUDEBUG
        logvt->append("MMU7: ROM NOT read, closed.");
//        #endif
    }

    ARM7size=MMUSubROMsize;
    if(!strncmp((ARM7name+strlen(ARM7name)-3),"nds",3))
    {
        ARM7source+=GameROM[0x0c].data;
        ARM7start=GameROM[0x0d].data;
        ARM7dest=GameROM[0x0e].data;
        ARM7size=GameROM[0x0f].data;
        sprintf(str,"MMU7: NDS: size=%08X, source=%08X, copyto=%08X, execfrom=%08X",
                ARM7size,ARM7source,ARM7dest,ARM7start);
        logvt->append(str);
        logvt->append("MMU7: About to copy ARM7 rom.");
        for(a=0;a<(int)ARM7size;a+=4)
            MMUwrW(0,ARM7dest+a,MMUrdW(0,ARM7source+a));
    } else {
        MMUwrW(0,ARM7start,0xE2900001);
        MMUwrW(0,ARM7start+4,0xEAFFFFFD);
    }
    arm7reg.r[15]=ARM7start;

	/* Load ARM7 Bios from file */
    hFile=CreateFile(BIOSpath,GENERIC_READ,0,NULL,OPEN_EXISTING,0,NULL);
    if(!hFile) RETFAIL("FAIL: MMU7: BIOS file open.");
    MMUSubBIOSsize=GetFileSize(hFile,NULL);
    BIOS=(RAMWORD*)malloc(MMUSubBIOSsize/4*sizeof(RAMWORD));
    tempROM=(u8*)malloc(MMUSubBIOSsize);
    if(!BIOS || !tempROM) RETFAIL("FAIL: MMU7: BIOS allocation.");
    ReadFile(hFile,tempROM,MMUSubBIOSsize,&bytesread,NULL);
    if(bytesread != (int)MMUSubBIOSsize)
    {
        sprintf(str,"FAIL: BIOS read after %d bytes.",bytesread);
        RETFAIL(str);
    }
    CloseHandle(hFile); hFile=NULL;
    #ifdef MMUDEBUG
    logvt->append("MMU7: BIOS read and closed.");
    #endif

    for(a=0;a<(int)MMUSubBIOSsize;a++)
    {
        #if WORDS_BIGENDIAN
        BIOS[a>>2].b[3-(a&3)]=tempROM[a];
        #else
        BIOS[a>>2].b[a&3]=tempROM[a];
        #endif
    }
    free(tempROM);
    for(a=0;a<(int)MMUSubBIOSsize/4;a++)
    {
        BIOS[a]=ARM7opDecode(BIOS[a].data);
    }

//    for(a=0;a<553;a++) VRAMmap[a]=(a&63);

    RETPASS("MMU7: Reset.");
}

void MMUfini()
{
	free(ARM7_IWRAM);
    logvt->append("MMU7: Shutdown.");
}

#define MMUintRdB(ram,addr) b=ram[(addr)>>2].b[(addr)&3]
#define MMUintRdH(ram,addr) \
    h=((addr)&1) \
        ?(((addr)&2) \
          ?(ram[(addr)>>2].b[3] | ((ram[((addr)>>2)+1].b[0])<<8)) \
          :(ram[(addr)>>2].b[1] | ((ram[(addr)>>2].b[2])<<8))) \
        :(ram[(addr)>>2].h[((addr)&2)>>1])
#define MMUintRdW(ram,addr) \
    switch((addr)&3) \
    { \
        case 0: w=ram[(addr)>>2].data; break; \
        case 1: w=(ram[(addr)>>2].data>> 8)+((ram[(addr)>>2].data&0x000000FF)<<24); break; \
        case 2: w=ram[(addr)>>2].h[1]+((ram[(addr)>>2].h[0])<<16); break; \
        case 3: w=(ram[(addr)>>2].data>>24)+((ram[(addr)>>2].data&0x00FFFFFF)<< 8); break; \
    }

u8 MMUrdB(u32 prot, u32 addr)
{
	char str[256];
    u8 b; //char str[80];
    switch(addr&0x0F000000)
    {
        case 0x00000000:
            MMUintRdB(BIOS, addr&0x00003FFF);
            break;
        case 0x02000000:
	    MMUintRdB(EWRAM, addr&0x0003FFFFF);
	    if(((addr&0xFFFFFF00)==0x02000000) ||
               ((addr&0xFFFFFF00)==0x0237F000))
            {
//                sprintf(str,"MMU7: EWRAMRd: %08X=%02X",addr,b);
//                logvt->append(str);
            }
	    ARM7addClock(2); break;
	case 0x03000000:
		if(addr&0x00800000) { 
		    MMUintRdB(ARM7_IWRAM, addr&0x0000FFFF);
		}
		else {
			MMUintRdB(IWRAM, addr&0x00007FFF);
		}
	    break;
	case 0x04000000:
		if(is_register_address(ARM7_REG, addr))
		{
			b = do_register_read_byte(ARM7_REG, addr);
		}
	    else if(prot || REG_ADDR(ARM7_REG,addr).flags&REG_FLAG_R)
            b=REG_ADDR(ARM7_REG,addr).b[addr&1];
        else 
			b=0;

#if 0
	sprintf(str,"MMU7: IORd %08X: %08X=%02X",arm7reg.r[0xf],addr,b);
	logvt->append(str);
#endif
	    break;
	case 0x05000000:
            b=GPUPAL8[addr&0x7FF];
            break;
	case 0x06000000:
		{
			u8* vram8 = VRAMmap[(addr&0x00FFFFFF)/0x4000] + (addr&0x3FFF);
			b = *vram8;
		}
	    break;
	case 0x07000000:
	    b=(addr&1)?OAM[(addr&0x7FF)>>1]>>8:OAM[(addr&0x7FF)>>1]&255;
	    #ifdef MMUDEBUG
            sprintf(str,"MMU7: OAMRd: %03X=%02X",addr&0x7FF,b);
            logvt->append(str);
            #endif
	    break;
	case 0x08000000: case 0x09000000:
	    if((addr&0x00FFFFFF)<=MMUSubROMsize)
                MMUintRdB(GameROM, addr&0x00FFFFFF);
            else b=MMUSubEmpty.b[addr&3];
	    ARM7addClock(4); break;

	case SRAM_ADDRESS:
		b = sram_read ( addr );
		break;

	default: break;
    }
    return b;
}

u16 MMUrdH(u32 prot, u32 addr)
{
	char str[256];
    u16 h; //char str[80];
    switch(addr&0x0F000000)
    {
        case 0x00000000:
            MMUintRdH(BIOS, addr&0x00003FFF);
            break;
        case 0x02000000:
	    MMUintRdH(EWRAM, addr&0x0003FFFFF);
	    if(((addr&0xFFFFFF00)==0x02000000) ||
               ((addr&0xFFFFFF00)==0x0237F000))
            {
//                sprintf(str,"MMU7: EWRAMRd: %08X=%04X",addr,h);
//                logvt->append(str);
            }
	    ARM7addClock(2); break;
	case 0x03000000:
		if(addr&0x00800000) {
		    MMUintRdH(ARM7_IWRAM, addr&0x0000FFFF);
		}
		else {
		    MMUintRdH(IWRAM, addr&0x00007FFF);
		}
	    break;
	case 0x04000000:
		if(is_register_address(ARM7_REG, addr))
		{
			h = do_register_read_half(ARM7_REG, addr);
		}
		else if((addr&0x1ffe)==0x1c2) {
			h=doSPIDataReadH();
		}
	    else if(prot || REG_ADDR(ARM7_REG,addr).flags&REG_FLAG_R)
            h=REG_ADDR(ARM7_REG,addr).data;
        else 
			h=0;
#if 0
	sprintf(str,"MMU7: IORd %08X: %08X=%04X",arm7reg.r[0xf],addr,h);
	logvt->append(str);
#endif
	    break;
	case 0x05000000:
	    h=GPUPAL[(addr>>1)&0x3FF];
            break;
	case 0x06000000:
		{
			u8* vram8 = VRAMmap[(addr&0x00FFFFFF)/0x4000] + (addr&0x3FFF);
			u16* vram = (u16*)vram8;
			h = *vram;
		}
	    break;
	case 0x07000000:
	    h=OAM[(addr&0x7FF)>>1];
	    #ifdef MMUDEBUG
            sprintf(str,"MMU7: OAMRd: %03X=%04X",addr&0x7FF,h);
            logvt->append(str);
            #endif
	    break;
	case 0x08000000: case 0x09000000:
	    if((addr&0x00FFFFFF)<=MMUSubROMsize)
                MMUintRdH(GameROM, addr&0x00FFFFFF);
            else h=MMUSubEmpty.h[(addr&2)>>1];
	    ARM7addClock(4); break;
	default: break;
    }
    return h;
}

u32 MMUrdW(u32 prot, u32 addr)
{
	char str[256];
    u32 w; u16 hl,hh; //char str[80];
    switch(addr&0x0F000000)
    {
        case 0x00000000:
            MMUintRdW(BIOS, addr&0x00003FFF);
            break;
        case 0x02000000:
	    MMUintRdW(EWRAM, addr&0x0003FFFFF);
	    if(((addr&0xFFFFFF00)==0x02000000) ||
               ((addr&0xFFFFFF00)==0x0237F000))
            {
//                sprintf(str,"MMU7: EWRAMRd: %08X=%08X",addr,w);
//                logvt->append(str);
            }
	    ARM7addClock(5); break;
	case 0x03000000:
		if(addr&0x00800000) {
			MMUintRdW(ARM7_IWRAM, addr&0x0000FFFF);
		}
		else  {
			MMUintRdW(IWRAM, addr&0x00007FFF);
		}
	    break;
	case 0x04000000:
		if(is_register_address(ARM7_REG, addr))
		{
			w = do_register_read_word(ARM7_REG, addr);
		}
		else {
		    if(prot || REG_ADDR(ARM7_REG,addr).flags&REG_FLAG_R)
                hl=REG_ADDR(ARM7_REG,addr).data;
            else 
				hl=0;
			addr+=2;
			if(prot || REG_ADDR(ARM7_REG,addr).flags&REG_FLAG_R)
                hh=REG_ADDR(ARM7_REG,addr).data;
            else 
				hh=0;
            w=hl+(hh<<16);
		}
#if 0
	sprintf(str,"MMU7: IORd %08X: %08X=%08X",arm7reg.r[0xf],(addr-2),w);
	logvt->append(str);
#endif
	    break;
	case 0x05000000:
	    hl=GPUPAL[(addr>>1)&0x3FF];
            addr+=2;
	    hh=GPUPAL[(addr>>1)&0x3FF];
            w=hl+(hh<<16);
            break;
	case 0x06000000:
		{
			u8* vram8 = VRAMmap[(addr&0x00FFFFFF)/0x4000] + (addr&0x3FFF);
			u32* vram = (u32*)vram8;
			w = *vram;
		}
        break;
	case 0x07000000:
	    hl=OAM[(addr&0x7FF)>>1]; addr+=2;
	    hh=OAM[(addr&0x7FF)>>1];
            w=hl+(hh<<16);
	    #ifdef MMUDEBUG
            sprintf(str,"MMU7: OAMRd: %03X=%08X",addr&0x7FF,w);
            logvt->append(str);
            #endif
            break;
	case 0x08000000: case 0x09000000:
//            sprintf(str,"MMU7: RdW at %08X",addr); logvt->append(str);
	    if((addr&0x00FFFFFF)<=MMUSubROMsize){ MMUintRdW(GameROM, addr&0x00FFFFFF); }
            else w=MMUSubEmpty.data;
	    ARM7addClock(7); break;
	default: break;
    }
    return w;
}

RAMWORD MMUrdS(u32 prot, u32 addr)
{
    if((addr&0xFE000000)==0x08000000)
    {
        if((addr&0x00FFFFFF)<=MMUSubROMsize) return GameROM[(addr&0xFFFFFF)>>2];
        else return MMUSubEmpty;
    }
    switch(addr&0x0F000000)
    {
        case 0x03000000: 
			if(addr&0x00800000) 
				return ARM7_IWRAM[(addr&0x0FFFF)>>2];
			else
				return IWRAM[(addr&0x07FFF)>>2];

        case 0x02000000: return EWRAM[(addr&0x3FFFFF)>>2];
        case 0x00000000: 
            return BIOS[(addr&0x1ff)>>2];
    }
    return MMUSubEmpty;
}
#define MMUintWrB(ram,addr,val) \
    ram[(addr)>>2].b[(addr)&3]=val; \
    ram[(addr)>>2].op=ARM7opUNL; \
    ram[(addr)>>2].cond=ARM7condAL

#define MMUintWrH(ram,addr,val) \
    ram[(addr)>>2].h[((addr)&2)>>1]=val; \
    ram[(addr)>>2].op=ARM7opUNL; \
    ram[(addr)>>2].cond=ARM7condAL

#define MMUintWrW(ram,addr,val) \
    ram[(addr)>>2].data=val; \
    ram[(addr)>>2].op=ARM7opUNL; \
    ram[(addr)>>2].cond=ARM7condAL

void MMUwrB(u32 bus, u32 addr, u8 val)
{
	char str[256];
//    char str[80];
    switch(addr&0x0F000000)
    {
        case 0x00000000:
            if(addr&0x00800000) DTCM[(addr&0x3FFF)>>2].b[addr&3]=val;
            else MMUintWrB(ITCM,addr&0x0FFF,val);
            break;
        case 0x02000000:
	    MMUintWrB(EWRAM, addr&0x0003FFFFF, val);
	    ARM7addClock(2); break;
	case 0x03000000:
		if(addr&0x0800000) {
			MMUintWrB(ARM7_IWRAM, addr&0x0000FFFF, val);
		}
		else {
			MMUintWrB(IWRAM, addr&0x00007FFF, val);
		}
	    break;
        case 0x04000000:
			if(is_register_address(ARM7_REG, addr))
			{
				do_register_write_byte(ARM7_REG, addr, val);
			}
            else if(REG_ADDR(ARM7_REG,addr).flags&REG_FLAG_W)
            {
				// The POWERCNT register for ARM7 can't write to bit 15.
				if((addr&0x1ffe)==0x304) 
					val &= ~(1 << 15);
				// Process a write to the serial control registers
				else if((addr&0x1ffe) == 0x1c0)
					doSPIWriteB(val);
				else if((addr&0x1ffe) == 0x1c2)
					doSPIDataWriteB(val);
				else if((addr&0x1ffe)==0x100)
					DoTimerReload(ARM7_REG, 0, val);
				else if((addr&0x1ffe)==0x102)
					DoTimerControl(ARM7_REG, 0, val);
				else if((addr&0x1ffe)==0x104)
					DoTimerReload(ARM7_REG, 1, val);
				else if((addr&0x1ffe)==0x106)
					DoTimerControl(ARM7_REG, 1, val);
				else if((addr&0x1ffe)==0x108)
					DoTimerReload(ARM7_REG, 2, val);
				else if((addr&0x1ffe)==0x10A)
					DoTimerControl(ARM7_REG, 2, val);
				else if((addr&0x1ffe)==0x10B)
					DoTimerReload(ARM7_REG, 3, val);
				else if((addr&0x1ffe)==0x10E)
					DoTimerControl(ARM7_REG, 3, val);
                else if((addr&0x1FFE)==0x214) 
					Int7Clear(val);
                else 
					REG_ADDR(ARM7_REG,addr).b[addr&1]=val;
			}
            DMAcheck(DMA_TIMENOW);
#if 0
	sprintf(str,"MMU7: IOWr %08X: %08X=%02X",arm7reg.r[0xf],addr,val);
    logvt->append(str);
#endif
			break;
	
	case SRAM_ADDRESS:
		sram_write ( addr, val );
		break;
	
	default: break;
    }
}

void MMUwrH(u32 bus, u32 addr, u16 val)
{
    char str[80];
    switch(addr&0x0F000000)
    {
        case 0x00000000:
            if(addr&0x00800000) DTCM[(addr&0x3FFF)>>2].h[(addr&2)>>1]=val;
            else MMUintWrH(ITCM,addr&0x0FFF,val);
            break;
        case 0x02000000:
	    MMUintWrH(EWRAM, addr&0x0003FFFFF, val);
	    ARM7addClock(2); break;
	case 0x03000000:
		if(addr&0x0800000) {
			MMUintWrH(ARM7_IWRAM, addr&0x0000FFFF, val);
		}
		else {
			MMUintWrH(IWRAM, addr&0x00007FFF, val);
		}
	    break;
	case 0x04000000:
		if(addr == 0x4000204 || addr == 0x4000247 || addr >= 0x4800000) {
           sprintf(str,"MMU7: IOWr %08X: %03X=%04X",arm7reg.r[0xf],addr&0x1FFF,val);
           logvt->append(str);
		}
		if(is_register_address(ARM7_REG, addr))
		{
			do_register_write_half(ARM7_REG, addr, val);
		}
        else if(REG_ADDR(ARM7_REG,addr).flags&REG_FLAG_W)
        {
			// The POWERCNT register for ARM7 can't write to bit 15.
			if((addr&0x1ffe)==0x304) 
				val &= ~(1 << 15);
			// Process a write to the serial control registers
			else if((addr&0x1ffe) == 0x1c0)
				doSPIWriteH(val);
			else if((addr&0x1ffe) == 0x1c2)
				doSPIDataWriteH(val);
			else if((addr&0x1ffe)==0x100)
				DoTimerReload(ARM7_REG, 0, val);
			else if((addr&0x1ffe)==0x102)
				DoTimerControl(ARM7_REG, 0, val);
			else if((addr&0x1ffe)==0x104)
				DoTimerReload(ARM7_REG, 1, val);
			else if((addr&0x1ffe)==0x106)
				DoTimerControl(ARM7_REG, 1, val);
			else if((addr&0x1ffe)==0x108)
				DoTimerReload(ARM7_REG, 2, val);
			else if((addr&0x1ffe)==0x10A)
				DoTimerControl(ARM7_REG, 2, val);
			else if((addr&0x1ffe)==0x10B)
				DoTimerReload(ARM7_REG, 3, val);
			else if((addr&0x1ffe)==0x10E)
				DoTimerControl(ARM7_REG, 3, val);
            else if((addr&0x1FFE)==0x214) 
				Int7Clear(val);
			else if((addr&0x1ffe)==0x180) {
				// TODO: Handle enable and interrupt bits
				int current7 = REG(ARM7_REG,IPCSYNC);
				int current9 = REG(ARM9_REG,IPCSYNC);
				REG(ARM7_REG,IPCSYNC)=(current7&0x6000)|(val&0xf00)|(current7&0xf);
				REG(ARM9_REG,IPCSYNC)=(current9&0x6f00)|(val>>8)&0xf;
			}
            else 
				REG_ADDR(ARM7_REG,addr).data=val;
		}
        DMAcheck(DMA_TIMENOW);
#if 0
	sprintf(str,"MMU7: IOWr %08X: %08X=%04X",arm7reg.r[0xf],addr,val);
    logvt->append(str);
#endif
	    break;
	case 0x05000000:
	    #ifdef MMUDEBUG
	    sprintf(str,"MMU7: PalWr: %08X=%04X",addr,val);
	    logvt->append(str);
	    #endif
	    GPUPAL[(addr>>1)&0x3FF]=val;
	    break;
	case 0x06000000:
		{
			u8* vram8 = VRAMmap[(addr&0x00FFFFFF)/0x4000] + (addr&0x3FFF);
			*vram8 = (u8)val;
		}
		break;
	case 0x07000000:
	    #ifdef MMUDEBUG
            sprintf(str,"MMU7: OAMWr: %03X=%04X",addr&0x7FF,val);
            logvt->append(str);
            #endif
	    OAM[(addr&0x7FF)>>1]=val;
            break;
	default: break;
    }
}

void MMUwrW(u32 bus, u32 addr, u32 val)
{
    char str[80];
    switch(addr&0x0F000000)
    {
        case 0x00000000:
            if(addr&0x00800000) DTCM[(addr&0x3FFF)>>2].data=val;
            else MMUintWrW(ITCM,addr&0x0FFF,val);
            break;
        case 0x02000000:
	    MMUintWrW(EWRAM, addr&0x0003FFFFF, val);
	    ARM7addClock(2); break;
	case 0x03000000:
		if(addr&0x0800000) {
			MMUintWrW(ARM7_IWRAM, addr&0x0000FFFF, val);
		}
		else {
			MMUintWrW(IWRAM, addr&0x00007FFF, val);
		}
	    break;
	case 0x04000000:
		if(is_register_address(ARM7_REG, addr))
		{
			do_register_write_word(ARM7_REG, addr, val);
		}
		else {
            if(REG_ADDR(ARM7_REG,addr).flags&REG_FLAG_W)
            {
				// The POWERCNT register for ARM7 can't write to bit 15.
				if((addr&0x1ffe)==0x304) 
					val &= ~(1 << 15);
				// Process a write to the serial control registers
				else if((addr&0x1ffe) == 0x1c0)
					doSPIWriteW(val);
				else if((addr&0x1ffe) == 0x1c2)
					doSPIDataWriteW(val);
				else if((addr&0x1ffe)==0x100)
					DoTimerReload(ARM7_REG, 0, val);
				else if((addr&0x1ffe)==0x102)
					DoTimerControl(ARM7_REG, 0, val);
				else if((addr&0x1ffe)==0x104)
					DoTimerReload(ARM7_REG, 1, val);
				else if((addr&0x1ffe)==0x106)
					DoTimerControl(ARM7_REG, 1, val);
				else if((addr&0x1ffe)==0x108)
					DoTimerReload(ARM7_REG, 2, val);
				else if((addr&0x1ffe)==0x10A)
					DoTimerControl(ARM7_REG, 2, val);
				else if((addr&0x1ffe)==0x10B)
					DoTimerReload(ARM7_REG, 3, val);
				else if((addr&0x1ffe)==0x10E)
					DoTimerControl(ARM7_REG, 3, val);
                else if((addr&0x1FFE)==0x214) 
					Int7Clear(val&65535);
                else 
					REG_ADDR(ARM7_REG,addr).data=val&65535;
            }
		    addr+=2;
            if(REG_ADDR(ARM7_REG,addr).flags&REG_FLAG_W)
            {
				// The POWERCNT register for ARM7 can't write to bit 15.
				if((addr&0x1ffe)==0x304) 
					val &= ~(1 << 15);
                if((addr&0x1FFE)==0x214) 
					Int7Clear(val>>16);
                else 
					REG_ADDR(ARM7_REG,addr).data=val>>16;
            }
		}
        DMAcheck(DMA_TIMENOW);
#if 0
	sprintf(str,"MMU7: IOWr %08X: %08X=%08X",arm7reg.r[0xf],addr,val);
    logvt->append(str);
#endif
	    break;
	case 0x05000000:
	    sprintf(str,"MMU7: PalWr: %08X=%08X",addr,val);
	    logvt->append(str);
	    GPUPAL[(addr>>1)&0x3FF]=val&65535;
	    addr+=2;
	    GPUPAL[(addr>>1)&0x3FF]=val>>16;
		break;

	case 0x06000000:
		{
			u8* vram8 = VRAMmap[(addr&0x00FFFFFF)/0x4000] + (addr&0x3FFF);
			u16* vram = (u16*)vram8;
			*vram = val;
		}
	    break;
	case 0x07000000:
	    #ifdef MMUDEBUG
            sprintf(str,"MMU7: OAMWr: %03X=%08X",addr&0x7FF,val);
            logvt->append(str);
            #endif
	    OAM[(addr&0x7FF)>>1]=val&65535; addr+=2;
	    OAM[(addr&0x7FF)>>1]=val>>16;
	    break;
	default: break;
    }
}

void MMUwrS(u32 bus, u32 addr, RAMWORD val)
{
    switch(addr&0x0F000000)
    {
        case 0x00000000: if(!(addr&0x00800000)) ITCM[(addr&0x0FFF)>>2]=val; break;
        case 0x02000000: EWRAM[(addr&0x3FFFFF)>>2]=val; break;
        case 0x03000000: 
		if(addr&0x0800000) 
			ARM7_IWRAM[(addr&0x0FFFF)>>2]=val;
		else
			IWRAM[(addr&0x07FFF)>>2]=val;
	    break;

	case 0x08000000:
	case 0x09000000:
	case 0x0A000000:
	case 0x0B000000:
	case 0x0C000000:
        case 0x0D000000: GameROM[(addr&0xFFFFFF)>>2]=val; break;
	default: break;
    }
}


// The SPI routines below are currently geared towards handling the
// touchscreen only, and in the way that libnds handles retrieving the
// touchscreen values. Hopefully this will be improved in time.
static u32 spi_output_byte = 0;
static int spi_last_command = 0;
static int spi_touchscreen_command = 0;
int spi_touch_x = 0;
int spi_touch_y = 0;

int spi_touch_first(int mini, int maxi, int top, int value) {
	int byte1 = value >> 5;
	return byte1;
}

int spi_touch_second(int mini, int maxi, int top, int value) {
	int byte2 = (value & 0x1f) << 3;
	return byte2;
}

#if 0
int spi_touch_first(int mini, int maxi, int top, int value) {
	int full_value = (float)(maxi-mini)*((float)value/(float)top);
	int byte1 = full_value >> 5;
	return byte1;
}

int spi_touch_second(int mini, int maxi, int top, int value) {
	int full_value = (float)(maxi-mini)*((float)value/(float)top);
	int byte2 = (full_value & 0x1f) << 3;
	return byte2;
}
#endif

void doSPIWriteH(u16 value)
{
	spi_last_command = value;
	switch(value) {
	case 0x8A01:
		// Reading the touchscreen data
		REG(ARM7_REG, SERIALCNT) = value;
		REG(ARM7_REG, SERIALCNT) &= ~0x80;
		break;

	case 0x8201:
		REG(ARM7_REG, SERIALCNT) = value;
		REG(ARM7_REG, SERIALCNT) &= ~0x80;
		break;
		
	}
}

void doSPIDataWriteH(u16 value) 
{
	REG(ARM7_REG, SERIALCNT) |= 0x80;

	switch(spi_last_command) {
	case 0x8A01:
		spi_touchscreen_command = value;
		// Write the touchscreen command
		switch(value) {
		case 0xD0:
		case 0xD1:
		case 0xD2:
		case 0xD3:
		case 0xD4:
		case 0xD5:
			// Touchscreen X
			spi_output_byte = spi_touch_first(300,3760,256,spi_touch_x);
			break;

		case 0x90:
		case 0x91:
		case 0x92:
		case 0x93:
		case 0x94:
		case 0x95:
			// Touchscreen Y
			spi_output_byte = spi_touch_first(230,3850,192,spi_touch_y);
			break;

		default:
			spi_output_byte = 0x00;
			break;
		}
		break;

	case 0x8201:
		switch(spi_touchscreen_command) {
		case 0xD0:
		case 0xD1:
		case 0xD2:
		case 0xD3:
		case 0xD4:
		case 0xD5:
			// Touchscreen X
			spi_output_byte = spi_touch_second(300,3760,256,spi_touch_x);
			break;

		case 0x90:
		case 0x91:
		case 0x92:
		case 0x93:
		case 0x94:
		case 0x95:
			// Touchscreen Y
			spi_output_byte = spi_touch_second(230,3850,192,spi_touch_y);
			break;

		default:
			spi_output_byte = 0x00;
			break;
		}
		spi_touchscreen_command = 0;
		break;
	}
	spi_last_command = 0;
	REG(ARM7_REG, SERIALCNT) &= ~0x80;
}

u16 doSPIDataReadH() {
	u16 value = spi_output_byte;
	spi_output_byte = 0;
	return value;
}

u16 doSPIDataReadB(u16 value) {
	return 0;
}

u16 doSPIDataReadW(u16 value) {
	return 0;
}


void doSPIWriteW(u32 value)
{
}

void doSPIWriteB(u8 value)
{
}

void doSPIDataWriteW(u32 value)
{
}

void doSPIDataWriteB(u8 value)
{
}
