/**************************************************************************
* DSemu: Memory management and handling (mmu.c)                           *
* Released under the terms of the BSD Public Licence                      *
* Imran Nazar (tf@oopsilon.com), 2004                                     *
**************************************************************************/

#include <stdio.h>
#include "vtbl.h"
#include "arm.h"
#include "err.h"
#include "dsmmumain.h"
#include "dma.h"
#include "dsioreg.h"
#include "gpu.h"
#include "unzip.h"
#include "timers.h"
#include "sram.h"
#include "arm9.h"
#include "dsint.h"
#include "dsregisters.h"

//#define MMUMainDEBUG
//#define BIOSsize 1024

//#define DEBUG_IO

extern u16 *VRAM;
extern u8  *VRAM8;
extern RAMWORD *EWRAM, *IWRAM;
RAMWORD *GameROM, *BIOS9;

extern ARMREGS arm9reg;

extern u16 OAM[512];
extern u32 LCDcolour[32768];
extern u16 GPUPAL[1024];
extern u8 *GPUPAL8;

extern u8 *VRAM_BANK_A;
extern u8 *VRAM_BANK_B;
extern u8 *VRAM_BANK_C;
extern u8 *VRAM_BANK_D;
extern u8 *VRAM_BANK_E;
extern u8 *VRAM_BANK_F;
extern u8 *VRAM_BANK_G;
extern u8 *VRAM_BANK_H;
extern u8 *VRAM_BANK_I;

RAMWORD MMUMainEmpty;
u32 MMUMainROMsize; char *MMUMainROMfile;
u32 MMUMainBIOSsize;

// Each entry in the map is for 16Kb of physical memory starting at
// 0x06000000. Each entry in the map identifies the VRAM bank that is
// mapped to that address and the start of the 16Kb boundary within that
// VRAM bank.
u8* VRAMmap[553];

RAMWORD DTCM[4096], ITCM[4096];

int MMUMaininit(char *file)
{
//    FILE *fp; u8 c; int a;

    MMUMainROMfile=file;

    EWRAM=(RAMWORD*)malloc(1048576*sizeof(RAMWORD));
    if(!EWRAM) RETFAIL("FAIL: MMU9: EWRAM allocation.");

    IWRAM=(RAMWORD*)malloc(8192*sizeof(RAMWORD));
    if(!IWRAM) RETFAIL("FAIL: MMU9: IWRAM allocation.");

    MMUMainEmpty.data=0xFFFFFFFF;
    MMUMainEmpty.op=ARM9opUNL;
    MMUMainEmpty.cond=ARM9condAL;

    RETPASS("MMU9: Memoryspace allocation complete.");
}

int MMUMainreset()
{
    HANDLE hFile; /*u8 c;*/ int a,done=0,bytesread; u8 *tempROM;
    char BIOSpath[1024], INIpath[1024], ARM9name[1024], str[256], strout[384];
    int pathsize;
    HZIP hz; ZIPENTRY ze; int cnt;
    u32 ARM9size, ARM9source=0x08000000, ARM9dest=0x02004000, ARM9start=0x02004000;

    logvt->append("MMU9: Resetting.");

//    memset(EWRAM,0,1048576*sizeof(RAMWORD));
    memset(EWRAM,0,0x3FFFFF);
	EWRAM[(0x027FF035&0x3FFFFF)>>2].b[0x027FF035&3]=0;
	memset(IWRAM,0,8192*sizeof(RAMWORD));
    memset(DTCM,0,4096*sizeof(RAMWORD));
    memset(ITCM,0,4096*sizeof(RAMWORD));
    logvt->append("MMU9: Memory zeroed.");

    pathsize=strstr(logvt->file,"\\log.txt")-logvt->file;
    strncpy(BIOSpath, logvt->file, pathsize); BIOSpath[pathsize]=0;
    sprintf(INIpath, "%s\\dsemu.ini", BIOSpath);
    GetPrivateProfileString("General","BIOS9","bioshack9.bin",str,256,INIpath);
    sprintf(BIOSpath, "%s\\%s", BIOSpath,str);

    if(!strncmp((MMUMainROMfile+strlen(MMUMainROMfile)-3),"zip",3))
    {
        hz=OpenZipFromName(MMUMainROMfile,0);
        if(!hz) RETFAIL("FAIL: MMU9: Zip file open.");
        #ifdef MMUMainDEBUG
        logvt->append("MMU9: Zip file open.");
        #endif
        GetZipItem(hz,-1,&ze); cnt=ze.index; a=0;
        do {
            GetZipItem(hz,a,&ze);
            if(ze.unc_size && (strstr(ze.name,".gba") || strstr(ze.name,".bin")))
	        done=1;
	    else a++;
	} while(!done && a<cnt);
	if(a==cnt) RETFAIL("FAIL: MMU9: Zip file has no GBA binary.");

        MMUMainROMsize=ze.unc_size; strcpy(ARM9name,ze.name);
        sprintf(strout,"MMU9: Zipped ROM: %s, %d bytes.",ze.name,MMUMainROMsize);
        logvt->append(strout);
        GameROM=(RAMWORD*)malloc((MMUMainROMsize/4)*sizeof(RAMWORD));
        tempROM=(u8*)malloc(MMUMainROMsize);
        if(!GameROM || !tempROM) RETFAIL("FAIL: MMU9: GameROM allocation.");
        #ifdef MMUMainDEBUG
        logvt->append("MMU9: GameROM allocated.");
        #endif

        a=UnzipItemToBlock(hz,a,tempROM,MMUMainROMsize); if(a && a!=ZR_MORE)
        {
	    FormatZipMessage(a,str,256);
	    sprintf(strout,"FAIL: MMU9: Zip read: %s.",str);
	    RETFAIL(strout);
	}

        CloseZip(hz);
        #ifdef MMUMainDEBUG
        logvt->append("MMU9: Zip read and closed.");
        #endif
    } else {
        strcpy(ARM9name,MMUMainROMfile);
        logvt->append("MMU9: Opening ROM file.");
        hFile=CreateFile(MMUMainROMfile,GENERIC_READ,0,NULL,OPEN_EXISTING,0,NULL);
        if(hFile==INVALID_HANDLE_VALUE) {
            FormatMessage(
                          FORMAT_MESSAGE_FROM_SYSTEM |
                          FORMAT_MESSAGE_IGNORE_INSERTS,
                          NULL, GetLastError(), 0, (LPTSTR)str, 256, NULL);
            sprintf(strout,"FAIL: MMU9: ROM file open: %s, %s.",str,MMUMainROMfile);
            RETFAIL(strout);
        }
        //#ifdef MMUMainDEBUG
        logvt->append("MMU9: ROM file open.");
        //#endif

        MMUMainROMsize=GetFileSize(hFile,NULL);
        sprintf(str, "MMU9: ROM size: %d bytes.",MMUMainROMsize);
        logvt->append(str);
        GameROM=(RAMWORD*)malloc(((MMUMainROMsize/4)+1)*sizeof(RAMWORD));
        tempROM=(u8*)malloc(MMUMainROMsize);
        if(!GameROM || !tempROM) RETFAIL("FAIL: MMU9: GameROM allocation.");
        //#ifdef MMUMainDEBUG
        logvt->append("MMU9: GameROM allocated.");
        //#endif

        ReadFile(hFile,tempROM,MMUMainROMsize,&bytesread,NULL);
        if((u32)bytesread != MMUMainROMsize) RETFAIL("FAIL: ROM read.");
        CloseHandle(hFile); hFile=NULL;
        //#ifdef MMUMainDEBUG
        logvt->append("MMU9: ROM read and closed.");
        //#endif
    }

    for(a=0;a<(int)MMUMainROMsize;a++)
    {
        #if WORDS_BIGENDIAN
        GameROM[a>>2].b[3-(a&3)]=tempROM[a];
        #else
        GameROM[a>>2].b[a&3]=tempROM[a];
        #endif
    }
    free(tempROM);
    for(a=0;a<((int)MMUMainROMsize>>2);a++)
    {
        GameROM[a]=ARM9opDecode(GameROM[a].data);
    }

    ARM9size=MMUMainROMsize;
    if(!strncmp((ARM9name+strlen(ARM9name)-3),"nds",3))
    {
        ARM9source+=GameROM[8].data;
        ARM9start=GameROM[9].data;
        ARM9dest=GameROM[10].data;
        ARM9size=GameROM[11].data;
    }
    sprintf(str,"MMU9: NDS: size=%08X, source=%08X, copyto=%08X, execfrom=%08X",
            ARM9size,ARM9source,ARM9dest,ARM9start);
    logvt->append(str);
    logvt->append("MMU9: About to copy ARM9 rom.");
    for(a=0;a<(int)ARM9size;a+=4)
        MMUMainwrW(0,ARM9dest+a,MMUMainrdW(0,ARM9source+a));
    arm9reg.r[15]=ARM9start;

    hFile=CreateFile(BIOSpath,GENERIC_READ,0,NULL,OPEN_EXISTING,0,NULL);
    if(!hFile) RETFAIL("FAIL: MMU9: BIOS file open.");
    MMUMainBIOSsize=GetFileSize(hFile,NULL);
    BIOS9=(RAMWORD*)malloc(MMUMainBIOSsize/4*sizeof(RAMWORD));
    tempROM=(u8*)malloc(MMUMainBIOSsize);
    if(!BIOS9 || !tempROM) RETFAIL("FAIL: MMU9: BIOS allocation.");
    ReadFile(hFile,tempROM,MMUMainBIOSsize,&bytesread,NULL);
    if(bytesread != (int)MMUMainBIOSsize)
    {
        sprintf(str,"FAIL: BIOS read after %d bytes.",bytesread);
        RETFAIL(str);
    }
    CloseHandle(hFile); hFile=NULL;
    #ifdef MMUMainDEBUG
    logvt->append("MMU9: BIOS read and closed.");
    #endif

    for(a=0;a<(int)MMUMainBIOSsize;a++)
    {
        #if WORDS_BIGENDIAN
        BIOS9[a>>2].b[3-(a&3)]=tempROM[a];
        #else
        BIOS9[a>>2].b[a&3]=tempROM[a];
        #endif
    }
    free(tempROM);
    for(a=0;a<(int)MMUMainBIOSsize/4;a++)
    {
        BIOS9[a]=ARM9opDecode(BIOS9[a].data);
    }

    for(a=0;a<(sizeof(VRAMmap)/sizeof(u8*));a++) 
		VRAMmap[a]=0;

    RETPASS("MMU9: Reset.");
}

void MMUMainfini()
{
    free(EWRAM);
    free(IWRAM);
    free(GameROM);
    logvt->append("MMU9: Shutdown.");
}

void MMUMainremapVRAM_BANK_A(int master, int offset)
{
	int i = 0;
	int physical_start = 0;
	int physical_end = 0;
	u8* vram_start = VRAM_BANK_A;

	switch(master) {
	case 0:		// LCDC
		physical_start = 0x800000;
		physical_end =   0x820000;
		break;

	case 1:	   // A-BG
		physical_start = 0x000000 + 0x20000 * offset;		
		physical_end = physical_start + 0x20000;
		break;

	case 2:	   // A-OBJ
		physical_start = 0x400000 + 0x20000 * offset;
		physical_end =   physical_start + 0x20000;
		break;

	case 3:		// TEXIMG
		// TODO: What to do here?
		break;
	}

	for(i = physical_start / 0x4000; 
	    i < (physical_end/0x4000); 
		++i, vram_start+=0x4000) {
		VRAMmap[i] = vram_start;
	}
}

void MMUMainremapVRAM_BANK_B(int master, int offset)
{
	int i = 0;
	int physical_start = 0;
	int physical_end = 0;
	u8* vram_start = VRAM_BANK_B;

	switch(master) {
	case 0:		// LCDC
		physical_start = 0x820000;
		physical_end =   0x840000;
		break;

	case 1:	   // A-BG
		physical_start = 0x000000 + 0x20000 * offset;
		physical_end =   physical_start + 0x20000;
		break;

	case 2:	   // A-OBJ
		physical_start = 0x400000 + 0x20000 * offset;
		physical_end =   physical_start + 0x20000;
		break;

	case 3:		// TEXIMG
		// TODO: What to do here?
		break;
	}

	for(i = physical_start / 0x4000; 
	    i < (physical_end/0x4000); 
		++i, vram_start+=0x4000) {
		VRAMmap[i] = vram_start;
	}
}

void MMUMainremapVRAM_BANK_C(int master, int offset)
{
	int i = 0;
	int physical_start = 0;
	int physical_end = 0;
	u8* vram_start = VRAM_BANK_C;

	switch(master) {
	case 0:		// LCDC
		physical_start = 0x840000;
		physical_end =   0x860000;
		break;

	case 1:	   // A-BG
		physical_start = 0x000000 + 0x20000 * offset;
		physical_end =   physical_start + 0x20000;
		break;

	case 2:	   // ARM7
		physical_start = 0x000000 + 0x20000 * offset;
		physical_end =   physical_start + 0x20000;
		break;

	case 3:		// TEXIMG
		// TODO: What to do here?
		break;

	case 4:		// B-BG
		physical_start = 0x200000;
		physical_end =   0x220000;
		break;

	}

	for(i = physical_start / 0x4000; 
	    i < (physical_end/0x4000); 
		++i, vram_start+=0x4000) {
		VRAMmap[i] = vram_start;
	}
}

void MMUMainremapVRAM_BANK_D(int master, int offset)
{
	int i = 0;
	int physical_start = 0;
	int physical_end = 0;
	u8* vram_start = VRAM_BANK_D;

	switch(master) {
	case 0:		// LCDC
		physical_start = 0x860000;
		physical_end =   0x880000;
		break;

	case 1:	   // A-BG
		physical_start = 0x000000 + 0x20000 * offset;
		physical_end =   physical_start + 0x20000;
		break;

	case 2:	   // ARM7
		physical_start = 0x000000 + 0x20000 * offset;
		physical_end =   physical_start + 0x20000;
		break;

	case 3:		// TEXIMG
		// TODO: What to do here?
		break;

	case 4:		// B-BG
		physical_start = 0x600000;
		physical_end =   0x620000;
		break;

	}

	for(i = physical_start / 0x4000; 
	    i < (physical_end/0x4000); 
		++i, vram_start+=0x4000) {
		VRAMmap[i] = vram_start;
	}
}

void MMUMainremapVRAM_BANK_E(int master, int offset)
{
	int i = 0;
	int physical_start = 0;
	int physical_end = 0;
	u8* vram_start = VRAM_BANK_E;

	switch(master) {
	case 0:		// LCDC
		physical_start = 0x880000;
		physical_end =   0x890000;
		break;

	case 1:	   // A-BG
		physical_start = 0x000000;
		physical_end =   0x010000;
		break;

	case 2:	   // A-OBJ
		physical_start = 0x400000;
		physical_end =   0x410000;
		break;

	case 3:		// TEXPAX
		// TODO: What to do here?
		break;

	case 4:		// BGEXTPAL
		// TODO: What to do here?
		break;

	}

	for(i = physical_start / 0x4000; 
	    i < (physical_end/0x4000); 
		++i, vram_start+=0x4000) {
		VRAMmap[i] = vram_start;
	}
}

void MMUMainremapVRAM_BANK_F(int master, int offset)
{
	int i = 0;
	int physical_start = 0;
	int physical_end = 0;
	u8* vram_start = VRAM_BANK_F;

	switch(master) {
	case 0:		// LCDC
		physical_start = 0x890000;
		physical_end =   0x894000;
		break;

	case 1:	   // A-BG
		physical_start = 0x000000 + 0x4000 * offset;
		physical_end =   physical_start + 0x4000;
		break;

	case 2:	   // A-OBJ
		physical_start = 0x000000 + 0x4000 * offset;
		physical_end =   physical_start + 0x4000;
		break;

	case 3:		// TEXPAL
		// TODO: What to do here?
		break;

	case 4:		// BGEXTPAL
		// TODO: What to do here?
		break;

	case 5:		// OBJEXTPAL
		// TODO: What to do here?
		break;

	}

	for(i = physical_start / 0x4000; 
	    i < (physical_end/0x4000); 
		++i, vram_start+=0x4000) {
		VRAMmap[i] = vram_start;
	}
}

void MMUMainremapVRAM_BANK_G(int master, int offset)
{
	int i = 0;
	int physical_start = 0;
	int physical_end = 0;
	u8* vram_start = VRAM_BANK_G;

	switch(master) {
	case 0:		// LCDC
		physical_start = 0x894000;
		physical_end =   0x898000;
		break;

	case 1:	   // A-BG
		physical_start = 0x000000 + 0x4000 * offset;
		physical_end =   physical_start + 0x4000;
		break;

	case 2:	   // A-OBJ
		physical_start = 0x000000 + 0x4000 * offset;
		physical_end =   physical_start + 0x4000;
		break;

	case 3:		// TEXPAL
		// TODO: What to do here?
		break;

	case 4:		// BGEXTPAL
		// TODO: What to do here?
		break;

	case 5:		// OBJEXTPAL
		// TODO: What to do here?
		break;

	}

	for(i = physical_start / 0x4000; 
	    i < (physical_end/0x4000); 
		++i, vram_start+=0x4000) {
		VRAMmap[i] = vram_start;
	}
}

void MMUMainremapVRAM_BANK_H(int master, int offset)
{
	int i = 0;
	int physical_start = 0;
	int physical_end = 0;
	u8* vram_start = VRAM_BANK_H;

	switch(master) {
	case 0:		// LCDC
		physical_start = 0x898000;
		physical_end =   0x8A0000;
		break;

	case 1:	   // A-BG
		physical_start = 0x200000;
		physical_end =   0x208000;;
		break;

	case 2:	   // BGEXTPAL
		// TODO: What to do here?
		break;
	}

	for(i = physical_start / 0x4000; 
	    i < (physical_end/0x4000); 
		++i, vram_start+=0x4000) {
		VRAMmap[i] = vram_start;
	}
}

void MMUMainremapVRAM_BANK_I(int master, int offset)
{
	int i = 0;
	int physical_start = 0;
	int physical_end = 0;
	u8* vram_start = VRAM_BANK_I;

	switch(master) {
	case 0:		// LCDC
		physical_start = 0x8A0000;
		physical_end =   0x8A4000;
		break;

	case 1:	   // A-BG
		physical_start = 0x208000;
		physical_end =   0x210000;;
		break;

	case 2:	   // B-OBJ
		physical_start = 0x600000;
		physical_end =   0x604000;
		break;

	case 3:	   // BGEXTPAL
		// TODO: What to do here?
		break;
	}

	for(i = physical_start / 0x4000; 
	    i < (physical_end/0x4000); 
		++i, vram_start+=0x4000) {
		VRAMmap[i] = vram_start;
	}
}

void MMUMainremapVRAM(u8 cntindex, u8 vcnt)
{
    int master=vcnt&7;
	int offset=(vcnt>>3)&3;
	int a = 0;
    char str[160];
    if(!(vcnt&0x80)) return;
    sprintf(str,"VRAM_CNT_%d: master=%d, offset=%d value=%x",cntindex,master,offset, vcnt);
    logvt->append(str);
    switch(cntindex)
    {
        case 0:
			MMUMainremapVRAM_BANK_A(master, offset);
			break;
        case 1:
			MMUMainremapVRAM_BANK_B(master, offset);
			break;
        case 2:
			MMUMainremapVRAM_BANK_C(master, offset);
			break;
        case 3:
			MMUMainremapVRAM_BANK_D(master, offset);
			break;
        case 4:
			MMUMainremapVRAM_BANK_E(master, offset);
			break;
        case 5:
			MMUMainremapVRAM_BANK_F(master, offset);
			break;
        case 6:
			MMUMainremapVRAM_BANK_G(master, offset);
			break;
        case 8:
			MMUMainremapVRAM_BANK_H(master, offset);
			break;
        case 9:
			MMUMainremapVRAM_BANK_I(master, offset);
			break;
	}
}

#define MMUMainintRdB(ram,addr) b=ram[(addr)>>2].b[(addr)&3]
#define MMUMainintRdH(ram,addr) \
    h=((addr)&1) \
        ?(((addr)&2) \
          ?(ram[(addr)>>2].b[3] | ((ram[((addr)>>2)+1].b[0])<<8)) \
          :(ram[(addr)>>2].b[1] | ((ram[(addr)>>2].b[2])<<8))) \
        :(ram[(addr)>>2].h[((addr)&2)>>1])
#define MMUMainintRdW(ram,addr) \
    switch((addr)&3) \
    { \
        case 0: w=ram[(addr)>>2].data; break; \
        case 1: w=(ram[(addr)>>2].data>> 8)+((ram[(addr)>>2].data&0x000000FF)<<24); break; \
        case 2: w=ram[(addr)>>2].h[1]+((ram[(addr)>>2].h[0])<<16); break; \
        case 3: w=(ram[(addr)>>2].data>>24)+((ram[(addr)>>2].data&0x00FFFFFF)<< 8); break; \
    }

u8 MMUMainrdB(u32 prot, u32 addr)
{
	char str[256];
    u8 b; //char str[80];
    switch(addr&0x0F000000)
    {
        case 0x00000000:
            if(addr&0x00800000) {
                b=DTCM[(addr&0x3FFF)>>2].b[addr&3];
            } else {
                b=ITCM[(addr&0x3FFF)>>2].b[addr&3];
            }
            break;
        case 0x02000000:
	    MMUMainintRdB(EWRAM, addr&0x0003FFFFF);
	    if(((addr&0xFFFFFF00)==0x02000000) ||
               ((addr&0xFFFFFF00)==0x0237F000))
            {
//                sprintf(str,"MMU9: EWRAMRd: %08X=%02X",addr,b);
//                logvt->append(str);
            }
	    ARM9addClock(2); break;
	case 0x03000000:
	    MMUMainintRdB(IWRAM, addr&0x00007FFF);
	    break;
	case 0x04000000:
		if(is_register_address(ARM9_REG, addr))
		{
			b = do_register_read_byte(ARM9_REG, addr);
		}
		else if(prot || REG_ADDR(ARM9_REG,addr).flags&REG_FLAG_R)
			b=REG_ADDR(ARM9_REG,addr).b[addr&1];
        else 
			b=0;
#if 0
	sprintf(str,"MMU9: IORd %08X: %08X=%02X",arm9reg.r[0xf],addr,b);
	logvt->append(str);
#endif
	    break;
	case 0x05000000:
            b=GPUPAL8[addr&0x7FF];
            break;
	case 0x06000000:
		{
			u8* base = VRAMmap[(addr&0x00FFFFFF)/0x4000];
			if(base != 0) {
				u8* vram8 =  base + (addr&0x3FFF);
				b = *vram8;
			}
			else {
				b = 0;
			}
		}
	    break;
	case 0x07000000:
	    b=(addr&1)?OAM[(addr&0x7FF)>>1]>>8:OAM[(addr&0x7FF)>>1]&255;
	    #ifdef MMUMainDEBUG
            sprintf(str,"MMU9: OAMRd: %03X=%02X",addr&0x7FF,b);
            logvt->append(str);
            #endif
	    break;
	case 0x08000000: case 0x09000000:
	    if((addr&0x00FFFFFF)<=MMUMainROMsize)
                MMUMainintRdB(GameROM, addr&0x00FFFFFF);
            else b=MMUMainEmpty.b[addr&3];
	    ARM9addClock(4); break;
        case 0x0F000000:
            if(((addr&0xFFFF0000)==0xFFFF0000) && prot)
                MMUMainintRdB(BIOS9, addr&0x000001FF);
            else b=0;
            break;
	
	case SRAM_ADDRESS:
		b = sram_read ( addr );
		break;
	
	default: break;
    }
    return b;
}

u16 MMUMainrdH(u32 prot, u32 addr)
{
	char str[256];
    u16 h; //char str[80];
    switch(addr&0x0F000000)
    {
        case 0x00000000:
            if(addr&0x00800000) {
                h=DTCM[(addr&0x3FFF)>>2].h[(addr&2)>>1];
            } else {
                h=ITCM[(addr&0x3FFF)>>2].h[(addr&2)>>1];
            }
            break;
        case 0x02000000:
	    MMUMainintRdH(EWRAM, addr&0x0003FFFFF);
	    if(((addr&0xFFFFFF00)==0x02000000) ||
               ((addr&0xFFFFFF00)==0x0237F000))
            {
//                sprintf(str,"MMU9: EWRAMRd: %08X=%04X",addr,h);
//                logvt->append(str);
            }
	    ARM9addClock(2); break;
	case 0x03000000:
	    MMUMainintRdH(IWRAM, addr&0x00007FFF);
	    break;
	case 0x04000000:
		if(is_register_address(ARM9_REG, addr))
		{
			h = do_register_read_half(ARM9_REG, addr);
		}
		else if(prot || REG_ADDR(ARM9_REG,addr).flags&REG_FLAG_R)
			h = REG_ADDR(ARM9_REG,addr).data;
        else 
			h=0;
#if 0
			/* TODO: DSLINUX DEBUG CODE REMOVE ME OR KEYS WILL BE BROKEN*/
		if(addr==0x4000130) {
			static u16 value = 0x0;
			h = value;
			value = ~value;
		}
#endif
#if 0
	sprintf(str,"MMU9: IORd %08X: %08X=%04X",arm9reg.r[0xf],addr,h);
	logvt->append(str);
#endif
	    break;
	case 0x05000000:
	    h=GPUPAL[(addr>>1)&0x3FF];
            break;
	case 0x06000000:
		{
			u8* base = VRAMmap[(addr&0x00FFFFFF)/0x4000];
			if(base != 0) {
				u8* vram8 =  base + (addr&0x3FFF);
				u16* vram = (u16*)vram8;
				h = *vram;
			}
			else {
				h = 0;
			}

		}
	    break;
	case 0x07000000:
	    h=OAM[(addr&0x7FF)>>1];
	    #ifdef MMUMainDEBUG
            sprintf(str,"MMU9: OAMRd: %03X=%04X",addr&0x7FF,h);
            logvt->append(str);
            #endif
	    break;
	case 0x08000000: case 0x09000000:
	    if((addr&0x00FFFFFF)<=MMUMainROMsize)
                MMUMainintRdH(GameROM, addr&0x00FFFFFF);
            else h=MMUMainEmpty.h[(addr&2)>>1];
	    ARM9addClock(4); break;
        case 0x0F000000:
            if(prot) MMUMainintRdH(BIOS9, addr&0x000001FF); else return 0;
            break;
	default: break;
    }
    return h;
}

u32 MMUMainrdW(u32 prot, u32 addr)
{
	char str[256];
    u32 w; u16 hl,hh; //char str[80];
    switch(addr&0x0F000000)
    {
        case 0x00000000:
            if(addr&0x00800000) {
                w=DTCM[(addr&0x3FFF)>>2].data;
            } else {
                w=ITCM[(addr&0x3FFF)>>2].data;
            }
            break;
        case 0x02000000:
	    MMUMainintRdW(EWRAM, addr&0x0003FFFFF);
	    if(((addr&0xFFFFFF00)==0x02000000) ||
               ((addr&0xFFFFFF00)==0x0237F000))
            {
//                sprintf(str,"MMU9: EWRAMRd: %08X=%08X",addr,w);
//                logvt->append(str);
            }
	    ARM9addClock(5); break;
	case 0x03000000:
	    MMUMainintRdW(IWRAM, addr&0x00007FFF);
	    break;
	case 0x04000000:
		if(is_register_address(ARM9_REG, addr))
		{
			w = do_register_read_word(ARM9_REG, addr);
		}
		else {
			if(prot || REG_ADDR(ARM9_REG,addr).flags&REG_FLAG_R) 
				hl=REG_ADDR(ARM9_REG,addr).data;
            else 
				hl=0;
			addr+=2;
			if(prot || REG_ADDR(ARM9_REG,addr).flags&REG_FLAG_R)
                hh=REG_ADDR(ARM9_REG,addr).data;
            else 
				hh=0;
            w=hl+(hh<<16);
		}
#if 0
	sprintf(str,"MMU9: IORd %08X: %08X=%08X",arm9reg.r[0xf],(addr-2),w);
	logvt->append(str);
#endif
	    break;
	case 0x05000000:
	    hl=GPUPAL[(addr>>1)&0x3FF];
            addr+=2;
	    hh=GPUPAL[(addr>>1)&0x3FF];
            w=hl+(hh<<16);
            break;
	case 0x06000000:
		{
			u8* base = VRAMmap[(addr&0x00FFFFFF)/0x4000];
			if(base != 0) {
				u8* vram8 =  base + (addr&0x3FFF);
				u32* vram = (u32*)vram8;
				w = *vram;
			}
			else {
				w = 0;
			}
		}
        break;
	case 0x07000000:
	    hl=OAM[(addr&0x7FF)>>1]; addr+=2;
	    hh=OAM[(addr&0x7FF)>>1];
            w=hl+(hh<<16);
	    #ifdef MMUMainDEBUG
            sprintf(str,"MMU9: OAMRd: %03X=%08X",addr&0x7FF,w);
            logvt->append(str);
            #endif
            break;
	case 0x08000000: case 0x09000000:
//            sprintf(str,"MMU9: RdW at %08X",addr); logvt->append(str);
	    if((addr&0x00FFFFFF)<=MMUMainROMsize){ MMUMainintRdW(GameROM, addr&0x00FFFFFF); }
            else w=MMUMainEmpty.data;
	    ARM9addClock(7); break;
        case 0x0F000000:
            MMUMainintRdW(BIOS9, addr&0x000001FF);
            break;
	default: break;
    }
    return w;
}

RAMWORD MMUMainrdS(u32 prot, u32 addr)
{
    if((addr&0xFE000000)==0x08000000)
    {
        if((addr&0x00FFFFFF)<=MMUMainROMsize) return GameROM[(addr&0xFFFFFF)>>2];
        else return MMUMainEmpty;
    }
    switch(addr&0x0F000000)
    {
        case 0x03000000: return IWRAM[(addr&0x07FFF)>>2];
        case 0x02000000: return EWRAM[(addr&0x3FFFFF)>>2];
        case 0x00000000: if(!(addr&0x00800000)) 
							 return ITCM[(addr&0x0FFF)>>2];
						 else
							 return DTCM[(addr&0x3fff)>>2];
        case 0x0F000000: if((addr&0xFFFF0000)==0xFFFF0000) {
							 return BIOS9[(addr&0x07FFF)>>2];
						 }
    }
    return MMUMainEmpty;
}

#define MMUMainintWrB(ram,addr,val) \
    ram[(addr)>>2].b[(addr)&3]=val; \
    ram[(addr)>>2].op=ARM9opUNL; \
    ram[(addr)>>2].cond=ARM9condAL

#define MMUMainintWrH(ram,addr,val) \
    ram[(addr)>>2].h[((addr)&2)>>1]=val; \
    ram[(addr)>>2].op=ARM9opUNL; \
    ram[(addr)>>2].cond=ARM9condAL

#define MMUMainintWrW(ram,addr,val) \
    ram[(addr)>>2].data=val; \
    ram[(addr)>>2].op=ARM9opUNL; \
    ram[(addr)>>2].cond=ARM9condAL

void MMUMainwrB(u32 bus, u32 addr, u8 val)
{    
    char str[256];
#if 0
    char str[80];
    sprintf(str,"MMU9: MMUMainwrB %08X: %08X=%02X",arm9reg.r[0xf],addr,val);
                    logvt->append(str);
#endif

    switch(addr&0x0F000000)
    {
        case 0x00000000:
            if(addr&0x00800000) DTCM[(addr&0x3FFF)>>2].b[addr&3]=val;
            else MMUMainintWrB(ITCM,addr&0x0FFF,val);
            break;
        case 0x02000000:
	    MMUMainintWrB(EWRAM, addr&0x0003FFFFF, val);
	    ARM9addClock(2); break;
	case 0x03000000:
	    MMUMainintWrB(IWRAM, addr&0x00007FFF, val);
	    break;
	case 0x06000000:
		{
			u8* base = VRAMmap[(addr&0x00FFFFFF)/0x4000];
			if(base != 0) {
				u8* vram8 =  base + (addr&0x3FFF);
				*vram8 = val;
			}
		}
		break;
        case 0x04000000:
			if(is_register_address(ARM9_REG, addr))
			{
				do_register_write_byte(ARM9_REG, addr, val);
			}
			else if(REG_ADDR(ARM9_REG,addr).flags&REG_FLAG_W)
            {
				extern void SwapScreens();
				if((addr&0x1ffe)==0x304 && 
					(((REG(ARM9_REG,POWERCNT))>>15) & 1) != 
					((val>>15)&1))
					SwapScreens();

                if((addr&0x1FFE)==0x214) 
					Int9Clear(val);
                else 
					REG_ADDR(ARM9_REG,addr).b[addr&1]=val;
                if((addr&0x1FF0)>=0x240 && (addr&0x1ff0)<=0x249 && (addr&0x1ff0) != 0x247)
					MMUMainremapVRAM(addr&15,val);
			}
            DMAcheck(DMA_TIMENOW);
#if 0
	sprintf(str,"MMU9: IOWr %08X: %08X=%02X",arm9reg.r[0xf],addr,val);
    logvt->append(str);
#endif
			break;

	case SRAM_ADDRESS:
		sram_write ( addr, val );
		break;
	
	default: break;
    }
}

void MMUMainwrH(u32 bus, u32 addr, u16 val)
{
    char str[80];
#if 0
sprintf(str,"MMU9: MMUMainwrH %08X: %08X=%04X",arm9reg.r[0xf],addr,val);
                    logvt->append(str);
#endif
    switch(addr&0x0F000000)
    {
        case 0x00000000:
            if(addr&0x00800000) DTCM[(addr&0x3FFF)>>2].h[(addr&2)>>1]=val;
            else MMUMainintWrH(ITCM,addr&0x0FFF,val);
            break;
        case 0x02000000:
	    MMUMainintWrH(EWRAM, addr&0x0003FFFFF, val);
	    ARM9addClock(2); break;
	case 0x03000000:
	    MMUMainintWrH(IWRAM, addr&0x00007FFF, val);
	    break;
	case 0x04000000:
		if(is_register_address(ARM9_REG, addr))
		{
			do_register_write_half(ARM9_REG, addr, val);
		}
		else if(REG_ADDR(ARM9_REG,addr).flags&REG_FLAG_W)
        {
			extern void SwapScreens();
			if((addr&0x1ffe)==0x304 && 
				(((REG(ARM9_REG,POWERCNT))>>15) & 1) != 
				((val>>15)&1))
			{
				SwapScreens();
			}

			if((addr&0x1ffe)==0x100)
				DoTimerReload(ARM9_REG, 0, val);
			else if((addr&0x1ffe)==0x102)
				DoTimerControl(ARM9_REG, 0, val);
			else if((addr&0x1ffe)==0x104)
				DoTimerReload(ARM9_REG, 1, val);
			else if((addr&0x1ffe)==0x106)
				DoTimerControl(ARM9_REG, 1, val);
			else if((addr&0x1ffe)==0x108)
				DoTimerReload(ARM9_REG, 2, val);
			else if((addr&0x1ffe)==0x10A)
				DoTimerControl(ARM9_REG, 2, val);
			else if((addr&0x1ffe)==0x10B)
				DoTimerReload(ARM9_REG, 3, val);
			else if((addr&0x1ffe)==0x10E)
				DoTimerControl(ARM9_REG, 3, val);
            else if((addr&0x1FFE)==0x214) 
				Int9Clear(val);
			else if((addr&0x1ffe)==0x180) {
				// TODO: Handle enable and interrupt bits
				int current9 = REG(ARM9_REG,IPCSYNC);
				int current7 = REG(ARM7_REG,IPCSYNC);
				REG(ARM9_REG,IPCSYNC)=(current9&0x6000)|(val&0xf00)|(current9&0xf);
				REG(ARM7_REG,IPCSYNC)=(current7&0x6f00)|(val>>8)&0xf;
			}
            else 
				REG_ADDR(ARM9_REG,addr).data=val;
            if((addr&0x1FF0)==0x240)
            {
                MMUMainremapVRAM((addr+0)&15,(val>> 0)&255);
                MMUMainremapVRAM((addr+1)&15,(val>> 8)&255);
            }
		}
        DMAcheck(DMA_TIMENOW);
#if 0
	sprintf(str,"MMU9: IOWr %08X: %08X=%04X",arm9reg.r[0xf],addr,val);
    logvt->append(str);
#endif
	    break;
	case 0x05000000:
	    #ifdef MMUMainDEBUG
	    sprintf(str,"MMU9: PalWr: %08X=%04X",addr,val);
	    logvt->append(str);
	    #endif
	    GPUPAL[(addr>>1)&0x3FF]=val;
	    break;
	case 0x06000000:
		{
#define VRAM_ADDR(x) (VRAMmap[(x)/0x4000])[(x) - (((x)/0x4000)*0x4000)]
#if 0
			int real_addr = addr&0x00ffffff;
			VRAM_ADDR(real_addr) = val & 0xff;
			VRAM_ADDR((real_addr+1)) = (val & 0xff00) >> 8;
#else
			u8* base = VRAMmap[(addr&0x00FFFFFF)/0x4000];
			if(base != 0) {
				u8* vram8 =  base + (addr&0x3FFF);
				u16* vram = (u16*)vram8;
				*vram = val;
			}
#endif
		}
		break;
	case 0x07000000:
	    #ifdef MMUMainDEBUG
            sprintf(str,"MMU9: OAMWr: %03X=%04X",addr&0x7FF,val);
            logvt->append(str);
            #endif
	    OAM[(addr&0x7FF)>>1]=val;
            break;
	default: break;
    }
}

void MMUMainwrW(u32 bus, u32 addr, u32 val)
{
    char str[256];
#if 0
    char str[80];
    sprintf(str,"MMU9: MMUMainwrW %08X: %08X=%08X",arm9reg.r[0xf],addr,val);
                    logvt->append(str);
#endif

    switch(addr&0x0F000000)
    {
        case 0x00000000:
            if(addr&0x00800000) DTCM[(addr&0x3FFF)>>2].data=val;
            else MMUMainintWrW(ITCM,addr&0x0FFF,val);
            break;
        case 0x02000000:
	    MMUMainintWrW(EWRAM, addr&0x0003FFFFF, val);
	    ARM9addClock(2); break;
	case 0x03000000:
	    MMUMainintWrW(IWRAM, addr&0x00007FFF, val);
	    break;
	case 0x04000000:
		if(is_register_address(ARM9_REG, addr))
		{
			do_register_write_word(ARM9_REG, addr, val);
		}
        else {
			if(REG_ADDR(ARM9_REG,addr).flags&REG_FLAG_W)
			{
				extern void SwapScreens();
				if((addr&0x1ffe)==0x304 && 
					(((REG(ARM9_REG,POWERCNT))>>15) & 1) != 
					(((int)val>>15)&1))
				{
					SwapScreens();
				}

				if((addr&0x1ffe)==0x100)
					DoTimerReload(ARM9_REG, 0, val);
				else if((addr&0x1ffe)==0x102)
					DoTimerControl(ARM9_REG, 0, val);
				else if((addr&0x1ffe)==0x104)
					DoTimerReload(ARM9_REG, 1, val);
				else if((addr&0x1ffe)==0x106)
					DoTimerControl(ARM9_REG, 1, val);
				else if((addr&0x1ffe)==0x108)
					DoTimerReload(ARM9_REG, 2, val);
				else if((addr&0x1ffe)==0x10A)
					DoTimerControl(ARM9_REG, 2, val);
				else if((addr&0x1ffe)==0x10B)
					DoTimerReload(ARM9_REG, 3, val);
				else if((addr&0x1ffe)==0x10E)
					DoTimerControl(ARM9_REG, 3, val);
				else if((addr&0x1FFE)==0x214) {
					Int9Clear(val);
				}
				else 
					REG_ADDR(ARM9_REG,addr).data=val&65535;
				if((addr&0x1FF0)==0x240)
				{
	                MMUMainremapVRAM((addr+0)&15,(val>> 0)&255);
					MMUMainremapVRAM((addr+1)&15,(val>> 8)&255);
				}
			}
			addr+=2;
			if(REG_ADDR(ARM9_REG,addr).flags&REG_FLAG_W)
            {
				extern void SwapScreens();
				if((addr&0x1ffe)==0x304 && 
					(((REG(ARM9_REG,POWERCNT))>>15) & 1) != 
					(((int)val>>15)&1))
				{
					SwapScreens();
				}

				if((addr&0x1ffe)==0x100)
					/* do nothing */;
				else if((addr&0x1ffe)==0x102)
					/* do nothing */;
				else if((addr&0x1ffe)==0x104)
					/* do nothing */;
				else if((addr&0x1ffe)==0x106)
					/* do nothing */;
				else if((addr&0x1ffe)==0x108)
					/* do nothing */;
				else if((addr&0x1ffe)==0x10A)
					/* do nothing */;
				else if((addr&0x1ffe)==0x10B)
					/* do nothing */;
				else if((addr&0x1ffe)==0x10E)
					/* do nothing */;
                else if((addr&0x1FFE)==0x214) 
					/* do nothing */;
                else REG_ADDR(ARM9_REG,addr).data=val>>16;
                if((addr&0x1FF0)==0x240)
                {
                    MMUMainremapVRAM((addr+0)&15,(val>>16)&255);
                    MMUMainremapVRAM((addr+1)&15,(val>>24)&255);
                }
            }
		}
        DMAcheck(DMA_TIMENOW);
#if 0
	sprintf(str,"MMU9: IOWr %08X: %08X=%08X",arm9reg.r[0xf],addr,val);
    logvt->append(str);
#endif
	    break;
	case 0x05000000:
#if 0
	    sprintf(str,"MMU9: PalWr: %08X=%08X",addr,val);
	    logvt->append(str);
#endif
	    GPUPAL[(addr>>1)&0x3FF]=val&65535;
	    addr+=2;
	    GPUPAL[(addr>>1)&0x3FF]=val>>16;
	case 0x06000000:
		{
			u8* base = VRAMmap[(addr&0x00FFFFFF)/0x4000];
			if(base != 0) {
				u8* vram8 =  base + (addr&0x3FFF);
				u32* vram = (u32*)vram8;
				*vram = val;
			}
		}
	    break;
	case 0x07000000:
	    #ifdef MMUMainDEBUG
            sprintf(str,"MMU9: OAMWr: %03X=%08X",addr&0x7FF,val);
            logvt->append(str);
            #endif
	    OAM[(addr&0x7FF)>>1]=val&65535; addr+=2;
	    OAM[(addr&0x7FF)>>1]=val>>16;
	    break;
	default: break;
    }
}

void MMUMainwrS(u32 bus, u32 addr, RAMWORD val)
{
	switch(addr&0x0F000000)
    {
        case 0x00000000: if(!(addr&0x00800000)) ITCM[(addr&0x0FFF)>>2]=val; break;
        case 0x02000000: EWRAM[(addr&0x3FFFFF)>>2]=val; break;
        case 0x03000000: IWRAM[(addr&0x07FFF)>>2]=val; break;
	case 0x08000000:
	case 0x09000000:
	case 0x0A000000:
	case 0x0B000000:
	case 0x0C000000:
        case 0x0D000000: GameROM[(addr&0xFFFFFF)>>2]=val; break;
	default: break;
    }
}

