/**************************************************************************
* DSemu: ARM7 assembly core, x86 (arm7x86.c)                              *
* Released under the terms of the BSD Public Licence                      *
* Imran Nazar (tf@oopsilon.com), 2004                                     *
**************************************************************************/

#include <stdio.h>
#include "err.h"
#include "arm7.h"
#include "arm7func.h"
#include "cache.h"
#include "vtbl.h"
#include "dbgout.h"
#include "bkpt.h"
#include "dsioreg.h"
#include "timers.h"
#include "dsmmusub.h"
#include <math.h>

// TODO: ARM7 and ARM9 cores are 90% identical. Merge the common code.
#if defined(GBA_COMPILE)
#undef REG
#define REG(ignore, x) GBAio[REG_ ## x].data
#endif

//#define ARM7REGDUMP

extern void emuPause();

ARMREGS arm7reg;

u32 *dbuf7;
FILE *ARM7regdump=NULL;

int cpuHalted; u16 intrBitmask;
int ARM7execforThumb(int), ARM7execforARM(int);
int (*ARM7execforPtr)(int) = ARM7execforARM;
int ARM7waitstate;

BREAKPOINT ARM7bkpts[16];

int ARM7init(char *file, u32 *dbgbuf)
{
    char str[8192];
    sprintf(str,"ARM7: ROM file: %s",file);
    logvt->append(str);

    MMUinit(file);
    dbuf7=dbgbuf;

//    if(ARM7reset()) RETFAIL("FAIL: ARM7: Reset.");
    RETPASS("ARM7: Initialised.");
}

void ARM7fini()
{
    MMUfini();
    #ifdef ARM7REGDUMP
    fclose(ARM7regdump);
    #endif
    logvt->append("ARM7: Shutdown.");
}

int ARM7reset()
{
    int a;

    #ifdef ARM7REGDUMP
    if(ARM7regdump){fclose(ARM7regdump);ARM7regdump=NULL;}
    ARM7regdump=fopen("C:\\Devel\\regdump.arm7.dat","wb");
    #endif

    arm7reg.r0=0;  arm7reg.r1=0;  arm7reg.r2=0;  arm7reg.r3=0;
    arm7reg.r4=0;  arm7reg.r5=0;  arm7reg.r6=0;  arm7reg.r7=0;
    arm7reg.r8=0;  arm7reg.r9=0;  arm7reg.r10=0; arm7reg.r11=0;
    arm7reg.r12=0; arm7reg.r13=0x03007F00; arm7reg.r14=0; arm7reg.r15=0x08000000;

    arm7reg.r[0]=arm7reg.r0;   arm7reg.r[1]=arm7reg.r1;   arm7reg.r[2]=arm7reg.r2;   arm7reg.r[3]=arm7reg.r3;
    arm7reg.r[4]=arm7reg.r4;   arm7reg.r[5]=arm7reg.r5;   arm7reg.r[6]=arm7reg.r6;   arm7reg.r[7]=arm7reg.r7;
    arm7reg.r[8]=arm7reg.r8;   arm7reg.r[9]=arm7reg.r9;   arm7reg.r[10]=arm7reg.r10; arm7reg.r[11]=arm7reg.r11;
    arm7reg.r[12]=arm7reg.r12; arm7reg.r[13]=arm7reg.r13; arm7reg.r[14]=arm7reg.r14; arm7reg.r[15]=arm7reg.r15;

    arm7reg.cpsr=ARMS_M_SYS|ARMS_F;
    arm7reg.clock=0;

    for(a=0;a<7;a++) arm7reg.spsr[a]=0;
    arm7reg.r8fiq=0;  arm7reg.r9fiq=0;  arm7reg.r10fiq=0; arm7reg.r11fiq=0;
    arm7reg.r12fiq=0; arm7reg.r13fiq=0; arm7reg.r14fiq=0;
    arm7reg.r13irq=0x03007FA0; arm7reg.r14irq=0;
    arm7reg.r13abt=0; arm7reg.r14abt=0;
    arm7reg.r13svc=0x03007FE0; arm7reg.r14svc=0;
    arm7reg.r13und=0; arm7reg.r14und=0;

    cpuHalted=0; intrBitmask=0xFFFF;
    ARM7execforPtr=ARM7execforARM;
    ARM7waitstate=0;

    arm7reg.curmode=ARMMD_SYS;
    for(a=0;a<8;a++) arm7reg.flags[a]=0;
    if(MMUreset()) RETFAIL("FAIL: MMU: Reset.");
    RETPASS("ARM7: Reset.");
}

__int64 ARM7getClock() { return arm7reg.clock; }
void ARM7addClock(int clk) { arm7reg.clock+=clk; ARM7waitstate+=clk; }

#define ARM7DUMP() \
	ARM7updateCPSR(); \
        fwrite((&arm7reg),4,15,ARM7regdump); \
        r15=arm7reg.r[15]+((arm7reg.cpsr&ARMS_T)?2:4); \
        fwrite((&r15),4,1,ARM7regdump); \
        fwrite((&arm7reg.cpsr),4,1,ARM7regdump); \
        fwrite((&arm7reg.curop),4,1,ARM7regdump);

int ARM7exec()
{
    //int idx,cond;
    char /*str[512],*/ opstr[128];
    //__int64 clk=arm7reg.clock,clkend;
    u32 r15=arm7reg.r[15];
    RAMWORD op;
    if(cpuHalted) return 8;
        op=MMUrdS(0,r15);
        ARM7waitstate=0;
	#ifdef ARM7DEBUG
//        sprintf(str,"%08X: Encountered opcode %08X (%s)", arm7reg.r[15], op.data, ARM7DASM(op.data));
//        logvt->append(str);
	#endif
	if(arm7reg.flags[ARMFLAG_T])
	{
            switch(r15&2)
            {
                case 0:
		    arm7reg.curop=op.data&65535;
                    sprintf(opstr,"%s", Thumb7DASM(arm7reg.curop));
//                    logvt->append(opstr);
                    arm7reg.r[15]+=2;
		    return /*arm7reg.clock+=*/thumb7ops[arm7reg.curop>>8]()+ARM7waitstate;
//                    arm7reg.clock+=thumb7ops[arm7reg.curop>>8].cyc; break;
                case 2:
		    arm7reg.curop=op.data>>16;
                    sprintf(opstr,"%s", Thumb7DASM(arm7reg.curop));
//                    logvt->append(opstr);
                    arm7reg.r[15]+=2;
		    return /*arm7reg.clock+=*/thumb7ops[arm7reg.curop>>8]()+ARM7waitstate;
//                    arm7reg.clock+=thumb7ops[arm7reg.curop>>8].cyc; break;
            }
            return 0;
        } else {
            sprintf(opstr,"%s", ARM7DASM(op.data));
//            logvt->append(opstr);
            arm7reg.r[15]+=4;
            if(op.cond())
            {
                arm7reg.curop=op.data;
                return /*arm7reg.clock+=*/op.op()+ARM7waitstate;
//                arm7reg.clock+=op.cyc;
            }
            else return 1; //arm7reg.clock++;
        }
        #ifdef ARM7REGDUMP
	ARM7DUMP();
	#endif
//	clkend=arm7reg.clock;
//	return clkend-clk;
}

// TODO: Reorganise code so timers can be taken out of the
//       ARM7 file.
#include "int.h"

static void UpdateIRQ() 
{
	extern u16 int7Fired;
	if(int7Fired && !(arm7reg.cpsr & ARMS_I))
	{
		if(REG(ARM7_REG,IME) && (REG(ARM7_REG,IE)&int7Fired))
		{
			REG(ARM7_REG,IF)|=int7Fired;
			ARM7InterruptFlagChanged();
			ARM7irq();
		}
		int7Fired = 0;
	}
}

int ARM7execforThumb(int count)
{
    u32 clk=0/*,opclk*/; /*u32 r15;*/ RAMWORD op;
	UpdateIRQ();
	if(!is_any_timer_active(ARM7_REG) && cpuHalted)
		return count;
    do
    {
		if(cpuHalted && is_any_timer_active(ARM7_REG)) {
			int i = 0;
			int min_timer = 1000000;
			int opclk = 0;
			for(i=0; i < 4; ++i) {
				int ticks = timer_data[ARM7_REG][i].rollover_ticks - timer_data[ARM7_REG][i].current_ticks;
				if(!timer_data[ARM7_REG][i].cascade && timer_data[ARM7_REG][i].active && ticks < min_timer)
					min_timer = ticks;
			}
			opclk = min(count-(int)clk, min_timer);
			clk += opclk;
			UpdateTimers(ARM7_REG, opclk);
			UpdateIRQ();
		}
		else if(!cpuHalted)
		{
			int c = 0;
	        op=MMUrdS(0,arm7reg.r[15]);
			ARM7waitstate=0;
			arm7reg.curop=(arm7reg.r[15]&2)?op.h[1]:op.h[0];
			arm7reg.r[15]+=2;
			c = thumb7ops[arm7reg.curop>>8]()+ARM7waitstate;
			clk+=c;
			UpdateTimers(ARM7_REG, c);
		}
		else
			break;
    } while(((int)clk<count) && arm7reg.flags[ARMFLAG_T]);
    return (int)clk;
}

int ARM7execforARM(int count)
{
    u32 clk=0,opclk; /*u32 r15;*/ RAMWORD op;
	UpdateIRQ();
	if(!is_any_timer_active(ARM7_REG) && cpuHalted)
		return count;
    do
    {
		if(cpuHalted && is_any_timer_active(ARM7_REG)) {
			int i = 0;
			int min_timer = 1000000;
			int opclk = 0;
			for(i=0; i < 4; ++i) {
				int ticks = timer_data[ARM7_REG][i].rollover_ticks - timer_data[ARM7_REG][i].current_ticks;
				if(!timer_data[ARM7_REG][i].cascade && timer_data[ARM7_REG][i].active && ticks < min_timer)
					min_timer = ticks;
			}
			opclk = min(count-(int)clk, min_timer);
			clk += opclk;
			UpdateTimers(ARM7_REG, opclk);
			UpdateIRQ();
		}
		else if(!cpuHalted)
		{
			u32 r15 = arm7reg.r[0xf];
			op=MMUrdS(0,r15);
			ARM7waitstate=0;
			arm7reg.r[15] = r15+4;
			if(op.cond && op.cond())
			{
	            arm7reg.curop=op.data;
				opclk=op.op();
				if(opclk) clk+=opclk+ARM7waitstate; else return clk;
			}
			UpdateTimers(ARM7_REG, opclk+ARM7waitstate);
		}
		else
			break;
    } while(((int)clk<count) && (!arm7reg.flags[ARMFLAG_T]));
    return (int)clk;
}

int ARM7execfor(int count)
{
	return ARM7execforPtr(count);
}

void ARM7updateCPSR()
{
    if(arm7reg.flags[0]) arm7reg.cpsr|=ARMS_N;
    else arm7reg.cpsr&=(0xFFFFFFFF-ARMS_N);
    if(arm7reg.flags[1]) arm7reg.cpsr|=ARMS_Z;
    else arm7reg.cpsr&=(0xFFFFFFFF-ARMS_Z);
    if(arm7reg.flags[2]) arm7reg.cpsr|=ARMS_C;
    else arm7reg.cpsr&=(0xFFFFFFFF-ARMS_C);
    if(arm7reg.flags[3]) arm7reg.cpsr|=ARMS_V;
    else arm7reg.cpsr&=(0xFFFFFFFF-ARMS_V);
    if(arm7reg.flags[4]) arm7reg.cpsr|=ARMS_T;
    else arm7reg.cpsr&=(0xFFFFFFFF-ARMS_T);
    if(arm7reg.flags[5]) arm7reg.cpsr|=ARMS_Q;
    else arm7reg.cpsr&=(0xFFFFFFFF-ARMS_Q);
    ARM7execforPtr=(arm7reg.flags[ARMFLAG_T]?ARM7execforThumb:ARM7execforARM);
}

void ARM7splitCPSR()
{
    arm7reg.flags[0]=(arm7reg.cpsr>>31)&1;
    arm7reg.flags[1]=(arm7reg.cpsr>>30)&1;
    arm7reg.flags[2]=(arm7reg.cpsr>>29)&1;
    arm7reg.flags[3]=(arm7reg.cpsr>>28)&1;
    arm7reg.flags[4]=(arm7reg.cpsr>>5)&1;
    arm7reg.flags[5]=(arm7reg.cpsr>>27)&1;
    ARM7execforPtr=(arm7reg.flags[ARMFLAG_T]?ARM7execforThumb:ARM7execforARM);
}

void ARM7modesw(int mdin, int mdout)
{
    ARM7updateCPSR();
    arm7reg.r0=arm7reg.r[0]; arm7reg.r1=arm7reg.r[1];
    arm7reg.r2=arm7reg.r[2]; arm7reg.r3=arm7reg.r[3];
    arm7reg.r4=arm7reg.r[4]; arm7reg.r5=arm7reg.r[5];
    arm7reg.r6=arm7reg.r[6]; arm7reg.r7=arm7reg.r[7];
    arm7reg.r15=arm7reg.r[15];

    switch(mdin)
    {
    	case ARMMD_USR: case ARMMD_SYS:
    	    arm7reg.r8=arm7reg.r[8];   arm7reg.r9=arm7reg.r[9];
    	    arm7reg.r10=arm7reg.r[10]; arm7reg.r11=arm7reg.r[11];
    	    arm7reg.r12=arm7reg.r[12]; arm7reg.r13=arm7reg.r[13];
    	    arm7reg.r14=arm7reg.r[14];
    	    break;
    	case ARMMD_IRQ:
    	    arm7reg.r8=arm7reg.r[8];   arm7reg.r9=arm7reg.r[9];
    	    arm7reg.r10=arm7reg.r[10]; arm7reg.r11=arm7reg.r[11];
    	    arm7reg.r12=arm7reg.r[12];
            arm7reg.r13irq=arm7reg.r[13]; arm7reg.r14irq=arm7reg.r[14];
    	    break;
        case ARMMD_SVC:
    	    arm7reg.r8=arm7reg.r[8];   arm7reg.r9=arm7reg.r[9];
    	    arm7reg.r10=arm7reg.r[10]; arm7reg.r11=arm7reg.r[11];
    	    arm7reg.r12=arm7reg.r[12];
            arm7reg.r13svc=arm7reg.r[13]; arm7reg.r14svc=arm7reg.r[14];
    	    break;
        case ARMMD_ABT:
    	    arm7reg.r8=arm7reg.r[8];   arm7reg.r9=arm7reg.r[9];
    	    arm7reg.r10=arm7reg.r[10]; arm7reg.r11=arm7reg.r[11];
    	    arm7reg.r12=arm7reg.r[12];
            arm7reg.r13abt=arm7reg.r[13]; arm7reg.r14abt=arm7reg.r[14];
    	    break;
        case ARMMD_UND:
    	    arm7reg.r8=arm7reg.r[8];   arm7reg.r9=arm7reg.r[9];
    	    arm7reg.r10=arm7reg.r[10]; arm7reg.r11=arm7reg.r[11];
    	    arm7reg.r12=arm7reg.r[12];
            arm7reg.r13und=arm7reg.r[13]; arm7reg.r14und=arm7reg.r[14];
    	    break;
    	case ARMMD_FIQ:
            arm7reg.r8fiq=arm7reg.r[8];   arm7reg.r9fiq=arm7reg.r[9];
    	    arm7reg.r10fiq=arm7reg.r[10]; arm7reg.r11fiq=arm7reg.r[11];
    	    arm7reg.r12fiq=arm7reg.r[12]; arm7reg.r13fiq=arm7reg.r[13];
    	    arm7reg.r14fiq=arm7reg.r[14];
    	    break;
    }
    arm7reg.cpsr&=(0xFFFFFFFF-ARMS_M);
    switch(mdout)
    {
    	case ARMMD_USR:
    	    arm7reg.r[8]=arm7reg.r8;   arm7reg.r[9]=arm7reg.r9;
    	    arm7reg.r[10]=arm7reg.r10; arm7reg.r[11]=arm7reg.r11;
    	    arm7reg.r[12]=arm7reg.r12; arm7reg.r[13]=arm7reg.r13;
    	    arm7reg.r[14]=arm7reg.r14;
    	    break;
        case ARMMD_SYS:
    	    arm7reg.r[8]=arm7reg.r8;   arm7reg.r[9]=arm7reg.r9;
    	    arm7reg.r[10]=arm7reg.r10; arm7reg.r[11]=arm7reg.r11;
    	    arm7reg.r[12]=arm7reg.r12; arm7reg.r[13]=arm7reg.r13;
    	    arm7reg.r[14]=arm7reg.r14;
    	    break;
    	case ARMMD_IRQ:
    	    arm7reg.r[8]=arm7reg.r8;   arm7reg.r[9]=arm7reg.r9;
    	    arm7reg.r[10]=arm7reg.r10; arm7reg.r[11]=arm7reg.r11;
    	    arm7reg.r[12]=arm7reg.r12;
            arm7reg.r[13]=arm7reg.r13irq; arm7reg.r[14]=arm7reg.r14irq;
    	    break;
        case ARMMD_SVC:
    	    arm7reg.r[8]=arm7reg.r8;   arm7reg.r[9]=arm7reg.r9;
    	    arm7reg.r[10]=arm7reg.r10; arm7reg.r[11]=arm7reg.r11;
    	    arm7reg.r[12]=arm7reg.r12;
            arm7reg.r[13]=arm7reg.r13svc; arm7reg.r[14]=arm7reg.r14svc;
    	    break;
        case ARMMD_ABT:
    	    arm7reg.r[8]=arm7reg.r8;   arm7reg.r[9]=arm7reg.r9;
    	    arm7reg.r[10]=arm7reg.r10; arm7reg.r[11]=arm7reg.r11;
    	    arm7reg.r[12]=arm7reg.r12;
            arm7reg.r[13]=arm7reg.r13abt; arm7reg.r[14]=arm7reg.r14abt;
    	    break;
        case ARMMD_UND:
    	    arm7reg.r[8]=arm7reg.r8;   arm7reg.r[9]=arm7reg.r9;
    	    arm7reg.r[10]=arm7reg.r10; arm7reg.r[11]=arm7reg.r11;
    	    arm7reg.r[12]=arm7reg.r12;
            arm7reg.r[13]=arm7reg.r13und; arm7reg.r14und=arm7reg.r[14];
    	    break;
    	case ARMMD_FIQ:
            arm7reg.r[8]=arm7reg.r8fiq;   arm7reg.r[9]=arm7reg.r9fiq;
    	    arm7reg.r[10]=arm7reg.r10fiq; arm7reg.r[11]=arm7reg.r11fiq;
    	    arm7reg.r[12]=arm7reg.r12fiq; arm7reg.r[13]=arm7reg.r13fiq;
    	    arm7reg.r[14]=arm7reg.r14fiq;
    	    break;
    }
    arm7reg.cpsr|=ARM7modetocpsr[mdout];
    arm7reg.curmode=mdout;
}

void ARM7status(int offset, int mode)
{
    char str[512]; int a; u16 oph;
    u32 r15=arm7reg.r[15],r15orig=r15; static u32 r15old;
    static u32 dbgoldr[16],rchanged[16],dbgoldf[6],fchanged[6];
    RAMWORD op; u32 col; u16 *outbuf=(u16*)dbuf7;

    ARM7updateCPSR();
    dbgOutClear(outbuf,468,144);

    if((mode==3&&arm7reg.cpsr&ARMS_T)||(mode==2))
        r15-=(16-offset*2);
    else r15-=(32-offset*4);

    for(a=0;a<16;a++)
    {
        if(r15==r15orig) col=0x7E10; else col=0x7FFF;
        if((r15&0x0F000000)==(r15orig&0x0F000000)) op=MMUrdS(0,r15);
        switch(mode)
        {
            case 1:
                if((r15&0x0F000000)==(r15orig&0x0F000000))
                    sprintf(str,"%08X: %08X | %s",r15,op.data,ARM7DASM(op.data));
                else sprintf(str,"%08X",r15);
                r15+=4; break;
            case 2:
                if((r15&0x0F000000)==(r15orig&0x0F000000))
                {
                    if(r15&2) oph=op.data>>16; else oph=op.data&65535;
                    sprintf(str,"%08X: %04X | %s",r15,oph,Thumb7DASM(oph));
                }
                else sprintf(str,"%08X",r15);
                r15+=2; break;
            case 3:
                switch(arm7reg.cpsr&ARMS_T)
                {
                    case 0:
                        if((r15&0x0F000000)==(r15orig&0x0F000000))
                            sprintf(str,"%08X: %08X | %s",r15,op.data,ARM7DASM(op.data));
                        else sprintf(str,"%08X",r15);
                        r15+=4; break;
                    case ARMS_T:
                        if((r15&0x0F000000)==(r15orig&0x0F000000))
                        {
                            if(r15&2) oph=op.data>>16; else oph=op.data&65535;
                            sprintf(str,"%08X: %04X | %s",r15,oph,Thumb7DASM(oph));
                        }
                        else sprintf(str,"%08X",r15);
                        r15+=2; break;
                }
                break;
        }
        dbgOut(outbuf,468,144,str,0,16+a*8,(u16)col);
    }
    for(a=0;a<=15;a++)
    {
        sprintf(str,"r%02d: %08X",a,arm7reg.r[a]);
        if(r15old!=r15orig)
        {
            if(arm7reg.r[a]!=dbgoldr[a]) { dbgoldr[a]=arm7reg.r[a]; rchanged[a]=1; }
            else rchanged[a]=0;
        }
        if(rchanged[a]) dbgOut(outbuf,468,144,str,56*6,16+a*8,0x001F);
	else dbgOut(outbuf,468,144,str,56*6,16+a*8,0x7FFF);
    }
    sprintf(str,"%s",ARM7modes[arm7reg.curmode]);
    dbgOut(outbuf, 468,144, str, 72*6,72,0x7FFF);

    a=(arm7reg.cpsr&ARMS_N)>>31; sprintf(str,"N: %d",a);
    if(r15old!=r15orig)
        if(a!=(int)dbgoldf[0]) { dbgoldf[0]=a; fchanged[0]=1; } else fchanged[0]=0;
    if(fchanged[0]) dbgOut(outbuf,468,144,str,72*6,16,0x001F);
    else dbgOut(outbuf,468,144,str,72*6,16,0x7FFF);

    a=(arm7reg.cpsr&ARMS_Z)>>30; sprintf(str,"Z: %d",a);
    if(r15old!=r15orig)
        if(a!=(int)dbgoldf[1]) { dbgoldf[1]=a; fchanged[1]=1; } else fchanged[1]=0;
    if(fchanged[1]) dbgOut(outbuf,468,144,str,72*6,24,0x001F);
    else dbgOut(outbuf,468,144,str,72*6,24,0x7FFF);

    a=(arm7reg.cpsr&ARMS_C)>>29; sprintf(str,"C: %d",a);
    if(r15old!=r15orig)
        if(a!=(int)dbgoldf[2]) { dbgoldf[2]=a; fchanged[2]=1; } else fchanged[2]=0;
    if(fchanged[2]) dbgOut(outbuf,468,144,str,72*6,32,0x001F);
    else dbgOut(outbuf,468,144,str,72*6,32,0x7FFF);

    a=(arm7reg.cpsr&ARMS_V)>>28; sprintf(str,"V: %d",a);
    if(r15old!=r15orig)
        if(a!=(int)dbgoldf[3]) { dbgoldf[3]=a; fchanged[3]=1; } else fchanged[3]=0;
    if(fchanged[3]) dbgOut(outbuf,468,144,str,72*6,40,0x001F);
    else dbgOut(outbuf,468,144,str,72*6,40,0x7FFF);
/*
    a=(arm7reg.cpsr&ARMS_Q)>>27;  sprintf(str,"Q: %d",a);
    if(r15old!=r15orig)
        if(a!=dbgoldf[4]) { dbgoldf[4]=a; fchanged[4]=1; } else fchanged[4]=0;
    if(fchanged[4]) dbgOut(outbuf,468,144,str,72*6,48,0x001F);
    else dbgOut(outbuf,468,144,str,72*6,48,0x7FFF);
*/
    a=(arm7reg.cpsr&ARMS_T)>>5;  sprintf(str,"T: %d",a);
    if(r15old!=r15orig)
        if(a!=(int)dbgoldf[5]) { dbgoldf[5]=a; fchanged[5]=1; } else fchanged[5]=0;
    if(fchanged[5]) dbgOut(outbuf,468,144,str,72*6,56,0x001F);
    else dbgOut(outbuf,468,144,str,72*6,56,0x7FFF);

    if(r15old!=r15orig) r15old=r15orig;
}

//---Undefined/unimplemented opcode messaging------------------------------

OPCODE ARM7opUND()
{
    char str[80];
    sprintf(str,"Undefined opcode encountered: %08X.",arm7reg.curop);
    logvt->append(str);
    return 1;
}

OPCODE ARM7opUNP()
{
    char str[80];
    sprintf(str,"Unpredictable opcode encountered: %08X.",arm7reg.curop);
    logvt->append(str);
    return 1;
}

OPCODE ARM7opUNI()
{
    char str[80];
    sprintf(str,"Unimplemented opcode encountered: %08X.",arm7reg.curop);
    logvt->append(str);
    return 1;
}

OPCODE ARM7opUNL()
{
    RAMWORD op;
    u32 idx=((arm7reg.curop&0x0FF00000)>>16)|((arm7reg.curop&0x000000F0)>>4);
    u16 cond=(arm7reg.curop&0xF0000000)>>28;
    op.data=arm7reg.curop;
    op.op=arm7ops[idx];
    op.tidxlo=(arm7reg.curop>>8)&255;
    op.tidxhi=(arm7reg.curop>>24);
    op.cond=ARM7cond[cond];
//    op.cyc=arm7ops[idx].cyc;
    op.flags=0;

	if(arm7reg.flags[ARMFLAG_T] || (arm7reg.r[0xf] % 2 == 1))
	{
		// Somehow we ended up executing thumb instructions in ARM mode.
		// This happens in the current libnds as the interrupt handler is
		// in thumb but the interrupt routine executes the handler in ARM.
		// I check if the instruction is a thumb address and execute the 
		// correct ARM codes.
		int clock = 0;
	    arm7reg.curop=op.data&65535;
	    clock = thumb7ops[arm7reg.curop>>8]();
	    arm7reg.curop=op.data>>16;
	    clock+=thumb7ops[arm7reg.curop>>8]();
		return clock;
        } else {
            MMUwrS(0,arm7reg.r[15]-4,op);
            if(op.cond())
            {
                arm7reg.curop=op.data;
                return op.op();
//                arm7reg.clock+=op.cyc;
            }
            return 1;
        }
}

RAMWORD ARM7opDecode(u32 input)
{
    RAMWORD op;
    u32 idx=((input&0x0FF00000)>>16)|((input&0x000000F0)>>4);
    u16 cond=(input&0xF0000000)>>28;
    op.data=input;
    op.op=arm7ops[idx];
    op.tidxlo=(input>>8)&255;
    op.tidxhi=(input>>24);
    op.cond=ARM7cond[cond];
    op.flags=0;

    return op;
}

//---Conditional execution implementation----------------------------------

int ARM7condEQ(){return  arm7reg.flags[ARMFLAG_Z];}
int ARM7condNE(){return !arm7reg.flags[ARMFLAG_Z];}
int ARM7condCS(){return  arm7reg.flags[ARMFLAG_C];}
int ARM7condCC(){return !arm7reg.flags[ARMFLAG_C];}
int ARM7condMI(){return  arm7reg.flags[ARMFLAG_N];}
int ARM7condPL(){return !arm7reg.flags[ARMFLAG_N];}
int ARM7condVS(){return  arm7reg.flags[ARMFLAG_V];}
int ARM7condVC(){return !arm7reg.flags[ARMFLAG_V];}

int ARM7condHI(){return (!arm7reg.flags[ARMFLAG_Z])&arm7reg.flags[ARMFLAG_C];}
int ARM7condLS(){return arm7reg.flags[ARMFLAG_Z]|(!arm7reg.flags[ARMFLAG_C]);}
int ARM7condGE(){return !(arm7reg.flags[ARMFLAG_N]^arm7reg.flags[ARMFLAG_V]);}
int ARM7condLT(){return  (arm7reg.flags[ARMFLAG_N]^arm7reg.flags[ARMFLAG_V]);}
int ARM7condGT(){return (!arm7reg.flags[ARMFLAG_Z])&(!(arm7reg.flags[ARMFLAG_N]^arm7reg.flags[ARMFLAG_V]));}
int ARM7condLE(){return arm7reg.flags[ARMFLAG_Z]|(arm7reg.flags[ARMFLAG_N]^arm7reg.flags[ARMFLAG_V]);}
int ARM7condAL(){return 1;}
int ARM7condNV(){return 0;}

//---Branching-------------------------------------------------------------

OPCODE ARM7opB()
{
    __asm {
        mov eax,[arm7reg+ARMREG_OP*4]     /* Get opcode */
        shl eax,8                         /* Extract branch address */
        sar eax,6                         /* addr*4, with sign extend */
        add eax,4                         /* Add 4 to new pc */
        add [arm7reg+ARMREG_R15*4],eax    /* and write to r15 */
        mov eax,3
    }
}

OPCODE ARM7opBL()
{
    __asm {
	mov ecx,[arm7reg+ARMREG_R15*4]    /* Get r15 */
        mov [arm7reg+ARMREG_R14*4],ecx    /* r14=r15 */
        mov eax,[arm7reg+ARMREG_OP*4]     /* Get opcode */
        shl eax,8                         /* Extract branch address */
        sar eax,6                         /* addr*4, with sign extend */
        add eax,4                         /* Add 4 to new pc */
        add [arm7reg+ARMREG_R15*4],eax    /* and write to r15 */
        mov eax,3
    }
}

OPCODE ARM7opBX()
{
    __asm {
        mov eax,[arm7reg+ARMREG_OP*4]      /* Read opcode */
        and eax,15                         /* Mask Rm */
        mov eax,[arm7reg+eax*4]            /* Read Rm's value */
        mov ebx,eax                        /* Make a copy */
	and eax,0FFFFFFFEh                 /* Mask out T bit */
	mov [arm7reg+15*4],eax             /* and write r15 */
	and ebx,1                          /* Mask in T bit */
	cmp ebx,1                          /* Is it 1? */
	sbb ecx,ecx                        /* (==0)?-1:0 */
	not ecx                            /* (==0)?0:-1 */
//	and ecx,ARMS_T                     /* Mask cpsr T bit */
//	and [arm7reg+ARMREG_CPSR*4],0FFFFFFFFh-ARMS_T
//	or [arm7reg+ARMREG_CPSR*4],ecx     /* Add T */
        and ecx,1
        mov [arm7reg+ARMREG_FT*4],ecx
        call ARM7updateCPSR
        mov eax,3
    }
}

OPCODE ARM7opBLX()
{
  // Implemented in ARM5
  return 1;
}

OPCODE ARM7opBLXr()
{
    __asm {
        mov eax,[arm7reg+15*4]
        mov [arm7reg+14*4],eax
        mov eax,[arm7reg+ARMREG_OP*4]      /* Read opcode */
        and eax,15                         /* Mask Rm */
        mov eax,[arm7reg+eax*4]            /* Read Rm's value */
        mov ebx,eax                        /* Make a copy */
	and eax,0FFFFFFFEh                 /* Mask out T bit */
	mov [arm7reg+15*4],eax             /* and write r15 */
	and ebx,1                          /* Mask in T bit */
	cmp ebx,1                          /* Is it 1? */
	sbb ecx,ecx                        /* (==0)?-1:0 */
	not ecx                            /* (==0)?0:-1 */
//	and ecx,ARMS_T                     /* Mask cpsr T bit */
//	and [arm7reg+ARMREG_CPSR*4],0FFFFFFFFh-ARMS_T
//	or [arm7reg+ARMREG_CPSR*4],ecx     /* Add T */
        and ecx,1
        mov [arm7reg+ARMREG_FT*4],ecx
        call ARM7updateCPSR
        mov eax,3
    }
}

//---Data Processing addressing modes, opcodes-----------------------------

#define ARM7addrDPops() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask to range */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Read value of Rn */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt2                          /* If not, skip next */ \
        __asm add eax,4                       /* Hack r15 */ \
        __asm pt2:                            /* OPERAND2 set */ \
        \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm and ecx,15                      /* Mask to Rm */ \
        __asm mov ebx,[arm7reg+ecx*4]         /* Read value of Rm */ \
        __asm cmp ecx,15                      /* Is Rm 15? */ \
        __asm jl pt1                          /* If not, skip next */ \
        __asm add ebx,4                       /* Hack r15 value */ \
        __asm pt1:                            /* OPERAND1 set */ \
    }

#define ARM7addrDPreg() \
    ARM7addrDPops(); \
    __asm { \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        __asm mov ecx,[arm7reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm setc [arm7reg+ARMREG_TC*4] \
    }

#define ARM7addrDPimm() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask to range */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Read value of Rn */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt2                          /* If not, skip next */ \
        __asm add eax,4                       /* Hack r15 */ \
        __asm pt2:                            /* OPERAND2 set */ \
        \
        __asm mov ecx,edx                     /* Make op copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        \
        __asm and ebx,255                     /* Get the low byte */ \
        __asm shr ecx,7                       /* Shift in rotamnt */ \
        __asm and ecx,30                      /* and mask to 31 */ \
        __asm ror ebx,cl                      /* Rotate constant */ \
        __asm or ecx,ecx                      /* Check rotamnt */ \
        __asm jz noshft                       /* Skip if no rot */ \
        __asm mov ecx,eax                     /* Copy operand1 */ \
        __asm shl ecx,1                       /* Carry = high bit */ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
        __asm mov ecx,[arm7reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
	\
        __asm shftdone:                       /* all done! */ \
        __asm setc [arm7reg+ARMREG_TC*4] \
    }

#define ARM7addrDPlli() \
    ARM7addrDPops(); \
    __asm { \
        __asm mov ecx,edx                     /* Make op copy */ \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        \
        __asm shr ecx,7                       /* Shift in shftamount */ \
        __asm and ecx,31                      /* Mask off amount */ \
        __asm jz noshft                       /* Skip if no shift */ \
        __asm shl ebx,cl                      /* Shift operand1 */ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
	\
        __asm mov ecx,[arm7reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm shftdone:                       /* all done! */ \
        __asm setc [arm7reg+ARMREG_TC*4] \
    }

#define ARM7addrDPllr() \
    ARM7addrDPops(); \
    __asm { \
        __asm cmp ecx,15 \
        __asm jl pt3 \
        __asm add ebx,4 \
        __asm pt3: \
        \
        __asm mov ecx,edx                     /* Make op copy */ \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        \
        __asm shr ecx,8                       /* Shift in Rs */ \
        __asm and ecx,15                      /* Mask off Rs */ \
        __asm mov ecx,[arm7reg+ecx*4]         /* Read Rs's value */ \
        __asm and ecx,255                     /* Get the low byte */ \
        __asm jz noshft                       /* Skip if no shift */ \
        __asm shl ebx,cl                      /* Shift operand1 */ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
	\
        __asm mov ecx,[arm7reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm shftdone:                       /* all done! */ \
        __asm setc [arm7reg+ARMREG_TC*4] \
    }

#define ARM7addrDPlri() \
    ARM7addrDPops(); \
    __asm { \
        __asm mov ecx,edx                     /* Make op copy */ \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        \
        __asm shr ecx,7                       /* Shift in shftamount */ \
        __asm and ecx,31                      /* Mask off amount */ \
        __asm jz noshft                       /* Skip if no shift */ \
        __asm shr ebx,cl                      /* Shift operand1 */ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
	\
        __asm mov ecx,ebx                     /* Make a copy */ \
        __asm xor ebx,ebx                     /* Rm=0 */ \
        __asm shl ecx,1                       /* Carry = Rm[31] */ \
        __asm shftdone:                       /* all done! */ \
        __asm setc [arm7reg+ARMREG_TC*4] \
    }

#define ARM7addrDPlrr() \
    ARM7addrDPops(); \
    __asm { \
        __asm cmp ecx,15 \
        __asm jl pt3 \
        __asm add ebx,4 \
        __asm pt3: \
        \
        __asm mov ecx,edx                     /* Make op copy */ \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        \
        __asm shr ecx,8                       /* Shift in Rs */ \
        __asm and ecx,15                      /* Mask off Rs */ \
        __asm mov ecx,[arm7reg+ecx*4]         /* Read Rs's value */ \
        __asm and ecx,255                     /* Get the low byte */ \
        __asm jz noshft                       /* Skip if no shift */ \
        __asm cmp cl,32                       /* If greater than 32 */ \
        __asm jg overshft                     /* Shift all the way */ \
        __asm je fullshft                     /* If equal, shift to carry */ \
        __asm shr ebx,cl                      /* Shift operand1 */ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
        \
        __asm mov ecx,[arm7reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm jmp shftdone                    /* Skip over */ \
        __asm fullshft:                       /* Shift by 32 */ \
        __asm mov ecx,ebx                     /* Make a copy */ \
        __asm xor ebx,ebx                     /* Rm=0 */ \
        __asm shl ecx,1                       /* Carry = Rm[31] */ \
        __asm jmp shftdone                    /* Skip over */ \
        __asm overshft:                       /* Shift over 32 */ \
        __asm xor ebx,ebx                     /* Rm=0 */ \
        __asm shr ebx,1                       /* And carry=0 */ \
        __asm shftdone:                       /* all done! */ \
        __asm setc [arm7reg+ARMREG_TC*4] \
    }

#define ARM7addrDPari() \
    ARM7addrDPops(); \
    __asm { \
        __asm mov ecx,edx                     /* Make op copy */ \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        \
        __asm shr ecx,7                       /* Shift in shftamount */ \
        __asm and ecx,31                      /* Mask off amount */ \
        __asm jz noshft                       /* Skip if no shift */ \
        __asm sar ebx,cl                      /* Shift operand1 */ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
	\
 	__asm mov ecx,ebx                     /* Make a copy */ \
        __asm sar ebx,31                      /* Fill with hi bit */ \
        __asm shl ecx,1                       /* Carry = Rm[31] */ \
        __asm shftdone:                       /* all done! */ \
        __asm setc [arm7reg+ARMREG_TC*4] \
    }

#define ARM7addrDParr() \
    ARM7addrDPops(); \
    __asm { \
        __asm cmp ecx,15 \
        __asm jl pt3 \
        __asm add ebx,4 \
        __asm pt3: \
        \
        __asm mov ecx,edx                     /* Make op copy */ \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        \
        __asm shr ecx,8                       /* Shift in Rs */ \
        __asm and ecx,15                      /* Mask off Rs */ \
        __asm mov ecx,[arm7reg+ecx*4]         /* Read Rs's value */ \
        __asm and ecx,255                     /* Get the low byte */ \
        __asm jz noshft                       /* Skip if no shift */ \
        __asm cmp cl,32                       /* If greater than 32 */ \
        __asm jge fullshft                    /* Shift all the way */ \
        __asm sar ebx,cl                      /* Shift operand1 */ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
	\
        __asm mov ecx,[arm7reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm jmp shftdone                    /* and skip over */ \
        __asm fullshft:                       /* shift over 31 */ \
 	__asm mov ecx,ebx                     /* Make a copy */ \
        __asm sar ebx,31                      /* Fill with hi bit */ \
        __asm shl ecx,1                       /* Carry = Rm[31] */ \
        __asm shftdone:                       /* all done! */ \
        __asm setc [arm7reg+ARMREG_TC*4] \
    }

#define ARM7addrDPrri() \
    ARM7addrDPops(); \
    __asm { \
        __asm mov ecx,edx                     /* Make op copy */ \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        \
        __asm shr ecx,7                       /* Shift in shftamount */ \
        __asm and ecx,31                      /* Mask off amount */ \
        __asm jz noshft                       /* Skip if no shift */ \
        __asm ror ebx,cl                      /* Shift operand1 */ \
/*        __asm mov ecx,ebx */ \
/*        __asm shr ecx,1 */ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
	\
	__asm push ebx                        /* Keep orig value */ \
        __asm mov ecx,[arm7reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shl ecx,31                      /* Shift to high */ \
        __asm shr ebx,1                       /* 33-bit shift */ \
        __asm or ebx,ecx                      /* Add together */ \
        __asm pop ecx                         /* Get orig back */ \
        __asm shr ecx,1                       /* Carry = Rm[0] */ \
        __asm shftdone:                       /* all done! */ \
        __asm setc [arm7reg+ARMREG_TC*4] \
    }

#define ARM7addrDPrrr() \
    ARM7addrDPops(); \
    __asm { \
        __asm cmp ecx,15 \
        __asm jl pt3 \
        __asm add ebx,4 \
        __asm pt3: \
        \
        __asm mov ecx,edx                     /* Make op copy */ \
        __asm shr edx,12                      /* Extract Rd */ \
        __asm and edx,15                      /* Limit to 15 */ \
        \
        __asm shr ecx,8                       /* Shift in Rs */ \
        __asm and ecx,15                      /* Mask off Rs */ \
        __asm mov ecx,[arm7reg+ecx*4]         /* Read Rs's value */ \
        __asm and ecx,255                     /* Get the low byte */ \
        __asm jz noshft                       /* Skip if no shift */ \
        __asm and cl,31                       /* Check [4:0] */ \
        __asm jz fullshft                     /* And skip if full */ \
        __asm ror ebx,cl                      /* Shift operand1 */ \
/*        __asm mov ecx,ebx*/ \
/*        __asm shr ecx,1*/ \
        __asm jmp shftdone                    /* And skip carry read */ \
        __asm noshft:                         /* Done shifting */ \
	\
        __asm mov ecx,[arm7reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm jmp shftdone                    /* Skip over */ \
        __asm fullshft:                       /* Shifting by full */ \
        __asm mov ecx,ebx                     /* Make a copy of Rm */ \
        __asm shl ecx,1                       /* Carry = hi bit */ \
        __asm shftdone:                       /* all done! */ \
        __asm setc [arm7reg+ARMREG_TC*4] \
    }

/*---eax=operand1, ebx=operand2, edx=destreg----*/

#define ARM7opAND() \
    __asm { \
        __asm and eax,ebx \
        __asm mov [arm7reg+edx*4],eax \
    }

#define ARM7opEOR() \
    __asm { \
        __asm xor eax,ebx \
        __asm mov [arm7reg+edx*4],eax \
    }

#define ARM7opSUB() \
    __asm { \
        __asm sub eax,ebx \
        __asm mov [arm7reg+edx*4],eax \
    }

#define ARM7opRSB() \
    __asm { \
        __asm mov ecx,[arm7reg+ARMREG_TC*4] \
        __asm not ecx \
        __asm shr ecx,1 \
        __asm sub ebx,eax \
        __asm mov [arm7reg+edx*4],ebx \
    }

#define ARM7opADD() \
    __asm { \
        __asm add eax,ebx \
        __asm mov [arm7reg+edx*4],eax \
    }

#define ARM7opADC() \
    __asm { \
        __asm adc eax,ebx \
        __asm mov [arm7reg+edx*4],eax \
    }

#define ARM7opSBC() \
    __asm { \
        __asm mov ecx,[arm7reg+ARMREG_TC*4] \
        __asm not ecx \
        __asm shr ecx,1 \
        __asm sbb eax,ebx \
        __asm mov [arm7reg+edx*4],eax \
    }

#define ARM7opRSC() \
    __asm { \
        __asm mov ecx,[arm7reg+ARMREG_TC*4] \
        __asm not ecx \
        __asm shr ecx,1 \
        __asm sbb ebx,eax \
        __asm mov [arm7reg+edx*4],ebx \
    }

#define ARM7opTST() \
    __asm { \
        __asm and eax,ebx \
    }

#define ARM7opTEQ() \
    __asm { \
        __asm xor eax,ebx \
    }

#define ARM7opCMP() \
    __asm { \
        __asm sub eax,ebx \
    }

#define ARM7opCMN() \
    __asm { \
        __asm add eax,ebx \
    }

#define ARM7opORR() \
    __asm { \
        __asm or eax,ebx \
        __asm mov [arm7reg+edx*4],eax \
    }

#define ARM7opMOV() \
    __asm { \
        __asm or ebx,ebx \
        __asm mov [arm7reg+edx*4],ebx \
    }

#define ARM7opBIC() \
    __asm { \
        __asm not ebx \
        __asm and ebx,eax \
        __asm mov [arm7reg+edx*4],ebx \
    }

#define ARM7opMVN() \
    __asm { \
        __asm not ebx \
        __asm mov [arm7reg+edx*4],ebx \
    }

#define ARM7addrDPflagR15() \
    __asm { \
        __asm cmp edx,15                       /* Is Rd 15? */ \
        __asm jl flagdone                      /* If not, skip */ \
        __asm mov eax,[arm7reg+ARMREG_MODE*4]  /* Get current mode */ \
        __asm mov eax,[arm7reg+ARMREG_SUSR*4+eax*4]     /* Read mode's SPSR */ \
        __asm mov ecx,eax                      /* Make a copy of it */ \
	__asm mov ebx,[arm7reg+ARMREG_CPSR*4]  /* Read CPSR */ \
	__asm and ecx,15                       /* Mask spsr mode */ \
	__asm and ebx,15                       /* and cpsr mode */ \
	__asm cmp ecx,ebx                      /* They the same? */ \
	__asm je flageq                        /* If so, skip */ \
	__asm push eax                         /* modesw mods eax */ \
	__asm push [ARM7cpsrtomode+ecx*4]       /* New mode */ \
	__asm push [ARM7cpsrtomode+ebx*4]       /* Old mode */ \
	__asm call ARM7modesw                  /* Switch modes */ \
	__asm pop eax                          /* Pull param2 */ \
	__asm pop eax                          /* and param1 */ \
	__asm pop eax                          /* Get eax back */ \
	__asm flageq:                          /* Modes unchanged */ \
	__asm mov [arm7reg+ARMREG_CPSR*4],eax  /* CPSR=SPSR */ \
	__asm call ARM7splitCPSR               /* Redo flags */ \
        __asm flagdone:                        /* All done! */ \
    }

#define ARM7addrDPflagBIT() \
    __asm { \
        __asm sets  [arm7reg+ARMREG_FN*4]      /* Copy N flag */ \
        __asm setz  [arm7reg+ARMREG_FZ*4]      /* and Z flag */ \
        __asm mov ecx,[arm7reg+ARMREG_TC*4] \
        __asm mov [arm7reg+ARMREG_FC*4],ecx \
    } \
    ARM7addrDPflagR15()

#define ARM7addrDPflagADD() \
    __asm { \
        __asm sets  [arm7reg+ARMREG_FN*4]    /* Copy N flag */ \
        __asm setz  [arm7reg+ARMREG_FZ*4]    /* and Z flag */ \
        __asm seto  [arm7reg+ARMREG_FV*4]    /* V too */ \
        __asm setc [arm7reg+ARMREG_FC*4]    /* REVERSE C flag */ \
    } \
    ARM7addrDPflagR15()

#define ARM7addrDPflagADC() \
    __asm { \
        __asm sets  [arm7reg+ARMREG_FN*4]    /* Copy N flag */ \
        __asm setz  [arm7reg+ARMREG_FZ*4]    /* and Z flag */ \
        __asm seto  [arm7reg+ARMREG_FV*4]    /* V too */ \
        __asm setc [arm7reg+ARMREG_FC*4]    /* REVERSE C flag */ \
    } \
    ARM7addrDPflagR15()

#define ARM7addrDPflagSUB() \
    __asm { \
        __asm sets  [arm7reg+ARMREG_FN*4]    /* Copy N flag */ \
        __asm setz  [arm7reg+ARMREG_FZ*4]    /* and Z flag */ \
        __asm seto  [arm7reg+ARMREG_FV*4]    /* V too */ \
        __asm setnc [arm7reg+ARMREG_FC*4]    /* REVERSE C flag */ \
    } \
    ARM7addrDPflagR15()

#define ARM7addrDPflagSBC() \
    __asm { \
        __asm sets  [arm7reg+ARMREG_FN*4]    /* Copy N flag */ \
        __asm setz  [arm7reg+ARMREG_FZ*4]    /* and Z flag */ \
        __asm seto  [arm7reg+ARMREG_FV*4]    /* V too */ \
        __asm setnc [arm7reg+ARMREG_FC*4]    /* REVERSE C flag */ \
    } \
    ARM7addrDPflagR15()

#define ARM7addrDPflagRSB() \
    __asm { \
        __asm sets  [arm7reg+ARMREG_FN*4]    /* Copy N flag */ \
        __asm setz  [arm7reg+ARMREG_FZ*4]    /* and Z flag */ \
        __asm seto  [arm7reg+ARMREG_FV*4]    /* V too */ \
        __asm setnc [arm7reg+ARMREG_FC*4]    /* REVERSE C flag */ \
    } \
    ARM7addrDPflagR15()

#define ARM7addrDPflagRSC() \
    __asm { \
        __asm sets  [arm7reg+ARMREG_FN*4]    /* Copy N flag */ \
        __asm setz  [arm7reg+ARMREG_FZ*4]    /* and Z flag */ \
        __asm seto  [arm7reg+ARMREG_FV*4]    /* V too */ \
        __asm setnc [arm7reg+ARMREG_FC*4]    /* REVERSE C flag */ \
    } \
    ARM7addrDPflagR15()

#define ARM7addrDPflagCMN() \
    __asm { \
        __asm sets  [arm7reg+ARMREG_FN*4]    /* Copy N flag */ \
        __asm setz  [arm7reg+ARMREG_FZ*4]    /* and Z flag */ \
        __asm seto  [arm7reg+ARMREG_FV*4]    /* V too */ \
        __asm setc [arm7reg+ARMREG_FC*4]    /* REVERSE C flag */ \
    } \
    ARM7addrDPflagR15()

#define ARM7addrDPflagCMP() \
    __asm { \
        __asm sets  [arm7reg+ARMREG_FN*4]    /* Copy N flag */ \
        __asm setz  [arm7reg+ARMREG_FZ*4]    /* and Z flag */ \
        __asm seto  [arm7reg+ARMREG_FV*4]    /* V too */ \
        __asm setnc [arm7reg+ARMREG_FC*4]    /* REVERSE C flag */ \
    } \
    ARM7addrDPflagR15()

OPCODE ARM7opANDreg() { ARM7addrDPreg(); ARM7opAND(); return 2; }
OPCODE ARM7opANDimm() { ARM7addrDPimm(); ARM7opAND(); return 1; }
OPCODE ARM7opANDlli() { ARM7addrDPlli(); ARM7opAND(); return 2; }
OPCODE ARM7opANDllr() { ARM7addrDPllr(); ARM7opAND(); return 2; }
OPCODE ARM7opANDlri() { ARM7addrDPlri(); ARM7opAND(); return 2; }
OPCODE ARM7opANDlrr() { ARM7addrDPlrr(); ARM7opAND(); return 2; }
OPCODE ARM7opANDari() { ARM7addrDPari(); ARM7opAND(); return 2; }
OPCODE ARM7opANDarr() { ARM7addrDParr(); ARM7opAND(); return 2; }
OPCODE ARM7opANDrri() { ARM7addrDPrri(); ARM7opAND(); return 2; }
OPCODE ARM7opANDrrr() { ARM7addrDPrrr(); ARM7opAND(); return 2; }

OPCODE ARM7opEORreg() { ARM7addrDPreg(); ARM7opEOR(); return 2; }
OPCODE ARM7opEORimm() { ARM7addrDPimm(); ARM7opEOR(); return 1; }
OPCODE ARM7opEORlli() { ARM7addrDPlli(); ARM7opEOR(); return 2; }
OPCODE ARM7opEORllr() { ARM7addrDPllr(); ARM7opEOR(); return 2; }
OPCODE ARM7opEORlri() { ARM7addrDPlri(); ARM7opEOR(); return 2; }
OPCODE ARM7opEORlrr() { ARM7addrDPlrr(); ARM7opEOR(); return 2; }
OPCODE ARM7opEORari() { ARM7addrDPari(); ARM7opEOR(); return 2; }
OPCODE ARM7opEORarr() { ARM7addrDParr(); ARM7opEOR(); return 2; }
OPCODE ARM7opEORrri() { ARM7addrDPrri(); ARM7opEOR(); return 2; }
OPCODE ARM7opEORrrr() { ARM7addrDPrrr(); ARM7opEOR(); return 2; }

OPCODE ARM7opSUBreg() { ARM7addrDPreg(); ARM7opSUB(); return 2; }
OPCODE ARM7opSUBimm() { ARM7addrDPimm(); ARM7opSUB(); return 1; }
OPCODE ARM7opSUBlli() { ARM7addrDPlli(); ARM7opSUB(); return 2; }
OPCODE ARM7opSUBllr() { ARM7addrDPllr(); ARM7opSUB(); return 2; }
OPCODE ARM7opSUBlri() { ARM7addrDPlri(); ARM7opSUB(); return 2; }
OPCODE ARM7opSUBlrr() { ARM7addrDPlrr(); ARM7opSUB(); return 2; }
OPCODE ARM7opSUBari() { ARM7addrDPari(); ARM7opSUB(); return 2; }
OPCODE ARM7opSUBarr() { ARM7addrDParr(); ARM7opSUB(); return 2; }
OPCODE ARM7opSUBrri() { ARM7addrDPrri(); ARM7opSUB(); return 2; }
OPCODE ARM7opSUBrrr() { ARM7addrDPrrr(); ARM7opSUB(); return 2; }

OPCODE ARM7opRSBreg() { ARM7addrDPreg(); ARM7opRSB(); return 2; }
OPCODE ARM7opRSBimm() { ARM7addrDPimm(); ARM7opRSB(); return 1; }
OPCODE ARM7opRSBlli() { ARM7addrDPlli(); ARM7opRSB(); return 2; }
OPCODE ARM7opRSBllr() { ARM7addrDPllr(); ARM7opRSB(); return 2; }
OPCODE ARM7opRSBlri() { ARM7addrDPlri(); ARM7opRSB(); return 2; }
OPCODE ARM7opRSBlrr() { ARM7addrDPlrr(); ARM7opRSB(); return 2; }
OPCODE ARM7opRSBari() { ARM7addrDPari(); ARM7opRSB(); return 2; }
OPCODE ARM7opRSBarr() { ARM7addrDParr(); ARM7opRSB(); return 2; }
OPCODE ARM7opRSBrri() { ARM7addrDPrri(); ARM7opRSB(); return 2; }
OPCODE ARM7opRSBrrr() { ARM7addrDPrrr(); ARM7opRSB(); return 2; }

OPCODE ARM7opADDreg() { ARM7addrDPreg(); ARM7opADD(); return 2; }
OPCODE ARM7opADDimm() { ARM7addrDPimm(); ARM7opADD(); return 1; }
OPCODE ARM7opADDlli() { ARM7addrDPlli(); ARM7opADD(); return 2; }
OPCODE ARM7opADDllr() { ARM7addrDPllr(); ARM7opADD(); return 2; }
OPCODE ARM7opADDlri() { ARM7addrDPlri(); ARM7opADD(); return 2; }
OPCODE ARM7opADDlrr() { ARM7addrDPlrr(); ARM7opADD(); return 2; }
OPCODE ARM7opADDari() { ARM7addrDPari(); ARM7opADD(); return 2; }
OPCODE ARM7opADDarr() { ARM7addrDParr(); ARM7opADD(); return 2; }
OPCODE ARM7opADDrri() { ARM7addrDPrri(); ARM7opADD(); return 2; }
OPCODE ARM7opADDrrr() { ARM7addrDPrrr(); ARM7opADD(); return 2; }

OPCODE ARM7opADCreg() { ARM7addrDPreg(); ARM7opADC(); return 2; }
OPCODE ARM7opADCimm() { ARM7addrDPimm(); ARM7opADC(); return 1; }
OPCODE ARM7opADClli() { ARM7addrDPlli(); ARM7opADC(); return 2; }
OPCODE ARM7opADCllr() { ARM7addrDPllr(); ARM7opADC(); return 2; }
OPCODE ARM7opADClri() { ARM7addrDPlri(); ARM7opADC(); return 2; }
OPCODE ARM7opADClrr() { ARM7addrDPlrr(); ARM7opADC(); return 2; }
OPCODE ARM7opADCari() { ARM7addrDPari(); ARM7opADC(); return 2; }
OPCODE ARM7opADCarr() { ARM7addrDParr(); ARM7opADC(); return 2; }
OPCODE ARM7opADCrri() { ARM7addrDPrri(); ARM7opADC(); return 2; }
OPCODE ARM7opADCrrr() { ARM7addrDPrrr(); ARM7opADC(); return 2; }

OPCODE ARM7opSBCreg() { ARM7addrDPreg(); ARM7opSBC(); return 2; }
OPCODE ARM7opSBCimm() { ARM7addrDPimm(); ARM7opSBC(); return 1; }
OPCODE ARM7opSBClli() { ARM7addrDPlli(); ARM7opSBC(); return 2; }
OPCODE ARM7opSBCllr() { ARM7addrDPllr(); ARM7opSBC(); return 2; }
OPCODE ARM7opSBClri() { ARM7addrDPlri(); ARM7opSBC(); return 2; }
OPCODE ARM7opSBClrr() { ARM7addrDPlrr(); ARM7opSBC(); return 2; }
OPCODE ARM7opSBCari() { ARM7addrDPari(); ARM7opSBC(); return 2; }
OPCODE ARM7opSBCarr() { ARM7addrDParr(); ARM7opSBC(); return 2; }
OPCODE ARM7opSBCrri() { ARM7addrDPrri(); ARM7opSBC(); return 2; }
OPCODE ARM7opSBCrrr() { ARM7addrDPrrr(); ARM7opSBC(); return 2; }

OPCODE ARM7opRSCreg() { ARM7addrDPreg(); ARM7opRSC(); return 2; }
OPCODE ARM7opRSCimm() { ARM7addrDPimm(); ARM7opRSC(); return 1; }
OPCODE ARM7opRSClli() { ARM7addrDPlli(); ARM7opRSC(); return 2; }
OPCODE ARM7opRSCllr() { ARM7addrDPllr(); ARM7opRSC(); return 2; }
OPCODE ARM7opRSClri() { ARM7addrDPlri(); ARM7opRSC(); return 2; }
OPCODE ARM7opRSClrr() { ARM7addrDPlrr(); ARM7opRSC(); return 2; }
OPCODE ARM7opRSCari() { ARM7addrDPari(); ARM7opRSC(); return 2; }
OPCODE ARM7opRSCarr() { ARM7addrDParr(); ARM7opRSC(); return 2; }
OPCODE ARM7opRSCrri() { ARM7addrDPrri(); ARM7opRSC(); return 2; }
OPCODE ARM7opRSCrrr() { ARM7addrDPrrr(); ARM7opRSC(); return 2; }

OPCODE ARM7opTSTreg() { ARM7addrDPreg(); ARM7opTST(); return 2; }
OPCODE ARM7opTSTimm() { ARM7addrDPimm(); ARM7opTST(); return 1; }
OPCODE ARM7opTSTlli() { ARM7addrDPlli(); ARM7opTST(); return 2; }
OPCODE ARM7opTSTllr() { ARM7addrDPllr(); ARM7opTST(); return 2; }
OPCODE ARM7opTSTlri() { ARM7addrDPlri(); ARM7opTST(); return 2; }
OPCODE ARM7opTSTlrr() { ARM7addrDPlrr(); ARM7opTST(); return 2; }
OPCODE ARM7opTSTari() { ARM7addrDPari(); ARM7opTST(); return 2; }
OPCODE ARM7opTSTarr() { ARM7addrDParr(); ARM7opTST(); return 2; }
OPCODE ARM7opTSTrri() { ARM7addrDPrri(); ARM7opTST(); return 2; }
OPCODE ARM7opTSTrrr() { ARM7addrDPrrr(); ARM7opTST(); return 2; }

OPCODE ARM7opTEQreg() { ARM7addrDPreg(); ARM7opTEQ(); return 2; }
OPCODE ARM7opTEQimm() { ARM7addrDPimm(); ARM7opTEQ(); return 1; }
OPCODE ARM7opTEQlli() { ARM7addrDPlli(); ARM7opTEQ(); return 2; }
OPCODE ARM7opTEQllr() { ARM7addrDPllr(); ARM7opTEQ(); return 2; }
OPCODE ARM7opTEQlri() { ARM7addrDPlri(); ARM7opTEQ(); return 2; }
OPCODE ARM7opTEQlrr() { ARM7addrDPlrr(); ARM7opTEQ(); return 2; }
OPCODE ARM7opTEQari() { ARM7addrDPari(); ARM7opTEQ(); return 2; }
OPCODE ARM7opTEQarr() { ARM7addrDParr(); ARM7opTEQ(); return 2; }
OPCODE ARM7opTEQrri() { ARM7addrDPrri(); ARM7opTEQ(); return 2; }
OPCODE ARM7opTEQrrr() { ARM7addrDPrrr(); ARM7opTEQ(); return 2; }

OPCODE ARM7opCMPreg() { ARM7addrDPreg(); ARM7opCMP(); return 2; }
OPCODE ARM7opCMPimm() { ARM7addrDPimm(); ARM7opCMP(); return 1; }
OPCODE ARM7opCMPlli() { ARM7addrDPlli(); ARM7opCMP(); return 2; }
OPCODE ARM7opCMPllr() { ARM7addrDPllr(); ARM7opCMP(); return 2; }
OPCODE ARM7opCMPlri() { ARM7addrDPlri(); ARM7opCMP(); return 2; }
OPCODE ARM7opCMPlrr() { ARM7addrDPlrr(); ARM7opCMP(); return 2; }
OPCODE ARM7opCMPari() { ARM7addrDPari(); ARM7opCMP(); return 2; }
OPCODE ARM7opCMParr() { ARM7addrDParr(); ARM7opCMP(); return 2; }
OPCODE ARM7opCMPrri() { ARM7addrDPrri(); ARM7opCMP(); return 2; }
OPCODE ARM7opCMPrrr() { ARM7addrDPrrr(); ARM7opCMP(); return 2; }

OPCODE ARM7opCMNreg() { ARM7addrDPreg(); ARM7opCMN(); return 2; }
OPCODE ARM7opCMNimm() { ARM7addrDPimm(); ARM7opCMN(); return 1; }
OPCODE ARM7opCMNlli() { ARM7addrDPlli(); ARM7opCMN(); return 2; }
OPCODE ARM7opCMNllr() { ARM7addrDPllr(); ARM7opCMN(); return 2; }
OPCODE ARM7opCMNlri() { ARM7addrDPlri(); ARM7opCMN(); return 2; }
OPCODE ARM7opCMNlrr() { ARM7addrDPlrr(); ARM7opCMN(); return 2; }
OPCODE ARM7opCMNari() { ARM7addrDPari(); ARM7opCMN(); return 2; }
OPCODE ARM7opCMNarr() { ARM7addrDParr(); ARM7opCMN(); return 2; }
OPCODE ARM7opCMNrri() { ARM7addrDPrri(); ARM7opCMN(); return 2; }
OPCODE ARM7opCMNrrr() { ARM7addrDPrrr(); ARM7opCMN(); return 2; }

OPCODE ARM7opORRreg() { ARM7addrDPreg(); ARM7opORR(); return 2; }
OPCODE ARM7opORRimm() { ARM7addrDPimm(); ARM7opORR(); return 1; }
OPCODE ARM7opORRlli() { ARM7addrDPlli(); ARM7opORR(); return 2; }
OPCODE ARM7opORRllr() { ARM7addrDPllr(); ARM7opORR(); return 2; }
OPCODE ARM7opORRlri() { ARM7addrDPlri(); ARM7opORR(); return 2; }
OPCODE ARM7opORRlrr() { ARM7addrDPlrr(); ARM7opORR(); return 2; }
OPCODE ARM7opORRari() { ARM7addrDPari(); ARM7opORR(); return 2; }
OPCODE ARM7opORRarr() { ARM7addrDParr(); ARM7opORR(); return 2; }
OPCODE ARM7opORRrri() { ARM7addrDPrri(); ARM7opORR(); return 2; }
OPCODE ARM7opORRrrr() { ARM7addrDPrrr(); ARM7opORR(); return 2; }

OPCODE ARM7opMOVreg() { ARM7addrDPreg(); ARM7opMOV(); return 2; }
OPCODE ARM7opMOVimm() { ARM7addrDPimm(); ARM7opMOV(); return 1; }
OPCODE ARM7opMOVlli() { ARM7addrDPlli(); ARM7opMOV(); return 2; }
OPCODE ARM7opMOVllr() { ARM7addrDPllr(); ARM7opMOV(); return 2; }
OPCODE ARM7opMOVlri() { ARM7addrDPlri(); ARM7opMOV(); return 2; }
OPCODE ARM7opMOVlrr() { ARM7addrDPlrr(); ARM7opMOV(); return 2; }
OPCODE ARM7opMOVari() { ARM7addrDPari(); ARM7opMOV(); return 2; }
OPCODE ARM7opMOVarr() { ARM7addrDParr(); ARM7opMOV(); return 2; }
OPCODE ARM7opMOVrri() { ARM7addrDPrri(); ARM7opMOV(); return 2; }
OPCODE ARM7opMOVrrr() { ARM7addrDPrrr(); ARM7opMOV(); return 2; }

OPCODE ARM7opBICreg() { ARM7addrDPreg(); ARM7opBIC(); return 2; }
OPCODE ARM7opBICimm() { ARM7addrDPimm(); ARM7opBIC(); return 1; }
OPCODE ARM7opBIClli() { ARM7addrDPlli(); ARM7opBIC(); return 2; }
OPCODE ARM7opBICllr() { ARM7addrDPllr(); ARM7opBIC(); return 2; }
OPCODE ARM7opBIClri() { ARM7addrDPlri(); ARM7opBIC(); return 2; }
OPCODE ARM7opBIClrr() { ARM7addrDPlrr(); ARM7opBIC(); return 2; }
OPCODE ARM7opBICari() { ARM7addrDPari(); ARM7opBIC(); return 2; }
OPCODE ARM7opBICarr() { ARM7addrDParr(); ARM7opBIC(); return 2; }
OPCODE ARM7opBICrri() { ARM7addrDPrri(); ARM7opBIC(); return 2; }
OPCODE ARM7opBICrrr() { ARM7addrDPrrr(); ARM7opBIC(); return 2; }

OPCODE ARM7opMVNreg() { ARM7addrDPreg(); ARM7opMVN(); return 2; }
OPCODE ARM7opMVNimm() { ARM7addrDPimm(); ARM7opMVN(); return 1; }
OPCODE ARM7opMVNlli() { ARM7addrDPlli(); ARM7opMVN(); return 2; }
OPCODE ARM7opMVNllr() { ARM7addrDPllr(); ARM7opMVN(); return 2; }
OPCODE ARM7opMVNlri() { ARM7addrDPlri(); ARM7opMVN(); return 2; }
OPCODE ARM7opMVNlrr() { ARM7addrDPlrr(); ARM7opMVN(); return 2; }
OPCODE ARM7opMVNari() { ARM7addrDPari(); ARM7opMVN(); return 2; }
OPCODE ARM7opMVNarr() { ARM7addrDParr(); ARM7opMVN(); return 2; }
OPCODE ARM7opMVNrri() { ARM7addrDPrri(); ARM7opMVN(); return 2; }
OPCODE ARM7opMVNrrr() { ARM7addrDPrrr(); ARM7opMVN(); return 2; }

OPCODE ARM7opANDSreg() { ARM7addrDPreg(); ARM7opAND(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opANDSimm() { ARM7addrDPimm(); ARM7opAND(); ARM7addrDPflagBIT(); return 1; }
OPCODE ARM7opANDSlli() { ARM7addrDPlli(); ARM7opAND(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opANDSllr() { ARM7addrDPllr(); ARM7opAND(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opANDSlri() { ARM7addrDPlri(); ARM7opAND(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opANDSlrr() { ARM7addrDPlrr(); ARM7opAND(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opANDSari() { ARM7addrDPari(); ARM7opAND(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opANDSarr() { ARM7addrDParr(); ARM7opAND(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opANDSrri() { ARM7addrDPrri(); ARM7opAND(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opANDSrrr() { ARM7addrDPrrr(); ARM7opAND(); ARM7addrDPflagBIT(); return 2; }

OPCODE ARM7opEORSreg() { ARM7addrDPreg(); ARM7opEOR(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opEORSimm() { ARM7addrDPimm(); ARM7opEOR(); ARM7addrDPflagBIT(); return 1; }
OPCODE ARM7opEORSlli() { ARM7addrDPlli(); ARM7opEOR(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opEORSllr() { ARM7addrDPllr(); ARM7opEOR(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opEORSlri() { ARM7addrDPlri(); ARM7opEOR(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opEORSlrr() { ARM7addrDPlrr(); ARM7opEOR(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opEORSari() { ARM7addrDPari(); ARM7opEOR(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opEORSarr() { ARM7addrDParr(); ARM7opEOR(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opEORSrri() { ARM7addrDPrri(); ARM7opEOR(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opEORSrrr() { ARM7addrDPrrr(); ARM7opEOR(); ARM7addrDPflagBIT(); return 2; }

OPCODE ARM7opSUBSreg() { ARM7addrDPreg(); ARM7opSUB(); ARM7addrDPflagSUB(); return 2; }
OPCODE ARM7opSUBSimm() { ARM7addrDPimm(); ARM7opSUB(); ARM7addrDPflagSUB(); return 1; }
OPCODE ARM7opSUBSlli() { ARM7addrDPlli(); ARM7opSUB(); ARM7addrDPflagSUB(); return 2; }
OPCODE ARM7opSUBSllr() { ARM7addrDPllr(); ARM7opSUB(); ARM7addrDPflagSUB(); return 2; }
OPCODE ARM7opSUBSlri() { ARM7addrDPlri(); ARM7opSUB(); ARM7addrDPflagSUB(); return 2; }
OPCODE ARM7opSUBSlrr() { ARM7addrDPlrr(); ARM7opSUB(); ARM7addrDPflagSUB(); return 2; }
OPCODE ARM7opSUBSari() { ARM7addrDPari(); ARM7opSUB(); ARM7addrDPflagSUB(); return 2; }
OPCODE ARM7opSUBSarr() { ARM7addrDParr(); ARM7opSUB(); ARM7addrDPflagSUB(); return 2; }
OPCODE ARM7opSUBSrri() { ARM7addrDPrri(); ARM7opSUB(); ARM7addrDPflagSUB(); return 2; }
OPCODE ARM7opSUBSrrr() { ARM7addrDPrrr(); ARM7opSUB(); ARM7addrDPflagSUB(); return 2; }

OPCODE ARM7opRSBSreg() { ARM7addrDPreg(); ARM7opRSB(); ARM7addrDPflagRSB(); return 2; }
OPCODE ARM7opRSBSimm() { ARM7addrDPimm(); ARM7opRSB(); ARM7addrDPflagRSB(); return 1; }
OPCODE ARM7opRSBSlli() { ARM7addrDPlli(); ARM7opRSB(); ARM7addrDPflagRSB(); return 2; }
OPCODE ARM7opRSBSllr() { ARM7addrDPllr(); ARM7opRSB(); ARM7addrDPflagRSB(); return 2; }
OPCODE ARM7opRSBSlri() { ARM7addrDPlri(); ARM7opRSB(); ARM7addrDPflagRSB(); return 2; }
OPCODE ARM7opRSBSlrr() { ARM7addrDPlrr(); ARM7opRSB(); ARM7addrDPflagRSB(); return 2; }
OPCODE ARM7opRSBSari() { ARM7addrDPari(); ARM7opRSB(); ARM7addrDPflagRSB(); return 2; }
OPCODE ARM7opRSBSarr() { ARM7addrDParr(); ARM7opRSB(); ARM7addrDPflagRSB(); return 2; }
OPCODE ARM7opRSBSrri() { ARM7addrDPrri(); ARM7opRSB(); ARM7addrDPflagRSB(); return 2; }
OPCODE ARM7opRSBSrrr() { ARM7addrDPrrr(); ARM7opRSB(); ARM7addrDPflagRSB(); return 2; }

OPCODE ARM7opADDSreg() { ARM7addrDPreg(); ARM7opADD(); ARM7addrDPflagADD(); return 2; }
OPCODE ARM7opADDSimm() { ARM7addrDPimm(); ARM7opADD(); ARM7addrDPflagADD(); return 1; }
OPCODE ARM7opADDSlli() { ARM7addrDPlli(); ARM7opADD(); ARM7addrDPflagADD(); return 2; }
OPCODE ARM7opADDSllr() { ARM7addrDPllr(); ARM7opADD(); ARM7addrDPflagADD(); return 2; }
OPCODE ARM7opADDSlri() { ARM7addrDPlri(); ARM7opADD(); ARM7addrDPflagADD(); return 2; }
OPCODE ARM7opADDSlrr() { ARM7addrDPlrr(); ARM7opADD(); ARM7addrDPflagADD(); return 2; }
OPCODE ARM7opADDSari() { ARM7addrDPari(); ARM7opADD(); ARM7addrDPflagADD(); return 2; }
OPCODE ARM7opADDSarr() { ARM7addrDParr(); ARM7opADD(); ARM7addrDPflagADD(); return 2; }
OPCODE ARM7opADDSrri() { ARM7addrDPrri(); ARM7opADD(); ARM7addrDPflagADD(); return 2; }
OPCODE ARM7opADDSrrr() { ARM7addrDPrrr(); ARM7opADD(); ARM7addrDPflagADD(); return 2; }

OPCODE ARM7opADCSreg() { ARM7addrDPreg(); ARM7opADC(); ARM7addrDPflagADC(); return 2; }
OPCODE ARM7opADCSimm() { ARM7addrDPimm(); ARM7opADC(); ARM7addrDPflagADC(); return 1; }
OPCODE ARM7opADCSlli() { ARM7addrDPlli(); ARM7opADC(); ARM7addrDPflagADC(); return 2; }
OPCODE ARM7opADCSllr() { ARM7addrDPllr(); ARM7opADC(); ARM7addrDPflagADC(); return 2; }
OPCODE ARM7opADCSlri() { ARM7addrDPlri(); ARM7opADC(); ARM7addrDPflagADC(); return 2; }
OPCODE ARM7opADCSlrr() { ARM7addrDPlrr(); ARM7opADC(); ARM7addrDPflagADC(); return 2; }
OPCODE ARM7opADCSari() { ARM7addrDPari(); ARM7opADC(); ARM7addrDPflagADC(); return 2; }
OPCODE ARM7opADCSarr() { ARM7addrDParr(); ARM7opADC(); ARM7addrDPflagADC(); return 2; }
OPCODE ARM7opADCSrri() { ARM7addrDPrri(); ARM7opADC(); ARM7addrDPflagADC(); return 2; }
OPCODE ARM7opADCSrrr() { ARM7addrDPrrr(); ARM7opADC(); ARM7addrDPflagADC(); return 2; }

OPCODE ARM7opSBCSreg() { ARM7addrDPreg(); ARM7opSBC(); ARM7addrDPflagSBC(); return 2; }
OPCODE ARM7opSBCSimm() { ARM7addrDPimm(); ARM7opSBC(); ARM7addrDPflagSBC(); return 1; }
OPCODE ARM7opSBCSlli() { ARM7addrDPlli(); ARM7opSBC(); ARM7addrDPflagSBC(); return 2; }
OPCODE ARM7opSBCSllr() { ARM7addrDPllr(); ARM7opSBC(); ARM7addrDPflagSBC(); return 2; }
OPCODE ARM7opSBCSlri() { ARM7addrDPlri(); ARM7opSBC(); ARM7addrDPflagSBC(); return 2; }
OPCODE ARM7opSBCSlrr() { ARM7addrDPlrr(); ARM7opSBC(); ARM7addrDPflagSBC(); return 2; }
OPCODE ARM7opSBCSari() { ARM7addrDPari(); ARM7opSBC(); ARM7addrDPflagSBC(); return 2; }
OPCODE ARM7opSBCSarr() { ARM7addrDParr(); ARM7opSBC(); ARM7addrDPflagSBC(); return 2; }
OPCODE ARM7opSBCSrri() { ARM7addrDPrri(); ARM7opSBC(); ARM7addrDPflagSBC(); return 2; }
OPCODE ARM7opSBCSrrr() { ARM7addrDPrrr(); ARM7opSBC(); ARM7addrDPflagSBC(); return 2; }

OPCODE ARM7opRSCSreg() { ARM7addrDPreg(); ARM7opRSC(); ARM7addrDPflagRSC(); return 2; }
OPCODE ARM7opRSCSimm() { ARM7addrDPimm(); ARM7opRSC(); ARM7addrDPflagRSC(); return 1; }
OPCODE ARM7opRSCSlli() { ARM7addrDPlli(); ARM7opRSC(); ARM7addrDPflagRSC(); return 2; }
OPCODE ARM7opRSCSllr() { ARM7addrDPllr(); ARM7opRSC(); ARM7addrDPflagRSC(); return 2; }
OPCODE ARM7opRSCSlri() { ARM7addrDPlri(); ARM7opRSC(); ARM7addrDPflagRSC(); return 2; }
OPCODE ARM7opRSCSlrr() { ARM7addrDPlrr(); ARM7opRSC(); ARM7addrDPflagRSC(); return 2; }
OPCODE ARM7opRSCSari() { ARM7addrDPari(); ARM7opRSC(); ARM7addrDPflagRSC(); return 2; }
OPCODE ARM7opRSCSarr() { ARM7addrDParr(); ARM7opRSC(); ARM7addrDPflagRSC(); return 2; }
OPCODE ARM7opRSCSrri() { ARM7addrDPrri(); ARM7opRSC(); ARM7addrDPflagRSC(); return 2; }
OPCODE ARM7opRSCSrrr() { ARM7addrDPrrr(); ARM7opRSC(); ARM7addrDPflagRSC(); return 2; }

OPCODE ARM7opTSTSreg() { ARM7addrDPreg(); ARM7opTST(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opTSTSimm() { ARM7addrDPimm(); ARM7opTST(); ARM7addrDPflagBIT(); return 1; }
OPCODE ARM7opTSTSlli() { ARM7addrDPlli(); ARM7opTST(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opTSTSllr() { ARM7addrDPllr(); ARM7opTST(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opTSTSlri() { ARM7addrDPlri(); ARM7opTST(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opTSTSlrr() { ARM7addrDPlrr(); ARM7opTST(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opTSTSari() { ARM7addrDPari(); ARM7opTST(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opTSTSarr() { ARM7addrDParr(); ARM7opTST(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opTSTSrri() { ARM7addrDPrri(); ARM7opTST(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opTSTSrrr() { ARM7addrDPrrr(); ARM7opTST(); ARM7addrDPflagBIT(); return 2; }

OPCODE ARM7opTEQSreg() { ARM7addrDPreg(); ARM7opTEQ(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opTEQSimm() { ARM7addrDPimm(); ARM7opTEQ(); ARM7addrDPflagBIT(); return 1; }
OPCODE ARM7opTEQSlli() { ARM7addrDPlli(); ARM7opTEQ(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opTEQSllr() { ARM7addrDPllr(); ARM7opTEQ(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opTEQSlri() { ARM7addrDPlri(); ARM7opTEQ(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opTEQSlrr() { ARM7addrDPlrr(); ARM7opTEQ(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opTEQSari() { ARM7addrDPari(); ARM7opTEQ(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opTEQSarr() { ARM7addrDParr(); ARM7opTEQ(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opTEQSrri() { ARM7addrDPrri(); ARM7opTEQ(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opTEQSrrr() { ARM7addrDPrrr(); ARM7opTEQ(); ARM7addrDPflagBIT(); return 2; }

OPCODE ARM7opCMPSreg() { ARM7addrDPreg(); ARM7opCMP(); ARM7addrDPflagCMP(); return 2; }
OPCODE ARM7opCMPSimm() { ARM7addrDPimm(); ARM7opCMP(); ARM7addrDPflagCMP(); return 1; }
OPCODE ARM7opCMPSlli() { ARM7addrDPlli(); ARM7opCMP(); ARM7addrDPflagCMP(); return 2; }
OPCODE ARM7opCMPSllr() { ARM7addrDPllr(); ARM7opCMP(); ARM7addrDPflagCMP(); return 2; }
OPCODE ARM7opCMPSlri() { ARM7addrDPlri(); ARM7opCMP(); ARM7addrDPflagCMP(); return 2; }
OPCODE ARM7opCMPSlrr() { ARM7addrDPlrr(); ARM7opCMP(); ARM7addrDPflagCMP(); return 2; }
OPCODE ARM7opCMPSari() { ARM7addrDPari(); ARM7opCMP(); ARM7addrDPflagCMP(); return 2; }
OPCODE ARM7opCMPSarr() { ARM7addrDParr(); ARM7opCMP(); ARM7addrDPflagCMP(); return 2; }
OPCODE ARM7opCMPSrri() { ARM7addrDPrri(); ARM7opCMP(); ARM7addrDPflagCMP(); return 2; }
OPCODE ARM7opCMPSrrr() { ARM7addrDPrrr(); ARM7opCMP(); ARM7addrDPflagCMP(); return 2; }

OPCODE ARM7opCMNSreg() { ARM7addrDPreg(); ARM7opCMN(); ARM7addrDPflagCMN(); return 2; }
OPCODE ARM7opCMNSimm() { ARM7addrDPimm(); ARM7opCMN(); ARM7addrDPflagCMN(); return 1; }
OPCODE ARM7opCMNSlli() { ARM7addrDPlli(); ARM7opCMN(); ARM7addrDPflagCMN(); return 2; }
OPCODE ARM7opCMNSllr() { ARM7addrDPllr(); ARM7opCMN(); ARM7addrDPflagCMN(); return 2; }
OPCODE ARM7opCMNSlri() { ARM7addrDPlri(); ARM7opCMN(); ARM7addrDPflagCMN(); return 2; }
OPCODE ARM7opCMNSlrr() { ARM7addrDPlrr(); ARM7opCMN(); ARM7addrDPflagCMN(); return 2; }
OPCODE ARM7opCMNSari() { ARM7addrDPari(); ARM7opCMN(); ARM7addrDPflagCMN(); return 2; }
OPCODE ARM7opCMNSarr() { ARM7addrDParr(); ARM7opCMN(); ARM7addrDPflagCMN(); return 2; }
OPCODE ARM7opCMNSrri() { ARM7addrDPrri(); ARM7opCMN(); ARM7addrDPflagCMN(); return 2; }
OPCODE ARM7opCMNSrrr() { ARM7addrDPrrr(); ARM7opCMN(); ARM7addrDPflagCMN(); return 2; }

OPCODE ARM7opORRSreg() { ARM7addrDPreg(); ARM7opORR(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opORRSimm() { ARM7addrDPimm(); ARM7opORR(); ARM7addrDPflagBIT(); return 1; }
OPCODE ARM7opORRSlli() { ARM7addrDPlli(); ARM7opORR(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opORRSllr() { ARM7addrDPllr(); ARM7opORR(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opORRSlri() { ARM7addrDPlri(); ARM7opORR(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opORRSlrr() { ARM7addrDPlrr(); ARM7opORR(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opORRSari() { ARM7addrDPari(); ARM7opORR(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opORRSarr() { ARM7addrDParr(); ARM7opORR(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opORRSrri() { ARM7addrDPrri(); ARM7opORR(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opORRSrrr() { ARM7addrDPrrr(); ARM7opORR(); ARM7addrDPflagBIT(); return 2; }

OPCODE ARM7opMOVSreg() { ARM7addrDPreg(); ARM7opMOV(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opMOVSimm() { ARM7addrDPimm(); ARM7opMOV(); ARM7addrDPflagBIT(); return 1; }
OPCODE ARM7opMOVSlli() { ARM7addrDPlli(); ARM7opMOV(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opMOVSllr() { ARM7addrDPllr(); ARM7opMOV(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opMOVSlri() { ARM7addrDPlri(); ARM7opMOV(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opMOVSlrr() { ARM7addrDPlrr(); ARM7opMOV(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opMOVSari() { ARM7addrDPari(); ARM7opMOV(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opMOVSarr() { ARM7addrDParr(); ARM7opMOV(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opMOVSrri() { ARM7addrDPrri(); ARM7opMOV(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opMOVSrrr() { ARM7addrDPrrr(); ARM7opMOV(); ARM7addrDPflagBIT(); return 2; }

OPCODE ARM7opBICSreg() { ARM7addrDPreg(); ARM7opBIC(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opBICSimm() { ARM7addrDPimm(); ARM7opBIC(); ARM7addrDPflagBIT(); return 1; }
OPCODE ARM7opBICSlli() { ARM7addrDPlli(); ARM7opBIC(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opBICSllr() { ARM7addrDPllr(); ARM7opBIC(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opBICSlri() { ARM7addrDPlri(); ARM7opBIC(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opBICSlrr() { ARM7addrDPlrr(); ARM7opBIC(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opBICSari() { ARM7addrDPari(); ARM7opBIC(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opBICSarr() { ARM7addrDParr(); ARM7opBIC(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opBICSrri() { ARM7addrDPrri(); ARM7opBIC(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opBICSrrr() { ARM7addrDPrrr(); ARM7opBIC(); ARM7addrDPflagBIT(); return 2; }

OPCODE ARM7opMVNSreg() { ARM7addrDPreg(); ARM7opMVN(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opMVNSimm() { ARM7addrDPimm(); ARM7opMVN(); ARM7addrDPflagBIT(); return 1; }
OPCODE ARM7opMVNSlli() { ARM7addrDPlli(); ARM7opMVN(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opMVNSllr() { ARM7addrDPllr(); ARM7opMVN(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opMVNSlri() { ARM7addrDPlri(); ARM7opMVN(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opMVNSlrr() { ARM7addrDPlrr(); ARM7opMVN(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opMVNSari() { ARM7addrDPari(); ARM7opMVN(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opMVNSarr() { ARM7addrDParr(); ARM7opMVN(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opMVNSrri() { ARM7addrDPrri(); ARM7opMVN(); ARM7addrDPflagBIT(); return 2; }
OPCODE ARM7opMVNSrrr() { ARM7addrDPrrr(); ARM7opMVN(); ARM7addrDPflagBIT(); return 2; }

//---Load/Store addressing modes, opcodes----------------------------------

#define ARM7addrLSofim() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm and ebx,00000FFFh               /* Mask offset */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* Subtract offset */ \
    }

#define ARM7addrLSofip() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm and ebx,00000FFFh               /* Mask offset */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* Subtract offset */ \
    }

#define ARM7addrLSlmofim() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm mov eax,edx                     /* yet another */ \
        __asm shr eax,4                       /* Shift top nybble */ \
        __asm and ebx,15                      /* Mask low nybble */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm and eax,240                     /* Mask high nyb */ \
        __asm or ebx,eax                      /* Add nybbles */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* Subtract offset */ \
    }

#define ARM7addrLSlmofip() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm mov eax,edx                     /* yet another */ \
        __asm shr eax,4                       /* Shift top nybble */ \
        __asm and ebx,15                      /* Mask low nybble */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm and eax,240                     /* Mask high nyb */ \
        __asm or ebx,eax                      /* Add nybbles */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* Add offset */ \
    }

#define ARM7addrLSofrm() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM7addrLSofrmll() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
	__asm shl ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM7addrLSofrmlr() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm shl ebx,1                       /* Set carry */ \
        __asm xor ebx,ebx                     /* Operand=0 */ \
        __asm jmp shftend                     /* and skip over */\
        __asm shft1:                          /* Otherwise.. */ \
	__asm shr ebx,cl                      /* Shift Rm */ \
	__asm shftend:                        /* Good to go */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM7addrLSofrmar() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,31                      /* Set to 32 */ \
        __asm shft1:                          /* Otherwise.. */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
	__asm sar ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM7addrLSofrmrr() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,[arm7reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm rcr ebx,1                       /* 33-bit shift */ \
        __asm jmp shftend \
        __asm shft1:                          /* Otherwise.. */ \
	__asm ror ebx,cl                      /* Shift Rm */ \
	__asm shftend: \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM7addrLSofrp() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM7addrLSofrpll() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
	__asm shl ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM7addrLSofrplr() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm shl ebx,1                       /* Set carry */ \
        __asm xor ebx,ebx                     /* Operand=0 */ \
        __asm jmp shftend                     /* and skip over */\
        __asm shft1:                          /* Otherwise.. */ \
	__asm shr ebx,cl                      /* Shift Rm */ \
	__asm shftend:                        /* Good to go */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM7addrLSofrpar() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,31                      /* Set to 32 */ \
        __asm shft1:                          /* Otherwise.. */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
	__asm sar ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM7addrLSofrprr() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,[arm7reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm rcr ebx,1                       /* 33-bit shift */ \
        __asm jmp shftend                     /* And skip over this */ \
        __asm shft1:                          /* Otherwise.. */ \
	__asm ror ebx,cl                      /* Shift Rm */ \
	__asm shftend: \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
    }

#define ARM7addrLSprim() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm and ebx,00000FFFh               /* Mask offset */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* Subtract offset */ \
        __asm mov [arm7reg+ecx*4],eax         /* Write back */ \
    }

#define ARM7addrLSprip() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm and ebx,00000FFFh               /* Mask offset */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* Add offset */ \
        __asm mov [arm7reg+ecx*4],eax         /* Write back */ \
    }

#define ARM7addrLSlmprim() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm mov eax,edx                     /* yet another */ \
        __asm shr eax,4                       /* Shift top nybble */ \
        __asm and ebx,15                      /* Mask low nybble */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm and eax,240                     /* Mask high nyb */ \
        __asm or ebx,eax                      /* Add nybbles */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* Subtract offset */ \
        __asm mov [arm7reg+ecx*4],eax         /* Write back */ \
    }

#define ARM7addrLSlmprip() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm mov eax,edx                     /* yet another */ \
        __asm shr eax,4                       /* Shift top nybble */ \
        __asm and ebx,15                      /* Mask low nybble */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm and eax,240                     /* Mask high nyb */ \
        __asm or ebx,eax                      /* Add nybbles */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* Add offset */ \
        __asm mov [arm7reg+ecx*4],eax         /* Write back */ \
    }

#define ARM7addrLSprrm() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm7reg+ecx*4],eax         /* Write back */ \
    }

#define ARM7addrLSprrmll() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
	__asm shl ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm7reg+ecx*4],eax         /* Write back */ \
    }

#define ARM7addrLSprrmlr() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm shl ebx,1                       /* Set carry */ \
        __asm xor ebx,ebx                     /* Operand=0 */ \
        __asm jmp shftend                     /* and skip over */\
        __asm shft1:                          /* Otherwise.. */ \
	__asm shr ebx,cl                      /* Shift Rm */ \
	__asm shftend:                        /* Good to go */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm7reg+ecx*4],eax         /* Write back */ \
    }

#define ARM7addrLSprrmar() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,31                      /* Set to 32 */ \
        __asm shft1:                          /* Otherwise.. */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
	__asm sar ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm7reg+ecx*4],eax         /* Write back */ \
    }

#define ARM7addrLSprrmrr() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,[arm7reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm rcr ebx,1                       /* 33-bit shift */ \
        __asm jmp shftend                     /* And skip over this */ \
        __asm shft1:                          /* Otherwise.. */ \
	__asm ror ebx,cl                      /* Shift Rm */ \
	__asm shftend: \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm7reg+ecx*4],eax         /* Write back */ \
    }

#define ARM7addrLSprrp() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm7reg+ecx*4],eax         /* Write back */ \
    }

#define ARM7addrLSprrpll() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
	__asm shl ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm7reg+ecx*4],eax         /* Write back */ \
    }

#define ARM7addrLSprrplr() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm shl ebx,1                       /* Set carry */ \
        __asm xor ebx,ebx                     /* Operand=0 */ \
        __asm jmp shftend                     /* and skip over */\
        __asm shft1:                          /* Otherwise.. */ \
	__asm shr ebx,cl                      /* Shift Rm */ \
	__asm shftend:                        /* Good to go */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm7reg+ecx*4],eax         /* Write back */ \
    }

#define ARM7addrLSprrpar() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,31                      /* Set to 32 */ \
        __asm shft1:                          /* Otherwise.. */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
	__asm sar ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm7reg+ecx*4],eax         /* Write back */ \
    }

#define ARM7addrLSprrprr() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,[arm7reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm rcr ebx,1                       /* 33-bit shift */ \
        __asm jmp shftend                     /* And skip over this */ \
        __asm shft1:                          /* Otherwise.. */ \
	__asm ror ebx,cl                      /* Shift Rm */ \
	__asm shftend: \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add eax,ebx                     /* val=Rn-(Rm#shft) */ \
        __asm mov [arm7reg+ecx*4],eax         /* Write back */ \
    }

#define ARM7addrLSptim() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm and ebx,00000FFFh               /* Mask offset */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub [arm7reg+ecx*4],ebx         /* Don't touch eax */ \
    }

#define ARM7addrLSptip() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm and ebx,00000FFFh               /* Mask offset */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add [arm7reg+ecx*4],ebx         /* Don't touch eax */ \
    }

#define ARM7addrLSlmptim() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm mov eax,edx                     /* yet another */ \
        __asm shr eax,4                       /* Shift top nybble */ \
        __asm and ebx,15                      /* Mask low nybble */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm and eax,240                     /* Mask high nyb */ \
        __asm or ebx,eax                      /* Add nybbles */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub [arm7reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM7addrLSlmptip() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov ebx,edx                     /* and another */ \
        __asm mov eax,edx                     /* yet another */ \
        __asm shr eax,4                       /* Shift top nybble */ \
        __asm and ebx,15                      /* Mask low nybble */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm and eax,240                     /* Mask high nyb */ \
        __asm or ebx,eax                      /* Add nybbles */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add [arm7reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM7addrLSptrm() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub [arm7reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM7addrLSptrmll() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
	__asm shl ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub [arm7reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM7addrLSptrmlr() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm shl ebx,1                       /* Set carry */ \
        __asm xor ebx,ebx                     /* Operand=0 */ \
        __asm jmp shftend                     /* and skip over */\
        __asm shft1:                          /* Otherwise.. */ \
	__asm shr ebx,cl                      /* Shift Rm */ \
	__asm shftend:                        /* Good to go */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub [arm7reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM7addrLSptrmar() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,31                      /* Set to 32 */ \
        __asm shft1:                          /* Otherwise.. */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
	__asm sar ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub [arm7reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM7addrLSptrmrr() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,[arm7reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm rcr ebx,1                       /* 33-bit shift */ \
        __asm jmp shftend                     /* And skip over this */ \
        __asm shft1:                          /* Otherwise.. */ \
	__asm ror ebx,cl                      /* Shift Rm */ \
	__asm shftend: \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm sub [arm7reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM7addrLSptrp() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add [arm7reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM7addrLSptrpll() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
	__asm shl ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add [arm7reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM7addrLSptrplr() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm shl ebx,1                       /* Set carry */ \
        __asm xor ebx,ebx                     /* Operand=0 */ \
        __asm jmp shftend                     /* and skip over */\
        __asm shft1:                          /* Otherwise.. */ \
	__asm shr ebx,cl                      /* Shift Rm */ \
	__asm shftend:                        /* Good to go */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add [arm7reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM7addrLSptrpar() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,31                      /* Set to 32 */ \
        __asm shft1:                          /* Otherwise.. */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
	__asm sar ebx,cl                      /* Shift Rm */ \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add [arm7reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM7addrLSptrprr() \
    __asm { \
        __asm mov edx,[arm7reg+ARMREG_OP*4]   /* Read opcode */ \
        __asm mov ecx,edx                     /* Make a copy */ \
        __asm mov eax,edx                     /* and another */ \
        __asm and eax,15                      /* Mask Rm off */ \
        __asm mov ebx,[arm7reg+eax*4]         /* Get Rm's value */ \
        __asm shr ecx,7                       /* Shift in shftamnt */ \
        __asm and ecx,31                      /* And to 31 */ \
        __asm jnz shft1                       /* If amnt is 0 */ \
        __asm mov ecx,[arm7reg+ARMREG_FC*4]   /* Read saved carry */ \
        __asm shr ecx,1                       /* Shift into x86 c */ \
        __asm rcr ebx,1                       /* 33-bit shift */ \
        __asm jmp shftend                     /* And skip over this */ \
        __asm shft1:                          /* Otherwise.. */ \
	__asm ror ebx,cl                      /* Shift Rm */ \
	__asm shftend: \
        __asm mov ecx,edx                     /* Opcode copy */ \
        __asm shr ecx,16                      /* Shift in Rn */ \
        __asm and ecx,15                      /* Mask it off */ \
        __asm mov eax,[arm7reg+ecx*4]         /* Get Rn's value */ \
        __asm cmp ecx,15                      /* Is Rn 15? */ \
        __asm jl pt1                          /* If so.. */ \
        __asm add eax,4                       /* Hack value+4 */ \
        __asm pt1:                            /* Otherwise skip */ \
        __asm add [arm7reg+ecx*4],ebx         /* Write back */ \
    }

#define ARM7opLDR() \
    __asm { \
        __asm shr edx,12                /* Shift in Rd */ \
        __asm and edx,15                /* Mask to 15 */ \
        __asm cmp edx,15                /* Is Rd 15? */ \
        __asm jl ldr1                   /* If not, skip */ \
        __asm and eax,0FFFFFFFCh        /* Discount low 2 bits */ \
        __asm ldr1:                     /* Otherwise.. */ \
        __asm push edx                  /* Save Rd */ \
        __asm push eax                  /* Address */ \
        __asm xor ecx,ecx               /* Bus = 0 */ \
        __asm push ecx                  /* Push "bus number" */ \
        __asm call MMUrdW               /* Read a word */ \
        __asm pop ecx                   /* Clear stack */ \
        __asm pop ecx                   /* of params */ \
        __asm pop edx                   /* Get Rd back */\
	__asm mov [arm7reg+edx*4],eax   /* Write value to Rd */ \
	__asm mov eax,3 \
    }

#define ARM7opLDRH() \
    __asm { \
        __asm push edx                       /* Save opcode */ \
        __asm push eax                       /* Push address */ \
        __asm xor ecx,ecx                    /* Bus = 0 */ \
        __asm push ecx                       /* Push "bus number" */ \
        __asm call MMUrdH                    /* Read a hword */ \
        __asm pop ecx                        /* Pop param1 */ \
        __asm pop ecx                        /* Pop param2 */ \
        __asm pop edx                        /* Get opcode back */ \
        __asm shr edx,12                     /* Shift in Rd */ \
        __asm and edx,15                     /* Mask to 15 */ \
        __asm movzx ebx,ax                   /* Extend to 32b */ \
	__asm mov [arm7reg+edx*4],ebx        /* and write */ \
	__asm mov eax,3 \
    }

#define ARM7opLDRB() \
    __asm { \
        __asm push edx                       /* Save opcode */ \
        __asm push eax                       /* Push address */ \
        __asm xor ecx,ecx                    /* Bus = 0 */ \
        __asm push ecx                       /* Push "bus number" */ \
        __asm call MMUrdB                    /* Read a byte */ \
        __asm pop ecx                        /* Pop param1 */ \
        __asm pop ecx                        /* Pop param2 */ \
        __asm pop edx                        /* Get opcode back */ \
        __asm shr edx,12                     /* Shift in Rd */ \
        __asm and edx,15                     /* Mask to 15 */ \
        __asm movzx ebx,al                   /* Extend to 32b */ \
	__asm mov [arm7reg+edx*4],ebx        /* and write */ \
	__asm mov eax,3 \
    }

#define ARM7opLDRSH() \
    __asm { \
        __asm push edx                       /* Save opcode */ \
        __asm push eax                       /* Push address */ \
        __asm xor ecx,ecx                    /* Bus = 0 */ \
        __asm push ecx                       /* Push "bus number" */ \
        __asm call MMUrdH                    /* Read a byte */ \
        __asm pop ecx                        /* Pop param1 */ \
        __asm pop ecx                        /* Pop param2 */ \
        __asm pop edx                        /* Get opcode back */ \
        __asm shr edx,12                     /* Shift in Rd */ \
        __asm and edx,15                     /* Mask to 15 */ \
        __asm movsx ebx,ax                   /* Extend to 32b */ \
	__asm mov [arm7reg+edx*4],ebx        /* and write */ \
	__asm mov eax,3 \
    }

#define ARM7opLDRSB() \
    __asm { \
        __asm push edx                       /* Save opcode */ \
        __asm push eax                       /* Push address */ \
        __asm xor ecx,ecx                    /* Bus = 0 */ \
        __asm push ecx                       /* Push "bus number" */ \
        __asm call MMUrdB                    /* Read a byte */ \
        __asm pop ecx                        /* Pop param1 */ \
        __asm pop ecx                        /* Pop param2 */ \
        __asm pop edx                        /* Get opcode back */ \
        __asm shr edx,12                     /* Shift in Rd */ \
        __asm and edx,15                     /* Mask to 15 */ \
        __asm movsx ebx,al                   /* Extend to 32b */ \
	__asm mov [arm7reg+edx*4],ebx        /* and write */ \
	__asm mov eax,3 \
    }

#define ARM7opLDRT() \
    __asm { \
        __asm mov ecx,eax               /* Make copy of addr */ \
        __asm and ecx,3                 /* Get low 2 bits */ \
        __asm shl ecx,3                 /* Multiples of 8 */ \
        __asm ror eax,cl                /* Rotate addr */ \
        __asm shr edx,12                /* Shift in Rd */ \
        __asm and edx,15                /* Mask to 15 */ \
        __asm push edx                  /* Save Rd */ \
        __asm push eax                  /* Address */ \
        __asm xor ecx,ecx               /* Bus = 0 */ \
        __asm push ecx                  /* Push "bus number" */ \
        __asm call MMUrdW               /* Read a word */ \
        __asm pop ecx                   /* Clear stack */ \
        __asm pop ecx                   /* of params */ \
        __asm pop edx                   /* Get Rd back */\
        /*__asm mov ecx,[arm7reg+ARMREG_MODE*4] \
        __asm cmp ecx,ARMMD_SYS         // Set flags on mode  \
        __asm jle usrmode               // If USR or SYS, skip  \
	__asm mov [arm7reg+ARMREG_USER*4+edx*4],eax \
	__asm jmp modedone              // Read usr-reg, end  \
	__asm usrmode:                  // Skip here if USR  \
	__asm mov [arm7reg+edx*4],eax   // Read normal reg  \
	__asm modedone:                 // Done */ \
	__asm mov [arm7reg+edx*4],eax \
	__asm mov eax,3 \
    }

#define ARM7opLDRBT() \
    __asm { \
        __asm shr edx,12                /* Shift in Rd */ \
        __asm and edx,15                /* Mask to 15 */ \
        __asm push edx                  /* Save Rd */ \
        __asm push eax                  /* Address */ \
        __asm xor ecx,ecx               /* Bus = 0 */ \
        __asm push ecx                  /* Push "bus number" */ \
        __asm call MMUrdB               /* Read a word */ \
        __asm pop ecx                   /* Clear stack */ \
        __asm pop ecx                   /* of params */ \
        __asm pop edx                   /* Get Rd back */\
        __asm movzx ebx,al              /* Extend to 32b */ \
        __asm mov ecx,[arm7reg+ARMREG_MODE*4] \
        __asm cmp ecx,ARMMD_SYS         /* Set flags on mode */ \
        __asm jle usrmode               /* If USR or SYS, skip */ \
	__asm mov [arm7reg+ARMREG_USER*4+edx*4],ebx \
	__asm jmp modedone              /* Read usr-reg, end */ \
	__asm usrmode:                  /* Skip here if USR */ \
	__asm mov [arm7reg+edx*4],ebx   /* Read normal reg */ \
	__asm modedone:                 /* Done */ \
	__asm mov eax,3 \
    }

#define ARM7opSTR() \
    __asm { \
        __asm shr edx,12                  /* Shift in Rd */ \
        __asm and edx,15                  /* Mask to 15 */ \
        __asm mov ebx,[arm7reg+edx*4]     /* Get Rd value */ \
        __asm push ebx                    /* Push Rd */ \
        __asm push eax                    /* Push address */ \
        __asm xor ecx,ecx                 /* Bus = 0 */ \
        __asm push ecx                    /* Push "bus number" */ \
        __asm call MMUwrW                 /* Write thru MMU */ \
        __asm pop eax                     /* Clear stack */ \
        __asm pop eax                     /* of pushed */ \
        __asm pop eax                     /* parameters */ \
	__asm mov eax,2 \
    }

#define ARM7opSTRH() \
    __asm { \
        __asm shr edx,12                  /* Shift in Rd */ \
        __asm and edx,15                  /* Mask to 15 */ \
        __asm mov ebx,[arm7reg+edx*4]     /* Get Rd value */ \
        __asm and ebx,0000FFFFh           /* Mask low hword */ \
        __asm push ebx                    /* Push Rd */ \
        __asm push eax                    /* Push address */ \
        __asm xor ecx,ecx                 /* Bus = 0 */ \
        __asm push ecx                    /* Push "bus number" */ \
        __asm call MMUwrH                 /* Write thru MMU */ \
        __asm pop eax                     /* Clear stack */ \
        __asm pop eax                     /* of pushed */ \
        __asm pop eax                     /* parameters */ \
	__asm mov eax,2 \
    }

#define ARM7opSTRB() \
    __asm { \
        __asm shr edx,12                  /* Shift in Rd */ \
        __asm and edx,15                  /* Mask to 15 */ \
        __asm mov ebx,[arm7reg+edx*4]     /* Get Rd value */ \
        __asm and ebx,000000FFh           /* Mask low byte */ \
        __asm push ebx                    /* Push Rd */ \
        __asm push eax                    /* Push address */ \
        __asm xor ecx,ecx                 /* Bus = 0 */ \
        __asm push ecx                    /* Push "bus number" */ \
        __asm call MMUwrB                 /* Write thru MMU */ \
        __asm pop eax                     /* Clear stack */ \
        __asm pop eax                     /* of pushed */ \
        __asm pop eax                     /* parameters */ \
	__asm mov eax,2 \
    }

#define ARM7opSTRT() \
    __asm { \
        __asm shr edx,12                  /* Shift in Rd */ \
        __asm and edx,15                  /* Mask to 15 */ \
        __asm mov ecx,[arm7reg+ARMREG_MODE*4] \
        __asm cmp ecx,ARMMD_SYS         /* Set flags on mode */ \
        __asm jle usrmode               /* If USR or SYS, skip */ \
	__asm mov ebx,[arm7reg+ARMREG_USER*4+edx*4] \
	__asm jmp modedone                /* Read usr-reg, end */ \
	__asm usrmode:                    /* Skip here if USR */ \
	__asm mov ebx,[arm7reg+edx*4]     /* Read normal reg */ \
	__asm modedone:                   /* Done */ \
        __asm push ebx                    /* Push value */ \
        __asm push eax                    /* Push address */ \
        __asm xor ecx,ecx                 /* Bus = 0 */ \
        __asm push ecx                    /* Push "bus number" */ \
        __asm call MMUwrW                 /* Write thru MMU */ \
        __asm pop eax                     /* Clear stack */ \
        __asm pop eax                     /* of pushed */ \
        __asm pop eax                     /* parameters */ \
	__asm mov eax,2 \
    }

#define ARM7opSTRBT() \
    __asm { \
        __asm shr edx,12                  /* Shift in Rd */ \
        __asm and edx,15                  /* Mask to 15 */ \
        __asm mov ecx,[arm7reg+ARMREG_MODE*4] \
        __asm cmp ecx,ARMMD_SYS         /* Set flags on mode */ \
        __asm jle usrmode               /* If USR or SYS, skip */ \
	__asm mov ebx,[arm7reg+ARMREG_USER*4+edx*4] \
	__asm jmp modedone              /* Read usr-reg, end */ \
	__asm usrmode:                  /* Skip here if USR */ \
	__asm mov ebx,[arm7reg+edx*4]   /* Read normal reg */ \
	__asm modedone:                 /* Done */ \
        __asm and ebx,000000FFh           /* Mask low byte */ \
        __asm push ebx                    /* Push Rd */ \
        __asm push eax                    /* Push address */ \
        __asm xor ecx,ecx                 /* Bus = 0 */ \
        __asm push ecx                    /* Push "bus number" */ \
        __asm call MMUwrB                 /* Write thru MMU */ \
        __asm pop eax                     /* Clear stack */ \
        __asm pop eax                     /* of pushed */ \
        __asm pop eax                     /* parameters */ \
	__asm mov eax,2 \
    }

OPCODE ARM7opLDRptim() { ARM7addrLSptim(); ARM7opLDR(); }
OPCODE ARM7opLDRptip() { ARM7addrLSptip(); ARM7opLDR(); }
OPCODE ARM7opLDRptrm() { ARM7addrLSptrm(); ARM7opLDR(); }
OPCODE ARM7opLDRptrp() { ARM7addrLSptrp(); ARM7opLDR(); }
OPCODE ARM7opLDRprim() { ARM7addrLSprim(); ARM7opLDR(); }
OPCODE ARM7opLDRprip() { ARM7addrLSprip(); ARM7opLDR(); }
OPCODE ARM7opLDRprrm() { ARM7addrLSprrm(); ARM7opLDR(); }
OPCODE ARM7opLDRprrp() { ARM7addrLSprrp(); ARM7opLDR(); }
OPCODE ARM7opLDRofim() { ARM7addrLSofim(); ARM7opLDR(); }
OPCODE ARM7opLDRofip() { ARM7addrLSofip(); ARM7opLDR(); }
OPCODE ARM7opLDRofrm() { ARM7addrLSofrm(); ARM7opLDR(); }
OPCODE ARM7opLDRofrp() { ARM7addrLSofrp(); ARM7opLDR(); }

OPCODE ARM7opSTRptim() { ARM7addrLSptim(); ARM7opSTR(); }
OPCODE ARM7opSTRptip() { ARM7addrLSptip(); ARM7opSTR(); }
OPCODE ARM7opSTRptrm() { ARM7addrLSptrm(); ARM7opSTR(); }
OPCODE ARM7opSTRptrp() { ARM7addrLSptrp(); ARM7opSTR(); }
OPCODE ARM7opSTRprim() { ARM7addrLSprim(); ARM7opSTR(); }
OPCODE ARM7opSTRprip() { ARM7addrLSprip(); ARM7opSTR(); }
OPCODE ARM7opSTRprrm() { ARM7addrLSprrm(); ARM7opSTR(); }
OPCODE ARM7opSTRprrp() { ARM7addrLSprrp(); ARM7opSTR(); }
OPCODE ARM7opSTRofim() { ARM7addrLSofim(); ARM7opSTR(); }
OPCODE ARM7opSTRofip() { ARM7addrLSofip(); ARM7opSTR(); }
OPCODE ARM7opSTRofrm() { ARM7addrLSofrm(); ARM7opSTR(); }
OPCODE ARM7opSTRofrp() { ARM7addrLSofrp(); ARM7opSTR(); }

OPCODE ARM7opLDRBptim() { ARM7addrLSptim(); ARM7opLDRB(); }
OPCODE ARM7opLDRBptip() { ARM7addrLSptip(); ARM7opLDRB(); }
OPCODE ARM7opLDRBptrm() { ARM7addrLSptrm(); ARM7opLDRB(); }
OPCODE ARM7opLDRBptrp() { ARM7addrLSptrp(); ARM7opLDRB(); }
OPCODE ARM7opLDRBprim() { ARM7addrLSprim(); ARM7opLDRB(); }
OPCODE ARM7opLDRBprip() { ARM7addrLSprip(); ARM7opLDRB(); }
OPCODE ARM7opLDRBprrm() { ARM7addrLSprrm(); ARM7opLDRB(); }
OPCODE ARM7opLDRBprrp() { ARM7addrLSprrp(); ARM7opLDRB(); }
OPCODE ARM7opLDRBofim() { ARM7addrLSofim(); ARM7opLDRB(); }
OPCODE ARM7opLDRBofip() { ARM7addrLSofip(); ARM7opLDRB(); }
OPCODE ARM7opLDRBofrm() { ARM7addrLSofrm(); ARM7opLDRB(); }
OPCODE ARM7opLDRBofrp() { ARM7addrLSofrp(); ARM7opLDRB(); }

OPCODE ARM7opSTRBptim() { ARM7addrLSptim(); ARM7opSTRB(); }
OPCODE ARM7opSTRBptip() { ARM7addrLSptip(); ARM7opSTRB(); }
OPCODE ARM7opSTRBptrm() { ARM7addrLSptrm(); ARM7opSTRB(); }
OPCODE ARM7opSTRBptrp() { ARM7addrLSptrp(); ARM7opSTRB(); }
OPCODE ARM7opSTRBprim() { ARM7addrLSprim(); ARM7opSTRB(); }
OPCODE ARM7opSTRBprip() { ARM7addrLSprip(); ARM7opSTRB(); }
OPCODE ARM7opSTRBprrm() { ARM7addrLSprrm(); ARM7opSTRB(); }
OPCODE ARM7opSTRBprrp() { ARM7addrLSprrp(); ARM7opSTRB(); }
OPCODE ARM7opSTRBofim() { ARM7addrLSofim(); ARM7opSTRB(); }
OPCODE ARM7opSTRBofip() { ARM7addrLSofip(); ARM7opSTRB(); }
OPCODE ARM7opSTRBofrm() { ARM7addrLSofrm(); ARM7opSTRB(); }
OPCODE ARM7opSTRBofrp() { ARM7addrLSofrp(); ARM7opSTRB(); }

OPCODE ARM7opLDRHptim() { ARM7addrLSlmptim(); ARM7opLDRH(); }
OPCODE ARM7opLDRHptip() { ARM7addrLSlmptip(); ARM7opLDRH(); }
OPCODE ARM7opLDRHptrm() { ARM7addrLSptrm(); ARM7opLDRH(); }
OPCODE ARM7opLDRHptrp() { ARM7addrLSptrp(); ARM7opLDRH(); }
OPCODE ARM7opLDRHprim() { ARM7addrLSlmprim(); ARM7opLDRH(); }
OPCODE ARM7opLDRHprip() { ARM7addrLSlmprip(); ARM7opLDRH(); }
OPCODE ARM7opLDRHprrm() { ARM7addrLSprrm(); ARM7opLDRH(); }
OPCODE ARM7opLDRHprrp() { ARM7addrLSprrp(); ARM7opLDRH(); }
OPCODE ARM7opLDRHofim() { ARM7addrLSlmofim(); ARM7opLDRH(); }
OPCODE ARM7opLDRHofip() { ARM7addrLSlmofip(); ARM7opLDRH(); }
OPCODE ARM7opLDRHofrm() { ARM7addrLSofrm(); ARM7opLDRH(); }
OPCODE ARM7opLDRHofrp() { ARM7addrLSofrp(); ARM7opLDRH(); }

OPCODE ARM7opSTRHptim() { ARM7addrLSlmptim(); ARM7opSTRH(); }
OPCODE ARM7opSTRHptip() { ARM7addrLSlmptip(); ARM7opSTRH(); }
OPCODE ARM7opSTRHptrm() { ARM7addrLSptrm(); ARM7opSTRH(); }
OPCODE ARM7opSTRHptrp() { ARM7addrLSptrp(); ARM7opSTRH(); }
OPCODE ARM7opSTRHprim() { ARM7addrLSlmprim(); ARM7opSTRH(); }
OPCODE ARM7opSTRHprip() { ARM7addrLSlmprip(); ARM7opSTRH(); }
OPCODE ARM7opSTRHprrm() { ARM7addrLSprrm(); ARM7opSTRH(); }
OPCODE ARM7opSTRHprrp() { ARM7addrLSprrp(); ARM7opSTRH(); }
OPCODE ARM7opSTRHofim() { ARM7addrLSlmofim(); ARM7opSTRH(); }
OPCODE ARM7opSTRHofip() { ARM7addrLSlmofip(); ARM7opSTRH(); }
OPCODE ARM7opSTRHofrm() { ARM7addrLSofrm(); ARM7opSTRH(); }
OPCODE ARM7opSTRHofrp() { ARM7addrLSofrp(); ARM7opSTRH(); }

OPCODE ARM7opLDRTptim() { ARM7addrLSptim(); ARM7opLDRT(); }
OPCODE ARM7opLDRTptip() { ARM7addrLSptip(); ARM7opLDRT(); }
OPCODE ARM7opLDRTptrm() { ARM7addrLSptrm(); ARM7opLDRT(); }
OPCODE ARM7opLDRTptrp() { ARM7addrLSptrp(); ARM7opLDRT(); }
OPCODE ARM7opLDRTprim() { ARM7addrLSprim(); ARM7opLDRT(); }
OPCODE ARM7opLDRTprip() { ARM7addrLSprip(); ARM7opLDRT(); }
OPCODE ARM7opLDRTprrm() { ARM7addrLSprrm(); ARM7opLDRT(); }
OPCODE ARM7opLDRTprrp() { ARM7addrLSprrp(); ARM7opLDRT(); }
OPCODE ARM7opLDRTofim() { ARM7addrLSofim(); ARM7opLDRT(); }
OPCODE ARM7opLDRTofip() { ARM7addrLSofip(); ARM7opLDRT(); }
OPCODE ARM7opLDRTofrm() { ARM7addrLSofrm(); ARM7opLDRT(); }
OPCODE ARM7opLDRTofrp() { ARM7addrLSofrp(); ARM7opLDRT(); }

OPCODE ARM7opSTRTptim() { ARM7addrLSptim(); ARM7opSTRT(); }
OPCODE ARM7opSTRTptip() { ARM7addrLSptip(); ARM7opSTRT(); }
OPCODE ARM7opSTRTptrm() { ARM7addrLSptrm(); ARM7opSTRT(); }
OPCODE ARM7opSTRTptrp() { ARM7addrLSptrp(); ARM7opSTRT(); }
OPCODE ARM7opSTRTprim() { ARM7addrLSprim(); ARM7opSTRT(); }
OPCODE ARM7opSTRTprip() { ARM7addrLSprip(); ARM7opSTRT(); }
OPCODE ARM7opSTRTprrm() { ARM7addrLSprrm(); ARM7opSTRT(); }
OPCODE ARM7opSTRTprrp() { ARM7addrLSprrp(); ARM7opSTRT(); }
OPCODE ARM7opSTRTofim() { ARM7addrLSofim(); ARM7opSTRT(); }
OPCODE ARM7opSTRTofip() { ARM7addrLSofip(); ARM7opSTRT(); }
OPCODE ARM7opSTRTofrm() { ARM7addrLSofrm(); ARM7opSTRT(); }
OPCODE ARM7opSTRTofrp() { ARM7addrLSofrp(); ARM7opSTRT(); }

OPCODE ARM7opLDRBTptim() { ARM7addrLSptim(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTptip() { ARM7addrLSptip(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTptrm() { ARM7addrLSptrm(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTptrp() { ARM7addrLSptrp(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTprim() { ARM7addrLSprim(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTprip() { ARM7addrLSprip(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTprrm() { ARM7addrLSprrm(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTprrp() { ARM7addrLSprrp(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTofim() { ARM7addrLSofim(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTofip() { ARM7addrLSofip(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTofrm() { ARM7addrLSofrm(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTofrp() { ARM7addrLSofrp(); ARM7opLDRBT(); }

OPCODE ARM7opSTRBTptim() { ARM7addrLSptim(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTptip() { ARM7addrLSptip(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTptrm() { ARM7addrLSptrm(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTptrp() { ARM7addrLSptrp(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTprim() { ARM7addrLSprim(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTprip() { ARM7addrLSprip(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTprrm() { ARM7addrLSprrm(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTprrp() { ARM7addrLSprrp(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTofim() { ARM7addrLSofim(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTofip() { ARM7addrLSofip(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTofrm() { ARM7addrLSofrm(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTofrp() { ARM7addrLSofrp(); ARM7opSTRBT(); }

OPCODE ARM7opLDRSBptim() { /*u8 c; */ARM7addrLSlmptim(); ARM7opLDRSB(); }
OPCODE ARM7opLDRSBptip() { /*u8 c; */ARM7addrLSlmptip(); ARM7opLDRSB(); }
OPCODE ARM7opLDRSBptrm() { /*u8 c; */ARM7addrLSptrm(); ARM7opLDRSB(); }
OPCODE ARM7opLDRSBptrp() { /*u8 c; */ARM7addrLSptrp(); ARM7opLDRSB(); }
OPCODE ARM7opLDRSBprim() { /*u8 c; */ARM7addrLSlmprim(); ARM7opLDRSB(); }
OPCODE ARM7opLDRSBprip() { /*u8 c; */ARM7addrLSlmprip(); ARM7opLDRSB(); }
OPCODE ARM7opLDRSBprrm() { /*u8 c; */ARM7addrLSprrm(); ARM7opLDRSB(); }
OPCODE ARM7opLDRSBprrp() { /*u8 c; */ARM7addrLSprrp(); ARM7opLDRSB(); }
OPCODE ARM7opLDRSBofim() { /*u8 c; */ARM7addrLSlmofim(); ARM7opLDRSB(); }
OPCODE ARM7opLDRSBofip() { /*u8 c; */ARM7addrLSlmofip(); ARM7opLDRSB(); }
OPCODE ARM7opLDRSBofrm() { /*u8 c; */ARM7addrLSofrm(); ARM7opLDRSB(); }
OPCODE ARM7opLDRSBofrp() { /*u8 c; */ARM7addrLSofrp(); ARM7opLDRSB(); }

OPCODE ARM7opLDRSHptim() { /*u16 c;*/ ARM7addrLSlmptim(); ARM7opLDRSH(); }
OPCODE ARM7opLDRSHptip() { /*u16 c;*/ ARM7addrLSlmptip(); ARM7opLDRSH(); }
OPCODE ARM7opLDRSHptrm() { /*u16 c;*/ ARM7addrLSptrm(); ARM7opLDRSH(); }
OPCODE ARM7opLDRSHptrp() { /*u16 c;*/ ARM7addrLSptrp(); ARM7opLDRSH(); }
OPCODE ARM7opLDRSHprim() { /*u16 c;*/ ARM7addrLSlmprim(); ARM7opLDRSH(); }
OPCODE ARM7opLDRSHprip() { /*u16 c;*/ ARM7addrLSlmprip(); ARM7opLDRSH(); }
OPCODE ARM7opLDRSHprrm() { /*u16 c;*/ ARM7addrLSprrm(); ARM7opLDRSH(); }
OPCODE ARM7opLDRSHprrp() { /*u16 c;*/ ARM7addrLSprrp(); ARM7opLDRSH(); }
OPCODE ARM7opLDRSHofim() { /*u16 c;*/ ARM7addrLSlmofim(); ARM7opLDRSH(); }
OPCODE ARM7opLDRSHofip() { /*u16 c;*/ ARM7addrLSlmofip(); ARM7opLDRSH(); }
OPCODE ARM7opLDRSHofrm() { /*u16 c;*/ ARM7addrLSofrm(); ARM7opLDRSH(); }
OPCODE ARM7opLDRSHofrp() { /*u16 c;*/ ARM7addrLSofrp(); ARM7opLDRSH(); }

OPCODE ARM7opLDRptrmll() { ARM7addrLSptrmll(); ARM7opLDR(); }
OPCODE ARM7opLDRptrmlr() { ARM7addrLSptrmlr(); ARM7opLDR(); }
OPCODE ARM7opLDRptrmar() { ARM7addrLSptrmar(); ARM7opLDR(); }
OPCODE ARM7opLDRptrmrr() { ARM7addrLSptrmrr(); ARM7opLDR(); }
OPCODE ARM7opLDRptrpll() { ARM7addrLSptrpll(); ARM7opLDR(); }
OPCODE ARM7opLDRptrplr() { ARM7addrLSptrplr(); ARM7opLDR(); }
OPCODE ARM7opLDRptrpar() { ARM7addrLSptrpar(); ARM7opLDR(); }
OPCODE ARM7opLDRptrprr() { ARM7addrLSptrprr(); ARM7opLDR(); }
OPCODE ARM7opLDRprrmll() { ARM7addrLSprrmll(); ARM7opLDR(); }
OPCODE ARM7opLDRprrmlr() { ARM7addrLSprrmlr(); ARM7opLDR(); }
OPCODE ARM7opLDRprrmar() { ARM7addrLSprrmar(); ARM7opLDR(); }
OPCODE ARM7opLDRprrmrr() { ARM7addrLSprrmrr(); ARM7opLDR(); }
OPCODE ARM7opLDRprrpll() { ARM7addrLSprrpll(); ARM7opLDR(); }
OPCODE ARM7opLDRprrplr() { ARM7addrLSprrplr(); ARM7opLDR(); }
OPCODE ARM7opLDRprrpar() { ARM7addrLSprrpar(); ARM7opLDR(); }
OPCODE ARM7opLDRprrprr() { ARM7addrLSprrprr(); ARM7opLDR(); }
OPCODE ARM7opLDRofrmll() { ARM7addrLSofrmll(); ARM7opLDR(); }
OPCODE ARM7opLDRofrmlr() { ARM7addrLSofrmlr(); ARM7opLDR(); }
OPCODE ARM7opLDRofrmar() { ARM7addrLSofrmar(); ARM7opLDR(); }
OPCODE ARM7opLDRofrmrr() { ARM7addrLSofrmrr(); ARM7opLDR(); }
OPCODE ARM7opLDRofrpll() { ARM7addrLSofrpll(); ARM7opLDR(); }
OPCODE ARM7opLDRofrplr() { ARM7addrLSofrplr(); ARM7opLDR(); }
OPCODE ARM7opLDRofrpar() { ARM7addrLSofrpar(); ARM7opLDR(); }
OPCODE ARM7opLDRofrprr() { ARM7addrLSofrprr(); ARM7opLDR(); }

OPCODE ARM7opSTRptrmll() { ARM7addrLSptrmll(); ARM7opSTR(); }
OPCODE ARM7opSTRptrmlr() { ARM7addrLSptrmlr(); ARM7opSTR(); }
OPCODE ARM7opSTRptrmar() { ARM7addrLSptrmar(); ARM7opSTR(); }
OPCODE ARM7opSTRptrmrr() { ARM7addrLSptrmrr(); ARM7opSTR(); }
OPCODE ARM7opSTRptrpll() { ARM7addrLSptrpll(); ARM7opSTR(); }
OPCODE ARM7opSTRptrplr() { ARM7addrLSptrplr(); ARM7opSTR(); }
OPCODE ARM7opSTRptrpar() { ARM7addrLSptrpar(); ARM7opSTR(); }
OPCODE ARM7opSTRptrprr() { ARM7addrLSptrprr(); ARM7opSTR(); }
OPCODE ARM7opSTRprrmll() { ARM7addrLSprrmll(); ARM7opSTR(); }
OPCODE ARM7opSTRprrmlr() { ARM7addrLSprrmlr(); ARM7opSTR(); }
OPCODE ARM7opSTRprrmar() { ARM7addrLSprrmar(); ARM7opSTR(); }
OPCODE ARM7opSTRprrmrr() { ARM7addrLSprrmrr(); ARM7opSTR(); }
OPCODE ARM7opSTRprrpll() { ARM7addrLSprrpll(); ARM7opSTR(); }
OPCODE ARM7opSTRprrplr() { ARM7addrLSprrplr(); ARM7opSTR(); }
OPCODE ARM7opSTRprrpar() { ARM7addrLSprrpar(); ARM7opSTR(); }
OPCODE ARM7opSTRprrprr() { ARM7addrLSprrprr(); ARM7opSTR(); }
OPCODE ARM7opSTRofrmll() { ARM7addrLSofrmll(); ARM7opSTR(); }
OPCODE ARM7opSTRofrmlr() { ARM7addrLSofrmlr(); ARM7opSTR(); }
OPCODE ARM7opSTRofrmar() { ARM7addrLSofrmar(); ARM7opSTR(); }
OPCODE ARM7opSTRofrmrr() { ARM7addrLSofrmrr(); ARM7opSTR(); }
OPCODE ARM7opSTRofrpll() { ARM7addrLSofrpll(); ARM7opSTR(); }
OPCODE ARM7opSTRofrplr() { ARM7addrLSofrplr(); ARM7opSTR(); }
OPCODE ARM7opSTRofrpar() { ARM7addrLSofrpar(); ARM7opSTR(); }
OPCODE ARM7opSTRofrprr() { ARM7addrLSofrprr(); ARM7opSTR(); }

OPCODE ARM7opLDRBptrmll() { ARM7addrLSptrmll(); ARM7opLDRB(); }
OPCODE ARM7opLDRBptrmlr() { ARM7addrLSptrmlr(); ARM7opLDRB(); }
OPCODE ARM7opLDRBptrmar() { ARM7addrLSptrmar(); ARM7opLDRB(); }
OPCODE ARM7opLDRBptrmrr() { ARM7addrLSptrmrr(); ARM7opLDRB(); }
OPCODE ARM7opLDRBptrpll() { ARM7addrLSptrpll(); ARM7opLDRB(); }
OPCODE ARM7opLDRBptrplr() { ARM7addrLSptrplr(); ARM7opLDRB(); }
OPCODE ARM7opLDRBptrpar() { ARM7addrLSptrpar(); ARM7opLDRB(); }
OPCODE ARM7opLDRBptrprr() { ARM7addrLSptrprr(); ARM7opLDRB(); }
OPCODE ARM7opLDRBprrmll() { ARM7addrLSprrmll(); ARM7opLDRB(); }
OPCODE ARM7opLDRBprrmlr() { ARM7addrLSprrmlr(); ARM7opLDRB(); }
OPCODE ARM7opLDRBprrmar() { ARM7addrLSprrmar(); ARM7opLDRB(); }
OPCODE ARM7opLDRBprrmrr() { ARM7addrLSprrmrr(); ARM7opLDRB(); }
OPCODE ARM7opLDRBprrpll() { ARM7addrLSprrpll(); ARM7opLDRB(); }
OPCODE ARM7opLDRBprrplr() { ARM7addrLSprrplr(); ARM7opLDRB(); }
OPCODE ARM7opLDRBprrpar() { ARM7addrLSprrpar(); ARM7opLDRB(); }
OPCODE ARM7opLDRBprrprr() { ARM7addrLSprrprr(); ARM7opLDRB(); }
OPCODE ARM7opLDRBofrmll() { ARM7addrLSofrmll(); ARM7opLDRB(); }
OPCODE ARM7opLDRBofrmlr() { ARM7addrLSofrmlr(); ARM7opLDRB(); }
OPCODE ARM7opLDRBofrmar() { ARM7addrLSofrmar(); ARM7opLDRB(); }
OPCODE ARM7opLDRBofrmrr() { ARM7addrLSofrmrr(); ARM7opLDRB(); }
OPCODE ARM7opLDRBofrpll() { ARM7addrLSofrpll(); ARM7opLDRB(); }
OPCODE ARM7opLDRBofrplr() { ARM7addrLSofrplr(); ARM7opLDRB(); }
OPCODE ARM7opLDRBofrpar() { ARM7addrLSofrpar(); ARM7opLDRB(); }
OPCODE ARM7opLDRBofrprr() { ARM7addrLSofrprr(); ARM7opLDRB(); }

OPCODE ARM7opSTRBptrmll() { ARM7addrLSptrmll(); ARM7opSTRB(); }
OPCODE ARM7opSTRBptrmlr() { ARM7addrLSptrmlr(); ARM7opSTRB(); }
OPCODE ARM7opSTRBptrmar() { ARM7addrLSptrmar(); ARM7opSTRB(); }
OPCODE ARM7opSTRBptrmrr() { ARM7addrLSptrmrr(); ARM7opSTRB(); }
OPCODE ARM7opSTRBptrpll() { ARM7addrLSptrpll(); ARM7opSTRB(); }
OPCODE ARM7opSTRBptrplr() { ARM7addrLSptrplr(); ARM7opSTRB(); }
OPCODE ARM7opSTRBptrpar() { ARM7addrLSptrpar(); ARM7opSTRB(); }
OPCODE ARM7opSTRBptrprr() { ARM7addrLSptrprr(); ARM7opSTRB(); }
OPCODE ARM7opSTRBprrmll() { ARM7addrLSprrmll(); ARM7opSTRB(); }
OPCODE ARM7opSTRBprrmlr() { ARM7addrLSprrmlr(); ARM7opSTRB(); }
OPCODE ARM7opSTRBprrmar() { ARM7addrLSprrmar(); ARM7opSTRB(); }
OPCODE ARM7opSTRBprrmrr() { ARM7addrLSprrmrr(); ARM7opSTRB(); }
OPCODE ARM7opSTRBprrpll() { ARM7addrLSprrpll(); ARM7opSTRB(); }
OPCODE ARM7opSTRBprrplr() { ARM7addrLSprrplr(); ARM7opSTRB(); }
OPCODE ARM7opSTRBprrpar() { ARM7addrLSprrpar(); ARM7opSTRB(); }
OPCODE ARM7opSTRBprrprr() { ARM7addrLSprrprr(); ARM7opSTRB(); }
OPCODE ARM7opSTRBofrmll() { ARM7addrLSofrmll(); ARM7opSTRB(); }
OPCODE ARM7opSTRBofrmlr() { ARM7addrLSofrmlr(); ARM7opSTRB(); }
OPCODE ARM7opSTRBofrmar() { ARM7addrLSofrmar(); ARM7opSTRB(); }
OPCODE ARM7opSTRBofrmrr() { ARM7addrLSofrmrr(); ARM7opSTRB(); }
OPCODE ARM7opSTRBofrpll() { ARM7addrLSofrpll(); ARM7opSTRB(); }
OPCODE ARM7opSTRBofrplr() { ARM7addrLSofrplr(); ARM7opSTRB(); }
OPCODE ARM7opSTRBofrpar() { ARM7addrLSofrpar(); ARM7opSTRB(); }
OPCODE ARM7opSTRBofrprr() { ARM7addrLSofrprr(); ARM7opSTRB(); }

OPCODE ARM7opLDRTptrmll() { ARM7addrLSptrmll(); ARM7opLDRT(); }
OPCODE ARM7opLDRTptrmlr() { ARM7addrLSptrmlr(); ARM7opLDRT(); }
OPCODE ARM7opLDRTptrmar() { ARM7addrLSptrmar(); ARM7opLDRT(); }
OPCODE ARM7opLDRTptrmrr() { ARM7addrLSptrmrr(); ARM7opLDRT(); }
OPCODE ARM7opLDRTptrpll() { ARM7addrLSptrpll(); ARM7opLDRT(); }
OPCODE ARM7opLDRTptrplr() { ARM7addrLSptrplr(); ARM7opLDRT(); }
OPCODE ARM7opLDRTptrpar() { ARM7addrLSptrpar(); ARM7opLDRT(); }
OPCODE ARM7opLDRTptrprr() { ARM7addrLSptrprr(); ARM7opLDRT(); }
OPCODE ARM7opLDRTprrmll() { ARM7addrLSprrmll(); ARM7opLDRT(); }
OPCODE ARM7opLDRTprrmlr() { ARM7addrLSprrmlr(); ARM7opLDRT(); }
OPCODE ARM7opLDRTprrmar() { ARM7addrLSprrmar(); ARM7opLDRT(); }
OPCODE ARM7opLDRTprrmrr() { ARM7addrLSprrmrr(); ARM7opLDRT(); }
OPCODE ARM7opLDRTprrpll() { ARM7addrLSprrpll(); ARM7opLDRT(); }
OPCODE ARM7opLDRTprrplr() { ARM7addrLSprrplr(); ARM7opLDRT(); }
OPCODE ARM7opLDRTprrpar() { ARM7addrLSprrpar(); ARM7opLDRT(); }
OPCODE ARM7opLDRTprrprr() { ARM7addrLSprrprr(); ARM7opLDRT(); }
OPCODE ARM7opLDRTofrmll() { ARM7addrLSofrmll(); ARM7opLDRT(); }
OPCODE ARM7opLDRTofrmlr() { ARM7addrLSofrmlr(); ARM7opLDRT(); }
OPCODE ARM7opLDRTofrmar() { ARM7addrLSofrmar(); ARM7opLDRT(); }
OPCODE ARM7opLDRTofrmrr() { ARM7addrLSofrmrr(); ARM7opLDRT(); }
OPCODE ARM7opLDRTofrpll() { ARM7addrLSofrpll(); ARM7opLDRT(); }
OPCODE ARM7opLDRTofrplr() { ARM7addrLSofrplr(); ARM7opLDRT(); }
OPCODE ARM7opLDRTofrpar() { ARM7addrLSofrpar(); ARM7opLDRT(); }
OPCODE ARM7opLDRTofrprr() { ARM7addrLSofrprr(); ARM7opLDRT(); }

OPCODE ARM7opSTRTptrmll() { ARM7addrLSptrmll(); ARM7opSTRT(); }
OPCODE ARM7opSTRTptrmlr() { ARM7addrLSptrmlr(); ARM7opSTRT(); }
OPCODE ARM7opSTRTptrmar() { ARM7addrLSptrmar(); ARM7opSTRT(); }
OPCODE ARM7opSTRTptrmrr() { ARM7addrLSptrmrr(); ARM7opSTRT(); }
OPCODE ARM7opSTRTptrpll() { ARM7addrLSptrpll(); ARM7opSTRT(); }
OPCODE ARM7opSTRTptrplr() { ARM7addrLSptrplr(); ARM7opSTRT(); }
OPCODE ARM7opSTRTptrpar() { ARM7addrLSptrpar(); ARM7opSTRT(); }
OPCODE ARM7opSTRTptrprr() { ARM7addrLSptrprr(); ARM7opSTRT(); }
OPCODE ARM7opSTRTprrmll() { ARM7addrLSprrmll(); ARM7opSTRT(); }
OPCODE ARM7opSTRTprrmlr() { ARM7addrLSprrmlr(); ARM7opSTRT(); }
OPCODE ARM7opSTRTprrmar() { ARM7addrLSprrmar(); ARM7opSTRT(); }
OPCODE ARM7opSTRTprrmrr() { ARM7addrLSprrmrr(); ARM7opSTRT(); }
OPCODE ARM7opSTRTprrpll() { ARM7addrLSprrpll(); ARM7opSTRT(); }
OPCODE ARM7opSTRTprrplr() { ARM7addrLSprrplr(); ARM7opSTRT(); }
OPCODE ARM7opSTRTprrpar() { ARM7addrLSprrpar(); ARM7opSTRT(); }
OPCODE ARM7opSTRTprrprr() { ARM7addrLSprrprr(); ARM7opSTRT(); }
OPCODE ARM7opSTRTofrmll() { ARM7addrLSofrmll(); ARM7opSTRT(); }
OPCODE ARM7opSTRTofrmlr() { ARM7addrLSofrmlr(); ARM7opSTRT(); }
OPCODE ARM7opSTRTofrmar() { ARM7addrLSofrmar(); ARM7opSTRT(); }
OPCODE ARM7opSTRTofrmrr() { ARM7addrLSofrmrr(); ARM7opSTRT(); }
OPCODE ARM7opSTRTofrpll() { ARM7addrLSofrpll(); ARM7opSTRT(); }
OPCODE ARM7opSTRTofrplr() { ARM7addrLSofrplr(); ARM7opSTRT(); }
OPCODE ARM7opSTRTofrpar() { ARM7addrLSofrpar(); ARM7opSTRT(); }
OPCODE ARM7opSTRTofrprr() { ARM7addrLSofrprr(); ARM7opSTRT(); }

OPCODE ARM7opLDRBTptrmll() { ARM7addrLSptrmll(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTptrmlr() { ARM7addrLSptrmlr(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTptrmar() { ARM7addrLSptrmar(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTptrmrr() { ARM7addrLSptrmrr(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTptrpll() { ARM7addrLSptrpll(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTptrplr() { ARM7addrLSptrplr(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTptrpar() { ARM7addrLSptrpar(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTptrprr() { ARM7addrLSptrprr(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTprrmll() { ARM7addrLSprrmll(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTprrmlr() { ARM7addrLSprrmlr(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTprrmar() { ARM7addrLSprrmar(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTprrmrr() { ARM7addrLSprrmrr(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTprrpll() { ARM7addrLSprrpll(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTprrplr() { ARM7addrLSprrplr(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTprrpar() { ARM7addrLSprrpar(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTprrprr() { ARM7addrLSprrprr(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTofrmll() { ARM7addrLSofrmll(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTofrmlr() { ARM7addrLSofrmlr(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTofrmar() { ARM7addrLSofrmar(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTofrmrr() { ARM7addrLSofrmrr(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTofrpll() { ARM7addrLSofrpll(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTofrplr() { ARM7addrLSofrplr(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTofrpar() { ARM7addrLSofrpar(); ARM7opLDRBT(); }
OPCODE ARM7opLDRBTofrprr() { ARM7addrLSofrprr(); ARM7opLDRBT(); }

OPCODE ARM7opSTRBTptrmll() { ARM7addrLSptrmll(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTptrmlr() { ARM7addrLSptrmlr(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTptrmar() { ARM7addrLSptrmar(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTptrmrr() { ARM7addrLSptrmrr(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTptrpll() { ARM7addrLSptrpll(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTptrplr() { ARM7addrLSptrplr(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTptrpar() { ARM7addrLSptrpar(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTptrprr() { ARM7addrLSptrprr(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTprrmll() { ARM7addrLSprrmll(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTprrmlr() { ARM7addrLSprrmlr(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTprrmar() { ARM7addrLSprrmar(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTprrmrr() { ARM7addrLSprrmrr(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTprrpll() { ARM7addrLSprrpll(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTprrplr() { ARM7addrLSprrplr(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTprrpar() { ARM7addrLSprrpar(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTprrprr() { ARM7addrLSprrprr(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTofrmll() { ARM7addrLSofrmll(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTofrmlr() { ARM7addrLSofrmlr(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTofrmar() { ARM7addrLSofrmar(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTofrmrr() { ARM7addrLSofrmrr(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTofrpll() { ARM7addrLSofrpll(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTofrplr() { ARM7addrLSofrplr(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTofrpar() { ARM7addrLSofrpar(); ARM7opSTRBT(); }
OPCODE ARM7opSTRBTofrprr() { ARM7addrLSofrprr(); ARM7opSTRBT(); }

//-------------------------------------------------------------------------

OPCODE ARM7opMUL()
{
    __asm {
        mov ebx,[arm7reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm7reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm7reg+eax*4]          /* Read Rs' value */
	mul ecx                          /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	mov [arm7reg+ecx*4],eax          /* Write low word */
    }
    return 3;
}

OPCODE ARM7opMLA()
{
    __asm {
        mov ebx,[arm7reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm7reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm7reg+eax*4]          /* Read Rs' value */
	mul ecx                          /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	add eax,[arm7reg+ebx*4]          /* Add Rd to result */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	mov [arm7reg+ecx*4],eax          /* Add low word */
    }
    return 4;
}

OPCODE ARM7opMULS()
{
    __asm {
        mov ebx,[arm7reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm7reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm7reg+eax*4]          /* Read Rs' value */
	mul ecx                          /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	mov [arm7reg+ecx*4],eax          /* Write low word */
	or eax,eax                       /* Check word=0 */
	setz [arm7reg+ARMREG_FZ*4]       /* And set ARM Z flag */
	and eax,80000000h                /* Check MSB of word */
	sets [arm7reg+ARMREG_FN*4]       /* set ARM N flag */
    }
    return 3;
}

OPCODE ARM7opMLAS()
{
    __asm {
        mov ebx,[arm7reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm7reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm7reg+eax*4]          /* Read Rs' value */
	mul ecx                          /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	add eax,[arm7reg+ebx*4]          /* Add Rd to result */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	mov [arm7reg+ecx*4],eax          /* Add low word */
	or eax,eax                       /* Check word=0 */
	setz [arm7reg+ARMREG_FZ*4]       /* And set ARM Z flag */
	and eax,80000000h                /* Check MSB of word */
	sets [arm7reg+ARMREG_FN*4]       /* set ARM N flag */
    }
    return 4;
}

OPCODE ARM7opUMULL()
{
    __asm {
        mov ebx,[arm7reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm7reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm7reg+eax*4]          /* Read Rs' value */
	mul ecx                          /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	mov [arm7reg+ebx*4],eax          /* Write high word */
	mov [arm7reg+ecx*4],edx          /* Write low word */
    }
    return 4;
}

OPCODE ARM7opUMLAL()
{
    __asm {
        mov ebx,[arm7reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm7reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm7reg+eax*4]          /* Read Rs' value */
	mul ecx                          /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	add [arm7reg+ebx*4],eax          /* Add high word */
	adc [arm7reg+ecx*4],edx          /* Add low word */
    }
    return 5;
}

OPCODE ARM7opSMULL()
{
    __asm {
        mov ebx,[arm7reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm7reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm7reg+eax*4]          /* Read Rs' value */
	imul ecx                         /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	mov [arm7reg+ebx*4],eax          /* Write high word */
	mov [arm7reg+ecx*4],edx          /* Write low word */
    }
    return 4;
}

OPCODE ARM7opSMLAL()
{
    __asm {
        mov ebx,[arm7reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm7reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm7reg+eax*4]          /* Read Rs' value */
	imul ecx                         /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	add [arm7reg+ebx*4],eax          /* Add high word */
	adc [arm7reg+ecx*4],edx          /* Add low word */
    }
    return 5;
}

OPCODE ARM7opUMULLS()
{
    __asm {
        mov ebx,[arm7reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm7reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm7reg+eax*4]          /* Read Rs' value */
	mul ecx                          /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	mov [arm7reg+ebx*4],eax          /* Write high word */
	mov [arm7reg+ecx*4],edx          /* Write low word */
	or eax,edx                       /* Check both words are 0 */
	setz [arm7reg+ARMREG_FZ*4]       /* And set ARM Z flag */
	and edx,80000000h                /* Check MSB of hiword */
	sets [arm7reg+ARMREG_FN*4]       /* set ARM N flag */
    }
    return 4;
}

OPCODE ARM7opUMLALS()
{
    __asm {
        mov ebx,[arm7reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm7reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm7reg+eax*4]          /* Read Rs' value */
	mul ecx                          /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	add eax,[arm7reg+ebx*4]          /* Add high word */
	adc edx,[arm7reg+ecx*4]          /* Add low word */
	mov [arm7reg+ebx*4],eax
	mov [arm7reg+ecx*4],edx
	or eax,edx                       /* Check both words are 0 */
	setz [arm7reg+ARMREG_FZ*4]       /* And set ARM Z flag */
	and edx,80000000h                /* Check MSB of hiword */
	sets [arm7reg+ARMREG_FN*4]       /* set ARM N flag */
    }
    return 5;
}

OPCODE ARM7opSMULLS()
{
    __asm {
        mov ebx,[arm7reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm7reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm7reg+eax*4]          /* Read Rs' value */
	imul ecx                         /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	mov [arm7reg+ebx*4],eax          /* Write high word */
	mov [arm7reg+ecx*4],edx          /* Write low word */
	or eax,edx                       /* Check both words are 0 */
	setz [arm7reg+ARMREG_FZ*4]       /* And set ARM Z flag */
	and edx,80000000h                /* Check MSB of hiword */
	sets [arm7reg+ARMREG_FN*4]       /* set ARM N flag */
    }
    return 4;
}

OPCODE ARM7opSMLALS()
{
    __asm {
        mov ebx,[arm7reg+ARMREG_OP*4]    /* Read opcode */
        mov eax,ebx                      /* Make a copy */
        shr eax,8                        /* Shift in Rs */
        mov ecx,ebx                      /* Make another copy */
        and ecx,15                       /* Mask Rm */
        mov ecx,[arm7reg+ecx*4]          /* Read Rm's value */
	and eax,15                       /* Mask Rs to 15 */
	mov eax,[arm7reg+eax*4]          /* Read Rs' value */
	imul ecx                         /* edx:eax=Rs*Rm */
	mov ecx,ebx                      /* Make opcode copy */
	shr ebx,12                       /* Shift in Rd */
	and ebx,15                       /* And mask to 15 */
	shr ecx,16                       /* Shift Rn */
	and ecx,15                       /* And mask */
	add eax,[arm7reg+ebx*4]          /* Add high word */
	adc edx,[arm7reg+ecx*4]          /* Add low word */
	mov [arm7reg+ebx*4],eax
	mov [arm7reg+ecx*4],edx
	or eax,edx                       /* Check both words are 0 */
	setz [arm7reg+ARMREG_FZ*4]       /* And set ARM Z flag */
	and edx,80000000h                /* Check MSB of hiword */
	sets [arm7reg+ARMREG_FN*4]       /* set ARM N flag */
    }
    return 5;
}

//-------------------------------------------------------------------------

int ARM7SideSum(u16 val)
{
    int res=val;
    res=((res>>1)&0x5555)+(res&0x5555);
    res=((res>>2)&0x3333)+(res&0x3333);
    res=((res>>4)&0x0F0F)+(res&0x0F0F);
    res=((res>>8)&0x00FF)+(res&0x00FF);
    return res;
}

#define ARM7addrLMIA() /*int a;*/ arm7reg.tmp1=arm7reg.r[ARM7OP_RN]
#define ARM7addrLMIB() /*int a;*/ arm7reg.tmp1=arm7reg.r[ARM7OP_RN]+4
#define ARM7addrLMDA() /*int a;*/ arm7reg.tmp1=arm7reg.r[ARM7OP_RN]-(ARM7SideSum(arm7reg.curop&65535)*4)+4
#define ARM7addrLMDB() /*int a;*/ arm7reg.tmp1=arm7reg.r[ARM7OP_RN]-(ARM7SideSum(arm7reg.curop&65535)*4)

#define ARM7addrLMwi() arm7reg.r[ARM7OP_RN]+=ARM7SideSum(arm7reg.curop&65535)*4
#define ARM7addrLMwd() arm7reg.r[ARM7OP_RN]-=ARM7SideSum(arm7reg.curop&65535)*4

#define ARM7opLDM() \
    for(a=0;a<=14;a++) \
    { \
        if(arm7reg.curop&(1<<a)) \
        { \
            arm7reg.r[a]=MMUrdW(0,arm7reg.tmp1); \
            arm7reg.tmp1+=4; \
        } \
    } \
    if(arm7reg.curop&0x00008000) \
    { \
        arm7reg.r[15]=MMUrdW(0,arm7reg.tmp1)&0xFFFFFFFC; \
        arm7reg.tmp1+=4; \
    } \
    return 3

#define ARM7opSTM() \
    for(a=0;a<=15;a++) \
    { \
        if(arm7reg.curop&(1<<a)) \
        { \
            MMUwrW(0,arm7reg.tmp1,arm7reg.r[a]); \
            arm7reg.tmp1+=4; \
        } \
    } \
    return 2

#define ARM7opLDMu() \
    if(arm7reg.curop&0x0001) { arm7reg.r[0]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0002) { arm7reg.r[1]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0004) { arm7reg.r[2]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0008) { arm7reg.r[3]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0010) { arm7reg.r[4]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0020) { arm7reg.r[5]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0040) { arm7reg.r[6]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0080) { arm7reg.r[7]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    switch(arm7reg.curmode) \
    { \
    	case ARMMD_USR: case ARMMD_SYS: \
    	    if(arm7reg.curop&0x0100) { arm7reg.r[8]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0200) { arm7reg.r[9]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0400) { arm7reg.r[10]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x0800) { arm7reg.r[11]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x1000) { arm7reg.r[12]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x2000) { arm7reg.r[13]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x4000) { arm7reg.r[14]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    break;\
    	case ARMMD_IRQ: case ARMMD_ABT: case ARMMD_SVC: case ARMMD_UND: \
    	    if(arm7reg.curop&0x0100) { arm7reg.r[8]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0200) { arm7reg.r[9]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0400) { arm7reg.r[10]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x0800) { arm7reg.r[11]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x1000) { arm7reg.r[12]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x2000) { arm7reg.r13=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; }   \
    	    if(arm7reg.curop&0x4000) { arm7reg.r14=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; }   \
            break; \
    	case ARMMD_FIQ: \
    	    if(arm7reg.curop&0x0100) { arm7reg.r8=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0200) { arm7reg.r9=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0400) { arm7reg.r10=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x0800) { arm7reg.r11=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x1000) { arm7reg.r12=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x2000) { arm7reg.r13=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x4000) { arm7reg.r14=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
            break; \
    } \
    if(arm7reg.curop&0x8000) \
    { \
    	arm7reg.r[15]=MMUrdW(0,arm7reg.tmp1)&0xFFFFFFFC; arm7reg.tmp1+=4; \
    	if(arm7reg.curmode>1) arm7reg.cpsr=arm7reg.spsr[arm7reg.curmode]; \
    } \
    return 3

#define ARM7opSTMu() \
    if(arm7reg.curop&0x0001) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[0]); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0002) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[1]); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0004) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[2]); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0008) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[3]); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0010) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[4]); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0020) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[5]); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0040) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[6]); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0080) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[7]); arm7reg.tmp1+=4; } \
    switch(arm7reg.curmode) \
    { \
    	case ARMMD_USR: case ARMMD_SYS: \
    	    if(arm7reg.curop&0x0100) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[8]); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0200) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[9]); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0400) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[10]); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x0800) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[11]); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x1000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[12]); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x2000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[13]); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x4000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[14]); arm7reg.tmp1+=4; } \
    	    break; \
    	case ARMMD_IRQ: case ARMMD_ABT: case ARMMD_SVC: case ARMMD_UND: \
    	    if(arm7reg.curop&0x0100) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[8]); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0200) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[9]); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0400) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[10]); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x0800) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[11]); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x1000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[12]); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x2000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r13); arm7reg.tmp1+=4; }   \
    	    if(arm7reg.curop&0x4000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r14); arm7reg.tmp1+=4; }   \
            break; \
    	case ARMMD_FIQ: \
    	    if(arm7reg.curop&0x0100) { MMUwrW(0,arm7reg.tmp1,arm7reg.r8); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0200) { MMUwrW(0,arm7reg.tmp1,arm7reg.r9); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0400) { MMUwrW(0,arm7reg.tmp1,arm7reg.r10); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x0800) { MMUwrW(0,arm7reg.tmp1,arm7reg.r11); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x1000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r12); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x2000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r13); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x4000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r14); arm7reg.tmp1+=4; } \
            break; \
    } \
    if(arm7reg.curop&0x8000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[15]); arm7reg.tmp1+=4; } \
    return 2

OPCODE ARM7opLDMIA() { int a; ARM7addrLMIA(); ARM7opLDM(); }
OPCODE ARM7opLDMIB() { int a; ARM7addrLMIB(); ARM7opLDM(); }
OPCODE ARM7opLDMDA() { int a; ARM7addrLMDA(); ARM7opLDM(); }
OPCODE ARM7opLDMDB() { int a; ARM7addrLMDB(); ARM7opLDM(); }

OPCODE ARM7opLDMIAw() { int a; ARM7addrLMIA(); ARM7addrLMwi(); ARM7opLDM(); }
OPCODE ARM7opLDMIBw() { int a; ARM7addrLMIB(); ARM7addrLMwi(); ARM7opLDM(); }
OPCODE ARM7opLDMDAw() { int a; ARM7addrLMDA(); ARM7addrLMwd(); ARM7opLDM(); }
OPCODE ARM7opLDMDBw() { int a; ARM7addrLMDB(); ARM7addrLMwd(); ARM7opLDM(); }

OPCODE ARM7opLDMIAu() { ARM7addrLMIA(); ARM7opLDMu(); }
OPCODE ARM7opLDMIBu() { ARM7addrLMIB(); ARM7opLDMu(); }
OPCODE ARM7opLDMDAu() { ARM7addrLMDA(); ARM7opLDMu(); }
OPCODE ARM7opLDMDBu() { ARM7addrLMDB(); ARM7opLDMu(); }

OPCODE ARM7opLDMIAuw() { ARM7addrLMIA(); ARM7addrLMwi(); ARM7opLDMu(); }
OPCODE ARM7opLDMIBuw() { ARM7addrLMIB(); ARM7addrLMwi(); ARM7opLDMu(); }
OPCODE ARM7opLDMDAuw() { ARM7addrLMDA(); ARM7addrLMwd(); ARM7opLDMu(); }
OPCODE ARM7opLDMDBuw() { ARM7addrLMDB(); ARM7addrLMwd(); ARM7opLDMu(); }

OPCODE ARM7opSTMIA() { int a; ARM7addrLMIA(); ARM7opSTM(); }
OPCODE ARM7opSTMIB() { int a; ARM7addrLMIB(); ARM7opSTM(); }
OPCODE ARM7opSTMDA() { int a; ARM7addrLMDA(); ARM7opSTM(); }
OPCODE ARM7opSTMDB() { int a; ARM7addrLMDB(); ARM7opSTM(); }

OPCODE ARM7opSTMIAw() { int a; ARM7addrLMIA(); ARM7addrLMwi(); ARM7opSTM(); }
OPCODE ARM7opSTMIBw() { int a; ARM7addrLMIB(); ARM7addrLMwi(); ARM7opSTM(); }
OPCODE ARM7opSTMDAw() { int a; ARM7addrLMDA(); ARM7addrLMwd(); ARM7opSTM(); }
OPCODE ARM7opSTMDBw() { int a; ARM7addrLMDB(); ARM7addrLMwd(); ARM7opSTM(); }

OPCODE ARM7opSTMIAu() { ARM7addrLMIA(); ARM7opSTMu(); }
OPCODE ARM7opSTMIBu() { ARM7addrLMIB(); ARM7opSTMu(); }
OPCODE ARM7opSTMDAu() { ARM7addrLMDA(); ARM7opSTMu(); }
OPCODE ARM7opSTMDBu() { ARM7addrLMDB(); ARM7opSTMu(); }

OPCODE ARM7opSTMIAuw() { ARM7addrLMIA(); ARM7addrLMwi(); ARM7opSTMu(); }
OPCODE ARM7opSTMIBuw() { ARM7addrLMIB(); ARM7addrLMwi(); ARM7opSTMu(); }
OPCODE ARM7opSTMDAuw() { ARM7addrLMDA(); ARM7addrLMwd(); ARM7opSTMu(); }
OPCODE ARM7opSTMDBuw() { ARM7addrLMDB(); ARM7addrLMwd(); ARM7opSTMu(); }

//-------------------------------------------------------------------------

OPCODE ARM7opMRSrs()
{
    ARM7updateCPSR();
    if(arm7reg.curmode>1) arm7reg.tmp1=arm7reg.spsr[arm7reg.curmode];
    else arm7reg.tmp1=arm7reg.cpsr;
    arm7reg.r[ARM7OP_RD]=arm7reg.tmp1;
    return 1;
}

OPCODE ARM7opMRSrc()
{
    ARM7updateCPSR();
    __asm {
	mov ebx,[arm7reg+ARMREG_CPSR*4]
	mov eax,[arm7reg+ARMREG_OP*4]
	shr eax,12
	and eax,15
	mov [arm7reg+eax*4],ebx
    }
    return 1;
}

OPCODE ARM7opMSRic()
{
    arm7reg.tmp3 = arm7reg.curop&255;
    arm7reg.tmp4 = (arm7reg.curop&0x00000F00)>>7;
    arm7reg.tmp1 = (arm7reg.tmp3>>arm7reg.tmp4)|((arm7reg.tmp3&((1<<arm7reg.tmp4)-1))<<(32-arm7reg.tmp4));
    arm7reg.tmp2=arm7reg.cpsr;
    if(arm7reg.curop&0x00080000)
    {
        arm7reg.tmp2&=0x00FFFFFF;
        arm7reg.tmp2|=(arm7reg.tmp1&0xFF000000);
    }
    if(arm7reg.curmode>0)
    {
        if(arm7reg.curop&0x00040000)
        {
            arm7reg.tmp2&=0xFF00FFFF;
            arm7reg.tmp2|=(arm7reg.tmp1&0x00FF0000);
        }
        if(arm7reg.curop&0x00020000)
        {
            arm7reg.tmp2&=0xFFFF00FF;
            arm7reg.tmp2|=(arm7reg.tmp1&0x0000FF00);
        }
        if(arm7reg.curop&0x00010000)
        {
            arm7reg.tmp2&=0xFFFFFF00;
            arm7reg.tmp2|=(arm7reg.tmp1&0x000000FF);
        }
    }
    arm7reg.cpsr=arm7reg.tmp2; ARM7splitCPSR();
    ARM7modesw(arm7reg.curmode,ARM7cpsrtomode[arm7reg.cpsr&0x0F]);
    return 1;
}

OPCODE ARM7opMSRis()
{
    arm7reg.tmp3 = arm7reg.curop&255;
    arm7reg.tmp4 = (arm7reg.curop&0x00000F00)>>7;
    arm7reg.tmp1 = (arm7reg.tmp3>>arm7reg.tmp4)|((arm7reg.tmp3&((1<<arm7reg.tmp4)-1))<<(32-arm7reg.tmp4));
    arm7reg.tmp2=0;
    if(arm7reg.curop&0x00080000)
    {
        arm7reg.tmp2&=0x00FFFFFF;
        arm7reg.tmp2|=(arm7reg.tmp1&0xFF000000);
    }
    if(arm7reg.curmode>0)
    {
        if(arm7reg.curop&0x00040000)
        {
            arm7reg.tmp2&=0xFF00FFFF;
            arm7reg.tmp2|=(arm7reg.tmp1&0x00FF0000);
        }
        if(arm7reg.curop&0x00020000)
        {
            arm7reg.tmp2&=0xFFFF00FF;
            arm7reg.tmp2|=(arm7reg.tmp1&0x0000FF00);
        }
        if(arm7reg.curop&0x00010000)
        {
            arm7reg.tmp2&=0xFFFFFF00;
            arm7reg.tmp2|=(arm7reg.tmp1&0x000000FF);
        }
    }
    if(arm7reg.curmode>1) arm7reg.spsr[arm7reg.curmode]=arm7reg.tmp2;
    return 1;
}

OPCODE ARM7opMSRrc()
{
    arm7reg.tmp1=arm7reg.r[ARM7OP_RM];
    arm7reg.tmp2=arm7reg.cpsr;
    if(arm7reg.curop&0x00080000)
    {
        arm7reg.tmp2&=0x00FFFFFF;
        arm7reg.tmp2|=(arm7reg.tmp1&0xFF000000);
    }
    if(arm7reg.curmode>0)
    {
        if(arm7reg.curop&0x00040000)
        {
            arm7reg.tmp2&=0xFF00FFFF;
            arm7reg.tmp2|=(arm7reg.tmp1&0x00FF0000);
        }
        if(arm7reg.curop&0x00020000)
        {
            arm7reg.tmp2&=0xFFFF00FF;
            arm7reg.tmp2|=(arm7reg.tmp1&0x0000FF00);
        }
        if(arm7reg.curop&0x00010000)
        {
            arm7reg.tmp2&=0xFFFFFF00;
            arm7reg.tmp2|=(arm7reg.tmp1&0x000000FF);
        }
    }
    arm7reg.cpsr=arm7reg.tmp2; ARM7splitCPSR();
    ARM7modesw(arm7reg.curmode,ARM7cpsrtomode[arm7reg.cpsr&0x0F]);
    return 1;
}

OPCODE ARM7opMSRrs()
{
    arm7reg.tmp1=arm7reg.r[ARM7OP_RM];
    arm7reg.tmp2=0;
    if(arm7reg.curop&0x00080000)
    {
        arm7reg.tmp2&=0x00FFFFFF;
        arm7reg.tmp2|=(arm7reg.tmp1&0xFF000000);
    }
    if(arm7reg.curmode>0)
    {
        if(arm7reg.curop&0x00040000)
        {
            arm7reg.tmp2&=0xFF00FFFF;
            arm7reg.tmp2|=(arm7reg.tmp1&0x00FF0000);
        }
        if(arm7reg.curop&0x00020000)
        {
            arm7reg.tmp2&=0xFFFF00FF;
            arm7reg.tmp2|=(arm7reg.tmp1&0x0000FF00);
        }
        if(arm7reg.curop&0x00010000)
        {
            arm7reg.tmp2&=0xFFFFFF00;
            arm7reg.tmp2|=(arm7reg.tmp1&0x000000FF);
        }
    }
    if(arm7reg.curmode>1) arm7reg.spsr[arm7reg.curmode]=arm7reg.tmp2;
    return 1;
}

//-------------------------------------------------------------------------

#define ARM7addrSWP() \
    arm7reg.tmp2=arm7reg.r[ARM7OP_RN]; \
    switch(arm7reg.tmp2&3) \
    { \
        case 0: arm7reg.tmp1=arm7reg.tmp2; break; \
	case 1: arm7reg.tmp1=(arm7reg.tmp2>> 8)|((arm7reg.tmp2&0x000000FF)<<24); break; \
	case 2: arm7reg.tmp1=(arm7reg.tmp2>>16)|((arm7reg.tmp2&0x0000FFFF)<<16); break; \
	case 3: arm7reg.tmp1=(arm7reg.tmp2>>24)|((arm7reg.tmp2&0x00FFFFFF)<< 8); break; \
    }

OPCODE ARM7opSWP()
{
    ARM7addrSWP();
    arm7reg.tmp4=MMUrdW(0,arm7reg.tmp1);
    MMUwrW(0,arm7reg.tmp1,arm7reg.r[ARM7OP_RM]);
    arm7reg.r[ARM7OP_RD]=arm7reg.tmp4;
    return 4;
}

OPCODE ARM7opSWPB()
{
    ARM7addrSWP();
    arm7reg.tmp4=0|MMUrdB(0,arm7reg.tmp1);
    MMUwrB(0,arm7reg.tmp1,arm7reg.r[ARM7OP_RM]&255);
    arm7reg.r[ARM7OP_RD]=0|arm7reg.tmp4;
    return 4;
}

//-------------------------------------------------------------------------

void ARM7InterruptFlagChanged() {
	// If the interrupt occured we're waiting for it, restart the CPU.
	if(cpuHalted && MMUrdH(0,0x04000214)&intrBitmask) {
		cpuHalted = 0;
	}
}

OPCODE ARM7opSWI()
{
    u32 a,ret; u32 src/*,dest,size*/; char str[256];
/*    arm7reg.r14svc=arm7reg.r[15];
    arm7reg.spsr[ARMMD_SVC]=arm7reg.cpsr;
    ARM7modesw(arm7reg.curmode, ARMMD_SVC);
    arm7reg.cpsr&=(0xFFFFFFFF-ARMS_T);
    arm7reg.cpsr|=ARMS_I;
    arm7reg.r[15]=0x00000008;
*/
    switch((arm7reg.curop&0x00FF0000)>>16)
    {
        /*SoftReset*/ case 0:
            for(a=0;a<=12;a++) arm7reg.r[a]=0;
            if(MMUrdB(0,0x03007FFA)) ret=0x02000000; else ret=0x08000000;
            for(a=0x03007E00;a<=0x03007FFF;a+=4) MMUwrW(0,a,0);
            arm7reg.r13=0x03007F00;
            arm7reg.r13irq=0x03007FA0;
            arm7reg.r13svc=0x03007FE0;
            ARM7modesw(arm7reg.curmode,ARMMD_SYS);
            arm7reg.flags[ARMFLAG_T]=0;
            arm7reg.r[15]=ret; break;

	        /*Delay*/ case 3:
			{
			int delay = arm7reg.r[0];
			arm7reg.r[0] = 0;
			return delay+3;
			}
			break;

        /*IntrWait*/ case 4:
            cpuHalted=1;
            sprintf(str,"ARM7: CPU halted at %08X.",arm7reg.r[15]);
//            logvt->append(str);
            intrBitmask=arm7reg.r[1]&0xFFFF;
            switch(arm7reg.r[0]&1)
            {
                case 0: 
					if(REG(ARM7_REG,IF)&intrBitmask) {
						cpuHalted=0;
						return 3; 
					}
					break;
                case 1: 
				    REG(ARM7_REG,IF)&=~intrBitmask;
					break;
            }
            break;

        /*VblankIntrWait*/ case 5:
            arm7reg.r[0]=1; arm7reg.r[1]=1;
            arm7reg.curop=0x00040000; ARM7opSWI(); break;

        /*Halt*/ case 6:
            cpuHalted=1;
            sprintf(str,"ARM7: CPU halted at %08X.",arm7reg.r[15]);
//            logvt->append(str);
            intrBitmask=0xFFFF; break;

        /*Stop*/ case 7:
            cpuHalted=1;
            sprintf(str,"ARM7: CPU halted at %08X.",arm7reg.r[15]);
//            logvt->append(str);
            intrBitmask=0x0000; break;

        /*Div*/ case 9:
			{
				s32 lvalue = arm7reg.r[0];
				s32 rvalue = arm7reg.r[1];
				if(rvalue != 0) {
					s32 divresult = lvalue / rvalue;
					s32 divmodulous = lvalue % rvalue;
					arm7reg.r[0] = divresult;
					arm7reg.r[1] = divmodulous;
				}
			}
            break;
#if 0
        /*DivArm*/ case 10:
            __asm {
                mov ebx,[arm7reg+0*4]
                mov eax,[arm7reg+1*4]
                xor edx,edx
                idiv ebx
                mov [arm7reg+0*4],eax
                mov [arm7reg+1*4],edx
                test eax,0x80000000
                jz divarmend
                neg eax
                divarmend:
                mov [arm7reg+3*4],eax
            }
            break;
#endif
        /*Sqrt*/ case 0x0d:
            src=arm7reg.r[0]; arm7reg.r[0]= (u32) sqrt(src);
            break;
#if 0
        /*CpuSet*/ case 11:
            src=arm7reg.r[0]; dest=arm7reg.r[1]; size=arm7reg.r[2]&0xFFFF;
            for(a=0;a<size;a++)
            {
                if(arm7reg.r[2]&0x04000000)
                {
                    MMUwrW(0,dest,MMUrdW(0,src));
                    dest+=4;
                    if(!(arm7reg.r[2]&0x01000000)) src+=4;
                } else {
                    MMUwrH(0,dest,MMUrdH(0,src));
                    dest+=2;
                    if(!(arm7reg.r[2]&0x01000000)) src+=2;
                }
            }
            break;

        /*cpuFastSet*/ case 12:
            src=arm7reg.r[0]; dest=arm7reg.r[1]; size=arm7reg.r[2]&0xFFF8;
            for(a=0;a<size;a++)
            {
                MMUwrW(0,dest,MMUrdW(0,src));
                dest+=4;
                if(!(arm7reg.r[2]&0x01000000)) src+=4;
            }
#endif
    }
    return 3;
}

void ARM7irq()
{
    arm7reg.r14irq=arm7reg.r[15];
    arm7reg.spsr[ARMMD_IRQ]=arm7reg.cpsr;
    ARM7modesw(arm7reg.curmode, ARMMD_IRQ);
    arm7reg.cpsr&=(0xFFFFFFFF-ARMS_T);
    arm7reg.cpsr|=ARMS_I;
    cpuHalted=0; intrBitmask=0xFFFF;
    arm7reg.r[15]=0x00000018;
	ARM7execforPtr=ARM7execforARM;
	arm7reg.flags[ARMFLAG_T] = 0;
}

//=========================================================================
//---Thumb time!-----------------------------------------------------------

#include "thumb7x86.c"

/*** EOF:arm7.c **********************************************************/
