/**************************************************************************
* DSemu: ARM7 core definitions and opcode pointer table (arm7func.h)      *
* Released under the terms of the BSD Public Licence                      *
* Imran Nazar (tf@oopsilon.com), 2004                                     *
**************************************************************************/

#ifndef __ARM7FUNC_H_
#define __ARM7FUNC_H_

#include "defs.h"
#include "cache.h"

int ARM7exec(), ARM7execfor(int);
void ARM7dbgOut();
void ARM7modesw(int,int);
void ARM7status(int,int);
void ARM7updateCPSR(), ARM7splitCPSR();
int ARM7init(char*,u32*);
void ARM7fini();
int ARM7reset();
__int64 ARM7getClock();
void ARM7addClock(int);
void ARM7irq();

extern int MMUinit();
extern void MMUfini();
extern void MMUreg(int,void*);
extern void MMUunreg(int);
extern u8  MMUrdB(u32, u32);
extern u16 MMUrdH(u32, u32);
extern u32 MMUrdW(u32, u32);
extern RAMWORD MMUrdS(u32, u32);
extern void MMUwrB(u32, u32, u8 );
extern void MMUwrH(u32, u32, u16);
extern void MMUwrW(u32, u32, u32);
extern void MMUwrS(u32, u32, RAMWORD);

extern char *ARM7DASM(u32);

char ARM7modes[7][4]={"USR","SYS","FIQ","IRQ","SUP","ABT","UND"};
int ARM7modetocpsr[7]={ARMS_M_USR,ARMS_M_SYS,ARMS_M_FIQ,ARMS_M_IRQ,
                      ARMS_M_SVC,ARMS_M_ABT,ARMS_M_UND};
int ARM7cpsrtomode[16]={0,2,3,4,-1,-1,-1,5,-1,-1,-1,6,-1,-1,-1,1};

int ARM7condEQ(), ARM7condNE(), ARM7condCS(), ARM7condCC();
int ARM7condMI(), ARM7condPL(), ARM7condVS(), ARM7condVC();
int ARM7condHI(), ARM7condLS(), ARM7condGE(), ARM7condLT();
int ARM7condGT(), ARM7condLE(), ARM7condAL(), ARM7condNV();

static intfptr ARM7cond[]={
  ARM7condEQ, ARM7condNE, ARM7condCS, ARM7condCC,
  ARM7condMI, ARM7condPL, ARM7condVS, ARM7condVC,
  ARM7condHI, ARM7condLS, ARM7condGE, ARM7condLT,
  ARM7condGT, ARM7condLE, ARM7condAL, ARM7condNV,
};

//---UNdefined/predictable-------------------------------------------------

OPCODE ARM7opUNP();
OPCODE ARM7opUND();
OPCODE ARM7opUNI();
OPCODE ARM7opUNL();
RAMWORD ARM7opDecode(u32);

//---Uberdecoders----------------------------------------------------------

OPCODE ARM7opDPS0();      /* 0 - Data processing Shifted */
OPCODE ARM7opDPS1();      /* 1 - Data processing Shifted */
OPCODE ARM7opDPI0();      /* 2 - Data processing Immediate */
OPCODE ARM7opDPI1();      /* 3 - Data processing Immediate */
OPCODE ARM7opLSIpt();     /* 4 - Load/store Immediate Post-indexed */
OPCODE ARM7opLSIof();     /* 5 - Load/store Immediate Preidx/Offset */
OPCODE ARM7opLSRpt();     /* 6 - Load/store Register Post-indexed */
OPCODE ARM7opLSRof();     /* 7 - Load/store Register Preidx/Offset */
OPCODE ARM7opLSMf();      /* 8 - Load/store Multiple Load:Full */
OPCODE ARM7opLSMe();      /* 9 - Load/store Multiple Load:Empty */
OPCODE ARM7opB();         /* A - Branch */
OPCODE ARM7opBL();        /* B - Branch+Link */
OPCODE ARM7opLSCof();     /* C - Load/store Copro Preidx/Offset */
OPCODE ARM7opLSCpt();     /* D - Load/store Copro Post-indexed */
OPCODE ARM7opCP();        /* E - Copro Data/Reg handling */
OPCODE ARM7opSWI();       /* F - Software Interrupt */

//---Branch (with Link)----------------------------------------------------

OPCODE ARM7opB(), ARM7opBL();

OPCODE ARM7opBX();
OPCODE ARM7opBLX(), ARM7opBLXr();

//---DP (DataProcessing); LsL/LsR/AsR/RoR; Immediate/Register--------------

OPCODE ARM7opANDlli(), ARM7opANDllr();
OPCODE ARM7opANDlri(), ARM7opANDlrr();
OPCODE ARM7opANDari(), ARM7opANDarr();
OPCODE ARM7opANDrri(), ARM7opANDrrr();
OPCODE ARM7opANDimm();

OPCODE ARM7opEORlli(), ARM7opEORllr();
OPCODE ARM7opEORlri(), ARM7opEORlrr();
OPCODE ARM7opEORari(), ARM7opEORarr();
OPCODE ARM7opEORrri(), ARM7opEORrrr();
OPCODE ARM7opEORimm();

OPCODE ARM7opSUBlli(), ARM7opSUBllr();
OPCODE ARM7opSUBlri(), ARM7opSUBlrr();
OPCODE ARM7opSUBari(), ARM7opSUBarr();
OPCODE ARM7opSUBrri(), ARM7opSUBrrr();
OPCODE ARM7opSUBimm();

OPCODE ARM7opRSBlli(), ARM7opRSBllr();
OPCODE ARM7opRSBlri(), ARM7opRSBlrr();
OPCODE ARM7opRSBari(), ARM7opRSBarr();
OPCODE ARM7opRSBrri(), ARM7opRSBrrr();
OPCODE ARM7opRSBimm();

OPCODE ARM7opADDlli(), ARM7opADDllr();
OPCODE ARM7opADDlri(), ARM7opADDlrr();
OPCODE ARM7opADDari(), ARM7opADDarr();
OPCODE ARM7opADDrri(), ARM7opADDrrr();
OPCODE ARM7opADDimm();

OPCODE ARM7opADClli(), ARM7opADCllr();
OPCODE ARM7opADClri(), ARM7opADClrr();
OPCODE ARM7opADCari(), ARM7opADCarr();
OPCODE ARM7opADCrri(), ARM7opADCrrr();
OPCODE ARM7opADCimm();

OPCODE ARM7opSBClli(), ARM7opSBCllr();
OPCODE ARM7opSBClri(), ARM7opSBClrr();
OPCODE ARM7opSBCari(), ARM7opSBCarr();
OPCODE ARM7opSBCrri(), ARM7opSBCrrr();
OPCODE ARM7opSBCimm();

OPCODE ARM7opRSClli(), ARM7opRSCllr();
OPCODE ARM7opRSClri(), ARM7opRSClrr();
OPCODE ARM7opRSCari(), ARM7opRSCarr();
OPCODE ARM7opRSCrri(), ARM7opRSCrrr();
OPCODE ARM7opRSCimm();

OPCODE ARM7opORRlli(), ARM7opORRllr();
OPCODE ARM7opORRlri(), ARM7opORRlrr();
OPCODE ARM7opORRari(), ARM7opORRarr();
OPCODE ARM7opORRrri(), ARM7opORRrrr();
OPCODE ARM7opORRimm();

OPCODE ARM7opMOVlli(), ARM7opMOVllr();
OPCODE ARM7opMOVlri(), ARM7opMOVlrr();
OPCODE ARM7opMOVari(), ARM7opMOVarr();
OPCODE ARM7opMOVrri(), ARM7opMOVrrr();
OPCODE ARM7opMOVimm();

OPCODE ARM7opBIClli(), ARM7opBICllr();
OPCODE ARM7opBIClri(), ARM7opBIClrr();
OPCODE ARM7opBICari(), ARM7opBICarr();
OPCODE ARM7opBICrri(), ARM7opBICrrr();
OPCODE ARM7opBICimm();

OPCODE ARM7opMVNlli(), ARM7opMVNllr();
OPCODE ARM7opMVNlri(), ARM7opMVNlrr();
OPCODE ARM7opMVNari(), ARM7opMVNarr();
OPCODE ARM7opMVNrri(), ARM7opMVNrrr();
OPCODE ARM7opMVNimm();

OPCODE ARM7opANDSlli(), ARM7opANDSllr();
OPCODE ARM7opANDSlri(), ARM7opANDSlrr();
OPCODE ARM7opANDSari(), ARM7opANDSarr();
OPCODE ARM7opANDSrri(), ARM7opANDSrrr();
OPCODE ARM7opANDSimm();

OPCODE ARM7opEORSlli(), ARM7opEORSllr();
OPCODE ARM7opEORSlri(), ARM7opEORSlrr();
OPCODE ARM7opEORSari(), ARM7opEORSarr();
OPCODE ARM7opEORSrri(), ARM7opEORSrrr();
OPCODE ARM7opEORSimm();

OPCODE ARM7opSUBSlli(), ARM7opSUBSllr();
OPCODE ARM7opSUBSlri(), ARM7opSUBSlrr();
OPCODE ARM7opSUBSari(), ARM7opSUBSarr();
OPCODE ARM7opSUBSrri(), ARM7opSUBSrrr();
OPCODE ARM7opSUBSimm();

OPCODE ARM7opRSBSlli(), ARM7opRSBSllr();
OPCODE ARM7opRSBSlri(), ARM7opRSBSlrr();
OPCODE ARM7opRSBSari(), ARM7opRSBSarr();
OPCODE ARM7opRSBSrri(), ARM7opRSBSrrr();
OPCODE ARM7opRSBSimm();

OPCODE ARM7opADDSlli(), ARM7opADDSllr();
OPCODE ARM7opADDSlri(), ARM7opADDSlrr();
OPCODE ARM7opADDSari(), ARM7opADDSarr();
OPCODE ARM7opADDSrri(), ARM7opADDSrrr();
OPCODE ARM7opADDSimm();

OPCODE ARM7opADCSlli(), ARM7opADCSllr();
OPCODE ARM7opADCSlri(), ARM7opADCSlrr();
OPCODE ARM7opADCSari(), ARM7opADCSarr();
OPCODE ARM7opADCSrri(), ARM7opADCSrrr();
OPCODE ARM7opADCSimm();

OPCODE ARM7opSBCSlli(), ARM7opSBCSllr();
OPCODE ARM7opSBCSlri(), ARM7opSBCSlrr();
OPCODE ARM7opSBCSari(), ARM7opSBCSarr();
OPCODE ARM7opSBCSrri(), ARM7opSBCSrrr();
OPCODE ARM7opSBCSimm();

OPCODE ARM7opRSCSlli(), ARM7opRSCSllr();
OPCODE ARM7opRSCSlri(), ARM7opRSCSlrr();
OPCODE ARM7opRSCSari(), ARM7opRSCSarr();
OPCODE ARM7opRSCSrri(), ARM7opRSCSrrr();
OPCODE ARM7opRSCSimm();

OPCODE ARM7opTSTSlli(), ARM7opTSTSllr();
OPCODE ARM7opTSTSlri(), ARM7opTSTSlrr();
OPCODE ARM7opTSTSari(), ARM7opTSTSarr();
OPCODE ARM7opTSTSrri(), ARM7opTSTSrrr();
OPCODE ARM7opTSTSimm();

OPCODE ARM7opTEQSlli(), ARM7opTEQSllr();
OPCODE ARM7opTEQSlri(), ARM7opTEQSlrr();
OPCODE ARM7opTEQSari(), ARM7opTEQSarr();
OPCODE ARM7opTEQSrri(), ARM7opTEQSrrr();
OPCODE ARM7opTEQSimm();

OPCODE ARM7opCMPSlli(), ARM7opCMPSllr();
OPCODE ARM7opCMPSlri(), ARM7opCMPSlrr();
OPCODE ARM7opCMPSari(), ARM7opCMPSarr();
OPCODE ARM7opCMPSrri(), ARM7opCMPSrrr();
OPCODE ARM7opCMPSimm();

OPCODE ARM7opCMNSlli(), ARM7opCMNSllr();
OPCODE ARM7opCMNSlri(), ARM7opCMNSlrr();
OPCODE ARM7opCMNSari(), ARM7opCMNSarr();
OPCODE ARM7opCMNSrri(), ARM7opCMNSrrr();
OPCODE ARM7opCMNSimm();

OPCODE ARM7opORRSlli(), ARM7opORRSllr();
OPCODE ARM7opORRSlri(), ARM7opORRSlrr();
OPCODE ARM7opORRSari(), ARM7opORRSarr();
OPCODE ARM7opORRSrri(), ARM7opORRSrrr();
OPCODE ARM7opORRSimm();

OPCODE ARM7opMOVSlli(), ARM7opMOVSllr();
OPCODE ARM7opMOVSlri(), ARM7opMOVSlrr();
OPCODE ARM7opMOVSari(), ARM7opMOVSarr();
OPCODE ARM7opMOVSrri(), ARM7opMOVSrrr();
OPCODE ARM7opMOVSimm();

OPCODE ARM7opBICSlli(), ARM7opBICSllr();
OPCODE ARM7opBICSlri(), ARM7opBICSlrr();
OPCODE ARM7opBICSari(), ARM7opBICSarr();
OPCODE ARM7opBICSrri(), ARM7opBICSrrr();
OPCODE ARM7opBICSimm();

OPCODE ARM7opMVNSlli(), ARM7opMVNSllr();
OPCODE ARM7opMVNSlri(), ARM7opMVNSlrr();
OPCODE ARM7opMVNSari(), ARM7opMVNSarr();
OPCODE ARM7opMVNSrri(), ARM7opMVNSrrr();
OPCODE ARM7opMVNSimm();

//---AR (ARithmetic); REGister/IMMediate-----------------------------------

OPCODE ARM7opMUL(), ARM7opMLA();
OPCODE ARM7opUMULL(), ARM7opUMLAL(), ARM7opSMULL(), ARM7opSMLAL();

OPCODE ARM7opMULS(), ARM7opMLAS();
OPCODE ARM7opUMULLS(), ARM7opUMLALS(), ARM7opSMULLS(), ARM7opSMLALS();

//---Move Register From Status; Imm/Reg; Cpsr/Spsr-------------------------

OPCODE ARM7opMRSrc(), ARM7opMRSrs();

//---Move Status from Register; Cpsr/Spsr----------------------------------

OPCODE ARM7opMSRrc(), ARM7opMSRrs();
OPCODE ARM7opMSRic(), ARM7opMSRis();

//---LS (Load/Store); PReidx/OFfset/PsTidx; Imm/Reg; Minus/Plus; (Shift)---

OPCODE ARM7opLDRptim(), ARM7opLDRofim(), ARM7opLDRprim();
OPCODE ARM7opLDRptip(), ARM7opLDRofip(), ARM7opLDRprip();
OPCODE ARM7opLDRptrm(), ARM7opLDRofrm(), ARM7opLDRprrm();
OPCODE ARM7opLDRptrp(), ARM7opLDRofrp(), ARM7opLDRprrp();

OPCODE ARM7opSTRptim(), ARM7opSTRofim(), ARM7opSTRprim();
OPCODE ARM7opSTRptip(), ARM7opSTRofip(), ARM7opSTRprip();
OPCODE ARM7opSTRptrm(), ARM7opSTRofrm(), ARM7opSTRprrm();
OPCODE ARM7opSTRptrp(), ARM7opSTRofrp(), ARM7opSTRprrp();

OPCODE ARM7opLDRBptim(), ARM7opLDRBofim(), ARM7opLDRBprim();
OPCODE ARM7opLDRBptip(), ARM7opLDRBofip(), ARM7opLDRBprip();
OPCODE ARM7opLDRBptrm(), ARM7opLDRBofrm(), ARM7opLDRBprrm();
OPCODE ARM7opLDRBptrp(), ARM7opLDRBofrp(), ARM7opLDRBprrp();

OPCODE ARM7opSTRBptim(), ARM7opSTRBofim(), ARM7opSTRBprim();
OPCODE ARM7opSTRBptip(), ARM7opSTRBofip(), ARM7opSTRBprip();
OPCODE ARM7opSTRBptrm(), ARM7opSTRBofrm(), ARM7opSTRBprrm();
OPCODE ARM7opSTRBptrp(), ARM7opSTRBofrp(), ARM7opSTRBprrp();

OPCODE ARM7opLDRHptim(), ARM7opLDRHofim(), ARM7opLDRHprim();
OPCODE ARM7opLDRHptip(), ARM7opLDRHofip(), ARM7opLDRHprip();
OPCODE ARM7opLDRHptrm(), ARM7opLDRHofrm(), ARM7opLDRHprrm();
OPCODE ARM7opLDRHptrp(), ARM7opLDRHofrp(), ARM7opLDRHprrp();

OPCODE ARM7opSTRHptim(), ARM7opSTRHofim(), ARM7opSTRHprim();
OPCODE ARM7opSTRHptip(), ARM7opSTRHofip(), ARM7opSTRHprip();
OPCODE ARM7opSTRHptrm(), ARM7opSTRHofrm(), ARM7opSTRHprrm();
OPCODE ARM7opSTRHptrp(), ARM7opSTRHofrp(), ARM7opSTRHprrp();

OPCODE ARM7opLDRTptim(), ARM7opLDRTofim(), ARM7opLDRTprim();
OPCODE ARM7opLDRTptip(), ARM7opLDRTofip(), ARM7opLDRTprip();
OPCODE ARM7opLDRTptrm(), ARM7opLDRTofrm(), ARM7opLDRTprrm();
OPCODE ARM7opLDRTptrp(), ARM7opLDRTofrp(), ARM7opLDRTprrp();

OPCODE ARM7opSTRTptim(), ARM7opSTRTofim(), ARM7opSTRTprim();
OPCODE ARM7opSTRTptip(), ARM7opSTRTofip(), ARM7opSTRTprip();
OPCODE ARM7opSTRTptrm(), ARM7opSTRTofrm(), ARM7opSTRTprrm();
OPCODE ARM7opSTRTptrp(), ARM7opSTRTofrp(), ARM7opSTRTprrp();

OPCODE ARM7opLDRBTptim(), ARM7opLDRBTofim(), ARM7opLDRBTprim();
OPCODE ARM7opLDRBTptip(), ARM7opLDRBTofip(), ARM7opLDRBTprip();
OPCODE ARM7opLDRBTptrm(), ARM7opLDRBTofrm(), ARM7opLDRBTprrm();
OPCODE ARM7opLDRBTptrp(), ARM7opLDRBTofrp(), ARM7opLDRBTprrp();

OPCODE ARM7opSTRBTptim(), ARM7opSTRBTofim(), ARM7opSTRBTprim();
OPCODE ARM7opSTRBTptip(), ARM7opSTRBTofip(), ARM7opSTRBTprip();
OPCODE ARM7opSTRBTptrm(), ARM7opSTRBTofrm(), ARM7opSTRBTprrm();
OPCODE ARM7opSTRBTptrp(), ARM7opSTRBTofrp(), ARM7opSTRBTprrp();

OPCODE ARM7opLDRSHptim(), ARM7opLDRSHofim(), ARM7opLDRSHprim();
OPCODE ARM7opLDRSHptip(), ARM7opLDRSHofip(), ARM7opLDRSHprip();
OPCODE ARM7opLDRSHptrm(), ARM7opLDRSHofrm(), ARM7opLDRSHprrm();
OPCODE ARM7opLDRSHptrp(), ARM7opLDRSHofrp(), ARM7opLDRSHprrp();

OPCODE ARM7opLDRSBptim(), ARM7opLDRSBofim(), ARM7opLDRSBprim();
OPCODE ARM7opLDRSBptip(), ARM7opLDRSBofip(), ARM7opLDRSBprip();
OPCODE ARM7opLDRSBptrm(), ARM7opLDRSBofrm(), ARM7opLDRSBprrm();
OPCODE ARM7opLDRSBptrp(), ARM7opLDRSBofrp(), ARM7opLDRSBprrp();

OPCODE ARM7opLDRptrmll(), ARM7opLDRofrmll(), ARM7opLDRprrmll();
OPCODE ARM7opLDRptrpll(), ARM7opLDRofrpll(), ARM7opLDRprrpll();
OPCODE ARM7opLDRptrmlr(), ARM7opLDRofrmlr(), ARM7opLDRprrmlr();
OPCODE ARM7opLDRptrplr(), ARM7opLDRofrplr(), ARM7opLDRprrplr();
OPCODE ARM7opLDRptrmar(), ARM7opLDRofrmar(), ARM7opLDRprrmar();
OPCODE ARM7opLDRptrpar(), ARM7opLDRofrpar(), ARM7opLDRprrpar();
OPCODE ARM7opLDRptrmrr(), ARM7opLDRofrmrr(), ARM7opLDRprrmrr();
OPCODE ARM7opLDRptrprr(), ARM7opLDRofrprr(), ARM7opLDRprrprr();

OPCODE ARM7opSTRptrmll(), ARM7opSTRofrmll(), ARM7opSTRprrmll();
OPCODE ARM7opSTRptrpll(), ARM7opSTRofrpll(), ARM7opSTRprrpll();
OPCODE ARM7opSTRptrmlr(), ARM7opSTRofrmlr(), ARM7opSTRprrmlr();
OPCODE ARM7opSTRptrplr(), ARM7opSTRofrplr(), ARM7opSTRprrplr();
OPCODE ARM7opSTRptrmar(), ARM7opSTRofrmar(), ARM7opSTRprrmar();
OPCODE ARM7opSTRptrpar(), ARM7opSTRofrpar(), ARM7opSTRprrpar();
OPCODE ARM7opSTRptrmrr(), ARM7opSTRofrmrr(), ARM7opSTRprrmrr();
OPCODE ARM7opSTRptrprr(), ARM7opSTRofrprr(), ARM7opSTRprrprr();

OPCODE ARM7opLDRBptrmll(), ARM7opLDRBofrmll(), ARM7opLDRBprrmll();
OPCODE ARM7opLDRBptrpll(), ARM7opLDRBofrpll(), ARM7opLDRBprrpll();
OPCODE ARM7opLDRBptrmlr(), ARM7opLDRBofrmlr(), ARM7opLDRBprrmlr();
OPCODE ARM7opLDRBptrplr(), ARM7opLDRBofrplr(), ARM7opLDRBprrplr();
OPCODE ARM7opLDRBptrmar(), ARM7opLDRBofrmar(), ARM7opLDRBprrmar();
OPCODE ARM7opLDRBptrpar(), ARM7opLDRBofrpar(), ARM7opLDRBprrpar();
OPCODE ARM7opLDRBptrmrr(), ARM7opLDRBofrmrr(), ARM7opLDRBprrmrr();
OPCODE ARM7opLDRBptrprr(), ARM7opLDRBofrprr(), ARM7opLDRBprrprr();

OPCODE ARM7opSTRBptrmll(), ARM7opSTRBofrmll(), ARM7opSTRBprrmll();
OPCODE ARM7opSTRBptrpll(), ARM7opSTRBofrpll(), ARM7opSTRBprrpll();
OPCODE ARM7opSTRBptrmlr(), ARM7opSTRBofrmlr(), ARM7opSTRBprrmlr();
OPCODE ARM7opSTRBptrplr(), ARM7opSTRBofrplr(), ARM7opSTRBprrplr();
OPCODE ARM7opSTRBptrmar(), ARM7opSTRBofrmar(), ARM7opSTRBprrmar();
OPCODE ARM7opSTRBptrpar(), ARM7opSTRBofrpar(), ARM7opSTRBprrpar();
OPCODE ARM7opSTRBptrmrr(), ARM7opSTRBofrmrr(), ARM7opSTRBprrmrr();
OPCODE ARM7opSTRBptrprr(), ARM7opSTRBofrprr(), ARM7opSTRBprrprr();

OPCODE ARM7opLDRTptrmll(), ARM7opLDRTofrmll(), ARM7opLDRTprrmll();
OPCODE ARM7opLDRTptrpll(), ARM7opLDRTofrpll(), ARM7opLDRTprrpll();
OPCODE ARM7opLDRTptrmlr(), ARM7opLDRTofrmlr(), ARM7opLDRTprrmlr();
OPCODE ARM7opLDRTptrplr(), ARM7opLDRTofrplr(), ARM7opLDRTprrplr();
OPCODE ARM7opLDRTptrmar(), ARM7opLDRTofrmar(), ARM7opLDRTprrmar();
OPCODE ARM7opLDRTptrpar(), ARM7opLDRTofrpar(), ARM7opLDRTprrpar();
OPCODE ARM7opLDRTptrmrr(), ARM7opLDRTofrmrr(), ARM7opLDRTprrmrr();
OPCODE ARM7opLDRTptrprr(), ARM7opLDRTofrprr(), ARM7opLDRTprrprr();

OPCODE ARM7opSTRTptrmll(), ARM7opSTRTofrmll(), ARM7opSTRTprrmll();
OPCODE ARM7opSTRTptrpll(), ARM7opSTRTofrpll(), ARM7opSTRTprrpll();
OPCODE ARM7opSTRTptrmlr(), ARM7opSTRTofrmlr(), ARM7opSTRTprrmlr();
OPCODE ARM7opSTRTptrplr(), ARM7opSTRTofrplr(), ARM7opSTRTprrplr();
OPCODE ARM7opSTRTptrmar(), ARM7opSTRTofrmar(), ARM7opSTRTprrmar();
OPCODE ARM7opSTRTptrpar(), ARM7opSTRTofrpar(), ARM7opSTRTprrpar();
OPCODE ARM7opSTRTptrmrr(), ARM7opSTRTofrmrr(), ARM7opSTRTprrmrr();
OPCODE ARM7opSTRTptrprr(), ARM7opSTRTofrprr(), ARM7opSTRTprrprr();

OPCODE ARM7opLDRBTptrmll(), ARM7opLDRBTofrmll(), ARM7opLDRBTprrmll();
OPCODE ARM7opLDRBTptrpll(), ARM7opLDRBTofrpll(), ARM7opLDRBTprrpll();
OPCODE ARM7opLDRBTptrmlr(), ARM7opLDRBTofrmlr(), ARM7opLDRBTprrmlr();
OPCODE ARM7opLDRBTptrplr(), ARM7opLDRBTofrplr(), ARM7opLDRBTprrplr();
OPCODE ARM7opLDRBTptrmar(), ARM7opLDRBTofrmar(), ARM7opLDRBTprrmar();
OPCODE ARM7opLDRBTptrpar(), ARM7opLDRBTofrpar(), ARM7opLDRBTprrpar();
OPCODE ARM7opLDRBTptrmrr(), ARM7opLDRBTofrmrr(), ARM7opLDRBTprrmrr();
OPCODE ARM7opLDRBTptrprr(), ARM7opLDRBTofrprr(), ARM7opLDRBTprrprr();

OPCODE ARM7opSTRBTptrmll(), ARM7opSTRBTofrmll(), ARM7opSTRBTprrmll();
OPCODE ARM7opSTRBTptrpll(), ARM7opSTRBTofrpll(), ARM7opSTRBTprrpll();
OPCODE ARM7opSTRBTptrmlr(), ARM7opSTRBTofrmlr(), ARM7opSTRBTprrmlr();
OPCODE ARM7opSTRBTptrplr(), ARM7opSTRBTofrplr(), ARM7opSTRBTprrplr();
OPCODE ARM7opSTRBTptrmar(), ARM7opSTRBTofrmar(), ARM7opSTRBTprrmar();
OPCODE ARM7opSTRBTptrpar(), ARM7opSTRBTofrpar(), ARM7opSTRBTprrpar();
OPCODE ARM7opSTRBTptrmrr(), ARM7opSTRBTofrmrr(), ARM7opSTRBTprrmrr();
OPCODE ARM7opSTRBTptrprr(), ARM7opSTRBTofrprr(), ARM7opSTRBTprrprr();

//---SWP-------------------------------------------------------------------

OPCODE ARM7opSWP(), ARM7opSWPB();

//---LoaD/STore Multiple; Increment/Decrement After/Before-----------------

OPCODE ARM7opLDMIA(), ARM7opLDMIB(), ARM7opLDMDA(), ARM7opLDMDB();
OPCODE ARM7opSTMIA(), ARM7opSTMIB(), ARM7opSTMDA(), ARM7opSTMDB();

OPCODE ARM7opLDMIAu(), ARM7opLDMIBu(), ARM7opLDMDAu(), ARM7opLDMDBu();
OPCODE ARM7opSTMIAu(), ARM7opSTMIBu(), ARM7opSTMDAu(), ARM7opSTMDBu();

OPCODE ARM7opLDMIAw(), ARM7opLDMIBw(), ARM7opLDMDAw(), ARM7opLDMDBw();
OPCODE ARM7opSTMIAw(), ARM7opSTMIBw(), ARM7opSTMDAw(), ARM7opSTMDBw();

OPCODE ARM7opLDMIAuw(), ARM7opLDMIBuw(), ARM7opLDMDAuw(), ARM7opLDMDBuw();
OPCODE ARM7opSTMIAuw(), ARM7opSTMIBuw(), ARM7opSTMDAuw(), ARM7opSTMDBuw();

//---SoftWare Interrupts---------------------------------------------------

OPCODE ARM7opSWI();

//---Thumb ops-------------------------------------------------------------

OPCODE Thumb7opUND();

OPCODE Thumb7opLSLimm(), Thumb7opLSRimm(), Thumb7opASRimm();
OPCODE Thumb7opADDreg(), Thumb7opSUBreg();
OPCODE Thumb7opADDimm3(), Thumb7opSUBimm3();

OPCODE Thumb7opMOVimm8r0(), Thumb7opMOVimm8r1();
OPCODE Thumb7opMOVimm8r2(), Thumb7opMOVimm8r3();
OPCODE Thumb7opMOVimm8r4(), Thumb7opMOVimm8r5();
OPCODE Thumb7opMOVimm8r6(), Thumb7opMOVimm8r7();
OPCODE Thumb7opCMPimm8r0(), Thumb7opCMPimm8r1();
OPCODE Thumb7opCMPimm8r2(), Thumb7opCMPimm8r3();
OPCODE Thumb7opCMPimm8r4(), Thumb7opCMPimm8r5();
OPCODE Thumb7opCMPimm8r6(), Thumb7opCMPimm8r7();
OPCODE Thumb7opADDimm8r0(), Thumb7opADDimm8r1();
OPCODE Thumb7opADDimm8r2(), Thumb7opADDimm8r3();
OPCODE Thumb7opADDimm8r4(), Thumb7opADDimm8r5();
OPCODE Thumb7opADDimm8r6(), Thumb7opADDimm8r7();
OPCODE Thumb7opSUBimm8r0(), Thumb7opSUBimm8r1();
OPCODE Thumb7opSUBimm8r2(), Thumb7opSUBimm8r3();
OPCODE Thumb7opSUBimm8r4(), Thumb7opSUBimm8r5();
OPCODE Thumb7opSUBimm8r6(), Thumb7opSUBimm8r7();

OPCODE Thumb7opDPg1(), Thumb7opDPg2(), Thumb7opDPg3(), Thumb7opDPg4();
OPCODE Thumb7opADDH(), Thumb7opMOVH(), Thumb7opCMPH(), Thumb7opBXreg();

OPCODE Thumb7opLDRPCr0(), Thumb7opLDRPCr1();
OPCODE Thumb7opLDRPCr2(), Thumb7opLDRPCr3();
OPCODE Thumb7opLDRPCr4(), Thumb7opLDRPCr5();
OPCODE Thumb7opLDRPCr6(), Thumb7opLDRPCr7();

OPCODE Thumb7opSTRSPr0(), Thumb7opSTRSPr1();
OPCODE Thumb7opSTRSPr2(), Thumb7opSTRSPr3();
OPCODE Thumb7opSTRSPr4(), Thumb7opSTRSPr5();
OPCODE Thumb7opSTRSPr6(), Thumb7opSTRSPr7();
OPCODE Thumb7opLDRSPr0(), Thumb7opLDRSPr1();
OPCODE Thumb7opLDRSPr2(), Thumb7opLDRSPr3();
OPCODE Thumb7opLDRSPr4(), Thumb7opLDRSPr5();
OPCODE Thumb7opLDRSPr6(), Thumb7opLDRSPr7();

OPCODE Thumb7opSTRreg(),  Thumb7opLDRreg();
OPCODE Thumb7opSTRHreg(), Thumb7opLDRHreg();
OPCODE Thumb7opSTRBreg(), Thumb7opLDRBreg();

OPCODE Thumb7opLDRSBreg(), Thumb7opLDRSHreg();

OPCODE Thumb7opSTRimm5(),  Thumb7opLDRimm5();
OPCODE Thumb7opSTRBimm5(), Thumb7opLDRBimm5();
OPCODE Thumb7opSTRHimm5(), Thumb7opLDRHimm5();

OPCODE Thumb7opADDPCr0(), Thumb7opADDPCr1();
OPCODE Thumb7opADDPCr2(), Thumb7opADDPCr3();
OPCODE Thumb7opADDPCr4(), Thumb7opADDPCr5();
OPCODE Thumb7opADDPCr6(), Thumb7opADDPCr7();
OPCODE Thumb7opADDSPr0(), Thumb7opADDSPr1();
OPCODE Thumb7opADDSPr2(), Thumb7opADDSPr3();
OPCODE Thumb7opADDSPr4(), Thumb7opADDSPr5();
OPCODE Thumb7opADDSPr6(), Thumb7opADDSPr7();
OPCODE Thumb7opADDSPimm7();

OPCODE Thumb7opPUSH(), Thumb7opPUSHlr();
OPCODE Thumb7opPOP(),  Thumb7opPOPpc();

OPCODE Thumb7opBKPT(), Thumb7opSWI();

OPCODE Thumb7opLDMIAr0(), Thumb7opLDMIAr1();
OPCODE Thumb7opLDMIAr2(), Thumb7opLDMIAr3();
OPCODE Thumb7opLDMIAr4(), Thumb7opLDMIAr5();
OPCODE Thumb7opLDMIAr6(), Thumb7opLDMIAr7();
OPCODE Thumb7opSTMIAr0(), Thumb7opSTMIAr1();
OPCODE Thumb7opSTMIAr2(), Thumb7opSTMIAr3();
OPCODE Thumb7opSTMIAr4(), Thumb7opSTMIAr5();
OPCODE Thumb7opSTMIAr6(), Thumb7opSTMIAr7();

OPCODE Thumb7opBEQ(), Thumb7opBNE();
OPCODE Thumb7opBCS(), Thumb7opBCC();
OPCODE Thumb7opBMI(), Thumb7opBPL();
OPCODE Thumb7opBVS(), Thumb7opBVC();
OPCODE Thumb7opBHI(), Thumb7opBLS();
OPCODE Thumb7opBGE(), Thumb7opBLT();
OPCODE Thumb7opBGT(), Thumb7opBLE();

OPCODE Thumb7opB(), Thumb7opBLsetup(), Thumb7opBLoff();

//---And so it begins------------------------------------------------------

typedef struct {
    opfptr op;
    u32 cyc;
    u32 pad1,pad2;
} ARM7OPDESC;

//---Opcode function pointer table. Indexed by 27-20|7-4-------------------

//const fptr arm7uber[16]={
//    ARM7opDPS0,      /* 0 - Data processing Shifted */
//    ARM7opDPS1,      /* 1 - Data processing Shifted */
//    ARM7opDPI0,      /* 2 - Data processing Immediate */
//    ARM7opDPI1,      /* 3 - Data processing Immediate */
//    ARM7opLSIpt,     /* 4 - Load/store Immediate Post-indexed */
//    ARM7opLSIof,     /* 5 - Load/store Immediate Preidx/Offset */
//    ARM7opLSRpt,     /* 6 - Load/store Register Post-indexed */
//    ARM7opLSRof,     /* 7 - Load/store Register Preidx/Offset */
//    ARM7opLSMf,      /* 8 - Load/store Multiple Load:Full */
//    ARM7opLSMe,      /* 9 - Load/store Multiple Load:Empty */
//    ARM7opB,         /* A - Branch */
//    ARM7opBL,        /* B - Branch+Link */
//    ARM7opLSCof,     /* C - Load/store Copro Preidx/Offset */
//    ARM7opLSCpt,     /* D - Load/store Copro Post-indexed */
//    ARM7opCP,        /* E - Copro Data/Reg handling */
//    ARM7opSWI        /* F - Software Interrupt */
//};

const opfptr arm7ops[]={
    /*0x00*/
    ARM7opANDlli      ,
    ARM7opANDllr      ,
    ARM7opANDlri      ,
    ARM7opANDlrr      ,
    ARM7opANDari      ,
    ARM7opANDarr      ,
    ARM7opANDrri      ,
    ARM7opANDrrr      ,
    ARM7opANDlli      ,
    ARM7opMUL         ,
    ARM7opANDlri      ,
    ARM7opSTRHptrm    ,
    ARM7opANDari      ,
    ARM7opUND,     // Defined in ARM5
    ARM7opANDrri      ,
    ARM7opUND,     // Defined in ARM5

    /*0x01*/
    ARM7opANDSlli     ,
    ARM7opANDSllr     ,
    ARM7opANDSlri     ,
    ARM7opANDSlrr     ,
    ARM7opANDSari     ,
    ARM7opANDSarr     ,
    ARM7opANDSrri     ,
    ARM7opANDSrrr     ,
    ARM7opANDSlli     ,
    ARM7opMULS        ,
    ARM7opANDSlri     ,
    ARM7opLDRHptrm    ,
    ARM7opANDSari     ,
    ARM7opLDRSBptrm   ,
    ARM7opANDSrri     ,
    ARM7opLDRSHptrm   ,

    /*0x02*/
    ARM7opEORlli      ,
    ARM7opEORllr      ,
    ARM7opEORlri      ,
    ARM7opEORlrr      ,
    ARM7opEORari      ,
    ARM7opEORarr      ,
    ARM7opEORrri      ,
    ARM7opEORrrr      ,
    ARM7opEORlli      ,
    ARM7opMLA         ,
    ARM7opEORlri      ,
    ARM7opSTRHptrm    ,
    ARM7opEORari      ,
    ARM7opUND,     // Defined in ARM5
    ARM7opEORrri      ,
    ARM7opUND,     // Defined in ARM5

    /*0x03*/
    ARM7opEORSlli     ,
    ARM7opEORSllr     ,
    ARM7opEORSlri     ,
    ARM7opEORSlrr     ,
    ARM7opEORSari     ,
    ARM7opEORSarr     ,
    ARM7opEORSrri     ,
    ARM7opEORSrrr     ,
    ARM7opEORSlli     ,
    ARM7opMLAS        ,
    ARM7opEORSlri     ,
    ARM7opLDRHptrm    ,
    ARM7opEORSari     ,
    ARM7opLDRSBptrm   ,
    ARM7opEORSrri     ,
    ARM7opLDRSHptrm   ,

    /*0x04*/
    ARM7opSUBlli      ,
    ARM7opSUBllr      ,
    ARM7opSUBlri      ,
    ARM7opSUBlrr      ,
    ARM7opSUBari      ,
    ARM7opSUBarr      ,
    ARM7opSUBrri      ,
    ARM7opSUBrrr      ,
    ARM7opSUBlli      ,
    ARM7opUNP         ,  // Unallocated
    ARM7opSUBlri      ,
    ARM7opSTRHptim    ,
    ARM7opSUBari      ,
    ARM7opUND,     // Defined in ARM5
    ARM7opSUBrri      ,
    ARM7opUND,     // Defined in ARM5

    /*0x05*/
    ARM7opSUBSlli     ,
    ARM7opSUBSllr     ,
    ARM7opSUBSlri     ,
    ARM7opSUBSlrr     ,
    ARM7opSUBSari     ,
    ARM7opSUBSarr     ,
    ARM7opSUBSrri     ,
    ARM7opSUBSrrr     ,
    ARM7opSUBSlli     ,
    ARM7opUNL         ,  // Unallocated [reindexed]
    ARM7opSUBSlri     ,
    ARM7opLDRHptim    ,
    ARM7opSUBSari     ,
    ARM7opLDRSBptim   ,
    ARM7opSUBSrri     ,
    ARM7opLDRSHptim   ,

    /*0x06*/
    ARM7opRSBlli      ,
    ARM7opRSBllr      ,
    ARM7opRSBlri      ,
    ARM7opRSBlrr      ,
    ARM7opRSBari      ,
    ARM7opRSBarr      ,
    ARM7opRSBrri      ,
    ARM7opRSBrrr      ,
    ARM7opRSBlli      ,
    ARM7opUNP         ,  // Unallocated
    ARM7opRSBlri      ,
    ARM7opSTRHptim    ,
    ARM7opRSBari      ,
    ARM7opUND,     // Defined in ARM5
    ARM7opRSBrri      ,
    ARM7opUND,     // Defined in ARM5

    /*0x07*/
    ARM7opRSBSlli     ,
    ARM7opRSBSllr     ,
    ARM7opRSBSlri     ,
    ARM7opRSBSlrr     ,
    ARM7opRSBSari     ,
    ARM7opRSBSarr     ,
    ARM7opRSBSrri     ,
    ARM7opRSBSrrr     ,
    ARM7opRSBSlli     ,
    ARM7opUNP         ,  // Unallocated
    ARM7opRSBSlri     ,
    ARM7opLDRHptim    ,
    ARM7opRSBSari     ,
    ARM7opLDRSBptim   ,
    ARM7opRSBSrri     ,
    ARM7opLDRSHptim   ,

    /*0x08*/
    ARM7opADDlli      ,
    ARM7opADDllr      ,
    ARM7opADDlri      ,
    ARM7opADDlrr      ,
    ARM7opADDari      ,
    ARM7opADDarr      ,
    ARM7opADDrri      ,
    ARM7opADDrrr      ,
    ARM7opADDlli      ,
    ARM7opUMULL       ,
    ARM7opADDlri      ,
    ARM7opSTRHptrp    ,
    ARM7opADDari      ,
    ARM7opUND,     // Defined in ARM5
    ARM7opADDrri      ,
    ARM7opUND,     // Defined in ARM5

    /*0x09*/
    ARM7opADDSlli     ,
    ARM7opADDSllr     ,
    ARM7opADDSlri     ,
    ARM7opADDSlrr     ,
    ARM7opADDSari     ,
    ARM7opADDSarr     ,
    ARM7opADDSrri     ,
    ARM7opADDSrrr     ,
    ARM7opADDSlli     ,
    ARM7opUMULLS      ,
    ARM7opADDSlri     ,
    ARM7opLDRHptrp    ,
    ARM7opADDSari     ,
    ARM7opLDRSBptrp   ,
    ARM7opADDSrri     ,
    ARM7opLDRSHptrp   ,

    /*0x0A*/
    ARM7opADClli      ,
    ARM7opADCllr      ,
    ARM7opADClri      ,
    ARM7opADClrr      ,
    ARM7opADCari      ,
    ARM7opADCarr      ,
    ARM7opADCrri      ,
    ARM7opADCrrr      ,
    ARM7opADClli      ,
    ARM7opUMLAL       ,
    ARM7opADClri      ,
    ARM7opSTRHptrp    ,
    ARM7opADCari      ,
    ARM7opUND,     // Defined in ARM5
    ARM7opADCrri      ,
    ARM7opUND,     // Defined in ARM5

    /*0x0B*/
    ARM7opADCSlli     ,
    ARM7opADCSllr     ,
    ARM7opADCSlri     ,
    ARM7opADCSlrr     ,
    ARM7opADCSari     ,
    ARM7opADCSarr     ,
    ARM7opADCSrri     ,
    ARM7opADCSrrr     ,
    ARM7opADCSlli     ,
    ARM7opUMLALS      ,
    ARM7opADCSlri     ,
    ARM7opLDRHptrp    ,
    ARM7opADCSari     ,
    ARM7opLDRSBptrp   ,
    ARM7opADCSrri     ,
    ARM7opLDRSHptrp   ,

    /*0x0C*/
    ARM7opSBClli      ,
    ARM7opSBCllr      ,
    ARM7opSBClri      ,
    ARM7opSBClrr      ,
    ARM7opSBCari      ,
    ARM7opSBCarr      ,
    ARM7opSBCrri      ,
    ARM7opSBCrrr      ,
    ARM7opSBClli      ,
    ARM7opSMULL       ,
    ARM7opSBClri      ,
    ARM7opSTRHptip    ,
    ARM7opSBCari      ,
    ARM7opUND,     // Defined in ARM5
    ARM7opSBCrri      ,
    ARM7opUND,     // Defined in ARM5

    /*0x0D*/
    ARM7opSBCSlli     ,
    ARM7opSBCSllr     ,
    ARM7opSBCSlri     ,
    ARM7opSBCSlrr     ,
    ARM7opSBCSari     ,
    ARM7opSBCSarr     ,
    ARM7opSBCSrri     ,
    ARM7opSBCSrrr     ,
    ARM7opSBCSlli     ,
    ARM7opSMULLS      ,
    ARM7opSBCSlri     ,
    ARM7opLDRHptip    ,
    ARM7opSBCSari     ,
    ARM7opLDRSBptip   ,
    ARM7opSBCSrri     ,
    ARM7opLDRSHptip   ,

    /*0x0E*/
    ARM7opRSClli      ,
    ARM7opRSCllr      ,
    ARM7opRSClri      ,
    ARM7opRSClrr      ,
    ARM7opRSCari      ,
    ARM7opRSCarr      ,
    ARM7opRSCrri      ,
    ARM7opRSCrrr      ,
    ARM7opRSClli      ,
    ARM7opSMLAL       ,
    ARM7opRSClri      ,
    ARM7opSTRHptip    ,
    ARM7opRSCari      ,
    ARM7opUND,     // Defined in ARM5
    ARM7opRSCrri      ,
    ARM7opUND,     // Defined in ARM5

    /*0x0F*/
    ARM7opRSCSlli     ,
    ARM7opRSCSllr     ,
    ARM7opRSCSlri     ,
    ARM7opRSCSlrr     ,
    ARM7opRSCSari     ,
    ARM7opRSCSarr     ,
    ARM7opRSCSrri     ,
    ARM7opRSCSrrr     ,
    ARM7opRSCSlli     ,
    ARM7opSMLALS      ,
    ARM7opRSCSlri     ,
    ARM7opLDRHptip    ,
    ARM7opRSCSari     ,
    ARM7opLDRSBptip   ,
    ARM7opRSCSrri     ,
    ARM7opLDRSHptip   ,

    /*0x10*/
    ARM7opMRSrc       ,
    ARM7opUNP         ,  // MRS with non-zero 7-4
    ARM7opUNP         ,  // MRS with non-zero 7-4
    ARM7opUNP         ,  // MRS with non-zero 7-4
    ARM7opUNP         ,  // MRS with non-zero 7-4
    ARM7opUND,     // Defined in ARM5
    ARM7opUNP         ,  // MRS with non-zero 7-4
    ARM7opUNP         ,  // MRS with non-zero 7-4
    ARM7opUND,     // Defined in ARM5
    ARM7opSWP         ,
    ARM7opUND,     // Defined in ARM5
    ARM7opSTRHofrm    ,
    ARM7opUND,     // Defined in ARM5
    ARM7opUND,     // Defined in ARM5
    ARM7opUND,     // Defined in ARM5
    ARM7opUND,     // Defined in ARM5

    /*0x11*/
    ARM7opTSTSlli     ,
    ARM7opTSTSllr     ,
    ARM7opTSTSlri     ,
    ARM7opTSTSlrr     ,
    ARM7opTSTSari     ,
    ARM7opTSTSarr     ,
    ARM7opTSTSrri     ,
    ARM7opTSTSrrr     ,
    ARM7opTSTSlli     ,
    ARM7opUNP         ,  // Unallocated
    ARM7opTSTSlri     ,
    ARM7opLDRHofrm    ,
    ARM7opTSTSari     ,
    ARM7opLDRSBofrm   ,
    ARM7opTSTSrri     ,
    ARM7opLDRSHofrm   ,

    /*0x12*/
    ARM7opMSRrc       ,
    ARM7opBX          ,
    ARM7opUNP         ,  // Unallocated
    ARM7opBLXr        ,
    ARM7opUNP         ,  // Unallocated
    ARM7opUNP         ,  // Defined in ARM5E!
    ARM7opUNP         ,  // Unallocated
    ARM7opUND        ,   // Defined in ARM5
    ARM7opUNP         ,  // Defined in ARM5E!
    ARM7opUNP         ,  // Unallocated
    ARM7opUNP         ,  // Defined in ARM5E!
    ARM7opSTRHprrm    ,
    ARM7opUNP         ,  // Defined in ARM5E!
    ARM7opUND,     // Defined in ARM5
    ARM7opUNP         ,  // Defined in ARM5E!
    ARM7opUND,     // Defined in ARM5

    /*0x13*/
    ARM7opTEQSlli     ,
    ARM7opTEQSllr     ,
    ARM7opTEQSlri     ,
    ARM7opTEQSlrr     ,
    ARM7opTEQSari     ,
    ARM7opTEQSarr     ,
    ARM7opTEQSrri     ,
    ARM7opTEQSrrr     ,
    ARM7opTEQSlli     ,
    ARM7opUNP         ,  // Unallocated
    ARM7opTEQSlri     ,
    ARM7opLDRHprrm    ,
    ARM7opTEQSari     ,
    ARM7opLDRSBprrm   ,
    ARM7opTEQSrri     ,
    ARM7opLDRSHprrm   ,

    /*0x14*/
    ARM7opMRSrs       ,
    ARM7opUNP         ,  // Unallocated
    ARM7opUNP         ,  // Unallocated
    ARM7opUNP         ,  // Unallocated
    ARM7opUNP         ,  // Unallocated
    ARM7opUNP         ,  // Defined in ARM5E!
    ARM7opUNP         ,  // Unallocated
    ARM7opUNP         ,  // Unallocated
    ARM7opUNP         ,  // Defined in ARM5E!
    ARM7opSWPB        ,
    ARM7opUNP         ,  // Defined in ARM5E!
    ARM7opSTRHofim    ,
    ARM7opUNP         ,  // Defined in ARM5E!
    ARM7opUND,     // Defined in ARM5
    ARM7opUNP         ,  // Defined in ARM5E!
    ARM7opUND,     // Defined in ARM5

    /*0x15*/
    ARM7opCMPSlli     ,
    ARM7opCMPSllr     ,
    ARM7opCMPSlri     ,
    ARM7opCMPSlrr     ,
    ARM7opCMPSari     ,
    ARM7opCMPSarr     ,
    ARM7opCMPSrri     ,
    ARM7opCMPSrrr     ,
    ARM7opCMPSlli     ,
    ARM7opUNP         ,  // Unallocated
    ARM7opCMPSlri     ,
    ARM7opLDRHofim    ,
    ARM7opCMPSari     ,
    ARM7opLDRSBofim   ,
    ARM7opCMPSrri     ,
    ARM7opLDRSHofim   ,

    /*0x16*/
    ARM7opMSRrs       ,
    ARM7opUND         ,  // Defined in ARM5
    ARM7opUNP         ,  // Unallocated
    ARM7opUNP         ,  // Unallocated
    ARM7opUNP         ,  // Unallocated
    ARM7opUNP         ,  // Defined in ARM5E!
    ARM7opUNP         ,  // Unallocated
    ARM7opUNP         ,  // Unallocated
    ARM7opUND,     // Defined in ARM5
    ARM7opUNP         ,  // Unallocated
    ARM7opUND,     // Defined in ARM5
    ARM7opSTRHprim    ,
    ARM7opUND,     // Defined in ARM5
    ARM7opUND,     // Defined in ARM5
    ARM7opUND,     // Defined in ARM5
    ARM7opUND,     // Defined in ARM5

    /*0x17*/
    ARM7opCMNSlli     ,
    ARM7opCMNSllr     ,
    ARM7opCMNSlri     ,
    ARM7opCMNSlrr     ,
    ARM7opCMNSari     ,
    ARM7opCMNSarr     ,
    ARM7opCMNSrri     ,
    ARM7opCMNSrrr     ,
    ARM7opCMNSlli     ,
    ARM7opUNP         ,  // Unallocated
    ARM7opCMNSlri     ,
    ARM7opLDRHprim    ,
    ARM7opCMNSari     ,
    ARM7opLDRSBprim   ,
    ARM7opCMNSrri     ,
    ARM7opLDRSHprim   ,

    /*0x18*/
    ARM7opORRlli      ,
    ARM7opORRllr      ,
    ARM7opORRlri      ,
    ARM7opORRlrr      ,
    ARM7opORRari      ,
    ARM7opORRarr      ,
    ARM7opORRrri      ,
    ARM7opORRrrr      ,
    ARM7opORRlli      ,
    ARM7opUNP         ,  // Unallocated
    ARM7opORRlri      ,
    ARM7opSTRHofrp    ,
    ARM7opORRari      ,
    ARM7opUND,     // Defined in ARM5
    ARM7opORRrri      ,
    ARM7opUND,     // Defined in ARM5

    /*0x19*/
    ARM7opORRSlli     ,
    ARM7opORRSllr     ,
    ARM7opORRSlri     ,
    ARM7opORRSlrr     ,
    ARM7opORRSari     ,
    ARM7opORRSarr     ,
    ARM7opORRSrri     ,
    ARM7opORRSrrr     ,
    ARM7opORRSlli     ,
    ARM7opUNP         ,  // Unallocated
    ARM7opORRSlri     ,
    ARM7opLDRHofrp    ,
    ARM7opORRSari     ,
    ARM7opLDRSBofrp   ,
    ARM7opORRSrri     ,
    ARM7opLDRSHofrp   ,

    /*0x1A*/
    ARM7opMOVlli      ,
    ARM7opMOVllr      ,
    ARM7opMOVlri      ,
    ARM7opMOVlrr      ,
    ARM7opMOVari      ,
    ARM7opMOVarr      ,
    ARM7opMOVrri      ,
    ARM7opMOVrrr      ,
    ARM7opMOVlli      ,
    ARM7opUNP         ,  // Unallocated
    ARM7opMOVlri      ,
    ARM7opSTRHprrp    ,
    ARM7opMOVari      ,
    ARM7opUND,     // Defined in ARM5
    ARM7opMOVrri      ,
    ARM7opUND,     // Defined in ARM5

    /*0x1B*/
    ARM7opMOVSlli     ,
    ARM7opMOVSllr     ,
    ARM7opMOVSlri     ,
    ARM7opMOVSlrr     ,
    ARM7opMOVSari     ,
    ARM7opMOVSarr     ,
    ARM7opMOVSrri     ,
    ARM7opMOVSrrr     ,
    ARM7opMOVSlli     ,
    ARM7opUNP         ,  // Unallocated
    ARM7opMOVSlri     ,
    ARM7opLDRHprrp    ,
    ARM7opMOVSari     ,
    ARM7opLDRSBprrp   ,
    ARM7opMOVSrri     ,
    ARM7opLDRSHprrp   ,

    /*0x1C*/
    ARM7opBIClli      ,
    ARM7opBICllr      ,
    ARM7opBIClri      ,
    ARM7opBIClrr      ,
    ARM7opBICari      ,
    ARM7opBICarr      ,
    ARM7opBICrri      ,
    ARM7opBICrrr      ,
    ARM7opBIClli      ,
    ARM7opUNP         ,  // Unallocated
    ARM7opBIClri      ,
    ARM7opSTRHofip    ,
    ARM7opBICari      ,
    ARM7opUND,     // Defined in ARM5
    ARM7opBICrri      ,
    ARM7opUND,     // Defined in ARM5

    /*0x1D*/
    ARM7opBICSlli     ,
    ARM7opBICSllr     ,
    ARM7opBICSlri     ,
    ARM7opBICSlrr     ,
    ARM7opBICSari     ,
    ARM7opBICSarr     ,
    ARM7opBICSrri     ,
    ARM7opBICSrrr     ,
    ARM7opBICSlli     ,
    ARM7opUNP         ,  // Unallocated
    ARM7opBICSlri     ,
    ARM7opLDRHofip    ,
    ARM7opBICSari     ,
    ARM7opLDRSBofip   ,
    ARM7opBICSrri     ,
    ARM7opLDRSHofip   ,

    /*0x1E*/
    ARM7opMVNlli      ,
    ARM7opMVNllr      ,
    ARM7opMVNlri      ,
    ARM7opMVNlrr      ,
    ARM7opMVNari      ,
    ARM7opMVNarr      ,
    ARM7opMVNrri      ,
    ARM7opMVNrrr      ,
    ARM7opMVNlli      ,
    ARM7opUNP         ,  // Unallocated
    ARM7opMVNlri      ,
    ARM7opSTRHprip    ,
    ARM7opMVNari      ,
    ARM7opUND,     // Defined in ARM5
    ARM7opMVNrri      ,
    ARM7opUND,     // Defined in ARM5

    /*0x1F*/
    ARM7opMVNSlli     ,
    ARM7opMVNSllr     ,
    ARM7opMVNSlri     ,
    ARM7opMVNSlrr     ,
    ARM7opMVNSari     ,
    ARM7opMVNSarr     ,
    ARM7opMVNSrri     ,
    ARM7opMVNSrrr     ,
    ARM7opMVNSlli     ,
    ARM7opUNP         ,  // Unallocated
    ARM7opMVNSlri     ,
    ARM7opLDRHprip    ,
    ARM7opMVNSari     ,
    ARM7opLDRSBprip   ,
    ARM7opMVNSrri     ,
    ARM7opLDRSHprip   ,

    /*0x20*/
    ARM7opANDimm      ,
    ARM7opANDimm      ,
    ARM7opANDimm      ,
    ARM7opANDimm      ,
    ARM7opANDimm      ,
    ARM7opANDimm      ,
    ARM7opANDimm      ,
    ARM7opANDimm      ,
    ARM7opANDimm      ,
    ARM7opANDimm      ,
    ARM7opANDimm      ,
    ARM7opANDimm      ,
    ARM7opANDimm      ,
    ARM7opANDimm      ,
    ARM7opANDimm      ,
    ARM7opANDimm      ,

    /*0x21*/
    ARM7opANDSimm     ,
    ARM7opANDSimm     ,
    ARM7opANDSimm     ,
    ARM7opANDSimm     ,
    ARM7opANDSimm     ,
    ARM7opANDSimm     ,
    ARM7opANDSimm     ,
    ARM7opANDSimm     ,
    ARM7opANDSimm     ,
    ARM7opANDSimm     ,
    ARM7opANDSimm     ,
    ARM7opANDSimm     ,
    ARM7opANDSimm     ,
    ARM7opANDSimm     ,
    ARM7opANDSimm     ,
    ARM7opANDSimm     ,

    /*0x22*/
    ARM7opEORimm      ,
    ARM7opEORimm      ,
    ARM7opEORimm      ,
    ARM7opEORimm      ,
    ARM7opEORimm      ,
    ARM7opEORimm      ,
    ARM7opEORimm      ,
    ARM7opEORimm      ,
    ARM7opEORimm      ,
    ARM7opEORimm      ,
    ARM7opEORimm      ,
    ARM7opEORimm      ,
    ARM7opEORimm      ,
    ARM7opEORimm      ,
    ARM7opEORimm      ,
    ARM7opEORimm      ,

    /*0x23*/
    ARM7opEORSimm     ,
    ARM7opEORSimm     ,
    ARM7opEORSimm     ,
    ARM7opEORSimm     ,
    ARM7opEORSimm     ,
    ARM7opEORSimm     ,
    ARM7opEORSimm     ,
    ARM7opEORSimm     ,
    ARM7opEORSimm     ,
    ARM7opEORSimm     ,
    ARM7opEORSimm     ,
    ARM7opEORSimm     ,
    ARM7opEORSimm     ,
    ARM7opEORSimm     ,
    ARM7opEORSimm     ,
    ARM7opEORSimm     ,

    /*0x24*/
    ARM7opSUBimm      ,
    ARM7opSUBimm      ,
    ARM7opSUBimm      ,
    ARM7opSUBimm      ,
    ARM7opSUBimm      ,
    ARM7opSUBimm      ,
    ARM7opSUBimm      ,
    ARM7opSUBimm      ,
    ARM7opSUBimm      ,
    ARM7opSUBimm      ,
    ARM7opSUBimm      ,
    ARM7opSUBimm      ,
    ARM7opSUBimm      ,
    ARM7opSUBimm      ,
    ARM7opSUBimm      ,
    ARM7opSUBimm      ,

    /*0x25*/
    ARM7opSUBSimm     ,
    ARM7opSUBSimm     ,
    ARM7opSUBSimm     ,
    ARM7opSUBSimm     ,
    ARM7opSUBSimm     ,
    ARM7opSUBSimm     ,
    ARM7opSUBSimm     ,
    ARM7opSUBSimm     ,
    ARM7opSUBSimm     ,
    ARM7opSUBSimm     ,
    ARM7opSUBSimm     ,
    ARM7opSUBSimm     ,
    ARM7opSUBSimm     ,
    ARM7opSUBSimm     ,
    ARM7opSUBSimm     ,
    ARM7opSUBSimm     ,

    /*0x26*/
    ARM7opRSBimm      ,
    ARM7opRSBimm      ,
    ARM7opRSBimm      ,
    ARM7opRSBimm      ,
    ARM7opRSBimm      ,
    ARM7opRSBimm      ,
    ARM7opRSBimm      ,
    ARM7opRSBimm      ,
    ARM7opRSBimm      ,
    ARM7opRSBimm      ,
    ARM7opRSBimm      ,
    ARM7opRSBimm      ,
    ARM7opRSBimm      ,
    ARM7opRSBimm      ,
    ARM7opRSBimm      ,
    ARM7opRSBimm      ,

    /*0x27*/
    ARM7opRSBSimm     ,
    ARM7opRSBSimm     ,
    ARM7opRSBSimm     ,
    ARM7opRSBSimm     ,
    ARM7opRSBSimm     ,
    ARM7opRSBSimm     ,
    ARM7opRSBSimm     ,
    ARM7opRSBSimm     ,
    ARM7opRSBSimm     ,
    ARM7opRSBSimm     ,
    ARM7opRSBSimm     ,
    ARM7opRSBSimm     ,
    ARM7opRSBSimm     ,
    ARM7opRSBSimm     ,
    ARM7opRSBSimm     ,
    ARM7opRSBSimm     ,

    /*0x28*/
    ARM7opADDimm      ,
    ARM7opADDimm      ,
    ARM7opADDimm      ,
    ARM7opADDimm      ,
    ARM7opADDimm      ,
    ARM7opADDimm      ,
    ARM7opADDimm      ,
    ARM7opADDimm      ,
    ARM7opADDimm      ,
    ARM7opADDimm      ,
    ARM7opADDimm      ,
    ARM7opADDimm      ,
    ARM7opADDimm      ,
    ARM7opADDimm      ,
    ARM7opADDimm      ,
    ARM7opADDimm      ,

    /*0x29*/
    ARM7opADDSimm     ,
    ARM7opADDSimm     ,
    ARM7opADDSimm     ,
    ARM7opADDSimm     ,
    ARM7opADDSimm     ,
    ARM7opADDSimm     ,
    ARM7opADDSimm     ,
    ARM7opADDSimm     ,
    ARM7opADDSimm     ,
    ARM7opADDSimm     ,
    ARM7opADDSimm     ,
    ARM7opADDSimm     ,
    ARM7opADDSimm     ,
    ARM7opADDSimm     ,
    ARM7opADDSimm     ,
    ARM7opADDSimm     ,

    /*0x2A*/
    ARM7opADCimm      ,
    ARM7opADCimm      ,
    ARM7opADCimm      ,
    ARM7opADCimm      ,
    ARM7opADCimm      ,
    ARM7opADCimm      ,
    ARM7opADCimm      ,
    ARM7opADCimm      ,
    ARM7opADCimm      ,
    ARM7opADCimm      ,
    ARM7opADCimm      ,
    ARM7opADCimm      ,
    ARM7opADCimm      ,
    ARM7opADCimm      ,
    ARM7opADCimm      ,
    ARM7opADCimm      ,

    /*0x2B*/
    ARM7opADCSimm     ,
    ARM7opADCSimm     ,
    ARM7opADCSimm     ,
    ARM7opADCSimm     ,
    ARM7opADCSimm     ,
    ARM7opADCSimm     ,
    ARM7opADCSimm     ,
    ARM7opADCSimm     ,
    ARM7opADCSimm     ,
    ARM7opADCSimm     ,
    ARM7opADCSimm     ,
    ARM7opADCSimm     ,
    ARM7opADCSimm     ,
    ARM7opADCSimm     ,
    ARM7opADCSimm     ,
    ARM7opADCSimm     ,

    /*0x2C*/
    ARM7opSBCimm      ,
    ARM7opSBCimm      ,
    ARM7opSBCimm      ,
    ARM7opSBCimm      ,
    ARM7opSBCimm      ,
    ARM7opSBCimm      ,
    ARM7opSBCimm      ,
    ARM7opSBCimm      ,
    ARM7opSBCimm      ,
    ARM7opSBCimm      ,
    ARM7opSBCimm      ,
    ARM7opSBCimm      ,
    ARM7opSBCimm      ,
    ARM7opSBCimm      ,
    ARM7opSBCimm      ,
    ARM7opSBCimm      ,

    /*0x2D*/
    ARM7opSBCSimm     ,
    ARM7opSBCSimm     ,
    ARM7opSBCSimm     ,
    ARM7opSBCSimm     ,
    ARM7opSBCSimm     ,
    ARM7opSBCSimm     ,
    ARM7opSBCSimm     ,
    ARM7opSBCSimm     ,
    ARM7opSBCSimm     ,
    ARM7opSBCSimm     ,
    ARM7opSBCSimm     ,
    ARM7opSBCSimm     ,
    ARM7opSBCSimm     ,
    ARM7opSBCSimm     ,
    ARM7opSBCSimm     ,
    ARM7opSBCSimm     ,

    /*0x2E*/
    ARM7opRSCimm      ,
    ARM7opRSCimm      ,
    ARM7opRSCimm      ,
    ARM7opRSCimm      ,
    ARM7opRSCimm      ,
    ARM7opRSCimm      ,
    ARM7opRSCimm      ,
    ARM7opRSCimm      ,
    ARM7opRSCimm      ,
    ARM7opRSCimm      ,
    ARM7opRSCimm      ,
    ARM7opRSCimm      ,
    ARM7opRSCimm      ,
    ARM7opRSCimm      ,
    ARM7opRSCimm      ,
    ARM7opRSCimm      ,

    /*0x2F*/
    ARM7opRSCSimm     ,
    ARM7opRSCSimm     ,
    ARM7opRSCSimm     ,
    ARM7opRSCSimm     ,
    ARM7opRSCSimm     ,
    ARM7opRSCSimm     ,
    ARM7opRSCSimm     ,
    ARM7opRSCSimm     ,
    ARM7opRSCSimm     ,
    ARM7opRSCSimm     ,
    ARM7opRSCSimm     ,
    ARM7opRSCSimm     ,
    ARM7opRSCSimm     ,
    ARM7opRSCSimm     ,
    ARM7opRSCSimm     ,
    ARM7opRSCSimm     ,

    /*0x30*/
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,

    /*0x31*/
    ARM7opTSTSimm     ,
    ARM7opTSTSimm     ,
    ARM7opTSTSimm     ,
    ARM7opTSTSimm     ,
    ARM7opTSTSimm     ,
    ARM7opTSTSimm     ,
    ARM7opTSTSimm     ,
    ARM7opTSTSimm     ,
    ARM7opTSTSimm     ,
    ARM7opTSTSimm     ,
    ARM7opTSTSimm     ,
    ARM7opTSTSimm     ,
    ARM7opTSTSimm     ,
    ARM7opTSTSimm     ,
    ARM7opTSTSimm     ,
    ARM7opTSTSimm     ,

    /*0x32*/
    ARM7opMSRic       ,
    ARM7opMSRic       ,
    ARM7opMSRic       ,
    ARM7opMSRic       ,
    ARM7opMSRic       ,
    ARM7opMSRic       ,
    ARM7opMSRic       ,
    ARM7opMSRic       ,
    ARM7opMSRic       ,
    ARM7opMSRic       ,
    ARM7opMSRic       ,
    ARM7opMSRic       ,
    ARM7opMSRic       ,
    ARM7opMSRic       ,
    ARM7opMSRic       ,
    ARM7opMSRic       ,

    /*0x33*/
    ARM7opTEQSimm     ,
    ARM7opTEQSimm     ,
    ARM7opTEQSimm     ,
    ARM7opTEQSimm     ,
    ARM7opTEQSimm     ,
    ARM7opTEQSimm     ,
    ARM7opTEQSimm     ,
    ARM7opTEQSimm     ,
    ARM7opTEQSimm     ,
    ARM7opTEQSimm     ,
    ARM7opTEQSimm     ,
    ARM7opTEQSimm     ,
    ARM7opTEQSimm     ,
    ARM7opTEQSimm     ,
    ARM7opTEQSimm     ,
    ARM7opTEQSimm     ,

    /*0x34*/
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,
    ARM7opUND         ,

    /*0x35*/
    ARM7opCMPSimm     ,
    ARM7opCMPSimm     ,
    ARM7opCMPSimm     ,
    ARM7opCMPSimm     ,
    ARM7opCMPSimm     ,
    ARM7opCMPSimm     ,
    ARM7opCMPSimm     ,
    ARM7opCMPSimm     ,
    ARM7opCMPSimm     ,
    ARM7opCMPSimm     ,
    ARM7opCMPSimm     ,
    ARM7opCMPSimm     ,
    ARM7opCMPSimm     ,
    ARM7opCMPSimm     ,
    ARM7opCMPSimm     ,
    ARM7opCMPSimm     ,

    /*0x36*/
    ARM7opMSRis       ,
    ARM7opMSRis       ,
    ARM7opMSRis       ,
    ARM7opMSRis       ,
    ARM7opMSRis       ,
    ARM7opMSRis       ,
    ARM7opMSRis       ,
    ARM7opMSRis       ,
    ARM7opMSRis       ,
    ARM7opMSRis       ,
    ARM7opMSRis       ,
    ARM7opMSRis       ,
    ARM7opMSRis       ,
    ARM7opMSRis       ,
    ARM7opMSRis       ,
    ARM7opMSRis       ,

    /*0x37*/
    ARM7opCMNSimm     ,
    ARM7opCMNSimm     ,
    ARM7opCMNSimm     ,
    ARM7opCMNSimm     ,
    ARM7opCMNSimm     ,
    ARM7opCMNSimm     ,
    ARM7opCMNSimm     ,
    ARM7opCMNSimm     ,
    ARM7opCMNSimm     ,
    ARM7opCMNSimm     ,
    ARM7opCMNSimm     ,
    ARM7opCMNSimm     ,
    ARM7opCMNSimm     ,
    ARM7opCMNSimm     ,
    ARM7opCMNSimm     ,
    ARM7opCMNSimm     ,

    /*0x38*/
    ARM7opORRimm      ,
    ARM7opORRimm      ,
    ARM7opORRimm      ,
    ARM7opORRimm      ,
    ARM7opORRimm      ,
    ARM7opORRimm      ,
    ARM7opORRimm      ,
    ARM7opORRimm      ,
    ARM7opORRimm      ,
    ARM7opORRimm      ,
    ARM7opORRimm      ,
    ARM7opORRimm      ,
    ARM7opORRimm      ,
    ARM7opORRimm      ,
    ARM7opORRimm      ,
    ARM7opORRimm      ,

    /*0x39*/
    ARM7opORRSimm     ,
    ARM7opORRSimm     ,
    ARM7opORRSimm     ,
    ARM7opORRSimm     ,
    ARM7opORRSimm     ,
    ARM7opORRSimm     ,
    ARM7opORRSimm     ,
    ARM7opORRSimm     ,
    ARM7opORRSimm     ,
    ARM7opORRSimm     ,
    ARM7opORRSimm     ,
    ARM7opORRSimm     ,
    ARM7opORRSimm     ,
    ARM7opORRSimm     ,
    ARM7opORRSimm     ,
    ARM7opORRSimm     ,

    /*0x3A*/
    ARM7opMOVimm      ,
    ARM7opMOVimm      ,
    ARM7opMOVimm      ,
    ARM7opMOVimm      ,
    ARM7opMOVimm      ,
    ARM7opMOVimm      ,
    ARM7opMOVimm      ,
    ARM7opMOVimm      ,
    ARM7opMOVimm      ,
    ARM7opMOVimm      ,
    ARM7opMOVimm      ,
    ARM7opMOVimm      ,
    ARM7opMOVimm      ,
    ARM7opMOVimm      ,
    ARM7opMOVimm      ,
    ARM7opMOVimm      ,

    /*0x3B*/
    ARM7opMOVSimm     ,
    ARM7opMOVSimm     ,
    ARM7opMOVSimm     ,
    ARM7opMOVSimm     ,
    ARM7opMOVSimm     ,
    ARM7opMOVSimm     ,
    ARM7opMOVSimm     ,
    ARM7opMOVSimm     ,
    ARM7opMOVSimm     ,
    ARM7opMOVSimm     ,
    ARM7opMOVSimm     ,
    ARM7opMOVSimm     ,
    ARM7opMOVSimm     ,
    ARM7opMOVSimm     ,
    ARM7opMOVSimm     ,
    ARM7opMOVSimm     ,

    /*0x3C*/
    ARM7opBICimm      ,
    ARM7opBICimm      ,
    ARM7opBICimm      ,
    ARM7opBICimm      ,
    ARM7opBICimm      ,
    ARM7opBICimm      ,
    ARM7opBICimm      ,
    ARM7opBICimm      ,
    ARM7opBICimm      ,
    ARM7opBICimm      ,
    ARM7opBICimm      ,
    ARM7opBICimm      ,
    ARM7opBICimm      ,
    ARM7opBICimm      ,
    ARM7opBICimm      ,
    ARM7opBICimm      ,

    /*0x3D*/
    ARM7opBICSimm     ,
    ARM7opBICSimm     ,
    ARM7opBICSimm     ,
    ARM7opBICSimm     ,
    ARM7opBICSimm     ,
    ARM7opBICSimm     ,
    ARM7opBICSimm     ,
    ARM7opBICSimm     ,
    ARM7opBICSimm     ,
    ARM7opBICSimm     ,
    ARM7opBICSimm     ,
    ARM7opBICSimm     ,
    ARM7opBICSimm     ,
    ARM7opBICSimm     ,
    ARM7opBICSimm     ,
    ARM7opBICSimm     ,

    /*0x3E*/
    ARM7opMVNimm      ,
    ARM7opMVNimm      ,
    ARM7opMVNimm      ,
    ARM7opMVNimm      ,
    ARM7opMVNimm      ,
    ARM7opMVNimm      ,
    ARM7opMVNimm      ,
    ARM7opMVNimm      ,
    ARM7opMVNimm      ,
    ARM7opMVNimm      ,
    ARM7opMVNimm      ,
    ARM7opMVNimm      ,
    ARM7opMVNimm      ,
    ARM7opMVNimm      ,
    ARM7opMVNimm      ,
    ARM7opMVNimm      ,

    /*0x3F*/
    ARM7opMVNSimm     ,
    ARM7opMVNSimm     ,
    ARM7opMVNSimm     ,
    ARM7opMVNSimm     ,
    ARM7opMVNSimm     ,
    ARM7opMVNSimm     ,
    ARM7opMVNSimm     ,
    ARM7opMVNSimm     ,
    ARM7opMVNSimm     ,
    ARM7opMVNSimm     ,
    ARM7opMVNSimm     ,
    ARM7opMVNSimm     ,
    ARM7opMVNSimm     ,
    ARM7opMVNSimm     ,
    ARM7opMVNSimm     ,
    ARM7opMVNSimm     ,

    /*0x40*/
    ARM7opSTRptim     ,
    ARM7opSTRptim     ,
    ARM7opSTRptim     ,
    ARM7opSTRptim     ,
    ARM7opSTRptim     ,
    ARM7opSTRptim     ,
    ARM7opSTRptim     ,
    ARM7opSTRptim     ,
    ARM7opSTRptim     ,
    ARM7opSTRptim     ,
    ARM7opSTRptim     ,
    ARM7opSTRptim     ,
    ARM7opSTRptim     ,
    ARM7opSTRptim     ,
    ARM7opSTRptim     ,
    ARM7opSTRptim     ,

    /*0x41*/
    ARM7opLDRptim     ,
    ARM7opLDRptim     ,
    ARM7opLDRptim     ,
    ARM7opLDRptim     ,
    ARM7opLDRptim     ,
    ARM7opLDRptim     ,
    ARM7opLDRptim     ,
    ARM7opLDRptim     ,
    ARM7opLDRptim     ,
    ARM7opLDRptim     ,
    ARM7opLDRptim     ,
    ARM7opLDRptim     ,
    ARM7opLDRptim     ,
    ARM7opLDRptim     ,
    ARM7opLDRptim     ,
    ARM7opLDRptim     ,

    /*0x42*/
    ARM7opSTRTptim    ,
    ARM7opSTRTptim    ,
    ARM7opSTRTptim    ,
    ARM7opSTRTptim    ,
    ARM7opSTRTptim    ,
    ARM7opSTRTptim    ,
    ARM7opSTRTptim    ,
    ARM7opSTRTptim    ,
    ARM7opSTRTptim    ,
    ARM7opSTRTptim    ,
    ARM7opSTRTptim    ,
    ARM7opSTRTptim    ,
    ARM7opSTRTptim    ,
    ARM7opSTRTptim    ,
    ARM7opSTRTptim    ,
    ARM7opSTRTptim    ,

    /*0x43*/
    ARM7opLDRTptim    ,
    ARM7opLDRTptim    ,
    ARM7opLDRTptim    ,
    ARM7opLDRTptim    ,
    ARM7opLDRTptim    ,
    ARM7opLDRTptim    ,
    ARM7opLDRTptim    ,
    ARM7opLDRTptim    ,
    ARM7opLDRTptim    ,
    ARM7opLDRTptim    ,
    ARM7opLDRTptim    ,
    ARM7opLDRTptim    ,
    ARM7opLDRTptim    ,
    ARM7opLDRTptim    ,
    ARM7opLDRTptim    ,
    ARM7opLDRTptim    ,

    /*0x44*/
    ARM7opSTRBptim    ,
    ARM7opSTRBptim    ,
    ARM7opSTRBptim    ,
    ARM7opSTRBptim    ,
    ARM7opSTRBptim    ,
    ARM7opSTRBptim    ,
    ARM7opSTRBptim    ,
    ARM7opSTRBptim    ,
    ARM7opSTRBptim    ,
    ARM7opSTRBptim    ,
    ARM7opSTRBptim    ,
    ARM7opSTRBptim    ,
    ARM7opSTRBptim    ,
    ARM7opSTRBptim    ,
    ARM7opSTRBptim    ,
    ARM7opSTRBptim    ,

    /*0x45*/
    ARM7opLDRBptim    ,
    ARM7opLDRBptim    ,
    ARM7opLDRBptim    ,
    ARM7opLDRBptim    ,
    ARM7opLDRBptim    ,
    ARM7opLDRBptim    ,
    ARM7opLDRBptim    ,
    ARM7opLDRBptim    ,
    ARM7opLDRBptim    ,
    ARM7opLDRBptim    ,
    ARM7opLDRBptim    ,
    ARM7opLDRBptim    ,
    ARM7opLDRBptim    ,
    ARM7opLDRBptim    ,
    ARM7opLDRBptim    ,
    ARM7opLDRBptim    ,

    /*0x46*/
    ARM7opSTRBTptim   ,
    ARM7opSTRBTptim   ,
    ARM7opSTRBTptim   ,
    ARM7opSTRBTptim   ,
    ARM7opSTRBTptim   ,
    ARM7opSTRBTptim   ,
    ARM7opSTRBTptim   ,
    ARM7opSTRBTptim   ,
    ARM7opSTRBTptim   ,
    ARM7opSTRBTptim   ,
    ARM7opSTRBTptim   ,
    ARM7opSTRBTptim   ,
    ARM7opSTRBTptim   ,
    ARM7opSTRBTptim   ,
    ARM7opSTRBTptim   ,
    ARM7opSTRBTptim   ,

    /*0x47*/
    ARM7opLDRBTptim   ,
    ARM7opLDRBTptim   ,
    ARM7opLDRBTptim   ,
    ARM7opLDRBTptim   ,
    ARM7opLDRBTptim   ,
    ARM7opLDRBTptim   ,
    ARM7opLDRBTptim   ,
    ARM7opLDRBTptim   ,
    ARM7opLDRBTptim   ,
    ARM7opLDRBTptim   ,
    ARM7opLDRBTptim   ,
    ARM7opLDRBTptim   ,
    ARM7opLDRBTptim   ,
    ARM7opLDRBTptim   ,
    ARM7opLDRBTptim   ,
    ARM7opLDRBTptim   ,

    /*0x48*/
    ARM7opSTRptip     ,
    ARM7opSTRptip     ,
    ARM7opSTRptip     ,
    ARM7opSTRptip     ,
    ARM7opSTRptip     ,
    ARM7opSTRptip     ,
    ARM7opSTRptip     ,
    ARM7opSTRptip     ,
    ARM7opSTRptip     ,
    ARM7opSTRptip     ,
    ARM7opSTRptip     ,
    ARM7opSTRptip     ,
    ARM7opSTRptip     ,
    ARM7opSTRptip     ,
    ARM7opSTRptip     ,
    ARM7opSTRptip     ,

    /*0x49*/
    ARM7opLDRptip     ,
    ARM7opLDRptip     ,
    ARM7opLDRptip     ,
    ARM7opLDRptip     ,
    ARM7opLDRptip     ,
    ARM7opLDRptip     ,
    ARM7opLDRptip     ,
    ARM7opLDRptip     ,
    ARM7opLDRptip     ,
    ARM7opLDRptip     ,
    ARM7opLDRptip     ,
    ARM7opLDRptip     ,
    ARM7opLDRptip     ,
    ARM7opLDRptip     ,
    ARM7opLDRptip     ,
    ARM7opLDRptip     ,

    /*0x4A*/
    ARM7opSTRTptip    ,
    ARM7opSTRTptip    ,
    ARM7opSTRTptip    ,
    ARM7opSTRTptip    ,
    ARM7opSTRTptip    ,
    ARM7opSTRTptip    ,
    ARM7opSTRTptip    ,
    ARM7opSTRTptip    ,
    ARM7opSTRTptip    ,
    ARM7opSTRTptip    ,
    ARM7opSTRTptip    ,
    ARM7opSTRTptip    ,
    ARM7opSTRTptip    ,
    ARM7opSTRTptip    ,
    ARM7opSTRTptip    ,
    ARM7opSTRTptip    ,

    /*0x4B*/
    ARM7opLDRTptip    ,
    ARM7opLDRTptip    ,
    ARM7opLDRTptip    ,
    ARM7opLDRTptip    ,
    ARM7opLDRTptip    ,
    ARM7opLDRTptip    ,
    ARM7opLDRTptip    ,
    ARM7opLDRTptip    ,
    ARM7opLDRTptip    ,
    ARM7opLDRTptip    ,
    ARM7opLDRTptip    ,
    ARM7opLDRTptip    ,
    ARM7opLDRTptip    ,
    ARM7opLDRTptip    ,
    ARM7opLDRTptip    ,
    ARM7opLDRTptip    ,

    /*0x4C*/
    ARM7opSTRBptip    ,
    ARM7opSTRBptip    ,
    ARM7opSTRBptip    ,
    ARM7opSTRBptip    ,
    ARM7opSTRBptip    ,
    ARM7opSTRBptip    ,
    ARM7opSTRBptip    ,
    ARM7opSTRBptip    ,
    ARM7opSTRBptip    ,
    ARM7opSTRBptip    ,
    ARM7opSTRBptip    ,
    ARM7opSTRBptip    ,
    ARM7opSTRBptip    ,
    ARM7opSTRBptip    ,
    ARM7opSTRBptip    ,
    ARM7opSTRBptip    ,

    /*0x4D*/
    ARM7opLDRBptip    ,
    ARM7opLDRBptip    ,
    ARM7opLDRBptip    ,
    ARM7opLDRBptip    ,
    ARM7opLDRBptip    ,
    ARM7opLDRBptip    ,
    ARM7opLDRBptip    ,
    ARM7opLDRBptip    ,
    ARM7opLDRBptip    ,
    ARM7opLDRBptip    ,
    ARM7opLDRBptip    ,
    ARM7opLDRBptip    ,
    ARM7opLDRBptip    ,
    ARM7opLDRBptip    ,
    ARM7opLDRBptip    ,
    ARM7opLDRBptip    ,

    /*0x4E*/
    ARM7opSTRBTptip   ,
    ARM7opSTRBTptip   ,
    ARM7opSTRBTptip   ,
    ARM7opSTRBTptip   ,
    ARM7opSTRBTptip   ,
    ARM7opSTRBTptip   ,
    ARM7opSTRBTptip   ,
    ARM7opSTRBTptip   ,
    ARM7opSTRBTptip   ,
    ARM7opSTRBTptip   ,
    ARM7opSTRBTptip   ,
    ARM7opSTRBTptip   ,
    ARM7opSTRBTptip   ,
    ARM7opSTRBTptip   ,
    ARM7opSTRBTptip   ,
    ARM7opSTRBTptip   ,

    /*0x4F*/
    ARM7opLDRBTptip   ,
    ARM7opLDRBTptip   ,
    ARM7opLDRBTptip   ,
    ARM7opLDRBTptip   ,
    ARM7opLDRBTptip   ,
    ARM7opLDRBTptip   ,
    ARM7opLDRBTptip   ,
    ARM7opLDRBTptip   ,
    ARM7opLDRBTptip   ,
    ARM7opLDRBTptip   ,
    ARM7opLDRBTptip   ,
    ARM7opLDRBTptip   ,
    ARM7opLDRBTptip   ,
    ARM7opLDRBTptip   ,
    ARM7opLDRBTptip   ,
    ARM7opLDRBTptip   ,

    /*0x50*/
    ARM7opSTRofim     ,
    ARM7opSTRofim     ,
    ARM7opSTRofim     ,
    ARM7opSTRofim     ,
    ARM7opSTRofim     ,
    ARM7opSTRofim     ,
    ARM7opSTRofim     ,
    ARM7opSTRofim     ,
    ARM7opSTRofim     ,
    ARM7opSTRofim     ,
    ARM7opSTRofim     ,
    ARM7opSTRofim     ,
    ARM7opSTRofim     ,
    ARM7opSTRofim     ,
    ARM7opSTRofim     ,
    ARM7opSTRofim     ,

    /*0x51*/
    ARM7opLDRofim     ,
    ARM7opLDRofim     ,
    ARM7opLDRofim     ,
    ARM7opLDRofim     ,
    ARM7opLDRofim     ,
    ARM7opLDRofim     ,
    ARM7opLDRofim     ,
    ARM7opLDRofim     ,
    ARM7opLDRofim     ,
    ARM7opLDRofim     ,
    ARM7opLDRofim     ,
    ARM7opLDRofim     ,
    ARM7opLDRofim     ,
    ARM7opLDRofim     ,
    ARM7opLDRofim     ,
    ARM7opLDRofim     ,

    /*0x52*/
    ARM7opSTRprim     ,
    ARM7opSTRprim     ,
    ARM7opSTRprim     ,
    ARM7opSTRprim     ,
    ARM7opSTRprim     ,
    ARM7opSTRprim     ,
    ARM7opSTRprim     ,
    ARM7opSTRprim     ,
    ARM7opSTRprim     ,
    ARM7opSTRprim     ,
    ARM7opSTRprim     ,
    ARM7opSTRprim     ,
    ARM7opSTRprim     ,
    ARM7opSTRprim     ,
    ARM7opSTRprim     ,
    ARM7opSTRprim     ,

    /*0x53*/
    ARM7opLDRprim     ,
    ARM7opLDRprim     ,
    ARM7opLDRprim     ,
    ARM7opLDRprim     ,
    ARM7opLDRprim     ,
    ARM7opLDRprim     ,
    ARM7opLDRprim     ,
    ARM7opLDRprim     ,
    ARM7opLDRprim     ,
    ARM7opLDRprim     ,
    ARM7opLDRprim     ,
    ARM7opLDRprim     ,
    ARM7opLDRprim     ,
    ARM7opLDRprim     ,
    ARM7opLDRprim     ,
    ARM7opLDRprim     ,

    /*0x54*/
    ARM7opSTRBofim    ,
    ARM7opSTRBofim    ,
    ARM7opSTRBofim    ,
    ARM7opSTRBofim    ,
    ARM7opSTRBofim    ,
    ARM7opSTRBofim    ,
    ARM7opSTRBofim    ,
    ARM7opSTRBofim    ,
    ARM7opSTRBofim    ,
    ARM7opSTRBofim    ,
    ARM7opSTRBofim    ,
    ARM7opSTRBofim    ,
    ARM7opSTRBofim    ,
    ARM7opSTRBofim    ,
    ARM7opSTRBofim    ,
    ARM7opSTRBofim    ,

    /*0x55*/
    ARM7opLDRBofim    ,
    ARM7opLDRBofim    ,
    ARM7opLDRBofim    ,
    ARM7opLDRBofim    ,
    ARM7opLDRBofim    ,
    ARM7opLDRBofim    ,
    ARM7opLDRBofim    ,
    ARM7opLDRBofim    ,
    ARM7opLDRBofim    ,
    ARM7opLDRBofim    ,
    ARM7opLDRBofim    ,
    ARM7opLDRBofim    ,
    ARM7opLDRBofim    ,
    ARM7opLDRBofim    ,
    ARM7opLDRBofim    ,
    ARM7opLDRBofim    ,

    /*0x56*/
    ARM7opSTRBprim    ,
    ARM7opSTRBprim    ,
    ARM7opSTRBprim    ,
    ARM7opSTRBprim    ,
    ARM7opSTRBprim    ,
    ARM7opSTRBprim    ,
    ARM7opSTRBprim    ,
    ARM7opSTRBprim    ,
    ARM7opSTRBprim    ,
    ARM7opSTRBprim    ,
    ARM7opSTRBprim    ,
    ARM7opSTRBprim    ,
    ARM7opSTRBprim    ,
    ARM7opSTRBprim    ,
    ARM7opSTRBprim    ,
    ARM7opSTRBprim    ,

    /*0x57*/
    ARM7opLDRBprim    ,
    ARM7opLDRBprim    ,
    ARM7opLDRBprim    ,
    ARM7opLDRBprim    ,
    ARM7opLDRBprim    ,
    ARM7opLDRBprim    ,
    ARM7opLDRBprim    ,
    ARM7opLDRBprim    ,
    ARM7opLDRBprim    ,
    ARM7opLDRBprim    ,
    ARM7opLDRBprim    ,
    ARM7opLDRBprim    ,
    ARM7opLDRBprim    ,
    ARM7opLDRBprim    ,
    ARM7opLDRBprim    ,
    ARM7opLDRBprim    ,

    /*0x58*/
    ARM7opSTRofip     ,
    ARM7opSTRofip     ,
    ARM7opSTRofip     ,
    ARM7opSTRofip     ,
    ARM7opSTRofip     ,
    ARM7opSTRofip     ,
    ARM7opSTRofip     ,
    ARM7opSTRofip     ,
    ARM7opSTRofip     ,
    ARM7opSTRofip     ,
    ARM7opSTRofip     ,
    ARM7opSTRofip     ,
    ARM7opSTRofip     ,
    ARM7opSTRofip     ,
    ARM7opSTRofip     ,
    ARM7opSTRofip     ,

    /*0x59*/
    ARM7opLDRofip     ,
    ARM7opLDRofip     ,
    ARM7opLDRofip     ,
    ARM7opLDRofip     ,
    ARM7opLDRofip     ,
    ARM7opLDRofip     ,
    ARM7opLDRofip     ,
    ARM7opLDRofip     ,
    ARM7opLDRofip     ,
    ARM7opLDRofip     ,
    ARM7opLDRofip     ,
    ARM7opLDRofip     ,
    ARM7opLDRofip     ,
    ARM7opLDRofip     ,
    ARM7opLDRofip     ,
    ARM7opLDRofip     ,

    /*0x5A*/
    ARM7opSTRprip     ,
    ARM7opSTRprip     ,
    ARM7opSTRprip     ,
    ARM7opSTRprip     ,
    ARM7opSTRprip     ,
    ARM7opSTRprip     ,
    ARM7opSTRprip     ,
    ARM7opSTRprip     ,
    ARM7opSTRprip     ,
    ARM7opSTRprip     ,
    ARM7opSTRprip     ,
    ARM7opSTRprip     ,
    ARM7opSTRprip     ,
    ARM7opSTRprip     ,
    ARM7opSTRprip     ,
    ARM7opSTRprip     ,

    /*0x5B*/
    ARM7opLDRprip     ,
    ARM7opLDRprip     ,
    ARM7opLDRprip     ,
    ARM7opLDRprip     ,
    ARM7opLDRprip     ,
    ARM7opLDRprip     ,
    ARM7opLDRprip     ,
    ARM7opLDRprip     ,
    ARM7opLDRprip     ,
    ARM7opLDRprip     ,
    ARM7opLDRprip     ,
    ARM7opLDRprip     ,
    ARM7opLDRprip     ,
    ARM7opLDRprip     ,
    ARM7opLDRprip     ,
    ARM7opLDRprip     ,

    /*0x5C*/
    ARM7opSTRBofip    ,
    ARM7opSTRBofip    ,
    ARM7opSTRBofip    ,
    ARM7opSTRBofip    ,
    ARM7opSTRBofip    ,
    ARM7opSTRBofip    ,
    ARM7opSTRBofip    ,
    ARM7opSTRBofip    ,
    ARM7opSTRBofip    ,
    ARM7opSTRBofip    ,
    ARM7opSTRBofip    ,
    ARM7opSTRBofip    ,
    ARM7opSTRBofip    ,
    ARM7opSTRBofip    ,
    ARM7opSTRBofip    ,
    ARM7opSTRBofip    ,

    /*0x5D*/
    ARM7opLDRBofip    ,
    ARM7opLDRBofip    ,
    ARM7opLDRBofip    ,
    ARM7opLDRBofip    ,
    ARM7opLDRBofip    ,
    ARM7opLDRBofip    ,
    ARM7opLDRBofip    ,
    ARM7opLDRBofip    ,
    ARM7opLDRBofip    ,
    ARM7opLDRBofip    ,
    ARM7opLDRBofip    ,
    ARM7opLDRBofip    ,
    ARM7opLDRBofip    ,
    ARM7opLDRBofip    ,
    ARM7opLDRBofip    ,
    ARM7opLDRBofip    ,

    /*0x5E*/
    ARM7opSTRBprip    ,
    ARM7opSTRBprip    ,
    ARM7opSTRBprip    ,
    ARM7opSTRBprip    ,
    ARM7opSTRBprip    ,
    ARM7opSTRBprip    ,
    ARM7opSTRBprip    ,
    ARM7opSTRBprip    ,
    ARM7opSTRBprip    ,
    ARM7opSTRBprip    ,
    ARM7opSTRBprip    ,
    ARM7opSTRBprip    ,
    ARM7opSTRBprip    ,
    ARM7opSTRBprip    ,
    ARM7opSTRBprip    ,
    ARM7opSTRBprip    ,

    /*0x5F*/
    ARM7opLDRBprip    ,
    ARM7opLDRBprip    ,
    ARM7opLDRBprip    ,
    ARM7opLDRBprip    ,
    ARM7opLDRBprip    ,
    ARM7opLDRBprip    ,
    ARM7opLDRBprip    ,
    ARM7opLDRBprip    ,
    ARM7opLDRBprip    ,
    ARM7opLDRBprip    ,
    ARM7opLDRBprip    ,
    ARM7opLDRBprip    ,
    ARM7opLDRBprip    ,
    ARM7opLDRBprip    ,
    ARM7opLDRBprip    ,
    ARM7opLDRBprip    ,

    /*0x60*/
    ARM7opSTRptrmll   ,
    ARM7opUND         ,
    ARM7opSTRptrmlr   ,
    ARM7opUND         ,
    ARM7opSTRptrmar   ,
    ARM7opUND         ,
    ARM7opSTRptrmrr   ,
    ARM7opUND         ,
    ARM7opSTRptrmll   ,
    ARM7opUND         ,
    ARM7opSTRptrmlr   ,
    ARM7opUND         ,
    ARM7opSTRptrmar   ,
    ARM7opUND         ,
    ARM7opSTRptrmrr   ,
    ARM7opUND         ,

    /*0x61*/
    ARM7opLDRptrmll   ,
    ARM7opUND         ,
    ARM7opLDRptrmlr   ,
    ARM7opUND         ,
    ARM7opLDRptrmar   ,
    ARM7opUND         ,
    ARM7opLDRptrmrr   ,
    ARM7opUND         ,
    ARM7opLDRptrmll   ,
    ARM7opUND         ,
    ARM7opLDRptrmlr   ,
    ARM7opUND         ,
    ARM7opLDRptrmar   ,
    ARM7opUND         ,
    ARM7opLDRptrmrr   ,
    ARM7opUND         ,

    /*0x62*/
    ARM7opSTRTptrmll  ,
    ARM7opUND         ,
    ARM7opSTRTptrmlr  ,
    ARM7opUND         ,
    ARM7opSTRTptrmar  ,
    ARM7opUND         ,
    ARM7opSTRTptrmrr  ,
    ARM7opUND         ,
    ARM7opSTRTptrmll  ,
    ARM7opUND         ,
    ARM7opSTRTptrmlr  ,
    ARM7opUND         ,
    ARM7opSTRTptrmar  ,
    ARM7opUND         ,
    ARM7opSTRTptrmrr  ,
    ARM7opUND         ,

    /*0x63*/
    ARM7opLDRTptrmll  ,
    ARM7opUND         ,
    ARM7opLDRTptrmlr  ,
    ARM7opUND         ,
    ARM7opLDRTptrmar  ,
    ARM7opUND         ,
    ARM7opLDRTptrmrr  ,
    ARM7opUND         ,
    ARM7opLDRTptrmll  ,
    ARM7opUND         ,
    ARM7opLDRTptrmlr  ,
    ARM7opUND         ,
    ARM7opLDRTptrmar  ,
    ARM7opUND         ,
    ARM7opLDRTptrmrr  ,
    ARM7opUND         ,

    /*0x64*/
    ARM7opSTRBptrmll  ,
    ARM7opUND         ,
    ARM7opSTRBptrmlr  ,
    ARM7opUND         ,
    ARM7opSTRBptrmar  ,
    ARM7opUND         ,
    ARM7opSTRBptrmrr  ,
    ARM7opUND         ,
    ARM7opSTRBptrmll  ,
    ARM7opUND         ,
    ARM7opSTRBptrmlr  ,
    ARM7opUND         ,
    ARM7opSTRBptrmar  ,
    ARM7opUND         ,
    ARM7opSTRBptrmrr  ,
    ARM7opUND         ,

    /*0x65*/
    ARM7opLDRBptrmll  ,
    ARM7opUND         ,
    ARM7opLDRBptrmlr  ,
    ARM7opUND         ,
    ARM7opLDRBptrmar  ,
    ARM7opUND         ,
    ARM7opLDRBptrmrr  ,
    ARM7opUND         ,
    ARM7opLDRBptrmll  ,
    ARM7opUND         ,
    ARM7opLDRBptrmlr  ,
    ARM7opUND         ,
    ARM7opLDRBptrmar  ,
    ARM7opUND         ,
    ARM7opLDRBptrmrr  ,
    ARM7opUND         ,

    /*0x66*/
    ARM7opSTRBTptrmll ,
    ARM7opUND         ,
    ARM7opSTRBTptrmlr ,
    ARM7opUND         ,
    ARM7opSTRBTptrmar ,
    ARM7opUND         ,
    ARM7opSTRBTptrmrr ,
    ARM7opUND         ,
    ARM7opSTRBTptrmll ,
    ARM7opUND         ,
    ARM7opSTRBTptrmlr ,
    ARM7opUND         ,
    ARM7opSTRBTptrmar ,
    ARM7opUND         ,
    ARM7opSTRBTptrmrr ,
    ARM7opUND         ,

    /*0x67*/
    ARM7opLDRBTptrmll ,
    ARM7opUND         ,
    ARM7opLDRBTptrmlr ,
    ARM7opUND         ,
    ARM7opLDRBTptrmar ,
    ARM7opUND         ,
    ARM7opLDRBTptrmrr ,
    ARM7opUND         ,
    ARM7opLDRBTptrmll ,
    ARM7opUND         ,
    ARM7opLDRBTptrmlr ,
    ARM7opUND         ,
    ARM7opLDRBTptrmar ,
    ARM7opUND         ,
    ARM7opLDRBTptrmrr ,
    ARM7opUND         ,

    /*0x68*/
    ARM7opSTRptrpll   ,
    ARM7opUND         ,
    ARM7opSTRptrplr   ,
    ARM7opUND         ,
    ARM7opSTRptrpar   ,
    ARM7opUND         ,
    ARM7opSTRptrprr   ,
    ARM7opUND         ,
    ARM7opSTRptrpll   ,
    ARM7opUND         ,
    ARM7opSTRptrplr   ,
    ARM7opUND         ,
    ARM7opSTRptrpar   ,
    ARM7opUND         ,
    ARM7opSTRptrprr   ,
    ARM7opUND         ,

    /*0x69*/
    ARM7opLDRptrpll   ,
    ARM7opUND         ,
    ARM7opLDRptrplr   ,
    ARM7opUND         ,
    ARM7opLDRptrpar   ,
    ARM7opUND         ,
    ARM7opLDRptrprr   ,
    ARM7opUND         ,
    ARM7opLDRptrpll   ,
    ARM7opUND         ,
    ARM7opLDRptrplr   ,
    ARM7opUND         ,
    ARM7opLDRptrpar   ,
    ARM7opUND         ,
    ARM7opLDRptrprr   ,
    ARM7opUND         ,

    /*0x6A*/
    ARM7opSTRTptrpll  ,
    ARM7opUND         ,
    ARM7opSTRTptrplr  ,
    ARM7opUND         ,
    ARM7opSTRTptrpar  ,
    ARM7opUND         ,
    ARM7opSTRTptrprr  ,
    ARM7opUND         ,
    ARM7opSTRTptrpll  ,
    ARM7opUND         ,
    ARM7opSTRTptrplr  ,
    ARM7opUND         ,
    ARM7opSTRTptrpar  ,
    ARM7opUND         ,
    ARM7opSTRTptrprr  ,
    ARM7opUND         ,

    /*0x6B*/
    ARM7opLDRTptrpll  ,
    ARM7opUND         ,
    ARM7opLDRTptrplr  ,
    ARM7opUND         ,
    ARM7opLDRTptrpar  ,
    ARM7opUND         ,
    ARM7opLDRTptrprr  ,
    ARM7opUND         ,
    ARM7opLDRTptrpll  ,
    ARM7opUND         ,
    ARM7opLDRTptrplr  ,
    ARM7opUND         ,
    ARM7opLDRTptrpar  ,
    ARM7opUND         ,
    ARM7opLDRTptrprr  ,
    ARM7opUND         ,

    /*0x6C*/
    ARM7opSTRBptrpll  ,
    ARM7opUND         ,
    ARM7opSTRBptrplr  ,
    ARM7opUND         ,
    ARM7opSTRBptrpar  ,
    ARM7opUND         ,
    ARM7opSTRBptrprr  ,
    ARM7opUND         ,
    ARM7opSTRBptrpll  ,
    ARM7opUND         ,
    ARM7opSTRBptrplr  ,
    ARM7opUND         ,
    ARM7opSTRBptrpar  ,
    ARM7opUND         ,
    ARM7opSTRBptrprr  ,
    ARM7opUND         ,

    /*0x6D*/
    ARM7opLDRBptrpll  ,
    ARM7opUND         ,
    ARM7opLDRBptrplr  ,
    ARM7opUND         ,
    ARM7opLDRBptrpar  ,
    ARM7opUND         ,
    ARM7opLDRBptrprr  ,
    ARM7opUND         ,
    ARM7opLDRBptrpll  ,
    ARM7opUND         ,
    ARM7opLDRBptrplr  ,
    ARM7opUND         ,
    ARM7opLDRBptrpar  ,
    ARM7opUND         ,
    ARM7opLDRBptrprr  ,
    ARM7opUND         ,

    /*0x6E*/
    ARM7opSTRBTptrpll ,
    ARM7opUND         ,
    ARM7opSTRBTptrplr ,
    ARM7opUND         ,
    ARM7opSTRBTptrpar ,
    ARM7opUND         ,
    ARM7opSTRBTptrprr ,
    ARM7opUND         ,
    ARM7opSTRBTptrpll ,
    ARM7opUND         ,
    ARM7opSTRBTptrplr ,
    ARM7opUND         ,
    ARM7opSTRBTptrpar ,
    ARM7opUND         ,
    ARM7opSTRBTptrprr ,
    ARM7opUND         ,

    /*0x6F*/
    ARM7opLDRBTptrpll ,
    ARM7opUND         ,
    ARM7opLDRBTptrplr ,
    ARM7opUND         ,
    ARM7opLDRBTptrpar ,
    ARM7opUND         ,
    ARM7opLDRBTptrprr ,
    ARM7opUND         ,
    ARM7opLDRBTptrpll ,
    ARM7opUND         ,
    ARM7opLDRBTptrplr ,
    ARM7opUND         ,
    ARM7opLDRBTptrpar ,
    ARM7opUND         ,
    ARM7opLDRBTptrprr ,
    ARM7opUND         ,

    /*0x70*/
    ARM7opSTRofrmll   ,
    ARM7opUND         ,
    ARM7opSTRofrmlr   ,
    ARM7opUND         ,
    ARM7opSTRofrmar   ,
    ARM7opUND         ,
    ARM7opSTRofrmrr   ,
    ARM7opUND         ,
    ARM7opSTRofrmll   ,
    ARM7opUND         ,
    ARM7opSTRofrmlr   ,
    ARM7opUND         ,
    ARM7opSTRofrmar   ,
    ARM7opUND         ,
    ARM7opSTRofrmrr   ,
    ARM7opUND         ,

    /*0x71*/
    ARM7opLDRofrmll   ,
    ARM7opUND         ,
    ARM7opLDRofrmlr   ,
    ARM7opUND         ,
    ARM7opLDRofrmar   ,
    ARM7opUND         ,
    ARM7opLDRofrmrr   ,
    ARM7opUND         ,
    ARM7opLDRofrmll   ,
    ARM7opUND         ,
    ARM7opLDRofrmlr   ,
    ARM7opUND         ,
    ARM7opLDRofrmar   ,
    ARM7opUND         ,
    ARM7opLDRofrmrr   ,
    ARM7opUND         ,

    /*0x72*/
    ARM7opSTRprrmll   ,
    ARM7opUND         ,
    ARM7opSTRprrmlr   ,
    ARM7opUND         ,
    ARM7opSTRprrmar   ,
    ARM7opUND         ,
    ARM7opSTRprrmrr   ,
    ARM7opUND         ,
    ARM7opSTRprrmll   ,
    ARM7opUND         ,
    ARM7opSTRprrmlr   ,
    ARM7opUND         ,
    ARM7opSTRprrmar   ,
    ARM7opUND         ,
    ARM7opSTRprrmrr   ,
    ARM7opUND         ,

    /*0x73*/
    ARM7opLDRprrmll   ,
    ARM7opUND         ,
    ARM7opLDRprrmlr   ,
    ARM7opUND         ,
    ARM7opLDRprrmar   ,
    ARM7opUND         ,
    ARM7opLDRprrmrr   ,
    ARM7opUND         ,
    ARM7opLDRprrmll   ,
    ARM7opUND         ,
    ARM7opLDRprrmlr   ,
    ARM7opUND         ,
    ARM7opLDRprrmar   ,
    ARM7opUND         ,
    ARM7opLDRprrmrr   ,
    ARM7opUND         ,

    /*0x74*/
    ARM7opSTRBofrmll  ,
    ARM7opUND         ,
    ARM7opSTRBofrmlr  ,
    ARM7opUND         ,
    ARM7opSTRBofrmar  ,
    ARM7opUND         ,
    ARM7opSTRBofrmrr  ,
    ARM7opUND         ,
    ARM7opSTRBofrmll  ,
    ARM7opUND         ,
    ARM7opSTRBofrmlr  ,
    ARM7opUND         ,
    ARM7opSTRBofrmar  ,
    ARM7opUND         ,
    ARM7opSTRBofrmrr  ,
    ARM7opUND         ,

    /*0x75*/
    ARM7opLDRBofrmll  ,
    ARM7opUND         ,
    ARM7opLDRBofrmlr  ,
    ARM7opUND         ,
    ARM7opLDRBofrmar  ,
    ARM7opUND         ,
    ARM7opLDRBofrmrr  ,
    ARM7opUND         ,
    ARM7opLDRBofrmll  ,
    ARM7opUND         ,
    ARM7opLDRBofrmlr  ,
    ARM7opUND         ,
    ARM7opLDRBofrmar  ,
    ARM7opUND         ,
    ARM7opLDRBofrmrr  ,
    ARM7opUND         ,

    /*0x76*/
    ARM7opSTRBprrmll  ,
    ARM7opUND         ,
    ARM7opSTRBprrmlr  ,
    ARM7opUND         ,
    ARM7opSTRBprrmar  ,
    ARM7opUND         ,
    ARM7opSTRBprrmrr  ,
    ARM7opUND         ,
    ARM7opSTRBprrmll  ,
    ARM7opUND         ,
    ARM7opSTRBprrmlr  ,
    ARM7opUND         ,
    ARM7opSTRBprrmar  ,
    ARM7opUND         ,
    ARM7opSTRBprrmrr  ,
    ARM7opUND         ,

    /*0x77*/
    ARM7opLDRBprrmll  ,
    ARM7opUND         ,
    ARM7opLDRBprrmlr  ,
    ARM7opUND         ,
    ARM7opLDRBprrmar  ,
    ARM7opUND         ,
    ARM7opLDRBprrmrr  ,
    ARM7opUND         ,
    ARM7opLDRBprrmll  ,
    ARM7opUND         ,
    ARM7opLDRBprrmlr  ,
    ARM7opUND         ,
    ARM7opLDRBprrmar  ,
    ARM7opUND         ,
    ARM7opLDRBprrmrr  ,
    ARM7opUND         ,

    /*0x78*/
    ARM7opSTRofrpll   ,
    ARM7opUND         ,
    ARM7opSTRofrplr   ,
    ARM7opUND         ,
    ARM7opSTRofrpar   ,
    ARM7opUND         ,
    ARM7opSTRofrprr   ,
    ARM7opUND         ,
    ARM7opSTRofrpll   ,
    ARM7opUND         ,
    ARM7opSTRofrplr   ,
    ARM7opUND         ,
    ARM7opSTRofrpar   ,
    ARM7opUND         ,
    ARM7opSTRofrprr   ,
    ARM7opUND         ,

    /*0x79*/
    ARM7opLDRofrpll   ,
    ARM7opUND         ,
    ARM7opLDRofrplr   ,
    ARM7opUND         ,
    ARM7opLDRofrpar   ,
    ARM7opUND         ,
    ARM7opLDRofrprr   ,
    ARM7opUND         ,
    ARM7opLDRofrpll   ,
    ARM7opUND         ,
    ARM7opLDRofrplr   ,
    ARM7opUND         ,
    ARM7opLDRofrpar   ,
    ARM7opUND         ,
    ARM7opLDRofrprr   ,
    ARM7opUND         ,

    /*0x7A*/
    ARM7opSTRprrpll   ,
    ARM7opUND         ,
    ARM7opSTRprrplr   ,
    ARM7opUND         ,
    ARM7opSTRprrpar   ,
    ARM7opUND         ,
    ARM7opSTRprrprr   ,
    ARM7opUND         ,
    ARM7opSTRprrpll   ,
    ARM7opUND         ,
    ARM7opSTRprrplr   ,
    ARM7opUND         ,
    ARM7opSTRprrpar   ,
    ARM7opUND         ,
    ARM7opSTRprrprr   ,
    ARM7opUND         ,

    /*0x7B*/
    ARM7opLDRprrpll   ,
    ARM7opUND         ,
    ARM7opLDRprrplr   ,
    ARM7opUND         ,
    ARM7opLDRprrpar   ,
    ARM7opUND         ,
    ARM7opLDRprrprr   ,
    ARM7opUND         ,
    ARM7opLDRprrpll   ,
    ARM7opUND         ,
    ARM7opLDRprrplr   ,
    ARM7opUND         ,
    ARM7opLDRprrpar   ,
    ARM7opUND         ,
    ARM7opLDRprrprr   ,
    ARM7opUND         ,

    /*0x7C*/
    ARM7opSTRBofrpll  ,
    ARM7opUND         ,
    ARM7opSTRBofrplr  ,
    ARM7opUND         ,
    ARM7opSTRBofrpar  ,
    ARM7opUND         ,
    ARM7opSTRBofrprr  ,
    ARM7opUND         ,
    ARM7opSTRBofrpll  ,
    ARM7opUND         ,
    ARM7opSTRBofrplr  ,
    ARM7opUND         ,
    ARM7opSTRBofrpar  ,
    ARM7opUND         ,
    ARM7opSTRBofrprr  ,
    ARM7opUND         ,

    /*0x7D*/
    ARM7opLDRBofrpll  ,
    ARM7opUND         ,
    ARM7opLDRBofrplr  ,
    ARM7opUND         ,
    ARM7opLDRBofrpar  ,
    ARM7opUND         ,
    ARM7opLDRBofrprr  ,
    ARM7opUND         ,
    ARM7opLDRBofrpll  ,
    ARM7opUND         ,
    ARM7opLDRBofrplr  ,
    ARM7opUND         ,
    ARM7opLDRBofrpar  ,
    ARM7opUND         ,
    ARM7opLDRBofrprr  ,
    ARM7opUND         ,

    /*0x7E*/
    ARM7opSTRBprrpll  ,
    ARM7opUND         ,
    ARM7opSTRBprrplr  ,
    ARM7opUND         ,
    ARM7opSTRBprrpar  ,
    ARM7opUND         ,
    ARM7opSTRBprrprr  ,
    ARM7opUND         ,
    ARM7opSTRBprrpll  ,
    ARM7opUND         ,
    ARM7opSTRBprrplr  ,
    ARM7opUND         ,
    ARM7opSTRBprrpar  ,
    ARM7opUND         ,
    ARM7opSTRBprrprr  ,
    ARM7opUND         ,

    /*0x7F*/
    ARM7opLDRBprrpll  ,
    ARM7opUND         ,
    ARM7opLDRBprrplr  ,
    ARM7opUND         ,
    ARM7opLDRBprrpar  ,
    ARM7opUND         ,
    ARM7opLDRBprrprr  ,
    ARM7opUND         ,
    ARM7opLDRBprrpll  ,
    ARM7opUND         ,
    ARM7opLDRBprrplr  ,
    ARM7opUND         ,
    ARM7opLDRBprrpar  ,
    ARM7opUND         ,
    ARM7opLDRBprrprr  ,
    ARM7opUND         ,

    /*0x80*/
    ARM7opSTMDA       ,
    ARM7opSTMDA       ,
    ARM7opSTMDA       ,
    ARM7opSTMDA       ,
    ARM7opSTMDA       ,
    ARM7opSTMDA       ,
    ARM7opSTMDA       ,
    ARM7opSTMDA       ,
    ARM7opSTMDA       ,
    ARM7opSTMDA       ,
    ARM7opSTMDA       ,
    ARM7opSTMDA       ,
    ARM7opSTMDA       ,
    ARM7opSTMDA       ,
    ARM7opSTMDA       ,
    ARM7opSTMDA       ,

    /*0x81*/
    ARM7opLDMDA       ,
    ARM7opLDMDA       ,
    ARM7opLDMDA       ,
    ARM7opLDMDA       ,
    ARM7opLDMDA       ,
    ARM7opLDMDA       ,
    ARM7opLDMDA       ,
    ARM7opLDMDA       ,
    ARM7opLDMDA       ,
    ARM7opLDMDA       ,
    ARM7opLDMDA       ,
    ARM7opLDMDA       ,
    ARM7opLDMDA       ,
    ARM7opLDMDA       ,
    ARM7opLDMDA       ,
    ARM7opLDMDA       ,

    /*0x82*/
    ARM7opSTMDAw      ,
    ARM7opSTMDAw      ,
    ARM7opSTMDAw      ,
    ARM7opSTMDAw      ,
    ARM7opSTMDAw      ,
    ARM7opSTMDAw      ,
    ARM7opSTMDAw      ,
    ARM7opSTMDAw      ,
    ARM7opSTMDAw      ,
    ARM7opSTMDAw      ,
    ARM7opSTMDAw      ,
    ARM7opSTMDAw      ,
    ARM7opSTMDAw      ,
    ARM7opSTMDAw      ,
    ARM7opSTMDAw      ,
    ARM7opSTMDAw      ,

    /*0x83*/
    ARM7opLDMDAw      ,
    ARM7opLDMDAw      ,
    ARM7opLDMDAw      ,
    ARM7opLDMDAw      ,
    ARM7opLDMDAw      ,
    ARM7opLDMDAw      ,
    ARM7opLDMDAw      ,
    ARM7opLDMDAw      ,
    ARM7opLDMDAw      ,
    ARM7opLDMDAw      ,
    ARM7opLDMDAw      ,
    ARM7opLDMDAw      ,
    ARM7opLDMDAw      ,
    ARM7opLDMDAw      ,
    ARM7opLDMDAw      ,
    ARM7opLDMDAw      ,

    /*0x84*/
    ARM7opSTMDAu      ,
    ARM7opSTMDAu      ,
    ARM7opSTMDAu      ,
    ARM7opSTMDAu      ,
    ARM7opSTMDAu      ,
    ARM7opSTMDAu      ,
    ARM7opSTMDAu      ,
    ARM7opSTMDAu      ,
    ARM7opSTMDAu      ,
    ARM7opSTMDAu      ,
    ARM7opSTMDAu      ,
    ARM7opSTMDAu      ,
    ARM7opSTMDAu      ,
    ARM7opSTMDAu      ,
    ARM7opSTMDAu      ,
    ARM7opSTMDAu      ,

    /*0x85*/
    ARM7opLDMDAu      ,
    ARM7opLDMDAu      ,
    ARM7opLDMDAu      ,
    ARM7opLDMDAu      ,
    ARM7opLDMDAu      ,
    ARM7opLDMDAu      ,
    ARM7opLDMDAu      ,
    ARM7opLDMDAu      ,
    ARM7opLDMDAu      ,
    ARM7opLDMDAu      ,
    ARM7opLDMDAu      ,
    ARM7opLDMDAu      ,
    ARM7opLDMDAu      ,
    ARM7opLDMDAu      ,
    ARM7opLDMDAu      ,
    ARM7opLDMDAu      ,

    /*0x86*/
    ARM7opSTMDAuw     ,
    ARM7opSTMDAuw     ,
    ARM7opSTMDAuw     ,
    ARM7opSTMDAuw     ,
    ARM7opSTMDAuw     ,
    ARM7opSTMDAuw     ,
    ARM7opSTMDAuw     ,
    ARM7opSTMDAuw     ,
    ARM7opSTMDAuw     ,
    ARM7opSTMDAuw     ,
    ARM7opSTMDAuw     ,
    ARM7opSTMDAuw     ,
    ARM7opSTMDAuw     ,
    ARM7opSTMDAuw     ,
    ARM7opSTMDAuw     ,
    ARM7opSTMDAuw     ,

    /*0x87*/
    ARM7opLDMDAuw     ,
    ARM7opLDMDAuw     ,
    ARM7opLDMDAuw     ,
    ARM7opLDMDAuw     ,
    ARM7opLDMDAuw     ,
    ARM7opLDMDAuw     ,
    ARM7opLDMDAuw     ,
    ARM7opLDMDAuw     ,
    ARM7opLDMDAuw     ,
    ARM7opLDMDAuw     ,
    ARM7opLDMDAuw     ,
    ARM7opLDMDAuw     ,
    ARM7opLDMDAuw     ,
    ARM7opLDMDAuw     ,
    ARM7opLDMDAuw     ,
    ARM7opLDMDAuw     ,

    /*0x88*/
    ARM7opSTMIA       ,
    ARM7opSTMIA       ,
    ARM7opSTMIA       ,
    ARM7opSTMIA       ,
    ARM7opSTMIA       ,
    ARM7opSTMIA       ,
    ARM7opSTMIA       ,
    ARM7opSTMIA       ,
    ARM7opSTMIA       ,
    ARM7opSTMIA       ,
    ARM7opSTMIA       ,
    ARM7opSTMIA       ,
    ARM7opSTMIA       ,
    ARM7opSTMIA       ,
    ARM7opSTMIA       ,
    ARM7opSTMIA       ,

    /*0x89*/
    ARM7opLDMIA       ,
    ARM7opLDMIA       ,
    ARM7opLDMIA       ,
    ARM7opLDMIA       ,
    ARM7opLDMIA       ,
    ARM7opLDMIA       ,
    ARM7opLDMIA       ,
    ARM7opLDMIA       ,
    ARM7opLDMIA       ,
    ARM7opLDMIA       ,
    ARM7opLDMIA       ,
    ARM7opLDMIA       ,
    ARM7opLDMIA       ,
    ARM7opLDMIA       ,
    ARM7opLDMIA       ,
    ARM7opLDMIA       ,

    /*0x8A*/
    ARM7opSTMIAw      ,
    ARM7opSTMIAw      ,
    ARM7opSTMIAw      ,
    ARM7opSTMIAw      ,
    ARM7opSTMIAw      ,
    ARM7opSTMIAw      ,
    ARM7opSTMIAw      ,
    ARM7opSTMIAw      ,
    ARM7opSTMIAw      ,
    ARM7opSTMIAw      ,
    ARM7opSTMIAw      ,
    ARM7opSTMIAw      ,
    ARM7opSTMIAw      ,
    ARM7opSTMIAw      ,
    ARM7opSTMIAw      ,
    ARM7opSTMIAw      ,

    /*0x8B*/
    ARM7opLDMIAw      ,
    ARM7opLDMIAw      ,
    ARM7opLDMIAw      ,
    ARM7opLDMIAw      ,
    ARM7opLDMIAw      ,
    ARM7opLDMIAw      ,
    ARM7opLDMIAw      ,
    ARM7opLDMIAw      ,
    ARM7opLDMIAw      ,
    ARM7opLDMIAw      ,
    ARM7opLDMIAw      ,
    ARM7opLDMIAw      ,
    ARM7opLDMIAw      ,
    ARM7opLDMIAw      ,
    ARM7opLDMIAw      ,
    ARM7opLDMIAw      ,

    /*0x8C*/
    ARM7opSTMIAu      ,
    ARM7opSTMIAu      ,
    ARM7opSTMIAu      ,
    ARM7opSTMIAu      ,
    ARM7opSTMIAu      ,
    ARM7opSTMIAu      ,
    ARM7opSTMIAu      ,
    ARM7opSTMIAu      ,
    ARM7opSTMIAu      ,
    ARM7opSTMIAu      ,
    ARM7opSTMIAu      ,
    ARM7opSTMIAu      ,
    ARM7opSTMIAu      ,
    ARM7opSTMIAu      ,
    ARM7opSTMIAu      ,
    ARM7opSTMIAu      ,

    /*0x8D*/
    ARM7opLDMIAu      ,
    ARM7opLDMIAu      ,
    ARM7opLDMIAu      ,
    ARM7opLDMIAu      ,
    ARM7opLDMIAu      ,
    ARM7opLDMIAu      ,
    ARM7opLDMIAu      ,
    ARM7opLDMIAu      ,
    ARM7opLDMIAu      ,
    ARM7opLDMIAu      ,
    ARM7opLDMIAu      ,
    ARM7opLDMIAu      ,
    ARM7opLDMIAu      ,
    ARM7opLDMIAu      ,
    ARM7opLDMIAu      ,
    ARM7opLDMIAu      ,

    /*0x8E*/
    ARM7opSTMIAuw     ,
    ARM7opSTMIAuw     ,
    ARM7opSTMIAuw     ,
    ARM7opSTMIAuw     ,
    ARM7opSTMIAuw     ,
    ARM7opSTMIAuw     ,
    ARM7opSTMIAuw     ,
    ARM7opSTMIAuw     ,
    ARM7opSTMIAuw     ,
    ARM7opSTMIAuw     ,
    ARM7opSTMIAuw     ,
    ARM7opSTMIAuw     ,
    ARM7opSTMIAuw     ,
    ARM7opSTMIAuw     ,
    ARM7opSTMIAuw     ,
    ARM7opSTMIAuw     ,

    /*0x8F*/
    ARM7opLDMIAuw     ,
    ARM7opLDMIAuw     ,
    ARM7opLDMIAuw     ,
    ARM7opLDMIAuw     ,
    ARM7opLDMIAuw     ,
    ARM7opLDMIAuw     ,
    ARM7opLDMIAuw     ,
    ARM7opLDMIAuw     ,
    ARM7opLDMIAuw     ,
    ARM7opLDMIAuw     ,
    ARM7opLDMIAuw     ,
    ARM7opLDMIAuw     ,
    ARM7opLDMIAuw     ,
    ARM7opLDMIAuw     ,
    ARM7opLDMIAuw     ,
    ARM7opLDMIAuw     ,

    /*0x90*/
    ARM7opSTMDB       ,
    ARM7opSTMDB       ,
    ARM7opSTMDB       ,
    ARM7opSTMDB       ,
    ARM7opSTMDB       ,
    ARM7opSTMDB       ,
    ARM7opSTMDB       ,
    ARM7opSTMDB       ,
    ARM7opSTMDB       ,
    ARM7opSTMDB       ,
    ARM7opSTMDB       ,
    ARM7opSTMDB       ,
    ARM7opSTMDB       ,
    ARM7opSTMDB       ,
    ARM7opSTMDB       ,
    ARM7opSTMDB       ,

    /*0x91*/
    ARM7opLDMDB       ,
    ARM7opLDMDB       ,
    ARM7opLDMDB       ,
    ARM7opLDMDB       ,
    ARM7opLDMDB       ,
    ARM7opLDMDB       ,
    ARM7opLDMDB       ,
    ARM7opLDMDB       ,
    ARM7opLDMDB       ,
    ARM7opLDMDB       ,
    ARM7opLDMDB       ,
    ARM7opLDMDB       ,
    ARM7opLDMDB       ,
    ARM7opLDMDB       ,
    ARM7opLDMDB       ,
    ARM7opLDMDB       ,

    /*0x92*/
    ARM7opSTMDBw      ,
    ARM7opSTMDBw      ,
    ARM7opSTMDBw      ,
    ARM7opSTMDBw      ,
    ARM7opSTMDBw      ,
    ARM7opSTMDBw      ,
    ARM7opSTMDBw      ,
    ARM7opSTMDBw      ,
    ARM7opSTMDBw      ,
    ARM7opSTMDBw      ,
    ARM7opSTMDBw      ,
    ARM7opSTMDBw      ,
    ARM7opSTMDBw      ,
    ARM7opSTMDBw      ,
    ARM7opSTMDBw      ,
    ARM7opSTMDBw      ,

    /*0x93*/
    ARM7opLDMDBw      ,
    ARM7opLDMDBw      ,
    ARM7opLDMDBw      ,
    ARM7opLDMDBw      ,
    ARM7opLDMDBw      ,
    ARM7opLDMDBw      ,
    ARM7opLDMDBw      ,
    ARM7opLDMDBw      ,
    ARM7opLDMDBw      ,
    ARM7opLDMDBw      ,
    ARM7opLDMDBw      ,
    ARM7opLDMDBw      ,
    ARM7opLDMDBw      ,
    ARM7opLDMDBw      ,
    ARM7opLDMDBw      ,
    ARM7opLDMDBw      ,

    /*0x94*/
    ARM7opSTMDBu      ,
    ARM7opSTMDBu      ,
    ARM7opSTMDBu      ,
    ARM7opSTMDBu      ,
    ARM7opSTMDBu      ,
    ARM7opSTMDBu      ,
    ARM7opSTMDBu      ,
    ARM7opSTMDBu      ,
    ARM7opSTMDBu      ,
    ARM7opSTMDBu      ,
    ARM7opSTMDBu      ,
    ARM7opSTMDBu      ,
    ARM7opSTMDBu      ,
    ARM7opSTMDBu      ,
    ARM7opSTMDBu      ,
    ARM7opSTMDBu      ,

    /*0x95*/
    ARM7opLDMDBu      ,
    ARM7opLDMDBu      ,
    ARM7opLDMDBu      ,
    ARM7opLDMDBu      ,
    ARM7opLDMDBu      ,
    ARM7opLDMDBu      ,
    ARM7opLDMDBu      ,
    ARM7opLDMDBu      ,
    ARM7opLDMDBu      ,
    ARM7opLDMDBu      ,
    ARM7opLDMDBu      ,
    ARM7opLDMDBu      ,
    ARM7opLDMDBu      ,
    ARM7opLDMDBu      ,
    ARM7opLDMDBu      ,
    ARM7opLDMDBu      ,

    /*0x96*/
    ARM7opSTMDBuw     ,
    ARM7opSTMDBuw     ,
    ARM7opSTMDBuw     ,
    ARM7opSTMDBuw     ,
    ARM7opSTMDBuw     ,
    ARM7opSTMDBuw     ,
    ARM7opSTMDBuw     ,
    ARM7opSTMDBuw     ,
    ARM7opSTMDBuw     ,
    ARM7opSTMDBuw     ,
    ARM7opSTMDBuw     ,
    ARM7opSTMDBuw     ,
    ARM7opSTMDBuw     ,
    ARM7opSTMDBuw     ,
    ARM7opSTMDBuw     ,
    ARM7opSTMDBuw     ,

    /*0x97*/
    ARM7opLDMDBuw     ,
    ARM7opLDMDBuw     ,
    ARM7opLDMDBuw     ,
    ARM7opLDMDBuw     ,
    ARM7opLDMDBuw     ,
    ARM7opLDMDBuw     ,
    ARM7opLDMDBuw     ,
    ARM7opLDMDBuw     ,
    ARM7opLDMDBuw     ,
    ARM7opLDMDBuw     ,
    ARM7opLDMDBuw     ,
    ARM7opLDMDBuw     ,
    ARM7opLDMDBuw     ,
    ARM7opLDMDBuw     ,
    ARM7opLDMDBuw     ,
    ARM7opLDMDBuw     ,

    /*0x98*/
    ARM7opSTMIB       ,
    ARM7opSTMIB       ,
    ARM7opSTMIB       ,
    ARM7opSTMIB       ,
    ARM7opSTMIB       ,
    ARM7opSTMIB       ,
    ARM7opSTMIB       ,
    ARM7opSTMIB       ,
    ARM7opSTMIB       ,
    ARM7opSTMIB       ,
    ARM7opSTMIB       ,
    ARM7opSTMIB       ,
    ARM7opSTMIB       ,
    ARM7opSTMIB       ,
    ARM7opSTMIB       ,
    ARM7opSTMIB       ,

    /*0x99*/
    ARM7opLDMIB       ,
    ARM7opLDMIB       ,
    ARM7opLDMIB       ,
    ARM7opLDMIB       ,
    ARM7opLDMIB       ,
    ARM7opLDMIB       ,
    ARM7opLDMIB       ,
    ARM7opLDMIB       ,
    ARM7opLDMIB       ,
    ARM7opLDMIB       ,
    ARM7opLDMIB       ,
    ARM7opLDMIB       ,
    ARM7opLDMIB       ,
    ARM7opLDMIB       ,
    ARM7opLDMIB       ,
    ARM7opLDMIB       ,

    /*0x9A*/
    ARM7opSTMIBw      ,
    ARM7opSTMIBw      ,
    ARM7opSTMIBw      ,
    ARM7opSTMIBw      ,
    ARM7opSTMIBw      ,
    ARM7opSTMIBw      ,
    ARM7opSTMIBw      ,
    ARM7opSTMIBw      ,
    ARM7opSTMIBw      ,
    ARM7opSTMIBw      ,
    ARM7opSTMIBw      ,
    ARM7opSTMIBw      ,
    ARM7opSTMIBw      ,
    ARM7opSTMIBw      ,
    ARM7opSTMIBw      ,
    ARM7opSTMIBw      ,

    /*0x9B*/
    ARM7opLDMIBw      ,
    ARM7opLDMIBw      ,
    ARM7opLDMIBw      ,
    ARM7opLDMIBw      ,
    ARM7opLDMIBw      ,
    ARM7opLDMIBw      ,
    ARM7opLDMIBw      ,
    ARM7opLDMIBw      ,
    ARM7opLDMIBw      ,
    ARM7opLDMIBw      ,
    ARM7opLDMIBw      ,
    ARM7opLDMIBw      ,
    ARM7opLDMIBw      ,
    ARM7opLDMIBw      ,
    ARM7opLDMIBw      ,
    ARM7opLDMIBw      ,

    /*0x9C*/
    ARM7opSTMIBu      ,
    ARM7opSTMIBu      ,
    ARM7opSTMIBu      ,
    ARM7opSTMIBu      ,
    ARM7opSTMIBu      ,
    ARM7opSTMIBu      ,
    ARM7opSTMIBu      ,
    ARM7opSTMIBu      ,
    ARM7opSTMIBu      ,
    ARM7opSTMIBu      ,
    ARM7opSTMIBu      ,
    ARM7opSTMIBu      ,
    ARM7opSTMIBu      ,
    ARM7opSTMIBu      ,
    ARM7opSTMIBu      ,
    ARM7opSTMIBu      ,

    /*0x9D*/
    ARM7opLDMIBu      ,
    ARM7opLDMIBu      ,
    ARM7opLDMIBu      ,
    ARM7opLDMIBu      ,
    ARM7opLDMIBu      ,
    ARM7opLDMIBu      ,
    ARM7opLDMIBu      ,
    ARM7opLDMIBu      ,
    ARM7opLDMIBu      ,
    ARM7opLDMIBu      ,
    ARM7opLDMIBu      ,
    ARM7opLDMIBu      ,
    ARM7opLDMIBu      ,
    ARM7opLDMIBu      ,
    ARM7opLDMIBu      ,
    ARM7opLDMIBu      ,

    /*0x9E*/
    ARM7opSTMIBuw     ,
    ARM7opSTMIBuw     ,
    ARM7opSTMIBuw     ,
    ARM7opSTMIBuw     ,
    ARM7opSTMIBuw     ,
    ARM7opSTMIBuw     ,
    ARM7opSTMIBuw     ,
    ARM7opSTMIBuw     ,
    ARM7opSTMIBuw     ,
    ARM7opSTMIBuw     ,
    ARM7opSTMIBuw     ,
    ARM7opSTMIBuw     ,
    ARM7opSTMIBuw     ,
    ARM7opSTMIBuw     ,
    ARM7opSTMIBuw     ,
    ARM7opSTMIBuw     ,

    /*0x9F*/
    ARM7opLDMIBuw     ,
    ARM7opLDMIBuw     ,
    ARM7opLDMIBuw     ,
    ARM7opLDMIBuw     ,
    ARM7opLDMIBuw     ,
    ARM7opLDMIBuw     ,
    ARM7opLDMIBuw     ,
    ARM7opLDMIBuw     ,
    ARM7opLDMIBuw     ,
    ARM7opLDMIBuw     ,
    ARM7opLDMIBuw     ,
    ARM7opLDMIBuw     ,
    ARM7opLDMIBuw     ,
    ARM7opLDMIBuw     ,
    ARM7opLDMIBuw     ,
    ARM7opLDMIBuw     ,

    /*0xA0*/
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,

    /*0xA1*/
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,

    /*0xA2*/
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,

    /*0xA3*/
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,

    /*0xA4*/
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,

    /*0xA5*/
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,

    /*0xA6*/
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,

    /*0xA7*/
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,

    /*0xA8*/
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,

    /*0xA9*/
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,

    /*0xAA*/
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,

    /*0xAB*/
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,

    /*0xAC*/
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,

    /*0xAD*/
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,

    /*0xAE*/
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,

    /*0xAF*/
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,
    ARM7opB           ,

    /*0xB0*/
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,

    /*0xB1*/
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,

    /*0xB2*/
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,

    /*0xB3*/
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,

    /*0xB4*/
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,

    /*0xB5*/
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,

    /*0xB6*/
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,

    /*0xB7*/
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,

    /*0xB8*/
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,

    /*0xB9*/
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,

    /*0xBA*/
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,

    /*0xBB*/
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,

    /*0xBC*/
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,

    /*0xBD*/
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,

    /*0xBE*/
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,

    /*0xBF*/
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,
    ARM7opBL          ,

    /*0xC0*/ // Coprocessors undefined here!
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xC1*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xC2*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xC3*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xC4*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xC5*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xC6*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xC7*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xC8*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xC9*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xCA*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xCB*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xCC*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xCD*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xCE*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xCF*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xD0*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xD1*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xD2*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xD3*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xD4*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xD5*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xD6*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xD7*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xD8*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xD9*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xDA*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xDB*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xDC*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xDD*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xDE*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xDF*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xE0*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xE1*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xE2*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xE3*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xE4*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xE5*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xE6*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xE7*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xE8*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xE9*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xEA*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xEB*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xEC*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xED*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xEE*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xEF*/
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,
    ARM7opUNI         ,

    /*0xF0*/
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,

    /*0xF1*/
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,

    /*0xF2*/
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,

    /*0xF3*/
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,

    /*0xF4*/
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,

    /*0xF5*/
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,

    /*0xF6*/
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,

    /*0xF7*/
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,

    /*0xF8*/
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,

    /*0xF9*/
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,

    /*0xFA*/
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,

    /*0xFB*/
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,

    /*0xFC*/
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,

    /*0xFD*/
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,

    /*0xFE*/
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,

    /*0xFF*/
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
    ARM7opSWI         ,
};

const opfptr thumb7ops[]={
    //0x00
    Thumb7opLSLimm    ,
    Thumb7opLSLimm    ,
    Thumb7opLSLimm    ,
    Thumb7opLSLimm    ,
    Thumb7opLSLimm    ,
    Thumb7opLSLimm    ,
    Thumb7opLSLimm    ,
    Thumb7opLSLimm    ,
    Thumb7opLSRimm    ,
    Thumb7opLSRimm    ,
    Thumb7opLSRimm    ,
    Thumb7opLSRimm    ,
    Thumb7opLSRimm    ,
    Thumb7opLSRimm    ,
    Thumb7opLSRimm    ,
    Thumb7opLSRimm    ,

    //0x10
    Thumb7opASRimm    ,
    Thumb7opASRimm    ,
    Thumb7opASRimm    ,
    Thumb7opASRimm    ,
    Thumb7opASRimm    ,
    Thumb7opASRimm    ,
    Thumb7opASRimm    ,
    Thumb7opASRimm    ,
    Thumb7opADDreg    ,
    Thumb7opADDreg    ,
    Thumb7opSUBreg    ,
    Thumb7opSUBreg    ,
    Thumb7opADDimm3   ,
    Thumb7opADDimm3   ,
    Thumb7opSUBimm3   ,
    Thumb7opSUBimm3   ,

    //0x20
    Thumb7opMOVimm8r0 ,
    Thumb7opMOVimm8r1 ,
    Thumb7opMOVimm8r2 ,
    Thumb7opMOVimm8r3 ,
    Thumb7opMOVimm8r4 ,
    Thumb7opMOVimm8r5 ,
    Thumb7opMOVimm8r6 ,
    Thumb7opMOVimm8r7 ,
    Thumb7opCMPimm8r0 ,
    Thumb7opCMPimm8r1 ,
    Thumb7opCMPimm8r2 ,
    Thumb7opCMPimm8r3 ,
    Thumb7opCMPimm8r4 ,
    Thumb7opCMPimm8r5 ,
    Thumb7opCMPimm8r6 ,
    Thumb7opCMPimm8r7 ,

    //0x30
    Thumb7opADDimm8r0 ,
    Thumb7opADDimm8r1 ,
    Thumb7opADDimm8r2 ,
    Thumb7opADDimm8r3 ,
    Thumb7opADDimm8r4 ,
    Thumb7opADDimm8r5 ,
    Thumb7opADDimm8r6 ,
    Thumb7opADDimm8r7 ,
    Thumb7opSUBimm8r0 ,
    Thumb7opSUBimm8r1 ,
    Thumb7opSUBimm8r2 ,
    Thumb7opSUBimm8r3 ,
    Thumb7opSUBimm8r4 ,
    Thumb7opSUBimm8r5 ,
    Thumb7opSUBimm8r6 ,
    Thumb7opSUBimm8r7 ,

    //0x40
    Thumb7opDPg1      ,
    Thumb7opDPg2      ,
    Thumb7opDPg3      ,
    Thumb7opDPg4      ,
    Thumb7opADDH      ,
    Thumb7opCMPH      ,
    Thumb7opMOVH      ,
    Thumb7opBXreg     ,
    Thumb7opLDRPCr0   ,
    Thumb7opLDRPCr1   ,
    Thumb7opLDRPCr2   ,
    Thumb7opLDRPCr3   ,
    Thumb7opLDRPCr4   ,
    Thumb7opLDRPCr5   ,
    Thumb7opLDRPCr6   ,
    Thumb7opLDRPCr7   ,

    //0x50
    Thumb7opSTRreg    ,
    Thumb7opSTRreg    ,
    Thumb7opSTRHreg   ,
    Thumb7opSTRHreg   ,
    Thumb7opSTRBreg   ,
    Thumb7opSTRBreg   ,
    Thumb7opLDRSBreg  ,
    Thumb7opLDRSBreg  ,
    Thumb7opLDRreg    ,
    Thumb7opLDRreg    ,
    Thumb7opLDRHreg   ,
    Thumb7opLDRHreg   ,
    Thumb7opLDRBreg   ,
    Thumb7opLDRBreg   ,
    Thumb7opLDRSHreg  ,
    Thumb7opLDRSHreg  ,

    //0x60
    Thumb7opSTRimm5   ,
    Thumb7opSTRimm5   ,
    Thumb7opSTRimm5   ,
    Thumb7opSTRimm5   ,
    Thumb7opSTRimm5   ,
    Thumb7opSTRimm5   ,
    Thumb7opSTRimm5   ,
    Thumb7opSTRimm5   ,
    Thumb7opLDRimm5   ,
    Thumb7opLDRimm5   ,
    Thumb7opLDRimm5   ,
    Thumb7opLDRimm5   ,
    Thumb7opLDRimm5   ,
    Thumb7opLDRimm5   ,
    Thumb7opLDRimm5   ,
    Thumb7opLDRimm5   ,

    //0x70
    Thumb7opSTRBimm5  ,
    Thumb7opSTRBimm5  ,
    Thumb7opSTRBimm5  ,
    Thumb7opSTRBimm5  ,
    Thumb7opSTRBimm5  ,
    Thumb7opSTRBimm5  ,
    Thumb7opSTRBimm5  ,
    Thumb7opSTRBimm5  ,
    Thumb7opLDRBimm5  ,
    Thumb7opLDRBimm5  ,
    Thumb7opLDRBimm5  ,
    Thumb7opLDRBimm5  ,
    Thumb7opLDRBimm5  ,
    Thumb7opLDRBimm5  ,
    Thumb7opLDRBimm5  ,
    Thumb7opLDRBimm5  ,

    //0x80
    Thumb7opSTRHimm5  ,
    Thumb7opSTRHimm5  ,
    Thumb7opSTRHimm5  ,
    Thumb7opSTRHimm5  ,
    Thumb7opSTRHimm5  ,
    Thumb7opSTRHimm5  ,
    Thumb7opSTRHimm5  ,
    Thumb7opSTRHimm5  ,
    Thumb7opLDRHimm5  ,
    Thumb7opLDRHimm5  ,
    Thumb7opLDRHimm5  ,
    Thumb7opLDRHimm5  ,
    Thumb7opLDRHimm5  ,
    Thumb7opLDRHimm5  ,
    Thumb7opLDRHimm5  ,
    Thumb7opLDRHimm5  ,

    //0x90
    Thumb7opSTRSPr0   ,
    Thumb7opSTRSPr1   ,
    Thumb7opSTRSPr2   ,
    Thumb7opSTRSPr3   ,
    Thumb7opSTRSPr4   ,
    Thumb7opSTRSPr5   ,
    Thumb7opSTRSPr6   ,
    Thumb7opSTRSPr7   ,
    Thumb7opLDRSPr0   ,
    Thumb7opLDRSPr1   ,
    Thumb7opLDRSPr2   ,
    Thumb7opLDRSPr3   ,
    Thumb7opLDRSPr4   ,
    Thumb7opLDRSPr5   ,
    Thumb7opLDRSPr6   ,
    Thumb7opLDRSPr7   ,

    //0xA0
    Thumb7opADDPCr0   ,
    Thumb7opADDPCr1   ,
    Thumb7opADDPCr2   ,
    Thumb7opADDPCr3   ,
    Thumb7opADDPCr4   ,
    Thumb7opADDPCr5   ,
    Thumb7opADDPCr6   ,
    Thumb7opADDPCr7   ,
    Thumb7opADDSPr0   ,
    Thumb7opADDSPr1   ,
    Thumb7opADDSPr2   ,
    Thumb7opADDSPr3   ,
    Thumb7opADDSPr4   ,
    Thumb7opADDSPr5   ,
    Thumb7opADDSPr6   ,
    Thumb7opADDSPr7   ,

    //0xB0
    Thumb7opADDSPimm7 ,
    Thumb7opUND       ,  // Misc map: unallocated
    Thumb7opUND       ,  // Misc map: unallocated
    Thumb7opUND       ,  // Misc map: unallocated
    Thumb7opPUSH      ,
    Thumb7opPUSHlr    ,
    Thumb7opUND       ,  // Misc map: unallocated
    Thumb7opUND       ,  // Misc map: unallocated
    Thumb7opUND       ,  // Misc map: unallocated
    Thumb7opUND       ,  // Misc map: unallocated
    Thumb7opUND       ,  // Misc map: unallocated
    Thumb7opUND       ,  // Misc map: unallocated
    Thumb7opPOP       ,
    Thumb7opPOPpc     ,
    Thumb7opBKPT      ,
    Thumb7opUND       ,  // Misc map: unallocated

    //0xC0
    Thumb7opSTMIAr0   ,
    Thumb7opSTMIAr1   ,
    Thumb7opSTMIAr2   ,
    Thumb7opSTMIAr3   ,
    Thumb7opSTMIAr4   ,
    Thumb7opSTMIAr5   ,
    Thumb7opSTMIAr6   ,
    Thumb7opSTMIAr7   ,
    Thumb7opLDMIAr0   ,
    Thumb7opLDMIAr1   ,
    Thumb7opLDMIAr2   ,
    Thumb7opLDMIAr3   ,
    Thumb7opLDMIAr4   ,
    Thumb7opLDMIAr5   ,
    Thumb7opLDMIAr6   ,
    Thumb7opLDMIAr7   ,

    //0xD0
    Thumb7opBEQ       ,
    Thumb7opBNE       ,
    Thumb7opBCS       ,
    Thumb7opBCC       ,
    Thumb7opBMI       ,
    Thumb7opBPL       ,
    Thumb7opBVS       ,
    Thumb7opBVC       ,
    Thumb7opBHI       ,
    Thumb7opBLS       ,
    Thumb7opBGE       ,
    Thumb7opBLT       ,
    Thumb7opBGT       ,
    Thumb7opBLE       ,
    Thumb7opUND       ,  // Covered by unconditional
    Thumb7opSWI       ,

    //0xE0
    Thumb7opB         ,
    Thumb7opB         ,
    Thumb7opB         ,
    Thumb7opB         ,
    Thumb7opB         ,
    Thumb7opB         ,
    Thumb7opB         ,
    Thumb7opB         ,
    Thumb7opUND       ,  // Defined in ARM5!
    Thumb7opUND       ,  // Defined in ARM5!
    Thumb7opUND       ,  // Defined in ARM5!
    Thumb7opUND       ,  // Defined in ARM5!
    Thumb7opUND       ,  // Defined in ARM5!
    Thumb7opUND       ,  // Defined in ARM5!
    Thumb7opUND       ,  // Defined in ARM5!
    Thumb7opUND       ,  // Defined in ARM5!

    //0xF0
    Thumb7opBLsetup   ,
    Thumb7opBLsetup   ,
    Thumb7opBLsetup   ,
    Thumb7opBLsetup   ,
    Thumb7opBLsetup   ,
    Thumb7opBLsetup   ,
    Thumb7opBLsetup   ,
    Thumb7opBLsetup   ,
    Thumb7opBLoff     ,
    Thumb7opBLoff     ,
    Thumb7opBLoff     ,
    Thumb7opBLoff     ,
    Thumb7opBLoff     ,
    Thumb7opBLoff     ,
    Thumb7opBLoff     ,
    Thumb7opBLoff     ,
};

#endif//__ARM7FUNC_H_

/*** EOF:arm7func.h ******************************************************/

