/*
Konami "Twin16"

driver by Phil Stroffolino

CPU   : 68000(10M)x2 Z-80
SOUND : YM2151 007232 uPD7759C
OSC.  : 3.579545MHz 18432.00KHz

Main processors are a pair of 68000 CPUs
Sounds are generated by a Z80, a Yamaha 2151 and 3012, a Konami custom IC and a UPD7759C

        Dark Adventure / Devil World / Majuu no Ohkoku
        Vulcan Venture / Gradius II
        Cuebrick
        MIA (Japan)
        Final Round / Hard Puncher (Japan)

Known Issues:
    - some rogue sprites in Devil World
    - sprite-background priority isn't correct (see intro screen of Devil World)

68000 Memory Map for Konami Twin System

    CPUA                                        CPUB
    0x000000..0x03ffff  ROM                     0x000000..0x03ffff
    0x040000..0x043fff  communication RAM       0x040000..0x043fff (shared)
    0x060000..0x063fff  work RAM                0x060000..0x063fff
    0x080000..0x080fff  palette
    0x080000..0x09ffff  ROM (extra tile data)
    0x0a0000..0x0a0001  IRQ control             0x0a0000..0x0a0001
    0x0a0008..0x0a0009  sound command
    0x0a0010..0xa00011  watchdog
    0x0c0000..0x0c0001  screenflip
    0x0c0002..0x0c000f  scroll registers

    0x100000..0x103fff  FIXRAM (text layer)
    0x120000..0x123fff  VIDRAM (tilemaps)       0x480000..0x483fff (shared)
    0x140000..0x143fff  OBJRAM (sprites)        0x400000..0x403fff (shared)
                        ZIP RAM (tiles)         0x500000..0x53ffff
                        gfx ROM (banked)        0x600000..0x77ffff
                        sprite gfx RAM          0x780000..0x79ffff
*/

/*

    Konami Twin16 Hardware

    TODO:

    - mia reset crash

*/

#include "driver.h"
#include "vidhrdw/generic.h"
#include "cpu/m68000/m68000.h"
#include "cpu/z80/z80.h"
#include "sound/2151intf.h"
#include "sound/k007232.h"
#include "sound/upd7759.h"

extern WRITE16_HANDLER( twin16_videoram2_w );
extern WRITE16_HANDLER( twin16_paletteram_word_w );
extern WRITE16_HANDLER( fround_gfx_bank_w );
extern WRITE16_HANDLER( twin16_video_register_w );

extern VIDEO_START( twin16 );
extern VIDEO_START( fround );
extern VIDEO_UPDATE( twin16 );
extern VIDEO_EOF( twin16 );

extern void twin16_spriteram_process( void );

UINT16 twin16_custom_vidhrdw;
UINT16 *twin16_gfx_rom;
UINT16 *twin16_sprite_gfx_ram;
UINT16 *twin16_tile_gfx_ram;
UINT16 *twin16_videoram2; /* text layer */

static UINT16 twin16_CPUA_register, twin16_CPUB_register;

#define CPUA_IRQ_ENABLE (twin16_CPUA_register & 0x20)
#define CPUB_IRQ_ENABLE (twin16_CPUB_register & 0x02)

static UINT8 twin16_soundlatch;
static UINT16 twin16_sound_command;

static int cuebrick_nvram_bank;
static UINT16 cuebrick_nvram[0x400*0x20];	/* 32k paged in a 1k window */


int twin16_spriteram_process_enable( void )
{
	return (twin16_CPUA_register & 0x40) == 0;
}

enum
{
	CPU_SOUND,
	CPU_B,
	CPU_A
};


/******************************************************************************************/

#define COMRAM_r					MRA16_BANK1
#define COMRAM_w					MWA16_BANK1

/* Read/Write Handlers */

static READ16_HANDLER( videoram16_r )
{
	return videoram16[offset];
}

static WRITE16_HANDLER( videoram16_w )
{
	COMBINE_DATA(videoram16 + offset);
}

static READ16_HANDLER( extra_rom_r )
{
	return ((UINT16 *)memory_region(REGION_GFX3))[offset];
}

static READ16_HANDLER( twin16_gfx_rom1_r )
{
	return twin16_gfx_rom[offset];
}

static READ16_HANDLER( twin16_gfx_rom2_r )
{
	return twin16_gfx_rom[offset + 0x80000 + ((twin16_CPUB_register&0x04)?0x40000:0)];
}

static WRITE16_HANDLER( sound_command_w )
{
	COMBINE_DATA(&twin16_sound_command);
	soundlatch_w( 0, twin16_sound_command&0xff );
}

static READ16_HANDLER( twin16_sprite_status_r )
{
	/*
        return value indicates whether the spriteram16-processing circuitry
        is busy.

        for now, we'll just alternate the value every time it is read
    */
	static int k;
	k = 1-k;
	return k;
}

static WRITE16_HANDLER( twin16_CPUA_register_w )
{
	/*
        7   6   5   4   3   2   1   0
            ?                           sprite protection disable
                X                       IRQ5 enable (CPUA)
                    X                   0->1 trigger IRQ6 on CPUB
                        X               0->1 trigger IRQ on sound CPU
                            x   x   x   coin counters
    */
	UINT16 old = twin16_CPUA_register;
	COMBINE_DATA(&twin16_CPUA_register);
	if( twin16_CPUA_register!=old )
	{
		if( (old&0x08)==0 && (twin16_CPUA_register&0x08) )
		{
			cpunum_set_input_line_and_vector( CPU_SOUND, 0, HOLD_LINE, 0xff );
		}

		if( (old&0x40) && (twin16_CPUA_register&0x40)==0 )
		{
			twin16_spriteram_process();
		}

		if( (old&0x10)==0 && (twin16_CPUA_register&0x10) )
		{
			cpunum_set_input_line( CPU_B, MC68000_IRQ_6, HOLD_LINE );
		}
		coin_counter_w( 0, twin16_CPUA_register&0x01 );
		coin_counter_w( 1, twin16_CPUA_register&0x02 );
		coin_counter_w( 2, twin16_CPUA_register&0x04 );
	}
}

static WRITE16_HANDLER( twin16_CPUB_register_w )
{
	/*
        7   6   5   4   3   2   1   0
                            X           gfx bank select
                                X       IRQ5 enable
                                    X   0->1 trigger IRQ6 on CPUA
    */
	UINT16 old = twin16_CPUB_register;
	COMBINE_DATA(&twin16_CPUB_register);
	if( twin16_CPUB_register!=old )
	{
		if( (old&0x01)==0 && (twin16_CPUB_register&0x1) )
		{
			cpunum_set_input_line( CPU_A, MC68000_IRQ_6, HOLD_LINE );
		}
	}
}

static WRITE16_HANDLER( fround_CPU_register_w )
{
	UINT16 old = twin16_CPUA_register;
	COMBINE_DATA(&twin16_CPUA_register);
	if( twin16_CPUA_register!=old )
	{
		if( (old&0x08)==0 && (twin16_CPUA_register&0x08) )
			cpunum_set_input_line_and_vector( CPU_SOUND, 0, HOLD_LINE, 0xff ); /* trigger IRQ on sound CPU */
	}
}

static READ16_HANDLER( twin16_input_r )
{
	switch( offset )
	{
		case 0x00: return readinputport(0); /* coin */
		case 0x01: return readinputport(1); /* p1 */
		case 0x02: return readinputport(2); /* p2 */
		case 0x03: return readinputport(3); /* p3 (Dark Adventure) */
		case 0x08: return readinputport(5); /* DSW1 */
		case 0x09: return readinputport(4); /* DSW2 */
		case 0x0c: return readinputport(6); /* DSW3 */
	}
	return 0;
}

static READ8_HANDLER( twin16_sres_r )
{
	return twin16_soundlatch;
}

static WRITE8_HANDLER( twin16_sres_w )
{
	/* bit 1 resets the UPD7795C sound chip */
	upd7759_reset_w(0, data & 0x02);
	twin16_soundlatch = data;
}

static READ16_HANDLER( cuebrick_nvram_r )
{
	return cuebrick_nvram[offset + (cuebrick_nvram_bank * 0x400 / 2)];
}

static WRITE16_HANDLER( cuebrick_nvram_w )
{
	COMBINE_DATA(&cuebrick_nvram[offset + (cuebrick_nvram_bank * 0x400 / 2)]);
}

static WRITE16_HANDLER( cuebrick_nvram_bank_w )
{
	cuebrick_nvram_bank = (data >> 8);
}

/* Memory Maps */

static ADDRESS_MAP_START( readmem_sound, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x7fff) AM_READ(MRA8_ROM)
	AM_RANGE(0x8000, 0x8fff) AM_READ(MRA8_RAM)
	AM_RANGE(0x9000, 0x9000) AM_READ(twin16_sres_r)
	AM_RANGE(0xa000, 0xa000) AM_READ(soundlatch_r)
	AM_RANGE(0xb000, 0xb00d) AM_READ(K007232_read_port_0_r)
	AM_RANGE(0xc001, 0xc001) AM_READ(YM2151_status_port_0_r)
	AM_RANGE(0xf000, 0xf000) AM_READ(upd7759_0_busy_r)
ADDRESS_MAP_END

static ADDRESS_MAP_START( writemem_sound, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x7fff) AM_WRITE(MWA8_ROM)
	AM_RANGE(0x8000, 0x8fff) AM_WRITE(MWA8_RAM)
	AM_RANGE(0x9000, 0x9000) AM_WRITE(twin16_sres_w)
	AM_RANGE(0xb000, 0xb00d) AM_WRITE(K007232_write_port_0_w)
	AM_RANGE(0xc000, 0xc000) AM_WRITE(YM2151_register_port_0_w)
	AM_RANGE(0xc001, 0xc001) AM_WRITE(YM2151_data_port_0_w)
	AM_RANGE(0xd000, 0xd000) AM_WRITE(upd7759_0_port_w)
	AM_RANGE(0xe000, 0xe000) AM_WRITE(upd7759_0_start_w)
	AM_RANGE(0xf000, 0xf000) AM_WRITE(MWA8_NOP)	/* ??? */
ADDRESS_MAP_END

static ADDRESS_MAP_START( readmem, ADDRESS_SPACE_PROGRAM, 16 )
	AM_RANGE(0x000000, 0x03ffff) AM_READ(MRA16_ROM)
	AM_RANGE(0x040000, 0x043fff) AM_READ(COMRAM_r)
	AM_RANGE(0x044000, 0x04ffff) AM_READ(MRA16_RAM)	/* miaj */
	AM_RANGE(0x060000, 0x063fff) AM_READ(MRA16_RAM)
	AM_RANGE(0x080000, 0x080fff) AM_READ(MRA16_RAM)
	AM_RANGE(0x0a0000, 0x0a001b) AM_READ(twin16_input_r)
	AM_RANGE(0x0b0000, 0x0b03ff) AM_READ(cuebrick_nvram_r)
	AM_RANGE(0x0c000e, 0x0c000f) AM_READ(twin16_sprite_status_r)
	AM_RANGE(0x100000, 0x103fff) AM_READ(MRA16_RAM)
	AM_RANGE(0x104000, 0x105fff) AM_READ(MRA16_RAM)	/* miaj */
	AM_RANGE(0x120000, 0x123fff) AM_READ(MRA16_RAM)
	AM_RANGE(0x140000, 0x143fff) AM_READ(MRA16_RAM)
ADDRESS_MAP_END

static ADDRESS_MAP_START( writemem, ADDRESS_SPACE_PROGRAM, 16 )
	AM_RANGE(0x000000, 0x03ffff) AM_WRITE(MWA16_ROM)
	AM_RANGE(0x040000, 0x043fff) AM_WRITE(COMRAM_w)
	AM_RANGE(0x044000, 0x04ffff) AM_WRITE(MWA16_RAM)	/* miaj */
	AM_RANGE(0x060000, 0x063fff) AM_WRITE(MWA16_RAM)
	AM_RANGE(0x080000, 0x080fff) AM_WRITE(twin16_paletteram_word_w) AM_BASE(&paletteram16)
	AM_RANGE(0x081000, 0x081fff) AM_WRITE(MWA16_NOP)
	AM_RANGE(0x0a0000, 0x0a0001) AM_WRITE(twin16_CPUA_register_w)
	AM_RANGE(0x0a0008, 0x0a0009) AM_WRITE(sound_command_w)
	AM_RANGE(0x0a0010, 0x0a0011) AM_WRITE(watchdog_reset16_w)
	AM_RANGE(0x0b0000, 0x0b03ff) AM_WRITE(cuebrick_nvram_w)
	AM_RANGE(0x0b0400, 0x0b0401) AM_WRITE(cuebrick_nvram_bank_w)
	AM_RANGE(0x0c0000, 0x0c000f) AM_WRITE(twin16_video_register_w)
	AM_RANGE(0x100000, 0x103fff) AM_WRITE(twin16_videoram2_w) AM_BASE(&twin16_videoram2)
	AM_RANGE(0x104000, 0x105fff) AM_WRITE(MWA16_RAM)	/* miaj */
	AM_RANGE(0x120000, 0x123fff) AM_WRITE(MWA16_RAM) AM_BASE(&videoram16)
	AM_RANGE(0x140000, 0x143fff) AM_WRITE(MWA16_RAM) AM_BASE(&spriteram16) AM_SIZE(&spriteram_size)
ADDRESS_MAP_END

static ADDRESS_MAP_START( readmem_sub, ADDRESS_SPACE_PROGRAM, 16 )
	AM_RANGE(0x000000, 0x03ffff) AM_READ(MRA16_ROM)
	AM_RANGE(0x040000, 0x043fff) AM_READ(COMRAM_r)
	AM_RANGE(0x044000, 0x04ffff) AM_READ(MRA16_RAM)	/* miaj */
	AM_RANGE(0x060000, 0x063fff) AM_READ(MRA16_RAM)
	AM_RANGE(0x080000, 0x09ffff) AM_READ(extra_rom_r)
	AM_RANGE(0x400000, 0x403fff) AM_READ(spriteram16_r)
	AM_RANGE(0x480000, 0x483fff) AM_READ(videoram16_r)
	AM_RANGE(0x500000, 0x53ffff) AM_READ(MRA16_RAM)
	AM_RANGE(0x600000, 0x6fffff) AM_READ(twin16_gfx_rom1_r)
	AM_RANGE(0x700000, 0x77ffff) AM_READ(twin16_gfx_rom2_r)
	AM_RANGE(0x780000, 0x79ffff) AM_READ(MRA16_RAM)
ADDRESS_MAP_END

static ADDRESS_MAP_START( writemem_sub, ADDRESS_SPACE_PROGRAM, 16 )
	AM_RANGE(0x000000, 0x03ffff) AM_WRITE(MWA16_ROM)
	AM_RANGE(0x040000, 0x043fff) AM_WRITE(COMRAM_w)
	AM_RANGE(0x044000, 0x04ffff) AM_WRITE(MWA16_RAM)	/* miaj */
	AM_RANGE(0x060000, 0x063fff) AM_WRITE(MWA16_RAM)
	AM_RANGE(0x0a0000, 0x0a0001) AM_WRITE(twin16_CPUB_register_w)
	AM_RANGE(0x400000, 0x403fff) AM_WRITE(spriteram16_w)
	AM_RANGE(0x480000, 0x483fff) AM_WRITE(videoram16_w)
	AM_RANGE(0x500000, 0x53ffff) AM_WRITE(MWA16_RAM) AM_BASE(&twin16_tile_gfx_ram)
	AM_RANGE(0x780000, 0x79ffff) AM_WRITE(MWA16_RAM) AM_BASE(&twin16_sprite_gfx_ram)
ADDRESS_MAP_END

static ADDRESS_MAP_START( fround_readmem, ADDRESS_SPACE_PROGRAM, 16 )
	AM_RANGE(0x000000, 0x03ffff) AM_READ(MRA16_ROM)
	AM_RANGE(0x040000, 0x043fff) AM_READ(COMRAM_r)
	AM_RANGE(0x060000, 0x063fff) AM_READ(MRA16_RAM)
	AM_RANGE(0x080000, 0x080fff) AM_READ(MRA16_RAM)
	AM_RANGE(0x0a0000, 0x0a001b) AM_READ(twin16_input_r)
	AM_RANGE(0x0c000e, 0x0c000f) AM_READ(twin16_sprite_status_r)
	AM_RANGE(0x100000, 0x103fff) AM_READ(MRA16_RAM)
	AM_RANGE(0x120000, 0x123fff) AM_READ(MRA16_RAM)
	AM_RANGE(0x140000, 0x143fff) AM_READ(MRA16_RAM)
	AM_RANGE(0x500000, 0x6fffff) AM_READ(twin16_gfx_rom1_r)
ADDRESS_MAP_END

static ADDRESS_MAP_START( fround_writemem, ADDRESS_SPACE_PROGRAM, 16 )
	AM_RANGE(0x000000, 0x03ffff) AM_WRITE(MWA16_ROM)
	AM_RANGE(0x040000, 0x043fff) AM_WRITE(COMRAM_w)
	AM_RANGE(0x060000, 0x063fff) AM_WRITE(MWA16_RAM)
	AM_RANGE(0x080000, 0x080fff) AM_WRITE(twin16_paletteram_word_w) AM_BASE(&paletteram16)
	AM_RANGE(0x0a0000, 0x0a0001) AM_WRITE(fround_CPU_register_w)
	AM_RANGE(0x0a0008, 0x0a0009) AM_WRITE(sound_command_w)
	AM_RANGE(0x0a0010, 0x0a0011) AM_WRITE(watchdog_reset16_w)
	AM_RANGE(0x0c0000, 0x0c000f) AM_WRITE(twin16_video_register_w)
	AM_RANGE(0x0e0000, 0x0e0001) AM_WRITE(fround_gfx_bank_w)
	AM_RANGE(0x100000, 0x103fff) AM_WRITE(twin16_videoram2_w) AM_BASE(&twin16_videoram2)
	AM_RANGE(0x120000, 0x123fff) AM_WRITE(MWA16_RAM) AM_BASE(&videoram16)
	AM_RANGE(0x140000, 0x143fff) AM_WRITE(MWA16_RAM) AM_BASE(&spriteram16) AM_SIZE(&spriteram_size)
ADDRESS_MAP_END

/* Input Ports */

#define KONAMI_TWIN_COINAGE PORT_START \
	PORT_DIPNAME( 0x0f, 0x0f, DEF_STR( Coin_A ) ) \
	PORT_DIPSETTING(    0x02, DEF_STR( 4C_1C ) ) \
	PORT_DIPSETTING(    0x05, DEF_STR( 3C_1C ) ) \
	PORT_DIPSETTING(    0x08, DEF_STR( 2C_1C ) ) \
	PORT_DIPSETTING(    0x04, DEF_STR( 3C_2C ) ) \
	PORT_DIPSETTING(    0x01, DEF_STR( 4C_3C ) ) \
	PORT_DIPSETTING(    0x0f, DEF_STR( 1C_1C ) ) \
	PORT_DIPSETTING(    0x03, DEF_STR( 3C_4C ) ) \
	PORT_DIPSETTING(    0x07, DEF_STR( 2C_3C ) ) \
	PORT_DIPSETTING(    0x0e, DEF_STR( 1C_2C ) ) \
	PORT_DIPSETTING(    0x06, DEF_STR( 2C_5C ) ) \
	PORT_DIPSETTING(    0x0d, DEF_STR( 1C_3C ) ) \
	PORT_DIPSETTING(    0x0c, DEF_STR( 1C_4C ) ) \
	PORT_DIPSETTING(    0x0b, DEF_STR( 1C_5C ) ) \
	PORT_DIPSETTING(    0x0a, DEF_STR( 1C_6C ) ) \
	PORT_DIPSETTING(    0x09, DEF_STR( 1C_7C ) ) \
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) ) \
	PORT_DIPNAME( 0xf0, 0xf0, DEF_STR( Coin_B ) ) \
	PORT_DIPSETTING(    0x20, DEF_STR( 4C_1C ) ) \
	PORT_DIPSETTING(    0x50, DEF_STR( 3C_1C ) ) \
	PORT_DIPSETTING(    0x80, DEF_STR( 2C_1C ) ) \
	PORT_DIPSETTING(    0x40, DEF_STR( 3C_2C ) ) \
	PORT_DIPSETTING(    0x10, DEF_STR( 4C_3C ) ) \
	PORT_DIPSETTING(    0xf0, DEF_STR( 1C_1C ) ) \
	PORT_DIPSETTING(    0x30, DEF_STR( 3C_4C ) ) \
	PORT_DIPSETTING(    0x70, DEF_STR( 2C_3C ) ) \
	PORT_DIPSETTING(    0xe0, DEF_STR( 1C_2C ) ) \
	PORT_DIPSETTING(    0x60, DEF_STR( 2C_5C ) ) \
	PORT_DIPSETTING(    0xd0, DEF_STR( 1C_3C ) ) \
	PORT_DIPSETTING(    0xc0, DEF_STR( 1C_4C ) ) \
	PORT_DIPSETTING(    0xb0, DEF_STR( 1C_5C ) ) \
	PORT_DIPSETTING(    0xa0, DEF_STR( 1C_6C ) ) \
	PORT_DIPSETTING(    0x90, DEF_STR( 1C_7C ) ) \
	PORT_DIPSETTING(    0x00, "Disabled" )

INPUT_PORTS_START( devilw )
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE4 )	/* map, advance through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )	/* start/powerup */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 )	/* attack */
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 )	/* jump */
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(2)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	KONAMI_TWIN_COINAGE

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x03, "2" )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x01, "5" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_BIT( 0x1c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x60, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x20, "Difficult" )
	PORT_DIPSETTING(    0x00, "Very Difficult" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW3 0xa0019 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

INPUT_PORTS_START( darkadv )
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE4 )	/* map, advance through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE2 )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )	/* start/jump */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 )	/* attack */
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 )	/* dynamite */
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE3 )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(2)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(3)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(3)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(3)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* Coinage */
	PORT_DIPNAME( 0x0f, 0x0f, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(    0x05, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 3C_2C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 4C_3C ) )
	PORT_DIPSETTING(    0x0f, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 3C_4C ) )
	PORT_DIPSETTING(    0x07, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(    0x0e, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x06, DEF_STR( 2C_5C ) )
	PORT_DIPSETTING(    0x0d, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(    0x0c, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(    0x0b, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(    0x0a, DEF_STR( 1C_6C ) )
	PORT_DIPSETTING(    0x09, DEF_STR( 1C_7C ) )
	PORT_DIPSETTING(    0x00, "Invalid" )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED)

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x03, "2" )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x01, "5" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_BIT( 0x1c, IP_ACTIVE_LOW, IPT_UNUSED)
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x60, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x20, "Difficult" )
	PORT_DIPSETTING(    0x00, "Very Difficult" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW3 0xa0019 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED)
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED)
INPUT_PORTS_END

INPUT_PORTS_START( vulcan )
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 )		/* advance through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )	/* powerup */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 )	/* shoot */
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 )	/* missile */
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(2)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	KONAMI_TWIN_COINAGE

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x03, "2" )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x01, "4" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x18, 0x18, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x18, "20K 70K" )
	PORT_DIPSETTING(    0x10, "30K 80K" )
	PORT_DIPSETTING(    0x08, "20K" )
	PORT_DIPSETTING(    0x00, "70K" )
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x60, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x20, "Difficult" )
	PORT_DIPSETTING(    0x00, "Very Difficult" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW3 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "Upright Controls" )
	PORT_DIPSETTING(    0x02, DEF_STR( Single ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Dual ) )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED)
INPUT_PORTS_END


INPUT_PORTS_START( gradius2 )	/* same as vulcan, different bonus */
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 )		/* advance through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )	/* powerup */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 )	/* shoot */
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 )	/* missile */
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(2)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	KONAMI_TWIN_COINAGE

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x03, "2" )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x01, "4" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x18, 0x18, DEF_STR (Bonus_Life ) )
	PORT_DIPSETTING(    0x18, "20K 150K" )
	PORT_DIPSETTING(    0x10, "30K 200K" )
	PORT_DIPSETTING(    0x08, "20K" )
	PORT_DIPSETTING(    0x00, "70K" )
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x60, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x20, "Difficult" )
	PORT_DIPSETTING(    0x00, "Very Difficult" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW3 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "Upright Controls" )
	PORT_DIPSETTING(    0x02, DEF_STR( Single ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Dual ) )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED)
INPUT_PORTS_END

INPUT_PORTS_START( fround )
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 )		/* advance through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )	/* face punch */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 )	/* body punch */
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 )	/* defend */
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(2)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED)

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	KONAMI_TWIN_COINAGE

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x03, 0x02, "Energy" )
	PORT_DIPSETTING(    0x03, "18" )
	PORT_DIPSETTING(    0x02, "20" )
	PORT_DIPSETTING(    0x01, "22" )
	PORT_DIPSETTING(    0x00, "24" )
	PORT_BIT( 0x1c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x60, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x20, "Difficult" )
	PORT_DIPSETTING(    0x00, "Very Difficult" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW3 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

INPUT_PORTS_START( miaj )
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 )		/* advance through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )	/* knife */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 )	/* weapon */
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 )	/* choice */
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(2)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	KONAMI_TWIN_COINAGE

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x03, "2" )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x01, "5" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x18, 0x10, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x18, "30K 80K" )
	PORT_DIPSETTING(    0x10, "50K 100K" )
	PORT_DIPSETTING(    0x08, "50K" )
	PORT_DIPSETTING(    0x00, "100K" )
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x60, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x20, "Difficult" )
	PORT_DIPSETTING(    0x00, "Very Difficult" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW3 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, "VRAM Character Check" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

INPUT_PORTS_START( cuebrick )
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 )		/* advance through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )	/* move */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 )	/* warp */
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 )	/* stop */
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(2)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	KONAMI_TWIN_COINAGE

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x03, "1" )
	PORT_DIPSETTING(    0x02, "2" )
	PORT_DIPSETTING(    0x01, "3" )
	PORT_DIPSETTING(    0x00, "4" )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x18, 0x18, "Machine Name" )
	PORT_DIPSETTING(    0x18, DEF_STR( None ) )
	PORT_DIPSETTING(    0x10, "Lewis" )
	PORT_DIPSETTING(    0x08, "Johnson" )
	PORT_DIPSETTING(    0x00, "George" )
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x60, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Hard ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Hardest ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW3 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "Upright Controls" )
	PORT_DIPSETTING(    0x02, DEF_STR( Single ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Dual ) )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x08, 0x08, "Mode" )
	PORT_DIPSETTING(    0x08, "3" )
	PORT_DIPSETTING(    0x00, "4" )
INPUT_PORTS_END

/* Graphics Layouts */

static gfx_layout alpha_layout =
{
	8,8,
	0x200,
	4,
	{ 0,1,2,3 },
	{ 0*4,1*4,2*4,3*4,4*4,5*4,6*4,7*4 },
	{ 0*32,1*32,2*32,3*32,4*32,5*32,6*32,7*32 },
	8*32
};

/* Graphics Decode Info */

static gfx_decode gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0x00000, &alpha_layout, 0, 16 },
	{ -1 }
};

/* Sound Interfaces */

static void volume_callback(int v)
{
	K007232_set_volume(0,0,(v >> 4) * 0x11,0);
	K007232_set_volume(0,1,0,(v & 0x0f) * 0x11);
}

static struct K007232_interface k007232_interface =
{
	REGION_SOUND1, /* memory regions */
	volume_callback	/* external port callback */
};

static struct upd7759_interface upd7759_interface =
{
	REGION_SOUND2 /* memory region */
};

/* Interrupt Generators */

static INTERRUPT_GEN( CPUA_interrupt )
{
	if (CPUA_IRQ_ENABLE) cpunum_set_input_line(cpu_getactivecpu(), 5, HOLD_LINE);
}

static INTERRUPT_GEN( CPUB_interrupt )
{
	if (CPUB_IRQ_ENABLE) cpunum_set_input_line(cpu_getactivecpu(), 5, HOLD_LINE);
}

/* Machine Drivers */

static MACHINE_DRIVER_START( twin16 )
	/* basic machine hardware */
	MDRV_CPU_ADD(Z80, 3579545)
	/* audio CPU */
	MDRV_CPU_PROGRAM_MAP(readmem_sound,writemem_sound)

	MDRV_CPU_ADD(M68000, 10000000)
	MDRV_CPU_PROGRAM_MAP(readmem_sub,writemem_sub)
	MDRV_CPU_VBLANK_INT(CPUB_interrupt,1)

	MDRV_CPU_ADD(M68000, 10000000)
	MDRV_CPU_PROGRAM_MAP(readmem,writemem)
	MDRV_CPU_VBLANK_INT(CPUA_interrupt,1)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_REAL_60HZ_VBLANK_DURATION)
	MDRV_INTERLEAVE(100)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER | VIDEO_BUFFERS_SPRITERAM)
	MDRV_SCREEN_SIZE(40*8, 32*8)
	MDRV_VISIBLE_AREA(0, 40*8-1, 2*8, 30*8-1)
	MDRV_GFXDECODE(gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(0x400)

	MDRV_VIDEO_START(twin16)
	MDRV_VIDEO_UPDATE(twin16)
	MDRV_VIDEO_EOF(twin16)

	/* sound hardware */
	MDRV_SPEAKER_STANDARD_STEREO("left", "right")

	MDRV_SOUND_ADD(YM2151, 7159160/2)
	MDRV_SOUND_ROUTE(0, "left", 1.0)
	MDRV_SOUND_ROUTE(1, "right", 1.0)

	MDRV_SOUND_ADD(K007232, 3579545)
	MDRV_SOUND_CONFIG(k007232_interface)
	MDRV_SOUND_ROUTE(0, "left", 0.20)
	MDRV_SOUND_ROUTE(0, "right", 0.20)
	MDRV_SOUND_ROUTE(1, "left", 0.20)
	MDRV_SOUND_ROUTE(1, "right", 0.20)

	MDRV_SOUND_ADD(UPD7759, UPD7759_STANDARD_CLOCK)
	MDRV_SOUND_CONFIG(upd7759_interface)
	MDRV_SOUND_ROUTE(ALL_OUTPUTS, "left", 0.20)
	MDRV_SOUND_ROUTE(ALL_OUTPUTS, "right", 0.20)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( heavysync )
	MDRV_IMPORT_FROM(twin16)
	MDRV_INTERLEAVE(1000)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( fround )
	/* basic machine hardware */
	MDRV_CPU_ADD(Z80, 3579545)
	/* audio CPU */
	MDRV_CPU_PROGRAM_MAP(readmem_sound,writemem_sound)

	MDRV_CPU_ADD(M68000, 10000000)
	MDRV_CPU_PROGRAM_MAP(fround_readmem,fround_writemem)
	MDRV_CPU_VBLANK_INT(CPUA_interrupt,1)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_REAL_60HZ_VBLANK_DURATION)
	MDRV_INTERLEAVE(100)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER | VIDEO_BUFFERS_SPRITERAM)
	MDRV_SCREEN_SIZE(40*8, 32*8)
	MDRV_VISIBLE_AREA(0, 40*8-1, 2*8, 30*8-1)
	MDRV_GFXDECODE(gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(0x400)

	MDRV_VIDEO_START(fround)
	MDRV_VIDEO_UPDATE(twin16)
	MDRV_VIDEO_EOF(twin16)

	/* sound hardware */
	MDRV_SPEAKER_STANDARD_STEREO("left", "right")

	MDRV_SOUND_ADD(YM2151, 7159160/2)
	MDRV_SOUND_ROUTE(0, "left", 1.0)
	MDRV_SOUND_ROUTE(1, "right", 1.0)

	MDRV_SOUND_ADD(K007232, 3579545)
	MDRV_SOUND_CONFIG(k007232_interface)
	MDRV_SOUND_ROUTE(0, "left", 0.20)
	MDRV_SOUND_ROUTE(0, "right", 0.20)
	MDRV_SOUND_ROUTE(1, "left", 0.20)
	MDRV_SOUND_ROUTE(1, "right", 0.20)

	MDRV_SOUND_ADD(UPD7759, UPD7759_STANDARD_CLOCK)
	MDRV_SOUND_CONFIG(upd7759_interface)
	MDRV_SOUND_ROUTE(ALL_OUTPUTS, "left", 0.20)
	MDRV_SOUND_ROUTE(ALL_OUTPUTS, "right", 0.20)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( hpuncher )
	MDRV_IMPORT_FROM(twin16)
	MDRV_VIDEO_START(fround)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( mia )
	MDRV_IMPORT_FROM(twin16)
	MDRV_VISIBLE_AREA(1*8, 39*8-1, 2*8, 30*8-1)
	MDRV_VIDEO_START(fround)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( cuebrick )
	MDRV_IMPORT_FROM(twin16)
	MDRV_VISIBLE_AREA(1*8, 39*8-1, 2*8, 30*8-1)
	MDRV_VIDEO_START(fround)
	MDRV_NVRAM_HANDLER(generic_0fill)
MACHINE_DRIVER_END

/* ROMs */

ROM_START( devilw )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* Z80 code (sound CPU) */
	ROM_LOAD( "687_l03.10a",		0x00000,  0x8000, CRC(7201983c) SHA1(06d089406d2f702e8d51ffdfbf34e4727a28d506) )

	ROM_REGION( 0x40000, REGION_CPU2, 0 )	/* 68000 code (CPU B) */
	ROM_LOAD16_BYTE( "687_q07.10n",	0x00000, 0x10000, CRC(53110c0b) SHA1(a473486f7c99ab758187376d5bd89581e0b7cbdf) )
	ROM_LOAD16_BYTE( "687_q06.8n",	0x00001, 0x10000, CRC(9c53a0c5) SHA1(3b195a88af2b0198af47e4f7140087bf5a7feee6) )
	ROM_LOAD16_BYTE( "687_q13.10s",	0x20000, 0x10000, CRC(36ae6014) SHA1(102d4c3215fb5f199ce6f93ce92725b5cce0b01d) )
	ROM_LOAD16_BYTE( "687_q12.8s",	0x20001, 0x10000, CRC(6d012167) SHA1(d608e29e0b2e834c099386b6ebb667e4bfc40ce7) )

	ROM_REGION( 0x40000, REGION_CPU3, 0 )	/* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "687_t05.6n",	0x00000, 0x10000, CRC(8ab7dc61) SHA1(de8e93f04f22fa07ea6c379eeae12f8fcb48f751) )
	ROM_LOAD16_BYTE( "687_t04.4n",	0x00001, 0x10000, CRC(c69924da) SHA1(9b7220eac95fd92f32fec92b21f4f02f871c5d43) )
	ROM_LOAD16_BYTE( "687_t09.6r",	0x20000, 0x10000, CRC(fae97de0) SHA1(25b085c9de0943a0112d76447ca0504ed57cad85) )
	ROM_LOAD16_BYTE( "687_t08.4r",	0x20001, 0x10000, CRC(8c898d67) SHA1(0dd1a4467f97c993ea8d40398f46c3fc88c889bd) )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "687_m14.d8",			0x00000,  0x4000, CRC(d7338557) SHA1(9b384baafabaab3888a0139674f0b530303684ca) ) /* characters */

	ROM_REGION16_LE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD( "687i17.p16",	0x000000, 0x80000, CRC(66cb3923) SHA1(3c1fc1dff77201cf8d8c4594c965695066c1701c) )
	ROM_LOAD16_WORD( "687i18.p18",	0x080000, 0x80000, CRC(a1c7d0db) SHA1(901bc6e1982b9a8c2150a802995256d0168cc605) )
	ROM_LOAD16_WORD( "687i15.p13",	0x100000, 0x80000, CRC(eec8c5b2) SHA1(c400dd8c7bb85d233815572acd547acf626e4c73) )
	ROM_LOAD16_WORD( "687i16.p15",	0x180000, 0x80000, CRC(746cf48b) SHA1(8f51df931b1de928f402f51bbaf02e37dfec1d6d) )

	ROM_REGION16_BE( 0x20000, REGION_GFX3, 0 )	/* tile data; mapped at 0x80000 on CPUB */
	ROM_LOAD16_BYTE( "687_l11.10r",	0x00000, 0x10000, CRC(399deee8) SHA1(dcc65e95f28ae4e9b671e70ce0bd5ba0fe178506) )
	ROM_LOAD16_BYTE( "687_l10.8r",	0x00001, 0x10000, CRC(117c91ee) SHA1(dcf8efb25fc73cff916b66b7bcfd3c1fb2556a53) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "687_i01.5a",			0x00000, 0x20000, CRC(d4992dfb) SHA1(c65bef07b6adb9ab6328d679595450945dbf6a88) )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	ROM_LOAD( "687_i02.7c",			0x00000, 0x20000, CRC(e5947501) SHA1(9ec7319bd97711981db60af6f477c2ba2ec7e6b8) )
ROM_END

ROM_START( majuu )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* Z80 code (sound CPU) */
	ROM_LOAD( "687_l03.10a",		0x00000,  0x8000, CRC(7201983c) SHA1(06d089406d2f702e8d51ffdfbf34e4727a28d506) )

	ROM_REGION( 0x40000, REGION_CPU2, 0 )	/* 68000 code (CPU B) */
	ROM_LOAD16_BYTE( "687_q07.10n",	0x00000, 0x10000, CRC(53110c0b) SHA1(a473486f7c99ab758187376d5bd89581e0b7cbdf) )
	ROM_LOAD16_BYTE( "687_q06.8n",	0x00001, 0x10000, CRC(9c53a0c5) SHA1(3b195a88af2b0198af47e4f7140087bf5a7feee6) )
	ROM_LOAD16_BYTE( "687_q13.10s",	0x20000, 0x10000, CRC(36ae6014) SHA1(102d4c3215fb5f199ce6f93ce92725b5cce0b01d) )
	ROM_LOAD16_BYTE( "687_q12.8s",	0x20001, 0x10000, CRC(6d012167) SHA1(d608e29e0b2e834c099386b6ebb667e4bfc40ce7) )

	ROM_REGION( 0x40000, REGION_CPU3, 0 )	/* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "687_s05.6n",	0x00000, 0x10000, CRC(bd99b434) SHA1(4dbcf3a57ff0b4c3a160e2a60c2ee3fd38f5a95d) )
	ROM_LOAD16_BYTE( "687_s04.4n",	0x00001, 0x10000, CRC(3df732e2) SHA1(32e42edbc6f9dfba6ff881c72b8b014b6905e7f3) )
	ROM_LOAD16_BYTE( "687_s09.6r",	0x20000, 0x10000, CRC(1f6efec3) SHA1(4177d300877e3ffcb481f534ef7b3e4bdc160be0) )
	ROM_LOAD16_BYTE( "687_s08.4r",	0x20001, 0x10000, CRC(8a16c8c6) SHA1(df3af1c137632eab98b1eea538cc4ef5b134f44f) )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "687_l14.d8",			0x00000,  0x4000, CRC(20ecccd6) SHA1(b8ac3186de5ea81ae1c64b9511b7a0718aabab48) ) /* characters */

	ROM_REGION16_LE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD( "687i17.p16",	0x000000, 0x80000, CRC(66cb3923) SHA1(3c1fc1dff77201cf8d8c4594c965695066c1701c) )
	ROM_LOAD16_WORD( "687i18.p18",	0x080000, 0x80000, CRC(a1c7d0db) SHA1(901bc6e1982b9a8c2150a802995256d0168cc605) )
	ROM_LOAD16_WORD( "687i15.p13",	0x100000, 0x80000, CRC(eec8c5b2) SHA1(c400dd8c7bb85d233815572acd547acf626e4c73) )
	ROM_LOAD16_WORD( "687i16.p15",	0x180000, 0x80000, CRC(746cf48b) SHA1(8f51df931b1de928f402f51bbaf02e37dfec1d6d) )

	ROM_REGION16_BE( 0x20000, REGION_GFX3, 0 )	/* tile data; mapped at 0x80000 on CPUB */
	ROM_LOAD16_BYTE( "687_l11.10r",	0x00000, 0x10000, CRC(399deee8) SHA1(dcc65e95f28ae4e9b671e70ce0bd5ba0fe178506) )
	ROM_LOAD16_BYTE( "687_l10.8r",	0x00001, 0x10000, CRC(117c91ee) SHA1(dcf8efb25fc73cff916b66b7bcfd3c1fb2556a53) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "687_i01.5a",			0x00000, 0x20000, CRC(d4992dfb) SHA1(c65bef07b6adb9ab6328d679595450945dbf6a88) )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	ROM_LOAD( "687_i02.7c",			0x00000, 0x20000, CRC(e5947501) SHA1(9ec7319bd97711981db60af6f477c2ba2ec7e6b8) )
ROM_END

ROM_START( darkadv )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* Z80 code (sound CPU) */
	ROM_LOAD( "687_n03.10a",		0x00000,  0x8000, CRC(a24c682f) SHA1(cf053270d3f77448ff802db832598ae3cf7dae6c) )

	ROM_REGION( 0x40000, REGION_CPU2, 0 )	/* 68000 code (CPU B) */
	ROM_LOAD16_BYTE( "687_n07.10n",	0x00000, 0x10000, CRC(6154322a) SHA1(e84c75b4d346ade2cfb5d56454282fafb04c2d77) )
	ROM_LOAD16_BYTE( "687_n06.8n",	0x00001, 0x10000, CRC(37a72e8b) SHA1(574d655ba7cda2f8eb628155fb8662a4ee5f23e5) )
	ROM_LOAD16_BYTE( "687_n13.10s",	0x20000, 0x10000, CRC(f1c252af) SHA1(dbe78352ae5b284870becc92139773f66c52ed7d) )
	ROM_LOAD16_BYTE( "687_n12.8s",	0x20001, 0x10000, CRC(da221944) SHA1(c288ee583414a077f0d8b815b8e01b7f19a5fafe) )

	ROM_REGION( 0x40000, REGION_CPU3, 0 )	/* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "687_n05.6n",	0x00000, 0x10000, CRC(a9195b0b) SHA1(a5b30c93dc962b5aef33bab7e257ed606f00c9f2) )
	ROM_LOAD16_BYTE( "687_n04.4n",	0x00001, 0x10000, CRC(65b55105) SHA1(ef102eebf34cceebafb55b04003ab9173af43458) )
	ROM_LOAD16_BYTE( "687_n09.6r",	0x20000, 0x10000, CRC(1c6b594c) SHA1(06e4eab7f544170ac40e8c77b9ebb236a6094367) )
	ROM_LOAD16_BYTE( "687_n08.4r",	0x20001, 0x10000, CRC(a9603196) SHA1(a3ab00a446531d370809d8e405cabe2d74cb612b) )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "687_n14.d8",	0x0000,  0x4000, CRC(c76ac6d2) SHA1(d8fec255f1f7177a3716a5894fb679cbe172b6ea) ) /* characters */

	ROM_REGION16_LE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD( "687i17.p16",	0x000000, 0x80000, CRC(66cb3923) SHA1(3c1fc1dff77201cf8d8c4594c965695066c1701c) )
	ROM_LOAD16_WORD( "687i18.p18",	0x080000, 0x80000, CRC(a1c7d0db) SHA1(901bc6e1982b9a8c2150a802995256d0168cc605) )
	ROM_LOAD16_WORD( "687i15.p13",	0x100000, 0x80000, CRC(eec8c5b2) SHA1(c400dd8c7bb85d233815572acd547acf626e4c73) )
	ROM_LOAD16_WORD( "687i16.p15",	0x180000, 0x80000, CRC(746cf48b) SHA1(8f51df931b1de928f402f51bbaf02e37dfec1d6d) )

	ROM_REGION16_BE( 0x20000, REGION_GFX3, 0 ) /* tile data; mapped at 0x80000 on CPUB */
	ROM_LOAD16_BYTE( "687_l11.10r",	0x00000, 0x10000, CRC(399deee8) SHA1(dcc65e95f28ae4e9b671e70ce0bd5ba0fe178506) )
	ROM_LOAD16_BYTE( "687_l10.8r",	0x00001, 0x10000, CRC(117c91ee) SHA1(dcf8efb25fc73cff916b66b7bcfd3c1fb2556a53) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "687_i01.5a",			0x00000, 0x20000, CRC(d4992dfb) SHA1(c65bef07b6adb9ab6328d679595450945dbf6a88) )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	ROM_LOAD( "687_i02.7c",			0x00000, 0x20000, CRC(e5947501) SHA1(9ec7319bd97711981db60af6f477c2ba2ec7e6b8) )
ROM_END

ROM_START( vulcan )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* Z80 code (sound CPU) */
	ROM_LOAD( "785_g03.10a",		0x00000,  0x8000, CRC(67a3b50d) SHA1(3c83f3b0df73d9361ec3cda26a6c4c603a088419) )

	ROM_REGION( 0x40000, REGION_CPU2, 0 )	/* 68000 code (CPU B) */
	ROM_LOAD16_BYTE( "785_p07.10n",	0x00000, 0x10000, CRC(686d549d) SHA1(9687be801c4fb963bf6b0199e2ae9f5051213f7a) )
	ROM_LOAD16_BYTE( "785_p06.8n" ,	0x00001, 0x10000, CRC(70c94bee) SHA1(951e00ca4d3a47a21b4db05bcdc8ead45b42c3f1) )
	ROM_LOAD16_BYTE( "785_p13.10s",	0x20000, 0x10000, CRC(478fdb0a) SHA1(2e285ad6dcfc67f3e24d231e0e1be19036ce64d2) )
	ROM_LOAD16_BYTE( "785_p12.8s",	0x20001, 0x10000, CRC(38ea402a) SHA1(90ff2bd71435988cde967704ce3b1401de206683) )

	ROM_REGION( 0x40000, REGION_CPU3, 0 )	/* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "785_w05.6n",	0x00000, 0x10000, CRC(6e0e99cd) SHA1(f9f4ef617aee83397b0ac98a5cc92f60d3a3d5ed) )
	ROM_LOAD16_BYTE( "785_w04.4n",	0x00001, 0x10000, CRC(23ec74ca) SHA1(901c7c798b113232f20e24f61940ed6ef5148599) )
	ROM_LOAD16_BYTE( "785_w09.6r",	0x20000, 0x10000, CRC(377e4f28) SHA1(d3af2e52c38b5d618cac2e66df99ab7950638fbc) )
	ROM_LOAD16_BYTE( "785_w08.4r",	0x20001, 0x10000, CRC(813d41ea) SHA1(58ba0e4b1afec3e5674286c44da14244591f1312) )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "785_h14.d8",			0x0000,	0x4000, CRC(02f4b16f) SHA1(45addc99f520770f38c6aa69aef9af59cfc410b5) ) /* characters */

	ROM_REGION16_BE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD( "785f17.p16",	0x000000, 0x80000, CRC(8fbec1a4) SHA1(71cbdb25470b5a2ca7e80c92f12fc47e28085793) )
	ROM_LOAD16_WORD( "785f18.p18",	0x080000, 0x80000, CRC(50d61e38) SHA1(051d332d93d435d19571e22d69ffc5395dbb16a4) )
	ROM_LOAD16_WORD( "785f15.p13",	0x100000, 0x80000, CRC(af96aef3) SHA1(a58e26a8b134ab6e7938cd4fa342c04c236efd99) )
	ROM_LOAD16_WORD( "785f16.p15",	0x180000, 0x80000, CRC(b858df1f) SHA1(c434da7ba54d59643fe6ca30bceccf16413904c3) )

	ROM_REGION( 0x20000, REGION_GFX3, 0 )	/* tile data; mapped at 0x80000 on CPUB */

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "785_f01.5a",			0x00000, 0x20000, CRC(a0d8d69e) SHA1(2994e5740b7c099d55fb162a363a26ef1995c756) )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	ROM_LOAD( "785_f02.7c",			0x00000, 0x20000, CRC(c39f5ca4) SHA1(92ebea11b7d3a72d0f58c456f16a6ff36d6b3062) )
ROM_END

ROM_START( gradius2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* Z80 code (sound CPU) */
	ROM_LOAD( "785_g03.10a",		0x00000,  0x8000, CRC(67a3b50d) SHA1(3c83f3b0df73d9361ec3cda26a6c4c603a088419) )

	ROM_REGION( 0x40000, REGION_CPU2, 0 )	/* 68000 code (CPU B) */
	ROM_LOAD16_BYTE( "785_p07.10n",	0x00000, 0x10000, CRC(686d549d) SHA1(9687be801c4fb963bf6b0199e2ae9f5051213f7a) )
	ROM_LOAD16_BYTE( "785_p06.8n" ,	0x00001, 0x10000, CRC(70c94bee) SHA1(951e00ca4d3a47a21b4db05bcdc8ead45b42c3f1) )
	ROM_LOAD16_BYTE( "785_p13.10s",	0x20000, 0x10000, CRC(478fdb0a) SHA1(2e285ad6dcfc67f3e24d231e0e1be19036ce64d2) )
	ROM_LOAD16_BYTE( "785_p12.8s",	0x20001, 0x10000, CRC(38ea402a) SHA1(90ff2bd71435988cde967704ce3b1401de206683) )

	ROM_REGION( 0x40000, REGION_CPU3, 0 )	/* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "785_x05.6n",	0x00000, 0x10000, CRC(8a23a7b8) SHA1(9846272c011715b17f146d5265a09e790d6e37cd) )
	ROM_LOAD16_BYTE( "785_x04.4n",	0x00001, 0x10000, CRC(88e466ce) SHA1(3a7d098b851bcf6dc8e7ff601d11563406c56666) )
	ROM_LOAD16_BYTE( "785_x09.6r",	0x20000, 0x10000, CRC(3f3d7d7a) SHA1(8e3acf980179e749ec642b6ff211c4ba57c6c592) )
	ROM_LOAD16_BYTE( "785_x08.4r",	0x20001, 0x10000, CRC(c39c8efd) SHA1(372dbb254525a7334fe79a42ba9851eafa856847) )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "785_g14.d8",			0x0000, 0x4000, CRC(9dcdad9d) SHA1(22f457408b453a71b7e91974aee1b1e735ff887f) ) /* characters */

	ROM_REGION16_BE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD( "785f17.p16",	0x000000, 0x80000, CRC(8fbec1a4) SHA1(71cbdb25470b5a2ca7e80c92f12fc47e28085793) )
	ROM_LOAD16_WORD( "785f18.p18",	0x080000, 0x80000, CRC(50d61e38) SHA1(051d332d93d435d19571e22d69ffc5395dbb16a4) )
	ROM_LOAD16_WORD( "785f15.p13",	0x100000, 0x80000, CRC(af96aef3) SHA1(a58e26a8b134ab6e7938cd4fa342c04c236efd99) )
	ROM_LOAD16_WORD( "785f16.p15",	0x180000, 0x80000, CRC(b858df1f) SHA1(c434da7ba54d59643fe6ca30bceccf16413904c3) )

	ROM_REGION( 0x20000, REGION_GFX3, 0 )	/* tile data; mapped at 0x80000 on CPUB */

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "785_f01.5a",			0x00000, 0x20000, CRC(a0d8d69e) SHA1(2994e5740b7c099d55fb162a363a26ef1995c756) )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	ROM_LOAD( "785_f02.7c",			0x00000, 0x20000, CRC(c39f5ca4) SHA1(92ebea11b7d3a72d0f58c456f16a6ff36d6b3062) )
ROM_END

ROM_START( grdius2a )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* Z80 code (sound CPU) */
	ROM_LOAD( "785_g03.10a",		0x00000,  0x8000, CRC(67a3b50d) SHA1(3c83f3b0df73d9361ec3cda26a6c4c603a088419) )

	ROM_REGION( 0x40000, REGION_CPU2, 0 )	/* 68000 code (CPU B) */
	ROM_LOAD16_BYTE( "785_p07.10n",	0x00000, 0x10000, CRC(686d549d) SHA1(9687be801c4fb963bf6b0199e2ae9f5051213f7a) )
	ROM_LOAD16_BYTE( "785_p06.8n" ,	0x00001, 0x10000, CRC(70c94bee) SHA1(951e00ca4d3a47a21b4db05bcdc8ead45b42c3f1) )
	ROM_LOAD16_BYTE( "785_p13.10s",	0x20000, 0x10000, CRC(478fdb0a) SHA1(2e285ad6dcfc67f3e24d231e0e1be19036ce64d2) )
	ROM_LOAD16_BYTE( "785_p12.8s",	0x20001, 0x10000, CRC(38ea402a) SHA1(90ff2bd71435988cde967704ce3b1401de206683) )

	ROM_REGION( 0x40000, REGION_CPU3, 0 )	/* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "785_p05.6n",	0x00000, 0x10000, CRC(4db0e736) SHA1(45edf094da448d039af18a7b47a592a3de02f17b) )
	ROM_LOAD16_BYTE( "785_p04.4n",	0x00001, 0x10000, CRC(765b99e6) SHA1(4acae0cf2bf23f8a1cf7b650b018d22a9ae947c9) )
	ROM_LOAD16_BYTE( "785_t09.6r",	0x20000, 0x10000, CRC(4e3f4965) SHA1(130da6db0590779948a29fd5010f0e628e4f2ee6) )
	ROM_LOAD16_BYTE( "785_j08.4r",	0x20001, 0x10000, CRC(2b1c9108) SHA1(3d367a1b60f95798b478dc4a84daa9b362271d4d) )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "785_g14.d8",			0x0000, 0x4000, CRC(9dcdad9d) SHA1(22f457408b453a71b7e91974aee1b1e735ff887f) ) /* characters */

	ROM_REGION16_BE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD( "785f17.p16",	0x000000, 0x80000, CRC(8fbec1a4) SHA1(71cbdb25470b5a2ca7e80c92f12fc47e28085793) )
	ROM_LOAD16_WORD( "785f18.p18",	0x080000, 0x80000, CRC(50d61e38) SHA1(051d332d93d435d19571e22d69ffc5395dbb16a4) )
	ROM_LOAD16_WORD( "785f15.p13",	0x100000, 0x80000, CRC(af96aef3) SHA1(a58e26a8b134ab6e7938cd4fa342c04c236efd99) )
	ROM_LOAD16_WORD( "785f16.p15",	0x180000, 0x80000, CRC(b858df1f) SHA1(c434da7ba54d59643fe6ca30bceccf16413904c3) )

	ROM_REGION( 0x20000, REGION_GFX3, 0 )	/* tile data; mapped at 0x80000 on CPUB */

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "785_f01.5a",			0x00000, 0x20000, CRC(a0d8d69e) SHA1(2994e5740b7c099d55fb162a363a26ef1995c756) )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	ROM_LOAD( "785_f02.7c",			0x00000, 0x20000, CRC(c39f5ca4) SHA1(92ebea11b7d3a72d0f58c456f16a6ff36d6b3062) )
ROM_END

ROM_START( grdius2b )
	ROM_REGION( 0x10000, REGION_CPU1, 0 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "785_g03.10a",		0x00000,  0x8000, CRC(67a3b50d) SHA1(3c83f3b0df73d9361ec3cda26a6c4c603a088419) )

	ROM_REGION( 0x40000, REGION_CPU2, 0 ) /* 68000 code (CPU B) */
	ROM_LOAD16_BYTE( "785_p07.10n",	0x00000, 0x10000, CRC(686d549d) SHA1(9687be801c4fb963bf6b0199e2ae9f5051213f7a) )
	ROM_LOAD16_BYTE( "785_p06.8n" ,	0x00001, 0x10000, CRC(70c94bee) SHA1(951e00ca4d3a47a21b4db05bcdc8ead45b42c3f1) )
	ROM_LOAD16_BYTE( "785_p13.10s",	0x20000, 0x10000, CRC(478fdb0a) SHA1(2e285ad6dcfc67f3e24d231e0e1be19036ce64d2) )
	ROM_LOAD16_BYTE( "785_p12.8s",	0x20001, 0x10000, CRC(38ea402a) SHA1(90ff2bd71435988cde967704ce3b1401de206683) )

	ROM_REGION( 0x40000, REGION_CPU3, 0 ) /* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "785_p05.6n",	0x00000, 0x10000, CRC(4db0e736) SHA1(45edf094da448d039af18a7b47a592a3de02f17b) )
	ROM_LOAD16_BYTE( "785_p04.4n",	0x00001, 0x10000, CRC(765b99e6) SHA1(4acae0cf2bf23f8a1cf7b650b018d22a9ae947c9) )
	ROM_LOAD16_BYTE( "785_j09.6r",	0x20000, 0x10000, CRC(6d96a7e3) SHA1(cc1362f5922cf1e348a58d54d03320f394a2aeab) )
	ROM_LOAD16_BYTE( "785_j08.4r",	0x20001, 0x10000, CRC(2b1c9108) SHA1(3d367a1b60f95798b478dc4a84daa9b362271d4d) )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "785_g14.d8",			0x0000, 0x4000, CRC(9dcdad9d) SHA1(22f457408b453a71b7e91974aee1b1e735ff887f) ) /* characters */

	ROM_REGION16_BE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD( "785f17.p16",	0x000000, 0x80000, CRC(8fbec1a4) SHA1(71cbdb25470b5a2ca7e80c92f12fc47e28085793) )
	ROM_LOAD16_WORD( "785f18.p18",	0x080000, 0x80000, CRC(50d61e38) SHA1(051d332d93d435d19571e22d69ffc5395dbb16a4) )
	ROM_LOAD16_WORD( "785f15.p13",	0x100000, 0x80000, CRC(af96aef3) SHA1(a58e26a8b134ab6e7938cd4fa342c04c236efd99) )
	ROM_LOAD16_WORD( "785f16.p15",	0x180000, 0x80000, CRC(b858df1f) SHA1(c434da7ba54d59643fe6ca30bceccf16413904c3) )

	ROM_REGION( 0x20000, REGION_GFX3, 0 )	/* tile data; mapped at 0x80000 on CPUB */

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "785_f01.5a",			0x00000, 0x20000, CRC(a0d8d69e) SHA1(2994e5740b7c099d55fb162a363a26ef1995c756) )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	ROM_LOAD( "785_f02.7c",			0x00000, 0x20000, CRC(c39f5ca4) SHA1(92ebea11b7d3a72d0f58c456f16a6ff36d6b3062) )
ROM_END

ROM_START( fround )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* Z80 code (sound CPU) */
	ROM_LOAD( "870_f03.10a",		0x00000,  0x8000, CRC(a645c727) SHA1(e173e79130f187d090766664f1c478d47062749c) )

	ROM_REGION( 0x40000, REGION_CPU2, 0 )	/* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "870_m21.bin",	0x00000, 0x20000, CRC(436dbffb) SHA1(be8c8544f4d8ae86f216095753c6178a3cbf0e8d) )
	ROM_LOAD16_BYTE( "870_m20.bin",	0x00001, 0x20000, CRC(b1c79d6a) SHA1(76e95e87eaa96ba694675fde7706540c584a36fb) )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "870_f14.d8",			0x0000, 0x4000, CRC(c9b46615) SHA1(c0cddb1af47b8e0865055624cf4e89a111ac1b0f) ) /* characters */

	ROM_REGION16_LE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD("870c18.p18",	0x000000, 0x80000, CRC(07927fe8) SHA1(0ab5e0e785347fbed5c4b930a32876d6ce2bef4a) )
	ROM_LOAD16_WORD("870c17.p16",	0x080000, 0x80000, CRC(2bc99ff8) SHA1(9a06502317a71ca5662b79aedf47379b8e5434a9) )
	ROM_LOAD16_WORD("870c16.p15",	0x100000, 0x80000, CRC(41df6a1b) SHA1(32e0fdeb53628d18adde851e4496dd01ac6ec68f) )
	ROM_LOAD16_WORD("870c15.p13",	0x180000, 0x80000, CRC(8c9281df) SHA1(5e3d80be414db108d5363d0ea1b74021ba942c33) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "870_c01.5a",			0x00000, 0x20000, CRC(6af96546) SHA1(63b49b28c0f2ef8f52bc4c5955ad6a633dd553cf) )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	ROM_LOAD( "870_c02.7c",			0x00000, 0x20000, CRC(54e12c6d) SHA1(8171441aeb1a1dc4d65fae59c7c217f992f70cbc) )
ROM_END

ROM_START( froundl )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* Z80 code (sound CPU) */
	ROM_LOAD( "870_f03.10a",		0x00000,  0x8000, CRC(a645c727) SHA1(e173e79130f187d090766664f1c478d47062749c) )

	ROM_REGION( 0x40000, REGION_CPU2, 0 )	/* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "870_l21.bin",	0x00000, 0x20000, CRC(e21a3a19) SHA1(5f5793c88093bc8632eab673f5e0d954ac9b9177) )
	ROM_LOAD16_BYTE( "870_l20.bin",	0x00001, 0x20000, CRC(0ce9786f) SHA1(0fb0eef999ed09dd50eed403255887103fc35461) )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "870_f14.d8",			0x0000, 0x4000, CRC(c9b46615) SHA1(c0cddb1af47b8e0865055624cf4e89a111ac1b0f) ) /* characters */

	ROM_REGION16_LE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD("870c18.p18",	0x000000, 0x80000, CRC(07927fe8) SHA1(0ab5e0e785347fbed5c4b930a32876d6ce2bef4a) )
	ROM_LOAD16_WORD("870c17.p16",	0x080000, 0x80000, CRC(2bc99ff8) SHA1(9a06502317a71ca5662b79aedf47379b8e5434a9) )
	ROM_LOAD16_WORD("870c16.p15",	0x100000, 0x80000, CRC(41df6a1b) SHA1(32e0fdeb53628d18adde851e4496dd01ac6ec68f) )
	ROM_LOAD16_WORD("870c15.p13",	0x180000, 0x80000, CRC(8c9281df) SHA1(5e3d80be414db108d5363d0ea1b74021ba942c33) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "870_c01.5a",			0x00000, 0x20000, CRC(6af96546) SHA1(63b49b28c0f2ef8f52bc4c5955ad6a633dd553cf) )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 )	/* samples */
	ROM_LOAD( "870_c02.7c",			0x00000, 0x20000, CRC(54e12c6d) SHA1(8171441aeb1a1dc4d65fae59c7c217f992f70cbc) )
ROM_END

ROM_START( hpuncher )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* Z80 code (sound CPU) */
	ROM_LOAD( "870_g03.10a",		0x00000,  0x8000, CRC(db9c10c8) SHA1(b2ec4d6800a4ab00dfcc8d3cd4abf9b9d7c5544a) )

	ROM_REGION( 0x40000, REGION_CPU2, 0 )	/* 68000 code (CPU B) */
	ROM_LOAD16_BYTE( "870_h07.10n",	0x00000, 0x10000, CRC(b4dda612) SHA1(e9612af11a151de1f0236629c84679f80936cae9) )
	ROM_LOAD16_BYTE( "870_h06.8n",	0x00001, 0x10000, CRC(696ba702) SHA1(e35c9aeab4305d828a347eaddf4abc806a889fa3) )

	ROM_REGION( 0x40000, REGION_CPU3, 0 )	/* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "870_h05.6n",	0x00000, 0x10000, CRC(2bcfeef3) SHA1(8843e795a310e1a79b7b65226d9526197c2144ef) )
	ROM_LOAD16_BYTE( "870_h04.4n",	0x00001, 0x10000, CRC(b9f97fd3) SHA1(d15a659f753f9f475dfd4f7e96789bacc54cb1f9) )
	ROM_LOAD16_BYTE( "870_h09.6r",	0x20000, 0x10000, CRC(96a4f8b1) SHA1(2709f5236235b1b69c3367054fb998c1beb35625) )
	ROM_LOAD16_BYTE( "870_h08.4r",	0x20001, 0x10000, CRC(46d65156) SHA1(6dd9d97a8e3394db668da1fc6a46c166bca1f456) )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "870_f14.d8",			0x0000, 0x4000, CRC(c9b46615) SHA1(c0cddb1af47b8e0865055624cf4e89a111ac1b0f) ) /* characters */

	ROM_REGION16_LE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD("870c17.p16",	0x000000, 0x80000, CRC(2bc99ff8) SHA1(9a06502317a71ca5662b79aedf47379b8e5434a9) )
	ROM_LOAD16_WORD("870c18.p18",	0x080000, 0x80000, CRC(07927fe8) SHA1(0ab5e0e785347fbed5c4b930a32876d6ce2bef4a) )
	ROM_LOAD16_WORD("870c15.p13",	0x100000, 0x80000, CRC(8c9281df) SHA1(5e3d80be414db108d5363d0ea1b74021ba942c33) )
	ROM_LOAD16_WORD("870c16.p15",	0x180000, 0x80000, CRC(41df6a1b) SHA1(32e0fdeb53628d18adde851e4496dd01ac6ec68f) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )	/* samples */
	ROM_LOAD( "870_c01.5a",			0x00000, 0x20000, CRC(6af96546) SHA1(63b49b28c0f2ef8f52bc4c5955ad6a633dd553cf) )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	ROM_LOAD( "870_c02.7c",			0x00000, 0x20000, CRC(54e12c6d) SHA1(8171441aeb1a1dc4d65fae59c7c217f992f70cbc) )
ROM_END

ROM_START( miaj )
	ROM_REGION( 0x10000, REGION_CPU1, 0 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "808_e03.10a",		0x00000,  0x8000, CRC(3d93a7cd) SHA1(dcdd327e78f32436b276d0666f62a5b733b296e8) )

	ROM_REGION( 0x40000, REGION_CPU2, 0 ) /* 68000 code (CPU B) */
	ROM_LOAD16_BYTE("808_e07.10n",	0x00000, 0x10000, CRC(297bdcea) SHA1(41ed650cbc4f56777fa352812e9a560554459f90) )
	ROM_LOAD16_BYTE("808_e06.8n",	0x00001, 0x10000, CRC(8f576b33) SHA1(cd8c9a45626c30571ca12d0241cccc092c0903a0) )
	ROM_LOAD16_BYTE("808_e13.10s",	0x20000, 0x10000, CRC(1fa708f4) SHA1(9511a19f50fb61571c2986c72d1a85e87b8d0495) )
	ROM_LOAD16_BYTE("808_e12.8s",	0x20001, 0x10000, CRC(d62f1fde) SHA1(1e55084f1294b6ac7c152fcd1800511fcab5d360) )

	ROM_REGION( 0x40000, REGION_CPU3, 0 ) /* 68000 code (CPU A) */
	ROM_LOAD16_BYTE("808_r05.6n",	0x00000, 0x10000, CRC(91fd83f4) SHA1(12cba5828d1d752efe4fc1972bed0205ac277850) )
	ROM_LOAD16_BYTE("808_r04.4n",	0x00001, 0x10000, CRC(f1c8c597) SHA1(4e8e87aedc9dac4ed05816989210de49f4dd60d2) )
	ROM_LOAD16_BYTE("808_r09.6r",	0x20000, 0x10000, CRC(f74d4467) SHA1(2b2491ee7ece979f2ec849075b0c30b6c1ed6bae) )
	ROM_LOAD16_BYTE("808_r08.4r",	0x20001, 0x10000, CRC(26f21704) SHA1(123cb2e4bcb25d82faa75e4505f9859c10bca754) )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD("808_e14.d8",			0x0000, 0x4000, CRC(b9d36525) SHA1(53291c4911d7e1a5110539e4c57a11d0d530dc6f) ) /* characters */

	ROM_REGION16_LE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD("808d17.p16",	0x000000, 0x80000, CRC(d1299082) SHA1(c3c07b0517e7428ccd1cdf9e15aaf16d98e7c4cd) )
	ROM_LOAD16_WORD("808d15.p13",	0x100000, 0x80000, CRC(2b22a6b6) SHA1(8e1af0627a4eac045128c4096e2cfb59c3d2f5ef) )

	ROM_REGION( 0x20000, REGION_GFX3, 0 ) /* tile data; mapped at 0x80000 on CPUB */

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD("808_d01.5a",			0x00000, 0x20000, CRC(fd4d37c0) SHA1(ef91c6e7bb57c27a9a51729fffd1bfe3e806fb61) )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
ROM_END

ROM_START( cuebrick )
	ROM_REGION( 0x10000, REGION_CPU1, 0 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "903_d03.10a",		0x00000,  0x8000, CRC(455e855a) SHA1(cfdd54a5071862653ee94c0455301f4a7245fbd8) )

	ROM_REGION( 0x40000, REGION_CPU2, 0 ) /* 68000 code (CPU B) */
	ROM_LOAD16_BYTE( "903_d07.10n",	0x00000, 0x10000, CRC(fc0edce7) SHA1(57e969a42c48a9464ec370d4605eaecb690d58ab) )
	ROM_LOAD16_BYTE( "903_d06.8n",	0x00001, 0x10000, CRC(b2cef6fe) SHA1(27d1d15f3a408e6166745cb6a159704a8cbb4584) )
	ROM_LOAD16_BYTE( "903_e13.10s",	0x20000, 0x10000, CRC(4fb5fb80) SHA1(3a59dae3846341289c31aa106684ebc45488ca45) )
	ROM_LOAD16_BYTE( "903_e12.8s",	0x20001, 0x10000, CRC(883e3097) SHA1(fe0fa1a2881a67223d741c400bb8c1a0c67946c1) )

	ROM_REGION( 0x40000, REGION_CPU3, 0 ) /* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "903_e05.6n",	0x00000, 0x10000, CRC(8b556220) SHA1(dbe24133e74018c4fe9332519394cbb882c4ed5a) )
	ROM_LOAD16_BYTE( "903_e04.4n",	0x00001, 0x10000, CRC(bf9c7927) SHA1(3a594b8846f7e6074ca54f8cd5fe2ba3b64ba740) )
	ROM_LOAD16_BYTE( "903_e09.6r",	0x20000, 0x10000, CRC(2a77554d) SHA1(53eee99737333c8c4bbb9c757fd12488b6c4d3d8) )
	ROM_LOAD16_BYTE( "903_e08.4r",	0x20001, 0x10000, CRC(c0a430c1) SHA1(bef737f12e9a79781a4cd795ffc108899757901d) )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "903_e14.d8",			0x0000, 0x4000, CRC(ddbebbd5) SHA1(df532f19a08cabb28d693e4cd445c47c9e35e5ff) ) /* characters */

	ROM_REGION( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	/* unpopulated */

	ROM_REGION16_BE( 0x20000, REGION_GFX3, 0 ) /* tile data; mapped at 0x80000 on CPUB */
	ROM_LOAD16_BYTE( "903_e11.10r",	0x00000, 0x10000, CRC(5c41faf8) SHA1(f9eee6a7b92d3b3aa4320747da6390310522a2cf) )
	ROM_LOAD16_BYTE( "903_e10.8r",	0x00001, 0x10000, CRC(417576d4) SHA1(e84762743e3a1117b6ef7ea0b304877e4a719f75) )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	/* unpopulated */

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	/* unpopulated */
ROM_END

/* Driver Initialization */

static void gfx_untangle( void )
{
	/* sprite, tile data */

	int i;
	UINT16 *temp = malloc(0x200000);

	if ( temp )
	{
		twin16_gfx_rom = (UINT16 *)memory_region(REGION_GFX2);
		memcpy( temp, twin16_gfx_rom, 0x200000 );

		for( i=0; i<0x080000; i++ )
		{
			twin16_gfx_rom[i*2+0] = temp[i+0x080000];
			twin16_gfx_rom[i*2+1] = temp[i];
		}
		free( temp );
	}
}

static DRIVER_INIT( twin16 )
{
	gfx_untangle();
	twin16_custom_vidhrdw = 0;
}

static DRIVER_INIT( fround )
{
	gfx_untangle();
	twin16_custom_vidhrdw = 1;
}

static DRIVER_INIT( hpuncher )
{
	gfx_untangle();
	twin16_custom_vidhrdw = 2;
}

static DRIVER_INIT( cuebrick )
{
	gfx_untangle();
	twin16_custom_vidhrdw = 2;

	generic_nvram = (UINT8 *)cuebrick_nvram;
	generic_nvram_size = 0x400*0x20;
}

/* Game Drivers */

GAME( 1987, devilw,   0,      heavysync, devilw,   twin16,   ROT0, "Konami", "Devil World" )
GAME( 1987, majuu,    devilw, heavysync, devilw,   twin16,   ROT0, "Konami", "Majuu no Ohkoku" )
GAME( 1987, darkadv,  devilw, heavysync, darkadv,  twin16,   ROT0, "Konami", "Dark Adventure" )
GAME( 1988, vulcan,   0,      twin16,    vulcan,   twin16,   ROT0, "Konami", "Vulcan Venture" )
GAME( 1988, gradius2, vulcan, twin16,    gradius2, twin16,   ROT0, "Konami", "Gradius II - GOFER no Yabou (Japan New Ver.)" )
GAME( 1988, grdius2a, vulcan, twin16,    vulcan,   twin16,   ROT0, "Konami", "Gradius II - GOFER no Yabou (Japan Old Ver.)" )
GAME( 1988, grdius2b, vulcan, twin16,    vulcan,   twin16,   ROT0, "Konami", "Gradius II - GOFER no Yabou (Japan Older Ver.)" )

GAME( 1988, fround,   0,      fround,    fround,   fround,   ROT0, "Konami", "The Final Round (version M)" )
GAME( 1988, froundl,  fround, fround,    fround,   fround,   ROT0, "Konami", "The Final Round (version L)" )
GAME( 1988, hpuncher, fround, hpuncher,  fround,   hpuncher, ROT0, "Konami", "Hard Puncher (Japan)" )
GAME( 1989, miaj,     mia,    mia,       miaj,     hpuncher, ROT0, "Konami", "M.I.A. - Missing in Action (Japan)" )
GAME( 1989, cuebrick, 0,      cuebrick,  cuebrick, cuebrick, ROT0, "Konami", "Cue Brick (Japan)" )
