/***************************************************************************

    Taito Qix hardware

    driver by John Butler, Ed Mueller, Aaron Giles

    Games supported:
        * Qix (3 sets)
        * Qix II Tournament
        * Zookeeper (3 sets)
        * Space Dungeon
        * Kram (3 sets, one encrypted)
        * The Electric Yo-Yo (2 sets)
        * Slither (2 sets)
        * Complex X

    Known bugs:
        * none at this time

    Notes:
        * the encrypted version of Kram doesn't have the protection MCU
        * it's not clear what the 16R4 PAL does.
        * Electric Yo-Yo was hanging in attract mode because of a synchronization issue.
          This was fixed by increasing the interleave to 20.
          The reason for the hang was that CPU #2 was reading location $8013 of shared
          RAM after clearing it and before CPU #1 had put the correct value in it.

    TODO:
        * The Kram encryption algorithm is not understood. I merely provide tables to
          decrypt it, derived by comparison with the not encrypted versions.
        * I applied the interleave change only to elecyoyo because these games are
          really sensitive to timing, and kram's service mode was disturbed by it (the
          high score and audit pages were broken).

****************************************************************************

    Memory map

****************************************************************************

Video CPU:
----------
All but Zookeeper:

Address          Dir Data     Name        Description
---------------- --- -------- ----------- -----------------------
100000xxxxxxxxxx R/W xxxxxxxx DS0         dual port RAM (shared with video CPU)
100001xxxxxxxxxx R/W xxxxxxxx             local RAM
100010---------x R/W xxxxxxxx DS2         6850 ACIA [1]
100011---------0 R/W -------- DS3         assert FIRQ on video CPU
100011---------1 R/W -------- DS3         FIRQ acknowledge
100100--------xx R/W xxxxxxxx DS4/U20     6821 PIA (sound control / data IRQ)
100101--------xx R/W xxxxxxxx DS5/U11     6821 PIA (coin / player 1 inputs)
100110-1------xx R/W xxxxxxxx DS6/U20     6821 PIA (spare / player 2 inputs)
100110xxxxxxxx-- R/W ----xxxx DS6/U24     PAL 16R4 (purpose unclear)
100111--------xx R/W xxxxxxxx DS7/U30     6821 PIA (player 2 inputs / coin control)
101xxxxxxxxxxxxx R   xxxxxxxx             program ROM
11xxxxxxxxxxxxxx R   xxxxxxxx             program ROM

Zookeeper is almost the same, just the top bit of the address is inverted most of
the time to make space for more ROM.

Address          Dir Data     Name        Description
---------------- --- -------- ----------- -----------------------
000000xxxxxxxxxx R/W xxxxxxxx DS0         dual port RAM (shared with video CPU)
000001xxxxxxxxxx R/W xxxxxxxx             local RAM
000010---------x R/W xxxxxxxx DS2         6850 ACIA [1]
000011---------0 R/W -------- DS3         assert FIRQ on video CPU
000011---------1 R/W -------- DS3         FIRQ acknowledge
000100--------xx R/W xxxxxxxx DS4/U20     6821 PIA (sound control / data IRQ)
000101--------xx R/W xxxxxxxx DS5/U11     6821 PIA (coin / player 1 inputs)
000110-1------xx R/W xxxxxxxx DS6/U20     6821 PIA (spare / player 2 inputs)
100110xxxxxxxx-- R/W ----xxxx DS6/U24     PAL 16R4 (purpose unclear)
000111--------xx R/W xxxxxxxx DS7/U30     6821 PIA (player 2 inputs / coin control)
1xxxxxxxxxxxxxxx R   xxxxxxxx             program ROM

[1] Diagnostic purposes only, not used

Interrupts:
    NMI not connected
    IRQ generated by pia_3
    FIRQ generated by CPU #2


Video CPU:
----------

Address          Dir Data     Name        Description
---------------- --- -------- ----------- -----------------------
0xxxxxxxxxxxxxxx R/W xxxxxxxx             direct video RAM access
100000xxxxxxxxxx R/W xxxxxxxx VS0         dual port RAM (shared with data CPU)
100001xxxxxxxxxx R/W xxxxxxxx VS1         CMOS NVRAM
100010----------   W xxxxxx-- VS2         self test LEDs      [1]
100010----------   W ------xx VS2         palette bank select [1]
100011---------0 R/W -------- VS3         assert FIRQ on data CPU
100011---------1 R/W -------- VS3         FIRQ acknowledge
100100xxxxxxxxxx R/W xxxxxxxx VS4         palette RAM (RRGGBBII)
100101--------00 R/W xxxxxxxx VS5         video RAM access at latched address
100101--------01 R/W xxxxxxxx             video RAM access mask [2]
100101--------1x   W xxxxxxxx VS5         video RAM address latch
100110---------- R   xxxxxxxx VS6         current scanline
100111---------x R/W xxxxxxxx VS7         68A45 video controller
11xxxxxxxxxxxxxx R   xxxxxxxx             program ROMs

Zookeeper has some more ROMs (banked):
100010--------01   W -----x--             ROM bank select [1]
101xxxxxxxxxxxxx R   xxxxxxxx             banked program ROMs

[1] The Zookeeper extension is handled in the ROM expansion board, without
    changes to the main board. This means that the ROM bank select address is
    also a mirror for the LEDs/palette bank latch. That's why they used bit 2,
    this way they could copy the palette bank in the low two bits when changing
    the ROM bank and avoid problems.
[2] Slither only. This hardware feature is NOT present in the Taito games.

Interrupts:
    NMI not connected
    IRQ not connected
    FIRQ generated by CPU #1


Sound CPU:
----------
Slither doesn't have the third CPU and has different sound hardware.

Address          Dir Data     Name        Description
---------------- --- -------- ----------- -----------------------
000000000xxxxxxx R/W xxxxxxxx             6802 internal RAM
0-1-----------xx R/W xxxxxxxx U7          6821 PIA (TMS5200 control) [2]
01------------xx R/W xxxxxxxx U8          6821 PIA (DAC, communication with data CPU)
1100------------                          n.c.
1101xxxxxxxxxxxx R   xxxxxxxx U25         program ROM [1]
1110xxxxxxxxxxxx R   xxxxxxxx U26         program ROM [1]
1111xxxxxxxxxxxx R   xxxxxxxx U27         program ROM [1]

[1] In older games (qix, kram, elecyoyo) ROMs are half size and mapped at E800-FFFF
[2] Not used by any game

Interrupts:
    NMI not connected
    IRQ generated by pia_4

****************************************************************************

    Qix uses two 6809 CPUs: one for data and sound and the other for
    video. Communication between the two CPUs is done using a 4K RAM
    space at $8000 (for ZooKeeper the data cpu maps it at $0000 and the
    video cpu at $8000) which both CPUs have direct access. FIRQs (fast
    interrupts) are generated by each CPU to interrupt the other at
    specific times.

    A third CPU, a 6802, is used for sample playback. It drives an 8-bit
    DAC and according to the schematics a TMS5220 speech chip, which is
    never accessed. ROM u27 is the only code needed. A sound command from
    the data CPU causes an IRQ to fire on the 6802 and the sound playback
    is started.

    The coin door switches and player controls are connected to the CPUs
    by Mototola 6821 PIAs. These devices are memory mapped as shown below.

    The screen is 256x256 with eight bit pixels (64K). The screen is
    divided into two halves each half mapped by the video CPU at
    $0000-$7FFF. The high order bit of the address latch at $9402
    specifies which half of the screen is being accessed.

    Timing is critical in the hardware. Communications between the data,
    video, and sound CPUs are all very sensitive.

    The address latch works as follows. When the video CPU accesses $9400,
    the screen address is computed by using the values at $9402 (high byte)
    and $9403 (low byte) to get a value between $0000-$FFFF. The value at
    that location is either returned or written.

    The scan line at $9800 on the video CPU records where the scan line is
    on the display (0-255). Several places in the ROM code wait until the
    scan line reaches zero before continuing.

****************************************************************************

    QIX NONVOLATILE CMOS MEMORY MAP (CPU #2 -- Video) $8400-$87ff
        $86A9 - $86AA:  When CMOS is valid, these bytes are $55AA
        $86AC - $86C3:  AUDIT TOTALS -- 4 4-bit BCD digits per setting
                        (All totals default to: 0000)
                        $86AC: TOTAL PAID CREDITS
                        $86AE: LEFT COINS
                        $86B0: CENTER COINS
                        $86B2: RIGHT COINS
                        $86B4: PAID CREDITS
                        $86B6: AWARDED CREDITS
                        $86B8: % FREE PLAYS
                        $86BA: MINUTES PLAYED
                        $86BC: MINUTES AWARDED
                        $86BE: % FREE TIME
                        $86C0: AVG. GAME [SEC]
                        $86C2: HIGH SCORES
        $86C4 - $86FF:  High scores -- 10 scores/names, consecutive in memory
                        Six 4-bit BCD digits followed by 3 ascii bytes
                        (Default: 030000 QIX)
        $8700        :  LANGUAGE SELECT (Default: $32)
                        ENGLISH = $32  FRANCAIS = $33  ESPANOL = $34  DEUTSCH = $35
        $87D9 - $87DF:  COIN SLOT PROGRAMMING -- 2 4-bit BCD digits per setting
                        $87D9: STANDARD COINAGE SETTING  (Default: 01)
                        $87DA: COIN MULTIPLIERS LEFT (Default: 01)
                        $87DB: COIN MULTIPLIERS CENTER (Default: 04)
                        $87DC: COIN MULTIPLIERS RIGHT (Default: 01)
                        $87DD: COIN UNITS FOR CREDIT (Default: 01)
                        $87DE: COIN UNITS FOR BONUS (Default: 00)
                        $87DF: MINIMUM COINS (Default: 00)
        $87E0 - $87EA:  LOCATION PROGRAMMING -- 2 4-bit BCD digits per setting
                        $87E0: BACKUP HSTD [0000] (Default: 03)
                        $87E1: MAXIMUM CREDITS (Default: 10)
                        $87E2: NUMBER OF TURNS (Default: 03)
                        $87E3: THRESHOLD (Default: 75)
                        $87E4: TIME LINE (Default: 37)
                        $87E5: DIFFICULTY 1 (Default: 01)
                        $87E6: DIFFICULTY 2 (Default: 01)
                        $87E7: DIFFICULTY 3 (Default: 01)
                        $87E8: DIFFICULTY 4 (Default: 01)
                        $87E9: ATTRACT SOUND (Default: 01)
                        $87EA: TABLE MODE (Default: 00)

***************************************************************************/

#include "driver.h"
#include "qix.h"
#include "machine/6821pia.h"
#include "vidhrdw/generic.h"
#include "sound/sn76496.h"
#include "sound/discrete.h"


/* Constants */
#define MAIN_CLOCK_OSC		20000000	/* 20 MHz */
#define SLITHER_CLOCK_OSC	21300000	/* 21.3 MHz */
#define SOUND_CLOCK_OSC		7372800		/* 7.3728 MHz */
#define COIN_CLOCK_OSC		4000000		/* 4 MHz */



/*************************************
 *
 *  Data CPU memory handlers
 *
 *************************************/

static ADDRESS_MAP_START( readmem_data, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x8000, 0x83ff) AM_READ(qix_sharedram_r)
	AM_RANGE(0x8400, 0x87ff) AM_READ(MRA8_RAM)
	AM_RANGE(0x8800, 0x8800) AM_READ(MRA8_NOP)   /* ACIA */
	AM_RANGE(0x8c00, 0x8c00) AM_READ(qix_video_firq_r)
	AM_RANGE(0x8c01, 0x8c01) AM_READ(qix_data_firq_ack_r)
	AM_RANGE(0x9000, 0x93ff) AM_READ(pia_3_r)
	AM_RANGE(0x9400, 0x97ff) AM_READ(pia_0_r)
	AM_RANGE(0x9900, 0x99ff) AM_READ(pia_1_r)
	AM_RANGE(0x9c00, 0x9fff) AM_READ(pia_2_r)
	AM_RANGE(0xa000, 0xffff) AM_READ(MRA8_ROM)
ADDRESS_MAP_END


static ADDRESS_MAP_START( writemem_data, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x8000, 0x83ff) AM_WRITE(qix_sharedram_w) AM_BASE(&qix_sharedram)
	AM_RANGE(0x8400, 0x87ff) AM_WRITE(MWA8_RAM)
	AM_RANGE(0x8c00, 0x8c00) AM_WRITE(qix_video_firq_w)
	AM_RANGE(0x8c01, 0x8c01) AM_WRITE(qix_data_firq_ack_w)
	AM_RANGE(0x9000, 0x93ff) AM_WRITE(pia_3_w)
	AM_RANGE(0x9400, 0x97ff) AM_WRITE(qix_pia_0_w)
	AM_RANGE(0x9900, 0x99ff) AM_WRITE(pia_1_w)
	AM_RANGE(0x9c00, 0x9fff) AM_WRITE(pia_2_w)
	AM_RANGE(0xa000, 0xffff) AM_WRITE(MWA8_ROM)
ADDRESS_MAP_END


static ADDRESS_MAP_START( zoo_readmem_data, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x03ff) AM_READ(qix_sharedram_r)
	AM_RANGE(0x0400, 0x07ff) AM_READ(MRA8_RAM)
	AM_RANGE(0x0800, 0x0800) AM_READ(MRA8_NOP)   /* ACIA */
	AM_RANGE(0x0c00, 0x0c00) AM_READ(qix_video_firq_r)
	AM_RANGE(0x0c01, 0x0c01) AM_READ(qix_data_firq_ack_r)
	AM_RANGE(0x1000, 0x13ff) AM_READ(pia_3_r)
	AM_RANGE(0x1400, 0x17ff) AM_READ(pia_0_r)
	AM_RANGE(0x1900, 0x19ff) AM_READ(pia_1_r)
	AM_RANGE(0x1c00, 0x1fff) AM_READ(pia_2_r)
	AM_RANGE(0x8000, 0xffff) AM_READ(MRA8_ROM)
ADDRESS_MAP_END


static ADDRESS_MAP_START( zoo_writemem_data, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x03ff) AM_WRITE(qix_sharedram_w) AM_BASE(&qix_sharedram)
	AM_RANGE(0x0400, 0x07ff) AM_WRITE(MWA8_RAM)
	AM_RANGE(0x0c00, 0x0c00) AM_WRITE(qix_video_firq_w)
	AM_RANGE(0x0c01, 0x0c01) AM_WRITE(qix_data_firq_ack_w)
	AM_RANGE(0x1000, 0x13ff) AM_WRITE(pia_3_w)
	AM_RANGE(0x1400, 0x17ff) AM_WRITE(qix_pia_0_w)
	AM_RANGE(0x1900, 0x19ff) AM_WRITE(pia_1_w)
	AM_RANGE(0x1c00, 0x1fff) AM_WRITE(pia_2_w)
	AM_RANGE(0x8000, 0xffff) AM_WRITE(MWA8_ROM)
ADDRESS_MAP_END



/*************************************
 *
 *  Video CPU memory handlers
 *
 *************************************/

static ADDRESS_MAP_START( readmem_video, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x7fff) AM_READ(qix_videoram_r)
	AM_RANGE(0x8000, 0x83ff) AM_READ(qix_sharedram_r)
	AM_RANGE(0x8400, 0x87ff) AM_READ(MRA8_RAM)
	AM_RANGE(0x8c00, 0x8c00) AM_READ(qix_data_firq_r)
	AM_RANGE(0x8c01, 0x8c01) AM_READ(qix_video_firq_ack_r)
	AM_RANGE(0x9000, 0x93ff) AM_READ(MRA8_RAM)
	AM_RANGE(0x9400, 0x9400) AM_READ(qix_addresslatch_r)
	AM_RANGE(0x9800, 0x9800) AM_READ(qix_scanline_r)
	AM_RANGE(0xa000, 0xffff) AM_READ(MRA8_ROM)
ADDRESS_MAP_END


static ADDRESS_MAP_START( writemem_video, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x7fff) AM_WRITE(qix_videoram_w)
	AM_RANGE(0x8000, 0x83ff) AM_WRITE(qix_sharedram_w)
	AM_RANGE(0x8400, 0x87ff) AM_WRITE(MWA8_RAM) AM_BASE(&generic_nvram) AM_SIZE(&generic_nvram_size)
	AM_RANGE(0x8800, 0x8800) AM_WRITE(qix_palettebank_w)
	AM_RANGE(0x8c00, 0x8c00) AM_WRITE(qix_data_firq_w)
	AM_RANGE(0x8c01, 0x8c01) AM_WRITE(qix_video_firq_ack_w)
	AM_RANGE(0x9000, 0x93ff) AM_WRITE(qix_paletteram_w) AM_BASE(&paletteram)
	AM_RANGE(0x9400, 0x9400) AM_WRITE(qix_addresslatch_w)
	AM_RANGE(0x9402, 0x9403) AM_WRITE(MWA8_RAM) AM_BASE(&qix_videoaddress)
	AM_RANGE(0x9c00, 0x9fff) AM_WRITE(MWA8_RAM) /* Video controller */
	AM_RANGE(0xa000, 0xffff) AM_WRITE(MWA8_ROM)
ADDRESS_MAP_END


static ADDRESS_MAP_START( zoo_readmem_video, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x7fff) AM_READ(qix_videoram_r)
	AM_RANGE(0x8000, 0x83ff) AM_READ(qix_sharedram_r)
	AM_RANGE(0x8400, 0x87ff) AM_READ(MRA8_RAM)
	AM_RANGE(0x8c00, 0x8c00) AM_READ(qix_data_firq_r)
	AM_RANGE(0x8c01, 0x8c01) AM_READ(qix_video_firq_ack_r)
	AM_RANGE(0x9000, 0x93ff) AM_READ(MRA8_RAM)
	AM_RANGE(0x9400, 0x9400) AM_READ(qix_addresslatch_r)
	AM_RANGE(0x9800, 0x9800) AM_READ(qix_scanline_r)
	AM_RANGE(0xa000, 0xbfff) AM_READ(MRA8_BANK1)
	AM_RANGE(0xc000, 0xffff) AM_READ(MRA8_ROM)
ADDRESS_MAP_END


static ADDRESS_MAP_START( zoo_writemem_video, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x7fff) AM_WRITE(qix_videoram_w)
	AM_RANGE(0x8000, 0x83ff) AM_WRITE(qix_sharedram_w)
	AM_RANGE(0x8400, 0x87ff) AM_WRITE(MWA8_RAM) AM_BASE(&generic_nvram) AM_SIZE(&generic_nvram_size)
	AM_RANGE(0x8800, 0x8800) AM_WRITE(qix_palettebank_w)
	AM_RANGE(0x8801, 0x8801) AM_WRITE(zoo_bankswitch_w)
	AM_RANGE(0x8c00, 0x8c00) AM_WRITE(qix_data_firq_w)
	AM_RANGE(0x8c01, 0x8c01) AM_WRITE(qix_video_firq_ack_w)
	AM_RANGE(0x9000, 0x93ff) AM_WRITE(qix_paletteram_w) AM_BASE(&paletteram)
	AM_RANGE(0x9400, 0x9400) AM_WRITE(qix_addresslatch_w)
	AM_RANGE(0x9402, 0x9403) AM_WRITE(MWA8_RAM) AM_BASE(&qix_videoaddress)
	AM_RANGE(0x9c00, 0x9fff) AM_WRITE(MWA8_RAM) /* Video controller */
	AM_RANGE(0xa000, 0xffff) AM_WRITE(MWA8_ROM)
ADDRESS_MAP_END



/*************************************
 *
 *  Sound CPU memory handlers
 *
 *************************************/

static ADDRESS_MAP_START( readmem_sound, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x007f) AM_READ(MRA8_RAM)
	AM_RANGE(0x2000, 0x2003) AM_READ(pia_5_r)
	AM_RANGE(0x4000, 0x4003) AM_READ(pia_4_r)
	AM_RANGE(0xd000, 0xffff) AM_READ(MRA8_ROM)
ADDRESS_MAP_END


static ADDRESS_MAP_START( writemem_sound, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x007f) AM_WRITE(MWA8_RAM)
	AM_RANGE(0x2000, 0x2003) AM_WRITE(pia_5_w)
	AM_RANGE(0x4000, 0x4003) AM_WRITE(pia_4_w)
	AM_RANGE(0xd000, 0xffff) AM_WRITE(MWA8_ROM)
ADDRESS_MAP_END



/*************************************
 *
 *  Coin CPU memory handlers
 *
 *************************************/

static ADDRESS_MAP_START( mcu_readmem, ADDRESS_SPACE_PROGRAM, 8 )
	ADDRESS_MAP_FLAGS( AMEF_ABITS(11) )
	AM_RANGE(0x0000, 0x0000) AM_READ(qix_68705_portA_r)
	AM_RANGE(0x0001, 0x0001) AM_READ(qix_68705_portB_r)
	AM_RANGE(0x0002, 0x0002) AM_READ(qix_68705_portC_r)
	AM_RANGE(0x0010, 0x007f) AM_READ(MRA8_RAM)
	AM_RANGE(0x0080, 0x07ff) AM_READ(MRA8_ROM)
ADDRESS_MAP_END


static ADDRESS_MAP_START( mcu_writemem, ADDRESS_SPACE_PROGRAM, 8 )
	ADDRESS_MAP_FLAGS( AMEF_ABITS(11) )
	AM_RANGE(0x0000, 0x0000) AM_WRITE(qix_68705_portA_w) AM_BASE(&qix_68705_port_out)
	AM_RANGE(0x0001, 0x0001) AM_WRITE(qix_68705_portB_w)
	AM_RANGE(0x0002, 0x0002) AM_WRITE(qix_68705_portC_w)
	AM_RANGE(0x0004, 0x0007) AM_WRITE(MWA8_RAM) AM_BASE(&qix_68705_ddr)
	AM_RANGE(0x0010, 0x007f) AM_WRITE(MWA8_RAM)
	AM_RANGE(0x0080, 0x07ff) AM_WRITE(MWA8_ROM)
ADDRESS_MAP_END



/*************************************
 *
 *  Input port definitions
 *
 *************************************/

#define COIN_PORT \
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_SERVICE ) PORT_NAME("Test Advance") PORT_CODE(KEYCODE_F1) \
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_SERVICE ) PORT_NAME("Test Next line") PORT_CODE(KEYCODE_F2) \
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_SERVICE ) PORT_NAME("Test Slew Up") PORT_CODE(KEYCODE_F5) \
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_SERVICE ) PORT_NAME("Test Slew Down") PORT_CODE(KEYCODE_F6) \
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_COIN1 ) \
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN2 ) \
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN3 ) \
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_TILT )


INPUT_PORTS_START( qix )
	PORT_START	/* PIA 0 Port A (PLAYER 1) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_4WAY
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_4WAY
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_4WAY
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_4WAY
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 )

	PORT_START	/* PIA 0 Port B (COIN) */
	COIN_PORT

	PORT_START	/* PIA 1 Port A (SPARE) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START	/* PIA 1 Port B (PLAYER 1/2) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 2 Port A (PLAYER 2) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_4WAY PORT_COCKTAIL
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_4WAY PORT_COCKTAIL
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_4WAY PORT_COCKTAIL
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_4WAY PORT_COCKTAIL
    PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_COCKTAIL
    PORT_BIT( 0x60, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_COCKTAIL
INPUT_PORTS_END


INPUT_PORTS_START( sdungeon )
	PORT_START	/* PIA 0 Port A (PLAYER 1) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP ) PORT_8WAY
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_RIGHT ) PORT_8WAY
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN ) PORT_8WAY
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_LEFT ) PORT_8WAY
    PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP ) PORT_8WAY
    PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_RIGHT ) PORT_8WAY
    PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN ) PORT_8WAY
    PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_LEFT ) PORT_8WAY

	PORT_START	/* PIA 0 Port B (COIN) */
	COIN_PORT

	PORT_START	/* PIA 1 Port A (SPARE) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START	/* PIA 1 Port B (PLAYER 1/2) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1 )
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_START2 )
    PORT_BIT( 0xfc, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 2 Port A (PLAYER 2) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_RIGHT ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_LEFT ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_RIGHT ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_LEFT ) PORT_8WAY PORT_COCKTAIL
INPUT_PORTS_END


INPUT_PORTS_START( elecyoyo )
	PORT_START	/* PIA 0 Port A (PLAYER 1) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_4WAY
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_4WAY
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_4WAY
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_4WAY
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 0 Port B (COIN) */
	COIN_PORT

	PORT_START	/* PIA 1 Port A (SPARE) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START	/* PIA 1 Port B (PLAYER 1/2) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 2 Port A (PLAYER 2) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_4WAY PORT_COCKTAIL
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_4WAY PORT_COCKTAIL
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_4WAY PORT_COCKTAIL
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_4WAY PORT_COCKTAIL
    PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


INPUT_PORTS_START( kram )
	PORT_START	/* PIA 0 Port A (PLAYER 1) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 )

	PORT_START	/* PIA 0 Port B (COIN) */
	COIN_PORT

	PORT_START	/* PIA 1 Port A (SPARE) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START	/* PIA 1 Port B (PLAYER 1/2) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 2 Port A (PLAYER 2) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_COCKTAIL
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_COCKTAIL
    PORT_BIT( 0x60, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_COCKTAIL
INPUT_PORTS_END


INPUT_PORTS_START( zookeep )
	PORT_START	/* PIA 0 Port A (PLAYER 1) */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_4WAY
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_4WAY
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_4WAY
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_4WAY
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 0 Port B (COIN) */
	COIN_PORT

	PORT_START	/* PIA 1 Port A (SPARE) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START	/* PIA 1 Port B (PLAYER 1/2) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 2 Port A (PLAYER 2) */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_4WAY PORT_COCKTAIL
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_4WAY PORT_COCKTAIL
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_4WAY PORT_COCKTAIL
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_4WAY PORT_COCKTAIL
	PORT_BIT( 0x30, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_COCKTAIL
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


INPUT_PORTS_START( slither )
	PORT_START	/* PIA 0 Port A (PLAYER 1) */
    PORT_BIT( 0x0f, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 )

	PORT_START	/* PIA 0 Port B (COIN) */
	PORT_SERVICE(0x01, IP_ACTIVE_LOW )
    PORT_BIT( 0x0e, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_TILT )

	PORT_START	/* PIA 3 Port A (PLAYER 2) */
    PORT_BIT( 0x0f, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_COCKTAIL
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 ) PORT_COCKTAIL
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 ) PORT_COCKTAIL
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_COCKTAIL

	PORT_START	/* PIA 1 Port A (TRACKBALL L/R) */
    PORT_BIT( 0xff, 0, IPT_TRACKBALL_X ) PORT_MINMAX(0x00,0xff) PORT_SENSITIVITY(50) PORT_KEYDELTA(10)

	PORT_START	/* PIA 2 Port A (TRACKBALL U/D) */
    PORT_BIT( 0xff, 0, IPT_TRACKBALL_Y ) PORT_MINMAX(0x00,0xff) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_REVERSE

	PORT_START	/* PIA 1 Port A (TRACKBALL L/R) */
    PORT_BIT( 0xff, 0, IPT_TRACKBALL_X ) PORT_MINMAX(0x00,0xff) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_COCKTAIL

	PORT_START	/* PIA 2 Port A (TRACKBALL U/D) */
    PORT_BIT( 0xff, 0, IPT_TRACKBALL_Y ) PORT_MINMAX(0x00,0xff) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_REVERSE PORT_COCKTAIL
INPUT_PORTS_END


INPUT_PORTS_START( complexx )
	PORT_START	/* PIA 0 Port A (Left Stick) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP ) PORT_8WAY
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_RIGHT ) PORT_8WAY
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN ) PORT_8WAY
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_LEFT ) PORT_8WAY
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

/* 0x80 isn't scanned unless the difficulty in the game is below 4. I
decided not to map this button, so if you set the difficulty that low,
you can't fire. If I do map that button, and you set difficulty below 4
and hit this fire button without pressing the second stick in any
direction it fires up. If you're pressing left it fires right, press
right it fires left and press down it fires down. That's just too wacky.
I gotta think it's not supposed to be hooked up at all and if the
difficulty is that low, you just can't shoot, so I think it should stay
as-is. Tim Lindquist 1-17-03 */

	PORT_START	/* PIA 0 Port B (COIN) */
	COIN_PORT

	PORT_START	/* PIA 1 Port A (SPARE) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START	/* PIA 1 Port B (PLAYER 1/2) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 2 Port A (Right Stick) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED )
    PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP ) PORT_8WAY
    PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_RIGHT ) PORT_8WAY
    PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN ) PORT_8WAY
    PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_LEFT ) PORT_8WAY
INPUT_PORTS_END



/*************************************
 *
 *  Machine drivers
 *
 *************************************/

static MACHINE_DRIVER_START( qix )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("main", M6809, MAIN_CLOCK_OSC/4/4)	/* 1.25 MHz */
	MDRV_CPU_PROGRAM_MAP(readmem_data,writemem_data)
	MDRV_CPU_VBLANK_INT(qix_vblank_start,1)

	MDRV_CPU_ADD_TAG("video", M6809, MAIN_CLOCK_OSC/4/4)	/* 1.25 MHz */
	MDRV_CPU_PROGRAM_MAP(readmem_video,writemem_video)

	MDRV_CPU_ADD_TAG("sound", M6802, SOUND_CLOCK_OSC/2/4)	/* 0.92 MHz */
	/* audio CPU */
	MDRV_CPU_PROGRAM_MAP(readmem_sound,writemem_sound)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_REAL_60HZ_VBLANK_DURATION)

	MDRV_MACHINE_INIT(qix)
	MDRV_NVRAM_HANDLER(generic_0fill)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(256, 256)
	MDRV_VISIBLE_AREA(0, 255, 8, 247)
	MDRV_PALETTE_LENGTH(1024)

	MDRV_VIDEO_START(qix)
	MDRV_VIDEO_UPDATE(qix)

	/* sound hardware */
	MDRV_SPEAKER_STANDARD_STEREO("left", "right")

	MDRV_SOUND_ADD_TAG("discrete", DISCRETE, 0)
	MDRV_SOUND_CONFIG(qix_discrete_interface)
	MDRV_SOUND_ROUTE(0, "left", 1.0)
	MDRV_SOUND_ROUTE(1, "right", 1.0)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( mcu )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(qix)

	MDRV_CPU_ADD(M68705, COIN_CLOCK_OSC/4)	/* 1.00 MHz */
	MDRV_CPU_PROGRAM_MAP(mcu_readmem,mcu_writemem)

	MDRV_MACHINE_INIT(qixmcu)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( elecyoyo )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(mcu)

	MDRV_INTERLEAVE(20)	/* fixes hang in attract mode */
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( zookeep )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(mcu)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_PROGRAM_MAP(zoo_readmem_data,zoo_writemem_data)

	MDRV_CPU_MODIFY("video")
	MDRV_CPU_PROGRAM_MAP(zoo_readmem_video,zoo_writemem_video)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( slither )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(qix)

	MDRV_CPU_REPLACE("main", M6809, SLITHER_CLOCK_OSC/4/4)	/* 1.34 MHz */
	MDRV_CPU_REPLACE("video", M6809, SLITHER_CLOCK_OSC/4/4)	/* 1.34 MHz */
	MDRV_CPU_REMOVE("sound")

	MDRV_MACHINE_INIT(slither)

	/* video hardware */
	MDRV_VISIBLE_AREA(0, 255, 0, 255)

	/* sound hardware */
	MDRV_SPEAKER_REMOVE("left")
	MDRV_SPEAKER_REMOVE("right")
	MDRV_SPEAKER_STANDARD_MONO("mono")

	MDRV_SOUND_REPLACE("discrete", SN76496, SLITHER_CLOCK_OSC/4/4)
	MDRV_SOUND_ROUTE(ALL_OUTPUTS, "mono", 0.50)

	MDRV_SOUND_ADD(SN76496, SLITHER_CLOCK_OSC/4/4)
	MDRV_SOUND_ROUTE(ALL_OUTPUTS, "mono", 0.50)
MACHINE_DRIVER_END



/*************************************
 *
 *  ROM definitions
 *
 *************************************/

ROM_START( qix )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "u12", 0xc000, 0x0800, CRC(aad35508) SHA1(5fa72e00b4373de21e27a86b49a44a9769f769f4) )
	ROM_LOAD( "u13", 0xc800, 0x0800, CRC(46c13504) SHA1(19c084c38b75f14bf5094b317afeecaca6870f7a) )
	ROM_LOAD( "u14", 0xd000, 0x0800, CRC(5115e896) SHA1(8359a1700fff7a38e8ea4f92a4f18bc628cf1cb1) )
	ROM_LOAD( "u15", 0xd800, 0x0800, CRC(ccd52a1b) SHA1(86d134cd769ef12820638b96a4ffedd8b15dffd2) )
	ROM_LOAD( "u16", 0xe000, 0x0800, CRC(cd1c36ee) SHA1(b379b1fe3109947a12c9683cd0c2400c2ee845b3) )
	ROM_LOAD( "u17", 0xe800, 0x0800, CRC(1acb682d) SHA1(a2c60964e8d838d09662f8a670c6da41ba850df9) )
	ROM_LOAD( "u18", 0xf000, 0x0800, CRC(de77728b) SHA1(8e183bb27858aad9a996e4a2e5a95f0145d1f5b4) )
	ROM_LOAD( "u19", 0xf800, 0x0800, CRC(c0994776) SHA1(9452a98c78a038679c4e58f4a9983adb28ea5e78) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u4",  0xc800, 0x0800, CRC(5b906a09) SHA1(84a2e817d6718e0276fcea702811a91bc054a670) )
	ROM_LOAD( "u5",  0xd000, 0x0800, CRC(254a3587) SHA1(66045c71cc1d04d4e03c728e578f570fbf7c650d) )
	ROM_LOAD( "u6",  0xd800, 0x0800, CRC(ace30389) SHA1(50c6275d13cfbca7750d5a3e725faedba7574e04) )
	ROM_LOAD( "u7",  0xe000, 0x0800, CRC(8ebcfa7c) SHA1(21ccf5e74424ab5470473d1059ee6a43d144f685) )
	ROM_LOAD( "u8",  0xe800, 0x0800, CRC(b8a3c8f9) SHA1(32ba771913ef44b1133ecfaedaae7f96dcc84343) )
	ROM_LOAD( "u9",  0xf000, 0x0800, CRC(26cbcd55) SHA1(2e55e222f850548cd1d461ab5337e98dd817b567) )
	ROM_LOAD( "u10", 0xf800, 0x0800, CRC(568be942) SHA1(8b6a01d983d355a64372fa76af810ab53e09d5df) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "u27", 0xf800, 0x0800, CRC(f3782bd0) SHA1(bfc6d29f9668e02857453e96c005c81568ae931d) )
ROM_END


ROM_START( qixa )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "qu12", 0xc000, 0x0800, CRC(1c55b44d) SHA1(6385e5e484e24cf396c14de86344170639c3cc65) )
	ROM_LOAD( "qu13", 0xc800, 0x0800, CRC(20279e8c) SHA1(722da239636de3fe40318768ddbe687b19afcdb6) )
	ROM_LOAD( "qu14", 0xd000, 0x0800, CRC(bafe3ce3) SHA1(648a54545a1b545c82c0ace5eb1ce17af5ea7391) )
	/* d800-dfff empty */
	ROM_LOAD( "qu16", 0xe000, 0x0800, CRC(db560753) SHA1(4acbe17f1e555f45606ddec197c5ab691ff46d39) )
	ROM_LOAD( "qu17", 0xe800, 0x0800, CRC(8c7aeed8) SHA1(b7da2b0f34f72f9853cdf6ce55e604b09fcf4728) )
	ROM_LOAD( "qu18", 0xf000, 0x0800, CRC(353be980) SHA1(a50e02fcc69771a13b238aa0e8dc3c56b01a58d5) )
	ROM_LOAD( "qu19", 0xf800, 0x0800, CRC(f46a69ca) SHA1(dacb53c0318445da3fbb86f9a45914c5b7a4c4a1) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "qu3",  0xc000, 0x0800, CRC(8b4c0ef0) SHA1(6d18d1052f342e3b3313f2174b20f2a179e2c6bd) )
	ROM_LOAD( "qu4",  0xc800, 0x0800, CRC(66a5c260) SHA1(8cce71bcd3a432650f0d0c94f3a2151ba8154220) )
	ROM_LOAD( "qu5",  0xd000, 0x0800, CRC(70160ea3) SHA1(a411130c5c669a181564369a8921b26e0f0b5450) )
	/* d800-dfff empty */
	ROM_LOAD( "qu7",  0xe000, 0x0800, CRC(d6733019) SHA1(89e9e63c91e044fe1c6ce883e3ec18eec0cb39d3) )
	ROM_LOAD( "qu8",  0xe800, 0x0800, CRC(66870dcc) SHA1(9f926390f5ce86d7c1bf55b75dbfb34119425c46) )
	ROM_LOAD( "qu9",  0xf000, 0x0800, CRC(c99bf94d) SHA1(7b6fa6e1cf0f131909d44694c261b1cc2de65003) )
	ROM_LOAD( "qu10", 0xf800, 0x0800, CRC(88b45037) SHA1(e2e5fefe377def3f784026b921527898af8b83a9) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "u27",  0xf800, 0x0800, CRC(f3782bd0) SHA1(bfc6d29f9668e02857453e96c005c81568ae931d) )
ROM_END


ROM_START( qixb )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "lk14.bin", 0xc000, 0x1000, CRC(6d164986) SHA1(c805abe1a441e10080ceca8ba547835bafb61bcc) )
	ROM_LOAD( "lk15.bin", 0xd000, 0x1000, CRC(16c6ce0f) SHA1(b8091d2db476d2acb4b3f0789e1f155336be9b39) )
	ROM_LOAD( "lk16.bin", 0xe000, 0x1000, CRC(698b1f9c) SHA1(7e7637ca5985f072e821e16f8b65aedb87df136b) )
	ROM_LOAD( "lk17.bin", 0xf000, 0x1000, CRC(7e3adde6) SHA1(dfe66317f87e10919f1ea4b4d565703e73039821) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "lk10.bin", 0xc000, 0x1000, CRC(7eac67d0) SHA1(ca5938422aaa1e380af0afa505876d4682ac69b9) )
	ROM_LOAD( "lk11.bin", 0xd000, 0x1000, CRC(90ccbb6a) SHA1(b65592384597dc2aafc02f49b6b6f477c9112580) )
	ROM_LOAD( "lk12.bin", 0xe000, 0x1000, CRC(be9b9f7d) SHA1(e681bdb9aa8b8c31af1c14e23d0f420577d6db63) )
	ROM_LOAD( "lk13.bin", 0xf000, 0x1000, CRC(51c9853b) SHA1(29a5221f2af866d2ee73110409ecddc2c96404fd) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "u27",      0xf800, 0x0800, CRC(f3782bd0) SHA1(bfc6d29f9668e02857453e96c005c81568ae931d) )
ROM_END


ROM_START( qix2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "u12.rmb", 0xc000, 0x0800, CRC(484280fd) SHA1(a60c1a278e519721294b2486dc817d248d19c3be) )
	ROM_LOAD( "u13.rmb", 0xc800, 0x0800, CRC(3d089fcb) SHA1(f4f31134c9c15160d2d15cb41296dfec6f2dfe37) )
	ROM_LOAD( "u14.rmb", 0xd000, 0x0800, CRC(362123a9) SHA1(3e2a853f6960f2d5fdcdef8dec8ccf5aad449548) )
	ROM_LOAD( "u15.rmb", 0xd800, 0x0800, CRC(60f3913d) SHA1(a97b658fe2c58b00c2749072828b2e0032894915) )
	ROM_LOAD( "u16.rmb", 0xe000, 0x0800, CRC(cc139e34) SHA1(0ed3e7179b0cbaa31fa91e1ed862b86f5032919a) )
	ROM_LOAD( "u17.rmb", 0xe800, 0x0800, CRC(cf31dc49) SHA1(71c089d827ab61ba69e5e95b7e53220763786df9) )
	ROM_LOAD( "u18.rmb", 0xf000, 0x0800, CRC(1f91ed7a) SHA1(85bb5370a244719663a4f859f66860613aa2b86e) )
	ROM_LOAD( "u19.rmb", 0xf800, 0x0800, CRC(68e8d5a6) SHA1(d09252c393be2fdaf3b9b9f477c79f721d15943f) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u3.rmb",  0xc000, 0x0800, CRC(19cebaca) SHA1(7d7e79ab0920952cf7618567c9c65397535b6d4f) )
	ROM_LOAD( "u4.rmb",  0xc800, 0x0800, CRC(6cfb4185) SHA1(6545dece8eaeb716877aa6e7b24c21f6e5991451) )
	ROM_LOAD( "u5.rmb",  0xd000, 0x0800, CRC(948f53f3) SHA1(db6eddec8ba41335316d80b6f97e932bf91139af) )
	ROM_LOAD( "u6.rmb",  0xd800, 0x0800, CRC(8630120e) SHA1(14a020fd1bff4acbb034883e33130adda85884e5) )
	ROM_LOAD( "u7.rmb",  0xe000, 0x0800, CRC(bad037c9) SHA1(17218c31895b1547b71d2d9d2b6a93d2e5d73bdd) )
	ROM_LOAD( "u8.rmb",  0xe800, 0x0800, CRC(3159bc00) SHA1(479a69bfe5af48d5ce63978265ce59f79c25749f) )
	ROM_LOAD( "u9.rmb",  0xf000, 0x0800, CRC(e80e9b1d) SHA1(66ef22a26df3f766ae813213473b9ac4b35b01f6) )
	ROM_LOAD( "u10.rmb", 0xf800, 0x0800, CRC(9a55d360) SHA1(fc5f8c853dcc573f6b36dbdd63e5d1edba88bce1) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "u27",     0xf800, 0x0800, CRC(f3782bd0) SHA1(bfc6d29f9668e02857453e96c005c81568ae931d) )
ROM_END


ROM_START( sdungeon )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "sd14.u14", 0xa000, 0x1000, CRC(7024b55a) SHA1(1e21cb2a9cba8c0a3684e137ff78d4b331d86922) )
	ROM_LOAD( "sd15.u15", 0xb000, 0x1000, CRC(a3ac9040) SHA1(f033c21983e87688884180c2336d766a0fa49765) )
	ROM_LOAD( "sd16.u16", 0xc000, 0x1000, CRC(cc20b580) SHA1(53e34405f1f39bce305f199d09d3a32c10c0c616) )
	ROM_LOAD( "sd17.u17", 0xd000, 0x1000, CRC(4663e4b8) SHA1(519b0e730db2047ebe2cce0953eda53e851f8fac) )
	ROM_LOAD( "sd18.u18", 0xe000, 0x1000, CRC(7ef1ffc0) SHA1(eca49a916b6b51b91ed45ff89bb37a67fee7db0e) )
	ROM_LOAD( "sd19.u19", 0xf000, 0x1000, CRC(7b20b7ac) SHA1(554e29adc75cc465ea603a628f9d60d6d903e7ab) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "sd05.u5",  0xa000, 0x1000, CRC(0b2bf48e) SHA1(51bfb35521864f09a20b38aeeb98ab1399d139a5) )
	ROM_LOAD( "sd06.u6",  0xb000, 0x1000, CRC(f86db512) SHA1(7e7ae64db7821f18a5eefdcc7a2e1abc37abd3ac) )
	ROM_LOAD( "sd07.u7",  0xc000, 0x1000, CRC(7b796831) SHA1(cd3d8975e99886f51b27530f0e261f749aadee73) )
	ROM_LOAD( "sd08.u8",  0xd000, 0x1000, CRC(5fbe7068) SHA1(2884909156179b81181b908431de15a7c586f619) )
	ROM_LOAD( "sd09.u9",  0xe000, 0x1000, CRC(89bc51ea) SHA1(57ad4806f0b39af3b3cde91cef16234fc82f21d0) )
	ROM_LOAD( "sd10.u10", 0xf000, 0x1000, CRC(754de734) SHA1(a37b8362a592f5d1c0aeaa525fc003f6060fc12b) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "sd26.u26", 0xf000, 0x0800, CRC(3df8630d) SHA1(5f87c4d49799d424ce36469bf2b36b14c782fcd8) )
	ROM_LOAD( "sd27.u27", 0xf800, 0x0800, CRC(0386f351) SHA1(24ba6aba7c62c313397d743d18093c646f4a6526) )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "sd101",    0x0000, 0x0800, CRC(e255af9a) SHA1(2410d3b7dec8e72a93d71c824c9403a6d96b9e8c) )
ROM_END


ROM_START( elecyoyo )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
    ROM_LOAD( "yy14",   0xa000, 0x1000, CRC(0d2edcb9) SHA1(36e1a1aa81111f38e1c06a8174e7de406478cc67) )
    ROM_LOAD( "yy15",   0xb000, 0x1000, CRC(a91f01e3) SHA1(7818299d25a0816b856e83fae02d8019e5e8b4a3) )
    ROM_LOAD( "yy16-1", 0xc000, 0x1000, CRC(2710f360) SHA1(4a6210b07618fba261c38b7bf9a779598dd6bb3c) )
    ROM_LOAD( "yy17",   0xd000, 0x1000, CRC(25fd489d) SHA1(ca7b8d1bcbc223fc0706c8f2e9f02821519b75e4) )
    ROM_LOAD( "yy18",   0xe000, 0x1000, CRC(0b6661c0) SHA1(3e24acbfea1a3b83223d780ea34c83759a751175) )
    ROM_LOAD( "yy19-1", 0xf000, 0x1000, CRC(95b8b244) SHA1(2852dcfc6a638118ee7fe60b957d8aa9a5a8984c) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
    ROM_LOAD( "yy5",    0xa000, 0x1000, CRC(3793fec5) SHA1(d31f3ba6364755c98beb814b5e7d5541e8b4e1a0) )
    ROM_LOAD( "yy6",    0xb000, 0x1000, CRC(2e8b1265) SHA1(6fb67b6a5b627bcbd10c72ac8c9e4d9f4dd7860f) )
    ROM_LOAD( "yy7",    0xc000, 0x1000, CRC(20f93411) SHA1(126c27442a4e35d054a236c41930603241d08ccf) )
    ROM_LOAD( "yy8",    0xd000, 0x1000, CRC(926f90c8) SHA1(dbdfc0e9184a4d9d44d04fdb8fed19b35a8edadc) )
    ROM_LOAD( "yy9",    0xe000, 0x1000, CRC(2f999480) SHA1(582baf285cadf8431ff80f5b63a02fbbefb62e45) )
    ROM_LOAD( "yy10",   0xf000, 0x1000, CRC(b31d20e2) SHA1(e2ba4e6b81a02c3c02353774eb63a72cdfe5f2c3) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
    ROM_LOAD( "yy27",   0xf800, 0x0800, CRC(5a2aa0f3) SHA1(16c7c2db39f33ea3506e07312352ccbfe5528fbd) )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "yy101",  0x0000, 0x0800, CRC(3cf13038) SHA1(29c1b309d9046087a50ca78a5f7cba694271af26) )
ROM_END


ROM_START( elecyoy2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
    ROM_LOAD( "yy14",  0xa000, 0x1000, CRC(0d2edcb9) SHA1(36e1a1aa81111f38e1c06a8174e7de406478cc67) )
    ROM_LOAD( "yy15",  0xb000, 0x1000, CRC(a91f01e3) SHA1(7818299d25a0816b856e83fae02d8019e5e8b4a3) )
    ROM_LOAD( "yy16",  0xc000, 0x1000, CRC(cab19f3a) SHA1(df41649a800b77f046edb6623f65d100f6a8ef5f) )
    ROM_LOAD( "yy17",  0xd000, 0x1000, CRC(25fd489d) SHA1(ca7b8d1bcbc223fc0706c8f2e9f02821519b75e4) )
    ROM_LOAD( "yy18",  0xe000, 0x1000, CRC(0b6661c0) SHA1(3e24acbfea1a3b83223d780ea34c83759a751175) )
    ROM_LOAD( "yy19",  0xf000, 0x1000, CRC(d0215d2e) SHA1(816131bc272252df6cea1b84a42750a71ce4f427) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
    ROM_LOAD( "yy5",   0xa000, 0x1000, CRC(3793fec5) SHA1(d31f3ba6364755c98beb814b5e7d5541e8b4e1a0) )
    ROM_LOAD( "yy6",   0xb000, 0x1000, CRC(2e8b1265) SHA1(6fb67b6a5b627bcbd10c72ac8c9e4d9f4dd7860f) )
    ROM_LOAD( "yy7",   0xc000, 0x1000, CRC(20f93411) SHA1(126c27442a4e35d054a236c41930603241d08ccf) )
    ROM_LOAD( "yy8",   0xd000, 0x1000, CRC(926f90c8) SHA1(dbdfc0e9184a4d9d44d04fdb8fed19b35a8edadc) )
    ROM_LOAD( "yy9",   0xe000, 0x1000, CRC(2f999480) SHA1(582baf285cadf8431ff80f5b63a02fbbefb62e45) )
    ROM_LOAD( "yy10",  0xf000, 0x1000, CRC(b31d20e2) SHA1(e2ba4e6b81a02c3c02353774eb63a72cdfe5f2c3) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
    ROM_LOAD( "yy27",  0xf800, 0x0800, CRC(5a2aa0f3) SHA1(16c7c2db39f33ea3506e07312352ccbfe5528fbd) )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "yy101", 0x0000, 0x0800, CRC(3cf13038) SHA1(29c1b309d9046087a50ca78a5f7cba694271af26) )
ROM_END


ROM_START( kram )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
    ROM_LOAD( "ks14-1", 0xa000, 0x1000, CRC(fe69ac79) SHA1(6df0f98e6c0901c058123988bf22a6dd9f0a1fac) )
    ROM_LOAD( "ks15",   0xb000, 0x1000, CRC(4b2c175e) SHA1(4f9d4dcc78a12e994d499b182c8229d5fa63b805) )
    ROM_LOAD( "ks16",   0xc000, 0x1000, CRC(9500a05d) SHA1(18e0107111f79ba5dc6d568e3a6e7e7778955d0b) )
    ROM_LOAD( "ks17",   0xd000, 0x1000, CRC(c752a3a1) SHA1(1d03ea97b9ca6fa3d4c43ac867ab737439d987af) )
    ROM_LOAD( "ks18",   0xe000, 0x1000, CRC(79158b03) SHA1(0d4873471b5b7ace0de8ec421ff3d74650790f7e) )
    ROM_LOAD( "ks19-1", 0xf000, 0x1000, CRC(759ea6ce) SHA1(7962f713dd93c73475fa1f64635d8e965336484b) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
    ROM_LOAD( "ks5",    0xa000, 0x1000, CRC(1c472080) SHA1(a85400be562ef6b817f8a654f29d966d3a198ab4) )
    ROM_LOAD( "ks6",    0xb000, 0x1000, CRC(b8926622) SHA1(e25a8b2ff192f6ab0328fd7b3c58d638342f79e2) )
    ROM_LOAD( "ks7",    0xc000, 0x1000, CRC(c98a7485) SHA1(e310d53ae65d456e12a2475e9ac578592b0e82ba) )
    ROM_LOAD( "ks8",    0xd000, 0x1000, CRC(1127c4e4) SHA1(d78a8a964aac29fb71a55acf7956355724a234eb) )
    ROM_LOAD( "ks9",    0xe000, 0x1000, CRC(d3bc8b5e) SHA1(2c5b882c54bdb48f9a76abfe734b99390b89d76c) )
    ROM_LOAD( "ks10",   0xf000, 0x1000, CRC(e0426444) SHA1(3c4ea66f8ee907452e67aeb7cfaa15bee02b004b) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
    ROM_LOAD( "ks27",   0xf800, 0x0800, CRC(c46530c8) SHA1(d2df3f2228a5cff7d7b04b5bbbc4820d2fe84d8d) )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "ks101.dat", 0x0000, 0x0800, CRC(e53d97b7) SHA1(acfc3a5c5e73bd2d37e04ac357043b708d1982de) )
ROM_END


ROM_START( kram2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
    ROM_LOAD( "ks14", 0xa000, 0x1000, CRC(a2eac1ff) SHA1(128f83b1760492cbb272828ad8c67ea9a5db862a) )
    ROM_LOAD( "ks15", 0xb000, 0x1000, CRC(4b2c175e) SHA1(4f9d4dcc78a12e994d499b182c8229d5fa63b805) )
    ROM_LOAD( "ks16", 0xc000, 0x1000, CRC(9500a05d) SHA1(18e0107111f79ba5dc6d568e3a6e7e7778955d0b) )
    ROM_LOAD( "ks17", 0xd000, 0x1000, CRC(c752a3a1) SHA1(1d03ea97b9ca6fa3d4c43ac867ab737439d987af) )
    ROM_LOAD( "ks18", 0xe000, 0x1000, CRC(79158b03) SHA1(0d4873471b5b7ace0de8ec421ff3d74650790f7e) )
    ROM_LOAD( "ks19", 0xf000, 0x1000, CRC(053c5e09) SHA1(cd6e5b54abf73c1ccf318ca18fceb56b51a3847f) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
    ROM_LOAD( "ks5",  0xa000, 0x1000, CRC(1c472080) SHA1(a85400be562ef6b817f8a654f29d966d3a198ab4) )
    ROM_LOAD( "ks6",  0xb000, 0x1000, CRC(b8926622) SHA1(e25a8b2ff192f6ab0328fd7b3c58d638342f79e2) )
    ROM_LOAD( "ks7",  0xc000, 0x1000, CRC(c98a7485) SHA1(e310d53ae65d456e12a2475e9ac578592b0e82ba) )
    ROM_LOAD( "ks8",  0xd000, 0x1000, CRC(1127c4e4) SHA1(d78a8a964aac29fb71a55acf7956355724a234eb) )
    ROM_LOAD( "ks9",  0xe000, 0x1000, CRC(d3bc8b5e) SHA1(2c5b882c54bdb48f9a76abfe734b99390b89d76c) )
    ROM_LOAD( "ks10", 0xf000, 0x1000, CRC(e0426444) SHA1(3c4ea66f8ee907452e67aeb7cfaa15bee02b004b) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
    ROM_LOAD( "ks27", 0xf800, 0x0800, CRC(c46530c8) SHA1(d2df3f2228a5cff7d7b04b5bbbc4820d2fe84d8d) )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "ks101.dat", 0x0000, 0x0800, CRC(e53d97b7) SHA1(acfc3a5c5e73bd2d37e04ac357043b708d1982de) )
ROM_END


ROM_START( kram3 )
	ROM_REGION( 2*0x10000, REGION_CPU1, 0 )	/* encrypted */
    ROM_LOAD( "kr-u14", 0xa000, 0x1000, CRC(02c1bd1e) SHA1(5f13f32ca2da0e93ed43b052c8c33af9ac67cb6c) )
    ROM_LOAD( "kr-u15", 0xb000, 0x1000, CRC(46b3ff33) SHA1(7db45971972df144a21fee4cc015b0190b502e12) )
    ROM_LOAD( "kr-u16", 0xc000, 0x1000, CRC(f202b9cf) SHA1(baf27507611c3029e2dfb1a4ff86e6fe17171246) )
    ROM_LOAD( "kr-u17", 0xd000, 0x1000, CRC(257cea23) SHA1(f9503c4a0f94d35a55033c02dda4d03737eedb90) )
    ROM_LOAD( "kr-u18", 0xe000, 0x1000, CRC(da3aed8c) SHA1(0107d58fa006a39b47513381aead760190abef35) )
    ROM_LOAD( "kr-u19", 0xf000, 0x1000, CRC(496ab571) SHA1(30e12b31ffd70a8a1ce23c845e89170ca3cabaa5) )

	ROM_REGION( 2*0x10000, REGION_CPU2, 0 )	/* encrypted */
    ROM_LOAD( "kr-u5",  0xa000, 0x1000, CRC(9e63c2bc) SHA1(f61a2b93ed322b62818f31fddb324c666380eff7) )
    ROM_LOAD( "kr-u6",  0xb000, 0x1000, CRC(a0ff1244) SHA1(5bc3f3f8caac0dfc8c1381d34e5b8ef5c8202982) )
    ROM_LOAD( "kr-u7",  0xc000, 0x1000, CRC(20a15024) SHA1(094951c4cd06e32af2cb2faec04c31d55ade6b7b) )
    ROM_LOAD( "ks8",    0xd000, 0x1000, CRC(1127c4e4) SHA1(d78a8a964aac29fb71a55acf7956355724a234eb) )
    ROM_LOAD( "ks9",    0xe000, 0x1000, CRC(d3bc8b5e) SHA1(2c5b882c54bdb48f9a76abfe734b99390b89d76c) )
    ROM_LOAD( "kr-u10", 0xf000, 0x1000, CRC(0a8adbd8) SHA1(8ab806108c68aa2740d9e157dd215b371e81c482) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
    ROM_LOAD( "ks27",   0xf800, 0x0800, CRC(c46530c8) SHA1(d2df3f2228a5cff7d7b04b5bbbc4820d2fe84d8d) )
ROM_END


ROM_START( zookeep )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "za12", 0x8000, 0x1000, CRC(4e40d8dc) SHA1(dd7923dcb55a2a1ae0f2029caf5a8904a9ebe8b1) )
	ROM_LOAD( "za13", 0x9000, 0x1000, CRC(eebd5248) SHA1(ebe7f8c436bfefa4236f603fbcbd38d2f4cfd2bd) )
	ROM_LOAD( "za14", 0xa000, 0x1000, CRC(fab43297) SHA1(daa5a780bc9f171da0f6db5319b1519caa09c6c9) )
	ROM_LOAD( "za15", 0xb000, 0x1000, CRC(ef8cd67c) SHA1(f3a2e12ccfa45eb77c2a6e0d9cc0601b99273fd3) )
	ROM_LOAD( "za16", 0xc000, 0x1000, CRC(ccfc15bc) SHA1(56ac6e89825fcde9cf7aeb5d0765a02f5474174a) )
	ROM_LOAD( "za17", 0xd000, 0x1000, CRC(358013f4) SHA1(dd3fae69bf460f89c71e7c78d229dd86605c5950) )
	ROM_LOAD( "za18", 0xe000, 0x1000, CRC(37886afe) SHA1(a0cc902b2d253466e21c4fbf9d3339069fe79ebe) )
	ROM_LOAD( "za19", 0xf000, 0x1000, CRC(bbfb30d9) SHA1(bc6bd5525b159bee7b08b6967cd3088b7bd10fee) )

	ROM_REGION( 0x12000, REGION_CPU2, 0 )
	ROM_LOAD( "za5",  0x0a000, 0x1000, CRC(dc0c3cbd) SHA1(8335cd91bbacc680a3a98a5242d4cb5a6f61b2b5) )
	ROM_LOAD( "za3",  0x10000, 0x1000, CRC(cc4d0aee) SHA1(05c0025e96b432088b46100051a2c780e46b7457) )
	ROM_LOAD( "za6",  0x0b000, 0x1000, CRC(27c787dd) SHA1(1142790d875573d0c39d846aba4b06946fd2bc88) )
	ROM_LOAD( "za4",  0x11000, 0x1000, CRC(ec3b10b1) SHA1(44303ab923d776052ca4d2d09a6a4315d67adc4b) )

	ROM_LOAD( "za7",  0x0c000, 0x1000, CRC(1479f480) SHA1(bd0e9eead0f1213eb3b9653c73257bc704346ab0) )
	ROM_LOAD( "za8",  0x0d000, 0x1000, CRC(4c96cdb2) SHA1(67e506462317d882c4e5c2e16318411d4958ac63) )
	ROM_LOAD( "za9",  0x0e000, 0x1000, CRC(a4f7d9e0) SHA1(a958b4d305bb397aa46d8fdab9dc7e472237ca11) )
	ROM_LOAD( "za10", 0x0f000, 0x1000, CRC(05df1a5a) SHA1(30797838c25cca038023c188cd9fa45277c4190d) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "za25", 0xd000, 0x1000, CRC(779b8558) SHA1(7312e63c23d92c9c52e93cc445a718bc8fe35a0a) )
	ROM_LOAD( "za26", 0xe000, 0x1000, CRC(60a810ce) SHA1(d97e5acea0ef1c208f8e5e95024c83dd6bc9b028) )
	ROM_LOAD( "za27", 0xf000, 0x1000, CRC(99ed424e) SHA1(e4e543dcb77f153aeb78904d11b95381d039299e) )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "za_coin.bin", 0x0000, 0x0800, CRC(364d3557) SHA1(049d0759750c576187053306e07984b1e5877df7) )
ROM_END


ROM_START( zookeep2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "za12",     0x8000, 0x1000, CRC(4e40d8dc) SHA1(dd7923dcb55a2a1ae0f2029caf5a8904a9ebe8b1) )
	ROM_LOAD( "za13",     0x9000, 0x1000, CRC(eebd5248) SHA1(ebe7f8c436bfefa4236f603fbcbd38d2f4cfd2bd) )
	ROM_LOAD( "za14",     0xa000, 0x1000, CRC(fab43297) SHA1(daa5a780bc9f171da0f6db5319b1519caa09c6c9) )
	ROM_LOAD( "za15",     0xb000, 0x1000, CRC(ef8cd67c) SHA1(f3a2e12ccfa45eb77c2a6e0d9cc0601b99273fd3) )
	ROM_LOAD( "za16",     0xc000, 0x1000, CRC(ccfc15bc) SHA1(56ac6e89825fcde9cf7aeb5d0765a02f5474174a) )
	ROM_LOAD( "za17",     0xd000, 0x1000, CRC(358013f4) SHA1(dd3fae69bf460f89c71e7c78d229dd86605c5950) )
	ROM_LOAD( "za18",     0xe000, 0x1000, CRC(37886afe) SHA1(a0cc902b2d253466e21c4fbf9d3339069fe79ebe) )
	ROM_LOAD( "za19.red", 0xf000, 0x1000, CRC(ec01760e) SHA1(169ab9d3a0abe325d960f9ed358258b3d6fcd4be) )

	ROM_REGION( 0x12000, REGION_CPU2, 0 )
	ROM_LOAD( "za5",      0x0a000, 0x1000, CRC(dc0c3cbd) SHA1(8335cd91bbacc680a3a98a5242d4cb5a6f61b2b5) )
	ROM_LOAD( "za3",      0x10000, 0x1000, CRC(cc4d0aee) SHA1(05c0025e96b432088b46100051a2c780e46b7457) )
	ROM_LOAD( "za6",      0x0b000, 0x1000, CRC(27c787dd) SHA1(1142790d875573d0c39d846aba4b06946fd2bc88) )
	ROM_LOAD( "za4",      0x11000, 0x1000, CRC(ec3b10b1) SHA1(44303ab923d776052ca4d2d09a6a4315d67adc4b) )

	ROM_LOAD( "za7",      0x0c000, 0x1000, CRC(1479f480) SHA1(bd0e9eead0f1213eb3b9653c73257bc704346ab0) )
	ROM_LOAD( "za8",      0x0d000, 0x1000, CRC(4c96cdb2) SHA1(67e506462317d882c4e5c2e16318411d4958ac63) )
	ROM_LOAD( "za9",      0x0e000, 0x1000, CRC(a4f7d9e0) SHA1(a958b4d305bb397aa46d8fdab9dc7e472237ca11) )
	ROM_LOAD( "za10",     0x0f000, 0x1000, CRC(05df1a5a) SHA1(30797838c25cca038023c188cd9fa45277c4190d) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "za25",     0xd000, 0x1000, CRC(779b8558) SHA1(7312e63c23d92c9c52e93cc445a718bc8fe35a0a) )
	ROM_LOAD( "za26",     0xe000, 0x1000, CRC(60a810ce) SHA1(d97e5acea0ef1c208f8e5e95024c83dd6bc9b028) )
	ROM_LOAD( "za27",     0xf000, 0x1000, CRC(99ed424e) SHA1(e4e543dcb77f153aeb78904d11b95381d039299e) )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "za_coin.bin", 0x0000, 0x0800, CRC(364d3557) SHA1(049d0759750c576187053306e07984b1e5877df7) )
ROM_END


ROM_START( zookeep3 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "za12",    0x8000, 0x1000, CRC(4e40d8dc) SHA1(dd7923dcb55a2a1ae0f2029caf5a8904a9ebe8b1) )
	ROM_LOAD( "za13",    0x9000, 0x1000, CRC(eebd5248) SHA1(ebe7f8c436bfefa4236f603fbcbd38d2f4cfd2bd) )
	ROM_LOAD( "za14",    0xa000, 0x1000, CRC(fab43297) SHA1(daa5a780bc9f171da0f6db5319b1519caa09c6c9) )
	ROM_LOAD( "za15",    0xb000, 0x1000, CRC(ef8cd67c) SHA1(f3a2e12ccfa45eb77c2a6e0d9cc0601b99273fd3) )
	ROM_LOAD( "za16",    0xc000, 0x1000, CRC(ccfc15bc) SHA1(56ac6e89825fcde9cf7aeb5d0765a02f5474174a) )
	ROM_LOAD( "za17",    0xd000, 0x1000, CRC(358013f4) SHA1(dd3fae69bf460f89c71e7c78d229dd86605c5950) )
	ROM_LOAD( "za18",    0xe000, 0x1000, CRC(37886afe) SHA1(a0cc902b2d253466e21c4fbf9d3339069fe79ebe) )
	ROM_LOAD( "za19",    0xf000, 0x1000, CRC(bbfb30d9) SHA1(bc6bd5525b159bee7b08b6967cd3088b7bd10fee) )

	ROM_REGION( 0x12000, REGION_CPU2, 0 )
	ROM_LOAD( "za5",     0x0a000, 0x1000, CRC(dc0c3cbd) SHA1(8335cd91bbacc680a3a98a5242d4cb5a6f61b2b5) )
	ROM_LOAD( "za3",     0x10000, 0x1000, CRC(cc4d0aee) SHA1(05c0025e96b432088b46100051a2c780e46b7457) )
	ROM_LOAD( "za6",     0x0b000, 0x1000, CRC(27c787dd) SHA1(1142790d875573d0c39d846aba4b06946fd2bc88) )
	ROM_LOAD( "za4",     0x11000, 0x1000, CRC(ec3b10b1) SHA1(44303ab923d776052ca4d2d09a6a4315d67adc4b) )

	ROM_LOAD( "za7",     0x0c000, 0x1000, CRC(1479f480) SHA1(bd0e9eead0f1213eb3b9653c73257bc704346ab0) )
	ROM_LOAD( "za8",     0x0d000, 0x1000, CRC(4c96cdb2) SHA1(67e506462317d882c4e5c2e16318411d4958ac63) )
	ROM_LOAD( "zv35.9",  0x0e000, 0x1000, CRC(d14123b7) SHA1(5d35bffd2203225937bb83598ffdc31a46a1dbca) )
	ROM_LOAD( "zv36.10", 0x0f000, 0x1000, CRC(23705777) SHA1(952cd8d9ee00268bff2022b2428b1dbfab061254) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "za25",    0xd000, 0x1000, CRC(779b8558) SHA1(7312e63c23d92c9c52e93cc445a718bc8fe35a0a) )
	ROM_LOAD( "za26",    0xe000, 0x1000, CRC(60a810ce) SHA1(d97e5acea0ef1c208f8e5e95024c83dd6bc9b028) )
	ROM_LOAD( "za27",    0xf000, 0x1000, CRC(99ed424e) SHA1(e4e543dcb77f153aeb78904d11b95381d039299e) )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "za_coin.bin", 0x0000, 0x0800, CRC(364d3557) SHA1(049d0759750c576187053306e07984b1e5877df7) )
ROM_END


ROM_START( slither )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "u31.cpu", 0xd800, 0x0800, CRC(3dfff970) SHA1(ee50840e26aa7be226dbe9a32a8344bb75b8de07) )
	ROM_LOAD( "u30.cpu", 0xe000, 0x0800, CRC(8cbc5af8) SHA1(3d563d0bbbce007bd6db6d620e1b0996c67029f6) )
	ROM_LOAD( "u29.cpu", 0xe800, 0x0800, CRC(98c14510) SHA1(7a39b4b691883ad5d079a5c199b93986071c4a49) )
	ROM_LOAD( "u28.cpu", 0xf000, 0x0800, CRC(2762f612) SHA1(2f094832b199d8514ed04c517fca828c75ac7bfa) )
	ROM_LOAD( "u27.cpu", 0xf800, 0x0800, CRC(9306d5b1) SHA1(e5a2c613b1e083b70d63e24dd45472364930398a) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u41.cpu", 0xd000, 0x0800, CRC(e4c60a57) SHA1(5ce9fe3d84b7a5ded372f5c3fc14a335a37ad472) )
	ROM_LOAD( "u40.cpu", 0xd800, 0x0800, CRC(5dcec622) SHA1(4c00b91106d1e505bdbd3aefb46bfb1a17f14fc1) )
	ROM_LOAD( "u39.cpu", 0xe000, 0x0800, CRC(69829c2a) SHA1(5ea1f7e6db2b2cebab8663f7a05496f2e13131f9) )
	ROM_LOAD( "u38.cpu", 0xe800, 0x0800, CRC(6adc64c6) SHA1(0502fa9f793a246ade30d506cce62c5d9b773952) )
	ROM_LOAD( "u37.cpu", 0xf000, 0x0800, CRC(55d31c96) SHA1(cf04b30369407dd9a8f0ef5850fdc13d13a7d56d) )
	ROM_LOAD( "u36.cpu", 0xf800, 0x0800, CRC(d5ffc013) SHA1(d278c0a6be86010cdf4b18afc5099dfd39f26523) )
ROM_END


ROM_START( slithera )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "u31.cpu", 0xd800, 0x0800, CRC(3dfff970) SHA1(ee50840e26aa7be226dbe9a32a8344bb75b8de07) )
	ROM_LOAD( "u30.cpu", 0xe000, 0x0800, CRC(8cbc5af8) SHA1(3d563d0bbbce007bd6db6d620e1b0996c67029f6) )
	ROM_LOAD( "u29.cpu", 0xe800, 0x0800, CRC(98c14510) SHA1(7a39b4b691883ad5d079a5c199b93986071c4a49) )
	ROM_LOAD( "u28.cpu", 0xf000, 0x0800, CRC(2762f612) SHA1(2f094832b199d8514ed04c517fca828c75ac7bfa) )
	ROM_LOAD( "u27.cpu", 0xf800, 0x0800, CRC(9306d5b1) SHA1(e5a2c613b1e083b70d63e24dd45472364930398a) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u41.cpu", 0xd000, 0x0800, CRC(e4c60a57) SHA1(5ce9fe3d84b7a5ded372f5c3fc14a335a37ad472) )
	ROM_LOAD( "u40.cpu", 0xd800, 0x0800, CRC(5dcec622) SHA1(4c00b91106d1e505bdbd3aefb46bfb1a17f14fc1) )
	ROM_LOAD( "u39.cpu", 0xe000, 0x0800, CRC(69829c2a) SHA1(5ea1f7e6db2b2cebab8663f7a05496f2e13131f9) )
	ROM_LOAD( "u38a.cpu",0xe800, 0x0800, CRC(423adfef) SHA1(f10ca9acf31e602a77b011b002bd53cebf5ba502) )
	ROM_LOAD( "u37.cpu", 0xf000, 0x0800, CRC(55d31c96) SHA1(cf04b30369407dd9a8f0ef5850fdc13d13a7d56d) )
	ROM_LOAD( "u36a.cpu",0xf800, 0x0800, CRC(5ac4e244) SHA1(077bf6f3cb98c3f7845393fb57a080d5bdfc2920) )
ROM_END

ROM_START( complexx )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "cx14.bin", 0xa000, 0x1000, CRC(f123a0de) SHA1(fbb64c33d01031e9da78e725dbdaf87d6e33e23c) )
	ROM_LOAD( "cx15.bin", 0xb000, 0x1000, CRC(0fcb966f) SHA1(f7ea6f0ce356629b8133214c7b2e5ede41c54e6c) )
	ROM_LOAD( "cx16.bin", 0xc000, 0x1000, CRC(aa11e0e3) SHA1(4c0b4fc61c682d501ec3dffd7e324d4dc16425a7) )
	ROM_LOAD( "cx17.bin", 0xd000, 0x1000, CRC(f610856e) SHA1(a01edb705cf7c321800c2739beaee584bfc37270) )
	ROM_LOAD( "cx18.bin", 0xe000, 0x1000, CRC(8f8c3984) SHA1(4cfc83c7a972eeb6e386c9f663388b57c1ebfd00) )
	ROM_LOAD( "cx19.bin", 0xf000, 0x1000, CRC(13af3ba8) SHA1(79ce3dce960d89161db89821d9b211ffd1d399d7) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "cx5.bin",  0xa000, 0x1000, CRC(62a2b87b) SHA1(eeecdfd3eeba15cd93d1514132919fdc9254c1cb) )
	ROM_LOAD( "cx6.bin",  0xb000, 0x1000, CRC(dfa7c088) SHA1(626cae67db85ab8f87c59f5945032b4cb6683c8b) )
	ROM_LOAD( "cx7.bin",  0xc000, 0x1000, CRC(c8bd6759) SHA1(5e2debc2f5acf5c14da2d0c3daf49a0f63ade07b) )
	ROM_LOAD( "cx8.bin",  0xd000, 0x1000, CRC(14a57221) SHA1(21e0ac7db246cc0a23f0992e6568a9e737db725a) )
	ROM_LOAD( "cx9.bin",  0xe000, 0x1000, CRC(fc2d4a9f) SHA1(ce16cafe09e2a4411bfe3063136507e649ac7870) )
	ROM_LOAD( "cx10.bin", 0xf000, 0x1000, CRC(96e0c1ad) SHA1(4e67b46cc21b4b3e3259d34c618c6d4e4bb1ae78) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "cx26.bin",    0xe000, 0x1000, CRC(f4f19c9b) SHA1(d14ae6b59016c428a7e08862f1a4ec89f4eac4fb) )
	ROM_LOAD( "cx27.bin",    0xf000, 0x1000, CRC(7e177328) SHA1(bd3d361bb44341a01a8d118a682ad4efa19be8ff) )
ROM_END




/*************************************
 *
 *  Game-specific initialization
 *
 *************************************/

static DRIVER_INIT( kram )
{
	/* we need to override one PIA handler to prevent controls from getting disabled */
	memory_install_write8_handler(0, ADDRESS_SPACE_PROGRAM, 0x9400, 0x97ff, 0, 0, zookeep_pia_0_w);
}


static DRIVER_INIT( kram3 )
{
	static const int cpu0_patch[] =
	{
		0xa080,0x02,0x34, 0xa082,0x23,0xb6, 0xa085,0x89,0x44, 0xa086,0x99,0x44,
		0xa087,0x99,0x44, 0xa088,0x73,0x25, 0xa08a,0x29,0x86, 0xa08c,0xe9,0x1c,
		0xa08e,0xd7,0x35, 0xa090,0x9a,0x44, 0xa091,0x00,0x25, 0xa093,0xa1,0x86,
		0xa095,0xd7,0x1a, 0xa097,0x10,0x20, 0xa099,0xd9,0x34, 0xa09b,0xfc,0x29,
		0xa09d,0xa3,0x8e, 0xa0a0,0x77,0xbd, 0xa0a3,0x31,0xb6, 0xa0a6,0xa5,0x84,
		0xa0a8,0x72,0x27, 0xa0aa,0x0a,0x8e, 0xa0ad,0xbf,0xbd, 0xa0b0,0x17,0x35,
		0xa0b2,0xe2,0x8e, 0xa0b5,0x77,0xbd, 0xa0b8,0x91,0xb6, 0xa0bb,0x84,0x84,
		0xa0bd,0xda,0x27, 0xa0bf,0xa5,0x8e, 0xa0c2,0x77,0xbd, 0xa0c5,0x11,0x20,
		0xa0c7,0x12,0x30, 0xa0c9,0x17,0x26, 0xa0cb,0x36,0x39, 0xa0cc,0x0d,0x34,
		0xa0ce,0x4b,0xb6, 0xa0d1,0xb6,0x84, 0xa0d3,0x05,0x27, 0xa0d5,0x4d,0x7f,
		0xa0d8,0xb3,0x85, 0xa0da,0xed,0x27, 0xa0dc,0x41,0x44, 0xa0dd,0x27,0x35,
		0xa0df,0x7d,0x85, 0xa0e1,0x10,0x26, 0xa0e3,0x34,0xa6, 0xa0e5,0xe2,0x8b,
		0xa0e7,0xc3,0x19, 0xa0e8,0xba,0xa7, 0xa0ea,0x8d,0xbd, 0xa0ed,0x78,0x8e,
		0xa0f0,0x77,0xbd, 0xa0f3,0x75,0xbd, 0xa0f6,0x55,0x2a, 0xa0f8,0xce,0x29,
		0xa0fa,0xaa,0xa6, 0xa0fc,0xa2,0x8b, 0xa0fe,0xf2,0x19, 0xa0ff,0x70,0xa7,
		0xa101,0xc2,0xbd, 0xa104,0xdf,0x8e, 0xa107,0xf0,0x20, 0xa109,0x19,0xa6,
		0xa10b,0x6c,0x8b, 0xa10d,0x60,0x19, 0xa10e,0x0d,0xa7, 0xa110,0x93,0xbd,
		0xa113,0xc1,0x8e, 0xa116,0x3f,0xbd, 0xa119,0x28,0xbd, 0xa11c,0x30,0x2a,
		0xa11e,0x30,0x28, 0xa120,0x70,0xa6, 0xa122,0x4f,0x8b, 0xa124,0x4f,0x19,
		0xa125,0x72,0xa7, 0xa127,0x7b,0xbd, 0xa12a,0x7d,0x8e, 0xa12d,0x04,0x20,
		0xa12f,0x13,0x35, 0xa131,0x7d,0x34, 0xa133,0xcc,0x86, 0xa135,0x5f,0x7d,
		0xa138,0x10,0x26, 0xa13a,0xac,0xfd, 0xa13d,0xb7,0x7d, 0xa140,0x18,0x26,
		0xa142,0x89,0xa6, 0xa144,0xd1,0x10, 0xa148,0x2b,0xbd, 0xa14b,0x17,0x35,
		0xa14d,0x15,0x34, 0xa14f,0x59,0x84, 0xa151,0xf2,0x44, 0xa152,0xf1,0x44,
		0xa153,0xf1,0x44, 0xa154,0x29,0x44, 0xa155,0x54,0x8b, 0xa157,0xb7,0x7d,
		0xa15a,0x01,0x26, 0xa15c,0x1a,0xb7, 0xa15f,0x49,0x86, 0xa161,0x73,0xb7,
		0xa164,0xf7,0x35, 0xa166,0x54,0x84, 0xa168,0x6a,0x8b, 0xa16a,0xa5,0x7d,
		0xa16d,0x10,0x26, 0xa16f,0x1b,0xb7, 0xa172,0x33,0x86, 0xa174,0xdb,0xb7,
		0xa177,0xf1,0x39, 0xa178,0x15,0x34, 0xa17a,0x48,0x81, 0xa17c,0x00,0x22,
		0xa17e,0x4d,0x86, 0xa180,0xe2,0x8b, 0xa182,0x5f,0x7d, 0xa185,0x17,0x26,
		0xa187,0x27,0xb7, 0xa18a,0xc9,0x86, 0xa18c,0xce,0xb7, 0xa18f,0xc1,0xa6,
		0xa191,0xf6,0x8b, 0xa193,0x5f,0x7d, 0xa196,0x01,0x26, 0xa198,0x39,0xb7,
		0xa19b,0x81,0x86, 0xa19d,0x7b,0xb7, 0xa1a0,0x17,0x35, 0xa1a2,0x5f,0x6f,
		0xa1a4,0x5c,0x6c, 0xa1a6,0xa5,0x80, 0xa1a8,0xce,0x81, 0xa1aa,0x85,0x22,
		0xa1ac,0xda,0xa7, 0xa1ae,0xdc,0xa6, 0xa1b0,0x04,0x20, 0xa1b2,0x06,0x34,
		0xa1b4,0xb0,0x86, 0xa1b6,0x5d,0x7d, 0xa1b9,0xd8,0x26, 0xa1bb,0x8a,0xb7,
		0xa1be,0x15,0x35, 0xa1c0,0xb0,0x86, 0xa1c2,0x5f,0x7d, 0xa1c5,0x07,0x26,
		0xa1c7,0x27,0xb7, 0xa1ca,0xdf,0x7d, 0xa1cd,0x8f,0x26, 0xa1cf,0x9b,0x34,
		0xa1d1,0xf0,0x8e, 0xa1d4,0xeb,0xce, 0xa1d7,0xcf,0x5f, 0xa1d8,0x9e,0x5c,
		0xa1d9,0xff,0xc1, 0xa1db,0x25,0x27, 0xa1dd,0xcf,0x10, 0xa1e0,0x30,0xa6,
		0xa1e2,0x4d,0x7d, 0xa1e5,0x10,0x26, 0xa1e7,0x3b,0xf7, 0xa1ea,0x77,0x10,
		0xa1ee,0xf2,0xbd, 0xa1f1,0x04,0x20, 0xa1f3,0x05,0x35, 0xa1f5,0xd3,0x1f,
		0xa1f7,0xaf,0xc4, 0xa1f9,0x7e,0x85, 0xa1fb,0x20,0x27, 0xa1fd,0x28,0xca,
		0xa1ff,0x8d,0x85, 0xa201,0x58,0x27, 0xa203,0xa0,0xca, 0xa205,0xc4,0x85,
		0xa207,0x65,0x27, 0xa209,0xfd,0xca, 0xa20b,0x5d,0x85, 0xa20d,0x06,0x27,
		0xa20f,0xe8,0xca, 0xa211,0x53,0x1e, 0xa213,0x76,0x39, 0xa214,0x34,0x86,
		0xa216,0xaf,0x1f, 0xa218,0x62,0x1c, 0xa21a,0x49,0x86, 0xa21c,0x77,0x0d,
		0xa21e,0x01,0x26, 0xa220,0x53,0x97, 0xa222,0x55,0x86, 0xa224,0xde,0x0d,
		0xa226,0xf5,0x26, 0xa228,0x5b,0x97, 0xa22a,0xfc,0xcc, 0xa22d,0x76,0x0d,
		0xa22f,0x15,0x26, 0xa231,0x97,0xdd, 0xa233,0xe2,0xbd, 0xa236,0x32,0x86,
		0xa238,0x45,0x10, 0xa23c,0x76,0x0d, 0xa23e,0x12,0x26, 0xa240,0xe8,0x97,
		0xa242,0x7a,0x10, 0xa245,0xc7,0x86, 0xa247,0x52,0x10, 0xa24b,0x75,0x0d,
		0xa24d,0x07,0x26, 0xa24f,0x4f,0x97, 0xa251,0xa2,0x10, 0xa254,0x81,0xcc,
		0xa257,0xbd,0x0d, 0xa259,0x14,0x26, 0xa25b,0xff,0xdd, 0xa25d,0xb5,0x6f,
		0xa25f,0xe9,0xce, 0xa262,0xc6,0xae, 0xa264,0x45,0x10, 0xa268,0x18,0xa6,
		0xa26a,0x0c,0xa7, 0xa26c,0x18,0xa6, 0xa26e,0x6e,0x8b, 0xa270,0x4f,0x19,
		0xa271,0x12,0xa7, 0xa273,0x72,0xc6, 0xa275,0xd4,0x64, 0xa277,0x93,0x24,
		0xa279,0xd8,0xc6, 0xa27b,0x49,0x81, 0xa27d,0x10,0x26, 0xa27f,0xda,0xc6,
		0xa281,0xd2,0x0d, 0xa283,0x01,0x26, 0xa285,0xbe,0xd7, 0xa287,0x92,0x10,
		0xa28a,0x41,0xe6, 0xa28c,0x1d,0x54, 0xa28d,0x15,0x54, 0xa28e,0x1b,0x54,
		0xa28f,0x13,0x54, 0xa290,0xfc,0xca, 0xa292,0xa1,0x86, 0xa294,0xef,0xdd,
		0xa296,0x29,0xe6, 0xa298,0xba,0xc4, 0xa29a,0x88,0xca, 0xa29c,0x1f,0x0d,
		0xa29e,0x21,0x26, 0xa2a0,0xff,0xdd, 0xa2a2,0x02,0x31, 0xa2a4,0xd3,0x1f,
		0xa2a6,0xbe,0xc5, 0xa2a8,0x90,0x26, 0xa2aa,0x82,0x31, 0xa2ae,0xbb,0x1f,
		0xa2b0,0x01,0x32, 0xa2b2,0xa1,0x85, 0xa2b4,0x10,0x26, 0xa2b6,0x07,0x30,
		0xa2b8,0x6f,0x11, 0xa2bc,0x30,0x26, 0xa2be,0x23,0x20, 0xa2c0,0x71,0xae,
		0xa2c2,0x10,0x20, 0xa2c4,0x27,0xb6, 0xa2c7,0x99,0x44, 0xa2c8,0xfb,0x24,
		0xa2ca,0x4d,0x6f, 0xa2cc,0x99,0x20, 0xa2ce,0xcb,0xb6, 0xa2d1,0x9e,0x44,
		0xa2d2,0x11,0x24, 0xa2d4,0x17,0x32, 0xa2d6,0x4f,0x7f, 0xa2d9,0xde,0xcc,
		0xa2dc,0xe0,0x0d, 0xa2de,0xc9,0x26, 0xa2e0,0xff,0xdd, 0xa2e2,0x61,0xbd,
		0xa2e5,0xf8,0xce, 0xa2e8,0x1c,0xec, 0xa2ea,0xf3,0x10, 0xa2ed,0xf8,0xae,
		0xa2ef,0xb6,0x0d, 0xa2f1,0x10,0x26, 0xa2f3,0xb4,0x97, 0xa2f5,0xb0,0x86,
		0xa2f7,0xd1,0x0d, 0xa2f9,0x30,0x26, 0xa2fb,0x2f,0x10, 0xa2fe,0x22,0xdd,
		0xa300,0xce,0x31, 0xa304,0x21,0x30, 0xa306,0x21,0x26, 0xa308,0x46,0x11,
		0xa30c,0x17,0x26, 0xa30e,0xf9,0xcc, 0xa311,0x50,0x0d, 0xa313,0x69,0x26,
		0xa315,0xf9,0xdd, 0xa317,0x83,0xb6, 0xa31a,0x59,0x84, 0xa31c,0x15,0x27,
		0xa31e,0xa7,0x7f, 0xa321,0xdc,0xcc, 0xa324,0x1e,0x0d, 0xa326,0x34,0x26,
		0xa328,0xff,0xdd, 0xa32a,0x29,0xbd, 0xa32d,0xf2,0x4f, 0xa32e,0x7c,0x8e,
		0xa331,0x4d,0x10, 0xa335,0x3b,0x1f, 0xa337,0x64,0x54, 0xa338,0xd5,0x54,
		0xa339,0xd5,0x54, 0xa33a,0xd0,0x54, 0xa33b,0x8c,0xe6, 0xa33d,0x76,0x0d,
		0xa33f,0x01,0x26, 0xa341,0x16,0xd7, 0xa343,0x8f,0x5f, 0xa344,0x1c,0xdd,
		0xa346,0x4f,0x9f, 0xa348,0xf2,0x4c, 0xa349,0x07,0x26, 0xa34b,0xf9,0xcc,
		0xa34e,0x75,0x0d, 0xa350,0xcc,0x26, 0xa352,0x37,0xdd, 0xa354,0xdb,0xb6,
		0xa357,0x5d,0x85, 0xa359,0x11,0x27, 0xa35b,0xa7,0x7f, 0xa35e,0xf9,0xcc,
		0xa361,0xde,0x0d, 0xa363,0xf5,0x26, 0xa365,0x97,0xdd, 0xa367,0xf2,0xbd,
		0xa36a,0x4a,0x86, 0xa36c,0x76,0x0d, 0xa36e,0x15,0x26, 0xa370,0x7b,0x97,
		0xa372,0x32,0xcc, 0xa375,0x50,0x8e, 0xa378,0x45,0x10, 0xa37c,0x76,0x0d,
		0xa37e,0x05,0x26, 0xa380,0xea,0xdd, 0xa382,0x92,0x10, 0xa385,0xc4,0x0d,
		0xa387,0x01,0x26, 0xa389,0x66,0x9f, 0xa38b,0xc1,0xc6, 0xa38d,0x2c,0x0d,
		0xa38f,0x01,0x26, 0xa391,0xaa,0xd7, 0xa393,0x92,0x10, 0xa396,0xf9,0xcc,
		0xa399,0xf0,0x0d, 0xa39b,0xc9,0x26, 0xa39d,0xd1,0xdd, 0xa39f,0x47,0x9f,
		0xa3a1,0xf0,0x8e, 0xa3a4,0x85,0x10, 0xa3a8,0x18,0xce, 0xa3ab,0xf5,0xcc,
		0xa3ae,0x37,0x0d, 0xa3b0,0x10,0x26, 0xa3b2,0xfa,0xdd, 0xa3b4,0xb8,0xc6,
		0xa3b6,0xd1,0x0d, 0xa3b8,0x18,0x26, 0xa3ba,0xd3,0xd7, 0xa3bc,0x0d,0x10,
		0xa3bf,0xd3,0xcb, 0xa3c1,0x05,0x24, 0xa3c3,0xf9,0xcc, 0xa3c6,0xd5,0x0d,
		0xa3c8,0x78,0x26, 0xa3ca,0xa7,0xd7, 0xa3cc,0xac,0x0d, 0xa3ce,0x81,0x26,
		0xa3d0,0xb5,0x97, 0xa3d2,0xe7,0x9f, 0xa3d4,0xe6,0x8b, 0xa3d6,0x11,0x24,
		0xa3d8,0xac,0x86, 0xa3da,0xdd,0x0d, 0xa3dc,0xeb,0x26, 0xa3de,0xa7,0x97,
		0xa3e0,0xfc,0xcc, 0xa3e3,0xf5,0x8e, 0xa3e6,0x80,0x10, 0xa3ea,0x41,0x0d,
		0xa3ec,0x90,0x26, 0xa3ee,0x7a,0xdd, 0xa3f0,0x85,0x10, 0xa3f3,0xc1,0x0d,
		0xa3f5,0x10,0x26, 0xa3f7,0xe0,0x9f, 0xa3f9,0xf8,0xc6, 0xa3fb,0x09,0x0d,
		0xa3fd,0xd0,0x26, 0xa3ff,0x08,0xd7, 0xa401,0x00,0x10, 0xa404,0x5c,0xcc,
		0xa407,0x1d,0x0d, 0xa409,0x16,0x26, 0xa40b,0xff,0xdd, 0xa40d,0x6f,0x9f,
		0xa40f,0xf9,0xcc, 0xa412,0xf5,0x0d, 0xa414,0x94,0x26, 0xa416,0x7f,0xdd,
		0xa418,0x09,0xb6, 0xa41b,0x5d,0x85, 0xa41d,0x02,0x27, 0xa41f,0xa7,0x7f,
		0xa422,0x6e,0x86, 0xa424,0x9b,0x1f, 0xa426,0x34,0xb6, 0xa429,0x5c,0x84,
		0xa42b,0x1e,0xb7, 0xa42e,0x75,0x1c, 0xa430,0x34,0xcc, 0xa433,0x46,0x0d,
		0xa435,0x78,0x26, 0xa437,0x82,0xdd, 0xa439,0x58,0x86, 0xa43b,0x66,0x0d,
		0xa43d,0x10,0x26, 0xa43f,0x4b,0x97, 0xa441,0x7c,0xbd, 0xa444,0xc9,0xb6,
		0xa447,0x51,0x84, 0xa449,0x06,0x27, 0xa44b,0xe5,0x4f, 0xa44c,0x08,0xb7,
		0xa44f,0x52,0x10, 0xa453,0xbd,0x0d, 0xa455,0x94,0x26, 0xa457,0xb2,0x10,
		0xa45a,0x49,0x86, 0xa45c,0x6b,0x8e, 0xa45f,0x75,0x0d, 0xa461,0xf0,0x26,
		0xa463,0x53,0x97, 0xa465,0x5b,0x9f, 0xa467,0x5d,0x8e, 0xa46a,0x44,0x10,
		0xa46e,0x8a,0xf6, 0xa471,0xfd,0x5c, 0xa472,0x1a,0x0d, 0xa474,0x78,0x26,
		0xa476,0x8b,0xd7, 0xa478,0x45,0x10, 0xa47b,0x1e,0xb6, 0xa47e,0x4c,0x84,
		0xa480,0x10,0x27, 0xa482,0x4f,0x7f, 0xa485,0xb5,0x85, 0xa487,0x01,0x26,
		0xa489,0xd1,0xc1, 0xa48b,0x0c,0x23, 0xa48d,0xa7,0x5f, 0xa48e,0xc7,0xf7,
		0xa491,0xc0,0x0d, 0xa493,0x01,0x26, 0xa495,0xf2,0x9f, 0xa497,0xd5,0x0d,
		0xa499,0xcb,0x26, 0xa49b,0xd8,0x20, 0xa49d,0x21,0xcc, 0xa4a0,0x5b,0x7f,
		0xa4a3,0x36,0xb7, 0xa4a6,0x3e,0xf7, 0xa4a9,0xf4,0xcc, 0xa4ac,0x3e,0x0d,
		0xa4ae,0x10,0x26, 0xa4b0,0xff,0xdd, 0xa4b2,0x61,0xbd, 0xa4b5,0xb0,0x86,
		0xa4b7,0x29,0xf6, 0xa4ba,0x5b,0x58, 0xa4bb,0xa5,0x8e, 0xa4be,0x06,0x3a,
		0xa4bf,0xe6,0x0d, 0xa4c1,0x07,0x26, 0xa4c3,0xa7,0x97, 0xa4c5,0xf6,0x9f,
		0xa4c7,0xe9,0xce, 0xa4ca,0xc1,0xc6, 0xa4cc,0xe7,0x10, 0xa4d0,0xa6,0x86,
		0xa4d2,0x13,0x34, 0xa4d4,0x23,0xa6, 0xa4d6,0xd5,0x0d, 0xa4d8,0x0e,0x26,
		0xa4da,0x47,0x97, 0xa4dc,0xd5,0x10, 0xa4df,0x61,0xa6, 0xa4e1,0x77,0xbd,
		0xa4e4,0x07,0x31, 0xa4e6,0xcb,0x5c, 0xa4e7,0x35,0xa6, 0xa4e9,0x3c,0x4c,
		0xa4ea,0x5b,0xa7, 0xa4ec,0xa6,0x81, 0xa4ee,0xee,0x26, 0xa4f0,0x01,0x32,
		0xa4f2,0xf1,0x8e, 0xa4f5,0xb0,0x86, 0xa4f7,0xd1,0x0d, 0xa4f9,0x30,0x26,
		0xa4fb,0x7c,0x97, 0xa4fd,0x5b,0x9f, 0xa4ff,0xfa,0xc6, 0xa501,0x50,0x10,
		0xa505,0x7d,0xbd, 0xa508,0x05,0x25, 0xa50a,0x16,0x31, 0xa50c,0xf4,0x5c,
		0xa50d,0xe4,0x4c, 0xa50e,0x5c,0x81, 0xa510,0x18,0x27, 0xa512,0x35,0x0d,
		0xa514,0xb4,0x26, 0xa516,0x07,0x97, 0xa518,0x68,0x9f, 0xa51a,0x10,0x20,
		0xa51c,0x5c,0x86, 0xa51e,0x75,0x0d, 0xa520,0x90,0x26, 0xa522,0x37,0x97,
		0xa524,0xdb,0x9f, 0xa526,0xf5,0xb6, 0xa529,0x5c,0x84, 0xa52b,0x13,0x27,
		0xa52d,0xb3,0x7f, 0xa530,0xbe,0x85, 0xa532,0x0e,0x26, 0xa534,0x16,0x85,
		0xa536,0xb2,0x26, 0xa538,0x58,0x86, 0xa53a,0x65,0x0d, 0xa53c,0x10,0x26,
		0xa53e,0x5c,0x97, 0xa540,0x72,0x0d, 0xa542,0x49,0x26, 0xa544,0xf2,0x7e,
		0xa547,0xf9,0xcc, 0xa54a,0x75,0x0d, 0xa54c,0x02,0x26, 0xa54e,0xff,0xdd,
		0xa550,0x7c,0xbd, 0xa553,0x21,0x86, 0xa555,0x29,0xf6, 0xa558,0xf1,0x58,
		0xa559,0x78,0x8e, 0xa55c,0x37,0x3a, 0xa55d,0x72,0x0d, 0xa55f,0x01,0x26,
		0xa561,0x13,0x97, 0xa563,0x1e,0x9f, 0xa565,0xb8,0x8e, 0xa568,0x58,0x86,
		0xa56a,0xee,0xce, 0xa56d,0x45,0x10, 0xa571,0xbe,0x0d, 0xa573,0x82,0x26,
		0xa575,0x5b,0x97, 0xa577,0xdb,0x10, 0xa57a,0xdd,0xc6, 0xa57c,0x18,0xa6,
		0xa57e,0x05,0x26, 0xa580,0x90,0x10, 0xa584,0xc5,0x0d, 0xa586,0x01,0x26,
		0xa588,0x38,0x10, 0xa58b,0xb5,0x0d, 0xa58d,0x86,0x26, 0xa58f,0x9a,0x10,
		0xa592,0x06,0x33, 0xa594,0xc8,0x5a, 0xa595,0xad,0xc1, 0xa597,0x01,0x26,
		0xa599,0x9b,0xa6, 0xa59b,0xbf,0xbd, 0xa59e,0xaa,0x5a, 0xa59f,0x2d,0x27,
		0xa5a1,0x30,0xa6, 0xa5a3,0x76,0xbd, 0xa5a6,0xcd,0x5a, 0xa5a7,0x11,0x26,
		0xa5a9,0x85,0x10, 0xa5ac,0x8f,0x31, 0xa5b0,0xd6,0x0d, 0xa5b2,0x06,0x26,
		0xa5b4,0x85,0x10, 0xa5b7,0xa1,0x86, 0xa5b9,0xc5,0x10, 0xa5bd,0xf8,0xe6,
		0xa5bf,0x02,0x0d, 0xa5c1,0x16,0x26, 0xa5c3,0xff,0xdd, 0xa5c5,0x13,0x31,
		0xa5c7,0x01,0x26, 0xa5c9,0x08,0x10, 0xa5cc,0x55,0x31, 0xa5d0,0xc4,0x0d,
		0xa5d2,0x01,0x26, 0xa5d4,0x95,0x10, 0xa5d7,0x96,0x11, 0xa5db,0x21,0x26,
		0xa5dd,0x99,0x86, 0xa5df,0xfd,0x0d, 0xa5e1,0x10,0x26, 0xa5e3,0xb6,0x97,
		0xa5e5,0xf3,0x9f, 0xa5e7,0x37,0xb6, 0xa5ea,0xa7,0x84, 0xa5ec,0x52,0x27,
		0xa5ee,0xe7,0x7f, 0xa5f1,0xb6,0x85, 0xa5f3,0x02,0x26, 0xa5f5,0xb6,0x85,
		0xa5f7,0x93,0x10, 0xa5fb,0x6a,0x86, 0xa5fd,0xfe,0x0d, 0xa5ff,0xc6,0x26,
		0xa601,0xd7,0x97, 0xa603,0x5d,0x0d, 0xa605,0x66,0x26, 0xa607,0x2b,0x7e,
		0xa60a,0xf9,0xcc, 0xa60d,0x65,0x0d, 0xa60f,0x01,0x26, 0xa611,0x90,0xdd,
		0xa613,0xc9,0x86, 0xa615,0x40,0x0d, 0xa617,0x69,0x26, 0xa619,0x5a,0x97,
		0xa61b,0x3f,0xbd, 0xa61e,0x8b,0xf6, 0xa621,0x0d,0x58, 0xa622,0x66,0x8e,
		0xa625,0x69,0x3a, 0xa626,0x51,0x86, 0xa628,0x76,0x0d, 0xa62a,0x06,0x26,
		0xa62c,0x5b,0x97, 0xa62e,0x7f,0x9f, 0xa630,0x58,0x8e, 0xa633,0xcd,0x86,
		0xa635,0x90,0xce, 0xa638,0x45,0x10, 0xa63c,0x76,0x0d, 0xa63e,0x12,0x26,
		0xa640,0xe8,0x97, 0xa642,0x7a,0x10, 0xa645,0x44,0xc6, 0xa647,0x21,0xa6,
		0xa649,0x04,0x26, 0xa64b,0x52,0x10, 0xa64f,0x75,0x0d, 0xa651,0x94,0x26,
		0xa653,0xb2,0x10, 0xa656,0x55,0x0d, 0xa658,0x11,0x26, 0xa65a,0x52,0x10,
		0xa65d,0x01,0x33, 0xa65f,0xe2,0x5a, 0xa660,0xae,0xc1, 0xa662,0x4e,0x26,
		0xa664,0xb0,0xa6, 0xa666,0xaf,0xbd, 0xa669,0xe1,0x5a, 0xa66a,0x04,0x27,
		0xa66c,0x18,0xa6, 0xa66e,0x2f,0xbd, 0xa671,0xe9,0x5a, 0xa672,0x4e,0x26,
		0xa674,0x4d,0x10, 0xa677,0x85,0x31, 0xa67b,0x71,0x0d, 0xa67d,0x10,0x26,
		0xa67f,0x47,0x10, 0xa682,0x96,0x11, 0xa686,0x01,0x26, 0xa688,0xb9,0x86,
		0xa68a,0xdd,0x0d, 0xa68c,0x86,0x26, 0xa68e,0xcf,0x97, 0xa690,0xf7,0x9f,
		0xa692,0x23,0xb6, 0xa695,0xa1,0x84, 0xa697,0x05,0x27, 0xa699,0x7a,0x7f,
		0xa69c,0x7f,0x85, 0xa69e,0x21,0x26, 0xa6a0,0xb0,0x86, 0xa6a2,0xc5,0x0d,
		0xa6a4,0x10,0x26, 0xa6a6,0xb7,0x97, 0xa6a8,0xde,0x0d, 0xa6aa,0xe6,0x26,
		0xa6ac,0x31,0x7e, 0xa6af,0x70,0xcc, 0xa6b2,0xd2,0x0d, 0xa6b4,0x10,0x26,
		0xa6b6,0xfd,0xdd, 0xa6b8,0xf7,0xbd, 0xa6bb,0x85,0x86, 0xa6bd,0x19,0xf6,
		0xa6c0,0xcf,0x58, 0xa6c1,0xe4,0x8e, 0xa6c4,0x46,0x3a, 0xa6c5,0xc7,0x0d,
		0xa6c7,0x01,0x26, 0xa6c9,0x5d,0x97, 0xa6cb,0x6f,0x9f, 0xa6cd,0x4c,0x86,
		0xa6cf,0x69,0x8e, 0xa6d2,0xd5,0x0d, 0xa6d4,0x03,0x26, 0xa6d6,0xa7,0x97,
		0xa6d8,0x80,0x9f, 0xa6da,0x45,0x4f, 0xa6db,0x57,0xbd, 0xa6de,0x31,0xcc,
		0xa6e1,0xd6,0x0d, 0xa6e3,0x13,0x26, 0xa6e5,0xff,0xdd, 0xa6e7,0x3f,0xf6,
		0xa6ea,0x27,0xc4, 0xa6ec,0x1a,0x27, 0xa6ee,0xae,0x7f, 0xa6f1,0xbe,0xc5,
		0xa6f3,0x12,0x26, 0xa6f5,0xbe,0xc5, 0xa6f7,0x12,0x26, 0xa6f9,0xfe,0xc5,
		0xa6fb,0x30,0x27, 0xa6fd,0x71,0xb6, 0xa700,0x74,0x8b, 0xa702,0xf6,0x19,
		0xa703,0x94,0x81, 0xa705,0xbf,0x2f, 0xa707,0x01,0x86, 0xa709,0x1d,0xb7,
		0xa70c,0x39,0xbd, 0xa70f,0x69,0x8e, 0xa712,0xd7,0xbd, 0xa715,0x99,0xbd,
		0xa718,0x35,0x2a, 0xa71a,0x34,0x29, 0xa71c,0x12,0x20, 0xa71e,0x0b,0xb6,
		0xa721,0x8a,0x8b, 0xa723,0x0c,0x19, 0xa724,0x86,0x81, 0xa726,0x28,0x2c,
		0xa728,0x58,0x86, 0xa72a,0x17,0x20, 0xa72c,0xb3,0x7f, 0xa72f,0x45,0x10,
		0xa733,0x47,0x0d, 0xa735,0x10,0x26, 0xa737,0x34,0x10, 0xa73a,0x29,0xbd,
		0xa73d,0xfc,0xcc, 0xa740,0xf0,0xce, 0xa743,0x49,0x8e, 0xa746,0xd2,0x10,
		0xa74a,0xf1,0x4c, 0xa74b,0x5c,0x81, 0xa74d,0x24,0x2d, 0xa74f,0x49,0x86,
		0xa751,0xbb,0x0d, 0xa753,0x81,0x26, 0xa755,0x48,0x97, 0xa757,0xda,0x10,
		0xa75a,0x10,0x20, 0xa75c,0x60,0x0d, 0xa75e,0x01,0x26, 0xa760,0xbb,0x97,
		0xa762,0x3f,0x10, 0xa765,0x45,0x10, 0xa768,0x3f,0xbd, 0xa76b,0x16,0x25,
		0xa76d,0xf5,0x5c, 0xa76e,0x04,0x20, 0xa770,0x74,0xcc, 0xa773,0xa9,0x0d,
		0xa775,0xf8,0x26, 0xa777,0x01,0xdd, 0xa779,0x58,0x86, 0xa77b,0x61,0x0d,
		0xa77d,0x10,0x26, 0xa77f,0x48,0x97, 0xa781,0x5c,0x7e, 0xa784,0xd6,0x1a,
		0xa786,0x92,0x10, 0xa78a,0xa9,0x86, 0xa78c,0x33,0x1f, 0xa78e,0xaf,0x7f,
		0xa791,0xc1,0x1c, 0xa793,0x10,0x20, 0xa795,0xa2,0x86, 0xa797,0xc7,0x1f,
		0xa799,0xe7,0x1a, 0xa79b,0x21,0xce, 0xa79e,0x1a,0x10, 0xa7a2,0x66,0xae,
		0xa7a4,0xb8,0xc6, 0xa7a6,0x4b,0x6f, 0xa7a8,0xb0,0xa6, 0xa7aa,0xa4,0xa7,
		0xa7ac,0x58,0xa6, 0xa7ae,0x4b,0xa7, 0xa7b0,0x30,0xa6, 0xa7b2,0x24,0xa7,
		0xa7b4,0xc9,0x5a, 0xa7b5,0x10,0x26, 0xa7b7,0x96,0x11, 0xa7bb,0xcc,0x26,
		0xa7bd,0x13,0x1f, 0xa7bf,0xeb,0xe7, 0xa7c1,0xe5,0x8c, 0xa7c4,0x07,0x26,
		0xa7c6,0xa1,0x86, 0xa7c8,0x29,0xb7, 0xa7cb,0xc7,0xb7, 0xa7ce,0xc9,0x86,
		0xa7d0,0x21,0xb7, 0xa7d3,0x27,0xb7, 0xa7d6,0xd3,0x1c, 0xa7d8,0xc1,0x7f,
		0xa7db,0x63,0xb6, 0xa7de,0x80,0x8a, 0xa7e0,0x33,0xb7, 0xa7e3,0x59,0x7e,
		0xa7e6,0x5b,0x7f, 0xa7e9,0x37,0x7d, 0xa7ec,0x8d,0x10, 0xa7f0,0x5f,0x7d,
		0xa7f3,0x00,0x27, 0xa7f5,0x85,0x10, 0xa7f9,0x58,0x8e, 0xa7fc,0xb9,0xb6,
		0xa7ff,0x60,0x7f, 0xa802,0xf3,0x1c, 0xa804,0xff,0x6e, 0xa806,0xae,0x3b,
		0xa811,0xfb,0xb6, 0xa814,0xa2,0x10, 0xa818,0x18,0xb6, 0xa81b,0x52,0x10,
		0xa81f,0x0b,0xb6, 0xa822,0x4a,0xb6, 0xa825,0x32,0x27, 0xa827,0xb2,0xb7,
		0xa82a,0x0a,0xb6, 0xa82d,0x10,0x26, 0xa82f,0x98,0xf6, 0xa832,0x74,0xf1,
		0xa835,0x7e,0x25, 0xa837,0xe9,0xb6, 0xa83a,0x4a,0x84, 0xa83c,0x1b,0xb7,
		0xa83f,0x14,0x20, 0xa841,0x77,0xb6, 0xa844,0xc3,0x8a, 0xa846,0xef,0xb7,
		0xa849,0x1f,0xb6, 0xa84c,0xc2,0x43, 0xa84d,0x07,0x26, 0xa84f,0x0f,0xb7,
		0xa852,0x78,0x20, 0xa854,0xbb,0xf6, 0xa857,0xe6,0x53, 0xa858,0x0a,0xb4,
		0xa85b,0x8f,0xf7, 0xa85e,0x67,0x1f, 0xa860,0x6a,0x43, 0xa861,0x69,0xba,
		0xa864,0xbb,0xf7, 0xa867,0xe7,0x43, 0xa868,0x12,0x27, 0xa86a,0x3e,0xba,
		0xa86d,0x1b,0xb7, 0xa870,0xf9,0xb6, 0xa873,0x90,0x27, 0xa875,0xc9,0x7a,
		0xa878,0x10,0x26, 0xa87a,0x0e,0xb6, 0xa87d,0x5e,0x85, 0xa87f,0x10,0x27,
		0xa881,0xb5,0x84, 0xa883,0x27,0xb7, 0xa886,0xa1,0x86, 0xa888,0xa0,0xb7,
		0xa88b,0x18,0x20, 0xa88d,0xcf,0xf6, 0xa890,0x01,0x27, 0xa892,0x23,0xb6,
		0xa895,0xe7,0x8a, 0xa897,0x27,0xb7, 0xa89a,0xa2,0x7a, 0xa89d,0x83,0x86,
		0xa89f,0x8f,0xb7, 0xa8a2,0x22,0xb6, 0xa8a5,0x40,0x2a, 0xa8a7,0x5b,0x7d,
		0xa8aa,0xd4,0x2b, 0xa8ac,0xb3,0x7f, 0xa8af,0x54,0x86, 0xa8b1,0x5f,0x7d,
		0xa8b4,0x10,0x26, 0xa8b6,0x34,0xb7, 0xa8b9,0x9b,0xb7, 0xa8bc,0x91,0x7e,
		0xa8bf,0x89,0xb6, 0xa8c2,0x40,0x2a, 0xa8c4,0x52,0x3b, 0xa8c5,0xde,0x5f,
		0xa8c6,0xb5,0x85, 0xa8c8,0x79,0x27, 0xa8ca,0xca,0x7a, 0xa8cd,0x27,0x26,
		0xa8cf,0x0f,0x1f, 0xa8d1,0xb4,0x86, 0xa8d3,0x27,0xb7, 0xa8d6,0xc7,0x1f,
		0xa8d8,0x6c,0x7a, 0xa8db,0x21,0x26, 0xa8dd,0x7c,0x86, 0xa8df,0x67,0xb7,
		0xa8e2,0xc4,0x1f, 0xa8e4,0xe8,0xca, 0xa8e6,0xb6,0x85, 0xa8e8,0x7a,0x27,
		0xa8ea,0xde,0x7a, 0xa8ed,0x30,0x26, 0xa8ef,0x1b,0x1f, 0xa8f1,0xb0,0x86,
		0xa8f3,0x34,0xb7, 0xa8f6,0xc3,0x1f, 0xa8f8,0x69,0x7a, 0xa8fb,0x32,0x26,
		0xa8fd,0x78,0x86, 0xa8ff,0x74,0xb7, 0xa902,0xe7,0x1f, 0xa904,0xd3,0xca,
		0xa906,0x38,0x20, 0xa908,0x09,0xb6, 0xa90b,0x5d,0x85, 0xa90d,0x16,0x27,
		0xa90f,0x52,0x10, 0xa913,0xdd,0x85, 0xa915,0x03,0x27, 0xa917,0x2a,0x1a,
		0xa919,0xa2,0x7f, 0xa91c,0x5c,0x86, 0xa91e,0x0f,0xb7, 0xa921,0xb3,0xb7,
		0xa924,0xb9,0xb6, 0xa927,0x96,0x27, 0xa929,0xb3,0x7f, 0xa92c,0x58,0x86,
		0xa92e,0x1a,0xb7, 0xa931,0x53,0x1f, 0xa933,0x22,0x7f, 0xa936,0xfc,0x7f,
		0xa939,0x75,0x1c, 0xa93b,0xed,0xcc, 0xa93e,0xb5,0x7d, 0xa941,0xf7,0x26,
		0xa943,0x67,0xb7, 0xa946,0xbf,0x7d, 0xa949,0x17,0x26, 0xa94b,0x8f,0xf7,
		0xa94e,0x49,0x86, 0xa950,0xd7,0x7d, 0xa953,0x81,0x26, 0xa955,0x68,0xb7,
		0xa958,0xb4,0x7e, 0xa95b,0x49,0x86, 0xa95d,0x08,0xb7, 0xa960,0x93,0xb7,
		0xa963,0xb8,0x7e, 0xa966,0x70,0xb6, 0xa969,0x5e,0x85, 0xa96b,0x16,0x27,
		0xa96d,0xe8,0xca, 0xa96f,0xf3,0x5d, 0xa970,0xad,0x10, 0xa974,0x17,0x7d,
		0xa977,0xee,0x26, 0xa979,0xe8,0xca, 0xa97b,0x8c,0xf7, 0xa97e,0x8b,0xf7,
		0xa981,0x4f,0x7e, 0xa984,0x42,0x3b, 0xa985,0xde,0x4f, 0xa986,0xc7,0x1f,
		0xa988,0xcd,0x1a, 0xa98a,0xa3,0xb6, 0xa98d,0x7d,0x84, 0xa98f,0x67,0xb7,
		0xa992,0xf9,0xcc, 0xa995,0x79,0xfd, 0xa998,0xe0,0x8e, 0xa99b,0x21,0xce,
		0xa99e,0xaf,0x5f, 0xa99f,0x69,0xa6, 0xa9a1,0x12,0x27, 0xa9a3,0x81,0x10,
		0xa9a7,0x6e,0xeb, 0xa9a9,0x0f,0x31, 0xa9ab,0x1c,0x26, 0xa9ad,0xe6,0xe1,
		0xa9af,0x36,0x27, 0xa9b1,0xd3,0x1f, 0xa9b3,0xb5,0x80, 0xa9b5,0x9c,0x44,
		0xa9b6,0xe2,0x8a, 0xa9b8,0x02,0x43, 0xa9b9,0xbf,0xbd, 0xa9bc,0xd8,0xc6,
		0xa9be,0x54,0xb7, 0xa9c1,0x2e,0xf7, 0xa9c4,0x37,0xa6, 0xa9c6,0x10,0x20,
		0xa9c8,0x3e,0x30, 0xa9cc,0xfd,0x8c, 0xa9cf,0x89,0x26, 0xa9d1,0xfa,0xcc,
		0xa9d4,0x58,0x7d, 0xa9d7,0x01,0x26, 0xa9d9,0xc3,0xfd, 0xa9dc,0x83,0x8e,
		0xa9df,0x09,0xc6, 0xa9e1,0xb0,0x86, 0xa9e3,0x36,0xa7, 0xa9e5,0x26,0xa1,
		0xa9e7,0x14,0x26, 0xa9e9,0x2c,0x48, 0xa9ea,0x2b,0x26, 0xa9ec,0xd8,0x86,
		0xa9ee,0xd6,0xa7, 0xa9f0,0x26,0xa1, 0xa9f2,0x13,0x26, 0xa9f4,0xc1,0x1a,
		0xa9f6,0xde,0x49, 0xa9f7,0xb5,0x81, 0xa9f9,0x30,0x26, 0xa9fb,0xe0,0xe7,
		0xa9fd,0x2a,0xcb, 0xa9ff,0x3d,0xc9, 0xaa01,0x8d,0x8c, 0xaa04,0xd9,0x26,
		0xaa06,0x61,0x8e, 0xaa09,0xcf,0xc6, 0xaa0b,0x9c,0xe1, 0xaa0d,0x07,0x26,
		0xaa0f,0xa5,0x6f, 0xaa11,0x86,0xcb, 0xaa13,0x94,0xc9, 0xaa15,0xc1,0x8c,
		0xaa18,0x07,0x26, 0xaa1a,0x10,0x20, 0xaa1c,0x0d,0xb7, 0xaa1f,0x10,0x20,
		0xaa21,0xdc,0xcc, 0xaa24,0x17,0xfd, 0xaa27,0x95,0x86, 0xaa29,0x1b,0xb7,
		0xaa2c,0x0f,0xb1, 0xaa2f,0x15,0x26, 0xaa31,0xd8,0x86, 0xaa33,0x22,0x7d,
		0xaa36,0x92,0x26, 0xaa38,0x1b,0xb7, 0xaa3b,0xe3,0x5f, 0xaa3c,0x78,0x8e,
		0xaa3f,0x50,0x10, 0xaa43,0x0c,0xcb, 0xaa45,0x09,0xc9, 0xaa47,0x34,0xe1,
		0xaa49,0x06,0x26, 0xaa4b,0x12,0x30, 0xaa4d,0x6c,0x8c, 0xaa50,0x19,0x26,
		0xaa52,0x67,0x5f, 0xaa53,0x45,0xe7, 0xaa55,0x3e,0x8c, 0xaa58,0x11,0x26,
		0xaa5a,0x49,0x86, 0xaa5c,0x1a,0xb7, 0xaa5f,0xf9,0xcc, 0xaa62,0xc1,0xfd,
		0xaa65,0x1c,0xcc, 0xaa68,0x1b,0xb7, 0xaa6b,0x1f,0xb7, 0xaa6e,0x1d,0xb6,
		0xaa71,0x90,0x26, 0xaa73,0x54,0xf7, 0xaa76,0xab,0xf7, 0xaa79,0x99,0xf6,
		0xaa7c,0x10,0x26, 0xaa7e,0xa2,0x7d, 0xaa81,0x07,0x26, 0xaa83,0xf9,0xcc,
		0xaa86,0x7f,0xfd, 0xaa89,0x35,0x1c, 0xaa8b,0x27,0x1f, 0xaa8d,0xa4,0x4d,
		0xaa8e,0xfa,0x10, 0xaa92,0x5f,0x7d, 0xaa95,0x03,0x26, 0xaa97,0xa1,0x86,
		0xaa99,0x9a,0xb7, 0xaa9c,0x08,0x1a, 0xaa9e,0xd9,0xcc, 0xaaa1,0x5b,0x7f,
		0xaaa4,0x33,0xb7, 0xaaa7,0x3b,0xf7, 0xaaaa,0x51,0xb6, 0xaaad,0x54,0x84,
		0xaaaf,0x5b,0xb7, 0xaab2,0x04,0x20, 0xb000,0x26,0xb6, 0xb003,0x77,0x7d,
		0xb006,0x81,0x26, 0xb008,0x0e,0xb6,	0xb00b,0x1e,0xb1, 0xb00e,0x05,0x27,
		0xb010,0x27,0xb7, 0xb013,0xf7,0xbd,	0xb016,0x1f,0xbd, 0xb019,0xb0,0x7d,
		0xb01c,0x15,0x27, 0xb01e,0x3f,0xbd,	0xb021,0x0d,0x10, 0xb025,0x1d,0x1c,
		0xb027,0x7c,0x7e, 0xb02a,0x18,0xa6,	0xb02e,0xc0,0x43, 0xb02f,0x12,0x27,
		0xb031,0xde,0x85, 0xb033,0x6f,0x27,	0xb035,0x98,0xc6, 0xb037,0xab,0xf7,
		0xb03a,0x01,0x20, 0xb03c,0x5e,0x85,	0xb03e,0x10,0x27, 0xb040,0xba,0xc6,
		0xb042,0x8f,0xf7, 0xb045,0xf1,0x7d,	0xb048,0x12,0x26, 0xb04a,0x59,0x84,
		0xb04c,0x03,0x27, 0xb04e,0x2a,0xba,	0xb051,0x05,0xb7, 0xb054,0x31,0x7c,
		0xb057,0x21,0x86, 0xb059,0x0b,0xb1,	0xb05c,0x01,0x24, 0xb05e,0x0b,0xb6,
		0xb061,0x3b,0xb7, 0xb064,0x73,0x7f,	0xb067,0xe7,0x7f, 0xb06a,0xa7,0x7d,
		0xb06d,0x12,0x27, 0xb06f,0xb5,0x7a,	0xb072,0x07,0x7d, 0xb075,0x92,0x27,
		0xb077,0xce,0x7a, 0xb07a,0xad,0xfc,	0xb07d,0x12,0x27, 0xb07f,0x48,0x83,
		0xb082,0x7f,0xfd, 0xb085,0x4d,0x7d,	0xb088,0x17,0x27, 0xb08a,0xa2,0x7a,
		0xb08d,0x18,0x20, 0xb08f,0xbf,0x7d,	0xb092,0x01,0x26, 0xb094,0x24,0xb6,
		0xb097,0x27,0xb7, 0xb09a,0xb9,0x4c,	0xb09b,0x94,0x81, 0xb09d,0xce,0x23,
		0xb09f,0x65,0x4f, 0xb0a0,0x33,0xb7,	0xb0a3,0xa0,0x86, 0xb0a5,0x33,0xb7,
		0xb0a8,0x53,0xb7, 0xb0ab,0x00,0x86,	0xb0ad,0x5b,0xb7, 0xb0b0,0x04,0x20,
		0xb0b2,0x5f,0x7d, 0xb0b5,0x12,0x27,	0xb0b7,0x23,0xb7, 0xb0ba,0xb7,0x7d,
		0xb0bd,0xda,0x27, 0xb0bf,0x79,0x7a,	0xb0c2,0x46,0x3b, 0xb100,0x4b,0xb7,
		0xb103,0x4b,0x16, 0xb106,0x66,0x3b,	0xb107,0xc2,0x1a, 0xb109,0xb3,0x7f,
		0xb10c,0x4a,0x86, 0xb10e,0x0f,0xb7,	0xb111,0xdb,0x86, 0xb113,0xef,0xb7,
		0xb116,0x01,0x86, 0xb118,0x0f,0xb7,	0xb11b,0xa7,0x7f, 0xb11e,0x49,0x86,
		0xb120,0x5b,0xb7, 0xb123,0xc9,0x86,	0xb125,0x7b,0xb7, 0xb128,0x58,0x86,
		0xb12a,0x0e,0xb7, 0xb12d,0xb3,0x7f,	0xb130,0xb0,0x86, 0xb132,0x96,0xb7,
		0xb135,0xb0,0x86, 0xb137,0x87,0xb7,	0xb13a,0x58,0x86, 0xb13c,0x1b,0xb7,
		0xb13f,0xa7,0x7f, 0xb142,0xe9,0x86,	0xb144,0x73,0xb7, 0xb147,0xc1,0x86,
		0xb149,0x1b,0xb7, 0xb14c,0x4a,0x86,	0xb14e,0x0f,0xb7, 0xb151,0xaa,0x1c,
		0xb153,0x9e,0x39, 0xb154,0xe3,0x8e,	0xb157,0xda,0x10, 0xb15b,0x3f,0xbd,
		0xb15e,0x52,0x10, 0xb162,0x7d,0x10,	0xb166,0x6e,0xbd, 0xb169,0x45,0x10,
		0xb16d,0x15,0x34, 0xb16f,0xc9,0xc6,	0xb171,0xdf,0x7d, 0xb174,0x58,0x26,
		0xb176,0x8b,0xf7, 0xb179,0x9b,0xf7,	0xb17c,0x17,0x35, 0xb17e,0x4d,0x86,
		0xb180,0x61,0xbd, 0xb183,0xf9,0xcc,	0xb186,0x7f,0xfd, 0xb189,0x79,0x7d,
		0xb18c,0xcb,0x27, 0xb18e,0x8b,0xb6,	0xb191,0xf7,0x8a, 0xb193,0x27,0xb7,
		0xb196,0x10,0x20, 0xb198,0x6e,0xb6,	0xb19b,0xa0,0x8a, 0xb19d,0x7b,0xb7,
		0xb1a0,0xb0,0x86, 0xb1a2,0x67,0xbd,	0xb1a5,0x77,0xbd, 0xb1a8,0xed,0x10,
		0xb1ac,0xbf,0xbd, 0xb1af,0xdc,0x10,	0xb1b3,0x01,0x34, 0xb1b5,0xb8,0xc6,
		0xb1b7,0x4e,0x7d, 0xb1ba,0x38,0x26,	0xb1bc,0xd3,0xf7, 0xb1bf,0xef,0xf7,
		0xb1c2,0x17,0x35, 0xb1c4,0xa2,0x86,	0xb1c6,0x77,0xbd, 0xb1c9,0xd2,0xb6,
		0xb1cc,0xc8,0x84, 0xb1ce,0xc7,0xb7,	0xb1d1,0x4b,0x7d, 0xb1d4,0x02,0x27,
		0xb1d6,0xf9,0xcc, 0xb1d9,0xc3,0xfd,	0xb1dc,0xed,0x20, 0xb1de,0x11,0xcc,
		0xb1e1,0x7f,0xfd, 0xb1e4,0xb0,0x86,	0xb1e6,0x66,0xbd, 0xb1e9,0xf7,0xbd,
		0xb1ec,0x99,0x16, 0xb1ef,0xee,0xbd,	0xb1f2,0x00,0x24, 0xb1f4,0x15,0x34,
		0xb1f6,0xad,0xc6, 0xb1f8,0x37,0x7d,	0xb1fb,0x22,0x26, 0xb1fd,0x13,0xf7,
		0xb200,0x0b,0xf7, 0xb203,0x77,0x35,	0xb205,0xc2,0x86, 0xb207,0xff,0xbd,
		0xb20a,0x0b,0xb6, 0xb20d,0x48,0x84,	0xb20f,0x0f,0xb7, 0xb212,0xd7,0x7d,
		0xb215,0x6a,0x27, 0xb217,0x91,0xcc,	0xb21a,0xbf,0xfd, 0xb21d,0x05,0x20,
		0xb21f,0xf9,0xcc, 0xb222,0x07,0xfd,	0xb225,0xd0,0x86, 0xb227,0xee,0xbd,
		0xb22a,0x2f,0xbd, 0xb22d,0x45,0x10,	0xb231,0xf7,0xbd, 0xb234,0x94,0x24,
		0xb236,0x97,0x34, 0xb238,0xd8,0xc6,	0xb23a,0xb7,0x7d, 0xb23d,0x10,0x26,
		0xb23f,0x8b,0xf7, 0xb242,0x8f,0xf7,	0xb245,0xf1,0x35, 0xb247,0x41,0x86,
		0xb249,0x39,0xbd, 0xb24c,0xe8,0xcc,	0xb24f,0xbf,0xfd, 0xb252,0x3f,0x7d,
		0xb255,0x95,0x27, 0xb257,0x03,0xb6,	0xb25a,0x68,0x8a, 0xb25c,0x1a,0xb7,
		0xb25f,0x10,0x20, 0xb261,0x79,0xb6,	0xb264,0xc0,0x8a, 0xb266,0xef,0xb7,
		0xb269,0x58,0x86, 0xb26b,0x2e,0xbd,	0xb26e,0x2e,0xbd, 0xb271,0xde,0x6d,
		0xb275,0x90,0x26, 0xb277,0x32,0x86,	0xb279,0x1a,0xa7, 0xb27d,0x19,0xb6,
		0xb280,0xf7,0x8b, 0xb282,0x27,0xb7,	0xb285,0x55,0x39, 0xb286,0x02,0x32,
		0xb288,0x93,0xaf, 0xb28a,0x99,0x8c,	0xb28d,0x8b,0x27, 0xb28f,0xe1,0x8e,
		0xb292,0x79,0xec, 0xb294,0x7f,0xed,	0xb296,0x79,0xec, 0xb298,0x95,0xed,
		0xb29a,0x91,0xec, 0xb29c,0xd7,0xed,	0xb29e,0xd1,0xec, 0xb2a0,0x7e,0xed,
		0xb2a2,0x15,0x30, 0xb2a4,0x74,0xac,	0xb2a6,0x00,0x24, 0xb2a8,0xe5,0x30,
		0xb2aa,0x01,0xec, 0xb2ac,0x76,0xed,	0xb2ae,0x60,0xec, 0xb2b0,0x7e,0xed,
		0xb2b2,0x79,0xec, 0xb2b4,0x7e,0xed,	0xb2b6,0x7b,0xec, 0xb2b8,0x96,0xed,
		0xb2ba,0x64,0x32, 0xb2bc,0x07,0x39,	0xb2bd,0xb0,0x8e, 0xb2c0,0x61,0xbd,
		0xb2c3,0x15,0x25, 0xb2c5,0x14,0x30,	0xb2c7,0xf1,0x8c, 0xb2ca,0xec,0x23,
		0xb2cc,0xf0,0x1c, 0xb2ce,0xfe,0x39,	0xb2cf,0xaa,0x1a, 0xb2d1,0x43,0x39,
		0xb2d2,0x21,0xa6, 0xb2d4,0x24,0xa1,	0xb2d6,0x15,0x25, 0xb2d8,0x8e,0x22,
		0xb2da,0xc9,0xa6, 0xb2dc,0x8c,0xa1,	0xb2de,0xdd,0x25, 0xb2e0,0x00,0x22,
		0xb2e2,0x20,0xa6, 0xb2e4,0x26,0xa1,	0xb2e6,0x02,0x25, 0xb2e8,0xe8,0x22,
		0xb2ea,0xa8,0xa6, 0xb2ec,0x8e,0xa1,	0xb2ee,0xc8,0x24, 0xb2f0,0xc1,0x1a,
		0xb2f2,0x56,0x39, 0xb2f3,0xd7,0x1c,	0xb2f5,0x47,0x39, 0xb31b,0x13,0x34,
		0xb31d,0xa1,0x6d, 0xb321,0x38,0x26,	0xb323,0xa5,0xbe, 0xb326,0xc0,0xcc,
		0xb329,0xbe,0xed, 0xb32b,0x75,0x1a,	0xb32d,0x58,0x86, 0xb32f,0x0b,0xa7,
		0xb333,0xe8,0xb6, 0xb336,0x08,0x8b,	0xb338,0x1b,0xb7, 0xb33b,0x61,0x1c,
		0xb33d,0x18,0xa6, 0xb33f,0x5a,0x81,	0xb341,0xe2,0x26, 0xb343,0xd2,0x10,
		0xb347,0x78,0x20, 0xb349,0x54,0x10,	0xb34d,0x6a,0x8e, 0xb350,0x22,0xce,
		0xb353,0xdd,0x6d, 0xb355,0x94,0x26,	0xb357,0x79,0xcc, 0xb35a,0xbd,0xed,
		0xb35c,0x8b,0xe6, 0xb35e,0x8d,0xe7,	0xb360,0x58,0x26, 0xb362,0x26,0x5f,
		0xb363,0x61,0xa6, 0xb365,0xc2,0x8b,	0xb367,0xa2,0xed, 0xb369,0x58,0x86,
		0xb36b,0xa2,0x6d, 0xb36f,0x01,0x26,	0xb371,0x59,0xfe, 0xb374,0xfa,0xa7,
		0xb376,0xcd,0x86, 0xb378,0x1a,0xa7,	0xb37c,0x19,0xb6, 0xb37f,0x68,0x8b,
		0xb381,0x33,0xb7, 0xb384,0xa2,0x86,	0xb386,0x25,0xa7, 0xb388,0xb3,0xa7,
		0xb38a,0xa5,0xa7, 0xb38c,0xa0,0xcc,	0xb38f,0xf7,0xfd, 0xb392,0xa1,0x86,
		0xb394,0x67,0xbd, 0xb397,0xa1,0x86,	0xb399,0xa8,0xbd, 0xb39c,0x5c,0x86,
		0xb39e,0x77,0xbd, 0xb3a1,0xb0,0x86,	0xb3a3,0x66,0xbd, 0xb3a6,0xe1,0x8e,
		0xb3a9,0xb0,0xce, 0xb3ac,0x3f,0xbd,	0xb3af,0xe4,0xfc, 0xb3b2,0x94,0x10,
		0xb3b6,0x32,0xa6, 0xb3ba,0x93,0x43,	0xb3bb,0x87,0x85, 0xb3bd,0x38,0x26,
		0xb3bf,0x62,0x85, 0xb3c1,0x02,0x26,	0xb3c3,0xb5,0x85, 0xb3c5,0x02,0x26,
		0xb3c7,0x10,0x20, 0xb3c9,0x94,0x30,	0xb3cb,0xf9,0x8c, 0xb3ce,0x84,0x23,
		0xb3d0,0xe2,0x8e, 0xb3d3,0x77,0xbd,	0xb3d6,0xa1,0x86, 0xb3d8,0xb3,0xb7,
		0xb3db,0x97,0x7d, 0xb3de,0xc9,0x26,	0xb3e0,0x04,0x20, 0xb3e2,0x15,0x30,
		0xb3e4,0xf4,0x8c, 0xb3e7,0x00,0x24,	0xb3e9,0xf8,0x8e, 0xb3ec,0xf7,0xbd,
		0xb3ef,0x81,0x86, 0xb3f1,0x33,0xb7,	0xb3f4,0x5f,0x7d, 0xb3f7,0x12,0x26,
		0xb3f9,0x24,0x20, 0xb3fb,0x7a,0x86,	0xb3fd,0x57,0xbd, 0xb400,0x41,0xbd,
		0xb403,0xa9,0xa6, 0xb405,0xaf,0xa7,	0xb407,0x81,0x8e, 0xb40a,0x06,0x33,
		0xb40d,0x51,0x11, 0xb411,0x6f,0x22,	0xb413,0xf7,0xbd, 0xb416,0x01,0xa6,
		0xb41a,0xc4,0x43, 0xb41b,0x5d,0x85,	0xb41d,0x03,0x26, 0xb41f,0x49,0x86,
		0xb421,0xbb,0xb7, 0xb424,0xf7,0x7d,	0xb427,0x39,0x26, 0xb429,0x51,0x16,
		0xb42c,0x58,0x86, 0xb42e,0x1f,0xb7,	0xb431,0x7f,0x35, 0xb433,0x68,0x34,
		0xb435,0x37,0x7d, 0xb438,0x10,0x26,	0xb43a,0xae,0xff, 0xb43d,0x18,0xa6,
		0xb43f,0xe3,0x5f, 0xb440,0xfd,0xfd,	0xb443,0xbd,0x6d, 0xb447,0xe1,0x26,
		0xb449,0x3a,0xbe, 0xb44c,0x5a,0x86,	0xb44e,0x0d,0xa7, 0xb450,0x1a,0x1a,
		0xb452,0x81,0x86, 0xb454,0xea,0xa7,	0xb458,0x1e,0xb6, 0xb45b,0x6c,0x8b,
		0xb45d,0x0d,0xb7, 0xb460,0x1d,0x1c,	0xb462,0xb9,0xae, 0xb464,0x17,0x7d,
		0xb467,0xe1,0x26, 0xb469,0xbb,0xff,	0xb46c,0x18,0xa6, 0xb46e,0xe6,0x5f,
		0xb46f,0xae,0xfd, 0xb472,0x7a,0x6d,	0xb476,0x6d,0x26, 0xb478,0x39,0xbe,
		0xb47b,0x5a,0x86, 0xb47d,0x1a,0xa7,	0xb47f,0x66,0x1a, 0xb481,0xa6,0x86,
		0xb483,0x25,0xa7, 0xb487,0x23,0xb6,	0xb48a,0x8c,0x8b, 0xb48c,0x17,0xb7,
		0xb48f,0x73,0x1c, 0xb491,0x63,0xae,	0xb493,0x5f,0x7d, 0xb496,0x01,0x26,
		0xb498,0x03,0xff, 0xb49b,0x89,0xa6,	0xb49d,0x72,0x5f, 0xb49e,0xd7,0xfd,
		0xb4a1,0x5e,0x6d, 0xb4a5,0x10,0x26,	0xb4a7,0x75,0xbe, 0xb4aa,0xa8,0x86,
		0xb4ac,0x12,0xa7, 0xb4ae,0x6d,0x1a,	0xb4b0,0xb0,0x86, 0xb4b2,0x22,0xa7,
		0xb4b6,0x36,0xb6, 0xb4b9,0xa2,0x8b,	0xb4bb,0x8b,0xb7, 0xb4be,0x1f,0x1c,
		0xb4c0,0x15,0x35, 0xb4c2,0x13,0x34,	0xb4c4,0x68,0xfc, 0xb4c7,0xe1,0x8e,
		0xb4ca,0x49,0xa6, 0xb4cc,0xce,0x3a,	0xb4cd,0xe7,0xab, 0xb4cf,0xc5,0xa7,
		0xb4d1,0x25,0xb7, 0xb4d4,0x5f,0x7a,	0xb4d7,0x01,0x26, 0xb4d9,0x7c,0x86,
		0xb4db,0x67,0xb7, 0xb4de,0x6a,0x7a,	0xb4e1,0x10,0x26, 0xb4e3,0xa1,0x86,
		0xb4e5,0x33,0xb7, 0xb4e8,0x1f,0x35,	0xb4ea,0x2e,0x39, 0xb4eb,0xe9,0x8e,
		0xb4ee,0xa0,0x86, 0xb4f0,0x32,0xa7,	0xb4f2,0xe3,0x8b, 0xb4f4,0xf4,0x8c,
		0xb4f7,0x00,0x23, 0xb4f9,0x78,0x86,	0xb4fb,0x74,0xb7, 0xb4fe,0x85,0x86,
		0xb500,0x57,0xb7, 0xb503,0xc1,0x86,	0xb505,0x7d,0xbd, 0xb508,0xe2,0x4a,
		0xb509,0x06,0x26, 0xb50b,0x36,0x39,	0xb50c,0x14,0x34, 0xb50e,0x2b,0xbe,
		0xb511,0x93,0xce, 0xb514,0x7c,0xee,	0xb516,0xa6,0x33, 0xb518,0x1a,0xa6,
		0xb51a,0x2c,0xab, 0xb51c,0x63,0x19,	0xb51d,0x0b,0xa7, 0xb51f,0x09,0xa6,
		0xb521,0xee,0xa9, 0xb523,0x4e,0x19,	0xb524,0xf2,0xa7, 0xb526,0xf1,0xa6,
		0xb528,0x2e,0xa9, 0xb52a,0x66,0x19,	0xb52b,0x1b,0xa7, 0xb52d,0x18,0xa6,
		0xb52f,0x2a,0xa9, 0xb531,0x47,0x19,	0xb532,0xca,0xa7, 0xb534,0x79,0xfe,
		0xb537,0x81,0xa6, 0xb539,0x0e,0xa1,	0xb53b,0x14,0x22, 0xb53d,0x16,0x25,
		0xb53f,0x09,0xa6, 0xb541,0x29,0xa1,	0xb543,0xbd,0x25, 0xb545,0x8d,0xbd,
		0xb548,0xb0,0x6c, 0xb54c,0x0e,0xa6,	0xb54e,0x2c,0xab, 0xb550,0xdd,0x19,
		0xb551,0x11,0xa7, 0xb553,0x01,0xa6,	0xb555,0x76,0xa9, 0xb557,0xd6,0x19,
		0xb558,0x0b,0xa7, 0xb55a,0x49,0x86,	0xb55c,0x38,0xbd, 0xb55f,0x3f,0xbd,
		0xb562,0x4f,0x35, 0xb564,0xbd,0x34,	0xb566,0x35,0xbe, 0xb569,0xf8,0xce,
		0xb56c,0xb6,0x6d, 0xb56e,0x11,0x26,	0xb570,0x30,0xe6, 0xb572,0xcf,0x1f,
		0xb574,0x3c,0x44, 0xb575,0x5c,0x84,	0xb577,0x0f,0xc4, 0xb579,0xe5,0x58,
		0xb57a,0xf4,0x58, 0xb57b,0xf7,0x58,	0xb57c,0xbe,0xed, 0xb57e,0x8d,0xe6,
		0xb580,0xd7,0x1f, 0xb582,0x99,0x44,	0xb583,0xb1,0x84, 0xb585,0xac,0xc4,
		0xb587,0xda,0x58, 0xb588,0x18,0x58,	0xb589,0x30,0x58, 0xb58a,0x55,0xed,
		0xb58c,0x4e,0xe6, 0xb58e,0xa7,0x1f,	0xb590,0x98,0x44, 0xb591,0xb1,0x84,
		0xb593,0xb9,0xc4, 0xb595,0xdc,0x58,	0xb596,0xda,0x58, 0xb597,0xda,0x58,
		0xb598,0x19,0xed, 0xb59a,0x29,0xe6,	0xb59c,0x2d,0x1f, 0xb59e,0x99,0x44,
		0xb59f,0x71,0x84, 0xb5a1,0xbc,0xc4,	0xb5a3,0xcc,0x58, 0xb5a4,0xcd,0x58,
		0xb5a5,0xcd,0x58, 0xb5a6,0x7a,0xed,	0xb5a8,0x5c,0xcc, 0xb5ab,0xf1,0x8e,
		0xb5ae,0xca,0xa1, 0xb5b0,0x10,0x26,	0xb5b2,0x2a,0xe7, 0xb5b4,0xf4,0x8c,
		0xb5b7,0x02,0x25, 0xb5b9,0x98,0xa6,	0xb5bb,0x08,0xf6, 0xb5be,0xc9,0xc1,
		0xb5c0,0x13,0x27, 0xb5c2,0xb4,0x81,	0xb5c4,0x06,0x26, 0xb5c6,0xe9,0xce,
		0xb5c9,0x40,0x20, 0xb5cb,0xa9,0xce,	0xb5ce,0x50,0x20, 0xb5d0,0xea,0xce,
		0xb5d3,0x5d,0x6d, 0xb5d7,0x01,0x26,	0xb5d9,0x40,0xbe, 0xb5dc,0x0a,0xef,
		0xb5de,0xad,0xa7, 0xb5e0,0xa6,0x81,	0xb5e2,0x00,0x26, 0xb5e4,0xb0,0x86,
		0xb5e6,0x33,0xa7, 0xb5ea,0xac,0xb6,	0xb5ed,0x42,0x8b, 0xb5ef,0x7e,0xb7,
		0xb5f2,0x10,0x20, 0xb5f4,0xb0,0x86,	0xb5f6,0x27,0xa7, 0xb5fa,0xc5,0xb6,
		0xb5fd,0xaa,0x8b, 0xb5ff,0x80,0xb7,	0xb602,0xdf,0x35, 0xb604,0xf9,0x39,
		0xb605,0x74,0x34, 0xb607,0xeb,0xbe,	0xb60a,0xe9,0xce, 0xb60d,0xa5,0x6d,
		0xb60f,0x01,0x26, 0xb611,0xab,0xe6,	0xb613,0x47,0x1f, 0xb615,0x09,0x44,
		0xb616,0xf1,0x44, 0xb617,0xd9,0x84,	0xb619,0xc9,0xc4, 0xb61b,0xf2,0x58,
		0xb61c,0xe2,0x58, 0xb61d,0xa8,0xed,	0xb61f,0x89,0xe6, 0xb621,0x5b,0x1f,
		0xb623,0x10,0x44, 0xb624,0x94,0x44,	0xb625,0xd4,0x84, 0xb627,0x90,0xc4,
		0xb629,0xe5,0x58, 0xb62a,0xf5,0x58,	0xb62b,0xba,0xed, 0xb62d,0xfc,0xcc,
		0xb630,0x3e,0xed, 0xb632,0x3a,0xed,	0xb634,0x7c,0xcc, 0xb637,0xc5,0x8e,
		0xb63a,0x0a,0xa1, 0xb63c,0x10,0x26,	0xb63e,0x98,0xe7, 0xb640,0xf0,0x8c,
		0xb643,0xb5,0x25, 0xb645,0x02,0x86,	0xb647,0x49,0xce, 0xb64a,0xb5,0x6d,
		0xb64e,0x01,0x26, 0xb650,0xc2,0xbe,	0xb653,0x4d,0xef, 0xb655,0x15,0xa7,
		0xb657,0x21,0x86, 0xb659,0x1d,0xa7,	0xb65d,0x0c,0xb6, 0xb660,0xe2,0x8b,
		0xb662,0xce,0xb7, 0xb665,0xb7,0x35,	0xb667,0xa4,0x34, 0xb669,0xb7,0x7d,
		0xb66c,0x10,0x26, 0xb66e,0xa7,0x7d,	0xb671,0x92,0x27, 0xb673,0xdc,0x7f,
		0xb676,0x22,0x7d, 0xb679,0x12,0x27,	0xb67b,0x7a,0x8e, 0xb67e,0x89,0xf6,
		0xb681,0x2e,0xe6, 0xb683,0x3c,0xe1,	0xb685,0x04,0x24, 0xb687,0x27,0xb7,
		0xb68a,0x89,0x8e, 0xb68d,0x4e,0xe6,	0xb68f,0x4f,0xf7, 0xb692,0xb4,0x81,
		0xb694,0x14,0x26, 0xb696,0x29,0xe6,	0xb698,0x4c,0xc4, 0xb69a,0x68,0xca,
		0xb69c,0x0d,0x1f, 0xb69e,0x0a,0x1a,	0xb6a0,0x33,0xb7, 0xb6a3,0xf8,0xcc,
		0xb6a6,0x37,0xb7, 0xb6a9,0x5b,0xf7,	0xb6ac,0xbd,0x1c, 0xb6ae,0x96,0x35,
		0xc800,0xa7,0x10, 0xc804,0x2e,0xbd,	0xc807,0x77,0x7d, 0xc80a,0x05,0x27,
		0xc80c,0x0d,0xb6, 0xc80f,0x68,0x8a,	0xc811,0xe1,0xb7, 0xc814,0x9b,0x20,
		0xc816,0x0b,0xb6, 0xc819,0x5a,0x84,	0xc81b,0x0f,0xb7, 0xc81e,0x69,0x8e,
		0xc821,0x52,0x4f, 0xc822,0x04,0xa7,	0xc824,0xa5,0x30, 0xc826,0x1f,0x8c,
		0xc829,0x16,0x25, 0xc82b,0xff,0xcc,	0xc82e,0x7e,0x8e, 0xc831,0xbe,0xed,
		0xc833,0xcb,0x8c, 0xc836,0x9d,0x25,	0xc838,0x78,0x8e, 0xc83b,0x2c,0xbf,
		0xc83e,0x39,0xbd, 0xc841,0x00,0xbd,	0xc844,0xa0,0xbd, 0xc847,0x1f,0xbd,
		0xc84a,0x3f,0xbd, 0xc84d,0x2f,0xbd,	0xc850,0xcf,0xbd, 0xc853,0x83,0xb6,
		0xc856,0x27,0xb7, 0xc859,0x0a,0xb7,	0xc85c,0x2b,0xbd, 0xc85f,0xa7,0x7f,
		0xc862,0x4d,0x7d, 0xc865,0xa5,0x10,	0xc869,0x3f,0xbd, 0xc86c,0x3f,0xbd,
		0xc86f,0x3d,0xbd, 0xc872,0xbb,0xcc,	0xc875,0xd7,0xbd, 0xc878,0x3f,0xbd,
		0xc87b,0x2b,0xbd, 0xc87e,0x2d,0xbd,	0xc881,0x66,0xbd, 0xc884,0x85,0x16,
		0xc887,0x5d,0x6d, 0xc88b,0x49,0x26,	0xc88d,0x10,0x86, 0xc88f,0x0d,0xa7,
		0xc893,0x23,0xb6, 0xc896,0xe4,0x8b,	0xc898,0x48,0xb7, 0xc89b,0xff,0xbd,
		0xc89e,0x17,0xbd, 0xc8a1,0x85,0x10,	0xc8a5,0x7c,0xec, 0xc8a7,0x97,0x10,
		0xc8ab,0x5f,0x24, 0xc8ad,0x04,0x80,	0xc8af,0x93,0xc0, 0xc8b1,0xd3,0x1f,
		0xc8b3,0x49,0x6d, 0xc8b7,0x02,0x26,	0xc8b9,0xf9,0xbe, 0xc8bc,0x92,0xef,
		0xc8be,0x3e,0x86, 0xc8c0,0xc9,0x5f,	0xc8c1,0x78,0xed, 0xc8c3,0xa1,0x86,
		0xc8c5,0x35,0xa7, 0xc8c9,0x23,0xb6,	0xc8cc,0x36,0x8b, 0xc8ce,0x0f,0xb7,
		0xc8d1,0x58,0x6d, 0xc8d5,0x10,0x26,	0xc8d7,0x63,0xbe, 0xc8da,0x4d,0xef,
		0xc8dc,0x58,0x86, 0xc8de,0xaf,0x5f,	0xc8df,0x1d,0xed, 0xc8e1,0xb0,0x86,
		0xc8e3,0x26,0xa7, 0xc8e7,0x36,0xb6,	0xc8ea,0x50,0x8b, 0xc8ec,0x1b,0xb7,
		0xc8ef,0x23,0x20, 0xc8f1,0x47,0x39,	0xc8f2,0x5c,0x7f, 0xc8f5,0x5b,0x7f,
		0xc8f8,0xd0,0x8e, 0xc8fb,0x4a,0xbf,	0xc8fe,0x4a,0x86, 0xc900,0xe8,0xb7,
		0xc903,0xff,0xbd, 0xc906,0xbf,0xbd,	0xc909,0xa2,0x7f, 0xc90c,0xa3,0x7f,
		0xc90f,0x3f,0xbd, 0xc912,0x97,0xbd,	0xc915,0x63,0xbd, 0xc918,0x29,0xbd,
		0xc91b,0x3f,0xbd, 0xc91e,0xf9,0xcc,	0xc921,0xbf,0xfd, 0xc924,0x90,0x86,
		0xc926,0xa6,0xb7, 0xc929,0x1b,0xb7,	0xc92c,0x3f,0xbd, 0xc92f,0xed,0xcc,
		0xc932,0xc7,0xfd, 0xc935,0x37,0xfd,	0xc938,0xbf,0xfd, 0xc93b,0xbe,0xfd,
		0xc93e,0x4e,0x86, 0xc940,0x56,0xf6,	0xc943,0x5f,0xfd, 0xc946,0x77,0xfd,
		0xc949,0xf8,0xcc, 0xc94c,0xa9,0xfd,	0xc94f,0xf9,0xcc, 0xc952,0xb7,0xfd,
		0xc955,0x23,0x7f, 0xc958,0x0a,0xb6,	0xc95b,0x5c,0x81, 0xc95d,0x12,0x26,
		0xc95f,0x49,0x86, 0xc961,0x13,0xb7,	0xc964,0x18,0x86, 0xc966,0x27,0xbd,
		0xc969,0xfc,0xcc, 0xc96c,0xbf,0xfd,	0xc96f,0x29,0xbd, 0xc972,0x81,0xcc,
		0xc975,0xb7,0xfd, 0xc978,0x3f,0xbd,	0xc97b,0xe9,0xcc, 0xc97e,0xaf,0xfd,
		0xc981,0xb3,0x86, 0xc983,0x27,0xb7,	0xc986,0xf9,0xcc, 0xc989,0x68,0xfd,
		0xc98c,0x69,0xbd, 0xc98f,0x43,0xb6,	0xc992,0x27,0xb7, 0xc995,0x21,0xb7,
		0xc998,0x09,0xbd, 0xc99b,0x69,0x86,	0xc99d,0xa3,0xbd, 0xc9a0,0x77,0xbd,
		0xc9a3,0x63,0xbd, 0xc9a6,0x67,0xbd,	0xc9a9,0xaf,0x39, 0xc9aa,0x8a,0xb6,
		0xc9ad,0x4e,0x81, 0xc9af,0x16,0x23,	0xc9b1,0xb0,0x86, 0xc9b3,0x37,0xb7,
		0xc9b6,0xdf,0x48, 0xc9b7,0x74,0xbb,	0xc9ba,0xf1,0x40, 0xc9bb,0x4b,0x8b,
		0xc9bd,0xb8,0x8e, 0xc9c0,0x4e,0x6e,	0xc9c2,0x77,0xbd, 0xc9c5,0x62,0xbd,
		0xc9c8,0x55,0xbd, 0xc9cb,0x57,0xbd,	0xc9ce,0x37,0xbd, 0xc9d1,0x45,0x39,
		0xc9d2,0xf9,0xcc, 0xc9d5,0x69,0xfd,	0xc9d8,0x29,0xfd, 0xc9db,0x51,0xcc,
		0xc9de,0x5f,0xfd, 0xc9e1,0xfc,0xcc,	0xc9e4,0x7f,0xfd, 0xc9e7,0x20,0xb7,
		0xc9ea,0x77,0xb7, 0xc9ed,0xf3,0x7f,	0xc9f0,0x47,0x39, 0xc9f1,0xb0,0x86,
		0xc9f3,0x26,0xb7, 0xc9f6,0xe9,0xcc,	0xc9f9,0x5f,0xfd, 0xc9fc,0x1f,0x7d,
		0xc9ff,0xca,0x27, 0xca01,0x16,0xb6,	0xca04,0x6e,0x8a, 0xca06,0x47,0xb7,
		0xca09,0x00,0x20, 0xca0b,0x0b,0xb6,	0xca0e,0x59,0x84, 0xca10,0xe7,0xb7,
		0xca13,0x99,0xcc, 0xca16,0x3f,0xfd,	0xca19,0xe8,0xcc, 0xca1c,0xb8,0xfd,
		0xca1f,0xf9,0xcc, 0xca22,0xfe,0xfd,	0xca25,0x1c,0xcc, 0xca28,0xbf,0xfd,
		0xca2b,0xf8,0xcc, 0xca2e,0xae,0xfd,	0xca31,0x9c,0xcc, 0xca34,0x3f,0xfd,
		0xca37,0xe8,0xcc, 0xca3a,0xbf,0xfd,	0xca3d,0xfc,0xcc, 0xca40,0xae,0xfd,
		0xca43,0xfe,0x39, 0xca44,0xc1,0x86,	0xca46,0xe7,0xb7, 0xca49,0xa4,0x7d,
		0xca4c,0x00,0x27, 0xca4e,0xf9,0xcc,	0xca51,0x46,0xfd, 0xca54,0x54,0xb6,
		0xca57,0x39,0x84, 0xca59,0x0c,0xb7,	0xca5c,0x02,0x20, 0xca5e,0xf9,0xcc,
		0xca61,0x3f,0xfd, 0xca64,0xf1,0xb6,	0xca67,0xfa,0x84, 0xca69,0x1b,0xb7,
		0xca6c,0xfc,0xcc, 0xca6f,0xbc,0xfd,	0xca72,0x66,0xcc, 0xca75,0x57,0xfd,
		0xca78,0xfc,0xcc, 0xca7b,0xb9,0xfd,	0xca7e,0xfa,0xcc, 0xca81,0x7d,0xfd,
		0xca84,0xe8,0xcc, 0xca87,0x7f,0xfd,	0xca8a,0xf1,0xcc, 0xca8d,0x76,0xfd,
		0xca90,0xfc,0xcc, 0xca93,0x7f,0xfd,	0xca96,0xf9,0xcc, 0xca99,0x79,0xfd,
		0xca9c,0xb4,0x39, 0xca9d,0x59,0xbd,	0xcaa0,0x5f,0x7d, 0xcaa3,0x11,0x26,
		0xcaa5,0xb0,0x86, 0xcaa7,0x31,0xb7,	0xcaaa,0x7e,0xbd, 0xcaad,0x37,0x7d,
		0xcab0,0x10,0x26, 0xcab2,0xb0,0x86,	0xcab4,0x33,0xb7, 0xcab7,0x66,0xbd,
		0xcaba,0xff,0xbd, 0xcabd,0x25,0x10,	0xcac1,0x64,0xbd, 0xcac4,0x5a,0x7d,
		0xcac7,0x01,0x26, 0xcac9,0xdf,0x86,	0xcacb,0xc7,0xb7, 0xcace,0xff,0xbd,
		0xcad1,0x5e,0x6d, 0xcad5,0x16,0x26,	0xcad7,0x77,0xbd, 0xcada,0xd7,0x7d,
		0xcadd,0x76,0x26, 0xcadf,0x29,0x86,	0xcae1,0x33,0xb7, 0xcae4,0x77,0xbd,
		0xcae7,0x74,0xbd, 0xcaea,0x83,0x10,	0xcaee,0xe8,0xbd, 0xcaf1,0x5f,0x7d,
		0xcaf4,0x10,0x26, 0xcaf6,0xb3,0x86,	0xcaf8,0x33,0xb7, 0xcafb,0x11,0xbd,
		0xcafe,0x14,0x7d, 0xcb01,0xa7,0x26,	0xcb03,0x01,0x86, 0xcb05,0xb4,0xb7,
		0xcb08,0x2c,0xbd, 0xcb0b,0x3f,0xbd,	0xcb0e,0x52,0x10, 0xcb12,0x3f,0xbd,
		0xcb15,0x68,0x6d, 0xcb19,0x02,0x26,	0xcb1b,0x3f,0xbd, 0xcb1e,0x43,0x16,
		0xcb21,0xfb,0x7f, 0xcb24,0xff,0x7d,	0xcb27,0x35,0x27, 0xcb29,0x19,0xb6,
		0xcb2c,0x68,0x8a, 0xcb2e,0x1d,0xb7,	0xcb31,0xa4,0x20, 0xcb33,0x25,0xb6,
		0xcb36,0x82,0x84, 0xcb38,0x1b,0xb7,	0xcb3b,0xa6,0x7d, 0xcb3e,0x03,0x26,
		0xcb40,0x4b,0xcc, 0xcb43,0x3f,0xbd,	0xcb46,0x57,0xbd, 0xcb49,0x2e,0xbd,
		0xcb4c,0xb2,0x7f, 0xcb4f,0xb5,0x6d,	0xcb53,0x89,0x26, 0xcb55,0xc6,0xbe,
		0xcb58,0xec,0xcc, 0xcb5b,0xbd,0xed,	0xcb5d,0x62,0x1a, 0xcb5f,0x49,0x86,
		0xcb61,0x32,0xa7, 0xcb65,0x99,0xb6,	0xcb68,0x6a,0x8b, 0xcb6a,0x09,0xb7,
		0xcb6d,0x3f,0xbd, 0xcb70,0x7d,0x1c,	0xcb72,0x1a,0x6d, 0xcb76,0xe3,0x26,
		0xcb78,0x61,0x1a, 0xcb7a,0x59,0x86,	0xcb7c,0x1a,0xa7, 0xcb80,0x27,0xb6,
		0xcb83,0xe4,0x8b, 0xcb85,0x34,0xb7,	0xcb88,0x25,0x1c, 0xcb8d,0xe7,0xcc,
		0xcb90,0x65,0xbd, 0xcb93,0x5d,0x6d,	0xcb97,0x01,0x26, 0xcb99,0x3e,0x1a,
		0xcb9b,0x49,0x86, 0xcb9d,0x4a,0xa7,	0xcba1,0x31,0xb6, 0xcba4,0xe2,0x8b,
		0xcba6,0x35,0xb7, 0xcba9,0x3d,0x1c,	0xcbab,0x72,0xbd, 0xcbae,0x74,0xbd,
		0xcbb1,0xb0,0x86, 0xcbb3,0x66,0xbd,	0xcbb6,0x48,0x6d, 0xcbba,0x10,0x26,
		0xcbbc,0x79,0xbe, 0xcbbf,0xe4,0xcc,	0xcbc2,0x7d,0xed, 0xcbc4,0xd7,0x1a,
		0xcbc6,0xa1,0x86, 0xcbc8,0x40,0xa7,	0xcbcc,0xb3,0xb6, 0xcbcf,0x04,0x8b,
		0xcbd1,0x35,0xb7, 0xcbd4,0x73,0xbd,	0xcbd7,0xd3,0x1c, 0xcbd9,0x17,0xbd,
		0xcbdc,0x6a,0xcc, 0xcbdf,0x9f,0xbd,	0xcbe2,0x60,0xbd, 0xcbe5,0x77,0xbd,
		0xcbe8,0xfc,0xcc, 0xcbeb,0x71,0xbd,	0xcbee,0x82,0xbd, 0xcbf1,0x77,0xbd,
		0xcbf4,0xfc,0xcc, 0xcbf7,0x77,0xbd,	0xcbfa,0xde,0xbd, 0xcbfd,0x9f,0xbd,
		0xcc00,0x8e,0xcc, 0xcc03,0x5f,0xbd,	0xcc06,0x77,0xbd, 0xcc09,0x3a,0xbd,
		0xcc0c,0xff,0xcc, 0xcc0f,0x3f,0xbd,	0xcc12,0xf7,0xbd, 0xcc15,0xd4,0xbd,
		0xcc18,0xe9,0xcc, 0xcc1b,0x3f,0xbd,	0xcc1e,0x3f,0xbd, 0xcc21,0x5f,0xbd,
		0xcc24,0x14,0xcc, 0xcc27,0x60,0xbd,	0xcc2a,0x3c,0xbd, 0xcc2d,0x51,0x16,
		0xcc30,0xf9,0xb6, 0xcc33,0x9b,0x43,	0xcc34,0x7e,0x85, 0xcc36,0xdd,0x27,
		0xcc38,0x15,0x34, 0xcc3a,0x09,0xb6,	0xcc3d,0x10,0x26, 0xcc3f,0x10,0x34,
		0xcc41,0xd0,0xb6, 0xcc44,0x63,0x8b,	0xcc46,0xf6,0x19, 0xcc47,0xef,0xb7,
		0xcc4a,0x49,0x86, 0xcc4c,0xce,0xc6,	0xcc4e,0xb7,0x7d, 0xcc51,0xc7,0x26,
		0xcc53,0x07,0xf7, 0xcc56,0x2f,0xf7,	0xcc59,0xcf,0xc6, 0xcc5b,0xb7,0x7d,
		0xcc5e,0x01,0x26, 0xcc60,0xbb,0xf7,	0xcc63,0x85,0xf7, 0xcc66,0xa1,0x4a,
		0xcc67,0x6b,0x26, 0xcc69,0x17,0x35,	0xcc6b,0xcb,0xc6, 0xcc6d,0x8f,0xf1,
		0xcc70,0xea,0x23, 0xcc72,0x92,0x4f,	0xcc73,0xc4,0x20, 0xcc75,0x70,0x86,
		0xcc77,0x33,0xc5, 0xcc79,0x12,0x27,	0xcc7b,0x5d,0x86, 0xcc7d,0x2b,0xbb,
		0xcc80,0xd3,0x19, 0xcc81,0x30,0xb7,	0xcc84,0x22,0xb1, 0xcc87,0x00,0x22,
		0xcc89,0xf1,0xb6, 0xcc8c,0x9f,0x84,	0xcc8e,0x87,0xb7, 0xcc91,0xaa,0xc6,
		0xcc93,0x5f,0x7d, 0xcc96,0x01,0x26,	0xcc98,0x1a,0xf7, 0xcc9b,0x07,0xf7,
		0xcc9e,0xa9,0x86, 0xcca0,0x17,0x35,	0xcca2,0x86,0x10, 0xcca6,0xb6,0x86,
		0xcca8,0xdb,0xb7, 0xccab,0xe1,0xb7,	0xccae,0xc7,0x20, 0xccb0,0xb6,0x85,
		0xccb2,0x90,0x10, 0xccb6,0x12,0x34,	0xccb8,0x99,0xb6, 0xccbb,0xc1,0x26,
		0xccbd,0xfd,0x34, 0xccbf,0xb9,0xb6,	0xccc2,0xe4,0x8b, 0xccc4,0xd1,0x19,
		0xccc5,0x31,0xb7, 0xccc8,0x97,0x86,	0xccca,0x01,0xc6, 0xcccc,0x55,0x7d,
		0xcccf,0xc9,0x26, 0xccd1,0x3e,0xf7,	0xccd4,0x3d,0xf7, 0xccd7,0xa9,0xc6,
		0xccd9,0x14,0x7d, 0xccdc,0xc4,0x26,	0xccde,0x07,0xf7, 0xcce1,0x3b,0xf7,
		0xcce4,0xc8,0x4a, 0xcce5,0x10,0x26,	0xcce7,0x04,0x35, 0xcce9,0x70,0xc6,
		0xcceb,0x7c,0xf1, 0xccee,0xe3,0x23,	0xccf0,0xda,0x4f, 0xccf1,0x04,0x20,
		0xccf3,0xb5,0x86, 0xccf5,0xbe,0xc5,	0xccf7,0x12,0x27, 0xccf9,0xf8,0x86,
		0xccfb,0xc3,0xbb, 0xccfe,0xec,0x19,	0xccff,0xf6,0xb7, 0xcd02,0xde,0xb1,
		0xcd05,0x53,0x22, 0xcd07,0xc3,0xb6,	0xcd0a,0x59,0x84, 0xcd0c,0x1a,0xb7,
		0xcd0f,0xc9,0xc6, 0xcd11,0x36,0x7d,	0xcd14,0x45,0x26, 0xcd16,0x87,0xf7,
		0xcd19,0x9c,0xf7, 0xcd1c,0x4d,0x86,	0xcd1e,0x17,0x35, 0xcd20,0x5a,0x27,
		0xcd22,0xe9,0x86, 0xcd24,0xd3,0xb7,	0xcd27,0xd2,0xb7, 0xcd2a,0x49,0x86,
		0xcd2c,0x1b,0xb7, 0xcd2f,0x23,0x39,	0xcd30,0x37,0x7d, 0xcd33,0x7d,0x27,
		0xcd35,0x89,0x7a, 0xcd38,0x45,0x10,	0xcd3c,0x3f,0xbd, 0xcd3f,0x21,0x39,
		0xcd40,0xb4,0x7d, 0xcd43,0xcd,0x27,	0xcd45,0x3b,0x7a, 0xcd48,0x57,0x16,
		0xcd4b,0xa7,0x7f, 0xcd4e,0xa7,0x7f,	0xcd51,0xf7,0xb6, 0xcd54,0x51,0x10,
		0xcd58,0xb6,0x7f, 0xcd5b,0xb7,0x7d,	0xcd5e,0x05,0x27, 0xcd60,0xa1,0x7a,
		0xcd63,0xf8,0x16, 0xcd66,0xdc,0x1f,	0xcd68,0x45,0x10, 0xcd6c,0x98,0xe6,
		0xcd6e,0x47,0x10, 0xcd72,0xb3,0xf7,	0xcd75,0x6d,0x10, 0xcd79,0x45,0x10,
		0xcd7c,0x45,0x10, 0xcd80,0xcc,0x48,	0xcd81,0xe0,0x8e, 0xcd84,0x69,0xec,
		0xcd86,0x92,0x10, 0xcd8a,0x8e,0xbb,	0xcd8d,0x4a,0xfb, 0xcd90,0xa1,0x81,
		0xcd92,0x15,0x25, 0xcd94,0xa2,0x81,	0xcd96,0x04,0x23, 0xcd98,0x9e,0xb6,
		0xcd9b,0x6f,0x7f, 0xcd9e,0x74,0xc1,	0xcda0,0x16,0x25, 0xcda2,0xba,0xc1,
		0xcda4,0x02,0x23, 0xcda6,0x29,0xf6,	0xcda9,0x53,0x7f, 0xcdac,0x93,0x1f,
		0xcdae,0x77,0xc3, 0xcdb1,0x77,0xbd,	0xcdb4,0x71,0xbe, 0xcdb7,0x42,0x3a,
		0xcdb8,0x18,0xe6, 0xcdba,0x12,0xc5,	0xcdbc,0xba,0x27, 0xcdbe,0xfb,0xb6,
		0xcdc1,0xb4,0x85, 0xcdc3,0x05,0x27,	0xcdc5,0xd6,0x1f, 0xcdc7,0x2b,0xf6,
		0xcdca,0x0c,0xc3, 0xcdcd,0xfc,0xbd,	0xcdd0,0x77,0xbe, 0xcdd3,0x42,0x3a,
		0xcdd4,0x2f,0xe6, 0xcdd6,0xbd,0xc5,	0xcdd8,0x8f,0x26, 0xcdda,0xa7,0x1f,
		0xcddc,0x5f,0xf6, 0xcddf,0x4f,0x7f,	0xcde2,0xd3,0x1f, 0xcde4,0x04,0x20,
		0xcde6,0xd4,0x1f, 0xcde8,0x91,0xb6,	0xcdeb,0x2e,0xc3, 0xcdee,0xf5,0xbd,
		0xcdf1,0x71,0xbe, 0xcdf4,0x41,0x3a,	0xcdf5,0x38,0xe6, 0xcdf7,0xbd,0xc5,
		0xcdf9,0x98,0x26, 0xcdfb,0xb3,0x1f,	0xcdfd,0xb9,0xb6, 0xce00,0x10,0x7f,
		0xce03,0x2f,0x1f, 0xce05,0x2f,0x20,	0xce07,0xe3,0xfe, 0xce0a,0xa7,0x7f,
		0xce0d,0xb0,0x7f, 0xce10,0xad,0x20,	0xce12,0x73,0xfc, 0xce15,0x54,0xfd,
		0xce18,0x4b,0x86, 0xce1a,0x3f,0xbd,	0xce1d,0xbe,0xff, 0xce20,0xf8,0x86,
		0xce22,0xc9,0xb7, 0xce25,0xf5,0xfc,	0xce28,0x2b,0xbb, 0xce2b,0xb9,0xfb,
		0xce2e,0x70,0x1f, 0xce30,0x39,0xb6,	0xce33,0x93,0x8b, 0xce35,0x1b,0xb7,
		0xce38,0x4e,0x81, 0xce3a,0x17,0x23,	0xce3c,0x58,0x86, 0xce3e,0x19,0xb7,
		0xce41,0xcb,0x6d, 0xce45,0xa8,0x26,	0xce47,0x9a,0x10, 0xce4b,0x0d,0xa7,
		0xce4d,0xba,0xfc, 0xce50,0xf5,0xed,	0xce52,0xe7,0xff, 0xce55,0xdf,0xef,
		0xce57,0xa1,0x86, 0xce59,0x0e,0xa7,	0xce5d,0x0e,0xb6, 0xce60,0x02,0x8b,
		0xce62,0x3a,0xb7, 0xce65,0x95,0x1c,	0xce67,0xb4,0x39, 0xce68,0x58,0x86,
		0xce6a,0x11,0x34, 0xce6c,0x39,0xbe,	0xce6f,0xa4,0x7d, 0xce72,0x0d,0x10,
		0xce76,0xf2,0xec, 0xce78,0x10,0x26,	0xce7a,0xb7,0x7d, 0xce7d,0x12,0x27,
		0xce7f,0xa4,0x7a, 0xce82,0x83,0x16,	0xce85,0xa1,0x86, 0xce87,0x32,0xb0,
		0xce8a,0xaf,0xb7, 0xce8d,0xeb,0x6c,	0xce8f,0x43,0x16, 0xce92,0x5d,0x6d,
		0xce94,0x04,0x27, 0xce96,0x48,0x6a,	0xce98,0xa9,0xcc, 0xce9b,0xbd,0xed,
		0xce9d,0x2f,0x86, 0xce9f,0x25,0xa7,	0xcea1,0x04,0x20, 0xcea3,0xa7,0x86,
		0xcea5,0x26,0xa1, 0xcea7,0x07,0x26,	0xcea9,0x70,0x86, 0xceab,0x6c,0xa5,
		0xcead,0x5a,0x27, 0xceaf,0xec,0x6d,	0xceb3,0x01,0x26, 0xceb5,0x85,0x10,
		0xceb9,0xbc,0xec, 0xcebb,0xaa,0xed,	0xcebd,0x30,0xa6, 0xcebf,0xdb,0x5f,
		0xcec0,0xb5,0x84, 0xcec2,0x7d,0xed,	0xcec4,0x26,0xa6, 0xcec6,0x25,0xa7,
		0xcec8,0x1f,0x86, 0xceca,0x05,0xa7,	0xcece,0x2b,0xb6, 0xced1,0xe2,0x8b,
		0xced3,0x27,0xb7, 0xced6,0x10,0x20,	0xced8,0x1e,0xa6, 0xceda,0x52,0x10,
		0xcede,0x3d,0xad, 0xcee0,0x7c,0xec,	0xcee2,0x7b,0xfd, 0xcee5,0x62,0xab,
		0xcee7,0x79,0xeb, 0xcee9,0xd6,0xed,	0xceeb,0x80,0x6d, 0xceef,0x43,0x26,
		0xcef1,0x85,0x10, 0xcef5,0x7d,0xfc,	0xcef8,0xbe,0xed, 0xcefa,0xb9,0xec,
		0xcefc,0xbe,0xed, 0xcefe,0x1b,0xa6,	0xcf00,0x0b,0xa7, 0xcf02,0x49,0xe6,
		0xcf04,0x03,0xa6, 0xcf06,0x35,0x85,	0xcf08,0x02,0x26, 0xcf0a,0xbd,0xed,
		0xcf0c,0x4b,0x86, 0xcf0e,0x0d,0xa7,	0xcf12,0x2b,0xb6, 0xcf15,0xb8,0x8b,
		0xcf17,0x87,0xb7, 0xcf1a,0x10,0x20,	0xcf1c,0x4f,0x84, 0xcf1e,0xbd,0xed,
		0xcf20,0x38,0x86, 0xcf22,0x0b,0xa7,	0xcf26,0x01,0xb6, 0xcf29,0x6a,0x8b,
		0xcf2b,0x0d,0xb7, 0xcf2e,0x14,0x20,	0xcf30,0x38,0xa6, 0xcf32,0x0a,0x81,
		0xcf34,0x92,0x27, 0xcf36,0x24,0x81,	0xcf38,0x10,0x26, 0xcf3a,0x3e,0xbd,
		0xcf3d,0x05,0x30, 0xcf40,0x85,0x8c,	0xcf43,0x1d,0x25, 0xcf45,0xcf,0x8e,
		0xcf48,0xa3,0x6a, 0xcf4a,0x05,0x27,	0xcf4c,0xa7,0x7d, 0xcf4f,0x52,0x10,
		0xcf53,0xef,0xbf, 0xcf56,0x77,0x35,	0xcf70,0xf0,0xa6, 0xcf72,0xf7,0xa7,
		0xcf74,0xca,0x8b, 0xcf76,0xd4,0x84,	0xcf78,0x1a,0xa7, 0xcf7a,0xa5,0x6a,
		0xcf7c,0x10,0x26, 0xcf7e,0xb6,0x6d,	0xcf80,0x03,0x27, 0xcf82,0x4d,0x6f,
		0xcf84,0x4a,0x6f, 0xcf86,0x48,0x6a,	0xcf88,0x2a,0x86, 0xcf8a,0x25,0xa7,
		0xcf8c,0x25,0x20, 0xcf8e,0xe1,0xe6,	0xcf90,0xa6,0x86, 0xcf92,0x25,0xa7,
		0xcf94,0x49,0x6a, 0xcf96,0x01,0x26,	0xcf98,0xff,0x5f, 0xcf99,0x05,0xa6,
		0xcf9b,0x16,0xb1, 0xcf9e,0x25,0x27,	0xcfa0,0x16,0x25, 0xcfa2,0xf9,0xca,
		0xcfa4,0x04,0x20, 0xcfa6,0xfd,0xca,	0xcfa8,0x30,0xa6, 0xcfaa,0x36,0xb1,
		0xcfad,0x32,0x27, 0xcfaf,0x37,0x25,	0xcfb1,0xe8,0xca, 0xcfb3,0x06,0x20,
		0xcfb5,0xe8,0xca, 0xcfb7,0x83,0x10,	0xcfbb,0x97,0xe6, 0xcfbd,0xc2,0xcb,
		0xcfbf,0xf7,0xc4, 0xcfc1,0x36,0xa7, 0xcfc3,0x77,0xbd, 0xcfc6,0x2d,0xe7,
		0xcfc8,0x51,0x10, 0xcfcc,0xfe,0xec,	0xcfce,0xfd,0xed, 0xcfd0,0x50,0x39,
		0xd059,0x13,0x34, 0xd05b,0x02,0x32,	0xd05d,0x0c,0xa6, 0xd05f,0x0d,0xa7,
		0xd061,0xea,0x8b, 0xd063,0xf0,0x84,	0xd065,0x92,0xa7, 0xd067,0x5c,0x6d,
		0xd069,0x12,0x27, 0xd06b,0xa1,0x6a,	0xd06d,0x45,0x10, 0xd071,0x5e,0x6d,
		0xd073,0x05,0x27, 0xd075,0x5a,0x6f,	0xd077,0x59,0x6f, 0xd079,0xa0,0x6a,
		0xd07b,0x43,0x16, 0xd07e,0x4c,0x86,	0xd080,0x27,0xa7, 0xd082,0xe9,0xce,
		0xd085,0x6f,0xec, 0xd087,0xb4,0x81,	0xd089,0x6d,0x24, 0xd08b,0x29,0xa6,
		0xd08d,0x85,0xce, 0xd090,0x68,0xec,	0xd092,0x7d,0xed, 0xd094,0xa0,0x86,
		0xd096,0x25,0xa7, 0xd098,0x72,0x16,	0xd09b,0x99,0xec, 0xd09d,0xa4,0xc3,
		0xd0a0,0x77,0xbd, 0xd0a3,0x3f,0xe7,	0xd0a5,0x30,0xa6, 0xd0a7,0x35,0xa7,
		0xd0a9,0x65,0x10, 0xd0ad,0x65,0x10,	0xd0b0,0x6a,0xeb, 0xd0b2,0x2b,0xe7,
		0xd0b4,0xae,0xc1, 0xd0b6,0x13,0x22,	0xd0b8,0x9c,0xec, 0xd0ba,0x1b,0xab,
		0xd0bc,0x8a,0xeb, 0xd0be,0xb0,0xc1,	0xd0c0,0x15,0x25, 0xd0c2,0xbc,0xc1,
		0xd0c4,0x10,0x22, 0xd0c6,0x7d,0xed,	0xd0c8,0x6c,0x34, 0xd0ca,0x0b,0xbe,
		0xd0cd,0xed,0xe6, 0xd0cf,0x0a,0x3a,	0xd0d0,0xae,0xc6, 0xd0d2,0x3d,0xe5,
		0xd0d4,0x00,0x35, 0xd0d6,0x05,0x27,	0xd0d8,0xce,0xa6, 0xd0da,0xcd,0xa7,
		0xd0dc,0x93,0xce, 0xd0df,0xd9,0xec,	0xd0e1,0x7e,0xed, 0xd0e3,0xb1,0x86,
		0xd0e5,0x32,0xa7, 0xd0e7,0x06,0x20,	0xd0e9,0x78,0xe6, 0xd0eb,0xf9,0xa6,
		0xd0ed,0x7a,0xa7, 0xd0ef,0x8f,0x6a,	0xd0f1,0x10,0x26, 0xd0f3,0x92,0x10,
		0xd0f7,0x33,0xa6, 0xd0f9,0xdc,0xec,	0xd0fb,0xfd,0xe3, 0xd0fd,0x66,0x81,
		0xd0ff,0x38,0x25, 0xd101,0x08,0x81,	0xd103,0xa8,0x22, 0xd105,0x4a,0xc1,
		0xd107,0xbd,0x25, 0xd109,0xc8,0xc1,	0xd10b,0x00,0x22, 0xd10d,0xae,0xed,
		0xd10f,0x09,0xa6, 0xd111,0x9d,0x10,	0xd115,0xe8,0xec, 0xd117,0xfd,0xed,
		0xd119,0x4b,0x86, 0xd11b,0x0d,0xa7,	0xd11d,0x04,0x20, 0xd11f,0x3f,0xbd,
		0xd122,0x98,0xb6, 0xd125,0x1c,0x84,	0xd127,0x3d,0xa7, 0xd129,0x04,0x20,
		0xd12b,0x01,0x32, 0xd12d,0x17,0x35,	0xd12f,0x1f,0xa6, 0xd131,0x1c,0x84,
		0xd133,0x17,0x8b, 0xd135,0xb2,0xa7,	0xd137,0xa1,0xa6, 0xd139,0x5e,0x85,
		0xd13b,0x05,0x26, 0xd13d,0x5e,0x85,	0xd13f,0x03,0x27, 0xd141,0xd6,0x4f,
		0xd142,0xb8,0x20, 0xd144,0x3c,0xa6,	0xd148,0xd4,0x44, 0xd149,0x58,0x81,
		0xd14b,0x04,0x23, 0xd14d,0x5d,0x86,	0xd14f,0x0d,0xa7, 0xd151,0x67,0x6a,
		0xd153,0x29,0x26, 0xd155,0xfc,0xe6,	0xd157,0x89,0x86, 0xd159,0x0a,0xa7,
		0xd15b,0xa0,0x6a, 0xd15d,0x14,0x26,	0xd15f,0x49,0x86, 0xd161,0x24,0xa8,
		0xd163,0x3e,0xa7, 0xd165,0xba,0x27,	0xd167,0x31,0xbd, 0xd16a,0x9c,0xf6,
		0xd16d,0xdc,0xc4, 0xd16f,0x5e,0x86,	0xd171,0xba,0xa7, 0xd173,0x3c,0x20,
		0xd175,0x53,0x5f, 0xd176,0x2a,0xa6,	0xd178,0x0f,0xb1, 0xd17b,0x07,0x27,
		0xd17d,0x16,0x25, 0xd17f,0xea,0xca,	0xd181,0x01,0x20, 0xd183,0xe8,0xca,
		0xd185,0x25,0xa6, 0xd187,0x36,0xb1,	0xd18a,0x6d,0x27, 0xd18c,0x4b,0x25,
		0xd18e,0xa0,0xca, 0xd190,0x04,0x20,	0xd192,0xe8,0xca, 0xd194,0x96,0x10,
		0xd198,0x82,0xe6, 0xd19a,0xac,0xcb,	0xd19c,0x8a,0xc4, 0xd19e,0x09,0x86,
		0xd1a0,0x32,0xa7, 0xd1a2,0x73,0xbd,	0xd1a5,0x3a,0xe7, 0xd1a7,0x82,0x10,
		0xd1ab,0xb4,0xec, 0xd1ad,0xb6,0xed,	0xd1af,0x60,0x39, 0xd1b0,0x30,0xa6,
		0xd1b2,0xa0,0x84, 0xd1b4,0xe2,0x8b,	0xd1b6,0x21,0xa7, 0xd1b8,0x10,0xa6,
		0xd1ba,0x0a,0x85, 0xd1bc,0x50,0x26,	0xd1be,0x08,0x85, 0xd1c0,0x17,0x27,
		0xd1c2,0xcd,0x4f, 0xd1c3,0x10,0x20,	0xd1c5,0x35,0xa6, 0xd1c9,0xdd,0x44,
		0xd1ca,0x18,0x80, 0xd1cc,0x1b,0x81,	0xd1ce,0x44,0x23, 0xd1d0,0xa2,0x86,
		0xd1d2,0x25,0xa7, 0xd1d4,0x5c,0x6a,	0xd1d6,0x01,0x26, 0xd1d8,0x8a,0xe6,
		0xd1da,0x01,0x86, 0xd1dc,0x11,0xa7,	0xd1de,0xe0,0x6a, 0xd1e0,0x10,0x26,
		0xd1e2,0xb0,0x86, 0xd1e4,0x64,0xa8,	0xd1e6,0x30,0xa7, 0xd1e8,0x52,0x27,
		0xd1ea,0x37,0xbd, 0xd1ed,0x91,0xf6,	0xd1f0,0xbc,0xc4, 0xd1f2,0xa1,0x86,
		0xd1f4,0x32,0xa7, 0xd1f6,0x13,0x20,	0xd1f8,0xf3,0x5f, 0xd1f9,0x18,0xa6,
		0xd1fb,0x1a,0xb1, 0xd1fe,0x40,0x27,	0xd200,0x0d,0x25, 0xd202,0xe0,0xca,
		0xd204,0x6b,0x20, 0xd206,0x80,0xca,	0xd208,0x0e,0xa6, 0xd20a,0x1e,0xb1,
		0xd20d,0x03,0x27, 0xd20f,0x15,0x25,	0xd211,0x84,0xca, 0xd213,0x78,0x20,
		0xd215,0x92,0xca, 0xd217,0x5a,0x10,	0xd21b,0x89,0xe6, 0xd21d,0xfb,0xcb,
		0xd21f,0xd9,0xc4, 0xd221,0xb0,0x86,	0xd223,0x84,0xa7, 0xd225,0xf7,0xbd,
		0xd228,0x9a,0xe7, 0xd22a,0x44,0x10,	0xd22e,0xbe,0xec, 0xd230,0xb6,0xed,
		0xd232,0x72,0x39, 0xd233,0xeb,0xa6,	0xd235,0xdc,0x84, 0xd237,0xc2,0x8b,
		0xd239,0x1a,0xa7, 0xd23b,0x0b,0xa6,	0xd23d,0x5e,0x85, 0xd23f,0x10,0x26,
		0xd241,0x3c,0x85, 0xd243,0x8d,0x27,	0xd245,0x7b,0x4f, 0xd246,0x78,0x20,
		0xd248,0x1f,0xa6, 0xd24c,0xd0,0x44,	0xd24d,0x5c,0x80, 0xd24f,0x5c,0x81,
		0xd251,0x7c,0x23, 0xd253,0xc9,0x86,	0xd255,0x98,0xa7, 0xd257,0x40,0x6a,
		0xd259,0x11,0x26, 0xd25b,0x89,0xe6,	0xd25d,0x59,0x86, 0xd25f,0x0d,0xa7,
		0xd261,0xc0,0x6a, 0xd263,0x9c,0x26,	0xd265,0xb0,0x86, 0xd267,0x99,0xa8,
		0xd269,0x1a,0xa7, 0xd26b,0x16,0x27,	0xd26d,0x3f,0xbd, 0xd270,0xd1,0xf6,
		0xd273,0x8f,0xc4, 0xd275,0xb8,0x86,	0xd277,0x8c,0xa7, 0xd279,0x04,0x20,
		0xd27b,0xe0,0x5f, 0xd27c,0x18,0xa6,	0xd27e,0x1d,0xb1, 0xd281,0x10,0x27,
		0xd283,0x15,0x25, 0xd285,0xee,0xca,	0xd287,0x10,0x20, 0xd289,0xd5,0xca,
		0xd28b,0x69,0xa6, 0xd28d,0x9d,0xb1,	0xd290,0x02,0x27, 0xd292,0x15,0x25,
		0xd294,0xed,0xca, 0xd296,0x10,0x20,	0xd298,0x27,0xca, 0xd29a,0xd2,0x10,
		0xd29e,0x09,0xe6, 0xd2a0,0xea,0xcb,	0xd2a2,0xb8,0xc4, 0xd2a4,0xb0,0x86,
		0xd2a6,0x35,0xa7, 0xd2a8,0x1f,0xbd,	0xd2ab,0xec,0xe7, 0xd2ad,0xc5,0x10,
		0xd2b1,0x7c,0xec, 0xd2b3,0x68,0xed,	0xd2b5,0x47,0x39, 0xd2b6,0x27,0xa6,
		0xd2b8,0x94,0x84, 0xd2ba,0xb7,0x8b,	0xd2bc,0x9a,0xa7, 0xd2be,0x8f,0xa6,
		0xd2c0,0xb4,0x85, 0xd2c2,0x01,0x26,	0xd2c4,0xb1,0x85, 0xd2c6,0x05,0x27,
		0xd2c8,0x76,0x4f, 0xd2c9,0x71,0x20,	0xd2cb,0x21,0xa6, 0xd2cf,0xb1,0x44,
		0xd2d0,0xa5,0x80, 0xd2d2,0xb4,0x81,	0xd2d4,0x15,0x23, 0xd2d6,0xa1,0x86,
		0xd2d8,0x6a,0xa7, 0xd2da,0x88,0x6a,	0xd2dc,0x78,0x26, 0xd2de,0xa9,0xe6,
		0xd2e0,0xb0,0x86, 0xd2e2,0x36,0xa7,	0xd2e4,0x48,0x6a, 0xd2e6,0x00,0x26,
		0xd2e8,0x90,0x86, 0xd2ea,0xad,0xa8,	0xd2ec,0x9a,0xa7, 0xd2ee,0xca,0x27,
		0xd2f0,0x77,0xbd, 0xd2f3,0x2a,0xf6,	0xd2f6,0xab,0xc4, 0xd2f8,0x70,0x86,
		0xd2fa,0x6f,0xa7, 0xd2fc,0x6c,0x20,	0xd2fe,0xc4,0x5f, 0xd2ff,0x2e,0xa6,
		0xd301,0x30,0xb1, 0xd304,0xa5,0x27,	0xd306,0xb5,0x25, 0xd308,0xf8,0xca,
		0xd30a,0x10,0x20, 0xd30c,0xe8,0xca,	0xd30e,0x09,0xa6, 0xd310,0x35,0xb1,
		0xd313,0x25,0x27, 0xd315,0x3e,0x25,	0xd317,0xc0,0xca, 0xd319,0x12,0x20,
		0xd31b,0xe8,0xca, 0xd31d,0x43,0x10,	0xd321,0x70,0xe6, 0xd323,0x4f,0xcb,
		0xd325,0x54,0xc4, 0xd327,0x17,0x86,	0xd329,0x1a,0xa7, 0xd32b,0x3a,0xbd,
		0xd32e,0x99,0xe7, 0xd330,0x85,0x10,	0xd334,0xf4,0xec, 0xd336,0xe4,0xed,
		0xd338,0x27,0x39, 0xd339,0x18,0xa6,	0xd33b,0x18,0xa7, 0xd33d,0x6a,0x8b,
		0xd33f,0x5f,0x84, 0xd341,0x36,0xa7,	0xd343,0xe8,0x6a, 0xd345,0xb5,0x26,
		0xd347,0x61,0xe6, 0xd349,0x5f,0x86,	0xd34b,0x0d,0xa7, 0xd34d,0xb1,0x6a,
		0xd34f,0x01,0x26, 0xd351,0x5a,0x5f,	0xd352,0x21,0xa6, 0xd354,0xb7,0xb1,
		0xd357,0x25,0x27, 0xd359,0x00,0x25,	0xd35b,0xe8,0xca, 0xd35d,0x07,0x20,
		0xd35f,0xe8,0xca, 0xd361,0x30,0xa6,	0xd363,0x23,0xb1, 0xd366,0xb0,0x27,
		0xd368,0x16,0x25, 0xd36a,0xe8,0xca,	0xd36c,0x04,0x20, 0xd36e,0xea,0xca,
		0xd370,0x85,0x10, 0xd374,0xf0,0xe6,	0xd376,0xd9,0xcb, 0xd378,0xdc,0xc4,
		0xd37a,0x2f,0xbd, 0xd37d,0x19,0xb6,	0xd380,0xb6,0x84, 0xd382,0xe4,0x8b,
		0xd384,0x25,0xa7, 0xd386,0x77,0xbd,	0xd389,0xc6,0xe7, 0xd38b,0x1a,0x10,
		0xd38f,0xf1,0xec, 0xd391,0x7f,0xed,	0xd393,0x56,0x39, 0xd394,0x21,0xa6,
		0xd396,0x25,0xa7, 0xd398,0x6f,0x8b,	0xd39a,0x59,0x84, 0xd39c,0x08,0xa7,
		0xd39e,0xe8,0x6a, 0xd3a0,0x10,0x26,	0xd3a2,0x3a,0xe6, 0xd3a4,0xb0,0x86,
		0xd3a6,0x31,0xa7, 0xd3a8,0x68,0x6a,	0xd3aa,0xda,0x26, 0xd3ac,0xbb,0x5f,
		0xd3ad,0x50,0xa6, 0xd3af,0x40,0xb1,	0xd3b2,0x03,0x27, 0xd3b4,0x16,0x25,
		0xd3b6,0xff,0xca, 0xd3b8,0x04,0x20,	0xd3ba,0xf9,0xca, 0xd3bc,0x18,0xa6,
		0xd3be,0x18,0xb1, 0xd3c1,0x16,0x27,	0xd3c3,0x15,0x25, 0xd3c5,0xf9,0xca,
		0xd3c7,0x10,0x20, 0xd3c9,0xfb,0xca,	0xd3cb,0x1a,0x10, 0xd3cf,0xc1,0xe6,
		0xd3d1,0xee,0xcb, 0xd3d3,0xb9,0xc4,	0xd3d5,0x72,0xbd, 0xd3d8,0x42,0xb6,
		0xd3db,0x51,0x84, 0xd3dd,0x64,0x8b,	0xd3df,0x4d,0xa7, 0xd3e1,0x77,0xbd,
		0xd3e4,0x3a,0xe7, 0xd3e6,0x87,0x10,	0xd3ea,0xf1,0xec, 0xd3ec,0xf6,0xed,
		0xd3ee,0x25,0x39, 0xd3ef,0x02,0x32,	0xd3f1,0x30,0xa6, 0xd3f3,0x27,0xa7,
		0xd3f5,0xe2,0x8b, 0xd3f7,0xa6,0x84,	0xd3f9,0x5a,0xa7, 0xd3fb,0xbd,0x6a,
		0xd3fd,0x05,0x10, 0xd401,0xf4,0xa6,	0xd403,0x65,0x27, 0xd405,0xd5,0x81,
		0xd407,0x60,0x22, 0xd409,0x2f,0xbd,	0xd40c,0x13,0x20, 0xd40e,0xa5,0x6f,
		0xd410,0x3a,0x6f, 0xd412,0x28,0x6a,	0xd414,0xd1,0x86, 0xd416,0xe5,0xa7,
		0xd418,0x43,0x16, 0xd41b,0x89,0xe6,	0xd41d,0x4e,0x86, 0xd41f,0x0d,0xa7,
		0xd421,0x28,0x6a, 0xd423,0x72,0x26,	0xd425,0xd0,0x86, 0xd427,0xe2,0xa7,
		0xd429,0xb6,0x6d, 0xd42b,0x10,0x27,	0xd42d,0x45,0x10, 0xd430,0xf0,0xa6,
		0xd432,0xc7,0x85, 0xd434,0x70,0x26,	0xd436,0x53,0x10, 0xd43a,0x2b,0xbd,
		0xd43d,0x58,0x86, 0xd43f,0x1c,0xb4,	0xd442,0x7e,0x31, 0xd444,0xfb,0xa6,
		0xd446,0xdd,0x85, 0xd448,0x06,0x27,	0xd44a,0xb5,0x6d, 0xd44c,0x12,0x27,
		0xd44e,0x52,0x10, 0xd451,0x11,0x20,	0xd453,0x2f,0x1f, 0xd455,0x95,0xcb,
		0xd457,0x80,0xca, 0xd459,0x66,0x1f,	0xd45b,0xa0,0x6a, 0xd45d,0x10,0x26,
		0xd45f,0xf9,0xcc, 0xd462,0xb6,0xed,	0xd464,0xf7,0xbd, 0xd467,0xc5,0xf6,
		0xd46a,0xdc,0xc4, 0xd46c,0x58,0x86,	0xd46e,0x0e,0xa7, 0xd470,0x04,0x20,
		0xd472,0x27,0x10, 0xd475,0x13,0x5f,	0xd476,0x9e,0xa6, 0xd478,0x0e,0xa1,
		0xd47a,0x10,0x27, 0xd47c,0x16,0x25,	0xd47e,0xee,0xca, 0xd480,0x15,0x20,
		0xd482,0xe8,0xca, 0xd484,0x35,0xa6,	0xd486,0x34,0xa1, 0xd488,0x3a,0x27,
		0xd48a,0x3d,0x25, 0xd48c,0x71,0xca,	0xd48e,0x98,0x20, 0xd490,0xff,0xca,
		0xd492,0x92,0x10, 0xd496,0x29,0xe6,	0xd498,0x76,0xcb, 0xd49a,0x51,0xc4,
		0xd49c,0xd1,0x86, 0xd49e,0xcd,0xa7,	0xd4a0,0x77,0xbd, 0xd4a3,0x38,0xe7,
		0xd4a5,0x85,0x10, 0xd4a9,0x7c,0xec,	0xd4ab,0x79,0xed, 0xd4ad,0x89,0x32,
		0xd4af,0xbf,0x39, 0xd4b0,0x15,0x34,	0xd4b2,0x10,0x32, 0xd4b4,0xb0,0x86,
		0xd4b6,0x24,0xa7, 0xd4b8,0x70,0xce,	0xd4bb,0x97,0x10, 0xd4bf,0x38,0xec,
		0xd4c1,0x62,0xab, 0xd4c3,0x6c,0xeb,	0xd4c5,0x94,0x10, 0xd4c9,0x67,0x26,
		0xd4cb,0x25,0xa7, 0xd4cd,0x6a,0xa6,	0xd4cf,0x74,0x81, 0xd4d1,0x16,0x27,
		0xd4d3,0xb4,0x81, 0xd4d5,0x11,0x27,	0xd4d7,0xb4,0x81, 0xd4d9,0x3e,0x27,
		0xd4db,0x23,0x16, 0xd4de,0x29,0xa6,	0xd4e0,0xa6,0x81, 0xd4e2,0x16,0x25,
		0xd4e4,0xa6,0x81, 0xd4e6,0x14,0x22,	0xd4e8,0x06,0xc1, 0xd4ea,0xde,0x25,
		0xd4ec,0x06,0xc1, 0xd4ee,0xdc,0x22,	0xd4f0,0x7e,0xed, 0xd4f2,0xa8,0xc3,
		0xd4f5,0x77,0xbd, 0xd4f8,0x13,0x1f,	0xd4fa,0x5e,0xbe, 0xd4fd,0xe9,0x3a,
		0xd4fe,0xbe,0xe6, 0xd500,0x5a,0x10,	0xd504,0xcf,0xc5, 0xd506,0x29,0x26,
		0xd508,0xcf,0xc5, 0xd50a,0x52,0x10,	0xd50e,0x09,0xa6, 0xd510,0x1c,0x81,
		0xd512,0xa9,0x26, 0xd514,0x96,0x86,	0xd516,0xa5,0xa7, 0xd518,0x5b,0x86,
		0xd51a,0x3f,0xbd, 0xd51d,0x07,0x20,	0xd51f,0x10,0x20, 0xd521,0x60,0x6a,
		0xd523,0x3f,0x27, 0xd525,0xf8,0xe6,	0xd527,0xe5,0xeb, 0xd529,0xdc,0xc4,
		0xd52b,0x9f,0xe7, 0xd52d,0x38,0xae,	0xd52f,0x53,0x16, 0xd532,0x4c,0xc6,
		0xd534,0xf2,0xe7, 0xd536,0x24,0x32,	0xd538,0x17,0x35, 0xd53a,0x01,0x34,
		0xd53c,0xbc,0xec, 0xd53e,0xca,0xc3,	0xd541,0xa4,0xbd, 0xd544,0x95,0x1f,
		0xd546,0x23,0xbe, 0xd549,0x23,0x3a,	0xd54a,0xc9,0xc6, 0xd54c,0xd1,0x53,
		0xd54d,0x9b,0xe4, 0xd54f,0x67,0x1f,	0xd551,0x99,0x84, 0xd553,0xad,0xa7,
		0xd555,0x91,0x8e, 0xd558,0xd8,0xc4,	0xd55a,0x22,0x3a, 0xd55b,0x09,0xa6,
		0xd55d,0x5a,0x85, 0xd55f,0x05,0x27,	0xd561,0xb8,0xa6, 0xd565,0x04,0x80,
		0xd567,0x24,0xa7, 0xd56b,0xb6,0x6f,	0xd56d,0xb2,0x6f, 0xd56f,0xa4,0x6f,
		0xd571,0x72,0x6f, 0xd573,0x66,0x6f,	0xd575,0x98,0x86, 0xd577,0xac,0xa7,
		0xd579,0x58,0x86, 0xd57b,0x0e,0xa7,	0xd57d,0xfc,0xcc, 0xd580,0x6f,0xed,
		0xd582,0x79,0xec, 0xd584,0x92,0x10,	0xd588,0x6f,0x6d, 0xd58c,0x4c,0x26,
		0xd58e,0xbd,0xed, 0xd590,0xb3,0x86,	0xd592,0x25,0xa7, 0xd596,0x23,0xb6,
		0xd599,0x6e,0x8b, 0xd59b,0x47,0xb7,	0xd59e,0xe1,0xce, 0xd5a1,0x31,0xb6,
		0xd5a4,0x78,0xee, 0xd5a6,0x79,0xef,	0xd5a8,0xa2,0x8b, 0xd5aa,0x9d,0xb7,
		0xd5ad,0xfe,0x6d, 0xd5b1,0x10,0x26,	0xd5b3,0x90,0x10, 0xd5b7,0x6d,0xef,
		0xd5b9,0xf2,0xef, 0xd5bb,0x45,0xa6,	0xd5bd,0x52,0xa7, 0xd5bf,0x47,0xa6,
		0xd5c1,0x3f,0xe6, 0xd5c3,0x7d,0xed,	0xd5c5,0xb1,0x86, 0xd5c7,0x25,0xa7,
		0xd5cb,0x0b,0xb6, 0xd5ce,0x2c,0x8b,	0xd5d0,0x23,0xb7, 0xd5d3,0x17,0x35,
		0xd5e5,0x15,0x34, 0xd5e7,0xc0,0x1f,	0xd5e9,0xbc,0xec, 0xd5eb,0x86,0xc3,
		0xd5ee,0x34,0xbd, 0xd5f1,0x71,0xbe,	0xd5f4,0x41,0x3a, 0xd5f5,0x38,0xe6,
		0xd5f7,0xad,0xc5, 0xd5f9,0x5a,0x27,	0xd5fb,0x50,0x12, 0xd5fc,0x41,0x12,
		0xd5fd,0xc3,0x53, 0xd5fe,0xb2,0xca,	0xd600,0x2e,0x53, 0xd601,0x99,0xe7,
		0xd603,0x15,0x6d, 0xd607,0x49,0x26,	0xd609,0x28,0xbe, 0xd60c,0xaf,0xec,
		0xd60e,0xbd,0xed, 0xd610,0xeb,0x86,	0xd612,0x27,0x5f, 0xd613,0x95,0xed,
		0xd615,0xfe,0x86, 0xd617,0xcd,0xa7,	0xd61b,0x0b,0xb6, 0xd61e,0x6c,0x8b,
		0xd620,0xdb,0xb7, 0xd623,0x5b,0x20,	0xd625,0x16,0x6d, 0xd629,0x10,0x26,
		0xd62b,0x2e,0xbe, 0xd62e,0xbe,0xec,	0xd630,0x96,0xed, 0xd632,0xed,0x86,
		0xd634,0x33,0x5f, 0xd635,0x96,0xed,	0xd637,0xf8,0x86, 0xd639,0x1a,0xa7,
		0xd63d,0x19,0xb6, 0xd640,0xf2,0x8b,	0xd642,0xcf,0xb7, 0xd645,0x4d,0x86,
		0xd647,0x77,0xbd, 0xd64a,0x49,0x86,	0xd64c,0x0d,0xa7, 0xd64e,0x17,0x35,
		0xd650,0x42,0x8e, 0xd653,0x09,0xc6,	0xd655,0xfb,0x34, 0xd657,0x09,0xc6,
		0xd659,0x55,0x10, 0xd65c,0x48,0x86,	0xd65e,0x0d,0xa7, 0xd660,0x16,0x6d,
		0xd664,0xf8,0x26, 0xd666,0x37,0xfe,	0xd669,0x45,0x10, 0xd66c,0x9a,0xe7,
		0xd66e,0x67,0x1a, 0xd670,0x58,0x86,	0xd672,0x6b,0xa7, 0xd676,0x6f,0xb6,
		0xd679,0x6a,0x8b, 0xd67b,0x0b,0xb7,	0xd67e,0x66,0x1c, 0xd680,0x04,0x31,
		0xd682,0x48,0x6a, 0xd684,0x05,0x26,	0xd686,0x48,0x6a, 0xd688,0xfb,0x26,
		0xd68a,0xff,0x35, 0xd68c,0xf3,0x10,	0xd690,0x6b,0xee, 0xd692,0xa1,0x86,
		0xd694,0x4a,0x6d, 0xd698,0xeb,0x26,	0xd69a,0x8b,0xbe, 0xd69d,0x1f,0xef,
		0xd69f,0xad,0xa7, 0xd6a1,0xc1,0x1a,	0xd6a3,0xa2,0x86, 0xd6a5,0x32,0xa7,
		0xd6a9,0xb9,0xb6, 0xd6ac,0x82,0x8b,	0xd6ae,0xb9,0xb7, 0xd6b1,0xd5,0x1c,
		0xd6b3,0x96,0x10, 0xd6b7,0x13,0x25,	0xd6b9,0xcf,0x39, 0xd6ba,0xad,0xa6,
		0xd6be,0xb4,0x81, 0xd6c0,0x12,0x23,	0xd6c2,0xa1,0x86, 0xd6c4,0x25,0xa7,
		0xd6c8,0x92,0x8b, 0xd6ca,0x50,0x48,	0xd6cb,0x85,0xa7, 0xd6cf,0x89,0x86,
		0xd6d1,0x22,0xa7, 0xd6d5,0x97,0x10,	0xd6d9,0xab,0x8e, 0xd6dc,0x00,0xe6,
		0xd6e0,0x70,0xae, 0xd6e2,0x99,0x54,	0xd6e3,0x03,0x33, 0xd6e5,0x5a,0x6f,
		0xd6e7,0x5d,0x6f, 0xd6e9,0x72,0x6f,	0xd6eb,0x77,0x6f, 0xd6ed,0x72,0x6f,
		0xd6ef,0x97,0xa6, 0xd6f1,0x32,0xa7,	0xd6f3,0x25,0xa6, 0xd6f5,0x32,0xa7,
		0xd6f7,0xa4,0x85, 0xd6f9,0xd2,0x27,	0xd6fb,0x85,0xa6, 0xd6ff,0x4e,0x44,
		0xd700,0xdd,0xa7, 0xd702,0x91,0xec,	0xd704,0x47,0xed, 0xd706,0x55,0xed,
		0xd708,0x5b,0x81, 0xd70a,0x01,0x26,	0xd70c,0xcc,0xc6, 0xd70e,0x10,0x20,
		0xd710,0x6d,0xc1, 0xd712,0x89,0x26,	0xd714,0x68,0xc6, 0xd716,0x98,0x20,
		0xd718,0xca,0xc6, 0xd71a,0x8d,0xe7,	0xd71c,0xea,0xcc, 0xd71f,0xbd,0xed,
		0xd721,0x38,0xa6, 0xd723,0x1d,0x81,	0xd725,0x98,0x26, 0xd727,0xd5,0x6f,
		0xd729,0xb2,0x6f, 0xd72b,0x0e,0xa6,	0xd72d,0x6a,0x8b, 0xd72f,0x1d,0xa7,
		0xd731,0x8c,0x20, 0xd733,0x3e,0x81,	0xd735,0x98,0x26, 0xd737,0xc5,0x6f,
		0xd739,0xb2,0x6f, 0xd73b,0x10,0x30,	0xd73d,0x07,0x31, 0xd740,0xa8,0x10,
		0xd744,0xc9,0x25, 0xd746,0xd6,0x39,	0xd75b,0x89,0xe6, 0xd75f,0xdc,0xc1,
		0xd761,0xa2,0x23, 0xd763,0x51,0xc6,	0xd765,0x1a,0xe7, 0xd769,0x78,0x8e,
		0xd76c,0x18,0xa6, 0xd770,0xbc,0x44,	0xd771,0x58,0xa6, 0xd773,0xca,0x34,
		0xd775,0xed,0x10, 0xd779,0x78,0x8e,	0xd77c,0x38,0xae, 0xd77e,0x58,0x86,
		0xd780,0x37,0xa5, 0xd782,0x01,0x26,	0xd784,0x92,0x10, 0xd788,0xff,0x25,
		0xd78a,0xa9,0xa6, 0xd78c,0x2f,0x85,	0xd78e,0x61,0x26, 0xd790,0xe8,0xcc,
		0xd793,0x7d,0xed, 0xd795,0x6a,0xed,	0xd797,0x21,0xa6, 0xd799,0x22,0xa7,
		0xd79b,0x99,0xcc, 0xd79e,0x95,0xed,	0xd7a0,0xb0,0x86, 0xd7a2,0x30,0xa7,
		0xd7a4,0xb0,0x86, 0xd7a6,0x61,0xab,	0xd7a8,0xba,0xa7, 0xd7aa,0xeb,0x20,
		0xd7ac,0xd2,0x6f, 0xd7ae,0xd1,0x6f,	0xd7b0,0x5a,0x6f, 0xd7b2,0x4b,0x6f,
		0xd7b4,0x30,0xa6, 0xd7b6,0x21,0xa7,	0xd7b8,0x30,0xa6, 0xd7ba,0x23,0xa7,
		0xd7bc,0x94,0xec, 0xd7be,0x80,0xed,	0xd7c0,0x7d,0xed, 0xd7c2,0xb4,0x81,
		0xd7c4,0x02,0x26, 0xd7c6,0xa9,0xc6,	0xd7c8,0x70,0x20, 0xd7ca,0xbc,0xc1,
		0xd7cc,0xe2,0x26, 0xd7ce,0x29,0xc6,	0xd7d0,0x06,0x20, 0xd7d2,0xa9,0xc6,
		0xd7d4,0x3f,0xe7, 0xd7d6,0xf9,0xcc,	0xd7d9,0x02,0xed, 0xd7db,0xa1,0xa6,
		0xd7dd,0xb8,0x81, 0xd7df,0xe1,0x26,	0xd7e1,0x5a,0x6f, 0xd7e3,0x5b,0x6f,
		0xd7e5,0x30,0xa6, 0xd7e7,0xf7,0x8b,	0xd7e9,0x52,0xa7, 0xd7eb,0x4a,0x81,
		0xd7ed,0xf0,0x26, 0xd7ef,0x47,0x6f,	0xd7f1,0x5a,0x6f, 0xd7f3,0x12,0x30,
		0xd7f5,0x07,0x31, 0xd7f8,0xed,0x10,	0xd7fc,0xf6,0x25, 0xd7fe,0xf7,0x35,
		0xd830,0xd8,0x8e, 0xd833,0xe5,0xa6,	0xd835,0x45,0x10, 0xd838,0xb9,0xfe,
		0xd83b,0xa6,0x6d, 0xd83f,0x07,0x26,	0xd841,0xa0,0x10, 0xd844,0xfc,0xa7,
		0xd846,0xc9,0x86, 0xd848,0x1f,0xa7,	0xd84c,0x0d,0xb6, 0xd84f,0x6c,0x8b,
		0xd851,0x87,0xb7, 0xd854,0xbf,0x8c,	0xd857,0x15,0x25, 0xd859,0xb5,0x6d,
		0xd85d,0x07,0x26, 0xd85f,0x2b,0xbe,	0xd862,0x99,0xcc, 0xd865,0xde,0xed,
		0xd867,0xa2,0x86, 0xd869,0x1a,0xa7,	0xd86b,0x58,0x86, 0xd86d,0x1a,0xa7,
		0xd871,0x91,0xb6, 0xd874,0xaa,0x8b,	0xd876,0x90,0xb7, 0xd879,0xb6,0x6d,
		0xd87d,0x10,0x26, 0xd87f,0x39,0xbe,	0xd882,0xf9,0xcc, 0xd885,0x6e,0xed,
		0xd887,0xa1,0x86, 0xd889,0xce,0xa7,	0xd88b,0xc1,0x86, 0xd88d,0xcc,0xa7,
		0xd891,0x33,0xb6, 0xd894,0xe5,0x8b,	0xd896,0x27,0xb7, 0xd899,0x28,0x6d,
		0xd89d,0x89,0x26, 0xd89f,0xeb,0xbe,	0xd8a2,0xea,0xcc, 0xd8a5,0x7e,0xed,
		0xd8a7,0xb7,0x86, 0xd8a9,0xda,0xa7,	0xd8ab,0xd5,0x86, 0xd8ad,0xda,0xa7,
		0xd8b1,0x31,0xb6, 0xd8b4,0xe2,0x8b,	0xd8b6,0x25,0xb7, 0xd8b9,0x3e,0x6d,
		0xd8bd,0x98,0x26, 0xd8bf,0xeb,0xbe,	0xd8c2,0xf9,0xcc, 0xd8c5,0x7b,0xed,
		0xd8c7,0xa1,0x86, 0xd8c9,0x77,0xa7,	0xd8cb,0x69,0x86, 0xd8cd,0x79,0xa7,
		0xd8d1,0x26,0xb6, 0xd8d4,0xf0,0x8b,	0xd8d6,0x27,0xb7, 0xd8d9,0x09,0x39,
		0xd93a,0x68,0x8e, 0xd93d,0x18,0xa6,	0xd93f,0x53,0x10, 0xd942,0x63,0xfe,
		0xd945,0x7e,0x6d, 0xd949,0x13,0x26,	0xd94b,0x52,0x10, 0xd94e,0x0d,0xa7,
		0xd950,0x92,0x86, 0xd952,0xa5,0xa7,	0xd956,0xa3,0xb6, 0xd959,0x7e,0x8b,
		0xd95b,0x0f,0xb7, 0xd95e,0x79,0x8c,	0xd961,0xb6,0x25, 0xd963,0xbb,0x39,
		0xd964,0x18,0x8e, 0xd967,0xb6,0xa6,	0xd969,0x45,0x10, 0xd96c,0xb9,0xfe,
		0xd96f,0xb5,0x6d, 0xd973,0x24,0x26,	0xd975,0x05,0x10, 0xd978,0x61,0x1a,
		0xd97a,0x48,0x86, 0xd97c,0x1a,0xa7,	0xd980,0x27,0xb6, 0xd983,0xe4,0x8b,
		0xd985,0x34,0xb7, 0xd988,0xa5,0x1c,	0xd98a,0xf1,0x8c, 0xd98d,0x0e,0x25,
		0xd98f,0x3e,0x39, 0xd9d5,0xf1,0x8e,	0xd9d8,0x5a,0xa6, 0xd9da,0x12,0x10,
		0xd9dd,0xe9,0xfe, 0xd9e0,0x5e,0x6d,	0xd9e4,0x10,0x26, 0xd9e6,0x97,0x10,
		0xd9e9,0x52,0xa7, 0xd9eb,0x4c,0x86,	0xd9ed,0x5a,0xa7, 0xd9f1,0x31,0xb6,
		0xd9f4,0xe2,0x8b, 0xd9f6,0x24,0xb7,	0xd9f9,0x74,0x8c, 0xd9fc,0x1e,0x25,
		0xd9fe,0x3d,0x39, 0xd9ff,0x66,0x8e,	0xda02,0xc9,0xa6, 0xda04,0x72,0x10,
		0xda07,0x8b,0xfe, 0xda0a,0xb5,0x6d,	0xda0e,0x01,0x26, 0xda10,0x74,0x10,
		0xda13,0xe1,0x86, 0xda15,0xc0,0xa7,	0xda19,0x1b,0xb6, 0xda1c,0x6d,0x8b,
		0xda1e,0x0f,0xb7, 0xda21,0xfc,0x8c,	0xda24,0x56,0x25, 0xda26,0x4c,0x39,
		0xda4b,0x69,0x8e, 0xda4e,0x09,0xa6,	0xda50,0xd6,0x10, 0xda53,0x2b,0xfe,
		0xda56,0xbd,0x6d, 0xda5a,0x01,0x26,	0xda5c,0x40,0x10, 0xda5f,0x0d,0xa7,
		0xda61,0x50,0x86, 0xda63,0x75,0xa7,	0xda67,0x61,0xb6, 0xda6a,0x6a,0x8b,
		0xda6c,0x1b,0xb7, 0xda6f,0x6c,0x8c,	0xda72,0xe0,0x25, 0xda74,0xef,0x39,
		0xda75,0x58,0x8e, 0xda78,0x18,0xa6,	0xda7a,0x56,0x10, 0xda7d,0xb9,0xfe,
		0xda80,0x5e,0x6d, 0xda84,0x04,0x26,	0xda86,0x92,0x10, 0xda89,0xe7,0x86,
		0xda8b,0xed,0xa7, 0xda8f,0xeb,0xb6,	0xda92,0xe4,0x8b, 0xda94,0x31,0xb7,
		0xda97,0xf1,0x8c, 0xda9a,0xbd,0x25,	0xda9c,0x9f,0x39, 0xdac1,0xe5,0x8e,
		0xdac4,0x22,0xa6, 0xdac6,0x92,0x10,	0xdac9,0x75,0xfe, 0xdacc,0x21,0x6d,
		0xdad0,0x00,0x26, 0xdad2,0x92,0x10,	0xdad5,0x36,0xa7, 0xdad7,0xa1,0x86,
		0xdad9,0xc0,0xa7, 0xdadd,0xc7,0xb6,	0xdae0,0xe2,0x8b, 0xdae2,0x36,0xb7,
		0xdae5,0xf4,0x8c, 0xdae8,0xbe,0x25,	0xdaea,0x8f,0x39, 0xdaeb,0xff,0x8e,
		0xdaee,0xc1,0xa6, 0xdaf0,0x85,0x10,	0xdaf3,0x6b,0xfe, 0xdaf6,0x5b,0x6d,
		0xdafa,0x83,0x26, 0xdafc,0x8d,0x10,	0xdaff,0xdc,0x86, 0xdb01,0x07,0xa7,
		0xdb05,0x14,0xb6, 0xdb08,0x7a,0x8b,	0xdb0a,0x0f,0xb7, 0xdb0d,0x6e,0x8c,
		0xdb10,0x86,0x25, 0xdb12,0x9e,0x39,	0xdb13,0xdf,0x7d, 0xdb16,0xb2,0x10,
		0xdb1a,0xe9,0xce, 0xdb1d,0xba,0xfc,	0xdb20,0xa2,0xc3, 0xdb23,0x1e,0xbd,
		0xdb26,0x19,0xbe, 0xdb29,0x21,0x3a,	0xdb2a,0xbd,0xec, 0xdb2c,0x45,0x10,
		0xdb30,0x5c,0xec, 0xdb32,0x2d,0x8b,	0xdb34,0x07,0xb1, 0xdb37,0xb6,0x10,
		0xdb3b,0x58,0x80, 0xdb3d,0x0f,0xb1,	0xdb40,0x85,0x22, 0xdb42,0x6c,0xcb,
		0xdb44,0x01,0xf1, 0xdb47,0xd5,0x25,	0xdb49,0xc8,0xc0, 0xdb4b,0x9e,0xf1,
		0xdb4e,0x00,0x22, 0xdb50,0x14,0xe6,	0xdb52,0x35,0xc5, 0xdb54,0xd3,0x27,
		0xdb56,0x61,0x86, 0xdb58,0x1f,0xa5,	0xdb5a,0x01,0x26, 0xdb5c,0x1a,0xa6,
		0xdb5e,0x5d,0x85, 0xdb60,0x90,0x26,	0xdb62,0xdf,0x6a, 0xdb66,0x70,0x86,
		0xdb68,0x28,0xaa, 0xdb6a,0x0d,0xa7,	0xdb6c,0x18,0xa6, 0xdb6e,0xe1,0x48,
		0xdb6f,0x9c,0xe6, 0xdb71,0x34,0xc4,	0xdb73,0x59,0xbd, 0xdb76,0x40,0xa6,
		0xdb7a,0x68,0x8b, 0xdb7c,0x1a,0xa7,	0xdb80,0xb7,0x86, 0xdb82,0x77,0xbd,
		0xdb85,0x24,0xa6, 0xdb87,0xb1,0x84,	0xdb89,0xe6,0xe6, 0xdb8b,0xbb,0x54,
		0xdb8c,0x4f,0x8e, 0xdb8f,0x41,0xae,	0xdb91,0x2a,0xe6, 0xdb93,0xa1,0x86,
		0xdb95,0x73,0xbd, 0xdb98,0x56,0xbd,	0xdb9b,0xd9,0xcc, 0xdb9e,0xd5,0xed,
		0xdba0,0x04,0x20, 0xdba2,0xaf,0xc1,	0xdba4,0x12,0x27, 0xdba6,0xae,0xc1,
		0xdba8,0xb8,0x26, 0xdbaa,0xf4,0x86,	0xdbac,0x57,0xbd, 0xdbaf,0x54,0xbd,
		0xdbb2,0x06,0x20, 0xdbb4,0x03,0x33,	0xdbb7,0x94,0x11, 0xdbbb,0x3c,0x10,
		0xdbbf,0x0e,0x1c, 0xdbc1,0x51,0x39,	0xdbf2,0x06,0x32, 0xdbf4,0x5e,0x6d,
		0xdbf8,0xb0,0x26, 0xdbfa,0xd6,0xbe,	0xdbfd,0x54,0xec, 0xdbff,0x47,0xed,
		0xdc01,0x8e,0xfc, 0xdc04,0x9e,0xed,	0xdc06,0x9d,0xed, 0xdc08,0xb9,0xed,
		0xdc0a,0x09,0xa6, 0xdc0c,0x1c,0xa7,	0xdc0e,0xc9,0xc6, 0xdc10,0xd3,0xa6,
		0xdc12,0xf1,0x84, 0xdc14,0x9a,0xed,	0xdc16,0xe1,0x86, 0xdc18,0x1a,0xa7,
		0xdc1c,0x09,0xb6, 0xdc1f,0x6c,0x8b,	0xdc21,0xd3,0xb7, 0xdc24,0xf0,0x86,
		0xdc26,0xd4,0xa7, 0xdc28,0xb3,0x7f,	0xdc2b,0xb4,0x7f, 0xdc2e,0xb0,0x7f,
		0xdc31,0x1b,0x7f, 0xdc34,0x1b,0x7f,	0xdc37,0x12,0x6d, 0xdc3b,0x02,0x26,
		0xdc3d,0x39,0xbe, 0xdc40,0xdf,0xb6,	0xdc43,0x65,0xa7, 0xdc45,0x28,0xfc,
		0xdc48,0xac,0xed, 0xdc4a,0xbb,0xfc,	0xdc4d,0xa9,0xfd, 0xdc50,0x38,0xed,
		0xdc52,0x41,0x86, 0xdc54,0x70,0xa7,	0xdc58,0x0b,0xb6, 0xdc5b,0x6c,0x8b,
		0xdc5d,0x1f,0xb7, 0xdc60,0x4c,0x20,	0xdc62,0x49,0x32, 0xdc64,0x50,0x86,
		0xdc66,0x74,0xa7, 0xdc68,0x58,0x86,	0xdc6a,0x1f,0xa7, 0xdc6c,0x58,0x86,
		0xdc6e,0x1c,0xa7, 0xdc70,0x71,0xb6,	0xdc73,0xb8,0x8b, 0xdc75,0x5c,0x84,
		0xdc77,0x7f,0xb7, 0xdc7a,0xa7,0x6d,	0xdc7e,0x06,0x26, 0xdc80,0x62,0xbe,
		0xdc83,0x25,0xa7, 0xdc85,0x68,0xfc,	0xdc88,0x19,0xed, 0xdc8a,0x8c,0xab,
		0xdc8c,0x4e,0xfd, 0xdc8f,0x5d,0xed,	0xdc91,0xc4,0x1a, 0xdc93,0xa1,0x86,
		0xdc95,0x27,0xa7, 0xdc99,0xe8,0xb6,	0xdc9c,0xd4,0x8b, 0xdc9e,0xef,0xb7,
		0xdca1,0xd5,0x1c, 0xdca3,0xb2,0x86,	0xdca5,0x77,0xbd, 0xdca8,0x48,0x6a,
		0xdcaa,0xa8,0xa6, 0xdcac,0xe6,0x81,	0xdcae,0xbe,0x26, 0xdcb0,0xb0,0x86,
		0xdcb2,0x26,0xa7, 0xdcb4,0xde,0x4d,	0xdcb5,0x10,0x26, 0xdcb7,0x49,0x6a,
		0xdcb9,0xb8,0x26, 0xdcbb,0x09,0x6d,	0xdcbf,0xf9,0x26, 0xdcc1,0x64,0xbe,
		0xdcc4,0x6f,0xfc, 0xdcc7,0x7d,0xed,	0xdcc9,0xa8,0x1a, 0xdccb,0xc9,0x86,
		0xdccd,0xd3,0xa7, 0xdcd1,0x20,0xb6,	0xdcd4,0xf4,0x8b, 0xdcd6,0x27,0xb7,
		0xdcd9,0x02,0x1c, 0xdcdb,0x2a,0x32,	0xdcdd,0xf1,0x39, 0xdcde,0x93,0x34,
		0xdce0,0x5b,0x7f, 0xdce3,0x5c,0x7f,	0xdce6,0x58,0x7f, 0xdce9,0x33,0x7f,
		0xdcec,0x33,0x7f, 0xdcef,0x23,0x6d,	0xdcf3,0x02,0x26, 0xdcf5,0x71,0xbe,
		0xdcf8,0xd8,0x86, 0xdcfa,0xc6,0xa7,	0xdcfc,0x75,0xfc, 0xdcff,0x32,0xed,
		0xdd01,0x4d,0xfc, 0xdd04,0x4f,0xfd,	0xdd07,0x5d,0xed, 0xdd09,0x5f,0x86,
		0xdd0b,0x0d,0xa7, 0xdd0f,0x0b,0xb6,	0xdd12,0x2c,0x8b, 0xdd14,0x01,0xb7,
		0xdd17,0x21,0x86, 0xdd19,0x0a,0xb7,	0xdd1c,0x03,0x35, 0xdd1e,0xb7,0x7d,
		0xdd21,0x92,0x27, 0xdd23,0xda,0x7f,	0xdd26,0x5f,0xfc, 0xdd29,0xca,0xc3,
		0xdd2c,0x3f,0xbd, 0xdd2f,0x3a,0xbe,	0xdd32,0x9c,0x3a, 0xdd33,0xcf,0x6d,
		0xdd35,0x90,0x26, 0xdd37,0x26,0x86,	0xdd39,0x1a,0xa7, 0xdd3b,0xa9,0xfc,
		0xdd3e,0xa1,0x6d, 0xdd42,0x89,0x26,	0xdd44,0x48,0xbe, 0xdd47,0x1d,0xed,
		0xdd49,0x4e,0x86, 0xdd4b,0xe7,0x5f,	0xdd4c,0xbb,0xed, 0xdd4e,0x49,0x86,
		0xdd50,0x4f,0xa7, 0xdd54,0x5c,0xb6,	0xdd57,0x64,0x8b, 0xdd59,0x1e,0xb7,
		0xdd5c,0x4c,0x86, 0xdd5e,0x3f,0xbd,	0xdd61,0x24,0x20, 0xdd63,0x7e,0x7d,
		0xdd66,0xc0,0x27, 0xdd68,0xb3,0x7f,	0xdd6b,0x3e,0xbd, 0xdd6e,0x4a,0x86,
		0xdd70,0x57,0xbd, 0xdd73,0xc2,0x39,	0xdd74,0xdd,0x34, 0xdd76,0xcf,0x32,
		0xdd78,0x9a,0xe7, 0xdd7a,0x48,0x80,	0xdd7c,0xd4,0x44, 0xdd7d,0xd4,0x44,
		0xdd7e,0xd2,0x44, 0xdd7f,0xd6,0x44,	0xdd80,0xac,0xc6, 0xdd82,0x57,0x3d,
		0xdd83,0x7d,0xed, 0xdd85,0xdc,0x4f,	0xdd86,0x29,0xe6, 0xdd88,0x22,0xc0,
		0xdd8a,0x73,0x54, 0xdd8b,0xbb,0x54,	0xdd8c,0x5c,0x8e, 0xdd8f,0xe1,0xe6,
		0xdd91,0x3e,0xe3, 0xdd93,0x02,0x32,	0xdd95,0x02,0x35, 0xddcb,0x1b,0x34,
		0xddcd,0x00,0x7f, 0xddd0,0xa6,0x86,	0xddd2,0x77,0xbd, 0xddd5,0x76,0xbd,
		0xddd8,0x95,0xbd, 0xdddb,0x97,0xbd,	0xddde,0x97,0xbd, 0xdde1,0x77,0xbd,
		0xdde4,0x77,0xbd, 0xdde7,0x5e,0x6a,	0xddeb,0x19,0x26, 0xdded,0x01,0x7a,
		0xddf0,0x10,0x26, 0xddf2,0x76,0xbd,	0xddf5,0xb0,0x86, 0xddf7,0x70,0xbd,
		0xddfa,0x96,0xbd, 0xddfd,0xc9,0x1a,	0xddff,0xf0,0x35, 0xde01,0x64,0xbd,
		0xde04,0xfa,0x20, 0xde06,0x6b,0xb6,	0xde09,0x58,0x81, 0xde0b,0x01,0x26,
		0xde0d,0x2e,0xbd, 0xde10,0x5a,0x86,	0xde12,0x77,0xbd, 0xde15,0x62,0xbd,
		0xde18,0x5a,0x86, 0xde1a,0x3a,0xb8,	0xde1d,0x08,0xb7, 0xde20,0x19,0xb6,
		0xde23,0x4f,0x81, 0xde25,0xf8,0x26,	0xde27,0x71,0xbd, 0xde2a,0x5a,0x86,
		0xde2c,0x3f,0xbd, 0xde2f,0x39,0xbd,	0xde32,0x67,0xbd, 0xde35,0x14,0xcc,
		0xde38,0x3f,0xbd, 0xde3b,0x2b,0xbd,	0xde3e,0x2d,0xbd, 0xde41,0x65,0xbd,
		0xde44,0xc2,0x20, 0xde46,0x5f,0xbd,	0xde49,0x5e,0x86, 0xde4b,0x3f,0xbd,
		0xde4e,0x3f,0xbd, 0xde51,0x58,0xbd,	0xde54,0x2a,0xcc, 0xde57,0x5f,0xbd,
		0xde5a,0x3f,0xbd, 0xde5d,0x3a,0xbd,	0xde60,0x5f,0xbd, 0xde63,0x7c,0xb6,
		0xde66,0x60,0x81, 0xde68,0x10,0x26,	0xde6a,0x19,0xa6, 0xde6e,0x13,0x26,
		0xde70,0x50,0xa6, 0xde74,0x47,0xb1,	0xde77,0xc7,0x26, 0xde79,0x3f,0xbd,
		0xde7c,0x04,0x20, 0xde7e,0x2c,0xbd,	0xde81,0x64,0xbd, 0xde84,0x67,0xbd,
		0xde87,0x77,0xbd, 0xde8a,0x17,0xbd,	0xde8d,0xa6,0xbd, 0xde90,0xc0,0x1c,
		0xde92,0x17,0x35, 0xde94,0x63,0xbd,	0xde97,0xa1,0x86, 0xde99,0x84,0xb1,
		0xde9c,0xd0,0x26, 0xde9e,0xb7,0xbd,	0xdea1,0x04,0x20, 0xdea3,0x73,0xbd,
		0xdea6,0x75,0xbd, 0xdea9,0x3c,0xcc,	0xdeac,0xb7,0xbd, 0xdeaf,0xb1,0xbd,
		0xdeb2,0x67,0xbd, 0xdeb5,0x77,0xbd,	0xdeb8,0xb7,0xbd, 0xdebb,0xf3,0x39,
		0xdebc,0xb7,0xbd, 0xdebf,0xf1,0x39,	0xdec0,0x5f,0x7f, 0xdec3,0xa1,0x86,
		0xdec5,0x76,0xbd, 0xdec8,0xee,0x86,	0xdeca,0xe7,0xb7, 0xdecd,0xd6,0xbd,
		0xded0,0x59,0x7d, 0xded3,0x01,0x26,	0xded5,0x72,0xbd, 0xded8,0xd1,0xbd,
		0xdedb,0xd7,0xbd, 0xdede,0xd7,0xbd,	0xdee1,0x77,0xbd, 0xdee4,0x30,0xa6,
		0xdee8,0xca,0x8b, 0xdeea,0xea,0x81,	0xdeec,0xa2,0x23, 0xdeee,0xfb,0x86,
		0xdef0,0x32,0xa7, 0xdef4,0x77,0xbd,	0xdef7,0x60,0xbd, 0xdefa,0xc6,0xbd,
		0xdefd,0xd7,0xbd, 0xdf00,0x5c,0xbd,	0xdf03,0x96,0x39, 0xdf04,0x88,0x34,
		0xdf06,0xba,0x10, 0xdf0a,0x09,0xa6,	0xdf0e,0x52,0x10, 0xdf11,0x01,0xbe,
		0xdf14,0x26,0xe6, 0xdf16,0xe5,0x27,	0xdf18,0x24,0x3a, 0xdf19,0x19,0xa6,
		0xdf1b,0x68,0x8a, 0xdf1d,0x0e,0xa7,	0xdf1f,0x10,0x20, 0xdf21,0x9f,0x35,
		0xdf23,0xd1,0x7d, 0xdf26,0x88,0x26,	0xdf28,0x58,0x86, 0xdf2a,0x1b,0xb7,
		0xdf2d,0x58,0x86, 0xdf2f,0x0a,0xb7,	0xdf32,0xc5,0x7d, 0xdf35,0xf0,0x26,
		0xdf37,0x55,0x86, 0xdf39,0x1b,0xb7,	0xdf3c,0x58,0x86, 0xdf3e,0x1f,0xb7,
		0xdf41,0xf5,0x7d, 0xdf44,0xdc,0x26,	0xdf46,0x69,0x86, 0xdf48,0x1d,0xb7,
		0xdf4b,0x49,0x86, 0xdf4d,0x0c,0xb7,	0xdf50,0x81,0x7d, 0xdf53,0xc9,0x26,
		0xdf55,0x79,0x86, 0xdf57,0x4f,0xb7,	0xdf5a,0x49,0x86, 0xdf5c,0x09,0xb7,
		0xdf5f,0xb7,0x7d, 0xdf62,0xf3,0x26,	0xdf64,0x78,0x86, 0xdf66,0x5c,0xb7,
		0xdf69,0x58,0x86, 0xdf6b,0x1f,0xb7,	0xdf6e,0xb5,0x7d, 0xdf71,0xd8,0x26,
		0xdf73,0x6c,0x86, 0xdf75,0x5b,0xb7,	0xdf78,0x58,0x86, 0xdf7a,0x0e,0xb7,
		0xdf7d,0xb7,0x7d, 0xdf80,0x07,0x26,	0xdf82,0xa1,0x86, 0xdf84,0x23,0xb7,
		0xdf87,0xa1,0x86, 0xdf89,0x24,0xb7,	0xdf8c,0xc1,0x7d, 0xdf8f,0x61,0x26,
		0xdf91,0xb1,0x86, 0xdf93,0x27,0xb7,	0xdf96,0xa1,0x86, 0xdf98,0x21,0xb7,
		0xdf9b,0x56,0x39, 0xdf9c,0xee,0x8e,	0xdf9f,0x71,0xec, 0xdfa1,0x12,0x27,
		0xdfa3,0x5d,0x6d, 0xdfa7,0x04,0x26,	0xdfa9,0x6d,0x10, 0xdfad,0x96,0xed,
		0xdfaf,0x2c,0x86, 0xdfb1,0x32,0xa7,	0xdfb5,0x31,0xb6, 0xdfb8,0x0a,0x8b,
		0xdfba,0x24,0xb7, 0xdfbd,0x85,0x30,	0xdfc0,0xe6,0x8c, 0xdfc3,0x15,0x25,
		0xdfc5,0x42,0x39, 0xdfc6,0x7f,0xfd,	0xdfc9,0xb4,0xfc, 0xdfcc,0xf4,0x26,
		0xdfce,0xd6,0x39, 0xdfcf,0xa7,0xb7,	0xdfd2,0x5f,0x7d, 0xdfd5,0x11,0x26,
		0xdfd7,0x56,0x39, 0xdfd8,0xec,0x10,	0xdfdc,0x80,0xbe, 0xdfdf,0xa1,0xa6,
		0xdfe1,0xb6,0x85, 0xdfe3,0x13,0x27,	0xdfe5,0x7c,0xec, 0xdfe7,0xa7,0x80,
		0xdfe9,0x8c,0xc0, 0xdfeb,0xf2,0x6d,	0xdfef,0xf6,0x26, 0xdff1,0x79,0xfe,
		0xdff4,0x7e,0xed, 0xdff6,0xa2,0x86,	0xdff8,0x5b,0x5f, 0xdff9,0x16,0xed,
		0xdffb,0xac,0x86, 0xdffd,0xb2,0xa7,	0xe001,0x3f,0xb6, 0xe004,0x01,0x8b,
		0xe006,0x2f,0xb7, 0xe009,0xad,0xec,	0xe00b,0x58,0x80, 0xe00d,0xcb,0xc0,
		0xe00f,0xb5,0x6d, 0xe013,0xe1,0x26,	0xe015,0x29,0xfe, 0xe018,0xaa,0xed,
		0xe01a,0x49,0x86, 0xe01c,0xe1,0x5f,	0xe01d,0xba,0xed, 0xe01f,0x49,0x86,
		0xe021,0x3a,0xa7, 0xe025,0x39,0xb6,	0xe028,0x6a,0x8b, 0xe02a,0x1a,0xb7,
		0xe02d,0x07,0x31, 0xe02f,0x02,0x30,	0xe031,0xc5,0x10, 0xe035,0xc5,0x10,
		0xe039,0x27,0x39, 0xe1d2,0x63,0xbe,	0xe1d5,0x07,0x30, 0xe1d9,0x92,0x6f,
		0xe1db,0x53,0xbc, 0xe1de,0x20,0x22,	0xe1e0,0x47,0x39, 0xe1e1,0xfc,0xcc,
		0xe1e4,0xf0,0x8e, 0xe1e7,0x6c,0xed,	0xe1e9,0x5c,0x8c, 0xe1ec,0x16,0x25,
		0xe1ee,0x6e,0x8e, 0xe1f1,0x7e,0xed,	0xe1f3,0xe5,0x8c, 0xe1f6,0x01,0x25,
		0xe1f8,0x07,0x39, 0xe1f9,0x51,0xbe,	0xe1fc,0x25,0x30, 0xe200,0xa6,0xa6,
		0xe202,0x91,0x84, 0xe204,0xb0,0xa7,	0xe206,0x32,0x30, 0xe208,0x3c,0xbc,
		0xe20b,0x11,0x24, 0xe20d,0x22,0x39,	0xe20e,0x69,0x8e, 0xe211,0x98,0xcc,
		0xe214,0xe5,0xed, 0xe216,0x91,0x8c,	0xe219,0x11,0x25, 0xe21b,0x36,0x39,
		0xe21c,0x6e,0x8e, 0xe21f,0x52,0x10,	0xe223,0xe6,0xee, 0xe225,0xfc,0xec,
		0xe227,0xfd,0xed, 0xe229,0xba,0xef,	0xe22b,0x6f,0x8c, 0xe22e,0x05,0x25,
		0xe230,0x0f,0x39, 0xe231,0x98,0xce,	0xe234,0x76,0x6d, 0xe238,0x10,0x26,
		0xe23a,0x3a,0xbe, 0xe23d,0xba,0xef,	0xe23f,0x4c,0x86, 0xe241,0x92,0xa7,
		0xe245,0xa8,0xb6, 0xe248,0x7e,0x8b,	0xe24a,0x0f,0xb7, 0xe24d,0x05,0x33,
		0xe24f,0x56,0x11, 0xe253,0x04,0x23,	0xe255,0x8f,0xce, 0xe258,0xb6,0x6d,
		0xe25c,0x15,0x26, 0xe25e,0x2b,0xbe,	0xe261,0xda,0xef, 0xe263,0xa4,0x86,
		0xe265,0xba,0xa7, 0xe269,0x19,0xb6,	0xe26c,0x6a,0x8b, 0xe26e,0x0e,0xb7,
		0xe271,0x03,0x33, 0xe273,0x26,0x11,	0xe277,0x67,0x23, 0xe279,0x27,0x39,
		0xe27a,0xbb,0xfc, 0xe27d,0xb6,0x6d,	0xe281,0x11,0x26, 0xe283,0x63,0xbe,
		0xe286,0x7d,0xed, 0xe288,0x46,0x86,	0xe28a,0x4d,0xa7, 0xe28e,0x4b,0xb6,
		0xe291,0xf7,0x8b, 0xe293,0x27,0xb7,	0xe296,0x56,0x39, 0xe297,0x7b,0xfc,
		0xe29a,0x9d,0x6d, 0xe29e,0x49,0x26,	0xe2a0,0x71,0xbe, 0xe2a3,0x4c,0x6f,
		0xe2a5,0x7e,0xed, 0xe2a7,0x7d,0xed,	0xe2a9,0x10,0x86, 0xe2ab,0x0c,0xa7,
		0xe2af,0x5a,0xb6, 0xe2b2,0xe4,0x8b,	0xe2b4,0x33,0xb7, 0xe2b7,0x53,0x39,
		0xe2b8,0x25,0x10, 0xe2bc,0x39,0xbe,	0xe2bf,0x4c,0xa6, 0xe2c1,0xb0,0x85,
		0xe2c3,0x05,0x27, 0xe2c5,0x7a,0xec,	0xe2c7,0xb0,0x80, 0xe2c9,0xf5,0xc0,
		0xe2cb,0x9d,0x6d, 0xe2cf,0x49,0x26,	0xe2d1,0x7e,0xfe, 0xe2d4,0x7e,0xed,
		0xe2d6,0xa1,0x86, 0xe2d8,0x24,0x5f,	0xe2d9,0xcd,0xed, 0xe2db,0x61,0x86,
		0xe2dd,0xce,0xa7, 0xe2e1,0x31,0xb6,	0xe2e4,0xe2,0x8b, 0xe2e6,0x26,0xb7,
		0xe2e9,0x2f,0x31, 0xe2eb,0x39,0x30,	0xe2ed,0x45,0x10, 0xe2f1,0x16,0x25,
		0xe2f3,0x46,0x39, 0xe2f4,0x85,0x10,	0xe2f8,0xf9,0xbe, 0xe2fb,0x79,0xa6,
		0xe2fd,0xd6,0x85, 0xe2ff,0x9f,0x27,	0xe301,0xed,0xec, 0xe303,0x90,0x80,
		0xe305,0xc3,0xc0, 0xe307,0x7d,0x6d,	0xe30b,0x01,0x26, 0xe30d,0xbd,0xfe,
		0xe310,0xe9,0xed, 0xe312,0x81,0x86,	0xe314,0x59,0x5f, 0xe315,0xea,0xed,
		0xe317,0x81,0x86, 0xe319,0x0a,0xa7,	0xe31d,0x19,0xb6, 0xe320,0xca,0x8b,
		0xe322,0xf7,0xb7, 0xe325,0x90,0x86,	0xe327,0xb8,0xbd, 0xe32a,0x59,0x86,
		0xe32c,0x3f,0xbd, 0xe32f,0x00,0x31,	0xe331,0x25,0x30, 0xe333,0x10,0x10,
		0xe337,0x24,0x25, 0xe339,0x27,0x39,	0xe33a,0x04,0x34, 0xe33c,0x38,0xae,
		0xe340,0xfc,0xec, 0xe342,0x25,0x27,	0xe344,0x7f,0x6d, 0xe348,0x16,0x26,
		0xe34a,0x2b,0xbe, 0xe34d,0xb1,0x6f,	0xe34f,0xbd,0xed, 0xe351,0xfb,0xed,
		0xe353,0x81,0x86, 0xe355,0xf0,0xa7,	0xe359,0x08,0xb6, 0xe35c,0x7d,0x8b,
		0xe35e,0x0f,0xb7, 0xe361,0x7c,0xec,	0xe365,0xe2,0xc3, 0xe368,0xbe,0xed,
		0xe36c,0x17,0x35, 0xe36e,0x25,0x39,	0xe36f,0xab,0xec, 0xe373,0x90,0x83,
		0xe376,0xa9,0xed, 0xe37a,0x28,0xae,	0xe37e,0xab,0xec, 0xe380,0x4d,0x6d,
		0xe384,0x04,0x26, 0xe386,0x63,0xbe,	0xe389,0xec,0xed, 0xe38b,0x41,0x86,
		0xe38d,0x4c,0xa7, 0xe391,0x33,0xb6,	0xe394,0xe5,0x8b, 0xe396,0x27,0xb7,
		0xe399,0x3c,0x39, 0xe39a,0x0a,0x32,	0xe39c,0x7b,0xbd, 0xe39f,0xa7,0x5f,
		0xe3a0,0x31,0xb6, 0xe3a3,0xa6,0x81,	0xe3a5,0x85,0x10, 0xe3a9,0x05,0x10,
		0xe3ad,0x50,0x86, 0xe3af,0x5d,0xa7,	0xe3b1,0xb0,0x86, 0xe3b3,0x62,0xbd,
		0xe3b6,0x48,0x6d, 0xe3ba,0x09,0x26,	0xe3bc,0x79,0xbe, 0xe3bf,0xcd,0xe7,
		0xe3c1,0x90,0x10, 0xe3c4,0x03,0x31,	0xe3c6,0x92,0x10, 0xe3c9,0x4e,0x86,
		0xe3cb,0x45,0xa7, 0xe3cf,0x6b,0xb6,	0xe3d2,0xe4,0x8b, 0xe3d4,0x21,0xb7,
		0xe3d7,0x48,0x6a, 0xe3d9,0x18,0x26,	0xe3db,0x41,0x86, 0xe3dd,0xea,0xbd,
		0xe3e0,0xb0,0x86, 0xe3e2,0x32,0xa7,	0xe3e4,0x85,0x10, 0xe3e8,0x1a,0x27,
		0xe3ea,0xe2,0xcb, 0xe3ec,0xd4,0xc4,	0xe3ee,0x0b,0x20, 0xe3f0,0xb0,0x86,
		0xe3f2,0x67,0xbd, 0xe3f5,0x5e,0x6d,	0xe3f9,0x18,0x26, 0xe3fb,0x6e,0xbe,
		0xe3fe,0x48,0xe7, 0xe400,0x04,0x10,	0xe403,0x3e,0x31, 0xe407,0x1a,0x10,
		0xe40a,0x49,0x86, 0xe40c,0x0d,0xa7,	0xe410,0xba,0xb6, 0xe413,0x84,0x8b,
		0xe415,0xb8,0xb7, 0xe418,0xa6,0x6a,	0xe41a,0x01,0x26, 0xe41c,0x48,0x86,
		0xe41e,0x3f,0xbd, 0xe421,0x98,0x86,	0xe423,0xbf,0xa7, 0xe425,0x0d,0x10,
		0xe429,0x45,0x10, 0xe42d,0xea,0xcb,	0xe42f,0xde,0xc4, 0xe431,0x2c,0x20,
		0xe433,0x43,0x5f, 0xe434,0x0d,0x10,	0xe438,0x58,0x86, 0xe43a,0x0b,0xa7,
		0xe43c,0x58,0x86, 0xe43e,0x2c,0xbd,	0xe441,0xe5,0x6d, 0xe445,0x04,0x26,
		0xe447,0x8b,0xbe, 0xe44a,0x8d,0xe7,	0xe44c,0x51,0x10, 0xe44f,0x16,0x31,
		0xe451,0x34,0x10, 0xe454,0xa3,0x86,	0xe456,0x85,0xa7, 0xe45a,0x0b,0xb6,
		0xe45d,0x6d,0x8b, 0xe45f,0x0f,0xb7,	0xe462,0x60,0x6a, 0xe464,0x10,0x26,
		0xe466,0xb5,0x86, 0xe468,0x3f,0xbd,	0xe46b,0x59,0x86, 0xe46d,0x1a,0xa7,
		0xe46f,0x46,0x10, 0xe473,0x03,0x27,	0xe475,0xea,0xcb, 0xe477,0xe2,0xc4,
		0xe479,0x04,0x20, 0xe47b,0x49,0x86,	0xe47d,0x3f,0xbd, 0xe480,0x59,0x6d,
		0xe484,0x04,0x26, 0xe486,0x63,0xbe,	0xe489,0x8e,0xe7, 0xe48b,0x52,0x10,
		0xe48e,0x5e,0x31, 0xe492,0x92,0x10,	0xe495,0xa1,0x86, 0xe497,0x25,0xa7,
		0xe49b,0x0b,0xb6, 0xe49e,0x24,0x8b,	0xe4a0,0x33,0xb7, 0xe4a3,0x4d,0x6a,
		0xe4a5,0x10,0x26, 0xe4a7,0xb7,0x86,	0xe4a9,0x3f,0xbd, 0xe4ac,0x10,0x86,
		0xe4ae,0x19,0xa7, 0xe4b0,0x85,0x10,	0xe4b4,0x12,0x27, 0xe4b6,0xf9,0xcb,
		0xe4b8,0x94,0xc4, 0xe4ba,0x58,0x20,	0xe4bc,0x49,0x32, 0xe4be,0x7c,0x39,
		0xe4bf,0x5b,0x32, 0xe4c1,0x76,0xbe,	0xe4c4,0xaf,0xc6, 0xe4c6,0x42,0x3a,
		0xe4c7,0x65,0xaf, 0xe4c9,0x36,0xbe,	0xe4cc,0xe7,0xc6, 0xe4ce,0x0a,0x3a,
		0xe4cf,0xe9,0xe6, 0xe4d1,0xa8,0xc4,	0xe4d3,0x2d,0xe7, 0xe4d5,0x62,0xac,
		0xe4d7,0x04,0x23, 0xe4d9,0x39,0x35,	0xe4db,0x8f,0x7f, 0xe4de,0x8f,0x7f,
		0xe4e1,0x5b,0x7f, 0xe4e4,0x5b,0x7f,	0xe4e7,0x5d,0x7f, 0xe4ea,0x6f,0x39,
		0xe4eb,0x19,0x86, 0xe4ed,0xf9,0xf6,	0xe4f0,0xae,0xc1, 0xe4f2,0x17,0x26,
		0xe4f4,0xb4,0x84, 0xe4f6,0x14,0x20,	0xe4f8,0xe6,0xc1, 0xe4fa,0x3f,0x26,
		0xe4fc,0x74,0x84, 0xe4fe,0xe9,0xf6,	0xe501,0xf1,0xc1, 0xe503,0x01,0x26,
		0xe505,0xc6,0x84, 0xe507,0x70,0x20,	0xe509,0xd9,0xc1, 0xe50b,0x01,0x26,
		0xe50d,0x4e,0x84, 0xe50f,0x13,0x34,	0xe511,0xb8,0xfc, 0xe514,0x86,0xc3,
		0xe517,0xff,0xbd, 0xe51a,0x17,0x35,	0xe51c,0x38,0xbd, 0xe51f,0xd8,0xc0,
		0xe521,0xb6,0x85, 0xe523,0x12,0x27,	0xe525,0xff,0xbd, 0xe528,0xea,0xcb,
		0xe52a,0x4c,0x85, 0xe52c,0x12,0x27,	0xe52e,0x3e,0xbd, 0xe531,0x8a,0xcb,
		0xe533,0xc2,0x85, 0xe535,0x72,0x27,	0xe537,0xec,0xbd, 0xe53a,0xed,0xcb,
		0xe53c,0x5e,0x85, 0xe53e,0x07,0x27,	0xe540,0xfa,0xbd, 0xe543,0x8c,0xcb,
		0xe545,0x9f,0x85, 0xe547,0x2d,0x27,	0xe549,0x2d,0xbd, 0xe54c,0xeb,0xcb,
		0xe54e,0x5d,0x85, 0xe550,0x75,0x27,	0xe552,0xff,0xbd, 0xe555,0x9e,0xcb,
		0xe557,0xd5,0x85, 0xe559,0x15,0x27,	0xe55b,0x3f,0xbd, 0xe55e,0xec,0xcb,
		0xe560,0xd6,0x85, 0xe562,0x65,0x27,	0xe564,0x7f,0xbd, 0xe567,0x9a,0x86,
		0xe569,0x1b,0xb7, 0xe56c,0x27,0x39,	0xe56d,0x15,0x34, 0xe56f,0x3e,0xbe,
		0xe572,0x6e,0x3a, 0xe573,0x92,0xa6,	0xe575,0xd6,0x85, 0xe577,0x66,0x27,
		0xe579,0x5c,0x84, 0xe57b,0x46,0x12,	0xe57c,0x41,0x12, 0xe57d,0x1a,0xa7,
		0xe57f,0x6c,0x8e, 0xe582,0x42,0x3a,	0xe583,0x42,0x3a, 0xe584,0x6c,0xec,
		0xe586,0xb0,0x80, 0xe588,0xe7,0xc0,	0xe58a,0x07,0x1f, 0xe58c,0x85,0x6d,
		0xe590,0x05,0x26, 0xe592,0x6b,0xfe,	0xe595,0x61,0xaf, 0xe597,0xa1,0x86,
		0xe599,0x33,0x5f, 0xe59a,0x7d,0xed,	0xe59c,0xc3,0x86, 0xe59e,0xcd,0xa7,
		0xe5a2,0x23,0xb6, 0xe5a5,0xe2,0x8b,	0xe5a7,0x35,0xb7, 0xe5aa,0x3e,0x20,
		0xe5ac,0x96,0x6d, 0xe5b0,0x10,0x26,	0xe5b2,0x7b,0xfe, 0xe5b5,0x72,0xaf,
		0xe5b7,0xa3,0x86, 0xe5b9,0x33,0x5f,	0xe5ba,0x7c,0xed, 0xe5bc,0xd0,0x86,
		0xe5be,0xcf,0xa7, 0xe5c2,0x23,0xb6,	0xe5c5,0xf1,0x8b, 0xe5c7,0x27,0xb7,
		0xe5ca,0xcb,0xb6, 0xe5cd,0x31,0x8b,	0xe5cf,0x0f,0xb7, 0xe5d2,0x17,0x35,
		0xe5d4,0x5a,0x6d, 0xe5d8,0x9c,0x26,	0xe5da,0xeb,0xbe, 0xe5dd,0x69,0xfc,
		0xe5e0,0x7e,0xed, 0xe5e2,0x60,0xbd,	0xe5e5,0x7d,0xfc, 0xe5e8,0x7e,0xed,
		0xe5ea,0x2d,0x6f, 0xe5ec,0xd8,0x86,	0xe5ee,0xd3,0xa7, 0xe5f2,0x36,0xb6,
		0xe5f5,0xe2,0x8b, 0xe5f7,0x22,0xb7,	0xe5fa,0xad,0x39, 0xe5fb,0x27,0x10,
		0xe5ff,0x6d,0xec, 0xe601,0x6e,0x27,	0xe603,0x35,0x6d, 0xe607,0x69,0x26,
		0xe609,0x3f,0xbe, 0xe60c,0xaf,0xed,	0xe60e,0xbd,0xed, 0xe610,0xea,0xa6,
		0xe612,0xed,0xa7, 0xe614,0xe8,0xa6,	0xe616,0x89,0xc6, 0xe618,0x49,0x85,
		0xe61a,0x01,0x26, 0xe61c,0xab,0xed,	0xe61e,0x49,0x86, 0xe620,0xb2,0xa7,
		0xe624,0xf9,0xb6, 0xe627,0xc5,0x8b,	0xe629,0x1b,0xb7, 0xe62c,0x04,0x20,
		0xe62e,0x5a,0x84, 0xe630,0xb6,0xed,	0xe632,0xc9,0x86, 0xe634,0xfa,0xa7,
		0xe636,0xcb,0x86, 0xe638,0x1a,0xa7,	0xe63c,0x19,0xb6, 0xe63f,0x7d,0x8b,
		0xe641,0x7f,0xb7, 0xe644,0x18,0x31,	0xe647,0x3a,0x10, 0xe64b,0x15,0x25,
		0xe64d,0x33,0x39, 0xe64e,0x13,0x34,	0xe650,0x63,0x48, 0xe651,0xa2,0x8e,
		0xe654,0xeb,0xc6, 0xe656,0x81,0xae,	0xe658,0x2d,0xae, 0xe65a,0x52,0x10,
		0xe65e,0xb5,0x6d, 0xe660,0x78,0x26,	0xe662,0xf8,0xa6, 0xe664,0x9a,0xa7,
		0xe666,0x1d,0x26, 0xe668,0xb6,0x6d,	0xe66c,0x10,0x26, 0xe66e,0x39,0xbe,
		0xe671,0xb8,0x86, 0xe673,0x8b,0xa7,	0xe675,0x21,0x1a, 0xe677,0xab,0x86,
		0xe679,0x1a,0xa7, 0xe67d,0x19,0xb6,	0xe680,0xf2,0x8b, 0xe682,0x27,0xb7,
		0xe685,0xd7,0x1c, 0xe687,0x17,0x35,	0xe689,0x61,0x48, 0xe68a,0x61,0x8e,
		0xe68d,0x05,0xc6, 0xe68f,0xa9,0xae,	0xe691,0x63,0xae, 0xe693,0x92,0x10,
		0xe697,0x5d,0x6d, 0xe699,0xcb,0x26,	0xe69b,0x89,0xa6, 0xe69d,0x88,0xa7,
		0xe69f,0xc9,0x26, 0xe6a1,0x5e,0x6d,	0xe6a5,0x10,0x26, 0xe6a7,0x73,0xbe,
		0xe6aa,0x54,0x86, 0xe6ac,0x9a,0xa7,	0xe6ae,0xe7,0x1a, 0xe6b0,0xb0,0x86,
		0xe6b2,0x23,0xa7, 0xe6b6,0x27,0xb6,	0xe6b9,0xa2,0x8b, 0xe6bb,0x8b,0xb7,
		0xe6be,0xe6,0x1c, 0xe6c0,0x41,0x39,	0xe6c1,0xa6,0x86, 0xe6c3,0x77,0xbd,
		0xe6c6,0x77,0xbd, 0xe6c9,0xf1,0x39,	0xe6ca,0x81,0x86, 0xe6cc,0x1c,0xbd,
		0xe6cf,0x21,0x86, 0xe6d1,0x71,0xbd,	0xe6d4,0x70,0xbd, 0xe6d7,0x56,0x39,
		0xe6d8,0x25,0x86, 0xe6da,0x1f,0xbd,	0xe6dd,0x1b,0xbd, 0xe6e0,0x47,0x39,
		0xe6e1,0xb0,0x86, 0xe6e3,0x76,0xbd,	0xe6e6,0x72,0xbd, 0xe6e9,0xe7,0x39,
		0xe6ea,0x90,0x86, 0xe6ec,0x1f,0xbd,	0xe6ef,0x33,0x86, 0xe6f1,0x77,0xbd,
		0xe6f4,0x77,0xbd, 0xe6f7,0x54,0x39,	0xe6f8,0x30,0x86, 0xe6fa,0x18,0xbd,
		0xe6fd,0x1f,0xbd, 0xe700,0x16,0x39,	0xe701,0xbe,0x86, 0xe703,0x97,0xbd,
		0xe706,0x17,0xbd, 0xe709,0x21,0x39,	0xe70a,0x49,0x86, 0xe70c,0x3d,0xbd,
		0xe70f,0x36,0x39, 0xe710,0xe5,0x86,	0xe712,0xdf,0xbd, 0xe715,0x5f,0x39,
		0xe716,0xe1,0x86, 0xe718,0x3e,0xbd,	0xe71b,0x36,0x39, 0xe71c,0x4b,0x86,
		0xe71e,0x3f,0xbd, 0xe721,0xb8,0x86,	0xe723,0x97,0xbd, 0xe726,0x86,0x39,
		0xe727,0xf3,0x86, 0xe729,0x3f,0xbd,	0xe72c,0x27,0x39, 0xe72d,0x58,0x86,
		0xe72f,0x3c,0xbd, 0xe732,0x48,0x39,	0xe733,0xe1,0x86, 0xe735,0xdf,0xbd,
		0xe738,0x58,0x86, 0xe73a,0x38,0xbd,	0xe73d,0x27,0x39, 0xe73e,0x4d,0x86,
		0xe740,0xda,0xbd, 0xe743,0xe1,0x86,	0xe745,0xf5,0xbd, 0xe748,0x36,0x39,
		0xe749,0x5e,0x86, 0xe74b,0x3f,0xbd,	0xe74e,0x36,0x39, 0xe74f,0x49,0x86,
		0xe751,0xdd,0xbd, 0xe754,0x58,0x39,	0xe755,0x09,0x86, 0xe757,0x37,0xbd,
		0xe75a,0x36,0x39, 0xe75b,0x36,0x39,	0xe75c,0x73,0x1a, 0xe75e,0x3f,0xbd,
		0xe761,0x7d,0x1c, 0xe763,0xf0,0x86,	0xe765,0xf7,0xbd, 0xe768,0xfc,0xcc,
		0xe76b,0x15,0x34, 0xe76d,0x78,0x8e,	0xe770,0x2d,0x10, 0xe774,0x1e,0x6d,
		0xe776,0x55,0x26, 0xe778,0xbc,0xec,	0xe77a,0xba,0xed, 0xe77c,0x18,0xa6,
		0xe77e,0x73,0x1f, 0xe780,0x8b,0x44,	0xe781,0x9c,0x44, 0xe782,0x99,0x44,
		0xe783,0x99,0x44, 0xe784,0xac,0xc4,	0xe786,0xac,0xc3, 0xe789,0x9d,0xed,
		0xe78b,0x29,0xa6, 0xe78d,0x95,0x1f,	0xe78f,0x31,0x44, 0xe790,0x8f,0x44,
		0xe791,0x88,0x44, 0xe792,0x99,0x44,	0xe793,0xb9,0xc4, 0xe795,0xad,0xc3,
		0xe798,0x4e,0xed, 0xe79a,0xe9,0xa6,	0xe79c,0x92,0x1f, 0xe79e,0x31,0x44,
		0xe79f,0x31,0x44, 0xe7a0,0x9c,0x44,	0xe7a1,0x9c,0x44, 0xe7a2,0xae,0xc4,
		0xe7a4,0xaa,0xc3, 0xe7a7,0x78,0xed,	0xe7a9,0x38,0xa6, 0xe7ab,0x56,0x1f,
		0xe7ad,0x34,0x44, 0xe7ae,0xf8,0x44,	0xe7af,0x32,0x44, 0xe7b0,0x9c,0x44,
		0xe7b1,0xbc,0xc4, 0xe7b3,0xbb,0xc3,	0xe7b6,0x6e,0xed, 0xe7b8,0x54,0xcc,
		0xe7bb,0xfa,0xa1, 0xe7bd,0xb8,0x26,	0xe7bf,0x69,0xe7, 0xe7c1,0x31,0xa1,
		0xe7c3,0x01,0x26, 0xe7c5,0x2f,0xe7,	0xe7c7,0x34,0xa1, 0xe7c9,0xbe,0x26,
		0xe7cb,0x6d,0xe7, 0xe7cd,0x89,0xa1,	0xe7cf,0xc9,0x26, 0xe7d1,0x2f,0xe7,
		0xe7d3,0x34,0xa1, 0xe7d5,0x01,0x26,	0xe7d7,0x2d,0xe7, 0xe7d9,0xe9,0xa1,
		0xe7db,0xa9,0x26, 0xe7dd,0x80,0xe7,	0xe7df,0x14,0xa1, 0xe7e1,0x10,0x26,
		0xe7e3,0x3a,0xe7, 0xe7e5,0x3a,0xe7,	0xe7e7,0x3b,0xe7, 0xe7e9,0x7a,0xe7,
		0xe7eb,0x59,0xec, 0xe7ed,0x3e,0xed,	0xe7ef,0x99,0xa6, 0xe7f1,0xb8,0xc6,
		0xe7f3,0x7a,0xed, 0xe7f5,0x5e,0x6d,	0xe7f9,0xb8,0x26, 0xe7fb,0x5d,0xfe,
		0xe7fe,0xf5,0x86, 0xe800,0xa7,0xa7,	0xe802,0x0a,0x1a, 0xe804,0x84,0x86,
		0xe806,0xa5,0xa7, 0xe80a,0x0b,0xb6,	0xe80d,0x68,0x8b, 0xe80f,0x0f,0xb7,
		0xe812,0x1b,0x1c, 0xe814,0xec,0xec,	0xe816,0x8c,0x8b, 0xe818,0xad,0xed,
		0xe81a,0x12,0x30, 0xe81c,0x6c,0x8c,	0xe81f,0x52,0x10, 0xe823,0x32,0x35,
		0xe825,0x21,0x32, 0xe827,0x97,0x86,	0xe829,0x1b,0xb7, 0xe82c,0x78,0x8e,
		0xe82f,0xbe,0xec, 0xe831,0xff,0xfd,	0xe834,0x90,0x86, 0xe836,0xb7,0xb7,
		0xe839,0x18,0xa6, 0xe83b,0x16,0x27,	0xe83d,0x1a,0xa7, 0xe83f,0xaf,0xec,
		0xe841,0xfd,0xfd, 0xe844,0xbd,0xbd,	0xe847,0x81,0x86, 0xe849,0x3d,0xbd,
		0xe84c,0xb2,0x7a, 0xe84f,0x01,0x26,	0xe851,0x93,0x86, 0xe853,0xa7,0xb7,
		0xe856,0x68,0x6a, 0xe858,0x12,0x26,	0xe85a,0x43,0x16, 0xe85d,0x05,0x32,
		0xe85f,0x36,0x39, 0xe884,0x10,0x32,	0xe886,0x77,0xbd, 0xe889,0x67,0x8e,
		0xe88c,0xee,0xec, 0xe88e,0x6d,0xaf,	0xe890,0x7d,0xfd, 0xe893,0xa1,0x86,
		0xe895,0x20,0xb7, 0xe898,0x46,0x86,	0xe89a,0x12,0x10, 0xe89e,0x12,0x10,
		0xe8a1,0x85,0x10, 0xe8a5,0xf0,0x8e,	0xe8a8,0x7a,0xaf, 0xe8aa,0x7e,0xbd,
		0xe8ad,0x70,0x8e, 0xe8b0,0xb0,0x86,	0xe8b2,0x33,0xa7, 0xe8b4,0x32,0xa7,
		0xe8b7,0x24,0xa7, 0xe8ba,0x5b,0xa7,	0xe8bd,0x98,0xae, 0xe8bf,0xae,0xa6,
		0xe8c1,0x27,0xa7, 0xe8c3,0x79,0xec,	0xe8c5,0x14,0x27, 0xe8c7,0x5d,0x6d,
		0xe8c9,0x0e,0x26, 0xe8cb,0x1b,0x34,	0xe8cd,0x6e,0xbd, 0xe8d0,0x01,0x35,
		0xe8d2,0x65,0xaf, 0xe8d4,0x16,0x20,	0xe8d6,0x7f,0xfd, 0xe8d9,0x7a,0xaf,
		0xe8db,0x7f,0xbd, 0xe8de,0x7f,0xbd,	0xe8e1,0xb0,0x86, 0xe8e3,0x76,0xbd,
		0xe8e6,0x4d,0x7a, 0xe8e9,0x18,0x26,	0xe8eb,0x51,0x86, 0xe8ed,0x5b,0xb7,
		0xe8f0,0x48,0x6a, 0xe8f2,0x00,0x26,	0xe8f4,0x91,0x16, 0xe8f7,0x12,0x32,
		0xe8f9,0x2f,0x39, 0xe999,0xc6,0x32,	0xe99b,0xa9,0x8e, 0xe99e,0xf9,0xec,
		0xe9a0,0x72,0xaf, 0xe9a2,0x7e,0xfd,	0xe9a5,0xb0,0x86, 0xe9a7,0x31,0xb7,
		0xe9aa,0x11,0x86, 0xe9ac,0x45,0x10,	0xe9b0,0x85,0x10, 0xe9b3,0x95,0x10,
		0xe9b7,0xf1,0x8e, 0xe9ba,0xd3,0xaf,	0xe9bc,0x7f,0xbd, 0xe9bf,0x71,0x8e,
		0xe9c2,0xa1,0x86, 0xe9c4,0x27,0xa7,	0xe9c6,0x25,0xa7, 0xe9c9,0x3f,0xae,
		0xe9cb,0x09,0xa6, 0xe9cd,0x4f,0xa7,	0xe9cf,0xf9,0xec, 0xe9d1,0x12,0x27,
		0xe9d3,0x7f,0xfd, 0xe9d6,0x65,0xaf,	0xe9d8,0x29,0xbd, 0xe9db,0x3f,0xbd,
		0xe9de,0x01,0x86, 0xe9e0,0x77,0xbd,	0xe9e3,0x48,0x7a, 0xe9e6,0x11,0x26,
		0xe9e8,0x10,0x86, 0xe9ea,0x1e,0xb7,	0xe9ed,0xa8,0x6a, 0xe9ef,0x19,0x26,
		0xe9f1,0x91,0x16, 0xe9f4,0x01,0x32,	0xe9f6,0x55,0x39, 0xea7b,0x11,0x32,
		0xea7d,0x78,0x8e, 0xea80,0x6c,0xec,	0xea82,0x65,0xaf, 0xea84,0x7d,0xfd,
		0xea87,0xa1,0x86, 0xea89,0x5f,0xb7,	0xea8f,0x3b,0x4b, 0xea90,0x90,0x10,
		0xea93,0x92,0x10, 0xea97,0xe1,0x8e,	0xea9a,0x41,0x86, 0xea9c,0x08,0xa7,
		0xea9e,0x0d,0xa7, 0xeaa1,0x70,0xae,	0xeaa3,0x31,0xa6, 0xeaa5,0x32,0xa7,
		0xeaa7,0x78,0xec, 0xeaa9,0x1a,0x27,	0xeaab,0xfe,0xfd, 0xeaae,0xba,0x6d,
		0xeab0,0x12,0x27, 0xeab2,0x5a,0x6d,	0xeab6,0x07,0x26, 0xeab8,0xf9,0xfe,
		0xeabb,0xff,0xfc, 0xeabe,0xac,0xed,	0xeac0,0xa6,0x86, 0xeac2,0xcf,0x5f,
		0xeac3,0x7d,0xed, 0xeac5,0xd2,0x1a,	0xeac7,0xa1,0x86, 0xeac9,0x4f,0xa7,
		0xeacd,0x5e,0xb6, 0xead0,0xe3,0x8b,	0xead2,0x27,0xb7, 0xead5,0xd3,0x1c,
		0xead7,0xa1,0x86, 0xead9,0x78,0xbd,	0xeadc,0x69,0xaf, 0xeade,0x7f,0xbd,
		0xeae1,0x77,0xbd, 0xeae4,0xb0,0x86,	0xeae6,0x76,0xbd, 0xeae9,0xa9,0x7a,
		0xeaec,0x18,0x26, 0xeaee,0x51,0x86,	0xeaf0,0x33,0xb7, 0xeaf3,0x5d,0x6a,
		0xeaf5,0x10,0x26, 0xeaf7,0x83,0x16,	0xeafa,0x0d,0x32, 0xeafc,0x2f,0x39,
		0xebbb,0x5d,0x32, 0xebbd,0x70,0x8e,	0xebc0,0x6c,0xec, 0xebc2,0x65,0xaf,
		0xebc4,0x7d,0xfd, 0xebc7,0xa1,0x86,	0xebc9,0x5f,0xb7, 0xebcc,0x46,0x86,
		0xebce,0x12,0x10, 0xebd2,0xe1,0x8e,	0xebd5,0x75,0xaf, 0xebd7,0x92,0x10,
		0xebdb,0x52,0x10, 0xebde,0x12,0x10,	0xebe2,0x76,0xbd, 0xebe5,0xf0,0x8e,
		0xebe8,0x50,0x86, 0xebea,0x5b,0xa7,	0xebec,0x5a,0xa7, 0xebef,0x5b,0xa7,
		0xebf2,0x33,0xa7, 0xebf5,0x70,0xae,	0xebf7,0x32,0xa6, 0xebf9,0x1a,0xa7,
		0xebfb,0xac,0xec, 0xebfd,0x5a,0x27,	0xebff,0xbd,0xfd, 0xec02,0x0d,0xaf,
		0xec06,0x08,0x18, 0xec07,0xbf,0xbd,	0xec0a,0x49,0x86, 0xec0c,0x3d,0xbd,
		0xec0f,0xa2,0x7a, 0xec12,0x21,0x26,	0xec14,0x86,0x86, 0xec16,0xa7,0xb7,
		0xec19,0xa2,0x6a, 0xec1b,0x01,0x26,	0xec1d,0x56,0x16, 0xec20,0x81,0x32,
		0xec22,0x8e,0x39, 0xed4c,0x13,0x32,	0xed4e,0xb5,0x6d, 0xed52,0x81,0x26,
		0xed54,0x05,0xbe, 0xed57,0x11,0xcc,	0xed5a,0xbd,0xed, 0xed5c,0x61,0x1a,
		0xed5e,0x49,0x86, 0xed60,0xb2,0xa7,	0xed64,0x11,0xb6, 0xed67,0x3b,0x8b,
		0xed69,0x1b,0xb7, 0xed6c,0x75,0x1c,	0xed6e,0x7d,0x8e, 0xed71,0x5c,0xec,
		0xed73,0x1e,0xaf, 0xed75,0x37,0xfd,	0xed78,0x58,0x86, 0xed7a,0x1c,0xb7,
		0xed7d,0x58,0x86, 0xed7f,0x57,0x10,	0xed83,0xe1,0x8e, 0xed86,0x65,0xaf,
		0xed88,0x3e,0x10, 0xed8f,0x12,0x10,	0xed93,0x77,0xbd, 0xed95,0x08,0x18,
		0xed96,0xe1,0x8e, 0xed99,0x46,0x86,	0xed9b,0x4d,0xa7, 0xed9d,0x4f,0xa7,
		0xeda0,0x32,0xa7, 0xeda3,0x33,0xa7,	0xeda6,0x71,0xae, 0xeda8,0x70,0xa6,
		0xedaa,0x61,0xa7, 0xedac,0xfc,0xec,	0xedae,0x04,0x10, 0xedb2,0x7e,0xfd,
		0xedb5,0x5e,0x6d, 0xedb7,0x84,0x10,	0xedbb,0x51,0x86, 0xedbd,0x7f,0xbd,
		0xedc0,0x90,0x10, 0xedc4,0x4c,0x6d,	0xedc8,0x0e,0x26, 0xedca,0xeb,0xfe,
		0xedcd,0x3c,0x10, 0xedd0,0xb4,0x86,	0xedd2,0xcf,0x5f, 0xedd3,0x7d,0xed,
		0xedd5,0xc4,0x1a, 0xedd7,0xa1,0x86,	0xedd9,0x76,0xa7, 0xeddd,0x80,0xb6,
		0xede0,0xe2,0x8b, 0xede2,0x37,0xb7,	0xede5,0xd5,0x1c, 0xede7,0x94,0x10,
		0xedeb,0xba,0x6d, 0xedef,0x21,0x26,	0xedf1,0x79,0xfe, 0xedf4,0x85,0x10,
		0xedf7,0xa2,0x86, 0xedf9,0x9b,0x5f,	0xedfa,0xd1,0xed, 0xedfc,0x49,0x1a,
		0xedfe,0x3e,0x86, 0xee00,0x12,0xa7,	0xee04,0xb6,0xb6, 0xee07,0x8c,0x8b,
		0xee09,0x08,0xb7, 0xee0c,0x74,0x1c,	0xee0e,0x52,0x10, 0xee12,0x7d,0x6d,
		0xee16,0x21,0x26, 0xee18,0xbe,0xfe,	0xee1b,0x52,0x10, 0xee1e,0x49,0x86,
		0xee20,0xfb,0x5f, 0xee21,0x5e,0xed,	0xee23,0xba,0x1a, 0xee25,0x90,0x86,
		0xee27,0xb3,0xa7, 0xee2b,0x0a,0xb6,	0xee2e,0x6b,0x8b, 0xee30,0xb3,0xb7,
		0xee33,0x1c,0x1c, 0xee35,0xba,0xaf,	0xee37,0xbe,0xbd, 0xee3a,0x3e,0xbd,
		0xee3d,0x58,0x86, 0xee3f,0x3e,0xbd,	0xee42,0x6a,0x7a, 0xee45,0x92,0x26,
		0xee47,0x21,0x86, 0xee49,0x1f,0xb7,	0xee4c,0xa2,0x6a, 0xee4e,0x01,0x26,
		0xee50,0xb3,0x16, 0xee53,0xea,0x32,	0xee55,0x86,0x39, 0xeedc,0x64,0x32,
		0xeede,0x09,0x8e, 0xeee1,0x7c,0xec,	0xeee3,0x73,0xaf, 0xeee5,0x7f,0xfd,
		0xeee8,0x50,0x86, 0xeeea,0x5f,0xb7,	0xeeed,0x78,0x86, 0xeeef,0x3d,0x10,
		0xeef3,0x93,0x10, 0xeef6,0x97,0x10,	0xeefa,0x4d,0x8e, 0xeefd,0x17,0xbd,
		0xef00,0x8e,0x8e, 0xef03,0x81,0x86,	0xef05,0xa7,0xa7, 0xef07,0xa5,0xa7,
		0xef0a,0x0d,0xa7, 0xef0d,0x0f,0xa7,	0xef10,0xa7,0xa7, 0xef13,0xa9,0xae,
		0xef15,0xa6,0xa6, 0xef17,0xa5,0xa7,	0xef19,0xac,0xec, 0xef1b,0x05,0x27,
		0xef1d,0xbd,0xfd, 0xef20,0x7e,0x6d,	0xef22,0x33,0x27, 0xef24,0xb0,0xa6,
		0xef26,0x82,0x81, 0xef28,0x10,0x26,	0xef2a,0x59,0x86, 0xef2c,0x1b,0xb7,
		0xef2f,0xb7,0x7f, 0xef32,0x7f,0x7f,	0xef35,0x24,0x20, 0xef37,0x7a,0x6d,
		0xef3b,0x11,0x26, 0xef3d,0xb9,0xfe,	0xef40,0xfc,0xfc, 0xef43,0xfd,0xed,
		0xef45,0x25,0x86, 0xef47,0xef,0x5f,	0xef48,0xad,0xed, 0xef4a,0x62,0x1a,
		0xef4c,0x5c,0x86, 0xef4e,0x0d,0xa7,	0xef52,0x63,0xb6, 0xef55,0x29,0x8b,
		0xef57,0x07,0xb7, 0xef5a,0x73,0x1c,	0xef5c,0x5b,0x86, 0xef5e,0x3f,0xbd,
		0xef61,0xba,0xaf, 0xef63,0xbe,0xbd,	0xef66,0x15,0xbd, 0xef69,0x58,0x86,
		0xef6b,0x3e,0xbd, 0xef6e,0xa6,0x7a,	0xef71,0x90,0x26, 0xef73,0x27,0x86,
		0xef75,0x13,0xb7, 0xef78,0xa0,0x6a,	0xef7a,0x11,0x26, 0xef7c,0x51,0x16,
		0xef7f,0x07,0x32, 0xef81,0x51,0x39,	0xf108,0x14,0x34, 0xf10a,0xbb,0xfc,
		0xf10d,0x2f,0xbb, 0xf110,0x0e,0xfb,	0xf113,0x9d,0x6d, 0xf117,0xc9,0x26,
		0xf119,0x28,0xbe, 0xf11c,0xa9,0xfe,	0xf11f,0xad,0xef, 0xf121,0x1e,0xed,
		0xf123,0x1d,0xfd, 0xf126,0x52,0xb6,	0xf129,0x4e,0x81, 0xf12b,0x10,0x27,
		0xf12d,0x6a,0x8b, 0xf12f,0x49,0x84,	0xf131,0x53,0xb7, 0xf134,0x52,0xa7,
		0xf136,0xfa,0x1a, 0xf138,0x58,0x86,	0xf13a,0x0e,0xa7, 0xf13e,0x0a,0xb6,
		0xf141,0x61,0x8b, 0xf143,0x4f,0xb7,	0xf146,0x1b,0x1c, 0xf148,0x14,0x35,
		0xf14a,0x13,0x34, 0xf14c,0x12,0x32,	0xf14e,0x69,0x8e, 0xf151,0xec,0xec,
		0xf153,0x12,0x10, 0xf157,0xfd,0xed,	0xf159,0x8e,0xe6, 0xf15b,0x09,0xa6,
		0xf15d,0xcd,0xc5, 0xf15f,0x01,0x26,	0xf161,0x5a,0xa7, 0xf163,0x61,0x8b,
		0xf165,0x7c,0x84, 0xf167,0xdd,0x20,	0xf169,0xf6,0x4d, 0xf16a,0x17,0x27,
		0xf16c,0x5c,0x84, 0xf16e,0x6b,0x8b,	0xf170,0xb2,0xa7, 0xf172,0x69,0x6a,
		0xf174,0x30,0x26, 0xf176,0x91,0x86,	0xf178,0x1a,0xa7, 0xf17a,0xa1,0x6a,
		0xf17c,0x10,0x26, 0xf17e,0x59,0x86,	0xf180,0x75,0xbd, 0xf183,0x79,0xec,
		0xf185,0x12,0x26, 0xf187,0x5d,0x6d,	0xf18b,0xa1,0x26, 0xf18d,0xac,0x10,
		0xf191,0x7a,0xec, 0xf193,0x7d,0xed,	0xf195,0xd2,0x1a, 0xf197,0x5d,0x6d,
		0xf19b,0xa1,0x26, 0xf19d,0xdc,0x1c,	0xf19f,0xeb,0x16, 0xf1a2,0x31,0xa6,
		0xf1a4,0xa6,0x81, 0xf1a6,0x14,0x27,	0xf1a8,0xfc,0xec, 0xf1aa,0x6b,0xab,
		0xf1ac,0x42,0xeb, 0xf1ae,0x55,0xed,	0xf1b0,0x5e,0x6d, 0xf1b4,0x10,0x26,
		0xf1b6,0x93,0x10, 0xf1ba,0x45,0xec,	0xf1bc,0x56,0xed, 0xf1be,0x47,0xec,
		0xf1c0,0x6c,0xed, 0xf1c2,0x21,0xa6,	0xf1c4,0x27,0xa7, 0xf1c6,0x29,0xe6,
		0xf1c8,0xa7,0xa6, 0xf1ca,0xbd,0x85,	0xf1cc,0x0e,0x26, 0xf1ce,0xfd,0xed,
		0xf1d0,0xd2,0x1a, 0xf1d2,0xa1,0x86,	0xf1d4,0x27,0xa7, 0xf1d8,0x5e,0xb6,
		0xf1db,0x64,0x8b, 0xf1dd,0x5f,0xb7,	0xf1e0,0xd5,0x1c, 0xf1e2,0x04,0x20,
		0xf1e4,0xb4,0x84, 0xf1e6,0x7a,0xed,	0xf1e8,0xc9,0x1a, 0xf1ea,0xbd,0x86,
		0xf1ec,0x5a,0xa7, 0xf1f0,0x31,0xb6,	0xf1f3,0xe3,0x8b, 0xf1f5,0x33,0xb7,
		0xf1f8,0x35,0x1c, 0xf1fa,0x0c,0x30,	0xf1fd,0x74,0x8c, 0xf200,0x10,0x10,
		0xf204,0xd9,0x32, 0xf206,0xdf,0x35,	0xf208,0x14,0x34, 0xf20a,0x52,0x10,
		0xf20e,0xb9,0xec, 0xf210,0x00,0xed,	0xf212,0x11,0xec, 0xf214,0x02,0xed,
		0xf216,0x11,0xec, 0xf218,0x0a,0xa7,	0xf21a,0x49,0x86, 0xf21c,0xaa,0xed,
		0xf21e,0xb5,0x6d, 0xf220,0x30,0x26,	0xf222,0xf8,0xec, 0xf224,0xd8,0x26,
		0xf226,0x16,0xec, 0xf228,0xb6,0x6d,	0xf22c,0x10,0x26, 0xf22e,0xbb,0xfe,
		0xf231,0x16,0xed, 0xf233,0x6b,0x86,	0xf235,0xb3,0x5f, 0xf236,0x04,0xed,
		0xf238,0x61,0x1a, 0xf23a,0x4d,0x86,	0xf23c,0x1a,0xa7, 0xf240,0x4f,0xb6,
		0xf243,0x64,0x8b, 0xf245,0xb7,0xb7,	0xf248,0x65,0x1c, 0xf24a,0x10,0x20,
		0xf24c,0xac,0xec, 0xf24e,0xb5,0x6d,	0xf252,0x21,0x26, 0xf254,0xfe,0xfe,
		0xf257,0xfd,0xed, 0xf259,0x5d,0x86,	0xf25b,0xe7,0x5f, 0xf25c,0xad,0xed,
		0xf25e,0x62,0x1a, 0xf260,0x78,0x86,	0xf262,0x5e,0xa7, 0xf266,0xb5,0xb6,
		0xf269,0x6a,0x8b, 0xf26b,0x1c,0xb7,	0xf26e,0x74,0x1c, 0xf270,0xfc,0xec,
		0xf272,0xfa,0xed, 0xf274,0x27,0x31,	0xf276,0x24,0x30, 0xf279,0xa0,0x6a,
		0xf27b,0x16,0x26, 0xf27d,0x45,0x10,	0xf281,0x15,0x35, 0xf283,0x13,0x34,
		0xf285,0x32,0xb6, 0xf288,0x41,0x81,	0xf28a,0x09,0x26, 0xf28c,0x46,0x6d,
		0xf290,0x03,0x26, 0xf292,0x63,0xbe,	0xf295,0xb6,0x86, 0xf297,0x25,0xa7,
		0xf299,0x0a,0xcc, 0xf29c,0x02,0xed,	0xf29e,0xa9,0x86, 0xf2a0,0x32,0xa7,
		0xf2a4,0x31,0xb6, 0xf2a7,0xf3,0x8b,	0xf2a9,0x5b,0xb7, 0xf2ac,0xb1,0xb6,
		0xf2af,0xcf,0x1f, 0xf2b1,0xbc,0xc4,	0xf2b3,0xde,0x58, 0xf2b4,0xcd,0x58,
		0xf2b5,0xb4,0x84, 0xf2b7,0x99,0x44,	0xf2b8,0x7c,0x44, 0xf2b9,0xf0,0x26,
		0xf2bb,0xab,0x86, 0xf2bd,0x57,0x7d,	0xf2c0,0x17,0x26, 0xf2c2,0x7f,0xfd,
		0xf2c5,0xec,0xcc, 0xf2c8,0x04,0xfd,	0xf2cb,0x17,0xfd, 0xf2ce,0xff,0xfd,
		0xf2d1,0x4c,0x7f, 0xf2d4,0x4d,0x6d,	0xf2d8,0x0e,0x26, 0xf2da,0x6b,0xbe,
		0xf2dd,0xe4,0xcc, 0xf2e0,0x7e,0xed,	0xf2e2,0xb4,0x86, 0xf2e4,0x32,0xa7,
		0xf2e6,0xb1,0x86, 0xf2e8,0xb2,0xa7,	0xf2ec,0x59,0xb6, 0xf2ef,0x63,0x8b,
		0xf2f1,0x33,0xb7, 0xf2f4,0x17,0x35,	0xf2f6,0xdd,0x39, 0xffdb,0xe3,0x7e,
		0xffe1,0x59,0x7e, -1
	};

	static const int cpu1_patch[] =
	{
		0xaeda,0xc1,0xc6, 0xaedc,0xf3,0x1e, 0xaede,0x0f,0xf7, 0xaee1,0x3b,0xf7,
		0xaee4,0xd2,0x0f, 0xaee6,0xf5,0x8e, 0xaee9,0xd9,0x96, 0xaeeb,0x97,0x27,
		0xaeed,0x96,0xad, 0xaeef,0xa0,0xb7, 0xaef2,0x5f,0x7f, 0xaef5,0x04,0x20,
		0xaf21,0x35,0x34, 0xaf23,0xde,0xdc, 0xaf25,0x99,0x1e, 0xaf27,0xc1,0x43,
		0xaf28,0xff,0xdd, 0xaf2a,0xcd,0xd6, 0xaf2c,0xce,0xc1, 0xaf2e,0x06,0x26,
		0xaf30,0xd9,0xde, 0xaf32,0x37,0x33, 0xaf36,0x7c,0xff, 0xaf39,0xd9,0xd6,
		0xaf3b,0x8a,0xf7, 0xaf3e,0x04,0x33, 0xaf42,0xef,0xff, 0xaf45,0xe7,0xf7,
		0xaf48,0xec,0xdc, 0xaf4a,0x10,0x20, 0xaf4c,0xd9,0xc1, 0xaf4e,0x01,0x26,
		0xaf50,0x4c,0xdc, 0xaf52,0x41,0xc6, 0xaf54,0x4d,0x20, 0xaf56,0xb8,0xc0,
		0xaf58,0xe0,0x58, 0xaf59,0xe6,0x58, 0xaf5a,0x69,0x8e, 0xaf5d,0x25,0x3a,
		0xaf5e,0x22,0x3a, 0xaf5f,0x49,0x86, 0xaf61,0xb5,0x34, 0xaf63,0x5c,0xde,
		0xaf65,0x05,0x10, 0xaf69,0x18,0xa6, 0xaf6b,0xbe,0xff, 0xaf6e,0x06,0x33,
		0xaf72,0xd1,0x44, 0xaf73,0x81,0x25, 0xaf75,0xb3,0x7f, 0xaf78,0x04,0x20,
		0xaf7a,0xdc,0xd6, 0xaf7c,0x9b,0xf7, 0xaf7f,0x11,0x31, 0xaf81,0x04,0x26,
		0xaf83,0x48,0x6a, 0xaf85,0x05,0x27, 0xaf87,0xd1,0x0c, 0xaf89,0x0a,0x20,
		0xaf8b,0x0a,0x32, 0xaf8d,0xfe,0xdc, 0xaf8f,0xec,0xcb, 0xaf91,0x13,0x26,
		0xaf93,0xe4,0x8b, 0xaf95,0x8f,0x43, 0xaf96,0xc3,0x1e, 0xaf98,0xf1,0xdd,
		0xaf9a,0x1f,0x35, 0xaf9c,0x23,0x39, 0xaf9d,0x42,0x8e, 0xafa0,0xda,0x4f,
		0xafa1,0x62,0xab, 0xafa3,0xf2,0x8c, 0xafa6,0x06,0x26, 0xafa8,0x27,0x39,
		0xafa9,0x1d,0x34, 0xafab,0x73,0x8e, 0xafae,0xf3,0x10, 0xafb2,0x21,0xa6,
		0xafb4,0x32,0xa7, 0xafb6,0xf3,0x8c, 0xafb9,0xca,0x23, 0xafbb,0x6b,0x7d,
		0xafbe,0x01,0x21, 0xafc0,0xf6,0x8e, 0xafc3,0x92,0x10, 0xafc7,0x79,0xec,
		0xafc9,0xf5,0xed, 0xafcb,0x79,0x8c, 0xafce,0x01,0x26, 0xafd0,0x80,0x10,
		0xafd4,0xeb,0xce, 0xafd7,0x77,0xbd, 0xafda,0x17,0x35, 0xafdc,0xdc,0x39,
		0xafdd,0x81,0xb6, 0xafe0,0x12,0x27, 0xafe2,0x36,0xb7, 0xafe5,0x77,0xbd,
		0xafe8,0xa1,0x5a, 0xafe9,0x58,0x86, 0xafeb,0x37,0x3d, 0xafec,0x78,0x8e,
		0xafef,0x2c,0x3a, 0xaff0,0x85,0x10, 0xaff4,0xf8,0xce, 0xaff7,0x6f,0xec,
		0xaff9,0xd6,0xed, 0xaffb,0xc1,0xed, 0xaffd,0x34,0xec, 0xafff,0x25,0xed,
		0xb001,0xf1,0xed, 0xb003,0xf1,0xec, 0xb005,0xfd,0xed, 0xb007,0xfd,0xed,
		0xb009,0x29,0xbd, 0xb00c,0x0d,0xb7, 0xb00f,0x36,0x39, 0xb010,0x1e,0x8e,
		0xb013,0x5a,0x10, 0xb017,0xb1,0xec, 0xb019,0xa8,0xed, 0xb01b,0xb9,0xec,
		0xb01d,0xa8,0xed, 0xb01f,0xb9,0xec, 0xb021,0xf6,0xed, 0xb023,0x37,0x39,
		0xb024,0xfa,0x4f, 0xb025,0x5b,0x5f, 0xb026,0xbc,0x1f, 0xb028,0xf8,0xce,
		0xb02b,0xae,0xff, 0xb02e,0x16,0x36, 0xb030,0x87,0x11, 0xb034,0x90,0x26,
		0xb036,0x5c,0xff, 0xb039,0xf8,0xce, 0xb03c,0x11,0x36, 0xb03e,0x45,0x11,
		0xb042,0x49,0x26, 0xb044,0x35,0x39, 0xb045,0x50,0xdc, 0xb047,0x03,0x9e,
		0xb049,0x3b,0xbf, 0xb04c,0x9b,0xf7, 0xb04f,0x12,0x30, 0xb053,0xe8,0x4a,
		0xb054,0x6b,0x26, 0xb056,0x76,0x39, 0xb057,0x29,0x8e, 0xb05a,0xe9,0xce,
		0xb05d,0x1c,0xa6, 0xb05f,0x0d,0xa7, 0xb061,0x8f,0x11, 0xb065,0xb8,0x26,
		0xb067,0x01,0x8e, 0xb06a,0xe8,0xce, 0xb06d,0x18,0xa6, 0xb06f,0x0b,0xa7,
		0xb071,0xa7,0x11, 0xb075,0x90,0x26, 0xb077,0x8d,0x39, 0xb0c8,0xa2,0x96,
		0xb0ca,0xaf,0xb7, 0xb0cd,0x75,0x39, 0xb0ce,0x5b,0x34, 0xb0d0,0xf2,0x8e,
		0xb0d3,0x23,0xb6, 0xb0d6,0x01,0x26, 0xb0d8,0x53,0x4f, 0xb0d9,0x75,0xa7,
		0xb0db,0x99,0x4c, 0xb0dc,0xcb,0x26, 0xb0de,0xdf,0x35, 0xb0e0,0x47,0x39,
		0xb0e1,0xb9,0xd6, 0xb0e3,0xe1,0x8e, 0xb0e6,0x55,0x3a, 0xb0e7,0x25,0xb6,
		0xb0ea,0xe8,0x26, 0xb0ec,0x1a,0xa7, 0xb0ee,0x20,0x8c, 0xb0f1,0x10,0x26,
		0xb0f3,0xca,0x4a, 0xb0f4,0xf0,0x8e, 0xb0f7,0x30,0xa7, 0xb0f9,0x89,0x5a,
		0xb0fa,0xe8,0x26, 0xb0fc,0x90,0x86, 0xb0fe,0x01,0xd6, 0xb100,0x5b,0xa7,
		0xb102,0x69,0x86, 0xb104,0xfb,0xb7, 0xb107,0xf6,0x39, 0xb108,0x7a,0x8e,
		0xb10b,0x0b,0xb6, 0xb10e,0x01,0x26, 0xb110,0x97,0xa7, 0xb112,0xb9,0x8c,
		0xb115,0x02,0x26, 0xb117,0x1e,0x39, 0xb118,0x7e,0x8e, 0xb11b,0x0b,0xb6,
		0xb11e,0x01,0x26, 0xb120,0x5a,0xa7, 0xb122,0x61,0x8c, 0xb125,0xf8,0x26,
		0xb127,0x6d,0xb6, 0xb12a,0x02,0x27, 0xb12c,0x19,0xb6, 0xb12f,0x01,0x26,
		0xb131,0x94,0xcc, 0xb134,0x7f,0xfd, 0xb137,0x87,0xb7, 0xb13a,0x9e,0xf7,
		0xb13d,0xfc,0xcc, 0xb140,0xcb,0xb7, 0xb143,0x27,0xf7, 0xb146,0x67,0xb7,
		0xb149,0x9b,0xf7, 0xb14c,0xe8,0xcc, 0xb14f,0x0f,0xb7, 0xb152,0x27,0xf7,
		0xb155,0x68,0xb7, 0xb158,0x9b,0xf7, 0xb15b,0xf9,0xcc, 0xb15e,0x0f,0xb7,
		0xb161,0x33,0xf7, 0xb164,0x73,0xb7, 0xb167,0xa7,0xf7, 0xb16a,0x37,0x39,
		0xb16b,0x0d,0xb6, 0xb16e,0x2e,0xbd, 0xb171,0xbf,0x1d, 0xb172,0x2e,0xfd,
		0xb175,0x35,0xfc, 0xb178,0x8b,0xf3, 0xb17b,0xad,0xfd, 0xb17e,0x1b,0xb3,
		0xb181,0x79,0xfd, 0xb184,0xe2,0x8e, 0xb187,0x7b,0xfc, 0xb18a,0xc5,0x2f,
		0xb18c,0x38,0xfc, 0xb18f,0xc5,0x27, 0xb191,0xf3,0x8e, 0xb194,0x02,0x30,
		0xb196,0x05,0x27, 0xb198,0xeb,0xf3, 0xb19b,0xd8,0x20, 0xb19d,0x34,0x10,
		0xb1a1,0x16,0x25, 0xb1a3,0x14,0x30, 0xb1a5,0x23,0xb3, 0xb1a8,0xe4,0x20,
		0xb1aa,0x5d,0x58, 0xb1ab,0x72,0x49, 0xb1ac,0xc5,0x10, 0xb1b0,0x16,0x25,
		0xb1b2,0x05,0x30, 0xb1b4,0x73,0xbf, 0xb1b7,0xe1,0x8e, 0xb1ba,0xdd,0xfc,
		0xb1bd,0x3a,0x27, 0xb1bf,0x41,0x8e, 0xb1c2,0x12,0x30, 0xb1c4,0x03,0x27,
		0xb1c6,0x2e,0xf3, 0xb1c9,0x84,0x20, 0xb1cb,0x9a,0x10, 0xb1cf,0x9d,0x25,
		0xb1d1,0x06,0x30, 0xb1d3,0x26,0xb3, 0xb1d6,0x10,0x20, 0xb1d8,0x3c,0x58,
		0xb1d9,0x9f,0x49, 0xb1da,0x9a,0x10, 0xb1de,0xfd,0x25, 0xb1e0,0x05,0x30,
		0xb1e2,0x66,0xbf, 0xb1e5,0xf0,0x8e, 0xb1e8,0x1d,0xfc, 0xb1eb,0xa3,0x2f,
		0xb1ed,0x7d,0xfc, 0xb1f0,0x12,0x27, 0xb1f2,0x94,0x10, 0xb1f6,0x06,0x25,
		0xb1f8,0x85,0x30, 0xb1fa,0xd7,0xb3, 0xb1fd,0xc4,0x20, 0xb1ff,0xa4,0xaf,
		0xb201,0x1c,0x58, 0xb202,0xb4,0x49, 0xb203,0xbd,0xed, 0xb205,0xca,0x8e,
		0xb208,0x9f,0xe3, 0xb20a,0x12,0x30, 0xb20c,0x02,0x26, 0xb20e,0x52,0x10,
		0xb212,0x9d,0x25, 0xb214,0x82,0x30, 0xb216,0x06,0xb3, 0xb219,0x05,0x20,
		0xb21b,0x2d,0xaf, 0xb21d,0xf5,0x58, 0xb21e,0xf4,0x49, 0xb21f,0xbd,0xed,
		0xb221,0xd8,0x8e, 0xb224,0x22,0xe3, 0xb226,0xf4,0x30, 0xb228,0x10,0x26,
		0xb22a,0x55,0x10, 0xb22e,0x02,0x25, 0xb230,0x8d,0x30, 0xb232,0x0e,0xb3,
		0xb235,0x6c,0x20, 0xb237,0x43,0x1f, 0xb239,0xd4,0x44, 0xb23a,0xd4,0x56,
		0xb23b,0xae,0xed, 0xb23d,0xbc,0xec, 0xb23f,0x6d,0x8e, 0xb242,0x8c,0xe3,
		0xb244,0x7c,0x30, 0xb246,0x69,0x26, 0xb248,0x9f,0xe3, 0xb24a,0xbd,0xed,
		0xb24c,0xa8,0xec, 0xb24e,0xf2,0x58, 0xb24f,0xf4,0x49, 0xb250,0x1c,0x58,
		0xb251,0xfb,0x49, 0xb252,0xbd,0xed, 0xb254,0xc3,0x8e, 0xb257,0x44,0xe3,
		0xb259,0x02,0x30, 0xb25b,0x01,0x26, 0xb25d,0x99,0xe3, 0xb25f,0xbf,0xfd,
		0xb262,0x5c,0x32, 0xb264,0x6c,0x20, 0xb266,0xe7,0xbf, 0xb269,0xf2,0x4f,
		0xb26a,0xe6,0x5f, 0xb26b,0x69,0x8e, 0xb26e,0x54,0x10, 0xb272,0x89,0xce,
		0xb275,0x91,0x36, 0xb277,0xa5,0x11, 0xb27b,0x12,0x26, 0xb27d,0x45,0x10,
		0xb281,0xe2,0x8e, 0xb284,0x68,0xec, 0xb286,0xa4,0x83, 0xb289,0x89,0x25,
		0xb28b,0x9a,0x30, 0xb28d,0x8c,0x20, 0xb28f,0x44,0xc3, 0xb292,0xf1,0x8c,
		0xb295,0x02,0x27, 0xb297,0x13,0x34, 0xb299,0x62,0x4f, 0xb29a,0x0c,0x8b,
		0xb29c,0x0b,0x19, 0xb29d,0xdd,0x30, 0xb29f,0xc9,0x26, 0xb2a1,0x32,0xa7,
		0xb2a3,0x06,0x35, 0xb2a5,0xa2,0x83, 0xb2a8,0xf6,0x25, 0xb2aa,0xf5,0x30,
		0xb2ac,0x84,0x20, 0xb2ae,0x5b,0xc3, 0xb2b1,0xf4,0x8c, 0xb2b4,0x12,0x27,
		0xb2b6,0x17,0x34, 0xb2b8,0xda,0x4f, 0xb2b9,0xa2,0x8b, 0xb2bb,0xf2,0x19,
		0xb2bc,0x25,0x30, 0xb2be,0x32,0x26, 0xb2c0,0x33,0xa7, 0xb2c2,0x17,0x35,
		0xb2c4,0xa7,0x83, 0xb2c7,0x15,0x25, 0xb2c9,0xd4,0x30, 0xb2cb,0xd0,0x20,
		0xb2cd,0x4f,0xc3, 0xb2d0,0xe0,0x8c, 0xb2d3,0x05,0x27, 0xb2d5,0x06,0x34,
		0xb2d7,0x21,0xa6, 0xb2d9,0x59,0x8b, 0xb2db,0x1e,0x19, 0xb2dc,0xdd,0x30,
		0xb2de,0xc9,0x26, 0xb2e0,0x32,0xa7, 0xb2e2,0x07,0x35, 0xb2e4,0xa2,0x83,
		0xb2e7,0x02,0x25, 0xb2e9,0xc5,0x30, 0xb2eb,0xd0,0x20, 0xb2ed,0x4a,0xc3,
		0xb2f0,0xf4,0x8c, 0xb2f3,0x10,0x27, 0xb2f5,0x15,0x34, 0xb2f7,0xd8,0x4f,
		0xb2f8,0xe2,0x8b, 0xb2fa,0xb6,0x19, 0xb2fb,0x27,0x30, 0xb2fd,0x30,0x26,
		0xb2ff,0x70,0xa7, 0xb301,0x39,0x35, 0xb303,0x8c,0x83, 0xb306,0x3d,0x25,
		0xb308,0x04,0x30, 0xb30a,0x10,0x20, 0xb30c,0x68,0x8c, 0xb30f,0x05,0x27,
		0xb311,0xfb,0xa6, 0xb313,0xc4,0x8b, 0xb315,0x3d,0x19, 0xb316,0xb2,0x30,
		0xb318,0x02,0x26, 0xb31a,0x0d,0xa7, 0xb31c,0x01,0x33, 0xb31e,0x52,0x10,
		0xb322,0xfd,0x10, 0xb326,0x36,0x39, 0xb327,0x09,0x8e, 0xb32a,0x48,0x86,
		0xb32c,0x1a,0xa7, 0xb32e,0x7b,0x8b, 0xb330,0x8e,0x89, 0xb332,0x88,0x8c,
		0xb335,0x10,0x26, 0xb337,0xa5,0x96, 0xb339,0x10,0x26, 0xb33b,0x10,0x32,
		0xb33d,0xb1,0x7e, 0xb340,0xba,0xc6, 0xb342,0xa1,0xce, 0xb345,0xc2,0x8e,
		0xb348,0x44,0x10, 0xb34c,0x0a,0xa6, 0xb34e,0x0d,0xa7, 0xb350,0x4b,0xa7,
		0xb352,0xa2,0x5a, 0xb353,0x81,0x26, 0xb355,0x0f,0xbd, 0xb358,0x1b,0xb7,
		0xb35b,0x36,0x39, 0xb35c,0xcb,0xc6, 0xb35e,0xe9,0xce, 0xb361,0x58,0x8e,
		0xb364,0xc5,0x10, 0xb368,0x04,0x20, 0xb36a,0x4c,0x96, 0xb36c,0x78,0x8e,
		0xb36f,0x54,0x10, 0xb373,0x52,0xe6, 0xb375,0x52,0xe7, 0xb377,0x52,0xe6,
		0xb379,0x9a,0xe7, 0xb37b,0x3d,0xbd, 0xb37e,0x0b,0xb7, 0xb381,0x55,0x39,
		0xb382,0xfb,0xdc, 0xb384,0x33,0xb7, 0xb387,0xb1,0x84, 0xb389,0x73,0x1f,
		0xb38b,0xb3,0xdc, 0xb38d,0xc3,0xe7, 0xb38f,0xa8,0x4a, 0xb390,0x02,0x26,
		0xb392,0x56,0x39, 0xb393,0xe1,0x8e, 0xb396,0xab,0xd6, 0xb398,0x6a,0xae,
		0xb39a,0x12,0x10, 0xb39d,0x2b,0x27, 0xb39f,0x32,0x10, 0xb3a2,0x28,0xe6,
		0xb3a4,0x12,0x27, 0xb3a6,0xba,0xc1, 0xb3a8,0xe2,0x23, 0xb3aa,0x2e,0xd7,
		0xb3ac,0x3f,0xbd, 0xb3af,0x10,0x20, 0xb3b1,0xae,0xc1, 0xb3b3,0x06,0x27,
		0xb3b5,0xd4,0x0c, 0xb3b7,0x10,0x20, 0xb3b9,0x13,0xd7, 0xb3bb,0xdd,0x20,
		0xb3bd,0x0f,0x39, 0xb3be,0xfa,0xce, 0xb3c1,0xe2,0x8e, 0xb3c4,0x6b,0xef,
		0xb3c6,0xf1,0x8c, 0xb3c9,0x82,0x26, 0xb3cb,0x2d,0x4f, 0xb3cc,0x6a,0xce,
		0xb3cf,0xd2,0x10, 0xb3d2,0x03,0x36, 0xb3d4,0x94,0x11, 0xb3d8,0xe2,0x26,
		0xb3da,0x47,0x7f, 0xb3dd,0x85,0x7f, 0xb3e0,0x47,0x39, 0xb411,0x32,0x7f,
		0xb414,0xa8,0x1a, 0xb416,0xcf,0x7f, 0xb419,0x55,0x10, 0xb41d,0x4b,0x86,
		0xb41f,0x69,0x8e, 0xb422,0x4e,0xb7, 0xb425,0xf8,0xe6, 0xb427,0xff,0xf7,
		0xb42a,0xf0,0x4a, 0xb42b,0x21,0x2a, 0xb42d,0xf2,0x4f, 0xb42e,0xf7,0x5f,
		0xb42f,0x79,0x8e, 0xb432,0x46,0xbf, 0xb435,0xb6,0xed, 0xb437,0xc4,0x8c,
		0xb43a,0x00,0x26, 0xb43c,0x3b,0xbf, 0xb43f,0xae,0xed, 0xb441,0x48,0x8c,
		0xb444,0x5a,0x26, 0xb446,0xd7,0xbd, 0xb449,0x4a,0x86, 0xb44b,0x0f,0xb7,
		0xb44e,0x73,0x1c, 0xb450,0xcc,0xcc, 0xb453,0xaf,0x1f, 0xb455,0x05,0xb7,
		0xb458,0x9f,0xf7, 0xb45b,0x8f,0xf7, 0xb45e,0x8b,0xf6, 0xb461,0xf0,0x26,
		0xb463,0x65,0xb6, 0xb466,0xce,0xb7, 0xb469,0x3f,0xbd, 0xb46c,0xbd,0xfc,
		0xb46f,0x54,0x10, 0xb473,0x90,0x27, 0xb475,0xbd,0x1c, 0xb477,0x83,0x20,
		0xb479,0x3f,0xbd, 0xb47c,0x75,0x1c, 0xb47e,0x1a,0xb1, 0xb481,0x90,0x10,
		0xb485,0x5f,0x7f, 0xb488,0x81,0xbd, 0xb48b,0x81,0xc6, 0xb48d,0x9f,0xf7,
		0xb490,0xf8,0xcc, 0xb493,0xe1,0x8e, 0xb496,0x7d,0xed, 0xb498,0x98,0x8c,
		0xb49b,0xc9,0x26, 0xb49d,0x21,0xcc, 0xb4a0,0x7f,0xfd, 0xb4a3,0xf1,0x8e,
		0xb4a6,0x84,0x10, 0xb4aa,0x4a,0x4f, 0xb4ab,0x2b,0xab, 0xb4ad,0x98,0xe6,
		0xb4af,0x9b,0xe7, 0xb4b1,0xf4,0x8c, 0xb4b4,0x10,0x26, 0xb4b6,0x34,0xb7,
		0xb4b9,0xbf,0xbd, 0xb4bc,0xf8,0xc6, 0xb4be,0x55,0xbd, 0xb4c1,0x35,0xb1,
		0xb4c4,0x13,0x27, 0xb4c6,0xc5,0x0f, 0xb4c8,0xc6,0xc6, 0xb4ca,0xc7,0xd7,
		0xb4cc,0x89,0xbd, 0xb4cf,0x49,0xc6, 0xb4d1,0x2d,0xf7, 0xb4d4,0x3a,0xf7,
		0xb4d7,0x4b,0x7e, 0xb4da,0xaf,0x7f, 0xb4dd,0x87,0x7d, 0xb4e0,0x42,0x2b,
		0xb4e2,0x11,0x26, 0xb4e4,0x5f,0x7d, 0xb4e7,0x94,0x10, 0xb4eb,0xb7,0x3b,
		0xb4ec,0xb9,0xb6, 0xb4ef,0x9c,0x10, 0xb4f3,0xf2,0x8e, 0xb4f6,0x4b,0x7f,
		0xb4f9,0x1d,0x1c, 0xb4fb,0x92,0x6e, 0xb4fd,0x35,0x34, 0xb4ff,0xb7,0x74,
		0xb502,0xd5,0x25, 0xb504,0x66,0x26, 0xb506,0x65,0x27, 0xb508,0x3e,0xbe,
		0xb50b,0x12,0x30, 0xb50d,0x3f,0xbf, 0xb510,0x08,0x74, 0xb513,0x7d,0x25,
		0xb515,0x02,0x26, 0xb517,0x05,0x27, 0xb519,0x2b,0xbe, 0xb51c,0x02,0x30,
		0xb51e,0x2f,0xbf, 0xb521,0x35,0x74, 0xb524,0x76,0x25, 0xb526,0x71,0x26,
		0xb528,0x12,0x27, 0xb52a,0x3e,0xbd, 0xb52d,0x95,0x74, 0xb530,0x5e,0x25,
		0xb532,0x48,0x26, 0xb534,0xb2,0x27, 0xb536,0x36,0xb6, 0xb539,0x1b,0xb7,
		0xb53c,0x95,0x74, 0xb53f,0x02,0x25, 0xb541,0xae,0x26, 0xb543,0xad,0x27,
		0xb545,0x8e,0xbe, 0xb548,0x04,0x30, 0xb54a,0x2f,0xbf, 0xb54d,0x84,0x74,
		0xb550,0xe1,0x25, 0xb552,0xf0,0x20, 0xb554,0xd3,0xbe, 0xb557,0xfa,0x30,
		0xb559,0x29,0xbf, 0xb55c,0x10,0x35, 0xb55e,0xa7,0x7f, 0xb561,0xf1,0x7e,
		0xb56e,0x00,0x20, 0xb570,0x31,0xf6, 0xb573,0x61,0xc5, 0xb575,0xfa,0x27,
		0xb577,0x62,0xbe, 0xb57a,0x14,0x30, 0xb57c,0x3b,0xbf, 0xb57f,0x0a,0xb6,
		0xb582,0x77,0xbd, 0xb585,0xd7,0x1f, 0xb587,0x66,0xbb, 0xb58a,0xe7,0xb7,
		0xb58d,0x73,0xfb, 0xb590,0x2b,0xf7, 0xb593,0x2b,0xf6, 0xb596,0xbd,0xc5,
		0xb598,0xfb,0x27, 0xb59a,0x8b,0xbe, 0xb59d,0x3c,0x30, 0xb59f,0x4f,0xbf,
		0xb5a2,0x35,0xb6, 0xb5a5,0x77,0xbd, 0xb5a8,0x9b,0x1f, 0xb5aa,0xc7,0xbb,
		0xb5ad,0xd3,0xb7, 0xb5b0,0x6b,0xfb, 0xb5b3,0x2a,0xf7, 0xb5b6,0x3e,0xf6,
		0xb5b9,0x16,0xc5, 0xb5bb,0xce,0x27, 0xb5bd,0x59,0xbe, 0xb5c0,0x04,0x30,
		0xb5c2,0x67,0xbf, 0xb5c5,0x26,0xb6, 0xb5c8,0x89,0xbd, 0xb5cb,0x6f,0x1f,
		0xb5cd,0x4b,0xbb, 0xb5d0,0x33,0xb7, 0xb5d3,0x6e,0xfb, 0xb5d6,0x2f,0xf7,
		0xb5d9,0xe0,0xf6, 0xb5dc,0x21,0xc4, 0xb5de,0x2f,0xf7, 0xb5e1,0x31,0xb6,
		0xb5e4,0x77,0xbd, 0xb5e7,0xdf,0x5d, 0xb5e8,0xfa,0x27, 0xb5ea,0x3f,0xf1,
		0xb5ed,0x2a,0x23, 0xb5ef,0x0f,0x39, 0xb5f0,0x71,0xbe, 0xb5f3,0x97,0x10,
		0xb5f7,0x22,0xb6, 0xb5fa,0x83,0x27, 0xb5fc,0x9f,0xbd, 0xb5ff,0x11,0xf1,
		0xb602,0xc1,0x2e, 0xb604,0x27,0xe7, 0xb606,0x8d,0x4f, 0xb607,0xa3,0xf6,
		0xb60a,0x98,0xe0, 0xb60c,0x05,0x25, 0xb60e,0x6c,0x8b, 0xb610,0xd9,0x19,
		0xb611,0x31,0x7c, 0xb614,0x92,0x20, 0xb616,0x4c,0xeb, 0xb618,0x9f,0xf7,
		0xb61b,0x0d,0xa7, 0xb61d,0x3e,0xbb, 0xb620,0xe7,0x19, 0xb621,0xf3,0xb7,
		0xb624,0x73,0xb7, 0xb627,0xf1,0xa6, 0xb629,0x3f,0xbd, 0xb62c,0x05,0x30,
		0xb62e,0x06,0x31, 0xb630,0x71,0xb6, 0xb633,0x6f,0x27, 0xb635,0xf7,0xbd,
		0xb638,0x8f,0xf1, 0xb63b,0x25,0x2e, 0xb63d,0x9a,0xe7, 0xb63f,0xe7,0x4f,
		0xb640,0x8a,0xf6, 0xb643,0x70,0xe0, 0xb645,0xa1,0x25, 0xb647,0x0c,0x8b,
		0xb649,0x75,0x19, 0xb64a,0xb3,0x7c, 0xb64d,0x04,0x20, 0xb64f,0xa7,0x7f,
		0xb652,0x07,0x7f, 0xb655,0x15,0xa7, 0xb657,0x0e,0xbb, 0xb65a,0x76,0x19,
		0xb65b,0x0f,0xb7, 0xb65e,0x0f,0xb7, 0xb661,0x30,0xa6, 0xb663,0x2e,0xbd,
		0xb666,0x3d,0x30, 0xb668,0x07,0x31, 0xb66a,0x2e,0xbf, 0xb66d,0x45,0x10,
		0xb671,0x8f,0x39, 0xb672,0x4d,0x34, 0xb674,0x13,0x5f, 0xb675,0xfe,0x4d,
		0xb676,0x6f,0x27, 0xb678,0xf5,0x5c, 0xb679,0x6a,0x8b, 0xb67b,0x65,0x19,
		0xb67c,0x04,0x20, 0xb67e,0x02,0x35, 0xb680,0x45,0x39, 0xb681,0xa6,0x86,
		0xb683,0xc7,0x1f, 0xb685,0xa6,0x86, 0xb687,0x27,0xb7, 0xb68a,0x47,0x7f,
		0xb68d,0xe6,0x8e, 0xb690,0x10,0x30, 0xb692,0x01,0x26, 0xb694,0xb4,0x86,
		0xb696,0x27,0xb7, 0xb699,0xdd,0x30, 0xb69b,0xc9,0x26, 0xb69d,0xf1,0x1c,
		0xb69f,0x0b,0xf6, 0xb6a2,0xa0,0x86, 0xb6a4,0x33,0xb7, 0xb6a7,0x26,0xb1,
		0xb6aa,0xe0,0x26, 0xb6ac,0x0a,0x43, 0xb6ad,0xdb,0xb7, 0xb6b0,0x31,0xb6,
		0xb6b3,0xb6,0x81, 0xb6b5,0x12,0x27, 0xb6b7,0x2b,0xf7, 0xb6ba,0xd9,0x86,
		0xb6bc,0x1b,0x1f, 0xb6be,0x55,0xbd, 0xb6c1,0x32,0xb6, 0xb6c4,0xdc,0x48,
		0xb6c5,0xe7,0x8b, 0xb6c7,0x27,0xb7, 0xb6ca,0xa1,0x86, 0xb6cc,0x0b,0xb7,
		0xb6cf,0x77,0xbd, 0xb6d2,0x77,0xbd, 0xb6d5,0x12,0x20, 0xb6d7,0x2f,0xf7,
		0xb6da,0x21,0xc6, 0xb6dc,0xf2,0x1f, 0xb6de,0x75,0x4d, 0xb6df,0x04,0x2b,
		0xb6e1,0xb8,0xc6, 0xb6e3,0xc7,0x1f, 0xb6e5,0xb0,0x86, 0xb6e7,0x27,0xb7,
		0xb6ea,0x80,0x8e, 0xb6ed,0xf8,0xce, 0xb6f0,0xdb,0x5f, 0xb6f1,0x30,0xa6,
		0xb6f3,0x10,0x27, 0xb6f5,0x85,0x10, 0xb6f9,0xc2,0xeb, 0xb6fb,0x25,0x31,
		0xb6fd,0x30,0x26, 0xb6ff,0xe1,0xe1, 0xb701,0x37,0x27, 0xb703,0x0f,0x1f,
		0xb705,0x20,0x80, 0xb707,0xf1,0x44, 0xb708,0x7e,0x8a, 0xb70a,0xc4,0x43,
		0xb70b,0x3f,0xbd, 0xb70e,0x0f,0xb7, 0xb711,0xfb,0xa6, 0xb713,0x78,0x20,
		0xb715,0x14,0x30, 0xb719,0x79,0x8c, 0xb71c,0x14,0x26, 0xb71e,0x49,0x86,
		0xb720,0x5b,0xb7, 0xb723,0x90,0x86, 0xb725,0x70,0xc6, 0xb727,0x57,0xf7,
		0xb72a,0x8e,0xf7, 0xb72d,0xd8,0xc6, 0xb72f,0xe5,0x5a, 0xb730,0x58,0x26,
		0xb732,0xe0,0x8e, 0xb735,0x25,0x10, 0xb739,0x73,0x1f, 0xb73b,0xcc,0xc6,
		0xb73d,0x58,0x86, 0xb73f,0x0b,0xa7, 0xb741,0x65,0xa1, 0xb743,0xe1,0x26,
		0xb745,0x0c,0x48, 0xb746,0xe1,0x26, 0xb748,0x5a,0x86, 0xb74a,0x0d,0xa7,
		0xb74c,0x0d,0xa1, 0xb74e,0x01,0x26, 0xb750,0xe6,0x1a, 0xb752,0xb4,0x49,
		0xb753,0x34,0x81, 0xb755,0xf9,0x26, 0xb757,0x2d,0xe7, 0xb759,0xfd,0xcb,
		0xb75b,0xfc,0xc9, 0xb75d,0x06,0x31, 0xb75f,0x01,0x26, 0xb761,0xdb,0x1f,
		0xb763,0x93,0x43, 0xb764,0x83,0x53, 0xb765,0x82,0xc3, 0xb768,0x07,0x31,
		0xb76a,0x66,0x1f, 0xb76c,0x58,0x86, 0xb76e,0x1a,0xa1, 0xb770,0xd8,0x26,
		0xb772,0x82,0x6f, 0xb774,0xe2,0x8b, 0xb776,0xf6,0x89, 0xb778,0x07,0x31,
		0xb77a,0x00,0x26, 0xb77c,0xf2,0x4f, 0xb77d,0x7c,0x8c, 0xb780,0x13,0x27,
		0xb782,0xdd,0x4d, 0xb783,0x56,0x39, 0xb784,0xcd,0x4d, 0xb785,0x07,0x27,
		0xb787,0x27,0xb7, 0xb78a,0xf0,0x20, 0xb78c,0x06,0x86, 0xb78e,0x0f,0xb7,
		0xb791,0xf3,0x8e, 0xb794,0x82,0x10, 0xb798,0xc3,0x1f, 0xb79a,0x19,0xec,
		0xb79c,0xf7,0xed, 0xb79e,0x79,0x8c, 0xb7a1,0x10,0x26, 0xb7a3,0xc4,0x1a,
		0xb7a5,0xf0,0x8e, 0xb7a8,0xed,0x10, 0xb7ac,0x3f,0xbd, 0xb7af,0x33,0x27,
		0xb7b1,0xd5,0x1c, 0xb7b3,0x22,0xb7, 0xb7b6,0x03,0x20, 0xb7b8,0x90,0x8e,
		0xb7bb,0xd1,0x10, 0xb7bf,0xc8,0xec, 0xb7c1,0x69,0xed, 0xb7c3,0x92,0x10,
		0xb7c7,0x01,0x26, 0xb7c9,0x9c,0x10, 0xb7cd,0xc2,0x86, 0xb7cf,0xcf,0xb7,
		0xb7d2,0xd3,0x1c, 0xb7d4,0xa3,0x86, 0xb7d6,0x27,0xb7, 0xb7d9,0x58,0x8e,
		0xb7dc,0xf5,0x10, 0xb7e0,0x77,0xbd, 0xb7e3,0x03,0x27, 0xb7e5,0x33,0xb7,
		0xb7e8,0xec,0x20, 0xb7ea,0x8f,0xbd, 0xb7ed,0xd0,0x86, 0xb7ef,0xcf,0xb7,
		0xb7f2,0xe0,0x8e, 0xb7f5,0x85,0x10, 0xb7f9,0x53,0xbf, 0xb7fc,0x9f,0xbd,
		0xb7ff,0xc4,0x27, 0xb801,0xa4,0xa8, 0xb803,0xaf,0xb7, 0xb806,0x87,0x1f,
		0xb808,0xe4,0x58, 0xb809,0xf4,0x58, 0xb80a,0xf2,0x58, 0xb80b,0xd9,0xc4,
		0xb80d,0x9b,0xf7, 0xb810,0xd3,0x4f, 0xb811,0xbb,0xf6, 0xb814,0xe6,0x54,
		0xb815,0x6a,0x27, 0xb817,0x11,0x4c, 0xb818,0x14,0x20, 0xb81a,0x2a,0xba,
		0xb81d,0x6f,0x8a, 0xb81f,0xc4,0x43, 0xb820,0xa4,0x48, 0xb821,0x44,0x48,
		0xb822,0x47,0xbd, 0xb825,0xbb,0xb7, 0xb828,0x1a,0xa7, 0xb82a,0x11,0x20,
		0xb82c,0x78,0x8e, 0xb82f,0x54,0x10, 0xb833,0x58,0x10, 0xb837,0xe2,0xbd,
		0xb83a,0x00,0x26, 0xb83c,0xfc,0xcc, 0xb83f,0x6d,0x8e, 0xb842,0x7a,0x10,
		0xb846,0xef,0xbf, 0xb849,0x1b,0xb7, 0xb84c,0x54,0x10, 0xb850,0xb7,0xf7,
		0xb853,0x0f,0xbf, 0xb856,0xfe,0xb1, 0xb859,0x14,0x26, 0xb85b,0x52,0x10,
		0xb85f,0x9e,0xf1, 0xb862,0x48,0x26, 0xb864,0x39,0x7e, 0xb867,0xe6,0x1f,
		0xb869,0xdc,0xc4, 0xb86b,0x4a,0x85, 0xb86d,0x12,0x27, 0xb86f,0xf8,0xca,
		0xb871,0x36,0x85, 0xb873,0x90,0x27, 0xb875,0x68,0xca, 0xb877,0x31,0x85,
		0xb879,0x12,0x27, 0xb87b,0xea,0xca, 0xb87d,0x5e,0x85, 0xb87f,0x10,0x27,
		0xb881,0xee,0xca, 0xb883,0xc3,0x1e, 0xb885,0x55,0x39, 0xc000,0xba,0x1a,
		0xc002,0xcf,0x7f, 0xc005,0x3b,0x10, 0xc009,0x4f,0x86, 0xc00b,0x67,0x1f,
		0xc00d,0x4d,0x86, 0xc00f,0x69,0x8e, 0xc012,0x8f,0xb7, 0xc015,0x5e,0xe6,
		0xc017,0x4f,0xf7, 0xc01a,0xe0,0x4a, 0xc01b,0x20,0x2a, 0xc01d,0x5e,0x86,
		0xc01f,0x0f,0xb7, 0xc022,0xbc,0x1c, 0xc024,0x17,0xbd, 0xc027,0x3b,0x16,
		0xc038,0xf2,0x4f, 0xc039,0xf3,0x5f, 0xc03a,0x6d,0x8e, 0xc03d,0x3b,0xbf,
		0xc040,0x41,0xed, 0xc042,0x31,0x8c, 0xc045,0x1f,0x26, 0xc047,0x87,0xbf,
		0xc04a,0xbd,0xed, 0xc04c,0x7e,0x8c, 0xc04f,0x01,0x26, 0xc051,0xb2,0x39,
		0xc052,0x67,0x7f, 0xc055,0x3a,0xbe, 0xc058,0x12,0x30, 0xc05c,0x29,0xbf,
		0xc05f,0xf9,0xcc, 0xc062,0x46,0xfd, 0xc065,0x53,0x7f, 0xc068,0x78,0x8e,
		0xc06b,0x3f,0xbf, 0xc06e,0xfb,0xcc, 0xc071,0x76,0xed, 0xc073,0x16,0x8c,
		0xc076,0xa1,0x25, 0xc078,0x27,0x39, 0xc079,0x78,0x8e, 0xc07c,0x3b,0xbf,
		0xc07f,0xe3,0x4f, 0xc080,0x27,0xa7, 0xc082,0x12,0x30, 0xc085,0xe7,0x8c,
		0xc088,0x25,0x25, 0xc08a,0x67,0xb7, 0xc08d,0xde,0x7d, 0xc090,0x02,0x27,
		0xc092,0x13,0x34, 0xc094,0x30,0xb6, 0xc097,0xe1,0x8e, 0xc09a,0x1d,0xad,
		0xc09c,0xd9,0x7f, 0xc09f,0x77,0x35, 0xc0a1,0x43,0x3b, 0xc0ac,0x31,0xb6,
		0xc0af,0x36,0x84, 0xc0b1,0x61,0xba, 0xc0b4,0x33,0xb7, 0xc0b7,0x36,0xb7,
		0xc0ba,0x5a,0x39, 0xc0bb,0x23,0xb6, 0xc0be,0x6e,0x34, 0xc0c0,0xdb,0x5f,
		0xc0c1,0xab,0xc3, 0xc0c4,0xc5,0x1f, 0xc0c6,0x21,0xa6, 0xc0c8,0x70,0x34,
		0xc0ca,0x29,0x86, 0xc0cc,0x34,0xe6, 0xc0ce,0x25,0xe7, 0xc0d0,0xc8,0x4a,
		0xc0d1,0x05,0x26, 0xc0d3,0x17,0x35, 0xc0d5,0x23,0xa7, 0xc0d7,0x17,0x35,
		0xc0d9,0x72,0x39, 0xc0da,0x0b,0xb6, 0xc0dd,0x53,0xb7, 0xc0e0,0x33,0xb7,
		0xc0e3,0x37,0xb7, 0xc0e6,0x34,0xb7, 0xc0e9,0x4f,0x39, 0xc0ea,0x64,0x34,
		0xc0ec,0x18,0xce, 0xc0ef,0x38,0xb6, 0xc0f2,0xd5,0x1f, 0xc0f4,0xd3,0x1f,
		0xc0f6,0xc4,0x1f, 0xc0f8,0x59,0x36, 0xc0fa,0x5f,0x36, 0xc0fc,0x11,0x36,
		0xc0fe,0x06,0x36, 0xc100,0x5a,0x36, 0xc102,0x4b,0x36, 0xc104,0xc0,0x36,
		0xc106,0xc3,0x36, 0xc108,0x12,0x36, 0xc10a,0x03,0x36, 0xc10c,0x00,0x36,
		0xc10e,0x06,0x33, 0xc112,0x76,0x11, 0xc116,0xc4,0x23, 0xc118,0x10,0x35,
		0xc11a,0x36,0x39, 0xc11b,0x13,0x34, 0xc11d,0xed,0xce, 0xc120,0xd9,0xb6,
		0xc123,0x76,0x1f, 0xc125,0xf3,0x1f, 0xc127,0xe3,0x1f, 0xc129,0x11,0x36,
		0xc12b,0x14,0x36, 0xc12d,0x11,0x36, 0xc12f,0x01,0x36, 0xc131,0xd1,0x36,
		0xc133,0xc5,0x36, 0xc135,0x51,0x36, 0xc137,0x53,0x36, 0xc139,0x11,0x36,
		0xc13b,0x05,0x36, 0xc13d,0x11,0x36, 0xc13f,0x01,0x33, 0xc143,0xf6,0x11,
		0xc147,0xc4,0x23, 0xc149,0x05,0x35, 0xc14b,0x36,0x39, 0xc14c,0xfa,0xcc,
		0xc14f,0xbf,0xfd, 0xc152,0xe7,0x7f, 0xc155,0xbd,0xa6, 0xc159,0x15,0x27,
		0xc15b,0x5d,0x85, 0xc15d,0x15,0x26, 0xc15f,0x5c,0x81, 0xc161,0xc2,0x23,
		0xc163,0xc1,0x20, 0xc165,0xe9,0x1a, 0xc167,0x17,0xf6, 0xc16a,0xf8,0xca,
		0xc16c,0xdc,0xc4, 0xc16e,0x8c,0xf7, 0xc171,0xf3,0xf7, 0xc174,0x1d,0x1c,
		0xc176,0x9f,0x8e, 0xc179,0x3e,0xad, 0xc17b,0xb0,0x6f, 0xc17f,0x1a,0xb6,
		0xc182,0xe4,0x8b, 0xc184,0x21,0xb7, 0xc187,0x5f,0x7d, 0xc18a,0xed,0x27,
		0xc18c,0x04,0x7a, 0xc18f,0xf0,0x20, 0xc191,0xa2,0x86, 0xc193,0x27,0xb7,
		0xc196,0xc2,0x1a, 0xc198,0x7b,0xf6, 0xc19b,0x48,0xca, 0xc19d,0x67,0xc4,
		0xc19f,0x6f,0xf7, 0xc1a2,0x2f,0xf7, 0xc1a5,0xd5,0x1c, 0xc1a7,0x91,0x12,
		0xc1a8,0xe1,0x12, 0xc1a9,0xe1,0x12, 0xc1aa,0xf8,0x20, 0xc1ec,0x30,0x8e,
		0xc1ef,0xae,0x10, 0xc1f3,0x3a,0xe6, 0xc1f5,0x12,0x27, 0xc1f7,0x36,0xa6,
		0xc1f9,0x5a,0xa7, 0xc1fb,0xa6,0x5a, 0xc1fc,0x18,0x26, 0xc1fe,0x0e,0x20,
		0xc200,0x18,0x39, 0xc201,0x7b,0x8e, 0xc204,0x26,0x10, 0xc208,0xad,0xec,
		0xc20a,0xbd,0xed, 0xc20c,0x7a,0x8c, 0xc20f,0x15,0x25, 0xc211,0x53,0x39,
		0xc212,0x8b,0xbe, 0xc215,0x0d,0xec, 0xc217,0x1f,0xfd, 0xc21a,0x0b,0xb6,
		0xc21d,0x2b,0xaa, 0xc21f,0x0f,0xb7, 0xc222,0x0a,0x39, 0xc223,0x48,0xbe,
		0xc226,0xcb,0xec, 0xc228,0xbf,0xfd, 0xc22b,0x08,0xb6, 0xc22e,0x0b,0xa4,
		0xc230,0x93,0xb7, 0xc233,0x44,0x39, 0xc234,0xd9,0xbe, 0xc237,0x1c,0xec,
		0xc239,0xbf,0xfd, 0xc23c,0x19,0xb6, 0xc23f,0x1c,0xa4, 0xc241,0x5c,0xb7,
		0xc244,0x7f,0xec, 0xc246,0x7f,0xfd, 0xc249,0x1d,0xb6, 0xc24c,0x39,0xaa,
		0xc24e,0x0f,0xb7, 0xc251,0xac,0x39, 0xc252,0x09,0x86, 0xc254,0x0f,0x1f,
		0xc256,0xab,0xbe, 0xc259,0x1b,0xa6, 0xc25b,0x0f,0xb7, 0xc25e,0x29,0xae,
		0xc260,0xb8,0xce, 0xc263,0xca,0x30, 0xc265,0xd3,0xbf, 0xc268,0x18,0xa6,
		0xc26a,0x3a,0xba, 0xc26d,0x1a,0xa7, 0xc26f,0x12,0x30, 0xc273,0x2a,0xbf,
		0xc276,0xb0,0xa6, 0xc278,0x29,0xba, 0xc27b,0x0e,0xa7, 0xc27d,0x05,0x30,
		0xc281,0x62,0xbf, 0xc284,0x30,0xa6, 0xc286,0x62,0xba, 0xc289,0xf2,0xa7,
		0xc28b,0xb2,0x30, 0xc28f,0x87,0xbf, 0xc292,0x21,0xa6, 0xc294,0x63,0xba,
		0xc297,0x25,0xa7, 0xc299,0xfa,0x30, 0xc29d,0xd6,0xbf, 0xc2a0,0x30,0xa6,
		0xc2a2,0x61,0xba, 0xc2a5,0x32,0xa7, 0xc2a7,0x16,0x30, 0xc2ab,0xc2,0xbf,
		0xc2ae,0x6a,0xa6, 0xc2b0,0x61,0xba, 0xc2b3,0x31,0xa7, 0xc2b5,0x05,0x30,
		0xc2b9,0x9b,0xbf, 0xc2bc,0xf8,0xa6, 0xc2be,0xc8,0xba, 0xc2c1,0x24,0xa7,
		0xc2c3,0x12,0x30, 0xc2c7,0x67,0xbf, 0xc2ca,0xe9,0xa6, 0xc2cc,0x61,0xba,
		0xc2cf,0xad,0xa7, 0xc2d1,0x05,0x30, 0xc2d5,0x77,0xbf, 0xc2d8,0x17,0xa6,
		0xc2da,0x22,0xba, 0xc2dd,0x4f,0xa7, 0xc2df,0x1a,0x30, 0xc2e1,0x73,0xbf,
		0xc2e4,0x30,0xa6, 0xc2e6,0x62,0xba, 0xc2e9,0xb2,0xa7, 0xc2eb,0xe2,0x30,
		0xc2ed,0x9b,0xbf, 0xc2f0,0x30,0xa6, 0xc2f2,0x60,0xba, 0xc2f5,0x32,0xa7,
		0xc2f7,0x04,0x30, 0xc2f9,0x73,0xbf, 0xc2fc,0x58,0xa6, 0xc2fe,0x6c,0xba,
		0xc301,0x40,0xa7, 0xc303,0xd2,0x30, 0xc305,0xd9,0xbf, 0xc308,0x1c,0xa6,
		0xc30a,0x2a,0xba, 0xc30d,0x09,0xa7, 0xc30f,0x12,0x30, 0xc313,0x07,0xbf,
		0xc316,0xc9,0xa6, 0xc318,0x3a,0xba, 0xc31b,0x0d,0xa7, 0xc31d,0x04,0x30,
		0xc321,0x5b,0xbf, 0xc324,0x18,0xa6, 0xc326,0x10,0xba, 0xc329,0x1a,0xa7,
		0xc32b,0x12,0x30, 0xc32f,0x2d,0xbf, 0xc332,0x59,0xa6, 0xc334,0xc1,0xba,
		0xc337,0x02,0xa7, 0xc339,0x05,0x30, 0xc33d,0x3b,0xbf, 0xc340,0x11,0xa6,
		0xc342,0x0a,0xba, 0xc345,0x92,0xa7, 0xc347,0x12,0x30, 0xc34b,0x2f,0xbf,
		0xc34e,0x09,0xa6, 0xc350,0xb6,0xba, 0xc353,0xa5,0xa7, 0xc355,0xcf,0x30,
		0xc359,0x2e,0xbf, 0xc35c,0x09,0xa6, 0xc35e,0x2a,0xba, 0xc361,0x5a,0xa7,
		0xc363,0xc9,0x30, 0xc367,0x8a,0xbf, 0xc36a,0x0e,0xa6, 0xc36c,0x29,0xba,
		0xc36f,0x0e,0xa7, 0xc371,0x25,0x30, 0xc375,0x53,0xbf, 0xc378,0x18,0xa6,
		0xc37a,0x3d,0xba, 0xc37d,0x1a,0xa7, 0xc37f,0x13,0x30, 0xc383,0x67,0xbf,
		0xc386,0x21,0xa6, 0xc388,0xc8,0xba, 0xc38b,0xe5,0xa7, 0xc38d,0xbe,0x30,
		0xc391,0x64,0xbf, 0xc394,0x27,0xa6, 0xc396,0x62,0xba, 0xc399,0x34,0xa7,
		0xc39b,0x72,0x30, 0xc39f,0x47,0xbf, 0xc3a2,0x32,0xa6, 0xc3a4,0x61,0xba,
		0xc3a7,0x21,0xa7, 0xc3a9,0xa5,0x30, 0xc3ad,0xdb,0xbf, 0xc3b0,0x30,0xa6,
		0xc3b2,0x65,0xba, 0xc3b5,0x32,0xa7, 0xc3b7,0x15,0x30, 0xc3bb,0x10,0xbf,
		0xc3be,0xbd,0xa6, 0xc3c0,0x65,0xba, 0xc3c3,0x25,0xa7, 0xc3c5,0x05,0x30,
		0xc3c9,0x86,0xbf, 0xc3cc,0x18,0xa6, 0xc3ce,0x2a,0xba, 0xc3d1,0x36,0xa7,
		0xc3d3,0x12,0x30, 0xc3d7,0x67,0xbf, 0xc3da,0x09,0xa6, 0xc3dc,0x75,0xba,
		0xc3df,0xa5,0xa7, 0xc3e1,0x05,0x30, 0xc3e5,0x73,0xbf, 0xc3e8,0x70,0xa6,
		0xc3ea,0x52,0xba, 0xc3ed,0x7a,0xa7, 0xc3ef,0x39,0x30, 0xc3f1,0x73,0xbf,
		0xc3f4,0x30,0xa6, 0xc3f6,0x73,0xba, 0xc3f9,0x5a,0xa7, 0xc3fb,0x09,0x30,
		0xc3fd,0x93,0xbf, 0xc400,0x06,0xa6, 0xc402,0x0a,0xba, 0xc405,0x04,0xa7,
		0xc407,0x92,0x30, 0xc409,0x2a,0xbf, 0xc40c,0x0e,0xa6, 0xc40e,0x2a,0xba,
		0xc411,0x01,0xa7, 0xc413,0x92,0x30, 0xc415,0x0d,0xbf, 0xc418,0x0b,0xa6,
		0xc41a,0x2a,0xba, 0xc41d,0x1d,0xa7, 0xc41f,0x12,0x30, 0xc423,0xc2,0xbf,
		0xc426,0x71,0xa6, 0xc428,0x29,0xba, 0xc42b,0x0e,0xa7, 0xc42d,0x05,0x30,
		0xc431,0x1b,0xbf, 0xc434,0x18,0xa6, 0xc436,0x16,0xba, 0xc439,0x1a,0xa7,
		0xc43b,0x16,0x30, 0xc43f,0x3c,0xbf, 0xc442,0x81,0xa6, 0xc444,0xe0,0xba,
		0xc447,0x85,0xa7, 0xc449,0x01,0x30, 0xc44d,0x2f,0xbf, 0xc450,0xac,0xa6,
		0xc452,0xa2,0xba, 0xc455,0x23,0xa7, 0xc457,0xb2,0x30, 0xc45b,0x2f,0xbf,
		0xc45e,0x09,0xa6, 0xc460,0x89,0xba, 0xc463,0x16,0xa7, 0xc465,0x05,0x30,
		0xc469,0x3b,0xbf, 0xc46c,0x18,0xa6, 0xc46e,0x2e,0xba, 0xc471,0xba,0xa7,
		0xc473,0x28,0x30, 0xc477,0x2f,0xbf, 0xc47a,0x1b,0xa6, 0xc47c,0x29,0xba,
		0xc47f,0x0b,0xa7, 0xc481,0xb3,0x86, 0xc483,0xc7,0x1f, 0xc485,0x52,0x39,
		0xc486,0x02,0x32, 0xc488,0x56,0xbe, 0xc48b,0xf2,0x10, 0xc48f,0x61,0xa6,
		0xc491,0x96,0x10, 0xc494,0x7a,0xec, 0xc496,0x7d,0xed, 0xc498,0x62,0xa7,
		0xc49a,0x41,0xae, 0xc49c,0x49,0x1f, 0xc49e,0x21,0xa6, 0xc4a0,0x73,0xbf,
		0xc4a3,0x22,0xb7, 0xc4a6,0x04,0x30, 0xc4a8,0x80,0x6a, 0xc4aa,0x31,0x26,
		0xc4ac,0xd0,0xa6, 0xc4ae,0xd3,0xa7, 0xc4b0,0x03,0x33, 0xc4b4,0xd3,0x1f,
		0xc4b6,0x5a,0x6a, 0xc4b8,0x30,0x26, 0xc4ba,0x31,0x32, 0xc4bc,0x47,0x39,
		0xc4bd,0x21,0x32, 0xc4bf,0x56,0xbe, 0xc4c2,0x92,0x10, 0xc4c6,0x21,0xa6,
		0xc4c8,0x34,0x10, 0xc4cb,0xd1,0xec, 0xc4cd,0xdd,0xed, 0xc4cf,0x6d,0xa7,
		0xc4d1,0x67,0xae, 0xc4d3,0xc7,0x1f, 0xc4d5,0xcb,0x4f, 0xc4d6,0x67,0xbf,
		0xc4d9,0x1e,0xb7, 0xc4dc,0x1d,0x30, 0xc4de,0xa8,0x6a, 0xc4e0,0x10,0x26,
		0xc4e2,0x21,0xa6, 0xc4e4,0x32,0xa7, 0xc4e6,0x04,0x33, 0xc4ea,0x07,0x1f,
		0xc4ec,0x68,0x6a, 0xc4ee,0xda,0x26, 0xc4f0,0x01,0x32, 0xc4f2,0x44,0x39,
		0xc4f3,0xb5,0x86, 0xc4f5,0x15,0x34, 0xc4f7,0x65,0xbe, 0xc4fa,0x1b,0xa6,
		0xc4fc,0xd0,0xc6, 0xc4fe,0x2a,0x3d, 0xc4ff,0x2f,0x1f, 0xc501,0xeb,0x33,
		0xc505,0xa1,0xec, 0xc507,0x47,0x1f, 0xc509,0x0f,0xb7, 0xc50c,0x5a,0x84,
		0xc50e,0x67,0x1f, 0xc510,0x29,0xec, 0xc512,0x3d,0xed, 0xc514,0xbf,0xec,
		0xc516,0xbd,0xed, 0xc518,0xa9,0xec, 0xc51a,0xbd,0xed, 0xc51c,0xb1,0x6a,
		0xc51e,0x05,0x27, 0xc520,0xeb,0x33, 0xc522,0xc7,0x1f, 0xc524,0x1c,0x4c,
		0xc525,0x53,0x1f, 0xc527,0x7c,0x20, 0xc529,0x17,0x35, 0xc52b,0x58,0x86,
		0xc52d,0x15,0x34, 0xc52f,0x29,0xbe, 0xc532,0x2c,0xec, 0xc534,0x5b,0x1f,
		0xc536,0xf0,0xb7, 0xc539,0x5c,0x84, 0xc53b,0x67,0x1f, 0xc53d,0xbc,0xec,
		0xc53f,0x5e,0x84, 0xc541,0x94,0xc4, 0xc543,0xbd,0xed, 0xc545,0xae,0xec,
		0xc547,0xd1,0x84, 0xc549,0xdc,0xc4, 0xc54b,0xbd,0xed, 0xc54d,0xae,0xec,
		0xc54f,0x59,0x84, 0xc551,0xd6,0xc4, 0xc553,0xfd,0xed, 0xc555,0xe2,0x6a,
		0xc557,0xad,0x27, 0xc559,0x65,0x1f, 0xc55b,0xf1,0x4c, 0xc55c,0x70,0x1f,
		0xc55e,0x10,0x20, 0xc560,0xf7,0x35, 0xc562,0x40,0x86, 0xc564,0x3d,0x34,
		0xc566,0xa6,0xbe, 0xc569,0xbc,0xec, 0xc56b,0xf8,0xce, 0xc56e,0xaa,0xee,
		0xc570,0x70,0xee, 0xc572,0x61,0xec, 0xc574,0x13,0x1f, 0xc576,0xa9,0xb7,
		0xc579,0x5c,0x84, 0xc57b,0x74,0x1f, 0xc57d,0xbc,0xec, 0xc57f,0x29,0xaa,
		0xc581,0x7a,0xea, 0xc583,0x7d,0xed, 0xc585,0x69,0xec, 0xc587,0x60,0xaa,
		0xc589,0x32,0xea, 0xc58b,0x35,0xed, 0xc58d,0x29,0xec, 0xc58f,0xa8,0xaa,
		0xc591,0x6e,0xea, 0xc593,0x7d,0xed, 0xc595,0x6f,0xec, 0xc597,0x60,0xaa,
		0xc599,0x6e,0xea, 0xc59b,0x7d,0xed, 0xc59d,0x67,0xec, 0xc59f,0xe0,0xaa,
		0xc5a1,0x68,0xea, 0xc5a3,0x7b,0xed, 0xc5a5,0x30,0xa6, 0xc5a7,0x75,0xaa,
		0xc5a9,0x92,0xa7, 0xc5ab,0x60,0x6a, 0xc5ad,0xda,0x27, 0xc5af,0xdb,0x33,
		0xc5b1,0xd3,0x1f, 0xc5b3,0xc8,0x4c, 0xc5b4,0xd3,0x1f, 0xc5b6,0x07,0x20,
		0xc5b8,0x97,0x35, 0xc5ba,0xcd,0x86, 0xc5bc,0x9d,0x34, 0xc5be,0xfa,0xbe,
		0xc5c1,0x7c,0xec, 0xc5c3,0xc7,0x1f, 0xc5c5,0x35,0xb7, 0xc5c8,0xd8,0x84,
		0xc5ca,0xaf,0x1f, 0xc5cc,0xba,0xec, 0xc5ce,0x11,0x84, 0xc5d0,0xb8,0xc4,
		0xc5d2,0x7d,0xed, 0xc5d4,0x6b,0xec, 0xc5d6,0xb1,0x84, 0xc5d8,0x98,0xc4,
		0xc5da,0xb5,0xed, 0xc5dc,0xab,0xec, 0xc5de,0x11,0x84, 0xc5e0,0xbc,0xc4,
		0xc5e2,0x6b,0xed, 0xc5e4,0x7c,0xec, 0xc5e6,0xa3,0x84, 0xc5e8,0x5c,0xc4,
		0xc5ea,0x63,0xed, 0xc5ec,0xbc,0xec, 0xc5ee,0x03,0x84, 0xc5f0,0xbc,0xc4,
		0xc5f2,0x6b,0xed, 0xc5f4,0x7c,0xec, 0xc5f6,0xa6,0x84, 0xc5f8,0x9c,0xc4,
		0xc5fa,0xa3,0xed, 0xc5fc,0xe8,0x6a, 0xc5fe,0x4d,0x27, 0xc600,0x23,0x1f,
		0xc602,0x71,0x4c, 0xc603,0x4f,0x1f, 0xc605,0x6e,0x20, 0xc607,0x7f,0x35,
		0xc609,0x5b,0x86, 0xc60b,0x13,0x34, 0xc60d,0x38,0xbe, 0xc610,0xce,0xec,
		0xc612,0x2f,0x1f, 0xc614,0x66,0xb7, 0xc617,0xd1,0x84, 0xc619,0x75,0x1f,
		0xc61b,0xb9,0xec, 0xc61d,0x4d,0x84, 0xc61f,0xd9,0xc4, 0xc621,0xbe,0xed,
		0xc623,0xad,0xec, 0xc625,0xdc,0x84, 0xc627,0x8a,0xc4, 0xc629,0xbe,0xed,
		0xc62b,0xad,0xec, 0xc62d,0x5c,0x84, 0xc62f,0xca,0xc4, 0xc631,0x36,0xed,
		0xc633,0x32,0xec, 0xc635,0xd4,0x84, 0xc637,0x94,0xc4, 0xc639,0xbe,0xed,
		0xc63b,0xba,0xec, 0xc63d,0x5c,0x84, 0xc63f,0xdc,0xc4, 0xc641,0xa6,0xed,
		0xc643,0xb1,0xec, 0xc645,0x4b,0x84, 0xc647,0x19,0xc4, 0xc649,0xae,0xed,
		0xc64b,0xa0,0x6a, 0xc64d,0x00,0x27, 0xc64f,0x67,0x1f, 0xc651,0xd9,0x4c,
		0xc652,0x07,0x1f, 0xc654,0xb0,0x20, 0xc656,0xb7,0x35, 0xc658,0x5b,0x86,
		0xc65a,0x13,0x34, 0xc65c,0x2e,0xbe, 0xc65f,0xb9,0xec, 0xc661,0x90,0xce,
		0xc664,0xf8,0xee, 0xc666,0xe8,0xee, 0xc668,0xbc,0xec, 0xc66a,0x60,0x1f,
		0xc66c,0x1b,0xb7, 0xc66f,0x4e,0x84, 0xc671,0x9b,0x1f, 0xc673,0x6d,0xec,
		0xc675,0x42,0x43, 0xc676,0xc6,0x53, 0xc677,0xb0,0xa4, 0xc679,0x9c,0xe4,
		0xc67b,0xaf,0xed, 0xc67d,0xbc,0xec, 0xc67f,0xc3,0x43, 0xc680,0x8f,0x53,
		0xc681,0x31,0xa4, 0xc683,0x39,0xe4, 0xc685,0x6b,0xed, 0xc687,0x79,0xec,
		0xc689,0x16,0x43, 0xc68a,0xe6,0x53, 0xc68b,0xd9,0xa4, 0xc68d,0x0a,0xe4,
		0xc68f,0x3d,0xed, 0xc691,0x6e,0xec, 0xc693,0x8c,0x43, 0xc694,0x8a,0x53,
		0xc695,0x25,0xa4, 0xc697,0x39,0xe4, 0xc699,0x27,0xed, 0xc69b,0x31,0xec,
		0xc69d,0x02,0x43, 0xc69e,0x0e,0x53, 0xc69f,0xd9,0xa4, 0xc6a1,0x3c,0xe4,
		0xc6a3,0x6f,0xed, 0xc6a5,0x30,0xa6, 0xc6a7,0x99,0x43, 0xc6a8,0x3c,0xa4,
		0xc6aa,0x2a,0xa7, 0xc6ac,0x88,0x6a, 0xc6ae,0x28,0x27, 0xc6b0,0x03,0x33,
		0xc6b2,0xd5,0x1f, 0xc6b4,0xdc,0x4c, 0xc6b5,0xd3,0x1f, 0xc6b7,0x04,0x20,
		0xc6b9,0xd7,0x35, 0xc6bb,0x1b,0xc6, 0xc6bd,0x9d,0x34, 0xc6bf,0xf9,0xbe,
		0xc6c2,0xe9,0xce, 0xc6c5,0x6b,0xec, 0xc6c7,0xc7,0x1f, 0xc6c9,0x9c,0xb7,
		0xc6cc,0x50,0x84, 0xc6ce,0x27,0x1f, 0xc6d0,0x7a,0xef, 0xc6d2,0x6d,0xef,
		0xc6d4,0x7f,0xef, 0xc6d6,0x6d,0xef, 0xc6d8,0xba,0xef, 0xc6da,0xad,0xef,
		0xc6dc,0xb2,0x6a, 0xc6de,0x05,0x27, 0xc6e0,0xd3,0x1f, 0xc6e2,0xcf,0x4c,
		0xc6e3,0xd1,0x1f, 0xc6e5,0x04,0x20, 0xc6e7,0x00,0x35, 0xc6e9,0xb9,0xbe,
		0xc6ec,0x58,0xa6, 0xc6ee,0x72,0x8b, 0xc6f0,0x27,0xb1, 0xc6f3,0x07,0x25,
		0xc6f5,0xa4,0x80, 0xc6f7,0x26,0xb1, 0xc6fa,0x4f,0x25, 0xc6fc,0x58,0x86,
		0xc6fe,0x15,0x34, 0xc700,0x6c,0xbe, 0xc703,0xb9,0xec, 0xc705,0xd9,0x1f,
		0xc707,0x67,0xb7, 0xc70a,0x59,0x84, 0xc70c,0x64,0x1f, 0xc70e,0xb9,0xec,
		0xc710,0xcc,0x84, 0xc712,0x99,0xc4, 0xc714,0x24,0xed, 0xc716,0x31,0xec,
		0xc718,0x4c,0x84, 0xc71a,0xd9,0xc4, 0xc71c,0xac,0xed, 0xc71e,0xb9,0xec,
		0xc720,0x54,0x84, 0xc722,0x13,0xc4, 0xc724,0xde,0xed, 0xc726,0xdf,0xec,
		0xc728,0x5c,0x84, 0xc72a,0xdb,0xc4, 0xc72c,0xbe,0xed, 0xc72e,0xbf,0xec,
		0xc730,0xdc,0x84, 0xc732,0x98,0xc4, 0xc734,0x36,0xed, 0xc736,0x34,0xec,
		0xc738,0x5c,0x84, 0xc73a,0xd8,0xc4, 0xc73c,0xbe,0xed, 0xc73e,0xa5,0x6a,
		0xc740,0x18,0x27, 0xc742,0x27,0x1f, 0xc744,0x52,0x4c, 0xc745,0x40,0x1f,
		0xc747,0x78,0x20, 0xc749,0x5f,0x86, 0xc74b,0x0d,0xa7, 0xc74d,0x29,0xbe,
		0xc750,0xaa,0xa6, 0xc752,0xc1,0xce, 0xc755,0xee,0xee, 0xc757,0xf9,0xec,
		0xc759,0x66,0x1f, 0xc75b,0x0f,0xb7, 0xc75e,0x59,0x84, 0xc760,0x33,0x1f,
		0xc762,0xc7,0xec, 0xc764,0xa0,0xaa, 0xc766,0xf2,0xea, 0xc768,0xbe,0xed,
		0xc76a,0xaf,0xec, 0xc76c,0x28,0xaa, 0xc76e,0xba,0xea, 0xc770,0xf6,0xed,
		0xc772,0xe0,0xec, 0xc774,0xa8,0xaa, 0xc776,0xfe,0xea, 0xc778,0xbe,0xed,
		0xc77a,0xa8,0xec, 0xc77c,0x28,0xaa, 0xc77e,0xbe,0xea, 0xc780,0x7f,0xed,
		0xc782,0x79,0xec, 0xc784,0x71,0xaa, 0xc786,0x68,0xea, 0xc788,0x77,0xed,
		0xc78a,0x71,0xec, 0xc78c,0x59,0xaa, 0xc78e,0xc8,0xea, 0xc790,0x6d,0xed,
		0xc792,0x48,0x6a, 0xc794,0x06,0x27, 0xc796,0x06,0x33, 0xc798,0xf7,0x1f,
		0xc79a,0x71,0x4c, 0xc79b,0x47,0x1f, 0xc79d,0x34,0x20, 0xc79f,0x37,0x35,
		0xc7a1,0x71,0xbe, 0xc7a4,0x30,0xa6, 0xc7a6,0xe4,0x8b, 0xc7a8,0xc7,0xb1,
		0xc7ab,0x8c,0x25, 0xc7ad,0xcc,0x80, 0xc7af,0xd4,0xb1, 0xc7b2,0x12,0x25,
		0xc7b4,0xb0,0x86, 0xc7b6,0x10,0x34, 0xc7b8,0xb9,0xbe, 0xc7bb,0x3a,0xa6,
		0xc7bd,0xf0,0xe6, 0xc7bf,0xc8,0xce, 0xc7c2,0x79,0xec, 0xc7c4,0xc1,0x1f,
		0xc7c6,0x27,0xb7, 0xc7c9,0x95,0x84, 0xc7cb,0xe7,0x1f, 0xc7cd,0x3b,0xec,
		0xc7cf,0x44,0x43, 0xc7d0,0x9f,0x53, 0xc7d1,0x35,0xa4, 0xc7d3,0x39,0xe4,
		0xc7d5,0x6d,0xed, 0xc7d7,0x79,0xec, 0xc7d9,0x92,0x43, 0xc7da,0xc6,0x53,
		0xc7db,0x59,0xa4, 0xc7dd,0xcc,0xe4, 0xc7df,0xfd,0xed, 0xc7e1,0x7c,0xec,
		0xc7e3,0x8e,0x43, 0xc7e4,0x8b,0x53, 0xc7e5,0x34,0xa4, 0xc7e7,0x2f,0xe4,
		0xc7e9,0x3e,0xed, 0xc7eb,0x3d,0xec, 0xc7ed,0x42,0x43, 0xc7ee,0xd4,0x53,
		0xc7ef,0x8f,0xa4, 0xc7f1,0x3c,0xe4, 0xc7f3,0x7f,0xed, 0xc7f5,0x7c,0xec,
		0xc7f7,0x8b,0x43, 0xc7f8,0xc3,0x53, 0xc7f9,0x5c,0xa4, 0xc7fb,0xdd,0xe4,
		0xc7fd,0xfe,0xed, 0xc7ff,0x59,0xa6, 0xc801,0x58,0x43, 0xc802,0x11,0xa4,
		0xc804,0x34,0xa7, 0xc806,0xe8,0x6a, 0xc808,0x10,0x27, 0xc80a,0x06,0x33,
		0xc80c,0x64,0x1f, 0xc80e,0xf1,0x4c, 0xc80f,0x67,0x1f, 0xc811,0x76,0x20,
		0xc813,0xc1,0x86, 0xc815,0x19,0xa7, 0xc817,0x03,0xbe, 0xc81a,0xb9,0xec,
		0xc81c,0xfd,0xce, 0xc81f,0xa9,0xee, 0xc821,0xf0,0xee, 0xc823,0xf7,0xec,
		0xc825,0x13,0x1f, 0xc827,0xac,0xb7, 0xc82a,0x4f,0x84, 0xc82c,0x73,0x1f,
		0xc82e,0xbf,0xec, 0xc830,0xa8,0xaa, 0xc832,0xfe,0xea, 0xc834,0x56,0xed,
		0xc836,0x57,0xec, 0xc838,0x28,0xaa, 0xc83a,0xbe,0xea, 0xc83c,0xbe,0xed,
		0xc83e,0xbf,0xec, 0xc840,0x9c,0xaa, 0xc842,0xc8,0xea, 0xc844,0xf0,0xed,
		0xc846,0xf1,0xec, 0xc848,0x3c,0xaa, 0xc84a,0xa8,0xea, 0xc84c,0xb8,0xed,
		0xc84e,0xb9,0xec, 0xc850,0xd8,0xaa, 0xc852,0x88,0xea, 0xc854,0x7b,0xed,
		0xc856,0x21,0xa6, 0xc858,0x38,0xaa, 0xc85a,0x0d,0xa7, 0xc85c,0xb4,0x6a,
		0xc85e,0x05,0x27, 0xc860,0x03,0x33, 0xc862,0x2c,0x1f, 0xc864,0x54,0x4c,
		0xc865,0xbb,0x1f, 0xc867,0x83,0x20, 0xc869,0x17,0x35, 0xc86b,0x28,0xbe,
		0xc86e,0x0c,0xa6, 0xc870,0xea,0x8b, 0xc872,0xc5,0xb1, 0xc875,0x5e,0x25,
		0xc877,0xfb,0x80, 0xc879,0x0f,0xb1, 0xc87c,0x16,0x25, 0xc87e,0x4a,0x86,
		0xc880,0x03,0x34, 0xc882,0x63,0xbe, 0xc885,0xf8,0xce, 0xc888,0xd3,0xec,
		0xc88a,0x0f,0x1f, 0xc88c,0x44,0xb7, 0xc88f,0x11,0x84, 0xc891,0xc1,0x1f,
		0xc893,0x79,0xec, 0xc895,0x9d,0x43, 0xc896,0x8e,0x53, 0xc897,0x31,0xa4,
		0xc899,0x5a,0xe4, 0xc89b,0x7d,0xed, 0xc89d,0x92,0xec, 0xc89f,0xec,0x43,
		0xc8a0,0x8b,0x53, 0xc8a1,0x34,0xa4, 0xc8a3,0x3f,0xe4, 0xc8a5,0x7e,0xed,
		0xc8a7,0x6b,0xec, 0xc8a9,0xc2,0x43, 0xc8aa,0xb8,0x53, 0xc8ab,0x1f,0xa4,
		0xc8ad,0x9c,0xe4, 0xc8af,0xa9,0xed, 0xc8b1,0x7c,0xec, 0xc8b3,0x98,0x43,
		0xc8b4,0x8b,0x53, 0xc8b5,0x34,0xa4, 0xc8b7,0x2b,0xe4, 0xc8b9,0x7e,0xed,
		0xc8bb,0x6b,0xec, 0xc8bd,0xea,0x43, 0xc8be,0xe8,0x53, 0xc8bf,0x23,0xa4,
		0xc8c1,0x39,0xe4, 0xc8c3,0x7d,0xed, 0xc8c5,0x34,0xa6, 0xc8c7,0x8c,0x43,
		0xc8c8,0x7a,0xa4, 0xc8ca,0x6d,0xa7, 0xc8cc,0xff,0x6a, 0xc8ce,0x4d,0x27,
		0xc8d0,0x01,0x33, 0xc8d2,0xc7,0x1f, 0xc8d4,0xda,0x4c, 0xc8d5,0xc1,0x1f,
		0xc8d7,0x10,0x20, 0xc8d9,0x73,0x86, 0xc8db,0x6d,0xa7, 0xc8dd,0x88,0xbe,
		0xc8e0,0x30,0xa6, 0xc8e2,0x2a,0xe6, 0xc8e4,0xf8,0xce, 0xc8e7,0x7a,0xee,
		0xc8e9,0x90,0xee, 0xc8eb,0x85,0xec, 0xc8ed,0x4a,0x8b, 0xc8ef,0x1c,0x1f,
		0xc8f1,0x33,0xb7, 0xc8f4,0xb4,0x84, 0xc8f6,0xc4,0x1f, 0xc8f8,0x5c,0xec,
		0xc8fa,0xca,0xaa, 0xc8fc,0xe0,0xea, 0xc8fe,0xe1,0xed, 0xc900,0xb3,0xec,
		0xc902,0xe0,0xaa, 0xc904,0xfe,0xea, 0xc906,0xfd,0xed, 0xc908,0xbb,0xec,
		0xc90a,0x28,0xaa, 0xc90c,0xbe,0xea, 0xc90e,0xbd,0xed, 0xc910,0xd0,0xec,
		0xc912,0x80,0xaa, 0xc914,0x9d,0xea, 0xc916,0x9d,0xed, 0xc918,0xb8,0xec,
		0xc91a,0x28,0xaa, 0xc91c,0xbd,0xea, 0xc91e,0xbd,0xed, 0xc920,0xf8,0xa6,
		0xc922,0xf6,0xaa, 0xc924,0xb2,0xa7, 0xc926,0x78,0x6a, 0xc928,0x12,0x27,
		0xc92a,0x10,0x33, 0xc92c,0x73,0x1f, 0xc92e,0xe1,0x4c, 0xc92f,0x62,0x1f,
		0xc931,0x84,0x20, 0xc933,0x31,0x86, 0xc935,0x7a,0xa7, 0xc937,0x66,0xbe,
		0xc93a,0xa9,0xec, 0xc93c,0xf8,0xce, 0xc93f,0xaa,0xee, 0xc941,0x5a,0xee,
		0xc943,0x59,0xec, 0xc945,0x80,0x1f, 0xc947,0x2f,0xb7, 0xc94a,0x59,0x84,
		0xc94c,0x63,0x1f, 0xc94e,0xb9,0xec, 0xc950,0xf6,0xaa, 0xc952,0xa0,0xea,
		0xc954,0x99,0xed, 0xc956,0x99,0xec, 0xc958,0x3e,0xaa, 0xc95a,0xa8,0xea,
		0xc95c,0xb9,0xed, 0xc95e,0xb9,0xec, 0xc960,0x08,0xaa, 0xc962,0x49,0xea,
		0xc964,0x76,0xed, 0xc966,0x61,0xec, 0xc968,0x28,0xaa, 0xc96a,0xa9,0xea,
		0xc96c,0xbe,0xed, 0xc96e,0xa9,0xec, 0xc970,0xe0,0xaa, 0xc972,0xb0,0xea,
		0xc974,0x9e,0xed, 0xc976,0xc0,0xa6, 0xc978,0x28,0xaa, 0xc97a,0x0a,0xa7,
		0xc97c,0x18,0xa6, 0xc97e,0xf0,0x4a, 0xc97f,0x0a,0xa7, 0xc981,0x12,0x27,
		0xc983,0xb4,0x81, 0xc985,0x13,0x26, 0xc987,0xc7,0x1f, 0xc989,0xf6,0x8b,
		0xc98b,0xa7,0x1f, 0xc98d,0x15,0x20, 0xc98f,0x2f,0x1f, 0xc991,0xcc,0x4c,
		0xc992,0xc7,0x1f, 0xc994,0x04,0x33, 0xc996,0x10,0x20, 0xc998,0x69,0x35,
		0xc99a,0x0b,0xbe, 0xc99d,0x98,0x86, 0xc99f,0xd3,0x34, 0xc9a1,0x71,0xbe,
		0xc9a4,0x30,0xa6, 0xc9a6,0x28,0xe6, 0xc9a8,0x30,0xce, 0xc9ab,0x6c,0xee,
		0xc9ad,0xf0,0xee, 0xc9af,0xe5,0xec, 0xc9b1,0xe2,0x8b, 0xc9b3,0xd7,0x1f,
		0xc9b5,0x33,0xb7, 0xc9b8,0x34,0x84, 0xc9ba,0x46,0x1f, 0xc9bc,0x5c,0xec,
		0xc9be,0xde,0xaa, 0xc9c0,0x6a,0xea, 0xc9c2,0x7d,0xed, 0xc9c4,0x78,0xec,
		0xc9c6,0x60,0xaa, 0xc9c8,0xc2,0xea, 0xc9ca,0xd5,0xed, 0xc9cc,0xb0,0xec,
		0xc9ce,0x20,0xaa, 0xc9d0,0x7e,0xea, 0xc9d2,0x7d,0xed, 0xc9d4,0x78,0xec,
		0xc9d6,0x60,0xaa, 0xc9d8,0x3e,0xea, 0xc9da,0x3d,0xed, 0xc9dc,0x58,0xec,
		0xc9de,0xc8,0xaa, 0xc9e0,0x68,0xea, 0xc9e2,0x7a,0xed, 0xc9e4,0x30,0xa6,
		0xc9e6,0x70,0xaa, 0xc9e8,0x72,0xa7, 0xc9ea,0x81,0x6a, 0xc9ec,0x52,0x27,
		0xc9ee,0x56,0x33, 0xc9f0,0xd3,0x1f, 0xc9f2,0xc9,0x4c, 0xc9f3,0xc6,0x1f,
		0xc9f5,0x04,0x20, 0xc9f7,0xa0,0x86, 0xc9f9,0xfa,0xa7, 0xc9fb,0xcc,0xbe,
		0xc9fe,0x49,0xec, 0xca00,0xce,0xce, 0xca03,0x49,0xee, 0xca05,0x2d,0xee,
		0xca07,0x39,0xec, 0xca09,0x77,0x1f, 0xca0b,0x0f,0xb7, 0xca0e,0x59,0x84,
		0xca10,0x4f,0x1f, 0xca12,0xb9,0xec, 0xca14,0x6f,0xaa, 0xca16,0x28,0xea,
		0xca18,0xbd,0xed, 0xca1a,0xb9,0xec, 0xca1c,0x2f,0xaa, 0xca1e,0xa8,0xea,
		0xca20,0xfe,0xed, 0xca22,0xf8,0xec, 0xca24,0x48,0xaa, 0xca26,0x19,0xea,
		0xca28,0xbe,0xed, 0xca2a,0xb8,0xec, 0xca2c,0x28,0xaa, 0xca2e,0xb9,0xea,
		0xca30,0xbe,0xed, 0xca32,0xb9,0xec, 0xca34,0x68,0xaa, 0xca36,0x3d,0xea,
		0xca38,0xbe,0xed, 0xca3a,0x18,0xa6, 0xca3c,0x28,0xaa, 0xca3e,0x0f,0xa7,
		0xca40,0xf5,0xa6, 0xca42,0x08,0x4a, 0xca43,0x65,0xa7, 0xca45,0x4d,0x27,
		0xca47,0xfc,0x81, 0xca49,0x17,0x26, 0xca4b,0x67,0x1f, 0xca4d,0x69,0x8b,
		0xca4f,0x67,0x1f, 0xca51,0x9b,0x20, 0xca53,0xa7,0x1f, 0xca55,0xf4,0x4c,
		0xca56,0xef,0x1f, 0xca58,0x14,0x33, 0xca5a,0x10,0x20, 0xca5c,0x17,0x35,
		0xca5e,0x2b,0xbe, 0xca61,0x70,0xa6, 0xca63,0x4e,0x8b, 0xca65,0xcf,0xb1,
		0xca68,0x16,0x25, 0xca6a,0x4f,0x80, 0xca6c,0x0f,0xb1, 0xca6f,0x10,0x25,
		0xca71,0x38,0x86, 0xca73,0x9b,0x34, 0xca75,0x11,0xbe, 0xca78,0xf8,0xce,
		0xca7b,0xbf,0xec, 0xca7d,0x73,0x1f, 0xca7f,0x1e,0xb7, 0xca82,0xb1,0x84,
		0xca84,0xd3,0x1f, 0xca86,0x79,0xec, 0xca88,0x83,0x43, 0xca89,0xbb,0x53,
		0xca8a,0x59,0xa4, 0xca8c,0x60,0xe4, 0xca8e,0x55,0xed, 0xca90,0x7c,0xec,
		0xca92,0x8c,0x43, 0xca93,0x8e,0x53, 0xca94,0x21,0xa4, 0xca96,0x39,0xe4,
		0xca98,0x7d,0xed, 0xca9a,0x79,0xec, 0xca9c,0x0e,0x43, 0xca9d,0xb7,0x53,
		0xca9e,0xd1,0xa4, 0xcaa0,0x3c,0xe4, 0xcaa2,0x7a,0xed, 0xcaa4,0x7c,0xec,
		0xcaa6,0x9b,0x43, 0xcaa7,0x89,0x53, 0xcaa8,0x5c,0xa4, 0xcaaa,0xd8,0xe4,
		0xcaac,0x56,0xed, 0xcaae,0x40,0xec, 0xcab0,0x8a,0x43, 0xcab1,0x8b,0x53,
		0xcab2,0x31,0xa4, 0xcab4,0x3c,0xe4, 0xcab6,0x6e,0xed, 0xcab8,0xd8,0xa6,
		0xcaba,0x02,0x43, 0xcabb,0xc8,0xa4, 0xcabd,0x7a,0xa7, 0xcabf,0x99,0x6a,
		0xcac1,0x16,0x27, 0xcac3,0x06,0x33, 0xcac5,0xd5,0x1f, 0xcac7,0xd9,0x4c,
		0xcac8,0xa4,0x1f, 0xcaca,0x98,0x20, 0xcacc,0xbe,0x39, 0xcacd,0xf8,0x32,
		0xcacf,0xa1,0x86, 0xcad1,0x34,0xa7, 0xcad3,0x63,0xbe, 0xcad6,0x69,0xee,
		0xcad8,0x7f,0x10, 0xcadc,0xe4,0xa6, 0xcade,0xe5,0xa7, 0xcae0,0xf0,0x8e,
		0xcae3,0x3c,0xe6, 0xcae5,0xae,0xc1, 0xcae7,0x13,0x26, 0xcae9,0x83,0x33,
		0xcaed,0xec,0x20, 0xcaef,0xcf,0x3a, 0xcaf0,0xb0,0x86, 0xcaf2,0x25,0xa7,
		0xcaf4,0xb0,0x86, 0xcaf6,0x31,0xa7, 0xcaf8,0x30,0xa6, 0xcafa,0x81,0xff,
		0xcafd,0x63,0x33, 0xcb01,0xc5,0x44, 0xcb02,0x35,0x25, 0xcb04,0xfd,0x7f,
		0xcb07,0x30,0x20, 0xcb09,0x8f,0xe6, 0xcb0b,0x8f,0xf7, 0xcb0e,0xa0,0x6a,
		0xcb10,0xb6,0x26, 0xcb12,0xe8,0x6a, 0xcb14,0x21,0x27, 0xcb16,0x26,0x33,
		0xcb1a,0x10,0x20, 0xcb1c,0xa4,0x6a, 0xcb1e,0x05,0x27, 0xcb20,0x23,0x33,
		0xcb24,0xa4,0x20, 0xcb26,0xfd,0x7f, 0xcb29,0x17,0x35, 0xcb2b,0x01,0x32,
		0xcb2d,0x58,0x86, 0xcb2f,0x1d,0xa7, 0xcb31,0x39,0xbe, 0xcb34,0xf8,0xee,
		0xcb36,0x16,0x10, 0xcb3a,0x18,0xa6, 0xcb3c,0x1a,0xa7, 0xcb3e,0x6b,0x8e,
		0xcb41,0x75,0xe6, 0xcb43,0x54,0xc1, 0xcb45,0x0b,0x26, 0xcb47,0x0e,0x33,
		0xcb4b,0x10,0x20, 0xcb4d,0x37,0x3a, 0xcb4e,0x49,0x86, 0xcb50,0xc7,0xa7,
		0xcb52,0xe1,0x86, 0xcb54,0x61,0xa7, 0xcb56,0x61,0xa6, 0xcb58,0xbf,0xff,
		0xcb5b,0x06,0x33, 0xcb5f,0xd1,0x44, 0xcb60,0xb6,0x25, 0xcb62,0xe9,0x7f,
		0xcb65,0xd3,0xff, 0xcb68,0x03,0x33, 0xcb6c,0xb3,0x7f, 0xcb6f,0x14,0x20,
		0xcb71,0x58,0xe6, 0xcb73,0x5d,0xf7, 0xcb76,0x2d,0xff, 0xcb79,0x03,0x33,
		0xcb7d,0x9b,0xf7, 0xcb80,0x5b,0x6a, 0xcb82,0x01,0x26, 0xcb84,0x48,0x6a,
		0xcb86,0x05,0x27, 0xcb88,0x0a,0x33, 0xcb8c,0x02,0x20, 0xcb8e,0xa0,0x6a,
		0xcb90,0x17,0x27, 0xcb92,0x06,0x33, 0xcb96,0x10,0x20, 0xcb98,0xa7,0x7f,
		0xcb9b,0x17,0x35, 0xcb9d,0x1f,0x32, 0xcb9f,0x41,0x86, 0xcba1,0x32,0xa7,
		0xcba3,0x71,0xbe, 0xcba6,0x7b,0xee, 0xcba8,0x05,0x10, 0xcbac,0x50,0xa6,
		0xcbae,0x51,0xa7, 0xcbb0,0xf0,0x8e, 0xcbb3,0x29,0xe6, 0xcbb5,0x41,0x3a,
		0xcbb6,0xa3,0x86, 0xcbb8,0x52,0xa7, 0xcbba,0x58,0x86, 0xcbbc,0x5a,0xa7,
		0xcbbe,0x4b,0xa6, 0xcbc0,0x78,0xff, 0xcbc3,0x06,0x33, 0xcbc7,0x99,0x44,
		0xcbc8,0x02,0x25, 0xcbca,0xa7,0x7f, 0xcbcd,0x6d,0x20, 0xcbcf,0xa9,0xe6,
		0xcbd1,0x3d,0xf7, 0xcbd4,0x4c,0x6a, 0xcbd6,0x01,0x26, 0xcbd8,0x0a,0x6a,
		0xcbda,0xad,0x27, 0xcbdc,0x95,0x33, 0xcbe0,0x04,0x20, 0xcbe2,0x5f,0x6a,
		0xcbe4,0x12,0x27, 0xcbe6,0x13,0x33, 0xcbea,0x16,0x20, 0xcbec,0x53,0x7f,
		0xcbef,0x6a,0x35, 0xcbf1,0x01,0x32, 0xcbf3,0x77,0xbe, 0xcbf6,0x23,0xa6,
		0xcbf8,0xfa,0xa7, 0xcbfa,0xa4,0x10, 0xcbfe,0x63,0xee, 0xcc00,0xcc,0x8e,
		0xcc03,0x01,0xe6, 0xcc05,0xe8,0x27, 0xcc07,0x18,0xc0, 0xcc09,0xf7,0x58,
		0xcc0a,0xf2,0x58, 0xcc0b,0x22,0x3a, 0xcc0c,0x21,0x3a, 0xcc0d,0x49,0x86,
		0xcc0f,0x0d,0xa7, 0xcc11,0x4a,0x86, 0xcc13,0x6d,0xa7, 0xcc15,0xcf,0xa6,
		0xcc17,0x87,0xff, 0xcc1a,0x06,0x33, 0xcc1e,0xd1,0x44, 0xcc1f,0x11,0x24,
		0xcc21,0x10,0xe6, 0xcc23,0x1b,0xfa, 0xcc26,0x38,0xf7, 0xcc29,0xa0,0x6a,
		0xcc2b,0x07,0x26, 0xcc2d,0x03,0x33, 0xcc31,0xa0,0x6a, 0xcc33,0xeb,0x27,
		0xcc35,0x4c,0x20, 0xcc37,0x13,0x7f, 0xcc3a,0x02,0x35, 0xcc3c,0x01,0x32,
		0xcc3e,0x3b,0xbe, 0xcc41,0xc0,0xa6, 0xcc43,0xc5,0xa7, 0xcc45,0x5e,0x10,
		0xcc49,0xa8,0xee, 0xcc4b,0x69,0x8e, 0xcc4e,0x89,0xe6, 0xcc50,0xff,0x27,
		0xcc52,0x18,0xc0, 0xcc54,0x92,0x58, 0xcc55,0xa8,0x58, 0xcc56,0xe2,0x3a,
		0xcc57,0xca,0x3a, 0xcc58,0x5e,0x86, 0xcc5a,0x0d,0xa7, 0xcc5c,0x4c,0x86,
		0xcc5e,0x0d,0xa7, 0xcc60,0xf8,0xa6, 0xcc62,0xb2,0xff, 0xcc65,0x6b,0x33,
		0xcc69,0xd4,0x44, 0xcc6a,0x17,0x25, 0xcc6c,0x03,0x33, 0xcc70,0xec,0x20,
		0xcc72,0x38,0xe6, 0xcc74,0x21,0xfa, 0xcc77,0x16,0xf7, 0xcc7a,0xbe,0xff,
		0xcc7d,0x03,0x33, 0xcc81,0x29,0xe6, 0xcc83,0x6a,0xfa, 0xcc86,0x2f,0xf7,
		0xcc89,0x05,0x6a, 0xcc8b,0xa1,0x26, 0xcc8d,0xd4,0x33, 0xcc91,0x4e,0x6a,
		0xcc93,0x05,0x27, 0xcc95,0x01,0x20, 0xcc97,0x4f,0x7f, 0xcc9a,0xdf,0x35,
		0xcc9c,0xf1,0x32, 0xcc9e,0x83,0xbe, 0xcca1,0x30,0xa6, 0xcca3,0x9d,0x43,
		0xcca4,0x32,0xa7, 0xcca6,0x83,0x10, 0xccaa,0x7b,0xee, 0xccac,0xb8,0x8e,
		0xccaf,0x07,0xe6, 0xccb1,0x12,0x27, 0xccb3,0xb8,0xc0, 0xccb5,0xcd,0x58,
		0xccb6,0xca,0x58, 0xccb7,0x41,0x3a, 0xccb8,0xe9,0x3a, 0xccb9,0x98,0x86,
		0xccbb,0x83,0xa7, 0xccbd,0xb0,0x86, 0xccbf,0xbf,0xa7, 0xccc1,0x20,0xa6,
		0xccc3,0x6f,0xff, 0xccc6,0x06,0x33, 0xccca,0x51,0x44, 0xcccb,0xb9,0x24,
		0xcccd,0x08,0xe6, 0xcccf,0x1b,0xf4, 0xccd2,0x2f,0xf7, 0xccd5,0x5a,0x6a,
		0xccd7,0x01,0x26, 0xccd9,0xb2,0x33, 0xccdd,0x12,0x6a, 0xccdf,0xa5,0x27,
		0xcce1,0x04,0x20, 0xcce3,0x4d,0x7f, 0xcce6,0x16,0x35, 0xcce8,0xc9,0x32,
		0xccea,0xb8,0xbe, 0xcced,0x98,0xa6, 0xccef,0x51,0x43, 0xccf0,0x32,0xa7,
		0xccf2,0x85,0x10, 0xccf6,0x7b,0xee, 0xccf8,0xb8,0x8e, 0xccfb,0x75,0xe6,
		0xccfd,0xb2,0x27, 0xccff,0x69,0xc0, 0xcd01,0xb6,0x58, 0xcd02,0x32,0x58,
		0xcd03,0xc2,0x3a, 0xcd04,0x5f,0x3a, 0xcd05,0xf7,0x86, 0xcd07,0xc5,0xa7,
		0xcd09,0x48,0x86, 0xcd0b,0x0d,0xa7, 0xcd0d,0x1f,0xa6, 0xcd0f,0xaf,0xff,
		0xcd12,0x6e,0x33, 0xcd16,0x31,0x44, 0xcd17,0xd5,0x25, 0xcd19,0x10,0x33,
		0xcd1d,0x13,0x20, 0xcd1f,0x89,0xe6, 0xcd21,0x1d,0xf4, 0xcd24,0x93,0xf7,
		0xcd27,0x9a,0xff, 0xcd2a,0x17,0x33, 0xcd2e,0x8d,0xe6, 0xcd30,0xbd,0xf4,
		0xcd33,0xb9,0xf7, 0xcd36,0x0f,0x6a, 0xcd38,0x10,0x26, 0xcd3a,0x07,0x33,
		0xcd3e,0xa7,0x6a, 0xcd40,0xee,0x27, 0xcd42,0xf8,0x20, 0xcd44,0xad,0x7f,
		0xcd47,0x77,0x35, 0xcd49,0xa6,0x7f, 0xcd4c,0x7c,0x8e, 0xcd4f,0xb9,0xec,
		0xcd51,0xc5,0x84, 0xcd53,0x91,0xc4, 0xcd55,0x96,0xed, 0xcd57,0xf1,0x8c,
		0xcd5a,0x15,0x25, 0xcd5c,0x3e,0xbf, 0xcd5f,0x69,0x8e, 0xcd62,0x34,0xec,
		0xcd64,0x54,0x84, 0xcd66,0x13,0xc4, 0xcd68,0xbe,0xed, 0xcd6a,0x7c,0x8c,
		0xcd6d,0x16,0x25, 0xcd6f,0x22,0x39, 0xcd70,0xf9,0xbe, 0xcd73,0xbc,0xec,
		0xcd75,0x5d,0x34, 0xcd77,0x30,0x5f, 0xcd78,0xf8,0xce, 0xcd7b,0x1b,0xb7,
		0xcd7e,0x73,0x1f, 0xcd80,0xa0,0x84, 0xcd82,0xc3,0x1e, 0xcd84,0x6a,0xef,
		0xcd86,0x6d,0xef, 0xcd88,0x0b,0xef, 0xcd8a,0x0d,0xef, 0xcd8c,0x4a,0xef,
		0xcd8e,0x4d,0xef, 0xcd90,0x6f,0xef, 0xcd92,0x6d,0xef, 0xcd94,0xa8,0xc3,
		0xcd97,0x34,0xa1, 0xcd99,0xc8,0x23, 0xcd9b,0xdf,0x35, 0xffde,0x2b,0x7e,
		0xffe1,0x59,0x7e, -1
	};

	UINT8 *rom;
	int diff;
	int i;

	/********************************

    The encryption algorithm is not known.

    I kept the two tables separate, but the two CPUs are encrypted identically.
    Addresses that are present in both tables show that when the encrypted value
    is the same, the decrypted value is the same. When one bit changes in the
    encrypted value, one bit changes in the decrypted value, and so on.
    I couldn't find any obvious relationship between values at different addresses.

    One important thing to note is that for 6809 instructions that take two
    opcodes (that is, 10 xx and 11 xx) only the first opcode is encrypted, not the
    second. This is different from how the Konami-1 CPU works, where both opcodes
    are encrypted.

     ********************************/

	i = 0;
	rom = memory_region(REGION_CPU1);
	diff = memory_region_length(REGION_CPU1) / 2;

	memory_set_opcode_base(0,rom+diff);

	memcpy(rom + diff,rom,diff);
	while (cpu0_patch[i] != -1)
	{
		int a = cpu0_patch[i];
		if (rom[a] != cpu0_patch[i+1])
			logerror("error in patch data at address %04x\n",a);
		else
			rom[a + diff] = cpu0_patch[i+2];

		i += 3;
	}

	i = 0;
	rom = memory_region(REGION_CPU2);
	diff = memory_region_length(REGION_CPU2) / 2;

	memory_set_opcode_base(1,rom+diff);

	memcpy(rom + diff,rom,diff);
	while (cpu1_patch[i] != -1)
	{
		int a = cpu1_patch[i];
		if (rom[a] != cpu1_patch[i+1])
			logerror("error in patch data at address %04x\n",a);
		else
			rom[a + diff] = cpu1_patch[i+2];

		i += 3;
	}

	init_kram();
}

static DRIVER_INIT( zookeep )
{
	/* we need to override two PIA handlers to prevent controls from getting disabled */
	memory_install_write8_handler(0, ADDRESS_SPACE_PROGRAM, 0x1400, 0x17ff, 0, 0, zookeep_pia_0_w);
	memory_install_write8_handler(0, ADDRESS_SPACE_PROGRAM, 0x1c00, 0x1fff, 0, 0, zookeep_pia_2_w);
}


static DRIVER_INIT( slither )
{
	memory_install_read8_handler(0, ADDRESS_SPACE_PROGRAM, 0x9800, 0x9bff, 0, 0, pia_1_r);
	memory_install_write8_handler(0, ADDRESS_SPACE_PROGRAM, 0x9800, 0x9bff, 0, 0, pia_1_w);
	memory_install_write8_handler(1, ADDRESS_SPACE_PROGRAM, 0x9401, 0x9401, 0, 0, slither_vram_mask_w);
}



/*************************************
 *
 *  Game drivers
 *
 *************************************/

GAME( 1981, qix,      0,        qix,      qix,      0,        ROT270, "Taito America Corporation", "Qix (set 1)" )
GAME( 1981, qixa,     qix,      qix,      qix,      0,        ROT270, "Taito America Corporation", "Qix (set 2)" )
GAME( 1981, qixb,     qix,      qix,      qix,      0,        ROT270, "Taito America Corporation", "Qix (set 3)" )
GAME( 1981, qix2,     qix,      qix,      qix,      0,        ROT270, "Taito America Corporation", "Qix II (Tournament)" )
GAME( 1981, sdungeon, 0,        mcu,      sdungeon, 0,        ROT270, "Taito America Corporation", "Space Dungeon" )
GAME( 1982, elecyoyo, 0,        elecyoyo, elecyoyo, 0,        ROT270, "Taito America Corporation", "The Electric Yo-Yo (set 1)" )
GAME( 1982, elecyoy2, elecyoyo, elecyoyo, elecyoyo, 0,        ROT270, "Taito America Corporation", "The Electric Yo-Yo (set 2)" )
GAME( 1982, kram,     0,        mcu,      kram,     kram,     ROT0,   "Taito America Corporation", "Kram (set 1)" )
GAME( 1982, kram2,    kram,     mcu,      kram,     kram,     ROT0,   "Taito America Corporation", "Kram (set 2)" )
GAME( 1982, kram3,    kram,     qix,      kram,     kram3,    ROT0,   "Taito America Corporation", "Kram (encrypted)" )
GAME( 1982, zookeep,  0,        zookeep,  zookeep,  zookeep,  ROT0,   "Taito America Corporation", "Zoo Keeper (set 1)" )
GAME( 1982, zookeep2, zookeep,  zookeep,  zookeep,  zookeep,  ROT0,   "Taito America Corporation", "Zoo Keeper (set 2)" )
GAME( 1982, zookeep3, zookeep,  zookeep,  zookeep,  zookeep,  ROT0,   "Taito America Corporation", "Zoo Keeper (set 3)" )
GAME( 1982, slither,  0,        slither,  slither,  slither,  ROT270, "Century II",                "Slither (set 1)" )
GAME( 1982, slithera, slither,  slither,  slither,  slither,  ROT270, "Century II",                "Slither (set 2)" )
GAME( 1984, complexx, 0,        qix,      complexx, 0,        ROT270, "Taito America Corporation", "Complex X" )
