/*
	Input plugin example for Dualis
	Mic, 2005


	Brief description
	-----------------
	* Hooks the Dualis window proc
	* Sends INPUT_KEYDOWN/INPUT_KEYUP messages to Dualis

	The messages are sent through the shell callback, e.g.
		
		  shell(TELL, INPUT_KEYDOWN, KEY_START, 0);

	Key enumerators are defined in plugin.h
	The fourth parameter is reserved for future use and should be set to 0 (zero).
*/


#include "plugin.h"
#include "resource.h"


HWND hwndMain,hwndVideo;	// Window handles
HINSTANCE hInst;			// Instance
DWORD lpfnOldWndProc;		// Pointer to old window proc
shell_callback shell;		// Callback to the shell (calling application)
bool hooked;				// Have we hooked the wndproc or not?
int keymap[256];			// Keyboard map
int i,j;					// Counters

char pluginName[] = "Input plugin";



// Standard entrypoint
bool __stdcall DllMain(HINSTANCE hDLL,DWORD reason,LPVOID unused)
{
	if (reason==DLL_PROCESS_ATTACH)
		hInst = hDLL;

	return true;
}



// WndProc hook
LRESULT CALLBACK wndproc_hook(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	if ((hwnd == hwndMain) || (hwnd == hwndVideo))
	{
		switch (uMsg)
		{
			case WM_KEYDOWN:
				// Signal a keypress
				if (keymap[wParam&0xFF] != -1)
					shell(TELL, INPUT_KEYDOWN, keymap[wParam & 0xFF], 0);
				break;

			case WM_KEYUP:
				// Signal a key being released
				if (keymap[wParam&0xFF] != -1)
					shell(TELL, INPUT_KEYUP, keymap[wParam & 0xFF], 0);
				break;

			case WM_DESTROY:
				// Remove the wndproc hooks, but only do it once
				if (hwnd == hwndMain)
				{
					SetWindowLong(hwndMain, GWL_WNDPROC, (DWORD)lpfnOldWndProc);
					SetWindowLong(hwndVideo, GWL_WNDPROC, (DWORD)lpfnOldWndProc);
					hooked = false;
				}
				break;
		}
	}

	// Call the old WndProc
	return CallWindowProc((WNDPROC)lpfnOldWndProc,hwnd,uMsg,wParam,lParam);
}




// Configuration dialog callback
BOOL CALLBACK DlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	switch(msg)
	{
		case WM_COMMAND:
			switch(LOWORD(wParam))
			{
				case IDOK:
					EndDialog(hdlg, 0);
					return TRUE;

				case IDCANCEL:
					EndDialog(hdlg, 1);
					return FALSE;
				default:
					break;
			}
			break;
   }

	return FALSE;
}




bool __stdcall inp_init(shell_callback lpfnShell)
{
	shell = lpfnShell;

	// Ask for the handles of the main and video output windows
	hwndMain = (HWND)shell(ASK, SHELL_WINDOW_HANDLE, WINDOW_HANDLE_MAIN, 0);
	hwndVideo = (HWND)shell(ASK, SHELL_WINDOW_HANDLE, WINDOW_HANDLE_VIDEO, 0);

	// Invalidate all key mappings
	for (i=0; i<256; i++)
		keymap[i] = -1;

	// Hook the WndProc for both windows. They use the same WndProc, so we only need
	// to save the value once.
	hooked = false;
	if ((lpfnOldWndProc = SetWindowLong(hwndMain, GWL_WNDPROC, (DWORD)wndproc_hook)) == NULL)
		return false;
	if (SetWindowLong(hwndVideo, GWL_WNDPROC, (DWORD)wndproc_hook) == NULL)
		return false;
	hooked = true;

	// Set up some keyboard mappings
	// Use the DPad, X, Y, L, R, Space, Return
	keymap[VK_UP] = KEY_UP;
	keymap[VK_DOWN] = KEY_DOWN;
	keymap[VK_LEFT] = KEY_LEFT;
	keymap[VK_RIGHT] = KEY_RIGHT;
	keymap['Z'] = KEY_B;
	keymap['X'] = KEY_A;
	keymap['L'] = KEY_L;
	keymap['R'] = KEY_R;
	keymap[VK_SPACE] = KEY_SELECT;
	keymap[VK_RETURN] = KEY_START;
	keymap['A'] = KEY_X;
	keymap['S'] = KEY_Y;

	return true;
}


bool __stdcall inp_reset()
{
	return true;
}


// Called by the shell when unloading the plugin
void __stdcall inp_close()
{
	if (hooked)
	{
		SetWindowLong(hwndMain, GWL_WNDPROC, (DWORD)lpfnOldWndProc);
		SetWindowLong(hwndVideo, GWL_WNDPROC, (DWORD)lpfnOldWndProc);
	}
}


// Used to return info to the shell
int __stdcall inp_get(int what)
{
	switch (what)
	{
		case PLUGIN_NAME:
			return (int)&pluginName;
	}

	return 0;
}



// Used to recieve info from the shell
void __stdcall inp_set(int what, DWORD param1, DWORD param2)
{
}



int __stdcall inp_config()
{
	return DialogBoxParam(hInst, MAKEINTRESOURCE(IDD_DIALOG1), hwndMain, DlgProc, NULL);
}


