// DS MMU emulator
// Mic, 2005

#include <windows.h>
#include <commctrl.h>
#include <string.h>
#include "zombie.h"
#include "resource.h"
#include "arm9_read.h"
#include "arm9_write.h"
#include <stdio.h>





touchpad					*touch;

// Timer prescalar
const unsigned char			prescalar[4] = {0,6,8,10};

// DMA maximum transfer counts
const int					dmamax[4] = {0x200000,0x200000,0x200000,0x200000};

const int					keymask[] = {0x040,0x080,0x020,0x010,0x001,0x002,0x200,0x100,0x008,0x004};
int							keycodes[] = {38,40,37,39,88,90,76,82,13,32};
int							btnXY[2];

HINSTANCE					hInst;
HANDLE						hout;

unsigned char				*ROM,*SYSROM,*IWRAM,*EXRAM,*cart,*VRAM,*IOREGS,*mmu_memory,*BIOS,
							*ITCM,*DTCM,*OBJ_VRAM,*OBJ_S_VRAM,*LCDC_VRAM,*PLT_VRAM,*PLT_S_VRAM,
							*S_VRAM,*OAM;
unsigned char				*iomap,**mmap,*arm9_pageTable[256],*arm7_pageTable[256];
unsigned int				arm9_pageLength[16],arm9_pageMask[16],
							arm7_pageLength[16],arm7_pageMask[16];
int							*arm9_pageTableAddr  = (int*)&arm9_pageTable[0],
							*arm9_pageLengthAddr = (int*)&arm9_pageLength[0],
							*arm9_pageMaskAddr   = (int*)&arm9_pageMask[0],
							*arm7_pageTableAddr  = (int*)&arm7_pageTable[0],
							*arm7_pageLengthAddr = (int*)&arm7_pageLength[0],
							*arm7_pageMaskAddr   = (int*)&arm7_pageMask[0];
unsigned int				dtcmPage=0, dtcmBase = 0x800000, itcmBase=0;
dtcmBackup					dtcmLast;



bool						useExtBios = false;
char						szBiosName[] = "bios.bin";
HANDLE						hBios;

unsigned short int			keys,irqFlags,irqAck,irqMask;
int							*arm9_regs,*arm7_regs;
int							romOffset,romSize;
int							iwramOffset,iwramSize;
int							exramOffset,exramSize;
int							vramOffset,vramSize;
char						mmuName[] = "DS MMU";
int							dummy;
unsigned int				*arm9_cycle,*arm7_cycle;
unsigned int				addressLatch;
bool						logTimer=false,logDMA=false;

mmu_read_word_callback		arm9_rd16[256];
mmu_read_dword_callback		arm9_rd32[256];
mmu_write_word_callback		arm9_wr16[256];
mmu_write_dword_callback	arm9_wr32[256];

mmu_read_word_callback		arm7_rd16[256];
mmu_read_dword_callback		arm7_rd32[256];
mmu_write_word_callback		arm7_wr16[256];
mmu_write_dword_callback	arm7_wr32[256];

mmu_read_word_callback		rdh_ipc,rdh_ipc_mirror;

cpu_irq_callback			cpu_irq;
cpu_dma_callback			arm9_dma;
cpu_dma_callback			arm7_dma;
gpu_set_callback			gpu_set;
gpu_get_callback			gpu_get;
apu_write_callback			apu_write;
zombie_callback				zombie;

char						buffer[80];
FILE						*log;
int							blockDMA=0;
dma_channel					dma[4];
timer_channel				timer[4];
SYSTEMTIME					*systime;

vram_bank					vramBank[9];
unsigned char				*vramBase[9];
unsigned char				*vramChunk[1024];
extern int					vramSettings[9][32][3];
unsigned char				*VRAM_A,*VRAM_B,*VRAM_C,*VRAM_D,*VRAM_E,*VRAM_F,*VRAM_G,*VRAM_H,*VRAM_I;

int							ipc_page = 0x20; //0x23;
int							ipc_x=4,ipc_y=6,ipc_z=20;


unsigned int				sysromcode_0[8] = {
		0,			// 00000000
		0,			// 00000004
		0,			// 00000008
		0,			// 0000000C
		0,			// 00000010
		0,			// 00000014
		0xEA000042,	// 00000018: B $128
		0			// 0000001C
};

unsigned int				sysromcode_128[9] = {
		0xE92D500F,	// 00000128: STMFD R13!,{R0-R3,R12,R14}

		//0xe3a03502, // 12C: mov	r3, #0x800000
		0xee193f11,	// 12C: mrc 15,0,r3,c9,c1,{0}
		0xe1a03623, // 130: mov	r3, r3, lsr #12
		0xe1a03603, // 134: mov	r3, r3, lsl #12
		0xe2833901, // 138: add	r3, r3, #0x4000

		0xE28FE000,	// 0000013C: ADD R14,R15,#0
		0xE513F004,	// 00000140: LDR R15,[R3,-4]
		0xE8BD500F,	// 00000144: LDMFD R13!,{R0-R3,R12,R14}
		0xE25EF004	// 00000148: SUBS R15,R14,#4
};




BOOL CALLBACK DlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam);



bool __stdcall DllMain(HINSTANCE hDLL,DWORD reason,LPVOID unused)
{
	if (reason==DLL_PROCESS_ATTACH)
		hInst = hDLL;

	return true;
}





///////////////////////////////////////////////////////////////////////////////////////



/////////////////////////////////////////////////////////////////////////////////////////

// Co-processor division
void cp_div(short int cnt)
{
	static short int divCnt=CP_DIV0;
	int divNumer,divDenom;

	if (cnt != -1) divCnt |= cnt;
	divCnt &= ~CP_BUSY;
	divNumer = *(int*)(IOREGS + REG_NUMER_L);
	divDenom = *(int*)(IOREGS + REG_DENOM_L);
	if (divDenom)
	{
		divCnt &= ~CP_DIV0;
		__asm
		{
			mov eax,divNumer
			cdq
			idiv dword ptr [divDenom]
			mov divNumer,eax
			mov divDenom,edx
		}
		*(int*)(IOREGS + REG_DIVRESULT_L) = divNumer;
		*(int*)(IOREGS + REG_DIVREM_L) = divDenom;
	} else
		divCnt |= CP_DIV0;
	*(short int*)(IOREGS + REG_DIVCNT) = divCnt;
}



// Co-processor square root
void cp_sqrt(short int cnt)
{
	static short int sqrtCnt=0;
	int sqrtParm;

	if (cnt != -1) sqrtCnt |= cnt;
	sqrtCnt &= ~CP_BUSY;
	sqrtParm = *(int*)(IOREGS + REG_SQRTPARM);
		__asm
		{
			fild dword ptr [sqrtParm]
			fsqrt
			fwait
			fistp dword ptr [sqrtParm]
		}
		*(int*)(IOREGS + REG_SQRTRESULT) = sqrtParm;
	*(short int*)(IOREGS + REG_SQRTCNT) = sqrtCnt;
}





void remap_vram(int bank,int setting)
{
	int i,j,k;

	if (vramBank[bank].type != DISABLED)
	{
		j = vramBank[bank].base;
		for (i=0; i<vramBank[bank].size; i++)
			vramChunk[j+i] = 0;
		for (k=0; k<9; k++)
			if ((k!=bank)&&(vramBank[k].base == j)) break;
		if (k<9)
		{
		wsprintf(buffer,"Bank %c has the same base as bank %c\n",
			'A'+(char)k,'A'+(char)bank);
		WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
			for (i=0; i<vramBank[k].size; i++)
				vramChunk[j+i] = vramBase[k] + (i<<14);
		}
	}

	if ((setting&0x80)==0)
	{
		vramBank[bank].type = DISABLED;
		return;
	}

	setting ^= 0x80;	// Clear bit 7

	vramBank[bank].base = vramSettings[bank][setting][0]>>2;
	vramBank[bank].size = vramSettings[bank][setting][1];
	vramBank[bank].type = vramSettings[bank][setting][2];

	if (vramBank[bank].type != DISABLED)
	{
		wsprintf(buffer,"Bank %c: base=%X, size=%X, type=%d\n",
			'A'+(char)bank,vramBank[bank].base,vramBank[bank].size,vramBank[bank].type);
		WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);

		j = vramBank[bank].base;
		for (k=0; k<bank; k++)
			if (vramBank[k].base == j) break;
		if (k==bank)
		{
			for (i=0; i<vramBank[bank].size; i++)
				vramChunk[j+i] = vramBase[bank] + (i<<14);
		}
	} 
}




/////////////////////////////////////////////////////////////////////////////////////////




bool __stdcall mmu_init(zombie_callback commlink)
{
	int i;

	zombie = commlink;

	zombie(TELL,ZOMBIE_DEBUG_MSG,"MMU init",0);

	romOffset = 0x8000000;
	romSize = 0x8000000;

	exramOffset = 0x2000000;
	exramSize = 0x400000;

	iwramOffset = 0x3000000;
	iwramSize = 0x8000;

	vramOffset = 0x6000000;
	vramSize = 0x40000;

	mmu_memory = (unsigned char*)LocalAlloc(LPTR,iwramSize + exramSize + 0x100000 + 0x4000 + 0x4000 + 0x4000 + 0x8000);
	IWRAM = mmu_memory;
	EXRAM = IWRAM + iwramSize;
	SYSROM = EXRAM + exramSize;
	BIOS = SYSROM + 0x100000; //0x4000;
	DTCM = BIOS + 0x4000;
	ITCM = DTCM + 0x8000;
	//IOREGS = (unsigned char*)LocalAlloc(LPTR,0x400);

	if (mmu_memory == NULL)
		return false;

	if (useExtBios)
	{
		if ((hBios = CreateFile(szBiosName,GENERIC_READ,0,NULL,OPEN_EXISTING,
						  FILE_ATTRIBUTE_NORMAL+FILE_FLAG_SEQUENTIAL_SCAN,NULL)) != INVALID_HANDLE_VALUE)
		{
			ReadFile(hBios,BIOS,0x4000,(LPDWORD)&i,NULL);
			if (i==0x4000)
				useExtBios = false;
		} else
			useExtBios = false;
	}


	for (i=0; i<0x10; i++)
	{
		arm9_pageTable[0x30+i] = IWRAM;
		arm9_pageTable[0x20+i] = EXRAM + ((i&3)*0x100000);
		arm9_pageTable[0x00+i] = DTCM; //SYSROM;	// 0x00
		arm9_pageTable[0xF0+i] = SYSROM;
	}
	
	arm9_pageLength[0xF] = 0x100000;

	//arm9_pageTable[0] = 0;
	arm9_pageLength[3] = 0x8000;
	arm9_pageLength[2] = 0x400000;
	arm9_pageLength[0] = 0x4000;

	for (i=0; i<4; i++)
	{
		dma[i].sad = 0xB0 + (i*0xC);
		dma[i].dad = 0xB4 + (i*0xC);
		dma[i].cnt_l = 0xB8 + (i*0xC);
		dma[i].cnt_h = 0xBA + (i*0xC);

		timer[i].enabled = false;
		timer[i].tcurr = 0;
		timer[i].tstart = 0;
	}

	for (i=0; i<256; i++)
	{
		arm9_rd16[i] = (mmu_read_word_callback)arm9_read_word_gen;
		arm9_rd32[i] = (mmu_read_dword_callback)arm9_read_dword_gen;
		arm9_wr16[i] = (mmu_write_word_callback)arm9_write_word_gen;
		arm9_wr32[i] = (mmu_write_dword_callback)arm9_write_dword_gen;
	}
	for (i=0; i<16; i++)
	{
		arm9_rd16[0x80+i] = (mmu_read_word_callback)arm9_read_word_rom;
		arm9_rd16[0x90+i] = (mmu_read_word_callback)arm9_read_word_rom;
		arm9_rd16[0xA0+i] = (mmu_read_word_callback)arm9_read_word_zero;
		arm9_rd16[0x60+i] = (mmu_read_word_callback)arm9_read_word_vram;
		arm9_rd32[0x80+i] = (mmu_read_dword_callback)arm9_read_dword_rom;
		arm9_rd32[0x90+i] = (mmu_read_dword_callback)arm9_read_dword_rom;
		arm9_rd32[0xA0+i] = (mmu_read_dword_callback)arm9_read_dword_zero;
		arm9_rd32[0x60+i] = (mmu_read_dword_callback)arm9_read_dword_vram;

		arm9_wr16[0x00+i] = (mmu_write_word_callback)arm9_write_word_na;
		arm9_wr16[0x80+i] = (mmu_write_word_callback)arm9_write_word_na;
		arm9_wr16[0x90+i] = (mmu_write_word_callback)arm9_write_word_na;
		arm9_wr16[0x60+i] = (mmu_write_word_callback)arm9_write_word_vram;
		arm9_wr32[0x00+i] = (mmu_write_dword_callback)arm9_write_dword_na;
		arm9_wr32[0x80+i] = (mmu_write_dword_callback)arm9_write_dword_na;
		arm9_wr32[0x90+i] = (mmu_write_dword_callback)arm9_write_dword_na;
		arm9_wr32[0x60+i] = (mmu_write_dword_callback)arm9_write_dword_vram;

	arm9_rd16[0x30+i] = (mmu_read_word_callback)arm9_read_word_wram;
	arm9_rd32[0x30+i] = (mmu_read_dword_callback)arm9_read_dword_wram;
	arm9_wr16[0x30+i] = (mmu_write_word_callback)arm9_write_word_wram;
	arm9_wr32[0x30+i] = (mmu_write_dword_callback)arm9_write_dword_wram;
	
	}
	arm9_rd16[ipc_page] = (mmu_read_word_callback)arm9_read_word_ipc;

	rdh_ipc = (mmu_read_word_callback)arm9_read_word_ipc;
	rdh_ipc_mirror = (mmu_read_word_callback)arm9_read_word_ipc_mirror;

	arm9_rd16[0x27] = arm9_rd16[0x23];
	//arm9_rd16[0x37] = (mmu_read_word_callback)arm9_read_word_wram;
	arm9_rd16[0x40] = (mmu_read_word_callback)arm9_read_word_io;
	arm9_rd16[0x50] = (mmu_read_word_callback)arm9_read_word_plt;
	arm9_rd16[0x68] = (mmu_read_word_callback)arm9_read_word_lcdc;
	arm9_rd16[0x70] = (mmu_read_word_callback)arm9_read_word_oam;

	arm9_rd32[0x40] = (mmu_read_dword_callback)arm9_read_dword_io;
	arm9_rd32[0x68] = (mmu_read_dword_callback)arm9_read_dword_lcdc;

	arm9_wr16[0x40] = (mmu_write_word_callback)arm9_write_word_io;
	arm9_wr16[0x50] = (mmu_write_word_callback)arm9_write_word_plt;
	arm9_wr16[0x68] = (mmu_write_word_callback)arm9_write_word_lcdc;
	arm9_wr16[0x70] = (mmu_write_word_callback)arm9_write_word_oam;

	arm9_wr32[0x40] = (mmu_write_dword_callback)arm9_write_dword_io;
	arm9_wr32[0x50] = (mmu_write_dword_callback)arm9_write_dword_plt;
	arm9_wr32[0x68] = (mmu_write_dword_callback)arm9_write_dword_lcdc;
	arm9_wr32[0x70] = (mmu_write_dword_callback)arm9_write_dword_oam;


	hout = GetStdHandle(STD_OUTPUT_HANDLE);
	return true;

}


bool __stdcall mmu_reset()
{
	int i;
	static int dwNull = 0;
	static int ask = ASK;

	zombie(TELL,ZOMBIE_DEBUG_MSG,"MMU reset",(void*)dwNull);

	//cpu_inpb = (cpu_inpb_callback)zombie(ASK,CPU_INPB_CALLBACK,0,0);
	//cpu_outpb = (cpu_outpb_callback)zombie(ASK,CPU_OUTPB_CALLBACK,0,0);

	cpu_irq = (cpu_irq_callback)zombie(ask,CPU_IRQ_CALLBACK,(void*)dwNull,(void*)dwNull);
	arm9_dma = (cpu_dma_callback)zombie(ask,CPU_DMA_CALLBACK,(void*)dwNull,(void*)dwNull);
	//arm7_dma = (cpu_dma_callback)zombie(ask,CPU_DMA_CALLBACK,(void*)dwNull,(void*)1);
	apu_write = (apu_write_callback)zombie(ask,APU_WRITE_CALLBACK,(void*)dwNull,(void*)dwNull);
	arm9_regs = (int*)zombie(ask,CPU_REG,(void*)dwNull,(void*)dwNull);
	//arm7_regs = (int*)zombie(ask,CPU_REG,(void*)dwNull,(void*)1);
	gpu_set = (gpu_set_callback)zombie(ask,GPU_SET_CALLBACK,(void*)dwNull,(void*)dwNull);
	gpu_get = (gpu_get_callback)zombie(ask,GPU_GET_CALLBACK,(void*)dwNull,(void*)dwNull);

	touch = (touchpad*)zombie(ask,TOUCHPAD_DATA,(void*)dwNull,(void*)dwNull);

	cart = (unsigned char*)zombie(ask,ZOMBIE_ROM_BASE,(void*)dwNull,(void*)dwNull);
	if ((int)cart == dwNull)
	{
		zombie(TELL,ZOMBIE_DEBUG_MSG,"MMU: bad ROM handle",(void*)dwNull);
		return false;
	}
	ROM = cart;

	VRAM = (unsigned char*)zombie(ask,GPU_MEMORY,(void*)dwNull,(void*)dwNull);
	if ((int)VRAM == dwNull)
		return false;
	for (i=0; i<9; i++)
		vramBank[i].type = DISABLED;
	for (i=0; i<1024; i++)
		vramChunk[i] = NULL;

	VRAM_A		= VRAM;
	VRAM_B		= VRAM_A + 0x20000;
	VRAM_C		= VRAM_B + 0x20000;
	VRAM_D		= VRAM_C + 0x20000;
	VRAM_E		= VRAM_D + 0x20000;
	VRAM_F		= VRAM_E + 0x10000;
	VRAM_G		= VRAM_F + 0x4000;
	VRAM_H		= VRAM_G + 0x4000;
	VRAM_I		= VRAM_H + 0x8000;
	vramBase[0] = VRAM_A;
	vramBase[1] = VRAM_B;
	vramBase[2] = VRAM_C;
	vramBase[3] = VRAM_D;
	vramBase[4] = VRAM_E;
	vramBase[5] = VRAM_F;
	vramBase[6] = VRAM_G;
	vramBase[7] = VRAM_H;
	vramBase[8] = VRAM_I;
	
	//remap_vram(2,0x84);
	//remap_vram(7,0x81);

	systime = (SYSTEMTIME*)zombie(ask,ZOMBIE_TIME,(void*)dwNull,(void*)dwNull);


	zombie(TELL,ZOMBIE_DEBUG_MSG,"Setting reset vector to 0x02004000",0);

	IOREGS = &VRAM[IOREGS_OFFSET];
	PLT_VRAM = &VRAM[PALETTE_OFFSET];
	OAM = &VRAM[SPRITE_TABLE_OFFSET];

	IOREGS[0x130] = 0xFF;
	keys = 0x3FF;
	btnXY[0] = btnXY[1] = 0;

	for (i=0; i<0x10; i++)
	{
		arm9_pageTable[0x80+i] = ROM + i*0x100000;
		arm9_pageTable[0x90+i] = ROM + i*0x100000;
		arm9_pageTable[0xA0+i] = ROM + i*0x100000;
		arm9_pageTable[0xC0+i] = ROM + i*0x100000;
	}
	arm9_pageTable[0x70] = OAM; //VRAM + 0x84000;
	arm9_pageTable[0x60] = VRAM;
	arm9_pageTable[0x62] = S_VRAM; //VRAM + 0xE8000;
	arm9_pageTable[0x64] = OBJ_VRAM;
	arm9_pageTable[0x66] = OBJ_S_VRAM; //VRAM+0xC8000; 
	arm9_pageTable[0x68] = LCDC_VRAM;
	arm9_pageTable[0x50] = PLT_VRAM; //VRAM + 0x80000;
	arm9_pageTable[0x40] = IOREGS;

	arm9_pageLength[8] = (unsigned int)zombie(ask,ZOMBIE_ROM_SIZE,(void*)dwNull,(void*)dwNull);
	arm9_pageLength[7] = 0x800;
	arm9_pageLength[6] = 0x100000;
	arm9_pageLength[5] = 0x1000;
	arm9_pageLength[4] = 0x2000;

	dtcmPage = 0x08;
	dtcmLast.nPage = -1;
	arm9_pageTable[dtcmPage] = DTCM;
	arm9_pageLength[dtcmPage>>4] = 0x4000;

	arm9_rd16[dtcmPage] = (mmu_read_word_callback)arm9_read_word_dtcm;
	arm9_rd32[dtcmPage] = (mmu_read_dword_callback)arm9_read_dword_dtcm;
	arm9_wr16[dtcmPage] = (mmu_write_word_callback)arm9_write_word_dtcm;
	arm9_wr32[dtcmPage] = (mmu_write_dword_callback)arm9_write_dword_dtcm;

	arm9_rd16[0] = (mmu_read_word_callback)arm9_read_word_itcm;
	arm9_rd32[0] = (mmu_read_dword_callback)arm9_read_dword_itcm;
	arm9_wr16[0] = (mmu_write_word_callback)arm9_write_word_itcm;
	arm9_wr32[0] = (mmu_write_dword_callback)arm9_write_dword_itcm;

	
	for (i=0; i<16; i++)
		arm9_pageMask[i] = arm9_pageLength[i] - 1;
	//pageMask[2] = 0xFFFFF;

	// Align the ROM size mask to the nearest 2^n boundary
	dummy = arm9_pageLength[8];
	__asm
	{
		pusha
		mov eax,[dummy]
		bsr ecx,eax
		inc ecx
		mov eax,1
		shl eax,cl
		dec eax
		mov [dummy],eax
		popa
	}
	arm9_pageMask[8] = dummy;


	// Initialize timers and DMA channels
	for (i=0; i<4; i++)
	{
		timer[i].enabled = false;
		timer[i].countup = false;
		timer[i].tcurr = 0;
		timer[i].tstart = 0;

		dma[i].sad_int = 0;
		dma[i].dad_int = 0;
		dma[i].enabled = false;
	}

	// Clear IWRAM
	memset(IWRAM,0,iwramSize);

	if (useExtBios)
	{
		memcpy(SYSROM,BIOS,0x4000);
	} else
	{
		memcpy(SYSROM,sysromcode_0,8*4);
		memcpy(SYSROM+0x128,sysromcode_128,9*4);

		memcpy(SYSROM+0xF0000,sysromcode_0,8*4);
		memcpy(SYSROM+0xF0000+0x128,sysromcode_128,9*4);
	}

	*(arm9_pageTable[0x23] + 0x7F016) = 5;
	*(arm9_pageTable[0x23] + 0x7F017) = 6;	// y
	*(arm9_pageTable[0x23] + 0x7F018) = 7;	// m
	*(arm9_pageTable[0x23] + 0x7F019) = 8;	// d
	*(arm9_pageTable[0x23] + 0x7F01A) = 9;
	*(arm9_pageTable[0x23] + 0x7F01B) = 1;	// h
	*(arm9_pageTable[0x23] + 0x7F01C) = 2;	// m
	*(arm9_pageTable[0x23] + 0x7F01D) = 3;	// s

	srand(123);

	irqFlags = 0;
	irqAck = 0xFFFF;
	*(int*)(IOREGS + IF_) = 0;
	*(short int*)(arm9_pageTable[0x08] + 0x3FF8) = 0; //0xFFFF;

	return true;
}


void __stdcall mmu_close()
{
	//fclose(log);
	LocalFree(mmu_memory);
}



int __stdcall mmu_config(HWND hwnd)
{
	return DialogBoxParam(hInst,
            MAKEINTRESOURCE(IDD_DIALOG1), hwnd,
            DlgProc, NULL);
}




unsigned char* __stdcall mmu_translate_address(unsigned int address)
{
	return arm9_pageTable[address>>20] + (address&0x0FFFFF);
}



void dma_transfer(void *src,void *dest,int cnt)
{
	memcpy(dest,src,cnt);
}



unsigned short int __stdcall mmu_read_word(unsigned int);
void WRHDECL mmu_write_word(unsigned int,int);
int __stdcall mmu_read_dword(unsigned int);
void WRWDECL mmu_write_dword(unsigned int,int);
void __stdcall mmu_irq(int);


void dma_setup(int channel,bool forceDMA)
{
	int i;
	unsigned int cnt;
	unsigned char sadcnt;
	unsigned short int cnts,cnth;
	unsigned int src,dest,srcend,destend;

	if ( ((IOREGS[dma[channel].cnt_h+1] & 0xB0) == 0x80) ||
		 (((IOREGS[dma[channel].cnt_h+1] & 0xB0) >= 0x80) && forceDMA) )
	{
		if (dma[channel].enabled == false)
		{
			cnts = *(unsigned short*)(IOREGS + dma[channel].cnt_l);
			cnt = cnts;
			cnt += (IOREGS[dma[channel].cnt_h]&0x1F)<<16;
			if ((cnt == 0) || (cnt > dmamax[channel]))
				cnt = dmamax[channel];
			dma[channel].cnt_int = cnt;
		}
		cnt = dma[channel].cnt_int;
		cnt <<= ((IOREGS[dma[channel].cnt_h+1] & 4) == 4) ? 2 : 1;

		src = (*(unsigned int*)(IOREGS + dma[channel].sad)) & 0xFFFFFFF;
		dest = (*(unsigned int*)(IOREGS + dma[channel].dad)) & 0xFFFFFFF;
		if (dma[channel].enabled == false)
		{
			dma[channel].sad_int = src;
			dma[channel].dad_int = dest;
		}
		dma[channel].enabled = true;

		sadcnt = IOREGS[dma[channel].cnt_h] & 0x80;
		sadcnt |= IOREGS[dma[channel].cnt_h+1] & 0x01;

		cnth = IOREGS[dma[channel].cnt_h] + (IOREGS[dma[channel].cnt_h+1]<<8);

		if ((dma[channel].dad_int >= 0x6800000)&&
		(dma[channel].dad_int < 0x7000000)) {
		/*	if (0) {
		wsprintf(buffer,"DMA%d %d bytes from %08x to %08x, CNT = %08x\n",
			channel,cnt,dma[channel].sad_int,dma[channel].dad_int,*(int*)&IOREGS[dma[channel].cnt_l]);
		WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
		}*/
		}

		if (sadcnt == 0x00)
		{
			/*srcend = (dma[channel].sad_int + cnt) & 0xFFFFFF;
			if (srcend >= pageLength[dma[channel].sad_int >> 24])
				cnt -= (srcend - pageLength[dma[channel].sad_int >> 24]);
			if (cnt<=0) return;*/
		}


		//dma_transfer((void*)mmu_translate_address(src),(void*)mmu_translate_address(dest),cnt);

		if (blockDMA == 0)
		{
		if ((IOREGS[dma[channel].cnt_h+1] & 4) == 4)
		{
			if (sadcnt == 0x00)
			{
				if ((int)arm9_dma == 0)
				{
					for (i=0; i<cnt; i+=4)
							mmu_write_dword(dma[channel].dad_int+i,mmu_read_dword(dma[channel].sad_int+i));
				} else
					arm9_dma((char*)dma[channel].dad_int,(char*)dma[channel].sad_int,cnt,4,4);	
			} else if (sadcnt == 0x01)
			{
				for (i=0; i<cnt; i+=4)
					arm9_wr32[((dma[channel].dad_int+i)>>20)&0xFF](dma[channel].dad_int+i,
						arm9_rd32[((dma[channel].sad_int)>>20)&0xFF](dma[channel].sad_int));
			}
		} else
		{
			if (sadcnt == 0x00)
			{
				if ((int)arm9_dma == 0)
				{
					for (i=0; i<cnt; i+=2)
						arm9_wr16[((dma[channel].dad_int+i)>>20)&0xFF](dma[channel].dad_int+i,
							           arm9_rd16[((dma[channel].sad_int+i)>>20)&0xFF](dma[channel].sad_int+i));
				} else
					arm9_dma((char*)dma[channel].dad_int,(char*)dma[channel].sad_int,cnt,2,2);
			} else if (sadcnt == 0x01)
			{
				for (i=0; i<cnt; i+=2)
					arm9_wr16[((dma[channel].dad_int+i)>>20)&0xFF](dma[channel].dad_int+i,
					               arm9_rd16[((dma[channel].sad_int)>>20)&0xFF](dma[channel].sad_int));
			}
		}
		}

		switch (IOREGS[dma[channel].cnt_h] & 0x60)
		{
			case 0x00:
				dma[channel].dad_int += cnt;
				break;
			case 0x20:
				dma[channel].dad_int -= cnt;
				break;
			default:
				break;
		}

		switch (sadcnt)
		{
			case 0x00:
				dma[channel].sad_int += cnt;
				break;
			case 0x80:
				dma[channel].sad_int -= cnt;
				break;
			default:
				break;
		}

		// Reset the DMA enable bit unless repeat mode is specified
		if ((IOREGS[dma[channel].cnt_h+1] & 0x02) == 0)
		{
			IOREGS[dma[channel].cnt_h+1] &= 0x7F;
			dma[channel].enabled = false;
		}

		// Generate an IRQ if needed
		if (IOREGS[dma[channel].cnt_h+1] & 0x40)
			if (IOREGS[IE+1]&(1<<channel))
				mmu_irq(0x100<<channel);
	} else
		dma[channel].enabled = false;
}



void timer_setup(int channel)
{
	timer[channel].tstart = *arm9_cycle;
	timer[channel].countup = ((IOREGS[TM0CNT+channel*4]&0x4)!=0);
	timer[channel].td = *(unsigned short int*)&IOREGS[TM0D + channel*4];

	if ((IOREGS[TM0CNT+channel*4]&0x80)&&(!timer[channel].enabled))
	{
		timer[channel].tcurr = timer[channel].td;
		timer[channel].te = timer[channel].td;
	}

	timer[channel].enabled = false;
	if ((IOREGS[TM0CNT+channel*4]&0x40) &&
		((IOREGS[TM0CNT+channel*4]&0x04)==0))
			zombie(TELL,ZOMBIE_EVENT,(void*)(EV_TIMER_OVERFLOW|(channel<<16)),
				   (void*)((*arm9_cycle)+((0x10000-timer[channel].tcurr)<<prescalar[IOREGS[TM0CNT+channel*4]&3])));
	else if ((channel<3) && ((IOREGS[TM0CNT+channel*4]&0x80)!=0))
		if (timer[channel+1].countup)
			zombie(TELL,ZOMBIE_EVENT,(void*)(EV_TIMER_OVERFLOW|(channel<<16)),
				   (void*)((*arm9_cycle)+((0x10000-timer[channel].tcurr)<<prescalar[IOREGS[TM0CNT+channel*4]&3])));

	if ((channel!=0) && ((IOREGS[TM0CNT+channel*4]&0x04)!=0))
	{
		if (timer[channel-1].enabled)
		{
		//WriteConsole(hout,"Starting countup timer\n",24,(LPDWORD)dummy,NULL);
			zombie(TELL,ZOMBIE_EVENT,(void*)(EV_TIMER_OVERFLOW|((channel-1)<<16)),
			   (void*)(timer[channel-1].tstart + ((0x10000-timer[channel-1].td)<<prescalar[IOREGS[TM0CNT+(channel-1)*4]&3])));
		}
	}
	if (IOREGS[TM0CNT+channel*4]&0x80)
		timer[channel].enabled = true;
}




void __stdcall mmu_irq(int mask)
{
	int i;

	if (mask & 2)
	{
		irqFlags |= 2;
		for (i=1; i<4; i++)
			if ((IOREGS[dma[i].cnt_h+1] & 0xB0) == 0xA0)
				dma_setup(i,true);
	}

	if (mask & 1)
	{
		irqFlags |= 1;
		for (i=1; i<4; i++)
			if ((IOREGS[dma[i].cnt_h+1] & 0xB0) == 0x90)
				dma_setup(i,true);
	}

	if (mask & 0x08)
	{
		irqFlags |= 0x08;
		if ((IOREGS[TM0CNT]&0x4)==0)
		{
			timer[0].enabled = false;
			timer_setup(0);
		}
		cpu_irq(0x0000018);
	}
	if (mask & 0x10)
	{
		//WriteConsole(hout,"IRQ on TM1\n",12,(LPDWORD)dummy,NULL);
		irqFlags |= 0x10;
		if ((IOREGS[TM1CNT]&0x4)==0)
		{
			timer[1].enabled = false;
			timer_setup(1);
		}
		cpu_irq(0x0000018);
	}
	if (mask & 0x20)
	{
		irqFlags |= 0x20;
		if ((IOREGS[TM2CNT]&0x4)==0)
		{
			timer[2].enabled = false;
			timer_setup(2);
		}
		cpu_irq(0x0000018);
	}
	if (mask & 0x40)
	{
		irqFlags |= 0x40;
		if ((IOREGS[TM3CNT]&0x4)==0)
		{
			timer[3].enabled = false;
			timer_setup(3);
		}
		cpu_irq(0x0000018);
	}

	// DMA
	if (mask & 0x400)
	{
		irqFlags |= 0x400;
		cpu_irq(0x0000018);
	}
	if (mask & 0x800)
	{
		irqFlags |= 0x800;
		cpu_irq(0x0000018);
	}

}



unsigned char __stdcall mmu_read_byte(unsigned int address)
{
	unsigned char n;

	__asm
	{
		mov ebx,address
		mov edx,arm9_pageTableAddr
		mov edi,arm9_pageLengthAddr
		mov esi,ebx
		mov ecx,arm9_pageMaskAddr
		shr ebx,20			// page
		and esi,0x0FFFFF	// offset
		mov eax,ebx
		shr eax,4

		// ROM reads
		cmp eax,8
		jne __rb_maybe_6
		//jne __rb_normal

		cmp esi,[edi+eax*4]
		jge __rb_random
		add esi,[edx+ebx*4]
		mov eax,[esi]
		jmp __rb_done

__rb_maybe_6:
		cmp eax,0x6
		jne __rb_maybe_A
		xor eax,eax
		jmp __rb_done

		// Save RAM reads
__rb_maybe_A:
		cmp eax,0xA
		jne __rb_normal
		xor eax,eax
		jmp __rb_done

		// Return "random" data
__rb_random:
		mov ecx,ebp
		mov eax,address
		ror eax,cl
		jmp __rb_done

__rb_normal:
		and esi,[ecx+eax*4]		// AND with pageMask
		add esi,[edx+ebx*4]		// ADD pageTable[page]
		mov eax,[esi]

__rb_done:
		//and eax,0xFF
		mov n,al
	}

	return n;
}


unsigned char __stdcall mmu_read_byte_old(unsigned int address)
{
	unsigned int page = address>>20,offset; //=address&0xFFFFFF;

		if ((address>=0x237f016)&&(address<0x237f01e))
		{
			wsprintf(buffer,"Rb [%08X]\n",address);
			WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
		}

	if (arm9_pageTable[page])
		//if (pageLength[page]>offset)
			return *(unsigned char*)(arm9_pageTable[page] + ((address&0x0FFFFF)&arm9_pageMask[page]));

	return rand()&0xFF;
}




unsigned short int __stdcall mmu_read_word(unsigned int address)
{
	unsigned int page = address>>24,pagen=address>>20,offset=address&0x0FFFFF;
	unsigned int tlong;
	unsigned short int *pagew;
	unsigned short int td;


	if (pagen == ipc_page)
	{
		if (offset == ipc_x)
			return (touch->x*14);
		if (offset== ipc_y)
			return (touch->y*8917)>>9;
		if (offset == ipc_z)
			return (touch->z&0x40)^0x40;
	}

	if (page == 4)
		switch (offset)
		{
			case P1:
				return keys;

			case TM0D:
				if (timer[0].enabled)
				{
					tlong = *arm9_cycle;
					if (tlong < timer[0].tstart)
						tlong -= (timer[0].tstart-0x40000000);
					else
						tlong -= timer[0].tstart;
					timer[0].tcurr = tlong >> prescalar[IOREGS[TM0CNT]&3];
				}
				return timer[0].tcurr;

			case TM1D:
				if (timer[1].countup)
					return timer[1].te;
				if (timer[1].enabled)
				{
					tlong = *arm9_cycle;
					if (tlong < timer[1].tstart)
						tlong -= (timer[1].tstart-0x40000000);
					else
						tlong -= timer[1].tstart;
					timer[1].tcurr = tlong >> prescalar[IOREGS[TM1CNT]&3];
				}
				return timer[1].tcurr;

			case TM2D:
				if (timer[2].countup)
					return timer[2].te;
				if (timer[2].enabled)
				{
					tlong = *arm9_cycle;
					if (tlong < timer[2].tstart)
						tlong -= (timer[2].tstart-0x40000000);
					else
						tlong -= timer[2].tstart;
					timer[2].tcurr = tlong >> prescalar[IOREGS[TM2CNT]&3];
				}
				return timer[2].tcurr;

			case TM3D:
				if (timer[3].countup)
					return timer[3].te;
				if (timer[3].enabled)
				{
					tlong = *arm9_cycle;
					if (tlong < timer[3].tstart)
						tlong -= (timer[3].tstart-0x40000000);
					else
						tlong -= timer[3].tstart;
					timer[3].tcurr = tlong >> prescalar[IOREGS[TM3CNT]&3];
				}
				return timer[3].tcurr;

			case IF_:
				//wsprintf(buffer,"Reading IF (%04x)\n",irqFlags&0xFFFF);
				//WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
				return irqFlags;

			case 0x600:
				gpu_get(GPU_3D_COMMAND);
				//wsprintf(buffer,"GXSTAT = %04X\n",*(unsigned short int*)(IOREGS + 0x600));
				//WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
				return *(unsigned short int*)(IOREGS + 0x600);

			case 0x10:
			case 0x12:
			case 0x14:
			case 0x16:
			case 0x18:
			case 0x1A:
			case 0x1C:
			case 0x1E:
				page = arm9_regs[15];
				pagew = (unsigned short int*)arm9_pageTable[page>>20];
				offset = (page&0x0FFFFF)>>1;
				return pagew[offset];

			default:
				pagew = (unsigned short int*)arm9_pageTable[0x40];
				return pagew[offset>>1];
		}


	if (page == 8)
		if (offset >= arm9_pageLength[0x80])
		{
			page = arm9_regs[15]-4;
			pagew = (unsigned short int*)arm9_pageTable[page>>20];
			offset = (page&0x0FFFFF)>>1;
			return pagew[offset];
		}

	if (arm9_pageTable[pagen])
		return *(unsigned short int*)(arm9_pageTable[pagen] + offset);

	return rand()&0xFFFF;
}



int __stdcall mmu_read_dword(unsigned int address)
{
	int n;


	if (address == 0x4000600)
		gpu_get(GPU_3D_COMMAND);


	/*if ((address&0xF00000) == 0x0800000)
	{
		wsprintf(buffer,"Reading from %08X\n",address);
		WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
	}*/

	__asm
	{
		mov ebx,address
		mov edx,arm9_pageTableAddr
		mov edi,arm9_pageLengthAddr
		mov esi,ebx
		mov ecx,arm9_pageMaskAddr
		shr ebx,20			// page
		and esi,0x0FFFFF	// offset
		mov eax,ebx
		shr eax,4

		// ROM reads
		cmp eax,8
		jne __rd_maybe_4
		
		cmp esi,[edi+eax*4]
		jge __rd_random
		test esi,3
		jnz __rd_unaligned
		add esi,[edx+ebx*4]
		mov eax,[esi]
		jmp __rd_done

		// I/O reg reads
__rd_maybe_4:
		cmp eax,4
		jne __rd_normal //__rd_maybe_6
		cmp esi,P1
		jne __rd_normal
		mov ebx,[IOREGS + P1CNT]
		movzx eax,keys
		shl ebx,16
		or eax,ebx
		jmp __rd_done

		// VRAM reads
__rd_maybe_6:
/*		cmp ecx,6
		jne __rd_maybe_0
		
		and esi,0x7FFFF
		//cmp esi,0x17FFF
		//jng __rd_offset_ok
		//sub esi,0x8000		// Mirror upper 32kB
__rd_offset_ok:
		test esi,3
		jnz __rd_unaligned
		add esi,[edx+ebx*4]
		mov eax,[esi]
		jmp __rd_done*/

		// DTCM reads
/*__rd_maybe_0:
		cmp ebx,0
		jne __rd_normal
		test esi,0x800000
		jz __rd_random
		sub esi,0x800000
		mov ebx,[DTCM]
		mov eax,[ebx+esi]
		jmp __rd_done*/

		// Unaligned read, rotate 16 bits
__rd_unaligned:
		mov ecx,esi
		and esi,0xFFFFFFFC		// Clear lower 2 bits of offset
		and ecx,3
		add esi,[edx+ebx*4]		// ADD pageTable[page]
		shl ecx,3
		mov eax,[esi]
		ror eax,cl				// Rotate value by (Offset & 3)*8
		jmp __rd_done

		// Return "random" data
__rd_random:
		mov ecx,ebp
		mov eax,address
		ror eax,cl
		jmp __rd_done

__rd_normal:
		test esi,3
		jnz __rd_unaligned
		and esi,[ecx+eax*4]		// AND with pageMask
		add esi,[edx+ebx*4]		// ADD pageTable[page]
		mov eax,[esi]

__rd_done:
		mov n,eax
	}


	return n;
}





void __stdcall mmu_write_byte(unsigned int address,int byte)
{
	unsigned int page = (address>>24)&0x0F,pagen=address>>20,offset=address&0x0FFFFF;

	/*if (pagen == 0x08)
	{
		wsprintf(buffer,"Writing %02X to %08X from %08X\n",byte&0xFF,address,cpu_regs[15]);
		WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
	}*/

	/*if (address == 0x02199220) {
		wsprintf(buffer,"[%08X] = %02X\n",address,byte&0xFF);
		WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
	}*/

	if (arm9_pageTable[pagen])
		if (arm9_pageLength[page]>offset)
			*(char*)(arm9_pageTable[pagen] + offset) = (char)byte;

		if (pagen==0x40)
		{
			//if (offset == IF_)
			//	irqFlags &= ((char)byte ^ 0xFF);

			/*if (offset<=0x308)
			{
				wsprintf(buffer,"[%04X] = %02X\n",offset,byte);
				WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
			}*/

			switch (offset)
			{
				case IF_:
					irqFlags &= ((char)byte ^ 0xFF);
					break;

				case 0x240:
				case 0x241:
				case 0x242:
				case 0x243:
				case 0x244:
				case 0x245:
				case 0x246:
					remap_vram(offset-0x240,byte&0xFF);
					//wsprintf(buffer,"VRAMCNT_%c: %08X\n",offset-1023,byte);
					//WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
					break;
				case 0x248:
				case 0x249:
					remap_vram(offset-0x241,byte&0xFF);
					break;
			}
		}

	//if ((page==4)&&(offset>=NR10)&&(offset<=NR52))
	//	apu_write(offset,byte&0xFF);

}






void WRHDECL mmu_write_word(unsigned int address,int word)
{
	unsigned int page = address>>24,pagen=address>>20,offset=address&0x0FFFFF;
	unsigned short int *pagew;


	if (arm9_pageTable[pagen])
		if (arm9_pageLength[page]>offset)
			*(unsigned short int*)(arm9_pageTable[pagen] + offset) = (unsigned short int)word;

	/*if (pagen == 0x08)
	{
		wsprintf(buffer,"[%08X] = %04X\n",(pagen<<20)+offset,mmu_read_word(address)&0xFFFF);
		WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
	}*/

	if (page == 4)
	{
		/*if ((offset>=NR10)&&(offset<=NR52))
		{
			apu_write(offset,word&0xFF);
			apu_write(offset+1,(word>>8)&0xFF);
		}*/


		switch (offset)
		{
			/*case 0x04:
				wsprintf(buffer,"DISPSTAT %04X\n",word);
				WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
				break;

			case 0x08:
			case 0x0A:
			case 0x0C:
			case 0x0E:
				wsprintf(buffer,"BG%dCNT %04X\n",(offset-0x08)>>1,word);
				WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
				break;
			case 0x12:
			case 0x16:
			case 0x1A:
			case 0x1E:
				wsprintf(buffer,"BG%dY %04X\n",(offset-0x12)>>2,word);
				WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
				break;*/

			case REG_DIVCNT:
				cp_div(word&3);
				break;

			case REG_SQRTCNT:
				cp_sqrt(word&3);
				break;
				
			case 0xC6:
				dma_setup(1,false);
				break;
			case 0xD2:
				dma_setup(2,false);
				break;
			case 0xDE:
				dma_setup(3,false);
				break;
			case IF_:
				//wsprintf(buffer,"Writing %04x to IF",word&0xFFFF);
				//zombie(TELL,ZOMBIE_DEBUG_MSG,buffer,0);
				irqFlags &= (word^0xFFFF);
				break;
			case TM0D:
				//timer[0].tcurr = word&0xFFFF;
				break;
			case TM1D:
				//timer[1].tcurr = word&0xFFFF;
				break;
			case TM2D:
				//timer[2].tcurr = word&0xFFFF;
				break;
			case TM3D:
				//timer[3].tcurr = word&0xFFFF;
				break;
			case TM0CNT:
				timer_setup(0);
				break;
			case TM1CNT:
				timer_setup(1);
				break;
			case TM2CNT:
				timer_setup(2);
				break;
			case TM3CNT:
				timer_setup(3);
				break;

			default:
				break;
		}
	}
}



void WRWDECL mmu_write_dword(unsigned int address,int dwrd)
{
	unsigned int page = address>>24,pagen=address>>20,offset=address&0x0FFFFC;
	//int *paged = (int*)(pageTable[pagen]+offset);

	if (arm9_pageTable[pagen])
		if (arm9_pageLength[page]>offset)
			*(int*)(arm9_pageTable[pagen] + offset) = dwrd;

	/*if (page == 6)
	{
		wsprintf(buffer,"[%08X] = %08X at %08X\n",address,dwrd,cpu_regs[15]);
		WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
	}*/

	if (pagen != 0x40) return;

	//if (pagen == 0x40)
	//{
		if ((offset>=0x400)&&(offset<=0x600))
		{
			//wsprintf(buffer,"[%04X] = %04X\n",offset&0xFFFF,dwrd);
			//WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
			gpu_set(GPU_3D_COMMAND,offset,dwrd);
		}
		
		switch (offset) 
		{
			/*case 0x600:
				wsprintf(buffer,"3D: %08X (PC=%08X) %04X (%08X)\n",dwrd,cpu_regs[15],pageLength[page],*paged);
				WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
				break;
			/*case 0x1000:
				wsprintf(buffer,"DISPCNT_S: %08X\n",dwrd);
				WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
				break;*/
			/*case 0x00:
				wsprintf(buffer,"DISPCNT %08X\n",dwrd);
				WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
				break;

			case 0x240:
			case 0x241:
			case 0x242:
			case 0x243:
				wsprintf(buffer,"VRAMCNT_%c: %08X\n",offset-175,dwrd);
				WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
				break;*/

			case REG_NUMER_L:
			case REG_DENOM_L:
				cp_div(-1);
				break;
			case REG_SQRTPARM:
				cp_sqrt(-1);
				break;

			case TM0D:
				timer_setup(0);
				break;
			case TM1D:
				timer_setup(1);
				break;
			case TM2D:
				timer_setup(2);
				break;
			case TM3D:
				timer_setup(3);
				break;

			//case IE:
				//wsprintf(buffer,"Writing %08x to IE",dwrd);
				//zombie(TELL,ZOMBIE_DEBUG_MSG,buffer,0);
			//	irqFlags ^= (dwrd>>16);
			//	break;
			case IF_:
				//wsprintf(buffer,"Writing %08x to IF",dwrd);
				//zombie(TELL,ZOMBIE_DEBUG_MSG,buffer,0);
				irqFlags &= (dwrd^0xFFFF);
				break;

			case 0xC4:
				dma_setup(1,false);
				break;
			case 0xC6:
				dma_setup(1,false);
				break;
			case 0xD0:
				dma_setup(2,false);
				break;
			case 0xD2:
				dma_setup(2,false);
				break;
			/*case 0xD4:
				wsprintf(buffer,"DM3SAD = %08x\n",dwrd);
				WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
				break;*/
			case 0xD8:
				//wsprintf(buffer,"DM3DAD = %08X (PC = %08X)\n",dwrd,cpu_regs[15]);
				//WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
				//wsprintf(buffer,"%08X: %08X %08X %08X\n",cpu_regs[13],
				//	mmu_read_dword(cpu_regs[13]),mmu_read_dword(cpu_regs[13]+4),mmu_read_dword(cpu_regs[13]+8));
				//WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
				break;
			case 0xDC:
				dma_setup(3,false);
				break;
			case 0xDE:
				dma_setup(3,false);
				break;
			default:
				break;
		}
//	}
}



int __stdcall mmu_get(int what)
{
	switch (what)
	{
		case MMU_DTCM:
			return (int)DTCM;
		case MMU_ITCM:
			return (int)ITCM;
		case MMU_IRQ_OK:
			return (irqFlags&0xFFFF)&irqAck;
		case MMU_READ_BYTE_CALLBACK:
			return (int)mmu_read_byte;
		case MMU_READ_WORD_CALLBACK:
			return (int)mmu_read_word;
		case MMU_READ_WORD_CALLBACKS:
			return (int)&arm9_rd16[0];
		case MMU_READ_DWORD_CALLBACKS:
			return (int)&arm9_rd32[0];
		case MMU_READ_DWORD_CALLBACK:
			return (int)mmu_read_dword;
		case MMU_WRITE_BYTE_CALLBACK:
			return (int)mmu_write_byte;
		case MMU_WRITE_WORD_CALLBACK:
			return (int)mmu_write_word;
		case MMU_WRITE_WORD_CALLBACKS:
			return (int)&arm9_wr16[0];
		case MMU_WRITE_DWORD_CALLBACKS:
			return (int)&arm9_wr32[0];
		case MMU_WRITE_DWORD_CALLBACK:
			return (int)mmu_write_dword;
		case MMU_TRANSLATE_ADDRESS_CALLBACK:
			return (int)mmu_translate_address;
		case MMU_IRQ_CALLBACK:
			return (int)mmu_irq;
		case MMU_GET_CALLBACK:
			return (int)mmu_get;
		case MMU_PAGE_TABLE:
			return (int)&arm9_pageTable[0];
		case MMU_VRAM_BANKS:
			return (int)&vramBank[0];
		case PLUGIN_NAME:
			return (int)&mmuName;
		case 0x700:
			return (int)mmu_translate_address((unsigned int)0x8000288);
		default:
			break;
	}
	return 0;
}


void __stdcall mmu_set(int what,void* param1,void* param2)
{
	static char buffer[64];
	int i,n;
	unsigned short int s;

	switch (what)
	{
		case CPU_CYCLE_PTR:
			arm9_cycle = (unsigned int *)param1;
			break;

		case MMU_TIMER_OVERFLOW:
			n = ((int)param1)>>16;

			//wsprintf(buffer,"Overflow on timer %d\n",n);
			//WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);

			if (IOREGS[TM0CNT+n*4]&0x40)
				mmu_irq(0x08<<n);
			else
				timer_setup(n);

			if (n<3)
				if (IOREGS[TM0CNT+((n+1)*4)]&0x04)
				{
					//wsprintf(buffer,"Doing count-up on timer %d (%d)\n",n+1,timer[n+1].tcurr+1);
					//WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);

					timer[n+1].te++;
					if (timer[n+1].te==0)
					{
					//wsprintf(buffer,"Timer %d overflowed, reloading with %04x\n",n+1,timer[n+1].td);
					//WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);

						timer[n+1].te = timer[n+1].td;
						if (IOREGS[TM0CNT+((n+1)*4)]&0x40)
							mmu_irq(0x08<<(n+1));
					}
				}
			break;

		case MMU_IRQ:
			irqFlags |= (int)param1;
			irqMask = irqFlags;
			break;

		case MMU_IRQ_OVER:
			s = *(unsigned short int*)(arm9_pageTable[dtcmPage] + 0x3FF8);

			//wsprintf(buffer,"ACK: %04X, MSK: %04X, s: %04X",irqAck,irqMask,s);
			//zombie(TELL,ZOMBIE_DEBUG_MSG,buffer,0);

			if ((irqMask & s) != irqMask)
				irqAck &= ((irqMask^0xFFFF)|s);
			//*(unsigned short int*)(pageTable[dtcmPage] + 0x3FF8) = 0;
			break;

		case MMU_TCM_REMAP:
			if (param1==0)
			{
				//pageTable[dtcmPage<<4] = 0;
				dtcmBase = (unsigned int)param2&0xFFFFF000;
				if (((int)param2>>20) != dtcmPage)
				{
					if (dtcmLast.nPage != -1)
					{
						arm9_wr16[dtcmPage] = dtcmLast.wrh;
						arm9_rd16[dtcmPage] = dtcmLast.rdh;
						arm9_wr32[dtcmPage] = dtcmLast.wrw;
						arm9_rd32[dtcmPage] = dtcmLast.rdw;
						arm9_pageTable[dtcmPage] = dtcmLast.physaddr;
						arm9_pageLength[dtcmPage] = dtcmLast.length;
						arm9_pageMask[dtcmPage] = dtcmLast.mask;
					}
					dtcmPage = (int)param2>>20;

					dtcmLast.nPage = dtcmPage;
					dtcmLast.wrh = arm9_wr16[dtcmPage];
					dtcmLast.rdh = arm9_rd16[dtcmPage];
					dtcmLast.wrw = arm9_wr32[dtcmPage];
					dtcmLast.rdw = arm9_rd32[dtcmPage];
					dtcmLast.physaddr = arm9_pageTable[dtcmPage];
					dtcmLast.length = arm9_pageLength[dtcmPage];
					dtcmLast.mask = arm9_pageMask[dtcmPage];

					arm9_rd16[dtcmPage] = (mmu_read_word_callback)arm9_read_word_dtcm;
					arm9_rd32[dtcmPage] = (mmu_read_dword_callback)arm9_read_dword_dtcm;
					arm9_wr16[dtcmPage] = (mmu_write_word_callback)arm9_write_word_dtcm;
					arm9_wr32[dtcmPage] = (mmu_write_dword_callback)arm9_write_dword_dtcm;
					wsprintf(buffer,"Remapping DTCM to 0x%08X",(int)param2&0xFFFFF000);
					zombie(TELL,ZOMBIE_DEBUG_MSG,buffer,0);
					//pageTable[(int)param2>>20] = DTCM;
					//pageLength[(int)param2>>24] = 0x4000;
					//pageMask[(int)param2>>24] = 0x3FFF;
				}
			}
			break;

		case ZOMBIE_KEYSTATE:
			if ((int)param2==1)
			{
				/*for (i=0; i<10; i++)
					if ((int)param1 == keycodes[i])
					{
						keys &= (keymask[i] ^ 0x3FF);
						break;
					}*/
				i = (int)param1;
				if ((i>=0)&&(i<10))
					keys &= (keymask[i] ^ 0x3FF);
				else if ((i>=10) && (i<12))
					btnXY[i-10] = 1;	

			} else
			{
				/*for (i=0; i<10; i++)
					if ((int)param1 == keycodes[i])
					{
						keys |= (keymask[i]);
						break;
					}*/
				i = (int)param1;
				if ((i>=0)&&(i<10))
					keys |= keymask[i];
				else if ((i>=10) && (i<12))
					btnXY[i-10] = 0;	


			}
			if (!IOREGS) return;

			if (IOREGS[IME]&1)
			if (IOREGS[IE+1]&0x10)
				if (IOREGS[P1CNT+1]&0x40)
					if (IOREGS[P1CNT+1]&0x80)
					{
					// Logical multiplication (AND)
						if (((keys^0xFFFF)&0x3FF) == (IOREGS[P1CNT]&0x3FF))
						{
							//IOREGS[0x201] |= 0x20;
							irqFlags |= 0x1000;
							cpu_irq(0x00000018);
						}
					}
					//else if ((keys&IOREGS[P1CNT])!=IOREGS[P1CNT])
					// Logical addition (OR)
					else if ((keys^0xFFFF)&(IOREGS[P1CNT]&0x3FF))
					{
						irqFlags |= 0x1000;
						cpu_irq(0x00000018);
					}
			break;

		case PLUGIN_SETTING:
			switch ((int)param1)
			{
				case 0:
				case 1:
				case 2:
				case 3:
				case 4:
				case 5:
				case 6:
				case 7:
				case 8:
				case 9:
					keycodes[(int)param1] = (int)param2;
					break;
				
				case 10:
					arm9_rd16[ipc_page] = (mmu_read_word_callback)arm9_read_word_gen;

					ipc_page = 0x20 + ((int)param2*3);
					ipc_x = 0x04 + ((int)param2*0x7f000);	// 0x7f000
					ipc_y = 0x06 + ((int)param2*0x7f000);
					ipc_z = 0x14 + ((int)param2*0x7f000);
					arm9_rd16[ipc_page] = rdh_ipc;
					//arm9_rd16[0x27] = arm9_rd16[0x23];
					break;
				
				case 11:
					if (param2)
					{
						arm9_rd16[0x27] = rdh_ipc_mirror;
					} else
					{
						arm9_rd16[0x27] = (mmu_read_word_callback)arm9_read_word_gen;
					}
					zombie(TELL, MMU_READ_WORD_CALLBACKS, (void*)arm9_rd16, 0);
					break;

				case 20:
					if ((int)param2 == 1)
					{
						rdh_ipc = (mmu_read_word_callback)arm9_read_word_ipc;
						rdh_ipc_mirror = (mmu_read_word_callback)arm9_read_word_ipc_mirror;

						arm9_rd16[ipc_page] = (mmu_read_word_callback)arm9_read_word_ipc;
						if (arm9_rd16[0x27] != (mmu_read_word_callback)arm9_read_word_gen)
							arm9_rd16[0x27] = (mmu_read_word_callback)arm9_read_word_ipc_mirror;
					} else if ((int)param2 == 2)
					{
						rdh_ipc = (mmu_read_word_callback)arm9_read_word_ipc_v2;
						rdh_ipc_mirror = (mmu_read_word_callback)arm9_read_word_ipc_mirror_v2;

						arm9_rd16[ipc_page] = (mmu_read_word_callback)arm9_read_word_ipc_v2;
						if (arm9_rd16[0x27] != (mmu_read_word_callback)arm9_read_word_gen)
							arm9_rd16[0x27] = (mmu_read_word_callback)arm9_read_word_ipc_mirror_v2;
					}
					zombie(TELL, MMU_READ_WORD_CALLBACKS, (void*)arm9_rd16, 0);
					break;

				case 63:
					blockDMA = 1-(int)param2;
					break;

				default:
					break;
			}
			break;

		default:
			break;
	}
}



// Configuration dialog callback
BOOL CALLBACK DlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	static char stemp[64];
	int i;
	HWND hCmb,hChk;

   switch(msg) {
		case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);

			hChk = GetDlgItem(hdlg,IDC_BIOS);
			SendMessage(hChk, BM_SETCHECK, (useExtBios)?BST_CHECKED:BST_UNCHECKED, 0);

			hChk = GetDlgItem(hdlg,IDC_EDIT1);
			SetWindowText(hChk,szBiosName);
			EnableWindow(hChk,0);

			return TRUE;

		case WM_COMMAND:
			switch(LOWORD(wParam)) {
				case IDOK:

					EndDialog(hdlg, 0);
					return TRUE;

				case IDCANCEL:
					EndDialog(hdlg, 1);
					return FALSE;

				default:
					break;
			}
			break;
   }

	return FALSE;
}

