; An ARM946E emulator
; /Mic, 2004/2005


.686p
.model flat,stdcall
option casemap:none


include \masm32\include\windows.inc
include \masm32\include\kernel32.inc
include \masm32\include\user32.inc

includelib \masm32\lib\kernel32.lib
includelib \masm32\lib\user32.lib

wsprintfA PROTO C :DWORD, :VARARG
wsprintf equ <wsprintfA>

.data

	; Message enumerators
	CPU_IRQ_CALLBACK		equ 000h
	CPU_SET_IO_CALLBACK		equ 001h
	CPU_IO_HANDLER			equ 001h
	CPU_CYCLE_COUNT			equ 002h
	CPU_REG_PC			equ 003h
	CPU_REG_SP			equ 004h
	CPU_BREAKPOINT			equ 005h
	CPU_SPEED			equ 006h
	CPU_INPB_CALLBACK		equ 007h
	CPU_OUTPB_CALLBACK		equ 008h
	CPU_CYCLE_SCALING		equ 009h
	CPU_REG				equ 00Ah
	CPU_CYCLE_PTR			equ 00Bh
	CPU_ENABLE_IRQ			equ 00Ch
	CPU_CACHE_READ_WORD		equ 00Dh
	CPU_CACHE_READ_HWORD		equ 00Eh
	CPU_CACHE_READ_BYTE		equ 00Fh
	CPU_CACHE_WRITE_WORD		equ 010h
	CPU_CACHE_WRITE_HWORD		equ 011h
	CPU_CACHE_WRITE_BYTE		equ 012h
	CPU_DMA_CALLBACK		equ 013h
	
	MMU_READ_BYTE_CALLBACK		equ 100h
	MMU_READ_WORD_CALLBACK		equ 101h
	MMU_WRITE_BYTE_CALLBACK		equ 102h
	MMU_WRITE_WORD_CALLBACK		equ 103h
	MMU_TRANSLATE_ADDRESS_CALLBACK	equ 104h
	MMU_READ_DWORD_CALLBACK		equ 106h
	MMU_WRITE_DWORD_CALLBACK	equ 107h
	MMU_PAGE_TABLE			equ 10Ch
 	MMU_TCM_REMAP			equ 110h
 	MMU_DTCM			equ 111h
	MMU_READ_WORD_CALLBACKS		equ 115h
	MMU_IRQ_OVER			equ 117h
	MMU_WRITE_WORD_CALLBACKS	equ 119h
	MMU_READ_DWORD_CALLBACKS	equ 11Bh
	MMU_WRITE_DWORD_CALLBACKS	equ 11Ch
 	MMU_ITCM			equ 11Eh
	
	GPU_CURRENT_SCANLINE		equ 200h
	GPU_SCREEN_WIDTH		equ 201h
	GPU_SCREEN_HEIGHT		equ 202h
	GPU_FRAME_COMPLETED		equ 203h
	GPU_SCREEN_PITCH		equ 204h
	GPU_MEMORY			equ 206h
	
	ZOMBIE_WINDOW_HANDLE		equ 500h

	PLUGIN_NAME			equ 600h
	LAST_ERROR			equ 601h
	PLUGIN_SETTING			equ 602h

;###############################################################################################

	INPUT				equ 0
	OUTPUT				equ 1

	; Communication direction
	ASK				equ 0
	TELL				equ 1

	; Do internal memory translation instead of asking the mmu
	MEM_TRANSLATE_INT		equ 1
	
	MEM_FPU_TRANSFER		equ 1
	
;###############################################################################################

	ENABLE_ICACHE			equ 0
	ENABLE_DCACHE			equ 0
	
	CACHE_INVALID			equ 1
	CACHE_DIRTY_L			equ 2
	CACHE_DIRTY_H			equ 4

	CACHE_NOT_CACHABLE		equ 0
	CACHE_CACHABLE			equ 1
	CACHE_WRITE_THROUGH		equ 0
	CACHE_WRITE_BACK		equ 2
	
	CACHE_UNLOCKED			equ 0
	CACHE_LOCKED			equ 1
	
;###############################################################################################
	
	LDR_CYCLES			equ 3
	STR_CYCLES			equ 2
	

	CYCLE_MULTIPLIER		equ 1


	DEBUG_ARM			equ 0
	DEBUG_THUMB			equ 0
	ENABLE_DUMP			equ 0
	DUMP_EXEC			equ 200000
	
;###############################################################################################

	; Function pointer prototypes
	_1ARGS_PROTO	typedef PROTO :DWORD
	_2ARGS_PROTO	typedef PROTO :DWORD,:DWORD
	_4ARGS_PROTO	typedef PROTO :DWORD,:DWORD,:DWORD,:DWORD
	_1ARGS		typedef ptr _1ARGS_PROTO
	_2ARGS		typedef ptr _2ARGS_PROTO
	_4ARGS		typedef ptr _4ARGS_PROTO

	cpu_inpb	PROTO :DWORD
	cpu_outpb	PROTO :DWORD,:DWORD


	cpuName		db "ARM946E",0
	
	Overflow	db 0
	Carry		db 0
	Zero		db 0
	Negative	db 0

	; Flags		   Z C N V 	   Q
	align 4
	flags		dd 1,1,1,1,1,0,0,1,0
	flags_usr	dd 1,1,1,1,1,0,0,1,0
	flags_irq	dd 1,1,1,1,1,0,0,1,0
	
	cycle		dd 0
	breakpoint	dd 0
	speed		dd 33514000*2	; 66 MHz
	halted 		dd 0
	irqEnabled	dd 1
	shifterC	dd 0
	
	frequencies	dd 1000000h,
			   1800000h,
			   2000000h,
			   3000000h,
			   4000000h,
			   5000000h
			   
	szFreq0 	db "16.77 MHz",0
	szFreq1 	db "25.16 MHz",0
	szFreq2 	db "33.51 MHz",0
	szFreq3 	db "50.33 MHz",0
	szFreq4 	db "67.03 MHz",0
	szFreq5 	db "83.88 MHz",0
	freqSel 	dd 4

	registerBank	dd offset r13_usr,offset r13_fiq,offset r13_irq,offset r13_svc
	                dd offset r13_usr,offset r13_usr,offset r13_usr,offset r13_abt
	                dd offset r13_usr,offset r13_usr,offset r13_usr,offset r13_und
	                dd offset r13_usr,offset r13_usr,offset r13_usr,offset r13_usr
	                
	;szCMOVError	db "This program requires CMOV instructions to run",0

	szBuffer 	db 128 dup (0)
	IF DEBUG_ARM EQ 1	
	; Debug messages
	szBranch 	db "Branch instruction",13,10,0
	szTrue 		db "true",13,10,0
	;szFormat 	db "Read %08x at pc=%08x",13,10,0
	;szFormat2 	db "Instruction type %d at %08x",13,10,0
	;szFormat3 	db "Writing %04x at %08x",13,10,0
	szFormat4 	db "Branching with link from %08x to %08x",13,10,0
	szFormat5 	db "mov pc,lr (%08x)",13,10,0
	;szFormat6 	db "stmfd %08x",13,10,0
	;szFormat7 	db "ldmfd %08x",13,10,0
	;szFormat8 	db "ldr r%d,=%d",13,10,0
	;szFormat9 	db "MUL r%d,%08x,%08x",13,10,0
	szFormat11 	db "%08x, %08x, %08x, %08x, %08x, %08x",13,10,0
	szFormat19 	db "r15 = %08x",13,10,0
	szFormat20 	db "Rs,Rd = %08x, %08x",13,10,0
	szFormat21 	db "ARM: %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x",13,10,0
	szFormat22 	db "TMB: %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x",13,10,0
	szFormat23 	db "ARM: %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x",13,10,0
	szFormat25	db "IRQ at 0x%08x with IF=0x%04x",13,10,0
	szFormat26	db "IRQ processing finished at 0x%08x. Jumping back to 0x%08x",13,10,0
	szFormat27	db "Branch back to ARM state from 0x%08x",13,10,0
	;szFormat28	db "ARM SWI %d",13,10,0
	;szFormat29	db "TMB SWI %d",13,10,0
	szFormat30	db "BX r%d (%08x) from %08x",13,10,0
	szFormat31	db "Mode switch: %x -> %x",13,10,0
	;szFormat32	db "IRQ at %08x, CPSR = %08x",13,10,0
	;szFormat33	db "Finished IRQ at %08x. Going back to %08x, CPSR = %08x",13,10,0
	szFormat34	db "MOVHI R%d,R%d",13,10,0
	szFormat35 	db "%08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x",13,10,0
	szFormat50	db "%08x",13,10,0
	szFormat47	db "STR %08x,[%08x]",13,10,0
	ENDIF
	;szFormat23 	db "ARM: %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x, %08x",13,10,0
	
	IF ENABLE_DCACHE EQ 1
	szFormat36	db "Doing cache lookup of address %08x",13,10,0
	szFormat37	db "Address %08x not cacheable",13,10,0
	szFormat38	db "Miss, loading cache line",13,10,0
	szFormat39	db "Hit, found matching tag in set %d",13,10,0
	szFormat40	db "Low half dirty, writing back to %08x",13,10,0
	szFormat41	db "High half dirty, writing back to %08x",13,10,0
	szFormat42	db "Doing cache lookup of address %08x (b)",13,10,0
	szFormat45	db "Hit, found matching tag in set %d (b)",13,10,0
	szFormat46	db "Set: %x, Line: %d",13,10,0
	ENDIF
	
	szSWI		db "SWI %02X at PC=%08X  ",0
	szRetIRQ	db "returning from IRQ to address %08X",0
	szIRQ		db "IRQ occured while in thumb state",0
	szLDRH		db "LDRH returned %04X",13,10,0
	
	szFormat100 	db "%08X: %08X\tR: %08X %08X %08X %08X %08X %08X %08X %08X %08X %08X %08X %08X %08X %08X %08X %08X",0
	szFormat101 	db "\tFlags: %d %d %d %d %d %d %d %d",13,10,0
	szFormat102 	db "%08X - PC=%08X BANK=%d R00=%08X R01=%08X R02=%08X R03=%08X R04=%08X R05=%08X R06=%08X R07=%08X R08=%08X R09=%08X R10=%08X R11=%08X R12=%08X R13=%08X R14=%08X R15=%08X OP=%08X ",13,10,0
	szFormat103 	db "%08X - PC=%08X BANK=%d R00=%08X R01=%08X R02=%08X R03=%08X R04=%08X R05=%08X R06=%08X R07=%08X R08=%08X R09=%08X R10=%08X R11=%08X R12=%08X R13=%08X R14=%08X R15=%08X OP=%04X ",13,10,0
	
	szFormat110 	db "DMA%d: %d bytes from 0x%08X to %08X",0

	szFormat117 	db "TOS(%08X) = %08X",13,10,0
	
	szFormat150	db "Remapping DTCM to %08x",13,10,0
	szFormat151	db "Reading from DTCM [%08X] = %08X",13,10,0

	hfile dd 0
	bankn dd 3
	szDump db "dump.txt",0

	guard 		dd 0
	;executed	dd 0
	
.data?
align 8
	; General registers
	r0			dd ?
	r1			dd ?
	r2			dd ?
	r3			dd ?
	r4			dd ?
	r5			dd ?
	r6			dd ?
	r7			dd ?
	r8			dd ?
	r9			dd ?
	r10			dd ?
	r11			dd ?
	r12			dd ?
	r13			dd ?	; SP
	r14			dd ?	; LR
	r15			dd ?	; PC
	r16			dd ?	; CPSR
	r17			dd ?	; SPSR

	r15_swi			dd ?
		
	executed		dd ?
	
	c0			dd ?
	c1			dd ?
	c2			dd ?
	c3			dd ?
	c4			dd ?
	c5			dd ?
	c6			dd ?
	c7			dd ?
	c8			dd ?
	c9			dd ?
	dtcmBase		dd ?
	itcmBase		dd ?
	dtcmEnd			dd ?
	
	r8_usr			dd ?
	r9_usr			dd ?
	r10_usr			dd ?
	r11_usr			dd ?
	r12_usr			dd ?
	r13_usr			dd ?
	r14_usr			dd ?
	spsr_usr		dd ?

	r8_fiq			dd ?
	r9_fiq			dd ?
	r10_fiq			dd ?
	r11_fiq			dd ?
	r12_fiq			dd ?
	r13_fiq			dd ?
	r14_fiq			dd ?
	spsr_fiq		dd ?

	r13_svc			dd ?
	r14_svc			dd ?
	spsr_svc		dd ?

	r13_abt			dd ?
	r14_abt			dd ?
	spsr_abt		dd ?
	
	r13_irq			dd ?
	r14_irq			dd ?
	spsr_irq		dd ?
	
	r13_und			dd ?
	r14_und			dd ?
	spsr_und		dd ?

	
	hInst			dd ?
	hout			dd ?
	dummy 			dd ?
	lastPC 			dd ?
	physaddr 		dd ?
	prefetch1		dd ?
	prefetch2		dd ?
	flagsd			dd ?
	lastCyc			dd ?
	cpu_irq_line		dd ?
	IWRAM 			dd ?
	VRAM			dd ?
	ITCM			dd ?
	DTCM			dd ?
	cb_execute 		dd ?
	cb_execute_bi		dd ?	; The execute callback before an IRQ (for Thumb restoration)
	cb_execute_swi 		dd ?
	hwnd 			dd ?
	branchTarget		dd ?
	branchOrigin		dd ?
	branchPA		dd ?
	r15_old			dd ?
	irq_cb_stack		dd 64 DUP(?)
	irq_cb_stack_ptr 	dd ?
	zombie			_4ARGS ?
	arm9_read_byte 		_1ARGS ?
	arm9_read_word 		_1ARGS ?
	arm9_read_dword 	_1ARGS ?
	arm9_write_byte 	_2ARGS ?
	arm9_write_word 	_2ARGS ?
	arm9_write_dword 	_2ARGS ?
	arm9_get 		_1ARGS ?
	arm9_translate_address _1ARGS ?
	lpszError		dd ?
	old_exec		dd ?
	;executed		dd ?
	
	align 16
	icache			dd 2048 dup (?)
	icache_tags		dd 256 dup (?)
	icache_lock		dd 4 dup (?)

	align 16
	dcache			dd 1024 dup (?)
	dcache_tags		dd 128 dup (?)
	dcache_tags_ex		dd 128 dup (?)
	dcache_lock		dd 4 dup (?)
	dcache_replac		dd 4 dup (?)

	align 16	
	cache_regions		dd 256 dup (?)
	cline			dd ?
	
	align 16
	pageTable		dd 256 dup (?)
	pageMask		dd 256 dup (?)
	readhw			dd 256 dup (?)
	writehw			dd 256 dup (?)
	readw			dd 256 dup (?)
	writew			dd 256 dup (?)
	
	bkpt_address		dd 32 dup (?)
	bkpt_opcode		dd 32 dup (?)
	
	
.code


; Increase the cycle counter
ADD_CYCLES MACRO num
	if num EQ 1
		inc 	dword ptr cycle
	else
		add 	cycle,num*CYCLE_MULTIPLIER
	endif
ENDM


; Cause a division by zero. Used for debugging purposes
DIV0 MACRO
	mov 	edi,r15
	mov 	edx,r12
	xor 	ebp,ebp
	div 	ebp
ENDM


; Shift an operand logically left (cl is non-zero)
LSL_OPERAND MACRO reg
	shl 	reg,cl
ENDM


; Shift an operand logically right (cl is non-zero)
LSR_OPERAND MACRO reg
	shr 	reg,cl
ENDM


; Shift an operand arithmetically right (cl is non-zero)
ASR_OPERAND MACRO reg
	sar 	reg,cl
ENDM


; Rotate an operand right (cl is non-zero)
ROR_OPERAND MACRO reg
	ror 	reg,cl
ENDM


; Shift an operand logically left zero bits
LSL_OPERAND_ZERO MACRO reg
	mov 	ebp,[flags+4]
	mov 	[shifterC],ebp
ENDM


; Shift an operand logically right zero bits
LSR_OPERAND_ZERO MACRO reg
	shr 	reg,31
	xor 	reg,1
	mov 	[shifterC],reg
	xor 	reg,reg
ENDM


; Shift an operand arithmetically right zero bits
ASR_OPERAND_ZERO MACRO reg
	sar 	reg,31
	sete 	byte ptr [shifterC]
ENDM


; Rotate an operand right zero bits
ROR_OPERAND_ZERO MACRO reg
	cmp 	[flags+4],1	; Set x86 carry if ARM9 carry is set
	rcr 	reg,1		; Rotate through carry
	UPDATE_C
ENDM



; Shift/rotate an operand
SHIFT_OPERAND MACRO reg
	mov 	ebp,60h
	and 	ebp,eax			; Get shift type
		
	.if ebp == 0
		shl 	reg,cl
	.elseif ebp == 20h
		shr 	reg,cl
	.elseif ebp == 40h
		sar 	reg,cl
	.else
		ror 	reg,cl
	.endif
ENDM


; Shift/rotate an operand zero bits
SHIFT_OPERAND_ZERO MACRO reg
	mov 	ebp,60h
	and 	ebp,eax			; Get shift type
		
	.if ebp == 0
		mov 	ebp,[flags+4]
		mov 	[shifterC],ebp
	.elseif ebp == 20h
		shr 	reg,31
		xor 	reg,1
		mov 	[shifterC],reg
		xor 	reg,reg
	.elseif ebp == 40h
		sar 	reg,31
		sete 	byte ptr [shifterC]
	.elseif ebp == 60h
		cmp 	[flags+4],1	; Set x86 carry if ARM9 carry is set
		rcr 	reg,1		; Rotate through carry
		UPDATE_C
	.endif
ENDM


; Shift an operand logically left zero bits and discard the output carry
LSL_OPERAND_ZERO_DISCARD_C MACRO reg
ENDM


; Shift an operand logically right zero bits and discard the output carry
LSR_OPERAND_ZERO_DISCARD_C MACRO reg
	xor 	reg,reg
ENDM


; Shift an operand arithmetically right zero bits and discard the output carry
ASR_OPERAND_ZERO_DISCARD_C MACRO reg
	sar 	reg,31
ENDM


; Rotate an operand right zero bits and discard the output carry
ROR_OPERAND_ZERO_DISCARD_C MACRO reg
	cmp 	[flags+4],1
	rcr 	reg,1
ENDM


; Shift/rotate an operand zero bits and discard the output carry
SHIFT_OPERAND_ZERO_DISCARD_C MACRO reg
	LOCAL lbl1,lbl2,lbl3
	
	mov 	ebp,60h
	and 	ebp,eax		; Get shift type
	jz 	lbl1
	cmp 	ebp,20h
	jne 	lbl2
	xor 	reg,reg
	jmp 	lbl1
lbl2:	cmp 	ebp,40h
	jne 	lbl3
	sar 	reg,31
	jmp 	lbl1
lbl3:	cmp 	ebp,60h
	jne 	lbl1
	cmp 	[flags+4],1
	rcr 	reg,1
lbl1:
ENDM


; Shift an operand logically left 32 bits
LSL_OPERAND_32 MACRO reg
	test 	reg,1
	setz 	byte ptr [shifterC]
	xor 	reg,reg
ENDM


LSR_OPERAND_32 MACRO reg
	test 	reg,80000000h
	sete 	byte ptr [shifterC]
	xor 	reg,reg
ENDM


ASR_OPERAND_32 MACRO reg
	sar 	reg,31
	sete 	byte ptr [shifterC]
ENDM


ROR_OPERAND_32 MACRO reg
	test 	reg,80000000h
	sete 	byte ptr [shifterC]
ENDM


SHIFT_OPERAND_32 MACRO reg
	mov 	ebp,60h
	and 	ebp,eax		; Get shift type
		
	.if ebp == 0
		xor 	reg,1
		mov 	[shifterC],reg
		xor 	reg,reg
	.elseif ebp == 20h
		shr 	reg,31
		xor 	reg,1
		mov 	[shifterC],reg
		xor 	reg,reg
	.elseif ebp == 40h
		sar 	reg,31
		sete 	byte ptr [shifterC]
	.elseif ebp == 60h
		test 	reg,80000000h
		sete 	byte ptr [shifterC]
	.endif
ENDM


; Shift an operand logically left 32 (or more) bits
LSL_OPERAND_GT32 MACRO reg
	mov 	[shifterC],1
	xor 	reg,reg
ENDM


LSR_OPERAND_GT32 MACRO reg
	mov 	[shifterC],1
	xor 	reg,reg
ENDM


ASR_OPERAND_GT32 MACRO reg
	sar 	reg,31
	sete 	byte ptr [shifterC]
ENDM


ROR_OPERAND_GT32 MACRO reg
	LOCAL _shift_and_31_eq_0,_shift_gt_0
	
	and 	ecx,31
	jnz 	_shift_gt_0
	mov 	ecx,[flags+4]
	mov 	[shifterC],ecx
	jmp 	_shift_and_31_eq_0
	_shift_gt_0:	
		ror 	reg,cl
		setnc 	byte ptr [shifterC]
	_shift_and_31_eq_0:
ENDM


SHIFT_OPERAND_GT32 MACRO reg
	mov 	ebp,60h
	and 	ebp,eax		; Get shift type
		
	.if ebp < 40h
		mov 	[shifterC],1
		xor 	reg,reg
	.elseif ebp == 40h
		sar 	reg,31
		sete 	byte ptr [shifterC]
	.elseif ebp == 60h
		and 	ecx,31
		ror 	reg,cl
		setnc 	byte ptr [shifterC]
	.endif
ENDM


;#####################################################################################################


; Read a word (32 bits) from memory
RDWORD MACRO adr
	push 		edx
	mov 		ecx,adr
	IF ENABLE_DCACHE EQ 1
		mov 		edx,adr
		call 		dcache_read_word
	ELSE
		ADD_CYCLES	LDR_CYCLES
		mov 		eax,ecx 
		shr 		eax,20
		and 		eax,0FFh
		call 		dword ptr [readw + eax*4]
	ENDIF
	pop 		edx
ENDM


; Read a halfword (16 bits) from memory
RDHWORD MACRO adr
	LOCAL __vram,__not_vram
	
	push 		edx
	mov 		ecx,adr
	IF ENABLE_DCACHE EQ 1
		mov 		edx,adr
		call 		dcache_read_hword
	ELSE
		ADD_CYCLES	LDR_CYCLES
		mov 		eax,ecx ;adr
		shr 		eax,20
		and 		ecx,-2
		and 		eax,0FFh
		call 		dword ptr [readhw + eax*4]
	ENDIF
	and 		eax,0FFFFh
	pop 		edx
ENDM


RDBYTE MACRO adr
	push 		edx
	IF ENABLE_DCACHE EQ 1
		mov 		edx,adr
		call 		dcache_read_byte
	ELSE
		invoke 		arm9_read_byte,adr
		ADD_CYCLES 	LDR_CYCLES
	ENDIF
	and 		eax,0FFh
	pop 		edx
ENDM



WRBYTE MACRO adr,r8,r32
	IF ENABLE_DCACHE EQ 1
		mov 		eax,r32
		mov 		edx,adr
		call 		dcache_write_byte
	ELSE
		invoke 		arm9_write_byte,adr,r32
		ADD_CYCLES 	STR_CYCLES
	ENDIF
ENDM


WRHWORD MACRO adr,r16,r32
	IF ENABLE_DCACHE EQ 1
		mov 		eax,r32
		mov 		edx,adr
		call 		dcache_write_hword
	ELSE
		mov 		ecx,adr
		mov 		edx,r32
		mov 		eax,ecx 
		ADD_CYCLES 	STR_CYCLES
		shr 		eax,20
		and 		eax,0FFh
		call 		dword ptr [writehw + eax*4]		
	ENDIF
ENDM


WRWORD MACRO adr,r32
	IF ENABLE_DCACHE EQ 1
		mov 		eax,r32
		mov 		edx,adr
		call 		dcache_write_word
	ELSE
		mov 		ecx,adr
		mov 		edx,r32
		mov 		eax,ecx 
		ADD_CYCLES 	STR_CYCLES
		shr 		eax,20
		and 		eax,0FFh
		call 		dword ptr [writew + eax*4]		
	ENDIF
ENDM

;#####################################################################################################


; Update all flags
; Destroys eax, ebx, edx
UPDATE_FLAGS MACRO 
	setnz 		byte ptr [flags]
	setnc 		ah
	setns 		byte ptr [flags+8]
	setno 		byte ptr [flags+12]

	setz 		dl
	setl  		byte ptr [flags+20]
	setle 		byte ptr [flags+24] 
	or 		dl,ah
	mov 		byte ptr [flags+4],ah
	mov 		byte ptr [flags+16],dl
ENDM


PACK_FLAGS MACRO
	xor 		ebx,ebx
	or 		ebx,[flags+8]	; N
	shl 		ebx,1
	or 		ebx,[flags+0]	; Z
	shl 		ebx,1
	or 		ebx,[flags+4]	; C
	shl 		ebx,1
	or 		ebx,[flags+12]	; V
	xor 		ebx,15
	shl 		ebx,1
	or 		ebx,[flags+32]	; Q
	and 		r16,07FFFFFFh

	shl 		ebx,27
	or 		r16,ebx
ENDM


UNPACK_FLAGS MACRO
	test 		r16,40000000h
	setz 		al
	setnz 		dl
	test 		r16,20000000h
	setz 		byte ptr [flags+4]	; C
	test 		r16,80000000h
	setz 		dh
	;setz 		byte ptr [flags+8]	; N
	test 		r16,10000000h
	setz 		byte ptr [flags+12] 	; V
	
	test 		r16,08000000h
	setnz 		byte ptr [flags+32] 	; Q
	
	mov 		byte ptr [flags],al	; Z
	mov 		byte ptr [flags+8],dh	; N 
	xor 		dh,byte ptr [flags+12]	; dh = N not equal V
	or 		dl,byte ptr [flags+4]	; dl = Z set or C clear
	xor 		al,1			; al = Z set
	mov 		byte ptr [flags+16],dl
	or 		al,dh			; al = Z set OR (N not equal V) 
	mov 		byte ptr [flags+20],dh
	mov 		byte ptr [flags+24],al
	
ENDM



SWAP_REGISTERS MACRO
; r16 holds current mode bits
; ebx hold new mode bits

	LOCAL noswap,not_from_fiq,not_to_fiq
	
	pusha
		
	mov 		eax,r16
	mov 		ecx,ebx

	and 		eax,1Fh	; Get current mode
	and 		ebx,1Fh	; Get new mode

	mov 		edx,eax
	cmp 		ebx,eax	; No mode change?

	je 		noswap
	xor 		edx,ebx
	cmp 		edx,0Fh	; User and System mode share the same registers
	je 		noswap

	; Copy R3,R14 and CPSR
	mov 		ecx,eax
	and 		ecx,15
	mov 		edi,[registerBank + ecx*4]
	mov 		ecx,r13
	mov 		edx,r14
	mov 		esi,r17
	mov 		[edi],ecx
	mov 		[edi+4],edx
	mov 		[edi+8],esi
	
	; Copy R8..R12 if current mode is FIQ
	cmp 		eax,11h
	jne 		not_from_fiq
	test 		r16,20h
	jnz 		not_from_fiq
	mov 		esi,offset r8
	mov 		edi,offset r8_fiq
	mov 		ecx,5
	rep 		movsd
	mov 		esi,offset r8_usr
	mov 		edi,offset r8
	mov 		ecx,5
	rep 		movsd
	not_from_fiq:

	; Copy R3,R14 and CPSR
	mov 		ecx,ebx
	and 		ecx,15
	mov 		edi,[registerBank + ecx*4]
	mov 		ecx,[edi]
	mov 		edx,[edi+4]
	mov 		esi,[edi+8]
	mov 		r13,ecx
	mov 		r14,edx
	mov 		r17,esi
	
	; Copy R8..R12 if the new mode is FIQ
	cmp 		ebx,11h
	jne 		not_to_fiq
	test 		r16,20h
	jnz 		not_to_fiq
	mov 		esi,offset r8
	mov 		edi,offset r8_usr
	mov 		ecx,5
	rep 		movsd
	mov 		esi,offset r8_fiq
	mov 		edi,offset r8
	mov 		ecx,5
	rep 		movsd
	not_to_fiq:
	
	noswap:
	popa
ENDM


; Update all flags (after a sub, sbc, rsb, rsc, cmp)
; Destroys eax, ebx, edx
UPDATE_FLAGS_SUB MACRO 
	setnz 		byte ptr [flags]
	setc 		byte ptr [flags+4]
	setns 		byte ptr [flags+8]
	setno 		byte ptr [flags+12]

	setbe 		byte ptr [flags+16]
	setl  		byte ptr [flags+20]
	setle 		byte ptr [flags+24] 
ENDM



UPDATE_C_FORCE MACRO
	setnc 		byte ptr [flags+4]
ENDM


UPDATE_C MACRO
	setnc 		byte ptr [shifterC]
ENDM


; Only update N, Z and derived flags like HI, BE etc
; Destroys eax, edx
UPDATE_NZ MACRO 
	setnz 		al
	setns 		dh
	mov 		byte ptr [flags],al	; Z
	mov 		byte ptr [flags+8],dh	; N 

	setz 		dl
	xor 		dh,byte ptr [flags+12]	; dh = N not equal V
	or 		dl,byte ptr [flags+4]	; dl = Z set or C clear
	xor 		al,1			; al = Z set
	mov 		byte ptr [flags+16],dl
	or 		al,dh			; al = Z set OR (N not equal V) 
	mov 		byte ptr [flags+20],dh
	mov 		byte ptr [flags+24],al
ENDM


; Update N and Z after a long MUL
UPDATE_NZ_MULL MACRO
	mov 		ebx,edx
	test 		edx,80000000h
	setz 		dh
	or 		eax,ebx
	setnz 		al
	mov 		byte ptr [flags+8],dh	; N 
	mov 		byte ptr [flags],al	; Z

	setz 		dl
	xor 		dh,byte ptr [flags+12]	; dh = N not equal V
	or 		dl,byte ptr [flags+4]	; dl = Z set or C clear
	xor 		al,1			; al = Z set
	mov 		byte ptr [flags+16],dl
	or 		al,dh			; al = Z set OR (N not equal V) 
	mov 		byte ptr [flags+20],dh
	mov 		byte ptr [flags+24],al
ENDM


UPDATE_NZ_MLAL MACRO
	mov 		edx,[r0+edi*4]
	mov 		eax,[r0+esi*4]
	mov 		ebx,edx
	test 		edx,80000000h
	setz 		dh
	or 		eax,ebx
	setnz 		al
	mov 		byte ptr [flags+8],dh	; N 
	mov 		byte ptr [flags],al	; Z

	setz 		dl
	xor 		dh,byte ptr [flags+12]	; dh = N not equal V
	or 		dl,byte ptr [flags+4]	; dl = Z set or C clear
	xor 		al,1			; al = Z set
	mov 		byte ptr [flags+16],dl
	or 		al,dh			; al = Z set OR (N not equal V) 
	mov 		byte ptr [flags+20],dh
	mov 		byte ptr [flags+24],al
ENDM


; Maps addresses from emulated system memory onto host system memory
; Destroys ebp
TRANSLATE_ADDRESS MACRO adr,mask
	LOCAL __itcm,__dtcm,__normal,__done
	
	mov 		ebp,adr
	and 		ebp,0FFFFFFFh
	mov 		eax,ebp

	; Is the address within DTCM or ITCM ?
	cmp 		ebp,8000h
	jb 		__itcm
	cmp 		ebp,dtcmBase
	jb 		__normal
	cmp 		ebp,dtcmEnd
	jge 		__normal
	
	__dtcm:
	and 		eax,(mask AND 3FFFh)
	add 		eax,DTCM
	jmp 		__done
	
	__itcm:
	and 		eax,(mask AND 7FFFh)
	add 		eax,ITCM
	jmp 		__done

	__normal:
	shr 		ebp,20
	and 		eax,mask ;00FFFFFh
	add 		eax,[pageTable + ebp*4]
	__done:
ENDM


; Destroys ebp
TRANSLATE_ADDRESS_ALIGN_2 MACRO adr
	mov 		ebp,adr
	and 		ebp,0FFFFFFFh
	mov 		eax,ebp
	shr 		ebp,20
	and 		eax,00FFFFEh
	add 		eax,[pageTable + ebp*4]
ENDM

; Destroys ebp
TRANSLATE_ADDRESS_ALIGN_4 MACRO adr
	mov 		ebp,adr
	and 		ebp,0FFFFFFFh
	mov 		eax,ebp
	shr 		ebp,20
	and 		eax,00FFFFCh
	add 		eax,[pageTable + ebp*4]
ENDM


; Destroys ebp
TRANSLATE_ADDRESS_ALIGN_32 MACRO adr
	mov 		ebp,adr
	mov 		eax,ebp
	shr 		ebp,20
	and 		eax,00FFFE0h
	add 		eax,[pageTable + ebp*4]
ENDM


; Handle writes to memory in a cached configuration
DCACHE_WRITE MACRO dreg,regsize,adrmask
	LOCAL tag_matched,region_not_cached,tag_matched,tag_matched2,write_through
	
	; Is the region set up as cachable ?

	mov 		ebx,edx
	and 		ebx,0FF000000h
	shr 		ebx,22
	test 		dword ptr [cache_regions + ebx],CACHE_CACHABLE
	jz 		region_not_cached
	

	; Does the region use write back or write through ?
	test 		dword ptr [cache_regions + ebx],CACHE_WRITE_BACK
	jz 		write_through

	and 		edx,[pageMask + ebx]
	
	; Get the set and tag indexes
	mov 		ebx,edx
	mov 		ecx,edx
	and 		ebx,0000003E0h	; Set
	and 		ecx,0FFFFFC00h	; Tag
	mov 		esi,0
	shr 		ebx,1

	; Try to match the tag against any line in the selected set
	cmp 		ecx,[dcache_tags + ebx]
	je 		tag_matched
	add 		esi,2
	cmp 		ecx,[dcache_tags + 8 + ebx]
	je 		tag_matched
	dec 		esi
	cmp 		ecx,[dcache_tags + 4 + ebx]
	je 		tag_matched
	add 		esi,2
	cmp 		ecx,[dcache_tags + 12 + ebx]
	je 		tag_matched

	; TODO handle buffered writes
	jmp 		region_not_cached


	tag_matched:
	mov 		ebp,edx				; ebp = address
	mov 		edi,ebx				; ecx = Set
	and 		ebp,10h				; ebp = a[4]
	shl 		ebx,3				; ebx = Set * Linesize * Associativity
	shr 		ebp,3				; ebp = a[4]/8 (0 : 2)
	add 		ebp,2				; ebp = 2 : 4
	or 		[dcache_tags_ex + edi + esi*4],ebp	; Mark as dirty
	shl 		esi,5				; esi = Line * Linesize
	and 		edx,adrmask			; edx = a[4:2]
	add 		ebx,esi
	mov 		&regsize& ptr [dcache + ebx + edx*1],dreg
	ADD_CYCLES 	1
	ret

	write_through:
	
	mov 		ebx,edx
	mov 		ecx,edx
	and 		ebx,0000003E0h	; Set
	and 		ecx,0FFFFFC00h	; Tag
	mov 		esi,0
	shr 		ebx,1

	; Try to match the tag against any line in the selected set
	cmp 		ecx,[dcache_tags + ebx]
	je 		tag_matched2
	add 		esi,2
	cmp 		ecx,[dcache_tags + 8 + ebx]
	je 		tag_matched2
	dec 		esi
	cmp 		ecx,[dcache_tags + 4 + ebx]
	je 		tag_matched2
	add 		esi,2
	cmp 		ecx,[dcache_tags + 12 + ebx]
	je 		tag_matched2

	; TODO handle buffered writes
	jmp 		region_not_cached


	tag_matched2:
	;mov 		ebp,edx				; ebp = address
	;mov 		edi,ebx				; ecx = Set
	;and 		ebp,10h				; ebp = a[4]
	shl 		ebx,3				; ebx = Set * Linesize * Associativity
	;shr 		ebp,3				; ebp = a[4]/8 (0 : 2)
	mov 		ecx,edx
	;add 		ebp,2				; ebp = 2 : 4
	;or 		[dcache_tags_ex + edi + esi*4],ebp	; Mark as dirty
	shl 		esi,5				; esi = Line * Linesize
	and 		ecx,adrmask			; edx = a[4:2]
	add 		ebx,esi
	mov 		&regsize& ptr [dcache + ebx + ecx],dreg
	;ADD_CYCLES 	1
	
	region_not_cached:
	invoke 		arm9_write_&regsize&,edx,eax
	ADD_CYCLES 	STR_CYCLES+1
	ret
ENDM


; Load a cache line (32 bytes)
LOAD_CACHE_LINE MACRO
	IF MEM_FPU_TRANSFER EQ 0
		mov 		ecx,[eax]
		mov 		ebp,[eax + 04h]
		mov 		[dcache + ebx + 00h],ecx
		mov 		[dcache + ebx + 04h],ebp
		mov 		ecx,[eax + 08h]
		mov 		ebp,[eax + 0Ch]
		mov 		[dcache + ebx + 08h],ecx
		mov 		[dcache + ebx + 0Ch],ebp
		mov 		ecx,[eax + 10h]
		mov 		ebp,[eax + 14h]
		mov 		[dcache + ebx + 10h],ecx
		mov 		[dcache + ebx + 14h],ebp
		mov 		ecx,[eax + 18h]
		mov 		ebp,[eax + 1Ch]
		mov 		[dcache + ebx + 18h],ecx
		mov 		[dcache + ebx + 1Ch],ebp
	ELSE
		fld 		qword ptr [eax+24]
		fld 		qword ptr [eax+16]
		fld 		qword ptr [eax+8]
		fld 		qword ptr [eax]
		fstp 		qword ptr [dcache + ebx]
		fstp 		qword ptr [dcache + ebx + 8]
		fstp 		qword ptr [dcache + ebx + 16]
		fstp 		qword ptr [dcache + ebx + 24]
	ENDIF
ENDM



; Handle Co-processor 15 operations
CP15OP MACRO op2,tbl
	LOCAL cpn_ok
	
	mov 		ebx,eax
	mov 		ecx,eax
	mov 		edx,eax
	mov 		ebp,eax
	and 		ebx,15	; CRm
	shr 		ecx,16
	shr 		edx,12
	shr 		ebp,21
	and 		ecx,15	; CRn
	and 		edx,15	; Rd
	shr 		eax,8

	and 		ebp,7	; Op1
	and 		eax,15	; Cp
	
	cmp 		eax,15
	je 		cpn_ok
	DIV0
	cpn_ok:
	
	mov 		eax,op2
	jmp 		dword ptr [&tbl& + ecx*4]
ENDM




;#####################################################################################################


LibMain PROC hInstDLL:DWORD, reason:DWORD, unused:DWORD

        .if reason == 1		;DLL_PROCESS_ATTACH
        	mov 		eax,hInstDLL
        	mov 		hInst,eax
		mov 		eax,TRUE
		mov 		cb_execute,OFFSET cpu_execute_until_arm
		ret

        .elseif reason == 0	;DLL_PROCESS_DETACH
        .elseif reason == 2	;DLL_THREAD_ATTACH
        .elseif reason == 3	;DLL_THREAD_DETACH
        .endif

        mov 		eax,FALSE
        ret
LibMain ENDP



;-------------------------------------------------------------------
; Unused crap

inpb_null PROC
	ret
inpb_null ENDP
.data
inpb_handlers dd 256 ;DUP(OFFSET inpb_null)
.code

outpb_null PROC dat:DWORD
	ret
outpb_null ENDP
.data
outpb_handlers dd 256 ;DUP(OFFSET outpb_null)
.code	

;--------------------------------------------------------------------



; Reset the data cache
dcache_reset:
	pusha

	; Invalidate the entire cache
	mov 	eax,CACHE_INVALID
	mov 	ecx,128
	mov 	edi,offset dcache_tags
	rep 	stosd

	mov 	eax,0
	mov 	ecx,128
	mov 	edi,offset dcache_tags_ex
	rep 	stosd

	mov 	eax,0
	mov 	ecx,4
	mov 	edi,offset dcache_lock
	rep 	stosd

	mov 	eax,0
	mov 	ecx,4
	mov 	edi,offset dcache_replac
	rep 	stosd

	; Initialize all regions as non-cachable
	mov 	eax,CACHE_NOT_CACHABLE OR CACHE_WRITE_THROUGH
	mov 	ecx,256
	mov 	edi,offset cache_regions
	rep 	stosd

	; Just a test...
;	mov 	[cache_regions + 8*4],CACHE_CACHABLE
	mov 	[cache_regions + 3*4],CACHE_CACHABLE ;OR CACHE_WRITE_BACK

	mov 	cline,0
	
	popa
	ret



DCACHE_READ MACRO dreg,regsize,adrmask
	LOCAL _region_not_cached,_tag_matched,_grab_line,_grab_line_next
	LOCAL _found_free_line,_low_data_clean,_high_data_clean,_dbgread
	
	mov 		ebx,edx
	and 		ebx,0FF000000h
	shr 		ebx,22
	test 		dword ptr [cache_regions + ebx],CACHE_CACHABLE
	jz 		_region_not_cached

	and 		edx,[pageMask + ebx]
	
	mov 		ebx,edx
	mov 		ecx,edx
	and 		ebx,0000003E0h	; Set
	and 		ecx,0FFFFFC00h	; Tag
	mov 		esi,0
	shr 		ebx,1		; 0..1F0h
	
	; Try to match the tag against any line in the selected set
	cmp 		ecx,[dcache_tags + ebx]
	je 		_tag_matched
	add 		esi,2
	cmp 		ecx,[dcache_tags + 8 + ebx]
	je 		_tag_matched
	dec 		esi
	cmp 		ecx,[dcache_tags + 4 + ebx]
	je 		_tag_matched
	add 		esi,2
	cmp 		ecx,[dcache_tags + 12 + ebx]
	je 		_tag_matched


	; Try looking for a cache line that is invalid and not locked
	mov 		esi,0
	_grab_line:
		test 		[dcache_tags + ebx + esi*4],CACHE_INVALID
		jz 		_grab_line_next
		test 		[dcache_lock + esi*4],CACHE_LOCKED
		jz 		_found_free_line
		_grab_line_next:
		inc 		esi
		cmp 		esi,4
		jne 		_grab_line


	; No invalid lines, so we need to flush a line
	mov 		esi,cline
	test		[dcache_lock + esi*4],CACHE_LOCKED
	jnz 		_region_not_cached		; TODO: come up with a better solution


	; The line has valid data in it, so write it back to main memory
	mov 		eax,edx
	and 		eax,0FF000000h
	shr 		eax,22
	test 		dword ptr [cache_regions + eax],CACHE_WRITE_BACK
	jz 		_found_free_line


	; Reconstruct original address from tag and set number
	mov 		ebp,ebx
	mov 		eax,[dcache_tags + ebx + esi*4]
	shl 		ebp,1
	and 		eax,0FFFFFC00h
	or 		eax,ebp

	; Map the address to physical memory
	IF MEM_TRANSLATE_INT EQ 1
		TRANSLATE_ADDRESS eax,0FFFE0h
	ENDIF

	push 		ebx
	mov 		edi,ebx
	mov 		ebp,esi
	shl 		ebx,3
	shl 		ebp,5
	add 		ebx,ebp

	test 		[dcache_tags_ex + edi + esi*4],CACHE_DIRTY_L
	jz 		_low_data_clean
	fld  		qword ptr [dcache + ebx + 00h]
	fld  		qword ptr [dcache + ebx + 08h]
	fstp 		qword ptr [eax+8]
	fstp 		qword ptr [eax]
	_low_data_clean:

	test 		[dcache_tags_ex + edi + esi*4],CACHE_DIRTY_H
	jz 		_high_data_clean
	fld 		qword ptr [dcache + ebx + 10h]
	fld  		qword ptr [dcache + ebx + 18h]
	fstp 		qword ptr [eax+18h]
	fstp 		qword ptr [eax+10h]
	_high_data_clean:

	pop ebx
	

	_found_free_line:
	mov 		[dcache_tags    + ebx + esi*4],ecx
	mov 		[dcache_tags_ex + ebx + esi*4],0


	IF MEM_TRANSLATE_INT EQ 1
		TRANSLATE_ADDRESS edx,0FFFE0h
	ELSE
		push 		edx
		push 		ebx
		and 		edx,0FFFFFFE0h
		invoke arm9_translate_address,edx
		pop 		ebx
	ENDIF

	shl 		esi,5
	inc 		cline
	lea 		ebx,[esi + ebx*8]

	LOAD_CACHE_LINE
	and 		cline,3
	IF MEM_TRANSLATE_INT EQ 0
		pop 		edx
	ENDIF
	IF adrmask EQ 1Ch
		mov 		ecx,edx
	ENDIF
	and 		edx,adrmask
	IF adrmask EQ 1Ch
		and 		ecx,3
	ENDIF
	mov 		dreg,&regsize& ptr [eax + edx]
	IF adrmask EQ 1Ch
		shl 		ecx,3
	ENDIF
	ADD_CYCLES 	10
	IF adrmask EQ 1Ch
		ror 		eax,cl
	ENDIF
	ret

	_tag_matched:
	shl 		ebx,3		; ebx = Set * Linesize * Associativity
	IF adrmask EQ 1Ch
		mov 		ecx,edx
	ENDIF
	shl 		esi,5		; esi = Line * Linesize
	IF adrmask EQ 1Ch
		and 		ecx,3
	ENDIF
	and 		edx,adrmask		
	add 		ebx,esi
	IF adrmask EQ 1Ch
		shl 		ecx,3
	ENDIF
	mov 		dreg,&regsize& ptr [dcache + ebx + edx]
	ADD_CYCLES 	1
	IF adrmask EQ 1Ch
		ror 		eax,cl
	ENDIF
	ret

	_region_not_cached:
	invoke 		arm9_read_&regsize&,edx
	ADD_CYCLES 	LDR_CYCLES
	ret
ENDM



IF ENABLE_DCACHE EQ 1

	
dcache_read_word:	
; IN: edx=address
; OUT: eax=data

	DCACHE_READ eax,<dword>,1Ch



dcache_read_hword:	
; IN: edx=address
; OUT: eax=data

	DCACHE_READ ax,<word>,1Eh



dcache_read_byte:	
; IN: edx=address
; OUT: eax=data

	DCACHE_READ al,<byte>,1Fh
	

;####################################################################################################



align 8
dcache_write_word:	
; IN: edx=address
; OUT: eax=data

	DCACHE_WRITE eax,<dword>,1Ch



align 8
dcache_write_hword:	
; IN: edx=address
; OUT: eax=data

	DCACHE_WRITE ax,<word>,1Eh


align 8
dcache_write_byte:	
; IN: edx=address
; OUT: eax=data

	DCACHE_WRITE al,<byte>,1Fh



ENDIF


; handle DMA transfers requested by the MMU
cpu_dma PROC dest:DWORD,src:DWORD,count:DWORD,xferWidth:DWORD,srcinc:DWORD
	pusha
	
	mov 		ecx,0
	cmp 		xferWidth,4
	jne 		cd2_repeat
	
	; 32-bit transfer
	cd4_repeat:
		push 		ecx
		RDWORD 		src
		WRWORD 		dest,eax
		mov 		eax,srcinc
		pop 		ecx
		add 		src,eax
		add 		dest,4
		add 		ecx,4
		cmp 		ecx,count
		jb 		cd4_repeat
	popa
	ret

	; 16-bit transfer
	cd2_repeat:
		push 		ecx
		RDHWORD 	src
		WRHWORD 	dest,ax,eax
		mov 		eax,srcinc
		pop 		ecx
		add 		src,eax
		add 		dest,2
		add 		ecx,2
		cmp 		ecx,count
		jb 		cd2_repeat

	popa
	ret
cpu_dma ENDP

;####################################################################################################



; ARM dataproc ops
align 8

op_and:
	shr 		eax,12		; Get Rd
	and 		esi,edi		; Rs AND Op2
	and 		eax,0Fh
	ADD_CYCLES 	1
	mov 		[r0+eax*4],esi	; Write result
	.if ebp == r15
		mov 		physaddr,-1
	.endif
	ret
op_ands:
	mov 		ebp,eax
	mov 		ecx,[shifterC]
	and 		esi,edi
	mov 		[flags+4],ecx
	UPDATE_NZ
	shr 		ebp,12
	and 		ebp,0Fh
	ADD_CYCLES 	1
	mov 		[r0+ebp*4],esi
	ret

op_eor:
	shr 		eax,12
	xor 		esi,edi		; Rs XOR Op2
	and 		eax,0Fh
	ADD_CYCLES 	1
	mov 		[r0+eax*4],esi
	.if ebp == r15
		mov 		physaddr,-1
	.endif
	ret
	
op_eors:
	mov 		ebp,eax
	mov 		ecx,[shifterC]
	xor 		esi,edi
	mov 		[flags+4],ecx
	UPDATE_NZ
	shr 		ebp,12
	and 		ebp,0Fh
	mov 		[r0+ebp*4],esi
	ADD_CYCLES 	1
	ret
	
op_sub:
	shr 		eax,12
	sub 		esi,edi		; Rs - Op2
	and 		eax,0Fh
	ADD_CYCLES 	1
	mov 		[r0+eax*4],esi
	.if eax == 15
		mov 	physaddr,-1
	.endif
	ret
	
op_subs:
	mov 		ebp,eax
	sub 		esi,edi
	UPDATE_FLAGS_SUB
	shr 		ebp,12
	ADD_CYCLES 	1
	and 		ebp,0Fh
	.if ebp==15
		ADD_CYCLES 	2
		mov 		physaddr,-1
	.endif	
	mov 		[r0+ebp*4],esi
	ret

op_subs_irq:
	mov 		ebp,eax
	sub 		esi,edi
	UPDATE_FLAGS_SUB
	shr 		ebp,12
	ADD_CYCLES 	1
	and 		ebp,0Fh
	.if ebp == 15
		sub 		esi,4
		mov 		r15,esi 
		mov 		ebx,r17
		SWAP_REGISTERS
		mov 		r16,ebx
		test		ebx,20h
		jz 		still_arm
		mov 		cb_execute,OFFSET cpu_execute_until_thumb
		mov 		dword ptr [esp],OFFSET cpu_execute_until_thumb
		still_arm:
		
		UNPACK_FLAGS
		mov 		dword ptr [subs_handler],offset op_subs
		mov 		physaddr,-1
		invoke 		zombie,TELL,MMU_IRQ_OVER,0,0
		.if cpu_irq_line & 80000000h
			mov 		cb_execute,OFFSET swi_5
			mov 		dword ptr [esp],OFFSET swi_5
		.endif
		ret
	.endif
	mov 		[r0+ebp*4],esi
	ret
	
op_rsb:
	shr 		eax,12
	sub 		edi,esi		; Op2 - Rs
	and 		eax,0Fh
	mov 		[r0+eax*4],edi
	ADD_CYCLES 	1
	.if eax == 15
		mov 		physaddr,-1
	.endif
	ret
	
op_rsbs:
	mov 		ebp,eax
	sub 		edi,esi
	UPDATE_FLAGS_SUB
	shr 		ebp,12
	and 		ebp,0Fh
	mov 		[r0+ebp*4],edi
	ADD_CYCLES 	1
	ret
op_add:
	shr 		eax,12
	add 		esi,edi		; Rs + Op2
	and 		eax,0Fh
	ADD_CYCLES 	1
	mov 		[r0+eax*4],esi
	.if eax == 15
		mov 		physaddr,-1
	.endif
	ret
	
op_adds:
	mov 		ebp,eax
	add 		esi,edi
	UPDATE_FLAGS
	shr 		ebp,12
	ADD_CYCLES 	1
	and 		ebp,0Fh
	mov 		[r0+ebp*4],esi
	.if ebp == r15
		mov 		physaddr,-1
	.endif
	ret
op_adc:
	shr 		eax,12
	cmp 		[flags+4],1	; Set x86 carry if ARM9 carry is set
	adc 		esi,edi
	and 		eax,0Fh
	mov 		[r0+eax*4],esi
	ADD_CYCLES 	1
	.if ebp == r15
		mov 		physaddr,-1
	.endif
	ret
	
op_adcs:
	mov 		ebp,eax
	cmp 		[flags+4],1	; Set x86 carry if ARM9 carry is set
	adc 		esi,edi
	UPDATE_FLAGS
	shr 		ebp,12
	and 		ebp,0Fh
	mov 		[r0+ebp*4],esi
	ADD_CYCLES 	1
	.if ebp == r15
		mov 		physaddr,-1
	.endif
	ret
	
op_sbc:
	shr 		eax,12
	add 		edi,[flags+4]
	sub 		esi,edi
	and 		eax,0Fh
	mov 		[r0+eax*4],esi
	ADD_CYCLES 	1
	.if ebp == r15
		mov 		physaddr,-1
	.endif
	ret
op_sbcs:
	mov 		ebp,eax
	add 		edi,[flags+4]
	sub 		esi,edi
	UPDATE_FLAGS_SUB
	shr 		ebp,12
	and 		ebp,0Fh
	mov 		[r0+ebp*4],esi
	ADD_CYCLES 	1
	ret
op_rsc:
	shr 		eax,12
	add 		esi,[flags+4]
	sub 		edi,esi
	and 		eax,0Fh
	mov 		[r0+eax*4],edi
	ADD_CYCLES 	1
	.if ebp == r15
		mov 		physaddr,-1
	.endif
	ret
	
op_rscs:
	mov 		ebp,eax
	add 		esi,[flags+4]
	sub 		edi,esi
	UPDATE_FLAGS_SUB
	shr 		ebp,12
	and 		ebp,0Fh
	mov 		[r0+ebp*4],edi
	ADD_CYCLES 	1
	ret

.data
szMRS1 db "MRS CPSR",13,10,0
.code
op_tst:
	;test 		eax,10000h
	;jz 		@@msr_psr
	shr 		eax,12

	PACK_FLAGS
	and 		eax,15
	mov 		ebx,r16
	mov 		[r0+eax*4],ebx
	ADD_CYCLES 	1
	ret
	;@@msr_psr:
	;ADD_CYCLES 	1
	;ret
	
op_tsts:
	mov 		eax,[shifterC]
	and 		esi,edi
	mov 		[flags+4],eax
	UPDATE_NZ
	ADD_CYCLES 	1
	ret

.data
szMSR1 db "MSR CPSR",13,10,0
.code
op_teq:
	test 		eax,10000h
	jz 		@@msr_psr

	and 		eax,15
	PACK_FLAGS
	mov 		ebx,[r0+eax*4]
	SWAP_REGISTERS
	mov 		r16,ebx
	UNPACK_FLAGS
	ADD_CYCLES 	1
	ret

	@@msr_psr:
	test 		eax,2000000h
	jnz 		@@msr_psr_imm
	and 		eax,15
	and 		r16,07FFFFFFh
	mov 		ebx,[r0+eax*4]
	and 		ebx,0F8000000h

	or 		r16,ebx
	UNPACK_FLAGS
	ADD_CYCLES 	1
	ret

	@@msr_psr_imm:
	and 		edi,0F8000000h
	and 		r16,007FFFFFFh
	or 		r16,edi
	UNPACK_FLAGS
	ADD_CYCLES 	1
	ret
	
	
op_teqs:
	mov 		eax,[shifterC]
	xor 		esi,edi
	mov 		[flags+4],eax
	UPDATE_NZ
	ADD_CYCLES 	1
	ret

.data
szMRS2 db "MRS SPSR",13,10,0
.code
op_cmp:
;	pusha
;	invoke 		WriteConsole,hout,ADDR szMRS2,10,ADDR dummy,NULL
;	popa
	ADD_CYCLES 	1
	ret

op_cmps:
	cmp 		esi,edi
	UPDATE_FLAGS_SUB
	ADD_CYCLES 	1
	ret

.data
szMSR2 db "MSR SPSR",13,10,0
.code
op_cmn:
;	pusha
;	invoke 		WriteConsole,hout,ADDR szMSR2,10,ADDR dummy,NULL
;	popa
	ADD_CYCLES 	1
	ret

op_cmns:
	add 		esi,edi
	UPDATE_FLAGS
	ADD_CYCLES 	1
	ret
	
op_orr:
	shr 		eax,12
	or 		esi,edi		; Rs OR Op2
	and 		eax,0Fh
	.if eax==15
		ADD_CYCLES 	3
		mov 		physaddr,-1
		mov 		r15,esi
		ret
	.endif
	ADD_CYCLES 	1
	mov 		[r0+eax*4],esi
	ret
	
op_orrs:
	mov 		ebp,eax
	mov 		ecx,[shifterC]
	or 		esi,edi
	mov 		[flags+4],ecx
	UPDATE_NZ
	shr 		ebp,12
	and 		ebp,0Fh
	ADD_CYCLES 	1
	mov 		[r0+ebp*4],esi
	ret
	
op_mov:
	shr 		eax,12
	and 		eax,0Fh
	.if eax==15
		ADD_CYCLES 	3
		;add 		edi,4
		mov 		physaddr,-1
		mov 		r15,edi
		ret
	.endif
	ADD_CYCLES 	1
	mov 		[r0+eax*4],edi
	ret
	
op_movs:
	mov 		ebp,eax
	mov 		ecx,[shifterC]
	or 		edi,edi
	mov 		[flags+4],ecx
	UPDATE_NZ
	shr 		ebp,12
	and 		ebp,0Fh
	.if ebp==15
		ADD_CYCLES 	2
		mov		physaddr,-1
	.endif
	mov 		[r0+ebp*4],edi
	ADD_CYCLES 	1
	ret
	
op_bic:
	not 		edi
	shr 		eax,12
	and 		esi,edi		; Rs AND (NOT Op2)
	and 		eax,0Fh
	ADD_CYCLES 	1
	mov 		[r0+eax*4],esi
	ret
	
op_bics:
	not 		edi
	mov 		ecx,[shifterC]
	mov 		ebp,eax
	and 		esi,edi
	mov 		[flags+4],ecx
	UPDATE_NZ
	shr 		ebp,12
	ADD_CYCLES 	1
	and 		ebp,0Fh
	mov 		[r0+ebp*4],esi
	ret
	
op_mvn:
	shr 		eax,12
	and 		eax,0Fh
	not 		edi 
	.if eax==15
		;cmp 		edi,[r0+56]
		;jz 		@@not_mvn_pc_lr
		;add 		edi,4
		;@@not_mvn_pc_lr:
		mov 		physaddr,-1
	.endif
	mov 		[r0+eax*4],edi
	ADD_CYCLES 	1
	ret
	
op_mvns:
	mov 		ebp,eax
	mov 		ecx,[shifterC]
	not 		edi 
	mov 		[flags+4],ecx
	UPDATE_NZ
	shr 		ebp,12
	and 		ebp,0Fh
	mov 		[r0+ebp*4],edi
	ADD_CYCLES 	1
	ret
	

.data
align 16
; Jump table
data_proc_ops dd OFFSET op_and
	dd OFFSET op_ands	
	dd OFFSET op_eor
	dd OFFSET op_eors
	dd OFFSET op_sub
subs_handler dd OFFSET op_subs
	dd OFFSET op_rsb
	dd OFFSET op_rsbs
	dd OFFSET op_add
	dd OFFSET op_adds
	dd OFFSET op_adc
	dd OFFSET op_adcs
	dd OFFSET op_sbc
	dd OFFSET op_sbcs
	dd OFFSET op_rsc
	dd OFFSET op_rscs
	dd OFFSET op_tst
	dd OFFSET op_tsts
	dd OFFSET op_teq
	dd OFFSET op_teqs
	dd OFFSET op_cmp
	dd OFFSET op_cmps
	dd OFFSET op_cmn
	dd OFFSET op_cmns
	dd OFFSET op_orr
	dd OFFSET op_orrs
	dd OFFSET op_mov
	dd OFFSET op_movs
	dd OFFSET op_bic
	dd OFFSET op_bics
	dd OFFSET op_mvn
	dd OFFSET op_mvns
.code





;hword_xfer_00:
;	mov ebp,r15
;	mov edi,eax
;	sub ebp,8
;	DIV0
;	ret
;hword_xfer_01:	; STRH Rd,[Rn],-disp
;	mov ebx,eax
;	mov edx,eax
;	shr ebx,16
;	shr edx,12
;	and ebx,0Fh
;	and edx,0Fh
;	mov ebx,[r0+ebx*4]
;	mov edx,[r0+edx*4]
;	;invoke arm9_write_word,ebx,edx
;	;ADD_CYCLES 4
;	WRHWORD ebx,dx,edx
;	ret
	
;hword_xfer_02:	; STRSB Rd,[Rn],-disp
;	mov ebx,eax
;	mov edx,eax
;	shr ebx,16
;	shr edx,12
;	and ebx,0Fh
;	and edx,0Fh
;	mov ebx,[r0+ebx*4]
;	mov edx,[r0+edx*4]
;	invoke arm9_write_byte,ebx,edx
;	ADD_CYCLES STR_CYCLES
;	ret
	
;hword_xfer_03:	; STRSH Rd,[Rn],-disp
;	mov ebx,eax
;	mov edx,eax
;	shr ebx,16
;	shr edx,12
;	and ebx,0Fh
;	and edx,0Fh
;	mov ebx,[r0+ebx*4]
;	mov edx,[r0+edx*4]
;	invoke arm9_write_word,ebx,edx
;	ADD_CYCLES STR_CYCLES
;	ret
	
;hword_xfer_04:
;	DIV0
;	ret
;hword_xfer_05:	; LDRH Rd,[Rn],-disp
;	mov ebx,eax
;	mov edx,eax
;	shr ebx,16
;	shr edx,12
;	and ebx,0Fh
;	and edx,0Fh
;	mov ebx,[r0+ebx*4]
;	push edx
;	invoke arm9_read_word,ebx
;	pop edx
;	mov [r0+edx*4],eax
;	ADD_CYCLES 4
;	ret
	
;hword_xfer_06:	; LDRSB Rd,[Rn],-disp
;	mov ebx,eax
;	mov edx,eax
;	shr ebx,16
;	shr edx,12
;	and ebx,0Fh
;	and edx,0Fh
;	RDBYTE <dword ptr [r0+ebx*4]>
;	movsx eax,al
;	mov [r0+edx*4],eax
;	ret

;hword_xfer_07:	; LDRSH Rd,[Rn],-disp
;	mov ebx,eax
;	mov edx,eax
;	shr ebx,16
;	shr edx,12
;	and ebx,0Fh
;	and edx,0Fh
;	RDHWORD <dword ptr [r0+ebx*4]>
;	movsx eax,ax
;	mov [r0+edx*4],eax
;	ret
	
hword_xfer_08:
	DIV0
	ret
	
hword_xfer_09:	; STRH Rd,[Rn],-disp!
	mov 		ebx,eax
	mov 		edx,eax
	xor 		esi,-1
	shr 		ebx,16
	shr 		edx,12
	inc 		esi
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ecx,[r0+ebx*4]
	add 		esi,ecx
	mov 		[r0+ebx*4],esi
	WRHWORD 	ecx,dx,<dword ptr [r0+edx*4]>
	ret

hword_xfer_0A:	; STRSB Rd,[Rn],-disp!
	mov 		ebx,eax
	mov 		edx,eax
	xor 		esi,-1
	shr 		ebx,16
	shr 		edx,12
	inc 		esi
	and 		ebx,0Fh
	and 		edx,0Fh
	push 		ebx
	mov 		ebx,[r0+ebx*4]
	mov 		edx,[r0+edx*4]
	add 		esi,ebx
	push 		esi
	;invoke 	arm9_write_byte,ebx,edx
	pop 		esi
	pop 		ebx
	mov 		[r0+ebx*4],esi
	ADD_CYCLES 	STR_CYCLES
	ret
	
hword_xfer_0B:	; STRSH Rd,[Rn],-disp!
	mov 		ebx,eax
	mov 		edx,eax
	xor 		esi,-1
	shr 		ebx,16
	shr 		edx,12
	inc 		esi
	and 		ebx,0Fh
	and 		edx,0Fh
	push 		ebx
	mov 		ebx,[r0+ebx*4]
	mov 		edx,[r0+edx*4]
	add 		esi,ebx
	push 		esi
	;invoke 	arm9_write_word,ebx,edx
	pop 		esi
	pop 		ebx
	mov 		[r0+ebx*4],esi
	ADD_CYCLES 	4
	ret
	
hword_xfer_0C:
	DIV0
	ret
	
hword_xfer_0D:	; LDRH Rd,[Rn],-disp!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ecx,[r0+ebx*4]
	sub 		[r0+ebx*4],esi
	RDHWORD 	ecx
	mov 		[r0+edx*4],eax
	ret
	
hword_xfer_0E:	; LDRSB Rd,[Rn],-disp!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ecx,[r0+ebx*4]
	sub 		[r0+ebx*4],esi
	RDBYTE 		ecx
	movsx 		eax,al
	mov 		[r0+edx*4],eax
	ret

hword_xfer_0F:	; LDRSH Rd,[Rn],-disp!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ecx,[r0+ebx*4]
	sub 		[r0+ebx*4],esi
	RDHWORD 	ecx
	movsx 		eax,ax
	mov 		[r0+edx*4],eax
	ret
	
;hword_xfer_10:
;	DIV0
;	ret
;hword_xfer_11:	; STRH Rd,[Rn],+disp
;	mov ebx,eax
;	mov edx,eax
;	shr ebx,16
;	shr edx,12
;	and ebx,0Fh
;	and edx,0Fh
;	mov ebx,[r0+ebx*4]
;	add r15,4
;	mov edx,[r0+edx*4]
;	;invoke arm9_write_word,ebx,edx
;	;ADD_CYCLES STR_CYCLES
;	WRHWORD ebx,dx,edx
;	sub r15,4
;	ret
;
;hword_xfer_12:
;	DIV0
;	ret
;hword_xfer_13:
;	DIV0
;	ret
;hword_xfer_14:
;	DIV0
;	ret
;hword_xfer_15:	; LDRH Rd,[Rn],+disp
;	mov ebx,eax
;	mov edx,eax
;	shr ebx,16
;	shr edx,12
;	and ebx,0Fh
;	and edx,0Fh
;	;mov ebx,[r0+ebx*4]
;	RDHWORD <dword ptr [r0+ebx*4]>
;	mov [r0+edx*4],eax
;	ret

;hword_xfer_16:
;	mov ebx,eax
;	mov edx,eax
;	shr ebx,16
;	shr edx,12
;	and ebx,0Fh
;	and edx,0Fh
;	RDBYTE <dword ptr [r0+ebx*4]>
;	movsx eax,al
;	mov [r0+edx*4],eax
;	ret

;hword_xfer_17:	; LDRSH Rd,[Rn],+disp
;	mov ebx,eax
;	mov edx,eax
;	shr ebx,16
;	shr edx,12
;	and ebx,0Fh
;	and edx,0Fh
;	RDHWORD <dword ptr [r0+ebx*4]>
;	movsx eax,ax
;	mov [r0+edx*4],eax
;	ret

hword_xfer_18:
	DIV0
	ret
hword_xfer_19:	; STRH Rd,[Rn],+disp!
	mov ebx,eax
	mov edx,eax
	shr ebx,16
	shr edx,12
	and ebx,0Fh
	and edx,0Fh
	mov ecx,[r0+ebx*4]
	add r15,4
	;mov edx,[r0+edx*4]
	add esi,ecx
	mov [r0+ebx*4],esi
	WRHWORD ecx,dx,<dword ptr [r0+edx*4]>
	sub r15,4
	ret
	
hword_xfer_1A:
	mov ebx,eax
	mov edx,eax
	shr ebx,16
	shr edx,12
	and ebx,0Fh
	and edx,0Fh
	push ebx
	mov ebx,[r0+ebx*4]
	mov edx,[r0+edx*4]
	add esi,ebx
	push esi
	;invoke arm9_write_byte,ebx,edx
	pop esi
	pop ebx
	mov [r0+ebx*4],esi
	ADD_CYCLES STR_CYCLES
	ret
hword_xfer_1B:
	mov ebx,eax
	mov edx,eax
	shr ebx,16
	shr edx,12
	and ebx,0Fh
	and edx,0Fh
	push ebx
	mov ecx,[r0+ebx*4]
	add r15,4
	mov edx,[r0+edx*4]
	add esi,ecx
	push esi
	;invoke arm9_write_word,ebx,edx
	WRHWORD ecx,dx,edx
	pop esi
	pop ebx
	mov [r0+ebx*4],esi
	;ADD_CYCLES STR_CYCLES
	sub r15,4
	ret
hword_xfer_1C:
	DIV0
	ret
hword_xfer_1D:	; LDRH Rd,[Rn],+disp!
	mov ebx,eax
	mov edx,eax
	shr ebx,16
	shr edx,12
	and ebx,0Fh
	and edx,0Fh
	mov ecx,[r0+ebx*4]
	add [r0+ebx*4],esi
	RDHWORD ecx
	mov [r0+edx*4],eax
	ret

hword_xfer_1E:	; LDRSB Rd,[Rn],+disp!
	mov ebx,eax
	mov edx,eax
	shr ebx,16
	shr edx,12
	and ebx,0Fh
	and edx,0Fh
	mov ecx,[r0+ebx*4]
	add [r0+ebx*4],esi
	RDBYTE ecx
	movsx eax,al

	mov [r0+edx*4],eax
	ret

hword_xfer_1F:
	mov ebx,eax
	mov edx,eax
	shr ebx,16
	shr edx,12
	and ebx,0Fh
	and edx,0Fh
	mov ecx,[r0+ebx*4]
	add [r0+ebx*4],esi
	RDHWORD ecx
	movsx eax,ax
	mov [r0+edx*4],eax
	ret
	
hword_xfer_20:
	DIV0
	ret
	
hword_xfer_21:		; STRH Rd,[Rn,-disp]
	mov 		ebx,eax
	neg 		esi
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	add 		r15,4
	and 		edx,0Fh
	add 		esi,[r0+ebx*4]
	mov 		edx,[r0+edx*4]
	WRHWORD 		esi,dx,edx
	sub 		r15,4
	ret

; Invalid op
hword_xfer_22:	; STRSB Rd,[Rn,-disp]
	mov 		ebx,eax
	neg 		esi
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	add 		esi,[r0+ebx*4]
	mov 		edx,[r0+edx*4]
	WRBYTE 		esi,dl,edx
	ret

; Invalid op	
hword_xfer_23:	; STRSH Rd,[Rn,-disp]	
	mov 		ebx,eax
	neg 		esi
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	add 		esi,[r0+ebx*4]
	mov 		edx,[r0+edx*4]
	;invoke 	arm9_write_word,esi,edx
	WRHWORD 	esi,dx,edx
	;ADD_CYCLES 	STR_CYCLES
	ret
	
hword_xfer_24:
	DIV0
	ret
hword_xfer_25:	; LDRH Rd,[Rn,-disp]
	mov 		ebx,eax
	neg 		esi
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	add 		esi,[r0+ebx*4]
	RDHWORD 	esi
	mov 		[r0+edx*4],eax
	ret
	
hword_xfer_26:	; LDRSB Rd,[Rn,-disp]
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	neg 		esi
	mov 		ebx,[r0+ebx*4]
	add 		esi,ebx
	RDBYTE 		esi
	movsx 		eax,al
	mov 		[r0+edx*4],eax
	ret
	
hword_xfer_27:	; LDRSH Rd,[Rn,-disp]
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	neg 		esi
	add 		esi,[r0+ebx*4]
	RDHWORD 	esi
	movsx 		eax,ax
	mov 		[r0+edx*4],eax
	ret

hword_xfer_28:	
	DIV0
	ret
	
hword_xfer_29:	; STRH Rd,[Rn,-disp]!	
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	neg 		esi
	and 		edx,0Fh
	add 		esi,[r0+ebx*4]
	add 		r15,4
	mov 		edx,[r0+edx*4]
	mov 		[r0+ebx*4],esi
	WRHWORD 	esi,dx,edx
	sub 		r15,4
	ret
	
hword_xfer_2A:	; STRSB Rd,[Rn,-disp]!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	neg 		esi
	and 		edx,0Fh
	add 		esi,[r0+ebx*4]
	mov 		edx,[r0+edx*4]
	mov 		[r0+ebx*4],esi
	WRBYTE 		esi,dl,edx
	ret
	
hword_xfer_2B:	; STRSH Rd,[Rn,-disp]!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	neg 		esi
	and 		edx,0Fh
	add 		esi,[r0+ebx*4]
	mov 		edx,[r0+edx*4]
	mov 		[r0+ebx*4],esi
	WRHWORD 	esi,dx,edx
	ret
	
hword_xfer_2C:
	DIV0
	ret
	
hword_xfer_2D:	; LDRH Rd,[Rn,-disp]!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		edi,[r0+ebx*4]
	sub 		edi,esi
	mov 		[r0+ebx*4],edi
	RDHWORD 	edi
	mov 		[r0+edx*4],eax
	ret
	
hword_xfer_2E:	; LDRSB Rd,[Rn,-disp]!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		edi,[r0+ebx*4]
	sub 		edi,esi
	mov 		[r0+ebx*4],edi
	RDBYTE 		edi
	movsx 		eax,al
	mov 		[r0+edx*4],eax
	ret

hword_xfer_2F:	; LDRSH Rd,[Rn,-disp]!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		edi,[r0+ebx*4]
	sub 		edi,esi
	mov 		[r0+ebx*4],edi
	RDHWORD 	edi
	movsx 		eax,ax
	mov 		[r0+edx*4],eax
	ret
	
hword_xfer_30:
	DIV0
	ret
	
hword_xfer_31:	; STRH Rd,[Rn,+disp]
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	add 		esi,[r0+ebx*4]
	add 		r15,4
	WRHWORD 	esi,dx,<dword ptr [r0+edx*4]>
	sub 		r15,4
	ret
	
hword_xfer_32:	; STRSB Rd,[Rn,+disp]
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	add 		esi,[r0+ebx*4]
	add 		r15,4
	mov 		edx,[r0+edx*4]
	invoke 		arm9_write_byte,esi,edx
	ADD_CYCLES 	STR_CYCLES
	sub 		r15,4
	ret

; Invalid op
hword_xfer_33:	; STRSH Rs,[Rn,+disp]
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	add 		esi,[r0+ebx*4]
	add 		r15,4
	mov 		edx,[r0+edx*4]
	invoke 		arm9_write_word,esi,edx
	ADD_CYCLES 		STR_CYCLES
	sub 		r15,4
	ret
	
hword_xfer_34:
	DIV0
	ret
	
hword_xfer_35:	; LDRH Rd,[Rn,+disp]
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	add 		esi,[r0+ebx*4]
	RDHWORD 	esi
	mov 		[r0+edx*4],eax
	ret

hword_xfer_36:	; LDRSB Rd,[Rn,+disp]
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ebx,[r0+ebx*4]
	add 		esi,ebx
	RDBYTE 		esi
	movsx 		eax,al
	mov 		[r0+edx*4],eax
	ret
	
hword_xfer_37:	; LDRSH Rd,[Rn,+disp]
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ebx,[r0+ebx*4]
	add 		esi,ebx
	RDHWORD 	esi
	movsx	 	eax,ax
	mov 		[r0+edx*4],eax
	ret
	
hword_xfer_38:
	DIV0
	ret

hword_xfer_39:	; STRH Rd,[Rn,+disp]!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	add 		esi,[r0+ebx*4]
	add 		r15,4
	mov 		[r0+ebx*4],esi
	mov 		edx,[r0+edx*4]
	WRHWORD		esi,dx,edx
	sub 		r15,4
	ret
	
hword_xfer_3A:	; STRSB Rd,[Rn,+disp]!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	add 		esi,[r0+ebx*4]
	add 		r15,4
	mov 		[r0+ebx*4],esi
	mov 		edx,[r0+edx*4]
	;invoke 	arm9_write_byte,esi,edx
	ADD_CYCLES 	STR_CYCLES
	sub 		r15,4
	ret
	
hword_xfer_3B:	; STRSH Rd,[Rn,+disp]!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	add 		esi,[r0+ebx*4]
	add 		r15,4
	mov 		[r0+ebx*4],esi
	mov 		edx,[r0+edx*4]
	;invoke 	arm9_write_word,esi,edx
	ADD_CYCLES 	STR_CYCLES
	sub 		r15,4
	ret
	
hword_xfer_3C:
	DIV0
	ret
	
hword_xfer_3D:	; LDRH Rd,[Rn,+disp]!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		edi,[r0+ebx*4]
	add 		esi,edi
	mov 		[r0+ebx*4],esi
	RDHWORD 	esi
	mov 		[r0+edx*4],eax
	ret
	
hword_xfer_3E:	; LDRSB Rd,[Rn,+disp]!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		edi,[r0+ebx*4]
	add 		esi,edi
	mov 		[r0+ebx*4],esi
	RDBYTE 		esi
	movsx 		eax,al
	mov 		[r0+edx*4],eax
	ret
	
hword_xfer_3F:	; LDRSB Rd,[Rn,+disp]!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		edi,[r0+ebx*4]
	add 		esi,edi
	mov 		[r0+ebx*4],esi
	RDHWORD 	esi
	movsx 		eax,ax
	mov 		[r0+edx*4],eax
	ret


.data
align 16
; Jump table	
hword_xfer dd OFFSET hword_xfer_08 ;00
	dd OFFSET hword_xfer_09 ;01
	dd OFFSET hword_xfer_0A ;02
	dd OFFSET hword_xfer_0B ;03
	dd OFFSET hword_xfer_0C ;04
	dd OFFSET hword_xfer_0D ;05
	dd OFFSET hword_xfer_0E ;06
	dd OFFSET hword_xfer_0F ;07
	dd OFFSET hword_xfer_08
	dd OFFSET hword_xfer_09
	dd OFFSET hword_xfer_0A
	dd OFFSET hword_xfer_0B
	dd OFFSET hword_xfer_0C
	dd OFFSET hword_xfer_0D
	dd OFFSET hword_xfer_0E
	dd OFFSET hword_xfer_0F
	dd OFFSET hword_xfer_18 ;10
	dd OFFSET hword_xfer_19 ;11
	dd OFFSET hword_xfer_1A ;12
	dd OFFSET hword_xfer_1B ;13
	dd OFFSET hword_xfer_1C ;14
	dd OFFSET hword_xfer_1D ;15
	dd OFFSET hword_xfer_1E ;16
	dd OFFSET hword_xfer_1F ;17
	dd OFFSET hword_xfer_18
	dd OFFSET hword_xfer_19
	dd OFFSET hword_xfer_1A
	dd OFFSET hword_xfer_1B
	dd OFFSET hword_xfer_1C
	dd OFFSET hword_xfer_1D
	dd OFFSET hword_xfer_1E
	dd OFFSET hword_xfer_1F
	dd OFFSET hword_xfer_20
	dd OFFSET hword_xfer_21
	dd OFFSET hword_xfer_22
	dd OFFSET hword_xfer_23
	dd OFFSET hword_xfer_24
	dd OFFSET hword_xfer_25
	dd OFFSET hword_xfer_26
	dd OFFSET hword_xfer_27
	dd OFFSET hword_xfer_28
	dd OFFSET hword_xfer_29
	dd OFFSET hword_xfer_2A
	dd OFFSET hword_xfer_2B
	dd OFFSET hword_xfer_2C
	dd OFFSET hword_xfer_2D
	dd OFFSET hword_xfer_2E
	dd OFFSET hword_xfer_2F
	dd OFFSET hword_xfer_30
	dd OFFSET hword_xfer_31
	dd OFFSET hword_xfer_32
	dd OFFSET hword_xfer_33
	dd OFFSET hword_xfer_34
	dd OFFSET hword_xfer_35
	dd OFFSET hword_xfer_36
	dd OFFSET hword_xfer_37
	dd OFFSET hword_xfer_38
	dd OFFSET hword_xfer_39
	dd OFFSET hword_xfer_3A
	dd OFFSET hword_xfer_3B
	dd OFFSET hword_xfer_3C
	dd OFFSET hword_xfer_3D
	dd OFFSET hword_xfer_3E
	dd OFFSET hword_xfer_3F
.code



dword_xfer_0:
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ecx,[r0+ebx*4]
	sub 		[r0+ebx*4],esi
	push 		ecx
	RDWORD 		ecx
	mov 		[r0+edx*4],eax
	pop 		ecx
	inc 		edx
	add 		ecx,4
	RDWORD 		ecx
	mov 		[r0+edx*4],eax
	ret

dword_xfer_1:	; LDRD Rd,[Rn],-disp!
	DIV0
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ecx,[r0+ebx*4]
	sub 		[r0+ebx*4],esi
	push		ecx
	RDWORD 		ecx
	mov 		[r0+edx*4],eax
	pop 		ecx
	inc 		edx
	add 		ecx,4
	RDWORD 		ecx
	mov 		[r0+edx*4],eax
	ret

dword_xfer_2:
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ecx,[r0+ebx*4]
	add 		[r0+ebx*4],esi
	push 		ecx
	RDWORD 		ecx
	mov 		[r0+edx*4],eax
	pop 		ecx
	inc 		edx
	add 		ecx,4
	RDWORD 		ecx
	mov 		[r0+edx*4],eax
	ret
	
dword_xfer_3:	; LDRD Rd,[Rn],+disp!
	DIV0
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ecx,[r0+ebx*4]
	add 		[r0+ebx*4],esi
	push	 	ecx
	RDWORD 		ecx
	mov 		[r0+edx*4],eax
	pop 		ecx
	inc 		edx
	add 		ecx,4
	RDWORD 		ecx
	mov 		[r0+edx*4],eax
	ret

dword_xfer_4:	; LDRD Rd,[Rn,-disp]
	neg 		esi
dword_xfer_6:	; LDRD Rd,[Rn,+disp]
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	add 		esi,[r0+ebx*4]
	push 		esi
	RDWORD 		esi
	mov 		[r0+edx*4],eax
	pop 		esi
	inc 		edx
	add 		esi,4
	RDWORD 		esi
	mov 		[r0+edx*4],eax
	ret

dword_xfer_5:	; LDRD Rd,[Rn,-disp]!
	neg 		esi
dword_xfer_7:	; LDRD Rd,[Rn,+disp]!	
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	add 		esi,[r0+ebx*4]
	mov 		[r0+ebx*4],esi
	push 		esi
	RDWORD 		esi
	mov 		[r0+edx*4],eax
	pop 		esi
	inc 		edx
	add 		esi,4
	RDWORD 		esi
	mov 		[r0+edx*4],eax
	ret

dword_xfer_8:
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ecx,[r0+ebx*4]
	sub 		[r0+ebx*4],esi
	mov 		eax,[r0+4+edx*4]
	mov 		ebx,[r0+edx*4]
	push 		eax
	push 		ecx
	WRWORD 		ecx,ebx
	pop 		ecx
	pop 		ebx
	add 		ecx,4
	WRWORD 		ecx,ebx
	ret

dword_xfer_9:	; STRD Rd,[Rn],-disp!
	DIV0
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ecx,[r0+ebx*4]
	sub 		[r0+ebx*4],esi
	mov 		eax,[r0+4+edx*4]
	mov 		ebx,[r0+edx*4]
	push 		eax
	push 		ecx
	WRWORD 		ecx,ebx
	pop 		ecx
	pop 		ebx
	add 		ecx,4
	WRWORD 		ecx,ebx
	ret

dword_xfer_A:
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ecx,[r0+ebx*4]
	add 		[r0+ebx*4],esi
	mov 		eax,[r0+4+edx*4]
	mov 		ebx,[r0+edx*4]
	push 		eax
	push 		ecx
	WRWORD 		ecx,ebx
	pop 		ecx
	pop 		ebx
	add 		ecx,4
	WRWORD 		ecx,ebx
	ret
	
dword_xfer_B:	; STRD Rd,[Rn],+disp!
	DIV0
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ecx,[r0+ebx*4]
	add 		[r0+ebx*4],esi
	mov 		eax,[r0+4+edx*4]
	mov 		ebx,[r0+edx*4]
	push 		eax
	push 		ecx
	WRWORD 		ecx,ebx
	pop 		ecx
	pop 		ebx
	add 		ecx,4
	WRWORD 		ecx,ebx
	ret

dword_xfer_C:	; STRD Rd,[Rn,-disp]
	neg 		esi
dword_xfer_E:	; STRD Rd,[Rn,+disp]
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	add 		esi,[r0+ebx*4]
	mov 		eax,[r0+4+edx*4]
	mov 		ebx,[r0+edx*4]
	push		 eax
	push		 esi
	WRWORD 		esi,ebx
	pop 		esi
	pop 		ebx
	add 		esi,4
	WRWORD 		esi,ebx
	ret

dword_xfer_D:	; STRD Rd,[Rn,-disp]!
	neg 		esi
dword_xfer_F:	; STRD Rd,[Rn,+disp]!	
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	add 		esi,[r0+ebx*4]
	mov 		[r0+ebx*4],esi
	mov 		eax,[r0+4+edx*4]

	mov 		ebx,[r0+edx*4]
	push 		eax
	push 		esi
	WRWORD 		esi,ebx
	pop 		esi
	pop 		ebx

	add 		esi,4
	WRWORD 		esi,ebx
	ret
	
	
.data
align 8
dword_xfer LABEL DWORD
dd offset dword_xfer_0
dd offset dword_xfer_1
dd offset dword_xfer_2
dd offset dword_xfer_3
dd offset dword_xfer_4
dd offset dword_xfer_5
dd offset dword_xfer_6
dd offset dword_xfer_7
dd offset dword_xfer_8
dd offset dword_xfer_9
dd offset dword_xfer_A
dd offset dword_xfer_B
dd offset dword_xfer_C
dd offset dword_xfer_D
dd offset dword_xfer_E
dd offset dword_xfer_F
.code


data_xfer_00:	; STR Rd,[Rn],-imm
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ebx,[r0+ebx*4]
	mov 		edx,[r0+edx*4]
	WRWORD 		ebx,edx
	ret

data_xfer_01:	; LDR Rd,[Rn],-imm
	mov	 	ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ebx,[r0+ebx*4]
	RDWORD 		ebx
	mov 		[r0+edx*4],eax
	ret
	
data_xfer_02:	; STR Rd,[Rn],-imm!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	xor 		ecx,-1
	and 		edx,0Fh
	inc 		ecx		; xor -1 / inc = neg
	mov 		esi,[r0+ebx*4]	; Rn
	add 		r15,4
	mov 		edx,[r0+edx*4]	; Rd
	add 		ecx,esi
	cmp 		ebx,15
	je 		@@data_xfer_02_no_wb
	mov 		[r0+ebx*4],ecx
	@@data_xfer_02_no_wb:
	WRWORD 		esi,edx
	sub 		r15,4
	ret
	
data_xfer_03:	; LDR Rd,[Rn],-imm!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	xor 		ecx,-1
	and 		edx,0Fh
	inc 		ecx
	mov 		esi,[r0+ebx*4]
	add 		ecx,esi
	cmp 		ebx,15
	je 		@@data_xfer_03_no_wb
	mov 		[r0+ebx*4],ecx
	@@data_xfer_03_no_wb:
	RDWORD 		esi
	mov 		[r0+edx*4],eax
	ret
	
data_xfer_04:	; STRB Rd,[Rn],-disp
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ebx,[r0+ebx*4]
	mov 		edx,[r0+edx*4]
	WRBYTE 		ebx,dl,edx
	ret
	
data_xfer_05:	; LDRB Rd,[Rn],-disp
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ebx,[r0+ebx*4]
	RDBYTE 		ebx
	mov 		[r0+edx*4],eax
	ret
	
data_xfer_06:	; STRB Rd,[Rn],-disp!
	mov 		ebx,eax
	mov 		edx,eax
	xor 		ecx,-1
	shr 		ebx,16
	shr 		edx,12
	inc 		ecx
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		esi,[r0+ebx*4]	; Rn
	mov 		edx,[r0+edx*4]	; Rd
	add 		ecx,esi
	cmp 		ebx,15
	je 		@@data_xfer_06_no_wb
	mov 		[r0+ebx*4],ecx
	@@data_xfer_06_no_wb:
	WRBYTE 		esi,dl,edx
	ret
	
data_xfer_07:	; LDRB Rd,[Rn],-disp!
	mov 		ebx,eax
	mov 		edx,eax
	xor 		ecx,-1
	shr 		ebx,16
	shr 		edx,12
	inc 		ecx
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		esi,[r0+ebx*4]
	add 		ecx,esi
	cmp 		ebx,15
	je 		@@data_xfer_07_no_wb
	mov 		[r0+ebx*4],ecx
	@@data_xfer_07_no_wb:
	RDBYTE 		esi
	mov 		[r0+edx*4],eax
	ret
	
data_xfer_08:	; STR Rd,[Rn],+disp
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ebx,[r0+ebx*4]
	mov 		edx,[r0+edx*4]
	WRWORD 		ebx,edx
	ret
	
data_xfer_09:	; LDR Rd,[Rn],+disp
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ebx,[r0+ebx*4]
	RDWORD 		ebx
	mov 		[r0+edx*4],eax
	ret
	
data_xfer_0A:	; STR Rd,[Rn],+disp!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		esi,[r0+ebx*4]
	add 		r15,4
	mov 		edx,[r0+edx*4]
	add 		ecx,esi
	cmp 		ebx,15
	je 		@@data_xfer_0A_no_wb
	mov 		[r0+ebx*4],ecx
	@@data_xfer_0A_no_wb:
	WRWORD 		esi,edx
	sub 		r15,4
	ret

data_xfer_0B:	; LDR Rd,[Rn],+disp!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		esi,[r0+ebx*4]
	add 		ecx,esi
	cmp 		ebx,15
	je 		@@data_xfer_0B_no_wb
	mov 		[r0+ebx*4],ecx
	@@data_xfer_0B_no_wb:
	RDWORD 		esi
	mov 		[r0+edx*4],eax
	ret

data_xfer_0C:	; STRB Rd,[Rn],+disp
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ebx,[r0+ebx*4]
	mov 		edx,[r0+edx*4]
	WRBYTE 		ebx,dl,edx
	ret
	
data_xfer_0D:	; LDRB Rd,[Rn],+disp
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ebx,[r0+ebx*4]
	RDBYTE 		ebx
	mov 		[r0+edx*4],eax
	ret

data_xfer_0E:	; STRB Rd,[Rn],+disp!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		esi,[r0+ebx*4]
	mov 		edx,[r0+edx*4]
	add 		ecx,esi
	cmp 		ebx,15
	je 		@@data_xfer_0E_no_wb
	mov 		[r0+ebx*4],ecx
	@@data_xfer_0E_no_wb:
	WRBYTE 		esi,dl,edx
	;ADD_CYCLES 	STR_CYCLES
	ret

data_xfer_0F:	; LDRB Rd,[Rn],+imm!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		esi,[r0+ebx*4]
	add 		ecx,esi
	cmp 		ebx,15
	je 		@@data_xfer_0F_no_wb
	mov 		[r0+ebx*4],ecx
	@@data_xfer_0F_no_wb:
	RDBYTE 		esi
	mov 		[r0+edx*4],eax
	ret
	
data_xfer_10:	; STR Rd,[Rn,-disp]
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		esi,[r0+ebx*4]
	add 		r15,4
	mov 		edx,[r0+edx*4]
	sub 		esi,ecx
	WRWORD 		esi,edx
	sub	 	r15,4
	ret
	
data_xfer_11:	; LDR Rd,[Rn,-disp]
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ebx,[r0+ebx*4]
	sub 		ebx,ecx
	RDWORD 		ebx
	mov 		[r0+edx*4],eax
	;ADD_CYCLES 	LDR_CYCLES
	ret
	
data_xfer_12:	; STR Rd,[Rn,-disp]!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		esi,[r0+ebx*4]
	add 		r15,4
	mov 		edx,[r0+edx*4]
	sub 		esi,ecx
	mov 		[r0+ebx*4],esi
	WRWORD 		esi,edx
	sub 		r15,4
	ret
	
data_xfer_13:	; LDR Rd,[Rn,-disp]!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		esi,[r0+ebx*4]
	sub 		esi,ecx
	mov 		[r0+ebx*4],esi
	RDWORD 		esi
	mov 		[r0+edx*4],eax
	ret

data_xfer_14:	; STRB Rd,[Rn,-disp]
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		esi,[r0+ebx*4]
	mov 		edx,[r0+edx*4]
	sub 		esi,ecx
	WRBYTE 		esi,dl,edx
	;ADD_CYCLES 	STR_CYCLES
	ret
	
data_xfer_15:	; LDRB Rd,[Rn,-disp]
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ebx,[r0+ebx*4]
	sub 		ebx,ecx
	RDBYTE 		ebx
	;and 		eax,0FFh
	;ADD_CYCLES 	LDR_CYCLES
	mov 		[r0+edx*4],eax
	ret
	
data_xfer_16:	; STRB Rd,[Rn,-disp]!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		esi,[r0+ebx*4]
	mov 		edx,[r0+edx*4]
	sub 		esi,ecx
	mov 		[r0+ebx*4],esi
	WRBYTE 		esi,dl,edx
	;ADD_CYCLES 	STR_CYCLES
	ret
	
data_xfer_17:	; LDRB Rd,[Rn,-disp]!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		esi,[r0+ebx*4]
	sub 		esi,ecx
	mov 		[r0+ebx*4],esi
	RDBYTE 		esi
	mov 		[r0+edx*4],eax
	ret
	
data_xfer_18:	; STR Rd,[Rn,+disp]
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		esi,[r0+ebx*4]
	add 		r15,4
	mov 		edx,[r0+edx*4]
	add 		esi,ecx
	WRWORD 		esi,edx
	sub 		r15,4
	ret
	
data_xfer_19:	; LDR Rd,[Rn,+disp]
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ebx,[r0+ebx*4]
	add 		ecx,ebx
	RDWORD 		ecx
	mov 		[r0+edx*4],eax
	ret

data_xfer_1A:	; STR Rd,[Rn,+disp]!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		esi,[r0+ebx*4]
	add 		r15,4
	mov 		edx,[r0+edx*4]
	add 		esi,ecx
	mov 		[r0+ebx*4],esi
	WRWORD 		esi,edx
	sub 		r15,4
	ret
	
data_xfer_1B:	; LDR Rd,[Rn,+disp]!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		esi,[r0+ebx*4]
	add 		esi,ecx
	mov 		[r0+ebx*4],esi
	RDWORD 		esi
	mov 		[r0+edx*4],eax
	ret
	
data_xfer_1C:	; STRB Rd,[Rn,+disp]
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		esi,[r0+ebx*4]
	mov 		edx,[r0+edx*4]
	add 		esi,ecx
	WRBYTE 		esi,dl,edx
	;ADD_CYCLES 	STR_CYCLES
	ret
	
data_xfer_1D:	; LDRB Rd,[Rn,+disp]
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		ebx,[r0+ebx*4]
	add 		ecx,ebx
	RDBYTE 		ecx
	;and 		eax,0FFh
	;ADD_CYCLES 	LDR_CYCLES
	mov 		[r0+edx*4],eax
	ret

data_xfer_1E:	; STRB Rd,[Rn,+disp]!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		esi,[r0+ebx*4]
	mov 		edx,[r0+edx*4]
	add 		esi,ecx
	mov 		[r0+ebx*4],esi
	WRBYTE 		esi,dl,edx
	;ADD_CYCLES 	STR_CYCLES
	ret
	
data_xfer_1F:	; LDRB Rd,[Rn,+disp]!
	mov 		ebx,eax
	mov 		edx,eax
	shr 		ebx,16
	shr 		edx,12
	and 		ebx,0Fh
	and 		edx,0Fh
	mov 		esi,[r0+ebx*4]
	add 		esi,ecx
	mov 		[r0+ebx*4],esi
	RDBYTE 		esi
	;and 		eax,0FFh
	;ADD_CYCLES 	LDR_CYCLES
	mov 		[r0+edx*4],eax
	ret

.data
align 16
; Jump table
data_xfer dd OFFSET data_xfer_02 ;00
	dd OFFSET data_xfer_03 ;01
	dd OFFSET data_xfer_02
	dd OFFSET data_xfer_03
	dd OFFSET data_xfer_06 ;04
	dd OFFSET data_xfer_07 ;05
	dd OFFSET data_xfer_06
	dd OFFSET data_xfer_07
	dd OFFSET data_xfer_0A ;08
	dd OFFSET data_xfer_0B ;09
	dd OFFSET data_xfer_0A
	dd OFFSET data_xfer_0B
	dd OFFSET data_xfer_0E ;0C
	dd OFFSET data_xfer_0F ;0D
	dd OFFSET data_xfer_0E
	dd OFFSET data_xfer_0F
	dd OFFSET data_xfer_10
	dd OFFSET data_xfer_11
	dd OFFSET data_xfer_12
	dd OFFSET data_xfer_13
	dd OFFSET data_xfer_14
	dd OFFSET data_xfer_15
	dd OFFSET data_xfer_16
	dd OFFSET data_xfer_17
	dd OFFSET data_xfer_18
	dd OFFSET data_xfer_19
	dd OFFSET data_xfer_1A
	dd OFFSET data_xfer_1B
	dd OFFSET data_xfer_1C
	dd OFFSET data_xfer_1D
	dd OFFSET data_xfer_1E
	dd OFFSET data_xfer_1F
.code


block_xfer_00:	; STMED Rn,{...}
	add r15,4
	mov ebx,eax
	shr ebx,16
	mov ecx,15
	and ebx,0Fh		
	mov ebp,[r0+ebx*4]	; get Rn
	shl eax,16
	@@block_xfer_00_loop:
		shl eax,1
		jnc @@block_xfer_00_no_xfer
			;.if ecx == ebx
			;	mov edx,ebp
			;.else
				mov edx,[r0+ecx*4]
			;.endif
			push eax
			push ecx
			push ebp
			push ebx
			WRWORD ebp,edx
			pop ebx
			pop ebp
			pop ecx
			sub ebp,4
			pop eax
		@@block_xfer_00_no_xfer:
		dec ecx
		jns @@block_xfer_00_loop
	sub r15,4
	ADD_CYCLES 8
	ret
	
block_xfer_01:	; LDMFA Rn,{...}
	mov ebx,eax
	mov esi,OFFSET r0
	shr ebx,16
	mov ecx,15
	and ebx,0Fh		
	mov edx,eax
	.if eax & 8000h
		mov physaddr,-1
	.endif
	mov ebp,[esi+ebx*4]	; get Rn
	shl edx,16
	@@block_xfer_01_loop:
		shl edx,1
		jnc @@block_xfer_01_no_xfer
			push ebp
			push ecx
			RDWORD ebp
			pop ecx
			pop ebp
			mov [r0+ecx*4],eax
			sub ebp,4 
		@@block_xfer_01_no_xfer:
		dec ecx
		jns @@block_xfer_01_loop
	ADD_CYCLES 8
	ret
	
block_xfer_02:	; STMED Rn!,{...}
	add r15,4
	mov ebx,eax
	shr ebx,16
	mov ecx,15
	and ebx,0Fh		
	mov ebp,[r0+ebx*4]	; get Rn
	shl eax,16
	push ebx
	@@block_xfer_02_loop:
		shl eax,1
		jnc @@block_xfer_02_no_xfer
			mov edx,[r0+ecx*4]
			push eax
			push ecx
			push ebp
			;push ebx
			WRWORD ebp,edx
			;pop ebx
			pop ebp
			pop ecx
			sub ebp,4
			pop eax
		@@block_xfer_02_no_xfer:
		dec ecx
		jns @@block_xfer_02_loop
	pop ebx

	mov [r0+ebx*4],ebp
	sub r15,4
	ADD_CYCLES 8
	ret
	
block_xfer_03:	; LDMFA Rn!,{...}
	mov ebx,eax
	shr ebx,16
	mov ecx,15
	and ebx,0Fh		
	mov edx,eax
	.if eax & 8000h
		mov physaddr,-1
	.endif
	shl edx,16
	mov ebp,[r0+ebx*4]	; get Rn
	push ebx
	@@block_xfer_03_loop:
		shl edx,1
		jnc @@block_xfer_03_no_xfer
			push ebp
			push ecx
			RDWORD ebp
			pop ecx
			pop ebp
			mov [r0+ecx*4],eax
			sub ebp,4 
		@@block_xfer_03_no_xfer:
		dec ecx
		jns @@block_xfer_03_loop
	pop ebx
	mov [r0+ebx*4],ebp
	ret
	
block_xfer_04:
	add r15,4
	mov ebx,eax
	shr ebx,16
	mov ecx,15
	and ebx,0Fh		
	mov ebp,[r0+ebx*4]	; get Rn
	shl eax,16
	@@block_xfer_04_loop:
		shl eax,1
		jnc @@block_xfer_04_no_xfer
			;.if ecx == ebx
			;	mov edx,ebp
			;.else
				mov edx,[r0+ecx*4]
			;.endif
			push eax
			push ebx
			push ecx
			push ebp
			WRWORD ebp,edx
			pop ebp
			pop ecx
			sub ebp,4
			pop ebx
			pop eax
		@@block_xfer_04_no_xfer:
		dec ecx
		jns @@block_xfer_04_loop
	sub r15,4
	ADD_CYCLES 8
	ret
	
block_xfer_05:
	DIV0
	ret
block_xfer_06:		; STMEDS Rn!,{...}
	add r15,4
	mov ebx,eax
	shr ebx,16
	mov ecx,15
	and ebx,0Fh		
	mov ebp,[r0+ebx*4]	; get Rn
	shl eax,16
	push ebx
	@@block_xfer_06_loop:
		shl eax,1
		jnc @@block_xfer_06_no_xfer
			.if ecx == ebx
				mov edx,ebp
			.else
				mov edx,[r0+ecx*4]
			.endif
			push eax
			push ebx
			push ecx
			push ebp
			WRWORD ebp,edx
			pop ebp
			pop ecx
			sub ebp,4
			pop ebx
			pop eax
		@@block_xfer_06_no_xfer:
		dec ecx
		jns @@block_xfer_06_loop
	pop ebx
	mov [r0+ebx*4],ebp
	sub r15,4
	ADD_CYCLES 8
	ret
	
block_xfer_07:	; LDMFAS Rn!,{...}
	mov ebx,eax
	shr ebx,16
	mov ecx,15
	and ebx,0Fh		
	mov edx,eax
	.if eax & 8000h
		mov physaddr,-1
		;pusha	
		;invoke wsprintf,ADDR szBuffer,ADDR szFormat102,executed,r15,bankn,r0,r1,r2,r3,r4,r5,r6,r7,r8,r9,r10,r11,r12,r13,r14,r15
		;invoke lstrlen,ADDR szBuffer
		;invoke WriteConsole,hout,ADDR szBuffer,eax,ADDR dummy,0
		;popa

	.endif
	shl edx,16
	mov ebp,[r0+ebx*4]	; get Rn
	push ebx
	@@block_xfer_07_loop:
		shl edx,1
		jnc @@block_xfer_07_no_xfer
			push ebp
			push ecx
			RDWORD ebp
			pop ecx
			pop ebp
			.if ecx==13
				mov r13_usr,eax
			.elseif ecx==14
				mov r14_usr,eax
			.elseif ecx==15
				mov r15,eax 

				mov ebx,r17
				SWAP_REGISTERS
				mov r16,ebx
				test ebx,20h
				jz still_arm_2
					mov cb_execute,OFFSET cpu_execute_until_thumb
					mov dword ptr [esp],OFFSET cpu_execute_until_thumb
				still_arm_2:
				UNPACK_FLAGS
				mov dword ptr [subs_handler],offset op_subs
				mov physaddr,-1
			.else
				mov [r0+ecx*4],eax
			.endif
			sub ebp,4 
		@@block_xfer_07_no_xfer:
		dec ecx
		jns @@block_xfer_07_loop
	pop ebx
	mov [r0+ebx*4],ebp
	ret
	
block_xfer_08:		; STMIA Rn,{...}
	add r15,4
	mov ebx,eax
	shr ebx,16
	mov ecx,0
	and ebx,0Fh		
	mov ebp,[r0+ebx*4]	; get Rn
	@@block_xfer_08_loop:
		shr eax,1
		jnc @@block_xfer_08_no_xfer
			.if ecx == ebx
				mov edx,ebp
			.else
				mov edx,[r0+ecx*4]
			.endif
			push eax
			push ebx
			push ecx
			push ebp
			WRWORD ebp,edx
			pop ebp
			pop ecx
			add ebp,4
			pop ebx
			pop eax
		@@block_xfer_08_no_xfer:
		inc ecx
		cmp ecx,16
		jne @@block_xfer_08_loop
	sub r15,4
	ADD_CYCLES 8
	ret
	
block_xfer_09:	; LDMFD Rn,{...}
	mov ebx,eax
	shr ebx,16
	mov ecx,0
	and ebx,0Fh		
	mov edx,eax
	.if eax & 8000h
		mov physaddr,-1
	.endif
	mov ebp,[r0+ebx*4]	; get Rn

			
	@@block_xfer_09_loop:
		shr edx,1
		jnc @@block_xfer_09_no_xfer
			push ebp
			push ecx
			RDWORD ebp ;[r0+ebx*4]
			pop ecx
			pop ebp
			mov [r0+ecx*4],eax
			add ebp,4 
		@@block_xfer_09_no_xfer:
		inc ecx
		cmp ecx,16
		jne @@block_xfer_09_loop
	ADD_CYCLES 8
	ret

block_xfer_0A:		; STMIA Rn!,{...}
	add r15,4
	mov ebx,eax
	shr ebx,16
	mov ecx,0
	and ebx,0Fh		
	mov ebp,[r0+ebx*4]	; get Rn
	push ebx
	@@block_xfer_0A_loop:
		shr eax,1
		jnc @@block_xfer_0A_no_xfer
			.if ecx == ebx
				mov edx,ebp
			.else
				mov edx,[r0+ecx*4]
			.endif
			push eax
			push ebx
			push ecx
			push ebp
			WRWORD ebp,edx
			pop ebp
			pop ecx
			add ebp,4
			pop ebx
			pop eax
		@@block_xfer_0A_no_xfer:
		inc ecx
		cmp ecx,16
		jne @@block_xfer_0A_loop
	pop ebx
	mov [r0+ebx*4],ebp
	sub r15,4
	ret

block_xfer_0B:	; LDMFD Rn!,{...}
	mov ebx,eax
	shr ebx,16
	mov ecx,0
	and ebx,0Fh		
	mov edx,eax
	.if eax & 8000h
		mov physaddr,-1
	.endif	
	push ebx
	mov ebp,[r0+ebx*4]
	@@block_xfer_0B_loop:
		shr edx,1
		jnc @@block_xfer_0B_no_xfer
			push ebp
			push ecx
			RDWORD ebp
			pop ecx
			pop ebp
			mov [r0+ecx*4],eax
			add ebp,4
		@@block_xfer_0B_no_xfer:
		inc ecx
		cmp ecx,16
		jne @@block_xfer_0B_loop
	pop ebx
	mov [r0+ebx*4],ebp
	ret

block_xfer_0C:		; STMIAS Rn,{...}
	add r15,4
	mov ebx,eax
	shr ebx,16
	mov ecx,0
	and ebx,0Fh		
	mov ebp,[r0+ebx*4]	; get Rn
	@@block_xfer_0C_loop:
		shr eax,1
		jnc @@block_xfer_0C_no_xfer
			.if ecx == ebx
				mov edx,ebp
			.else
				mov edx,[r0+ecx*4]
			.endif
			push eax
			push ebx
			push ecx
			push ebp
			WRWORD ebp,edx
			pop ebp
			pop ecx
			add ebp,4
			pop ebx
			pop eax
		@@block_xfer_0C_no_xfer:
		inc ecx
		cmp ecx,16
		jne @@block_xfer_0C_loop
	sub r15,4
	ADD_CYCLES 8
	ret
	
block_xfer_0D:
	mov ebx,eax
	shr ebx,16
	mov ecx,0
	and ebx,0Fh		
	mov edx,eax
	mov ebp,[r0+ebx*4]
	.if eax & 8000h
		mov physaddr,-1
	.endif	
	@@block_xfer_0D_loop:
		shr edx,1
		jnc @@block_xfer_0D_no_xfer
			push ecx
			push ebp
			RDWORD ebp
			pop ebp
			pop ecx
			mov [r0+ecx*4],eax
			add ebp,4 
		@@block_xfer_0D_no_xfer:
		inc ecx
		cmp ecx,16
		jne @@block_xfer_0D_loop
	ADD_CYCLES 8
	ret
	
block_xfer_0E:	; STMIAS Rn!,{...}
	add r15,4
	mov ebx,eax
	shr ebx,16
	mov ecx,0
	and ebx,0Fh		
	mov ebp,[r0+ebx*4]	; get Rn
	push ebx
	@@block_xfer_0E_loop:
		shr eax,1
		jnc @@block_xfer_0E_no_xfer
			.if ecx == 13
				mov edx,r13_usr
			.elseif ecx == 14
				mov edx,r14_usr
			.else
				mov edx,[r0+ecx*4]
			.endif
			push eax
			;push ebx
			push ecx
			push ebp
			WRWORD ebp,edx
			pop ebp
			pop ecx
			add ebp,4
			;pop ebx
			pop eax
		@@block_xfer_0E_no_xfer:
		inc ecx
		cmp ecx,16
		jne @@block_xfer_0E_loop
	pop ebx
	mov [r0+ebx*4],ebp
	sub r15,4
	ret

block_xfer_0F:
	DIV0
	ret
block_xfer_10:	; STMFD Rn,{...}
	add r15,4
	mov ebx,eax
	shr ebx,16
	mov ecx,15
	and ebx,0Fh		
	mov ebp,[r0+ebx*4]	; get Rn
	shl eax,16
	@@block_xfer_10_loop:
		shl eax,1
		jnc @@block_xfer_10_no_xfer
			;.if ecx == ebx
			;	mov edx,ebp
			;.else
				mov edx,[r0+ecx*4]
			;.endif
			push eax
			sub ebp,4
			push ecx
			push ebp
			push ebx
			WRWORD ebp,edx
			pop ebx
			pop ebp
			pop ecx
			pop eax
		@@block_xfer_10_no_xfer:
		dec ecx
		jns @@block_xfer_10_loop
	sub r15,4
	ADD_CYCLES 8
	ret

block_xfer_11:	; LDMEA Rn,{...}
	mov ebx,eax
	shr ebx,16
	mov ecx,15
	and ebx,0Fh		
	mov edx,eax
	mov ebp,[r0+ebx*4]	; get Rn
	shl edx,16
	.if eax & 8000h
		mov physaddr,-1
	.endif	
	@@block_xfer_11_loop:
		shl edx,1
		jnc @@block_xfer_11_no_xfer
			sub ebp,4 
			push ebp
			push ecx
			RDWORD ebp
			pop ecx
			pop ebp
			mov [r0+ecx*4],eax
		@@block_xfer_11_no_xfer:
		dec ecx
		jns @@block_xfer_11_loop
	ADD_CYCLES 8
	ret
	
block_xfer_12:	; STMFD Rn!,{...}
	add r15,4
	mov ebx,eax
	shr ebx,16
	mov ecx,15
	and ebx,0Fh		
	mov ebp,[r0+ebx*4]	; get Rn
	shl eax,16
	@@block_xfer_12_loop:
		shl eax,1
		jnc @@block_xfer_12_no_xfer
			;.if ecx == ebx
			;	mov edx,ebp
			;.else
				mov edx,[r0+ecx*4]
			;.endif
			push eax
			sub ebp,4
			push ecx
			push ebp
			push ebx

			WRWORD ebp,edx
			pop ebx
			pop ebp
			pop ecx
			pop eax
		@@block_xfer_12_no_xfer:
		dec ecx
		jns @@block_xfer_12_loop
	mov [r0+ebx*4],ebp
	sub r15,4
	ADD_CYCLES 8
	ret

block_xfer_13:
	mov ebx,eax
	mov esi,OFFSET r0
	shr ebx,16
	mov ecx,15
	and ebx,0Fh		
	mov edx,eax
	mov ebp,[esi+ebx*4]	; get Rn
	shl edx,16
	.if eax & 8000h
		mov physaddr,-1
	.endif
	push ebx
	@@block_xfer_13_loop:
		shl edx,1
		jnc @@block_xfer_13_no_xfer
			sub ebp,4 
			push ebp
			push ecx
			RDWORD ebp
			pop ecx
			pop ebp
			mov [r0+ecx*4],eax
		@@block_xfer_13_no_xfer:
		dec ecx
		jns @@block_xfer_13_loop
	pop ebx
	ADD_CYCLES 8
	mov [r0+ebx*4],ebp
	ret
	
block_xfer_14:
	add r15,4
	mov ebx,eax
	shr ebx,16
	mov ecx,15
	and ebx,0Fh		
	mov ebp,[r0+ebx*4]	; get Rn
	shl eax,16
	@@block_xfer_14_loop:
		shl eax,1
		jnc @@block_xfer_14_no_xfer
			.if ecx == 13
				mov edx,r13_usr
			.elseif ecx == 14
				mov edx,r14_usr
			.else
				mov edx,[r0+ecx*4]
			.endif
			push eax
			sub ebp,4
			push ecx
			push ebp
			push ebx
			WRWORD ebp,edx
			pop ebx
			pop ebp
			pop ecx
			pop eax
		@@block_xfer_14_no_xfer:
		dec ecx
		jns @@block_xfer_14_loop
	sub r15,4
	ADD_CYCLES 8
	ret
	
block_xfer_15:
	DIV0
	ret
block_xfer_16:
	DIV0
	ret
block_xfer_17:
	DIV0
	ret

block_xfer_18:	; STMIB
	add r15,4
	mov ebx,eax
	shr ebx,16
	mov ecx,0
	and ebx,0Fh		
	mov ebp,[r0+ebx*4]	; get Rn
	;shl eax,16
	@@block_xfer_18_loop:
		shr eax,1
		jnc @@block_xfer_18_no_xfer
			mov edx,[r0+ecx*4]
			push eax
			;push edx
			add ebp,4
			push ecx
			push ebp
			WRWORD ebp,edx
			pop ebp
			pop ecx
			;pop edx
			pop eax
		@@block_xfer_18_no_xfer:
		inc ecx
		cmp ecx,16
		jne @@block_xfer_18_loop
	sub r15,4
	ADD_CYCLES 8
	ret

block_xfer_19:	; LDMIB
	mov ebx,eax
	mov esi,OFFSET r0
	shr ebx,16
	mov ecx,0
	and ebx,0Fh		
	mov edx,eax
	mov ebp,[esi+ebx*4]	; get Rn
	.if eax & 8000h
		mov physaddr,-1
	.endif
	@@block_xfer_19_loop:
		shr edx,1
		jnc @@block_xfer_19_no_xfer
			add ebp,4 
			push ebp
			push ecx
			RDWORD ebp
			pop ecx
			pop ebp
			mov [r0+ecx*4],eax
		@@block_xfer_19_no_xfer:
		inc ecx
		cmp ecx,16
		jne @@block_xfer_19_loop
	ADD_CYCLES 8
	ret
	
block_xfer_1A:
	DIV0
	ret
block_xfer_1B:
	DIV0
	ret
block_xfer_1C:
	DIV0
	ret
block_xfer_1D:	; LDMEDS Rn,{...}
	mov ebx,eax
	mov esi,OFFSET r0
	shr ebx,16
	mov ecx,0
	and ebx,0Fh		
	mov edx,eax
	mov ebp,[esi+ebx*4]	; get Rn
	.if eax & 8000h
		mov physaddr,-1
	.endif
	@@block_xfer_1D_loop:
		shr edx,1
		jnc @@block_xfer_1D_no_xfer
			add ebp,4 
			;push esi
			push ebp
			push ecx
			RDWORD ebp
			pop ecx
			pop ebp
			mov [r0+ecx*4],eax
		@@block_xfer_1D_no_xfer:
		inc ecx
		cmp ecx,16
		jne @@block_xfer_1D_loop
	ADD_CYCLES 8
	ret
	
block_xfer_1E:
	DIV0
	ret
block_xfer_1F:
	DIV0
	ret


.data
align 16	
block_xfer dd OFFSET block_xfer_00
	dd OFFSET block_xfer_01
	dd OFFSET block_xfer_02
	dd OFFSET block_xfer_03
	dd OFFSET block_xfer_04
	dd OFFSET block_xfer_05
	dd OFFSET block_xfer_06
	dd OFFSET block_xfer_07
	dd OFFSET block_xfer_08
	dd OFFSET block_xfer_09
	dd OFFSET block_xfer_0A
	dd OFFSET block_xfer_0B
	dd OFFSET block_xfer_0C
	dd OFFSET block_xfer_0D
	dd OFFSET block_xfer_0E
	dd OFFSET block_xfer_0F
	dd OFFSET block_xfer_10
	dd OFFSET block_xfer_11
	dd OFFSET block_xfer_12
	dd OFFSET block_xfer_13
	dd OFFSET block_xfer_14
	dd OFFSET block_xfer_15
	dd OFFSET block_xfer_16
	dd OFFSET block_xfer_17
	dd OFFSET block_xfer_18
	dd OFFSET block_xfer_19
	dd OFFSET block_xfer_1A
	dd OFFSET block_xfer_1B
	dd OFFSET block_xfer_1C
	dd OFFSET block_xfer_1D
	dd OFFSET block_xfer_1E
	dd OFFSET block_xfer_1F
.code	


;----------------------------------------------------
; Multiply operations
; eax contains Rs, ebx contains the Rm number (0..15)
; edi contains the full opcode
;----------------------------------------------------

op_mul_:
	shr 		edi,16
	mul 		dword ptr [r0+ebx*4]		
	and 		edi,0Fh
	ADD_CYCLES 	2
	mov 		[r0+edi*4],eax
	ret
	
op_muls_:
	shr 		edi,16
	mul 		dword ptr [r0+ebx*4]
	mov 		esi,eax
	cmp 		eax,0
	UPDATE_NZ
	and 		edi,0Fh
	ADD_CYCLES 	2
	mov 		[r0+edi*4],esi
	ret	
	
op_mla:
	mov 		esi,edi
	mul 		dword ptr [r0+ebx*4]
	shr 		esi,12
	shr 		edi,16
	and 		esi,0Fh
	and 		edi,0Fh
	add 		eax,[r0+esi*4]
	ADD_CYCLES 	2
	mov 		[r0+edi*4],eax
	ret
	
op_mlas:
	mov 		esi,edi
	mul 		dword ptr [r0+ebx*4]	; Rs *= Rm
	shr 		esi,12
	shr 		edi,16
	and 		esi,0Fh
	and 		edi,0Fh
	add 		eax,[r0+esi*4]		; Rs += Rn
	mov 		ebp,eax
	;cmp 		eax,0
	UPDATE_NZ
	ADD_CYCLES 	2
	mov 		[r0+edi*4],ebp		; Rd = Rs*Rm + Rn
	ret
	
op_umull:
	mov 		esi,edi
	mul 		dword ptr [r0+ebx*4]	; Rs *= Rm
	shr 		esi,12
	shr 		edi,16
	and 		esi,0Fh			; esi : RdLo
	and 		edi,0Fh			; edi : RdHi
	ADD_CYCLES 	2
	mov 		[r0+esi*4],eax
	mov 		[r0+edi*4],edx
	ret
	
op_umulls:
	mov 		esi,edi
	mul 		dword ptr [r0+ebx*4]	; Rs *= Rm
	mov 		ecx,edx
	mov 		ebp,eax
	UPDATE_NZ_MULL
	shr 		esi,12
	shr 		edi,16
	and 		esi,0Fh			; esi : RdLo
	and 		edi,0Fh			; edi : RdHi
	ADD_CYCLES 	2
	mov 		[r0+esi*4],ebp
	mov 		[r0+edi*4],ecx
	ret
	
op_smull:
	mov 		esi,edi
	imul 		dword ptr [r0+ebx*4]	; Rs *= Rm
	shr 		esi,12
	shr 		edi,16
	and 		esi,0Fh			; esi : RdLo
	and 		edi,0Fh			; edi : RdHi
	ADD_CYCLES 	2
	mov 		[r0+esi*4],eax
	mov 		[r0+edi*4],edx
	ret
	
op_smulls:
	mov 		esi,edi
	imul 		dword ptr [r0+ebx*4]	; Rs *= Rm
	mov 		ecx,edx
	mov 		ebp,eax
	UPDATE_NZ_MULL
	shr 		esi,12
	shr 		edi,16
	and 		esi,0Fh			; esi : RdLo
	and 		edi,0Fh			; edi : RdHi
	ADD_CYCLES 	2
	mov 		[r0+esi*4],ebp
	mov 		[r0+edi*4],ecx
	ret		

op_umlal:
	mov 		esi,edi
	mul 		dword ptr [r0+ebx*4]	; Rs *= Rm
	shr 		esi,12
	shr 		edi,16
	and 		esi,0Fh			; esi : RdLo
	and 		edi,0Fh			; edi : RdHi
	ADD_CYCLES 	3
	add 		[r0+esi*4],eax
	adc 		[r0+edi*4],edx
	ret
	
op_umlals:
	mov 		esi,edi
	mul 		dword ptr [r0+ebx*4]	; Rs *= Rm
	shr 		esi,12
	shr 		edi,16
	and 		esi,0Fh			; esi : RdLo
	and 		edi,0Fh			; edi : RdHi
	ADD_CYCLES 	3
	add 		[r0+esi*4],eax
	adc 		[r0+edi*4],edx
	UPDATE_NZ_MLAL
	ret	
	;DIV0
	
op_smlal:
	mov 		esi,edi
	imul 		dword ptr [r0+ebx*4]	; Rs *= Rm
	shr 		esi,12
	shr 		edi,16
	and 		esi,0Fh			; esi : RdLo
	and 		edi,0Fh			; edi : RdHi
	ADD_CYCLES 	3
	add 		[r0+esi*4],eax
	adc 		[r0+edi*4],edx
	ret
	
op_smlals:
	mov 		esi,edi
	imul 		dword ptr [r0+ebx*4]	; Rs *= Rm
	shr 		esi,12
	shr 		edi,16
	and 		esi,0Fh			; esi : RdLo
	and 		edi,0Fh			; edi : RdHi
	ADD_CYCLES 	3
	add 		[r0+esi*4],eax
	adc 		[r0+edi*4],edx
	UPDATE_NZ_MLAL
	ret	
	;DIV0
	
op_swpb:
	mov 		esi,edi
	push 		edi
	shr 		esi,16
	push 		eax
	and 		esi,0Fh
	mov 		ebx,[r0+esi*4]
	push 		ebx	
	RDBYTE 		ebx
	pop 		esi
	pop 		ecx
	push 		eax
	WRBYTE 		esi,cl,ecx
	pop 		eax
	pop 		edi
	;and 		eax,0FFh
	shr 		edi,12
	and 		edi,0Fh
	mov 		[r0+edi*4],eax
	ADD_CYCLES 	4
	ret
	
op_swpd:
	mov 		esi,edi
	push 		edi
	shr 		esi,16
	push 		eax
	and 		esi,0Fh
	mov 		ebx,[r0+esi*4]
	push 		ebx	
	RDWORD 		ebx
	pop 		esi
	pop 		edx
	push 		eax
	WRWORD 		esi,edx
	pop 		eax
	pop 		edi
	shr 		edi,12
	and 		edi,0Fh
	mov 		[r0+edi*4],eax
	ADD_CYCLES 	4
	ret


	
	
.data
align 8
mul_ops dd OFFSET op_mul_
	dd OFFSET op_muls_
	dd OFFSET op_mla
	dd OFFSET op_mlas
	dd OFFSET op_mul_
	dd OFFSET op_muls_
	dd OFFSET op_mla
	dd OFFSET op_mlas
	dd OFFSET op_umull
	dd OFFSET op_umulls
	dd OFFSET op_umlal
	dd OFFSET op_umlals
	dd OFFSET op_smull
	dd OFFSET op_smulls
	dd OFFSET op_smlal
	dd OFFSET op_smlals
	
	dd OFFSET op_swpd
	dd OFFSET op_swpd
	dd OFFSET op_swpd
	dd OFFSET op_swpd
	dd OFFSET op_swpb
	dd OFFSET op_swpb
	dd OFFSET op_swpb
	dd OFFSET op_swpb
	dd OFFSET op_swpd
	dd OFFSET op_swpd
	dd OFFSET op_swpd
	dd OFFSET op_swpd
	dd OFFSET op_swpb
	dd OFFSET op_swpb
	dd OFFSET op_swpb
	dd OFFSET op_swpb

.code


.data
	szswiarm db "SWI ARM",0
	szswitmb db "SWI Thumb",0
.code

swi_handler:

	cmp 		eax,5
	jne 		not_swi_5
	
	; SWI 5 : Wait for vblank
	pop 		eax
	mov 		old_exec,eax
	.if r16 & 20h ;cb_execute == offset cpu_execute_until_arm
		;DIV0
		sub 		r15,2
		mov 		cb_execute_swi,offset cpu_execute_until_thumb
	.else
		sub 		r15,4
		mov 		cb_execute_swi,offset cpu_execute_until_arm
	.endif
	and 		cpu_irq_line,0FFFFFFFEh
	or 		cpu_irq_line,80000000h
	mov 		eax,dtcmBase
	add 		eax,3FF8h
	mov 		ebx,0
	WRHWORD 	eax,bx,ebx
	mov 		cb_execute,offset swi_5
	swi_5:
		test 		cpu_irq_line,1
		jnz 		swi_5_irq
		mov 		eax,dtcmBase
		add 		eax,3FF8h
		push 		dword ptr cycle
		RDHWORD 	eax
		;pusha
		;invoke wsprintf,ADDR szBuffer,ADDR szLDRH,eax
		;invoke lstrlen,ADDR szBuffer
		;invoke WriteConsole,hout,ADDR szBuffer,eax,ADDR dummy,NULL
		;popa				
		pop 		dword ptr cycle
		test 		eax,1
		jnz 		swi_5_done

		mov 		ecx,cycle
		cmp 		ecx,[esp+36] 
		jge 		swi_5_cyc

		;inc 		dummy
		;.if dummy & 2
			ADD_CYCLES 		2
		;.else
		;	ADD_CYCLES 2
		;.endif
		jmp 		swi_5
	swi_5_irq:
		and 		cpu_irq_line,0FFFFFFFEh
		mov 		cb_execute,offset cpu_execute_until_arm
		jmp 		cpu_execute_until_arm
	swi_5_cyc:
		popa
		mov 		eax,cycle
		ret 		4
	swi_5_done:
		mov 		eax,dtcmBase
		add 		eax,3FF8h
		WRHWORD 	eax,bx,0
		
		; Clear "waiting for acknowledgement" bit
		and 		cpu_irq_line,07FFFFFFEh
		
		;mov 		cb_execute,offset cpu_execute_until_arm
		;jmp 		cpu_execute_until_arm
		mov 		eax,cb_execute_swi
		mov 		cb_execute,eax
		push 		dword ptr [old_exec]
		ret
	not_swi_5:

	cmp 		eax,4
	jne 		not_swi_4	

	; SWI 4
	
	pop 		eax
	mov 		old_exec,eax
	.if cb_execute == offset cpu_execute_until_arm
		sub 		r15,4
	.elseif cb_execute == offset cpu_execute_until_thumb
		sub 		r15,2
	.endif
	and 		cpu_irq_line,0FFFFFFF8h
	mov 		cb_execute,offset swi_4
	swi_4:
		test 		cpu_irq_line,07h
		jnz 		swi_4_done

		mov 		ecx,cycle
		cmp 		ecx,[esp+36] 
		jge 		swi_4_cyc

		inc 		dummy
		.if dummy & 4
			ADD_CYCLES 	1
		.endif
		jmp 		swi_4
	swi_4_cyc:
		popa
		mov 		eax,cycle
		ret 		4
	swi_4_done:
		xor 		cpu_irq_line,1
		mov 		cb_execute,offset cpu_execute_until_arm
		jmp 		cpu_execute_until_arm
	not_swi_4:

	
	; SWI 9 : Divide
	.if eax == 9
		mov 		eax,r0
		cdq
		idiv 		dword ptr r1
		mov 		r0,eax
		mov 		r1,edx
		test 		eax,80000000h
		jz 		@@swi9_positive
		neg 		eax
		@@swi9_positive:
		mov 		r2,eax
		ADD_CYCLES 	20
		ret
	
	; SWI 7
	;.elseif eax == 7
	;	mov 		eax,r1
	;	cdq
	;	idiv 		dword ptr r0
	;	mov 		r0,eax
	;	mov 		r1,edx
	;	test 		eax,80000000h
	;	jz 		@@swi7_positive
	;	neg 		eax
	;	@@swi7_positive:
	;	mov 		r2,eax
	;	ADD_CYCLES 	20
	;	ret	
	
	; SWI 13 : Sqrt
	.elseif eax == 0Dh
		fild 		dword ptr [r0]
		fsqrt
		fwait
		fistp 		dword ptr [r0]
		ADD_CYCLES 	30
		ret
	
	; SWI 12
	.elseif eax == 0Ch
		test 		r2,1000000h
		jnz 		@@cfs_fixed
		and 		r2,0FFFFh
		mov 		eax,r2
		shl 		eax,2
		add 		cycle,eax
		@@cfs_copy:
			;invoke		arm9_read_dword,r0
			;invoke		arm9_write_dword,r1,eax
			RDWORD		r0
			WRWORD 		r1,eax
			add 		r0,4
			add 		r1,4
			dec 		r2
			jnz 		@@cfs_copy
			ret
		@@cfs_fixed:
			and 		r2,0FFFFh
			mov 		eax,r2
			shl 		eax,2
			add 		cycle,eax
			;invoke 	arm9_read_dword,r0
			RDWORD 		r0
			mov 		dummy,eax
			@@cfs_set:
				;invoke 	arm9_write_dword,r1,dummy
				WRWORD 		r1,dummy
				add 		r1,4
				dec 		r2
				jnz 		@@cfs_set
				ret
				

	.elseif eax == 0FEh
		pusha
		;invoke 	wsprintf,ADDR szBuffer,ADDR szFormat35,r0,r1,r2,r3,r4,r5,r6,r7,r14,r15
		;invoke 	lstrlen,ADDR szBuffer
		;invoke 	WriteConsole,hout,ADDR szBuffer,eax,ADDR dummy,NULL
		popa

	.elseif eax == 0FFh
		TRANSLATE_ADDRESS r0,0FFFFFh
		invoke 		zombie,TELL,511h,eax,0
	.endif
	ADD_CYCLES 	4
	ret



; Get operands for dataproc ops
DP_GET_OPERANDS MACRO
	mov 		ebx,eax
	mov 		esi,eax
	mov 		edi,eax
	shr 		ebx,20			; Get opcode
	shr 		esi,16			; Get Rn
	and 		edi,0Fh			; Get Rm
	and 		esi,0Fh
	and 		ebx,01Fh
	mov 		esi,[r0+esi*4]
ENDM



DP_IMM_SHIFT MACRO shtype
	LOCAL zeroshift
	
	test 		eax,0F80h		
	mov 		edi,[r0+edi*4]
	jz 		zeroshift	

	mov 		ecx,eax
	and 		ecx,0FFFh
	shr 		ecx,7	; Get Rm shift
	&shtype&_OPERAND edi
	UPDATE_C
	jmp 		dword ptr [data_proc_ops + ebx*4]

	zeroshift:
	&shtype&_OPERAND_ZERO edi
	jmp 		dword ptr [data_proc_ops + ebx*4]	
ENDM


DP_REG_SHIFT MACRO shtype
	LOCAL shift_reg_zero,shift_reg_ge_32,shift_reg_gt_32
	
	mov 		ecx,eax
	add 		r15,4
	shr 		ecx,8
	ADD_CYCLES 	1
	and 		ecx,0Fh
	mov 		ecx,[r0+ecx*4]
	and 		ecx,0FFh
	mov 		edi,[r0+edi*4]
	jz 		shift_reg_zero
	cmp 		ecx,32
	jge 		shift_reg_ge_32
	&shtype&_OPERAND edi
	UPDATE_C
	call 		dword ptr [data_proc_ops + ebx*4]
	sub 		r15,4
	ret

	shift_reg_zero:
	mov 		ecx,[flags+4]
	mov 		[shifterC],ecx
	call 		dword ptr [data_proc_ops + ebx*4]
	sub 		r15,4
	ret

	shift_reg_ge_32:
	jg 		shift_reg_gt_32		
	&shtype&_OPERAND_32 edi
	call 		dword ptr [data_proc_ops + ebx*4]
	sub 		r15,4
	ret

	shift_reg_gt_32:		
	&shtype&_OPERAND_GT32 edi
	call 		dword ptr [data_proc_ops + ebx*4]
	sub 		r15,4
	ret
ENDM


; Partial register operation
PR_OP MACRO op
	LOCAL hword_op_reg_offs
	
	mov 		ebx,eax
	nop

	shr 		ebx,18		
	and 		ebx,07Ch		; ebx = PU0WL00
	mov 		edx,ebx
	or	 	ebx,op

	shr 		edx,1		; edx = 0PU0WL0
	and 		ebx,0Fh		; ebx = 000WLSH
	and 		edx,30h		; edx = 0PU0000 

	test 		eax,400000h
	jz 		hword_op_reg_offs
	mov 		esi,eax
	mov 		edi,eax
	and 		esi,0Fh
	shr 		edi,4
	and 		edi,0F0h
	or 		ebx,edx
	or 		esi,edi	; esi = 8bit offset
	jmp 		dword ptr [hword_xfer + ebx*4]
	
	hword_op_reg_offs:
	mov 		edi,eax
	and 		edi,0Fh
	or 		ebx,edx
	mov 		esi,[r0+edi*4] ; esi = offset register	
	jmp 		dword ptr [hword_xfer + ebx*4]
ENDM


; Dual register operation
; operation = (ldrstr==1) ? STRD : LDRD
DR_OP MACRO ldrstr
	LOCAL dword_op_reg_offs
	
	mov 		ebx,eax
	
	shr 		ebx,21		
	and 		ebx,0Dh		; ebx = PU0W
	mov 		edx,ebx

	shr 		edx,1		; edx = PU0
	and 		ebx,1		; ebx = W

	test 		eax,400000h
	jz 		dword_op_reg_offs
	mov 		esi,eax
	mov 		edi,eax
	and 		esi,0Fh
	shr 		edi,4
	and 		edi,0F0h
	or 		ebx,edx
	or 		esi,edi	; esi = 8bit offset
	jmp 		dword ptr [dword_xfer + ldrstr*20h + ebx*4]

	dword_op_reg_offs:
	mov 		edi,eax
	and 		edi,0Fh
	or 		ebx,edx
	mov 		esi,[r0+edi*4] ; esi = offset register	
	jmp 		dword ptr [dword_xfer + ldrstr*20h + ebx*4]
ENDM


SMULxy MACRO x,y
	LOCAL _smlaxy,_smlaxy_sat,_smlaxy_sat_mi,_smlawy,_smlalxy
	
	mov 		ebx,eax
	and 		ebx,600000h
	jz 		_smlaxy
	cmp 		ebx,200000h
	je 		_smlawy
	cmp 		ebx,400000h
	je 		_smlalxy
	
	mov 		ebx,eax
	mov 		ebp,eax
	shr 		ebx,8
	shr 		ebp,16
	and 		eax,15
	and 		ebx,15
	movsx 		eax,word ptr [r0+x+eax*4]
	movsx 		ebx,word ptr [r0+y+ebx*4]
	and 		ebp,15
	imul 		eax,ebx
	mov 		[r0+ebp*4],eax
	ADD_CYCLES 	2
	ret
	

	_smlaxy:
	mov 		ebx,eax
	mov 		ecx,eax
	mov 		ebp,eax
	shr 		ebx,8
	and 		eax,15
	and 		ebx,15
	movsx 		eax,word ptr [r0+x+eax*4]
	movsx 		ebx,word ptr [r0+y+ebx*4]
	imul 		eax,ebx
	shr 		ecx,12
	shr 		ebp,16
	and 		ecx,15
	and 		ebp,15
	add 		eax,[r0+ecx*4]
	jo 		_smlaxy_sat
	;seto 		byte ptr [flags+32]
	mov 		[r0+ebp*4],eax
	ADD_CYCLES 	2
	ret

	_smlaxy_sat:
	mov 		[flags+32],1
	mov 		[r0+ebp*4],eax
	ADD_CYCLES 	2
	ret
	;test 		eax,80000000h
	;jnz 		_smlaxy_sat_mi
	;mov 		dword ptr [r0+ebp*4],80000000h
	;mov 		[flags+32],1
	;ADD_CYCLES 	2
	;ret
	
	_smlaxy_sat_mi:
	;mov 		dword ptr [r0+ebp*4],7fffffffh
	;mov 		[flags+32],1
	;ADD_CYCLES 	2
	ret
	
	_smlawy:
	; TODO: Add SMLAWy/SMULWy
	DIV0
	_smlalxy:
	; TODO: Add SMLALxy

	DIV0
ENDM



; Decoder bits:
;
; d27 d26 d25 d7 d6 d5 d4
;
arm9_00:	; dp lsl imm
	DP_GET_OPERANDS
	DP_IMM_SHIFT 	<LSL>
	
arm9_01:	; dp lsl r / bx / clz
	mov 		ecx,eax
	mov 		edx,eax
	and 		ecx,0FFF0FF0h
	and 		edx,0FFFFFF0h
	cmp 		ecx,016F0F10h
	je 		@@arm9_01_clz
	cmp 		edx,012FFF10h
	je 		@@arm9_01_bx

	DP_GET_OPERANDS	
	DP_REG_SHIFT 	<LSL>
	
	@@arm9_01_clz:
	mov 		ebx,eax
	shr 		ebx,12
	and 		eax,15	; Rm
	and 		ebx,15	; Rd
	bsr 		ecx,dword ptr [r0+eax*4]
	jz 		@@arm9_01_clz_all_zero
	mov 		eax,31
	sub 		eax,ecx
	mov 		[r0+ebx*4],eax
	ADD_CYCLES 	1
	ret
	@@arm9_01_clz_all_zero:
	mov 		dword ptr [r0+ebx*4],32
	ADD_CYCLES 	1
	ret
	
	@@arm9_01_bx:
	and 		eax,0Fh
	mov 		physaddr,-1
	mov 		ebx,[r0+eax*4]

	test 		ebx,1
	jz 		@@arm9_01_bx_no_switch
		;DIV0
		mov 		cb_execute,OFFSET cpu_execute_until_thumb
		mov 		dword ptr [esp],OFFSET cpu_execute_until_thumb
		or 		r16,20h	; set the T bit
		and 		ebx,0FFFFFFFEh
	@@arm9_01_bx_no_switch:
	mov 		r15,ebx
	ADD_CYCLES 	4
	ret
		
		
arm9_02:	; dp lsr imm
	DP_GET_OPERANDS
	DP_IMM_SHIFT 	<LSR>
	
arm9_03:	; dp lsr r / blx (2)
	mov 		ecx,eax
	and 		ecx,01FFFFF0h
	cmp 		ecx,012FFF30h
	je 		@@arm9_03_blx2

	DP_GET_OPERANDS	
	DP_REG_SHIFT 	<LSR>

	@@arm9_03_blx2:
	;DIV0
	mov 		ebx,r15
	sub 		ebx,4
	mov 		r14,ebx
	
	and 		eax,0Fh
	mov 		physaddr,-1
	mov 		ebx,[r0+eax*4]

	test 		ebx,1
	jz 		@@arm9_03_blx2_no_switch
		mov 		cb_execute,OFFSET cpu_execute_until_thumb
		mov 		dword ptr [esp],OFFSET cpu_execute_until_thumb
		or 		r16,20h	; set the T bit
		and 		ebx,0FFFFFFFEh
	@@arm9_03_blx2_no_switch:
	mov 		r15,ebx
	ADD_CYCLES 	4
	ret	
	
arm9_04:	; dp asr imm
	DP_GET_OPERANDS
	DP_IMM_SHIFT 	<ASR>

	
arm9_05:	; dp asr r / qadd
	mov	 	ecx,eax
	and 		ecx,00F00F00h
	jz 		@@arm9_05_qadd
	cmp 		ecx,00200000h
	jz 		@@arm9_05_qsub
	cmp 		ecx,00400000h
	jz 		@@arm9_05_qdadd
	cmp 		ecx,00600000h
	jz 		@@arm9_05_qdsub
	
	DP_GET_OPERANDS	
	DP_REG_SHIFT 	<ASR>

	@@arm9_05_qadd:
	mov 		ebx,eax
	mov 		ecx,eax
	and 		eax,15	; Rm
	shr 		ebx,12
	shr 		ecx,16
	and 		ebx,15	; Rd
	and 		ecx,15	; Rn
	mov 		edx,[r0+eax*4]
	add 		edx,[r0+ecx*4]
	jo 		@@arm9_05_qadd_sat
	mov 		[r0+ebx*4],edx
	ADD_CYCLES 		1
	ret
	@@arm9_05_qadd_sat:
	test 		edx,80000000h

	jnz 		@@arm9_05_qadd_sat_mi
	mov 		dword ptr [r0+ebx*4],80000000h
	mov 		[flags+32],1
	ADD_CYCLES 	1
	ret
	@@arm9_05_qadd_sat_mi:
	mov 		dword ptr [r0+ebx*4],7fffffffh
	mov 		[flags+32],1
	ADD_CYCLES 	1
	ret
	
	
	@@arm9_05_qdadd:
	mov 		ebx,eax
	mov 		ecx,eax
	and 		eax,15	; Rm
	shr 		ebx,12
	shr 		ecx,16
	and 		ebx,15	; Rd
	and 		ecx,15	; Rn
	mov 		eax,[r0+eax*4]
	mov 		edx,[r0+ecx*4]
	add 		edx,edx	; edx = Rn*2
	jno 		@@arm9_05_qdadd_no_sat
	mov 		[flags+32],1
	test 		edx,80000000h
	jnz 		@@arm9_05_qdadd_sat1_mi
	mov 		edx,80000000h
	jmp 		@@arm9_05_qdadd_no_sat
	@@arm9_05_qdadd_sat1_mi:
	mov 		edx,7fffffffh
	@@arm9_05_qdadd_no_sat:

	add 		eax,edx
	jo 		@@arm9_05_qdadd_sat2
	mov 		[r0+ebx*4],eax
	ADD_CYCLES 	1
	ret
	@@arm9_05_qdadd_sat2:
	test 		eax,80000000h
	jnz 		@@arm9_05_qdadd_sat2_mi
	mov 		dword ptr [r0+ebx*4],80000000h
	mov 		[flags+32],1
	ADD_CYCLES 	1
	ret
	@@arm9_05_qdadd_sat2_mi:
	mov 		dword ptr [r0+ebx*4],7fffffffh
	mov 		[flags+32],1
	ADD_CYCLES 	1
	ret

		
	@@arm9_05_qsub:
	mov 		ebx,eax
	mov 		ecx,eax
	and 		eax,15	; Rm
	shr 		ebx,12
	shr 		ecx,16
	and 		ebx,15	; Rd
	and 		ecx,15	; Rn
	mov 		edx,[r0+eax*4]
	sub 		edx,[r0+ecx*4]
	jo 		@@arm9_05_qsub_sat
	mov 		[r0+ebx*4],edx
	ADD_CYCLES 	1
	ret
	@@arm9_05_qsub_sat:
	test 		edx,80000000h
	jnz 		@@arm9_05_qsub_sat_mi
	mov 		dword ptr [r0+ebx*4],80000000h
	mov 		[flags+32],1
	ADD_CYCLES 	1
	ret
	@@arm9_05_qsub_sat_mi:
	mov 		dword ptr [r0+ebx*4],7fffffffh
	mov 		[flags+32],1
	ADD_CYCLES 	1
	ret
	
	@@arm9_05_qdsub:
	mov 		ebx,eax
	mov 		ecx,eax
	and 		eax,15	; Rm
	shr 		ebx,12
	shr 		ecx,16
	and 		ebx,15	; Rd
	and 		ecx,15	; Rn
	mov 		eax,[r0+eax*4]
	mov 		edx,[r0+ecx*4]
	add 		edx,edx	; edx = Rn*2
	jno 		@@arm9_05_qdsub_no_sat
		mov 		[flags+32],1
		test 		edx,80000000h
		jnz 		@@arm9_05_qdsub_sat1_mi
		mov 		edx,80000000h
		jmp 		@@arm9_05_qdsub_no_sat
		@@arm9_05_qdsub_sat1_mi:
		mov 		edx,7fffffffh
	@@arm9_05_qdsub_no_sat:
	sub 		eax,edx	; Rm - Sat(Rn*2)
	jo 		@@arm9_05_qdsub_sat2
	mov 		[r0+ebx*4],eax
	ADD_CYCLES 	1
	ret
	@@arm9_05_qdsub_sat2:
	test 		eax,80000000h
	jnz 		@@arm9_05_qdsub_sat2_mi
	mov 		dword ptr [r0+ebx*4],80000000h
	mov 		[flags+32],1
	ADD_CYCLES 	1
	ret
	@@arm9_05_qdsub_sat2_mi:
	mov 		dword ptr [r0+ebx*4],7fffffffh
	mov 		[flags+32],1
	ADD_CYCLES 	1
	ret
	
	
arm9_06:	; dp ror imm
	DP_GET_OPERANDS
	DP_IMM_SHIFT 	<ROR>
	
	
arm9_07:	; dp ror r / bkpt
	mov 		ecx,eax
	and 		ecx,012000F0h
	cmp 		ecx,01200070h
	je 		@@arm9_07_bkpt
	
	DP_GET_OPERANDS	
	DP_REG_SHIFT 	<ROR>

	@@arm9_07_bkpt:
	DIV0
	
arm9_08:	; dp lsl imm / SMLABB
	mov 		ebx,eax
	and 		ebx,03900090h
	cmp 		ebx,01000080h
	je 		@@arm9_08_smulbb

	DP_GET_OPERANDS
	DP_IMM_SHIFT 	<LSL>
	
	@@arm9_08_smulbb:
	SMULxy 		0,0
	

arm9_09:	; MUL/SWP
	mov 		ebx,eax
	mov 		ecx,eax
	mov 		ebp,eax

	mov 		edi,eax
	shr 		ebx,8
	and 		ecx,0Fh		; Rm

	and 			ebp,1F00000h
	and 		ebx,0Fh		; Rs
	shr 		ebp,18
	mov 		eax,[r0+ecx*4]
	jmp 		dword ptr [mul_ops + ebp]
	
arm9_0A:	; dp lsr imm / SMLATB / SMULTB
	mov 		ebx,eax
	and 		ebx,03900090h
	cmp 		ebx,01000080h
	je 		@@arm9_0A_smultb

	DP_GET_OPERANDS
	DP_IMM_SHIFT 	<LSR>

	@@arm9_0A_smultb:
	SMULxy 		2,0
	
arm9_0B:	; LDRH
	PR_OP 		1

arm9_0C:	; dp asr imm / SMLABT
	mov 		ebx,eax
	and 		ebx,03900090h
	cmp 		ebx,01000080h
	je 		@@arm9_0C_smulbt

	DP_GET_OPERANDS
	DP_IMM_SHIFT 	<ASR>

	@@arm9_0C_smulbt:
	SMULxy 		0,2

arm9_0D:	; LDRSB / LDRD
	test 		eax,100000h
	jz 		@@arm9_0D_ldrd
	PR_OP 		2

	@@arm9_0D_ldrd:
	DR_OP 		0
	
arm9_0E:	; dp ror imm / SMLATT
	mov 		ebx,eax
	and 		ebx,03900090h
	cmp 		ebx,01000080h
	je 		@@arm9_0E_smultt

	DP_GET_OPERANDS
	DP_IMM_SHIFT 	<ROR>

	@@arm9_0E_smultt:
	SMULxy 		2,2

arm9_0F:	; LDRSH / STRD
	test 		eax,100000h
	jz 		@@arm9_0F_strd
	PR_OP 		3
	
	@@arm9_0F_strd:
	DR_OP 		1
	;DIV0


arm9_10:
	mov 		ebx,eax
	mov 		edi,eax
	mov 		ecx,eax
	shr 		ebx,20		; Get opcode
	shr 		ecx,16		; Get Rn
	and 		edi,0FFh		; Get Op2
	and 		ecx,0Fh
	and 		ebx,01Fh
	mov 		esi,[r0+ecx*4]

	test	 	eax,0F00h
	jz 		@@arm9_10_no_rot
		mov 		ecx,eax
		shr 		ecx,8	; Get Op2 shift
		and 		ecx,0Fh
		add 		ecx,ecx
		ror 		edi,cl
		UPDATE_C
	@@arm9_10_no_rot:
	jmp 		dword ptr [data_proc_ops + ebx*4]

arm9_11:
arm9_12:
arm9_13:
arm9_14:
arm9_15:
arm9_16:
arm9_17:
arm9_18:
arm9_19:
arm9_1A:
arm9_1B:
arm9_1C:
arm9_1D:
arm9_1E:
arm9_1F:

arm9_20:
	mov 		ebx,eax
	mov 		ecx,eax
	shr 		ebx,20
	mov 		edx,eax
	and 		ecx,0FFFh
	and 		edx,50F000h
	and 		ebx,01Fh
	cmp 		edx,10F000h
	je 		@@ldr_r15
	jmp 		dword ptr [data_xfer + ebx*4]
	@@ldr_r15:
	mov 		physaddr,-1
	call 		dword ptr [data_xfer + ebx*4]
	test 		r15,1
	jz 		@@arm9_20_no_switch
		mov 		cb_execute,OFFSET cpu_execute_until_thumb
		mov 		dword ptr [esp],OFFSET cpu_execute_until_thumb
		or 		r16,20h	; set the T bit
		and 		r15,0FFFFFFFEh
	@@arm9_20_no_switch:	
	ret
	
arm9_21:
arm9_22:
arm9_23:
arm9_24:
arm9_25:
arm9_26:
arm9_27:
arm9_28:
arm9_29:
arm9_2A:
arm9_2B:
arm9_2C:
arm9_2D:
arm9_2E:
arm9_2F:


; Get operands for a data transfer op
DT_GET_OPERANDS MACRO
	mov 		ebx,eax
	mov 		edi,eax
	mov 		edx,eax
	shr 		ebx,20
	and 		edi,0Fh
	and 		edx,50F000h
	and 		ebx,01Fh
	mov 		edi,[r0+edi*4]
	.if edx == 10F000h
		mov 		physaddr,-1
	.endif
ENDM


arm9_30:	; lsl
	DT_GET_OPERANDS
	test 		eax,0FE0h
	jz 		@@arm9_30_ns
	mov 		ecx,eax
	and 		ecx,0FFFh
	shr 		ecx,7		; Get Rm shift
	jz 		@@arm9_30_zs
	LSL_OPERAND 	edi
	mov 		ecx,edi
	jmp 		dword ptr [data_xfer + ebx*4]
	@@arm9_30_zs:
	LSL_OPERAND_ZERO_DISCARD_C edi
	@@arm9_30_ns:
	mov 		ecx,edi
	jmp 		dword ptr [data_xfer + ebx*4]
	
arm9_31:	; na
	DIV0
arm9_32:
	DT_GET_OPERANDS
	test 		eax,0FE0h
	jz 		@@arm9_32_ns
	mov	 	ecx,eax
	and 		ecx,0FFFh
	shr 		ecx,7		; Get Rm shift
	jz 		@@arm9_32_zs
	LSR_OPERAND 	edi
	mov 		ecx,edi
	jmp 		dword ptr [data_xfer + ebx*4]
	@@arm9_32_zs:
	LSR_OPERAND_ZERO_DISCARD_C edi
	@@arm9_32_ns:
	mov 		ecx,edi
	jmp 		dword ptr [data_xfer + ebx*4]
arm9_33:
	DIV0
arm9_34:
	DT_GET_OPERANDS
	test 		eax,0FE0h
	jz 		@@arm9_34_ns
	mov 		ecx,eax
	and 		ecx,0FFFh
	shr 		ecx,7		; Get Rm shift
	jz 		@@arm9_34_zs
	ASR_OPERAND 	edi
	mov 		ecx,edi
	jmp 		dword ptr [data_xfer + ebx*4]
	@@arm9_34_zs:
	ASR_OPERAND_ZERO_DISCARD_C edi
	@@arm9_34_ns:
	mov 		ecx,edi
	jmp 		dword ptr [data_xfer + ebx*4]

arm9_35:
	DIV0
arm9_36:
	DT_GET_OPERANDS
	test 		eax,0FE0h
	jz 		@@arm9_36_ns
	mov 		ecx,eax
	and 		ecx,0FFFh
	shr 		ecx,7		; Get Rm shift
	jz 		@@arm9_36_zs
	ROR_OPERAND 	edi
	mov 		ecx,edi
	jmp 		dword ptr [data_xfer + ebx*4]
	@@arm9_36_zs:
	ROR_OPERAND_ZERO_DISCARD_C edi
	@@arm9_36_ns:
	mov 		ecx,edi
	jmp 		dword ptr [data_xfer + ebx*4]

arm9_37:
	DIV0
arm9_38:
arm9_39:
	DIV0
arm9_3A:
arm9_3B:
	DIV0
arm9_3C:
arm9_3D:
	DIV0
arm9_3E:
arm9_3F:
	DIV0

;.data
;	ldm_stack_ptr dd 4096
;.data?
;	ldm_stack dd 1024 dup(?)
;.code

arm9_40:
;	test eax,100000h
;	jnz ldm_stack_pop
;		sub ldm_stack_ptr,4
;		mov ebx,ldm_stack_ptr
;		mov ecx,r15
;		mov [ldm_stack+ebx],ecx
;		jmp ldm_stack_push
;	ldm_stack_pop:
;		add ldm_stack_ptr,4
;	ldm_stack_push:
	
	mov 		ebx,eax
	shr 		ebx,20
	and 		ebx,01Fh
	jmp 		dword ptr [block_xfer + ebx*4]
arm9_41:
arm9_42:
arm9_43:
arm9_44:
arm9_45:
arm9_46:
arm9_47:
arm9_48:
arm9_49:
arm9_4A:
arm9_4B:
arm9_4C:
arm9_4D:
arm9_4E:
arm9_4F:


; Branch
arm9_50:
	test 		eax,1000000h
	jz 		@@arm9_50_no_link
	mov 		ebx,r15
	sub 		ebx,4
	mov 		r14,ebx
	@@arm9_50_no_link:

	shl 		eax,8
	mov 		physaddr,-1
	sar 		eax,6		; Shift left by 2 and sign-extend
	ADD_CYCLES 	3
	add 		r15,eax
	ret
arm9_51:
arm9_52:
arm9_53:
arm9_54:
arm9_55:
arm9_56:
arm9_57:
arm9_58:
arm9_59:
arm9_5A:
arm9_5B:
arm9_5C:
arm9_5D:
arm9_5E:
arm9_5F:

arm9_60:	; mcrr d[20..23] = 4, mcrc d[20..23] = 5
arm9_61:
arm9_62:
arm9_63:
arm9_64:
arm9_65:
arm9_66:
arm9_67:
arm9_68:
arm9_69:
arm9_6A:
arm9_6B:
arm9_6C:
arm9_6D:
arm9_6E:
arm9_6F:


arm9_swi:
	shr 		eax,16
	and 		eax,0FFh
	jmp 		swi_handler	

arm9_70:
	test 		eax,1000000h
	jnz 		arm9_swi
	DIV0
	
arm9_71:	; mcr / mrc
	test 		eax,1000000h
	jnz 		arm9_swi
	test 		eax,100000h
	jnz 		@@mrc_op2_0
	CP15OP 		0,<mcr_table>
	@@mrc_op2_0:
	CP15OP 		0,<mrc_table>
	
arm9_72:
arm9_73:	; mcr / mrc
	test 		eax,1000000h
	jnz 		arm9_swi
	test 		eax,100000h
	jnz 		@@mrc_op2_1
	CP15OP 		1,<mcr_table>
	@@mrc_op2_1:
	CP15OP 		1,<mrc_table>
	
arm9_74:
arm9_75:	; mcr / mrc
	test 		eax,1000000h
	jnz 		arm9_swi
	test 		eax,100000h
	jnz 		@@mrc_op2_2
	CP15OP 		2,<mcr_table>
	@@mrc_op2_2:
	CP15OP 		2,<mrc_table>

arm9_76:
arm9_77:	; mcr / mrc
	test 		eax,1000000h
	jnz 		arm9_swi
	test 		eax,100000h
	jnz 		@@mrc_op2_3
	CP15OP 		3,<mcr_table>
	@@mrc_op2_3:
	CP15OP 		3,<mrc_table>
	
arm9_78:	
arm9_79:	; mcr / mrc
	test 		eax,1000000h
	jnz 		arm9_swi
	test 		eax,100000h
	jnz 		@@mrc_op2_4
	CP15OP 		4,<mcr_table>
	@@mrc_op2_4:
	CP15OP 		4,<mrc_table>
	
arm9_7A:	
arm9_7B:	; mcr / mrc
	test 		eax,1000000h
	jnz 		arm9_swi
	test 		eax,100000h
	jnz 		@@mrc_op2_5
	CP15OP 		5,<mcr_table>
	@@mrc_op2_5:
	CP15OP 		5,<mrc_table>
	
arm9_7C:
arm9_7D:	; mcr / mrc
	test 		eax,1000000h
	jnz 		arm9_swi
	test 		eax,100000h
	jnz 		@@mrc_op2_6
	CP15OP 		6,<mcr_table>
	@@mrc_op2_6:
	CP15OP 		6,<mrc_table>

arm9_7E:
arm9_7F:	; mcr / mrc
	test 		eax,1000000h
	jnz 		arm9_swi
	test 		eax,100000h
	jnz 		@@mrc_op2_7
	CP15OP 		7,<mcr_table>
	@@mrc_op2_7:
	CP15OP 		7,<mrc_table>
	
	ADD_CYCLES 	1
	ret


.data
	;cr_0	dd 0
	;cr_1	dd 0
	;cr_9	dd 0
	excpBase 	dd 0
.code


; ebx = CRm
; ebp = op1
; eax = op2
; edx = Rd
mcr_00:
	ret
mcr_01:
	.if ebx == 0
		.if ebp == 0
			.if eax == 0
				mov 		ecx,[r0+edx*4]
				mov 		c1,ecx
				mov 		ebx,0FFFF0000h
				mov 		eax,0
				test 		ecx,2000h
				cmovne 		eax,ebx
				mov 		excpBase,eax
			.endif
		.endif
	.endif
	ADD_CYCLES 	1
	ret
	
mcr_02:
mcr_03:
mcr_04:
mcr_05:
mcr_06:
mcr_07:
mcr_08:
	ret
mcr_09:
	.if ebp == 0
		.if ebx == 1
			; mcr 15,0,rd,cr9,cr1,0 -- set dtcm base and size
			.if eax == 0
				; Write data tcm
				;pusha
				;invoke wsprintf,ADDR szBuffer,ADDR szFormat150,dword ptr [r0+edx*4]
				;invoke lstrlen,ADDR szBuffer
				;invoke WriteConsole,hout,ADDR szBuffer,eax,ADDR dummy,NULL
				;popa
				
				; Save for later use
				mov 		eax,dtcmBase
				mov 		dummy,eax
				
				mov 		eax,[r0+edx*4]
				and 		eax,0FFFFF000h
				mov 		dtcmBase,eax
				mov 		dtcmEnd,eax
				add 		dtcmEnd,4000h
				mov 		ebx,eax
				mov 		ecx,DTCM
				shr 		ebx,20
				mov 		[pageTable+ebx*4],ecx
				invoke 		zombie,TELL,MMU_TCM_REMAP,0,eax
				
				invoke 		zombie,ASK,MMU_PAGE_TABLE,0,0
				;mov 		esi,eax
				mov 		edi,offset pageTable
				mov 		ebx,dummy
				.if ebx != dtcmBase
					shr 		ebx,20
					mov 		ecx,[eax + ebx*4]
					mov 		[edi + ebx*4],ecx
				.endif
			
			;; mcr 15,0,rd,cr9,cr1,1 -- set itcm base and size
			.elseif eax == 1
				; Write instruction tcm
				mov 		eax,[r0+edx*4]
				and 		eax,0FFFFF000h
				;mov 		itcmBase,eax
				;invoke 	zombie,TELL,MMU_TCM_REMAP,1,eax
			.endif
		.endif
	.endif
	ADD_CYCLES 	1
	ret
mcr_10:
mcr_11:
mcr_12:
mcr_13:
mcr_14:
mcr_15:
	ADD_CYCLES 	1
	ret

mrc_00:
	.if ebx == 0
		.if ebp == 0
			.if eax == 1
				mov 	dword ptr [r0+edx*4],0F0D2112h
			.elseif eax == 2
				mov 	dword ptr [r0+edx*4],00140140h
			.else
				mov 	dword ptr [r0+edx*4],41069461h
			.endif
		.endif
	.endif
	ADD_CYCLES 	1
	ret
mrc_01:
mrc_02:
mrc_03:
mrc_04:
mrc_05:
mrc_06:
mrc_07:
mrc_08:
	ret
mrc_09:
	.if ebp == 0
		.if ebx == 1
			.if eax == 0
				; Read data tcm
				;pusha
				;mov 		eax,dtcmBase
				;add 		eax,3FFCh
				;invoke 	arm9_read_dword,eax
				;invoke 	wsprintf,ADDR szBuffer,ADDR szFormat151,dtcmBase,eax
				;invoke 	lstrlen,ADDR szBuffer
				;invoke 	WriteConsole,hout,ADDR szBuffer,eax,ADDR dummy,NULL
				;popa
				mov 		eax,dtcmBase
				mov 		[r0+edx*4],eax
			.elseif eax == 1
				; Read instruction tcm
				mov 		eax,itcmBase
				mov 		[r0+edx*4],eax
			.endif
		.endif
	.endif
	ADD_CYCLES 	1
	ret
mrc_10:
mrc_11:
mrc_12:
mrc_13:
mrc_14:
mrc_15:
	ADD_CYCLES 	1
	ret
	
.data
align 4
mcr_table LABEL DWORD
dd offset mcr_00
dd offset mcr_01
dd offset mcr_02
dd offset mcr_03
dd offset mcr_04
dd offset mcr_05
dd offset mcr_06
dd offset mcr_07
dd offset mcr_08
dd offset mcr_09
dd offset mcr_10
dd offset mcr_11
dd offset mcr_12
dd offset mcr_13
dd offset mcr_14
dd offset mcr_15

mrc_table LABEL DWORD
dd offset mrc_00
dd offset mrc_01
dd offset mrc_02
dd offset mrc_03
dd offset mrc_04
dd offset mrc_05
dd offset mrc_06
dd offset mrc_07
dd offset mrc_08
dd offset mrc_09
dd offset mrc_10
dd offset mrc_11
dd offset mrc_12
dd offset mrc_13
dd offset mrc_14
dd offset mrc_15

.code


.data
align 16
arm9_decoder LABEL DWORD
dd offset arm9_00
dd offset arm9_01
dd offset arm9_02
dd offset arm9_03
dd offset arm9_04
dd offset arm9_05
dd offset arm9_06
dd offset arm9_07
dd offset arm9_08
dd offset arm9_09
dd offset arm9_0A
dd offset arm9_0B
dd offset arm9_0C
dd offset arm9_0D
dd offset arm9_0E
dd offset arm9_0F

dd offset arm9_10
dd offset arm9_10
dd offset arm9_10
dd offset arm9_10
dd offset arm9_10
dd offset arm9_10
dd offset arm9_10
dd offset arm9_10
dd offset arm9_10
dd offset arm9_10
dd offset arm9_10
dd offset arm9_10
dd offset arm9_10
dd offset arm9_10
dd offset arm9_10
dd offset arm9_10

dd offset arm9_20
dd offset arm9_20
dd offset arm9_20
dd offset arm9_20
dd offset arm9_20
dd offset arm9_20
dd offset arm9_20
dd offset arm9_20
dd offset arm9_20
dd offset arm9_20
dd offset arm9_20
dd offset arm9_20
dd offset arm9_20
dd offset arm9_20
dd offset arm9_20
dd offset arm9_20

dd offset arm9_30
dd offset arm9_31
dd offset arm9_32
dd offset arm9_33
dd offset arm9_34
dd offset arm9_35
dd offset arm9_36
dd offset arm9_37
dd offset arm9_30
dd offset arm9_39
dd offset arm9_32
dd offset arm9_3B
dd offset arm9_34
dd offset arm9_3D
dd offset arm9_36
dd offset arm9_3F

dd offset arm9_40
dd offset arm9_40
dd offset arm9_40
dd offset arm9_40
dd offset arm9_40
dd offset arm9_40
dd offset arm9_40
dd offset arm9_40
dd offset arm9_40
dd offset arm9_40
dd offset arm9_40
dd offset arm9_40
dd offset arm9_40
dd offset arm9_40
dd offset arm9_40
dd offset arm9_40

dd offset arm9_50
dd offset arm9_50
dd offset arm9_50
dd offset arm9_50
dd offset arm9_50
dd offset arm9_50
dd offset arm9_50
dd offset arm9_50
dd offset arm9_50
dd offset arm9_50
dd offset arm9_50
dd offset arm9_50
dd offset arm9_50
dd offset arm9_50
dd offset arm9_50
dd offset arm9_50

dd offset arm9_60
dd offset arm9_61
dd offset arm9_62
dd offset arm9_63
dd offset arm9_64
dd offset arm9_65
dd offset arm9_66
dd offset arm9_67
dd offset arm9_68
dd offset arm9_69
dd offset arm9_6A
dd offset arm9_6B
dd offset arm9_6C
dd offset arm9_6D
dd offset arm9_6E
dd offset arm9_6F

dd offset arm9_70
dd offset arm9_71
dd offset arm9_72
dd offset arm9_73
dd offset arm9_74
dd offset arm9_75
dd offset arm9_76
dd offset arm9_77
dd offset arm9_78
dd offset arm9_79
dd offset arm9_7A
dd offset arm9_7B
dd offset arm9_7C
dd offset arm9_7D
dd offset arm9_7E
dd offset arm9_7F
.code


;######################################## ARMv5TE extensions ########################################


arm9_ext_0:
arm9_ext_1:
arm9_ext_2:	; pld
arm9_ext_3:	; pld I
arm9_ext_4:
	DIV0
	
arm9_ext_5:	; blx (1)
	mov 		ebx,r15
	mov 		ecx,eax
	sub 		ebx,4
	shr		ecx,23

	mov 		r14,ebx
	
	shl 		eax,8
	mov 		physaddr,-1
	sar 		eax,6		; Shift left by 2 and sign-extend
	and		ecx,2	
	;and		eax,0FFFFFFFCh	
	ADD_CYCLES 	4
	or		eax,ecx		; Set bit 1 to the H-bit	
	add 		r15,eax

	mov 		cb_execute,OFFSET cpu_execute_until_thumb
	mov 		dword ptr [esp],OFFSET cpu_execute_until_thumb
	or 		r16,20h		; Set the T bit
	ret
	
arm9_ext_6:
arm9_ext_7:
	ADD_CYCLES 	1
	DIV0
	ret
	

.data
align 16
arm9_ext_decoder LABEL DWORD
dd offset arm9_ext_0
dd offset arm9_ext_1
dd offset arm9_ext_2
dd offset arm9_ext_3
dd offset arm9_ext_4
dd offset arm9_ext_5
dd offset arm9_ext_6
dd offset arm9_ext_7
.code



;######################################### THUMB emulation #########################################


op_and_thumb:
	mov 		eax,[r0+esi*4]
	and 		[r0+edi*4],eax
	UPDATE_NZ
	ADD_CYCLES 	1
	ret
	
op_eor_thumb:
	mov 		eax,[r0+esi*4]
	xor 		[r0+edi*4],eax
	UPDATE_NZ
	ADD_CYCLES 	1
	ret
	
op_lsl_thumb:
	mov 		ecx,[r0+esi*4]
	and 		ecx,0FFh
	jz 		op_lsl_thumb_zero
	cmp 		ecx,32
	jge 		op_lsl_thumb_ge_32
	shl 		dword ptr [r0+edi*4],cl
	UPDATE_C_FORCE
	UPDATE_NZ
	ADD_CYCLES 	1
	ret
	op_lsl_thumb_zero:
	or 		dword ptr [r0+edi*4],0
	UPDATE_NZ
	ADD_CYCLES 	1
	ret
	op_lsl_thumb_ge_32:
	jg 		op_lsl_thumb_gt_32
	shr 		dword ptr [r0+edi*4],1
	UPDATE_C_FORCE
	mov 		dword ptr [r0+edi*4],0
	xor 		ecx,ecx
	UPDATE_NZ
	ADD_CYCLES 	1
	ret
	op_lsl_thumb_gt_32:
	mov 		[flags+4],1
	mov 		dword ptr [r0+edi*4],0
	xor 		ecx,ecx
	UPDATE_NZ
	ADD_CYCLES 	1
	ret
	
op_lsr_thumb:
	mov 		ecx,[r0+esi*4]
	and 		ecx,0FFh
	jz 		op_lsr_thumb_zero
	cmp 		ecx,32
	jge 		op_lsr_thumb_ge_32
	shr 		dword ptr [r0+edi*4],cl
	UPDATE_C_FORCE
	UPDATE_NZ
	ADD_CYCLES 	1
	ret
	op_lsr_thumb_zero:
	or 		dword ptr [r0+edi*4],0
	UPDATE_NZ
	ADD_CYCLES 	1
	ret
	op_lsr_thumb_ge_32:
	jg 		op_lsr_thumb_gt_32
	shl 		dword ptr [r0+edi*4],1
	UPDATE_C_FORCE
	mov 		dword ptr [r0+edi*4],0
	xor 		ecx,ecx
	UPDATE_NZ
	ADD_CYCLES 	1
	ret
	op_lsr_thumb_gt_32:
	mov 		[flags+4],1
	mov 		dword ptr [r0+edi*4],0
	xor 		ecx,ecx
	UPDATE_NZ
	ADD_CYCLES 	1
	ret
	
op_asr_thumb:
	mov 		ecx,[r0+esi*4]
	and 		ecx,0FFh
	jz 		op_asr_thumb_zero
	cmp 		ecx,32
	jge 		op_asr_thumb_ge_32
	sar 		dword ptr [r0+edi*4],cl
	UPDATE_C_FORCE
	UPDATE_NZ
	ADD_CYCLES 	1
	ret
	op_asr_thumb_zero:
	or 		dword ptr [r0+edi*4],0
	UPDATE_NZ
	ADD_CYCLES 	1
	ret
	op_asr_thumb_ge_32:
	mov 		eax,[r0+edi*4]
	shl 		eax,1
	UPDATE_C_FORCE
	sar 		dword ptr [r0+edi*4],31
	UPDATE_NZ
	ADD_CYCLES 	1
	ret
	
op_adc_thumb:
	mov 		eax,[r0+esi*4]
	cmp 		[flags+4],1		; Set x86 carry if ARM9 carry is set
	adc 		[r0+edi*4],eax
	UPDATE_FLAGS
	ADD_CYCLES 	1
	ret
	
op_sbc_thumb:
	mov 		eax,[r0+esi*4]
	cmp 		[flags+4],1		; Set x86 carry if ARM9 carry is set
	cmc
	sbb 		[r0+edi*4],eax
	UPDATE_FLAGS_SUB
	ADD_CYCLES 	1
	ret

op_ror_thumb:
	mov 		ecx,[r0+esi*4]
	and 		ecx,0FFh
	jz 		op_ror_thumb_zero
	cmp 		ecx,32
	jge 		op_ror_thumb_ge_32
op_ror_thumb_bypass_load:
	ror 		dword ptr [r0+edi*4],cl
	UPDATE_C_FORCE
	UPDATE_NZ
	ADD_CYCLES 	1
	ret
	op_ror_thumb_zero:
	or 		dword ptr [r0+edi*4],0
	UPDATE_NZ
	ADD_CYCLES 	1
	ret
	op_ror_thumb_ge_32:
	jg 		op_ror_thumb_gt_32
	mov 		eax,[r0+edi*4]
	mov 		ecx,eax
	shl 		eax,1
	UPDATE_C_FORCE
	or 		ecx,ecx
	UPDATE_NZ
	ADD_CYCLES 	1
	ret
	op_ror_thumb_gt_32:
	and 		ecx,31
	jmp 		op_ror_thumb_bypass_load

op_tst_thumb:
	mov 		eax,[r0+esi*4]
	and 		eax,[r0+edi*4]
	UPDATE_NZ
	ADD_CYCLES 	1
	ret
	
op_neg_thumb:
	xor 		ebp,ebp
	sub 		ebp,[r0+esi*4]
	UPDATE_FLAGS_SUB
	mov 		[r0+edi*4],ebp
	ADD_CYCLES 	1
	ret
	
op_cmp_thumb:
	mov 		eax,[r0+edi*4]
	sub 		eax,[r0+esi*4]
	UPDATE_FLAGS_SUB
	ADD_CYCLES 	1
	ret
	
op_cmn_thumb:
	mov 		eax,[r0+edi*4]
	add 		eax,[r0+esi*4]
	UPDATE_FLAGS
	ADD_CYCLES 	1
	ret
	
op_orr_thumb:
	mov 		eax,[r0+esi*4]
	or 		[r0+edi*4],eax
	UPDATE_NZ
	ADD_CYCLES 	1
	ret
	
op_mul_thumb:
	mov 		eax,[r0+esi*4]
	imul 		dword ptr [r0+edi*4]
	mov 		[r0+edi*4],eax
	cmp 		eax,0
	UPDATE_NZ
	ADD_CYCLES 	2
	ret
	
op_bic_thumb:
	mov 		eax,[r0+esi*4]
	not 		eax
	and 		[r0+edi*4],eax
	UPDATE_NZ
	ADD_CYCLES 	1
	ret
	
op_mvn_thumb:
	mov 		eax,[r0+esi*4]
	not 		eax
	mov 		[r0+edi*4],eax
	UPDATE_NZ
	ADD_CYCLES 	1
	ret

.data	
thumb_alu_ops dd offset op_and_thumb
	dd offset op_eor_thumb
	dd offset op_lsl_thumb
	dd offset op_lsr_thumb
	dd offset op_asr_thumb
	dd offset op_adc_thumb
	dd offset op_sbc_thumb
	dd offset op_ror_thumb
	dd offset op_tst_thumb
	dd offset op_neg_thumb
	dd offset op_cmp_thumb
	dd offset op_cmn_thumb
	dd offset op_orr_thumb
	dd offset op_mul_thumb
	dd offset op_bic_thumb
	dd offset op_mvn_thumb
.code


	

; LSL Rd,Rs,#Offset5
thumb_type_00:
	mov 		esi,eax
	mov 		ecx,eax
	mov 		edi,eax
	and 		esi,38h
	and 		ecx,7C0h
	shr 		esi,1
	and 		edi,7
	mov 		esi,[r0+esi]
	shr 		ecx,6
	jz 		thumb_type_00_zero
	shl 		esi,cl
	UPDATE_C_FORCE
	UPDATE_NZ
	mov 		[r0+edi*4],esi
	ADD_CYCLES 	1
	ret
	thumb_type_00_zero:
	and 		esi,esi
	UPDATE_NZ
	mov 		[r0+edi*4],esi
	ADD_CYCLES 	1
	ret
	
; LSR Rd,Rs,#Offset5
thumb_type_01:
	mov 		esi,eax
	mov 		ecx,eax
	mov 		edi,eax
	and 		esi,38h
	and 		ecx,7C0h
	shr 		esi,1
	and 		edi,7
	mov 		esi,[r0+esi]
	shr 		ecx,6
	jz 		thumb_type_01_zero
	shr 		esi,cl
	UPDATE_C_FORCE
	UPDATE_NZ
	mov 		[r0+edi*4],esi
	ADD_CYCLES 	1	
	ret
	thumb_type_01_zero:
	shr 		esi,31
	xor 		esi,1
	mov 		[flags+4],esi
	xor 		esi,esi
	UPDATE_NZ
	mov 		[r0+edi*4],esi
	ADD_CYCLES 	1
	ret
	

; ASR Rd,Rs,#Offset5
thumb_type_02:
	mov 		esi,eax
	mov 		ecx,eax
	mov 		edi,eax
	and 		esi,38h
	and 		ecx,7C0h
	shr 		esi,1
	and 		edi,7
	mov 		esi,[r0+esi]
	shr 		ecx,6
	jz 		thumb_type_02_zero
	sar 		esi,cl
	UPDATE_C_FORCE
	UPDATE_NZ
	mov 		[r0+edi*4],esi
	ADD_CYCLES 	1
	ret
	thumb_type_02_zero:
	sar 		esi,31
	sete 		byte ptr [flags+4]
	UPDATE_NZ
	mov 		[r0+edi*4],esi
	ADD_CYCLES 	1
	ret
	
		
; ADD/SUB Rd,Rs,Rn/#Offset3
thumb_type_03:	
	mov 		esi,eax
	mov 		edi,eax
	mov 		ebx,eax
	and 		esi,38h		; Rs
	and 		edi,1C0h		; Rn/Offset3
	shr 		esi,1
	shr 		edi,6
	mov 		esi,[r0+esi]
	test 		eax,400h
	jnz 		@@thumb_type_03_imm
	mov 		edi,[r0+edi*4]
	@@thumb_type_03_imm:
	and 		eax,7		; Rd
	test	 	ebx,200h		; Op
	jnz 		tmb_sub_3

	add 		esi,edi
	mov 		[r0+eax*4],esi
	UPDATE_FLAGS
	ADD_CYCLES 	1
	ret

	tmb_sub_3:
	sub 		esi,edi
	mov 		[r0+eax*4],esi
	UPDATE_FLAGS_SUB
	ADD_CYCLES 	1
	ret


; MOV Rd,#Offset8
thumb_type_04:
	mov 		edi,eax
	ADD_CYCLES 	1
	shr 		edi,8
	and 		eax,0FFh
	and 		edi,7
	mov 		[r0+edi*4],eax
	or 		eax,eax
	UPDATE_NZ 
	ret


; CMP Rd,#Offset8
thumb_type_05:
	mov 		edi,eax
	ADD_CYCLES 	1
	shr 		edi,8
	and 		eax,0FFh
	and 		edi,7
	cmp 		[r0+edi*4],eax
	UPDATE_FLAGS_SUB
	ret
	

; ADD Rd,#Offset8
thumb_type_06:
	mov 		edi,eax
	ADD_CYCLES 	1
	shr 		edi,8
	and 		eax,0FFh
	and 		edi,7
	add 		[r0+edi*4],eax
	UPDATE_FLAGS
	ret
	

; SUB Rd,#Offset8
thumb_type_07:
	mov 		edi,eax
	ADD_CYCLES 	1
	shr 		edi,8
	and 		eax,0FFh
	and 		edi,7
	sub 		[r0+edi*4],eax
	UPDATE_FLAGS_SUB
	ret

	
; ALU ops / HiReg ops / BX
thumb_type_08:
	test 		eax,400h
	jnz 		@@thumb_hireg_op

	mov 		esi,eax
	mov 		ebx,eax
	mov 		edi,eax
	shr 		esi,3		; 000010000oooosss
	shr 		ebx,6		; 000000010000oooo
	and 		edi,7		; 0000000000000ddd
	and 		ebx,0Fh		; 000000000000oooo
	and 		esi,7		; 0000000000000sss
	jmp 		dword ptr [thumb_alu_ops + ebx*4]
	
	@@thumb_hireg_op:
	mov 		ebx,eax
	mov 		esi,eax
	mov 		edi,eax
	and 		ebx,300h
	mov 		ecx,eax
	mov 		edx,eax
	shr 		esi,3
	;shr	 	ecx,3	; H2
	shr 		edx,4	; H1
	and 		edi,7
	;and 		ecx,8
	;and 		esi,7
	and 		edx,8
	and 		esi,15

	;add 		esi,ecx
	add 		edi,edx
	ADD_CYCLES 	1
	.if ebx == 000h		; ADD Rd,Rd,Rs
		mov 		eax,[r0+esi*4]
		add 		[r0+edi*4],eax
		.if edi == 15
			and 		r15,0FFFFFFFEh
			mov 		physaddr,-1
		.endif		
		ret
	.elseif ebx == 100h	; CMP Rd,Rs
		mov 		eax,[r0+edi*4]
		cmp 		eax,[r0+esi*4]
		UPDATE_FLAGS_SUB
		ret
	.elseif ebx == 200h	; MOV Rd,Rs
		mov 		eax,[r0+esi*4]
		mov 		[r0+edi*4],eax
		.if edi == 15
			and 		r15,0FFFFFFFEh
			mov 	physaddr,-1
		.endif
		ret
	.else			; BX Rs
		mov 		ebx,[r0+esi*4]
		test		ebx,1
		jnz 		@@thumb_bx_no_switch
			mov 		cb_execute,OFFSET cpu_execute_until_arm
			mov 		dword ptr [esp],OFFSET cpu_execute_until_arm
			and 		r16,0FFFFFFDFh	; Clear the T bit

			;pusha
			;and 		ebx,0FFFFFFFEh
			;invoke 	wsprintf,ADDR szBuffer,ADDR szFormat30,esi,ebx,r15
			;invoke 	lstrlen,ADDR szBuffer
			;invoke 	WriteConsole,hout,ADDR szBuffer,eax,ADDR dummy,NULL
			;popa
			
		@@thumb_bx_no_switch:
		and 		ebx,0FFFFFFFEh
		mov 		physaddr,-1
		mov 		r15,ebx
		ADD_CYCLES 	3
		ret		
	.endif
	;ADD_CYCLES 	1
	ret
	
; LDR Rd,[pc,#Offset10]
thumb_type_09:
	mov 		ebx,eax
	mov 		esi,r15
	mov 		edx,eax
	and 		ebx,0FFh
	and 		edx,700h
	and 		esi,0FFFFFFFDh
	shl 		ebx,2
	shr 		edx,8
	add 		esi,ebx
	RDWORD 		esi
	mov 		[r0+edx*4],eax
	;ADD_CYCLES 	LDR_CYCLES
	ret

; STR with register offset
thumb_type_0A:
	mov 		ebx,eax
	mov 		esi,eax
	mov 		edi,eax
	mov 		ecx,eax
	and 		edi,7	; EDI = Rd
	shr 		esi,3	; ESI = Rb
	shr 		ecx,6	; ECX = Ro
	and 		esi,7
	and 		ecx,7
	test	 	eax,200h
	jnz 		@@thumb_type_0A_shb
	and 		ebx,400h
	mov 		edx,[r0+esi*4]
	add 		edx,[r0+ecx*4]
	.if ebx == 0000h	
		; STR Rd,[Rb,Ro]
		WRWORD 	edx,<dword ptr [r0+edi*4]>
	.else
		; STRB Rd,[Rb,Ro]
		WRBYTE 	edx,bl,<dword ptr [r0+edi*4]>
	.endif
	;ADD_CYCLES 	STR_CYCLES
	ret

	@@thumb_type_0A_shb:	
	and 		ebx,0400h
	mov 		ecx,[r0+ecx*4]
	add 		ecx,[r0+esi*4]
	.if ebx == 0000h
		; STRH Rd,[Rb,Ro]
		WRHWORD 	ecx,bx,<dword ptr [r0+edi*4]>
	.else
		; LDSB Rd,[Rb,Ro]
		push 		edi
		RDBYTE 		ecx
		movsx 		eax,al
		pop 		edi
		mov 		[r0+edi*4],eax
	.endif
	ret

; LDR with register offset	
thumb_type_0B:
	mov 		ebx,eax
	mov 		esi,eax
	mov 		edi,eax
	mov 		ecx,eax
	and 		edi,7
	shr 		esi,3
	shr 		ecx,6
	and 		esi,7
	and 		ecx,7
	test 		eax,200h
	jnz 		@@thumb_type_0B_shb
	and 		ebx,400h
	mov 		edx,[r0+esi*4]
	add 		edx,[r0+ecx*4]
	.if ebx == 0000h	
		; LDR Rd,[Rb,Ro]
		push 		edi
		RDWORD 		edx
		pop 		edi
		mov 		[r0+edi*4],eax
	.else
		; LDRB Rd,[Rb,Ro]
		push 		edi
		RDBYTE 		edx
		pop 		edi
		mov 		[r0+edi*4],eax
	.endif
	;ADD_CYCLES 	LDR_CYCLES
	ret
	
	@@thumb_type_0B_shb:	
	and 		ebx,0400h
	mov 		edx,[r0+esi*4]
	add 		edx,[r0+ecx*4]
	.if ebx == 0000h
		; LDRH Rd,[Rb,Ro]
		push 		edi
		RDHWORD 	edx
		pop 		edi
		mov 		[r0+edi*4],eax
	.else
		; LDSH Rd,[Rb,Ro]
		push 		edi
		RDHWORD 	edx
		movsx 		eax,ax
		pop 		edi
		mov 		[r0+edi*4],eax
	.endif
	;ADD_CYCLES 	LDR_CYCLES
	ret

; STR Rd,[Rb,#Offset6]
thumb_type_0C:
	mov	 	esi,eax
	mov 		edi,eax
	mov 		edx,eax
	and 		edi,7
	shr 		edx,4
	shr 		esi,3
	and 		edx,7Ch
	and 		esi,7
	mov 		ebx,[r0+esi*4]
	;mov 		ecx,[r0+edi*4]
	add 		edx,ebx
	WRWORD 		edx,<dword ptr [r0+edi*4]>
	ret

; LDR Rd,[Rb,#Offset6]	
thumb_type_0D:
	mov 		esi,eax
	mov 		edi,eax
	mov 		edx,eax
	and 		edi,7
	shr 		edx,4
	shr 		esi,3
	and 		edx,7Ch
	and 		esi,7
	mov 		ebx,[r0+esi*4]
	add 		edx,ebx
	push 		edi
	RDWORD 		edx
	pop 		edi
	mov 		[r0+edi*4],eax
	ret

; STRB Rd,[Rb,#Offset6]
thumb_type_0E:
	mov 		esi,eax
	mov 		edi,eax
	mov 		edx,eax
	and 		edi,7
	shr 		edx,6
	shr 		esi,3
	and 		edx,1Fh
	and 		esi,7
	mov 		ebx,[r0+esi*4]
	add 		edx,ebx
	WRBYTE 		edx,cl,<dword ptr [r0+edi*4]>
	ret

; LDRB Rd,[Rb,#Offset6]
thumb_type_0F:
	mov 		esi,eax
	mov 		edi,eax
	mov 		edx,eax
	and 		edi,7
	shr 		edx,6
	shr 		esi,3
	and 		edx,1Fh
	and 		esi,7
	mov 		ebx,[r0+esi*4]
	push 		edi
	add 		edx,ebx
	RDBYTE 		edx
	pop 		edi
	mov 		[r0+edi*4],eax
	ret

; STRH Rd,[Rb,#Offset6]
thumb_type_10:
	mov 		esi,eax
	mov 		edi,eax
	mov 		edx,eax
	and 		edi,7
	shr 		edx,5
	shr 		esi,3
	and 		edx,3Eh
	and 		esi,7
	mov 		ecx,[r0+esi*4]
	add 		ecx,edx
	WRHWORD 	ecx,cx,<dword ptr [r0+edi*4]>
	ret
	
; LDRH Rd,[Rb,#Offset6]
thumb_type_11:
	mov 		esi,eax
	mov 		edi,eax
	mov 		edx,eax
	and 		edi,7
	shr 		edx,5
	shr 		esi,3
	and 		edx,3Eh
	and 		esi,7
	mov 		ebx,[r0+esi*4]
	push 		edi
	add 		edx,ebx
	RDHWORD		edx
	pop 		edi
	mov 		[r0+edi*4],eax	
	ret

; STR Rd,[sp,#Offset10]
thumb_type_12:
	mov 		ebx,eax
	mov 		esi,r13
	mov 		edi,eax
	and 		ebx,0FFh
	and 		edi,700h
	shl 		ebx,2
	shr 		edi,8
	add 		esi,ebx
	WRWORD 		esi,<dword ptr [r0+edi*4]>
	ret

; LDR Rd,[sp,#Offset10]
thumb_type_13:
	mov 		ebx,eax
	mov 		esi,r13
	mov 		edx,eax
	and 		ebx,0FFh
	and 		edx,700h
	shl 		ebx,2
	shr 		edx,8
	add 		esi,ebx
	RDWORD 		esi
	mov 		[r0+edx*4],eax
	;ADD_CYCLES 	LDR_CYCLES
	ret

; Load address (PC)
thumb_type_14:
	mov 		ebx,eax
	mov 		esi,r15
	mov 		edx,eax
	and 		esi,0FFFFFFFDh
	and 		ebx,0FFh
	and 		edx,700h
	shl 		ebx,2
	shr 		edx,8
	add 		esi,ebx
	mov 		[r0+edx*4],esi
	ADD_CYCLES 	1
	ret
	
; Load address (SP)
thumb_type_15:
	mov 		ebx,eax
	mov 		esi,r13
	mov 		edx,eax
	and 		ebx,0FFh
	and 		edx,700h
	shl 		ebx,2
	shr 		edx,8
	add 		esi,ebx
	mov 		[r0+edx*4],esi
	ADD_CYCLES 	1
	ret

; Add offset to stack pointer / PUSH registers
thumb_type_16:
	test 		eax,400h
	jnz 		@@thumb_type_16_push
	mov 		ebx,eax
	and 		ebx,7Fh
	shl 		ebx,2
	test 		eax,80h
	jz 		@@thumb_type_16_positive
	neg 		ebx
	@@thumb_type_16_positive:
	add 		r13,ebx
	ADD_CYCLES 	1
	ret

	@@thumb_type_16_push:
	mov 		ecx,7
	mov 		ebp,r13 
	shl 		eax,24
	jnc 		@@thumb_push_not_lr
		push 		eax
		sub 		ebp,4
		push 		ecx
		push 		ebp
		WRWORD 		ebp,r14
		pop 		ebp
		pop 		ecx
		pop 		eax
	@@thumb_push_not_lr:	
	@@thumb_push_loop:
		shl 		eax,1
		jnc 		@@thumb_push_no_xfer
			push 		eax
			sub 		ebp,4
			push 		ecx
			push 		ebp
			mov 		eax,[r0+ecx*4]
			WRWORD 		ebp,eax ;<dword ptr [r0+ecx*4]>
			pop 		ebp
			pop 		ecx
			pop 		eax
		@@thumb_push_no_xfer:
		dec 		ecx
		jns 		@@thumb_push_loop
	mov 		r13,ebp
	;ADD_CYCLES 	5
	ret
	
	
; POP registers
thumb_type_17:
	mov 		ecx,0
	mov 		edx,eax
	@@thumb_pop_loop:
		shr 		edx,1
		jnc 		@@thumb_pop_no_xfer
			push 		ecx
			RDWORD 		r13 
			pop 		ecx
			add 		r13,4
			mov 		[r0+ecx*4],eax
		@@thumb_pop_no_xfer:
		inc 		ecx
		cmp 		ecx,8
		jne	 	@@thumb_pop_loop
	shr 		edx,1
	jnc 		@@thumb_pop_not_pc
		RDWORD 		r13
		add 		r13,4
		test		eax,1
		jnz 		@@thumb_pop_no_switch
			mov 		cb_execute,OFFSET cpu_execute_until_arm
			mov 		dword ptr [esp],OFFSET cpu_execute_until_arm
			and 		r16,0FFFFFFDFh	; Clear the T bit
		@@thumb_pop_no_switch:		
		and 		eax,0FFFFFFFEh
		mov 		physaddr,-1
		mov 		r15,eax
	@@thumb_pop_not_pc:
	ret
	

; STMIA
thumb_type_18:
	mov 		ebx,eax
	shr 		ebx,8
	and 		ebx,7
	mov 		ebp,[r0+ebx*4]
	xor 		ecx,ecx
	push 		ebx
	@@stmia_thumb:
		shr 		eax,1
		jnc 		@@stmia_thumb_no_xfer
			push 		eax
			push 		ecx
			push 		ebp
			mov 		eax,[r0+ecx*4]
			WRWORD 		ebp,eax ;<dword ptr [r0+ecx*4]>
			pop 		ebp
			pop 		ecx
			add 		ebp,4
			pop 		eax
		@@stmia_thumb_no_xfer:
		inc 		ecx
		cmp 		ecx,8
	jne 		@@stmia_thumb
	pop 		ebx
	mov 		[r0+ebx*4],ebp
	ret

; LDMIA
thumb_type_19:
	mov 		ebx,eax
	shr 		ebx,8
	and 		ebx,7
	mov 		ebp,[r0+ebx*4]
	xor 		ecx,ecx
	push 		ebx
	@@ldmia_thumb:
		shr 		eax,1
		jnc 		@@ldmia_thumb_no_xfer
			push 		eax
			push 		ecx
			push 		ebp
			RDWORD 		ebp
			pop 		ebp
			pop 		ecx
			mov 		[r0+ecx*4],eax
			add 		ebp,4
			pop 		eax
		@@ldmia_thumb_no_xfer:
		inc 		ecx
		cmp 		ecx,8
	jne 		@@ldmia_thumb
	pop 		ebx
	mov 		[r0+ebx*4],ebp
	ret
	
; Conditional branch
thumb_type_1A:
	mov 		ebx,eax
	and 		ebx,0F00h
	mov 		edx,ebx	
	shr 		ebx,9
	shr 		edx,8
	and 		edx,1
	xor 		edx,[ebx*4 + flags]
	jnz 		@@thumb_type_1A_cond_false	
		shl 		eax,24
		mov 		physaddr,-1
		sar 		eax,23
		add 		r15,eax
	@@thumb_type_1A_cond_false:
	ADD_CYCLES 	3
	ret


; Conditional branch / SWI
thumb_type_1B:
	mov 		ebx,eax
	and 		ebx,0F00h
	cmp 		ebx,0F00h
	je 		@@thumb_type_1B_swi
	mov 		edx,ebx	
	shr 		ebx,9
	shr 		edx,8
	and 		edx,1
	xor 		edx,[ebx*4 + flags]
	jnz 		@@thumb_type_1B_cond_false	
		shl 		eax,24
		mov 		physaddr,-1
		sar 		eax,23
		;add 		eax,2
		add 		r15,eax
	@@thumb_type_1B_cond_false:
	ADD_CYCLES 	3
	ret

	@@thumb_type_1B_swi:
	;pusha
	;and eax,0FFh
	;invoke wsprintf,ADDR szBuffer,ADDR szFormat29,eax
	;invoke lstrlen,ADDR szBuffer
	;invoke WriteConsole,hout,ADDR szBuffer,eax,ADDR dummy,NULL
	;popa	

	and 		eax,0FFh
	jmp 		swi_handler
	

; Unconditional branch	
thumb_type_1C:		
	shl 		eax,21	
	mov 		physaddr,-1	; Invalidate current address mapping
	sar 		eax,20	; Convert to 12-bit sign extended
	add 		r15,eax
	ADD_CYCLES 	3
	ret
	

thumb_type_1D:
	IF DEBUG_THUMB EQ 1
	pusha
	invoke 		wsprintf,ADDR szBuffer,ADDR szFormat22,r0,r1,r2,r3,r4,r5,r6,r7,r8,r9,r10,r11,r12,r13,r14,r15,dword ptr [flags],0
	invoke 		lstrlen,ADDR szBuffer
	;invoke 	WriteConsole,hout,ADDR szBuffer,eax,ADDR dummy,NULL
	popa
	ENDIF
	DIV0

; Long branch with link HI	
thumb_type_1E:
	and	 	eax,7FFh
	shl 		eax,21
	sar 		eax,9
	add 		eax,r15
	mov 		r14,eax
	ADD_CYCLES 	1
	ret
	
; Long branch with link LO
thumb_type_1F:
	and 		eax,7FFh
	mov 		ebx,r15
	shl 		eax,1
	sub 		ebx,2
	add 		eax,r14
	mov 		physaddr,-1
	or 		ebx,1
	mov 		r14,ebx
	mov 		r15,eax
	ADD_CYCLES 	1
	ret

.data
align 16
thumb_instr_decoder dd OFFSET thumb_type_00
	dd OFFSET thumb_type_01
	dd OFFSET thumb_type_02
	dd OFFSET thumb_type_03
	dd OFFSET thumb_type_04
	dd OFFSET thumb_type_05
	dd OFFSET thumb_type_06
	dd OFFSET thumb_type_07
	dd OFFSET thumb_type_08
	dd OFFSET thumb_type_09
	dd OFFSET thumb_type_0A
	dd OFFSET thumb_type_0B
	dd OFFSET thumb_type_0C
	dd OFFSET thumb_type_0D
	dd OFFSET thumb_type_0E
	dd OFFSET thumb_type_0F
	dd OFFSET thumb_type_10
	dd OFFSET thumb_type_11
	dd OFFSET thumb_type_12
	dd OFFSET thumb_type_13
	dd OFFSET thumb_type_14
	dd OFFSET thumb_type_15
	dd OFFSET thumb_type_16
	dd OFFSET thumb_type_17
	dd OFFSET thumb_type_18
	dd OFFSET thumb_type_19
	dd OFFSET thumb_type_1A
	dd OFFSET thumb_type_1B
	dd OFFSET thumb_type_1C
	dd OFFSET thumb_type_1D
	dd OFFSET thumb_type_1E
	dd OFFSET thumb_type_1F
.code



;###################################################################################################


cpu_init PROC commlink:DWORD	
	;pusha
	

	mov 		eax,commlink
	mov 		zombie,eax

	;invoke 	AllocConsole
	invoke 		GetStdHandle,STD_OUTPUT_HANDLE
	mov 		hout,eax

	mov 		executed,0
	
	;popa
	mov 		eax,1
	ret
cpu_init ENDP


	
	
cpu_reset PROC
	pusha

	invoke 		zombie,ASK,MMU_READ_BYTE_CALLBACK,0,0
	mov 		arm9_read_byte,eax
	invoke 		zombie,ASK,MMU_READ_WORD_CALLBACK,0,0
	mov 		arm9_read_word,eax
	invoke 		zombie,ASK,MMU_READ_DWORD_CALLBACK,0,0
	mov 		arm9_read_dword,eax

	invoke 		zombie,ASK,MMU_WRITE_BYTE_CALLBACK,0,0
	mov 		arm9_write_byte,eax
	invoke 		zombie,ASK,MMU_WRITE_WORD_CALLBACK,0,0
	mov 		arm9_write_word,eax
	invoke 		zombie,ASK,MMU_WRITE_DWORD_CALLBACK,0,0
	mov 		arm9_write_dword,eax

	invoke 		zombie,ASK,MMU_TRANSLATE_ADDRESS_CALLBACK,0,0
	mov 		arm9_translate_address,eax

	invoke 		zombie,ASK,MMU_DTCM,0,0
	mov 		DTCM,eax

	invoke 		zombie,ASK,MMU_ITCM,0,0
	mov 		ITCM,eax
	
	IF MEM_TRANSLATE_INT EQ 1
		invoke 		zombie,ASK,MMU_PAGE_TABLE,0,0
		mov 		esi,eax
		mov 		edi,offset pageTable
		mov 		ecx,256
		rep 		movsd
		
		mov 		[pageMask + 0*4],3FFFh
		mov 		[pageMask + 1*4],0
		mov 		[pageMask + 2*4],23FFFFFh
		mov 		[pageMask + 3*4],3007FFFh
	ENDIF

	invoke 		zombie,ASK,MMU_READ_WORD_CALLBACKS,0,0
	mov 		esi,eax
	mov 		edi,offset readhw
	mov 		ecx,256
	rep 		movsd
	invoke 		zombie,ASK,MMU_READ_DWORD_CALLBACKS,0,0
	mov 		esi,eax
	mov 		edi,offset readw
	mov 		ecx,256
	rep 		movsd
	invoke 		zombie,ASK,MMU_WRITE_WORD_CALLBACKS,0,0
	mov 		esi,eax
	mov 		edi,offset writehw
	mov 		ecx,256
	rep 		movsd
	invoke 		zombie,ASK,MMU_WRITE_DWORD_CALLBACKS,0,0
	mov 		esi,eax
	mov 		edi,offset writew
	mov 		ecx,256
	rep 		movsd
		
	invoke 		zombie,TELL,CPU_CYCLE_PTR,ADDR cycle,0
	
	mov 		eax,3000000h
	IF MEM_TRANSLATE_INT EQ 1
		TRANSLATE_ADDRESS eax,0FFFFFh
	ELSE
		invoke 		arm9_translate_address,eax	
	ENDIF
	mov 		IWRAM,eax

	
	mov 		cb_execute,OFFSET cpu_execute_until_arm

	mov 		irq_cb_stack_ptr,OFFSET irq_cb_stack
	add 		irq_cb_stack_ptr,64*4
	
	;mov 		dword ptr [flags+0],01010101h
	;mov 		dword ptr [flags+4],01000001h

	;1,1,1,1,1,0,0,1
	mov 		dword ptr [flags+0],1
	mov 		dword ptr [flags+4],1
	mov 		dword ptr [flags+8],1
	mov 		dword ptr [flags+12],1
	mov 		dword ptr [flags+16],0
	mov 		dword ptr [flags+20],0
	mov 		dword ptr [flags+24],1
	
	mov 		dword ptr [flags_irq+0],01010101h
	mov 		dword ptr [flags_irq+4],01000001h
	

	mov 		r16,0D3h ; for ARM9 ! ARM7=01Fh
	
	mov 		branchTarget,-1
	mov 		branchOrigin,-1
	mov 		physaddr,-1
	
	mov 		excpBase,0FFFF0000h
	mov 		itcmBase,0
	mov 		dtcmBase,800000h
	mov 		dtcmEnd, 804000h
	mov 		cpu_irq_line,0
	
	call	 	dcache_reset

	IF ENABLE_DUMP EQ 1
		invoke 		_lcreat,ADDR szDump,0
		mov 		hfile,eax
		mov 		executed,0
	ENDIF
	
	popa
	ret
cpu_reset ENDP


cpu_close PROC
	invoke 		FreeConsole
	ret
cpu_close ENDP



DlgProc PROC hdlg:DWORD, msg:DWORD, wParam:DWORD, lParam:DWORD
	LOCAL hCmb:DWORD,hIRQ:DWORD

	invoke 		GetDlgItem,hdlg,1000
	mov 		hCmb,eax
	invoke 		GetDlgItem,hdlg,1001
	mov 		hIRQ,eax
	
	.if msg == WM_INITDIALOG
		invoke 		SetWindowLong,hdlg, DWL_USER, lParam
		
		invoke 		SendMessage,hCmb, CB_ADDSTRING, 0, ADDR szFreq0		
		invoke 		SendMessage,hCmb, CB_ADDSTRING, 0, ADDR szFreq1		
		invoke 		SendMessage,hCmb, CB_ADDSTRING, 0, ADDR szFreq2		
		invoke 		SendMessage,hCmb, CB_ADDSTRING, 0, ADDR szFreq3		
		invoke 		SendMessage,hCmb, CB_ADDSTRING, 0, ADDR szFreq4		
		invoke 		SendMessage,hCmb, CB_ADDSTRING, 0, ADDR szFreq5
		invoke 		SendMessage,hCmb, CB_SETCURSEL, freqSel, 0

		invoke 		SendMessage,hIRQ, BM_SETCHECK, irqEnabled, 0		
		
		mov 		eax,1
		ret
	.elseif msg == WM_COMMAND
		mov 		eax,wParam
		and 		eax,0FFFFh
		.if eax == IDOK
			invoke 		SendMessage,eax, CB_GETCURSEL, 0, 0
			mov 		freqSel,eax
			mov 		eax,[frequencies+eax*4]
			mov 		speed,eax

			invoke 		SendMessage,hIRQ,BM_GETCHECK,0,0
			cmp 		eax,BST_CHECKED
			sete 		byte ptr [irqEnabled]

			invoke 		EndDialog,hdlg, 0
			mov 		eax,1
			ret
		.elseif eax == IDCANCEL
			invoke 		EndDialog,hdlg, 1
			mov 		eax,0
			ret
		.endif
	.endif
	mov 		eax,0
	ret
DlgProc ENDP



cpu_config PROC hWnd:DWORD
	invoke 		DialogBoxParam,hInst,101,hWnd,ADDR DlgProc,NULL
	ret
cpu_config ENDP



; Unused
cpu_set_io_handler PROC mode:DWORD,port:DWORD,pfhandler:DWORD
	push 		ebx

	mov 		eax,port
	mov 		ebx,pfhandler
	
	.if mode==INPUT	
		mov 		[inpb_handlers + eax*4],ebx
	.else
		mov 		[outpb_handlers + eax*4],ebx
	.endif
	
	pop 		ebx
	ret
cpu_set_io_handler ENDP



cpu_get PROC what:DWORD
	.if what == CPU_CYCLE_COUNT
		mov 		eax,cycle
	.elseif what == CPU_CYCLE_SCALING
		mov 		eax,CYCLE_MULTIPLIER
	.elseif what == CPU_SET_IO_CALLBACK
		mov 		eax,offset cpu_set_io_handler
	.elseif what == CPU_IRQ_CALLBACK
		mov 		eax,offset cpu_irq
	.elseif what == CPU_INPB_CALLBACK
		mov 		eax,offset cpu_inpb
	.elseif what == CPU_OUTPB_CALLBACK
		mov 		eax,offset cpu_outpb

	.elseif what == CPU_DMA_CALLBACK
		mov 		eax,offset cpu_dma
		
	.elseif what == CPU_SPEED
		mov 		eax,speed
	.elseif what == CPU_REG_PC
		mov 		eax,r15
	.elseif what == CPU_REG
		PACK_FLAGS
		mov 		eax,OFFSET r0
	.elseif what == 00Fh
		mov 		eax,OFFSET flags
	.elseif what == PLUGIN_NAME
		mov 		eax,offset cpuName
	.elseif what == LAST_ERROR
		mov 		eax,lpszError
	.endif
	ret
cpu_get ENDP


cpu_set PROC what:DWORD,param1:DWORD,param2:DWORD
	.if what == CPU_REG_PC
		mov 		eax,param1
		mov 		r15,eax
	
	.elseif what == CPU_REG_SP
		mov 		eax,param1
		mov 		r13,eax
	
	.elseif what == CPU_CYCLE_COUNT
		mov 		eax,param1
		mov 		cycle,eax
	
	.elseif what == CPU_BREAKPOINT
		mov 		eax,param1
		mov 		breakpoint,eax
	
	.elseif what == CPU_SPEED
		mov 		eax,param1
		mov 		speed,eax
		shr 		eax,24
		.if eax != 0
			dec 		eax
			mov 		freqSel,eax
		.endif

	.elseif what == MMU_READ_WORD_CALLBACKS
		mov 		esi,param1
		mov 		edi,offset readhw
		mov 		ecx,256
		rep 		movsd
	
	.elseif what == CPU_IO_HANDLER
		mov 		eax,param1
		and 		param1,0FFh
		shr 		eax,8
		invoke 		cpu_set_io_handler,eax,param1,param2
	

	.elseif what == CPU_ENABLE_IRQ
		mov 		eax,param1
		and 		eax,1
		mov 		irqEnabled,eax
		
	.elseif what == PLUGIN_SETTING
		mov 		eax,param1
	.endif
	ret
cpu_set ENDP



cpu_outpb PROC port:DWORD,dat:DWORD
	push 		ecx
	push 		dword ptr dat
	mov 		eax,port
	call 		dword ptr [outpb_handlers + eax*4]
	pop 		ecx
	ret
cpu_outpb ENDP


cpu_inpb PROC port:DWORD
	push 		ecx
	mov 		eax,port
	call 		dword ptr [inpb_handlers + eax*4]
	pop 		ecx
	ret
cpu_inpb ENDP



.data
szFormat111 db "IRQ mask: %X",0
.code


cpu_irq PROC address:DWORD
	pusha
	
	;mov 		irqEnabled,0	; DEBUG
	
	cmp 		irqEnabled,0
	jz 		ime_false
	
	test 		r16,80h
	jnz 		ime_false
	
	;or 		address,0FFFF0000h
	mov 		eax,excpBase
	or 		address,eax
	
	mov 		eax,04000208h
	invoke 		arm9_read_byte,eax
	test 		eax,1
	jz 		ime_false

	;pusha
	;mov 		eax,04000214h
	;invoke		arm9_read_word,eax
	;and 		eax,0FFFFh

	;invoke 	wsprintf,ADDR szBuffer,ADDR szFormat111,eax
	;invoke 	zombie,TELL,511h,ADDR szBuffer,0
	;popa
	
	;mov 		eax,04000214h
	;invoke 	arm9_read_word,eax
	;mov 		cpu_irq_line,eax
	or 		cpu_irq_line,1

	PACK_FLAGS

	;mov 		executed,1
	
	mov 		ebx,12h
	SWAP_REGISTERS
	mov 		eax,r15
	add 		eax,8
	mov 		r14,eax

	mov 		dword ptr [subs_handler],offset op_subs_irq
	mov 		eax,address
	mov 		physaddr,-1
	mov 		r15,eax

	mov 		eax,r16
	mov 		r17,eax

	cmp 		cb_execute,offset swi_5
	je 		@@in_swi_5
	cmp 		cb_execute,offset swi_4
	je 		@@in_swi_5
	;invoke 	zombie,TELL,511h,ADDR szIRQ,0
	mov 		cb_execute,OFFSET cpu_execute_until_arm
	@@in_swi_5:
	
	and 		r16,0FFFFFFC0h	; Clear T bit
	or  		r16,12h ;92h		; Set I bit, mode = IRQ

	popa
	ret

	
	ime_false:
	popa
	ret
cpu_irq ENDP


cpu_execute PROC
	ret
cpu_execute ENDP




align 16
cpu_execute_until_arm:	; DWORD threshold 
	mov 		ecx,cycle
	sub 		r15,4
	mov 		physaddr,-1
	
	@@exe_loop2:
		cmp 		ecx,[esp+36] ;threshold
		jge 		@@exe_done2
		
		mov 		eax,physaddr
		cmp 		eax,-1
		jne 		no_xlatea
		add 		r15,4
		TRANSLATE_ADDRESS r15,0FFFFCh
		mov 		physaddr,eax
		no_xlatea:

		IF DEBUG_ARM EQ 1
			pusha
			IF ENABLE_DUMP EQ 1
				.if executed <= DUMP_EXEC
					pusha
					inc executed
					mov ebx,r15
					mov ecx,executed
					add ebx,8
					dec ecx
					mov edx,physaddr
					mov edx,[edx]
					invoke wsprintf,ADDR szBuffer,ADDR szFormat102,ecx,r15,bankn,r0,r1,r2,r3,r4,r5,r6,r7,r8,r9,r10,r11,r12,r13,r14,ebx,edx
					invoke lstrlen,ADDR szBuffer
					invoke _lwrite,hfile,ADDR szBuffer,eax
					popa
				.endif				
			ELSE
				;.if r15>=02017000h
				mov ebx,physaddr
				invoke wsprintf,ADDR szBuffer,ADDR szFormat23,r0,r1,r2,r3,r4,r5,r6,r7,r8,r9,r10,r11,r12,r13,r14,r15,dword ptr [flags],dword ptr [ebx]
				invoke lstrlen,ADDR szBuffer
				invoke WriteConsole,hout,ADDR szBuffer,eax,ADDR dummy,NULL
				;.endif
			ENDIF
			popa
		ENDIF
		
		mov 		eax,[eax]
		add 		physaddr,4

		mov 		edx,eax
		and 		edx,0F0000000h
		cmp 		edx,0E0000000h
		je 		@@cond_true2
		cmp 		edx,0F0000000h
		je 		@@decode_f2
		shr 		edx,28
		mov 		ecx,edx
		and 		ecx,1
		shr 		edx,1
		xor 		ecx,[flags + edx*4]
		jnz 		@@cond_false2
		@@cond_true2:
			mov 		ebx,eax
			mov 		edx,eax
			shr 		ebx,19
			shr 		edx,2
			add 		r15,8
			and 		edx,03Ch
			and 		ebx,1C0h
			call 		dword ptr [arm9_decoder + ebx + edx*1]
			mov 		ecx,cycle
			sub 		r15,4
			jmp 		@@exe_loop2

		@@decode_f2:
			mov 		ebx,eax
			shr 		ebx,23
			add 		r15,8
			and 		ebx,1Ch
			call 		dword ptr [arm9_ext_decoder + ebx]
			mov 		ecx,cycle
			sub 		r15,4
			jmp 		@@exe_loop2
			
		@@cond_false2:
		ADD_CYCLES 	1
		add 		r15,4 
		mov 		ecx,cycle

		
		jmp 		@@exe_loop2	
	 
	@@exe_done2:
	.if physaddr == -1
		add 		r15,4
	.endif
	popa
	
	mov 		eax,cycle
	ret 		4
	
	


; Unused
cpu_execute_until_arm_old:	; DWORD threshold 

	@@ceu_arm_entry:
	
	mov ecx,cycle
	sub r15,4
	mov physaddr,-1
	
	@@exe_loop:
		cmp ecx,[esp+36] ;threshold
		jge @@exe_done
		
		;inc executed
		;.if executed == 3
		;	mov bankn,0
		;.endif

			
		cmp physaddr,-1
		jne @@arm_physaddr_ok
			add r15,4
			IF MEM_TRANSLATE_INT EQ 1
				TRANSLATE_ADDRESS r15,0FFFFCh
			ELSE
				invoke arm9_translate_address,r15	
			ENDIF
			mov physaddr,eax
			mov ebx,[eax+4]
			mov edx,[eax+8]
			mov prefetch1,ebx
			mov eax,[eax]
			mov prefetch2,edx
			jmp @@arm_physaddr_remapped
			
		@@arm_physaddr_ok:
			add physaddr,4
			mov ebx,prefetch2
			mov edx,physaddr
			mov eax,prefetch1
			mov ebp,[edx+8]
			mov prefetch1,ebx
			mov prefetch2,ebp
		@@arm_physaddr_remapped:

	
	;.if executed <= 1000
	;	inc executed
	;	
	;	pusha
	;	mov ebx,r15
	;	mov ecx,executed
	;	add ebx,8
	;	dec ecx
	;	invoke wsprintf,ADDR szBuffer,ADDR szFormat102,ecx,r15,bankn,r0,r1,r2,r3,r4,r5,r6,r7,r8,r9,r10,r11,r12,r13,r14,ebx
	;	invoke lstrlen,ADDR szBuffer
	;	invoke _lwrite,hfile,ADDR szBuffer,eax
	;	popa
	;.endif	
	
	;mov edx,r15
	;mov r15_old,edx
		;.if guard <= 10
		IF DEBUG_ARM EQ 1
			.if r15 >= 0FFFF0000h
			;.if r15 <= 02004DD8h
			;.if r15 == 2008618h
			;.if executed == 2F17h	
			;.if executed == 1
			;.if r15 & 2
				cmp r0,80000000h
				je is_bad
				cmp r8,80000000h
				je is_bad
				;jmp not_bad
				is_bad:
				
				;invoke _lclose,hfile
				pusha
				mov ebx,physaddr
				;jne _arm_noshow
				invoke wsprintf,ADDR szBuffer,ADDR szFormat22,r0,r1,r2,r3,r4,r5,r6,r7,r8,r9,r10,r11,r12,r13,r14,r15,dword ptr [flags],dword ptr [ebx]
				;invoke wsprintf,ADDR szBuffer,ADDR szFormat50,executed
				invoke lstrlen,ADDR szBuffer
				invoke WriteConsole,hout,ADDR szBuffer,eax,ADDR dummy,NULL

			.if r15 & 2
				stayput:
				jmp stayput
				not_bad:
			.endif
				popa
			.endif
			
		ENDIF
		

		mov edx,eax
		and edx,0F0000000h
		cmp edx,0E0000000h
		je @@cond_true
		cmp edx,0F0000000h
		je @@decode_f
		shr edx,28
		mov ecx,edx
		and ecx,1
		shr edx,1
		xor ecx,[flags + edx*4]
		jnz @@cond_false
		@@cond_true:
			mov ebx,eax
			mov edx,eax
			shr ebx,19
			shr edx,2
			add r15,8
			and edx,03Ch
			and ebx,1C0h
			call dword ptr [arm9_decoder + ebx + edx*1]
			mov ecx,cycle
			sub r15,4
			jmp @@exe_loop

		@@decode_f:
			mov ebx,eax
			shr ebx,23
			add r15,8
			and ebx,1Ch
			call dword ptr [arm9_ext_decoder + ebx]
			mov ecx,cycle
			sub r15,4
			jmp @@exe_loop
			
		@@cond_false:
		ADD_CYCLES 1
		add r15,4 
		mov ecx,cycle

		
		jmp @@exe_loop		
	 
	@@exe_done:
	.if physaddr == -1
		add r15,4
	.endif
	popa
	
	mov eax,cycle
	ret 4


; Unused
align 4
cpu_execute_until_thumb_old:
	;@@ceu_thumb_entry:
	
	mov ecx,cycle
	sub r15,2
	mov physaddr,-1
	
	@@thumb_exe_loop_old:
		cmp ecx,[esp+36] 
		jge @@thumb_exe_done
		;mov lastCyc,ecx

		cmp physaddr,-1
		jne @@thumb_physaddr_ok
			add r15,2
			IF MEM_TRANSLATE_INT EQ 1
				TRANSLATE_ADDRESS r15,0FFFFEh
			ELSE
				invoke arm9_translate_address,r15	
			ENDIF
			mov physaddr,eax
			mov ebx,[eax+2]
			mov prefetch1,ebx
			movzx eax,word ptr [eax]
			jmp @@thumb_physaddr_remapped
			
		@@thumb_physaddr_ok:
			mov ebx,prefetch1
			add physaddr,2
			mov eax,ebx
			mov edx,physaddr
			and eax,0FFFFh
			mov bp,[edx+4]
			shrd ebx,ebp,16
			mov prefetch1,ebx
		@@thumb_physaddr_remapped:
		
			
		IF DEBUG_THUMB EQ 1
			;.if r15 == 20054d2h
			mov ebx,physaddr
			pusha
			;invoke wsprintf,ADDR szBuffer,ADDR szFormat21,r0,r1,r2,r3,r4,r6,r14,r15,dword ptr [flags+4],dword ptr [ebx]
			invoke wsprintf,ADDR szBuffer,ADDR szFormat22,r0,r1,r2,r3,r4,r5,r6,r7,r8,r9,r10,r11,r12,r13,r14,r15,dword ptr [flags],dword ptr [ebx]
			invoke lstrlen,ADDR szBuffer
			invoke WriteConsole,hout,ADDR szBuffer,eax,ADDR dummy,NULL
			popa
			;.endif
			;call dump_state
		ENDIF
		
		
			mov ebx,eax
			add r15,4
			shr ebx,11 
			and ebx,31 
			
			call dword ptr [thumb_instr_decoder + ebx*4]
			mov ecx,cycle
			sub r15,2
		
			jmp @@thumb_exe_loop_old
			
	 
	@@thumb_exe_done:
	.if physaddr == -1
		add r15,2
	.endif
	popa
	
	mov eax,cycle
	ret 4



.data
align 8
transbuffer dd 32 dup (0)
opcodes dd 32 dup (0)
transrange dd 0,0
.code


align 4
cpu_execute_until_thumb:
	mov 		ecx,cycle
	sub 		r15,2
	mov 		physaddr,-1
	
	@@thumb_exe_loop2:
		cmp 		ecx,[esp+36] 
		jge 		@@thumb_exe_done2

		mov 		eax,physaddr
		cmp 		eax,-1
		jne 		no_xlatet
		add 		r15,2
		TRANSLATE_ADDRESS r15,0FFFFEh
		mov 		physaddr,eax
		no_xlatet:

			
		IF DEBUG_THUMB EQ 1
			pusha
			IF ENABLE_DUMP EQ 1
				.if executed <= DUMP_EXEC
					pusha
					inc executed
					mov ebx,r15
					mov ecx,executed
					add ebx,8
					dec ecx
					mov edx,physaddr
					movzx edx,word ptr [edx]
					invoke wsprintf,ADDR szBuffer,ADDR szFormat103,ecx,r15,bankn,r0,r1,r2,r3,r4,r5,r6,r7,r8,r9,r10,r11,r12,r13,r14,ebx,edx
					invoke lstrlen,ADDR szBuffer
					invoke _lwrite,hfile,ADDR szBuffer,eax
					popa
				.endif				
			ELSE
				mov ebx,physaddr
				invoke wsprintf,ADDR szBuffer,ADDR szFormat22,r0,r1,r2,r3,r4,r5,r6,r7,r8,r9,r10,r11,r12,r13,r14,r15,dword ptr [flags],dword ptr [ebx]
				invoke lstrlen,ADDR szBuffer
				invoke WriteConsole,hout,ADDR szBuffer,eax,ADDR dummy,NULL
			ENDIF
			popa
		ENDIF
				
		xor 		ebx,ebx
		add 		physaddr,2
		mov 		bx,[eax]
		add 		r15,4
		mov 		eax,ebx
		shr 		ebx,9
		and 		ebx,31*4
		call 		dword ptr [thumb_instr_decoder + ebx]
		mov 		ecx,cycle
		sub 		r15,2
		jmp 		@@thumb_exe_loop2		
	 
	@@thumb_exe_done2:
	.if physaddr == -1
		add 		r15,2
	.endif
	popa
	
	mov 		eax,cycle
	ret 		4
	
	
	
align 8
PUBLIC cpu_execute_until
cpu_execute_until:
	pusha
	jmp dword ptr [cb_execute]	


align 4

END LibMain
