/*
**
**
**
** This file's purpose is to emulate the inner workings of a
** GameBoy Game Pak cartrige.
**
** All code is by Mark McGough. MadManMarkAu@hotmail.com
**
**
*/

// TODO: The TimerData[] values are currently not updated over time.
#include <windows.h>
#include <io.h>
#include "commonIncludes.h"
#include "NRage PluginV2.h"
#include "PakIO.h"
#include "GBCart.h"

void ClearData(char *Data, int Length);

void UpdateRTC(LPGBCART Cart) {
	time_t now, dif;
	int carry = 0, carry2 = 0;

	now = time(NULL);
	dif = now - Cart->timerLastUpdate;
	Cart->TimerData[0] += dif % 60;
	dif /= 60;
	Cart->TimerData[1] += dif % 60;
	dif /= 60;
	Cart->TimerData[2] += dif % 24;
	dif /= 24;

	Cart->TimerData[3] += dif & 0xFF;
	dif=dif / 0x100;

	if(dif > 1) {
		carry = dif;
		carry2 = ((Cart->TimerData[4] & 0x80) / 0x80) + (carry / 2);
		if (carry2 > 1) carry=3;

		 // I'm not sure if I've handled this day counter correctly... I'm writing this tired as all hell...
		// Forgive me if I get this bit wrong...
		Cart->TimerData[3] &= 255;
		Cart->TimerData[4] = (Cart->TimerData[4] & 0x7E) | (carry2 & 1) | ((carry2 / 2) * 0x80);
	}
	Cart->timerLastUpdate = now;
	DebugWrite("Update RTC:");
	DebugWriteByte(Cart->TimerData[0]);
	DebugWrite(":");
	DebugWriteByte(Cart->TimerData[1]);
	DebugWrite(":");
	DebugWriteByte(Cart->TimerData[2]);
	DebugWrite(":");
	DebugWriteByte(Cart->TimerData[3]);
	DebugWrite(":");
	DebugWriteByte(Cart->TimerData[4]);
	DebugWrite("\n");
}

bool LoadCart(LPGBCART Cart, LPCTSTR RomFile, LPCTSTR RamFile, LPCTSTR TdfFile)
{
	HANDLE hRomFile;
	HANDLE hRamFile;
//	HANDLE hTdfFile;
	DWORD dwBytesRead;
	gbCartRTC RTCTimer;
	DWORD lLength;
	DWORD lLengthHigh;

	Cart->iCurrentRamBankNo = 0;
	Cart->iCurrentRomBankNo = 1;
	Cart->bRamEnableState = 1;

// Attempt to load the ROM file.
	hRomFile = CreateFile(RomFile, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_ALWAYS, 0, NULL);
	if (hRomFile == INVALID_HANDLE_VALUE) {
		MessageBox( g_strEmuInfo.hMainWindow, "Can't open ROM file!", PLUGINERROR, MB_OK | MB_ICONERROR);
		return false;
	} else {
		// We'll load the ROM here as if it were the largest ROM size a GameBoy can support...
		// Saves us loading the exact size, because the data that can't be read from the file will all be 0x00, and
		// paging checks are done in the WriteCart() function.
		ReadFile(hRomFile, &Cart->RomData[0], 0x80 * 0x4000, &dwBytesRead, NULL);
		CloseHandle(hRomFile);
	}

	// Initialise with some scummy defauly values
	Cart->iCartType = 0x00;
	Cart->bHasRam = false;
	Cart->bHasBattery = false;
	Cart->bHasTimer = false;
	Cart->bHasRumble = false;
	DebugWrite(" Cartirdge Type #:");
	DebugWriteByte(Cart->RomData[0x147]);
	DebugWrite("\n");
	switch (Cart->RomData[0x147]) {
	case 0x01:
		Cart->iCartType = 0x01;
		Cart->bHasRam = false;
		Cart->bHasBattery = false;
		Cart->bHasTimer = false;
		Cart->bHasRumble = false;
		break;
	case 0x02:
		Cart->iCartType = 0x01;
		Cart->bHasRam = true;
		Cart->bHasBattery = false;
		Cart->bHasTimer = false;
		Cart->bHasRumble = false;
		break;
	case 0x03:
		Cart->iCartType = 0x01;
		Cart->bHasRam = true;
		Cart->bHasBattery = true;
		Cart->bHasTimer = false;
		Cart->bHasRumble = false;
		break;
	case 0x05:
		Cart->iCartType = 0x02;
		Cart->bHasRam = false;
		Cart->bHasBattery = false;
		Cart->bHasTimer = false;
		Cart->bHasRumble = false;
		break;
	case 0x06:
		Cart->iCartType = 0x02;
		Cart->bHasRam = false;
		Cart->bHasBattery = true;
		Cart->bHasTimer = false;
		Cart->bHasRumble = false;
		break;
	case 0x08:
		Cart->iCartType = 0x00;
		Cart->bHasRam = true;
		Cart->bHasBattery = false;
		Cart->bHasTimer = false;
		Cart->bHasRumble = false;
		break;
	case 0x09:
		Cart->iCartType = 0x00;
		Cart->bHasRam = true;
		Cart->bHasBattery = true;
		Cart->bHasTimer = false;
		Cart->bHasRumble = false;
		break;
	case 0x0B:
		Cart->iCartType = 0x03;
		Cart->bHasRam = false;
		Cart->bHasBattery = false;
		Cart->bHasTimer = false;
		Cart->bHasRumble = false;
		break;
	case 0x0C:
		Cart->iCartType = 0x03;
		Cart->bHasRam = true;
		Cart->bHasBattery = false;
		Cart->bHasTimer = false;
		Cart->bHasRumble = false;
		break;
	case 0x0D:
		Cart->iCartType = 0x03;
		Cart->bHasRam = true;
		Cart->bHasBattery = true;
		Cart->bHasTimer = false;
		Cart->bHasRumble = false;
		break;
	case 0x0F:
		Cart->iCartType = 0x04;
		Cart->bHasRam = false;
		Cart->bHasBattery = true;
		Cart->bHasTimer = true;
		Cart->bHasRumble = false;
		break;
	case 0x10:
		Cart->iCartType = 0x04;
		Cart->bHasRam = true;
		Cart->bHasBattery = true;
		Cart->bHasTimer = true;
		Cart->bHasRumble = false;
		break;
	case 0x11:
		Cart->iCartType = 0x04;
		Cart->bHasRam = false;
		Cart->bHasBattery = false;
		Cart->bHasTimer = false;
		Cart->bHasRumble = false;
		break;
	case 0x12:
		Cart->iCartType = 0x04;
		Cart->bHasRam = true;
		Cart->bHasBattery = true;
		Cart->bHasTimer = false;
		Cart->bHasRumble = false;
		break;
	case 0x13:
		Cart->iCartType = 0x04;
		Cart->bHasRam = true;
		Cart->bHasBattery = true;
		Cart->bHasTimer = false;
		Cart->bHasRumble = false;
		break;
	case 0x19:
		Cart->iCartType = 0x05;
		Cart->bHasRam = false;
		Cart->bHasBattery = false;
		Cart->bHasTimer = false;
		Cart->bHasRumble = false;
		break;
	case 0x1A:
		Cart->iCartType = 0x05;
		Cart->bHasRam = true;
		Cart->bHasBattery = false;
		Cart->bHasTimer = false;
		Cart->bHasRumble = false;
		break;
	case 0x1B:
		Cart->iCartType = 0x05;
		Cart->bHasRam = true;
		Cart->bHasBattery = true;
		Cart->bHasTimer = false;
		Cart->bHasRumble = false;
		break;
	case 0x1C:
		Cart->iCartType = 0x05;
		Cart->bHasRam = false;
		Cart->bHasBattery = false;
		Cart->bHasTimer = false;
		Cart->bHasRumble = true;
		break;
	case 0x1D:
		Cart->iCartType = 0x05;
		Cart->bHasRam = true;
		Cart->bHasBattery = false;
		Cart->bHasTimer = false;
		Cart->bHasRumble = true;
		break;
	case 0x1E:
		Cart->iCartType = 0x05;
		Cart->bHasRam = true;
		Cart->bHasBattery = true;
		Cart->bHasTimer = false;
		Cart->bHasRumble = true;
		break;
	}

	// Determin ROM size for paging checks
	Cart->iNumRomBanks = 2;
	switch (Cart->RomData[0x148]) {
	case 0x01:
		Cart->iNumRomBanks = 4;
		break;
	case 0x02:
		Cart->iNumRomBanks = 8;
		break;
	case 0x03:
		Cart->iNumRomBanks = 16;
		break;
	case 0x04:
		Cart->iNumRomBanks = 32;
		break;
	case 0x05:
		Cart->iNumRomBanks = 64;
		break;
	case 0x06:
		Cart->iNumRomBanks = 128;
		break;
	case 0x52:
		Cart->iNumRomBanks = 72;
		break;
	case 0x53:
		Cart->iNumRomBanks = 80;
		break;
	case 0x54:
		Cart->iNumRomBanks = 96;
		break;
	}

	// Determin RAM size for paging checks
	Cart->iNumRamBanks = 0;
	switch (Cart->RomData[0x149]) {
	case 0x01:
		Cart->iNumRamBanks = 1;
		break;
	case 0x02:
		Cart->iNumRamBanks = 1;
		break;
	case 0x03:
		Cart->iNumRamBanks = 4;
		break;
	case 0x04:
		Cart->iNumRamBanks = 16;
		break;
	}

// Attempt to load the SRAM file, but only if RAM is supposed to be present.
	if (Cart->bHasRam && Cart->bHasBattery) {
		hRamFile = CreateFile(RamFile, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_ALWAYS, 0, NULL);
		if (hRamFile == INVALID_HANDLE_VALUE) {
			MessageBox( g_strEmuInfo.hMainWindow, "Can't open SRAM file!\nContinuing with null savedata.", PLUGINWARNING, MB_OK | MB_ICONWARNING);
			ClearData(Cart->RamData, Cart->iNumRamBanks * 0x2000);
		} else {
			ReadFile(hRamFile, &Cart->RamData[0], Cart->iNumRamBanks * 0x2000, &dwBytesRead, NULL);
// Load the VisualBoy format TDF here, if it exists
			if (Cart->bHasTimer && Cart->bHasBattery) {
				lLength = GetFileSize(hRamFile, &lLengthHigh);
				if (lLength > ((unsigned) Cart->iNumRamBanks * 0x2000)) {
					// Looks like there is extra data in the SAV file than just RAM data... assume it is RTC data.
					ReadFile(hRamFile, &RTCTimer.mapperSeconds, sizeof(gbCartRTC), &dwBytesRead, NULL);
					// TODO: Make sure there is enough data read to be a valid RTC save
					Cart->TimerData[0] = RTCTimer.mapperSeconds;
					Cart->TimerData[1] = RTCTimer.mapperMinutes;
					Cart->TimerData[2] = RTCTimer.mapperHours;
					Cart->TimerData[3] = RTCTimer.mapperDays;
					Cart->TimerData[4] = RTCTimer.mapperControl;
					Cart->timerLastUpdate = RTCTimer.mapperLastTime;
					UpdateRTC(Cart);
				}
			}
			CloseHandle(hRamFile);
		}
	}

	Cart->TimerDataLatched = false;

	return true;
}

// Done
bool ReadCartNorm(LPGBCART Cart, DWORD dwAddress, BYTE *Data) // For all non-MBC carts; fixed 0x8000 ROM; fixed, optional 0x2000 RAM
{
	int i;

	if ((dwAddress >= 0) && (dwAddress <= 0x7FFF)) {
		for (i=0; i<32; i++) Data[i] = Cart->RomData[dwAddress + i];
		DebugWrite("Fixed ROM read - RAW\n");
	}
	if ((dwAddress >= 0xA000) && (dwAddress <= 0xBFFF)) {
		if (Cart->bHasRam) {
			for (i=0; i<32; i++) Data[i] = Cart->RamData[(dwAddress - 0xA000) + i];
			DebugWrite("RAM read: Bank=");
			DebugWriteByte(Cart->iCurrentRamBankNo);
			DebugWrite("\n");
		} else {
			for (i=0; i<32; i++) Data[i] = 0;
			DebugWrite("Failed RAM read: (RAM Not Present)");
			DebugWrite("\n");
		}
	}
	return true;
}

// Done
bool WriteCartNorm(LPGBCART Cart, DWORD dwAddress, BYTE *Data)
{
	int i;

	if (Cart->bHasRam == false) return true; // no RAM, exit
	if (Cart->RomData[0x149] == 1) { // Whoops... Only 1/4 of the RAM space is used.
		if ((dwAddress >= 0xA000) && (dwAddress <= 0xA7FF)) { // Write to RAM
			DebugWrite("RAM write: Unbanked");
			DebugWrite("\n");
			for (i=0; i<32; i++) Cart->RamData[(dwAddress - 0xA000) + i] = Data[i];
		}
	} else {
		if ((dwAddress >= 0xA000) && (dwAddress <= 0xBFFF)) { // Write to RAM
			DebugWrite("RAM write: Unbanked");
			DebugWrite("\n");
			for (i=0; i<32; i++) Cart->RamData[(dwAddress - 0xA000) + i] = Data[i];
		}
	}
	return true;
}

// Done
bool ReadCartMBC1(LPGBCART Cart, DWORD dwAddress, BYTE *Data)
{
	int i;

	if ((dwAddress >= 0) && (dwAddress <= 0x3FFF)) {
		for (i=0; i<32; i++) Data[i] = Cart->RomData[dwAddress + i];
		DebugWrite("Fixed ROM read - MBC1\n");
	}
	if ((dwAddress >= 0x4000) && (dwAddress <= 0x7FFF)) {
		if (Cart->iCurrentRomBankNo >= Cart->iNumRomBanks) {
			for (i=0; i<32; i++) Data[i] = 0;
			DebugWrite("Banked ROM read: (Banking Error)");
			DebugWriteByte(Cart->iCurrentRomBankNo);
			DebugWrite("\n");
		} else {
			for (i=0; i<32; i++) Data[i] = Cart->RomData[(dwAddress - 0x4000) + i + (Cart->iCurrentRomBankNo * 0x4000)];
			DebugWrite("Banked ROM read: Bank=");
			DebugWriteByte(Cart->iCurrentRomBankNo);
			DebugWrite("\n");
		}
	}
	if ((dwAddress >= 0xA000) && (dwAddress <= 0xBFFF)) {
		if (Cart->bHasTimer && (Cart->iCurrentRamBankNo >= 0x08 && Cart->iCurrentRamBankNo <= 0x0c)) {
			// w00t! the Timer was just read!! Now, if only the timer actualy existed...
			// TODO: Put timer read code in here... Timer register = (iCurrentRamBankNo - 0x08)
			for (i=0; i<32; i++) Data[i] = 0;
			return true;
		}
		if (Cart->bHasRam) {
			if (Cart->iCurrentRamBankNo >= Cart->iNumRamBanks) {
				for (i=0; i<32; i++) Data[i] = 0;
				DebugWrite("Failed RAM read: (Banking Error)");
				DebugWriteByte(Cart->iCurrentRamBankNo);
				DebugWrite("\n");
			} else {
				for (i=0; i<32; i++) Data[i] = Cart->RamData[(dwAddress - 0xA000) + i + (Cart->iCurrentRamBankNo * 0x2000)];
				DebugWrite("RAM read: Bank=");
				DebugWriteByte(Cart->iCurrentRamBankNo);
				DebugWrite("\n");
			}
		} else {
			for (i=0; i<32; i++) Data[i] = 0;
			DebugWrite("Failed RAM read: (RAM Not Present)");
			DebugWrite("\n");
		}
	}
	return true;
}

// Done
bool WriteCartMBC1(LPGBCART Cart, DWORD dwAddress, BYTE *Data)
{
	int i;

//	if ((dwAddress >= 0) && (dwAddress <= 0x1FFF)) { // RAM enable (Don't know what this does... Doesn't seem to affect a GB Rom cartrige)
//	}

	if ((dwAddress >= 0x2000) && (dwAddress <= 0x3FFF)) { // ROM bank select
		Cart->iCurrentRomBankNo = Data[0] & 0x0F;
		if (Cart->iCurrentRomBankNo == 0) {
			Cart->iCurrentRomBankNo = 1;
		}
		DebugWrite("Set Rom Bank=");
		DebugWriteByte(Cart->iCurrentRomBankNo);
		DebugWrite("\n");
	}
	if (Cart->bHasRam) {
		if ((dwAddress >= 0x4000) && (dwAddress <= 0x5FFF)) { // RAM bank select
			Cart->iCurrentRamBankNo = Data[0] & 0x07;
			DebugWrite("Set Ram Bank=");
			DebugWriteByte(Cart->iCurrentRamBankNo);
			DebugWrite("\n");
		}
		if ((dwAddress >= 0xA000) && (dwAddress <= 0xBFFF)) { // Write to RAM
			DebugWrite("RAM write: Bank=");
			DebugWriteByte(Cart->iCurrentRamBankNo);
			DebugWrite("\n");
			for (i=0; i<32; i++) Cart->RamData[(dwAddress - 0xA000) + i + (Cart->iCurrentRamBankNo * 0x2000)] = Data[i];
		}
	}
	return true;
}

// Done
bool ReadCartMBC2(LPGBCART Cart, DWORD dwAddress, BYTE *Data)
{
	int i;

	if ((dwAddress >= 0) && (dwAddress <= 0x3FFF)) {
		for (i=0; i<32; i++) Data[i] = Cart->RomData[dwAddress + i];
		DebugWrite("Fixed ROM read - MBC2\n");
	}
	if ((dwAddress >= 0x4000) && (dwAddress <= 0x7FFF)) {
		if (Cart->iCurrentRomBankNo >= Cart->iNumRomBanks) {
			for (i=0; i<32; i++) Data[i] = 0;
			DebugWrite("Banked ROM read: (Banking Error)");
			DebugWriteByte(Cart->iCurrentRomBankNo);
			DebugWrite("\n");
		} else {
			for (i=0; i<32; i++) Data[i] = Cart->RomData[(dwAddress - 0x4000) + i + (Cart->iCurrentRomBankNo * 0x4000)];
			DebugWrite("Banked ROM read: Bank=");
			DebugWriteByte(Cart->iCurrentRomBankNo);
			DebugWrite("\n");
		}
	}
	return true;
}

// Done
bool WriteCartMBC2(LPGBCART Cart, DWORD dwAddress, BYTE *Data)
{
	int i;

	if ((dwAddress >= 0x2000) && (dwAddress <= 0x3FFF)) { // ROM bank select
		Cart->iCurrentRomBankNo = Data[0] & 0x0F;
		if (Cart->iCurrentRomBankNo == 0) {
			Cart->iCurrentRomBankNo = 1;
		}
		DebugWrite("Set Rom Bank=");
		DebugWriteByte(Cart->iCurrentRomBankNo);
		DebugWrite("");
	}
	if (Cart->bHasRam) {
		if ((dwAddress >= 0x4000) && (dwAddress <= 0x5FFF)) { // RAM bank select
			Cart->iCurrentRamBankNo = Data[0] & 0x07;
			DebugWrite("Set Ram Bank=");
			DebugWriteByte(Cart->iCurrentRamBankNo);
			DebugWrite("\n");
		}
		if ((dwAddress >= 0xA000) && (dwAddress <= 0xBFFF)) { // Write to RAM
			DebugWrite("RAM write: Bank=");
			DebugWriteByte(Cart->iCurrentRamBankNo);
			DebugWrite("\n");
			for (i=0; i<32; i++) Cart->RamData[(dwAddress - 0xA000) + i + (Cart->iCurrentRamBankNo * 0x2000)] = Data[i];
		}
	}
	return true;
}

// Done
bool ReadCartMBC3(LPGBCART Cart, DWORD dwAddress, BYTE *Data)
{
	int i;

	if ((dwAddress >= 0) && (dwAddress <= 0x3FFF)) {
		for (i=0; i<32; i++) Data[i] = Cart->RomData[dwAddress + i];
		DebugWrite("Fixed ROM read - MBC3\n");
	}
	if ((dwAddress >= 0x4000) && (dwAddress <= 0x7FFF)) {
		if (Cart->iCurrentRomBankNo >= Cart->iNumRomBanks) {
			for (i=0; i<32; i++) Data[i] = 0;
			DebugWrite("Banked ROM read: (Banking Error)");
			DebugWriteByte(Cart->iCurrentRomBankNo);
			DebugWrite("\n");
		} else {
			for (i=0; i<32; i++) Data[i] = Cart->RomData[(dwAddress - 0x4000) + i + (Cart->iCurrentRomBankNo * 0x4000)];
			DebugWrite("Banked ROM read: Bank=");
			DebugWriteByte(Cart->iCurrentRomBankNo);
			DebugWrite("\n");
		}
	}
	if ((dwAddress >= 0xA000) && (dwAddress <= 0xBFFF)) {
		if (Cart->bHasTimer && (Cart->iCurrentRamBankNo >= 0x08 && Cart->iCurrentRamBankNo <= 0x0c)) {
			// w00t! the Timer was just read!! Now, if only the timer actualy existed...
			 UpdateRTC(Cart);
			if (Cart->TimerDataLatched) {
				for (i=0; i<32; i++) Data[i] = Cart->LatchedTimerData[Cart->iCurrentRamBankNo - 0x08];
			} else {
				for (i=0; i<32; i++) Data[i] = Cart->TimerData[Cart->iCurrentRamBankNo - 0x08];
			}
			return true;
		}
		if (Cart->bHasRam) {
			if (Cart->iCurrentRamBankNo >= Cart->iNumRamBanks) {
				for (i=0; i<32; i++) Data[i] = 0;
				DebugWrite("Failed RAM read: (Banking Error)");
				DebugWriteByte(Cart->iCurrentRamBankNo);
				DebugWrite("\n");
			} else {
				for (i=0; i<32; i++) Data[i] = Cart->RamData[(dwAddress - 0xA000) + i + (Cart->iCurrentRamBankNo * 0x2000)];
				DebugWrite("RAM read: Bank=");
				DebugWriteByte(Cart->iCurrentRamBankNo);
				DebugWrite("\n");
			}
		} else {
			for (i=0; i<32; i++) Data[i] = 0;
			DebugWrite("Failed RAM read: (RAM Not Present)\n");
		}
	}
	return true;
}

// Done
bool WriteCartMBC3(LPGBCART Cart, DWORD dwAddress, BYTE *Data)
{
	int i;

	if ((dwAddress >= 0) && (dwAddress <= 0x1FFF)) { // RAM enable (Don't know what this does... Doesn't seem to affect a GB Rom cartrige)
		DebugWrite("Set Ram Enable:");
		DebugWriteByte(Data[0]);
		DebugWrite("\n");
	}

	if ((dwAddress >= 0x2000) && (dwAddress <= 0x3FFF)) { // ROM bank select
		Cart->iCurrentRomBankNo = Data[0] & 0x03F;
		if (Cart->iCurrentRomBankNo == 0) {
			Cart->iCurrentRomBankNo = 1;
		}
		DebugWrite("Set Rom Bank:");
		DebugWriteByte(Cart->iCurrentRomBankNo);
		DebugWrite("\n");
	}
	if ((dwAddress >= 0x6000) && (dwAddress <= 0x7FFF)) { // Latch timer data
		for (i=0; i<6; i++) Cart->LatchedTimerData[i] = Cart->TimerData[i];
		if (Data[0] & 1) {
			// Latch
			Cart->TimerDataLatched = true;
			DebugWrite("Timer Data Latch: Enable\n");
		} else {
			Cart->TimerDataLatched = false;
			DebugWrite("Timer Data Latch: Disable\n");
		}
	}
	if (Cart->bHasRam) {
		if ((dwAddress >= 0x4000) && (dwAddress <= 0x5FFF)) { // RAM/Clock bank select
			Cart->iCurrentRamBankNo = Data[0] & 0x03;
			DebugWrite("Set Ram Bank=");
			DebugWriteByte(Cart->iCurrentRamBankNo);
			DebugWrite("\n");
			if (Cart->bHasTimer && (Data[0] >= 0x08 && Data[0] <= 0x0c)) {
				// Set the bank for the timer
				Cart->iCurrentRamBankNo = Data[0];
			}
		}
		if ((dwAddress >= 0xA000) && (dwAddress <= 0xBFFF)) { // Write to RAM
			if (Cart->iCurrentRamBankNo >= 0x08 && Cart->iCurrentRamBankNo <= 0x0c) {
				// Write to the timer
				Cart->TimerData[Cart->iCurrentRamBankNo - 0x08] = Data[0];
			} else {
				DebugWrite("RAM write: Bank=");
				DebugWriteByte(Cart->iCurrentRamBankNo);
				DebugWrite("\n");
				for (i=0; i<32; i++) Cart->RamData[(dwAddress - 0xA000) + i + (Cart->iCurrentRamBankNo * 0x2000)] = Data[i];
			}
		}
	}
	return true;
}

// Done
bool ReadCartMBC5(LPGBCART Cart, DWORD dwAddress, BYTE *Data)
{
	int i, iBank;

	if ((dwAddress >= 0) && (dwAddress <= 0x3FFF)) {
		for (i=0; i<32; i++) Data[i] = Cart->RomData[dwAddress + i];
		DebugWrite("Fixed ROM read - MBC5\n");
	}
	if ((dwAddress >= 0x4000) && (dwAddress <= 0x7FFF)) {
		if (Cart->iCurrentRomBankNo >= Cart->iNumRomBanks) {
			for (i=0; i<32; i++) Data[i] = 0;
			DebugWrite("Banked ROM read: (Banking Error)");
			DebugWriteByte(Cart->iCurrentRomBankNo);
			DebugWrite("\n");
		} else {
			iBank = Cart->iCurrentRamBankNo;
			if (iBank == 0) iBank = 1;
			for (i=0; i<32; i++) Data[i] = Cart->RomData[(dwAddress - 0x4000) + i + (iBank * 0x4000)];
			DebugWrite("Banked ROM read: Bank=");
			DebugWriteByte(Cart->iCurrentRomBankNo);
			DebugWrite("\n");
		}
	}
	if ((dwAddress >= 0xA000) && (dwAddress <= 0xBFFF)) {
		if (Cart->bHasRam) {
			if (Cart->iCurrentRamBankNo >= Cart->iNumRamBanks) {
				for (i=0; i<32; i++) Data[i] = 0;
				DebugWrite("Failed RAM read: (Banking Error)");
				DebugWriteByte(Cart->iCurrentRamBankNo);
				DebugWrite("\n");
			} else {
				for (i=0; i<32; i++) Data[i] = Cart->RamData[(dwAddress - 0xA000) + i + (Cart->iCurrentRamBankNo * 0x2000)];
				DebugWrite("RAM read: Bank=");
				DebugWriteByte(Cart->iCurrentRamBankNo);
				DebugWrite("\n");
			}
		} else {
			for (i=0; i<32; i++) Data[i] = 0;
			DebugWrite("Failed RAM read: (RAM Not Present)\n");
		}
	}
	return true;
}

// Done
bool WriteCartMBC5(LPGBCART Cart, DWORD dwAddress, BYTE *Data)
{
	int i;

//	if ((dwAddress >= 0) && (dwAddress <= 0x1FFF)) { // RAM enable (Don't know what this does... Doesn't seem to affect a GB Rom cartrige)
//	}

	if ((dwAddress >= 0x2000) && (dwAddress <= 0x2FFF)) { // ROM bank select, low bits
		Cart->iCurrentRomBankNo = ((int) Data[0]) | (Cart->iCurrentRomBankNo & 0x100);
		DebugWrite("Set Rom Bank=");
		DebugWriteByte(Cart->iCurrentRomBankNo);
		DebugWrite("\n");
	}
	if ((dwAddress >= 0x3000) && (dwAddress <= 0x3FFF)) { // ROM bank select, high bit
		Cart->iCurrentRomBankNo = (Cart->iCurrentRomBankNo & 0xFF) | ((((int) Data[0]) & 1) * 0x100);
		DebugWrite("Set Rom Bank=");
		DebugWriteByte(Cart->iCurrentRomBankNo);
		DebugWrite("\n");
	}
	if (Cart->bHasRam) {
		if ((dwAddress >= 0x4000) && (dwAddress <= 0x5FFF)) { // RAM bank select
			Cart->iCurrentRamBankNo = Data[0] & 0x0F;
			DebugWrite("Set Ram Bank=");
			DebugWriteByte(Cart->iCurrentRamBankNo);
			DebugWrite("\n");
		}
		if ((dwAddress >= 0xA000) && (dwAddress <= 0xBFFF)) { // Write to RAM
			if (Cart->iCurrentRamBankNo >= Cart->iNumRamBanks) {
				DebugWrite("RAM write: Buffer error on ");
				DebugWriteByte(Cart->iCurrentRamBankNo);
				DebugWrite("\n");
			} else {
				DebugWrite("RAM write: Bank=");
				DebugWriteByte(Cart->iCurrentRamBankNo);
				DebugWrite("\n");
				for (i=0; i<32; i++) Cart->RamData[(dwAddress - 0xA000) + i + (Cart->iCurrentRamBankNo * 0x2000)] = Data[i];
			}
		}
	}
	return true;
}

bool ReadCart(LPGBCART Cart, DWORD dwAddress, BYTE *Data)
{
#ifdef ENABLE_GBCART_READS_WRITES_DEBUG
	DebugWrite("Cart Read: Addy=");
	DebugWriteWord(dwAddress);
	DebugWrite("\n");
#endif

	switch (Cart->iCartType) {
	case 0x00: // Raw cartridge
		return ReadCartNorm(Cart, dwAddress, Data);
		break;
	case 0x01: // MBC1
		return ReadCartMBC1(Cart, dwAddress, Data);
		break;
	case 0x02: // MBC2
		return ReadCartMBC2(Cart, dwAddress, Data);
		break;
	case 0x04: // MBC3
		return ReadCartMBC3(Cart, dwAddress, Data);
		break;
	default: // Currently unsupported... Pretend it's a normal ROM-only cart
		return ReadCartNorm(Cart, dwAddress, Data);
		break;
	}
	return true;
}

bool WriteCart(LPGBCART Cart, DWORD dwAddress, BYTE *Data)
{
#ifdef ENABLE_GBCART_READS_WRITES_DEBUG
	DebugWrite("Cart Write: Addy=");
	DebugWriteWord(dwAddress);
	DebugWrite("\n");
#endif

	switch (Cart->iCartType) {
	case 0x00: // Raw cartridge
		return WriteCartNorm(Cart, dwAddress, Data);
		break;
	case 0x01: // MBC1
		return WriteCartMBC1(Cart, dwAddress, Data);
		break;
	case 0x02: // MBC2
		return WriteCartMBC2(Cart, dwAddress, Data);
		break;
	case 0x04: // MBC3
		return WriteCartMBC3(Cart, dwAddress, Data);
		break;
	case 0x05: // MBC5
		return WriteCartMBC5(Cart, dwAddress, Data);
		break;
	default: // Currently unsupported... Pretend it's a normal ROM-only cart
		return WriteCartNorm(Cart, dwAddress, Data);
		break;
	}
	return true;
}

bool SaveCart(LPGBCART Cart, LPTSTR SaveFile, LPTSTR TimeFile)
{
	HANDLE hRamFile;
	DWORD dwBytesWritten;
	int NumQuarterBlocks = 0;
	gbCartRTC RTCTimer;

	if (Cart->bHasRam && Cart->bHasBattery) {
		hRamFile = CreateFile(SaveFile, GENERIC_WRITE, FILE_SHARE_READ, NULL, CREATE_ALWAYS, 0, NULL );
		// TODO: Write only the bytes that NEED writing!
		switch (Cart->RomData[0x149]) {
		case 1:
			NumQuarterBlocks = 1;
			break;
		case 2:
			NumQuarterBlocks = 4;
			break;
		case 3:
			NumQuarterBlocks = 16;
			break;
		case 4:
			NumQuarterBlocks = 64;
			break;
		}
		WriteFile(hRamFile, Cart->RamData, NumQuarterBlocks * 0x800, &dwBytesWritten, NULL);
		if (Cart->bHasTimer) {
			// Save RTC in VisualBoy Advance format
			// TODO: Check if VBA saves are compatible with other emus.
			// TODO: Only write RTC data if VBA RTC data was originaly present
			RTCTimer.mapperSeconds = Cart->TimerData[0];
			RTCTimer.mapperMinutes = Cart->TimerData[1];
			RTCTimer.mapperHours = Cart->TimerData[2];
			RTCTimer.mapperDays = Cart->TimerData[3];
			RTCTimer.mapperControl = Cart->TimerData[4];
			RTCTimer.mapperLSeconds = Cart->LatchedTimerData[0];
			RTCTimer.mapperLMinutes = Cart->LatchedTimerData[1];
			RTCTimer.mapperLHours = Cart->LatchedTimerData[2];
			RTCTimer.mapperLDays = Cart->LatchedTimerData[3];
			RTCTimer.mapperLControl = Cart->LatchedTimerData[4];
			RTCTimer.mapperLastTime = Cart->timerLastUpdate;
			WriteFile(hRamFile, &RTCTimer.mapperSeconds, sizeof(gbCartRTC), &dwBytesWritten, NULL);
		}
		CloseHandle(hRamFile);
	}
	return true;
}

bool UnloadCart(LPGBCART Cart)
{
	// Nothing to do when unloading...
	return true;
}

// This is used to clear the RAM data to look like it has just been turned on.
// When a RAM chip is first turned on, it is filled with alternating 128-byte
// blocks of 0x00 and 0xFF.
void ClearData(char *Data, int Length)
{
	int i;

	for (i=0; i<Length; i++) {
		if ((i & 0x80) != 0x80) {
			Data[i] = (char) 0x00;
		} else {
			Data[i] = (char) 0xFF;
		}
	}
}
