/*	
	N-Rage`s Dinput8 Plugin -- V1.80a (23. 1. 2002)
    (C) 2002  Norbert Wladyka

	Author`s Email: norbert.wladyka@chello.at
	Website: http://go.to/nrage


    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <wtypes.h>
#include <winbase.h>
#include "commonIncludes.h"
#include "NRage PluginV2.h"
#include "DataBufferClass.h"

//HitTypes
#define DBC_HT_NOHIT		0	// Buffer doesnt contains Data
#define DBC_HT_DATAPART		1	// the Buffer contains a Part of Data
#define DBC_HT_FIRSTBYTE	2	// first DataByte is in Buffer
#define DBC_HT_LASTBYTE		3	// last DataByte is in Buffer
#define DBC_HT_COMPLETE		4	// the Data is completely in Buffer


DataBuffer::DataBuffer( const TCHAR *szFileName, ULONG nBytes, bool bCreate, bool bFileReadOnly )
:	dwBuffersize( nBytes ),
	hFile( INVALID_HANDLE_VALUE ),
	aBufferedData( NULL ),
	bBufferChange( false ),
	dwBufferIndex( 0 ),
	dwBufferFill( 0 ),
	dwFileSize( 0 )
{
	ZeroMemory( szFile, sizeof(szFile) );

	if( szFileName )
		SetFile( szFileName, bCreate, bFileReadOnly );
}

DataBuffer::~DataBuffer()	// Destruktor
{
	if( hFile != INVALID_HANDLE_VALUE )
		CloseFile();
}


//tests if the requested Data is (partially) in Buffer 
inline BYTE DataBuffer::Bufferhit( ULONG Offset, ULONG nBytes )
{
	BYTE bHitType = DBC_HT_NOHIT;

	const DWORD dwBufferEnd = dwBufferIndex + dwBufferFill,
				dwDataEnd = Offset + nBytes;

	if(( dwDataEnd <= dwBufferEnd ) && ( dwDataEnd > dwBufferIndex )) 
		bHitType = DBC_HT_LASTBYTE; // last DataByte is in Buffer
	if(( Offset < dwBufferEnd ) && ( Offset >= dwBufferIndex ))
	{
		if( bHitType = DBC_HT_LASTBYTE )
			bHitType = DBC_HT_COMPLETE; // Data Complete in Buffer
		else
			bHitType = DBC_HT_FIRSTBYTE; // first DataByte is in Buffer
	}

	if( !bHitType )
	{
		if(( dwBufferIndex > Offset ) && ( dwBufferEnd < dwDataEnd ))
			bHitType = DBC_HT_DATAPART; // Buffer contains part of Data
	}

	return bHitType;
}


// Sets size of Buffer
DWORD DataBuffer::SetBufferSize( ULONG nBytes )
{
	DWORD dwReturn = DBC_OK;
	BYTE *aNewBuffer;

	if( !aBufferedData || ( nBytes != dwBuffersize )) // just P_reallocate if new size or no Buffer
	{
		if( aBufferedData )
			dwReturn = StoreBuffer();

		if( dwReturn == DBC_OK )
		{
			aNewBuffer = (BYTE*)P_realloc( aBufferedData, sizeof(BYTE) * nBytes );
		
			if( aNewBuffer ) //P_reallocated succesfully
			{
				aBufferedData = aNewBuffer;
				dwBuffersize = nBytes;
				dwBufferFill = min( dwBuffersize, dwBufferFill );
			}	
			else
				dwReturn = DBC_CANTALLOCATEMEM;
		}
	}

	return dwReturn;
}

// Sets File to buffer
DWORD DataBuffer::SetFile( const TCHAR *szFileName, bool bCreate, bool bFileReadOnly )
{
	DWORD dwReturn = DBC_OK;
	this->bFileReadOnly = bFileReadOnly;

	if( lstrcmp( szFile, szFileName )) // if its another file
	{
		if( hFile != INVALID_HANDLE_VALUE ) // if thers a open file -> close it
			CloseFile();

		DWORD dwCreationDisposition = ( bCreate ) ? OPEN_ALWAYS : OPEN_EXISTING;
		DWORD dwDesiredAccess = ( bFileReadOnly ) ? GENERIC_READ : GENERIC_READ | GENERIC_WRITE;
		DWORD dwShareMode = ( bFileReadOnly ) ? FILE_SHARE_READ | FILE_SHARE_WRITE : FILE_SHARE_READ;
	
		// create file
		if( bFileReadOnly )
		hFile = CreateFile( szFileName, dwDesiredAccess, dwShareMode, NULL, dwCreationDisposition, 0, NULL );

		if( hFile == INVALID_HANDLE_VALUE )
			dwReturn = DBC_FILEOPENFAILED;
		else
		{
			dwFileSize = GetFileSize( hFile, NULL );
			lstrcpy( szFile, szFileName );
			dwReturn = SetBufferSize( dwBuffersize );
		}
	}

	return dwReturn;
}

// Closes File
DWORD DataBuffer::CloseFile()
{
	DWORD dwReturn = DBC_OK;

	if( hFile != INVALID_HANDLE_VALUE )
	{
		dwReturn = StoreBuffer();
		CloseHandle( hFile );
		hFile = INVALID_HANDLE_VALUE;
	}

	ZeroMemory( szFile, sizeof(szFile) );

	dwBufferIndex = 0;
	dwBufferFill = 0;
	dwFileSize = 0;
	bBufferChange = false;

	P_free( aBufferedData );
	aBufferedData = NULL;
	return dwReturn;
}

// Reads from File/Buffer
DWORD DataBuffer::ReadBFile( ULONG Offset, ULONG nBytes, void *Destination )
{
	DWORD dwReturn = DBC_OK;

	if( hFile == INVALID_HANDLE_VALUE )
		dwReturn = DBC_NOFILEBUFFERED;
	else
	{
		if(( Offset + nBytes ) > dwFileSize )
			dwReturn = DBC_READFAILED;
	}

	if( dwReturn == DBC_OK )
	{
		if( nBytes > dwBuffersize ) // better read direct if Buffer is smaller than data to read
		{
			if( Bufferhit( Offset, nBytes ) != DBC_HT_NOHIT )
				dwReturn = StoreBuffer(); // Stores eventual changes
			
			if( dwReturn == DBC_OK )
			{
				DWORD dwBytesRead;

				SetFilePointer( hFile, (LONG)Offset, NULL, FILE_BEGIN );
				ReadFile( hFile, Destination, nBytes, &dwBytesRead, NULL );
				if( nBytes != dwBytesRead )
					dwReturn = DBC_READFAILED;
			}
		}
		else
		{
			ULONG lPos,
				  lBufferedBytes;
			switch( Bufferhit( Offset, nBytes ))
			{
			case DBC_HT_NOHIT:
			case DBC_HT_DATAPART:
				dwReturn = LoadBuffer( Offset );
				if( dwReturn == DBC_OK )
					dwReturn = ReadBFile( Offset, nBytes, Destination );
				break;

			case DBC_HT_FIRSTBYTE:
			case DBC_HT_COMPLETE:
				lPos = Offset - dwBufferIndex; // Position of first buffered Byte
				lBufferedBytes = dwBufferFill - lPos;
				
				if( lBufferedBytes < nBytes )
				{
					CopyMemory( Destination, aBufferedData + lPos, lBufferedBytes );

					dwReturn = LoadBuffer( Offset + lBufferedBytes );
					if( dwReturn == DBC_OK )
						dwReturn = ReadBFile( Offset + lBufferedBytes, nBytes - lBufferedBytes, (BYTE*)Destination + lBufferedBytes );
				}
				else
					CopyMemory( Destination, aBufferedData + lPos, nBytes );
				break;

			case DBC_HT_LASTBYTE:
				lBufferedBytes = Offset + nBytes - dwBufferIndex;

				CopyMemory( (BYTE*)Destination + nBytes - lBufferedBytes, aBufferedData, lBufferedBytes );
				if( lBufferedBytes < nBytes )
				{
					lPos = ( Offset > dwBuffersize ) ? ( Offset - dwBuffersize ) : 0; // locate previous Block of Data
					dwReturn = LoadBuffer( lPos );
					if( dwReturn == DBC_OK )
						dwReturn = ReadBFile( Offset, nBytes - lBufferedBytes, Destination );
				}
				break;
			default:
				dwReturn = DBC_UNKNOWN; // shouldnt happen
			}
		}
	}

	return dwReturn;
}

// Writes to File/Buffer
DWORD DataBuffer::WriteBFile( ULONG Offset, ULONG nBytes, void *Source )
{
	DWORD dwReturn = DBC_OK;

	if( hFile == INVALID_HANDLE_VALUE )
		dwReturn = DBC_NOFILEBUFFERED;
	else if( bFileReadOnly )
		dwReturn = DBC_WRITEFAILED;
	else
	{
		DWORD dwBytesWritten;

		if( nBytes > dwBuffersize ) // writes direct to file
		{
			if( Bufferhit( Offset, nBytes ) != DBC_HT_NOHIT )
				dwReturn = StoreBuffer(); // Stores eventual changes

			if( dwReturn == DBC_OK )
			{
				SetFilePointer( hFile, (LONG)Offset, NULL, FILE_BEGIN );
				WriteFile( hFile, Source, nBytes, &dwBytesWritten, NULL );
				if( nBytes != dwBytesWritten )
					dwReturn = DBC_WRITEFAILED;
				else
					dwFileSize = max( dwFileSize, Offset + nBytes );
			}
		}
		else
		{
			ULONG lPos,
				  lBufferedBytes;
			switch( Bufferhit( Offset, nBytes ))
			{
			case DBC_HT_DATAPART:
				bBufferChange = false; // make sure the Buffer wont be written afterwards

			case DBC_HT_NOHIT:
				SetFilePointer( hFile, (LONG)Offset, NULL, FILE_BEGIN );
				WriteFile( hFile, Source, nBytes, &dwBytesWritten, NULL );
				if( nBytes == dwBytesWritten )
				{
					dwFileSize = max( dwFileSize, Offset + nBytes );
					LoadBuffer( Offset );
				}
				else
					dwReturn = DBC_WRITEFAILED;
				break;

			case DBC_HT_FIRSTBYTE:
			case DBC_HT_COMPLETE:
				lPos = Offset - dwBufferIndex;
				lBufferedBytes = dwBufferFill - lPos;

				CopyMemory( aBufferedData + lPos, Source, min( nBytes, lBufferedBytes ));
				bBufferChange = true;
				if( lBufferedBytes < nBytes )
				{
					dwReturn = LoadBuffer( Offset + lBufferedBytes );
					if( dwReturn == DBC_OK || dwReturn == DBC_ENDOFFILE )
						dwReturn = WriteBFile( Offset + lBufferedBytes, nBytes - lBufferedBytes, (BYTE*)Source + lBufferedBytes );
				}
				break;

			case DBC_HT_LASTBYTE:
				lPos = ( Offset > dwBuffersize ) ? ( Offset - dwBuffersize ) : 0;
				lBufferedBytes = Offset + nBytes - dwBufferIndex;

				CopyMemory( (BYTE*)aBufferedData, (BYTE*)Source + nBytes - lBufferedBytes, lBufferedBytes );
				bBufferChange = true;
				if( lBufferedBytes < nBytes )
				{
					dwReturn = LoadBuffer( lPos );
					if( dwReturn == DBC_OK )
						dwReturn = WriteBFile( Offset, nBytes - lBufferedBytes, Source );
				}
				break;
			}
		}
	}

	return dwReturn;
}

// Inserts nBytes at Offset. If Source is not NULL the Source will be written.
DWORD DataBuffer::InsertBlock( ULONG Offset, DWORD nBytes, void *Source )
{
	DWORD dwReturn = DBC_OK;
	BYTE *bBuffer = NULL;
	DWORD dwMoveBufferSize = min( dwFileSize - Offset, DBC_DEFAULTBUFFERSIZE );

	if( hFile == INVALID_HANDLE_VALUE )
		dwReturn = DBC_NOFILEBUFFERED;

	if( dwReturn == DBC_OK )
		dwReturn = StoreBuffer();

	if( dwReturn == DBC_OK )
	{
		bBuffer = (BYTE*)P_malloc( dwMoveBufferSize );
		if( !bBuffer )
			dwReturn = DBC_CANTALLOCATEMEM;
	}

	if( dwReturn == DBC_OK )
	{
		DWORD dwBlocktoMove = dwFileSize - Offset;
		
		DWORD dwBytestoMove;
		DWORD dwCurrentOffset;
		DWORD dwBytesMoved;

		while( dwBlocktoMove > 0 && ( dwReturn == DBC_OK ))
		{
			dwBytestoMove = min( dwMoveBufferSize, dwBlocktoMove );
			dwCurrentOffset = Offset + dwBlocktoMove - dwBytestoMove;

			SetFilePointer( hFile, (LONG)dwCurrentOffset, NULL, FILE_BEGIN );
			ReadFile( hFile, bBuffer, dwBytestoMove, &dwBytesMoved, NULL );
			if( dwBytestoMove == dwBytesMoved )
			{
				SetFilePointer( hFile, (LONG)dwCurrentOffset + nBytes, NULL, FILE_BEGIN );
				WriteFile( hFile, bBuffer, dwBytestoMove, &dwBytesMoved, NULL );
				if( dwBytestoMove != dwBytesMoved )
					dwReturn = DBC_READFAILED;
			}
			else
				dwReturn = DBC_READFAILED;

			dwBlocktoMove -= dwBytesMoved;
		}

		dwFileSize = GetFileSize( hFile, NULL );
		if( Source )
			WriteBFile( Offset, nBytes, Source );
	}

	P_free( bBuffer );

	return dwReturn;
}

// Removes nBytes at Offset and truncates file
DWORD DataBuffer::RemoveBlock( ULONG Offset, DWORD nBytes )
{
	DWORD dwReturn = DBC_OK;
	BYTE *bBuffer = NULL;
	DWORD dwMoveBufferSize = min( dwFileSize - Offset - nBytes, DBC_DEFAULTBUFFERSIZE );

	if( hFile == INVALID_HANDLE_VALUE )
		dwReturn = DBC_NOFILEBUFFERED;
	else
	{
		if(( Offset + nBytes ) > dwFileSize )
			dwReturn = DBC_READFAILED;
	}

	if( dwReturn == DBC_OK )
		dwReturn = StoreBuffer();

	if( dwReturn == DBC_OK )
	{
		bBuffer = (BYTE*)P_malloc( dwMoveBufferSize );
		if( !bBuffer )
			dwReturn = DBC_CANTALLOCATEMEM;
	}

	if( dwReturn == DBC_OK )
	{
		DWORD dwBlocktoMove = dwFileSize - Offset - nBytes;
		
		DWORD dwBytestoMove;
		DWORD dwCurrentOffset;
		DWORD dwBytesMoved;

		while( dwBlocktoMove > 0 && ( dwReturn == DBC_OK ))
		{
			dwBytestoMove = min( dwMoveBufferSize, dwBlocktoMove );
			dwCurrentOffset = dwFileSize - dwBlocktoMove;

			SetFilePointer( hFile, (LONG)dwCurrentOffset, NULL, FILE_BEGIN );
			ReadFile( hFile, bBuffer, dwBytestoMove, &dwBytesMoved, NULL );
			if( dwBytestoMove == dwBytesMoved )
			{
				SetFilePointer( hFile, (LONG)dwCurrentOffset - nBytes, NULL, FILE_BEGIN );
				WriteFile( hFile, bBuffer, dwBytestoMove, &dwBytesMoved, NULL );
				if( dwBytestoMove != dwBytesMoved )
					dwReturn = DBC_READFAILED;
			}
			else
				dwReturn = DBC_READFAILED;

			dwBlocktoMove -= dwBytesMoved;
		}

		if( dwReturn == DBC_OK )
			SetEndOfFile( hFile );

		dwFileSize = GetFileSize( hFile, NULL );
	}

	P_free( bBuffer );

	return dwReturn;
}

// writes Buffer where it belongs
DWORD DataBuffer::StoreBuffer()
{
	DWORD dwReturn = DBC_OK;

	if( hFile == INVALID_HANDLE_VALUE )
		dwReturn = DBC_NOFILEBUFFERED;
	else
	{
		if( bBufferChange )
		{
			DWORD dwBytesWritten;
	
			SetFilePointer( hFile, (LONG)dwBufferIndex, NULL, FILE_BEGIN );
			WriteFile( hFile, aBufferedData, dwBufferFill, &dwBytesWritten, NULL );
			if( dwBufferFill == dwBytesWritten )
				bBufferChange = false;
			else
				dwReturn = DBC_WRITEFAILED;
		}
	}
	return dwReturn;
}

// loads from File to Buffer
DWORD DataBuffer::LoadBuffer( ULONG Offset )
{
	DWORD dwReturn = DBC_OK;

	if( hFile == INVALID_HANDLE_VALUE )
		dwReturn = DBC_NOFILEBUFFERED;
	else
	{
		if( Offset < dwFileSize )
			dwReturn = StoreBuffer();
		else
			dwReturn = DBC_ENDOFFILE;

		if( dwReturn == DBC_OK )
		{
			DWORD dwBytestoRead;

			if((dwFileSize - Offset) < dwBuffersize )
				dwBytestoRead = dwFileSize - Offset;
			else
				dwBytestoRead = dwBuffersize;
			
			
			SetFilePointer( hFile, (LONG)Offset, NULL, FILE_BEGIN );
			if( ReadFile( hFile, aBufferedData, dwBytestoRead, &dwBufferFill, NULL ))
				dwBufferIndex = Offset;
			else
				dwReturn = DBC_READFAILED;
		}
	}

	return dwReturn;
}

DWORD DataBuffer::ReadLine( ULONG Offset, TCHAR *szLine, int iSize )
{
	DWORD dwReturn = DBC_OK;
	iSize--;

	bool EndofLine = false;
	BYTE bData;
	do
	{
		if(	( Offset >= dwBufferIndex + dwBufferFill ) 
			|| ( Offset < dwBufferIndex ))
			dwReturn = LoadBuffer( Offset );
		else
		{
			bData = aBufferedData[Offset - dwBufferIndex];
			if( bData == '\n' || bData == '\r' || iSize <= 1 )
			{
				EndofLine = true;
			}
			else
			{
				*szLine = bData;
				++Offset;
				++szLine;
				--iSize;
			}
		}
	}
	while( !EndofLine && ( dwReturn == DBC_OK ));
	*szLine = '\0';
	return dwReturn;
}