#include <PalmOS.h>
#include "b_globals.h"

enum Language {
	UNK_LANG = -1,	// Use default language (i.e. none specified)
	EN_ANY = 0,
	EN_USA = 1,
	DE_DEU = 2,
	FR_FRA = 3,
	IT_ITA = 4,
	PT_BRA = 5,
	ES_ESP = 6,
	JA_JPN = 7,
	ZH_TWN = 8,
	KO_KOR = 9,
	SE_SWE = 10,
	EN_GRB = 11,
	HB_ISR = 20,
	RU_RUS = 21,
	CZ_CZE = 22,
	NL_NLD = 23,
	NB_NOR = 24
};

enum Platform {
	kPlatformUnknown = -1,
	kPlatformPC = 0,
	kPlatformAmiga = 1,
	kPlatformAtariST = 2,
	kPlatformMacintosh = 3,
	kPlatformFMTowns = 4,
	kPlatformWindows = 5,
	kPlatformNES = 6,
	kPlatformC64 = 7
/*
	kPlatformSEGA,
	kPlatformPCEngine
*/
};

typedef struct {
	const char md5[32 + 1];
	const char target[16 + 1];
	enum Language language;
	enum Platform platform;
} MD5Table;

static void addScumm_mdtable() {
	MD5Table md5table[] = {
	{ "0305e850382b812fec6e5998ef88a966", "pjs-demo", NL_NLD, kPlatformWindows },
	{ "035deab53b47bc43abc763560d0f8d4b", "playfate", EN_ANY, kPlatformPC },
	{ "037385a953789190298494d92b89b3d0", "catalog2", EN_ANY, kPlatformWindows },
	{ "0425954a9db5c340861672892c3e678d", "samdemo", EN_ANY, kPlatformMacintosh },
	{ "04401d747f1a2c1c4b388daff71ed378", "ft", DE_DEU, kPlatformMacintosh },
	{ "04687cdf7f975a89d2474929f7b80946", "indy3towns", EN_ANY, kPlatformFMTowns },
	{ "0557df19f046a84c2fdc63507c6616cb", "farmdemo", NL_NLD, kPlatformWindows },
	{ "06b187468113f9ae5a400b148a847fac", "atlantis", EN_ANY, kPlatformMacintosh },
	{ "07433205acdca3bc553d0e731588b35f", "airport", EN_ANY, kPlatformWindows },
	{ "07b810e37be7489263f7bc7627d4765d", "freddi4", RU_RUS, kPlatformWindows },
	{ "084ed0fa98a6d1e9368d67fe9cfbd417", "freddemo", EN_ANY, kPlatformWindows },
	{ "0855496dde35356b1a9691e22ba84cdc", "freddemo", EN_ANY, kPlatformWindows },
	{ "08cc5c3eedaf72ebe12734eee94f7fa2", "balloon", EN_ANY, kPlatformUnknown },
	{ "09820417db26687bb7fe0c83cc4c553b", "ft", EN_ANY, kPlatformUnknown },
	{ "0a212fa35fa8421f31c1f3961272caf0", "monkeyvga", DE_DEU, kPlatformAmiga },
	{ "0ab19be9e2a3f6938226638b2a3744fe", "timedemo", EN_ANY, kPlatformUnknown },
	{ "0ac41e2e3d2174e5a042a6b565328dba", "racedemo", EN_ANY, kPlatformUnknown },
	{ "0b3222aaa7efcf283eb621e0cefd26cc", "puttputt", RU_RUS, kPlatformPC },
	{ "0cccfa5223099a60e76cfcca57a1a141", "freddi3", NL_NLD, kPlatformWindows },
	{ "0d1b69471605201ef2fa9cec1f5f02d2", "maniac", ES_ESP, kPlatformPC },
	{ "0e4c5d54a0ad4b26132e78b5ea76642a", "snmidemo", EN_ANY, kPlatformPC },
	{ "0f5935bd5e88ba6f09e558d64459746d", "1grademo", EN_ANY, kPlatformWindows },
	{ "0f6f2e716ba896a44e5059bba1de7ca9", "samnmax", IT_ITA, kPlatformUnknown },
	{ "0f9c7a76657f0840b8f7ccb5bffeb9f4", "indy3EGA", FR_FRA, kPlatformAtariST },
	{ "0fb73eddfcf584c02ba097984df131ba", "samnmax", DE_DEU, kPlatformUnknown },
	{ "1005456bfe351c1b679e1ff2dc2849e9", "puttzoo", EN_ANY, kPlatformWindows },
	{ "114acdc2659a273c220f86ee9edb24c1", "maniac", FR_FRA, kPlatformPC },
	{ "11ddf1fde76e3156eb3a38da213f484e", "monkey2", IT_ITA, kPlatformAmiga },
	{ "11e6e244078ff09b0f3832e35420e0a7", "catalog", EN_ANY, kPlatformWindows },
	{ "132bff65e6367c09cc69318ce1b59333", "monkey2", EN_ANY, kPlatformAmiga },
	{ "145bd3373574feb668cc2eea2ec6cf86", "balloon", RU_RUS, kPlatformWindows },
	{ "14d48c95b43ddeb983254cf6c43851f1", "freddi4", NL_NLD, kPlatformWindows },
	{ "157367c3c21e0d03a0cba44361b4cf65", "indy3EGA", EN_ANY, kPlatformAtariST },
	{ "15e03ffbfeddb9c2aebc13dcb2a4a8f4", "monkeyvga", EN_ANY, kPlatformPC },
	{ "16542a7342a918bfe4ba512007d36c47", "FreddisFunShop", EN_ANY, kPlatformUnknown },
	{ "16effd200aa6b8abe9c569c3e578814d", "f4-demo", NL_NLD, kPlatformWindows },
	{ "179879b6e35c1ead0d93aab26db0951b", "fbear", EN_ANY, kPlatformWindows },
	{ "17b5d5e6af4ae89d62631641d66d5a05", "indy3", IT_ITA, kPlatformPC },
	{ "17fa250eb72dae2dad511ba79c0b6b0a", "dottdemo", FR_FRA, kPlatformPC },
	{ "182344899c2e2998fca0bebcd82aa81a", "atlantis", EN_ANY, kPlatformPC },
	{ "183d7464902d40d00800e8ee1f04117c", "maniac", DE_DEU, kPlatformPC },
	{ "1875b90fade138c9253a8e967007031a", "indy3", UNK_LANG, kPlatformPC },
	{ "187d315f6b5168f68680dfe8c3d76a3e", "loom", HB_ISR, kPlatformPC },
	{ "1900e501a52fbf55bde6e4196f6d2aa6", "zak", IT_ITA, kPlatformPC },
	{ "19263586f749a560c1adf8b3393a9593", "socks", RU_RUS, kPlatformWindows },
	{ "1a6e5ae2777a6a33f06ffc0226210934", "atlantis", EN_ANY, kPlatformMacintosh },
	{ "1c792d28376d45e145cb916bca0400a2", "sf2-demo", NL_NLD, kPlatformUnknown },
	{ "1ca86e2cf9aaa2068738a1e5ba477e60", "zakTowns", JA_JPN, kPlatformFMTowns },
	{ "1d05cd189e4908f79b57e78a4402f292", "monkeyega", EN_ANY, kPlatformPC },
	{ "1dd3c11ea4439adfe681e4e405b624e1", "monkeyega", FR_FRA, kPlatformPC },
	{ "1dd7aa088e09f96d06818aa9a9deabe0", "indy3EGA", EN_ANY, kPlatformMacintosh },
	{ "1fbebd7b2b692df5297870447a80cfed", "atlantis", DE_DEU, kPlatformPC },
	{ "2012f854d83d9cc6f73b2b544cd8bbf8", "water", RU_RUS, kPlatformWindows },
	{ "2108d83dcf09f8adb4bc524669c8cf51", "PuttTime", EN_ANY, kPlatformUnknown },
	{ "21a6592322f92550f144f68a8a4e685e", "dig", FR_FRA, kPlatformMacintosh },
	{ "21abe302e1b1e2b66d6f5c12e241ebfd", "freddicove", RU_RUS, kPlatformWindows },
	{ "225e18566e810c634bf7de63e7568e3e", "mustard", EN_ANY, kPlatformUnknown },
	{ "22c9eb04455440131ffc157aeb8d40a8", "fbdemo", EN_ANY, kPlatformWindows },
	{ "22d07d6c386c9c25aca5dac2a0c0d94b", "maniac", SE_SWE, kPlatformNES },
	{ "22f4ea88a09da12df9308ba30bcb7d0f", "loom", EN_ANY, kPlatformPC },
	{ "2723fea3dae0cb47768c424b145ae0e7", "tentacle", EN_ANY, kPlatformPC },
	{ "27b3a4224ad63d5b04627595c1c1a025", "zak", IT_ITA, kPlatformAmiga },
	{ "28d24a33448fab6795850bc9f159a4a2", "indydemo", JA_JPN, kPlatformFMTowns },
	{ "28ef68ee3ed76d7e2ee8ee13c15fbd5b", "loom", EN_ANY, kPlatformPC },
	{ "2a208ffbcd0e83e86f4356e6f64aa6e1", "loom", ES_ESP, kPlatformPC },
	{ "2ccd8891ce4d3f1a334d21bff6a88ca2", "monkey", EN_ANY, kPlatformMacintosh },
	{ "2d1e891fe52df707c30185e52c50cd92", "monkey", EN_ANY, kPlatformPC },
	{ "2d388339d6050d8ccaa757b64633954e", "zakTowns", EN_ANY, kPlatformFMTowns },
	{ "2d4536a56e01da4b02eb021e7770afa2", "zakTowns", EN_ANY, kPlatformFMTowns },
	{ "2e85f7aa054930c692a5b1bed1dfc295", "footdemo", EN_ANY, kPlatformUnknown },
	{ "2fe369ad70f52a8cf7ad6077ee64f81a", "loom", DE_DEU, kPlatformAmiga },
	{ "305d3dd57c96c65b017bc70c8c7cfb5e", "monkey", DE_DEU, kPlatformPC },
	{ "30ba1e825d4ad2b448143ae8df18482a", "pj2demo", NL_NLD, kPlatformWindows },
	{ "319a4dde52c7960b5aae8a1ec348d918", "monkeyvga", DE_DEU, kPlatformAmiga },
	{ "31aa57f460a3d12429f0552a46a90b39", "puttdemo", EN_ANY, kPlatformPC },
	{ "31b8fda4c8c7413fa6b39997e776eba4", "loomTowns", JA_JPN, kPlatformFMTowns },
	{ "32709cbeeb3044b34129950860a83f14", "pajama2", RU_RUS, kPlatformWindows },
	{ "32a433dea56b86a55b59e4ff7d755711", "ftpcdemo", EN_ANY, kPlatformPC },
	{ "330f631502e381a4e199a3f7cb483c20", "indy3EGA", DE_DEU, kPlatformAmiga },
	{ "3433be9866ca4261b2d5d25374e3f243", "monkeyvga", FR_FRA, kPlatformAmiga },
	{ "3486ede0f904789267d4bcc5537a46d4", "zoodemo", EN_ANY, kPlatformMacintosh },
	{ "35a2d3040fa512f8232d9e443319d84d", "dig", EN_ANY, kPlatformMacintosh },
	{ "362c1d281fb9899254cda66ad246c66a", "digdemo", EN_ANY, kPlatformUnknown },
	{ "3686cf8f89e102ececf4366e1d2c8126", "monkey2", EN_ANY, kPlatformPC },
	{ "36a6750e03fb505fc19fc2bf3e4dbe91", "pj2demo", EN_ANY, kPlatformUnknown },
	{ "37aed3f91c1ef959e0bd265f9b13781f", "pajama", EN_ANY, kPlatformUnknown },
	{ "37f56ceb13e401a7ac7d9e6b37fecaf7", "loom", EN_ANY, kPlatformPC },
	{ "37ff1b308999c4cca7319edfcc1280a0", "puttdemo", EN_ANY, kPlatformWindows },
	{ "3824e60cdf639d22f6df92a03dc4b131", "fbear", EN_ANY, kPlatformPC },
	{ "387a544b8b10b26912d8413bab63a853", "mi2demo", EN_ANY, kPlatformPC },
	{ "3905799e081b80a61d4460b7b733c206", "maniac", EN_ANY, kPlatformNES },
	{ "3938ee1aa4433fca9d9308c9891172b1", "zakTowns", EN_ANY, kPlatformFMTowns },
	{ "399b217b0c8d65d0398076da486363a9", "indy3", DE_DEU, kPlatformPC },
	{ "39cb9dec16fa16f38d79acd80effb059", "loom", FR_FRA, kPlatformAmiga },
	{ "39cb9dec16fa16f38d79acd80effb059", "loom", IT_ITA, kPlatformAmiga },
	{ "3a03dab514e4038df192d8a8de469788", "atlantis", EN_ANY, kPlatformAmiga },
	{ "3a0c35f3c147b98a2bdf8d400cfc4ab5", "indy3towns", JA_JPN, kPlatformFMTowns },
	{ "3b301b7892f883ce42ab4be6a274fea6", "samnmax", EN_ANY, kPlatformPC },
	{ "3de99ef0523f8ca7958faa3afccd035a", "spyfox", EN_ANY, kPlatformUnknown },
	{ "3df6ead57930488bc61e6e41901d0e97", "fbear", EN_ANY, kPlatformMacintosh },
	{ "40564ec47da48a67787d1f9bd043902a", "maniac", EN_ANY, kPlatformPC },
	{ "4167a92a1d46baa4f4127d918d561f88", "tentacle", EN_ANY, kPlatformUnknown },
	{ "430bc518017b6fac046f58bab6baad5d", "monkey2", JA_JPN, kPlatformFMTowns },
	{ "45082a5c9f42ba14dacfe1fdeeba819d", "ff5demo", EN_ANY, kPlatformWindows },
	{ "45152f7cf2ba8f43cf8a8ea2e740ae09", "monkeyvga", ES_ESP, kPlatformPC },
	{ "46b53fd430adcfbed791b48a0d4b079f", "funpack", EN_ANY, kPlatformPC },
	{ "477dbafbd66a53c98416dc01aef019ad", "monkeyega", IT_ITA, kPlatformPC },
	{ "47e75b1bdcb44c78cb94883d1731ccf8", "fbdemo", EN_ANY, kPlatformPC },
	{ "49210e124e4c2b30f1290a9ef6306301", "monkeyega", EN_ANY, kPlatformPC },
	{ "499c958affc394f2a3868f1eb568c3ee", "f4-demo", NL_NLD, kPlatformWindows },
	{ "4af4a6b248103c1fe9edef619677f540", "moondemo", EN_ANY, kPlatformMacintosh },
	{ "4ba37f835be11a59d969f90f272f575b", "water", EN_ANY, kPlatformUnknown },
	{ "4ba7fb331296c283e73d8f5b2096e551", "samnmax", ES_ESP, kPlatformUnknown },
	{ "4bedb49943df95a9c900a5a82ccbe9de", "ft", FR_FRA, kPlatformUnknown },
	{ "4cb9c3618f71668f8e4346c8f323fa82", "monkey2", EN_ANY, kPlatformMacintosh },
	{ "4ce2d5b355964bbcb5e5ce73236ef868", "freddicove", RU_RUS, kPlatformWindows },
	{ "4d34042713958b971cb139fba4658586", "indy4", JA_JPN, kPlatformFMTowns },
	{ "4dbff3787aedcd96b0b325f2d92d7ad9", "maze", EN_ANY, kPlatformUnknown },
	{ "4dc780f1bc587a193ce8a97652791438", "loom", EN_ANY, kPlatformAmiga },
	{ "4e5867848ee61bc30d157e2c94eee9b4", "timedemo", EN_ANY, kPlatformUnknown },
	{ "4edbf9d03550f7ba01e7f34d69b678dd", "spydemo", NL_NLD, kPlatformWindows },
	{ "4f04b321a95d4315ce6d65f8e1dd0368", "maze", EN_ANY, kPlatformUnknown },
	{ "4f267a901719623de7dde83e47d5b474", "atlantis", DE_DEU, kPlatformAmiga },
	{ "4f580a021eee026f3b4589e17d130d78", "freddi4", UNK_LANG, kPlatformUnknown },
	{ "4fa6870d9bc8c313b65d54b1da5a1891", "pajama", NL_NLD, kPlatformWindows },
	{ "4fbbe9f64b8bc547503a379a301183ce", "tentacle", IT_ITA, kPlatformUnknown },
	{ "5057fb0e99e5aa29df1836329232f101", "freddi2", UNK_LANG, kPlatformWindows },
	{ "507bb360688dc4180fdf0d7597352a69", "freddi", SE_SWE, kPlatformWindows },
	{ "50fcdc982a25063b78ad46bf389b8e8d", "tentacle", IT_ITA, kPlatformPC },
	{ "51305e929e330e24a75a0351c8f9975e", "freddi2", EN_ANY, kPlatformUnknown },
	{ "5262a27afcaee04e5c4900220bd463e7", "PuttsFunShop", EN_ANY, kPlatformUnknown },
	{ "52a4bae0746a11d7b1e8554e91a6645c", "zak", FR_FRA, kPlatformPC },
	{ "53e94115b55dd51d4b8ff0871aa1df1e", "spydemo", EN_ANY, kPlatformUnknown },
	{ "54a936ad06161ff7bfefcb96200f7bff", "monkeyvga", EN_ANY, kPlatformAmiga },
	{ "55518cd73cf9c6d23ea29c51ee06bdfe", "ft", IT_ITA, kPlatformUnknown },
	{ "55e4cc866ff9046824e1c638ba2b8c7f", "ft", RU_RUS, kPlatformUnknown },
	{ "566165a7338fa11029e7c14d94fa70d0", "freddemo", EN_ANY, kPlatformWindows },
	{ "5798972220cd458be2626d54c80f71d7", "atlantis", IT_ITA, kPlatformAmiga },
	{ "57b0d89af79befe1cabce3bece869e7f", "tentacle", DE_DEU, kPlatformPC },
	{ "589601b676c98b1c0c987bc031ab68b3", "chase", EN_ANY, kPlatformUnknown },
	{ "58fdf4c7ad13540a734e18f8584cad89", "puttzoo", EN_ANY, kPlatformMacintosh },
	{ "590e6546aacd0d374b7f3a4f53013ab1", "freddicove", EN_ANY, kPlatformUnknown },
	{ "5a35e36fd777e9c37a49c5b2faca52f9", "loom", EN_ANY, kPlatformPC },
	{ "5b08000a9c47b2887df6506ac767ca68", "fbear", EN_ANY, kPlatformUnknown },
	{ "5bd335265a61caa3d78956ad9f88ba23", "footdemo", EN_ANY, kPlatformUnknown },
	{ "5c21fc49aee8f46e58fef21579e614a1", "thinker1", EN_ANY, kPlatformUnknown },
	{ "5d88b9d6a88e6f8e90cded9d01b7f082", "loomcd", EN_ANY, kPlatformPC },
	{ "5e8fb66971a60e523e5afbc4c129c0e8", "socks", EN_ANY, kPlatformUnknown },
	{ "5fbe557049892eb4b709d90916ec97ca", "indy3EGA", EN_ANY, kPlatformPC },
	{ "600abd3e9f47e63e670188b7e4e86ac7", "spyozon", EN_ANY, kPlatformUnknown },
	{ "6027e9ca9c35746d95dee2068cec17e5", "zak", DE_DEU, kPlatformAmiga },
	{ "60ba818dc3bede86d40357e3913f8505", "ft", EN_ANY, kPlatformUnknown },
	{ "613f64f78ea26c7353b2a5940eb61d6a", "zak", FR_FRA, kPlatformAtariST },
	{ "624cdb93654667c869d204a64af7e57f", "maniac", EN_ANY, kPlatformPC },
	{ "6271130f440066830eca9056c1d7926f", "water", RU_RUS, kPlatformWindows },
	{ "66236cd1aec24e1d4aff4c4cc93b7e18", "indy3EGA", FR_FRA, kPlatformPC },
	{ "66fd5ff9a810dfeb6d6bdada18221140", "monkeyvga", IT_ITA, kPlatformPC },
	{ "672dec94b82f7f0877ebb5b5cf7f4bc1", "pajama", EN_ANY, kPlatformUnknown },
	{ "675d71151e9b5a968c8ce46d9fbf4cbf", "zak", EN_ANY, kPlatformPC },
	{ "68155a6bf082221525f431c2cbdac8ab", "SamsFunShop", EN_ANY, kPlatformUnknown },
	{ "684732efb5799c0f78804c99d8de9aba", "puttputt", EN_ANY, kPlatformMacintosh },
	{ "6886e5d08cee329b1f2e743ae2e3ceed", "monkey2", DE_DEU, kPlatformPC },
	{ "695fe0b3963333b7e15b37514db3c745", "kinddemo", EN_ANY, kPlatformUnknown },
	{ "697c9b7c55a05d8199c48b48e379d2c8", "puttmoon", HB_ISR, kPlatformPC },
	{ "69ea626f1f87eecb78ea0d6c6b983a1d", "monkey2", IT_ITA, kPlatformPC },
	{ "6a30a07f353a75cdc602db27d73e1b42", "puttputt", EN_ANY, kPlatformWindows },
	{ "6af2419fe3db5c2fdb091ae4e5833770", "racedemo", NL_NLD, kPlatformUnknown },
	{ "6b19d0e25cbf720d05822379b8b90ed9", "timedemo", NL_NLD, kPlatformWindows },
	{ "6b257bb2827dd894b8109a50a1a18b5a", "FF5Demo", NL_NLD, kPlatformWindows },
	{ "6b3ec67da214f558dc5ceaa2acd47453", "indy3EGA", EN_ANY, kPlatformPC },
	{ "6bf70eee5de3d24d2403e0dd3d267e8a", "spyfox", EN_ANY, kPlatformWindows },
	{ "6c2bff0e327f2962e809c2e1a82d7309", "monkeyvga", EN_ANY, kPlatformAmiga },
	{ "6df20c50c1ab19799de9be7ae7716881", "fbdemo", EN_ANY, kPlatformMacintosh },
	{ "6e959d65358eedf9b68b81e304b97fa4", "tentacle", DE_DEU, kPlatformUnknown },
	{ "6ea966b4d660c870b9ee790d1fbfc535", "monkey2", ES_ESP, kPlatformAmiga },
	{ "6f0be328c64d689bb606d22a389e1b0f", "loom", EN_ANY, kPlatformMacintosh },
	{ "6f6ef668c608c7f534fea6e6d3878dde", "indy3EGA", DE_DEU, kPlatformPC },
	{ "701246819d1a70573f41bf33fc19214f", "soccer", EN_ANY, kPlatformWindows },
	{ "7020931d5a2be0a49d68e7a1882363e4", "zak", EN_ANY, kPlatformPC },
	{ "71523b539491527d9860f4407faf0411", "monkeyega", EN_ANY, kPlatformPC },
	{ "71fe97c3108678cf604f14abe342341b", "spyfox2", NL_NLD, kPlatformWindows },
	{ "7222f260253f325c21fcfa68b5bfab67", "sf2-demo", EN_ANY, kPlatformUnknown },
	{ "72ac6bc980d5101c2142189d746bd62f", "spyfox", RU_RUS, kPlatformWindows },
	{ "73e5ab7dbb9a8061cc6d25df02dbd1e7", "loom", EN_ANY, kPlatformPC },
	{ "746e88c172a5b7a1ae89ac0ee3ee681a", "freddi", RU_RUS, kPlatformWindows },
	{ "754feb59d3bf86b8a00840df74fd7b26", "f3-mdemo", NL_NLD, kPlatformWindows },
	{ "75ba23fff4fd63fa446c02864f2a5a4b", "zak", IT_ITA, kPlatformPC },
	{ "75bff95816b84672b877d22a911ab811", "freddi3", RU_RUS, kPlatformWindows },
	{ "771bc18ec6f93837b839c992b211904b", "monkeyega", DE_DEU, kPlatformPC },
	{ "77f5c9cc0986eb729c1a6b4c8823bbae", "zakTowns", EN_ANY, kPlatformFMTowns },
	{ "780e4a0ae2ff17dc296f4a79543b44f8", "puttmoon", UNK_LANG, kPlatformPC },
	{ "78bd5f036ea35a878b74e4f47941f784", "freddi4", RU_RUS, kPlatformWindows },
	{ "7974365d3dc0f43a2748c975f91ff042", "monkey2", ES_ESP, kPlatformPC },
	{ "7bad72e332a59f9fcc1d437f4edad32a", "puttcircus", RU_RUS, kPlatformUnknown },
	{ "7c2e76087027eeee9c8f8985f93a1cc5", "f4-demo", EN_ANY, kPlatformUnknown },
	{ "7ddeaf52c8b9a50551ce0aa2ac811d07", "BluesABCTimeDemo", EN_ANY, kPlatformUnknown },
	{ "7e151c17adf624f1966c8fc5827c95e9", "puttputt", EN_ANY, kPlatformUnknown },
	{ "7ea2da67ebabea4ac20cee9f4f9d2934", "airdemo", EN_ANY, kPlatformMacintosh },
	{ "7edd665bbede7ea8b7233f8e650be6f8", "samnmax", FR_FRA, kPlatformUnknown },
	{ "7f45ddd6dbfbf8f80c0c0efea4c295bc", "maniac", EN_ANY, kPlatformPC },
	{ "7fc6cdb46b4c9d384c52327f4bca6416", "football", EN_ANY, kPlatformWindows },
	{ "810a9da887aefa597b0cf3c77d262897", "BluesABCTimeDemo", EN_ANY, kPlatformWindows },
	{ "81bbfa181184cb494e7a81dcfa94fbd9", "maniac", FR_FRA, kPlatformNES },
	{ "8299d9b8a1b0e7b881bae7a9971dc5e2", "zak", EN_ANY, kPlatformAtariST },
	{ "8368f552b1e3eba559f8d559bcc4cadb", "freddi3", UNK_LANG, kPlatformUnknown },
	{ "861e59ed72a1cd0e6d454f7ee7e2bf3d", "comi", RU_RUS, kPlatformUnknown },
	{ "86be8ada36371d4fdc35659d0e912a26", "indy3EGA", ES_ESP, kPlatformPC },
	{ "86c9902b7bec1a17926d4dae85beaa45", "airdemo", EN_ANY, kPlatformWindows },
	{ "870d1e3c86bc50846d808d14a36b4e08", "monkeyvga", ES_ESP, kPlatformAmiga },
	{ "87f6e8037b7cc996e13474b491a7a98e", "maniac", IT_ITA, kPlatformPC },
	{ "8801fb4a1200b347f7a38523339526dd", "jungle", EN_ANY, kPlatformWindows },
	{ "883af4b0af4f77a92f1dcf1d0a283140", "tentacle", ES_ESP, kPlatformUnknown },
	{ "898ce8eb1234a955ef75e87141902bb3", "freddi3", RU_RUS, kPlatformWindows },
	{ "8a484262363a8e18be87112454f1456b", "pjgames", EN_ANY, kPlatformWindows },
	{ "8aa05d3cdb0e795436043f0546af2da2", "tentacle", FR_FRA, kPlatformUnknown },
	{ "8afb3cf9f95abf208358e984f0c9e738", "funpack", EN_ANY, kPlatformUnknown },
	{ "8bdb0bf87b5e303dd35693afb9351215", "ft", DE_DEU, kPlatformUnknown },
	{ "8d479e36f35e80257dfc102cf4b8a912", "farmdemo", EN_ANY, kPlatformWindows },
	{ "8e3241ddd6c8dadf64305e8740d45e13", "balloon", EN_ANY, kPlatformUnknown },
	{ "8e4ee4db46954bfe2912e259a16fad82", "monkey2", FR_FRA, kPlatformPC },
	{ "8eb84cee9b429314c7f0bdcf560723eb", "monkey", EN_ANY, kPlatformFMTowns },
	{ "8ee63cafb1fe9d62aa0d5a23117e70e7", "freddi2", EN_ANY, kPlatformUnknown },
	{ "8f3758ff98c9c5d78e5d635222cad026", "atlantis", IT_ITA, kPlatformPC },
	{ "8fec68383202d38c0d25e9e3b757c5df", "comidemo", UNK_LANG, kPlatformUnknown },
	{ "8ffd618a776a4c0d8922bb28b09f8ce8", "airdemo", EN_ANY, kPlatformWindows },
	{ "90a329d8ad5b7ce0690429e98cfbb32f", "funpack", HB_ISR, kPlatformPC },
	{ "90c755e1c9b9b8a4129d37b2259d0655", "chase", EN_ANY, kPlatformUnknown },
	{ "910e31cffb28226bd68c569668a0d6b4", "monkeyega", ES_ESP, kPlatformPC },
	{ "91469353f7be1b122fa88d23480a1320", "zak", FR_FRA, kPlatformAmiga },
	{ "92b078d9d6d9d751da9c26b8b3075779", "tentacle", FR_FRA, kPlatformPC },
	{ "92e7727e67f5cd979d8a1070e4eb8cb3", "puttzoo", EN_ANY, kPlatformUnknown },
	{ "92fc0073a4cf259ff36070ecb8628ba8", "thinkerk", EN_ANY, kPlatformUnknown },
	{ "96a3069a3c63caa7329588ce1fef41ee", "spyozon", RU_RUS, kPlatformUnknown },
	{ "9708cf716ed8bcc9ff3fcfc69413b746", "puttputt", EN_ANY, kPlatformPC },
	{ "981e1e1891f2be7e25a01f50ae55a5af", "puttrace", EN_ANY, kPlatformUnknown },
	{ "98744fe66ff730e8c2b3b1f58803ab0b", "playfate", EN_ANY, kPlatformPC },
	{ "99a3699f80b8f776efae592b44b9b991", "maniac", FR_FRA, kPlatformPC },
	{ "99b6f822b0b2612415407865438697d6", "fate", EN_ANY, kPlatformPC },
	{ "9bc548e179cdb0767009401c094d0895", "maniac", DE_DEU, kPlatformAmiga },
	{ "9bd2a8f72613e715c199246dd511e10f", "atlantis", ES_ESP, kPlatformPC },
	{ "9bda5fee51d2fda5253d02c642016bf4", "spyfox", NL_NLD, kPlatformWindows },
	{ "9c0fee288ad564a7d25ec3e841810d79", "indy3EGA", EN_ANY, kPlatformAmiga },
	{ "9c143c5905055d5df7a0f014ab379aee", "moondemo", EN_ANY, kPlatformWindows },
	{ "9c92eeaf517a31b7221ec2546ab669fd", "puttmoon", EN_ANY, kPlatformWindows },
	{ "9cdd327c1034c046cb595d251c44da2f", "chase", RU_RUS, kPlatformWindows },
	{ "9d4ab3e0e1d1ebc6ba8a6a4c470ed184", "spydemo", EN_ANY, kPlatformWindows },
	{ "9d7b67be003fea60be4dcbd193611936", "ftdemo", EN_ANY, kPlatformMacintosh },
	{ "9dc02577bf50d4cfaf3de3fbac06fbe2", "puttmoon", EN_ANY, kPlatformMacintosh },
	{ "9e5e0fb43bd22f4628719b7501adb717", "monkeyega", FR_FRA, kPlatformAtariST },
	{ "a194f15f51ee62badab74b9e7da97693", "bb2demo", EN_ANY, kPlatformWindows },
	{ "a28135a7ade38cc0208b04507c46efd1", "spyfox", DE_DEU, kPlatformWindows },
	{ "a2bb6aa0537402c1b3c2ea899ccef64b", "smaller", EN_ANY, kPlatformWindows },
	{ "a3036878840720fbefa41e6965fa4a0a", "samnmax", EN_ANY, kPlatformPC },
	{ "a525c1753c1db5011c00417da37887ef", "PuttTime", EN_ANY, kPlatformUnknown },
	{ "a561d2e2413cc1c71d5a1bf87bf493ea", "lost", EN_ANY, kPlatformUnknown },
	{ "a570381b028972d891052ee1e51dc011", "maniac", EN_ANY, kPlatformAtariST },
	{ "a654fb60c3b67d6317a7894ffd9f25c5", "pj3-demo", EN_ANY, kPlatformWindows },
	{ "a7cacad9c40c4dc9e1812abf6c8af9d5", "circdemo", EN_ANY, kPlatformUnknown },
	{ "a85856675429fe88051744f755b72f93", "farm", EN_ANY, kPlatformWindows },
	{ "a86f9c49355579c30d4a55b477c0d869", "baseball2001", EN_ANY, kPlatformUnknown },
	{ "a9543ef0d79bcb47cd76ec197ad0a967", "puttmoon", EN_ANY, kPlatformUnknown },
	{ "a9f2f04b1ecaab9495b59befffe9bf88", "pj3-demo", EN_ANY, kPlatformUnknown },
	{ "aa6a91b7f6f119d1b7b1f2a4c9e24d59", "moondemo", EN_ANY, kPlatformPC },
	{ "aa7a07d94ae853f6460be4ce0a1bf530", "monkeyega", FR_FRA, kPlatformPC },
	{ "aa8a0cb65f3afbbe2c14c3f9f92775a3", "monkey", FR_FRA, kPlatformPC },
	{ "aaa587701cde7e74692c68c1024b85eb", "racedemo", NL_NLD, kPlatformUnknown },
	{ "ab0693e9324cfcf498fdcbb12acf8bb4", "puttcircus", EN_ANY, kPlatformUnknown },
	{ "ac1642b6edfb8521ca03760126f1c250", "dottdemo", DE_DEU, kPlatformPC },
	{ "ac62d50e39492ee3738b4e83a5ac780f", "freddi2", NL_NLD, kPlatformWindows },
	{ "acad97ab1c6fc2a5b2d98abf6db4a190", "tentacle", EN_ANY, kPlatformUnknown },
	{ "ae94f110a14ce71fc515d5b648827a8f", "tentacle", ES_ESP, kPlatformPC },
	{ "b23f7cd7c304d7dff08e92a96120d5b4", "zak", EN_ANY, kPlatformPC },
	{ "b250d0f9cc83f80ced56fe11a4fb057c", "maniac", EN_ANY, kPlatformPC },
	{ "b289a2a8cbedbf45786e0b4ad2f510f1", "samnmax", IT_ITA, kPlatformPC },
	{ "b597e0403cc0002f69170e6caba7edd9", "indy3EGA", EN_ANY, kPlatformPC },
	{ "b628506f7def772e40de0aa5440fb8e1", "activity", EN_ANY, kPlatformWindows },
	{ "b886b0a5d909c7158a914e1d7c1c6c65", "loom", FR_FRA, kPlatformPC },
	{ "b8955d7d23b4972229060d1592489fef", "freddicove", NL_NLD, kPlatformWindows },
	{ "bd126753de619a495f9f22adc951c8d5", "monkey2", IT_ITA, kPlatformPC },
	{ "be39a5d4db60e8aa736b9086778cb45c", "spyozon", EN_ANY, kPlatformWindows },
	{ "bf8b52fdd9a69c67f34e8e9fec72661c", "farmdemo", EN_ANY, kPlatformWindows },
	{ "bfdf584b01503f0762baded581f6a0a2", "SoccerMLS", EN_ANY, kPlatformWindows },
	{ "c0039ad982999c92d0de81910d640fa0", "freddi", NL_NLD, kPlatformWindows },
	{ "c13225cb1bbd3bc9fe578301696d8021", "game", EN_ANY, kPlatformUnknown },
	{ "c24c490373aeb48fbd54caa8e7ae376d", "loom", DE_DEU, kPlatformAtariST },
	{ "c25755b08a8d0d47695e05f1e2111bfc", "f4-demo", EN_ANY, kPlatformWindows },
	{ "c30ef068add4277104243c31ce46c12b", "monkey2", FR_FRA, kPlatformAmiga },
	{ "c3196c5349e53e387aaff1533d95e53a", "snmdemo", EN_ANY, kPlatformPC },
	{ "c3b22fa4654bb580b20325ebf4174841", "puttzoo", NL_NLD, kPlatformWindows },
	{ "c3df37df9d3b481b45f75283a9907c47", "loom", IT_ITA, kPlatformPC },
	{ "c4787c3e8b5e2dfda90850ee800af00f ", "zak", FR_FRA, kPlatformPC },
	{ "c4ffae9fac495475d6bc3343ccc8faf9", "Soccer2004", EN_ANY, kPlatformUnknown },
	{ "c5d10e190d4b4d59114b824f2fdbd00e", "loomTowns", EN_ANY, kPlatformFMTowns },
	{ "c63ee46143ba65f9ce14cf539ca51bd7", "atlantis", EN_ANY, kPlatformPC },
	{ "c6907d44f1166941d982864cd42cdc89", "pajama2", DE_DEU, kPlatformWindows },
	{ "c7890e038806df2bb5c0c8c6f1986ea2", "monkeyvga", EN_ANY, kPlatformPC },
	{ "c7be10f775404fd9785a8b92a06d240c", "indy4", EN_ANY, kPlatformFMTowns },
	{ "c83079157ec765a28de445aec9768d60", "dottdemo", EN_ANY, kPlatformPC },
	{ "c8aac5e3e701874e2fa4117896f9e1b1", "freddemo", EN_ANY, kPlatformMacintosh },
	{ "cb1559e8405d17a5a278a6b5ad9338d1", "f3-mdemo", EN_ANY, kPlatformUnknown },
	{ "cc8ba2b0df2f9c450bcf055fe2711979", "snmdemo", DE_DEU, kPlatformPC },
	{ "cd9c05e755d7bf8e9b9590ad1ebe273e", "digdemo", EN_ANY, kPlatformMacintosh },
	{ "cdd760228cf1010c2903f37e788ea31c", "zak", DE_DEU, kPlatformPC },
	{ "ce3edc99cd4f478c5b37104d70c68ca5", "zakTowns", JA_JPN, kPlatformFMTowns },
	{ "ce6a4cef315b20fef58a95bc40a2d8d3", "monkeyega", FR_FRA, kPlatformPC },
	{ "ce7733f185b838e248927c7ba1a04204", "maniac", FR_FRA, kPlatformAmiga },
	{ "ce7fd0c382389a6791fc3e199c117ef4", "indy3EGA", ES_ESP, kPlatformPC },
	{ "cea91e3dd47f2518ea418e41611aa77f", "spyfox2", RU_RUS, kPlatformUnknown },
	{ "cf8d13446ec6cb6222287a925fd47c1d", "baseball", EN_ANY, kPlatformWindows },
	{ "cf90b4db5486ef798db78fe6fbf897e5", "pj3-demo", EN_ANY, kPlatformWindows },
	{ "d06fbe28818fef7bfc45c2cdf0c0849d", "zak", DE_DEU, kPlatformPC },
	{ "d0b531227a27c6662018d2bd05aac52a", "monkeyvga", DE_DEU, kPlatformPC },
	{ "d37c55388294b66e53e7ced3af88fa68", "ff2-demo", EN_ANY, kPlatformUnknown },
	{ "d43352a805d78b5f4936c6d7779bf575", "samnmax", RU_RUS, kPlatformPC },
	{ "d4b8ee426b1afd3e53bc0cf020418cf6", "dog", EN_ANY, kPlatformWindows },
	{ "d4cccb5af88f3e77f370896e9ba8c5f9", "freddi", RU_RUS, kPlatformWindows },
	{ "d4e79c3d8645b8266cd78c325bc35154", "pajama2", EN_ANY, kPlatformUnknown },
	{ "d62047a6729349ab36f7ee065bf26509", "dig", RU_RUS, kPlatformUnknown },
	{ "d62d248c3df6ec177405e2cb23d923b2", "indy3EGA", IT_ITA, kPlatformPC },
	{ "d6dd0646404768a63e963891a96daadd", "atlantis", EN_ANY, kPlatformMacintosh },
	{ "d7ab7cd6105546016e6a0d46fb36b964", "pjs-demo", EN_ANY, kPlatformUnknown },
	{ "d8323015ecb8b10bf53474f6e6b0ae33", "dig", UNK_LANG, kPlatformUnknown },
	{ "d8d07efcb88f396bee0b402b10c3b1c9", "maniac", EN_ANY, kPlatformNES },
	{ "d917f311a448e3cc7239c31bddb00dd2", "samnmax", EN_ANY, kPlatformUnknown },
	{ "d9d0dd93d16ab4dec55cabc2b86bbd17", "samdemo", EN_ANY, kPlatformPC },
	{ "da09e666fc8f5b78d7b0ac65d1a3b56e", "monkey2", EN_ANY, kPlatformFMTowns },
	{ "da6269b18fcb08189c0aa9c95533cce2", "monkey", IT_ITA, kPlatformPC },
	{ "da669b20271b85182e9c17a2a37ea02e", "monkey2", DE_DEU, kPlatformAmiga },
	{ "dd30a53035393baa5a5e222e716559af", "maniac", FR_FRA, kPlatformAtariST },
	{ "de4efb910210736813c9a1185384bace", "zoodemo", EN_ANY, kPlatformWindows },
	{ "debe337f73d660e951ece7c1f1c81add", "zak", EN_ANY, kPlatformPC },
	{ "defb8cb9ec4b0f91acfb6b61c6129ad9", "PuttTime", RU_RUS, kPlatformWindows },
	{ "df03ee021aa9b81d90cab9c26da07614", "indy3EGA", IT_ITA, kPlatformAmiga },
	{ "df047cc4792150f601290357566d36a6", "freddi", EN_ANY, kPlatformUnknown },
	{ "e01acc8c12ef44e8f778fe87e5f90f4e", "fbpack", EN_ANY, kPlatformUnknown },
	{ "e144f5f49d9241d2a9dee2576b3d09cb", "airdemo", EN_ANY, kPlatformWindows },
	{ "e17db1ddf91b39ca6bbc8ad3ed19e883", "monkey", JA_JPN, kPlatformFMTowns },
	{ "e246e02db9630533a40d99c9f54a8e01", "monkey2", EN_ANY, kPlatformMacintosh },
	{ "e361a7058ed8e8ebb462663c0a3ae8d6", "puttputt", HB_ISR, kPlatformPC },
	{ "e41de1c2a15abbcdbf9977e2d7e8a340", "freddi2", RU_RUS, kPlatformWindows },
	{ "e534d29afb3c6e0ee9dc3d53c5956714", "atlantis", DE_DEU, kPlatformAmiga },
	{ "e63a0b9249b5ca4cc4d3ac34305ae360", "freddi", NB_NOR, kPlatformWindows },
	{ "e689bdf67f98b1d760ce4487ec0e8d06", "indy3EGA", FR_FRA, kPlatformAmiga },
	{ "e6cd81b25ab1453a8a6d3482118c391e", "pass", EN_ANY, kPlatformPC },
	{ "e72bb4c2b613db2cf50f89ff6350e70a", "ft", ES_ESP, kPlatformUnknown },
	{ "e781230da44a44e2f0770edb2b3b3633", "maniac", EN_ANY, kPlatformAmiga },
	{ "e94c7cc3686fce406d3c91b5eae5a72d", "zak", EN_ANY, kPlatformAmiga },
	{ "eae95b2b3546d8ba86ae1d397c383253", "dog", EN_ANY, kPlatformUnknown },
	{ "ebd0b2c8a387f18887282afe6cad894a", "sf3-demo", EN_ANY, kPlatformUnknown },
	{ "ebd324dcf06a4c49e1ba5c231eee1060", "f4-demo", EN_ANY, kPlatformUnknown },
	{ "ed361270102e355afe5236954216aba2", "lost", EN_ANY, kPlatformUnknown },
	{ "edfdb24a499d92c59f824c52987c0eec", "atlantis", FR_FRA, kPlatformPC },
	{ "ef347474f3c7be3b29584eaa133cca05", "samnmax", FR_FRA, kPlatformPC },
	{ "ef74d9071d4e564b037cb44bd6774de7", "fbear", HB_ISR, kPlatformPC },
	{ "efe0a04a703e765ebebe92b6c8aa6b86", "baseball2003", EN_ANY, kPlatformWindows },
	{ "f049e38c1f8302b5db6170f1872af89a", "monkey", ES_ESP, kPlatformPC },
	{ "f06e66fd45b2f8b0f4a2833ff4476050", "fbpack", HB_ISR, kPlatformPC },
	{ "f08145577e4f13584cc90b3d6e9caa55", "pj3-demo", NL_NLD, kPlatformUnknown },
	{ "f237bf8a5ef9af78b2a6a4f3901da341", "pjs-demo", EN_ANY, kPlatformUnknown },
	{ "f27b1ba0eadaf2a6617b2b58192d1dbf", "samnmax", DE_DEU, kPlatformPC },
	{ "f3d55aea441e260e9e9c7d2a187097e0", "zoodemo", EN_ANY, kPlatformWindows },
	{ "f40a7f495f59188ca57a9d1d50301bb6", "puttdemo", EN_ANY, kPlatformMacintosh },
	{ "f7711f9264d4d43c2a1518ec7c10a607", "pajama3", EN_ANY, kPlatformWindows },
	{ "f79e60c17cca601e411f1f75e8ee9b5a", "spyfox2", EN_ANY, kPlatformUnknown },
	{ "fa127d7c4bb47d05bb1c33ddcaa9f767", "loom", DE_DEU, kPlatformPC },
	{ "fb66aa42de21675116346213f176a366", "monkeyvga", IT_ITA, kPlatformAmiga },
	{ "fbbbb38a81fc9d6a61d509278390a290", "farm", EN_ANY, kPlatformMacintosh },
	{ "fbdd947d21e8f5bac6d6f7a316af1c5a", "spydemo", EN_ANY, kPlatformUnknown },
	{ "fc53ce0e5f6562b1c1e1b4b8203acafb", "samnmax", ES_ESP, kPlatformPC },
	{ "fc6b6148e80d67939d9a18697c0f626a", "monkeyega", DE_DEU, kPlatformPC },
	{ "fc8d197a22146e74766e9cb0cfcaf1da", "ff2-demo", EN_ANY, kPlatformUnknown },
	{ "fcb78ebecab2757264c590890c319cc5", "PuttTime", NL_NLD, kPlatformWindows },
	{ "fe381e45117878b1e942cb876b050fd6", "ft", EN_ANY, kPlatformMacintosh },
	{ "fe60d6b5ff51b0553ac59963123b5777", "comi", UNK_LANG, kPlatformUnknown },
	{ 0, 0, UNK_LANG, kPlatformUnknown }
};
	
	writeRecord(md5table, sizeof(md5table), GBVARS_MD5TABLE_INDEX , GBVARS_SCUMM);
}

void addScummTables() {
	addScumm_mdtable();
}
