/* ScummVM - Scumm Interpreter
 * Copyright (C) 2001-2006 The ScummVM project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * $URL: https://svn.sourceforge.net/svnroot/scummvm/scummvm/trunk/engines/simon/simon.cpp $
 * $Id: simon.cpp 22078 2006-04-22 02:14:42Z kirben $
 *
 */

#include "common/stdafx.h"

#include "backends/fs/fs.h"


#include "common/config-manager.h"
#include "common/file.h"
#include "common/md5.h"
#include "common/system.h"

#include "gui/about.h"
#include "gui/message.h"

#include "simon/simon.h"
#include "simon/intern.h"
#include "simon/vga.h"
#include "simon/debugger.h"
#include "simon/debugger.h"

#include "sound/mididrv.h"
#ifdef _WIN32_WCE
extern bool isSmartphone(void);
#endif

#ifdef PALMOS_68K
#include "globals.h"
#endif

using Common::File;

namespace Simon {

#ifdef PALMOS_68K
#define PTR(a) a
static const GameSpecificSettings *simon1_settings;
static const GameSpecificSettings *simon2_settings;
static const GameSpecificSettings *feeblefiles_settings;
#else
#define PTR(a) &a
static const GameSpecificSettings simon1_settings = {
	"EFFECTS",                              // effects_filename
	"SIMON",                                // speech_filename
};

static const GameSpecificSettings simon2_settings = {
	"",                                     // effects_filename
	"SIMON2",                               // speech_filename
};

static const GameSpecificSettings feeblefiles_settings = {
	"",                                     // effects_filename
	"VOICES",                               // speech_filename
};
#endif

SimonEngine::SimonEngine(OSystem *syst)
	: Engine(syst), midi(syst) {
	_vcPtr = 0;
	_vc_get_out_of_code = 0;
	_gameOffsetsPtr = 0;

	_debugger = 0;
	setupVgaOpcodes();

	_keyPressed = 0;

	_gameFile = 0;

	_strippedTxtMem = 0;
	_textSize = 0;
	_stringTabNum = 0;
	_stringTabPos = 0;
	_stringtab_numalloc = 0;
	_stringTabPtr = 0;

	_itemArrayPtr = 0;
	_itemArraySize = 0;
	_itemArrayInited = 0;

	_itemHeapPtr = 0;
	_itemHeapCurPos = 0;
	_itemHeapSize = 0;

	_iconFilePtr = 0;

	_tblList = 0;

	_codePtr = 0;

	_localStringtable = 0;
	_stringIdLocalMin = 1;
	_stringIdLocalMax = 0;

	_tablesHeapPtr = 0;
	_tablesHeapPtrOrg = 0;
	_tablesheapPtrNew = 0;
	_tablesHeapSize = 0;
	_tablesHeapCurPos = 0;
	_tablesHeapCurPosOrg = 0;
	_tablesHeapCurPosNew = 0;

	_subroutineList = 0;
	_subroutineListOrg = 0;
	_subroutine = 0;

	_dxSurfacePitch = 0;

	_recursionDepth = 0;

	_lastVgaTick = 0;

	_marks = 0;

	_scriptVar2 = 0;
	_runScriptReturn1 = 0;
	_skipVgaWait = 0;
	_noParentNotify = 0;
	_beardLoaded = 0;
	_hitarea_unk_3 = 0;
	_mortalFlag = 0;
	_updateScreen = 0;
	_usePaletteDelay = 0;
	_syncFlag2 = 0;
	_inCallBack = 0;
	_cepeFlag = 0;
	_copyPartialMode = 0;
	_speed = 1;
	_fastMode = 0;
	_useBackGround = 0;

	_debugMode = 0;
	_pause = 0;
	_startMainScript = 0;
	_continousMainScript = 0;
	_startVgaScript = 0;
	_continousVgaScript = 0;
	_drawImagesDebug = 0;
	_dumpImages = 0;
	_speech = true;
	_subtitles = true;
	_fade = true;

	_animatePointer = 0;
	_mouseCursor = 0;
	_mouseAnim = 0;
	_mouseAnimMax = 0;
	_oldMouseCursor = 0;
	_oldMouseAnimMax = 0;

	_vgaVar9 = 0;
	_chanceModifier = 0;
	_restoreWindow6 = 0;
	_scrollX = 0;
	_scrollY = 0;
	_scrollXMax = 0;
	_scrollYMax = 0;
	_scrollCount = 0;
	_scrollFlag = 0;
	_scrollHeight = 0;
	_scrollWidth = 0;
	_scrollImage = 0;
	_boxStarHeight = 0;

	_scriptVerb = 0;
	_scriptNoun1 = 0;
	_scriptNoun2 = 0;
	_scriptAdj1 = 0;
	_scriptAdj2 = 0;

	_curWindow = 0;
	_textWindow = 0;

	_subjectItem = 0;
	_objectItem = 0;
	_currentPlayer = 0;

	_currentBoxNumber = 0;
	_iOverflow = 0;
	_hitAreaObjectItem = 0;
	_lastHitArea = 0;
	_lastNameOn = 0;
	_lastHitArea3 = 0;
	_hitAreaSubjectItem = 0;
	_currentVerbBox = 0;
	_lastVerbOn = 0;
	_needHitAreaRecalc = 0;
	_verbHitArea = 0;
	_defaultVerb = 0;
	_mouseHideCount = 0;

	_windowNum = 0;

	_printCharCurPos = 0;
	_printCharMaxPos = 0;
	_printCharPixelCount = 0;
	_numLettersToPrint = 0;

	_numTextBoxes = 0;

	_clockStopped = 0;
	_gameStoppedClock = 0;
	_lastTime = 0;

	_firstTimeStruct = 0;
	_pendingDeleteTimeEvent = 0;

	_mouseX = 0;
	_mouseY = 0;
	_mouseXOld = 0;
	_mouseYOld = 0;

	_leftButtonDown = 0;
	_rightButtonDown = 0;
	_noRightClick = false;

	_dummyItem1 = new Item();
	_dummyItem2 = new Item();
	_dummyItem3 = new Item();

	_lockWord = 0;
	_scrollUpHitArea = 0;
	_scrollDownHitArea = 0;

	_paletteColorCount = 0;

	_noOverWrite = 0xFFFF;
	_rejectCount = 0;
	_rejectBlock = false;

	_fastFadeOutFlag = 0;
	_unkPalFlag = 0;
	_exitCutscene = 0;
	_paletteFlag = 0;

	_soundFileId = 0;
	_lastMusicPlayed = -1;
	_nextMusicToPlay = -1;

	_showPreposition = 0;
	_showMessageFlag = 0;

	_videoNumPalColors = 0;

	_vgaSpriteChanged = 0;

	_vgaBufFreeStart = 0;
	_vgaBufEnd = 0;
	_vgaBufStart = 0;
	_vgaFileBufOrg = 0;
	_vgaFileBufOrg2 = 0;

	_curVgaFile1 = 0;
	_curVgaFile2 = 0;
	_curSfxFile = 0;

	_syncCount = 0;
	_timer5 = 0;
	_timer4 = 0;

	_frameRate = 1;

	_zoneNumber = 0;
	_vgaWaitFor = 0;
	_vgaCurZoneNum = 0;
	_vgaCurSpriteId = 0;
	_vgaCurSpritePriority = 0;

	_baseY = 0;
	_scale = 0;

	_feebleRect.left = 0;
	_feebleRect.right = 0;
	_feebleRect.top = 0;
	_feebleRect.bottom = 0;

	_scaleX = 0;
	_scaleY = 0;
	_scaleWidth = 0;
	_scaleHeight = 0;

	_nextVgaTimerToProcess = 0;

	memset(_objectArray, 0, sizeof(_objectArray));
	memset(_itemStore, 0, sizeof(_itemStore));

	memset(_stringIdArray2, 0, sizeof(_stringIdArray2));
	memset(_stringIdArray3, 0, sizeof(_stringIdArray3));
	memset(_speechIdArray4, 0, sizeof(_speechIdArray4));

	memset(_bitArray, 0, sizeof(_bitArray));
	memset(_bitArrayTwo, 0, sizeof(_bitArrayTwo));
	memset(_bitArrayThree, 0, sizeof(_bitArrayThree));

	memset(_variableArray, 0, sizeof(_variableArray));
	memset(_variableArray2, 0, sizeof(_variableArray2));
	_variableArrayPtr = 0;

	memset(_windowArray, 0, sizeof(_windowArray));

	memset(_fcsData1, 0, sizeof(_fcsData1));
	memset(_fcsData2, 0, sizeof(_fcsData2));

	_freeStringSlot = 0;

	memset(_stringReturnBuffer, 0, sizeof(_stringReturnBuffer));

	memset(_pathFindArray, 0, sizeof(_pathFindArray));

	memset(_pathValues, 0, sizeof(_pathValues));
	_PVCount = 0;
	_GPVCount = 0;

	memset(_pathValues1, 0, sizeof(_pathValues1));
	_PVCount1 = 0;
	_GPVCount1 = 0;

	memset(_paletteBackup, 0, sizeof(_paletteBackup));
	memset(_palette, 0, sizeof(_palette));

	memset(_videoBuf1, 0, sizeof(_videoBuf1));

	_windowList = new WindowBlock[16];

	memset(_lettersToPrintBuf, 0, sizeof(_lettersToPrintBuf));

	_numScreenUpdates = 0;
	_vgaTickCounter = 0;

	_moviePlay = 0;
	_sound = 0;

	_effectsPaused = false;
	_ambientPaused = false;
	_musicPaused = false;

	_dumpFile = 0;

	_saveLoadType = 0;
	_saveLoadSlot = 0;
	memset(_saveLoadName, 0, sizeof(_saveLoadName));

	_saveLoadRowCurPos = 0;
	_numSaveGameRows = 0;
	_saveDialogFlag = false;
	_saveOrLoad = false;
	_saveLoadFlag = false;

	_hyperLink = 0;
	_interactY = 0;
	_oracleMaxScrollY = 0;
	_noOracleScroll = 0;

	_sdlMouseX = 0;
	_sdlMouseY = 0;

	_backGroundBuf = 0;
	_frontBuf = 0;
	_backBuf = 0;
	_sdl_buf_scaled = 0;

	_vc10BasePtrOld = 0;
	memcpy (_hebrewCharWidths,
		"\x5\x5\x4\x6\x5\x3\x4\x5\x6\x3\x5\x5\x4\x6\x5\x3\x4\x6\x5\x6\x6\x6\x5\x5\x5\x6\x5\x6\x6\x6\x6\x6", 32);


	// Add default file directories for Acorn version
	File::addDefaultDirectory(_gameDataPath + "execute/");
	File::addDefaultDirectory(_gameDataPath + "EXECUTE/");
	

	// Add default file directories for Amiga & Macintosh
	// versions of The Feeble Files
	File::addDefaultDirectory(_gameDataPath + "gfx/");
	File::addDefaultDirectory(_gameDataPath + "GFX/");
	File::addDefaultDirectory(_gameDataPath + "movies/");
	File::addDefaultDirectory(_gameDataPath + "MOVIES/");
	File::addDefaultDirectory(_gameDataPath + "sfx/");
	File::addDefaultDirectory(_gameDataPath + "SFX/");
	File::addDefaultDirectory(_gameDataPath + "speech/");
	File::addDefaultDirectory(_gameDataPath + "SPEECH/");
}

int SimonEngine::init() {

	// Detect game
	if (!initGame()) {
		return -1;
	}

	setupOpcodes();

	if (getGameType() == GType_FF) {
		_screenWidth = 640;
		_screenHeight = 480;
	} else {
		_screenWidth = 320;
		_screenHeight = 200;
	}

	// Setup mixer
	if (!_mixer->isReady())
		warning("Sound initialization failed. "
						"Features of the game that depend on sound synchronization will most likely break");
	set_volume(ConfMan.getInt("sfx_volume"));
	_mixer->setVolumeForSoundType(Audio::Mixer::kMusicSoundType, ConfMan.getInt("music_volume"));

	_system->beginGFXTransaction();
		initCommonGFX(getGameType() == GType_FF);
		_system->initSize(_screenWidth, _screenHeight);
	_system->endGFXTransaction();

	// Setup midi driver
	MidiDriver *driver = 0;
	if (getPlatform() == Common::kPlatformAmiga) {
		driver = MidiDriver::createMidi(MD_NULL);	// Create fake MIDI driver for Simon1Amiga and Simon2CD32 for now
		_native_mt32 = false;
	} else {
		int midiDriver = MidiDriver::detectMusicDriver(MDT_ADLIB | MDT_MIDI);
		_native_mt32 = ((midiDriver == MD_MT32) || ConfMan.getBool("native_mt32"));
		driver = MidiDriver::createMidi(midiDriver);
		if (_native_mt32) {
			driver->property(MidiDriver::PROP_CHANNEL_MASK, 0x03FE);
		}
	}

	midi.mapMT32toGM (getGameType() == GType_SIMON1 && !_native_mt32);

	midi.set_driver(driver);
	int ret = midi.open();
	if (ret)
		warning ("MIDI Player init failed: \"%s\"", midi.getErrorName (ret));
	midi.set_volume(ConfMan.getInt("music_volume"));

	_debugMode = (gDebugLevel >= 0);
	_language = Common::parseLanguage(ConfMan.get("language"));

	if (ConfMan.hasKey("music_mute") && ConfMan.getBool("music_mute") == 1)
		midi.pause(_musicPaused ^= 1);

	if (getFeatures() & GF_TALKIE) {
		_speech = !ConfMan.getBool("speech_mute");
		_subtitles = ConfMan.getBool("subtitles");

		if (getGameType() == GType_SIMON1) {
			// English and German versions don't have full subtitles
			 if (_language == Common::EN_ANY || _language == Common::DE_DEU)
				_subtitles = false;
			// Other versions require speech to be enabled
			else
				_speech = true;
		}

		// Default to speech only, if both speech and subtitles disabled
		if (!_speech && !_subtitles)
			_speech = true;
	} else {
		_speech = false;
		_subtitles = true;
	}

	if (ConfMan.hasKey("fade") && ConfMan.getBool("fade") == 0)
		_fade = 0;

	if (ConfMan.hasKey("slow_down") && ConfMan.getInt("slow_down") >= 1)
		_speed = ConfMan.getInt("slow_down");

	// FIXME Use auto dirty rects cleanup code to reduce CPU usage
	g_system->setFeatureState(OSystem::kFeatureAutoComputeDirtyRects, true);

	if (getGameType() == GType_FF) {
		NUM_VIDEO_OP_CODES = 85;
#ifndef PALMOS_68K
		VGA_MEM_SIZE = 7500000;
#else
		VGA_MEM_SIZE = gVars->memory[kMemSimon2Games];
#endif
		TABLES_MEM_SIZE = 200000;
		VGA_DELAY_BASE = 5;
	} else if (getGameType() == GType_SIMON2) {
		TABLE_INDEX_BASE = 1580 / 4;
		TEXT_INDEX_BASE = 1500 / 4;
		NUM_VIDEO_OP_CODES = 75;
#ifndef PALMOS_68K
		VGA_MEM_SIZE = 2000000;
#else
		VGA_MEM_SIZE = gVars->memory[kMemSimon2Games];
#endif
		TABLES_MEM_SIZE = 100000;
		// Check whether to use MT-32 MIDI tracks in Simon the Sorcerer 2
		if ((getGameType() == GType_SIMON2) && _native_mt32)
			MUSIC_INDEX_BASE = (1128 + 612) / 4;
		else
			MUSIC_INDEX_BASE = 1128 / 4;
		SOUND_INDEX_BASE = 1660 / 4;
		VGA_DELAY_BASE = 1;
	} else {
		TABLE_INDEX_BASE = 1576 / 4;
		TEXT_INDEX_BASE = 1460 / 4;
		NUM_VIDEO_OP_CODES = 64;
#ifndef PALMOS_68K
		VGA_MEM_SIZE = 1000000;
#else
		VGA_MEM_SIZE = gVars->memory[kMemSimon1Games];
#endif
		TABLES_MEM_SIZE = 50000;
		MUSIC_INDEX_BASE = 1316 / 4;
		SOUND_INDEX_BASE = 0;
		VGA_DELAY_BASE = 1;
	}

	if (getGameType() == GType_FF) {
		gss = PTR(feeblefiles_settings);
	} else if (getGameType() == GType_SIMON2) {
		if (getFeatures() & GF_TALKIE) {
			// Add default file directories
			File::addDefaultDirectory(_gameDataPath + "voices/");
			File::addDefaultDirectory(_gameDataPath + "VOICES/");
		}
		gss = PTR(simon2_settings);
	} else if (getGameType() == GType_SIMON1) {
		gss = PTR(simon1_settings);
	}

	if ((getGameType() == GType_SIMON1) && (getFeatures() & GF_TALKIE)) {
		// Add default file directories
		switch (_language) {
		case Common::HB_ISR:
			File::addDefaultDirectory(_gameDataPath + "hebrew/");
			File::addDefaultDirectory(_gameDataPath + "HEBREW/");
			break;
		case  Common::ES_ESP:
			File::addDefaultDirectory(_gameDataPath + "spanish/");
			File::addDefaultDirectory(_gameDataPath + "SPANISH/");
			break;
		case  Common::IT_ITA:
			File::addDefaultDirectory(_gameDataPath + "italian/");
			File::addDefaultDirectory(_gameDataPath + "ITALIAN/");
			break;
		case  Common::FR_FRA:
			File::addDefaultDirectory(_gameDataPath + "french/");
			File::addDefaultDirectory(_gameDataPath + "FRENCH/");
			break;
		}
	}

	return 0;
}

SimonEngine::~SimonEngine() {
	delete _gameFile;

	midi.close();

	free(_stringTabPtr);
	free(_itemArrayPtr);
	free(_itemHeapPtr - _itemHeapCurPos);
	free(_tablesHeapPtr - _tablesHeapCurPos);
	free(_tblList);
	free(_iconFilePtr);
	free(_gameOffsetsPtr);

	delete _dummyItem1;
	delete _dummyItem2;
	delete _dummyItem3;

	delete [] _windowList;

	delete _debugger;
	delete _moviePlay;
	delete _sound;
}

void SimonEngine::errorString(const char *buf1, char *buf2) {
	strcpy(buf2, buf1);

#ifdef _WIN32_WCE
	if (isSmartphone())
		return;
#endif

	// Unless an error -originated- within the debugger, spawn the
	// debugger. Otherwise exit out normally.
	if (_debugger && !_debugger->isAttached()) {
		// (Print it again in case debugger segfaults)
		printf("%s\n", buf2);
		_debugger->attach(buf2);
		_debugger->onFrame();
	}
}

void SimonEngine::palette_fadeout(uint32 *pal_values, uint num) {
	byte *p = (byte *)pal_values;

	do {
		if (p[0] >= 8)
			p[0] -= 8;
		else
			p[0] = 0;
		if (p[1] >= 8)
			p[1] -= 8;
		else
			p[1] = 0;
		if (p[2] >= 8)
			p[2] -= 8;
		else
			p[2] = 0;
		p += 4;
	} while (--num);
}

byte *SimonEngine::allocateItem(uint size) {
	byte *org = _itemHeapPtr;
	size = (size + 3) & ~3;

	_itemHeapPtr += size;
	_itemHeapCurPos += size;

	if (_itemHeapCurPos > _itemHeapSize)
		error("Itemheap overflow");

	return org;
}

void SimonEngine::alignTableMem() {
	if ((unsigned long)_tablesHeapPtr & 3) {
		_tablesHeapPtr += 2;
		_tablesHeapCurPos += 2;
	}
}

byte *SimonEngine::allocateTable(uint size) {
	byte *org = _tablesHeapPtr;

	size = (size + 1) & ~1;

	_tablesHeapPtr += size;
	_tablesHeapCurPos += size;

	if (_tablesHeapCurPos > _tablesHeapSize)
		error("Tablesheap overflow");

	return org;
}

int SimonEngine::allocGamePcVars(File *in) {
	uint item_array_size, item_array_inited, stringtable_num;
	uint32 version;
	uint i;

	item_array_size = in->readUint32BE();
	version = in->readUint32BE();
	item_array_inited = in->readUint32BE();
	stringtable_num = in->readUint32BE();

	item_array_inited += 2;				// first two items are predefined
	item_array_size += 2;

	if (version != 0x80)
		error("Not a runtime database");

	_itemArrayPtr = (Item **)calloc(item_array_size, sizeof(Item *));
	if (_itemArrayPtr == NULL)
		error("Out of memory for Item array");

	_itemArraySize = item_array_size;
	_itemArrayInited = item_array_inited;

	for (i = 1; i < item_array_inited; i++) {
		_itemArrayPtr[i] = (Item *)allocateItem(sizeof(Item));
	}

	// The rest is cleared automatically by calloc
	allocateStringTable(stringtable_num + 10);
	_stringTabNum = stringtable_num;

	return item_array_inited;
}

void SimonEngine::setUserFlag(Item *item, int a, int b) {
	SubUserFlag *subUserFlag;

	subUserFlag = (SubUserFlag *) findChildOfType(item, 9);
	if (subUserFlag == NULL) {
		subUserFlag = (SubUserFlag *) allocateChildBlock(item, 9, sizeof(SubUserFlag));
	}

	if (a >= 0 && a <= 3)
		subUserFlag->userFlags[a] = b;
}

void SimonEngine::createPlayer() {
	Child *child;

	_currentPlayer = _itemArrayPtr[1];
	_currentPlayer->adjective = -1;
	_currentPlayer->noun = 10000;

	child = (Child *)allocateChildBlock(_currentPlayer, 3, sizeof(Child));
	if (child == NULL)
		error("player create failure");

	setUserFlag(_currentPlayer, 0, 0);
}

void SimonEngine::allocateStringTable(int num) {
	_stringTabPtr = (byte **)calloc(num, sizeof(byte *));
	_stringTabPos = 0;
	_stringtab_numalloc = num;
}

void SimonEngine::setupStringTable(byte *mem, int num) {
	int i = 0;
	for (;;) {
		_stringTabPtr[i++] = mem;
		if (--num == 0)
			break;
		for (; *mem; mem++);
		mem++;
	}

	_stringTabPos = i;
}

void SimonEngine::setupLocalStringTable(byte *mem, int num) {
	int i = 0;
	for (;;) {
		_localStringtable[i++] = mem;
		if (--num == 0)
			break;
		for (; *mem; mem++);
		mem++;
	}
}

void SimonEngine::readSubroutineLine(File *in, SubroutineLine *sl, Subroutine *sub) {
	byte line_buffer[1024], *q = line_buffer;
	int size;

	if (sub->id == 0) {
		sl->verb = in->readUint16BE();
		sl->noun1 = in->readUint16BE();
		sl->noun2 = in->readUint16BE();
	}

	while ((*q = in->readByte()) != 0xFF) {
		if (*q == 87) {
			in->readUint16BE();
		} else {
			q = readSingleOpcode(in, q);
		}
	}

	size = q - line_buffer + 1;

	memcpy(allocateTable(size), line_buffer, size);
}

SubroutineLine *SimonEngine::createSubroutineLine(Subroutine *sub, int where) {
	SubroutineLine *sl, *cur_sl = NULL, *last_sl = NULL;

	if (sub->id == 0)
		sl = (SubroutineLine *)allocateTable(SUBROUTINE_LINE_BIG_SIZE);
	else
		sl = (SubroutineLine *)allocateTable(SUBROUTINE_LINE_SMALL_SIZE);

	// where is what offset to insert the line at, locate the proper beginning line
	if (sub->first != 0) {
		cur_sl = (SubroutineLine *)((byte *)sub + sub->first);
		while (where) {
			last_sl = cur_sl;
			cur_sl = (SubroutineLine *)((byte *)sub + cur_sl->next);
			if ((byte *)cur_sl == (byte *)sub)
				break;
			where--;
		}
	}

	if (last_sl != NULL) {
		// Insert the subroutine line in the middle of the link
		last_sl->next = (byte *)sl - (byte *)sub;
		sl->next = (byte *)cur_sl - (byte *)sub;
	} else {
		// Insert the subroutine line at the head of the link
		sl->next = sub->first;
		sub->first = (byte *)sl - (byte *)sub;
	}

	return sl;
}

void SimonEngine::readSubroutine(File *in, Subroutine *sub) {
	while (in->readUint16BE() == 0) {
		readSubroutineLine(in, createSubroutineLine(sub, 0xFFFF), sub);
	}
}

Subroutine *SimonEngine::createSubroutine(uint id) {
	Subroutine *sub;

	alignTableMem();

	sub = (Subroutine *)allocateTable(sizeof(Subroutine));
	sub->id = id;
	sub->first = 0;
	sub->next = _subroutineList;
	_subroutineList = sub;
	return sub;
}

void SimonEngine::readSubroutineBlock(File *in) {
	while (in->readUint16BE() == 0) {
		readSubroutine(in, createSubroutine(in->readUint16BE()));
	}
}

Child *SimonEngine::findChildOfType(Item *i, uint type) {
	Child *child = i->children;
	for (; child; child = child->next)
		if (child->type == type)
			return child;
	return NULL;
}

bool SimonEngine::isRoom(Item *item) {
	return findChildOfType(item, 1) != NULL;
}

bool SimonEngine::isObject(Item *item) {
	return findChildOfType(item, 2) != NULL;
}

uint SimonEngine::getOffsetOfChild2Param(SubObject *child, uint prop) {
	uint m = 1;
	uint offset = 0;
	while (m != prop) {
		if (child->objectFlags & m)
			offset++;
		m *= 2;
	}
	return offset;
}

Child *SimonEngine::allocateChildBlock(Item *i, uint type, uint size) {
	Child *child = (Child *)allocateItem(size);
	child->next = i->children;
	i->children = child;
	child->type = type;
	return child;
}

void SimonEngine::allocItemHeap() {
	_itemHeapSize = 10000;
	_itemHeapCurPos = 0;
	_itemHeapPtr = (byte *)calloc(10000, 1);
}

void SimonEngine::allocTablesHeap() {
	_tablesHeapSize = TABLES_MEM_SIZE;
	_tablesHeapCurPos = 0;
	_tablesHeapPtr = (byte *)calloc(TABLES_MEM_SIZE, 1);
}

void SimonEngine::setItemState(Item *item, int value) {
	item->state = value;
}

int SimonEngine::getNextWord() {
	int16 a = (int16)READ_BE_UINT16(_codePtr);
	_codePtr += 2;
	return a;
}

uint SimonEngine::getNextStringID() {
	return (uint16)getNextWord();
}

uint SimonEngine::getVarOrByte() {
	uint a = *_codePtr++;
	if (a != 255)
		return a;
	return readVariable(*_codePtr++);
}

uint SimonEngine::getVarOrWord() {
	uint a = READ_BE_UINT16(_codePtr);
	_codePtr += 2;
	if (a >= 30000 && a < 30512) {
		return readVariable(a - 30000);
	}
	return a;
}

Item *SimonEngine::getNextItemPtr() {
	int a = getNextWord();
	switch (a) {
	case -1:
		return _subjectItem;
	case -3:
		return _objectItem;
	case -5:
		return me();
	case -7:
		return actor();
	case -9:
		return derefItem(me()->parent);
	default:
		return derefItem(a);
	}
}

Item *SimonEngine::getNextItemPtrStrange() {
	int a = getNextWord();
	switch (a) {
	case -1:
		return _subjectItem;
	case -3:
		return _objectItem;
	case -5:
		return _dummyItem2;
	case -7:
		return NULL;
	case -9:
		return _dummyItem3;
	default:
		return derefItem(a);
	}
}

uint SimonEngine::getNextItemID() {
	int a = getNextWord();
	switch (a) {
	case -1:
		return itemPtrToID(_subjectItem);
	case -3:
		return itemPtrToID(_objectItem);
	case -5:
		return getItem1ID();
	case -7:
		return 0;
	case -9:
		return me()->parent;
	default:
		return a;
	}
}

Item *SimonEngine::me() {
	if (_currentPlayer)
		return _currentPlayer;
	return _dummyItem1;
}

Item *SimonEngine::actor() {
	error("actor: is this code ever used?");
	//if (_actorPlayer)
	//	return _actorPlayer;
	return _dummyItem1;
}

uint SimonEngine::getNextVarContents() {
	return (uint16)readVariable(getVarOrByte());
}

uint SimonEngine::readVariable(uint variable) {
	if (variable >= 255)
		error("Variable %d out of range in read", variable);

	if (getGameType() == GType_FF) {
		if (getBitFlag(83))
			return (uint16)_variableArray2[variable];
		else
			return (uint16)_variableArray[variable];
	} else {
			return _variableArray[variable];
	}
}

void SimonEngine::writeNextVarContents(uint16 contents) {
	writeVariable(getVarOrByte(), contents);
}

void SimonEngine::writeVariable(uint variable, uint16 contents) {
	if (variable >= 256)
		error("Variable %d out of range in write", variable);

	if (getGameType() == GType_FF && getBitFlag(83))
		_variableArray2[variable] = contents;
	else
		_variableArray[variable] = contents;
}

void SimonEngine::setItemParent(Item *item, Item *parent) {
	Item *old_parent = derefItem(item->parent);

	if (item == parent)
		error("Trying to set item as its own parent");

	// unlink it if it has a parent
	if (old_parent)
		unlinkItem(item);
	itemChildrenChanged(old_parent);
	linkItem(item, parent);
	itemChildrenChanged(parent);
}

void SimonEngine::itemChildrenChanged(Item *item) {
	int i;
	WindowBlock *window;

	if (_noParentNotify)
		return;

	mouseOff();

	for (i = 0; i != 8; i++) {
		window = _windowArray[i];
		if (window && window->iconPtr && window->iconPtr->itemRef == item) {
			if (_fcsData1[i]) {
				_fcsData2[i] = true;
			} else {
				_fcsData2[i] = false;
				drawIconArray(i, item, window->iconPtr->line, window->iconPtr->classMask);
			}
		}
	}

	mouseOn();
}

void SimonEngine::unlinkItem(Item *item) {
	Item *first, *parent, *next;

	// can't unlink item without parent
	if (item->parent == 0)
		return;

	// get parent and first child of parent
	parent = derefItem(item->parent);
	first = derefItem(parent->child);

	// the node to remove is first in the parent's children?
	if (first == item) {
		parent->child = item->sibling;
		item->parent = 0;
		item->sibling = 0;
		return;
	}

	for (;;) {
		if (!first)
			error("unlinkItem: parent empty");
		if (first->sibling == 0)
			error("unlinkItem: parent does not contain child");

		next = derefItem(first->sibling);
		if (next == item) {
			first->sibling = next->sibling;
			item->parent = 0;
			item->sibling = 0;
			return;
		}
		first = next;
	}
}

void SimonEngine::linkItem(Item *item, Item *parent) {
	uint id;
	// Don't allow that an item that is already linked is relinked
	if (item->parent)
		return;

	id = itemPtrToID(parent);
	item->parent = id;

	if (parent != 0) {
		item->sibling = parent->child;
		parent->child = itemPtrToID(item);
	} else {
		item->sibling = 0;
	}
}

const byte *SimonEngine::getStringPtrByID(uint stringId) {
	const byte *string_ptr;
	byte *dst;

	_freeStringSlot ^= 1;

	if (stringId < 0x8000) {
		string_ptr = _stringTabPtr[stringId];
	} else {
		string_ptr = getLocalStringByID(stringId);
	}

	dst = _stringReturnBuffer[_freeStringSlot];
	strcpy((char *)dst, (const char *)string_ptr);
	return dst;
}

const byte *SimonEngine::getLocalStringByID(uint stringId) {
	if (stringId < _stringIdLocalMin || stringId >= _stringIdLocalMax) {
		loadTextIntoMem(stringId);
	}
	return _localStringtable[stringId - _stringIdLocalMin];
}

void SimonEngine::loadTextIntoMem(uint stringId) {
	byte *p;
	char filename[30];
	int i;
	uint base_min = 0x8000, base_max, size;

	_tablesHeapPtr = _tablesheapPtrNew;
	_tablesHeapCurPos = _tablesHeapCurPosNew;

	p = _strippedTxtMem;

	// get filename
	while (*p) {
		for (i = 0; *p; p++, i++)
			filename[i] = *p;
		filename[i] = 0;
		p++;

		base_max = (p[0] * 256) | p[1];
		p += 2;

		if (stringId < base_max) {
			_stringIdLocalMin = base_min;
			_stringIdLocalMax = base_max;

			_localStringtable = (byte **)_tablesHeapPtr;

			size = (base_max - base_min + 1) * sizeof(byte *);
			_tablesHeapPtr += size;
			_tablesHeapCurPos += size;

			size = loadTextFile(filename, _tablesHeapPtr);

			setupLocalStringTable(_tablesHeapPtr, base_max - base_min + 1);

			_tablesHeapPtr += size;
			_tablesHeapCurPos += size;

			if (_tablesHeapCurPos > _tablesHeapSize) {
				error("loadTextIntoMem: Out of table memory");
			}
			return;
		}

		base_min = base_max;
	}

	error("loadTextIntoMem: didn't find %d", stringId);
}

void SimonEngine::loadTablesIntoMem(uint subr_id) {
	byte *p;
	int i;
	uint min_num, max_num;
	char filename[30];
	File *in;

	p = _tblList;
	if (p == NULL)
		return;

	while (*p) {
		for (i = 0; *p; p++, i++)
			filename[i] = *p;
		filename[i] = 0;
		p++;

		for (;;) {
			min_num = (p[0] * 256) | p[1];
			p += 2;

			if (min_num == 0)
				break;

			max_num = (p[0] * 256) | p[1];
			p += 2;

			if (subr_id >= min_num && subr_id <= max_num) {
				_subroutineList = _subroutineListOrg;
				_tablesHeapPtr = _tablesHeapPtrOrg;
				_tablesHeapCurPos = _tablesHeapCurPosOrg;
				_stringIdLocalMin = 1;
				_stringIdLocalMax = 0;

				in = openTablesFile(filename);
				readSubroutineBlock(in);
				closeTablesFile(in);
				if (getGameType() == GType_FF) {
					// TODO
				} else if (getGameType() == GType_SIMON2) {
					_sound->loadSfxTable(_gameFile, _gameOffsetsPtr[atoi(filename + 6) - 1 + SOUND_INDEX_BASE]);
				} else if (getPlatform() == Common::kPlatformWindows) {
					memcpy(filename, "SFXXXX", 6);
					_sound->readSfxFile(filename);
				}

				alignTableMem();

				_tablesheapPtrNew = _tablesHeapPtr;
				_tablesHeapCurPosNew = _tablesHeapCurPos;

				if (_tablesHeapCurPos > _tablesHeapSize)
					error("loadTablesIntoMem: Out of table memory");
				return;
			}
		}
	}

	debug(1,"loadTablesIntoMem: didn't find %d", subr_id);
}

void SimonEngine::playSting(uint a) {
	if (!midi._enable_sfx)
		return;

	char filename[15];

	File mus_file;
	uint16 mus_offset;

	sprintf(filename, "STINGS%i.MUS", _soundFileId);
	mus_file.open(filename);
	if (!mus_file.isOpen()) {
		warning("Can't load sound effect from '%s'", filename);
		return;
	}

	mus_file.seek(a * 2, SEEK_SET);
	mus_offset = mus_file.readUint16LE();
	if (mus_file.ioFailed())
		error("Can't read sting %d offset", a);

	mus_file.seek(mus_offset, SEEK_SET);
	midi.loadSMF(&mus_file, a, true);
	midi.startTrack(0);
}

Subroutine *SimonEngine::getSubroutineByID(uint subroutine_id) {
	Subroutine *cur;

	_subroutine = subroutine_id;

	for (cur = _subroutineList; cur; cur = cur->next) {
		if (cur->id == subroutine_id)
			return cur;
	}

	loadTablesIntoMem(subroutine_id);

	for (cur = _subroutineList; cur; cur = cur->next) {
		if (cur->id == subroutine_id)
			return cur;
	}

	if (subroutine_id != 160)
		debug(0,"getSubroutineByID: subroutine %d not found", subroutine_id);
	return NULL;
}

uint SimonEngine::loadTextFile_gme(const char *filename, byte *dst) {
	uint res;
	uint32 offs;
	uint32 size;

	res = atoi(filename + 4) + TEXT_INDEX_BASE - 1;
	offs = _gameOffsetsPtr[res];
	size = _gameOffsetsPtr[res + 1] - offs;

	readGameFile(dst, offs, size);

	return size;
}

File *SimonEngine::openTablesFile_gme(const char *filename) {
	uint res;
	uint32 offs;

	res = atoi(filename + 6) + TABLE_INDEX_BASE - 1;
	offs = _gameOffsetsPtr[res];

	_gameFile->seek(offs, SEEK_SET);
	return _gameFile;
}

uint SimonEngine::loadTextFile_simon1(const char *filename, byte *dst) {
	File fo;
	fo.open(filename);
	uint32 size;

	if (fo.isOpen() == false)
		error("loadTextFile: Can't open '%s'", filename);

	size = fo.size();

	if (fo.read(dst, size) != size)
		error("loadTextFile: fread failed");
	fo.close();

	return size;
}

File *SimonEngine::openTablesFile_simon1(const char *filename) {
	File *fo = new File();
	fo->open(filename);
	if (fo->isOpen() == false)
		error("openTablesFile: Can't open '%s'", filename);
	return fo;
}

uint SimonEngine::loadTextFile(const char *filename, byte *dst) {
	if (getFeatures() & GF_OLD_BUNDLE)
		return loadTextFile_simon1(filename, dst);
	else
		return loadTextFile_gme(filename, dst);
}

File *SimonEngine::openTablesFile(const char *filename) {
	if (getFeatures() & GF_OLD_BUNDLE)
		return openTablesFile_simon1(filename);
	else
		return openTablesFile_gme(filename);
}

void SimonEngine::closeTablesFile(File *in) {
	if (getFeatures() & GF_OLD_BUNDLE) {
		in->close();
		delete in;
	}
}

void SimonEngine::addTimeEvent(uint timeout, uint subroutine_id) {
	TimeEvent *te = (TimeEvent *)malloc(sizeof(TimeEvent)), *first, *last = NULL;
	time_t cur_time;

	time(&cur_time);

	te->time = cur_time + timeout - _gameStoppedClock;
	if (_clockStopped)
		te->time -= ((uint32)time(NULL) - _clockStopped);
	te->subroutine_id = subroutine_id;

	first = _firstTimeStruct;
	while (first) {
		if (te->time <= first->time) {
			if (last) {
				last->next = te;
				te->next = first;
				return;
			}
			te->next = _firstTimeStruct;
			_firstTimeStruct = te;
			return;
		}

		last = first;
		first = first->next;
	}

	if (last) {
		last->next = te;
		te->next = NULL;
	} else {
		_firstTimeStruct = te;
		te->next = NULL;
	}
}

void SimonEngine::delTimeEvent(TimeEvent *te) {
	TimeEvent *cur;

	if (te == _pendingDeleteTimeEvent)
		_pendingDeleteTimeEvent = NULL;

	if (te == _firstTimeStruct) {
		_firstTimeStruct = te->next;
		free(te);
		return;
	}

	cur = _firstTimeStruct;
	if (cur == NULL)
		error("delTimeEvent: none available");

	for (;;) {
		if (cur->next == NULL)
			error("delTimeEvent: no such te");
		if (te == cur->next) {
			cur->next = te->next;
			free(te);
			return;
		}
		cur = cur->next;
	}
}

void SimonEngine::killAllTimers() {
	TimeEvent *cur, *next;

	for (cur = _firstTimeStruct; cur; cur = next) {
		next = cur->next;
		delTimeEvent(cur);
	}
}

bool SimonEngine::kickoffTimeEvents() {
	time_t cur_time;
	TimeEvent *te;
	bool result = false;

	if (_clockStopped)
		return result;

	time(&cur_time);
	cur_time -= _gameStoppedClock;

	while ((te = _firstTimeStruct) != NULL && te->time <= (uint32)cur_time) {
		result = true;
		_pendingDeleteTimeEvent = te;
		invokeTimeEvent(te);
		if (_pendingDeleteTimeEvent) {
			_pendingDeleteTimeEvent = NULL;
			delTimeEvent(te);
		}
	}

	return result;
}

void SimonEngine::invokeTimeEvent(TimeEvent *te) {
	Subroutine *sub;

	_scriptVerb = 0;

	if (_runScriptReturn1)
		return;

	sub = getSubroutineByID(te->subroutine_id);
	if (sub != NULL)
		startSubroutineEx(sub);

	_runScriptReturn1 = false;
}

void SimonEngine::setup_cond_c_helper() {
	HitArea *last;
	uint id;

	_noRightClick = 1;

	if (getGameType() == GType_FF) {
		int cursor = 5;
		int animMax = 16;

		if (getBitFlag(200)) {
			cursor = 11;
			animMax = 5;
		} else if (getBitFlag(201)) {
			cursor = 12;
			animMax = 5;
		} else if (getBitFlag(202)) {
			cursor = 13;
			animMax = 5;
		} else if (getBitFlag(203)) {
			cursor = 14;
			animMax = 9;
		} else if (getBitFlag(205)) {
			cursor = 17;
			animMax = 11;
		} else if (getBitFlag(206)) {
			cursor = 16;
			animMax = 2;
		} else if (getBitFlag(208)) {
			cursor = 26;
			animMax = 2;
		} else if (getBitFlag(209)) {
			cursor = 27;
			animMax = 9;
		} else if (getBitFlag(210)) {
			cursor = 28;
			animMax = 9;
		}

		_animatePointer = 0;
		_mouseCursor = cursor;
		_mouseAnimMax = animMax;
		_mouseAnim = 1;
		_needHitAreaRecalc++;
	}

	if (getGameType() == GType_SIMON2) {
		_mouseCursor = 0;
		if (_defaultVerb != 999) {
			_mouseCursor = 9;
			_needHitAreaRecalc++;
			_defaultVerb = 0;
		}
	}

	_lastHitArea = 0;
	_hitAreaObjectItem = NULL;

	last = _lastNameOn;
	clearName();
	_lastNameOn = last;

	for (;;) {
		_lastHitArea = NULL;
		_lastHitArea3 = 0;
		_leftButtonDown = 0;

		do {
			if (_exitCutscene && getBitFlag(9)) {
				endCutscene();
				goto out_of_here;
			}

			if (getGameType() == GType_FF) {
				if (_variableArray[254] == 63) {
					hitarea_stuff_helper();
				} else if (_variableArray[254] == 75) {
					hitarea_stuff_helper();
					_variableArray[60] = 9999;
					goto out_of_here;
				}
			}

			delay(100);
		} while (_lastHitArea3 == (HitArea *) -1 || _lastHitArea3 == 0);

		if (_lastHitArea == NULL) {
		} else if (_lastHitArea->id == 0x7FFB) {
			inventoryUp(_lastHitArea->window);
		} else if (_lastHitArea->id == 0x7FFC) {
			inventoryDown(_lastHitArea->window);
		} else if (_lastHitArea->item_ptr != NULL) {
			_hitAreaObjectItem = _lastHitArea->item_ptr;
			id = 0xFFFF;
			if (_lastHitArea->flags & kBFTextBox) {
				if (getGameType() == GType_FF && (_lastHitArea->flags & kBFHyperBox))
					id = _lastHitArea->data;
				else
					id = _lastHitArea->flags / 256;
			}
			_variableArray[60] = id;
			break;
		}
	}

out_of_here:
	_lastHitArea3 = 0;
	_lastHitArea = 0;
	_lastNameOn = NULL;
	_mouseCursor = 0;
	_noRightClick = 0;
}

void SimonEngine::endCutscene() {
	Subroutine *sub;

	_sound->stopVoice();

	sub = getSubroutineByID(170);
	if (sub != NULL)
		startSubroutineEx(sub);

	_runScriptReturn1 = true;
}

uint SimonEngine::getWindowNum(WindowBlock *window) {
	uint i;

	for (i = 0; i != ARRAYSIZE(_windowArray); i++)
		if (_windowArray[i] == window)
			return i;

	error("getWindowNum: not found");
	return 0;
}

void SimonEngine::mouseOff() {
	_mouseHideCount++;
}

void SimonEngine::mouseOn() {
	_lockWord |= 1;

	if (_mouseHideCount != 0)
		_mouseHideCount--;

	_lockWord &= ~1;
}

void SimonEngine::handleMouseMoved() {
	uint x;

	if (_mouseHideCount) {
		_system->showMouse(false);
		return;
	}

	_system->showMouse(true);
	pollMouseXY();

	if (_mouseX <= 0)
		_mouseX = 0;
	if (_mouseX >= _screenWidth - 1)
		_mouseX = _screenWidth - 1;

	if (_mouseY <= 0)
		_mouseY = 0;
	if (_mouseY >= _screenHeight - 1)
		_mouseY = _screenHeight - 1;

	if (_defaultVerb) {
		uint id = 101;
		if (_mouseY >= 136)
			id = 102;
		if (_defaultVerb != id)
			resetVerbs();
	}

	if (getGameType() == GType_FF) {
		if (getBitFlag(99)) { // Oracle
			if (_mouseX >= 10 && _mouseX <= 635 && _mouseY >= 5 && _mouseY <= 475) {
				setBitFlag(98, true);
			} else {
				if (getBitFlag(98)) {
					_variableArray[254] = 63;
				}
			}
		} else if (getBitFlag(88)) { // Close Up
			if (_mouseX >= 10 && _mouseX <= 635 && _mouseY >= 5 && _mouseY <= 475) {
				setBitFlag(87, true);
			} else {
				if (getBitFlag(87)) {
					_variableArray[254] = 75;
				}
			}
		}

		if (_rightButtonDown) {
			_rightButtonDown = 0;
			setVerb(NULL);
		}
	}
	if (getGameType() == GType_SIMON2) {
		if (getBitFlag(79)) {
			if (!_vgaVar9) {
				if (_mouseX >= 315 || _mouseX < 9)
					goto get_out2;
				_vgaVar9 = 1;
			}
			if (_scrollCount == 0) {
				if (_mouseX >= 315) {
					if (_scrollX != _scrollXMax)
						_scrollFlag = 1;
				} else if (_mouseX < 8) {
					if (_scrollX != 0)
						_scrollFlag = -1;
				}
			}
		} else {
		get_out2:;
			_vgaVar9 = 0;
		}
	}

	if (_mouseX != _mouseXOld || _mouseY != _mouseYOld)
		_needHitAreaRecalc++;

	x = 0;
	if (_lastHitArea3 == 0 && _leftButtonDown != 0) {
		_leftButtonDown = 0;
		x = 1;
	} else {
		if (_hitarea_unk_3 == 0 && _needHitAreaRecalc == 0)
			goto get_out;
	}

	boxController(_mouseX, _mouseY, x);
	_lastHitArea3 = _lastHitArea;
	if (x == 1 && _lastHitArea == NULL)
		_lastHitArea3 = (HitArea *) -1;

get_out:
	drawMousePointer();
	_needHitAreaRecalc = 0;
}

bool SimonEngine::has_item_childflag_0x10(Item *item) {
	SubObject *child = (SubObject *)findChildOfType(item, 2);
	return child && (child->objectFlags & kOFIcon) != 0;
}

uint SimonEngine::itemGetIconNumber(Item *item) {
	SubObject *child = (SubObject *)findChildOfType(item, 2);
	uint offs;

	if (child == NULL || !(child->objectFlags & kOFIcon))
		return 0;

	offs = getOffsetOfChild2Param(child, 0x10);
	return child->objectFlagValue[offs];
}

void SimonEngine::displayBoxStars() {
	HitArea *ha, *dha;
	uint count;
	uint y_, x_;
	byte *dst;
	uint b, color;

	_lockWord |= 0x8000;

	if (getGameType() == GType_SIMON2)
		color = 236;
	else
		color = 225;

	uint limit = (getGameType() == GType_SIMON2) ? 200 : 134;

	for (int i = 0; i < 5; i++) {
		ha = _hitAreas;
		count = ARRAYSIZE(_hitAreas);

		animateSprites();

		do {
			if (ha->id != 0 && ha->flags & kBFBoxInUse && !(ha->flags & kBFBoxDead)) {

				dha = _hitAreas;
				if (ha->flags & kBFTextBox) {
					while (dha != ha && dha->flags != ha->flags)
						++dha;
					if (dha != ha && dha->flags == ha->flags)
						continue;
				} else {
					dha = _hitAreas;
					while (dha != ha && dha->item_ptr != ha->item_ptr)
						++dha;
					if (dha != ha && dha->item_ptr == ha->item_ptr)
						continue;
				}

				if (ha->y >= limit || ((getGameType() == GType_SIMON2) && ha->y >= _boxStarHeight))
					continue;

				y_ = (ha->height / 2) - 4 + ha->y;

				x_ = (ha->width / 2) - 4 + ha->x - (_scrollX * 8);

				if (x_ >= 311)
					continue;

				dst = getBackBuf();

				dst += (((_dxSurfacePitch / 4) * y_) * 4) + x_;

				b = _dxSurfacePitch;
				dst[4] = color;
				dst[b+1] = color;
				dst[b+4] = color;
				dst[b+7] = color;
				b += _dxSurfacePitch;
				dst[b+2] = color;
				dst[b+4] = color;
				dst[b+6] = color;
				b += _dxSurfacePitch;
				dst[b+3] = color;
				dst[b+5] = color;
				b += _dxSurfacePitch;
				dst[b] = color;
				dst[b+1] = color;
				dst[b+2] = color;
				dst[b+6] = color;
				dst[b+7] = color;
				dst[b+8] = color;
				b += _dxSurfacePitch;
				dst[b+3] = color;
				dst[b+5] = color;
				b += _dxSurfacePitch;
				dst[b+2] = color;
				dst[b+4] = color;
				dst[b+6] = color;
				b += _dxSurfacePitch;
				dst[b+1] = color;
				dst[b+4] = color;
				dst[b+7] = color;
				b += _dxSurfacePitch;
				dst[b+4] = color;
			}
		} while (ha++, --count);

		dx_update_screen_and_palette();
		delay(100);
		animateSprites();
		dx_update_screen_and_palette();
		delay(100);
	}

	_lockWord &= ~0x8000;
}

void SimonEngine::hitarea_stuff() {
	HitArea *ha;
	uint id;

	_leftButtonDown = 0;
	_lastHitArea = 0;
	_verbHitArea = 0;
	_hitAreaSubjectItem = NULL;
	_hitAreaObjectItem = NULL;

	resetVerbs();

startOver:
	for (;;) {
		_lastHitArea = NULL;
		_lastHitArea3 = NULL;

		for (;;) {
			if ((getGameType() == GType_SIMON1 || getGameType() == GType_SIMON2) && _keyPressed == 35)
				displayBoxStars();
			processSpecialKeys();
			if (_lastHitArea3 == (HitArea *) -1)
				goto startOver;
			if (_lastHitArea3 != 0)
				break;
			hitarea_stuff_helper();
			delay(100);
		}

		ha = _lastHitArea;

		if (ha == NULL) {
		} else if (ha->id == 0x7FFB) {
			inventoryUp(ha->window);
		} else if (ha->id == 0x7FFC) {
			inventoryDown(ha->window);
		} else if (ha->id >= 101 && ha->id < 113) {
			_verbHitArea = ha->verb;
			setVerb(ha);
			_defaultVerb = 0;
		} else {
			if ((_verbHitArea != 0 || _hitAreaSubjectItem != ha->item_ptr && ha->flags & kBFBoxItem) &&
					ha->item_ptr) {
			if_1:;
				_hitAreaSubjectItem = ha->item_ptr;
				id = 0xFFFF;
				if (ha->flags & kBFTextBox) {
					if (getGameType() == GType_FF && (ha->flags & kBFHyperBox))
						id = ha->data;
					else
						id = ha->flags / 256;
				}
				_variableArray[60] = id;
				displayName(ha);
				if (_verbHitArea != 0)
					break;
			} else {
				// else 1
				if (ha->verb == 0) {
					if (ha->item_ptr)
						goto if_1;
				} else {
					_verbHitArea = ha->verb & 0xBFFF;
					if (ha->verb & 0x4000) {
						_hitAreaSubjectItem = ha->item_ptr;
						break;
					}
					if (_hitAreaSubjectItem != NULL)
						break;
				}
			}
		}
	}

	_needHitAreaRecalc++;
}

void SimonEngine::hitarea_stuff_helper() {
	time_t cur_time;

	if (getGameType() == GType_SIMON1) {
		uint subr_id = (uint16)_variableArray[254];
		if (subr_id != 0) {
			Subroutine *sub = getSubroutineByID(subr_id);
			if (sub != NULL) {
				startSubroutineEx(sub);
				permitInput();
			}
			_variableArray[254] = 0;
			_runScriptReturn1 = false;
		}
	} else {
		if (_variableArray[254] || _variableArray[249]) {
			hitarea_stuff_helper_2();
		}
	}

	time(&cur_time);
	if ((uint) cur_time != _lastTime) {
		_lastTime = cur_time;
		if (kickoffTimeEvents())
			permitInput();
	}
}

void SimonEngine::hitarea_stuff_helper_2() {
	uint subr_id;
	Subroutine *sub;

	subr_id = (uint16)_variableArray[249];
	if (subr_id != 0) {
		sub = getSubroutineByID(subr_id);
		if (sub != NULL) {
			_variableArray[249] = 0;
			startSubroutineEx(sub);
			permitInput();
		}
		_variableArray[249] = 0;
	}

	subr_id = (uint16)_variableArray[254];
	if (subr_id != 0) {
		sub = getSubroutineByID(subr_id);
		if (sub != NULL) {
			_variableArray[254] = 0;
			startSubroutineEx(sub);
			permitInput();
		}
		_variableArray[254] = 0;
	}

	_runScriptReturn1 = false;
}

void SimonEngine::permitInput() {
	if (!_mortalFlag) {
		_mortalFlag = true;
		showmessage_print_char(0);
		_curWindow = 0;
		if (_windowArray[0] != 0) {
			_textWindow = _windowArray[0];
			if (getGameType() == GType_FF)
				showmessage_helper_3(_textWindow->textColumn, _textWindow->width);
			else
				showmessage_helper_3(_textWindow->textLength, _textWindow->textMaxLength);
		}
		_mortalFlag = false;
	}
}

void SimonEngine::pollMouseXY() {
	_mouseX = _sdlMouseX;
	_mouseY = _sdlMouseY;
}

void SimonEngine::handleVerbClicked(uint verb) {
	Subroutine *sub;
	int result;

	_objectItem = _hitAreaObjectItem;
	if (_objectItem == _dummyItem2) {
		_objectItem = me();
	}
	if (_objectItem == _dummyItem3) {
		_objectItem = derefItem(me()->parent);
	}

	_subjectItem = _hitAreaSubjectItem;
	if (_subjectItem == _dummyItem2) {
		_subjectItem = me();
	}
	if (_subjectItem == _dummyItem3) {
		_subjectItem = derefItem(me()->parent);
	}

	if (_subjectItem) {
		_scriptNoun1 = _subjectItem->noun;
		_scriptAdj1 = _subjectItem->adjective;
	} else {
		_scriptNoun1 = -1;
		_scriptAdj1 = -1;
	}

	if (_objectItem) {
		_scriptNoun2 = _objectItem->noun;
		_scriptAdj2 = _objectItem->adjective;
	} else {
		_scriptNoun2 = -1;
		_scriptAdj2 = -1;
	}

	_scriptVerb = _verbHitArea;

	sub = getSubroutineByID(0);
	if (sub == NULL)
		return;

	result = startSubroutine(sub);
	if (result == -1)
		showMessageFormat("I don't understand");

	_runScriptReturn1 = false;

	sub = getSubroutineByID(100);
	if (sub)
		startSubroutine(sub);

	if (getGameType() == GType_SIMON2 || getGameType() == GType_FF)
		_runScriptReturn1 = false;

	permitInput();
}

TextLocation *SimonEngine::getTextLocation(uint a) {
	switch (a) {
	case 1:
		return &_textLocation1;
	case 2:
		return &_textLocation2;
	case 101:
		return &_textLocation3;
	case 102:
		return &_textLocation4;
	default:
		error("text, invalid value %d", a);
	}
	return NULL;
}

void SimonEngine::loadZone(uint vga_res) {
	VgaPointersEntry *vpe;
	uint32 size;

	CHECK_BOUNDS(vga_res, _vgaBufferPointers);

	vpe = _vgaBufferPointers + vga_res;
	if (vpe->vgaFile1 != NULL)
		return;

	vpe->vgaFile1 = loadVGAFile(vga_res * 2, 1, size);
	vpe->vgaFile1End = vpe->vgaFile1 + size;

	vpe->vgaFile2 = loadVGAFile(vga_res * 2 + 1, 2, size);
	vpe->vgaFile2End = vpe->vgaFile2 + size;

	vpe->sfxFile = NULL;
	if (getGameType() == GType_FF && getPlatform() == Common::kPlatformWindows) {
		vpe->sfxFile = loadVGAFile(vga_res * 2, 3, size);
		vpe->sfxFileEnd = vpe->sfxFile + size;
	}
}

byte *SimonEngine::allocBlock(uint32 size) {
	byte *block, *blockEnd;

	_rejectCount = 0;

	for (;;) {
		block = _vgaBufFreeStart;

		blockEnd = block + size;

		if (blockEnd >= _vgaBufEnd) {
			_vgaBufFreeStart = _vgaBufStart;
		} else {
			_rejectBlock = false;
			checkNoOverWrite(blockEnd);
			if (_rejectBlock)
				continue;
			checkRunningAnims(blockEnd);
			if (_rejectBlock)
				continue;
			checkZonePtrs(blockEnd);
			_vgaBufFreeStart = blockEnd;
			return block;
		}
	}
}

void SimonEngine::setup_vga_file_buf_pointers() {
	byte *alloced;

	alloced = (byte *)malloc(VGA_MEM_SIZE);

	_vgaBufFreeStart = alloced;
	_vgaBufStart = alloced;
	_vgaFileBufOrg = alloced;
	_vgaFileBufOrg2 = alloced;
	_vgaBufEnd = alloced + VGA_MEM_SIZE;
}

void SimonEngine::checkNoOverWrite(byte *end) {
	VgaPointersEntry *vpe;

	if (_noOverWrite == 0xFFFF)
		return;

	if (_rejectCount == 2)
		error("checkNoOverWrite: _rejectCount == 2");

	vpe = &_vgaBufferPointers[_noOverWrite];

	if (vpe->vgaFile1 < end && vpe->vgaFile1End > _vgaBufFreeStart) {
		_rejectBlock = true;
		_rejectCount++;
		_vgaBufFreeStart = vpe->vgaFile1End;
	} else if (vpe->vgaFile2 < end && vpe->vgaFile2End > _vgaBufFreeStart) {
		_rejectBlock = true;
		_rejectCount++;
		_vgaBufFreeStart = vpe->vgaFile2End;
	} else if (vpe->sfxFile && vpe->sfxFile < end && vpe->sfxFileEnd > _vgaBufFreeStart) {
		_rejectBlock = true;
		_rejectCount++;
		_vgaBufFreeStart = vpe->sfxFileEnd;
	} else {
		_rejectBlock = false;
	}
}

void SimonEngine::checkRunningAnims(byte *end) {
	VgaSprite *vsp;
	if ((getGameType() == GType_SIMON1 || getGameType() == GType_SIMON2) &&
		(_lockWord & 0x20)) {
		return;
	}

	for (vsp = _vgaSprites; vsp->id; vsp++) {
		checkAnims(vsp->zoneNum, end);
		if (_rejectBlock == true)
			return;
	}
}

void SimonEngine::checkZonePtrs(byte *end) {
	uint count = ARRAYSIZE(_vgaBufferPointers);
	VgaPointersEntry *vpe = _vgaBufferPointers;
	do {
		if (vpe->vgaFile1 < end && vpe->vgaFile1End > _vgaBufFreeStart ||
				vpe->vgaFile2 < end && vpe->vgaFile2End > _vgaBufFreeStart ||
				vpe->sfxFile < end && vpe->sfxFileEnd > _vgaBufFreeStart) {
			vpe->vgaFile1 = NULL;
			vpe->vgaFile1End = NULL;
			vpe->vgaFile2 = NULL;
			vpe->vgaFile2End = NULL;
			vpe->sfxFile = NULL;
			vpe->sfxFileEnd = NULL;
		}

	} while (++vpe, --count);
}

void SimonEngine::checkAnims(uint a, byte *end) {
	VgaPointersEntry *vpe;

	vpe = &_vgaBufferPointers[a];

	if (vpe->vgaFile1 < end && vpe->vgaFile1End > _vgaBufFreeStart) {
		_rejectBlock = true;
		_vgaBufFreeStart = vpe->vgaFile1End;
	} else if (vpe->vgaFile2 < end && vpe->vgaFile2End > _vgaBufFreeStart) {
		_rejectBlock = true;
		_vgaBufFreeStart = vpe->vgaFile2End;
	} else if (vpe->sfxFile && vpe->sfxFile < end && vpe->sfxFileEnd > _vgaBufFreeStart) {
		_rejectBlock = true;
		_vgaBufFreeStart = vpe->sfxFileEnd;
	} else {
		_rejectBlock = false;
	}
}

void SimonEngine::set_video_mode_internal(uint mode, uint vga_res_id) {
	uint num, num_lines;
	VgaPointersEntry *vpe;
	byte *bb, *b;
	// uint16 count;
	const byte *vc_ptr_org;

	_windowNum = mode;
	_lockWord |= 0x20;

	if (getGameType() == GType_FF) {
		vc27_resetSprite();
		vga_res_id &= 0xFFFF;
	}

	if (vga_res_id == 0) {
		if (getGameType() == GType_SIMON1) {
			_unkPalFlag = true;
		} else if (getGameType() == GType_SIMON2) {
			_useBackGround = true;
			_restoreWindow6 = true;
		}
	}

	_zoneNumber = num = vga_res_id / 100;

	for (;;) {
		vpe = &_vgaBufferPointers[num];

		_curVgaFile1 = vpe->vgaFile1;
		_curVgaFile2 = vpe->vgaFile2;
		_curSfxFile = vpe->sfxFile;

		if (vpe->vgaFile1 != NULL)
			break;

		loadZone(num);
	}

	// ensure flipping complete

	bb = _curVgaFile1;

	if (getGameType() == GType_FF) {
		b = bb + READ_LE_UINT16(&((VgaFileHeader_Feeble *) bb)->hdr2_start);
		//count = READ_LE_UINT16(&((VgaFileHeader2_Feeble *) b)->imageCount);
		b = bb + READ_LE_UINT16(&((VgaFileHeader2_Feeble *) b)->imageTable);

		while (READ_LE_UINT16(&((ImageHeader_Feeble *) b)->id) != vga_res_id)
			b += sizeof(ImageHeader_Feeble);
	} else {
		b = bb + READ_BE_UINT16(&((VgaFileHeader_Simon *) bb)->hdr2_start);
		//count = READ_BE_UINT16(&((VgaFileHeader2_Simon *) b)->imageCount);
		b = bb + READ_BE_UINT16(&((VgaFileHeader2_Simon *) b)->imageTable);

		while (READ_BE_UINT16(&((ImageHeader_Simon *) b)->id) != vga_res_id)
			b += sizeof(ImageHeader_Simon);
	}

	if (getGameType() == GType_SIMON1) {
		if (vga_res_id == 16300) {
			clearBackFromTop(134);
			_usePaletteDelay = true;
		}
	} else {
		_scrollX = 0;
		_scrollY = 0;
		_scrollXMax = 0;
		_scrollYMax = 0;
		_scrollCount = 0;
		_scrollFlag = 0;
		_scrollHeight = 134;
		_variableArrayPtr = _variableArray;
		if (_variableArray[34] >= 0) {
			if (getGameType() == GType_FF)
				_variableArray[250] = 0;
			_variableArray[251] = 0;
		}
	}

	vc_ptr_org = _vcPtr;

	if (getGameType() == GType_FF) {
		_vcPtr = _curVgaFile1 + READ_LE_UINT16(&((ImageHeader_Feeble *) b)->scriptOffs);
	} else {
		_vcPtr = _curVgaFile1 + READ_BE_UINT16(&((ImageHeader_Simon *) b)->scriptOffs);
	}

	//dump_vga_script(_vcPtr, num, vga_res_id);
	runVgaScript();
	_vcPtr = vc_ptr_org;


	if (getGameType() == GType_FF) {
		fillFrontFromBack(0, 0, _screenWidth, _screenHeight);
		fillBackGroundFromBack(_screenHeight);
		_syncFlag2 = 1;
	} else if (getGameType() == GType_SIMON2) {
		if (!_useBackGround) {
			num_lines = _windowNum == 4 ? 134 : 200;
			_boxStarHeight = num_lines;
			fillFrontFromBack(0, 0, _screenWidth, num_lines);
			fillBackGroundFromBack(num_lines);
			_syncFlag2 = 1;
		}
		_useBackGround = false;
	} else {
		// Allow one section of Simon the Sorcerer 1 introduction to be displayed
		// in lower half of screen
		if (_subroutine == 2923 || _subroutine == 2926)
			num_lines = 200;
		else
			num_lines = _windowNum == 4 ? 134 : 200;

		fillFrontFromBack(0, 0, _screenWidth, num_lines);
		fillBackGroundFromBack(num_lines);

		_syncFlag2 = 1;
		_timer5 = 0;
	}

	_lockWord &= ~0x20;

	if (getGameType() == GType_SIMON1) {
		if (_unkPalFlag) {
			_unkPalFlag = false;
			while (_paletteColorCount != 0) {
				delay(10);
			}
		}
	}
}

void SimonEngine::fadeToBlack() {
	uint i;

	memcpy(_videoBuf1, _paletteBackup, 1024);

	i = NUM_PALETTE_FADEOUT;
	do {
		palette_fadeout((uint32 *)_videoBuf1, 32);
		palette_fadeout((uint32 *)_videoBuf1 + 32 + 16, 144);
		palette_fadeout((uint32 *)_videoBuf1 + 32 + 16 + 144 + 16, 48);

		_system->setPalette(_videoBuf1, 0, 256);
		if (_fade)
			_system->updateScreen();
		delay(5);
	} while (--i);

	memcpy(_paletteBackup, _videoBuf1, 1024);
	memcpy(_palette, _videoBuf1, 1024);
}

void SimonEngine::addVgaEvent(uint16 num, const byte *code_ptr, uint16 cur_sprite, uint16 curZoneNum, int32 param) {
	VgaTimerEntry *vte;

	// When Simon talks to the Golum about stew in French version of
	// Simon the Sorcerer 1 the code_ptr is at wrong location for
	// sprite 200. This  was a bug in the original game, which
	// caused several glitches in this scene.
	// We work around the problem by correcting the code_ptr for sprite
	// 200 in this scene, if it is wrong.
	if (getGameType() == GType_SIMON1 && _language == Common::FR_FRA &&
		(code_ptr - _vgaBufferPointers[curZoneNum].vgaFile1 == 4) && (cur_sprite == 200) && (curZoneNum == 2))
		code_ptr += 0x66;

	_lockWord |= 1;

	for (vte = _vgaTimerList; vte->delay; vte++) {
	}

	vte->delay = num;
	vte->script_pointer = code_ptr;
	vte->sprite_id = cur_sprite;
	vte->cur_vga_file = curZoneNum;
	vte->param = param;

	_lockWord &= ~1;
}

void SimonEngine::deleteVgaEvent(VgaTimerEntry * vte) {
	_lockWord |= 1;

	if (vte + 1 <= _nextVgaTimerToProcess) {
		_nextVgaTimerToProcess--;
	}

	do {
		memcpy(vte, vte + 1, sizeof(VgaTimerEntry));
		vte++;
	} while (vte->delay);

	_lockWord &= ~1;
}

void SimonEngine::processVgaEvents() {
	VgaTimerEntry *vte = _vgaTimerList;
	uint timer = (getGameType() == GType_FF) ? 5 : 1;

	_vgaTickCounter++;

	while (vte->delay) {
		vte->delay -= timer;
		if (vte->delay <= 0) {
			uint16 curZoneNum = vte->cur_vga_file;
			uint16 cur_sprite = vte->sprite_id;
			const byte *script_ptr = vte->script_pointer;
			int32 param = vte->param;

			_nextVgaTimerToProcess = vte + 1;
			deleteVgaEvent(vte);

			if (getGameType() == GType_FF && script_ptr == NULL) {
				panEvent(curZoneNum, cur_sprite, param);
			} else if (getGameType() == GType_SIMON2 && script_ptr == NULL) {
				scrollEvent();
			} else {
				animateEvent(script_ptr, curZoneNum, cur_sprite);
			}
			vte = _nextVgaTimerToProcess;
		} else {
			vte++;
		}
	}
}

void SimonEngine::animateEvent(const byte *code_ptr, uint16 curZoneNum, uint16 cur_sprite) {
	VgaPointersEntry *vpe;

	_vgaCurSpriteId = cur_sprite;

	_vgaCurZoneNum = curZoneNum;
	_zoneNumber = curZoneNum;
	vpe = &_vgaBufferPointers[curZoneNum];

	_curVgaFile1 = vpe->vgaFile1;
	_curVgaFile2 = vpe->vgaFile2;
	_curSfxFile = vpe->sfxFile;

	_vcPtr = code_ptr;

	runVgaScript();
}

void SimonEngine::panEvent(uint16 curZoneNum, uint16 cur_sprite, int32 param) {
	_vgaCurSpriteId = cur_sprite;
	_vgaCurZoneNum = curZoneNum;

	VgaSprite *vsp = findCurSprite();

	param &= 0x10;

	int32 pan = (vsp->x - _scrollX + param) * 8 - 2560;
	if (pan < -10000)
		pan = -10000;
	if (pan > 10000)
		pan = 10000;

	//setSfxPan(param, pan);

	if (pan != 0)
		addVgaEvent(10, NULL, _vgaCurSpriteId, _vgaCurZoneNum); /* pan event */
	debug(0, "panEvent: param %d pan %d", param, pan);
}

void SimonEngine::scrollEvent() {
	if (_scrollCount == 0)
		return;

	if (getGameType() == GType_FF) {
		if (_scrollCount < 0) {
			if (_scrollFlag != -8) {
				_scrollFlag = -8;
				_scrollCount += 8;
			}
		} else {
			if (_scrollFlag != 8) {
				_scrollFlag = 8;
				_scrollCount -= 8;
			}
		}
	} else {
		if (_scrollCount < 0) {
			if (_scrollFlag != -1) {
				_scrollFlag = -1;
				if (++_scrollCount == 0)
					return;
			}
		} else {
			if (_scrollFlag != 1) {
				_scrollFlag = 1;
				if (--_scrollCount == 0)
					return;
			}
		}

		addVgaEvent(6, NULL, 0, 0); /* scroll event */
	}
}

void SimonEngine::waitForSync(uint a) {
	const uint maxCount = (getGameType() == GType_SIMON1) ? 500 : 1000;

	_vgaWaitFor = a;
	_syncCount = 0;
	_exitCutscene = false;
	_rightButtonDown = false;
	while (_vgaWaitFor != 0) {
		if (_rightButtonDown && (getGameType() == GType_SIMON2 || getGameType() == GType_FF)) {
			if (_vgaWaitFor == 200 && (getGameType() == GType_FF || !getBitFlag(14))) {
				skipSpeech();
				break;
			}
		}
		if (_exitCutscene) {
			if (getBitFlag(9)) {
				endCutscene();
				break;
			}
		}
		processSpecialKeys();

		if (_syncCount >= maxCount) {
			warning("waitForSync: wait timed out");
			break;
		}

		delay(1);
	}
}

void SimonEngine::skipSpeech() {
	_sound->stopVoice();
	if (getBitFlag(28) == false) {
		setBitFlag(14, true);
		if (getGameType() == GType_FF) {
			_variableArray[103] = 5;
			loadSprite(4, 2, 13, 0, 0, 0);
			waitForSync(213);
			stopAnimateSimon2(2, 1);
		} else {
			_variableArray[100] = 5;
			loadSprite(4, 1, 30, 0, 0, 0);
			waitForSync(130);
			stopAnimateSimon2(2, 1);
		}
	}
}

void SimonEngine::animateSprites() {
	VgaSprite *vsp;
	VgaPointersEntry *vpe;
	const byte *vc_ptr_org = _vcPtr;
	uint16 params[5];							// parameters to vc10

	if (_paletteFlag == 2)
		_paletteFlag = 1;

	if (getGameType() == GType_FF && _scrollCount) {
		scrollEvent();
	}
	if (getGameType() == GType_SIMON2 && _scrollFlag) {
		scrollScreen();
	}

	if (getGameType() == GType_FF && getBitFlag(84)) {
		animateSpritesByY();
		return;
	}

	vsp = _vgaSprites;

	while (vsp->id != 0) {
		vsp->windowNum &= 0x7FFF;

		vpe = &_vgaBufferPointers[vsp->zoneNum];
		_curVgaFile1 = vpe->vgaFile1;
		_curVgaFile2 = vpe->vgaFile2;
		_curSfxFile = vpe->sfxFile;
		_windowNum = vsp->windowNum;
		_vgaCurSpriteId = vsp->id;
		_vgaCurSpritePriority = vsp->priority;

		params[0] = readUint16Wrapper(&vsp->image);
		params[1] = readUint16Wrapper(&vsp->palette);
		params[2] = readUint16Wrapper(&vsp->x);
		params[3] = readUint16Wrapper(&vsp->y);

		if (getGameType() == GType_SIMON1) {
			params[4] = READ_BE_UINT16(&vsp->flags);
		} else {
			*(byte *)(&params[4]) = (byte)vsp->flags;
		}

		_vcPtr = (const byte *)params;
		vc10_draw();

		vsp++;
	}

	if (_drawImagesDebug)
		memset(_backBuf, 0, _screenWidth * _screenHeight);

	_updateScreen++;
	_vcPtr = vc_ptr_org;
}

void SimonEngine::animateSpritesDebug() {
	VgaSprite *vsp;
	VgaPointersEntry *vpe;
	const byte *vc_ptr_org = _vcPtr;
	uint16 params[5];							// parameters to vc10_draw

	if (_paletteFlag == 2)
		_paletteFlag = 1;

	vsp = _vgaSprites;
	while (vsp->id != 0) {
		vsp->windowNum &= 0x7FFF;

		vpe = &_vgaBufferPointers[vsp->zoneNum];
		_curVgaFile1 = vpe->vgaFile1;
		_curVgaFile2 = vpe->vgaFile2;
		_curSfxFile = vpe->sfxFile;
		_windowNum = vsp->windowNum;
		_vgaCurSpriteId = vsp->id;

		if (vsp->image)
			fprintf(_dumpFile, "id:%5d image:%3d base-color:%3d x:%3d y:%3d flags:%x\n",
							vsp->id, vsp->image, vsp->palette, vsp->x, vsp->y, vsp->flags);
		params[0] = readUint16Wrapper(&vsp->image);
		params[1] = readUint16Wrapper(&vsp->palette);
		params[2] = readUint16Wrapper(&vsp->x);
		params[3] = readUint16Wrapper(&vsp->y);
		params[4] = readUint16Wrapper(&vsp->flags);
		_vcPtr = (const byte *)params;
		vc10_draw();

		vsp++;
	}

	_updateScreen++;
	_vcPtr = vc_ptr_org;
}

void SimonEngine::animateSpritesByY() {
	VgaSprite *vsp;
	VgaPointersEntry *vpe;
	const byte *vc_ptr_org = _vcPtr;
	uint16 params[5];							// parameters to vc10
	int16 spriteTable[180][2];
	
	byte *src;
	int height, slot, y;
	uint i, numSprites = 0;

	vsp = _vgaSprites;
	while (vsp->id != 0) {
		if (vsp->flags & kDFScaled) {
			y = vsp->y;
		} else {
			y = vsp->priority;
			vpe = &_vgaBufferPointers[vsp->zoneNum];
			src = vpe->vgaFile2 + vsp->image * 8;
			height = READ_LE_UINT16(src + 4) & 0x7FFF;
			y += height;
		}

		spriteTable[numSprites][0] = y;
		spriteTable[numSprites][1] = numSprites;
		numSprites++;
		vsp++;
	}

	while(1) {
		y = spriteTable[0][0];
		slot = spriteTable[0][1];

		for (i = 0; i < numSprites; i++) {
			if (y >= spriteTable[i][0]) {
				y = spriteTable[i][0];
				slot = spriteTable[i][1];
			}
		}

		if (y == 9999)
			break;

		for (i = 0; i < numSprites; i++) {
			if (slot == spriteTable[i][1]) {
				spriteTable[i][0] = 9999;
				break;
			}
		}

		vsp = &_vgaSprites[slot];
		vsp->windowNum &= 0x7FFF;

		vpe = &_vgaBufferPointers[vsp->zoneNum];
		_curVgaFile1 = vpe->vgaFile1;
		_curVgaFile2 = vpe->vgaFile2;
		_curSfxFile = vpe->sfxFile;
		_windowNum = vsp->windowNum;
		_vgaCurSpriteId = vsp->id;
		_vgaCurSpritePriority = vsp->priority;

		params[0] = readUint16Wrapper(&vsp->image);
		params[1] = readUint16Wrapper(&vsp->palette);
		params[2] = readUint16Wrapper(&vsp->x);
		params[3] = readUint16Wrapper(&vsp->y);
		*(byte *)(&params[4]) = (byte)vsp->flags;

		_vcPtr = (const byte *)params;
		vc10_draw();
	}

	_updateScreen++;
	_vcPtr = vc_ptr_org;
}

void SimonEngine::scrollScreen() {
	byte *dst = getFrontBuf();
	const byte *src;
	uint x, y;

	if (_scrollXMax == 0) {
		uint screenSize = 8 * _screenWidth;
		if (_scrollFlag < 0) {
			memmove(dst + screenSize, dst, _scrollWidth * _screenHeight - screenSize);
		} else {
			memmove(dst, dst + screenSize, _scrollWidth * _screenHeight - screenSize);
		}

		y = _scrollY - 8;

		if (_scrollFlag > 0) {
			dst += _screenHeight * _screenWidth - screenSize;
			y += 488;
		}

		src = _scrollImage + y / 2;
		decodeRow(dst, src + readUint32Wrapper(src), _scrollWidth);

		_scrollY += _scrollFlag;
		vcWriteVar(250, _scrollY);
	} else {
		if (_scrollFlag < 0) {
			memmove(dst + 8, dst, _screenWidth * _scrollHeight - 8);
		} else {
			memmove(dst, dst + 8, _screenWidth * _scrollHeight - 8);
		}

		x = _scrollX;
		x -= (getGameType() == GType_FF) ? 8 : 1;

		if (_scrollFlag > 0) {
			dst += _screenWidth - 8;
			x += (getGameType() == GType_FF) ? 648 : 41;
		}

		if (getGameType() == GType_FF)
			src = _scrollImage + x / 2;
		else
			src = _scrollImage + x * 4;
		decodeColumn(dst, src + readUint32Wrapper(src), _scrollHeight);

		_scrollX += _scrollFlag;
		vcWriteVar(251, _scrollX);
	}

	memcpy(_backBuf, _frontBuf, _screenWidth * _screenHeight);
	memcpy(_backGroundBuf, _backBuf, _scrollHeight * _screenWidth);

	_scrollFlag = 0;
}

void SimonEngine::timer_proc1() {
	_timer4++;

	if (_lockWord & 0x80E9 || _lockWord & 2)
		return;

	_syncCount++;

	_lockWord |= 2;

	if (!(_lockWord & 0x10)) {
		if (getGameType() == GType_FF) {
			_syncFlag2 ^= 1;

			if (!_syncFlag2)
				processVgaEvents();
		} else {
			processVgaEvents();
			processVgaEvents();
			_syncFlag2 ^= 1;
			_cepeFlag ^= 1;
			if (!_cepeFlag)
				processVgaEvents();

			if (_mouseHideCount != 0 && _syncFlag2) {
				_lockWord &= ~2;
				return;
			}
		}
	}

	animateSprites();
	if (_drawImagesDebug)
		animateSpritesDebug();

	if (_copyPartialMode == 1) {
		fillBackFromFront(80, 46, 208 - 80, 94 - 46);
	}

	if (_copyPartialMode == 2) {
		fillFrontFromBack(176, 61, _screenWidth - 176, 134 - 61);
		_copyPartialMode = 0;
	}

	if (_updateScreen) {
		if (getGameType() == GType_FF) {
			if (!getBitFlag(78)) {
				oracleLogo();
			}
			if (getBitFlag(76)) {
				swapCharacterLogo();
			}
		}
		handleMouseMoved();
		dx_update_screen_and_palette();
		_updateScreen = false;
	}

	_lockWord &= ~2;
}

void SimonEngine::timer_callback() {
	if (_timer5 != 0) {
		_syncFlag2 = true;
		_timer5--;
	} else {
		timer_proc1();
	}
}

bool SimonEngine::itemIsSiblingOf(uint16 a) {
	Item *item;

	CHECK_BOUNDS(a, _objectArray);

	item = _objectArray[a];
	if (item == NULL)
		return true;

	return me()->parent == item->parent;
}

bool SimonEngine::itemIsParentOf(uint16 a, uint16 b) {
	Item *item_a, *item_b;

	CHECK_BOUNDS(a, _objectArray);
	CHECK_BOUNDS(b, _objectArray);

	item_a = _objectArray[a];
	item_b = _objectArray[b];

	if (item_a == NULL || item_b == NULL)
		return true;

	return derefItem(item_a->parent) == item_b;
}

bool SimonEngine::vc_maybe_skip_proc_1(uint16 a, int16 b) {
	Item *item;

	CHECK_BOUNDS(a, _objectArray);

	item = _objectArray[a];
	if (item == NULL)
		return true;
	return item->state == b;
}

void SimonEngine::closeWindow(uint a) {
	if (_windowArray[a] == NULL)
		return;
	removeIconArray(a);
	resetWindow(_windowArray[a]);
	_windowArray[a] = NULL;
	if (_curWindow == a) {
		_textWindow = NULL;
		changeWindow(0);
	}
}

void SimonEngine::changeWindow(uint a) {
	a &= 7;

	if (_windowArray[a] == NULL || _curWindow == a)
		return;

	_curWindow = a;
	showmessage_print_char(0);
	_textWindow = _windowArray[a];

	if (getGameType() == GType_FF)
		showmessage_helper_3(_textWindow->textColumn, _textWindow->width);
	else
		showmessage_helper_3(_textWindow->textLength, _textWindow->textMaxLength);
}

WindowBlock *SimonEngine::openWindow(uint x, uint y, uint w, uint h, uint flags, uint fill_color, uint text_color) {
	WindowBlock *window;

	window = _windowList;
	while (window->mode != 0)
		window++;

	window->mode = 2;
	window->x = x;
	window->y = y;
	window->width = w;
	window->height = h;
	window->flags = flags;
	window->fill_color = fill_color;
	window->text_color = text_color;
	window->textColumn = 0;
	window->textRow = 0;
	window->textColumnOffset = 0;
	window->textMaxLength = window->width * 8 / 6; // characters are 6 pixels
	window->scrollY = 0;
	return window;
}

Item *SimonEngine::derefItem(uint item) {
	if (item >= _itemArraySize)
		error("derefItem: invalid item %d", item);
	return _itemArrayPtr[item];
}

uint SimonEngine::itemPtrToID(Item *id) {
	uint i;
	for (i = 0; i != _itemArraySize; i++)
		if (_itemArrayPtr[i] == id)
			return i;
	error("itemPtrToID: not found");
	return 0;
}

void SimonEngine::delete_hitarea_by_index(uint index) {
	CHECK_BOUNDS(index, _hitAreas);
	_hitAreas[index].flags = 0;
}

void SimonEngine::windowPutChar(uint a) {
	if (_textWindow != _windowArray[0])
		windowPutChar(_textWindow, a);
}

void SimonEngine::clearWindow(WindowBlock *window) {
	if (window->flags & 0x10)
		restoreWindow(window);
	else
		colorWindow(window);

	window->textColumn = 0;
	window->textRow = 0;
	window->textColumnOffset = 0;
	window->textLength = 0;
	window->scrollY = 0;
}

void SimonEngine::restoreWindow(WindowBlock *window) {
	_lockWord |= 0x8000;

	if (getGameType() == GType_FF) {
		restoreBlock(window->y + window->height, window->x + window->width, window->y, window->x);
	} else if (getGameType() == GType_SIMON2) {
		if (_restoreWindow6 && _windowArray[2] == window) {
			window = _windowArray[6];
			_restoreWindow6 = 0;
		}

		restoreBlock(window->y + window->height * 8, (window->x + window->width) * 8, window->y, window->x * 8);
	} else {
		restoreBlock(window->y + window->height * 8 + ((window == _windowArray[2]) ? 1 : 0), (window->x + window->width) * 8, window->y, window->x * 8);
	}

	_lockWord &= ~0x8000;
}

void SimonEngine::colorWindow(WindowBlock *window) {
	byte *dst;
	uint h, w;

	_lockWord |= 0x8000;

	if (getGameType() == GType_FF) {
		dst = getFrontBuf() + _dxSurfacePitch * window->y + window->x;

		for (h = 0; h < window->height; h++) {
			for (w = 0; w < window->width; w++) {
				if (dst[w] == 113  || dst[w] == 116 || dst[w] == 252)
					dst[w] = window->fill_color;
			}
			dst += _screenWidth;
		}
	} else {
		dst = getFrontBuf() + _dxSurfacePitch * window->y + window->x * 8;
		h = window->height * 8;
		w = window->width * 8;

		do {
			memset(dst, window->fill_color, w);
			dst += _dxSurfacePitch;
		} while (--h);
	}

	_lockWord &= ~0x8000;
}

VgaSprite *SimonEngine::findCurSprite() {
	VgaSprite *vsp = _vgaSprites;
	while (vsp->id) {
		if (getGameType() == GType_SIMON1) {
			if (vsp->id == _vgaCurSpriteId)
				break;
		} else {
			if (vsp->id == _vgaCurSpriteId && vsp->zoneNum == _vgaCurZoneNum)
				break;
		}
		vsp++;
	}
	return vsp;
}

bool SimonEngine::isSpriteLoaded(uint16 id, uint16 zoneNum) {
	VgaSprite *vsp = _vgaSprites;
	while (vsp->id) {
		if (getGameType() == GType_SIMON1) {
			if (vsp->id == id)
				return true;
		} else {
			if (vsp->id == id && vsp->zoneNum == zoneNum)
				return true;
		}
		vsp++;
	}
	return false;
}

void SimonEngine::processSpecialKeys() {
	switch (_keyPressed) {
	case 27: // escape
		_exitCutscene = true;
		break;
	case 59: // F1
		if (getGameType() == GType_SIMON1) {
			vcWriteVar(5, 40);
		} else {
			vcWriteVar(5, 50);
		}
		vcWriteVar(86, 0);
		break;
	case 60: // F2
		if (getGameType() == GType_SIMON1) {
			vcWriteVar(5, 60);
		} else {
			vcWriteVar(5, 75);
		}
		vcWriteVar(86, 1);
		break;
	case 61: // F3
		if (getGameType() == GType_SIMON1) {
			vcWriteVar(5, 100);
		} else {
			vcWriteVar(5, 125);
		}
		vcWriteVar(86, 2);
		break;
	case 63: // F5
		if (getGameType() == GType_SIMON2 || getGameType() == GType_FF)
			_exitCutscene = true;
		break;
	case 65: // F7
		if (getGameType() == GType_FF && getBitFlag(76))
			_variableArray[254] = 70;
		break;
	case 67: // F9
		if (getGameType() == GType_FF)
			setBitFlag(73, !getBitFlag(73));
		break;
	case 'p':
		pause();
		break;
	case 't':
		if (getGameType() == GType_FF || (getGameType() == GType_SIMON2 && (getFeatures() & GF_TALKIE)) ||
			((getFeatures() & GF_TALKIE) && _language != Common::EN_ANY && _language != Common::DE_DEU)) {
			if (_speech)
				_subtitles ^= 1;
		}
		break;
	case 'v':
		if (getGameType() == GType_FF || (getGameType() == GType_SIMON2 && (getFeatures() & GF_TALKIE))) {
			if (_subtitles)
				_speech ^= 1;
		}
	case '+':
		midi.set_volume(midi.get_volume() + 16);
		break;
	case '-':
		midi.set_volume(midi.get_volume() - 16);
		break;
	case 'm':
		midi.pause(_musicPaused ^= 1);
		break;
	case 's':
		if (getGameId() == GID_SIMON1DOS)
			midi._enable_sfx ^= 1;
		else
			_sound->effectsPause(_effectsPaused ^= 1);
		break;
	case 'b':
		_sound->ambientPause(_ambientPaused ^= 1);
		break;
	case 'r':
		if (_debugMode)
			_startMainScript ^= 1;
		break;
	case 'o':
		if (_debugMode)
			_continousMainScript ^= 1;
		break;
	case 'a':
		if (_debugMode)
			_startVgaScript ^= 1;
		break;
	case 'g':
		if (_debugMode)
			_continousVgaScript ^= 1;
		break;
	case 'i':
		if (_debugMode)
			_drawImagesDebug ^= 1;
		break;
	case 'd':
		if (_debugMode)
			_dumpImages ^=1;
		break;
	}

	_keyPressed = 0;
}

void SimonEngine::pause() {
	_keyPressed = 1;
	_pause = 1;
	bool ambient_status = _ambientPaused;
	bool music_status = _musicPaused;

	midi.pause(true);
	_sound->ambientPause(true);
	while (_pause) {
		delay(1);
		if (_keyPressed == 'p')
			_pause = 0;
	}
	midi.pause(music_status);
	_sound->ambientPause(ambient_status);

}

void SimonEngine::video_toggle_colors(HitArea * ha, byte a, byte b, byte c, byte d) {
	byte *src, color;
	int w, h, i;

	_lockWord |= 0x8000;
	src = getFrontBuf() + ha->y * _dxSurfacePitch + ha->x;

	w = ha->width;
	h = ha->height;

	// Works around bug in original Simon the Sorcerer 2
	// Animations continue in background when load/save dialog is open
	// often causing the savegame name highlighter to be cut short
	if (!(h > 0 && w > 0 && ha->x + w <= _screenWidth && ha->y + h <= _screenHeight)) {
		debug(1,"Invalid coordinates in video_toggle_colors (%d,%d,%d,%d)", ha->x, ha->y, ha->width, ha->height);
		_lockWord &= ~0x8000;
		return;
	}

	do {
		for (i = 0; i != w; ++i) {
			color = src[i];
			if (a >= color && b < color) {
				if (c >= color)
					color += d;
				else
					color -= d;
				src[i] = color;
			}
		}
		src += _dxSurfacePitch;
	} while (--h);

	_lockWord &= ~0x8000;
}

void SimonEngine::resetWindow(WindowBlock *window) {
	if (window->flags & 8)
		restoreWindow(window);
	window->mode = 0;
}

void SimonEngine::loadSprite(uint windowNum, uint zoneNum, uint vgaSpriteId, uint x, uint y, uint palette) {
	VgaSprite *vsp;
	VgaPointersEntry *vpe;
	byte *p, *pp;
	uint count;

	_lockWord |= 0x40;

	if (isSpriteLoaded(vgaSpriteId, zoneNum)) {
		_lockWord &= ~0x40;
		return;
	}

	vsp = _vgaSprites;
	while (vsp->id != 0)
		vsp++;

	vsp->windowNum = windowNum;
	vsp->priority = 0;
	vsp->flags = 0;

	vsp->y = y;
	vsp->x = x;
	vsp->image = 0;
	vsp->palette = palette;
	vsp->id = vgaSpriteId;
	if (getGameType() == GType_SIMON1)
		vsp->zoneNum = zoneNum = vgaSpriteId / 100;
	else
		vsp->zoneNum = zoneNum;


	for (;;) {
		vpe = &_vgaBufferPointers[zoneNum];
		_zoneNumber = zoneNum;
		_curVgaFile1 = vpe->vgaFile1;
		if (vpe->vgaFile1 != NULL)
			break;
		loadZone(zoneNum);
	}

	pp = _curVgaFile1;
	if (getGameType() == GType_FF) {
		p = pp + READ_LE_UINT16(&((VgaFileHeader_Feeble *) pp)->hdr2_start);
		count = READ_LE_UINT16(&((VgaFileHeader2_Feeble *) p)->animationCount);
		p = pp + READ_LE_UINT16(&((VgaFileHeader2_Feeble *) p)->animationTable);
	} else {
		p = pp + READ_BE_UINT16(&((VgaFileHeader_Simon *) pp)->hdr2_start);
		count = READ_BE_UINT16(&((VgaFileHeader2_Simon *) p)->animationCount);
		p = pp + READ_BE_UINT16(&((VgaFileHeader2_Simon *) p)->animationTable);
	}

	for (;;) {
		if (getGameType() == GType_FF) {
			if (READ_LE_UINT16(&((AnimationHeader_Feeble *) p)->id) == vgaSpriteId) {
				if (_startVgaScript)
					dump_vga_script(pp + READ_LE_UINT16(&((AnimationHeader_Feeble*)p)->scriptOffs), zoneNum, vgaSpriteId);

				addVgaEvent(VGA_DELAY_BASE, pp + READ_LE_UINT16(&((AnimationHeader_Feeble *) p)->scriptOffs), vgaSpriteId, zoneNum);
				break;
			}
			p += sizeof(AnimationHeader_Feeble);
		} else {
			if (READ_BE_UINT16(&((AnimationHeader_Simon *) p)->id) == vgaSpriteId) {
				if (_startVgaScript)
					dump_vga_script(pp + READ_BE_UINT16(&((AnimationHeader_Simon*)p)->scriptOffs), zoneNum, vgaSpriteId);

				addVgaEvent(VGA_DELAY_BASE, pp + READ_BE_UINT16(&((AnimationHeader_Simon *) p)->scriptOffs), vgaSpriteId, zoneNum);
				break;
			}
			p += sizeof(AnimationHeader_Simon);
		}

		if (!--count) {
			vsp->id = 0;
			break;
		}
	}

	_lockWord &= ~0x40;
}

void SimonEngine::playSpeech(uint speech_id, uint vgaSpriteId) {
	if (getGameType() == GType_SIMON1) {
		if (speech_id == 9999) {
			if (_subtitles)
				return;
			if (!getBitFlag(14) && !getBitFlag(28)) {
				setBitFlag(14, true);
				_variableArray[100] = 15;
				loadSprite(4, 1, 130, 0, 0, 0);
				waitForSync(130);
			}
			_skipVgaWait = true;
		} else {
			if (_subtitles && _scriptVar2) {
				loadSprite(4, 2, 204, 0, 0, 0);
				waitForSync(204);
				stopAnimateSimon1(204);
			}
			stopAnimateSimon1(vgaSpriteId + 201);
			loadVoice(speech_id);
			loadSprite(4, 2, vgaSpriteId + 201, 0, 0, 0);
		}
	} else {
		if (speech_id == 0xFFFF) {
			if (_subtitles)
				return;
			if (!getBitFlag(14) && !getBitFlag(28)) {
				setBitFlag(14, true);
				_variableArray[100] = 5;
				loadSprite(4, 1, 30, 0, 0, 0);
				waitForSync(130);
			}
			_skipVgaWait = true;
		} else {
			if (getGameType() == GType_SIMON2 && _subtitles && _language != Common::HB_ISR) {
				loadVoice(speech_id);
				return;
			}

			if (_subtitles && _scriptVar2) {
				loadSprite(4, 2, 5, 0, 0, 0);
				waitForSync(205);
				stopAnimateSimon2(2,5);
			}

			stopAnimateSimon2(2, vgaSpriteId + 2);
			loadVoice(speech_id);
			loadSprite(4, 2, vgaSpriteId + 2, 0, 0, 0);
		}
	}
}

void SimonEngine::runSubroutine101() {
	Subroutine *sub;

	sub = getSubroutineByID(101);
	if (sub != NULL)
		startSubroutineEx(sub);

	permitInput();
}

void SimonEngine::restoreBlock(uint h, uint w, uint y, uint x) {
	byte *dst, *src;
	uint i;

	dst = getFrontBuf();
	src = _backGroundBuf;

	dst += y * _dxSurfacePitch;
	src += y * _dxSurfacePitch;

	while (y < h) {
		for (i = x; i < w; i++)
			dst[i] = src[i];
		y++;
		dst += _dxSurfacePitch;
		src += _dxSurfacePitch;
	}
}

void SimonEngine::dx_clear_surfaces(uint num_lines) {
	memset(_backBuf, 0, num_lines * _screenWidth);

	_system->copyRectToScreen(_backBuf, _screenWidth, 0, 0, _screenWidth, num_lines);

	if (_useBackGround) {
		memset(_frontBuf, 0, num_lines * _screenWidth);
		memset(_backGroundBuf, 0, num_lines * _screenWidth);
	}
}

void SimonEngine::clearBackFromTop(uint lines) {
	memset(_backBuf, 0, lines * _screenWidth);
}

void SimonEngine::fillFrontFromBack(uint x, uint y, uint w, uint h) {
	uint offs = x + y * _screenWidth;
	byte *s = _backBuf + offs;
	byte *d = _frontBuf + offs;

	do {
		memcpy(d, s, w);
		d += _screenWidth;
		s += _screenWidth;
	} while (--h);
}

void SimonEngine::fillBackFromFront(uint x, uint y, uint w, uint h) {
	uint offs = x + y * _screenWidth;
	byte *s = _frontBuf + offs;
	byte *d = _backBuf + offs;

	do {
		memcpy(d, s, w);
		d += _screenWidth;
		s += _screenWidth;
	} while (--h);
}

void SimonEngine::fillBackGroundFromBack(uint lines) {
	memcpy(_backGroundBuf, _backBuf, lines * _screenWidth);
}

void SimonEngine::dx_update_screen_and_palette() {
	_numScreenUpdates++;

	if (_paletteColorCount == 0 && _paletteFlag == 1) {
		_paletteFlag = 0;
		if (memcmp(_palette, _paletteBackup, 1024) != 0) {
			memcpy(_paletteBackup, _palette, 1024);
			_system->setPalette(_palette, 0, 256);
		}
	}

	_system->copyRectToScreen(_backBuf, _screenWidth, 0, 0, _screenWidth, _screenHeight);
	_system->updateScreen();

	memcpy(_backBuf, _frontBuf, _screenWidth * _screenHeight);

	if (getGameType() == GType_FF && _scrollFlag) {
		scrollScreen();
	}

	if (_paletteColorCount != 0) {
		if (getGameType() == GType_SIMON1 && _usePaletteDelay) {
			delay(100);
			_usePaletteDelay = false;
		}
		fastFadeIn();
	}
}

void SimonEngine::fastFadeIn() {
	if (_paletteColorCount & 0x8000) {
		slowFadeIn();
	} else {
		_paletteFlag = false;
		memcpy(_paletteBackup, _palette, 1024);
		_system->setPalette(_palette, 0, _paletteColorCount);
		_paletteColorCount = 0;
	}
}

void SimonEngine::slowFadeIn() {
	uint8 paletteTmp[768];
	uint8 *src, *dst;
	int c, p;

	_paletteColorCount &= 0x7fff;
	_paletteFlag = false;

	memcpy(_videoBuf1, _palette, 1024); // Difference
	memset(_videoBuf1, 0, 768);

	memcpy(_paletteBackup, _palette, 768);
	memcpy(paletteTmp, _palette, 768);

	for (c = 255; c > 0; c -= 4) {
	  	src = paletteTmp;
 		dst = _videoBuf1;

		for (p = _paletteColorCount; p !=0 ; p--) {
			if (*src >= c)
				*dst = *dst + 4;
			
			src++;
			dst++;
 		}
 		_system->setPalette(_videoBuf1, 0, _videoNumPalColors);
		if (_fade)
			_system->updateScreen();
 		delay(5);
 	}
	_paletteColorCount = 0;
}

int SimonEngine::go() {
	if (!_dumpFile)
		_dumpFile = stdout;

	// allocate buffers
	_backGroundBuf = (byte *)calloc(_screenWidth * _screenHeight, 1);
	_frontBuf = (byte *)calloc(_screenWidth * _screenHeight, 1);
	_backBuf = (byte *)calloc(_screenWidth * _screenHeight, 1);
	if (getGameType() == GType_FF)
		_sdl_buf_scaled = (byte *)calloc(_screenWidth * _screenHeight, 1);

	allocItemHeap();
	allocTablesHeap();

	setup_vga_file_buf_pointers();

	_debugger = new Debugger(this);
	_moviePlay = new MoviePlayer(this, _mixer);
	_sound = new Sound(this, gss, _mixer);

	if (ConfMan.hasKey("sfx_mute") && ConfMan.getBool("sfx_mute") == 1) {
		if (getGameId() == GID_SIMON1DOS)
			midi._enable_sfx ^= 1;
		else
			_sound->effectsPause(_effectsPaused ^= 1);
	}

	loadGamePcFile();

	addTimeEvent(0, 1);
	openGameFile();

	_lastMusicPlayed = -1;
	_frameRate = 1;

	if (getGameType() == GType_FF)
		_numTextBoxes = 40;
	else
		_numTextBoxes = 20;

	_variableArrayPtr = _variableArray;

	_startMainScript = false;
	_continousMainScript = false;
	_startVgaScript = false;
	_continousVgaScript = false;
	_drawImagesDebug = false;

	if (gDebugLevel == 2)
		_continousMainScript = true;
	if (gDebugLevel == 3)
		_continousVgaScript = true;
	if (gDebugLevel == 4)
		_startMainScript = true;
	if (gDebugLevel == 5)
		_startVgaScript = true;

	while (1) {
		hitarea_stuff();
		handleVerbClicked(_verbHitArea);
		delay(100);
	}

	return 0;
}

void SimonEngine::shutdown() {
	delete _gameFile;

	midi.close();

	free(_stringTabPtr);
	free(_itemArrayPtr);
	free(_itemHeapPtr - _itemHeapCurPos);
	free(_tablesHeapPtr - _tablesHeapCurPos);
	free(_tblList);
	free(_iconFilePtr);
	free(_gameOffsetsPtr);

	_system->quit();
}

void SimonEngine::delay(uint amount) {
	OSystem::Event event;

	uint32 start = _system->getMillis();
	uint32 cur = start;
	uint this_delay, vga_period;

	if (_debugger->isAttached())
		_debugger->onFrame();

	if (_fastMode)
	 	vga_period = 10;
	else if (getGameType() == GType_SIMON2)
		vga_period = 45 * _speed;
	else
		vga_period = 50 * _speed;

	_rnd.getRandomNumber(2);

	do {
		while (!_inCallBack && cur >= _lastVgaTick + vga_period && !_pause) {
			_lastVgaTick += vga_period;

			// don't get too many frames behind
			if (cur >= _lastVgaTick + vga_period * 2)
				_lastVgaTick = cur;

			_inCallBack = true;
			timer_callback();
			_inCallBack = false;
		}

		while (_system->pollEvent(event)) {
			switch (event.type) {
			case OSystem::EVENT_KEYDOWN:
				if (event.kbd.keycode >= '0' && event.kbd.keycode <='9'
					&& (event.kbd.flags == OSystem::KBD_ALT ||
						event.kbd.flags == OSystem::KBD_CTRL)) {
					_saveLoadSlot = event.kbd.keycode - '0';

					// There is no save slot 0
					if (_saveLoadSlot == 0)
						_saveLoadSlot = 10;

					sprintf(_saveLoadName, "Quicksave %d", _saveLoadSlot);
					_saveLoadType = (event.kbd.flags == OSystem::KBD_ALT) ? 1 : 2;

					// We should only allow a load or save when it was possible in original
					// This stops load/save during copy protection, conversations and cut scenes
					if (!_mouseHideCount && !_showPreposition)
						quickLoadOrSave();
				} else if (event.kbd.flags == OSystem::KBD_CTRL) {
					if (event.kbd.keycode == 'a') {
						GUI::Dialog *_aboutDialog;
						_aboutDialog = new GUI::AboutDialog();
						_aboutDialog->runModal();
					} else if (event.kbd.keycode == 'f')
						_fastMode ^= 1;
					else if (event.kbd.keycode == 'd')
						_debugger->attach();
				}
				// Make sure backspace works right (this fixes a small issue on OS X)
				if (event.kbd.keycode == 8)
					_keyPressed = 8;
				else
					_keyPressed = (byte)event.kbd.ascii;
				break;
			case OSystem::EVENT_MOUSEMOVE:
				_sdlMouseX = event.mouse.x;
				_sdlMouseY = event.mouse.y;
				break;
			case OSystem::EVENT_LBUTTONDOWN:
				if (getGameType() == GType_FF)
					setBitFlag(89, true);
				_leftButtonDown++;
#if defined (_WIN32_WCE) || defined(PALMOS_MODE)
				_sdlMouseX = event.mouse.x;
				_sdlMouseY = event.mouse.y;
#endif
				break;
			case OSystem::EVENT_LBUTTONUP:
				if (getGameType() == GType_FF)
					setBitFlag(89, false);
				break;
			case OSystem::EVENT_RBUTTONDOWN:
				if (getGameType() == GType_FF)
					setBitFlag(92, false);
				if (getGameType() == GType_SIMON2 || getGameType() == GType_FF)
					_rightButtonDown++;
				else
					_exitCutscene = true;
				break;
			case OSystem::EVENT_QUIT:
				shutdown();
				return;
				break;
			default:
				break;
			}
		}

		if (amount == 0)
			break;

		{
			this_delay = _fastMode ? 1 : 20 * _speed;
			if (this_delay > amount)
				this_delay = amount;
			_system->delayMillis(this_delay);
		}
		cur = _system->getMillis();
	} while (cur < start + amount);
}

void SimonEngine::loadMusic(uint music) {
	char buf[4];

	if (getPlatform() == Common::kPlatformAmiga) {
		if (getFeatures() & GF_CRUNCHED) {
			// TODO Add support for decruncher
			debug(5,"loadMusic - Decrunch %dtune attempt", music);
		}
		// TODO Add Protracker support for simon1amiga/cd32
		debug(5,"playMusic - Load %dtune attempt", music);
	} else if (getGameType() == GType_SIMON2) {        // Simon 2 music
		midi.stop();
		_gameFile->seek(_gameOffsetsPtr[MUSIC_INDEX_BASE + music - 1], SEEK_SET);
		_gameFile->read(buf, 4);
		if (!memcmp(buf, "FORM", 4)) {
			_gameFile->seek(_gameOffsetsPtr[MUSIC_INDEX_BASE + music - 1], SEEK_SET);
			midi.loadXMIDI (_gameFile);
		} else {
			_gameFile->seek(_gameOffsetsPtr[MUSIC_INDEX_BASE + music - 1], SEEK_SET);
			midi.loadMultipleSMF (_gameFile);
		}

		_lastMusicPlayed = music;
		_nextMusicToPlay = -1;
	} else if (getGameType() == GType_SIMON1) {        // Simon 1 music
		midi.stop();
		midi.setLoop (true); // Must do this BEFORE loading music. (GMF may have its own override.)

		if (getFeatures() & GF_TALKIE) {
			// FIXME: The very last music resource, a cymbal crash for when the
			// two demons crash into each other, should NOT be looped like the
			// other music tracks. In simon1dos/talkie the GMF resource includes
			// a loop override that acomplishes this, but there seems to be nothing
			// for this in the SMF resources.
			if (music == 35)
				midi.setLoop (false);

			_gameFile->seek(_gameOffsetsPtr[MUSIC_INDEX_BASE + music], SEEK_SET);
			_gameFile->read(buf, 4);
			if (!memcmp(buf, "GMF\x1", 4)) {
				_gameFile->seek(_gameOffsetsPtr[MUSIC_INDEX_BASE + music], SEEK_SET);
				midi.loadSMF (_gameFile, music);
			} else {
				_gameFile->seek(_gameOffsetsPtr[MUSIC_INDEX_BASE + music], SEEK_SET);
				midi.loadMultipleSMF (_gameFile);
			}

		} else {
			char filename[15];
			File f;
			sprintf(filename, "MOD%d.MUS", music);
			f.open(filename);
			if (f.isOpen() == false) {
				warning("Can't load music from '%s'", filename);
				return;
			}
			if (getGameId() == GID_SIMON1DEMO)
				midi.loadS1D (&f);
			else
				midi.loadSMF (&f, music);
		}

		midi.startTrack (0);
	}
}

byte *SimonEngine::getFrontBuf() {
	_dxSurfacePitch = _screenWidth;
	return _frontBuf;
}

byte *SimonEngine::getBackBuf() {
	_dxSurfacePitch = _screenWidth;
	return _useBackGround ? _backGroundBuf : _backBuf;
}

byte *SimonEngine::getBackGround() {
	_dxSurfacePitch = _screenWidth;
	return _backGroundBuf;
}

byte *SimonEngine::getScaleBuf() {
	_dxSurfacePitch = _screenWidth;
	return _sdl_buf_scaled;
}

void SimonEngine::set_volume(int volume) {
	_mixer->setVolumeForSoundType(Audio::Mixer::kSFXSoundType, volume);
}

byte SimonEngine::getByte() {
	return *_codePtr++;
}

} // End of namespace Simon

#ifdef PALMOS_68K
#include "scumm_globals.h"

_GINIT(Simon_Simon)
_GSETPTR(Simon::simon1_settings, GBVARS_SIMON1SETTINGS_INDEX, Simon::GameSpecificSettings, GBVARS_SIMON)
_GSETPTR(Simon::simon1acorn_settings, GBVARS_SIMON1ACORNSETTINGS_INDEX, Simon::GameSpecificSettings, GBVARS_SIMON)
_GSETPTR(Simon::simon1amiga_settings, GBVARS_SIMON1AMIGASETTINGS_INDEX, Simon::GameSpecificSettings, GBVARS_SIMON)
_GSETPTR(Simon::simon1demo_settings, GBVARS_SIMON1DEMOSETTINGS_INDEX, Simon::GameSpecificSettings, GBVARS_SIMON)
_GSETPTR(Simon::simon2win_settings, GBVARS_SIMON2WINSETTINGS_INDEX, Simon::GameSpecificSettings, GBVARS_SIMON)
_GSETPTR(Simon::simon2dos_settings, GBVARS_SIMON2DOSSETTINGS_INDEX, Simon::GameSpecificSettings, GBVARS_SIMON)
_GSETPTR(Simon::feeblefiles_settings, GBVARS_FEEBLEFILESSETTINGS_INDEX, Simon::GameSpecificSettings, GBVARS_SIMON)
_GEND

_GRELEASE(Simon_Simon)
_GRELEASEPTR(GBVARS_SIMON1SETTINGS_INDEX, GBVARS_SIMON)
_GRELEASEPTR(GBVARS_SIMON1ACORNSETTINGS_INDEX, GBVARS_SIMON)
_GRELEASEPTR(GBVARS_SIMON1AMIGASETTINGS_INDEX, GBVARS_SIMON)
_GRELEASEPTR(GBVARS_SIMON1DEMOSETTINGS_INDEX, GBVARS_SIMON)
_GRELEASEPTR(GBVARS_SIMON2WINSETTINGS_INDEX, GBVARS_SIMON)
_GRELEASEPTR(GBVARS_SIMON2DOSSETTINGS_INDEX, GBVARS_SIMON)
_GRELEASEPTR(GBVARS_FEEBLEFILESSETTINGS_INDEX, GBVARS_SIMON)
_GEND

#endif
