/* ScummVMDS - Scumm Interpreter DS Port
 * Copyright (C) 2002-2004 The ScummVM project and Neil Millstone
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */
#include <NDS.h>
#include "touchkeyboard.h"
#include "keyboard_raw.h"
#include "keyboard_pal_raw.h"
#include "dsmain.h"
#include "osystem_ds.h"

struct key_data {
	char keyNum;
	char x, y;
	int character;
	bool pressed;
};

#define MT_NUM_KEYS 60
#define MT_SHIFT 0
#define MT_BACKSPACE 8
#define MT_RETURN 13
#define MT_CAPSLOCK 1


key_data MT_keys[MT_NUM_KEYS] = {
	// Key number		x		y		character
	
	// Numbers
	{28,				3,		0,		'1'},
	{29,				5,		0,		'2'},
	{30,				7,		0,		'3'},
	{31,				9,		0,		'4'},
	{32,				11,		0,		'5'},
	{33,				13,		0,		'6'},
	{34,				15,		0,		'7'},
	{35,				17,		0,		'8'},
	{36,				19,		0,		'9'},
	{27,				21,		0,		'0'},
	{45,				23,		0,		SDLK_MINUS},
	{50,				25,		0,		SDLK_EQUALS},
	{52,				27,		0,		SDLK_BACKSPACE},

	// Top row
	{'Q'-'A' + 1,		4,		2,		'Q'},
	{'W'-'A' + 1,		6,		2,		'W'},
	{'E'-'A' + 1,		8,		2,		'E'},
	{'R'-'A' + 1,		10,		2,		'R'},
	{'T'-'A' + 1,		12,		2,		'T'},
	{'Y'-'A' + 1,		14,		2,		'Y'},
	{'U'-'A' + 1,		16,		2,		'U'},
	{'I'-'A' + 1,		18,		2,		'I'},
	{'O'-'A' + 1,		20,		2,		'O'},
	{'P'-'A' + 1,		22,		2,		'P'},
	{43,				24,		2,		SDLK_LEFTBRACKET},
	{44,				26,		2,		SDLK_RIGHTBRACKET},

	// Middle row
	{55,				3,		4,		MT_CAPSLOCK},
	{'A'-'A' + 1,		5,		4,		'A'},
	{'S'-'A' + 1,		7,		4,		'S'},
	{'D'-'A' + 1,		9,		4,		'D'},
	{'F'-'A' + 1,		11,		4,		'F'},
	{'G'-'A' + 1,		13,		4,		'G'},
	{'H'-'A' + 1,		15,		4,		'H'},
	{'J'-'A' + 1,		17,		4,		'J'},
	{'K'-'A' + 1,		19,		4,		'K'},
	{'L'-'A' + 1,		21,		4,		'L'},
	{42,				23,		4,		SDLK_SEMICOLON},
	{41,				25,		4,		SDLK_QUOTE},
	{46,				27,		4,		SDLK_RETURN},

	// Bottom row
	{51,				4,		6,		MT_SHIFT},
	{'Z'-'A' + 1,		6,		6,		'Z'},
	{'X'-'A' + 1,		8,		6,		'X'},
	{'C'-'A' + 1,		10,		6,		'C'},
	{'V'-'A' + 1,		12,		6,		'V'},
	{'B'-'A' + 1,		14,		6,		'B'},
	{'N'-'A' + 1,		16,		6,		'N'},
	{'M'-'A' + 1,		18,		6,		'M'},
	{38,				20,		6,		SDLK_COMMA},
	{39,				22,		6,		SDLK_PERIOD},
	{40,				24,		6,		SDLK_SLASH},

	// Space bar
	{47,				9,		8,		SDLK_SPACE},
	{48,				11,		8,		SDLK_SPACE},
	{48,				13,		8,		SDLK_SPACE},
	{48,				15,		8,		SDLK_SPACE},
	{48,				17,		8,		SDLK_SPACE},
	{49,				19,		8,		SDLK_SPACE},

	// Cursor arrows
	{52,				27,		8,		SDLK_LEFT},
	{54,				29,		8,		SDLK_DOWN},
	{53,				31,		8,		SDLK_RIGHT},
	{51,				29,		6,		SDLK_UP},
	
	// Close button
	{56,				30,		0,		SDLK_UNKNOWN},
	
};

int MT_keyboardX;
int MT_keyboardY;

int MT_mapBase;
int MT_tileBase;

bool MT_shiftState;
bool MT_capsLockState;

bool MT_closed;

void MT_restoreVRAM(int tileBase, int mapBase, u16* saveSpace) {
	for (int r = 0; r < 32 * 32; r++) {
		((u16 *) SCREEN_BASE_BLOCK_SUB(mapBase))[r] = *saveSpace++;
	}
	
	for (int r = 0; r < 4096; r++) {
		((u16 *) CHAR_BASE_BLOCK_SUB(tileBase))[r]	= *saveSpace++;
	}
}

void MT_drawKeyboard(int tileBase, int mapBase, u16* saveSpace) {


	for (int r = 0; r < 32 * 32; r++) {
//		*saveSpace++ = ((u16 *) SCREEN_BASE_BLOCK_SUB(mapBase))[r];
		((u16 *) SCREEN_BASE_BLOCK_SUB(mapBase))[r] = 127;
	}
	
	for (int r = 0; r < 4096; r++) {
//		*saveSpace++ = ((u16 *) CHAR_BASE_BLOCK_SUB(tileBase))[r];
		((u16 *) CHAR_BASE_BLOCK_SUB(tileBase))[r] = ((u16 *) (keyboard_raw))[r];
	}
	
	for (int r = 0; r < 16; r++) {
		BG_PALETTE_SUB[r] = ((u16 *) (keyboard_pal_raw))[r];
	}

	for (int r = 0; r < 16; r++) {
		int col = ((u16 *) (keyboard_pal_raw))[r];
		
		int red = col & 0x001F;
		int green = (col & 0x03E0) >> 5;
		int blue = (col & 0x7C00) >> 10;
		
		red = (red * 8) / 16;
		green = (green * 8) / 16;
		blue = (blue * 8) / 16;
				
		BG_PALETTE_SUB[16 + r] = red | (green << 5) | (blue << 10);
	}
	
	MT_keyboardX = -2;
	MT_keyboardY = 2;
	
	MT_mapBase = mapBase;
	MT_tileBase = tileBase;
	
	MT_shiftState = false;
	MT_capsLockState = false;
	
	int x = MT_keyboardX;
	int y = MT_keyboardY;
	
	u16* base = ((u16 *) SCREEN_BASE_BLOCK_SUB(mapBase));
	
	for (int r = 0; r < MT_NUM_KEYS; r++) {
		base[(y + MT_keys[r].y) * 32 + x + MT_keys[r].x] = MT_keys[r].keyNum * 2;
		base[(y + MT_keys[r].y) * 32 + x + MT_keys[r].x + 1] = MT_keys[r].keyNum * 2 + 1;
		
		base[(y + MT_keys[r].y + 1) * 32 + x + MT_keys[r].x] = 128 + MT_keys[r].keyNum * 2;
		base[(y + MT_keys[r].y + 1) * 32 + x + MT_keys[r].x + 1] = 128 + MT_keys[r].keyNum * 2 + 1;
		
		MT_keys[r].pressed = false;
	}
	
	MT_closed = false;
}

bool MT_getKeyboardClosed() {
	return MT_closed;
}

void MT_setKeyHighlight(int key, bool highlight) {
	u16* base = ((u16 *) SCREEN_BASE_BLOCK_SUB(MT_mapBase));

	if (highlight) {
		base[(MT_keyboardY + MT_keys[key].y) * 32 + MT_keyboardX + MT_keys[key].x] |= 0x1000;
		base[(MT_keyboardY + MT_keys[key].y) * 32 + MT_keyboardX + MT_keys[key].x + 1] |= 0x1000;
		base[(MT_keyboardY + MT_keys[key].y + 1) * 32 + MT_keyboardX + MT_keys[key].x] |= 0x1000;
		base[(MT_keyboardY + MT_keys[key].y + 1) * 32 + MT_keyboardX + MT_keys[key].x + 1] |= 0x1000;
	} else {
		base[(MT_keyboardY + MT_keys[key].y) * 32 + MT_keyboardX + MT_keys[key].x] &= ~0x1000;
		base[(MT_keyboardY + MT_keys[key].y) * 32 + MT_keyboardX + MT_keys[key].x + 1] &= ~0x1000;
		base[(MT_keyboardY + MT_keys[key].y + 1) * 32 + MT_keyboardX + MT_keys[key].x] &= ~0x1000;
		base[(MT_keyboardY + MT_keys[key].y + 1) * 32 + MT_keyboardX + MT_keys[key].x + 1] &= ~0x1000;
	}
}

void MT_addKeyboardEvents() {
	if (MT_getPenDown()) {
		int x = IPC->touchXpx;
		int y = IPC->touchYpx;
		
		int tx = (x >> 3) - MT_keyboardX;
		int ty = (y >> 3) - MT_keyboardY;
		
//		consolePrintf("x=%d y=%d\n", tx, ty);
		
		for (int r = 0; r < MT_NUM_KEYS; r++) {
			if (( (tx >= MT_keys[r].x) && (tx <= MT_keys[r].x + 1)) && 
				   (ty >= MT_keys[r].y) && (ty <= MT_keys[r].y + 1)) {
			   	OSystem_DS* system = OSystem_DS::instance();
				OSystem::Event event;
				
//				consolePrintf("Key: %d\n", r);
				if ((MT_keys[r].character == SDLK_UNKNOWN)) {
					// Close button
					MT_closed = true;
				} else	if ((MT_keys[r].character >= '0') && (MT_keys[r].character <= '9')) {
					event.kbd.ascii = MT_keys[r].character;
					event.kbd.keycode = 0;
				
				} else if ((MT_keys[r].character >= 'A') && (MT_keys[r].character <= 'Z')) {
					
					if ((!MT_shiftState) && (!MT_capsLockState)) {
						event.kbd.ascii = MT_keys[r].character + 32; // Make key lowercase.
					} else {
						event.kbd.ascii = MT_keys[r].character;
					}
					
					event.kbd.keycode = event.kbd.ascii;
				} else {
					event.kbd.ascii = MT_keys[r].character;
					event.kbd.keycode = MT_keys[r].character;
				}				
			
				
				
				//event.kbd.keycode = MT_keys[r].character;		
				//event.kbd.ascii = MT_keys[r].character;		
				event.type = OSystem::EVENT_KEYDOWN;
				event.kbd.flags = 0;
				system->addEvent(event);

				event.type = OSystem::EVENT_KEYUP;
				system->addEvent(event);
				
				switch (MT_keys[r].character) {
					case MT_SHIFT: {
						MT_shiftState = !MT_shiftState;
						MT_setKeyHighlight(r, MT_shiftState);
						break;
					}
					
					case MT_CAPSLOCK: {
						MT_capsLockState = !MT_capsLockState;
						MT_setKeyHighlight(r, MT_capsLockState);
						break;
					}
					
					default: {
						MT_setKeyHighlight(r, true);
						MT_keys[r].pressed = true;
						
						if (MT_shiftState) {
							MT_shiftState = false;
							for (int t = 0; t < MT_NUM_KEYS; t++) {
								if (MT_keys[t].character == MT_SHIFT) {
									MT_setKeyHighlight(t, false);
								}
							}
						}
						break;
					}
				}
				
			}
		}
	}
	
	if (MT_getPenReleased()) {
		for (int r = 0; r < MT_NUM_KEYS; r++) {
			if (MT_keys[r].pressed) {
				MT_setKeyHighlight(r, false);
				MT_keys[r].pressed = false;
			}
		}	
	}
}


