/* ScummVM - Scumm Interpreter
 * Copyright (C) 2001-2006 The ScummVM project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * $URL: https://svn.sourceforge.net/svnroot/scummvm/scummvm/trunk/sound/softsynth/emumidi.h $
 * $Id: emumidi.h 20515 2006-02-11 10:05:31Z fingolfin $
 */

#include "common/stdafx.h"
#include "sound/audiostream.h"
#include "sound/mididrv.h"
#include "sound/mixer.h"

#define FIXP_SHIFT 16

class MidiDriver_Emulated : public AudioStream, public MidiDriver {
protected:
	bool _isOpen;
	Audio::Mixer *_mixer;

private:
	Common::Timer::TimerProc _timerProc;
	void *_timerParam;

	int _nextTick;
	int _samplesPerTick;

protected:
	virtual void generateSamples(int16 *buf, int len) = 0;
	virtual void onTimer() {}

	int _baseFreq;

public:
	MidiDriver_Emulated(Audio::Mixer *mixer) : _mixer(mixer) {
		_isOpen = false;

		_timerProc = 0;
		_timerParam = 0;

		_nextTick = 0;
		_samplesPerTick = 0;

		_baseFreq = 250;
	}

	int open() {
		_isOpen = true;

		int d = getRate() / _baseFreq;
		int r = getRate() % _baseFreq;

		// This is equivalent to (getRate() << FIXP_SHIFT) / BASE_FREQ
		// but less prone to arithmetic overflow.

		_samplesPerTick = (d << FIXP_SHIFT) + (r << FIXP_SHIFT) / _baseFreq;
		return 0;
	}

	void setTimerCallback(void *timer_param, Common::Timer::TimerProc timer_proc) {
		_timerProc = timer_proc;
		_timerParam = timer_param;
	}

	uint32 getBaseTempo() { return 1000000 / _baseFreq; }


	// AudioStream API
	int readBuffer(int16 *data, const int numSamples) {
		const int stereoFactor = isStereo() ? 2 : 1;
		int len = numSamples / stereoFactor;
		int step;

		do {
			step = len;
			if (step > (_nextTick >> FIXP_SHIFT))
				step = (_nextTick >> FIXP_SHIFT);

			generateSamples(data, step);

			_nextTick -= step << FIXP_SHIFT;
			if (!(_nextTick >> FIXP_SHIFT)) {
				if (_timerProc)
					(*_timerProc)(_timerParam);
				onTimer();
				_nextTick += _samplesPerTick;
			}
			data += step * stereoFactor;
			len -= step;
		} while (len);

		return numSamples;
	}
	bool endOfData() const { return false; }
};
