/* ScummVMDS - Scumm Interpreter DS Port
 * Copyright (C) 2002-2004 The ScummVM project and Neil Millstone
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */
 
 /*
 
To do:
- Top screen zoom mode - Done
- Sound speed-ups - Done
- Filesystem speed-ups for GBAMP - Done
- Lid close sleep mode - Done
- Sfx speed-ups - Done
- Fixed screen swapping on M3 - Done
- Put savegames in save folder - Done
- Fix dialog skip when scrolling screen - Done
- Hover mode left/right click (suggested by *****) - Done
- Two directional scrolling in S&M - Done
- Check right click in S&M - OK
- Console VRAM copy on mode change - Done
- Implement shake - Done


- The top screen now shows a zoomed in view of the action, scrolled to the character who's speaking
- Large speed increase for Adlib music emulation
- Large speed increase for GBAMP CF card loading
- Closing the lid now puts the DS in sleep mode
- Screens are now the right way round on the latest firmware for the M3 player
- Savegames now save in the selected folder rather than the root
- Dialog no longer skips when you scroll the screen down as it's playing
- The console no longer appears during gameplay by default, toggle it using the Y button.
- Implemented screen shake function

- Fix zak pausing - Done
- Add FM-TOWNS music support - Done
- Screen swap during game - Done
- Keyboard - Done
- Scrolling/zooming with L/R and drag - Done
- Lefty mode - Done
- Help screen - Done
- Support M3 and Supercard - Done
- Fix suspend - Done
- Fix screens swapping when save return - Done
- Sound volume improved - Done
- Fix zip file reading - can't find problem
- Fix keys when l held - Done
- Fix sound when suspended - Done
- Fix pause menu size - Done
- Button for on-screen keyboard - Done
- Fix ini file corruption - Done
- Sound volume improved - Done
- Touch screen controls improved - Done
- Fix zooming to characters - Done
- CD audio - Done
- D-pad in fe - Done
- Start a new game needs removing - Done
- Check delete option - Done

What's new in v0.5
------------------
- CD audio is supported using IMA-ADPCM WAV files.
- Direct file system support on the M3 and Supercard
- An on-screen keyboard to name saves
- Interact with the zoomed view by swapping the screens over with X
- Zoom in/out by holding L and pressing A or B
- Scroll the screen with the pen while holding L
- Improved the way the zoormed view follows the talking character
- Improved the way the zoomed view follows the talking character
- Music support in FM TOWNS games
- Sound volume improved, and sound glitching fixed
- Left handed mode
- Help screen added
- Numerous other bug fixes

- Fixed Loom CD audio - Done
- Moved to 0.9.0 source - Done
- Make keyboard work in 8-bit - Done
- Fixed building issues - Done
- Fix queen palette issues - Done
- Use differenet config file for non-scumm games - Done
- Fixed touchscreen to latest libnds - Done
- Add key bindings for simon - F10 (show hotspots) - Done
- Add control scheme for sky, and others - Done
- Redo left handed mode and indy fighting and help screen - Done
- Fix screen size in FMTOWNS - Done
- SD cards? - done
- Fix dma glitches - Done
- Delete saves - Done
- Fix zooming with A key - Done
- make indy fight controls work - Done
- Ini option to save using SRAM - done
- Disable autosave on flashcart - Done
- Zoom to character on other engines - done
- Remove consolePrintf's - done


- Added MP3 compressed speech support (although it's slow)
- Fixed save path bug
- Improved touchscreen accuracy
- Fixed GBA flashcart reading bug
- Fixed some graphical glitches
loom audio


- Keyboard and font corrupted when data appears at misaligned address - must fix!
*/

#include <nds.h>

#include <ARM9/console.h> //basic print funcionality

#include <stdlib.h>
#include "dsmain.h"
#include "string.h"
#include "system.h"
#include "osystem_ds.h"
#include "icons_raw.h"
#include "gba_nds_fat.h"
#include "disc_io.h"
#include "config-manager.h"
#include "engines/scumm/scumm.h"
#include "keyboard_raw.h"
#include "keyboard_pal_raw.h"
#define V16(a, b) ((a << 12) | b)
#include "touchkeyboard.h"
#include "registers_alt.h"
//#include "compact_flash.h"
#include "dsoptions.h"



// From console.c in NDSLib

//location of cursor
extern u8 row;
extern u8 col;

// Mouse mode
enum MouseMode {
	MOUSE_LEFT, MOUSE_RIGHT, MOUSE_HOVER, MOUSE_NUM_MODES
};

// Defines
#define FRAME_TIME 17
#define SCUMM_GAME_HEIGHT 142
#define SCUMM_GAME_WIDTH 232

int textureID;
u16* texture;

int MT_frameCount;
int MT_currentTimeMillis;

// Timer Callback
int MT_callbackInterval;
int MT_callbackTimer;
OSystem::TimerProc MT_callback;

// Scaled
bool MT_scaledMode;
int MT_scX;
int MT_scY;

int MT_subScX;
int MT_subScY;
int MT_subScTargetX;
int MT_subScTargetY;
int MT_subScreenWidth = SCUMM_GAME_WIDTH;
int MT_subScreenHeight = SCUMM_GAME_HEIGHT;
int MT_subScreenScale = 256;



// Sound
int MT_bufferSize;
s16* MT_soundBuffer;
int MT_bufferFrame;
int MT_bufferRate;
int MT_bufferSamples;
bool MT_soundHiPart;

// Events
int MT_lastEventFrame;
bool MT_indyFightState;
bool MT_indyFightRight;

OSystem::SoundProc MT_soundCallback;
void* MT_soundParam;
int MT_lastCallbackFrame;
bool MT_bufferFirstHalf;
bool MT_bufferSecondHalf;

// Saved buffers
u8* MT_savedBuffer = NULL;
bool MT_highBuffer;
bool MT_displayModeIs8Bit;

// Game id
u8 MT_gameID;

bool MT_consoleEnable = false;
bool MT_gameScreenSwap = false;

MouseMode MT_mouseMode;

// Sprites
SpriteEntry sprites[128];
SpriteEntry spritesMain[128];
int tweak;

// Shake
int MT_shakePos = 0;

// Keyboard
bool MT_keyboardEnable = false;
bool MT_leftHandedMode = false;
bool MT_keyboardIcon = false;

// Touch
int MT_touchScX, MT_touchScY, MT_touchX, MT_touchY;

// Dragging
int MT_dragStartX, MT_dragStartY;
bool MT_dragging = false;
int MT_dragScX, MT_dragScY;

// Interface styles
char MT_gameName[32];

// 8-bit surface size
int MT_gameWidth = 320;
int MT_gameHeight = 200;

enum MT_controlType {
	CONT_SCUMM_ORIGINAL,
	CONT_SCUMM_SAMNMAX,
	CONT_SKY,
	CONT_SIMON,
};

struct MT_gameListType {
	char 			gameId[16];
	MT_controlType 	control;
};

#define NUM_SUPPORTED_GAMES 15

MT_gameListType MT_gameList[NUM_SUPPORTED_GAMES] = {
	// Unknown game - use normal SCUMM controls
	{"unknown", 	CONT_SCUMM_ORIGINAL},
	
	// SCUMM games
	{"maniac",		CONT_SCUMM_ORIGINAL},
	{"zak",			CONT_SCUMM_ORIGINAL},
	{"loom",		CONT_SCUMM_ORIGINAL},
	{"indy3",		CONT_SCUMM_ORIGINAL},
	{"atlantis",	CONT_SCUMM_ORIGINAL},
	{"monkey",		CONT_SCUMM_ORIGINAL},
	{"monkey2",		CONT_SCUMM_ORIGINAL},
	{"tentacle",	CONT_SCUMM_ORIGINAL},
	{"samnmax",		CONT_SCUMM_SAMNMAX},
	
	// Non-SCUMM games
	{"sky",			CONT_SKY},
	{"simon1",		CONT_SIMON},
	{"simon2",		CONT_SIMON},
	{"gob1",		CONT_SCUMM_ORIGINAL},
	{"queen",		CONT_SCUMM_ORIGINAL}
};

MT_gameListType* MT_currentGame;

// Stylus
#define ABS(x) ((x)>0?(x):-(x))

bool penDown;
bool penHeld;
bool penReleased;
bool penDownLastFrame;
f32 penX, penY;
int keysDownSaved;
int keysReleasedSaved;

bool penDownSaved;
bool penReleasedSaved;
int penDownFrames;
int MT_touchXOffset = 0;
int MT_touchYOffset = 0;

u16 MT_savedPalEntry255 = RGB15(31, 31, 31);


extern "C" int scummvm_main(int argc, char *argv[]);
void MT_updateStatus();

TransferSound soundControl;

//plays an 8 bit mono sample at 11025Hz
void MT_playSound(const void* data, u32 length, bool loop, bool adpcm, int rate)
{
	
	if (!IPC->soundData) {
		soundControl.count = 0;
	}
	
	soundControl.data[soundControl.count].data = data;
	soundControl.data[soundControl.count].len = length | (loop? 0x80000000: 0x00000000);
	soundControl.data[soundControl.count].rate = rate;		// 367 samples per frame
	soundControl.data[soundControl.count].pan = 64;
	soundControl.data[soundControl.count].vol = 127;
	soundControl.data[soundControl.count].format = adpcm? 2: 0;

	soundControl.count++;

	DC_FlushAll();
	IPC->soundData = &soundControl;
}

void MT_stopSound(int channel) {
	MT_playSound(NULL, 0, false, false, -channel);
}

void MT_updateOAM() {
	DC_FlushAll();
    dmaCopy(sprites, OAM_SUB, 128 * sizeof(SpriteEntry));
    dmaCopy(spritesMain, OAM, 128 * sizeof(SpriteEntry));
}

void MT_setGameSize(int width, int height) {
	MT_gameWidth = width;
	MT_gameHeight = height;
}

void MT_initSprites() {
	for(int i = 0; i < 128; i++) {
	   sprites[i].attribute[0] = ATTR0_DISABLED;
	   sprites[i].attribute[1] = 0;
	   sprites[i].attribute[2] = 0;
	   sprites[i].attribute[3] = 0;
    }
	
	for(int i = 0; i < 128; i++) {
	   spritesMain[i].attribute[0] = ATTR0_DISABLED;
	   spritesMain[i].attribute[1] = 0;
	   spritesMain[i].attribute[2] = 0;
	   spritesMain[i].attribute[3] = 0;
    }
	
	MT_updateOAM();
}


void MT_saveGameBackBuffer() {
#ifdef DISABLE_SCUMM
	if (MT_savedBuffer == NULL) MT_savedBuffer = new u8[MT_gameWidth * MT_gameHeight];
	for (int r = 0; r < 200; r++) {
		memcpy(MT_savedBuffer + (r * MT_gameWidth), ((u8 *) (MT_get8BitBackBuffer())) + (r * 512), MT_gameWidth);
	}
#endif
}

void MT_restoreGameBackBuffer() {
#ifdef DISABLE_SCUMM
	if (MT_savedBuffer) {
		for (int r = 0; r < 200; r++) {
			memcpy(((u8 *) (BG_GFX_SUB)) + (r * 512), MT_savedBuffer + (r * MT_gameWidth), MT_gameWidth);
			memcpy(((u8 *) (MT_get8BitBackBuffer())) + (r * 512), MT_savedBuffer + (r * MT_gameWidth), MT_gameWidth);
		}
		delete MT_savedBuffer;
		MT_savedBuffer = NULL;
	}
#endif

#ifndef DISABLE_SCUMM	
	memset(MT_get8BitBackBuffer(), 0, 512 * 256);
	memset(BG_GFX_SUB, 0, 512 * 256);
	if (Scumm::g_scumm) {
		Scumm::g_scumm->markRectAsDirty(Scumm::kMainVirtScreen, 0, MT_gameWidth - 1, 0, MT_gameHeight - 1, 1);
		Scumm::g_scumm->markRectAsDirty(Scumm::kTextVirtScreen, 0, MT_gameWidth - 1, 0, MT_gameHeight - 1, 1);
		Scumm::g_scumm->markRectAsDirty(Scumm::kVerbVirtScreen, 0, MT_gameWidth - 1, 0, MT_gameHeight - 1, 1);
	}
#endif

}


void MT_initGame() {
	// This is a good time to check for left handed mode since the mode change is done as the game starts.
	// There's probably a better way, but hey.
//	consolePrintf("initing game\n");

	MT_setOptions();

	//strcpy(MT_gameName, ConfMan.getActiveDomain().c_str());
	strcpy(MT_gameName, ConfMan.get("gameid").c_str());
//	consolePrintf("\n\n\n\nCurrent game: '%s' %d\n", MT_gameName, MT_gameName[0]);

	MT_currentGame = &MT_gameList[0];		// Default game
	
	for (int r = 0; r < NUM_SUPPORTED_GAMES; r++) {
		if (!stricmp(MT_gameName, MT_gameList[r].gameId)) {
			MT_currentGame = &MT_gameList[r];
//			consolePrintf("Game list num: %d\n", MT_currentGame);
		}
	}
	

}

void MT_setLeftHanded(bool enable) {
	MT_leftHandedMode = enable;
}

void MT_setTouchXOffset(int x) {
	MT_touchXOffset = x;
}

void MT_setTouchYOffset(int y) {
	MT_touchYOffset = y;
}

void MT_setUnscaledMode(bool enable) {
	MT_scaledMode = !enable;
}

void MT_displayMode8Bit() {

	u16 buffer[32 * 32];
	
	MT_setKeyboardEnable(false);

	if (!MT_displayModeIs8Bit) {
		for (int r = 0; r < 32 * 32; r++) {
			buffer[r] = ((u16 *) SCREEN_BASE_BLOCK_SUB(4))[r];
		}
	}
	
	

	videoSetMode(MODE_5_2D | (MT_consoleEnable? DISPLAY_BG0_ACTIVE: 0) | DISPLAY_BG3_ACTIVE | DISPLAY_SPR_ACTIVE | DISPLAY_SPR_1D | DISPLAY_SPR_1D_BMP); 
	videoSetModeSub(MODE_3_2D /*| DISPLAY_BG0_ACTIVE*/ | DISPLAY_BG3_ACTIVE | DISPLAY_SPR_ACTIVE | DISPLAY_SPR_1D | DISPLAY_SPR_1D_BMP); //sub bg 0 will be used to print text

	vramSetBankA(VRAM_A_MAIN_BG_0x6000000);
	vramSetBankB(VRAM_B_MAIN_BG_0x6020000);

	vramSetBankC(VRAM_C_SUB_BG_0x6200000);
	vramSetBankD(VRAM_D_MAIN_BG_0x6040000);
	
	vramSetBankH(VRAM_H_LCD);

	BG3_CR = BG_BMP8_512x256 | BG_BMP_BASE(8);
	
	
	
	BG3_XDX = (int) (((float) (MT_gameWidth) / 256.0f) * 256);
    BG3_XDY = 0;
    BG3_YDX = 0;
    BG3_YDY = (int) ((200.0f / 192.0f) * 256);

	SUB_BG3_CR = BG_BMP8_512x256;
	
	
	
	SUB_BG3_XDX = (int) (MT_subScreenWidth / 256.0f * 256);
    SUB_BG3_XDY = 0;
    SUB_BG3_YDX = 0;
    SUB_BG3_YDY = (int) (MT_subScreenHeight / 192.0f * 256);


	// Do text stuff
	BG0_CR = BG_MAP_BASE(0) | BG_TILE_BASE(1);
	BG0_Y0 = 0;
	
	// Restore palette entry used by text in the front-end	
	PALETTE_SUB[255] = MT_savedPalEntry255;
	
	consoleInitDefault((u16*)SCREEN_BASE_BLOCK(0), (u16*)CHAR_BASE_BLOCK(1), 16);
	consolePrintSet(0, 23);
	
	if (!MT_displayModeIs8Bit) {
		for (int r = 0; r < 32 * 32; r++) {
			((u16 *) SCREEN_BASE_BLOCK(0))[r] = buffer[r];
		}
//		dmaCopyHalfWords(3, (u16 *) SCREEN_BASE_BLOCK(0), buffer, 32 * 32 * 2);
	}
	
	
	if (!MT_displayModeIs8Bit) MT_restoreGameBackBuffer();
	MT_displayModeIs8Bit = true;
	
	POWER_CR &= ~POWER_SWAP_LCDS;
	
	MT_keyboardEnable = false;
	MT_initGame();
	
}

void MT_setGameID(int id) {
	MT_gameID = id;
}

void MT_dummyHandler() {
	REG_IF = IRQ_VBLANK;
}

void MT_checkSleepMode() {
	if (IPC->performArm9SleepMode) {
	
		consolePrintf("ARM9 Entering sleep mode\n");
		
		int intSave = REG_IE;
		irqSet(IRQ_VBLANK, MT_dummyHandler);
//		int irqHandlerSave = (int) IRQ_HANDLER;
		REG_IE = IRQ_VBLANK;
		//IRQ_HANDLER = MT_dummyHandler;
		
		int powerSave = POWER_CR;
		POWER_CR &= ~POWER_ALL;
		
		while (IPC->performArm9SleepMode) {
			swiWaitForVBlank();
		}
		
		POWER_CR = powerSave;
//		IRQ_HANDLER = (void (*)()) irqHandlerSave;
		irqSet(IRQ_VBLANK, MT_VBlankHandler);
		REG_IE = intSave;

		consolePrintf("ARM9 Waking from sleep mode\n");
	}
}

void MT_setCursorIcon(const u8* icon, uint w, uint h, byte keycolor) {
	if (MT_currentGame->control != CONT_SCUMM_SAMNMAX)
		return;

	uint16 border = RGB15(24,24,24) | 0x8000;
	
	
	int off = 48*64;
	memset(SPRITE_GFX_SUB+off, 0, 64*64*2);
	
	int pos = 190 - (w+2);
	

	
	// make border
	for (uint i=0; i<w+2; i++) {
		SPRITE_GFX_SUB[off+i] = border;
		SPRITE_GFX_SUB[off+(31)*64+i] = border;
	}
	for (uint i=1; i<31; i++) {
		SPRITE_GFX_SUB[off+(i*64)] = border;
		SPRITE_GFX_SUB[off+(i*64)+(w+1)] = border;
	}
	
	int offset = (32 - h) >> 1;

	for (uint y=0; y<h; y++) {
		for (uint x=0; x<w; x++) {
			int color = icon[y*w+x];
			if (color == keycolor) {
				SPRITE_GFX_SUB[off+(y+1+offset)*64+(x+1)] = 0x8000; // black background
			} else {
				SPRITE_GFX_SUB[off+(y+1+offset)*64+(x+1)] = BG_PALETTE[color] | 0x8000;
			}
		}
	}

	sprites[1].attribute[0] = ATTR0_BMP | 150;
	sprites[1].attribute[1] = ATTR1_SIZE_64 | pos;
	sprites[1].attribute[2] = ATTR2_ALPHA(1) | 48;
}




void MT_displayMode16Bit() {

	u16 buffer[32 * 32 * 2];


	if (MT_displayModeIs8Bit) {
		MT_saveGameBackBuffer();
		for (int r = 0; r < 32 * 32; r++) {
			buffer[r] = ((u16 *) SCREEN_BASE_BLOCK(0))[r];
		}
	}


	videoSetMode(MODE_5_2D | /*DISPLAY_BG0_ACTIVE |*/ DISPLAY_BG3_ACTIVE | DISPLAY_SPR_ACTIVE | DISPLAY_SPR_1D | DISPLAY_SPR_1D_BMP); 
	videoSetModeSub(MODE_0_2D | DISPLAY_BG0_ACTIVE |/* DISPLAY_BG1_ACTIVE |*/ DISPLAY_SPR_ACTIVE | DISPLAY_SPR_1D | DISPLAY_SPR_1D_BMP); //sub bg 0 will be used to print text

	vramSetBankA(VRAM_A_MAIN_BG);
	vramSetBankB(VRAM_B_MAIN_BG);
	vramSetBankC(VRAM_C_MAIN_BG);
	vramSetBankD(VRAM_D_MAIN_BG);
	vramSetBankH(VRAM_H_SUB_BG);

	BG3_CR = BG_BMP16_512x256;
	MT_highBuffer = false;
	
	BG3_XDX = (int) (1.25f * 256);
    BG3_XDY = 0;
    BG3_YDX = 0;
    BG3_YDY = (int) ((200.0f / 192.0f) * 256);

	memset(BG_GFX, 0, 512 * 256 * 2);
	
	MT_savedPalEntry255 = PALETTE_SUB[255];
	PALETTE_SUB[255] = RGB15(31,31,31);//by default font will be rendered with color 255

	// Do text stuff
	SUB_BG0_CR = BG_MAP_BASE(4) | BG_TILE_BASE(0);
	SUB_BG0_Y0 = 0;

	consoleInitDefault((u16*)SCREEN_BASE_BLOCK_SUB(4), (u16*)CHAR_BASE_BLOCK_SUB(0), 16);

	if (MT_displayModeIs8Bit) {
		//dmaCopyHalfWords(3, (u16 *) SCREEN_BASE_BLOCK_SUB(0), buffer, 32 * 32 * 2);
		for (int r = 0; r < 32 * 32; r++) {
			((u16 *) SCREEN_BASE_BLOCK_SUB(4))[r] = buffer[r];
		}
	}

	consolePrintSet(0, 23);
	consolePrintf("\n");
	
	// Show keyboard
	SUB_BG1_CR = BG_TILE_BASE(1) | BG_MAP_BASE(12);
	//MT_drawKeyboard(1, 12);
	
	POWER_CR &= ~POWER_SWAP_LCDS;



	MT_displayModeIs8Bit = false;
}


void MT_displayMode16BitFlipBuffer() {
	if (!MT_displayModeIs8Bit) {
		u16* back = MT_get16BitBackBuffer();
	
//		MT_highBuffer = !MT_highBuffer;
//		BG3_CR = BG_BMP16_512x256 |	BG_BMP_RAM(MT_highBuffer? 1: 0);
		
		for (int r = 0; r < 512 * 256; r++) {
			*(BG_GFX + r) = *(back + r);
		}
	}
}

void MT_setShakePos(int shakePos) {
	MT_shakePos = shakePos;
}


u16* MT_get16BitBackBuffer() {
	return BG_GFX + 0x20000;
}

u16* MT_get8BitBackBuffer() {
	return BG_GFX + 0x10000;		// 16bit qty!
}

void MT_setSoundProc(OSystem::SoundProc proc, void* param) {
//	consolePrintf("Set sound callback");
	MT_soundCallback = proc;
	MT_soundParam = param;
}

// The sound system in ScummVM seems to always return stereo interleaved samples.
// Here, I'm treating an 11Khz stereo stream as a 22Khz mono stream, which works sorta ok, but is
// a horrible bodge.  Any advice on how to change the engine to output mono would be greatly
// appreciated.
void MT_doSoundCallback() {
	if ((MT_soundCallback)) {
		MT_lastCallbackFrame = MT_frameCount;
		
		for (int r = IPC->playingSection; r < IPC->playingSection + 4; r++) {
			int chunk = r & 3;
			
			if (IPC->fillNeeded[chunk]) {
				IPC->fillNeeded[chunk] = false;
				DC_FlushAll();
				MT_soundCallback(MT_soundParam, (byte *) (MT_soundBuffer + ((MT_bufferSamples >> 2) * chunk)), MT_bufferSamples >> 1);
				IPC->fillNeeded[chunk] = false;
				DC_FlushAll();
			}
		
		}
		
	}
}

void MT_doTimerCallback() {
	if (MT_callback) {
		if (MT_callbackTimer <= 0) {
			MT_callbackTimer += MT_callbackInterval;
			MT_callback(MT_callbackInterval);
		}	
	}
}

void MT_soundUpdate() {
	if ((MT_bufferFrame == 0)) {
//		MT_playSound(MT_soundBuffer, (MT_bufferSamples * 2), true);
	}
//	consolePrintf("%x\n", IPC->test);
	
	
	if (MT_bufferFrame == 0) {
//		MT_bufferFirstHalf = true;
	}	
	if (MT_bufferFrame == MT_bufferSize >> 1) {
	//MT_bufferSecondHalf = true;
	}	
	
	MT_bufferFrame++;
	if (MT_bufferFrame == MT_bufferSize) {
		MT_bufferFrame = 0;
	}
}

void MT_memoryReport() {
	int r = 0;
	int* p;
	do {
		p = (int *) malloc(r * 8192);
		free(p);
		r++;		
	} while ((p) && (r < 512));
	
	int t = -1;
	void* block[1024];
	do {
		t++;
		block[t] = (int *) malloc(4096);
	} while ((t < 1024) && (block[t]));		
	
	for (int q = 0; q < t; q++) {
		free(block[q]);
	}
	
	consolePrintf("Free: %dK, Largest: %dK\n", t * 4, r * 8);
}


void MT_addIndyFightingKeys() {
	OSystem_DS* system = OSystem_DS::instance();
	OSystem::Event event;

	event.type = OSystem::EVENT_KEYDOWN;
	event.kbd.flags = 0;

	if ((MT_getKeysDown() & KEY_L)) {
		MT_indyFightRight = false;
	}

	if ((MT_getKeysDown() & KEY_R)) {
		MT_indyFightRight = true;
	}

	if ((MT_getKeysDown() & KEY_UP)) {
		event.kbd.keycode = '8';
		event.kbd.ascii = '8';
		system->addEvent(event);
	}
	if ((MT_getKeysDown() & KEY_LEFT)) {
		event.kbd.keycode = '4';
		event.kbd.ascii = '4';
		system->addEvent(event);
	}
	if ((MT_getKeysDown() & KEY_RIGHT)) {
		event.kbd.keycode = '6';
		event.kbd.ascii = '6';
		system->addEvent(event);
	}	
	if ((MT_getKeysDown() & KEY_DOWN)) {
		event.kbd.keycode = '2';
		event.kbd.ascii = '2';
		system->addEvent(event);
	}
	
	if (MT_indyFightRight) {
	
		if ((MT_getKeysDown() & KEY_X)) {
			event.kbd.keycode = '9';
			event.kbd.ascii = '9';
			system->addEvent(event);
		}
		if ((MT_getKeysDown() & KEY_A)) {
			event.kbd.keycode = '6';
			event.kbd.ascii = '6';
			system->addEvent(event);
		}
		if ((MT_getKeysDown() & KEY_B)) {
			event.kbd.keycode = '3';
			event.kbd.ascii = '3';
			system->addEvent(event);
		}	

	} else {

		if ((MT_getKeysDown() & KEY_X)) {
			event.kbd.keycode = '7';
			event.kbd.ascii = '7';
			system->addEvent(event);
		}
		if ((MT_getKeysDown() & KEY_A)) {
			event.kbd.keycode = '4';
			event.kbd.ascii = '4';
			system->addEvent(event);
		}
		if ((MT_getKeysDown() & KEY_B)) {
			event.kbd.keycode = '1';
			event.kbd.ascii = '1';
			system->addEvent(event);
		}	
	
	}
	
	
	if ((MT_getKeysDown() & KEY_Y)) {
		event.kbd.keycode = '5';
		event.kbd.ascii = '5';
		system->addEvent(event);
	}
} 			


void MT_setKeyboardEnable(bool en) {
	if (en == MT_keyboardEnable) return;
	MT_keyboardEnable = en;
	u16* backupBank = (u16 *) 0x6040000;

	if (MT_keyboardEnable) {


		MT_drawKeyboard(1, 12, backupBank);
		
		
		SUB_BG1_CR = BG_TILE_BASE(1) | BG_MAP_BASE(12);

		if (MT_displayModeIs8Bit) {
			SUB_DISPLAY_CR |= DISPLAY_BG1_ACTIVE;	// Turn on keyboard layer
			SUB_DISPLAY_CR &= ~DISPLAY_BG3_ACTIVE;	// Turn off game layer
		} else {
			SUB_DISPLAY_CR |= DISPLAY_BG1_ACTIVE;	// Turn on keyboard layer
			SUB_DISPLAY_CR &= ~DISPLAY_BG0_ACTIVE;	// Turn off console layer
		}
		lcdSwap();
	} else {


		// Restore the palette that the keyboard has used
		for (int r = 0; r < 256; r++) {
			BG_PALETTE_SUB[r] = BG_PALETTE[r];
		}
		
		
		//MT_restoreVRAM(1, 12, backupBank);
		
		if (MT_displayModeIs8Bit) {
			// Copy the sub screen VRAM from the top screen - they should always be
			// the same.
			for (int r = 0; r < (512 * 256) >> 1; r++) {
				BG_GFX_SUB[r] = MT_get8BitBackBuffer()[r];
			}
			
			SUB_DISPLAY_CR &= ~DISPLAY_BG1_ACTIVE;	// Turn off keyboard layer
			SUB_DISPLAY_CR |= DISPLAY_BG3_ACTIVE;	// Turn on game layer
		} else {
			SUB_DISPLAY_CR &= ~DISPLAY_BG1_ACTIVE;	// Turn off keyboard layer
			SUB_DISPLAY_CR |= DISPLAY_BG0_ACTIVE;	// Turn on console layer
		}
		
		lcdSwap();
	}
}

bool MT_getKeyboardEnable() {
	return MT_keyboardEnable;
}

bool MT_getIsDisplayMode8Bit() {
	return MT_displayModeIs8Bit;
}

void MT_addEventsToQueue() {
	OSystem_DS* system = OSystem_DS::instance();
	OSystem::Event event;

	

	
	if (system->isEventQueueEmpty()) {

/*
		if (MT_getKeysDown() & KEY_L) {
			tweak--;
			consolePrintf("Tweak: %d\n", tweak);
			IPC->tweakChanged = true;
		}
		

		if (MT_getKeysDown() & KEY_R) {
			tweak++;
			consolePrintf("Tweak: %d\n", tweak);
			IPC->tweakChanged = true;
		}
	*/
		if ((keysHeld() & KEY_L) && (keysHeld() & KEY_R)) {
			MT_memoryReport();
		}

		if (MT_displayModeIs8Bit) {

			if (!MT_indyFightState) {

				
				if ((MT_getKeysDown() & KEY_B) && (!(MT_getKeysHeld() & KEY_L)) && (!(MT_getKeysHeld() & KEY_R))) {
		//			consolePrintf("Pressing Esc");
		
					event.type = OSystem::EVENT_KEYDOWN;
					event.kbd.keycode = 27;		
					event.kbd.ascii = 27;		
					event.kbd.flags = 0;
					system->addEvent(event);
				}
		
			}
			
		
			
			if ((!MT_getIndyFightState()) && (MT_getKeysDown() & KEY_Y)) {
				MT_consoleEnable = !MT_consoleEnable;
				if (MT_displayModeIs8Bit) {
					MT_displayMode8Bit();
				} else {
					MT_displayMode16Bit();
				}
			}
	
	
			if (!((MT_getKeysHeld() & KEY_L) || (MT_getKeysHeld() & KEY_R)) && (!MT_getIndyFightState())) {

				if ((MT_getKeysDown() & KEY_A) && (!MT_indyFightState)) {
					MT_gameScreenSwap = !MT_gameScreenSwap;
				}
	
				if (!MT_getPenHeld() || (MT_mouseMode != MOUSE_HOVER)) {
					if (MT_getKeysDown() & KEY_LEFT) {
						MT_mouseMode = MOUSE_LEFT;
					}
					if (MT_getKeysDown() & KEY_RIGHT) {
						if (MT_currentGame->control != CONT_SCUMM_SAMNMAX) {
							MT_mouseMode = MOUSE_RIGHT;
						} else {
							// If we're playing sam and max, click and release the right mouse
							// button to change verb
							OSystem::Event event;
		
							event.type = OSystem::EVENT_RBUTTONDOWN;
							event.mouse = Common::Point(MT_getPenX(), MT_getPenY());
							system->addEvent(event);
		
							event.type = OSystem::EVENT_RBUTTONUP;
							system->addEvent(event);
						}
					}
					if (MT_getKeysDown() & KEY_UP) {
						MT_mouseMode = MOUSE_HOVER;
					}
				}
	
					
				
			}

			if ((MT_getKeysDown() & KEY_SELECT)) {
				//MT_scaledMode = !MT_scaledMode;
				//MT_scY = 4;
				MT_showOptionsDialog();
			}

			
		}
	
		if (!MT_getIndyFightState() && !((MT_getKeysHeld() & KEY_L) || (MT_getKeysHeld() & KEY_R)) && (MT_getKeysDown() & KEY_X)) {
			MT_setKeyboardEnable(!MT_keyboardEnable);
		}
		
		MT_updateStatus();			
		
		OSystem::Event event;

		if ((!(MT_getKeysHeld() & KEY_L)) && (!(MT_getKeysHeld() & KEY_R))) {
			event.type = OSystem::EVENT_MOUSEMOVE;
			event.mouse = Common::Point(MT_getPenX(), MT_getPenY());
			system->addEvent(event);
			//consolePrintf("x=%d   y=%d  \n", MT_getPenX(), MT_getPenY());
		}
	
		if (!MT_keyboardEnable) {
			if ((MT_mouseMode != MOUSE_HOVER) || (!MT_displayModeIs8Bit)) {
					if (MT_getPenDown() && (!(MT_getKeysHeld() & KEY_L)) && (!(MT_getKeysHeld() & KEY_R))) {	
						event.type = ((MT_mouseMode == MOUSE_LEFT) || (!MT_displayModeIs8Bit))? OSystem::EVENT_LBUTTONDOWN: OSystem::EVENT_RBUTTONDOWN;
						event.mouse = Common::Point(MT_getPenX(), MT_getPenY());
						system->addEvent(event);
					}
					
					if (MT_getPenReleased()) {
						event.type = MT_mouseMode == MOUSE_LEFT? OSystem::EVENT_LBUTTONUP: OSystem::EVENT_RBUTTONUP;
						event.mouse = Common::Point(MT_getPenX(), MT_getPenY());
						system->addEvent(event);
					}
			} else {
				// In hover mode, D-pad left and right click the mouse when the pen is on the screen
	
				if (MT_getPenHeld()) {
					if (MT_getKeysDown() & KEY_LEFT) {
						event.type = OSystem::EVENT_LBUTTONDOWN;
						event.mouse = Common::Point(MT_getPenX(), MT_getPenY());
						system->addEvent(event);
					}
				/*	if (MT_getKeysReleased() & KEY_LEFT) {
						event.type = OSystem::EVENT_LBUTTONUP;
						event.mouse = Common::Point(MT_getPenX(), MT_getPenY());
						system->addEvent(event);
					}*/
	
					if (MT_getKeysDown() & KEY_RIGHT) {
						event.type = OSystem::EVENT_RBUTTONDOWN;
						event.mouse = Common::Point(MT_getPenX(), MT_getPenY());
						system->addEvent(event);
					}
					/*if (MT_getKeysReleased() & KEY_RIGHT) {
						event.type = OSystem::EVENT_RBUTTONUP;
						event.mouse = Common::Point(MT_getPenX(), MT_getPenY());
						system->addEvent(event);
					}*/
				}
			}
			
			if ((!(MT_getKeysHeld() & KEY_L)) && (!(MT_getKeysHeld() & KEY_R))  && (MT_displayModeIs8Bit)) {
				// Controls specific to the control method
			
			
				if (MT_currentGame->control == CONT_SKY) {
					// Extra controls for Benieth a Steel Sky
					if ((MT_getKeysDown() & KEY_DOWN)) {
						penY = 0;
						penX = 0;		// Show inventory by moving mouse onto top line
					}
				}

				if (MT_currentGame->control == CONT_SIMON) {
					// Extra controls for Simon the Sorcerer
					if ((MT_getKeysDown() & KEY_DOWN)) {
						OSystem::Event event;
					
						event.type = OSystem::EVENT_KEYDOWN;
						event.kbd.keycode = '#';		// F10 or # - show hotspots
						event.kbd.ascii = '#';
						event.kbd.flags = 0;
						system->addEvent(event);
//						consolePrintf("F10\n");
					}
				}
	
				if (MT_currentGame->control == CONT_SCUMM_ORIGINAL) {
					// Extra controls for Scumm v1-5 games
					if ((MT_getKeysDown() & KEY_DOWN)) {
						OSystem::Event event;
					
						event.type = OSystem::EVENT_KEYDOWN;
						event.kbd.keycode = '.';		// Full stop - skips current dialogue line
						event.kbd.ascii = '.';
						event.kbd.flags = 0;
						system->addEvent(event);
					}
					
					if (MT_indyFightState) {
						MT_addIndyFightingKeys();
					}
					
				}
				
			}
		}
		
		if (!MT_displayModeIs8Bit) {
			// Front end controls
			
			if (MT_leftHandedSwap(MT_getKeysDown()) & KEY_UP) {
				event.type = OSystem::EVENT_KEYDOWN;
				event.kbd.keycode = SDLK_UP;
				event.kbd.ascii = 0;
				event.kbd.flags = 0;
				system->addEvent(event);

				event.type = OSystem::EVENT_KEYUP;
				system->addEvent(event);
			}

			if (MT_leftHandedSwap(MT_getKeysDown()) & KEY_DOWN) {
				event.type = OSystem::EVENT_KEYDOWN;
				event.kbd.keycode = SDLK_DOWN;
				event.kbd.ascii = 0;
				event.kbd.flags = 0;
				system->addEvent(event);

				event.type = OSystem::EVENT_KEYUP;
				system->addEvent(event);
			}

			if (MT_leftHandedSwap(MT_getKeysDown()) & KEY_A) {
				event.type = OSystem::EVENT_KEYDOWN;
				event.kbd.keycode = SDLK_RETURN;
				event.kbd.ascii = 0;
				event.kbd.flags = 0;
				system->addEvent(event);

				event.type = OSystem::EVENT_KEYUP;
				system->addEvent(event);
			}
		
		}

		
		if ((MT_getKeysDown() & KEY_START)) {
			event.type = OSystem::EVENT_KEYDOWN;
			event.kbd.keycode = 319;		// F5
			event.kbd.ascii = 319;
			event.kbd.flags = 0;
			system->addEvent(event);
/*
			event.type = OSystem::EVENT_KEYUP;
			event.kbd.keycode = 319;		// F5
			event.kbd.ascii = 319;
			system->addEvent(event);*/
			
//			consolePrintf("Pressing F5");
		}

		
		if (MT_keyboardEnable) {
			MT_addKeyboardEvents();
		}
		
		MT_consumeKeys();
		
		MT_consumePenEvents();

	}
}

void MT_updateStatus() {
	int offs;

	if (MT_displayModeIs8Bit) {
		switch (MT_mouseMode) {
			case MOUSE_LEFT: {
				offs = 16;
				break;
			}
			case MOUSE_RIGHT: {
				offs = 32;
				break;
			}
			case MOUSE_HOVER: {
				offs = 0;
				break;
			}
			default: {
				// Nothing!
				break;
			}
		}
	
		
		sprites[0].attribute[0] = ATTR0_BMP | 150; 
		sprites[0].attribute[1] = ATTR1_SIZE_32 | 208;
		sprites[0].attribute[2] = ATTR2_ALPHA(1)| offs;
	
		if (MT_indyFightState) {
			sprites[2].attribute[0] = ATTR0_BMP | 150; 
			sprites[2].attribute[1] = ATTR1_SIZE_32 | (190 - 32) | (MT_indyFightRight? 0: ATTR1_FLIP_X);
			sprites[2].attribute[2] = ATTR2_ALPHA(1)| 48;
		} else {
			sprites[2].attribute[0] = ATTR0_DISABLED; 
			sprites[2].attribute[1] = 0;
			sprites[2].attribute[2] = 0;
		}
	} else {
		sprites[0].attribute[0] = ATTR0_DISABLED; 
		sprites[1].attribute[0] = ATTR0_DISABLED; 
		sprites[2].attribute[0] = ATTR0_DISABLED; 
		sprites[3].attribute[0] = ATTR0_DISABLED; 
	}

	if ((MT_keyboardIcon) && (!MT_keyboardEnable) && (!MT_displayModeIs8Bit)) {
		spritesMain[0].attribute[0] = ATTR0_BMP | 160;
		spritesMain[0].attribute[1] = ATTR1_SIZE_32 | 0;
		spritesMain[0].attribute[2] = ATTR2_ALPHA(1) | 64;
	} else {
		spritesMain[0].attribute[0] = ATTR0_DISABLED;
		spritesMain[0].attribute[1] = 0;
		spritesMain[0].attribute[2] = 0;
		spritesMain[0].attribute[3] = 0;
	}

}

void MT_soundBufferEmptyHandler() {
	REG_IF = IRQ_TIMER2;

	if (MT_soundHiPart) {
//		MT_bufferSecondHalf = true;
	} else {
//		MT_bufferFirstHalf = true;
	}
	
	MT_soundHiPart = !MT_soundHiPart;
}

void MT_setMainScreenScroll(int x, int y) {
	if (MT_gameScreenSwap) {
		SUB_BG3_CX = x + (((MT_frameCount & 1) == 0)? 64: 0);
		SUB_BG3_CY = y;
	} else {
		BG3_CX = x + (((MT_frameCount & 1) == 0)? 64: 0);
		BG3_CY = y;
		
		MT_touchX = x >> 8;
		MT_touchY = y >> 8;
	}
}

void MT_setMainScreenScale(int x, int y) {
	if (MT_gameScreenSwap) {
		SUB_BG3_XDX = x;
		SUB_BG3_XDY = 0;
		SUB_BG3_YDX = 0;
		SUB_BG3_YDY = y;
	} else {
		BG3_XDX = x;
		BG3_XDY = 0;
		BG3_YDX = 0;
		BG3_YDY = y;
		
		MT_touchScX = x;
		MT_touchScY = y;
	}
}

void MT_setZoomedScreenScroll(int x, int y) {
	if (MT_gameScreenSwap) {
		BG3_CX = x + (((MT_frameCount & 1) == 0)? 64: 0);
		BG3_CY = y;
		
		MT_touchX = x >> 8;
		MT_touchY = y >> 8;
	} else {
		SUB_BG3_CX = x + (((MT_frameCount & 1) == 0)? 64: 0);
		SUB_BG3_CY = y;
	}
}

void MT_setZoomedScreenScale(int x, int y) {
	if (MT_gameScreenSwap) {
		BG3_XDX = x;
		BG3_XDY = 0;
		BG3_YDX = 0;
		BG3_YDY = y;

		MT_touchScX = x;
		MT_touchScY = y;
	} else {
		SUB_BG3_XDX = x;
		SUB_BG3_XDY = 0;
		SUB_BG3_YDX = 0;
		SUB_BG3_YDY = y;
	}
}

void MT_VBlankHandler(void) {
//	BG_PALETTE[0] = RGB15(31, 31, 31);
//	if (*((int *) (0x023FFF00)) != 0xBEEFCAFE) {
	//	consolePrintf("Guard band overwritten!");
//  }

//	consolePri ntf("X:%d Y:%d\n", MT_getPenX(), MT_getPenY());



	IPC->tweak = tweak;
	MT_soundUpdate();

	if ((!MT_gameScreenSwap) && (!(MT_getKeysHeld() & KEY_L) && !(MT_getKeysHeld() & KEY_R))) {
		if (MT_currentGame->control != CONT_SCUMM_SAMNMAX) {
			if (MT_getPenHeld() && (MT_getPenY() < SCUMM_GAME_HEIGHT)) {
				MT_setTopScreenTarget(MT_getPenX(), MT_getPenY());
			}
		} else {
			if (MT_getPenHeld()) {
				MT_setTopScreenTarget(MT_getPenX(), MT_getPenY());
			}
		}
	}
	
	

	MT_penUpdate();
	MT_keysUpdate();


	MT_frameCount++;
	


	if (MT_callback) {
		MT_callbackTimer -= FRAME_TIME;
	}
	
	if ((MT_getKeysHeld() & KEY_L) || (MT_getKeysHeld() & KEY_R)) {
	
		if ((!MT_dragging) && (MT_getPenHeld()) && (penDownFrames > 5)) {
			MT_dragging = true;
			MT_dragStartX = penX;
			MT_dragStartY = penY;
			
			if (MT_gameScreenSwap) {
				MT_dragScX = MT_subScTargetX;
				MT_dragScY = MT_subScTargetY;
			} else {
				MT_dragScX = MT_scX;
				MT_dragScY = MT_scY;				
			}	
			
			
		}
		
		if ((MT_dragging) && (!MT_getPenHeld())) {
			MT_dragging = false;
		}
		
		if (MT_dragging) {
		
			if (MT_gameScreenSwap) {
				MT_subScTargetX = MT_dragScX + ((MT_dragStartX - penX) << 8);
				MT_subScTargetY = MT_dragScY + ((MT_dragStartY - penY) << 8);
			} else {
				MT_scX = MT_dragScX + ((MT_dragStartX - penX));
				MT_scY = MT_dragScY + ((MT_dragStartY - penY));
			}
			
//			consolePrintf("X:%d Y:%d\n", MT_dragStartX - penX, MT_dragStartY - penY);
		}
	}	
	
	
/*	if ((MT_frameCount & 1) == 0) {
		SUB_BG3_CX = MT_subScX;
	} else {
		SUB_BG3_CX = MT_subScX + 64;
	}
	
	SUB_BG3_CY = MT_subScY + (MT_shakePos << 8);*/

	/*SUB_BG3_XDX = (int) (MT_subScreenWidth / 256.0f * 256);
    SUB_BG3_XDY = 0;
    SUB_BG3_YDX = 0;
    SUB_BG3_YDY = (int) (MT_subScreenHeight / 192.0f * 256);*/
	
	
	if ((MT_getKeysHeld() & KEY_L) || (MT_getKeysHeld() & KEY_R)) {
		if ((MT_getKeysHeld() & KEY_A) && (MT_subScreenScale < 256)) {
			MT_subScreenScale += 3;
		}
		
		if ((MT_getKeysHeld() & KEY_B) && (MT_subScreenScale > 128)) {
			MT_subScreenScale -=3;
		}
		
		int xCenter = MT_subScTargetX + ((MT_subScreenWidth >> 1) << 8);
		int yCenter = MT_subScTargetY + ((MT_subScreenHeight >> 1) << 8);
		
		MT_subScreenWidth = SCUMM_GAME_WIDTH * MT_subScreenScale >> 8;
		MT_subScreenHeight = SCUMM_GAME_HEIGHT * MT_subScreenScale >> 8;
		
		MT_subScTargetX = xCenter - ((MT_subScreenWidth >> 1) << 8);
		MT_subScTargetY = yCenter - ((MT_subScreenHeight >> 1) << 8);
		

		

		if (MT_subScTargetX < 0) MT_subScTargetX = 0;
		if (MT_subScTargetX > (MT_gameWidth - MT_subScreenWidth) << 8) MT_subScTargetX = (MT_gameWidth - MT_subScreenWidth) << 8;
	
		if (MT_subScTargetY < 0) MT_subScTargetY = 0;
		if (MT_subScTargetY > (MT_gameHeight - MT_subScreenHeight) << 8) MT_subScTargetY = (MT_gameHeight - MT_subScreenHeight) << 8;
	}

	MT_subScX += (MT_subScTargetX - MT_subScX) >> 2;
	MT_subScY += (MT_subScTargetY - MT_subScY) >> 2;
	
	if (MT_displayModeIs8Bit) {
	
		if ((MT_getKeysHeld() & KEY_L) || (MT_getKeysHeld() & KEY_R)) {
			
			int offsX = 0, offsY = 0;


			if (MT_getKeysHeld() & KEY_LEFT) {
				offsX -= 1;
			}
			
			if (MT_getKeysHeld() & KEY_RIGHT) {
				offsX += 1;
			}
	
			if (MT_getKeysHeld() & KEY_UP) {
				offsY -= 1;
			}
	
			if (MT_getKeysHeld() & KEY_DOWN) {
				offsY += 1;
			}
			
			if (((MT_gameScreenSwap) && (MT_getKeysHeld() & KEY_L)) || ((!MT_gameScreenSwap) && (MT_getKeysHeld() & KEY_R))) {
				MT_subScTargetX += offsX << 8;
				MT_subScTargetY += offsY << 8;
			} else {
				MT_scX += offsX;
				MT_scY += offsY;
			}
		}

		if (!MT_scaledMode) {
			
			if (MT_scX + 256 > MT_gameWidth - 1) {
				MT_scX = MT_gameWidth - 1 - 256;
			}
	
			if (MT_scX < 0) {
				MT_scX = 0;
			}
	
			if (MT_scY + 192 > MT_gameHeight - 1) {
				MT_scY = MT_gameHeight - 1 - 192;
			}
	
			if (MT_scY < 0) {
				MT_scY = 0;
			}
			
			MT_setZoomedScreenScroll(MT_subScX, MT_subScY);
			MT_setZoomedScreenScale(MT_subScreenWidth, (MT_subScreenHeight * 256) / 192);
	
		
			MT_setMainScreenScroll(MT_scX << 8, (MT_scY << 8) + (MT_shakePos << 8));
			MT_setMainScreenScale(256, 256);		// 1:1 scale
	
		} else {
		
			if (MT_scY > MT_gameHeight - 192 - 1) {
				MT_scY = MT_gameHeight - 192 - 1;
			}

			if (MT_scY < 0) {
				MT_scY = 0;
			}
		
			MT_setZoomedScreenScroll(MT_subScX, MT_subScY);
			MT_setZoomedScreenScale(MT_subScreenWidth, (MT_subScreenHeight * 256) / 192);
	
			MT_setMainScreenScroll(64, (MT_scY << 8) + (MT_shakePos << 8));
			MT_setMainScreenScale(320, 256);		// 1:1 scale
			
		}
	} else {
		MT_setZoomedScreenScroll(0, 0);
		MT_setZoomedScreenScale(320, 256);

		MT_setMainScreenScroll(0, 0);
		MT_setMainScreenScale(320, 256);		// 1:1 scale
	}
	
	// Enable on screen keyboard when pen taps icon
	if ((MT_keyboardIcon) && (penX < 32) && (penY > 160) && (penHeld)) {
		MT_setKeyboardEnable(true);
	}
	
	if (MT_keyboardEnable) {
		if (MT_getKeyboardClosed()) {
			MT_setKeyboardEnable(false);
		}
	}

	MT_updateOAM();

	//PALETTE[0] = RGB15(0, 0, 0);
	REG_IF = IRQ_VBLANK;
}

int MT_getMillis() {
	return MT_currentTimeMillis;
//	return MT_frameCount * FRAME_TIME;
}

void MT_setTimerCallback(OSystem::TimerProc proc, int interval) {
//	consolePrintf("Set timer proc %x, int %d\n", proc, interval);
	MT_callback = proc;
	MT_callbackInterval = interval;
	MT_callbackTimer = interval;
}

void MT_timerTickHandler() {
	REG_IF = IRQ_TIMER0;
	if ((MT_callback) && (MT_callbackTimer > 0)) {
		MT_callbackTimer--;
	}
	MT_currentTimeMillis++;
}

void MT_setTalkPos(int x, int y) {
//	if (MT_gameID != Scumm::GID_SAMNMAX) {
//		MT_setTopScreenTarget(x, 0);
//	} else {
		MT_setTopScreenTarget(x, y);
//	}
}

void MT_setTopScreenTarget(int x, int y) {
	MT_subScTargetX = (x - (MT_subScreenWidth >> 1));
	MT_subScTargetY = (y - (MT_subScreenHeight >> 1));

	if (MT_subScTargetX < 0) MT_subScTargetX = 0;
	if (MT_subScTargetX > MT_gameWidth - MT_subScreenWidth) MT_subScTargetX = MT_gameWidth - MT_subScreenWidth;

	if (MT_subScTargetY < 0) MT_subScTargetY = 0;
	if (MT_subScTargetY > MT_gameHeight - MT_subScreenHeight) MT_subScTargetY = MT_gameHeight - MT_subScreenHeight;
	
	MT_subScTargetX <<=8;
	MT_subScTargetY <<=8;
}

void MT_initHardware() {
	// Guard band
//((int *) (0x023FFF00)) = 0xBEEFCAFE;


	MT_penInit();

	powerON(POWER_ALL);
/*	vramSetBankA(VRAM_A_MAIN_BG); 
	vramSetBankB(VRAM_B_MAIN_BG); 
	vramSetBankC(VRAM_C_SUB_BG); */
	vramSetBankI(VRAM_I_SUB_SPRITE); 
	vramSetBankG(VRAM_G_MAIN_SPRITE); 
	
	MT_currentTimeMillis = 0;

	// Set up a millisecond timer
	TIMER0_CR = 0;
	TIMER0_DATA = TIMER_FREQ(1000);
	TIMER0_CR = TIMER_ENABLE | TIMER_DIV_1 | TIMER_IRQ_REQ;
/*
	// Set up a millisecond counter
	TIMER0_CR = 0;
	TIMER0_DATA = 0xFFFF;
	TIMER0_CR = TIMER_ENABLE | TIMER_CASCADE;
*/



	PALETTE[255] = RGB15(0,31,0);

	// Allocate save buffer for game screen
//	MT_savedBuffer = new u8[320 * 200];
	MT_displayMode16Bit();
	
	memset(BG_GFX, 0, 512 * 256 * 2);	
	MT_scaledMode = true;
	MT_scX = 0;
	MT_scY = 0;
	MT_subScX = 0;
	MT_subScY = 0;
	MT_subScTargetX = 0;
	MT_subScTargetY = 0;
	
	//lcdSwap();
	POWER_CR &= ~POWER_SWAP_LCDS;
	
	MT_frameCount = 0;
	MT_callback = NULL;
	
//	vramSetBankH(VRAM_H_SUB_BG); 
	

//	// Do text stuff
	//BG0_CR = BG_MAP_BASE(0) | BG_TILE_BASE(1);
//	BG0_Y0 = 48;
	
	PALETTE[255] = RGB15(31,31,31);//by default font will be rendered with color 255
	
	//consoleInit() is a lot more flexible but this gets you up and running quick
//	consoleInitDefault((u16*)SCREEN_BASE_BLOCK(0), (u16*)CHAR_BASE_BLOCK(1), 16);
	//consolePrintSet(0, 6);
	
	//irqs are nice
	irqInit();
//	irqInitHandler();
	irqSet(IRQ_VBLANK, MT_VBlankHandler);
	irqSet(IRQ_TIMER0, MT_timerTickHandler);
	irqSet(IRQ_TIMER2, MT_soundBufferEmptyHandler);

	PALETTE[255] = RGB15(0,0,31);

	MT_initSprites();
	
//	videoSetModeSub(MODE_3_2D | DISPLAY_BG0_ACTIVE | DISPLAY_BG3_ACTIVE | DISPLAY_SPR_ACTIVE | DISPLAY_SPR_1D | DISPLAY_SPR_1D_BMP); //sub bg 0 will be used to print text
	
	// Convert texture from 24bit 888 to 16bit 1555, remembering to set top bit!
	u8* srcTex = (u8 *) icons_raw;
	for (int r = 32 * 160 ; r >= 0; r--) {
		SPRITE_GFX_SUB[r] = 0x8000 | (srcTex[r * 3] >> 3) | ((srcTex[r * 3 + 1] >> 3) << 5) | ((srcTex[r * 3 + 2] >> 3) << 10);
		SPRITE_GFX[r] = 0x8000 | (srcTex[r * 3] >> 3) | ((srcTex[r * 3 + 1] >> 3) << 5) | ((srcTex[r * 3 + 2] >> 3) << 10);
	}

	WAIT_CR &= ~(0x0080);
	REG_WRAM_CNT = 0;

}


void MT_setKeyboardIcon(bool enable) {
	MT_keyboardIcon = enable;
}

bool MT_getKeyboardIcon() {
	return MT_keyboardIcon;
}


////////////////////
// Pen stuff
////////////////////


void MT_penInit() {
	penDown = false;
	penHeld = false;
	penReleased = false;
	penDownLastFrame = false;
	penDownSaved = false;
	penReleasedSaved = false;
	penDownFrames = 0;
	MT_consumeKeys();
}

void MT_penUpdate() {

//	if (MT_getKeysHeld() & KEY_L) consolePrintf("%d, %d   penX=%d, penY=%d tz=%d\n", IPC->touchXpx, IPC->touchYpx, penX, penY, IPC->touchZ1);

	if ((penDownFrames > 1)) {			// Is this right?  Dunno, but it works for me.

		if ((penHeld)) {
			penHeld = true;
			penDown = false;

			if ((IPC->touchZ1 > 0) && (IPC->touchXpx > 0) && (IPC->touchYpx > 0)) {
				penX = IPC->touchXpx + MT_touchXOffset;
				penY = IPC->touchYpx + MT_touchYOffset;
			}

		} else {
			penDown = true;
			penHeld = true;
			penDownSaved = true;

			//if ( (ABS(penX - IPC->touchXpx) < 10) && (ABS(penY - IPC->touchYpx) < 10) ) {
			if ((IPC->touchZ1 > 0) && (IPC->touchXpx > 0) && (IPC->touchYpx > 0)) {
				penX = IPC->touchXpx;
				penY = IPC->touchYpx;
			}
			//}
		}

	} else {
		if (penHeld) {
			penReleased = true;
			penReleasedSaved = true;
		} else {
			penReleased = false;
		}

		penDown = false;
		penHeld = false;
	}


	if ((IPC->touchZ1 > 0) || ((penDownFrames == 2)) ) {
		penDownLastFrame = true;
		penDownFrames++;
	} else {
		penDownLastFrame = false;
		penDownFrames = 0;
	}
	
}

int MT_leftHandedSwap(int keys) {
	// Start and select are unchanged
	if (MT_leftHandedMode) {
		int result = keys & (~(KEY_R | KEY_L | KEY_Y | KEY_A | KEY_B | KEY_X | KEY_LEFT | KEY_RIGHT | KEY_UP | KEY_DOWN));
		
		if (keys & KEY_L) result |= KEY_R;
		if (keys & KEY_R) result |= KEY_L;
	
		if (keys & KEY_LEFT) result |= KEY_Y;
		if (keys & KEY_RIGHT) result |= KEY_A;
		if (keys & KEY_DOWN) result |= KEY_B;
		if (keys & KEY_UP) result |= KEY_X;
	
		if (keys & KEY_Y) result |= KEY_LEFT;
		if (keys & KEY_A) result |= KEY_RIGHT;
		if (keys & KEY_B) result |= KEY_DOWN;
		if (keys & KEY_X) result |= KEY_UP;
	
		return result;
	} else {
		return keys;
	}
}

void MT_keysUpdate() {
	scanKeys();
	keysDownSaved |= MT_leftHandedSwap(keysDown());
	keysReleasedSaved |= MT_leftHandedSwap(keysUp());
}

int MT_getKeysDown() {
	return keysDownSaved;
}

int MT_getKeysHeld() {
	return MT_leftHandedSwap(keysHeld());
}

int MT_getKeysReleased() {
	return keysReleasedSaved;
}

void MT_consumeKeys() {
	keysDownSaved = 0;
	keysReleasedSaved = 0;
}

bool MT_getPenDown() {
	return penDownSaved;
}

bool MT_getPenHeld() {
	return penHeld;
}

bool MT_getPenReleased() {
	return penReleasedSaved;
}

void MT_consumePenEvents() {
	penDownSaved = false;
	penReleasedSaved = false;
}

int MT_getPenX() {
	int x = ((penX * MT_touchScX) >> 8) + MT_touchX;
	x = x < 0? 0: (x > MT_gameWidth - 1? MT_gameWidth - 1: x);
	return x;
}

int MT_getPenY() {
	int y = ((penY * MT_touchScY) >> 8) + MT_touchY;
	y = y < 0? 0: (y > MT_gameHeight - 1? MT_gameHeight - 1: y);
	return y;
}

GLvector MT_getPenPos() {
	GLvector v;
	
	v.x = (penX * inttof32(1)) / SCREEN_WIDTH;
	v.y = (penY * inttof32(1)) / SCREEN_HEIGHT;
	
	return v;
}

void formatSramOption() {
	consolePrintf("The following files are present in save RAM:\n");
	DSSaveFileManager::instance()->listFiles();
	
	consolePrintf("\nAre you sure you want to\n");
	consolePrintf("DELETE all files?\n");
	consolePrintf("A = Yes, X = No\n");
	
	while (true) {
		if (keysHeld() & KEY_A) {
			DSSaveFileManager::instance()->formatSram();
			consolePrintf("SRAM cleared!\n");
			return;
		}
	
		if (keysHeld() & KEY_X) {
			consolePrintf("Whew, that was close!\n");
			return;
		}
	}
}


void MT_setIndyFightState(bool st) {
	MT_indyFightState = st;
	MT_indyFightRight = true;
}

bool MT_getIndyFightState() {
	return MT_indyFightState;
}

/////////////////
// GBAMP
/////////////////

bool MT_GBAMPAvail = false;

void MT_initGBAMP() {	
	FAT_InitFiles();
	if (disc_IsInserted()) {
		MT_GBAMPAvail = true;
		consolePrintf("Found flash card reader!\n");
	} else {
		MT_GBAMPAvail = false;
		consolePrintf("Flash card reader not found!\n");
	}
}
	 
bool MT_isGBAMPAvailable() {
	return MT_GBAMPAvail;
}



/////////////////
// Main
/////////////////

int main(void)
{
	MT_soundCallback = NULL;
	
	

	MT_initHardware();
	
	// Let arm9 read cartridge
	*((u16 *) (0x04000204)) &= ~0x0080;
	
	MT_lastCallbackFrame = 0;
	tweak = 0;
	
	MT_indyFightState = false;
	MT_indyFightRight = true;

	// CPU speed = 67108864
	// 8 frames = 2946   368.5 bytes per fr

//	MT_playSound(stretch, 47619, false);
//	MT_playSound(twang, 11010, true);   // 18640

//	MT_bufferSize = 10;
	MT_bufferRate = 22050;
	MT_bufferFrame = 0;
//	MT_bufferSamples = (MT_bufferRate * MT_bufferSize) / 60;
	MT_bufferSamples = 4096;
	
	MT_bufferFirstHalf = false;
	MT_bufferSecondHalf = true;
	
	MT_lastEventFrame = 0;
	MT_mouseMode = MOUSE_LEFT;

	

	
	int bytes = (2 * (MT_bufferSamples)) + 100;
	
	MT_soundBuffer = (s16 *) malloc(bytes * 2);


	MT_soundHiPart = true;
/*
	TIMER1_CR = 0;
	TIMER1_DATA = TIMER_FREQ(MT_bufferRate);
	TIMER1_CR = TIMER_ENABLE | TIMER_DIV_1;
	
	TIMER2_CR = 0;
	TIMER2_DATA = 0xFFFF - (MT_bufferSamples / 2);
	TIMER2_CR = TIMER_ENABLE | TIMER_IRQ_REQ | TIMER_CASCADE;
	*/
	// 2945 - 2947
	
	for (int r = 0; r < bytes; r++) {
		MT_soundBuffer[r] = 0;
	}

	swiWaitForVBlank();
	swiWaitForVBlank();
	MT_playSound(MT_soundBuffer, (MT_bufferSamples * 2), true);
	swiWaitForVBlank();
	swiWaitForVBlank();
	swiWaitForVBlank();

	
//	for (int r = 2946; r < 3000; r++) {
//		MT_soundBuffer[r] = 30000;
//	}
	

	
	consolePrintf("------------------------\n");
	consolePrintf("ScummVM DS\n");
	consolePrintf("Ported by Neil Millstone\n");
#ifdef DS_SCUMM_BUILD
	consolePrintf("Version 0.61 build A\n");
#else
	consolePrintf("Version 0.61 build B\n");
#endif
	consolePrintf("------------------------\n");
	consolePrintf("L/R + D-pad/pen: Scroll view\n");
	consolePrintf("D-pad left:  Left mouse button\n");
	consolePrintf("D-pad right: Right mouse button\n");
	consolePrintf("D-pad up:    Hover mouse\n");
	consolePrintf("D-pad down:  Skip dialog line\n");
	consolePrintf("B button:    Skip cutscenes\n");
	consolePrintf("Select:		DS Options menu\n");
	consolePrintf("Start:       Game menu\n");
	consolePrintf("Y (in game): Toggle console\n");
	consolePrintf("X:           Toggle keyboard\n");
	consolePrintf("A:			Swap screens\n");
	consolePrintf("L + R on bootup: Clear SRAM\n\n");
	consolePrintf("For a complete key list see the\n");
	consolePrintf("help screen.\n\n");
	
	// Create a file system node to force search for a zip file in GBA rom space
	DSFileSystemNode* node = new DSFileSystemNode();
	if (!node->getZip() || (!node->getZip()->isReady())) {
		// If not found, init CF/SD driver
		MT_initGBAMP();
	}
	delete node;
	

	MT_updateStatus();
	
	
//	OSystem_DS::instance();
	g_system = new OSystem_DS();
	assert(g_system);

	if ((keysHeld() & KEY_L) && (keysHeld() & KEY_R)) {
		formatSramOption();
	}

#ifdef DS_NON_SCUMM_BUILD	
	char* argv[2] = {NULL, "--config=scummvmb.ini"};

	while (1) {
		scummvm_main(2, (char **) &argv);
	}
#else
	while (1) {
		scummvm_main(0, NULL);
	}
#endif
	

	return 0;
}
