/* ScummVMDS - Scumm Interpreter DS Port
 * Copyright (C) 2002-2004 The ScummVM project and Neil Millstone
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */
 
#include "stdafx.h"
#include "cdaudio.h"
#include "ds-fs.h"
#include "config-manager.h"
#include "dsmain.h"
#include "nds/scummvm_ipc.h"
#include "console2.h"

#define WAV_FORMAT_IMA_ADPCM 0x14
#define CD_BUFFER_SIZE 8192
#define CD_BUFFER_CHUNK_SIZE (CD_BUFFER_SIZE >> 2)

struct CD_WaveHeader {

	char		riff[4];		// 'RIFF'
	u32			size;			// Size of the file
	char		wave[4];		// 'WAVE'

	// fmt chunk
	char		fmt[4];			// 'fmt '
	u32			fmtSize;		// Chunk size
	u16			fmtFormatTag;	// Format of this file
	u16			fmtChannels;	// Num channels
	u32			fmtSamPerSec;	// Samples per second
	u32			fmtBytesPerSec; // Bytes per second
	u16			fmtBlockAlign;	// Block alignment
	u16			fmtBitsPerSam;	// Bits per sample

	u16			fmtExtraData;	// Number of extra fmt bytes
	u16			fmtExtra;		// Samples per block (only for IMA-ADPCM files)
} __attribute__ ((packed));
	
struct CD_chunkHeader {
	char 		name[4];	
	u32			size;
} __attribute__ ((packed));

struct CD_Header {
	s16 		firstSample;
	char		stepTableIndex;
	char		reserved;
} __attribute__ ((packed));

struct CD_decoderFormat {
	s16 initial;
	unsigned char tableIndex;
	unsigned char test;
	unsigned char	sample[1024];
} __attribute__ ((packed));

bool CD_active = false;
CD_WaveHeader CD_waveHeader;
CD_Header CD_blockHeader;
FILE* CD_file;
int CD_fillPos;
bool CD_isPlayingFlag = false;

s16* CD_audioBuffer;
u32 CD_sampleNum;
s16* CD_decompressionBuffer;
int CD_numLoops;
int CD_blockCount;
int CD_dataChunkStart;
int CD_blocksLeft;


// These are from Microsoft's document on DVI ADPCM
const int CD_stepTab[ 89 ] = {
7, 8, 9, 10, 11, 12, 13, 14,
16, 17, 19, 21, 23, 25, 28, 31,
34, 37, 41, 45, 50, 55, 60, 66,
73, 80, 88, 97, 107, 118, 130, 143,
157, 173, 190, 209, 230, 253, 279, 307,
337, 371, 408, 449, 494, 544, 598, 658,
724, 796, 876, 963, 1060, 1166, 1282, 1411,
1552, 1707, 1878, 2066, 2272, 2499, 2749, 3024,
3327, 3660, 4026, 4428, 4871, 5358, 5894, 6484,
7132, 7845, 8630, 9493, 10442, 11487, 12635, 13899,
15289, 16818, 18500, 20350, 22385, 24623, 27086, 29794,
32767 };

const int CD_indexTab[ 16 ] = { -1, -1, -1, -1, 2, 4, 6, 8,
-1, -1, -1, -1, 2, 4, 6, 8 };

void CD_playNextBlock();
void CD_decompressBlock();


void CD_allocBuffers() {
	
}

void CD_setActive(bool active) {
	CD_active = active;
}

bool CD_getActive() {
	return CD_active;
}

void CD_playTrack(int track, int numLoops, int startFrame, int duration) {
	Common::String path = ConfMan.get("path");
	
	if (CD_isPlayingFlag) {
		CD_stopTrack();
	}
	
	
	
	
	
	char str[100];
	sprintf(str, "/track%d.wav", track);
	path = path + str;
	
	
	//1820160
	
	CD_file = DS_fopen(path.c_str(), "rb");
	
	if (!CD_file) {
		consolePrintf("Failed to open %s!\n", path.c_str());
		return;
	}
	
	
	DS_fread((const void *) &CD_waveHeader, sizeof(CD_waveHeader), 1, CD_file);
	
	consolePrintf("Playing track %d\n", track);
	consolePrintf("Format: %d\n", CD_waveHeader.fmtFormatTag);
	consolePrintf("Rate  : %d\n", CD_waveHeader.fmtSamPerSec);
	consolePrintf("Bits  : %d\n", CD_waveHeader.fmtBitsPerSam);
	consolePrintf("BlkSz : %d\n", CD_waveHeader.fmtExtra);
	
	if ((CD_waveHeader.fmtFormatTag != 17) && (CD_waveHeader.fmtFormatTag != 20)) {
		consolePrintf("Wave file is in the wrong format!  You must use IMA-ADPCM 4-bit mono.\n");
		return;
	}
	
	for (int r = 0; r < 8; r++) {
		IPC->adpcm.buffer[r] = (u8 * volatile) (CD_decoderFormat *) malloc(CD_waveHeader.fmtBlockAlign);
		IPC->adpcm.filled[r] = false;
		IPC->adpcm.arm7Dirty[r] = false;
	}
	
	// Skip chunks until we reach the data chunk
	CD_chunkHeader chunk;
	DS_fread((const void *) &chunk, sizeof(CD_chunkHeader), 1, CD_file);
	
	while (!((chunk.name[0] == 'd') && (chunk.name[1] == 'a') && (chunk.name[2] == 't') && (chunk.name[3] == 'a'))) {
		DS_fseek(CD_file, chunk.size, SEEK_CUR);
		DS_fread((const void *) &chunk, sizeof(CD_chunkHeader), 1, CD_file);
	}
	
	CD_dataChunkStart = DS_ftell(CD_file);
	
	
	static bool started = false;
	CD_sampleNum = 0;
	CD_blockCount = 0;

	IPC->streamFillNeeded[0] = true;
	IPC->streamFillNeeded[1] = true;
	IPC->streamFillNeeded[2] = true;
	IPC->streamFillNeeded[3] = true;
	if (!started) {
		CD_fillPos = 0;
		CD_audioBuffer = (s16 *) malloc(CD_BUFFER_SIZE * 2);
		CD_decompressionBuffer = (s16 *) malloc(CD_waveHeader.fmtExtra * 2);
		started = true;
//		consolePrintf("****Starting buffer*****\n");
		memset(CD_audioBuffer, 0, CD_BUFFER_SIZE * 2);
		memset(CD_decompressionBuffer, 0, CD_waveHeader.fmtExtra * 2);
		MT_playSound(CD_audioBuffer, CD_BUFFER_SIZE * 2, false, false, CD_waveHeader.fmtSamPerSec);
		
	}	
	CD_fillPos = (IPC->streamPlayingSection + 1) & 3;
	CD_isPlayingFlag = true;
	
	
	// Startframe is a 75Hz timer.  Dunno why, since nothing else
	// seems to run at that rate.
	int tenths = (startFrame * 10) / 75;
	
	// Seek to the nearest block start to the start time
	int samples = (tenths * CD_waveHeader.fmtSamPerSec) / 10;
	int block = samples / CD_waveHeader.fmtExtra;
	
	
	if (duration == 0) {
		CD_blocksLeft = 0;
	} else {
		CD_blocksLeft = ((((duration * 100) / 75) * (CD_waveHeader.fmtSamPerSec)) / (CD_waveHeader.fmtExtra) / 100) + 10;
	}
//	consolePrintf("Playing %d blocks (%d)\n\n", CD_blocksLeft, duration);
	
	// No need to seek if we're starting from the beginning
	if (block != 0) {
		DS_fseek(CD_file, CD_dataChunkStart + block * CD_waveHeader.fmtBlockAlign, SEEK_SET);
//		consolePrintf("Startframe: %d  msec: %d (%d,%d)\n", startFrame, tenthssec, samples, block);
	}
	
	
	//CD_decompressBlock();
	CD_playNextBlock();
	CD_numLoops = numLoops;
}

void CD_update() {
	CD_playNextBlock();
}

void CD_decompressBlock() {
	int block[2048];
	bool loop = false;
	
	CD_blockCount++;
	
	if (CD_blockCount < 10) return;
	
	
	do {
		DS_fread((const void *) &CD_blockHeader, sizeof(CD_blockHeader), 1, CD_file);
	
		DS_fread(&block[0], CD_waveHeader.fmtBlockAlign - sizeof(CD_blockHeader), 1, CD_file);

		if (DS_feof(CD_file) ) {
			// Reached end of file, so loop
			
			
			if ((CD_numLoops == -1) || (CD_numLoops > 1)) {
				// Seek file to first packet
				if (CD_numLoops != -1) {
					CD_numLoops--;
				}
				DS_fseek(CD_file, CD_dataChunkStart, SEEK_SET);
				loop = true;
			} else {
				// Fill decompression buffer with zeros to prevent glitching
				for (int r = 0; r < CD_waveHeader.fmtExtra; r++) {
					CD_decompressionBuffer[r] = 0;
				}
//				consolePrintf("Stopping music\n");
				CD_stopTrack();
				return;
			}
			
		} else {
			loop = false;
		}
		
	} while (loop);
		
		
	if (CD_blocksLeft > 0) {
		CD_blocksLeft--;
	//	consolePrintf("%d ", CD_blocksLeft);
		if (CD_blocksLeft == 0) {
			CD_stopTrack();
			return;
		}
	}
		
	// First sample is in header
	CD_decompressionBuffer[0] = CD_blockHeader.firstSample;
	
	// Set up initial table indeces
	int stepTableIndex = CD_blockHeader.stepTableIndex;
	int prevSample = CD_blockHeader.firstSample;
	
//	consolePrintf("Decompressing block step=%d fs=%d\n", stepTableIndex, prevSample);

	for (int r = 0; r < CD_waveHeader.fmtExtra - 1; r++) {
		
		int word = block[r >> 3];
		int offset;
		
		switch (7 - (r & 0x0007)) {
			case 0: {
				offset = (word & 0xF0000000) >> 28;
				break;
			}
			
			case 1: {
				offset = (word & 0x0F000000) >> 24;
				break;
			}
			
			case 2: {
				offset = (word & 0x00F00000) >> 20;
				break;
			}
			
			case 3: {
				offset = (word & 0x000F0000) >> 16;
				break;
			}

			case 4: {
				offset = (word & 0x0000F000) >> 12;
				break;
			}

			case 5: {
				offset = (word & 0x00000F00) >> 8;
				break;
			}

			case 6: {
				offset = (word & 0x000000F0) >> 4;
				break;
			}

			case 7: {
				offset = (word & 0x0000000F);
				break;
			}
		}
		
		int diff = 0;
		
		if (offset & 4) {
			diff = diff + CD_stepTab[stepTableIndex];
		}
		
		if (offset & 2) {
			diff = diff + (CD_stepTab[stepTableIndex] >> 1);
		}
		
		if (offset & 1) {
			diff = diff + (CD_stepTab[stepTableIndex] >> 2);
		}
		
		diff = diff + (CD_stepTab[stepTableIndex] >> 3);
		
		if (offset & 8) {
			diff = -diff;		
		}
		
		int newSample = prevSample + diff;
		
		if (newSample > 32767) newSample = 32767;
		if (newSample < -32768) newSample = -32768;
		
		CD_decompressionBuffer[r + 1] = newSample;
		
		prevSample = newSample;
		
		stepTableIndex += CD_indexTab[offset];
		
		if (stepTableIndex > 88) stepTableIndex = 88;
		if (stepTableIndex < 0) stepTableIndex = 0;
		

	}
}

void CD_playNextBlock() {
	if (!CD_isPlayingFlag) return;
	static int f = 16;
	int lastBlockId = -1;
	
	while (IPC->adpcm.semaphore);		// Wait for buffer to become free if needed
	IPC->adpcm.semaphore = true;		// Lock the buffer structure to prevent clashing with the ARM7
//	DC_FlushAll();
	
	//-8644, 25088
	for (int block = CD_fillPos + 1; block < CD_fillPos + 4; block++) {

		int blockId = block & 3;
		
		if (IPC->streamFillNeeded[blockId]) {
			
			IPC->streamFillNeeded[blockId] = false;
//			DC_FlushAll();
			
/*			if (!(REG_KEYINPUT & KEY_R)) {
				//consolePrintf("Align: %d First: %d  Step:%d  Res:%d\n", CD_waveHeader.fmtBlockAlign, CD_blockHeader.firstSample, CD_blockHeader.stepTableIndex, CD_blockHeader.reserved);
				consolePrintf("Filling buffer %d\n", blockId);
			}*/
			for (int r = blockId * CD_BUFFER_CHUNK_SIZE; r < (blockId + 1) * CD_BUFFER_CHUNK_SIZE; r++) {
				if (CD_isPlayingFlag) {
					CD_audioBuffer[r] = CD_decompressionBuffer[CD_sampleNum++];
					if (CD_sampleNum >= CD_waveHeader.fmtExtra) {
						CD_decompressBlock();
						CD_sampleNum = 0;
					}
				}
			}
			
			lastBlockId = blockId;
			IPC->streamFillNeeded[blockId] = false;
//			DC_FlushAll();

		}
	
		
		
	}
	
	
	
	if (lastBlockId != -1) {
		CD_fillPos = lastBlockId;
/*		if (!(REG_KEYINPUT & KEY_R)) {
			consolePrintf("Frame fill done\n");
		}*/
	}
	IPC->adpcm.semaphore = false;		// Release the buffer structure
//	DC_FlushAll();
}

void CD_stopTrack() {
	if (!CD_isPlayingFlag) return;

	DS_fclose(CD_file);
	
	CD_isPlayingFlag = false;
	
	for (int r = 0; r < CD_BUFFER_SIZE; r++) {
		CD_audioBuffer[r] = 0;
	}
	
	for (int r= 0; r < CD_waveHeader.fmtExtra; r++) {
		CD_decompressionBuffer[r] = 0;
	}
//	MT_stopSound(1);
	
//	free(CD_audioBuffer);
//	free(CD_decompressionBuffer);

	DC_FlushAll();
}

bool CD_checkCD() {
	// Need to check whethe CD audio files are present - do this by trying to open Track1.wav.
	consolePrintf("Attempted to open cd drive\n");

	Common::String path = ConfMan.get("path");
	path = path + "/track2.wav";
	// 6577 153 154
	consolePrintf("Looking for %s...", path.c_str());
	
	FILE* file;
	if ((file = DS_fopen(path.c_str(), "r"))) {
		consolePrintf("Success!\n");
		CD_setActive(true);
		DS_fclose(file);
		return true;
	} else {
		CD_setActive(false);
		consolePrintf("Failed!\n");
		return false;
	}
}

bool CD_isPlaying() {
	return CD_isPlayingFlag;
}
