#include <dsound.h>
#include <stdlib.h>
#include <windows.h>

#include "global.h"
#include "sound.h"
#include "log.h"

static struct {
	LPDIRECTSOUND dsound;
	DSCAPS caps;
	LPDIRECTSOUNDBUFFER buffer_primary;
	LPDIRECTSOUNDBUFFER buffer_secondary;
	unsigned long buffer_secondary_size;
	WAVEFORMATEX format;
	DSBUFFERDESC desc_primary;
	DSBUFFERDESC desc_secondary;
} sound;	


int sound_init(HWND hwnd)
{
	void* buffer;
	unsigned long buffer_size;
	HRESULT result;
	
	memset(&sound,0,sizeof(sound));

	/* create it */
	if ((DirectSoundCreate(NULL,&sound.dsound,NULL))!=DS_OK) {
		LOG(LOG_MISC|LOG_ERROR,"can't create DirectSound!\n");
		sound_clean(); exit(1);
	}

	/* get capabilities. MinGW/C: have to use an interface, instead of eg. dsound->GetCaps(&caps) */
	sound.caps.dwSize=sizeof(sound.caps);
	if ((IDirectSound_GetCaps(sound.dsound,&sound.caps))!=DS_OK) {
		LOG(LOG_MISC|LOG_ERROR,"can't get DirectSound capabilities!\n");
		sound_clean(); exit(1);
	}
	
	/* set cooperative level */
	if ((IDirectSound_SetCooperativeLevel(sound.dsound,hwnd,DSSCL_PRIORITY))!=DS_OK) {
		LOG(LOG_MISC|LOG_ERROR,"can't set DirectSound cooperative level!\n");
		sound_clean(); exit(1);
	}

	/* create primary buffer */
	memset(&sound.desc_primary,0,sizeof(sound.desc_primary));
	sound.desc_primary.dwSize=sizeof(DSBUFFERDESC);
	sound.desc_primary.dwFlags=DSBCAPS_PRIMARYBUFFER;

	if ((IDirectSound_CreateSoundBuffer(sound.dsound,&sound.desc_primary,&sound.buffer_primary,NULL))!=DS_OK) {
		LOG(LOG_MISC|LOG_ERROR,"can't create DirectSound primary buffer!\n");
		sound_clean(); exit(1);
	}
	
	/* set format */
	memset(&sound.format,0,sizeof(sound.format));
	sound.format.cbSize=sizeof(WAVEFORMATEX);
	sound.format.wFormatTag=WAVE_FORMAT_PCM;	/* waveform audio format type */
	sound.format.nChannels=1;			/* mono/stereo */
	sound.format.nSamplesPerSec=44100;		/* samplerate */
	sound.format.wBitsPerSample=16;			/* bits */
	sound.format.nBlockAlign=sound.format.wBitsPerSample*sound.format.nChannels/8;		/* block alignment */
	sound.format.nAvgBytesPerSec=sound.format.nBlockAlign*sound.format.nSamplesPerSec;	/* average datatransferrate */
	
	if ((IDirectSoundBuffer_SetFormat(sound.buffer_primary,&sound.format))!=DS_OK) {
		LOG(LOG_MISC|LOG_ERROR,"can't set DirectSound format!\n");
		sound_clean(); exit(1);
	}
	
	/* get format */
	if ((IDirectSoundBuffer_GetFormat(sound.buffer_primary,&sound.format,sizeof(sound.format),NULL))!=DS_OK) {
		LOG(LOG_MISC|LOG_ERROR,"can't get DirectSound format!\n");
		sound_clean(); exit(1);
	}
	
	/* create secondary buffer */
	sound.buffer_secondary_size=44100;
	
	memset(&sound.desc_secondary,0,sizeof(sound.desc_secondary));
	sound.desc_secondary.dwSize=sizeof(DSBUFFERDESC);
	sound.desc_secondary.dwFlags=DSBCAPS_GLOBALFOCUS|DSBCAPS_GETCURRENTPOSITION2;
	sound.desc_secondary.lpwfxFormat=&sound.format;
	sound.desc_secondary.dwBufferBytes=sound.buffer_secondary_size;
	
	if ((IDirectSound_CreateSoundBuffer(sound.dsound,&sound.desc_secondary,&sound.buffer_secondary,NULL))!=DS_OK) {
		LOG(LOG_MISC|LOG_ERROR,"can't create DirectSound secondary buffer!\n");
		sound_clean(); exit(1);
	}
	
	/* lock secondary buffer for write */
	result=IDirectSoundBuffer_Lock(
		sound.buffer_secondary,	/* buffer to lock */
		0,			/* lock start offset */
		sound.buffer_secondary_size,	/* number of bytes to lock */
		&buffer,		/* locked part start position */
		&buffer_size,		/* locked part size */
		NULL,			/* second locked part start position */
		NULL,			/* second locked part size */
		0			/* flags */
	);
	if (result!=DS_OK) {
		LOG(LOG_MISC|LOG_ERROR,"can't lock DirectSound secondary buffer (initial)!\n");
		sound_clean(); exit(1);
	}
	
	/* clean secondary buffer */
	memset(buffer,0,buffer_size);
	
	/* unlock secondary buffer */
	if ((IDirectSoundBuffer_Unlock(sound.buffer_secondary,buffer,buffer_size,NULL,0))!=DS_OK) { /* = params 1,4,5,6,7 of Lock */
		LOG(LOG_MISC|LOG_ERROR,"can't unlock DirectSound secondary buffer (initial)!\n");
		sound_clean(); exit(1);
	}
	
	/* play secondary buffer */
	result=IDirectSoundBuffer_Play(
		sound.buffer_secondary,	/* buffer to play */
		0,			/* reserved */
		0,			/* priority */
		DSBPLAY_LOOPING		/* flags */
	);
	if (result!=DS_OK) {
		LOG(LOG_MISC|LOG_ERROR,"can't play DirectSound secondary buffer!\n");
		sound_clean(); exit(1);
	}

	LOG(LOG_VERBOSE,"sound initialised\n");

	return 0;
}


void sound_clean(void)
{
	if (sound.buffer_secondary) {
		IDirectSoundBuffer_Stop(sound.buffer_secondary); /* stop playing */
		IDirectSoundBuffer_Release(sound.buffer_secondary); sound.buffer_secondary=NULL; } /* free secondary buffer */
	if (sound.buffer_primary) { IDirectSoundBuffer_Release(sound.buffer_primary); sound.buffer_primary=NULL; } /* free primary buffer */
	if (sound.dsound) { IDirectSound_Release(sound.dsound); sound.dsound=NULL; } /* free directsound */

	LOG(LOG_VERBOSE,"sound cleaned\n");
}
