#include <stdlib.h>
#include <string.h>

#include "global.h"
#include "log.h"
#include "cartridge.h"
#include "ppu_memorymap.h"
#include "ppu.h"

/*	0x10000
	0xc000	mirror of 0x0-0x3fff/not accessible(addressbus=15 bits)?
	0x8000	mirror of 0x0-0x3fff/not accessible(addressbus=15 bits)?
	0x4000	mirror of 0x0-0x3fff
	0x3f20	mirror of palette
	0x3f10	sprite palette
	0x3f00	tiles palette
	0x3000	mirror of 0x2000-0x2eff
	0x2fc0	attribute table 3
	0x2c00	name table 3
	0x2bc0	attribute table 2
	0x2800	name table 2
	0x27c0	attribute table 1
	0x2400	name table 1
	0x23c0	attribute table 0
	0x2000	name table 0
	0x1000	pattern table 1
	0x0	pattern table 0
*/

static struct {
	BYTE palette[0x20];
	
	BYTE* name_ram[2];
	BYTE* cur_name[4];

	BYTE buffer;
	
	BYTE* ppu_control2_ptr;
} ppu_memmap;

void __fastcall (*ppubusoverride)(register WORD);

BYTE* ppu_memmap_get_buffer_ptr(void) { return &ppu_memmap.buffer; }
BYTE* ppu_memmap_get_palette_ptr(void) { return ppu_memmap.palette; }
BYTE** ppu_memmap_get_cur_name_ptr(void) { return ppu_memmap.cur_name; }
BYTE** ppu_memmap_get_name_ram_ptr(void) { return ppu_memmap.name_ram; }

void ppu_memmap_set_ppubusoverride(fp_ppubusoverride fun) { ppubusoverride=fun; }

void ppu_memmap_init(void)
{
	int i;
	memset(&ppu_memmap,0,sizeof(ppu_memmap));
	
	for (i=0;i<2;i++) {
		if ((ppu_memmap.name_ram[i]=malloc(0x400))==NULL) { LOG(LOG_MISC|LOG_ERROR,"nametable %d RAM allocation error!\n",i); exit(1); }
		memset(ppu_memmap.name_ram[i],0,0x400);

		LOG(LOG_VERBOSE,"nametable %d RAM allocated\n",i);
	}
	memset(ppu_memmap.palette,BIN8(00111111),0x20);
	ppu_memmap.ppu_control2_ptr=ppu_get_control2_ptr();
	
	ppu_set_fpt(PPU_FPT_NORMAL,FALSE,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL);
	
	LOG(LOG_VERBOSE,"PPU memorymap initialised\n");
}

void ppu_memmap_clean(void)
{
	int i;
	for (i=0;i<2;i++)
		if (ppu_memmap.name_ram[i]!=NULL) {
			free(ppu_memmap.name_ram[i]);
			ppu_memmap.name_ram[i]=NULL;

			LOG(LOG_VERBOSE,"nametable %d RAM deallocated\n",i);
		}

	LOG(LOG_VERBOSE,"PPU memorymap cleaned\n");
}

/* --- pattern tables */
#define GET_PATDATA_START()	BYTE ret=ppu_memmap.buffer
#define GET_PATDATA_END(x)	ppu_memmap.buffer=cartridge->cur_patbank[x][address&0x3ff]; return ret
#define GET_PATDATA(x)		GET_PATDATA_START(); GET_PATDATA_END(x);
#define SET_PATDATA(x)		if (cartridge->cur_patbank_isram[x]) cartridge->cur_patbank[x][address&0x3ff]=data

/* read */
/* PT0 */
BYTE __fastcall ppu_read_pt0a(register WORD address) { GET_PATDATA(0); } /* 0x0000-0x03ff */
BYTE __fastcall ppu_read_pt0b(register WORD address) { GET_PATDATA(1); } /* 0x0400-0x07ff */
BYTE __fastcall ppu_read_pt0c(register WORD address) { GET_PATDATA(2); } /* 0x0800-0x0bff */
BYTE __fastcall ppu_read_pt0d(register WORD address) { GET_PATDATA(3); } /* 0x0c00-0x0fff */
/* PT1 */
BYTE __fastcall ppu_read_pt1a(register WORD address) { GET_PATDATA(4); } /* 0x1000-0x13ff */
BYTE __fastcall ppu_read_pt1b(register WORD address) { GET_PATDATA(5); } /* 0x1400-0x17ff */
BYTE __fastcall ppu_read_pt1c(register WORD address) { GET_PATDATA(6); } /* 0x1800-0x1bff */
BYTE __fastcall ppu_read_pt1d(register WORD address) { GET_PATDATA(7); } /* 0x1c00-0x1fff */

/* write */
/* PT0 */
void __fastcall ppu_write_pt0a(register WORD address,register BYTE data) { SET_PATDATA(0); } /* 0x0000-0x03ff */
void __fastcall ppu_write_pt0b(register WORD address,register BYTE data) { SET_PATDATA(1); } /* 0x0400-0x07ff */
void __fastcall ppu_write_pt0c(register WORD address,register BYTE data) { SET_PATDATA(2); } /* 0x0800-0x0bff */
void __fastcall ppu_write_pt0d(register WORD address,register BYTE data) { SET_PATDATA(3); } /* 0x0c00-0x0fff */
/* PT1 */
void __fastcall ppu_write_pt1a(register WORD address,register BYTE data) { SET_PATDATA(4); } /* 0x1000-0x13ff */
void __fastcall ppu_write_pt1b(register WORD address,register BYTE data) { SET_PATDATA(5); } /* 0x1400-0x17ff */
void __fastcall ppu_write_pt1c(register WORD address,register BYTE data) { SET_PATDATA(6); } /* 0x1800-0x1bff */
void __fastcall ppu_write_pt1d(register WORD address,register BYTE data) { SET_PATDATA(7); } /* 0x1c00-0x1fff */

/* overrides */
BYTE __fastcall ppu_read_pt0a_override(register WORD address) { GET_PATDATA_START(); (*ppubusoverride)(address); GET_PATDATA_END(0); }
BYTE __fastcall ppu_read_pt0b_override(register WORD address) { GET_PATDATA_START(); (*ppubusoverride)(address); GET_PATDATA_END(1); }
BYTE __fastcall ppu_read_pt0c_override(register WORD address) { GET_PATDATA_START(); (*ppubusoverride)(address); GET_PATDATA_END(2); }
BYTE __fastcall ppu_read_pt0d_override(register WORD address) { GET_PATDATA_START(); (*ppubusoverride)(address); GET_PATDATA_END(3); }
BYTE __fastcall ppu_read_pt1a_override(register WORD address) { GET_PATDATA_START(); (*ppubusoverride)(address); GET_PATDATA_END(4); }
BYTE __fastcall ppu_read_pt1b_override(register WORD address) { GET_PATDATA_START(); (*ppubusoverride)(address); GET_PATDATA_END(5); }
BYTE __fastcall ppu_read_pt1c_override(register WORD address) { GET_PATDATA_START(); (*ppubusoverride)(address); GET_PATDATA_END(6); }
BYTE __fastcall ppu_read_pt1d_override(register WORD address) { GET_PATDATA_START(); (*ppubusoverride)(address); GET_PATDATA_END(7); }
void __fastcall ppu_write_pt0a_override(register WORD address,register BYTE data) { (*ppubusoverride)(address); SET_PATDATA(0); }
void __fastcall ppu_write_pt0b_override(register WORD address,register BYTE data) { (*ppubusoverride)(address); SET_PATDATA(1); }
void __fastcall ppu_write_pt0c_override(register WORD address,register BYTE data) { (*ppubusoverride)(address); SET_PATDATA(2); }
void __fastcall ppu_write_pt0d_override(register WORD address,register BYTE data) { (*ppubusoverride)(address); SET_PATDATA(3); }
void __fastcall ppu_write_pt1a_override(register WORD address,register BYTE data) { (*ppubusoverride)(address); SET_PATDATA(4); }
void __fastcall ppu_write_pt1b_override(register WORD address,register BYTE data) { (*ppubusoverride)(address); SET_PATDATA(5); }
void __fastcall ppu_write_pt1c_override(register WORD address,register BYTE data) { (*ppubusoverride)(address); SET_PATDATA(6); }
void __fastcall ppu_write_pt1d_override(register WORD address,register BYTE data) { (*ppubusoverride)(address); SET_PATDATA(7); }

/* --- nametables --- */
#define GET_NAMEDATA_START()	BYTE ret=ppu_memmap.buffer
#define GET_NAMEDATA_END(x)	ppu_memmap.buffer=ppu_memmap.cur_name[x][address&0x3ff]; return ret
#define GET_NAMEDATA(x)		GET_NAMEDATA_START(); GET_NAMEDATA_END(x);
#define SET_NAMEDATA(x)		ppu_memmap.cur_name[x][address&0x3ff]=data

/* read */
BYTE __fastcall ppu_read_name0(register WORD address) { GET_NAMEDATA(0); } /* nametable 0: 0x2000-0x23ff */
BYTE __fastcall ppu_read_name1(register WORD address) { GET_NAMEDATA(1); } /* nametable 1: 0x2400-0x27ff */
BYTE __fastcall ppu_read_name2(register WORD address) { GET_NAMEDATA(2); } /* nametable 2: 0x2800-0x2bff */
BYTE __fastcall ppu_read_name3(register WORD address) { GET_NAMEDATA(3); } /* nametable 3: 0x2c00-0x2fff */

/* write */
void __fastcall ppu_write_name0(register WORD address,register BYTE data) { SET_NAMEDATA(0); } /* nametable 0: 0x2000-0x23ff */
void __fastcall ppu_write_name1(register WORD address,register BYTE data) { SET_NAMEDATA(1); } /* nametable 1: 0x2400-0x27ff */
void __fastcall ppu_write_name2(register WORD address,register BYTE data) { SET_NAMEDATA(2); } /* nametable 2: 0x2800-0x2bff */
void __fastcall ppu_write_name3(register WORD address,register BYTE data) { SET_NAMEDATA(3); } /* nametable 3: 0x2c00-0x2fff */

/* overrides */
BYTE __fastcall ppu_read_name0_override(register WORD address) { GET_NAMEDATA_START(); (*ppubusoverride)(address); GET_NAMEDATA_END(0); }
BYTE __fastcall ppu_read_name1_override(register WORD address) { GET_NAMEDATA_START(); (*ppubusoverride)(address); GET_NAMEDATA_END(1); }
BYTE __fastcall ppu_read_name2_override(register WORD address) { GET_NAMEDATA_START(); (*ppubusoverride)(address); GET_NAMEDATA_END(2); }
BYTE __fastcall ppu_read_name3_override(register WORD address) { GET_NAMEDATA_START(); (*ppubusoverride)(address); GET_NAMEDATA_END(3); }
void __fastcall ppu_write_name0_override(register WORD address,register BYTE data) { (*ppubusoverride)(address); SET_NAMEDATA(0); }
void __fastcall ppu_write_name1_override(register WORD address,register BYTE data) { (*ppubusoverride)(address); SET_NAMEDATA(1); }
void __fastcall ppu_write_name2_override(register WORD address,register BYTE data) { (*ppubusoverride)(address); SET_NAMEDATA(2); }
void __fastcall ppu_write_name3_override(register WORD address,register BYTE data) { (*ppubusoverride)(address); SET_NAMEDATA(3); }

/* --- palette --- */
#define GET_PALETTE()	ppu_memmap.buffer=ppu_memmap.cur_name[3][address&0x3ff]; \
			if (*ppu_memmap.ppu_control2_ptr&1) return ppu_memmap.palette[address&0x1f]&BIN8(11110000); \
			else return ppu_memmap.palette[address&0x1f]
#define SET_PALETTE()	if((address&BIN8(00000011))==0) ppu_memmap.palette[(address&0x1f)^0x10]=data&BIN8(00111111); \
			ppu_memmap.palette[address&0x1f]=data&BIN8(00111111)

BYTE __fastcall ppu_read_palette(register WORD address) { GET_PALETTE(); }
void __fastcall ppu_write_palette(register WORD address,register BYTE data) { SET_PALETTE(); }

/* overrides */
BYTE __fastcall ppu_read_palette_override(register WORD address) { (*ppubusoverride)(address); GET_PALETTE(); }
void __fastcall ppu_write_palette_override(register WORD address,register BYTE data) { (*ppubusoverride)(address); SET_PALETTE(); }
