#include <stdlib.h>
#include <stdio.h>
#include <math.h>
double rint(double);
#include <string.h>

#include "global.h"
#include "palette.h"
#include "palette_hardcoded.h"
#include "crystal.h"
#include "log.h"
#include "file.h"
#include "cartridge.h"
#include "ppu.h"


static void palette_emphasis_fill(void);

void palette_init(void)
{
	if((palette=malloc(sizeof(Palette)))==NULL) { LOG(LOG_MISC|LOG_ERROR,"palette struct allocation error!\n"); exit(1); }
	memset(palette,0,sizeof(Palette));
	palette_c_preset(cartridge->region);
	palette_y_preset(0);
	palette_load();
	
	palette->gamma=2.2;
	palette->brightness=0.0;

	LOG(LOG_VERBOSE,"palette initialised\n");
}

void palette_clean(void)
{
	if (palette!=NULL) { free(palette); palette=NULL; }

	LOG(LOG_VERBOSE,"palette cleaned\n");
}



/* presets */

/* colour */

const char* palette_c_name[PALETTE_C_PRESET_MAX]={
	"NTSC-U, estimate",
	"NTSC-J, estimate",
	"PAL, estimate",
	
	"NTSC, industrial",
	
	"Sony CXA2025AS/U, tv",
	"Sony CXA2025AS/J, tv",
	"Sony CXA2095S/U, tv",
	"Sony CXA2095S/J, tv",
	
	"AspiringSquire, hardcoded",
	"BMF54123, hardcoded",
	"Chris Covell, hardcoded",
	"loopy, hardcoded",
	"Matthew Conte, hardcoded",
	"Sardu, hardcoded",
	
	"RP2C03B, hardcoded" /* ripped by Kevin Horton, from a PlayChoice 10 */
};

const char* palette_c_preset(BYTE type)
{
	const double settings[PALETTE_C_PRESET_HARDCODED][9]={
		/* 3:colour demodulation angle (R-Y, G-Y, B-Y), 3:gain per c.d.a. channel, 1:saturation, 1:hue offset, 1:hue step */
		{112.0,252.0, 0.0, 0.83, 0.3,  1.0, 0.26, 0.0,  30.0},
		{95.0, 236.0, 0.0, 0.6,  0.33, 1.0, 0.30, 345.0,30.0},
		{90.0, 246.0, 0.0, 0.666,0.333,1.0, 0.30, 345.0,30.0},
		
		{90.0, 246.0, 0.0, 0.666,0.333,1.0, 0.30, 0.0,  30.0},
		
		{112.0,252.0, 0.0, 0.83, 0.3,  1.0, 0.26, 0.0,  30.0},
		{95.0, 240.0, 0.0, 0.78, 0.3,  1.0, 0.30, 0.0,  30.0},
		{105.0,236.0, 0.0, 0.78, 0.33, 1.0, 0.28, 0.0,  30.0},
		{95.0, 236.0, 0.0, 0.6,  0.33, 1.0, 0.30, 0.0,  30.0}
	};
	
	if (type>=PALETTE_C_PRESET_MAX) return NULL;

	palette->c_type=type;
	
	if (type<PALETTE_C_PRESET_HARDCODED) {
		memcpy(palette->cd[0],settings[type],sizeof(double)*2*3);
		palette->saturation=settings[type][6];
		palette->hue=settings[type][7];
		palette->step=settings[type][8];
	}
	
	return palette_c_name[type];
}

const char* get_palette_c_preset_name(BYTE type)
{
	if (type>=PALETTE_C_PRESET_MAX) return NULL;
	else return palette_c_name[type];
}

/* luminance */

const char* palette_y_name[PALETTE_Y_PRESET_MAX]={
	"estimate",
	"Kevin Horton, measured"
};

const char* palette_y_preset(BYTE type)
{
	const double y[PALETTE_Y_PRESET_MAX][4*3]={
		/* 4:0x?0, 4:0x?1-0x?c, 4:0x?d */
		{0.4,  0.7,  1.0,  1.0,  0.1,  0.333,0.666,0.95, 0.0,  0.05, 0.3,  0.71},
		{0.5,  0.75, 1.0,  1.0,  0.29, 0.45, 0.73, 0.9,  0.0,  0.24, 0.47, 0.77}
	};
	
	if (type>=PALETTE_Y_PRESET_MAX) return NULL;
	
	memcpy(palette->y[0],y[type],sizeof(double)*4*3);
	
	return palette_y_name[type];
}

const char* get_palette_y_preset_name(BYTE type)
{
	if (type>=PALETTE_Y_PRESET_MAX) return NULL;
	else return palette_y_name[type];
}



void palette_load(void)
{
	int i,s=0;
	char fn[STRING_SIZE];
	
	sprintf(fn,"%s\\%s.pal",file->palettedir,cartridge->name); /* try [gamename].pal */
	file_setfile(fn,NULL);
	if (!file_open()) {
		file_close();
		sprintf(fn,"%s\\%s.zip",file->palettedir,cartridge->name); /* else [gamename].zip */
		file_setfile(fn,".pal");
		if (!file_open()) {
			file_close();
			sprintf(fn,"%s\\default.pal",file->palettedir); /* else default.pal */
			file_setfile(fn,NULL);
			if (!file_open()) {
				file_close();
				sprintf(fn,"%s\\default.zip",file->palettedir); /* else default.zip */
				file_setfile(fn,".pal");
				if (!file_open()) {
					file_close();
					s=1+(palette->c_type<PALETTE_C_PRESET_HARDCODED); /* else hardcoded, or calculated */
				}
			}
		}
	}
	
	pal_switch_start:
	switch (s) {
		case 0: /* load from file */
			palette->hard=TRUE;
			for (i=0;i<64;i++)
			if (!file_read(palette->rgb[i],3)) {
				LOG(LOG_MISC|LOG_WARNING,"palette read error!\n");
				file_close();
				s=1+(palette->c_type<PALETTE_C_PRESET_HARDCODED);
				goto pal_switch_start; /* retry */
			}
			
			LOG(LOG_MISC,"loaded palette\n");
			file_close();
			break;
		case 1: /* hardcoded */
			palette->hard=TRUE;
			memcpy(palette->rgb[0],palette_hardcoded[palette->c_type-PALETTE_C_PRESET_HARDCODED],64*3);
			break;
		case 2: /* calculated */
			palette->hard=FALSE;
			palette_calculate();
			break;
		
		default: break;
	}
	
	palette_emphasis_fill();
	ppu_fill_palette_local();
}

void palette_save(void)
{
	char fn[STRING_SIZE];
	BYTE pd[256*3];
	memset(pd,0,256*3);
	memcpy(pd,palette->rgb[0],64*3);
	memcpy(pd+0x180,nesticle_padded_bytes,384);
	sprintf(fn,"%s\\%s.pal",file->palettedir,file->appname);
	file_setfile(fn,NULL);
	if (file_save()) {
		if (file_write(pd,(64*3)+(576*palette->write_nesticle_padded_bytes))) LOG(LOG_MISC,"saved palette\n");
		else LOG(LOG_MISC|LOG_WARNING,"palette write error!\n");
	}
	else LOG(LOG_MISC|LOG_WARNING,"couldn't save palette!\n");
	file_close();
}

void palette_calculate(void)
{
	int i;
	BYTE lownib,highnib;
	double rgb[3];
	double y,u,v,sat,hue;
	
	for (highnib=0;highnib<4;highnib++)
	for (lownib=0;lownib<0x10;lownib++)
	{
		sat=palette->saturation;
		y=palette->y[1][highnib];
		if (lownib>13) sat=y=0.0;
		else if (lownib==0) { sat=0.0; y=palette->y[0][highnib]; }
		else if (lownib==13) { sat=0.0; y=palette->y[2][highnib]; }
		
		hue=RAD(palette->hue+303.0+lownib*palette->step);
		
		u=sat*cos(hue);
		v=sat*sin(hue);
		
		for (i=0;i<3;i++) {
			rgb[i]=(y+sin(RAD(palette->cd[0][i]))*palette->cd[1][i]*2*v+cos(RAD(palette->cd[0][i]))*palette->cd[1][i]*2*u)*256.0;
			if (rgb[i]<0) rgb[i]=0.0; if (rgb[i]>255) rgb[i]=255.0;
			
			palette->rgb[highnib<<4|lownib][i]=(BYTE)rint(rgb[i]);
		}
	}
}

static void palette_emphasis_fill(void)
{
	const double emphasis_factor[8][3]={ /* as measured by Chris Covell. r,g,b */
		{1.0,  1.0,  1.0},	/* 000 */
		{1.239,0.915,0.743},	/* 001 */
		{0.794,1.086,0.882},	/* 010 */
		{1.019,0.98, 0.653},	/* 011 */
		{0.905,1.026,1.277},	/* 100 */
		{1.023,0.908,0.979},	/* 101 */
		{0.741,0.987,1.001},	/* 110 */
		{0.75, 0.75, 0.75}	/* 111 */
	};
	
	int i,j,k;
	double l;
	
	for (i=0;i<8;i++) memcpy(palette->rgb_emphasis[i],palette->rgb,64*3);
	if (palette->c_type==PALETTE_C_PRESET_HARDCODED_RP2C03B) { /* different for the RP2C03B */
		for (j=0;j<8;j++)
		for (i=0;i<64;i++) {
			if (j&1) palette->rgb_emphasis[j][i][0]=0xff;
			if (j&2) palette->rgb_emphasis[j][i][1]=0xff;
			if (j&4) palette->rgb_emphasis[j][i][2]=0xff;
		}
	}
	else {
		for (j=0;j<8;j++)
		for (i=0;i<64;i++)
		for (k=0;k<3;k++) {
			l=palette->rgb_emphasis[j][i][k]*emphasis_factor[j][k];
			if (l<0) l=0.0; if (l>255) l=255.0;
			palette->rgb_emphasis[j][i][k]=(BYTE)rint(l);
		}
	}
}
