/* --- iNES mapper 4: MMC3:MMC3A/MMC3B/MMC3C(NES-TEROM/NES-TFROM/NES-TGROM/NES-TKROM/NES-TLROM/HVC-TNROM/NES-TSROM), MMC6/MMC6B(NES-HKROM) ---
	MMC3: SMB 2,3, Megaman 3,4,5,6, Double Dragon 2,3, Ninja Gaiden 2,3, Shatterhand, River City Ransom,
	Mighty Final Fight, Final Fantasy 3, Crystalis, Gun Nac, Earthbound/Mother, Kirby, Turtles 2,3,
	Tiny Toon 1,2, Toki, etc.
	MMC6 (different WRAM handling, slightly different IRQ handling): Startropics 1,2.
	MMC3 NES-TR1ROM/NES-TVROM (4-screen 'mirroring'): Gauntlet 1,2, Rad Racer 2.
   --- iNES mapper 95: MMC3 Namco: different mirroring ---
	Dragon Buster
   --- iNES mapper 118: MMC3 HVC-TKSROM/NES-TLSROM: different mirroring ---
	Ys 3, Goal 2, etc.
   --- iNES mapper 119: MMC3 NES-TQROM: mixed VROM and VRAM banking. ---
	Pin Bot, High Speed. */

/* TODO:
	I'm not sure if IRQ handling is 100% correct. Problem games:
	- Days of Thunder: screen shakes when engine reaches high revs (normal?) (not in PAL version) (it's fixed when IRQs are delayed 1 instruction), and when entering/exiting pits. garbage scanline on the game over textbox (not in PAL version)
	- My Life My Love - Boku no Yume - Watashi no Negai: in-game splitscreen doesn't work: garbage
	- Ys III: screen shakes at the first conversation (pending IRQs are involved) */

enum { MMC3_TYPE_MMC3=0, MMC3_TYPE_MMC6, MMC3_TYPE_NAMCO, MMC3_TYPE_TLSROM, MMC3_TYPE_TQROM, MMC3_TYPE_TVROM };

static void mmc3_nothing(BYTE);
static __inline__ void mmc3_set_pat_mirroring(void);	static __inline__ void mmc3_set_pat_custom(BYTE);
static void mmc3_cmd(BYTE);		static void mmc3_page(BYTE);		static void mmc3_mirroring(BYTE);
static void mmc3_wram(BYTE);		static void mmc3_irq_count(BYTE);	static void mmc3_irq_latch(BYTE);
static void mmc3_irq_c0(BYTE);		static void mmc3_irq_c1(BYTE);
static void mmc3_std_bs(void);
static void mmc3_namco_bs(void);	static void mmc3_namco_page(BYTE);
static void mmc3_tqrom_bs(void);	static void mmc3_tqrom_page(BYTE);	static void mmc3_tqrom_clean(void);
static void mmc3_tlsrom_bs(void);	static void mmc3_tlsrom_page(BYTE);
static void mmc3_tvrom_clean(void);
static void mmc6_cmd(BYTE);		static void mmc6_wram(BYTE);		static void mmc6_clean(void);		static __inline__ void mmc6_ppubus(register WORD);
static void mmc3_new_frame(void);
static __inline__ void mmc3_ppubus(register WORD);
static __inline__ void mmc3_prg_bs(void);
typedef void(*fpt_mmc3_io)(BYTE);
static fpt_mmc3_io mmc3_io[8];
void (*mmc3_bs)(void);
static const fpt_mmc3_io mmc3_io_std[8]= { mmc3_cmd, mmc3_page, mmc3_mirroring, mmc3_wram, mmc3_irq_latch, mmc3_irq_count, mmc3_irq_c0, mmc3_irq_c1 };

/* init: for all */
void mapinit_mmc3(void)
{
	int i;
	
	/* set MMC3 standard values */
	for (i=0;i<8;i++) mmc3_io[i]=mmc3_io_std[i];
	mmc3_bs=&mmc3_std_bs;
	cpu_set_write_io8000(mapio_mmc3,mapio_mmc3,mapio_mmc3,mapio_mmc3);
	cpu_set_mapper_irq(mapirq_mmc3);
	mapnf=&mmc3_new_frame;
	ppu_set_fpt(PPU_FPT_ALL,FALSE,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL);
	ppu_memmap_set_ppubusoverride(mapppubus_mmc3);
	cartridge->wram_isenabled=cartridge->wram_writeenable=TRUE;
	
	INIT_PRGBANKS_FIRSTLAST(); /* first 16k bank at 0x8000, last 16k bank at 0xc000 */
	mapper.mmc3_prg[0]=0; mapper.mmc3_prg[1]=1;
	INIT_PATBANKS_FIRST();
	for (i=0;i<8;i++) { mapper.mmc3_pat[i]=i; }
	
	if (cartridge->type==118) { mapper.mmc3_type=MMC3_TYPE_TLSROM; }
	else if (cartridge->type==119) { mapper.mmc3_type=MMC3_TYPE_TQROM; }
	else if (cartridge->type==95) { mapper.mmc3_type=MMC3_TYPE_NAMCO; }
	else if ((cartridge->crc32==CRC32_STARTROPICS_E)|(cartridge->crc32==CRC32_STARTROPICS_U)|(cartridge->crc32==CRC32_STARTROPICS_2_ZODAS_REVENGE_U)|(cartridge->crc32==CRC32_STARTROPICS_2_ZODAS_REVENGE_U_A1)) { mapper.mmc3_type=MMC3_TYPE_MMC6; }
	else if ((cartridge->mirroring&CARTRIDGE_MIRRORING_4)!=0) { mapper.mmc3_type=MMC3_TYPE_TVROM; }
	else mapper.mmc3_type=MMC3_TYPE_MMC3;
	
	switch (mapper.mmc3_type) {
		case MMC3_TYPE_MMC3:
			strcpy(mapper.mappername,"MMC3 NES-TxROM");
			break;
		case MMC3_TYPE_MMC6:
			mapclean=&mmc6_clean;
			mmc3_io[0]=mmc6_cmd; mmc3_io[3]=mmc6_wram;
			cpu_set_mapper_irq(mapirq_mmc6);
			ppu_memmap_set_ppubusoverride(mapppubus_mmc6);
			cartridge->wram_isenabled=cartridge->wram_writeenable=FALSE;
			cartridge->battery=TRUE; /* MMC6 ROMs have no battery... according to iNES header settings, oh but they do */
			cpu_set_write_io6000(mapio_mmc6_wram_handler_w);
			cpu_set_read_io6000(mapio_mmc6_wram_handler_r);
			if ((mapper.mmc6_wram=malloc(0x400))==NULL) { LOG(LOG_MISC|LOG_ERROR,"MMC6 WRAM allocation error!\n"); exit(1); }
			LOG(LOG_VERBOSE,"MMC6 WRAM allocated\n");
			memset(mapper.mmc6_wram,0,0x400);
			cartridge->cur_wrambank=mapper.mmc6_wram;
			cartridge->battery_size=0x400;
			strcpy(mapper.mappername,"MMC6 NES-HxROM");
			break;
		case MMC3_TYPE_NAMCO:
			mmc3_io[1]=mmc3_namco_page; mmc3_io[2]=mmc3_nothing;
			mmc3_bs=&mmc3_namco_bs;
			cartridge->mirroring=CARTRIDGE_MIRRORING_CUSTOM;
			RESET_MIRRORING();
			strcpy(mapper.mappername,"MMC3 Namco");
			break;
		case MMC3_TYPE_TLSROM:
			mmc3_io[1]=mmc3_tlsrom_page; mmc3_io[2]=mmc3_nothing;
			mmc3_bs=&mmc3_tlsrom_bs;
			cartridge->mirroring=CARTRIDGE_MIRRORING_CUSTOM;
			RESET_MIRRORING();
			strcpy(mapper.mappername,"MMC3 NES-TLSROM");
			break;
		case MMC3_TYPE_TQROM:
			mmc3_io[1]=mmc3_tqrom_page;
			mmc3_bs=&mmc3_tqrom_bs;
			mapclean=&mmc3_tqrom_clean;
			for (i=0;i<8;i++) {
				if ((mapper.mmc3_tqrom_vram[i]=malloc(0x400))==NULL) { LOG(LOG_MISC|LOG_ERROR,"MMC3 NES-TQROM VRAM bank %d allocation error!\n",i); exit(1); }
				memset(mapper.mmc3_tqrom_vram[i],0,0x400);
			}
			LOG(LOG_VERBOSE,"MMC3 NES-TQROM VRAM allocated\n");
			strcpy(mapper.mappername,"MMC3 NES-TQROM");
			break;
		case MMC3_TYPE_TVROM:
			mmc3_io[2]=mmc3_nothing;
			mapclean=&mmc3_tvrom_clean;
			for (i=0;i<4;i++) {
				if ((mapper.mmc3_tvrom_nt[i]=malloc(0x400))==NULL) { LOG(LOG_MISC|LOG_ERROR,"MMC3 NES-TR1ROM/NES-TVROM nametable %d RAM allocation error!\n",i); exit(1); }
				memset(mapper.mmc3_tvrom_nt[i],0,0x400);
				mapper.ppu_memmap_cur_name_ptr[i]=mapper.mmc3_tvrom_nt[i];
			}
			LOG(LOG_VERBOSE,"MMC3 NES-TR1ROM/NES-TVROM nametable RAM allocated\n");
			strcpy(mapper.mappername,"MMC3 NES-TR1ROM/NES-TVROM");
			break;
	}
	
	if ((cartridge->mirroring&(CARTRIDGE_MIRRORING_4|CARTRIDGE_MIRRORING_CUSTOM))==0) { cartridge->mirroring=CARTRIDGE_MIRRORING_VERTICAL; RESET_MIRRORING(); } /* 0xa000=0 */
}

/* IO shortcut */
void __fastcall mapio_mmc3(register WORD address,register BYTE bus,register BYTE data) { mmc3_io[(address>>13<<1|(address&1))&7](data); }


/* --- standard MMC3 calls --- */

/* irq handler, based on ppu addressbus bit 12 rises (0 to 1) */
#define MMC3_RISE_WAIT (CRYSTAL_PPU_CYCLE_FIXED*16)
#define MMC3_PPUBUS_S()	register BYTE count_old; \
			register const int cycles=*mapper.ppu_vblank_ptr?*mapper.cpu_cycles_ptr:*mapper.ppu_cycles_ptr; \
			if (a16) { \
				if (((cycles+MMC3_RISE_WAIT)<=mapper.cycles_old)&(mapper.ppu_address_old==0)) { \
					count_old=mapper.mmc3_irq_count; \
					if ((mapper.mmc3_irq_count==0)|mapper.mmc3_irq_reload) { \
						mapper.mmc3_irq_reload=FALSE; \
						mapper.mmc3_irq_count=mapper.mmc3_irq_latch; \
					} \
					else mapper.mmc3_irq_count--
#define MMC3_IRQ()			if ((count_old!=0)&(mapper.mmc3_irq_count==0)&mapper.mmc3_irq_isenabled) cpu_set_interrupt(INTERRUPT_IRQ|IRQ_MAPPER) /* count went from non 0 to 0 */
#define MMC3_PPUBUS_E()		} \
				mapper.cycles_old=cycles; \
			} \
			mapper.ppu_address_old=a16

static void mmc3_new_frame(void) {
	if (mapper.cycles_old<crystal->frame) mapper.cycles_old+=crystal->frame;
	if (mapper.cycles_old<crystal->frame) mapper.cycles_old=crystal->frame;
}
static __inline__ void mmc3_ppubus(register WORD a16) { MMC3_PPUBUS_S(); MMC3_IRQ(); MMC3_PPUBUS_E(); }
void __fastcall mapppubus_mmc3(register WORD address) { mmc3_ppubus(address&0x1000); }
void __fastcall mapirq_mmc3(void) {
	ppu_force_update();
	if (!*mapper.ppu_enabled_ptr|!*mapper.ppu_valid_scanline_ptr) { mmc3_ppubus(*mapper.ppu_address_ptr&0x1000); }
}

/* registers */
static void mmc3_cmd(BYTE data) { mapper.mmc3_cmdreg=data; (*mmc3_bs)(); } /* 0x8000: set command register */
static void mmc3_page(BYTE data) /* 0x8001: set page */
{
	BYTE pat;
	if (MAXVROM) pat=data&MAXVROM;
	else pat=data&7;
	switch(mapper.mmc3_cmdreg&7) {
		case 0: mapper.mmc3_pat[0]=pat&~1; mapper.mmc3_pat[1]=pat|1; break; /* select 2k chr bank at 0x0 */
		case 1: mapper.mmc3_pat[2]=pat&~1; mapper.mmc3_pat[3]=pat|1; break; /* select 2k chr bank at 0x800 */
		case 2: mapper.mmc3_pat[4]=pat; break; /* select 1k chr bank at 0x1000 */
		case 3: mapper.mmc3_pat[5]=pat; break; /* select 1k chr bank at 0x1400 */
		case 4: mapper.mmc3_pat[6]=pat; break; /* select 1k chr bank at 0x1800 */
		case 5: mapper.mmc3_pat[7]=pat; break; /* select 1k chr bank at 0xc000 */
		case 6: mapper.mmc3_prg[0]=data&MAXPRG; break; /* select 1st prg bank */
		case 7: mapper.mmc3_prg[1]=data&MAXPRG; break; /* select 2nd prg bank */
	}
	(*mmc3_bs)();
}
static void mmc3_mirroring(BYTE data) /* 0xa000: set mirroring */
{
	ppu_force_update();
	if (data&1) cartridge->mirroring=CARTRIDGE_MIRRORING_HORIZONTAL;
	else cartridge->mirroring=CARTRIDGE_MIRRORING_VERTICAL;
	RESET_MIRRORING();
}
static void mmc3_wram(BYTE data) /* 0xa001: wram settings */
{
	/* bit 7: enable wram, bit 6: write protected wram */
	cartridge->wram_isenabled=data>>7;
	cartridge->wram_writeenable=~data>>6&1;
}
static void mmc3_irq_latch(BYTE data) { mapper.mmc3_irq_latch=data; } /* 0xc000: store irq count */
static void mmc3_irq_count(BYTE data) { mapper.mmc3_irq_reload=TRUE; } /* 0xc001: reload irq */
static void mmc3_irq_c0(BYTE data) { mapper.mmc3_irq_isenabled=FALSE; cpu_acknowledge_interrupt(~IRQ_MAPPER); } /* 0xe000: disable irq */
static void mmc3_irq_c1(BYTE data) { mapper.mmc3_irq_isenabled=TRUE; } /* 0xe001: enable irq */

/* bankswitch */
static void mmc3_std_bs(void)
{
	int i;
	const BYTE ca=mapper.mmc3_cmdreg>>5&4;
	ppu_force_update();
	if (MAXVROM) { for (i=0;i<8;i++) PAT_BS(i^ca,mapper.mmc3_pat[i],vrom); }
	else { for (i=0;i<8;i++) PAT_BS(i^ca,mapper.mmc3_pat[i],vram); }
	mmc3_prg_bs();
}
static __inline__ void mmc3_prg_bs(void)
{
	PRG_BS(1,mapper.mmc3_prg[1]);
	if (mapper.mmc3_cmdreg&BIT(6)) { /* 3 is hardwired */
		PRG_BS(0,MAXPRG-1);
		PRG_BS(2,mapper.mmc3_prg[0]);
	}
	else {
		PRG_BS(0,mapper.mmc3_prg[0]);
		PRG_BS(2,MAXPRG-1);
	}
}


/* --- custom shared --- */

static void mmc3_nothing(BYTE data) { return; }

/* set custom pattern table bit */
static __inline__ void mmc3_set_pat_custom(BYTE b)
{
	const BYTE bank=mapper.mmc3_cmdreg&7;
	if (bank<2) { mapper.mmc3_pat_custom[bank<<1]=b; mapper.mmc3_pat_custom[bank<<1|1]=b; }
	else if (bank<6) mapper.mmc3_pat_custom[bank+2]=b;
}

/* set mirroring based on pattern table bit */
static __inline__ void mmc3_set_pat_mirroring(void)
{
	int i;
	const BYTE ca=mapper.mmc3_cmdreg>>5&4;
	ppu_force_update();
	
	cartridge->mirroring&=BIN8(11110000);
	for (i=0;i<4;i++) cartridge->mirroring|=(mapper.mmc3_pat_custom[i^ca]<<i);
	RESET_MIRRORING();
}


/* --- MMC6 --- */

/* clean */
static void mmc6_clean(void)
{
	if (mapper.mmc6_wram!=NULL) {
		free(mapper.mmc6_wram); mapper.mmc6_wram=NULL;
		LOG(LOG_VERBOSE,"MMC6 WRAM deallocated\n");
	}
}

/* irq, for mmc6 the irq keeps triggering if the count is 0 */
#define MMC6_IRQ() if ((mapper.mmc3_irq_count==0)&mapper.mmc3_irq_isenabled) cpu_set_interrupt(INTERRUPT_IRQ|IRQ_MAPPER)
static __inline__ void mmc6_ppubus(register WORD a16) { MMC3_PPUBUS_S(); MMC6_IRQ(); MMC3_PPUBUS_E(); }
void __fastcall mapppubus_mmc6(register WORD address) { mmc6_ppubus(address&0x1000); }
void __fastcall mapirq_mmc6(void) {
	ppu_force_update();
	if (!*mapper.ppu_enabled_ptr|!*mapper.ppu_valid_scanline_ptr) { mmc6_ppubus(*mapper.ppu_address_ptr&0x1000); }
}

/* registers */
static void mmc6_cmd(BYTE data) /* 0x8000: set command register */
{
	/* bit 5: enable wram, can't be unset */
	mapper.mmc3_cmdreg=data|(cartridge->wram_isenabled<<5);
	cartridge->wram_isenabled=mapper.mmc3_cmdreg>>5&1;
	(*mmc3_bs)();
}
static void mmc6_wram(BYTE data) /* 0xa001: wram settings */
{
	if (cartridge->wram_isenabled) { /* bits 4567: enable/write protect 512byte wram bank */
		mapper.mmc6_wram_lh_isenabled[0]=data>>5&1;
		mapper.mmc6_wram_lh_isenabled[1]=data>>7&1;
		mapper.mmc6_wram_lh_writeenable[0]=data>>4&1;
		mapper.mmc6_wram_lh_writeenable[1]=data>>6&1;
	}
}

/* wram handler */
BYTE __fastcall mapio_mmc6_wram_handler_r(register WORD address)
{
	if (cartridge->wram_isenabled)
		if (address&0x1000) {
			if (mapper.mmc6_wram_lh_isenabled[(address&0x3ff)>0x1ff]) return cartridge->cur_wrambank[address&0x3ff];
			else if (mapper.mmc6_wram_lh_isenabled[(address&0x3ff)<0x200]) return 0;
		}
	return address>>8;
}
void __fastcall mapio_mmc6_wram_handler_w(register WORD address,register BYTE bus,register BYTE data)
{
	if (cartridge->wram_isenabled)
		if (address&0x1000)
			if (mapper.mmc6_wram_lh_isenabled[(address&0x3ff)>0x1ff]&mapper.mmc6_wram_lh_writeenable[(address&0x3ff)>0x1ff])
				cartridge->cur_wrambank[address&0x3ff]=data;
}


/* --- MMC3 NAMCO --- */

/* registers */
static void mmc3_namco_page(BYTE data) /* 0x8001: set page */
{
	mmc3_set_pat_custom(data>>5&1); /* bit 5: set nametable/mirroring */
	mmc3_page(data);
}

/* bankswitch */
static void mmc3_namco_bs(void)
{
	mmc3_set_pat_mirroring();
	mmc3_std_bs();
}


/* --- MMC3 HVC-TKSROM/NES-TLSROM --- */

/* registers */
static void mmc3_tlsrom_page(BYTE data) /* 0x8001: set page */
{
	mmc3_set_pat_custom(data>>7); /* bit 7: set nametable/mirroring */
	mmc3_page(data);
}

/* bankswitch */
static void mmc3_tlsrom_bs(void)
{
	mmc3_set_pat_mirroring();
	mmc3_std_bs();
}


/* --- MMC3 NES-TQROM --- */

/* clean */
static void mmc3_tqrom_clean(void)
{
	int i,j=FALSE;
	for (i=0;i<8;i++) if (mapper.mmc3_tqrom_vram[i]!=NULL) { j=TRUE; free(mapper.mmc3_tqrom_vram[i]); mapper.mmc3_tqrom_vram[i]=NULL; }
	if (j) LOG(LOG_VERBOSE,"MMC3 NES-TQROM VRAM deallocated\n");
}

/* registers */
static void mmc3_tqrom_page(BYTE data) /* 0x8001: set page */
{
	mmc3_set_pat_custom(data>>6&1); /* bit 6: set vram */
	mmc3_page(data);
}

/* bankswitch */
static void mmc3_tqrom_bs(void)
{
	int i;
	const BYTE ca=mapper.mmc3_cmdreg>>5&4;
	ppu_force_update();
	
	for (i=0;i<8;i++) {
		cartridge->cur_patbank_isram[i^ca]=mapper.mmc3_pat_custom[i];
		if (mapper.mmc3_pat_custom[i]) cartridge->cur_patbank[i^ca]=mapper.mmc3_tqrom_vram[mapper.mmc3_pat[i]&7];
		else PAT_BS(i^ca,mapper.mmc3_pat[i],vrom);
	}
	
	mmc3_prg_bs();
}


/* MMC3 NES-TR1ROM/NES-TVROM */

/* clean */
static void mmc3_tvrom_clean(void)
{
	int i,j=FALSE;
	for (i=0;i<4;i++) if (mapper.mmc3_tvrom_nt[i]!=NULL) { j=TRUE; free(mapper.mmc3_tvrom_nt[i]); mapper.mmc3_tvrom_nt[i]=NULL; }
	if (j) LOG(LOG_VERBOSE,"MMC3 NES-TR1ROM/NES-TVROM nametable RAM deallocated\n");
}
