#include <stdlib.h>
#include <string.h>

#include "global.h"
#include "log.h"
#include "cartridge.h"
#include "cpu_memorymap.h"
#include "ppu_memorymap.h"
#include "mapper.h"
#include "ppu.h"
#include "cpu.h"
#include "crystal.h"
#include "crc32.h"
#include "file.h"


static struct {
	/* ptr */
	BYTE** ppu_memmap_cur_name_ptr;
	BYTE** ppu_memmap_name_ram_ptr;
	BYTE* ppu_memmap_palette_ptr;
	BYTE* ppu_memmap_buffer_ptr;
	BYTE* ppu_valid_scanline_ptr;
	BYTE* ppu_enabled_ptr;
	BYTE* ppu_control1_ptr;
	BYTE* ppu_control2_ptr;
	BYTE* ppu_reading_sprites_ptr;
	BYTE* ppu_rendering_ptr;
	BYTE* ppu_sc_ptr;
	WORD* ppu_scc_ptr;
	BYTE* ppu_vblank_ptr;
	BYTE* ppu_hblank_ptr;
	WORD* ppu_address_ptr;
	int* cpu_cycles_ptr;
	int* ppu_cycles_ptr;
	
	/* std */
	char mappername[64];
	WORD ppu_address_old;
	int cycles_old;
	WORD sc;
	
	
	/* ave */
	BYTE aved1012_register[2];
	
	/* irem */
	BYTE iremg101_bsb;
	BYTE iremg101_prg;
	BYTE iremg101_type;
	
	BYTE iremh3001_irq_isenabled;
	WORD iremh3001_irq_latch;
	int iremh3001_irq_count;
	
	/* mmc1 */
	BYTE mmc1_type;
	BYTE mmc1_register[4];
	BYTE mmc1_register_temp;
	BYTE mmc1_filltemp_count;
	BYTE mmc1_registernum_old;
	
	BYTE* mmc1_sorom_wram;
	
	/* mmc2 */
	BYTE mmc2_latch[2];
	int mmc2_register[2][2];
	
	/* mmc3 */
	BYTE mmc3_type;
	BYTE mmc3_cmdreg;
	BYTE mmc3_irq_isenabled;
	BYTE mmc3_irq_latch;
	BYTE mmc3_irq_count;
	BYTE mmc3_irq_reload;
	BYTE mmc3_prg[2];
	BYTE mmc3_pat[8];
	BYTE mmc3_pat_custom[8];
	
	BYTE mmc6_wram_lh_isenabled[2];
	BYTE mmc6_wram_lh_writeenable[2];
	BYTE* mmc6_wram;
	
	BYTE* mmc3_tvrom_nt[4];
	
	BYTE* mmc3_tqrom_vram[8];
	
	/* mmc5 */
	BYTE* mmc5_wram;
	BYTE mmc5_wram_size;
	BYTE mmc5_wram_protection[2];
	BYTE* mmc5_exram;
	BYTE mmc5_exram_usage;
	BYTE mmc5_fillnt_ti;
	BYTE mmc5_fillnt_at;
	BYTE mmc5_nt_mode[4];
	WORD mmc5_nt_index;
	BYTE mmc5_prgbs_mode;
	BYTE mmc5_patbs_mode;
	BYTE mmc5_mirroring;
	BYTE mmc5_prg[4][4];
	BYTE mmc5_irq_scanline;
	BYTE mmc5_irq_isenabled;
	BYTE mmc5_irq_status;
	WORD mmc5_pat[2][4][8];
	BYTE mmc5_pat_use;
	BYTE mmc5_pat_high;
	BYTE mmc5_split_isenabled;
	BYTE mmc5_split_side;
	BYTE mmc5_split_pos;
	BYTE mmc5_split_yscroll;
	BYTE mmc5_split_pat;
	WORD mmc5_split_offset;
	BYTE mmc5_addressmode_split;
	BYTE mmc5_mul[2];
	
	/* pirate */
	BYTE smb2j_irq_isenabled;
	int smb2j_irq_count;
	int smb2j_irq_hit;
	
	BYTE mariobaby_irq_isenabled;
	int mariobaby_irq_count;
	int mariobaby_irq_hit;
	
	/* simple */
	BYTE colordreams_isagci50282;
	
	BYTE bandai74161_type;
	
	BYTE camerica_isbf9097;
	
	BYTE cndis_latch;
	BYTE cndis_data;
	
	BYTE camerica_bf9096_block;
	BYTE camerica_bf9096_bank;
	
	/* sunsoft */
	BYTE sunsoft4_ntvrom_isenabled;
	BYTE sunsoft4_ntvrom_bank[2];
	BYTE sunsoft4_mirroring;
	
	/* taito */	
	BYTE taito_istc0350;
	BYTE taitotc0350_irq_isenabled;
	BYTE taitotc0350_irq_latch;
	BYTE taitotc0350_irq_count;
	
} mapper;

/* fp */
static fp_init_mapper init_mapper[0x100];
void mapper_set_init(BYTE mapnum,fp_init_mapper fun) { if (fun==NULL) fun=mapinit_unsup; init_mapper[mapnum]=fun; }
void mapper_set_init_std(void)
{
	int mapnum; for (mapnum=0;mapnum<256;mapnum++) mapper_set_init(mapnum,NULL);
	
	mapper_set_init(0,mapinit_nrom);
	mapper_set_init(1,mapinit_mmc1);
	mapper_set_init(2,mapinit_unrom);
	mapper_set_init(3,mapinit_cnrom);
	mapper_set_init(4,mapinit_mmc3);
	mapper_set_init(5,mapinit_mmc5);
	mapper_set_init(7,mapinit_aorom);
	mapper_set_init(9,mapinit_mmc2);
	mapper_set_init(10,mapinit_mmc2);
	mapper_set_init(11,mapinit_colordreams);
	mapper_set_init(13,mapinit_cprom);
	mapper_set_init(32,mapinit_iremg101);
	mapper_set_init(33,mapinit_taitotc0190);
	mapper_set_init(34,mapinit_nina1);
	mapper_set_init(40,mapinit_smb2j);
	mapper_set_init(42,mapinit_mariobaby);
	mapper_set_init(65,mapinit_iremh3001);
	mapper_set_init(66,mapinit_gnrom);
	mapper_set_init(68,mapinit_sunsoft4);
	mapper_set_init(70,mapinit_bandai74161);
	mapper_set_init(71,mapinit_camerica);
	mapper_set_init(87,mapinit_jaleco74161);
	mapper_set_init(93,mapinit_sunsoft74161);
	mapper_set_init(94,mapinit_capcom74161);
	mapper_set_init(95,mapinit_mmc3);
	mapper_set_init(97,mapinit_irem74161);
	mapper_set_init(118,mapinit_mmc3);
	mapper_set_init(119,mapinit_mmc3);
	mapper_set_init(144,mapinit_colordreams);
	mapper_set_init(152,mapinit_bandai74161);
	mapper_set_init(155,mapinit_mmc1);
	mapper_set_init(185,mapinit_cndis);
	mapper_set_init(232,mapinit_camerica_bf9096);
	mapper_set_init(234,mapinit_aved1012);
}


void (*mapclean)(void);
static void mapclean_nothing(void); static void mapclean_nothing(void) { return; }
void mapper_clean_mapper(void) { (*mapclean)(); }

void (*mapnf)(void);
static void mapnf_nothing(void); static void mapnf_nothing(void) { return; }
void mapper_new_frame(void) { (*mapnf)(); }

BYTE (*mapbatsave)(void); BYTE (*mapbatload)(void);
static BYTE mapbat_loadsave_nothing(void); static BYTE mapbat_loadsave_nothing(void) { return FALSE; }
BYTE mapper_battery_custom_load(void) { return (*mapbatload)(); }
BYTE mapper_battery_custom_save(void) { return (*mapbatsave)(); }

BYTE mapper_supported(BYTE b) { return init_mapper[b]!=mapinit_unsup; }
void mapinit_unsup(void) { return; }

void mapper_kill_cycle(void) { mapper.cycles_old-=CRYSTAL_PPU_CYCLE_FIXED; }

void __fastcall mapio_nothing(register WORD address,register BYTE bus,register BYTE data) { return; }
BYTE __fastcall mapio_nothing_r(register WORD address) { return address>>8; }
void __fastcall mapirq_nothing(void) { return; }
void __fastcall mapppubus_nothing(register WORD address) { return; }


#define MAXPRG cartridge->prgbanks
#define MAXVROM cartridge->vrombanks

#define PRG_BS(n,i) cartridge->cur_prgbank[n]=cartridge->prg[i]
#define PRG_6000_BS(i) cartridge->cur_wrambank=cartridge->prg[i]
#define PAT_BS(n,i,d) cartridge->cur_patbank[n]=cartridge->d[i]

#define INIT_PRGBANKS_FIRSTLAST() PRG_BS(0,0); PRG_BS(1,1); PRG_BS(2,MAXPRG-1); PRG_BS(3,MAXPRG)
#define INIT_PRGBANKS_FIRST() PRG_BS(0,0); PRG_BS(1,1); PRG_BS(2,2); PRG_BS(3,3)
#define INIT_PRGBANKS_LAST() PRG_BS(0,(MAXPRG-3)&MAXPRG); PRG_BS(1,(MAXPRG-2)&MAXPRG); PRG_BS(2,MAXPRG-1); PRG_BS(3,MAXPRG)
#define DEF_PRGBANK_6000() cartridge->wram_isenabled=TRUE; cartridge->wram_writeenable=FALSE
#define INIT_PATBANKS_FIRSTVROM() if (MAXVROM) { PAT_BS(0,0,vrom); PAT_BS(1,1,vrom); PAT_BS(2,2,vrom); PAT_BS(3,3,vrom); PAT_BS(4,4,vrom); PAT_BS(5,5,vrom); PAT_BS(6,6,vrom); PAT_BS(7,7,vrom); cartridge->cur_patbank_isram[0]=cartridge->cur_patbank_isram[1]=cartridge->cur_patbank_isram[2]=cartridge->cur_patbank_isram[3]=cartridge->cur_patbank_isram[4]=cartridge->cur_patbank_isram[5]=cartridge->cur_patbank_isram[6]=cartridge->cur_patbank_isram[7]=FALSE; }
#define INIT_PATBANKS_FIRSTVRAM() if (MAXVROM==0) { PAT_BS(0,0,vram); PAT_BS(1,1,vram); PAT_BS(2,2,vram); PAT_BS(3,3,vram); PAT_BS(4,4,vram); PAT_BS(5,5,vram); PAT_BS(6,6,vram); PAT_BS(7,7,vram); cartridge->cur_patbank_isram[0]=cartridge->cur_patbank_isram[1]=cartridge->cur_patbank_isram[2]=cartridge->cur_patbank_isram[3]=cartridge->cur_patbank_isram[4]=cartridge->cur_patbank_isram[5]=cartridge->cur_patbank_isram[6]=cartridge->cur_patbank_isram[7]=TRUE; }
#define INIT_PATBANKS_FIRST() INIT_PATBANKS_FIRSTVROM() else INIT_PATBANKS_FIRSTVRAM()
#define RESET_MIRRORING() mapper.ppu_memmap_cur_name_ptr[0]=mapper.ppu_memmap_name_ram_ptr[cartridge->mirroring&1]; mapper.ppu_memmap_cur_name_ptr[1]=mapper.ppu_memmap_name_ram_ptr[cartridge->mirroring>>1&1]; mapper.ppu_memmap_cur_name_ptr[2]=mapper.ppu_memmap_name_ram_ptr[cartridge->mirroring>>2&1]; mapper.ppu_memmap_cur_name_ptr[3]=mapper.ppu_memmap_name_ram_ptr[cartridge->mirroring>>3&1]

#define CYCLE_WRITE_DELAY (2*crystal->cycle)


void mapper_init(void)
{
	mapclean=&mapclean_nothing;
	mapnf=&mapnf_nothing;
	mapbatsave=mapbatload=&mapbat_loadsave_nothing;
	mapper_set_init_std();
}

void mapper_clean(void)
{
	(*mapclean)();
}

void mapper_init_mapper(BYTE mapnum)
{
	int i;
	mapper_clean_mapper();
	memset(&mapper,0,sizeof(mapper));
	strcpy(mapper.mappername,"N/A");
	
	mapper.cpu_cycles_ptr=cpu_get_cycles_ptr();
	mapper.ppu_cycles_ptr=ppu_get_cycles_ptr();
	mapper.ppu_memmap_cur_name_ptr=ppu_memmap_get_cur_name_ptr();
	mapper.ppu_memmap_name_ram_ptr=ppu_memmap_get_name_ram_ptr();
	mapper.ppu_memmap_buffer_ptr=ppu_memmap_get_buffer_ptr();
	mapper.ppu_valid_scanline_ptr=ppu_get_valid_scanline_ptr();
	mapper.ppu_enabled_ptr=ppu_get_enabled_ptr();
	mapper.ppu_memmap_palette_ptr=ppu_memmap_get_palette_ptr();
	mapper.ppu_control1_ptr=ppu_get_control1_ptr();
	mapper.ppu_control2_ptr=ppu_get_control2_ptr();
	mapper.ppu_reading_sprites_ptr=ppu_get_reading_sprites_ptr();
	mapper.ppu_rendering_ptr=ppu_get_rendering_ptr();
	mapper.ppu_sc_ptr=ppu_get_sc_ptr();
	mapper.ppu_scc_ptr=ppu_get_scc_ptr();
	mapper.ppu_vblank_ptr=ppu_get_vblank_ptr();
	mapper.ppu_hblank_ptr=ppu_get_hblank_ptr();
	mapper.ppu_address_ptr=ppu_get_address_ptr();
	
	/* make rom point to dummies */
	for (i=0;i<8;i++)
	{
		cartridge->cur_prgbank[i>>1]=cartridge->dummy_rom;
		cartridge->cur_prgbank_isram[i>>1]=FALSE;
		cartridge->cur_patbank[i]=cartridge->dummy_vrom;
		cartridge->cur_patbank_isram[i]=FALSE;
	}
	
	if ((cartridge->mirroring&BIN8(11110000))==0) { RESET_MIRRORING(); }
	mapclean=&mapclean_nothing;
	mapnf=&mapnf_nothing;
	mapbatsave=mapbatload=&mapbat_loadsave_nothing;
	ppu_memmap_set_ppubusoverride(mapppubus_nothing);
	cpu_set_read_io4020(NULL); cpu_set_write_io4020(NULL);
	cpu_set_write_io6000(NULL); cpu_set_read_io6000(NULL);
	cpu_set_write_io8000(NULL,NULL,NULL,NULL);
	cpu_set_mapper_irq(NULL);
	cartridge->cur_wrambank=cartridge->wram; cartridge->battery_size=0x2000; cartridge->wram_isenabled=FALSE; cartridge->wram_writeenable=TRUE;
	if (cartridge->battery|cartridge->trainer) cartridge->wram_isenabled=TRUE;

	init_mapper[mapnum]();
	
	LOG(LOG_MISC,"initialised %s board (iNES mapper %d)\n",mapper.mappername,mapnum);
}


#include "mapper_ave.c"
#include "mapper_bandai.c"
#include "mapper_irem.c"
#include "mapper_jaleco.c"
#include "mapper_konami.c"
#include "mapper_mmc1.c"
#include "mapper_mmc2.c" /* and 4 */
#include "mapper_mmc3.c" /* and 6 */
#include "mapper_mmc5.c"
#include "mapper_namco.c"
#include "mapper_pirate.c"
#include "mapper_simple.c"
#include "mapper_sunsoft.c"
#include "mapper_taito.c"
