#include <stdlib.h>
#include <string.h>

#include "global.h"
#include "log.h"
#include "crystal.h"

/*
NES crystal runs at 236250000/11=~21477272.72Hz
NES NTSC CPU runs at crystal/12=~1789772.72Hz
NES PAL CPU runs at crystal/12.8=~1677911.93Hz
NES PPU runs at crystal/4=~5369318.18Hz

In NTSC 3 PPU cycles take 1 CPU cycle, in PAL 3.2 PPU cycles take 1 CPU cycle.
One scanline is 256 pixel cycles and 85 hblank cycles, that makes 341 PPU cycles.
There are 262 scanlines per frame for NTSC, and 312 for PAL.

The NES does not use the default NTSC(60/1.001=~59.94Hz) or PAL(50Hz) refresh rate.
NTSC refresh rate: ((236250000/11)/12)/((262*341)/3)=~60.098Hz
PAL refresh rate: ((236250000/11)/12.8)/((312*341)/3.2)=~50.467Hz

Windows timing maximum accuracy is 1ms, for NTSC it's a mixture of 16ms and 17ms:
(16a+17b)/(a+b)=1000/refresh rate(=~16.6394), the closest solution is 39731 and 70436.
For PAL it's a mixture of 19ms and 20ms(~19.815): 19815 and 87182.

*/

#define PPU_CYCLE		CRYSTAL_PPU_CYCLE_FIXED /* from crystal.h */
#define PPU_SCANLINE		(PPU_CYCLE*256)
#define PPU_HBLANK		(PPU_CYCLE*85)
#define PPU_FULL_SCANLINE	(PPU_SCANLINE+PPU_HBLANK)

#define NTSC_LINES		262
#define NTSC_CYCLE		(PPU_CYCLE*3)
#define NTSC_FRAME		(PPU_FULL_SCANLINE*NTSC_LINES)
#define NTSC_VBLANK_TRIGGER	(NTSC_FRAME-(PPU_FULL_SCANLINE*242))

#define PAL_LINES		312
#define PAL_CYCLE		16 /*(PPU_CYCLE*3.2)*/
#define PAL_FRAME		(PPU_FULL_SCANLINE*PAL_LINES)
#define PAL_VBLANK_TRIGGER	(PAL_FRAME-(PPU_FULL_SCANLINE*242))

#define WIN_NTSC_TIMA		16
#define WIN_NTSC_TIMB		17
#define WIN_NTSC_MULA		39731
#define WIN_NTSC_MULB		70436

#define WIN_PAL_TIMA		19
#define WIN_PAL_TIMB		20
#define WIN_PAL_MULA		19815
#define WIN_PAL_MULB		87182

void crystal_init(void)
{
	if((crystal=malloc(sizeof(Crystal)))==NULL) { LOG(LOG_MISC|LOG_ERROR,"crystal struct allocation error!\n"); exit(1); }
	memset(crystal,0,sizeof(Crystal));
	crystal->ppu_cycle=PPU_CYCLE;
	crystal->ppu_scanline=PPU_SCANLINE;
	crystal->hblank=PPU_HBLANK;
	crystal->full_scanline=PPU_FULL_SCANLINE;

	LOG(LOG_VERBOSE,"crystal initialised\n");
}

void crystal_clean(void)
{
	if (crystal!=NULL) { free(crystal); crystal=NULL; }

	LOG(LOG_VERBOSE,"crystal cleaned\n");
}

void crystal_set_mode(BYTE b)
{
	int i;
	
	/*b=1;*/ /* force PAL */
	
	crystal->mode=b;
	LOG(LOG_VERBOSE,"crystal mode set to %s\n",b?"PAL":"NTSC");

	crystal->lines=b?PAL_LINES:NTSC_LINES;
	crystal->cycle=b?PAL_CYCLE:NTSC_CYCLE;
	crystal->frame=b?PAL_FRAME:NTSC_FRAME;
	crystal->vblank_trigger=b?PAL_VBLANK_TRIGGER:NTSC_VBLANK_TRIGGER;
	crystal->odd_frame=b?FALSE:TRUE; /* should be FALSE at frame 0 */
	
	for (i=0;i<313;i++) crystal->hc[i]=crystal->sc[i]=-crystal->frame;
	for (i=0;i<=crystal->lines;i++) {
		crystal->hc[i]=crystal->frame-(i*crystal->full_scanline)-crystal->ppu_scanline;
		crystal->sc[i]=crystal->frame-(i*crystal->full_scanline);
	}
}

void crystal_new_frame(void)
{
	crystal->odd_frame^=!crystal->mode;
	crystal->fc++;
}
