#include <math.h>
double rint(double);
#include <string.h>

#include "global.h"
#include "log.h"
#include "cpu.h"
#include "crystal.h"
#include "apu.h"

/* (pseudo: it's part of the 2a03) APU

TODO:
	- triangle, squares, noise, dmc output
	
	- PAL timing of DMC and FS is correct ?
	- DMC: nestank: screen shaking sometimes, normal ?
*/

static struct {
	BYTE status;
	
	BYTE frame_irq_isenabled;
	BYTE fs_mode;
	BYTE fs_count;
	BYTE fs_max;
	
	BYTE dmc_irq_isenabled;
	BYTE dmc_loop;
	
	BYTE dmc_dac;
	WORD dmc_address;
	WORD dmc_length;
	
	BYTE dmc_buffer;
	BYTE dmc_buffer_isempty;
	BYTE dmc_bits_counter;
	WORD dmc_bytes_counter;
} apu;


/* DMC */

static __inline__ void dmc_refill(void);

static void apu_fill_dmc_update_frequency_lookup(void);
static int dmc_update_frequency_lookup[16];
static void apu_fill_dmc_update_frequency_lookup(void)
{
	const int dmc_update_frequency[16]={428,380,340,320,286,254,226,214,190,160,142,128,106,84,72,54};
	int i;
	if (crystal->mode) for (i=0;i<16;i++) dmc_update_frequency_lookup[i]=(rint(dmc_update_frequency[i]*APU_DMC_FREQ_PAL_FACTOR))*crystal->cycle;
	else for (i=0;i<16;i++) dmc_update_frequency_lookup[i]=dmc_update_frequency[i]*crystal->cycle;
}

static __inline__ void dmc_refill(void)
{
	if (apu.dmc_bytes_counter) {
		apu.dmc_buffer_isempty=FALSE;
		apu.dmc_buffer=cpu_dmc_dma_transfer(0);
		apu.dmc_bytes_counter--;
		if (apu.dmc_bytes_counter==0) {
			if (apu.dmc_loop) apu.dmc_bytes_counter=apu.dmc_length;
			else {
				apu.status&=~BIT(4);
				if (apu.dmc_irq_isenabled) { apu.status|=BIT(7); cpu_set_interrupt(INTERRUPT_IRQ|IRQ_DMC); }
			}
		}
	}
}

/* 0x4010, DMC control
	7	enable DMC IRQs
	6	enable looping
	3,2,1,0	update frequency */
void __fastcall apu_write_dmc_control(register BYTE data)
{
	apu.dmc_irq_isenabled=data&BIT(7);
	if (apu.dmc_irq_isenabled==0) { apu.status&=~BIT(7); cpu_acknowledge_interrupt(~IRQ_DMC); }
	apu.dmc_loop=data&BIT(6);
	cpu_reset_apu_dmc_freq(dmc_update_frequency_lookup[data&BIN8(00001111)]);
}
/* 0x4011, DAC and counter (-msb) */
void __fastcall apu_write_dmc_dac(register BYTE data)
{
	apu.dmc_dac=data&~BIT(7);
}
/* 0x4012, address */
void __fastcall apu_write_dmc_address(register BYTE data)
{
	apu.dmc_address=data<<6|0xc000;
}
/* 0x4013, length */
void __fastcall apu_write_dmc_length(register BYTE data)
{
	apu.dmc_length=data<<4|1;
}

void __fastcall apu_run_dmc(void)
{
	run_dmc_start:
	if (apu.dmc_bits_counter) {
		apu.dmc_bits_counter--;
		if (apu.dmc_bits_counter==0) {
			apu.dmc_bits_counter=8;
			apu.dmc_buffer_isempty=TRUE;
			dmc_refill();
		}
	}
	if (cpu_set_apu_dmc_next_step()) goto run_dmc_start;
}


/* status */

/* 0x4015, status register
read	7	DMC IRQ
	6	frame IRQ
	5	-
	4	DMC is reading
	3	not yet
	2	not yet
	1	not yet
	0	not yet */
BYTE __fastcall apu_read_status(register WORD address)
{
	BYTE b=apu.status;
	apu.status&=BIN8(00111111);
	cpu_acknowledge_interrupt(~(IRQ_DMC|IRQ_FRAME));
	return b;
}
/*write	4	enable DMC
	3	not yet
	2	not yet
	1	not yet
	0	not yet */
void __fastcall apu_write_status(register BYTE data)
{
	apu.status&=~BIT(7);
	cpu_acknowledge_interrupt(~IRQ_DMC);

	if (data&BIT(4)) {
		if (apu.dmc_bytes_counter==0) {
			apu.status|=BIT(4); apu.dmc_bytes_counter=apu.dmc_length;
			if (apu.dmc_buffer_isempty) dmc_refill();
		}
	}
	else { apu.status&=~BIT(4); apu.dmc_bytes_counter=0; }
}


/* FS */

static void apu_fill_fs_update_frequency_lookup(void);

static int fs_update_frequency_lookup[2][5];
static void apu_fill_fs_update_frequency_lookup(void)
{
	const int fs_update_frequency[2][5]={ {7456,7458,7458,7458,7458},{7458,7456,7458,7456,7454} };
	int i,j;
	if (crystal->mode) for (i=0;i<2;i++) for (j=0;j<5;j++) fs_update_frequency_lookup[i][j]=(rint(fs_update_frequency[i][j]*APU_FS_FREQ_PAL_FACTOR))*crystal->cycle;
	else for (i=0;i<2;i++) for (j=0;j<5;j++) fs_update_frequency_lookup[i][j]=fs_update_frequency[i][j]*crystal->cycle;
}

/* 0x4017, frame sequencer
	7	mode (4 step or 5 step)
	6	disable frame irqs */
void __fastcall apu_write_fs(register BYTE data)
{
	cpu_reset_apu_fs_next_step();
	apu.frame_irq_isenabled=(data&BIN8(11000000))==0;
	if (data&BIT(6)) { apu.status&=~BIT(6); cpu_acknowledge_interrupt(~IRQ_FRAME); }
	
	apu.fs_mode=data>>7;
	apu.fs_max=apu.fs_mode+3; /* 3 or 4 */
	apu.fs_count=apu.fs_max<<1&4; /* 3?:4 4?:0 */
}

void __fastcall apu_run_fs(void)
{
	if (apu.fs_count<4) {
		; /* envelope and triangle counter */
		if ((apu.fs_count+apu.fs_mode)&1) { ; } /* length counters and sweep units */
		if (apu.frame_irq_isenabled&(apu.fs_count==3)) {
			apu.status|=BIT(6);
			cpu_set_interrupt(INTERRUPT_IRQ|IRQ_FRAME);
		}
	}
	cpu_set_apu_fs_next_step(fs_update_frequency_lookup[apu.fs_mode][apu.fs_count++]);
	if (apu.fs_count>apu.fs_max) apu.fs_count=0;
}



void apu_init(void)
{
	memset(&apu,0,sizeof(apu));
	
	apu_fill_fs_update_frequency_lookup();
	apu_write_fs(APU_FS_INITIAL);
	
	apu_fill_dmc_update_frequency_lookup();
	apu_write_dmc_control(0xf);
	apu.dmc_buffer_isempty=TRUE;
	apu.dmc_length=1;
	apu.dmc_bits_counter=8;
	
	LOG(LOG_VERBOSE,"APU initialised\n");
}

void apu_clean(void)
{
	memset(&apu,0,sizeof(apu));
	
	LOG(LOG_VERBOSE,"APU cleaned\n");
}
