/*********** SIMPLE TYPES: usually a bit of discrete logic ***********/

/* --- iNES mapper 0(no 'mapper'): NES-NROM-128/NES-NROM-256/NES-RROM-128 ---
	Super Mario Bros, Ice Climber, Pac-Man, Donkey Kong, Dig Dug, Millipede, Pooyan, Galaga, etc. */

void mapinit_nrom(void)
{
	strcpy(mapper.mappername,"NES-NROM/NES-RROM");
	INIT_PRGBANKS_FIRSTLAST(); /* 32k?:full, 16k?:mirrored */
	INIT_PATBANKS_FIRST();
}
/* has no IO */





/* --- iNES mapper 2: NES-UNROM/NES-UOROM ---
	Castlevania, Ghosts 'n Goblins, Rockman, Rodland, Ducktales 1,2, Contra, Metal Gear, Guardian Legend, etc. */

void mapinit_unrom(void)
{
	cpu_set_write_io8000(mapio_unrom,mapio_unrom,mapio_unrom,mapio_unrom);
	strcpy(mapper.mappername,"NES-UxROM");
	INIT_PRGBANKS_FIRSTLAST(); /* first 16k bank at 0x8000, last 16k bank at 0xc000 */
	INIT_PATBANKS_FIRST(); /* 8k vram (some games use VROM with this mapper, though that's wrong) */
}

void __fastcall mapio_unrom(register WORD address,register BYTE bus,register BYTE data)
{
	/* swap 16k prgbank in 0x8000 */
	WORD prg=data<<1&MAXPRG;
	#if MAPPER_BUS_CONFLICTS
	prg=prg|(bus<<1&MAXPRG);
	#endif
	PRG_BS(0,prg++); PRG_BS(1,prg);
}





/* --- iNES mapper 3: NES-CNROM ---
	Solomon's Key, Gradius, Hudson's Adventure Island, Puzznic, Legend of Kage, etc. */

void mapinit_cnrom(void)
{
	strcpy(mapper.mappername,"NES-CNROM");
	cpu_set_write_io8000(mapio_cnrom,mapio_cnrom,mapio_cnrom,mapio_cnrom);
	INIT_PRGBANKS_FIRSTLAST(); /* 32k?:full, 16k?:mirrored */
	INIT_PATBANKS_FIRST(); /* first 8k vrom */
}

void __fastcall mapio_cnrom(register WORD address,register BYTE bus,register BYTE data)
{ /*	swap 8k vrom bank in 0x0 */
	WORD pat=data<<3&MAXVROM;
	#if MAPPER_BUS_CONFLICTS
	pat=pat|(bus<<3&MAXVROM);
	#endif
	ppu_force_update();
	PAT_BS(0,pat++,vrom); PAT_BS(1,pat++,vrom); PAT_BS(2,pat++,vrom); PAT_BS(3,pat++,vrom);
	PAT_BS(4,pat++,vrom); PAT_BS(5,pat++,vrom); PAT_BS(6,pat++,vrom); PAT_BS(7,pat,vrom);
}





/* --- iNES mapper 7: NES-AOROM/NES-AMROM/NES-ANROM ---
	Solstice, Marble Madness, Wizards & Warriors, Battletoads, etc. */

void mapinit_aorom(void)
{
	strcpy(mapper.mappername,"NES-AxROM");
	cpu_set_write_io8000(mapio_aorom,mapio_aorom,mapio_aorom,mapio_aorom);
	INIT_PRGBANKS_FIRST(); /* first 32k prgbank */
	INIT_PATBANKS_FIRST(); /* 8k vram */
	cartridge->mirroring=CARTRIDGE_MIRRORING_ONE_0;
	RESET_MIRRORING();
}

void __fastcall mapio_aorom(register WORD address,register BYTE bus,register BYTE data)
{
	/* bits 0123: swap 32k prgbank in 0x8000 */
	WORD prg=data<<2&MAXPRG;
	#if MAPPER_BUS_CONFLICTS
	prg=prg|(bus<<2&MAXPRG);
	#endif
	PRG_BS(0,prg++); PRG_BS(1,prg++); PRG_BS(2,prg++); PRG_BS(3,prg);
	
	/* bit 4: set mirroring */
	ppu_force_update();
	if (data&BIN8(00010000)) cartridge->mirroring=CARTRIDGE_MIRRORING_ONE_1;
	else cartridge->mirroring=CARTRIDGE_MIRRORING_ONE_0;
	RESET_MIRRORING();
}





/* --- iNES mapper 11: Color Dreams(Rev A/Rev B/Rev C/AGCI 47516/196/BC6) ---
	Crystal Mines, Menace Beach, all those funky religious games, etc.
   --- iNES mapper 144: AGCI 50282, same as Color Dreams mapper, except for D0 awkwardness ---
	Death Race */

void mapinit_colordreams(void)
{
	cpu_set_write_io8000(mapio_colordreams,mapio_colordreams,mapio_colordreams,mapio_colordreams);
	mapper.colordreams_isagci50282=cartridge->type==144;
	if (mapper.colordreams_isagci50282) strcpy(mapper.mappername,"AGCI 50282");
	else strcpy(mapper.mappername,"Color Dreams");
		
	INIT_PRGBANKS_FIRST(); /* first 32k prgbank */
	INIT_PATBANKS_FIRST();
}

void __fastcall mapio_colordreams(register WORD address,register BYTE bus,register BYTE data)
{
	BYTE prg,pat;
	
	data|=(bus&mapper.colordreams_isagci50282); /* dataline 0 on ROM always wins for AGCI 50282 */
	
	/* bits 01: select 32k prg bank, bits 23: lockout chip stuff, bits 4567: select 8k vrom bank. */
	prg=data<<2&MAXPRG;
	pat=data>>1&BIN8(01111000)&MAXVROM;
	ppu_force_update();
	PRG_BS(0,prg++); PRG_BS(1,prg++); PRG_BS(2,prg++); PRG_BS(3,prg);
	if (MAXVROM) {
		PAT_BS(0,pat++,vrom); PAT_BS(1,pat++,vrom); PAT_BS(2,pat++,vrom); PAT_BS(3,pat++,vrom);
		PAT_BS(4,pat++,vrom); PAT_BS(5,pat++,vrom); PAT_BS(6,pat++,vrom); PAT_BS(7,pat,vrom);
	}
}





/* --- iNES mapper 13: NES-CPROM ---
	Videomation */

void mapinit_cprom(void)
{
	int i;
	strcpy(mapper.mappername,"NES-CPROM");
	cpu_set_write_io8000(mapio_cprom,mapio_cprom,mapio_cprom,mapio_cprom);
	INIT_PRGBANKS_FIRST(); /* first (and only) 32k prgbank */
	INIT_PATBANKS_FIRST(); /* it's got 16k vram, 1st 4k fixed */

	/* allocate another 8k vram bank, cleaned in cartridge_unload */
	for (i=8;i<0x10;i++) {
		if ((cartridge->vram[i]=malloc(0x400))==NULL) { LOG(LOG_MISC|LOG_ERROR,"NES-CPROM vrambanks[%d] allocation error!\n",i); exit(1); }
		memset(cartridge->vram[i],0,0x400);
	}
	LOG(LOG_VERBOSE,"NES-CPROM VRAM allocated\n");
}

void __fastcall mapio_cprom(register WORD address,register BYTE bus,register BYTE data)
{
	/* select 4k vram for ppu 0x1000 */
	ppu_force_update();
	data=data<<2&0xf;
	PAT_BS(4,data++,vram); PAT_BS(5,data++,vram); PAT_BS(6,data++,vram); PAT_BS(7,data,vram);
}





/* --- iNES mapper 66: NES-GNROM/Bandai 74161/32 ---
	SMB+Duckhunt, Gumshoe, Dragonball 1 (J)/Dragon Power (U), etc. */

void mapinit_gnrom(void)
{
	strcpy(mapper.mappername,"NES-GNROM/Bandai 74161/32");
	cpu_set_write_io6000(mapio_gnrom); /* Youkai Kurabu */
	cpu_set_write_io8000(mapio_gnrom,mapio_gnrom,mapio_gnrom,mapio_gnrom);
	INIT_PRGBANKS_FIRST(); /* first 32k prgbank */
	INIT_PATBANKS_FIRST();
}

void __fastcall mapio_gnrom(register WORD address,register BYTE bus,register BYTE data)
{
	/* bits 0123: select 8k vrom bank, bits 4567: select 32k prg bank. */
	BYTE prg=data>>2&BIN8(00111100)&MAXPRG;
	BYTE pat=data<<3&BIN8(01111000)&MAXVROM;
	PRG_BS(0,prg++); PRG_BS(1,prg++); PRG_BS(2,prg++); PRG_BS(3,prg);
	if (MAXVROM) {
		ppu_force_update();
		PAT_BS(0,pat++,vrom); PAT_BS(1,pat++,vrom); PAT_BS(2,pat++,vrom); PAT_BS(3,pat++,vrom);
		PAT_BS(4,pat++,vrom); PAT_BS(5,pat++,vrom); PAT_BS(6,pat++,vrom); PAT_BS(7,pat,vrom);
	}

	if (address<0x8000) {
		if (cartridge->wram_isenabled) cartridge->cur_wrambank[address&0x1fff]=data; /* in case there's battery sram */
	}
}





/* --- iNES mapper 70: Bandai 74161/32 type 1 (normal mirroring) ---
	Kamen Rider Club, Space Shadow, etc.
   --- iNES mapper 152: Bandai 74161/32 type 2 (1-screen mirroring) ---
   	Arkanoid 2, etc. */

void mapinit_bandai74161(void)
{
	cpu_set_write_io8000(mapio_bandai74161,mapio_bandai74161,mapio_bandai74161,mapio_bandai74161);
	INIT_PRGBANKS_FIRSTLAST(); /* first 16k in 0x8000 (switchable), last 16k in 0xc000 (hardwired) */
	INIT_PATBANKS_FIRST();
	
	mapper.bandai74161_type=cartridge->type==70;
	if (mapper.bandai74161_type) {
		strcpy(mapper.mappername,"Bandai 74161/32 type 1");
		cartridge->mirroring=CARTRIDGE_MIRRORING_VERTICAL; RESET_MIRRORING();
	}
	else {
		strcpy(mapper.mappername,"Bandai 74161/32 type 2");
		cartridge->mirroring=CARTRIDGE_MIRRORING_ONE_0; RESET_MIRRORING();
	}
}

void __fastcall mapio_bandai74161(register WORD address,register BYTE bus,register BYTE data)
{
	BYTE pat=data<<3&MAXVROM;
	BYTE prg=(data&BIN8(01110000))>>3&MAXPRG;
	ppu_force_update();
	
	/* bits 0123: swap 8k vrom */
	PAT_BS(0,pat++,vrom); PAT_BS(1,pat++,vrom); PAT_BS(2,pat++,vrom); PAT_BS(3,pat++,vrom);
	PAT_BS(4,pat++,vrom); PAT_BS(5,pat++,vrom); PAT_BS(6,pat++,vrom); PAT_BS(7,pat,vrom);
	
	/* bits 456: swap 16k rom in 0x8000 */
	PRG_BS(0,prg++); PRG_BS(1,prg);

	/* bit 7: set mirroring */
	if (data&0x80) cartridge->mirroring=mapper.bandai74161_type?CARTRIDGE_MIRRORING_HORIZONTAL:CARTRIDGE_MIRRORING_ONE_1;
	else cartridge->mirroring=mapper.bandai74161_type?CARTRIDGE_MIRRORING_VERTICAL:CARTRIDGE_MIRRORING_ONE_0;
	RESET_MIRRORING();
}





/* --- iNES mapper 71: Camerica BF9093/BF9097(same as other, but with mirroring switch: Fire Hawk only) ---
	Linus Spacehead's Cosmic Crusade, Fire Hawk, Fantastic Adventures of Dizzy, etc. */

void mapinit_camerica(void)
{
	if (cartridge->crc32==CRC32_FIRE_HAWK_UNL) mapper.camerica_isbf9097=TRUE;
	if (mapper.camerica_isbf9097) {
		cpu_set_write_io8000(mapio_camerica_bf9097,NULL,mapio_camerica_ce,mapio_camerica_ce);
		strcpy(mapper.mappername,"Camerica BF9097");
		cartridge->mirroring=CARTRIDGE_MIRRORING_ONE_0;
		RESET_MIRRORING();
	}
	else {
		cpu_set_write_io8000(NULL,NULL,mapio_camerica_ce,mapio_camerica_ce);
		strcpy(mapper.mappername,"Camerica BF9093");
	}		
	
	/* same as unrom */
	INIT_PRGBANKS_FIRSTLAST();
	INIT_PATBANKS_FIRST();
}

void __fastcall mapio_camerica_ce(register WORD address,register BYTE bus,register BYTE data)
{
	/* swap 16k prgbank in 0x8000, same as unrom */
	WORD prg=data<<1&MAXPRG;
	PRG_BS(0,prg++); PRG_BS(1,prg);
}

void __fastcall mapio_camerica_bf9097(register WORD address,register BYTE bus,register BYTE data)
{
	/* bit 4: set mirroring, as AOROM. bit 3:? (sometimes set) */
	ppu_force_update();
	if (data&BIT(4)) cartridge->mirroring=CARTRIDGE_MIRRORING_ONE_1;
	else cartridge->mirroring=CARTRIDGE_MIRRORING_ONE_0;
	RESET_MIRRORING();
}





/* --- iNES mapper 87: Jaleco/Konami 74161/32 ---
	Goonies (J), City Connection (J), Ninja Jajamaru Kun (J), etc. */

void mapinit_jaleco74161(void)
{
	strcpy(mapper.mappername,"Jaleco/Konami 74161/32");
	cpu_set_write_io6000(mapio_jaleco74161);
	INIT_PRGBANKS_FIRSTLAST(); /* 32k?:full, 16k?:mirrored */
	INIT_PATBANKS_FIRST(); /* first 8k vrom */
}

void __fastcall mapio_jaleco74161(register WORD address,register BYTE bus,register BYTE data)
{
	/* bits 01: select 8k vrom bank: 0=0, 1=2, 2=1, 3=3 */
	ppu_force_update();
	if ((data>>1^data)&1) data=~data;
	data=data<<3&BIN8(00011000)&MAXVROM;
	PAT_BS(0,data++,vrom); PAT_BS(1,data++,vrom); PAT_BS(2,data++,vrom); PAT_BS(3,data++,vrom);
	PAT_BS(4,data++,vrom); PAT_BS(5,data++,vrom); PAT_BS(6,data++,vrom); PAT_BS(7,data,vrom);
}





/* --- iNES mapper 93: Sunsoft 74161/32 ---
	Fantasy Zone (J) */

void mapinit_sunsoft74161(void)
{
	cpu_set_write_io8000(mapio_sunsoft74161,mapio_sunsoft74161,mapio_sunsoft74161,mapio_sunsoft74161);
	strcpy(mapper.mappername,"Sunsoft 74161/32");
	cartridge->mirroring=CARTRIDGE_MIRRORING_VERTICAL; RESET_MIRRORING();
	
	/* same as unrom */
	INIT_PRGBANKS_FIRSTLAST();
	INIT_PATBANKS_FIRST();
}

void __fastcall mapio_sunsoft74161(register WORD address,register BYTE bus,register BYTE data)
{
	/* high nibble: swap 16k prgbank in 0x8000 */
	BYTE prg=(data&BIN8(11110000))>>3&MAXPRG;
	PRG_BS(0,prg++); PRG_BS(1,prg);
	
	/* bit 0: set mirroring */
	ppu_force_update();
	if (data&1) cartridge->mirroring=CARTRIDGE_MIRRORING_HORIZONTAL;
	else cartridge->mirroring=CARTRIDGE_MIRRORING_VERTICAL;
	RESET_MIRRORING();
}





/* --- iNES mapper 94: Capcom 74161/32 ---
	Senjou no Ookami (Japanese version of Commando) */

void mapinit_capcom74161(void)
{
	cpu_set_write_io8000(mapio_capcom74161,mapio_capcom74161,mapio_capcom74161,mapio_capcom74161);
	strcpy(mapper.mappername,"Capcom 74161/32");

	/* same as unrom */
	INIT_PRGBANKS_FIRSTLAST();
	INIT_PATBANKS_FIRST();
}

void __fastcall mapio_capcom74161(register WORD address,register BYTE bus,register BYTE data)
{
	/* swap 16k prgbank in 0x8000 (from bit 2) */
	data=((data>>1)&BIN8(11111110))&MAXPRG;
	PRG_BS(0,data++); PRG_BS(1,data);
}





/* --- iNES mapper 97: Irem 74161/32 ---
	Kaiketsu Yanchamaru (Japanese version of Kid Niki) */

/* TODO:
	mirroring ? */

void mapinit_irem74161(void)
{
	cpu_set_write_io8000(mapio_irem74161,mapio_irem74161,mapio_irem74161,mapio_irem74161);
	strcpy(mapper.mappername,"Irem 74161/32");

	PRG_BS(0,MAXPRG-1); PRG_BS(1,MAXPRG); PRG_BS(2,0); PRG_BS(3,1); /* inverse of FIRSTLAST */
	INIT_PATBANKS_FIRST(); /* 8k vram */
}

void __fastcall mapio_irem74161(register WORD address,register BYTE bus,register BYTE data)
{
	/* swap 16k prgbank in 0xc000 */
	BYTE prg=data<<1&MAXPRG;
	PRG_BS(2,prg++); PRG_BS(3,prg);
	
	/* bits 67: set mirroring */
	if ((data<<1^data)&0x80) {
		ppu_force_update();
		if (data&0x80) cartridge->mirroring=CARTRIDGE_MIRRORING_VERTICAL;
		else cartridge->mirroring=CARTRIDGE_MIRRORING_HORIZONTAL;
		RESET_MIRRORING();
	}
}





/* --- iNES mapper 185: NES-CNROM with CHR ROM disable ---
	Sansuu 1 Nen,2,3, Banana, etc. */

/* TODO:
	find out correct behaviour (Banana looks dodgy, might be a bad dump) */

#define CNDIS_GET_PATDATA(x)	BYTE ret=*mapper.ppu_memmap_buffer_ptr; \
				if (mapper.cndis_latch&!*mapper.ppu_rendering_ptr) *mapper.ppu_memmap_buffer_ptr=mapper.cndis_data; \
				else *mapper.ppu_memmap_buffer_ptr=cartridge->cur_patbank[x][address&0x3ff]; \
				return ret

void mapinit_cndis(void)
{
	strcpy(mapper.mappername,"NES-CNROM with CHR ROM disable");
	cpu_set_write_io8000(mapio_cndis,mapio_cndis,mapio_cndis,mapio_cndis);
	ppu_set_fpt(PPU_FPT_PT0,FALSE,mapper_cndis_ppu_read_pt0a,mapper_cndis_ppu_read_pt0b,mapper_cndis_ppu_read_pt0c,mapper_cndis_ppu_read_pt0d,NULL,NULL,NULL,NULL);
	ppu_set_fpt(PPU_FPT_PT1,FALSE,mapper_cndis_ppu_read_pt1a,mapper_cndis_ppu_read_pt1b,mapper_cndis_ppu_read_pt1c,mapper_cndis_ppu_read_pt1d,NULL,NULL,NULL,NULL);
	mapper.cndis_data=0xff;
	INIT_PRGBANKS_FIRSTLAST(); /* 32k?:full, 16k?:mirrored */
	INIT_PATBANKS_FIRST(); /* 8k vrom */
}

/* ppu memmap overrides */
/* ppu patterntable reads: PT0 */
BYTE __fastcall mapper_cndis_ppu_read_pt0a(register WORD address) { CNDIS_GET_PATDATA(0); } /* 0x0000-0x03ff */
BYTE __fastcall mapper_cndis_ppu_read_pt0b(register WORD address) { CNDIS_GET_PATDATA(1); } /* 0x0400-0x07ff */
BYTE __fastcall mapper_cndis_ppu_read_pt0c(register WORD address) { CNDIS_GET_PATDATA(2); } /* 0x0800-0x0bff */
BYTE __fastcall mapper_cndis_ppu_read_pt0d(register WORD address) { CNDIS_GET_PATDATA(3); } /* 0x0c00-0x0fff */
/* ppu patterntable reads: PT1 */
BYTE __fastcall mapper_cndis_ppu_read_pt1a(register WORD address) { CNDIS_GET_PATDATA(4); } /* 0x1000-0x13ff */
BYTE __fastcall mapper_cndis_ppu_read_pt1b(register WORD address) { CNDIS_GET_PATDATA(5); } /* 0x1400-0x17ff */
BYTE __fastcall mapper_cndis_ppu_read_pt1c(register WORD address) { CNDIS_GET_PATDATA(6); } /* 0x1800-0x1bff */
BYTE __fastcall mapper_cndis_ppu_read_pt1d(register WORD address) { CNDIS_GET_PATDATA(7); } /* 0x1c00-0x1fff */
/* --- */

void __fastcall mapio_cndis(register WORD address,register BYTE bus,register BYTE data)
{
	ppu_force_update();

	#if DEBUG_MAPPER
	LOG(LOG_MAPPER,"%x %x\n",address,data);
	#endif
	
	/* disable/enable chr rom, method guessed */
	mapper.cndis_latch^=1;
	if (data) mapper.cndis_data=data;
}





/* --- iNES mapper 232: Camerica BF9096 ---
	Camerica's Quattro multi-games */

/* TODO:
	Quattro Sports (U) (Aladdin): BMX and Soccer are switched around, bad dump ?
	Quattro Sports (any): BMX: how do I play this thing ? */

static __inline__ void camerica_bf9096_bs(void);

void mapinit_camerica_bf9096(void)
{
	cpu_set_write_io8000(mapio_camerica_bf9096_8a,mapio_camerica_bf9096_8a,mapio_camerica_bf9096_ce,mapio_camerica_bf9096_ce);
	strcpy(mapper.mappername,"Camerica BF9096");
	
	camerica_bf9096_bs();
	INIT_PATBANKS_FIRST(); /* 8k vram */
}

void __fastcall mapio_camerica_bf9096_8a(register WORD address,register BYTE bus,register BYTE data)
{
	/* bits 34: select 64k block */
	mapper.camerica_bf9096_block=data&BIN8(00011000);
	camerica_bf9096_bs();
}
void __fastcall mapio_camerica_bf9096_ce(register WORD address,register BYTE bus,register BYTE data)
{
	/* select 16k bank in 64k block for 0x8000-0xbfff */
	mapper.camerica_bf9096_bank=data<<1&BIN8(00000110);
	camerica_bf9096_bs();
}

static __inline__ void camerica_bf9096_bs(void)
{
	BYTE prg0=(mapper.camerica_bf9096_block|mapper.camerica_bf9096_bank)&MAXPRG;
	BYTE prg1=(mapper.camerica_bf9096_block|BIN8(00000110))&MAXPRG;
	PRG_BS(0,prg0); PRG_BS(1,prg0|1); PRG_BS(2,prg1); PRG_BS(3,prg1|1);
}
