#include <windows.h>
#include <string.h>
#include <stdlib.h>
#include <dinput.h>

#include "global.h"
#include "log.h"
#include "input.h"

static struct {
	LPDIRECTINPUT dinput;
	LPDIRECTINPUTDEVICE keyboard;
	DIDEVCAPS keyboard_caps;
	BYTE keyboard_buffer[256];
	
	BYTE latch;
	BYTE shift[2];
	
	BYTE is_pressed[2][8];
	int key_vk[2][8];
	int key_dik[2][8];
} input;

void input_init(HWND hwnd)
{
	int i;
	memset(&input,0,sizeof(input));
	
	/* create it */
	if ((DirectInputCreate(GetModuleHandle(NULL),DIRECTINPUT_VERSION,&input.dinput,NULL))!=DI_OK) {
		LOG(LOG_MISC|LOG_ERROR,"can't create DirectInput!\n");
		input_clean(); exit(1);
	}
	
	/* create keyboard */
	if ((IDirectInput_CreateDevice(input.dinput,&GUID_SysKeyboard,&input.keyboard,NULL))!=DI_OK) {
		LOG(LOG_MISC|LOG_ERROR,"can't create DirectInput keyboard!\n");
		input_clean(); exit(1);
	}
	
	/* get keyboard capabilities */
	input.keyboard_caps.dwSize=sizeof(input.keyboard_caps);
	if ((IDirectInputDevice_GetCapabilities(input.keyboard,&input.keyboard_caps))!=DI_OK) {
		LOG(LOG_MISC|LOG_ERROR,"can't get DirectInput keyboard capabilities!\n");
		input_clean(); exit(1);
	}
	
	/* set keyboard dataformat */
	if ((IDirectInputDevice_SetDataFormat(input.keyboard,&c_dfDIKeyboard))!=DI_OK) {
		LOG(LOG_MISC|LOG_ERROR,"can't set DirectInput keyboard dataformat!\n");
		input_clean(); exit(1);
	}
	
	/* set keyboard cooperative level */
	if ((IDirectInputDevice_SetCooperativeLevel(input.keyboard,hwnd,DISCL_NONEXCLUSIVE|DISCL_BACKGROUND))!=DI_OK) {
		LOG(LOG_MISC|LOG_ERROR,"can't set DirectInput keyboard cooperative level!\n");
		input_clean(); exit(1);
	}
	
	/* acquire keyboard */
	for (i=100;(IDirectInputDevice_Acquire(input.keyboard)!=DI_OK)&(i>0);i--) Sleep(5);
	if (i==0) {
		LOG(LOG_MISC|LOG_ERROR,"can't acquire DirectInput keyboard!\n");
		input_clean(); exit(1);
	}
	
	input_default();

	LOG(LOG_VERBOSE,"input initialised\n");
}

void input_clean(void)
{
	if (input.dinput) { IDirectInput_Release(input.dinput); input.dinput=NULL; }
	if (input.keyboard) { IDirectInputDevice_Unacquire(input.keyboard); IDirectInputDevice_Release(input.keyboard); input.keyboard=NULL; }
	
	LOG(LOG_VERBOSE,"input cleaned\n");
}

void input_default(void)
{
	/* default controls
	A, B, select, start, up, down, left, right
	1: X, Z, C, V, up, down, left, right
	2: S, A, D, F, I, K, J, L
	*/
	
	/* joypad 1: */			/* joypad 2: */
	input.key_vk[0][0]=88;		input.key_vk[1][0]=83;
	input.key_vk[0][1]=90;		input.key_vk[1][1]=65;
	input.key_vk[0][2]=67;		input.key_vk[1][2]=68;
	input.key_vk[0][3]=86;		input.key_vk[1][3]=70;
	input.key_vk[0][4]=VK_UP;	input.key_vk[1][4]=73;
	input.key_vk[0][5]=VK_DOWN;	input.key_vk[1][5]=75;
	input.key_vk[0][6]=VK_LEFT;	input.key_vk[1][6]=74;
	input.key_vk[0][7]=VK_RIGHT;	input.key_vk[1][7]=76;
	
	input.key_dik[0][0]=DIK_X;	input.key_dik[1][0]=DIK_S;
	input.key_dik[0][1]=DIK_Z;	input.key_dik[1][1]=DIK_A;
	input.key_dik[0][2]=DIK_C;	input.key_dik[1][2]=DIK_D;
	input.key_dik[0][3]=DIK_V;	input.key_dik[1][3]=DIK_F;
	input.key_dik[0][4]=DIK_UP;	input.key_dik[1][4]=DIK_I;
	input.key_dik[0][5]=DIK_DOWN;	input.key_dik[1][5]=DIK_K;
	input.key_dik[0][6]=DIK_LEFT;	input.key_dik[1][6]=DIK_J;
	input.key_dik[0][7]=DIK_RIGHT;	input.key_dik[1][7]=DIK_L;
}


void input_read(HWND hwnd)
{
	int i,j;
	SHORT vk_keystate;
	HRESULT result;

	if (hwnd==GetForegroundWindow()) {
		/* dinput */
		result=IDirectInputDevice_GetDeviceState(input.keyboard,256,&input.keyboard_buffer);
		if ((result==DIERR_NOTACQUIRED)|(result==DIERR_INPUTLOST)) {
			for (i=10;(IDirectInputDevice_Acquire(input.keyboard)!=DI_OK)&(i>0);i--) ;
			if (i)
				result=IDirectInputDevice_GetDeviceState(input.keyboard,256,&input.keyboard_buffer);
		}
		if (result==DI_OK)
			for (i=0;i<2;i++)
				for(j=0;j<8;j++)
					input.is_pressed[i][j]=(input.keyboard_buffer[input.key_dik[i][j]]&0x80)!=0;
		
		/* winapi */
		else
			for (i=0;i<2;i++)
				for(j=0;j<8;j++) {
					vk_keystate=GetAsyncKeyState(input.key_vk[i][j]);
					input.is_pressed[i][j]=(BYTE)(((vk_keystate&1)&((SHORT)(!input.is_pressed[i][j])))|((vk_keystate&0x8000)!=0)); /* held key: always true, pressed key before the getkeystate: only true when it was not held/pressed before that */
				}
		
		/* prevent up+down or left+right */
		#if !INPUT_WORN_JOYPAD
		for (i=0;i<2;i++) {
			if (input.is_pressed[i][4]&input.is_pressed[i][5]) input.is_pressed[i][4]=input.is_pressed[i][5]=0;
			if (input.is_pressed[i][6]&input.is_pressed[i][7]) input.is_pressed[i][6]=input.is_pressed[i][7]=0;
		}
		#endif
	}
	else memset(input.is_pressed,0,16);
}

BYTE __fastcall input_read_port(register WORD address)
{
	BYTE ret=1;
	const BYTE j=address&1;
	
	if (!input.latch)
		if (input.shift[j]<8) {
			ret=input.is_pressed[j][input.shift[j]];
			input.shift[j]++;
		}
	return ret|(address>>8&~1);
}

void __fastcall input_write_port(register BYTE data)
{
	input.latch=data&1;
	if (input.latch) input.shift[0]=input.shift[1]=0;
}
