#include <stdlib.h>
#include <string.h>

#include "global.h"
#include "log.h"
#include "ppu.h"
#include "apu.h"
#include "input.h"
#include "cpu.h"
#include "cpu_memorymap.h"
#include "cartridge.h"
#include "mapper.h"

/*	0x10000
	0xc000		PRG bank 2							cpu_read/write_prgbank2a/b
	0x8000		PRG bank 1							cpu_read/write_prgbank1a/b
	0x6000		WRAM/SRAM(battery)						cpu_read/write_wram
	0x4000		sound/general IO, 32 registers.					cpu_read/write_io
	0x2000		PPU IO, 8 registers (mirrored 1024 times)			cpu_read/write_ppu
	0x0		RAM, 4kB, mirrored 4 times					cpu_read/write_ram
*/

static struct {
	BYTE* ram;
} cpu_memmap;


/* standard IO handlers */
static fp_cpu_read_io2000 cpu_read_io2000[8]; /* PPU IO (read) */
static fp_cpu_write_io2000 cpu_write_io2000[8]; /* PPU IO (write) */
static fp_cpu_read_io4000 cpu_read_io4000[0x20]; /* sound/general IO (read) */
static fp_cpu_write_io4000 cpu_write_io4000[0x20]; /* sound/general IO (write) */

BYTE __fastcall cpu_read_io_nothing(register WORD address) { return address>>8; }
void __fastcall cpu_write_io_nothing(register BYTE data) { return; }

void cpu_set_read_io2000(WORD address,fp_cpu_read_io2000 fun) { if (fun==NULL) fun=ppu_read_openbus; cpu_read_io2000[address&7]=fun; }
void cpu_set_write_io2000(WORD address,fp_cpu_write_io2000 fun) { if (fun==NULL) fun=cpu_write_io_nothing; cpu_write_io2000[address&7]=fun; }
void cpu_set_read_io4000(WORD address,fp_cpu_read_io4000 fun) { if (fun==NULL) fun=cpu_read_io_nothing; cpu_read_io4000[address&0x1f]=fun; }
void cpu_set_write_io4000(WORD address,fp_cpu_write_io4000 fun) { if (fun==NULL) fun=cpu_write_io_nothing; cpu_write_io4000[address&0x1f]=fun; }

void cpu_set_io2000_std(void)
{
	WORD address;
	
	/* read */
	for (address=0x2000;address<0x2008;address++) cpu_set_read_io2000(address,NULL);
	cpu_set_read_io2000(0x2002,ppu_read_status);
	cpu_set_read_io2000(0x2004,ppu_read_sprite_memory_data);
	cpu_set_read_io2000(0x2007,ppu_read_memory_data);
	
	/* write */
	for (address=0x2000;address<0x2008;address++) cpu_set_write_io2000(address,NULL);
	cpu_set_write_io2000(0x2000,ppu_write_control1);
	cpu_set_write_io2000(0x2001,ppu_write_control2);
	cpu_set_write_io2000(0x2003,ppu_write_sprite_memory_address);
	cpu_set_write_io2000(0x2004,ppu_write_sprite_memory_data);
	cpu_set_write_io2000(0x2005,ppu_write_scroll);
	cpu_set_write_io2000(0x2006,ppu_write_memory_address);
	cpu_set_write_io2000(0x2007,ppu_write_memory_data);
}
void cpu_set_io4000_std(void)
{
	WORD address;

	/* read */
	for (address=0x4000;address<0x4020;address++) cpu_set_read_io4000(address,NULL);
	cpu_set_read_io4000(0x4015,apu_read_status);
	cpu_set_read_io4000(0x4016,input_read_port);
	cpu_set_read_io4000(0x4017,input_read_port);
	
	/* write */
	for (address=0x4000;address<0x4020;address++) cpu_set_write_io4000(address,NULL);
	cpu_set_write_io4000(0x4010,apu_write_dmc_control);
	cpu_set_write_io4000(0x4011,apu_write_dmc_dac);
	cpu_set_write_io4000(0x4012,apu_write_dmc_address);
	cpu_set_write_io4000(0x4013,apu_write_dmc_length);
	cpu_set_write_io4000(0x4014,cpu_sprite_dma_transfer);
	cpu_set_write_io4000(0x4015,apu_write_status);
	cpu_set_write_io4000(0x4016,input_write_port);
	cpu_set_write_io4000(0x4017,apu_write_fs);
}

/* mapper IO handlers */
static fp_cpu_read_io4020 cpu_read_io4020; /* mapper IO at 0x4020-0x5fff (read) */
static fp_cpu_write_io4020 cpu_write_io4020; /* mapper IO at 0x4020-0x5fff (write) */
static fp_cpu_read_io6000 cpu_read_io6000; /* mapper IO at 0x6000-0x7fff (read) */
static fp_cpu_write_io6000 cpu_write_io6000; /* mapper IO at 0x6000-0x7fff (write) */
static fp_cpu_read_io8000 cpu_read_io8000[4]; /* mapper IO at 0x8000-0xffff (read), in chunks of 0x2000 */
static fp_cpu_write_io8000 cpu_write_io8000[4]; /* mapper IO at 0x8000-0xffff (write), in chunks of 0x2000 */

void cpu_set_read_io4020(fp_cpu_read_io4020 fun) { if (fun==NULL) fun=mapio_nothing_r; cpu_read_io4020=fun; }
void cpu_set_write_io4020(fp_cpu_write_io4020 fun) { if (fun==NULL) fun=mapio_nothing; cpu_write_io4020=fun; }
void cpu_set_read_io6000(fp_cpu_read_io6000 fun)
{
	if (fun!=NULL) {
		cpu_read_io6000=fun;
		cpu_set_fpt(CPU_FPT_WRAM_READ);
	}
}
void cpu_set_write_io6000(fp_cpu_write_io6000 fun)
{
	if (fun!=NULL) {
		cpu_write_io6000=fun;
		cpu_set_fpt(CPU_FPT_WRAM_WRITE);
	}
}
void cpu_set_read_io8000(fp_cpu_read_io8000 fun1,fp_cpu_read_io8000 fun2,fp_cpu_read_io8000 fun3,fp_cpu_read_io8000 fun4)
{
	if (fun1!=NULL) { cpu_read_io8000[0]=fun1; cpu_set_fpt(CPU_FPT_PRG1A_READ); }
	if (fun2!=NULL) { cpu_read_io8000[1]=fun2; cpu_set_fpt(CPU_FPT_PRG1B_READ); }
	if (fun3!=NULL) { cpu_read_io8000[2]=fun3; cpu_set_fpt(CPU_FPT_PRG2A_READ); }
	if (fun4!=NULL) { cpu_read_io8000[3]=fun4; cpu_set_fpt(CPU_FPT_PRG2B_READ); }
}
void cpu_set_write_io8000(fp_cpu_write_io8000 fun1,fp_cpu_write_io8000 fun2,fp_cpu_write_io8000 fun3,fp_cpu_write_io8000 fun4)
{
	if (fun1==NULL) fun1=mapio_nothing;
	if (fun2==NULL) fun2=mapio_nothing;
	if (fun3==NULL) fun3=mapio_nothing;
	if (fun4==NULL) fun4=mapio_nothing;

	cpu_write_io8000[0]=fun1; cpu_write_io8000[1]=fun2; cpu_write_io8000[2]=fun3; cpu_write_io8000[3]=fun4;
}


void cpu_memmap_init(void)
{
	memset(&cpu_memmap,0,sizeof(cpu_memmap));
	if ((cpu_memmap.ram=malloc(0x800))==NULL) { LOG(LOG_MISC|LOG_ERROR,"CPU RAM allocation error!\n"); exit(1); }
	memset(cpu_memmap.ram,CPU_RAM_INIT,0x800);
	
	cpu_set_io2000_std();
	cpu_set_io4000_std();
	cpu_set_fpt(CPU_FPT_NORMAL);
	
	LOG(LOG_VERBOSE,"CPU RAM allocated\nCPU memorymap initialised\n");
}

void cpu_memmap_clean(void)
{
	if (cpu_memmap.ram!=NULL) { free(cpu_memmap.ram); cpu_memmap.ram=NULL; }
	
	LOG(LOG_VERBOSE,"CPU RAM deallocated\nCPU memorymap cleaned\n");
}


/* ram */
BYTE __fastcall cpu_read_ram(register WORD address)
{
	#if DEBUG_CPU_MMAP
	LOG(LOG_CPU_MMAP,"cpu ramread ad: %x data: %x\n",address&0x7ff,cpu_memmap.ram[address&0x7ff]);
	#endif
	
	return cpu_memmap.ram[address&0x7ff];
}

void __fastcall cpu_write_ram(register WORD address,register BYTE data)
{
	cpu_memmap.ram[address&0x7ff]=data;
	
	#if DEBUG_CPU_MMAP
	LOG(LOG_CPU_MMAP,"cpu ramwrite ad: %x data: %x\n",address&0x7ff, data);
	#endif
}

/* ppu */
BYTE __fastcall cpu_read_ppu(register WORD address) { return cpu_read_io2000[address&7](address); }
void __fastcall cpu_write_ppu(register WORD address,register BYTE data) { cpu_write_io2000[address&7](data); }

/* io */
BYTE __fastcall cpu_read_io(register WORD address)
{
	if (address<=0x4020) return cpu_read_io4000[address&0x1f](address);
	else return cpu_read_io4020(address);
}

void __fastcall cpu_write_io(register WORD address,register BYTE data)
{
	if (address<=0x4020) cpu_write_io4000[address&0x1f](data);
	else cpu_write_io4020(address,0,data);
}

/* wram */
BYTE __fastcall cpu_read_wram(register WORD address) { if (cartridge->wram_isenabled) return cartridge->cur_wrambank[address&0x1fff]; else return address>>8; }
BYTE __fastcall cpu_read_wram_override(register WORD address) { return cpu_read_io6000(address); }
void __fastcall cpu_write_wram(register WORD address,register BYTE data) { if (cartridge->wram_isenabled) if (cartridge->wram_writeenable) cartridge->cur_wrambank[address&0x1fff]=data; }
void __fastcall cpu_write_wram_override(register WORD address,register BYTE data) { cpu_write_io6000(address,cartridge->cur_wrambank[address&0x1fff],data); }

/* --- prg/rom --- */
#define PRGREAD(x) cartridge->cur_prgbank[x][address&0x1fff]
#define LOG_PRGREAD(x) LOG(LOG_CPU_MMAP,"cpu prgbank%dread ad: %x(%x) data: %x\n",x,address,address&0x1fff,PRGREAD(x))
#define PRGWRITE(x) cpu_write_io8000[x](address,PRGREAD(x),data)
#define PRGREAD_OVER(x) return cpu_read_io8000[x](address)

/* read */
BYTE __fastcall cpu_read_prgbank1a(register WORD address) /* 0x8000 - 0x9fff */
{
	#if DEBUG_CPU_MMAP
	LOG_PRGREAD(0);
	#endif
	
	return PRGREAD(0);
}
BYTE __fastcall cpu_read_prgbank1b(register WORD address) /* 0xa000 - 0xbfff */
{
	#if DEBUG_CPU_MMAP
	LOG_PRGREAD(1);
	#endif
	
	return PRGREAD(1);
}
BYTE __fastcall cpu_read_prgbank2a(register WORD address) /* 0xc000 - 0xdfff */
{
	#if DEBUG_CPU_MMAP
	LOG_PRGREAD(2);
	#endif
	
	return PRGREAD(2);
}
BYTE __fastcall cpu_read_prgbank2b(register WORD address) /* 0xe000 - 0xffff */
{
	#if DEBUG_CPU_MMAP
	LOG_PRGREAD(3);
	#endif
	
	return PRGREAD(3);
}
BYTE __fastcall cpu_read_prgbank1a_override(register WORD address) { PRGREAD_OVER(0); } /* 0x8000 - 0x9fff */
BYTE __fastcall cpu_read_prgbank1b_override(register WORD address) { PRGREAD_OVER(1); } /* 0xa000 - 0xbfff */
BYTE __fastcall cpu_read_prgbank2a_override(register WORD address) { PRGREAD_OVER(2); } /* 0xc000 - 0xdfff */
BYTE __fastcall cpu_read_prgbank2b_override(register WORD address) { PRGREAD_OVER(3); } /* 0xe000 - 0xffff */

/* write (mapper io usually) */
void __fastcall cpu_write_prgbank1a(register WORD address,register BYTE data) { PRGWRITE(0); } /* 0x8000 - 0x9fff */
void __fastcall cpu_write_prgbank1b(register WORD address,register BYTE data) { PRGWRITE(1); } /* 0xa000 - 0xbfff */
void __fastcall cpu_write_prgbank2a(register WORD address,register BYTE data) { PRGWRITE(2); } /* 0xc000 - 0xdfff */
void __fastcall cpu_write_prgbank2b(register WORD address,register BYTE data) { PRGWRITE(3); } /* 0xe000 - 0xffff */
