#include <stdio.h>
#include <string.h>

#include "global.h"
#include "crystal.h"
#include "cpu_memorymap.h"
#include "log.h"
#include "cpu.h"
#include "ppu.h"
#include "mapper.h"
#include "cartridge.h"
#include "apu.h"

/*
NES RP2a03x (or 2a07 for PAL?) (minus sound) = 6502 minus decimal maths

Games/programs highly sensitive to CPU timing (NMI, IRQ, cycle deduction), these fall into the category 'fix something --> break something else':
currently working correctly (so the described problems do not happen) (a lot more, but these are the ones I use for testing):
	- Battletoads: lock up at level 2
	- Battletoads & Double Dragon: lock up at level 1
	- Cobra Triangle: lock up at or before the start of gameplay
	- Downtown Special (Kunio-Kun): garbage/flickering scanline above the statusbar
	- Fire Hawk: lock up in-game, or wrong cutoff anywhere
	- Gun Sight/Laser Invasion: screen shaking anywhere
	- Ironsword (W&W2): lock up at titlescreen (related to frame irq/vblank)
	- Kamen Rider Club: lock up at boot (related to cold boot 0x2006 disabled)
	- Kick Master: screen shaking at title, and at in-game statusbar
	- Marble Madness: garbage to the left of the textbox at the start of each level
	- nestank: lock up or wrong cutoff
	- Pin Bot: screen shaking after you die
	- Time Zone: screen shaking in the 3rd level
	- Uchuu Keibitai SDF: flickering or garbage in the intro
	- Wizards & Warriors 3: garbage in the intro, after the ghost shoots the wizard

	- pputime2 (by Quietust/Kevin Horton): block moves left or right
	- read2004 (by Quietust): wrong values (starting with zeroes instead of ff)
	- scanline (2 versions, by Quietust): wrong cutoff
currently not working correctly:
	- nothing
	
Other problems:
	- Dragon Fighter: demo mode messes up (normal?)


cpu bugs that are in the real 6502:
	- Return address pushed on the stack by JSR is one less than actual next instruction. RTS increments PC after popping. RTI doesn't.
	- The status bits pushed on the stack by PHP have the breakpoint bit set.
	- The D flag is not defined after reset.
	- The D flag is not cleared by interrupts.
	- An indirect JMP (xxFF) will fail because the MSB will be fetched from address xx00 instead of page xx+1.
	- If an interrupt occurs on a BRK instruction, the breakpoint is ignored.
*/

#define FLAG_B 0x10

static struct {
	BYTE A;		/* accumulator */
	BYTE X;		/* index register x */
	BYTE Y;		/* index register y */
	WORD PC;	/* program counter (HIGHBYTE PCH+LOWBYTE PCL) */
	BYTE S;		/* stack pointer ([(WORD)HIGHBYTE 1+]LOWBYTE S) */
	
	/* status register */
	WORD P_NZ;	/* negative flag: lowbyte=0x80 or highbyte=0x80. zero flag: lowbyte=0 */
	BYTE P_V;	/* overflow flag: x or 0 */
	/*B*/		/* brk flag: technically inexistant, the only way you're able to read/write it yourself, is when it's on the stack */
	BYTE P_D;	/* decimal flag: bit: unused in maths on the NES */
	BYTE P_I;	/* irq flag: bit */
	WORD P_C;	/* carry flag: highbyte=1, don't allow bits above */
	
	BYTE irq_pend;
	BYTE irq_type;
	BYTE irq_edge;
	BYTE interrupt;
	BYTE nmi;
	
	BYTE halt;
	int illegal_count;
	int cycles;
	int apu_fs_next_step;
	int apu_dmc_next_step;
	int apu_dmc_frequency;
} cpu;

static __inline__ void update_sync(void);
void illegal_nothing(BYTE);
void illegal_show(BYTE);
void (*_illegal)(BYTE); /* fp, or compiler will inline */
void __fastcall (*mapper_irq)(void);

/* opcode mnemonics */
static const char* c_op_mnemonic[]= { "BRK","ORA","HLT","SLO","NOP","ASL","PHP","AAC","BPL","CLC","JSR","AND","RLA","BIT","ROL","PLP","BMI","SEC","RTI","EOR","SRE","LSR","PHA","ASR","JMP","BVC","CLI","RTS","ADC","RRA","ROR","PLA","ARR","BVS","SEI","STA","AAX","STY","STX","DEY","TXA","XAA","BCC","AXA","TYA","TXS","XAS","SYA","SXA","LDY","LDA","LDX","LAX","TAY","TAX","ATX","BCS","CLV","TSX","LAR","CPY","CMP","DCP","DEC","INY","DEX","AXS","BNE","CLD","CPX","SBC","ISC","INC","INX","BEQ","SED" };
enum {                               _BRK=0,_ORA, _HLT, _SLO, _NOP, _ASL, _PHP, _AAC, _BPL, _CLC, _JSR, _AND, _RLA, _BIT, _ROL, _PLP, _BMI, _SEC, _RTI, _EOR, _SRE, _LSR, _PHA, _ASR, _JMP, _BVC, _CLI, _RTS, _ADC, _RRA, _ROR, _PLA, _ARR, _BVS, _SEI, _STA, _AAX, _STY, _STX, _DEY, _TXA, _XAA, _BCC, _AXA, _TYA, _TXS, _XAS, _SYA, _SXA, _LDY, _LDA, _LDX, _LAX, _TAY, _TAX, _ATX, _BCS, _CLV, _TSX, _LAR, _CPY, _CMP, _DCP, _DEC, _INY, _DEX, _AXS, _BNE, _CLD, _CPX, _SBC, _ISC, _INC, _INX, _BEQ, _SED  }; /* for readability */
static const BYTE op_mnemonic[0x100]=
{ /*  0    1    2    3    4    5    6    7    8    9    A    B    C    D    E    F
0 */ _BRK,_ORA,_HLT,_SLO,_NOP,_ORA,_ASL,_SLO,_PHP,_ORA,_ASL,_AAC,_NOP,_ORA,_ASL,_SLO, /* 0
1 */ _BPL,_ORA,_HLT,_SLO,_NOP,_ORA,_ASL,_SLO,_CLC,_ORA,_NOP,_SLO,_NOP,_ORA,_ASL,_SLO, /* 1
2 */ _JSR,_AND,_HLT,_RLA,_BIT,_AND,_ROL,_RLA,_PLP,_AND,_ROL,_AAC,_BIT,_AND,_ROL,_RLA, /* 2
3 */ _BMI,_AND,_HLT,_RLA,_NOP,_AND,_ROL,_RLA,_SEC,_AND,_NOP,_RLA,_NOP,_AND,_ROL,_RLA, /* 3
4 */ _RTI,_EOR,_HLT,_SRE,_NOP,_EOR,_LSR,_SRE,_PHA,_EOR,_LSR,_ASR,_JMP,_EOR,_LSR,_SRE, /* 4
5 */ _BVC,_EOR,_HLT,_SRE,_NOP,_EOR,_LSR,_SRE,_CLI,_EOR,_NOP,_SRE,_NOP,_EOR,_LSR,_SRE, /* 5
6 */ _RTS,_ADC,_HLT,_RRA,_NOP,_ADC,_ROR,_RRA,_PLA,_ADC,_ROR,_ARR,_JMP,_ADC,_ROR,_RRA, /* 6
7 */ _BVS,_ADC,_HLT,_RRA,_NOP,_ADC,_ROR,_RRA,_SEI,_ADC,_NOP,_RRA,_NOP,_ADC,_ROR,_RRA, /* 7
8 */ _NOP,_STA,_NOP,_AAX,_STY,_STA,_STX,_AAX,_DEY,_NOP,_TXA,_XAA,_STY,_STA,_STX,_AAX, /* 8
9 */ _BCC,_STA,_HLT,_AXA,_STY,_STA,_STX,_AAX,_TYA,_STA,_TXS,_XAS,_SYA,_STA,_SXA,_AXA, /* 9
A */ _LDY,_LDA,_LDX,_LAX,_LDY,_LDA,_LDX,_LAX,_TAY,_LDA,_TAX,_ATX,_LDY,_LDA,_LDX,_LAX, /* A
B */ _BCS,_LDA,_HLT,_LAX,_LDY,_LDA,_LDX,_LAX,_CLV,_LDA,_TSX,_LAR,_LDY,_LDA,_LDX,_LAX, /* B
C */ _CPY,_CMP,_NOP,_DCP,_CPY,_CMP,_DEC,_DCP,_INY,_CMP,_DEX,_AXS,_CPY,_CMP,_DEC,_DCP, /* C
D */ _BNE,_CMP,_HLT,_DCP,_NOP,_CMP,_DEC,_DCP,_CLD,_CMP,_NOP,_DCP,_NOP,_CMP,_DEC,_DCP, /* D
E */ _CPX,_SBC,_NOP,_ISC,_CPX,_SBC,_INC,_ISC,_INX,_SBC,_NOP,_SBC,_CPX,_SBC,_INC,_ISC, /* E
F */ _BEQ,_SBC,_HLT,_ISC,_NOP,_SBC,_INC,_ISC,_SED,_SBC,_NOP,_ISC,_NOP,_SBC,_INC,_ISC  /* F
      0    1    2    3    4    5    6    7    8    9    A    B    C    D    E    F    */
};

/* opcode addressing modes */
static const char* addressing_mode_names[]= { "Accumulator", "Immediate", "Absolute", "Zero Page", "Indexed Zero Page (X)", "Indexed Zero Page (Y)", "Indexed Absolute (X)", "Indexed Absolute (Y)", "Implied", "Relative", "Indexed Indirect (X)", "Indirectly Indexed (Y)", "Indirect" };
enum {                                        _ACC=0,        _IMM,        _ABS,       _ZP,         _ZPX,                    _ZPY,                    _ABSX,                  _ABSY,                  _IMP,      _REL,       _INDX,                  _INDY,                    _INDR }; /* for readability */
static const BYTE op_addressing_mode[0x100]=
{ /*  0     1     2     3     4     5     6     7     8     9     A     B     C     D     E     F
0 */ _IMP, _INDX,_IMP, _INDX,_ZP,  _ZP,  _ZP,  _ZP,  _IMP, _IMM, _ACC, _IMM, _ABS, _ABS, _ABS, _ABS,  /* 0
1 */ _REL, _INDY,_IMP, _INDY,_ZPX, _ZPX, _ZPX, _ZPX, _IMP, _ABSY,_IMP, _ABSY,_ABSX,_ABSX,_ABSX,_ABSX, /* 1
2 */ _ABS, _INDX,_IMP, _INDX,_ZP,  _ZP,  _ZP,  _ZP,  _IMP, _IMM, _ACC, _IMM, _ABS, _ABS, _ABS, _ABS,  /* 2
3 */ _REL, _INDY,_IMP, _INDY,_ZPX, _ZPX, _ZPX, _ZPX, _IMP, _ABSY,_IMP, _ABSY,_ABSX,_ABSX,_ABSX,_ABSX, /* 3
4 */ _IMP, _INDX,_IMP, _INDX,_ZP,  _ZP,  _ZP,  _ZP,  _IMP, _IMM, _ACC, _IMM, _ABS, _ABS, _ABS, _ABS,  /* 4
5 */ _REL, _INDY,_IMP, _INDY,_ZPX, _ZPX, _ZPX, _ZPX, _IMP, _ABSY,_IMP, _ABSY,_ABSX,_ABSX,_ABSX,_ABSX, /* 5
6 */ _IMP, _INDX,_IMP, _INDX,_ZP,  _ZP,  _ZP,  _ZP,  _IMP, _IMM, _ACC, _IMM, _INDR,_ABS, _ABS, _ABS,  /* 6
7 */ _REL, _INDY,_IMP, _INDY,_ZPX, _ZPX, _ZPX, _ZPX, _IMP, _ABSY,_IMP, _ABSY,_ABSX,_ABSX,_ABSX,_ABSX, /* 7
8 */ _IMM, _INDX,_IMM, _INDX,_ZP,  _ZP,  _ZP,  _ZP,  _IMP, _IMM, _IMP, _IMM, _ABS, _ABS, _ABS, _ABS,  /* 8
9 */ _REL, _INDY,_IMP, _INDY,_ZPX, _ZPX, _ZPY, _ZPY, _IMP, _ABSY,_IMP, _ABSY,_ABSX,_ABSX,_ABSY,_ABSY, /* 9
A */ _IMM, _INDX,_IMM, _INDX,_ZP,  _ZP,  _ZP,  _ZP,  _IMP, _IMM, _IMP, _IMM, _ABS, _ABS, _ABS, _ABS,  /* A
B */ _REL, _INDY,_IMP, _INDY,_ZPX, _ZPX, _ZPY, _ZPY, _IMP, _ABSY,_IMP, _ABSY,_ABSX,_ABSX,_ABSY,_ABSY, /* B
C */ _IMM, _INDX,_IMM, _INDX,_ZP,  _ZP,  _ZP,  _ZP,  _IMP, _IMM, _IMP, _IMM, _ABS, _ABS, _ABS, _ABS,  /* C
D */ _REL, _INDY,_IMP, _INDY,_ZPX, _ZPX, _ZPX, _ZPX, _IMP, _ABSY,_IMP, _ABSY,_ABSX,_ABSX,_ABSX,_ABSX, /* D
E */ _IMM, _INDX,_IMM, _INDX,_ZP,  _ZP,  _ZP,  _ZP,  _IMP, _IMM, _IMP, _IMM, _ABS, _ABS, _ABS, _ABS,  /* E
F */ _REL, _INDY,_IMP, _INDY,_ZPX, _ZPX, _ZPX, _ZPX, _IMP, _ABSY,_IMP, _ABSY,_ABSX,_ABSX,_ABSX,_ABSX  /* F
      0     1     2     3     4     5     6     7     8     9     A     B     C     D     E     F     */
};

/* true(1): official/documented, false: illegal/undocumented */
static const BYTE op_official[0x100]=
{ /* 0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F
0 */ 1, 1, 0, 0, 0, 1, 1, 0, 1, 1, 1, 0, 0, 1, 1, 0, /* 0
1 */ 1, 1, 0, 0, 0, 1, 1, 0, 1, 1, 0, 0, 0, 1, 1, 0, /* 1
2 */ 1, 1, 0, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, /* 2
3 */ 1, 1, 0, 0, 0, 1, 1, 0, 1, 1, 0, 0, 0, 1, 1, 0, /* 3
4 */ 1, 1, 0, 0, 0, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, /* 4
5 */ 1, 1, 0, 0, 0, 1, 1, 0, 1, 1, 0, 0, 0, 1, 1, 0, /* 5
6 */ 1, 1, 0, 0, 0, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, /* 6
7 */ 1, 1, 0, 0, 0, 1, 1, 0, 1, 1, 0, 0, 0, 1, 1, 0, /* 7
8 */ 0, 1, 0, 0, 1, 1, 1, 0, 1, 0, 1, 0, 1, 1, 1, 0, /* 8
9 */ 1, 1, 0, 0, 1, 1, 1, 0, 1, 1, 1, 0, 0, 1, 0, 0, /* 9
A */ 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, /* A
B */ 1, 1, 0, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, /* B
C */ 1, 1, 0, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, /* C
D */ 1, 1, 0, 0, 0, 1, 1, 0, 1, 1, 0, 0, 0, 1, 1, 0, /* D
E */ 1, 1, 0, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, /* E
F */ 1, 1, 0, 0, 0, 1, 1, 0, 1, 1, 0, 0, 0, 1, 1, 0  /* F
     0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F  */
};

/* number of cycles to burn for each opcode */
static int op_cycles_lookup[0x100]; /* opcode cycles multiplied by ntsc/pal cycle */
static const BYTE op_cycles[0x100]=
/* 00-BRK(0) is 7, cycles are deducted in the interrupt handler instead */
{ /* 0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F
0 */ 0, 6, 0, 8, 3, 3, 5, 5, 3, 2, 2, 2, 4, 4, 6, 6, /* 0
1 */ 2, 5, 0, 8, 4, 4, 6, 6, 2, 4, 2, 7, 4, 4, 7, 7, /* 1
2 */ 6, 6, 0, 8, 3, 3, 5, 5, 4, 2, 2, 2, 4, 4, 6, 6, /* 2
3 */ 2, 5, 0, 8, 4, 4, 6, 6, 2, 4, 2, 7, 4, 4, 7, 7, /* 3
4 */ 6, 6, 0, 8, 3, 3, 5, 5, 3, 2, 2, 2, 3, 4, 6, 6, /* 4
5 */ 2, 5, 0, 8, 4, 4, 6, 6, 2, 4, 2, 7, 4, 4, 7, 7, /* 5
6 */ 6, 6, 0, 8, 3, 3, 5, 5, 4, 2, 2, 2, 5, 4, 6, 6, /* 6
7 */ 2, 5, 0, 8, 4, 4, 6, 6, 2, 4, 2, 7, 4, 4, 7, 7, /* 7
8 */ 2, 6, 2, 6, 3, 3, 3, 3, 2, 2, 2, 2, 4, 4, 4, 4, /* 8
9 */ 2, 6, 0, 6, 4, 4, 4, 4, 2, 5, 2, 5, 5, 5, 5, 5, /* 9
A */ 2, 6, 2, 6, 3, 3, 3, 3, 2, 2, 2, 2, 4, 4, 4, 4, /* A
B */ 2, 5, 0, 5, 4, 4, 4, 4, 2, 4, 2, 4, 4, 4, 4, 4, /* B
C */ 2, 6, 2, 8, 3, 3, 5, 5, 2, 2, 2, 2, 4, 4, 6, 6, /* C
D */ 2, 5, 0, 8, 4, 4, 6, 6, 2, 4, 2, 7, 4, 4, 7, 7, /* D
E */ 2, 6, 2, 8, 3, 3, 5, 5, 2, 2, 2, 2, 4, 4, 6, 6, /* E
F */ 2, 5, 0, 8, 4, 4, 6, 6, 2, 4, 2, 7, 4, 4, 7, 7  /* F
     0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F  */
};

/* extra cycles to burn on page overflow: only on addressing modes ABS_X, ABS_Y, IND_Y (and REL). can be calculated with masks, but this is faster */
static const WORD op_extracycles[0x100]=
{ /* 0   1   2   3   4   5   6   7   8   9   A   B   C   D   E   F
0 */ 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,   /* 0
1 */ 0,  256,0,  0,  0,  0,  0,  0,  0,  256,0,  0,  256,256,0,  0,   /* 1
2 */ 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,   /* 2
3 */ 0,  256,0,  0,  0,  0,  0,  0,  0,  256,0,  0,  256,256,0,  0,   /* 3
4 */ 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,   /* 4
5 */ 0,  256,0,  0,  0,  0,  0,  0,  0,  256,0,  0,  256,256,0,  0,   /* 5
6 */ 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,   /* 6
7 */ 0,  256,0,  0,  0,  0,  0,  0,  0,  256,0,  0,  256,256,0,  0,   /* 7
8 */ 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,   /* 8
9 */ 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,   /* 9
A */ 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,   /* A
B */ 256,256,256,256,256,256,256,256,256,256,256,256,256,256,256,256, /* B
C */ 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,   /* C
D */ 0,  256,0,  0,  0,  0,  0,  0,  0,  256,0,  0,  256,256,0,  0,   /* D
E */ 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,   /* E
F */ 0,  256,0,  0,  0,  0,  0,  0,  0,  256,0,  0,  256,256,0,  0    /* F
     0   1   2   3   4   5   6   7   8   9   A   B   C   D   E   F    */
};


/* interrupt handler: function declarations, defines, fpt define, functionpointertable */
static void int_none(void); static void int_brk(void); static void int_irq(void); static void int_nmi(void); static void int_res(void); static void int_mul(void);
typedef void(*fpt_handle_interrupt)(void);
static const fpt_handle_interrupt handle_interrupt[0x10]= { int_none, int_brk, int_irq, int_mul, int_nmi, int_mul, int_mul, int_mul, int_res, int_mul, int_mul, int_mul, int_mul, int_mul, int_mul, int_mul };
#define IRQ_PEND(x) if (tempb&cpu.irq_pend&!cpu.P_I) if ((cpu.interrupt&INTERRUPT_IRQ)==0) { cpu.interrupt|=INTERRUPT_IRQ; cpu.irq_type|=x; }



/* shortcuts */

/* memory handler */
typedef BYTE __fastcall(*fpt_cpu_read_memorymap)(register WORD);
typedef void __fastcall(*fpt_cpu_write_memorymap)(register WORD,register BYTE);
static fpt_cpu_read_memorymap cpu_read_memorymap[9]; /* last one in case of overflow */
static fpt_cpu_write_memorymap cpu_write_memorymap[9]; /* last one in case of overflow */
#define READ_BYTE_RAM(x) cpu_read_ram(x)
#define WRITE_BYTE_RAM(x,y) cpu_write_ram(x,y)
#define READ_BYTE(x) cpu_read_memorymap[(x)>>13](x)
#define WRITE_BYTE(x,y) cpu_write_memorymap[(x)>>13](x,y)
#define READ_WORD(x) READ_BYTE(x)|(READ_BYTE((x)+1)<<8)
#define DATA READ_BYTE(address)

/* branch */
#define BRANCH() address=cpu.PC+(signed char)DATA; cpu.cycles-=crystal->cycle<<((cpu.PC^address)>>8&1); cpu.PC=address /* shift left with result (1 if other mempage, 0 if not)=2 or 1 cycle */

/* stack */
#define PUSH(x) WRITE_BYTE_RAM(0x100|cpu.S--,x)
#define PULL() READ_BYTE_RAM(++cpu.S|0x100)

#define PULL_PC() tempb=PULL(); cpu.PC=PULL(); cpu.PC=cpu.PC<<8|tempb /* pull twice, store in program counter */
#define PUSH_PC() PUSH(cpu.PC>>8); PUSH(cpu.PC&0xff) /* push program counter */

/* push/pull status register NV1BDIZC (gotta convert every var to bits. tough job, but hardly used, so the var method of flags is faster) */
#define PUSH_STATUS(b) PUSH(((cpu.P_NZ>>8|cpu.P_NZ)&0x80)|((cpu.P_V!=0)<<6)|BIT(5)|b|(cpu.P_D<<3)|(cpu.P_I<<2)|(((cpu.P_NZ&0xff)==0)<<1)|(cpu.P_C>>8))
#define PULL_STATUS() cpu.P_NZ=PULL(); cpu.P_V=cpu.P_NZ&BIT(6); cpu.P_D=cpu.P_NZ>>3&1; cpu.P_I=cpu.P_NZ>>2&1; cpu.P_C=cpu.P_NZ<<8&0x100; cpu.P_NZ=(~cpu.P_NZ&2)|cpu.P_NZ<<8


/* opcodes */

/*
official opcodes:
	ADC, AND, ASL, BCC, BCS, BEQ, BIT, BMI, BNE, BPL, BRK, BVC, BVS, CLC, CLD, CLI, CLV, CMP, CPX, CPY, DEC, DEX, DEY, EOR, INC,
	INX, INY, JMP, JSR, LDA, LDX, LDY, LSR, NOP, ORA, PHA, PHP, PLA, PLP, ROL, ROR, RTI, RTS, SBC, SEC, SED, SEI, STA, STX, STY,
	TAX, TAY, TSX, TXA, TXS, TYA

opcode:mnemonic:addressing mode:bytes:cycles(*=extra cycle on page overflow, **=extra cycle on branch success and extra cycle on page overflow). !=unofficial opcode
 00:BRK:Imp:1:(7)   20:JSR:Abs:3:6     40:RTI:Imp:1:6     60:RTS:Imp:1:6     80:!               A0:LDY:Imm:2:2     C0:CPY:Imm:2:2     E0:CPX:Imm:2:2
 01:ORA:(Ind,X):2:6 21:AND:(Ind,X):2:6 41:EOR:(Ind,X):2:6 61:ADC:(Ind,X):2:6 81:STA:(Ind,X):2:6 A1:LDA:(Ind,X):2:6 C1:CMP:(Ind,X):2:6 E1:SBC:(Ind,X):2:6
 02:!               22:!               42:!               62:!               82:!               A2:LDX:Imm:2:2     C2:!               E2:!           
 03:!               23:!               43:!               63:!               83:!               A3:!               C3:!               E3:!           
 04:!               24:BIT:ZP:2:3      44:!               64:!               84:STY:ZP:2:3      A4:LDY:ZP:2:3      C4:CPY:ZP:2:3      E4:CPX:ZP:2:3
 05:ORA:ZP:2:3      25:AND:ZP:2:3      45:EOR:ZP:2:3      65:ADC:ZP:2:3      85:STA:ZP:2:3      A5:LDA:ZP:2:3      C5:CMP:ZP:2:3      E5:SBC:ZP:2:3
 06:ASL:ZP:2:5      26:ROL:ZP:2:5      46:LSR:ZP:2:5      66:ROR:ZP:2:5      86:STX:ZP:2:3      A6:LDX:ZP:2:3      C6:DEC:ZP:2:5      E6:INC:ZP:2:5
 07:!               27:!               47:!               67:!               87:!               A7:!               C7:!               E7:!           
 08:PHP:Imp:1:3     28:PLP:Imp:1:4     48:PHA:Imp:1:3     68:PLA:Imp:1:4     88:DEY:Imp:1:2     A8:TAY:Imp:1:2     C8:INY:Imp:2:2     E8:INX:Imp:1:2
 09:ORA:Imm:2:2     29:AND:Imm:2:2     49:EOR:Imm:2:2     69:ADC:Imm:2:2     89:!               A9:LDA:Imm:2:2     C9:CMP:Imm:2:2     E9:SBC:Imm:2:2
 0A:ASL:Accum:1:2   2A:ROL:Accum:1:2   4A:LSR:Accum:1:2   6A:ROR:Accum:1:2   8A:TXA:Imp:1:2     AA:TAX:Imp:1:2     CA:DEX:Imp:1:2     EA:NOP:Imp:1:2
 0B:!               2B:!               4B:!               6B:!               8B:!               AB:!               CB:!               EB:!           
 0C:!               2C:BIT:Abs:3:4     4C:JMP:Abs:3:3     6C:JMP:Ind:3:5     8C:STY:Abs:3:4     AC:LDY:Abs:3:4     CC:CPY:Abs:3:4     EC:CPX:Abs:3:4
 0D:ORA:Abs:3:4     2D:AND:Abs:3:4     4D:EOR:Abs:3:4     6D:ADC:Abs:3:4     80:STA:Abs:3:4     AD:LDA:Abs:3:4     CD:CMP:Abs:3:4     ED:SBC:Abs:3:4
 0E:ASL:Abs:3:6     2E:ROL:Abs:3:4     4E:LSR:Abs:3:6     6E:ROR:Abs:3:6     8E:STX:Abs:3:4     AE:LDX:Abs:3:4     CE:DEC:Abs:3:6     EE:INC:Abs:3:6
 0F:!               2F:!               4F:!               6F:!               8F:!               AF:!               CF:!               EF:!           
 10:BPL:Rel:2:2**   30:BMI:Rel:2:2**   50:BVC:Rel:2:2**   70:BVS:Rel:2:2**   90:BCC:Rel:2:2**   B0:BCS:Rel:2:2**   D0:BNE:Rel:2:2**   F0:BEQ:Rel:2:2**
 11:ORA:(Ind),Y:2:5*31:AND:(Ind),Y:2:5*51:EOR:(Ind),Y:2:5*71:ADC:(Ind),Y:2:5*91:STA:(Ind),Y:2:5 B1:LDA:(Ind),Y:2:5*D1:CMP:(Ind),Y:2:5*F1:SBC:(Ind),Y:2:5*
 12:!               32:!               52:!               72:!               92:!               B2:!               D2:!               F2:!           
 13:!               33:!               53:!               73:!               93:!               B3:!               D3:!               F3:!           
 14:!               34:!               54:!               74:!               94:STY:ZP,X:2:4    B4:LDY:ZP,X:2:4    D4:!               F4:!           
 15:ORA:ZP,X:2:4    35:AND:ZP,X:2:4    55:EOR:ZP,X:2:4    75:ADC:ZP,X:2:4    95:STA:ZP,X:2:4    B5:LDA:ZP,X:2:4    D5:CMP:ZP,X:2:4    F5:SBC:ZP,X:2:4
 16:ASL:ZP,X:2:6    36:ROL:ZP,X:2:6    56:LSR:ZP,X:2:6    76:ROR:ZP,X:2:6    96:STX:ZP,Y:2:4    B6:LDX:ZP,Y:2:4    D6:DEC:ZP,X:2:6    F6:INC:ZP,X:2:6
 17:!               37:!               57:!               77:!               97:!               B7:!               D7:!               F7:!           
 18:CLC:Imp:1:2     38:SEC:Imp:1:2     58:CLI:Imp:1:2     78:SEI:Imp:1:2     98:TYA:Imp:1:2     B8:CLV:Imp:1:2     D8:CLD:Imp:1:2     F8:SED:Imp:1:2
 19:ORA:Abs,Y:3:4*  39:AND:Abs,Y:3:4*  59:EOR:Abs,Y:3:4*  79:ADC:Abs,Y:3:4*  99:STA:Abs,Y:3:5   B9:LDA:Abs,Y:3:4*  D9:CMP:Abs,Y:3:4*  F9:SBC:Abs,Y:3:4*
 1A:!               3A:!               5A:!               7A:!               9A:TXS:Imp:1:2     BA:TSX:Imp:1:2     DA:!               FA:!           
 1B:!               3B:!               5B:!               7B:!               9B:!               BB:!               DB:!               FB:!           
 1C:!               3C:!               5C:!               7C:!               9C:!               BC:LDY:Abs,X:3:4*  DC:!               FC:!           
 1D:ORA:Abs,X:3:4*  3D:AND:Abs,X:3:4*  50:EOR:Abs,X:3:4*  70:ADC:Abs,X:3:4*  90:STA:Abs,X:3:5   BD:LDA:Abs,X:3:4*  DD:CMP:Abs,X:3:4*  FD:SBC:Abs,X:3:4*
 1E:ASL:Abs,X:3:7   3E:ROL:Abs,X:3:7   5E:LSR:Abs,X:3:7   7E:ROR:Abs,X:3:7   9E:!               BE:LDX:Abs,Y:3:4*  DE:DEC:Abs,X:3:7   FE:INC:Abs,X:3:7
 1F:!               3F:!               5F:!               7F:!               9F:!               BF:!               DF:!               FF:!           
*/

/* add memory to accumulator with carry. affected flags: negative, zero, overflow, carry */
#define OP_ADC()	tempb=DATA; \
			cpu.P_C=(cpu.P_C>>8)+cpu.A+tempb; \
			cpu.P_V=(cpu.A^cpu.P_C)&(tempb^cpu.P_C)&0x80; \
			cpu.P_NZ=cpu.A=cpu.P_C&0xff
/* AND memory with accumulator. affected flags: negative, zero */
#define OP_AND()	cpu.P_NZ=cpu.A&=DATA
/* shift left one bit, memory. affected flags: negative, zero, carry */
#define OP_ASL()	tempb=DATA; \
			WRITE_BYTE(address,tempb); /* rmw garbage store */ \
			cpu.P_NZ=cpu.P_C=tempb<<1; \
			WRITE_BYTE(address,cpu.P_C&0xff)
#define OP_aASL()	cpu.P_NZ=cpu.P_C=cpu.A<<1; /* same as above, accumulator */ \
			cpu.A=cpu.P_C&0xff
/* branch on carry clear */
#define OP_BCC();	if (~cpu.P_C&0x100) { BRANCH(); }
/* branch on carry set */
#define OP_BCS();	if (cpu.P_C&0x100) { BRANCH(); }
/* branch on result zero */
#define OP_BEQ();	if ((cpu.P_NZ&0xff)==0) { BRANCH(); }
/* test bits in memory with accumulator. zero flag=A&mem=0, negative flag=bit 7 of mem(same as negative), overflow flag=bit 6 of mem. affected flags: negative, overflow, zero */
#define OP_BIT()	tempb=DATA; \
			cpu.P_NZ=(((cpu.A&tempb)|(cpu.A&tempb)>>1)|(tempb<<8))&BIN16(11111111,01111111); \
			cpu.P_V=tempb&BIT(6)
/* branch on result minus */
#define OP_BMI();	if (cpu.P_NZ&0x8080) { BRANCH(); }
/* branch on result not zero */
#define OP_BNE();	if (cpu.P_NZ&0xff) { BRANCH(); }
/* branch on result plus */
#define OP_BPL();	if ((cpu.P_NZ&0x8080)==FALSE) { BRANCH(); }
/* force break. affected flags: "break", irq */
#define OP_BRK()	cpu.PC++; /* padme */ \
			cpu.interrupt|=INTERRUPT_BRK
/* branch on overflow clear */
#define OP_BVC();	if (cpu.P_V==FALSE) { BRANCH(); }
/* branch on overflow set */
#define OP_BVS();	if (cpu.P_V) { BRANCH(); }
/* clear carry flag */
#define OP_CLC()	cpu.P_C=FALSE
/* clear decimal mode */
#define OP_CLD()	cpu.P_D=FALSE
/* clear interrupt disable bit */
#define OP_CLI()	tempb=cpu.P_I; \
			cpu.P_I=FALSE; \
			IRQ_PEND(INT_EDGE_IRQ)
/* clear overflow flag */
#define OP_CLV()	cpu.P_V=FALSE
/* compare memory and accumulator (A-mem without result in A). affected flags: negative, zero, carry */
#define OP_CMP()	cpu.P_C=cpu.P_NZ=cpu.A+0x100-DATA
/* compare memory and index x (X-mem without result in X). affected flags: negative, zero, carry */
#define OP_CPX()	cpu.P_C=cpu.P_NZ=cpu.X+0x100-DATA
/* compare memory and index y (Y-mem without result in Y). affected flags: negative, zero, carry */
#define OP_CPY()	cpu.P_C=cpu.P_NZ=cpu.Y+0x100-DATA
/* decrement memory by one. affected flags: negative, zero */
#define OP_DEC()	tempb=DATA; \
			WRITE_BYTE(address,tempb); /* rmw garbage store */ \
			cpu.P_NZ=tempb+0xff; \
			WRITE_BYTE(address,cpu.P_NZ&0xff)
/* decrement index X by one. affected flags: negative, zero */
#define OP_DEX()	cpu.P_NZ=cpu.X+=0xff
/* decrement index Y by one. affected flags: negative, zero */
#define OP_DEY()	cpu.P_NZ=cpu.Y+=0xff
/* exclusive-OR memory with accumulator. affected flags: negative, zero */
#define OP_EOR()	cpu.P_NZ=cpu.A^=DATA
/* increment memory by one. affected flags: negative, zero */
#define OP_INC()	tempb=DATA; \
			WRITE_BYTE(address,tempb); /* rmw garbage store */ \
			cpu.P_NZ=tempb+1; \
			WRITE_BYTE(address,cpu.P_NZ&0xff)
/* increment index X by one. affected flags: negative, zero */
#define OP_INX()	cpu.P_NZ=++cpu.X
/* increment index Y by one. affected flags: negative, zero */
#define OP_INY()	cpu.P_NZ=++cpu.Y
/* jump to new location */
#define OP_JMP()	cpu.PC=address
/* jump to new location saving return address. there's a bug in the 6502 that the return address in the stack for jsr is one lower than it should be, it is incremented again though on rts */
#define OP_JSR()	cpu.PC--; \
			PUSH_PC(); \
			cpu.PC=address
/* load accumulator with memory. affected flags: negative, zero */
#define OP_LDA()	cpu.P_NZ=cpu.A=DATA
/* load index X with memory. affected flags: negative, zero */
#define OP_LDX()	cpu.P_NZ=cpu.X=DATA
/* load index Y with memory. affected flags: negative, zero */
#define OP_LDY()	cpu.P_NZ=cpu.Y=DATA
/* shift one bit right (memory), carries to the right instead of default left. affected flags: negative, zero, carry */
#define OP_LSR()	tempb=DATA; \
			WRITE_BYTE(address,tempb); /* rmw garbage store */ \
			cpu.P_NZ=tempb>>1; \
			cpu.P_C=tempb<<8&0x100; \
			WRITE_BYTE(address,cpu.P_NZ)
#define OP_aLSR()	cpu.P_NZ=cpu.A>>1; /* same as above, accumulator */ \
			cpu.P_C=cpu.A<<8&0x100; \
			cpu.A=cpu.P_NZ
/* no operation (NOP) */
/* nothing */
/* OR memory with accumulator. affected flags: negative, zero */
#define OP_ORA()	cpu.P_NZ=cpu.A|=DATA
/* push accumulator on stack */
#define OP_PHA()	PUSH(cpu.A)
/* push processor status on stack. there's a bug on the 6502 that causes the BRK bit to be set when pushing the status on the stack */
#define OP_PHP()	PUSH_STATUS(FLAG_B)
/* pull accumulator from stack. affected flags: negative, zero */
#define OP_PLA()	cpu.P_NZ=cpu.A=PULL()
/* pull processor status from stack */
#define OP_PLP()	tempb=cpu.P_I; \
			PULL_STATUS(); \
			IRQ_PEND(INT_EDGE_IRQ)
/* rotate one bit left, memory. affected flags: negative, zero, carry */
#define OP_ROL()	tempb=DATA; \
			WRITE_BYTE(address,tempb); /* rmw garbage store */ \
			cpu.P_NZ=cpu.P_C=tempb<<1|cpu.P_C>>8; \
			WRITE_BYTE(address,cpu.P_NZ&0xff)
#define OP_aROL()	cpu.P_NZ=cpu.P_C=cpu.A<<1|cpu.P_C>>8; /* same as above, accumulator */ \
			cpu.A=cpu.P_C&0xff
/* rotate one bit right (memory), carries to the right instead of default left. affected flags: negative, zero, carry */
#define OP_ROR()	tempb=DATA; \
			WRITE_BYTE(address,tempb); /* rmw garbage store */ \
			cpu.P_NZ=tempb>>1|(cpu.P_C>>1&0x80); \
			cpu.P_C=tempb<<8&0x100; \
			WRITE_BYTE(address,cpu.P_NZ)
#define OP_aROR()	cpu.P_NZ=cpu.A>>1|(cpu.P_C>>1&0x80); /* same as above, accumulator */ \
			cpu.P_C=cpu.A<<8&0x100; \
			cpu.A=cpu.P_NZ
/* return from interrupt */
#define OP_RTI()	tempb=cpu.P_I; \
			PULL_STATUS(); \
			IRQ_PEND(FALSE); \
			PULL_PC()
/* return from subroutine */
#define OP_RTS()	PULL_PC(); \
			cpu.PC++
/* subtract memory from accumulator with borrow. affected flags: negative, overflow, zero, carry. carry is inverse borrow. almost the same as adc. */
#define OP_SBC()	tempb=~DATA; \
			cpu.P_C=(cpu.P_C>>8)+cpu.A+tempb; \
			cpu.P_V=(cpu.A^cpu.P_C)&(tempb^cpu.P_C)&0x80; \
			cpu.P_NZ=cpu.A=cpu.P_C&0xff
/* set carry flag */
#define OP_SEC()	cpu.P_C=0x100
/* set decimal mode */
#define OP_SED()	cpu.P_D=TRUE
/* set interrupt disable status */
#define OP_SEI()	cpu.P_I=TRUE
/* store accumulator in memory */
#define OP_STA()	WRITE_BYTE(address,cpu.A)
/* store index X in memory */
#define OP_STX()	WRITE_BYTE(address,cpu.X)
/* store index Y in memory */
#define OP_STY()	WRITE_BYTE(address,cpu.Y)
/* transfer accumulator to index X. affected flags: negative, zero */
#define OP_TAX()	cpu.P_NZ=cpu.X=cpu.A
/* transfer accumulator to index Y. affected flags: negative, zero */
#define OP_TAY()	cpu.P_NZ=cpu.Y=cpu.A
/* transfer stack pointer to index X. affected flags: negative, zero */
#define OP_TSX()	cpu.P_NZ=cpu.X=cpu.S
/* transfer index X to accumulator. affected flags: negative, zero */
#define OP_TXA()	cpu.P_NZ=cpu.A=cpu.X
/* transfer index X to stack register */
#define OP_TXS()	cpu.S=cpu.X
/* transfer index Y to accumulator. affected flags: negative, zero */
#define OP_TYA()	cpu.P_NZ=cpu.A=cpu.Y


/*
unofficial ones are not well documented, but i've tried to emulate them anyway:
	AAC, AAX, ARR, ASR, ATX, AXA, AXS, DCP, HLT, ISC, LAR, LAX, RLA, RRA, SLO, SRE, SXA, SYA, XAA, XAS

unofficial 'official':
	NOP (though with more possible addressing modes instead of implied):
		1 byte IMP: 0x1a, 0x3a, 0x5a, 0x7a, 0xda, 0xfa
		2 bytes ZP(/x/y) or IMM: 0x04, 0x14, 0x34, 0x44, 0x54, 0x64, 0x74, 0x80, 0x82, 0x89, 0xc2, 0xd4, 0xe2, 0xf4
		3 bytes ABS(/x): 0x0c, 0x1c, 0x3c, 0x5c, 0x7c, 0xdc, 0xfc
	SBC IMM: 0xeb (same as 0xe9)

depending on unofficial opcodes:
Simpsons - Bart vs World: 0xb: IMM AAC (bad dump?)
Super Cars: 0xb3: IND_Y LAX
(Kevin Horton's nestest)

*/

/* AND+set carry when negative. affected flags: negative, zero, carry */
#define OP_AAC()	OP_AND(); \
			cpu.P_C=cpu.P_NZ<<1
/* store A AND X in memory. registers and flags are unchanged! */
#define OP_AAX()	tempb=cpu.A&cpu.X; \
			WRITE_BYTE(address,tempb)
/* AND byte (ARR is IMM only), ROR, and check bits 5 and 6 (56?cv: 11?10, 00?00, 10?01, 01?11). affected flags: negative, overflow, zero, carry */
#define OP_ARR()	OP_AND(); \
			OP_aROR(); \
			cpu.P_C=cpu.A<<2&0x100; \
			cpu.P_V=cpu.P_C>>1^(cpu.A<<2&0x80)
/* AND, LSR. affected flags: negative, zero, carry */
#define OP_ASR()	OP_AND(); \
			OP_aLSR()
/* AND, store result in X. known to behave differently on different cpu's. affected flags: negative, zero */
#define OP_ATX()	OP_AND(); \
			OP_TAX()
/* AND X with accumulator, AND result with 7, store in memory. registers and flags are unchanged! */
#define OP_AXA()	tempb=cpu.A&cpu.X; \
			tempb&=7; \
			WRITE_BYTE(address,tempb)
/* AND X, store result in X, then subtract byte (AXS is IMM only) from X register (without borrow). affected flags: negative, overflow, zero, carry (docs don't say overflow) */
#define OP_AXS();	cpu.P_NZ=cpu.A&=cpu.X; \
			OP_TAX(); \
			OP_SEC(); /* no borrow */ \
			OP_SBC(); \
			cpu.X^=cpu.A; cpu.A^=cpu.X; cpu.X^=cpu.A; /* swap X with A, since A holds the subtraction result, which should be in X */
/* DEC, CMP. affected flags: negative, zero, carry */
#define OP_DCP()	OP_DEC(); \
			OP_CMP()
/* halt cpu (stop program counter) */
#define OP_HLT()	cpu.halt=TRUE; \
			cpu.cycles=0
/* INC, SBC. affected flags: negative, overflow, zero, carry */
#define OP_ISC()	OP_INC(); \
			OP_SBC()
/* AND memory with stack, transfer result to X, A, stack. affected flags: negative, zero */
#define OP_LAR()	OP_TSX(); \
			OP_TXA(); \
			OP_AND(); \
			OP_TAX(); \
			OP_TXS()
/* load accumulator and X register with memory. affected registers: negative, zero */
#define OP_LAX()	OP_LDA(); \
			OP_LDX()
/* ROL, AND. affected flags: negative, zero, carry */
#define OP_RLA()	OP_ROL(); \
			OP_AND()
/* ROR, ADC (with carry). affected flags: negative, overflow, zero, carry */
#define OP_RRA()	OP_ROR(); \
			OP_ADC()
/* ASL, ORA. affected flags: negative, zero, carry */
#define OP_SLO()	OP_ASL(); \
			OP_ORA()
/* LSR, EOR. affected flags: negative, zero, carry */
#define OP_SRE()	OP_LSR(); \
			OP_EOR()
/* AND X with high byte of addressbus+1, store in memory. registers and flags are unchanged! */
#define OP_SXA()	tempb=cpu.X&((address>>8)+1); \
			WRITE_BYTE(address,tempb)
/* AND Y with high byte of addressbus+1, store in memory. registers and flags are unchanged! */
#define OP_SYA()	tempb=cpu.Y&((address>>8)+1); \
			WRITE_BYTE(address,tempb)
/* TXA, AND. known to behave differently on different cpu's. affected flags: negative, zero */
#define OP_XAA()	OP_TXA(); \
			OP_AND()
/* AND X with accumulator and store in stackpointer, AND stackpointer with high byte of addressbus+1, store in memory. registers (except S) and flags are unchanged! */
#define OP_XAS()	cpu.S=cpu.A&cpu.Y; \
			tempb=cpu.S&((address>>8)+1); \
			WRITE_BYTE(address,tempb)




/* addressing modes, 13(-2) */

#define ADD_EXTRA_CYCLE() if ((tempw^address)&op_extracycles[opcode]) cpu.cycles-=crystal->cycle

/* Accumulator Addressing, eg. ROR A. 1 byte */
/* nothing */
/* Immediate Addressing, eg. LDA #$00. 2 bytes */
#define ADD_IMM()	address=cpu.PC++
/* Absolute Addressing, eg. LDA $1000. 3 bytes */
#define ADD_ABS()	address=READ_WORD(cpu.PC); \
			cpu.PC++; cpu.PC++
/* Zero Page Addressing, eg. LDA $00. 2 bytes */
#define ADD_ZP()	address=READ_BYTE(cpu.PC); \
			cpu.PC++
/* Indexed Zero Page Addressing (X), eg. LDA $00,X. 2 bytes. zero page wraps */
#define ADD_ZP_X()	address=(READ_BYTE(cpu.PC)+cpu.X)&0xff; \
			cpu.PC++
/* Indexed Zero Page Addressing (Y), eg. LDX $00,Y. 2 bytes. zero page wraps */
#define ADD_ZP_Y()	address=(READ_BYTE(cpu.PC)+cpu.Y)&0xff; \
			cpu.PC++
/* Indexed Absolute Addressing (X), eg. LDA $1000,X. 3 bytes. 1 cycle penalty on page boundary cross */
#define ADD_ABS_X()	tempw=READ_WORD(cpu.PC); \
			address=tempw+cpu.X; \
			ADD_EXTRA_CYCLE(); \
			cpu.PC++; cpu.PC++
/* Indexed Absolute Addressing (Y), eg. LDA $1000,Y. 3 bytes. 1 cycle penalty on page boundary cross */
#define ADD_ABS_Y()	tempw=READ_WORD(cpu.PC); \
			address=tempw+cpu.Y; \
			ADD_EXTRA_CYCLE(); \
			cpu.PC++; cpu.PC++
/* Implied Addressing, eg. CLC. 1 byte */
/* nothing */
/* Relative Addressing, eg. BEQ $FE. 2 bytes. only used for branches */
#define ADD_REL()	address=cpu.PC++
/* Indexed Indirect Addressing: (IND, X), eg. LDA ($80,X). 2 bytes. zero page wraps. */
#define ADD_IND_X()	tempb=READ_BYTE(cpu.PC)+cpu.X; \
			cpu.PC++; \
			address=(READ_BYTE_RAM((tempb+1)&0xff)<<8)|READ_BYTE_RAM(tempb)
/* Indirectly Indexed Addressing: (IND), Y, eg. LDA ($80),Y. 2 bytes. zero page wraps. 1 cycle penalty on page boundary cross */
#define ADD_IND_Y()	tempb=READ_BYTE(cpu.PC); \
			tempw=(READ_BYTE_RAM((tempb+1)&0xff)<<8)|READ_BYTE_RAM(tempb); \
			address=tempw+cpu.Y; \
			ADD_EXTRA_CYCLE(); \
			cpu.PC++
/* Absolute Indirect Addressing, eg. JMP ($0080). 3 bytes. only used for indirect jump. there's a bug on the 6502 that it loads xx00 after xxff  */
#define ADD_INDIR()	tempw=READ_WORD(cpu.PC); \
			/* cpu.PC++; cpu.PC++; */ /* no need to increment programcounter */ \
			address=(READ_BYTE((tempw&0xff00)|((tempw+1)&0xff))<<8)|READ_BYTE(tempw)




/* interrupt handlers */

#define DO_INT_NOPUSH(x) cpu.P_I=TRUE; cpu.PC=READ_WORD(x); cpu.cycles-=7*crystal->cycle
#define DO_INT(x,b) PUSH_PC(); PUSH_STATUS(b); DO_INT_NOPUSH(x)

/* no interrupt, shouldn't be called */
static void int_none(void) { return; }
/* software interrupt (brk) handler */
static void int_brk(void) { cpu.interrupt&=~INTERRUPT_BRK; DO_INT(0xfffe,FLAG_B); }
/* interrupt request handler */
static void int_irq(void)
{
	if (cpu.irq_type&INT_EDGE_IRQ) { cpu.irq_type&=~INT_EDGE_IRQ; cpu.irq_edge=!cpu.P_I; }
	else {
		cpu.interrupt&=~INTERRUPT_IRQ;
		if (!cpu.P_I|cpu.irq_edge) { DO_INT(0xfffe,FALSE); cpu.irq_edge=FALSE; }
	}
}
/* non maskable interrupt handler */
static void int_nmi(void)
{
	if (cpu.irq_type&INT_EDGE_NMI) { cpu.irq_type&=~INT_EDGE_NMI; }
	else { cpu.interrupt&=~INTERRUPT_NMI; DO_INT(0xfffa,FALSE); }
}
/* reset handler */
static void int_res(void) { cpu.interrupt=cpu.irq_type=cpu.irq_edge=FALSE; cpu.S-=3; DO_INT_NOPUSH(0xfffc); }
/* multiple interrupts: choose one */
static void int_mul(void)
{
	if (cpu.interrupt&INTERRUPT_RES) int_res();
	else if (cpu.interrupt&INTERRUPT_NMI) {
		if (cpu.interrupt&INTERRUPT_IRQ) {
			if (cpu.irq_type&INT_EDGE_NMI) {
				cpu.irq_type&=~INT_EDGE_NMI;
				if (cpu.interrupt&INTERRUPT_BRK) {
					if (cpu.P_I|((cpu.irq_type&INT_EDGE_IRQ)!=0)) { cpu.irq_type&=~INT_EDGE_IRQ; int_brk(); }
					else int_irq();
				}
				else int_irq();
			}
			else { cpu.irq_type&=~INT_EDGE_IRQ; int_nmi(); }
		}
		else if (cpu.interrupt&INTERRUPT_BRK) {
			if (cpu.irq_type&INT_EDGE_NMI) { cpu.irq_type&=~INT_EDGE_NMI; int_brk(); }
			else int_nmi();
		}
		else int_nmi();
	}
	else if (cpu.interrupt&INTERRUPT_IRQ) {
		if (cpu.interrupt&INTERRUPT_BRK) {
			if (cpu.P_I|((cpu.irq_type&INT_EDGE_IRQ)!=0)) { cpu.irq_type&=~INT_EDGE_IRQ; int_brk(); }
			else int_irq();
		}
		else int_irq();
	}
}



/**************************/

static __inline__ void update_sync(void)
{
	if (cpu.cycles<=cpu.apu_dmc_next_step) apu_run_dmc();
	if (cpu.cycles<=cpu.apu_fs_next_step) apu_run_fs();
	(*mapper_irq)();
}

void cpu_set_fpt(BYTE type)
{
	const fpt_cpu_read_memorymap cpu_read_memorymap_std[8]= { cpu_read_ram, cpu_read_ppu, cpu_read_io, cpu_read_wram, cpu_read_prgbank1a, cpu_read_prgbank1b, cpu_read_prgbank2a, cpu_read_prgbank2b };
	const fpt_cpu_write_memorymap cpu_write_memorymap_std[8]= { cpu_write_ram, cpu_write_ppu, cpu_write_io, cpu_write_wram, cpu_write_prgbank1a, cpu_write_prgbank1b, cpu_write_prgbank2a, cpu_write_prgbank2b };
	BYTE i;
	
	if (type) {
		switch (type) {
			case CPU_FPT_WRAM_READ: cpu_read_memorymap[3]=cpu_read_wram_override; break;
			case CPU_FPT_WRAM_WRITE: cpu_write_memorymap[3]=cpu_write_wram_override; break;
			case CPU_FPT_PRG1A_READ: cpu_read_memorymap[4]=cpu_read_prgbank1a_override; break;
			case CPU_FPT_PRG1B_READ: cpu_read_memorymap[5]=cpu_read_prgbank1b_override; break;
			case CPU_FPT_PRG2A_READ: cpu_read_memorymap[6]=cpu_read_prgbank2a_override; break;
			case CPU_FPT_PRG2B_READ: cpu_read_memorymap[7]=cpu_read_prgbank2b_override; break;
			default: break;
		}
	}
	else {
		for (i=0;i<9;i++) {
			cpu_read_memorymap[i]=cpu_read_memorymap_std[i&7];
			cpu_write_memorymap[i]=cpu_write_memorymap_std[i&7];
		}
	}
}

void illegal_nothing(BYTE opcode) { return; }
void illegal_show(BYTE opcode)
{
	if (!op_official[opcode]) {
		LOG(LOG_MISC|LOG_WARNING,"illegal CPU opcode at 0x%04x:0x%02x(%s %s)\n",(cpu.PC-1)&0xffff,opcode,addressing_mode_names[op_addressing_mode[opcode]],c_op_mnemonic[op_mnemonic[opcode]]);
		cpu.illegal_count--;
		if (cpu.illegal_count==0) {
			_illegal=illegal_nothing;
			LOG(LOG_MISC,"etc.\n");
		}
	}
}

void __fastcall cpu_set_apu_fs_next_step(int freq)
{
	cpu.apu_fs_next_step-=freq;
}

void __fastcall cpu_reset_apu_fs_next_step(void)
{
	cpu.apu_fs_next_step=cpu.cycles-crystal->cycle;
}

BYTE __fastcall cpu_set_apu_dmc_next_step(void)
{
	cpu.apu_dmc_next_step-=cpu.apu_dmc_frequency;
	return cpu.cycles<=cpu.apu_dmc_next_step;
}

void __fastcall cpu_reset_apu_dmc_freq(int freq)
{
	cpu.apu_dmc_frequency=freq;
}

void cpu_set_mapper_irq(fp_mapper_irq fun)
{
	if (fun==NULL) fun=mapirq_nothing;
	mapper_irq=fun;
}

int* cpu_get_cycles_ptr(void)
{
	return &cpu.cycles;
}

void cpu_set_overflow(void)
{
	cpu.P_V=TRUE;
}

void cpu_set_vblank_nmi_want(void)
{
	cpu.nmi=TRUE;
}

void cpu_do_vblank_nmi(void)
{
	if (cpu.nmi) {
		cpu.nmi=FALSE;
		cpu.interrupt|=INTERRUPT_NMI;
		if ((cpu.cycles+(CPU_NMI_CYCLES_WAIT*CRYSTAL_PPU_CYCLE_FIXED))<crystal->vblank_trigger) handle_interrupt[cpu.interrupt]();
	}
}

void cpu_set_interrupt(WORD b)
{
	if (b&INTERRUPT_IRQ) cpu.irq_pend=TRUE;
	cpu.interrupt|=(b&BIN8(00001111));
	cpu.irq_type|=((b&BIN8(11110000))|(b>>8));
}

void cpu_acknowledge_interrupt(BYTE b)
{
	cpu.irq_type&=b;
	if ((cpu.irq_type&~(INT_EDGE_NMI|INT_EDGE_IRQ))==0) cpu.irq_pend=FALSE;
}

void cpu_fill_op_cycles_lookup(void)
{
	int i;
	for(i=0;i<0x100;i++) op_cycles_lookup[i]=op_cycles[i]*crystal->cycle;
}

void __fastcall cpu_sprite_dma_transfer(register BYTE addressbyte)
{
	WORD i=0x100;
	WORD address=addressbyte<<8;
	const int cc=crystal->cycle<<1;
	cpu.cycles-=crystal->cycle;
	for (;i>0;i--) {
		ppu_write_sprite_memory_data(READ_BYTE(address));
		address++;
		cpu.cycles-=cc;
	}

	update_sync();
}

BYTE __fastcall cpu_dmc_dma_transfer(register WORD address)
{
	cpu.cycles-=(crystal->cycle<<2);
	return READ_BYTE(address);
}

void cpu_kill_cycle(void)
{
	cpu.cycles-=CRYSTAL_PPU_CYCLE_FIXED;
	cpu.apu_dmc_next_step-=CRYSTAL_PPU_CYCLE_FIXED;
	cpu.apu_fs_next_step-=CRYSTAL_PPU_CYCLE_FIXED;
	mapper_kill_cycle();
	
	update_sync();
}

void cpu_reset(void)
{
	cpu.illegal_count=CPU_ILLEGAL_MAX;
	if (cpu.illegal_count) _illegal=illegal_show;
	else _illegal=illegal_nothing;
	
	cpu.interrupt|=INTERRUPT_RES;
	handle_interrupt[cpu.interrupt]();
	LOG(LOG_MISC,"CPU reset\n");
}

void cpu_init(void)
{
	memset(&cpu,0,sizeof(cpu));
	cpu_fill_op_cycles_lookup();
	cpu.P_NZ=1;
	
	#if !CPU_EMPTY_RAM
	WRITE_BYTE(0x0008,0xf7);
	WRITE_BYTE(0x0009,0xef);
	WRITE_BYTE(0x000a,0xdf);
	WRITE_BYTE(0x000f,0xbf);
	#endif
	
	#if CPU_INTERRUPT_DISABLED
	mapper_irq=mapirq_nothing;
	#endif
	
	LOG(LOG_VERBOSE,"CPU initialised\n");
}

void cpu_clean(void)
{
	memset(&cpu,0,sizeof(cpu));
	
	LOG(LOG_VERBOSE,"CPU cleaned\n");
}

void cpu_new_frame(void)
{
	cpu.cycles+=crystal->frame;
	cpu.apu_fs_next_step+=crystal->frame;
	cpu.apu_dmc_next_step+=crystal->frame;
}

#define _ILL() (*_illegal)(opcode)

void cpu_execute(register const int border)
{
	#if DEBUG_CPU
	WORD temp_pc;
	char in_int[3];
	#endif
	
	register WORD tempw;
	register BYTE tempb;
	register BYTE opcode;
	register WORD address;
	
	execute_start:
	
	/*ppu_write_control2(*(ppu_get_control2_ptr())|!cpu.P_I);*/ /* hacked in to see irq status */
	
	opcode=READ_BYTE(cpu.PC);
	cpu.cycles-=op_cycles_lookup[opcode];

	update_sync();

	#if DEBUG_CPU
	address=0;
	temp_pc=cpu.PC;
	#endif

	cpu.PC++;
	
	switch (opcode) {
/*0*/
	case 0x00: /*ADD_IMP*/ OP_BRK(); break;
	case 0x01: ADD_IND_X(); OP_ORA(); break;
	case 0x02: _ILL(); /*ADD_IMP*/ OP_HLT(); break; /*!*/
	case 0x03: _ILL(); ADD_IND_X(); OP_SLO(); break; /*!*/
	case 0x04: _ILL(); ADD_ZP(); /*OP_NOP*/ break; /*!*/
	case 0x05: ADD_ZP(); OP_ORA(); break;
	case 0x06: ADD_ZP(); OP_ASL(); break;
	case 0x07: _ILL(); ADD_ZP(); OP_SLO(); break; /*!*/
	case 0x08: /*ADD_IMP*/ OP_PHP(); break;
	case 0x09: ADD_IMM(); OP_ORA(); break;
	case 0x0a: /*ADD_ACCUM*/ OP_aASL(); break;
	case 0x0b: _ILL(); ADD_IMM(); OP_AAC(); break; /*!*/
	case 0x0c: _ILL(); ADD_ABS(); /*OP_NOP*/ break; /*!*/
	case 0x0d: ADD_ABS(); OP_ORA(); break;
	case 0x0e: ADD_ABS(); OP_ASL(); break;
	case 0x0f: _ILL(); ADD_ABS(); OP_SLO(); break; /*!*/
/*1*/
	case 0x10: ADD_REL(); OP_BPL(); break;
	case 0x11: ADD_IND_Y(); OP_ORA(); break;
	case 0x12: _ILL(); /*ADD_IMP*/ OP_HLT(); break; /*!*/
	case 0x13: _ILL(); ADD_IND_Y(); OP_SLO(); break; /*!*/
	case 0x14: _ILL(); ADD_ZP_X(); /*OP_NOP*/ break; /*!*/
	case 0x15: ADD_ZP_X(); OP_ORA(); break;
	case 0x16: ADD_ZP_X(); OP_ASL(); break;
	case 0x17: _ILL(); ADD_ZP_X(); OP_SLO(); break; /*!*/
	case 0x18: /*ADD_IMP*/ OP_CLC(); break;
	case 0x19: ADD_ABS_Y(); OP_ORA(); break;
	case 0x1a: _ILL(); /*ADD_IMP*/ /*OP_NOP*/ break; /*!*/
	case 0x1b: _ILL(); ADD_ABS_Y(); OP_SLO(); break; /*!*/
	case 0x1c: _ILL(); ADD_ABS_X(); /*OP_NOP*/ break; /*!*/
	case 0x1d: ADD_ABS_X(); OP_ORA(); break;
	case 0x1e: ADD_ABS_X(); OP_ASL(); break;
	case 0x1f: _ILL(); ADD_ABS_X(); OP_SLO(); break; /*!*/
/*2*/
	case 0x20: ADD_ABS(); OP_JSR(); break;
	case 0x21: ADD_IND_X(); OP_AND(); break;
	case 0x22:_ILL();  /*ADD_IMP*/ OP_HLT(); break; /*!*/
	case 0x23: _ILL(); ADD_IND_X(); OP_RLA(); break; /*!*/
	case 0x24: ADD_ZP(); OP_BIT(); break;
	case 0x25: ADD_ZP(); OP_AND(); break;
	case 0x26: ADD_ZP(); OP_ROL(); break;
	case 0x27: _ILL(); ADD_ZP(); OP_RLA(); break; /*!*/
	case 0x28: /*ADD_IMP*/ OP_PLP(); break;
	case 0x29: ADD_IMM(); OP_AND(); break;
	case 0x2a: /*ADD_ACCUM*/ OP_aROL(); break;
	case 0x2b: _ILL(); ADD_IMM(); OP_AAC(); break; /*!*/
	case 0x2c: ADD_ABS(); OP_BIT(); break;
	case 0x2d: ADD_ABS(); OP_AND(); break;
	case 0x2e: ADD_ABS(); OP_ROL(); break;
	case 0x2f: _ILL(); ADD_ABS(); OP_RLA(); break; /*!*/
/*3*/
	case 0x30: ADD_REL(); OP_BMI(); break;
	case 0x31: ADD_IND_Y(); OP_AND(); break;
	case 0x32: _ILL(); /*ADD_IMP*/ OP_HLT(); break; /*!*/
	case 0x33: _ILL(); ADD_IND_Y(); OP_RLA(); break; /*!*/
	case 0x34: _ILL(); ADD_ZP_X(); /*OP_NOP*/ break; /*!*/
	case 0x35: ADD_ZP_X(); OP_AND(); break;
	case 0x36: ADD_ZP_X(); OP_ROL(); break;
	case 0x37: _ILL(); ADD_ZP_X(); OP_RLA(); break; /*!*/
	case 0x38: /*ADD_IMP*/ OP_SEC(); break;
	case 0x39: ADD_ABS_Y(); OP_AND(); break;
	case 0x3a: _ILL(); /*ADD_IMP*/ /*OP_NOP*/ break; /*!*/
	case 0x3b: _ILL(); ADD_ABS_Y(); OP_RLA(); break; /*!*/
	case 0x3c: _ILL(); ADD_ABS_X(); /*OP_NOP*/ break; /*!*/
	case 0x3d: ADD_ABS_X(); OP_AND(); break;
	case 0x3e: ADD_ABS_X(); OP_ROL(); break;
	case 0x3f: _ILL(); ADD_ABS_X(); OP_RLA(); break; /*!*/
/*4*/
	case 0x40: /*ADD_IMP*/ OP_RTI(); break;
	case 0x41: ADD_IND_X(); OP_EOR(); break;
	case 0x42: _ILL(); /*ADD_IMP*/ OP_HLT(); break; /*!*/
	case 0x43: _ILL(); ADD_IND_X(); OP_SRE(); break; /*!*/
	case 0x44: _ILL(); ADD_ZP(); /*OP_NOP*/ break; /*!*/
	case 0x45: ADD_ZP(); OP_EOR(); break;
	case 0x46: ADD_ZP(); OP_LSR(); break;
	case 0x47: _ILL(); ADD_ZP(); OP_SRE(); break; /*!*/
	case 0x48: /*ADD_IMP*/ OP_PHA(); break;
	case 0x49: ADD_IMM(); OP_EOR(); break;
	case 0x4a: /*ADD_ACCUM*/ OP_aLSR(); break;
	case 0x4b: _ILL(); ADD_IMM(); OP_ASR(); break; /*!*/
	case 0x4c: ADD_ABS(); OP_JMP(); break;
	case 0x4d: ADD_ABS(); OP_EOR(); break;
	case 0x4e: ADD_ABS(); OP_LSR(); break;
	case 0x4f: _ILL(); ADD_ABS(); OP_SRE(); break; /*!*/
/*5*/
	case 0x50: ADD_REL(); OP_BVC(); break;
	case 0x51: ADD_IND_Y(); OP_EOR(); break;
	case 0x52: _ILL(); /*ADD_IMP*/ OP_HLT(); break; /*!*/
	case 0x53: _ILL(); ADD_IND_Y(); OP_SRE(); break; /*!*/
	case 0x54: _ILL(); ADD_ZP_X(); /*OP_NOP*/ break; /*!*/
	case 0x55: ADD_ZP_X(); OP_EOR(); break;
	case 0x56: ADD_ZP_X(); OP_LSR(); break;
	case 0x57: _ILL(); ADD_ZP_X(); OP_SRE(); break; /*!*/
	case 0x58: /*ADD_IMP*/ OP_CLI(); break;
	case 0x59: ADD_ABS_Y(); OP_EOR(); break;
	case 0x5a: _ILL(); /*ADD_IMP*/ /*OP_NOP*/ break; /*!*/
	case 0x5b: _ILL(); ADD_ABS_Y(); OP_SRE(); break; /*!*/
	case 0x5c: _ILL(); ADD_ABS_X(); /*OP_NOP*/ break; /*!*/
	case 0x5d: ADD_ABS_X(); OP_EOR(); break;
	case 0x5e: ADD_ABS_X(); OP_LSR(); break;
	case 0x5f: _ILL(); ADD_ABS_X(); OP_SRE(); break; /*!*/
/*6*/
	case 0x60: /*ADD_IMP*/ OP_RTS(); break;
	case 0x61: ADD_IND_X(); OP_ADC(); break;
	case 0x62: _ILL(); /*ADD_IMP*/ OP_HLT(); break; /*!*/
	case 0x63: _ILL(); ADD_IND_X(); OP_RRA(); break; /*!*/
	case 0x64: _ILL(); ADD_ZP(); /*OP_NOP*/ break; /*!*/
	case 0x65: ADD_ZP(); OP_ADC(); break;
	case 0x66: ADD_ZP(); OP_ROR(); break;
	case 0x67: _ILL(); ADD_ZP(); OP_RRA(); break; /*!*/
	case 0x68: /*ADD_IMP*/ OP_PLA(); break;
	case 0x69: ADD_IMM(); OP_ADC(); break;
	case 0x6a: /*ADD_ACCUM*/ OP_aROR(); break;
	case 0x6b: _ILL(); ADD_IMM(); OP_ARR(); break; /*!*/
	case 0x6c: ADD_INDIR(); OP_JMP(); break;
	case 0x6d: ADD_ABS(); OP_ADC(); break;
	case 0x6e: ADD_ABS(); OP_ROR(); break;
	case 0x6f: _ILL(); ADD_ABS(); OP_RRA(); break; /*!*/
/*7*/
	case 0x70: ADD_REL(); OP_BVS(); break;
	case 0x71: ADD_IND_Y(); OP_ADC(); break;
	case 0x72: _ILL(); /*ADD_IMP*/ OP_HLT(); break; /*!*/
	case 0x73: _ILL(); ADD_IND_Y(); OP_RRA(); break; /*!*/
	case 0x74: _ILL(); ADD_ZP_X(); /*OP_NOP*/ break; /*!*/
	case 0x75: ADD_ZP_X(); OP_ADC(); break;
	case 0x76: ADD_ZP_X(); OP_ROR(); break;
	case 0x77: _ILL(); ADD_ZP_X(); OP_RRA(); break; /*!*/
	case 0x78: /*ADD_IMP*/ OP_SEI(); break;
	case 0x79: ADD_ABS_Y(); OP_ADC(); break;
	case 0x7a: _ILL(); /*ADD_IMP*/ /*OP_NOP*/ break; /*!*/
	case 0x7b: _ILL(); ADD_ABS_Y(); OP_RRA(); break; /*!*/
	case 0x7c: _ILL(); ADD_ABS_X(); /*OP_NOP*/ break; /*!*/
	case 0x7d: ADD_ABS_X(); OP_ADC(); break;
	case 0x7e: ADD_ABS_X(); OP_ROR(); break;
	case 0x7f: _ILL(); ADD_ABS_X(); OP_RRA(); break; /*!*/
/*8*/
	case 0x80: _ILL(); ADD_IMM(); /*OP_NOP*/ break; /*!*/
	case 0x81: ADD_IND_X(); OP_STA(); break;
	case 0x82: _ILL(); ADD_IMM(); /*OP_NOP*/ break; /*!*/
	case 0x83: _ILL(); ADD_IND_X(); OP_AAX(); break; /*!*/
	case 0x84: ADD_ZP(); OP_STY(); break;
	case 0x85: ADD_ZP(); OP_STA(); break;
	case 0x86: ADD_ZP(); OP_STX(); break;
	case 0x87: _ILL(); ADD_ZP(); OP_AAX(); break; /*!*/
	case 0x88: /*ADD_IMP*/ OP_DEY(); break;
	case 0x89: _ILL(); ADD_IMM(); /*OP_NOP*/ break; /*!*/
	case 0x8a: /*ADD_IMP*/ OP_TXA(); break;
	case 0x8b: _ILL(); ADD_IMM(); OP_XAA(); break; /*!*/
	case 0x8c: ADD_ABS(); OP_STY(); break;
	case 0x8d: ADD_ABS(); OP_STA(); break;
	case 0x8e: ADD_ABS(); OP_STX(); break;
	case 0x8f: _ILL(); ADD_ABS(); OP_AAX(); break; /*!*/
/*9*/
	case 0x90: ADD_REL(); OP_BCC(); break;
	case 0x91: ADD_IND_Y(); OP_STA(); break;
	case 0x92: _ILL(); /*ADD_IMP*/ OP_HLT(); break; /*!*/
	case 0x93: _ILL(); ADD_IND_Y(); OP_AXA(); break; /*!*/
	case 0x94: ADD_ZP_X(); OP_STY(); break;
	case 0x95: ADD_ZP_X(); OP_STA(); break;
	case 0x96: ADD_ZP_Y(); OP_STX(); break;
	case 0x97: _ILL(); ADD_ZP_Y(); OP_AAX(); break; /*!*/
	case 0x98: /*ADD_IMP*/ OP_TYA(); break;
	case 0x99: ADD_ABS_Y(); OP_STA(); break;
	case 0x9a: /*ADD_IMP*/ OP_TXS(); break;
	case 0x9b: _ILL(); ADD_ABS_Y(); OP_XAS(); break; /*!*/
	case 0x9c: _ILL(); ADD_ABS_X(); OP_SYA(); break; /*!*/
	case 0x9d: ADD_ABS_X(); OP_STA(); break;
	case 0x9e: _ILL(); ADD_ABS_Y(); OP_SXA(); break; /*!*/
	case 0x9f: _ILL(); ADD_ABS_Y(); OP_AXA(); break; /*!*/
/*a*/
	case 0xa0: ADD_IMM(); OP_LDY(); break;
	case 0xa1: ADD_IND_X(); OP_LDA(); break;
	case 0xa2: ADD_IMM(); OP_LDX(); break;
	case 0xa3: _ILL(); ADD_IND_X(); OP_LAX(); break; /*!*/
	case 0xa4: ADD_ZP(); OP_LDY(); break;
	case 0xa5: ADD_ZP(); OP_LDA(); break;
	case 0xa6: ADD_ZP(); OP_LDX(); break;
	case 0xa7: _ILL(); ADD_ZP(); OP_LAX(); break; /*!*/
	case 0xa8: /*ADD_IMP*/ OP_TAY(); break;
	case 0xa9: ADD_IMM(); OP_LDA(); break;
	case 0xaa: /*ADD_IMP*/ OP_TAX(); break;
	case 0xab: _ILL(); ADD_IMM(); OP_ATX(); break; /*!*/
	case 0xac: ADD_ABS(); OP_LDY(); break;
	case 0xad: ADD_ABS(); OP_LDA(); break;
	case 0xae: ADD_ABS(); OP_LDX(); break;
	case 0xaf: _ILL(); ADD_ABS(); OP_LAX(); break; /*!*/
/*b*/
	case 0xb0: ADD_REL(); OP_BCS(); break;
	case 0xb1: ADD_IND_Y(); OP_LDA(); break;
	case 0xb2: _ILL(); /*ADD_IMP*/ OP_HLT(); break; /*!*/
	case 0xb3: _ILL(); ADD_IND_Y(); OP_LAX(); break; /*!*/
	case 0xb4: ADD_ZP_X(); OP_LDY(); break;
	case 0xb5: ADD_ZP_X(); OP_LDA(); break;
	case 0xb6: ADD_ZP_Y(); OP_LDX(); break;
	case 0xb7: _ILL(); ADD_ZP_Y(); OP_LAX(); break; /*!*/
	case 0xb8: /*ADD_IMP*/ OP_CLV(); break;
	case 0xb9: ADD_ABS_Y(); OP_LDA(); break;
	case 0xba: /*ADD_IMP*/ OP_TSX(); break;
	case 0xbb: _ILL(); ADD_ABS_Y(); OP_LAR(); break; /*!*/
	case 0xbc: ADD_ABS_X(); OP_LDY(); break;
	case 0xbd: ADD_ABS_X(); OP_LDA(); break;
	case 0xbe: ADD_ABS_Y(); OP_LDX(); break;
	case 0xbf: _ILL(); ADD_ABS_Y(); OP_LAX(); break; /*!*/
/*c*/
	case 0xc0: ADD_IMM(); OP_CPY(); break;
	case 0xc1: ADD_IND_X(); OP_CMP(); break;
	case 0xc2: _ILL(); ADD_IMM(); /*OP_NOP*/ break; /*!*/
	case 0xc3: _ILL(); ADD_IND_X(); OP_DCP(); break; /*!*/
	case 0xc4: ADD_ZP(); OP_CPY(); break;
	case 0xc5: ADD_ZP(); OP_CMP(); break;
	case 0xc6: ADD_ZP(); OP_DEC(); break;
	case 0xc7: _ILL(); ADD_ZP(); OP_DCP(); break; /*!*/
	case 0xc8: /*ADD_IMP*/ OP_INY(); break;
	case 0xc9: ADD_IMM(); OP_CMP(); break;
	case 0xca: /*ADD_IMP*/ OP_DEX(); break;
	case 0xcb: _ILL(); ADD_IMM(); OP_AXS(); break; /*!*/
	case 0xcc: ADD_ABS(); OP_CPY(); break;
	case 0xcd: ADD_ABS(); OP_CMP(); break;
	case 0xce: ADD_ABS(); OP_DEC(); break;
	case 0xcf: _ILL(); ADD_ABS(); OP_DCP(); break; /*!*/
/*d*/
	case 0xd0: ADD_REL(); OP_BNE(); break;
	case 0xd1: ADD_IND_Y(); OP_CMP(); break;
	case 0xd2: _ILL(); /*ADD_IMP*/ OP_HLT(); break; /*!*/
	case 0xd3: _ILL(); ADD_IND_Y(); OP_DCP(); break; /*!*/
	case 0xd4: _ILL(); ADD_ZP_X(); /*OP_NOP*/ break; /*!*/
	case 0xd5: ADD_ZP_X(); OP_CMP(); break;
	case 0xd6: ADD_ZP_X(); OP_DEC(); break;
	case 0xd7: _ILL(); ADD_ZP_X(); OP_DCP(); break; /*!*/
	case 0xd8: /*ADD_IMP*/ OP_CLD(); break;
	case 0xd9: ADD_ABS_Y(); OP_CMP(); break;
	case 0xda: _ILL(); /*ADD_IMP*/ /*OP_NOP*/ break; /*!*/
	case 0xdb: _ILL(); ADD_ABS_Y(); OP_DCP(); break; /*!*/
	case 0xdc: _ILL(); ADD_ABS_X(); /*OP_NOP*/ break; /*!*/
	case 0xdd: ADD_ABS_X(); OP_CMP(); break;
	case 0xde: ADD_ABS_X(); OP_DEC(); break;
	case 0xdf: _ILL(); ADD_ABS_X(); OP_DCP(); break; /*!*/
/*e*/
	case 0xe0: ADD_IMM(); OP_CPX(); break;
	case 0xe1: ADD_IND_X(); OP_SBC(); break;
	case 0xe2: _ILL(); ADD_IMM(); /*OP_NOP*/ break; /*!*/
	case 0xe3: _ILL(); ADD_IND_X(); OP_ISC(); break; /*!*/
	case 0xe4: ADD_ZP(); OP_CPX(); break;
	case 0xe5: ADD_ZP(); OP_SBC(); break;
	case 0xe6: ADD_ZP(); OP_INC(); break;
	case 0xe7: _ILL(); ADD_ZP(); OP_ISC(); break; /*!*/
	case 0xe8: /*ADD_IMP*/ OP_INX(); break;
	case 0xe9: ADD_IMM(); OP_SBC(); break;
	case 0xea: /*ADD_IMP*/ /*OP_NOP*/ break;
	case 0xeb: _ILL(); ADD_IMM(); OP_SBC(); break; /*!*/
	case 0xec: ADD_ABS(); OP_CPX(); break;
	case 0xed: ADD_ABS(); OP_SBC(); break;
	case 0xee: ADD_ABS(); OP_INC(); break;
	case 0xef: _ILL(); ADD_ABS(); OP_ISC(); break; /*!*/
/*f*/
	case 0xf0: ADD_REL(); OP_BEQ(); break;
	case 0xf1: ADD_IND_Y(); OP_SBC(); break;
	case 0xf2: _ILL(); /*ADD_IMP*/ OP_HLT(); break; /*!*/
	case 0xf3: _ILL(); ADD_IND_Y(); OP_ISC(); break; /*!*/
	case 0xf4: _ILL(); ADD_ZP_X(); /*OP_NOP*/ break; /*!*/
	case 0xf5: ADD_ZP_X(); OP_SBC(); break;
	case 0xf6: ADD_ZP_X(); OP_INC(); break;
	case 0xf7: _ILL(); ADD_ZP_X(); OP_ISC(); break; /*!*/
	case 0xf8: /*ADD_IMP*/ OP_SED(); break;
	case 0xf9: ADD_ABS_Y(); OP_SBC(); break;
	case 0xfa: _ILL(); /*ADD_IMP*/ /*OP_NOP*/ break; /*!*/
	case 0xfb: _ILL(); ADD_ABS_Y(); OP_ISC(); break; /*!*/
	case 0xfc: _ILL(); ADD_ABS_X(); /*OP_NOP*/ break; /*!*/
	case 0xfd: ADD_ABS_X(); OP_SBC(); break;
	case 0xfe: ADD_ABS_X(); OP_INC(); break;
	case 0xff: _ILL(); ADD_ABS_X(); OP_ISC(); break; /*!*/
	
	}
	
	#if DEBUG_CPU
	sprintf(in_int,"%s%s",cpu.interrupt?"i":"",cpu.irq_pend?"p":"");
	#endif
	
	if (cpu.interrupt) handle_interrupt[cpu.interrupt]();

	#if DEBUG_CPU
	LOG(LOG_CPU,"%s%x %s %s PC:%s%x A:%x X:%x Y:%x P:%d%d%dB%d%d%d%d ad:%x\n", \
	op_official[opcode]?"":"!",opcode,c_op_mnemonic[op_mnemonic[opcode]],addressing_mode_names[op_addressing_mode[opcode]], \
	in_int,temp_pc,cpu.A,cpu.X,cpu.Y,((cpu.P_NZ>>8|cpu.P_NZ)&0x80)>>7,cpu.P_V!=0,1,cpu.P_D,cpu.P_I, \
	(cpu.P_NZ&0xff)==0,cpu.P_C>>8,address);
	#endif

	if(cpu.cycles<=border) goto execute_end;
	else goto execute_start;
	
	execute_end:
	return;
}
